!===============================================================================
! Copyright 2021-2022 Intel Corporation.
!
! This software and the related documents are Intel copyrighted  materials,  and
! your use of  them is  governed by the  express license  under which  they were
! provided to you (License).  Unless the License provides otherwise, you may not
! use, modify, copy, publish, distribute,  disclose or transmit this software or
! the related documents without Intel's prior written permission.
!
! This software and the related documents  are provided as  is,  with no express
! or implied  warranties,  other  than those  that are  expressly stated  in the
! License.
!===============================================================================

! Content:
! A simple example of asynchronous, double-precision real-to-complex, 
! complex-to-real in-place 1D FFT using Intel(R) oneAPI Math Kernel Library 
! (oneMKL) DFTI
!
!*****************************************************************************

include "mkl_dfti_omp_offload.f90"

program dp_real_1d_async

  use MKL_DFTI_OMP_OFFLOAD, forget => DFTI_DOUBLE, DFTI_DOUBLE => DFTI_DOUBLE_R
  use omp_lib, ONLY : omp_get_num_devices
  use, intrinsic :: ISO_C_BINDING
  ! Size of 1D transform
  integer, parameter :: N = 16
  
  integer, parameter :: halfNplus1 = N/2 + 1

  ! Arbitrary harmonic used to verify FFT
  integer, parameter :: H = 1

  ! Working precision is double precision
  integer, parameter :: WP = selected_real_kind(15,307)

  ! Execution status
  integer :: status = 0, ignored_status

  ! The data array
  real(WP), allocatable :: x (:)

  ! DFTI descriptor handle
  type(DFTI_DESCRIPTOR), POINTER :: hand

  hand => null()

  print *,"Example dp_real_1d_async"
  print *,"Asynchronous forward and backward double-precision real-to-complex",&
    &      " and complex-to-real in-place 1D transform"
  print *,"Configuration parameters:"
  print *,"DFTI_PRECISION      = DFTI_DOUBLE"
  print *,"DFTI_FORWARD_DOMAIN = DFTI_REAL"
  print *,"DFTI_DIMENSION      = 1"
  print '(" DFTI_LENGTHS        = /"I0"/" )', N

  print *,"Create DFTI descriptor"
  status = DftiCreateDescriptor(hand, DFTI_DOUBLE, DFTI_REAL, 1, N)
  if (0 /= status) goto 999
  
  print *,"Set DFTI descriptor for CCE storage "
  status = DftiSetValue(hand, DFTI_CONJUGATE_EVEN_STORAGE, DFTI_COMPLEX_COMPLEX)
  if (0 /= status) goto 999

  print *,"Commit DFTI descriptor"
  !$omp target variant dispatch
  status = DftiCommitDescriptor(hand)
  !$omp end target variant dispatch
  if (0 /= status) goto 999

  print *,"Allocate array for input/output data"
  allocate ( x(2*halfNplus1), STAT = status)
  if (0 /= status) goto 999

  print *,"Initialize input for real-to-complex forward FFT "
  call init_r(x, N, H)

  print *,"Compute forward transform"
  !$omp target data map(tofrom:x)
  !$omp target variant dispatch use_device_ptr(x) nowait
  status = DftiComputeForward(hand, x)
  !$omp end target variant dispatch
  !$omp taskwait
  !$omp end target data
  if (0 /= status) goto 999

  print *,"Verify the complex result"
  status = verify_c(x, N, H)
  if (0 /= status) goto 999

  print *,"Initialize input for complex-to-real backward transform"
  call init_c(x, N, H)

  print *,"Compute backward transform"
  !$omp target data map(tofrom:x)
  !$omp target variant dispatch use_device_ptr(x) nowait
  status = DftiComputeBackward(hand, x)
  !$omp end target variant dispatch
  !$omp taskwait
  !$omp end target data
  if (0 /= status) goto 999

  print *,"Verify the result"
  status = verify_r(x, N, H)
  if (0 /= status) goto 999

100 continue

  print *,"Release the DFTI descriptor"
  ignored_status = DftiFreeDescriptor(hand)

  if (allocated(x)) then
      print *,"Deallocate input data array"
      deallocate(x)
  endif

  if (status == 0) then
    print *,"TEST PASSED"
    call exit(0)
  else
    print *,"TEST FAILED"
    call exit(1)
  endif

999 print '("  Error, status = ",I0)', status
  goto 100

contains

  ! Compute mod(K*L,M) accurately
  pure real(WP) function moda(k,l,m)
    integer, intent(in) :: k,l,m
    integer*8 :: k8
    k8 = k
    moda = real(mod(k8*l,m),WP)
  end function moda

  ! Initialize real array x to produce unit peaks at y(H) and y(N-H)
  subroutine init_r(x, N, H)
    integer N, H
    real(WP) :: x(:)

    integer k
    real(WP), parameter :: TWOPI = 6.2831853071795864769_WP
    real(WP) :: factor
    if (mod(2*(N - H), N) == 0) then
      factor = 1.0_WP
    else
      factor = 2.0_WP
    end if
    
    do k = 1, N
      x(k) = factor * cos(TWOPI*moda(k-1, H, N)/N) / N
    end do
  end subroutine init_r

  ! Verify that y(k) is unit peak at k = H
  integer function verify_c(y, N, H)
    integer N, H
    real(WP) :: y(:)

    integer k
    real(WP) err, errthr, maxerr
    complex(WP) res_exp, res_got

    ! Note, this simple error bound doesn't take into account error of
    ! input data
    errthr = 2.5 * log(real(N, WP)) / log(2.0_WP) * EPSILON(1.0_WP)
    print '("  Check if err is below errthr " G10.3)', errthr

    maxerr = 0.0_WP
    do k = 1, N/2+1
      if (mod(k-1-H,N)==0 .OR. mod(1-k-H,N)==0) then
        res_exp = 1.0_WP
      else
        res_exp = 0.0_WP
      end if
      res_got = DCMPLX(y(2*k-1), y(2*k))
      err = abs(res_got - res_exp)
      maxerr = max(err,maxerr)
      if (.not.(err < errthr)) then
        print '("  y("I0"): "$)', k
        print '(" expected ("G24.17", "G24.17"),"$)', res_exp
        print '(" got ("G24.17", "G24.17"),"$)', res_got
        print '(" err "G10.3)', err
        print *," Verification FAILED"
        verify_c = 100
        return
      end if
    end do
    print '("  Verified,  maximum error was " G10.3)', maxerr
    verify_c = 0
  end function verify_c
  
  ! Initialize complex array y to produce unit peaks at x(H)
  subroutine init_c(y, N, H)
    integer N, H
    real(WP) :: y(:)

    integer k
    real(WP), parameter :: TWOPI = 6.2831853071795864769_WP
    real(WP) :: TWOPI_phase
    
    do k = 1, N/2 + 1
      TWOPI_phase = TWOPI*moda(k-1, H, N)/N
      y(2*k-1)  =  cos(TWOPI_phase)/N ! real part
      y(2*k)    = -sin(TWOPI_phase)/N ! imaginary part
    end do
  end subroutine init_c
  
  ! Verify that x(k) is unit peak at k = H
  integer function verify_r(x, N, H)
    integer N, H
    real(WP) :: x(:)

    integer k
    real(WP) err, errthr, maxerr
    real(WP) res_exp, res_got

    ! Note, this simple error bound doesn't take into account error of
    ! input data
    errthr = 2.5 * log(real(N, WP)) / log(2.0_WP) * EPSILON(1.0_WP)
    print '("  Check if err is below errthr " G10.3)', errthr

    maxerr = 0.0_WP
    do k = 1, N
      if (mod(k-1-H,N)==0) then
        res_exp = 1.0_WP
      else
        res_exp = 0.0_WP
      end if
      res_got = x(k)
      err = abs(res_got - res_exp)
      maxerr = max(err,maxerr)
      if (.not.(err < errthr)) then
        print '("  x("I0"): "$)', k
        print '(" expected "G24.17","$)', res_exp
        print '(" got "G24.17","$)', res_got
        print '(" err "G10.3)', err
        print *," Verification FAILED"
        verify_r = 100
        return
      end if
    end do
    print '("  Verified,  maximum error was " G10.3)', maxerr
    verify_r = 0
  end function verify_r

end program dp_real_1d_async
