!===============================================================================
! Copyright 2020-2022 Intel Corporation.
!
! This software and the related documents are Intel copyrighted  materials,  and
! your use of  them is  governed by the  express license  under which  they were
! provided to you (License).  Unless the License provides otherwise, you may not
! use, modify, copy, publish, distribute,  disclose or transmit this software or
! the related documents without Intel's prior written permission.
!
! This software and the related documents  are provided as  is,  with no express
! or implied  warranties,  other  than those  that are  expressly stated  in the
! License.
!===============================================================================

! Content:
!       Example of using sfftw_plan_many_dft function on a
!       (GPU) device using the OpenMP target (offload) interface.
!
!*****************************************************************************

include "fftw/offload/fftw3_omp_offload.f90"

program sp_plan_many_dft_1d

  use FFTW3_OMP_OFFLOAD
  use omp_lib, ONLY : omp_get_num_devices
  use, intrinsic :: ISO_C_BINDING

  include 'fftw/fftw3.f'

  ! Size of 1D transform and the number of them
  integer, parameter :: N = 64
  integer, parameter :: M = 7

  ! Arbitrary harmonic used to verify FFT
  integer :: H = -N/2

  ! Working precision is single precision (using sfftw_* functions)
  integer, parameter :: WP = selected_real_kind(6,37)

  ! Execution status
  integer :: statusf = 0, statusb = 0, status = 0

  ! Distance, stride and embedding array
  integer :: dist, stride, nembed(1)

  ! FFTW plan variables
  integer*8 :: fwd = 0, bwd = 0

  ! The data array
  complex(WP), allocatable :: x(:)

  stride = 1
  nembed(1) = N * stride
  dist = nembed(1)

  print *,"Example sp_plan_many_dft_1d"
  print *,"Forward and Backward multiple 1D complex in-place FFT"
  print *,"Configuration parameters:"
  print '("  N = "I0)', N
  print '("  M = "I0)', M
  print '("  H = "I0)', H
  print '("  nembed = "I0)', nembed(1)
  print '("  dist = "I0)', dist
  print '("  stride = "I0)', stride

  print *,"Allocate data array"
  allocate ( x(dist*M), STAT = status)
  if (0 /= status) goto 999

  print *,"Initialize data for forward FFT"
  call init(x, N, H, M, dist)

  print *,"Create FFTW plan for forward in-place transform"
  !$omp target data map(tofrom:x)
  !$omp target variant dispatch use_device_ptr(x)
  call sfftw_plan_many_dft(fwd, 1, (/N/), M, &
                           x, nembed, stride, dist,   &
                           x, nembed, stride, dist,   &
                           FFTW_FORWARD, FFTW_ESTIMATE)
  !$omp end target variant dispatch
  if (0 == fwd) print *, "Call to sfftw_plan_many_dft for forward transform &
                          &has failed"

  print *,"Create FFTW plan for backward in-place transform"
  !$omp target variant dispatch use_device_ptr(x)
  call sfftw_plan_many_dft(bwd, 1, (/N/), M, &
                           x, nembed, stride, dist,   &
                           x, nembed, stride, dist,   &
                           FFTW_BACKWARD, FFTW_ESTIMATE)

  !$omp end target variant dispatch
  if (0 == bwd) print *, "Call to sfftw_plan_many_dft for backward transform &
                          &has failed"

  print *,"Compute forward transform"
  !$omp target variant dispatch
  call sfftw_execute(fwd)
  !$omp end target variant dispatch

  ! Update the host with the results from forward FFT
  !$omp target update from(x)

  print *,"Verify the result of the forward transform"
  statusf = verify(x, N, H, M, dist)

  print *,"Initialize data for backward FFT"
  call init(x, N, -H, M, dist)

  ! Update the device with input for backward FFT
  !$omp target update to(x)

  print *,"Compute backward transform"
  !$omp target variant dispatch
  call sfftw_execute(bwd)
  !$omp end target variant dispatch
  !$omp end target data

  print *,"Verify the result of the backward transform"
  statusb = verify(x, N, H, M, dist)
  if ((0 /= statusf) .or. (0 /= statusb)) goto 999

100 continue

  print *,"Destroy FFTW plans"
  call sfftw_destroy_plan(fwd)
  call sfftw_destroy_plan(bwd)

  print *,"Deallocate data array"
  deallocate(x)

  if (status == 0) then
    print *,"TEST PASSED"
    call exit(0)
  else
    print *,"TEST FAILED"
    call exit(1)
  endif

999 print '("  Error, status forward = ",I0)', statusf
  print '(" Error, status backward = ",I0)', statusb
  status = 1
  goto 100

contains

  ! Compute mod(K*L,M) accurately
  pure integer*8 function moda(k,l,m)
    integer, intent(in) :: k,l,m
    integer*8 :: k8
    k8 = k
    moda = mod(k8*l,m)
  end function moda

  ! Initialize array x(N) with harmonic H
  subroutine init(x, N, H, M, dist)
    integer N, H, M, dist
    complex(WP) :: x(:)

    integer k1, k2
    complex(WP), parameter :: I_TWOPI = (0.0_WP,6.2831853071795864769_WP)

    do k1 = 1, M
      do k2 = 1, N
        x(((k1-1)*dist)+k2) = exp( I_TWOPI * real(moda(k2-1, H, N), WP)/N ) / N
      end do
    end do
  end subroutine init

  ! Verify that x(N) is unit peak at x(H)
  integer function verify(x, N, H, M, dist)
    integer N, H, M, dist
    complex(WP) :: x(:)

    integer k1, k2
    real(WP) err, errthr, maxerr
    complex(WP) res_exp, res_got

    ! Note, this simple error bound doesn't take into account error of
    ! input data
    errthr = 5.0 * log(real(N, WP)) / log(2.0_WP) * EPSILON(1.0_WP)
    print '("  Check if err is below errthr " G10.3)', errthr

    maxerr = 0.0_WP
    do k1 = 1, M
      do k2 = 1, N
    !forall(k1=1:M,k2=1:N)
        if (mod(k2-1-H,N)==0) then
          res_exp = 1.0_WP
        else
          res_exp = 0.0_WP
        end if
        res_got = x((k1-1)*dist+k2)
        err = abs(res_got - res_exp)
        maxerr = max(err,maxerr)
        if (.not.(err < errthr)) then
          print '("  x("I0"): "$)', k
          print '(" expected "G14.7","$)', res_exp
          print '(" got "G14.7","$)', res_got
          print '(" err "G10.3)', err
          print *," Verification FAILED"
          verify = 1
          return
        end if
      end do
    end do
    print '("  Verified,  maximum error was " G10.3)', maxerr
    verify = 0
  end function verify

end program sp_plan_many_dft_1d
