!===============================================================================
! Copyright 2021-2022 Intel Corporation.
!
! This software and the related documents are Intel copyrighted  materials,  and
! your use of  them is  governed by the  express license  under which  they were
! provided to you (License).  Unless the License provides otherwise, you may not
! use, modify, copy, publish, distribute,  disclose or transmit this software or
! the related documents without Intel's prior written permission.
!
! This software and the related documents  are provided as  is,  with no express
! or implied  warranties,  other  than those  that are  expressly stated  in the
! License.
!===============================================================================

!  Content:
!      oneMKL FORTRAN Summary Statistics basic example for VSL openMP offload
!*******************************************************************************

include "mkl_omp_offload.f90"
include "errcheck.inc"

program vslsbasicstats
#if defined(MKL_ILP64)
    use onemkl_vsl_omp_offload_ilp64
#else
    use onemkl_vsl_omp_offload_lp64
#endif
    use MKL_VSL_TYPE

    integer,parameter :: DIM = 4       ! Task dimension
    integer,parameter :: NN  = 1000     ! Number of observations

    type (VSL_STREAM_STATE) stream
    integer brng,method,seed
    real a, b
    type (VSL_SS_TASK) task
    integer p
    integer n
    integer x_storage
    integer i, j
    integer errcode
    real(kind=4) x(DIM,NN)

    real(kind=4) mean(DIM)
    real(kind=4) min_estimate(DIM), max_estimate(DIM)
    real(kind=4) raw2(DIM), raw3(DIM), raw4(DIM)

    integer errnums
    integer(kind=8) estimate
    integer task_method

    n=1000
    p=4

    brng=VSL_BRNG_PHILOX4X32X10
    method=VSL_RNG_METHOD_UNIFORM_STD
    seed=1

    a=0.5
    b=3.0

    ! Initialize data
    errcode=vslnewstream(stream, brng, seed)
    call CheckVslError(errcode)

    errcode=vsrnguniform(method, stream, n * p, x, a, b)
    call CheckVslError(errcode)

    errcode=vsldeletestream(stream)
    call CheckVslError(errcode)

    ! Initializing parameters for Summary Statistics task
    p               = DIM
    n               = NN
    x_storage       = VSL_SS_MATRIX_STORAGE_COLS
    task_method     = VSL_SS_METHOD_FAST


    do i = 1, p
      min_estimate(i) = x(i, 1)
      max_estimate(i) = x(i, 1)
    end do

    !$omp target data map(x, min_estimate, max_estimate, mean, raw2, raw3, raw4)
    !$omp target data use_device_ptr(x, min_estimate, max, max_estimate, mean, raw2, raw3, raw4)

    ! Create Summary Statistics task
    errcode = vslsssnewtask( task, p, n, x_storage, x )
    call CheckVslError( errcode )

    ! Edit task parameters for min and max computation
    errcode = vslsssedittask( task, VSL_SS_ED_MIN, min_estimate )
    call CheckVslError( errcode )

    errcode = vslsssedittask( task, VSL_SS_ED_MAX, max_estimate )
    call CheckVslError( errcode )
    

    ! Edit task parameters for computating of mean estimate and 2nd, 3rd
    !           and 4th raw moments estimates
    errcode = vslssseditmoments( task, mean, raw2, raw3, raw4)
    call CheckVslError( errcode )

    estimate = ior( VSL_SS_MIN, VSL_SS_MAX )

    estimate = ior( estimate, ior( VSL_SS_MEAN,                       &
        &           ior( VSL_SS_2R_MOM, ior( VSL_SS_3R_MOM, VSL_SS_4R_MOM ) ) ) )

    ! Compute the estimates using FAST method on the GPU
    !$omp target variant dispatch
    errcode = vslssscompute( task, estimate, task_method )
    !$omp end target variant dispatch
    !$omp end target data
    !$omp end target data
    call CheckVslError(errcode)

    ! Testing stat characteristics of computed estimates
      errnums = 0

    ! Comparison of observations with min and max estimates
      do i = 1, p
        do j = 1, n
          if (x(i, j) < min_estimate(i)) then
            errnums = errnums + 1
          end if
          if (x(i, j) > max_estimate(i)) then
            errnums = errnums + 1
          end if
        end do
      end do


    ! Printing results
      print 9, 'Task dimension :         ', p
      print 9, 'Number of observations : ', n
      print *, ''

    ! Printing computed minimum, maximum, mean and moments estimates
      print 10, '               '
      print 10, 'Min         Max         Mean       '
      print 10, '2nd_raw     3rd_raw     4th_raw      '
      print *, ''

      do i = 1, p
        print 11, 'Variable #', i, ' '
        print 12, min_estimate(i), ' '
        print 12, max_estimate(i), ' '
        print 12, mean(i), ' '
        print 13, raw2(i), ' ', raw3(i), ' ', raw4(i), ' '
        print *, ''
      end do
      print *, ''


    ! Printing summary of the test
      if ( errnums == 0 ) then
        print *, ' All the computed estimates agree with theory'
      else
        print *, ' Error: At least one of the computed estimates',      &
     &           ' disagrees with theory'
        stop 1
      end if

    stop

9     format(A,I4)
10    format(A,$)
11    format(A,I1,A,$)
12    format(F11.6,A,$)
13    format(F11.6,A,F11.6,A,F11.6,A,$)

end program
