/*******************************************************************************
* Copyright 2013-2022 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*   Content : Intel(R) oneAPI Math Kernel Library (oneMKL) IE Sparse BLAS C
*             example for CSR format
*
********************************************************************************
*
* Example program for using Intel oneMKL Inspector-Executor Sparse BLAS routines
* for matrices represented in the compressed sparse row (CSR) sparse storage format.
*
* The following Inspector Executor Sparse Blas routines are used in the example:
*
*   Initialization/Destruction stage:
*          mkl_sparse_d_create_csr
*          mkl_sparse_destroy
*
*   Inspector stage:
*          mkl_sparse_set_mv_hint  mkl_sparse_set_sv_hint
*          mkl_sparse_set_mm_hint  mkl_sparse_set_sm_hint
*          mkl_sparse_optimize
*
*   Executor stage:
*          mkl_sparse_d_mv         mkl_sparse_d_trsv
*          mkl_sparse_d_mm         mkl_sparse_d_trsm
*
* Consider the matrix A (see 'Sparse Storage Formats for Sparse BLAS Level 2
* and Level 3 in the  Intel oneMKL Reference Manual')
*
*                 |   1       -1      0   -3     0   |
*                 |  -2        5      0    0     0   |
*   A    =        |   0        0      4    6     4   |,
*                 |  -4        0      2    7     0   |
*                 |   0        8      0    0    -5   |
*
*  The matrix A is represented in a zero-based compressed sparse row (CSR) storage
*  scheme with three arrays (see 'Sparse Matrix Storage Schemes' in the
*   Intel oneMKL Reference Manual) as follows:
*
*         rowPtr  = ( 0        3     5        8       11    13 )
*         columns = ( 0  1  3  0  1  2  3  4  0  2  3  1  4 )
*         values  = ( 1 -1 -3 -2  5  4  6  4 -4  2  7  8 -5 )
*
********************************************************************************
*/
#include <stdio.h>
#include <assert.h>
#include <math.h>
#include "mkl_spblas.h"

int main() {
    //*******************************************************************************
    //     Declaration and initialization of parameters for sparse representation of
    //     the matrix A in the CSR format:
    //*******************************************************************************
#define M 5
#define NNZ 13
#define NRHS 2

    MKL_INT m = M, nrhs = NRHS;

    //*******************************************************************************
    //    Sparse representation of the matrix A
    //*******************************************************************************

    MKL_INT rowPtr[M+1] = { 0, 3, 5, 8, 11, 13 };

    MKL_INT columns[NNZ]   = { 0,      1,        3,
                               0,      1,
                                            2,   3,   4,
                               0,           2,   3,
                                       1,             4 };

    double values[NNZ]     = { 1.0, -1.0,     -3.0,
                              -2.0,  5.0,
                                          4.0, 6.0, 4.0,
                              -4.0,       2.0, 7.0,
                                     8.0,          -5.0 };


    // Descriptor of main sparse matrix properties
    struct matrix_descr descrA;
    // Structure with sparse matrix stored in CSR format
    sparse_matrix_t       csrA;
    //*******************************************************************************
    //    Declaration of local variables:
    //*******************************************************************************
    double x_m[M*NRHS]  = { 1.0, 5.0, 3.0, 4.0, 2.0, 2.0, 10.0, 6.0, 8.0, 4.0};
    double y_m[M*NRHS]  = { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    double x_v[M]  = { 3.0, 2.0, 5.0, 4.0, 1.0};
    double y_v[M]  = { 0.0, 0.0, 0.0, 0.0, 0.0};
    double tmp_v[M]  = { 0.0, 0.0, 0.0, 0.0, 0.0};
    double alpha = 1.0, beta = 0.0;
    MKL_INT    i;

    sparse_status_t status;
    int exit_status = 0;

    printf( "\n EXAMPLE PROGRAM FOR CSR format routines from IE Sparse BLAS\n" );
    printf( "-------------------------------------------------------\n" );

//*******************************************************************************
//   Create CSR sparse matrix handle and analyze step
//*******************************************************************************

    // Create handle with matrix stored in CSR format
    status = mkl_sparse_d_create_csr ( &csrA,
                                       SPARSE_INDEX_BASE_ZERO,
                                       m,  // number of rows
                                       m,  // number of cols
                                       rowPtr,
                                       rowPtr+1,
                                       columns,
                                       values );


    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_d_create_csr: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    //*******************************************************************************
    // First we set hints for the different operations before calling the
    // mkl_sparse_optimize() api which actually does the analyze step.  Not all
    // configurations have optimized steps, so the hint apis may return status
    // MKL_SPARSE_STATUS_NOT_SUPPORTED (=6) if no analysis stage is actually available
    // for that configuration.
    //*******************************************************************************

    //*******************************************************************************
    // Set hints for Task 1: Lower triangular transpose MV and SV solve with
    // non-unit diagonal
    //*******************************************************************************
    descrA.type = SPARSE_MATRIX_TYPE_TRIANGULAR;
    descrA.mode = SPARSE_FILL_MODE_LOWER;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    status = mkl_sparse_set_mv_hint(csrA, SPARSE_OPERATION_TRANSPOSE, descrA, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 1: mkl_sparse_set_mv_hint: %d \n", status);
    }

    status = mkl_sparse_set_sv_hint(csrA, SPARSE_OPERATION_TRANSPOSE, descrA, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 1: mkl_sparse_set_mv_hint: %d \n", status);
    }

    //*******************************************************************************
    // Set hints for Task 2: Upper triangular non-transpose MV and SV solve with
    // unit diagonal
    //*******************************************************************************
    descrA.type = SPARSE_MATRIX_TYPE_TRIANGULAR;
    descrA.mode = SPARSE_FILL_MODE_UPPER;
    descrA.diag = SPARSE_DIAG_UNIT;

    status = mkl_sparse_set_mv_hint(csrA, SPARSE_OPERATION_TRANSPOSE, descrA, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 2: mkl_sparse_set_mv_hint: %d \n", status);
    }

    //*******************************************************************************
    // Set hints for Task 3: General matrix (transpose sparse) * dense MM  with
    // non-unit diagonal and column-major format
    //*******************************************************************************
    descrA.type = SPARSE_MATRIX_TYPE_GENERAL;

    status = mkl_sparse_set_mm_hint(csrA, SPARSE_OPERATION_TRANSPOSE, descrA,
                                    SPARSE_LAYOUT_COLUMN_MAJOR, nrhs, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 3: mkl_sparse_set_mm_hint: %d \n", status);
    }

    //*******************************************************************************
    // Set hints for Task 4: General matrix sparse * dense MM  with
    // non-unit diagonal and row-major format
    //*******************************************************************************
    descrA.type = SPARSE_MATRIX_TYPE_GENERAL;

    status = mkl_sparse_set_mm_hint(csrA, SPARSE_OPERATION_NON_TRANSPOSE, descrA,
                                    SPARSE_LAYOUT_ROW_MAJOR, nrhs, 1 );
    if (status != SPARSE_STATUS_SUCCESS && status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set hints for Task 4: mkl_sparse_set_mm_hint: %d \n", status);
    }

    //*******************************************************************************
    // Analyze sparse matrix; choose proper kernels and workload balancing strategy
    //*******************************************************************************
    status = mkl_sparse_optimize ( csrA );
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_optimize: %d \n", status);
        exit_status = 1;
        goto exit;
    }




//*******************************************************************************
//  Task 1: Obtain matrix-matrix multiply (L+D)' *x_v --> y_v
//          and solve triangular system   (L+D)' *tmp_v = y_v
//          Array tmp_v must be equal to the array x_v
//*******************************************************************************
    printf( "-------------------------------------------------------\n" );
    printf( "                                  \n" );
    printf( "   Task 1:                        \n" );
    printf( "   INPUT DATA FOR mkl_sparse_d_mv \n" );
    printf( "   WITH TRIANGULAR SPARSE MATRIX  \n" );
    printf( "   ALPHA = %4.1f  BETA = %4.1f    \n", alpha, beta );
    printf( "   SPARSE_OPERATION_TRANSPOSE     \n" );
    printf( "   Input vector                   \n" );
    for ( i = 0; i < m; i++ )
    {
        printf( "%7.1f\n", x_v[i] );
    }

    // Create matrix descriptor
    descrA.type = SPARSE_MATRIX_TYPE_TRIANGULAR;
    descrA.mode = SPARSE_FILL_MODE_LOWER;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    status = mkl_sparse_d_mv ( SPARSE_OPERATION_TRANSPOSE, alpha, csrA,
                               descrA, x_v, beta, y_v );
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 1 mkl_sparse_d_mv: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf( "   OUTPUT DATA FOR mkl_sparse_d_mv \n" );
    for ( i = 0; i < m; i++ )
    {
        printf( "%7.1f\n", y_v[i] );
    }

    printf("   Solve triangular system   \n");
    printf("   with obtained             \n");
    printf("   right hand side           \n");

    status = mkl_sparse_d_trsv ( SPARSE_OPERATION_TRANSPOSE, alpha, csrA,
                                 descrA, y_v, tmp_v );
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 1 mkl_sparse_d_trsv: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf( "   OUTPUT DATA FOR mkl_sparse_d_trsv \n" );
    for ( i = 0; i < m; i++ )
    {
        printf( "%7.1f\n", tmp_v[i] );
    }
    printf( "-------------------------------------------------------\n" );

//*******************************************************************************
//  Task 2: Obtain matrix-matrix multiply (U+I)*x_v --> y_v
//          and solve triangular system   (U+I)*tmp_v = y_v
//          Array tmp_v must be equal to the array x_v
//*******************************************************************************
    printf( "                                  \n" );
    printf( "   Task 2:                        \n" );
    printf( "   INPUT DATA FOR mkl_sparse_d_mv \n" );
    printf( "   WITH TRIANGULAR SPARSE MATRIX  \n" );
    printf( "   ALPHA = %4.1f  BETA = %4.1f    \n", alpha, beta );
    printf( "   SPARSE_OPERATION_NON_TRANSPOSE \n" );
    printf( "   Input vector                   \n" );
    // Release matrix handle and deallocate matrix
    for ( i = 0; i < m; i++ )
    {
        printf( "%7.1f\n", x_v[i] );
    }

    // Create matrix descriptor
    descrA.type = SPARSE_MATRIX_TYPE_TRIANGULAR;
    descrA.mode = SPARSE_FILL_MODE_UPPER;
    descrA.diag = SPARSE_DIAG_UNIT;

    status = mkl_sparse_d_mv ( SPARSE_OPERATION_NON_TRANSPOSE, alpha, csrA,
                               descrA, x_v, beta, y_v );
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 2 mkl_sparse_d_mv: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf( "   OUTPUT DATA FOR mkl_sparse_d_mv \n" );
    for ( i = 0; i < m; i++ )
    {
        printf( "%7.1f\n", y_v[i] );
    }

    printf("   Solve triangular system   \n");
    printf("   with obtained             \n");
    printf("   right hand side           \n");

    status = mkl_sparse_d_trsv ( SPARSE_OPERATION_NON_TRANSPOSE, alpha, csrA,
                                 descrA, y_v, tmp_v );
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 2 mkl_sparse_d_trsv: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf( "   OUTPUT DATA FOR mkl_sparse_d_trsv \n" );
    for ( i = 0; i < m; i++ )
    {
        printf( "%7.1f\n", tmp_v[i] );
    }
    printf( "-------------------------------------------------------\n" );

//*******************************************************************************
//  Task 3: Obtain matrix-matrix multiply A' *x_m --> y_m
//          A - zero-based indexing,
//          x_m - column major ordering
//*******************************************************************************
    printf( "                                  \n" );
    printf( "   Task 3:                        \n" );
    printf( "   INPUT DATA FOR mkl_sparse_d_mm \n" );
    printf( "   WITH GENERAL SPARSE MATRIX     \n" );
    printf( "   COLUMN MAJOR ORDERING for RHS  \n" );
    printf( "   ALPHA = %4.1f  BETA = %4.1f    \n", alpha, beta );
    printf( "   SPARSE_OPERATION_TRANSPOSE     \n" );
    printf( "   Input vectors                  \n" );
    for ( i = 0; i < m; i++ )
    {
        printf( "%7.1f, %7.1f\n", x_m[i], x_m[m+i] );
    }

    // Create matrix descriptor
    descrA.type = SPARSE_MATRIX_TYPE_GENERAL;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    // note that column-major format implies  ldx = m, ldy = m
    status = mkl_sparse_d_mm ( SPARSE_OPERATION_TRANSPOSE, alpha, csrA,  descrA,
                               SPARSE_LAYOUT_COLUMN_MAJOR, x_m, nrhs, m,
                               beta, y_m, m );
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 3 mkl_sparse_d_mm: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf( "   OUTPUT DATA FOR mkl_sparse_d_mm \n" );
    for ( i = 0; i < m; i++ )
    {
        printf( "%7.1f, %7.1f\n", y_m[i], y_m[m+i] );
    }
    printf( "-------------------------------------------------------\n" );

//*******************************************************************************
//  Task 4: Obtain matrix-matrix multiply A*x_m --> y_m
//          A - zero-based indexing,
//          x_m - row major ordering
//*******************************************************************************
    printf( "                                  \n" );
    printf( "   Task 4:                        \n" );
    printf( "   INPUT DATA FOR mkl_sparse_d_mm \n" );
    printf( "   WITH GENERAL SPARSE MATRIX     \n" );
    printf( "   ROW MAJOR ORDERING for RHS     \n" );
    printf( "   ALPHA = %4.1f  BETA = %4.1f    \n", alpha, beta );
    printf( "   SPARSE_OPERATION_TRANSPOSE     \n" );
    printf( "   Input vectors                  \n" );
    for ( i = 0; i < m; i++ )
    {
        printf( "%7.1f, %7.1f\n", x_m[2*i], x_m[2*i+1] );
    }

    // Create matrix descriptor
    descrA.type = SPARSE_MATRIX_TYPE_GENERAL;
    descrA.diag = SPARSE_DIAG_NON_UNIT;

    // note that row-major format implies  ldx = nrhs, ldy = nrhs
    status = mkl_sparse_d_mm ( SPARSE_OPERATION_NON_TRANSPOSE, alpha, csrA, descrA,
                               SPARSE_LAYOUT_ROW_MAJOR, x_m, nrhs, nrhs,
                               beta, y_m, nrhs );
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in Task 4 mkl_sparse_d_mm: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    printf( "   OUTPUT DATA FOR mkl_sparse_d_mm \n" );
    for ( i = 0; i < m; i++ )
    {
        printf( "%7.1f, %7.1f\n", y_m[2*i], y_m[2*i+1] );
    }
    printf( "-------------------------------------------------------\n" );

exit:
    // Release matrix handle and deallocate matrix
    mkl_sparse_destroy ( csrA );

    return exit_status;
}
