/*******************************************************************************
* Copyright 2021-2022 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

#ifndef _COMMON_FOR_SPARSE_EXAMPLES_H_
#define _COMMON_FOR_SPARSE_EXAMPLES_H_

#include "mkl.h"
#include <float.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#define FREE(p)          \
    do {                 \
        if (p) {         \
            mkl_free(p); \
            p = NULL;    \
        }                \
    } while (0)

#ifdef MKL_ILP64
#define INT_PRINT_FORMAT "%lld"
#else
#define INT_PRINT_FORMAT "%d"
#endif

void free_allocated_memories(void **array, int length)
{
    for (int i = 0; i < length; i++)
        FREE(array[i]);
}

int validation_result_double(const double *ref,
                             const double *z,
                             const int length,
                             const int flps_per_value)
{
    if (!ref || !z) {
        printf("Either ref or z is NULL pointer!\n");
        fflush(stdout);
        return 1;
    }

    const double err_thresh = 1.0 * flps_per_value * DBL_EPSILON;
    printf(" Check if error is below err_thresh %.3lg\n", err_thresh);

    double maxerr = 0.0;
    for (int i = 0; i < length; i++) {
        double err = fabs(ref[i] - z[i]);
        if (err > maxerr)
            maxerr = err;
        if (!(maxerr < err_thresh)) {
            printf(" z[%d]: expected %.7g, got %.7g, err %.3lg\n", i, ref[i], z[i], err);
            printf(" Verification FAILED\n");
            return 2;
        }
    }
    printf(" Verified, maximum error was %.3lg\n", maxerr);

    return 0;
}

int validation_result_float(const float *ref,
                            const float *z,
                            const int length,
                            const int flps_per_value)
{
    if (!ref || !z) {
        printf("Either ref or z is NULL pointer!\n");
        fflush(stdout);
        return 1;
    }

    const float err_thresh = 1.0f * flps_per_value * FLT_EPSILON;
    printf(" Check if error is below err_thresh %.3lg\n", err_thresh);

    float maxerr = 0.0f;
    for (int i = 0; i < length; i++) {
        float err = fabsf(ref[i] - z[i]);
        if (err > maxerr)
            maxerr = err;
        if (!(maxerr < err_thresh)) {
            printf(" z[%d]: expected %.7g, got %.7g, err %.3lg\n", i, ref[i], z[i], err);
            printf(" Verification FAILED\n");
            return 2;
        }
    }
    printf(" Verified, maximum error was %.3lg\n", maxerr);

    return 0;
}

void print_csr_matrix_float(const MKL_INT *ia,
                            const MKL_INT *ja,
                            const float *a,
                            const MKL_INT nrows)
{
    if (!ia || !ja || !a) {
        printf("Among ia, ja, and a arrays, at least one of them are NULL!\n");
        fflush(stdout);
        return;
    }

    MKL_INT ii = 0;

    for (int i = 0; i < nrows; i++) {
        printf("row# %d:", i + 1);
        fflush(0);
        for (int j = ia[i]; j < ia[i + 1]; j++) {
            printf(" (%5.0f, " INT_PRINT_FORMAT ")", a[ii], ja[ii]);
            fflush(0);
            ii++;
        }
        printf("\n");
    }

    return;
}

void print_csr_matrix_double(const MKL_INT *ia,
                             const MKL_INT *ja,
                             const double *a,
                             const MKL_INT nrows)
{
    if (!ia || !ja || !a) {
        printf("Among ia, ja, and a arrays, at least one of them are NULL!\n");
        fflush(stdout);
        return;
    }

    MKL_INT ii = 0;

    for (int i = 0; i < nrows; i++) {
        printf("row# %d:", i + 1);
        fflush(0);
        for (int j = ia[i]; j < ia[i + 1]; j++) {
            printf(" (%5.0f, " INT_PRINT_FORMAT ")", a[ii], ja[ii]);
            fflush(0);
            ii++;
        }
        printf("\n");
    }

    return;
}

#endif //_COMMON_FOR_SPARSE_EXAMPLES_H_
