/*******************************************************************************
* Copyright 2019-2022 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*
*  Content:
*       This example demonstrates use of DPC++ API oneapi::mkl::rng::uniform distribution with
*       oneapi::mkl::rng::philox4x32x10 random number generator to produce random numbers on
*       a SYCL device (Host, CPU, GPU) with Unified Shared Memory(USM) API.
*
*       The supported data types for random numbers are:
*           float
*           double
*           std::int32_t
*           std::uint32_t
*
*******************************************************************************/

// stl includes
#include <iostream>
#include <vector>

#include <CL/sycl.hpp>
#include "oneapi/mkl.hpp"

// local includes
#include "../include/common_for_rng_examples.hpp"

//
// Main example for Uniform random number generation consisting of
// initialization of random number engine philox4x32x10 object, distribution
// object. Then random number generation performed and
// the output is post-processed and validated.
//
template <typename Type>
bool run_uniform_example(const sycl::device &dev) {

    //
    // Initialization
    //
    // example parameters defines
    constexpr std::uint64_t seed = 777;
    constexpr std::size_t n = 1000;
    constexpr std::size_t n_print = 10;
    constexpr std::size_t alignment = 64;

    sycl::queue queue(dev, exception_handler);

    // set scalar Type values
    Type a(0.0);
    Type b(10.0);

    oneapi::mkl::rng::default_engine engine(queue, seed);

    oneapi::mkl::rng::uniform<Type> distribution(a, b);

    // prepare array for random numbers
    sycl::usm_allocator<Type, sycl::usm::alloc::shared, alignment> allocator(queue);
    std::vector<Type, decltype(allocator)> r(n, allocator);

    //
    // Perform generation
    //
    sycl::event event_out{};
    try {
        event_out = oneapi::mkl::rng::generate(distribution, engine, n, r.data());
    }
    catch(sycl::exception const& e) {
        std::cout << "\t\tSYCL exception during generation\n"
                  << e.what() << std::endl << "Error code: " << get_error_code(e) << std::endl;
        return false;
    }
    event_out.wait_and_throw();

    //
    // Post Processing
    //

    std::cout << "\n\t\tgeneration parameters:\n";
    std::cout << "\t\t\tseed = " << seed << ", a = " << a << ", b = " << b << std::endl;

    std::cout << "\n\t\tOutput of generator:" << std::endl;

    std::cout << "first "<< n_print << " numbers of " << n << ": " << std::endl;
    for(int i = 0 ; i < n_print; i++) {
        std::cout << r[i] << " ";
    }
    std::cout << std::endl;

    //
    // Validation
    //
    return check_statistics(r, distribution);
}

//
// Description of example setup, APIs used and supported floating point type precisions
//
void print_example_banner() {

    std::cout << "" << std::endl;
    std::cout << "########################################################################" << std::endl;
    std::cout << "# Generate uniformly distributed random numbers with philox4x32x10\n# generator example: " << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Using APIs:" << std::endl;
    std::cout << "#   default_engine uniform" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Supported precisions:" << std::endl;
    std::cout << "#   float double std::int32_t" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "########################################################################" << std::endl;
    std::cout << std::endl;

}

//
// Main entry point for example.
//
// Dispatches to appropriate device types as set at build time with flag:
// -DSYCL_DEVICES_host -- only runs host implementation
// -DSYCL_DEVICES_cpu -- only runs SYCL CPU implementation
// -DSYCL_DEVICES_gpu -- only runs SYCL GPU implementation
// -DSYCL_DEVICES_all (default) -- runs on all: host, cpu and gpu devices
//

int main (int argc, char ** argv) {

    print_example_banner();

    std::list<my_sycl_device_types> list_of_devices;
    set_list_of_devices(list_of_devices);

    for (auto it = list_of_devices.begin(); it != list_of_devices.end(); ++it) {

        sycl::device my_dev;
        bool my_dev_is_found = false;
        get_sycl_device(my_dev, my_dev_is_found, *it);

            if(my_dev_is_found) {
                std::cout << "Running tests on " << sycl_device_names[*it] << ".\n";

                std::cout << "\tRunning with single precision real data type:" << std::endl;
                if(!run_uniform_example<float>(my_dev)) {
                    std::cout << "FAILED" << std::endl;
                    return 1;
                }
                std::cout << "\tRunning with integer data type:" << std::endl;
                if(!run_uniform_example<std::int32_t>(my_dev)) {
                    std::cout << "FAILED" << std::endl;
                    return 1;
                }
            } else {
#ifdef FAIL_ON_MISSING_DEVICES
                std::cout << "No " << sycl_device_names[*it] << " devices found; Fail on missing devices is enabled.\n";
                std::cout << "FAILED" << std::endl;
                return 1;
#else
                std::cout << "No " << sycl_device_names[*it] << " devices found; skipping " << sycl_device_names[*it] << " tests.\n";
#endif
            }
    }
    std::cout << "PASSED" << std::endl;
    return 0;
}
