#############################################################################
##
#W  3ctsgrpdata.g               GAP4 Package `RCWA'               Stefan Kohl
##
##  Data library of groups generated by 3 class transpositions which inter-
##  change residue classes with moduli <= 6.
##
##  The following relations hold:
##
##  1. mods  = List(grps,Mod);
##  2. sizes = List(grps,Size);
##  3. TriangleTypes() = List(grps,G->List([[1,2],[1,3],[2,3]],
##           ind->CTPairProductType(GeneratorsOfGroup(G){ind})));
##     where CTPairProductType is the component of the record returned by
##     the function `LoadDatabaseOfProductsOf2ClassTranspositions'.
##
##  Given a group G which is generated by 3 class transpositions with moduli
##  <= 6, Id3CTsGroup(G,grps) is the position of the group G in the list
##  grps of groups in this database.
##
##  The list 'equalityclasses' contains lists of indices of equal groups in
##  the list grps. Due to the computational difficulty of equality tests for
##  the groups in question it is not guaranteed that grps[i] <> grps[j] if i
##  and j do not lie in the same list in 'equalityclasses'.
##
##  For the description of the list 'conjugacyclasses', see the file
##  3ctsgrp-conjugacyclasses.g.
##
##  For the description of the list 'intransitivemodulo', see the file
##  3ctsgrp-intransitivemodulo.g.
##
##  Given an index i, the entry trsstatus[i] provides information on what is
##  known about whether the group grps[i] acts transitively on the nonnega-
##  tive integers in its support or not. trsstatuspos[i] is a "translation"
##  of this to an integer. -- E.g. trsstatuspos[i] = 5 says that grps[i] is
##  finite, while trsstatuspos[i] = 7 says that grps[i] acts transitively on
##  the nonnegative integers in its support. This "translation" is defined
##  by the position of the string in the list 'trsstatusset'.
##
##  The list 'freeproductcandidates' is a list of the indices of those groups
##  which are possibly isomorphic to the free product of 3 cyclic groups of
##  order 2, thus whose generators possibly do not satisfy any further
##  relations. Only one index per class in 'conjugacyclasses' is given.
##
##  For the description of the lists 'partitionlength', 'degrees',
##  'orbitlengths', 'permgroupgens' and 'respectedpartitions', see the files
##  3ctsgrp-partitionlengths.g, 3ctsgrp-degrees.g, 3ctsgrp-orbitlengths.g,
##  3ctsgrp-permgroupgens and 3ctsgrp-respectedpartitions.g, respectively.
##
##  For the description of the record 'finiteorbits', see the file
##  3ctsgrp-finite-orbits.g.
##
#############################################################################

return rec(

Id3CTsGroup := function ( G, grps )

  local  gens;

  if not IsRcwaGroupOverZ(G) then return fail; fi;

  gens := Set(GeneratorsOfGroup(G));
  if Length(gens) <> 3 or not ForAll(gens,IsClassTransposition)
    or not ForAll(gens,g->Maximum(List(TransposedClasses(g),Mod))<=6)
  then return fail; fi;

  return First([1..Length(grps)],i->gens=GeneratorsOfGroup(grps[i]));
end,

TriangleTypes := function ( )

  local  triangles, triangle, cts, ctpairs, cttriples,
         data, ctptypes, mods, sizes, i;

  cts       := List(ClassPairs(6),ClassTransposition);
  ctpairs   := Combinations(cts,2);
  cttriples := Combinations(cts,3);

  data     := LoadDatabaseOfProductsOf2ClassTranspositions();
  ctptypes := data.CTPairsProductType;

  data  := LoadDatabaseOfGroupsGeneratedBy3ClassTranspositions();
  mods  := data.mods;
  sizes := data.sizes;

  triangles := [];
  for i in [1..Length(cttriples)] do
    triangle := List([[1,2],[1,3],[2,3]],ind->Set(cttriples[i]{ind}));
    triangle := List(triangle,edge->ctptypes[PositionSorted(ctpairs,edge)]);
    Add(triangles,triangle);
  od;

  return triangles;
end,

3CTsGroupsWithGivenOrbit := function ( orbit, maxmod )

  local  cts, stab, perms, gens, gensp, indcombs,
         grps, grpsp, trsinds, start;

  start := Runtime();
  orbit := AsSet(orbit);

  Info(InfoRCWA,2,"Setting up the list of class transpositions ...");
  cts   := List(ClassPairs(maxmod),ClassTransposition);
  Info(InfoRCWA,2,"Elapsed time: ",Runtime()-start); start := Runtime();

  Info(InfoRCWA,2,"Determining which class transpositions stabilize ...");
  stab  := Filtered(cts,ct->orbit^ct=orbit);
  perms := List(stab,s->Permutation(s,orbit));
  Info(InfoRCWA,2,"Found ",Length(stab),
                  " class transpositions which stabilize.");
  Info(InfoRCWA,2,"Elapsed time: ",Runtime()-start); start := Runtime();

  Info(InfoRCWA,2,"Forming the list of triples of class transpositions ",
                  "which stabilize ...");
  indcombs := Combinations([1..Length(stab)],3);
  gens     := List(indcombs,ind->stab{ind});
  gensp    := List(indcombs,ind->perms{ind});

  Info(InfoRCWA,2,"Forming the list of groups ...");
  grpsp    := List(gensp,Group);
  Info(InfoRCWA,2,"There are ",Length(grpsp)," groups to check in total.");
  Info(InfoRCWA,2,"Elapsed time: ",Runtime()-start); start := Runtime();

  Info(InfoRCWA,2,"Filtering the groups for transitivity ...");
  trsinds  := Filtered([1..Length(grpsp)],
                       i->IsTransitive(grpsp[i],[1..Length(orbit)]));
  grps     := List(gens{trsinds},Group);
  Info(InfoRCWA,2,"Found ",Length(grps),
                  " groups which have the given orbit.");
  Info(InfoRCWA,2,"Elapsed time: ",Runtime()-start); start := Runtime();

  return grps;
end,

ProbablyFixesDigitSumsModulo := function ( g, b, m )

  local  n, B;

  if b <= 1 then return fail; fi;
  if IsOne(g) or m = 1 then return true; fi;
  B := 100 * Mod(g);
  n := 0;
  repeat
    if   Sum(CoefficientsQadic(n  ,b)) mod m
      <> Sum(CoefficientsQadic(n^g,b)) mod m
    then return false; fi;
    if n >= B then return true; fi;
    n := n + 1;
  until false;
end,

ProbablyStabilizesDigitSumsModulo := function ( G, b, m )
  return ForAll(GeneratorsOfGroup(G),
                g->~.ProbablyFixesDigitSumsModulo(g,b,m));
end,

cts  := List(ClassPairs(6),ClassTransposition),

grps := List(Combinations(~.cts,3),Group),

mods :=
[ 6,6,12,12,12,0,12,12,0,12,0,0,30,30,30,30,30,
  30,30,30,30,30,6,6,6,6,6,6,0,0,6,6,6,0,6,0,0,0,0,
  0,6,6,0,6,0,6,0,6,0,0,12,12,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,6,12,12,12,0,12,12,0,12,0,0,30,30,30,30,
  30,30,30,30,30,30,6,6,6,6,6,6,0,0,6,6,6,12,6,0,0,
  0,0,0,6,6,0,6,0,6,12,6,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,12,12,12,0,12,12,0,12,0,0,30,30,30,
  30,30,30,30,30,30,30,6,6,6,6,6,6,0,0,6,6,6,0,6,0,
  0,0,0,0,6,6,0,6,0,6,0,6,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,4,4,0,4,4,0,4,0,0,20,20,20,20,20,20,
  20,20,20,20,12,12,12,12,12,12,0,0,12,12,12,24,12,24,
  24,0,0,0,12,12,0,12,0,12,24,12,0,0,24,24,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,4,0,4,4,0,4,0,0,20,20,20,20,
  20,20,20,20,20,20,12,12,12,12,12,12,0,0,12,12,12,0,
  12,0,0,0,0,0,12,12,0,12,0,12,0,12,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,4,4,0,4,0,0,20,20,20,20,
  20,20,20,20,20,20,12,12,12,12,12,12,0,0,12,12,12,0,
  12,0,0,0,0,0,12,12,0,12,0,12,0,12,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  4,0,4,0,0,20,20,20,20,20,20,20,20,20,20,12,12,12,12,
  12,12,0,0,12,12,12,24,12,24,24,0,0,0,12,12,0,12,0,
  12,24,12,0,0,24,24,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,4,0,0,20,20,20,20,20,20,20,20,20,20,12,12,12,12,12,
  12,0,0,12,12,12,0,12,0,0,0,0,0,12,12,0,12,0,12,0,
  12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,20,20,20,20,20,20,20,20,20,20,12,
  12,12,12,12,12,0,0,12,12,12,0,12,0,0,0,0,0,12,12,0,
  12,0,12,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,10,10,10,10,10,10,
  10,10,10,30,30,30,30,30,30,0,0,30,30,30,0,30,120,120,
  0,0,0,30,30,0,30,0,30,0,30,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,10,10,10,10,10,10,10,10,30,30,30,30,
  30,30,0,0,30,30,30,120,30,0,0,0,0,0,30,30,0,30,0,30,
  120,30,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,10,
  10,10,10,10,10,10,30,30,30,30,30,30,0,0,30,30,30,0,
  30,0,0,0,0,0,30,30,0,30,0,30,0,30,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,10,10,10,10,10,10,30,30,30,
  30,30,30,0,0,30,30,30,0,30,0,0,0,0,0,30,30,0,30,0,
  30,0,30,0,0,120,120,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  10,10,10,10,10,30,30,30,30,30,30,0,0,30,30,30,0,30,0,
  0,0,0,0,30,30,0,30,0,30,0,30,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,10,10,10,10,30,30,30,30,30,30,0,
  0,30,30,30,480,30,0,0,0,0,0,30,30,0,30,0,30,480,30,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,10,10,10,
  30,30,30,30,30,30,0,0,30,30,30,0,30,120,120,0,0,0,30,
  30,0,30,0,30,0,30,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,10,10,30,30,30,30,30,30,0,0,30,30,30,0,30,0,
  0,0,0,0,30,30,0,30,0,30,0,30,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,10,30,30,30,30,30,30,0,0,30,30,
  30,0,30,0,0,0,0,0,30,30,0,30,0,30,0,30,0,0,480,480,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,30,30,30,30,30,30,0,
  0,30,30,30,0,30,0,0,0,0,0,30,30,0,30,0,30,0,30,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,6,6,6,0,0,
  6,6,6,12,6,12,12,0,0,0,6,6,0,6,0,6,12,6,0,0,12,12,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,6,6,0,0,6,6,6,
  12,6,0,0,0,0,0,6,6,0,6,0,6,12,6,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,6,6,6,0,0,6,6,6,12,6,0,0,
  0,0,0,6,6,0,6,0,6,12,6,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,6,6,0,0,6,6,6,12,6,12,12,0,0,0,6,6,
  0,6,0,6,12,6,0,0,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,6,0,0,6,6,6,12,6,0,0,0,0,0,6,6,0,6,0,6,12,6,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,6,
  12,6,0,0,0,0,0,6,6,0,6,0,6,12,6,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,0,6,0,0,0,0,0,6,
  6,0,6,0,6,0,6,0,0,12,12,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,6,0,6,0,0,0,0,0,6,6,0,6,0,6,0,6,0,0,12,12,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,12,12,0,0,0,6,
  6,0,6,0,6,0,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,12,0,0,0,0,0,12,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,12,12,0,0,0,6,6,0,6,0,6,
  0,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,0,
  0,0,12,0,0,0,0,12,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,12,0,0,0,0,12,0,12,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,6,0,6,0,6,12,6,0,0,12,12,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,6,0,6,0,6,0,0,12,12,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,6,0,6,0,0,12,12,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,12,12,
  12,0,12,12,0,12,0,0,15,15,15,15,15,15,15,15,15,15,6,
  6,6,6,6,6,0,0,6,6,6,0,6,0,0,0,0,0,6,6,0,6,0,6,0,
  6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,12,12,
  0,12,12,0,12,0,0,15,15,15,15,15,15,15,15,15,15,6,6,
  6,6,6,6,0,0,6,6,6,0,6,0,0,0,0,0,6,6,0,6,0,6,0,6,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,12,0,12,
  12,96,12,96,0,60,60,60,60,60,60,60,60,60,60,12,12,12,
  12,12,12,0,0,12,12,12,0,12,0,0,0,0,0,12,12,48,12,0,
  12,48,12,0,0,24,24,0,108,0,0,0,0,0,0,0,0,72,0,0,72,
  0,12,0,12,12,96,12,0,0,60,60,60,60,60,60,60,60,60,60,
  12,12,12,12,12,12,0,0,12,12,12,0,12,0,0,0,0,0,12,12,
  0,12,108,12,0,12,24,24,0,0,0,0,0,0,0,0,0,0,0,0,0,
  72,0,72,0,0,12,12,0,12,0,0,60,60,60,60,60,60,60,60,
  60,60,12,12,12,12,12,12,0,0,12,12,12,0,12,0,0,0,0,0,
  12,12,24,12,36,12,24,12,24,24,0,0,0,0,0,0,0,0,0,0,
  0,0,36,36,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,0,
  12,0,0,60,60,60,60,60,60,60,60,60,60,12,12,12,12,12,
  12,48,0,12,12,12,0,12,0,48,0,0,0,12,12,0,12,0,12,0,
  12,0,0,24,24,0,108,0,0,0,72,0,0,0,0,0,72,0,0,0,0,
  12,0,0,60,60,60,60,60,60,60,60,60,60,12,12,12,12,12,
  12,24,0,12,12,12,0,12,0,24,0,0,0,12,12,0,12,0,12,0,
  12,0,0,0,0,0,36,0,0,0,36,0,0,0,0,0,72,0,36,0,0,0,
  0,0,7680,0,0,0,0,0,0,0,0,48,0,0,0,0,0,0,0,48,0,0,
  0,0,0,0,0,0,0,0,0,48,0,0,0,48,0,0,0,0,0,0,0,0,0,
  0,144,0,0,0,0,0,0,0,0,0,0,0,60,60,60,60,60,60,60,
  60,60,60,12,12,12,12,12,12,24,0,12,12,12,0,12,0,24,0,
  0,0,12,12,24,12,108,12,24,12,24,24,24,24,0,0,0,0,0,
  72,0,0,0,0,72,0,0,0,0,0,0,0,0,1920,0,0,0,0,0,0,48,
  0,0,0,0,0,48,0,48,0,0,0,0,0,48,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,144,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,15,15,15,15,15,15,15,15,15,30,30,30,30,30,
  30,120,0,30,30,30,0,30,0,120,0,0,0,30,30,60,30,810,
  30,60,30,0,0,0,0,0,270,0,0,0,1080,0,0,0,0,6480,6480,
  0,6480,0,15,15,15,15,15,15,15,15,30,30,30,30,30,30,0,
  0,30,30,30,0,30,0,0,0,0,0,30,30,60,30,0,30,60,30,60,
  60,0,0,0,0,0,0,0,0,0,0,0,0,6480,1080,0,1080,0,15,15,
  15,15,15,15,15,30,30,30,30,30,30,0,0,30,30,30,0,30,0,
  0,0,0,0,30,30,0,30,270,30,0,30,60,60,0,0,0,0,0,0,0,
  0,0,0,0,0,1080,1080,0,1080,0,15,15,15,15,15,15,30,30,
  30,30,30,30,120,0,30,30,30,0,30,0,120,0,0,0,30,30,
  120,30,270,30,120,30,0,0,60,60,0,810,0,0,0,1080,0,0,
  0,0,1080,0,0,0,0,15,15,15,15,15,30,30,30,30,30,30,0,
  0,30,30,30,0,30,0,0,0,0,0,30,30,60,30,0,30,60,30,
  120,120,120,120,0,270,0,0,0,0,0,0,0,0,1080,0,0,0,0,
  15,15,15,15,30,30,30,30,30,30,30,0,30,30,30,0,30,0,
  30,0,0,0,30,30,30,30,810,30,30,30,30,30,30,30,0,810,
  0,0,0,0,0,0,0,0,1080,6480,0,6480,0,15,15,15,30,30,30,
  30,30,30,60,0,30,30,30,0,30,0,60,0,0,0,30,30,0,30,0,
  30,0,30,0,0,0,0,0,810,0,0,0,6480,0,0,0,0,0,1080,0,
  1080,0,15,15,30,30,30,30,30,30,60,0,30,30,30,0,30,0,
  60,0,0,0,30,30,0,30,0,30,0,30,0,0,60,60,0,270,0,0,
  0,1080,0,0,0,0,0,0,0,0,0,15,30,30,30,30,30,30,30,0,
  30,30,30,0,30,0,30,0,0,0,30,30,30,30,270,30,30,30,30,
  30,30,30,0,0,0,0,0,6480,0,0,0,0,6480,0,0,0,0,30,30,
  30,30,30,30,0,0,30,30,30,0,30,0,0,0,0,0,30,30,60,30,
  0,30,60,30,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1080,0,0,0,
  0,6,6,6,6,6,12,0,6,6,6,0,6,0,12,0,0,0,6,6,12,6,
  18,6,12,6,6,6,6,6,0,18,0,0,0,36,0,0,0,0,36,36,0,
  36,0,6,6,6,6,0,0,6,6,6,0,6,0,0,0,0,0,6,6,12,6,18,
  6,12,6,12,12,0,0,0,0,0,0,0,0,0,0,0,0,36,36,0,0,0,
  6,6,6,0,0,6,6,6,0,6,0,0,0,0,0,6,6,12,6,18,6,12,6,
  12,12,0,0,0,0,0,0,0,0,0,0,0,0,36,36,0,0,0,6,6,0,
  0,6,6,6,0,6,0,0,0,0,0,6,6,12,6,18,6,12,6,0,0,12,
  12,0,0,0,0,0,0,0,0,0,0,36,36,0,0,0,6,12,0,6,6,6,
  0,6,0,12,0,0,0,6,6,12,6,18,6,12,6,6,6,6,6,0,18,0,
  0,0,36,0,0,0,0,36,36,0,36,0,12,0,6,6,6,0,6,0,12,0,
  0,0,6,6,12,6,18,6,12,6,6,6,6,6,0,18,0,0,0,36,0,0,
  0,0,36,36,0,36,0,0,12,0,12,0,12,0,6,0,0,0,12,12,6,
  6,36,12,6,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,72,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,0,6,0,12,0,0,
  0,6,6,12,6,18,6,12,6,6,6,6,6,0,18,0,0,0,36,0,0,0,
  0,36,36,0,36,0,6,0,6,0,0,0,0,0,6,6,12,6,18,6,12,6,
  0,0,12,12,0,0,0,0,0,0,0,0,0,0,36,36,0,0,0,0,6,0,
  12,0,0,0,6,6,12,6,18,6,12,6,6,6,6,6,0,18,0,0,0,36,
  0,0,0,0,36,36,0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,0,0,
  0,6,6,12,6,18,6,12,6,6,6,6,6,0,18,0,0,0,36,0,0,0,
  0,36,36,0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,12,6,6,36,
  12,6,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,72,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,6,0,6,0,6,0,6,0,0,12,12,
  0,18,0,0,0,36,0,0,0,0,0,0,0,36,0,0,6,0,6,0,6,0,0,
  12,12,0,18,0,0,0,36,0,0,0,0,0,0,0,36,0,6,0,0,6,0,
  0,0,0,0,0,36,0,0,0,72,0,0,0,0,0,0,0,72,0,0,6,6,6,
  12,12,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,12,12,
  0,0,0,18,0,0,0,36,0,0,0,0,0,0,0,36,0,0,0,0,0,0,0,
  36,0,0,0,72,0,0,0,0,0,0,0,72,0,12,12,0,0,0,18,0,0,
  0,36,0,0,0,0,0,0,0,36,0,6,6,6,0,0,0,0,0,72,0,0,0,
  0,0,72,0,0,0,6,6,0,0,0,0,0,72,0,0,0,0,0,72,0,0,0,
  6,0,0,0,0,0,0,0,0,0,0,0,72,0,72,0,0,0,0,0,0,0,0,
  0,0,0,0,72,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,36,0,0,0,0,0,0,0,0,0,12,12,12,
  0,12,12,0,12,0,0,15,15,15,15,15,15,15,15,15,15,6,6,
  6,6,6,6,0,0,6,6,6,0,6,0,0,0,0,0,6,6,0,6,0,6,0,6,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,12,0,12,
  12,0,12,0,0,60,60,60,60,60,60,60,60,60,60,12,12,12,
  12,12,12,0,0,12,12,12,24,12,48,0,0,108,0,12,12,0,12,
  0,12,24,12,0,48,0,0,0,0,0,0,0,0,0,0,72,0,0,0,0,0,
  72,12,0,12,12,0,12,0,96,60,60,60,60,60,60,60,60,60,
  60,12,12,12,12,12,12,24,24,12,12,12,0,12,0,0,0,0,0,
  12,12,0,12,0,12,0,12,0,0,0,0,0,0,108,0,72,0,0,0,0,
  0,0,0,0,0,72,0,12,12,0,12,0,0,60,60,60,60,60,60,60,
  60,60,60,12,12,12,12,12,12,0,0,12,12,12,0,12,24,0,0,
  36,0,12,12,0,12,0,12,0,12,0,24,0,0,0,0,0,0,36,0,0,
  0,36,0,0,0,0,0,72,96,0,0,0,0,0,0,1920,0,0,0,0,0,0,
  0,0,0,48,0,0,0,0,0,0,0,0,0,0,0,48,0,0,0,0,48,0,0,
  0,0,0,0,0,0,48,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  144,12,0,12,0,96,60,60,60,60,60,60,60,60,60,60,12,12,
  12,12,12,12,0,0,12,12,12,24,12,0,0,48,108,0,12,12,0,
  12,0,12,24,12,0,0,0,48,0,0,0,0,72,0,72,0,0,0,0,0,
  0,0,0,0,12,0,0,60,60,60,60,60,60,60,60,60,60,12,12,
  12,12,12,12,24,24,12,12,12,0,12,0,0,24,0,0,12,12,0,
  12,0,12,0,12,0,0,0,24,0,0,36,0,72,0,36,0,0,0,0,0,
  0,0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,60,60,60,60,60,60,
  60,60,60,60,12,12,12,12,12,12,24,24,12,12,12,24,12,24,
  0,24,0,0,12,12,0,12,0,12,24,12,0,24,0,24,0,0,108,0,
  0,0,72,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  7680,0,0,0,0,0,0,0,48,0,0,0,0,0,0,0,0,0,0,0,0,0,
  48,0,0,48,0,0,0,0,0,0,0,0,0,0,48,0,0,0,0,0,0,0,0,
  144,0,0,0,0,0,0,15,15,15,15,15,15,15,15,15,30,30,30,
  30,30,30,0,0,30,30,30,0,30,60,0,0,810,0,30,30,0,30,
  0,30,0,30,0,60,0,0,0,0,0,0,1080,0,0,0,6480,0,0,0,0,
  0,1080,15,15,15,15,15,15,15,15,30,30,30,30,30,30,0,0,
  30,30,30,60,30,120,0,120,810,0,30,30,0,30,0,30,60,30,
  0,120,0,120,0,0,270,0,0,0,1080,0,1080,0,0,0,0,0,0,
  15,15,15,15,15,15,15,30,30,30,30,30,30,0,0,30,30,30,
  60,30,60,0,0,270,0,30,30,0,30,0,30,60,30,0,60,0,0,0,
  0,0,0,0,0,0,0,1080,0,0,0,0,0,0,15,15,15,15,15,15,
  30,30,30,30,30,30,60,60,30,30,30,0,30,0,0,60,0,0,30,
  30,0,30,0,30,0,30,0,0,0,60,0,0,0,0,1080,0,6480,0,0,
  0,0,0,0,0,1080,15,15,15,15,15,30,30,30,30,30,30,120,
  120,30,30,30,120,30,0,0,60,270,0,30,30,0,30,0,30,120,
  30,0,0,0,60,0,0,0,0,0,0,1080,0,0,0,0,0,0,0,0,15,
  15,15,15,30,30,30,30,30,30,30,30,30,30,30,30,30,30,0,
  30,0,0,30,30,0,30,0,30,30,30,0,30,0,30,0,0,270,0,0,
  0,6480,0,6480,0,0,0,0,0,0,15,15,15,30,30,30,30,30,30,
  0,0,30,30,30,0,30,120,0,60,270,0,30,30,0,30,0,30,0,
  30,0,120,0,60,0,0,810,0,6480,0,6480,0,1080,0,0,0,0,0,
  6480,15,15,30,30,30,30,30,30,60,60,30,30,30,0,30,0,0,
  0,0,0,30,30,0,30,0,30,0,30,0,0,0,0,0,0,270,0,1080,
  0,1080,0,0,0,0,0,0,0,1080,15,30,30,30,30,30,30,30,30,
  30,30,30,30,30,30,0,30,810,0,30,30,0,30,0,30,30,30,0,
  30,0,30,0,0,810,0,6480,0,1080,0,0,0,0,0,0,0,6480,30,
  30,30,30,30,30,0,0,30,30,30,0,30,0,0,60,0,0,30,30,0,
  30,0,30,0,30,0,0,0,60,0,0,0,0,0,0,1080,0,0,0,0,0,
  0,0,0,6,6,6,6,6,0,0,6,6,6,12,6,12,0,0,18,0,6,6,0,
  6,0,6,12,6,0,12,0,0,0,0,0,0,36,0,0,0,36,0,0,0,0,
  0,0,6,6,6,6,6,6,6,6,6,6,6,12,0,12,18,0,6,6,0,6,0,
  6,6,6,0,12,0,12,0,0,18,0,36,0,36,0,36,0,0,0,0,0,
  36,6,6,6,0,0,6,6,6,12,6,12,0,0,18,0,6,6,0,6,0,6,
  12,6,0,12,0,0,0,0,0,0,36,0,0,0,36,0,0,0,0,0,0,6,
  6,0,0,6,6,6,12,6,0,0,12,0,0,6,6,0,6,0,6,12,6,0,0,
  0,12,0,0,18,0,0,0,36,0,0,0,0,0,0,0,36,6,0,0,6,6,
  6,12,6,0,0,12,0,0,6,6,0,6,0,6,12,6,0,0,0,12,0,0,
  18,0,0,0,36,0,0,0,0,0,0,0,36,12,12,6,6,6,12,6,6,0,
  6,0,0,6,6,0,6,0,6,12,6,0,6,0,6,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,6,12,6,6,6,12,0,0,0,0,0,6,12,0,6,0,
  6,6,12,0,0,0,0,0,0,0,0,0,0,0,0,72,0,0,0,0,0,72,
  12,6,6,6,12,0,0,0,0,0,6,12,0,6,0,6,6,12,0,0,0,0,
  0,0,0,0,0,0,0,0,72,0,0,0,0,0,72,6,6,0,6,0,0,12,0,
  0,6,6,0,6,0,6,0,6,0,0,0,12,0,0,18,0,0,0,36,0,0,0,
  0,0,0,0,36,6,6,6,12,0,12,18,0,6,6,0,6,0,6,6,6,0,
  12,0,12,0,0,18,0,36,0,36,0,36,0,0,0,0,0,36,6,6,12,
  0,12,18,0,6,6,0,6,0,6,6,6,0,12,0,12,0,0,18,0,36,0,
  36,0,36,0,0,0,0,0,36,0,0,0,0,0,0,6,0,0,6,0,6,6,0,
  0,0,0,0,0,0,0,0,72,0,0,0,0,0,0,0,0,0,72,12,0,0,
  18,0,6,6,0,6,0,6,0,6,0,12,0,0,0,0,0,0,36,0,0,0,
  36,0,0,0,0,0,0,0,6,0,0,12,0,0,12,0,12,0,12,0,6,0,
  6,0,0,36,0,0,0,72,0,0,0,0,0,0,0,72,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  36,0,12,12,0,12,0,12,0,0,0,6,0,6,0,0,0,0,72,0,0,0,
  72,0,0,0,0,0,0,0,18,0,0,18,0,18,0,18,0,0,0,36,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,0,6,0,6,6,6,0,
  12,0,12,0,0,18,0,36,0,36,0,36,0,0,0,0,0,36,0,6,0,
  6,0,6,0,0,0,12,0,0,18,0,0,0,36,0,0,0,0,0,0,0,36,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,6,6,6,0,12,0,12,0,0,18,0,36,0,36,0,36,0,0,0,0,0,
  36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  6,6,0,12,0,12,0,0,18,0,36,0,36,0,36,0,0,0,0,0,36,
  0,0,0,0,0,0,0,0,0,72,0,0,0,0,0,0,0,0,0,72,0,12,0,
  0,0,0,0,0,36,0,0,0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,6,0,0,36,0,0,0,72,0,0,0,
  0,0,0,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,72,0,0,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,12,12,0,12,12,0,12,0,0,60,60,60,60,60,60,60,60,60,
  60,12,12,12,12,12,12,0,0,12,12,12,48,12,48,0,0,0,0,
  12,12,48,12,0,12,0,12,48,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,12,0,12,12,0,12,0,0,60,60,60,60,60,60,
  60,60,60,60,12,12,12,12,12,12,0,0,12,12,12,0,12,0,0,
  0,0,0,12,12,0,12,0,12,0,12,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,24,12,12,24,12,12,12,60,60,60,60,60,
  60,60,60,60,60,12,12,12,12,12,12,0,24,12,12,12,0,12,
  24,0,0,36,36,12,12,24,12,36,12,0,12,0,0,24,0,36,36,
  36,36,36,72,72,72,36,36,36,36,72,72,72,0,12,12,0,0,0,
  0,0,60,0,60,0,0,60,0,60,0,0,12,0,12,0,0,24,0,12,24,
  0,12,0,0,0,0,0,0,12,0,0,0,12,0,0,0,0,24,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,12,0,12,0,0,60,60,60,60,
  60,60,60,60,60,60,12,12,12,12,12,12,0,48,12,12,12,0,
  12,0,48,48,0,0,12,12,0,12,0,12,0,12,0,0,48,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,12,12,24,24,60,60,60,60,60,
  60,60,60,60,60,12,12,12,12,12,12,0,0,12,12,12,24,12,
  0,24,24,36,36,12,12,0,12,36,12,0,12,24,0,0,0,36,36,
  36,72,72,36,36,36,72,72,72,72,36,36,36,0,0,0,0,960,
  60,0,60,0,0,60,0,60,0,0,12,0,12,0,0,0,0,12,24,0,12,
  0,0,0,0,0,0,12,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,60,60,60,60,60,60,60,60,60,
  60,12,12,12,12,12,12,0,24,12,12,12,24,12,24,24,24,0,
  108,12,12,24,12,0,12,0,12,24,0,24,0,108,0,0,0,0,0,0,
  72,0,72,0,0,0,0,0,12,0,0,60,0,60,0,0,60,0,60,0,0,
  12,0,12,0,0,0,0,12,24,24,12,0,0,0,0,0,0,12,0,0,0,
  12,0,0,24,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,60,960,60,0,0,60,0,60,0,0,12,0,12,0,0,0,0,12,24,
  0,12,0,0,0,0,0,0,12,0,0,0,12,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,15,15,15,15,15,15,15,15,15,
  30,30,30,30,30,30,0,0,30,30,30,60,30,60,0,0,0,0,30,
  30,60,30,0,30,0,30,60,0,0,0,0,0,0,0,38880,0,0,0,0,
  0,0,38880,0,38880,38880,15,15,15,15,15,15,15,15,30,30,
  30,30,30,30,0,0,30,30,30,120,30,120,0,0,0,0,30,30,
  120,30,0,30,0,30,120,0,0,0,0,0,0,0,0,0,0,0,38880,0,
  38880,0,0,0,0,15,15,15,15,15,15,15,30,30,30,30,30,30,
  0,0,30,30,30,120,30,60,0,0,90,90,30,30,60,30,90,30,0,
  30,120,0,0,0,90,90,90,180,180,180,180,180,180,180,180,
  180,180,180,180,15,15,15,15,15,15,30,30,30,30,30,30,0,
  120,30,30,30,0,30,0,120,120,0,0,30,30,0,30,0,30,0,30,
  0,0,120,0,0,0,0,0,0,38880,38880,0,0,0,0,0,0,0,0,15,
  15,15,15,15,30,30,30,30,30,30,0,60,30,30,30,60,30,0,
  0,0,90,90,30,30,0,30,90,30,0,30,60,0,60,0,90,90,90,
  180,180,180,180,180,180,180,180,180,180,180,180,15,15,15,
  15,30,30,30,30,30,30,0,30,30,30,30,30,30,30,30,30,0,
  0,30,30,30,30,0,30,0,30,30,0,30,0,0,0,0,6480,0,0,0,
  6480,0,0,0,0,1080,0,0,15,15,15,30,30,30,30,30,30,0,
  60,30,30,30,0,30,0,60,60,0,0,30,30,0,30,0,30,0,30,0,
  0,60,0,0,0,0,0,38880,0,0,0,0,0,0,38880,0,38880,38880,
  15,15,30,30,30,30,30,30,0,120,30,30,30,0,30,0,60,60,
  90,90,30,30,0,30,90,30,0,30,0,0,120,0,90,90,90,180,
  180,180,180,180,180,180,180,180,180,180,180,15,30,30,30,
  30,30,30,0,30,30,30,30,30,30,30,30,30,0,0,30,30,30,
  30,0,30,0,30,30,0,30,0,0,0,0,1080,0,0,0,0,0,6480,0,
  0,6480,0,0,30,30,30,30,30,30,0,0,30,30,30,0,30,60,60,
  60,90,90,30,30,60,30,90,30,0,30,0,0,0,0,90,90,90,180,
  180,180,180,180,180,180,180,180,180,180,180,6,6,6,6,6,
  0,0,6,6,6,12,6,12,0,0,18,0,6,6,12,6,18,6,0,6,12,0,
  0,0,18,0,0,0,36,0,0,36,36,0,36,36,36,0,0,6,6,6,6,
  0,0,6,6,6,12,6,12,0,0,18,0,6,6,12,6,18,6,0,6,12,0,
  0,0,18,0,0,0,36,0,0,36,36,0,36,36,36,0,0,6,6,6,0,
  12,6,6,6,12,6,6,6,6,18,18,6,6,6,6,18,6,0,6,12,0,
  12,0,18,18,18,36,36,36,36,36,36,36,36,36,36,36,36,6,
  6,0,12,6,6,6,12,6,6,6,6,0,18,6,6,6,6,0,6,0,6,12,
  0,12,0,18,0,0,36,0,0,0,36,0,36,0,0,36,0,0,6,0,0,6,
  6,6,12,6,0,12,12,18,18,6,6,0,6,18,6,0,6,12,0,0,0,
  18,18,18,36,36,36,36,36,36,36,36,36,36,36,36,0,12,6,
  6,6,12,6,6,6,6,0,18,6,6,6,6,0,6,0,6,12,0,12,0,18,
  0,0,36,0,0,0,36,0,36,0,0,36,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,12,12,6,6,12,0,0,0,36,0,12,12,0,12,36,
  0,0,12,6,0,6,0,36,0,0,0,72,0,0,72,72,0,72,72,72,72,
  72,6,6,0,6,0,12,12,0,18,6,6,0,6,0,6,0,6,0,0,12,0,
  0,18,18,36,0,36,36,0,0,36,0,0,0,36,36,6,0,6,12,0,0,
  18,18,6,6,12,6,18,6,0,6,0,0,12,0,18,18,18,36,36,36,
  36,36,36,36,36,36,36,36,36,6,6,12,12,12,18,0,6,6,12,
  6,18,6,0,6,6,0,6,0,0,18,18,0,36,36,36,0,36,0,36,36,
  0,36,36,0,0,0,0,0,36,0,12,0,12,0,12,0,12,6,0,6,0,
  0,36,36,72,72,72,72,0,0,72,0,72,0,72,72,12,0,0,18,
  18,6,6,12,6,18,6,0,6,0,0,12,0,18,18,18,36,36,36,36,
  36,36,36,36,36,36,36,36,6,6,0,36,0,6,6,6,0,0,0,6,0,
  0,0,0,0,36,36,72,0,72,72,0,0,72,0,0,0,72,72,6,36,0,
  12,6,6,6,36,12,0,6,0,0,0,0,36,0,0,0,72,0,0,72,72,
  0,72,72,72,0,0,36,0,12,6,6,6,36,12,0,6,0,0,0,0,36,
  0,0,0,72,0,0,72,72,0,72,72,72,0,0,18,0,18,0,0,18,
  18,0,0,0,0,36,0,0,0,0,36,0,0,0,0,0,36,0,0,0,0,0,
  18,18,36,18,18,18,0,18,36,0,0,0,0,0,0,0,36,0,0,0,
  36,0,36,36,0,0,0,6,0,6,0,6,0,6,0,0,12,0,0,18,18,
  36,0,36,36,0,0,36,0,0,0,36,36,6,6,18,6,0,6,12,0,12,
  0,18,18,18,36,36,36,36,36,36,36,36,36,36,36,36,6,0,0,
  0,6,0,0,0,0,0,36,36,72,0,72,72,0,0,72,0,0,0,72,72,
  0,6,0,6,12,0,12,0,18,0,0,36,0,0,0,36,0,36,0,0,36,
  0,0,18,0,0,0,0,36,0,0,0,0,36,0,0,0,0,0,36,0,0,0,
  0,0,0,6,12,0,0,0,18,18,18,36,36,36,36,36,36,36,36,
  36,36,36,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,12,0,12,0,18,0,0,36,0,0,0,36,0,36,0,0,36,0,0,
  0,6,0,0,36,36,72,72,72,72,0,0,72,0,72,0,72,72,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,0,0,72,0,0,
  72,72,0,72,72,72,72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,18,18,0,0,36,36,0,0,0,0,0,0,36,36,18,0,0,0,0,
  36,0,0,0,0,36,0,0,0,0,0,0,36,0,0,0,0,36,0,0,36,0,
  0,0,36,36,36,36,0,0,0,0,0,0,36,36,36,36,0,0,0,36,
  36,0,0,0,0,36,36,36,36,0,0,0,0,36,36,36,0,0,0,0,36,
  36,36,36,36,36,0,0,0,36,36,0,0,0,36,0,0,0,0,0,0,36,
  36,36,4,0,4,4,8,4,0,0,20,20,20,20,20,20,20,20,20,20,
  12,12,12,12,12,12,0,0,12,12,12,0,12,0,0,0,0,0,12,12,
  48,12,0,12,0,12,0,48,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  24,24,24,0,4,4,8,4,0,0,20,20,20,20,20,20,20,20,20,
  20,12,12,12,12,12,12,0,0,12,12,12,0,12,0,0,0,0,0,12,
  12,48,12,0,12,0,12,0,48,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,24,24,24,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,4,0,0,
  20,20,20,20,20,20,20,20,20,20,12,12,12,12,12,12,0,0,
  12,12,12,24,12,24,24,0,0,0,12,12,0,12,0,12,24,12,0,
  0,24,24,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,8,0,20,
  20,20,20,20,20,20,20,20,20,12,12,12,12,12,12,0,0,12,
  12,12,24,12,0,24,0,0,0,12,12,0,12,0,12,0,12,48,0,0,
  24,0,0,0,0,0,0,0,0,24,24,24,0,0,0,0,0,0,0,0,160,0,
  160,0,0,0,160,0,40,48,0,24,48,0,0,0,0,48,24,48,96,
  24,96,0,0,0,0,0,96,96,48,0,24,48,48,0,96,96,96,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,20,20,20,20,20,20,
  20,20,20,20,12,12,12,12,12,12,0,0,12,12,12,24,12,0,
  24,0,0,0,12,12,0,12,0,12,0,12,48,0,0,24,0,0,0,0,0,
  0,0,0,24,24,24,0,0,0,0,0,160,0,0,80,160,160,0,80,80,
  40,48,0,0,48,24,0,0,0,48,96,48,0,24,96,96,0,0,0,0,
  24,0,48,0,24,96,48,96,96,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,20,20,20,20,20,20,20,20,20,
  60,60,60,60,60,60,0,0,60,60,60,480,60,240,240,0,4860,
  0,60,60,0,60,0,60,0,60,960,960,240,0,0,1620,0,0,0,0,
  0,0,4320,720,4320,0,4320,4320,720,20,20,20,20,20,20,20,
  20,60,60,60,60,60,60,0,0,60,60,60,240,60,240,0,0,0,0,
  60,60,960,60,0,60,240,60,0,0,0,0,0,0,0,0,0,0,0,0,
  720,0,4320,0,4320,720,0,20,20,20,20,20,20,20,60,60,60,
  60,60,60,0,0,60,60,60,240,60,240,0,0,0,0,60,60,0,60,
  0,60,0,60,0,960,0,0,0,0,0,0,0,0,0,0,4320,0,720,0,
  720,720,4320,20,20,20,20,20,20,60,60,60,60,60,60,0,0,
  60,60,60,0,60,0,0,0,0,0,60,60,0,60,0,60,240,60,960,
  0,240,240,0,0,0,0,0,0,0,0,0,720,720,0,720,0,720,20,
  20,20,20,20,60,60,60,60,60,60,0,0,60,60,60,0,60,0,0,
  0,0,0,60,60,960,60,0,60,240,60,960,960,480,960,0,0,0,
  0,0,0,0,0,0,4320,4320,0,720,4320,0,20,20,20,20,60,60,
  60,60,60,60,0,0,60,60,60,480,60,0,0,0,0,0,60,60,960,
  60,0,60,480,60,480,0,480,480,0,4860,0,0,0,0,0,0,4320,
  0,720,0,720,4320,0,20,20,20,60,60,60,60,60,60,0,0,60,
  60,60,0,60,240,240,0,1620,0,60,60,0,60,0,60,0,60,0,
  960,0,480,0,4860,0,0,0,0,0,0,720,4320,0,0,0,720,4320,
  20,20,60,60,60,60,60,60,0,0,60,60,60,0,60,480,960,0,
  0,0,60,60,960,60,0,60,0,60,0,0,480,480,0,0,0,0,0,0,
  0,0,4320,720,0,0,4320,0,4320,20,60,60,60,60,60,60,0,0,
  60,60,60,0,60,480,480,0,4860,0,60,60,0,60,0,60,0,60,
  960,0,240,480,0,0,0,0,0,0,0,0,0,4320,4320,0,4320,0,
  4320,60,60,60,60,60,60,0,0,60,60,60,0,60,0,0,0,180,0,
  60,60,0,60,0,60,0,60,0,0,0,960,0,180,0,0,0,0,0,0,
  120,120,120,0,120,120,120,12,12,12,12,12,0,0,12,12,12,
  24,12,24,24,0,108,0,12,12,48,12,0,12,24,12,48,48,24,
  24,0,108,0,0,0,0,0,0,72,0,72,0,72,72,0,12,12,12,12,
  0,0,12,12,12,24,12,48,0,0,108,0,12,12,48,12,0,12,24,
  12,48,48,0,0,0,0,0,0,0,0,0,0,72,0,72,0,72,0,72,12,
  12,12,0,0,12,12,12,24,12,24,0,0,0,0,12,12,48,12,0,
  12,24,12,48,48,0,0,0,0,0,0,0,0,0,0,0,0,72,0,24,24,
  24,12,12,0,0,12,12,12,24,12,24,24,0,0,0,12,12,48,12,
  0,12,24,12,48,0,24,24,0,0,0,0,0,0,0,0,0,72,72,0,72,
  0,72,12,0,0,12,12,12,24,12,0,48,0,0,0,12,12,48,12,0,
  12,24,12,48,48,48,24,0,0,0,0,0,0,0,0,24,24,24,0,72,
  0,0,0,0,12,12,12,24,12,0,24,0,0,0,12,12,48,12,0,12,
  24,12,48,48,24,48,0,108,0,0,0,0,0,0,0,72,72,0,72,72,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,12,12,0,12,0,0,0,0,0,12,12,48,12,0,12,48,
  12,0,0,24,24,0,108,0,0,0,0,0,0,0,72,72,0,0,72,72,
  12,0,12,0,0,0,0,0,12,12,48,12,0,12,24,12,0,48,24,24,
  0,0,0,0,0,0,0,0,0,72,0,0,24,24,24,0,12,24,24,0,108,
  0,12,12,0,12,0,12,0,12,48,48,48,48,0,108,0,0,0,0,0,
  0,72,0,72,0,0,72,72,0,0,0,0,0,0,24,48,48,24,0,0,24,
  0,48,192,96,0,0,216,0,0,0,0,0,0,0,144,0,0,0,144,144,
  24,24,0,108,0,12,12,0,12,0,12,0,12,48,48,48,48,0,36,
  0,0,0,0,0,0,72,72,72,0,24,24,24,24,0,0,0,24,48,48,
  24,0,24,96,24,0,48,96,96,0,432,0,0,0,0,0,0,0,144,
  144,0,144,288,0,0,0,0,24,24,48,48,0,24,96,24,0,48,0,
  0,0,0,0,0,0,0,0,0,0,0,144,0,144,0,144,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  108,0,432,108,0,36,432,108,0,0,216,432,0,0,0,0,0,0,0,
  0,0,72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,12,0,12,0,12,24,12,0,48,
  24,24,0,108,0,0,0,0,0,0,72,72,0,0,0,72,72,48,12,0,
  12,0,12,48,0,24,24,0,0,0,0,0,0,0,0,24,24,24,0,0,0,
  72,48,0,0,48,0,0,192,0,192,0,432,0,0,0,0,0,0,144,
  144,0,0,0,288,0,0,12,0,12,0,0,24,24,0,0,0,0,0,0,0,
  0,72,72,0,0,72,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,12,48,48,0,48,0,108,0,0,0,0,0,0,
  24,24,24,0,72,72,72,0,48,0,96,0,0,0,0,0,0,0,0,0,
  144,144,0,0,288,0,144,48,48,48,0,0,108,0,0,0,0,0,0,
  72,72,0,0,72,72,0,0,48,48,0,0,0,0,0,0,0,0,864,288,
  0,0,0,144,144,48,48,0,0,0,0,0,0,0,0,0,144,144,0,864,
  0,288,24,0,0,0,0,0,0,0,0,144,0,144,0,432,144,288,0,
  0,0,0,0,0,0,0,144,0,0,0,144,144,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,0,72,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,24,24,0,0,0,0,24,0,0,0,0,0,0,0,0,0,0,0,
  24,24,24,0,4,4,8,4,0,0,20,20,20,20,20,20,20,20,20,
  20,12,12,12,12,12,12,0,0,12,12,12,0,12,0,0,0,0,0,12,
  12,48,12,0,12,0,12,0,48,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,24,24,24,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,4,0,8,
  20,20,20,20,20,20,20,20,20,20,12,12,12,12,12,12,48,0,
  12,12,12,0,12,0,0,48,0,0,12,12,0,12,0,12,0,12,0,0,
  0,0,0,0,0,24,24,0,0,0,0,0,0,24,0,0,0,0,4,0,8,20,
  20,20,20,20,20,20,20,20,20,12,12,12,12,12,12,48,0,12,
  12,12,0,12,0,0,48,0,0,12,12,0,12,0,12,0,12,0,0,0,0,
  0,0,0,24,24,0,0,0,0,0,0,24,0,0,0,0,0,0,0,160,160,
  160,0,0,0,0,0,40,48,0,24,48,96,0,96,48,48,24,48,0,
  24,0,0,96,0,0,0,0,96,48,0,24,0,48,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,20,20,20,20,20,20,20,20,
  20,20,12,12,12,12,12,12,24,24,12,12,12,0,12,0,0,24,0,
  0,12,12,24,12,0,12,0,12,24,24,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,160,0,160,0,0,0,
  160,0,40,0,48,0,48,24,48,0,0,0,96,48,0,24,0,0,96,0,
  0,48,24,96,0,0,24,0,48,48,96,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,20,20,20,20,20,20,20,20,20,60,60,60,
  60,60,60,0,0,60,60,60,0,60,0,0,0,0,0,60,60,240,60,
  4860,60,0,60,0,480,0,0,0,0,0,0,720,0,0,0,0,0,0,4320,
  4320,4320,720,20,20,20,20,20,20,20,20,60,60,60,60,60,
  60,0,240,60,60,60,0,60,0,0,240,0,0,60,60,480,60,0,60,
  0,60,240,240,0,0,4860,0,1620,720,0,0,0,0,0,0,0,720,
  4320,720,0,20,20,20,20,20,20,20,60,60,60,60,60,60,0,0,
  60,60,60,0,60,0,0,0,0,0,60,60,480,60,1620,60,0,60,0,
  240,0,0,1620,0,0,0,4320,0,0,0,0,0,0,720,720,4320,0,
  20,20,20,20,20,20,60,60,60,60,60,60,240,240,60,60,60,
  0,60,0,0,480,0,4860,60,60,240,60,1620,60,0,60,240,0,0,
  0,0,0,0,4320,720,0,0,0,0,0,0,0,720,0,720,20,20,20,
  20,20,60,60,60,60,60,60,0,240,60,60,60,0,60,0,0,240,
  0,0,60,60,240,60,1620,60,0,60,240,0,0,0,0,0,1620,4320,
  0,0,0,0,0,0,0,4320,4320,0,4320,20,20,20,20,60,60,60,
  60,60,60,0,0,60,60,60,0,60,0,0,480,0,0,60,60,240,60,
  0,60,0,60,480,480,0,0,0,0,4860,4320,0,0,0,0,0,0,0,
  4320,720,4320,0,20,20,20,60,60,60,60,60,60,480,0,60,60,
  60,0,60,0,0,240,0,0,60,60,0,60,0,60,0,60,0,0,0,0,0,
  0,4860,4320,4320,0,0,0,0,0,0,720,0,720,4320,20,20,60,
  60,60,60,60,60,240,0,60,60,60,0,60,0,0,480,0,1620,60,
  60,0,60,0,60,0,60,0,0,0,0,0,0,1620,720,4320,0,0,0,0,
  0,0,0,0,4320,720,20,60,60,60,60,60,60,480,480,60,60,
  60,0,60,0,0,240,0,0,60,60,480,60,4860,60,0,60,0,0,0,
  0,0,0,0,720,4320,0,0,0,0,0,0,0,4320,0,4320,60,60,60,
  60,60,60,0,0,60,60,60,0,60,0,0,960,0,180,60,60,960,
  60,180,60,0,60,0,0,0,0,180,0,180,120,120,0,0,0,0,0,
  0,120,120,120,120,12,12,12,12,12,0,0,12,12,12,0,12,0,
  0,0,0,0,12,12,0,12,108,12,0,12,24,24,0,0,108,0,0,0,
  72,0,0,0,0,0,0,72,72,72,0,12,12,12,12,24,24,12,12,
  12,0,12,0,0,24,0,0,12,12,24,12,108,12,0,12,24,24,0,
  0,108,0,108,0,72,0,0,0,0,0,0,72,72,0,72,12,12,12,0,
  0,12,12,12,0,12,0,0,0,0,0,12,12,24,12,0,12,0,12,24,
  24,0,0,36,0,0,0,72,0,0,0,0,0,0,0,24,24,24,12,12,0,
  48,12,12,12,0,12,0,0,24,0,108,12,12,24,12,108,12,0,
  12,48,0,0,0,108,0,108,72,0,0,0,0,0,0,0,72,72,0,72,
  12,48,0,12,12,12,0,12,0,0,24,0,0,12,12,24,12,36,12,
  0,12,24,0,0,0,0,0,108,24,24,0,0,0,0,0,0,24,0,0,72,
  24,24,12,12,12,0,12,0,0,0,0,108,12,12,48,12,108,12,0,
  12,0,0,0,0,108,0,0,72,0,0,0,0,0,0,0,72,72,72,0,24,
  24,0,24,0,24,0,0,0,0,0,24,24,0,0,432,0,0,24,96,0,0,
  0,216,0,0,0,0,0,0,0,0,0,0,144,144,0,144,24,24,0,0,
  24,0,0,0,0,0,24,24,0,0,216,0,0,24,48,96,0,0,0,0,0,
  0,144,0,0,0,0,0,0,144,432,288,144,12,12,0,12,0,0,24,
  0,108,12,12,24,12,108,12,0,12,24,24,0,0,0,0,108,72,0,
  0,0,0,0,0,0,72,0,72,72,12,0,12,0,0,24,0,0,12,12,24,
  12,108,12,0,12,0,48,0,0,0,0,36,0,0,0,0,0,0,0,0,72,
  24,24,24,0,12,0,0,24,0,0,12,12,24,12,108,12,0,12,0,
  24,0,0,0,0,108,0,72,0,0,0,0,0,0,72,0,72,72,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,108,12,12,48,12,108,12,0,12,0,
  0,0,0,36,0,36,72,72,0,0,0,0,0,0,72,24,24,24,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,24,24,48,0,48,0,24,0,0,
  0,0,0,0,0,0,144,0,0,0,0,0,0,0,144,144,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  108,36,0,108,36,36,0,108,0,216,0,0,0,0,0,0,72,0,0,0,
  0,0,0,72,0,0,0,12,0,12,0,12,0,12,0,0,0,0,0,0,108,
  72,72,0,0,0,0,0,0,0,0,72,72,0,12,0,12,0,12,0,0,0,
  0,0,0,0,24,24,0,0,0,0,0,0,24,0,72,0,0,0,0,0,24,0,
  0,0,0,0,0,0,144,144,0,0,0,0,0,0,0,0,144,0,0,12,0,
  12,24,24,0,0,108,0,108,72,72,0,0,0,0,0,0,0,72,0,72,
  36,0,0,0,0,0,0,0,0,0,72,72,0,0,0,0,0,0,0,0,0,0,0,
  12,24,24,0,0,108,0,108,24,24,0,0,0,0,0,0,24,72,72,
  72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,24,24,
  0,0,108,0,0,72,72,0,0,0,0,0,0,0,72,72,0,24,0,0,0,
  0,216,432,288,0,0,0,0,0,0,144,0,144,144,0,0,0,0,432,
  144,0,0,0,0,0,0,0,144,0,0,144,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,
  0,0,0,0,0,0,0,0,0,0,72,72,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,72,72,0,24,0,0,0,0,0,0,24,
  0,0,0,0,0,0,0,0,0,24,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,24,24,24,8,8,8,8,0,0,0,0,40,0,40,
  80,160,40,0,40,0,0,24,0,24,0,0,48,0,24,0,0,24,48,0,
  0,0,0,48,24,0,0,0,24,48,0,0,48,48,0,0,0,0,0,0,0,0,
  0,0,0,0,0,24,24,24,4,0,4,0,0,20,20,20,20,20,20,20,
  20,20,20,12,12,12,12,12,12,0,48,12,12,12,0,12,24,0,0,
  0,0,12,12,0,12,0,12,24,12,0,0,24,0,0,0,0,0,0,24,24,
  24,0,0,0,0,0,0,0,8,4,8,8,20,20,20,20,20,20,20,20,
  20,20,12,12,12,12,12,12,0,48,12,12,12,0,12,0,0,0,12,
  12,12,12,0,12,12,12,0,12,48,0,0,0,12,12,12,24,24,24,
  24,24,24,24,24,24,24,24,24,0,0,0,0,0,40,0,40,0,0,40,
  0,40,48,0,24,0,24,0,0,0,0,24,0,0,24,48,0,0,0,0,0,
  24,48,0,0,24,0,0,0,48,0,0,0,0,0,0,0,24,24,24,0,0,
  0,0,0,0,0,0,0,20,20,20,20,20,20,20,20,20,20,12,12,
  12,12,12,12,0,48,12,12,12,0,12,24,0,0,0,0,12,12,0,
  12,0,12,24,12,0,0,24,0,0,0,0,0,0,24,24,24,0,0,0,0,
  0,0,0,4,0,0,20,80,20,0,0,20,0,20,0,0,12,24,12,0,0,
  0,0,12,24,0,12,0,0,0,0,0,0,12,0,0,0,12,48,24,24,48,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,20,80,20,0,
  0,20,0,20,0,0,12,24,12,0,0,0,0,12,24,0,12,0,0,0,0,
  0,0,12,0,0,0,12,48,24,24,48,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,20,20,20,20,20,20,20,20,20,60,60,60,60,
  60,60,0,0,60,60,60,0,60,480,0,0,1620,0,60,60,480,60,
  1620,60,240,60,0,240,0,0,0,0,0,0,2160,0,0,0,2160,0,0,
  0,4320,4320,0,20,20,20,20,20,20,20,20,60,60,60,60,60,
  60,0,0,60,60,60,0,60,0,0,0,0,0,60,60,240,60,0,60,
  480,60,0,240,0,0,0,0,0,0,0,0,0,0,0,0,2160,2160,4320,
  0,0,20,20,20,20,20,20,20,60,60,60,60,60,60,0,0,60,60,
  60,0,60,240,0,0,60,60,60,60,240,60,60,60,240,60,240,
  240,0,0,60,60,60,60,60,120,120,120,60,60,60,60,120,120,
  120,20,20,20,20,20,20,60,60,60,60,60,60,0,480,60,60,
  60,0,60,0,0,0,0,1620,60,60,240,60,1620,60,0,60,480,0,
  480,0,0,0,0,2160,0,0,0,0,0,2160,0,0,0,0,0,20,20,20,
  20,20,60,60,60,60,60,60,0,240,60,60,60,0,60,0,0,0,60,
  60,60,60,240,60,60,60,240,60,240,240,240,0,60,60,60,60,
  60,120,120,120,60,60,60,60,120,120,120,20,20,20,20,60,
  60,60,60,60,60,0,240,60,60,60,0,60,240,0,0,0,0,60,60,
  240,60,0,60,480,60,0,0,0,0,0,1620,1620,0,0,0,0,0,0,
  0,2160,2160,4320,0,0,20,20,20,60,60,60,60,60,60,0,240,
  60,60,60,0,60,480,0,0,1620,0,60,60,0,60,0,60,0,60,0,
  240,240,0,0,0,0,0,2160,0,0,0,2160,0,0,0,0,0,4320,20,
  20,60,60,60,60,60,60,0,240,60,60,60,0,60,240,0,0,60,
  60,60,60,240,60,60,60,0,60,0,0,240,0,60,60,60,60,60,
  120,120,120,60,60,60,60,120,120,120,20,60,60,60,60,60,
  60,0,480,60,60,60,0,60,0,0,0,0,0,60,60,0,60,0,60,
  240,60,0,240,240,0,0,0,0,2160,0,0,720,0,0,2160,0,0,0,
  0,0,60,60,60,60,60,60,0,240,60,60,60,0,60,240,0,0,60,
  60,60,60,240,60,60,60,240,60,0,0,0,0,60,60,60,60,60,
  120,120,120,60,60,60,60,120,120,120,12,12,12,12,12,0,0,
  12,12,12,0,12,24,0,0,36,0,12,12,24,12,36,12,24,12,24,
  24,0,0,108,0,0,0,36,0,0,72,36,0,36,36,0,72,0,12,12,
  12,12,0,0,12,12,12,0,12,24,0,0,36,0,12,12,24,12,36,
  12,24,12,24,24,0,0,0,0,0,0,36,0,0,72,36,0,36,36,0,
  0,72,12,12,12,0,24,12,12,12,0,12,24,0,0,36,36,12,12,
  24,12,36,12,24,12,0,24,24,0,12,12,12,36,36,24,24,24,
  36,36,36,36,24,24,24,12,12,0,24,12,12,12,0,12,0,0,0,
  0,36,12,12,24,12,36,12,24,12,24,0,24,0,108,0,108,36,
  0,0,72,0,0,36,36,36,72,0,0,12,0,0,12,12,12,0,12,0,
  0,0,12,12,12,12,24,12,12,12,24,12,24,24,24,0,36,36,
  36,12,12,72,72,72,12,12,12,12,72,72,72,0,24,12,12,12,
  0,12,0,0,0,0,36,12,12,24,12,36,12,24,12,24,0,24,0,0,
  108,0,36,0,72,0,0,0,36,36,36,72,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,24,24,0,0,24,0,0,0,72,0,24,24,48,24,
  72,24,24,24,48,48,24,0,0,0,0,0,72,0,0,432,72,0,72,
  72,144,144,144,12,12,0,12,0,0,0,0,36,12,12,24,12,36,
  12,24,12,24,24,24,0,0,108,0,36,0,0,72,0,0,36,36,36,
  0,72,0,12,0,12,24,0,0,36,36,12,12,24,12,36,12,24,12,
  0,24,24,0,12,12,12,36,36,24,24,24,36,36,36,36,24,24,
  24,0,12,24,0,0,36,0,12,12,24,12,36,12,24,12,24,24,24,
  0,0,0,108,0,36,72,0,0,36,0,36,36,0,0,72,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,24,0,0,36,36,12,12,24,12,36,12,24,12,0,24,
  24,0,12,12,12,36,36,24,24,24,36,36,36,36,24,24,24,0,
  0,0,72,24,24,24,24,72,24,48,24,48,24,48,0,0,432,432,
  72,0,0,432,0,0,72,72,72,144,144,144,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,12,36,12,72,36,12,12,72,36,72,0,72,0,0,0,0,12,
  0,0,0,0,0,12,12,12,0,0,0,36,12,72,36,12,12,72,36,72,
  72,0,0,0,0,0,0,12,0,0,0,12,0,12,12,0,0,0,12,0,12,
  0,12,0,12,0,24,24,0,0,0,108,36,36,72,0,0,36,36,0,0,
  0,0,72,24,12,12,12,0,12,24,0,24,0,36,36,36,12,12,72,
  72,72,12,12,12,12,72,72,72,0,0,0,24,0,0,48,0,0,0,0,
  0,72,72,144,144,144,72,72,0,0,0,0,432,0,12,0,12,0,24,
  24,0,0,0,0,36,36,0,0,72,36,36,0,0,0,0,72,12,0,0,0,
  72,72,0,0,0,0,12,12,0,0,0,12,12,0,0,0,0,0,24,12,24,
  24,0,0,36,36,36,12,12,72,72,72,12,12,12,12,72,72,72,
  0,0,0,48,0,0,0,0,72,72,432,0,0,72,72,0,0,144,144,
  144,0,24,24,0,108,0,0,36,36,0,0,72,36,36,0,0,0,72,0,
  24,24,0,0,0,0,72,72,0,0,0,72,72,0,0,0,0,0,24,0,0,
  0,432,72,0,144,144,144,0,72,72,72,0,0,0,0,432,0,0,0,
  72,0,0,0,72,0,72,72,144,144,144,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,12,12,0,0,24,0,0,0,0,0,0,0,0,24,12,0,
  0,0,0,24,0,0,0,0,0,0,24,0,0,0,0,24,0,0,0,0,0,24,
  0,12,0,0,0,12,12,12,12,0,0,0,0,0,0,12,12,12,12,0,0,
  0,24,24,0,0,0,0,24,24,24,24,0,0,0,0,24,24,24,0,0,0,
  0,24,24,24,12,12,12,0,0,0,12,12,0,0,0,12,0,0,0,0,0,
  0,24,24,24,0,0,8,0,0,0,80,80,0,160,80,160,0,160,40,
  0,48,24,48,96,48,96,96,0,24,48,0,24,96,96,0,0,0,48,
  0,0,0,0,24,0,48,0,0,0,96,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,4,0,0,0,0,80,20,0,20,0,0,20,0,20,0,0,12,24,
  12,0,48,24,0,12,24,0,12,0,0,0,0,0,0,12,0,0,0,12,0,
  24,0,0,0,48,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,80,20,0,20,0,0,20,0,20,0,0,12,24,12,0,48,24,0,12,
  24,0,12,0,0,0,0,0,0,12,0,0,0,12,0,24,0,0,0,48,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,80,160,0,0,80,160,
  0,160,40,0,48,24,48,0,48,96,0,0,24,48,0,24,0,96,0,0,
  0,48,96,0,0,0,24,96,48,96,48,96,96,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,80,80,0,0,480,0,480,240,480,0,480,
  0,0,480,0,0,480,0,0,0,0,480,480,480,0,0,0,0,0,3840,
  480,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2160,360,0,0,20,0,
  80,20,80,20,0,0,60,240,60,240,0,0,0,60,0,0,60,240,0,
  0,0,0,240,60,0,0,0,60,0,0,0,480,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,20,0,20,0,240,60,0,60,0,
  0,240,0,60,240,0,60,0,0,0,0,0,240,60,240,0,0,60,240,
  0,960,240,240,960,0,0,0,0,0,0,0,0,0,0,0,0,0,180,0,
  80,40,0,80,0,480,480,480,0,1920,0,0,0,480,1920,0,0,0,
  0,0,0,0,480,240,960,0,0,480,1920,480,960,480,480,480,0,
  0,0,0,0,0,0,0,0,0,0,0,360,2160,0,0,0,80,0,960,0,
  1920,480,3840,30720,960,0,960,480,0,480,480,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  720,2160,80,20,0,480,60,240,60,0,960,240,0,60,0,0,60,
  240,0,0,0,0,0,60,480,0,0,60,0,0,0,0,0,3840,0,0,0,0,
  0,0,0,0,0,0,0,0,0,540,540,0,0,960,480,480,0,1920,0,
  1920,0,480,480,0,0,1920,960,0,0,0,480,1920,0,0,0,480,
  0,1920,960,960,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  2160,0,0,60,0,60,480,0,240,0,60,0,0,60,240,0,0,0,0,
  0,60,0,0,0,60,240,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,60,60,60,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,24,24,48,48,48,0,24,24,0,24,24,0,0,0,0,48,24,
  24,0,0,0,0,0,96,48,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  72,0,72,24,12,24,0,24,0,12,24,0,12,24,0,0,0,0,24,12,
  0,0,0,12,24,0,48,24,24,0,0,0,0,0,0,0,0,0,0,0,0,0,
  12,12,12,24,24,0,24,0,0,48,0,24,48,48,0,0,0,0,0,24,
  0,0,24,24,48,0,0,48,96,0,0,0,0,0,0,0,0,0,0,0,0,72,
  0,72,0,48,0,0,12,0,0,12,0,48,0,0,0,24,12,24,0,0,12,
  24,24,48,24,24,48,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,
  48,24,0,24,0,0,0,0,96,0,0,0,48,24,48,0,0,24,24,24,
  96,48,48,48,0,0,0,0,0,0,0,0,0,0,0,0,72,72,0,48,0,
  0,96,0,48,96,0,0,0,0,0,0,48,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,144,0,144,0,24,0,0,24,0,96,
  0,0,0,48,24,48,0,0,24,24,24,96,48,24,48,0,0,0,0,0,
  0,0,0,0,0,0,0,0,72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  24,0,12,24,0,0,0,0,0,12,24,0,0,12,24,24,0,0,24,48,
  0,0,0,0,0,0,0,0,0,0,0,0,12,12,12,0,0,24,48,0,0,0,
  24,24,24,0,0,24,48,48,48,24,48,96,0,0,0,0,0,0,0,0,
  0,0,0,0,0,72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,24,48,0,0,0,24,
  12,24,0,0,12,24,24,48,24,24,48,0,0,0,0,0,0,0,0,0,0,
  0,0,12,12,12,48,0,0,0,24,24,24,0,0,24,48,48,48,48,
  48,96,0,0,0,0,0,0,0,0,0,0,0,0,72,72,0,0,0,0,0,48,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,144,
  0,144,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,24,0,24,0,
  24,24,48,0,0,0,0,0,0,0,0,0,0,0,0,0,72,72,24,0,0,
  12,0,24,0,0,24,48,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  24,0,48,24,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,36,36,0,
  0,0,48,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,72,48,24,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,72,72,0,0,48,48,0,0,0,
  0,0,0,0,0,0,0,0,0,0,144,144,24,48,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,72,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,144,144,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  12,12,12,0,4,0,8,20,20,20,20,20,20,20,20,20,20,12,12,
  12,12,12,12,48,0,12,12,12,0,12,0,0,48,0,0,12,12,0,
  12,0,12,0,12,0,0,0,0,0,0,0,24,24,0,0,0,0,0,0,24,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,20,20,20,20,20,20,20,
  20,20,20,12,12,12,12,12,12,0,48,12,12,12,0,12,24,0,0,
  0,0,12,12,0,12,0,12,24,12,0,0,24,0,0,0,0,0,0,24,24,
  24,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,160,160,160,0,
  0,0,0,0,40,0,48,96,48,24,48,96,0,0,0,48,96,24,0,96,
  96,0,0,48,24,0,0,0,24,96,48,0,0,96,48,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,20,20,20,20,20,20,20,20,20,60,60,
  60,60,60,60,960,0,60,60,60,0,60,240,240,0,4860,0,60,
  60,0,60,0,60,480,60,0,0,0,0,0,1620,0,0,720,720,0,
  4320,0,0,0,4320,0,0,0,20,20,20,20,20,20,20,20,60,60,
  60,60,60,60,0,960,60,60,60,240,60,0,0,0,0,0,60,60,0,
  60,0,60,240,60,0,0,0,240,0,0,0,720,0,0,720,720,0,0,
  0,720,0,0,0,20,20,20,20,20,20,20,60,60,60,60,60,60,0,
  0,60,60,60,480,60,960,480,960,0,0,60,60,0,60,0,60,480,
  60,0,0,0,0,0,0,0,4320,0,4320,0,720,0,0,0,4320,0,0,0,
  20,20,20,20,20,20,60,60,60,60,60,60,0,0,60,60,60,0,
  60,0,240,960,0,0,60,60,0,60,0,60,0,60,0,0,240,240,0,
  0,0,4320,720,720,4320,0,0,0,0,0,0,0,0,20,20,20,20,20,
  60,60,60,60,60,60,960,960,60,60,60,480,60,0,0,960,0,0,
  60,60,0,60,0,60,960,60,0,0,0,240,0,0,0,720,4320,0,
  4320,4320,0,0,0,0,0,0,0,20,20,20,20,60,60,60,60,60,
  60,0,960,60,60,60,240,60,480,480,0,0,0,60,60,0,60,0,
  60,480,60,0,0,0,0,0,4860,0,4320,0,0,4320,4320,0,0,0,
  4320,0,0,0,20,20,20,60,60,60,60,60,60,960,960,60,60,
  60,240,60,240,240,0,1620,0,60,60,0,60,0,60,0,60,0,0,
  480,0,0,4860,0,4320,4320,4320,4320,720,0,0,0,720,0,0,0,
  20,20,60,60,60,60,60,60,960,0,60,60,60,0,60,0,240,0,
  0,0,60,60,0,60,0,60,0,60,0,0,240,0,0,0,0,720,720,
  4320,720,0,0,0,0,4320,0,0,0,20,60,60,60,60,60,60,0,
  480,60,60,60,480,60,0,0,960,4860,0,60,60,0,60,0,60,
  480,60,0,0,480,480,0,0,0,720,4320,4320,720,0,0,0,0,0,
  0,0,0,60,60,60,60,60,60,0,0,60,60,60,0,60,0,0,0,180,
  0,60,60,0,60,0,60,960,60,0,0,0,0,0,180,0,120,120,120,
  120,120,0,0,0,120,0,0,0,12,12,12,12,12,48,0,12,12,12,
  24,12,24,24,0,108,0,12,12,0,12,0,12,24,12,0,0,24,24,
  0,108,0,0,72,72,0,72,0,0,0,72,0,0,0,12,12,12,12,0,
  0,12,12,12,24,12,0,0,48,108,0,12,12,0,12,0,12,24,12,
  0,0,0,48,0,0,0,0,72,0,72,72,0,0,0,72,0,0,0,12,12,
  12,0,48,12,12,12,24,12,24,48,48,0,0,12,12,0,12,0,12,
  24,12,0,0,48,0,0,0,0,0,0,24,24,24,0,0,0,72,0,0,0,
  12,12,0,48,12,12,12,24,12,24,24,48,0,0,12,12,0,12,0,
  12,24,12,0,0,24,24,0,0,0,72,0,0,72,72,0,0,0,72,0,0,
  0,12,48,0,12,12,12,24,12,0,0,48,0,0,12,12,0,12,0,12,
  24,12,0,0,0,24,0,0,0,24,24,0,0,72,0,0,0,24,0,0,0,
  0,0,12,12,12,24,12,48,24,48,0,0,12,12,0,12,0,12,24,
  12,0,0,24,0,0,108,0,72,0,72,0,72,0,0,0,72,0,0,0,0,
  48,48,48,48,48,48,48,192,0,0,48,48,0,48,0,48,48,48,0,
  0,192,0,0,0,0,864,0,0,144,144,0,0,0,288,0,0,0,48,48,
  48,48,48,0,0,0,0,0,48,48,0,48,0,48,48,48,0,0,48,48,
  0,0,0,0,144,864,0,288,0,0,0,144,0,0,0,12,12,0,12,0,
  48,48,0,0,12,12,0,12,0,12,0,12,0,0,24,24,0,108,0,72,
  0,72,72,0,0,0,0,72,0,0,0,12,48,12,48,0,48,0,0,12,
  12,0,12,0,12,24,12,0,0,24,24,0,0,0,72,0,24,24,24,0,
  0,0,0,0,0,0,48,12,24,24,0,108,0,12,12,0,12,0,12,48,
  12,0,0,0,0,0,108,0,0,72,72,72,0,0,0,0,72,0,0,0,0,
  0,96,0,0,0,24,0,0,24,0,48,24,48,0,0,96,96,0,216,0,
  432,144,144,144,0,0,0,0,288,0,0,0,24,24,0,108,0,12,
  12,0,12,0,12,48,12,0,0,0,0,0,36,0,72,72,24,24,24,0,
  0,0,72,0,0,0,24,48,0,0,24,0,0,24,0,24,0,24,0,0,0,
  96,0,0,0,144,0,0,144,0,0,0,0,144,0,0,0,48,432,0,24,
  24,0,0,0,24,96,24,0,0,0,96,0,0,0,144,288,0,144,144,
  0,0,0,0,0,0,0,432,0,48,48,0,48,0,0,192,0,0,0,48,48,
  0,432,0,0,288,144,0,144,0,0,0,0,0,0,0,0,108,0,0,108,
  0,36,0,108,0,0,216,0,0,0,0,72,0,0,0,0,0,0,0,72,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,12,0,12,0,12,24,12,0,0,24,24,0,108,0,72,
  72,72,72,0,0,0,0,0,0,0,0,0,12,0,12,0,12,0,0,24,24,
  0,0,0,24,24,0,72,0,0,0,0,24,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,48,12,0,0,
  24,24,0,0,0,72,72,0,72,72,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,48,12,0,0,0,0,
  0,108,0,24,24,72,72,72,0,0,0,24,0,0,0,0,0,0,0,0,0,
  432,0,144,144,144,0,0,0,0,0,0,0,0,0,0,0,0,0,0,108,
  0,72,72,72,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,24,0,0,0,0,144,0,0,144,0,0,0,144,0,0,0,0,432,0,
  288,0,144,0,144,0,0,0,144,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,72,72,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,24,0,0,0,0,0,0,24,0,0,0,0,0,0,0,0,
  0,24,0,0,0,24,24,0,0,0,0,0,0,0,24,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,80,20,0,20,0,0,20,0,20,0,0,12,24,
  12,0,48,24,0,12,24,0,12,0,0,0,0,0,0,12,0,0,0,12,0,
  24,0,0,0,48,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,20,
  20,20,20,20,20,20,20,20,20,12,12,12,12,12,12,0,0,12,
  12,12,24,12,0,24,0,0,0,12,12,0,12,0,12,0,12,48,0,0,
  24,0,0,0,0,0,0,0,0,24,24,24,0,0,0,0,8,160,0,40,0,
  40,0,0,40,80,40,48,0,24,0,24,0,48,0,0,24,0,48,24,0,
  48,0,0,0,0,24,0,0,0,24,0,0,48,0,0,48,0,0,0,24,24,
  0,0,0,0,0,0,24,0,0,0,0,0,40,0,40,0,0,40,0,40,0,0,
  24,0,24,0,48,0,0,24,0,0,24,0,48,48,0,0,48,24,0,0,0,
  24,0,0,0,0,0,0,0,0,0,0,0,0,0,0,24,24,24,0,0,0,0,
  20,20,20,20,20,20,20,20,20,60,60,60,60,60,60,240,0,60,
  60,60,240,60,0,480,0,0,0,60,60,0,60,0,60,0,60,240,0,
  0,0,0,1620,0,0,0,2160,0,0,0,0,0,4320,0,2160,0,20,20,
  20,20,20,20,20,20,60,60,60,60,60,60,0,480,60,60,60,
  480,60,0,0,240,0,0,60,60,0,60,0,60,0,60,480,0,0,0,
  1620,0,1620,0,0,0,0,2160,0,0,0,0,2160,0,0,20,20,20,
  20,20,20,20,60,60,60,60,60,60,0,0,60,60,60,240,60,0,
  240,240,60,60,60,60,0,60,60,60,0,60,240,0,0,0,60,60,
  60,120,120,60,60,60,120,120,120,120,60,60,60,20,20,20,
  20,20,20,60,60,60,60,60,60,240,0,60,60,60,0,60,0,0,
  240,0,0,60,60,0,60,0,60,0,60,0,0,0,480,0,0,0,4320,0,
  0,2160,0,0,0,0,0,0,0,2160,20,20,20,20,20,60,60,60,60,
  60,60,240,240,60,60,60,240,60,0,0,240,60,60,60,60,0,
  60,60,60,0,60,240,0,0,240,60,60,60,120,120,60,60,60,
  120,120,120,120,60,60,60,20,20,20,20,60,60,60,60,60,60,
  240,0,60,60,60,240,60,0,0,0,0,0,60,60,0,60,0,60,0,
  60,480,0,0,240,0,0,0,0,0,0,0,2160,0,0,720,0,2160,0,
  0,20,20,20,60,60,60,60,60,60,240,0,60,60,60,0,60,0,
  480,480,0,0,60,60,0,60,0,60,0,60,0,0,0,240,0,1620,
  1620,4320,4320,2160,0,0,0,0,0,0,0,2160,0,20,20,60,60,
  60,60,60,60,240,240,60,60,60,0,60,0,240,240,60,60,60,
  60,0,60,60,60,0,60,0,0,0,240,60,60,60,120,120,60,60,
  60,120,120,120,120,60,60,60,20,60,60,60,60,60,60,0,0,
  60,60,60,0,60,0,240,240,1620,0,60,60,0,60,1620,60,0,
  60,240,0,0,480,0,0,0,4320,0,0,2160,0,0,0,0,0,0,0,
  2160,60,60,60,60,60,60,0,0,60,60,60,0,60,0,240,240,60,
  60,60,60,0,60,60,60,0,60,240,0,0,240,60,60,60,120,120,
  60,60,60,120,120,120,120,60,60,60,12,12,12,12,12,24,0,
  12,12,12,24,12,0,24,0,0,0,12,12,0,12,108,12,0,12,24,
  0,0,0,36,36,0,0,0,36,0,36,72,0,0,72,36,36,0,12,12,
  12,12,24,24,12,12,12,24,12,0,0,24,108,0,12,12,0,12,0,
  12,0,12,24,0,0,24,36,0,36,0,72,0,36,36,0,0,72,0,36,
  0,36,12,12,12,0,24,12,12,12,24,12,0,24,24,36,36,12,
  12,0,12,36,12,0,12,24,0,0,0,12,12,12,72,72,12,12,12,
  72,72,72,72,12,12,12,12,12,0,24,12,12,12,24,12,0,0,
  24,0,108,12,12,0,12,108,12,0,12,24,0,0,24,36,0,36,72,
  0,0,36,36,0,0,72,0,36,0,36,12,24,0,12,12,12,24,12,0,
  24,24,12,12,12,12,0,12,12,12,0,12,24,0,0,24,36,36,36,
  24,24,36,36,36,24,24,24,24,36,36,36,24,0,12,12,12,24,
  12,0,24,0,0,0,12,12,0,12,0,12,0,12,24,0,0,0,36,36,
  0,0,0,36,0,36,0,72,0,72,36,36,0,24,24,24,24,24,24,0,
  24,48,0,0,24,24,0,0,432,24,0,24,48,0,0,0,72,0,72,0,
  0,0,72,72,144,144,144,0,72,0,72,24,24,24,24,24,0,48,
  0,0,0,24,0,0,24,0,0,0,0,48,0,0,0,72,72,0,0,0,72,0,
  72,0,0,0,0,72,72,0,12,12,0,12,0,24,24,0,0,12,12,0,
  12,0,12,0,12,0,0,0,24,0,36,36,0,0,36,36,0,0,72,0,
  72,0,36,36,12,24,12,0,0,24,36,36,12,12,0,12,36,12,0,
  12,0,0,0,24,12,12,12,72,72,12,12,12,72,72,72,72,12,
  12,12,24,12,0,24,24,0,0,12,12,0,12,108,12,0,12,0,0,
  0,24,0,36,36,0,0,36,36,0,72,0,0,72,0,36,36,0,0,48,
  0,0,432,0,24,0,24,0,24,0,24,24,0,0,48,0,72,72,144,
  144,72,72,0,0,0,0,144,0,72,72,0,24,0,36,36,12,12,0,
  12,36,12,0,12,24,0,0,24,12,12,12,72,72,12,12,12,72,
  72,72,72,12,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,24,432,0,24,24,0,0,
  432,24,0,24,0,0,0,48,72,0,72,144,144,0,72,72,0,0,432,
  144,72,0,72,0,0,24,24,0,24,0,24,0,0,48,0,0,24,72,72,
  0,0,0,72,0,72,144,144,144,432,72,72,0,12,0,12,0,108,
  12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,24,0,24,0,0,
  0,108,12,0,0,12,12,0,108,0,0,0,0,0,0,0,0,0,0,0,0,
  24,0,0,24,0,0,0,12,0,12,0,12,0,12,0,0,0,24,0,36,36,
  0,72,36,36,0,0,72,0,0,0,36,36,0,12,12,12,0,12,24,0,
  0,24,36,36,36,24,24,36,36,36,24,24,24,24,36,36,36,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  12,0,12,24,0,0,24,36,0,36,72,0,0,36,36,72,0,0,0,36,
  0,36,12,0,0,0,0,0,0,0,0,0,0,24,0,0,0,0,24,0,0,0,
  0,0,0,12,24,0,0,24,36,36,36,24,24,36,36,36,24,24,24,
  24,36,36,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,24,0,0,0,36,36,0,0,72,36,0,36,0,72,0,0,36,36,0,
  0,0,24,0,72,72,144,144,72,72,0,0,432,0,144,0,72,72,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,72,72,0,144,144,72,0,72,432,0,0,144,
  72,72,0,12,12,0,0,12,12,0,0,0,0,0,0,12,12,12,0,0,0,
  12,12,0,0,0,0,12,0,12,0,0,12,0,12,0,0,0,0,12,12,0,
  24,0,0,0,24,24,24,24,0,0,0,0,0,0,24,24,24,24,0,0,0,
  12,12,0,0,0,0,12,12,12,12,0,0,0,0,12,12,12,0,0,0,0,
  12,12,12,24,24,24,0,0,0,24,24,0,0,0,24,0,0,0,0,0,0,
  12,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,80,80,0,0,0,0,80,1920,0,960,
  0,0,0,0,0,480,1920,960,7680,480,1920,0,0,0,0,0,480,
  960,960,0,1920,480,960,0,1920,0,0,0,0,0,0,0,0,720,720,
  0,0,0,0,0,0,0,0,20,0,0,20,0,20,240,0,60,0,60,0,0,
  0,240,60,960,960,60,240,0,3840,0,0,0,60,240,0,0,60,
  240,0,0,240,0,0,0,0,0,0,0,0,540,0,0,0,0,0,0,0,0,
  80,0,0,0,0,80,960,0,0,480,480,0,7680,480,0,1920,960,0,
  0,0,0,7680,0,0,0,0,480,960,0,1920,0,0,0,0,1920,0,0,
  0,0,0,0,720,0,0,0,0,0,0,0,0,0,0,0,20,0,20,0,0,60,
  480,60,0,0,480,480,60,0,0,60,0,0,0,0,0,0,60,240,0,0,
  60,480,480,0,240,0,1920,0,0,0,0,0,540,0,540,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,20,480,0,60,240,60,0,960,240,0,60,0,0,60,
  480,0,3840,0,0,0,60,240,960,0,60,0,0,0,0,0,3840,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,240,0,60,0,60,0,0,0,0,60,0,
  0,60,240,0,0,0,0,0,60,240,0,0,60,0,480,0,0,0,7680,0,
  0,0,0,0,60,60,60,0,0,0,0,0,0,0,0,0,0,0,0,0,0,48,
  24,24,96,24,24,0,0,0,0,0,24,24,48,0,24,48,24,0,0,48,
  48,0,0,0,0,0,72,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,12,0,0,0,24,12,24,48,12,
  0,0,0,0,0,0,12,24,24,0,12,24,24,0,24,24,0,0,0,0,0,
  0,12,12,12,0,0,0,0,0,0,0,0,0,0,24,24,24,48,48,24,0,
  0,0,0,0,0,24,48,24,0,24,24,48,0,0,24,0,0,0,0,0,0,
  0,72,72,0,0,0,0,0,0,0,0,0,0,24,12,24,48,12,0,0,0,
  0,0,0,12,24,24,0,12,24,24,0,24,24,48,0,0,0,0,0,0,0,
  36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,48,48,96,48,48,0,0,0,0,0,0,48,96,48,0,48,48,96,0,
  0,48,0,0,0,0,0,0,0,144,144,0,0,0,0,0,0,0,24,24,48,
  48,24,0,0,0,0,0,0,24,48,24,0,24,24,48,0,0,0,0,0,0,
  0,0,0,0,0,72,0,0,0,0,0,0,0,24,24,0,0,0,0,0,0,0,0,
  0,24,48,0,24,24,0,0,0,48,0,0,0,0,0,0,72,72,0,0,0,
  0,0,0,0,0,0,48,12,24,0,48,0,0,0,12,24,0,0,12,24,24,
  0,24,0,48,0,0,0,0,0,12,12,12,0,0,0,0,0,0,0,48,24,
  24,0,48,0,0,0,24,24,0,0,0,24,48,0,48,0,96,0,0,0,0,
  0,72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,48,96,0,
  48,0,0,0,0,96,0,0,0,0,0,0,144,144,0,0,0,0,0,0,0,0,
  24,0,0,0,0,0,12,24,24,0,12,0,0,0,24,24,0,0,0,0,0,
  0,12,12,12,0,0,0,0,0,0,0,0,0,0,0,0,24,24,48,0,24,
  0,24,0,0,48,48,0,0,0,0,0,72,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,48,48,96,0,48,0,48,0,0,96,0,0,0,0,
  0,0,144,0,144,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  24,24,0,12,0,0,0,0,24,0,0,0,0,0,0,36,0,0,0,0,0,0,
  0,0,0,0,0,0,48,48,0,48,0,96,0,0,0,0,0,0,72,72,0,0,
  0,0,0,0,0,0,0,48,24,0,24,24,48,0,0,0,0,0,0,72,72,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,24,24,0,24,0,48,0,0,0,0,0,36,36,36,0,0,
  0,0,0,0,0,0,0,0,48,0,0,0,0,0,0,72,0,0,0,0,0,0,0,
  0,0,0,24,24,0,0,0,0,0,0,72,0,72,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,24,0,0,0,0,0,
  0,72,72,0,0,0,0,0,0,0,0,48,0,0,0,0,0,0,72,0,0,0,
  0,0,0,0,0,0,0,0,0,0,144,0,144,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,12,12,0,0,0,0,0,0,0,12,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,160,0,0,80,0,160,0,160,80,40,48,
  0,96,48,24,0,48,96,48,0,48,96,24,96,0,0,0,0,0,24,0,
  48,0,24,96,48,0,96,0,96,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,20,20,20,20,20,20,20,20,20,60,60,
  60,60,60,60,480,0,60,60,60,480,60,0,240,0,0,0,60,60,
  240,60,0,60,480,60,0,480,0,0,0,0,0,0,0,720,0,4320,
  4320,720,4320,0,0,0,0,20,20,20,20,20,20,20,20,60,60,
  60,60,60,60,0,480,60,60,60,240,60,0,0,0,0,4860,60,60,
  480,60,0,60,0,60,480,0,0,0,0,0,14580,0,0,0,720,720,
  720,0,4320,0,0,0,0,20,20,20,20,20,20,20,60,60,60,60,
  60,60,0,0,60,60,60,240,60,240,0,480,0,0,60,60,960,60,
  0,60,480,60,480,0,0,0,0,0,0,0,0,720,4320,4320,0,4320,
  0,0,0,0,0,20,20,20,20,20,20,60,60,60,60,60,60,0,480,
  60,60,60,0,60,0,0,480,0,0,60,60,0,60,14580,60,0,60,
  480,0,240,0,4860,0,0,0,0,720,4320,0,0,720,720,0,0,0,
  0,20,20,20,20,20,60,60,60,60,60,60,480,0,60,60,60,0,
  60,0,0,0,0,0,60,60,0,60,0,60,240,60,0,480,0,240,0,0,
  0,0,0,4320,0,720,4320,720,0,0,0,0,0,20,20,20,20,60,
  60,60,60,60,60,120,120,60,60,60,120,60,120,120,120,0,0,
  60,60,120,60,0,60,120,60,120,120,120,120,0,0,1620,0,0,
  0,4320,4320,4320,0,720,0,0,0,0,20,20,20,60,60,60,60,
  60,60,480,0,60,60,60,0,60,240,0,240,0,0,60,60,0,60,0,
  60,0,60,0,480,480,480,0,0,0,0,0,4320,4320,720,720,4320,
  0,0,0,0,0,20,20,60,60,60,60,60,60,0,480,60,60,60,0,
  60,0,240,960,0,0,60,60,480,60,0,60,0,60,0,0,240,480,
  0,0,0,0,0,0,0,4320,720,4320,4320,0,0,0,0,20,60,60,60,
  60,60,60,120,120,60,60,60,120,60,120,120,120,0,0,60,60,
  120,60,1620,60,120,60,120,120,120,120,0,0,0,0,0,4320,
  720,0,0,4320,4320,0,0,0,0,60,60,60,60,60,60,0,0,60,
  60,60,0,60,480,480,960,0,180,60,60,960,60,180,60,480,
  60,0,0,0,480,180,0,180,0,0,120,120,120,120,120,120,0,
  0,0,0,12,12,12,12,12,24,0,12,12,12,0,12,0,24,0,0,0,
  12,12,24,12,108,12,24,12,24,24,24,24,108,0,0,0,0,72,
  0,72,72,0,72,0,0,0,0,12,12,12,12,24,24,12,12,12,24,
  12,24,0,24,0,0,12,12,0,12,108,12,24,12,0,24,0,24,108,
  0,108,0,0,0,72,72,72,0,72,0,0,0,0,12,12,12,0,24,12,
  12,12,24,12,24,24,24,0,108,12,12,24,12,0,12,48,12,24,
  0,24,0,36,0,0,0,0,24,24,24,0,72,0,0,0,0,0,12,12,0,
  24,12,12,12,24,12,24,24,24,0,108,12,12,24,12,108,12,0,
  12,24,0,24,0,108,0,108,0,0,0,72,72,0,72,72,0,0,0,0,
  12,24,0,12,12,12,48,12,0,24,0,0,0,12,12,24,12,36,12,
  24,12,24,24,24,24,0,0,0,0,0,72,0,0,24,24,24,0,0,0,
  0,24,24,12,12,12,24,12,24,24,24,0,108,12,12,24,12,108,
  12,24,12,24,24,24,24,108,0,0,0,0,72,0,72,0,72,72,0,
  0,0,0,24,24,24,24,24,24,48,24,48,0,216,24,0,24,24,0,
  24,24,48,48,0,48,0,216,0,432,0,0,0,144,144,0,144,864,
  0,0,0,0,0,24,24,24,0,0,48,0,0,0,24,24,48,24,216,24,
  24,24,24,48,24,48,0,0,0,0,0,144,0,0,144,0,144,0,0,0,
  0,12,12,0,12,0,24,0,0,108,12,12,24,12,108,12,24,12,
  24,24,24,24,0,0,108,0,0,72,72,0,0,72,72,0,0,0,0,12,
  24,12,24,0,24,0,0,12,12,0,12,0,12,24,12,0,24,48,24,
  0,0,36,0,0,24,24,24,72,0,0,0,0,0,0,24,12,24,24,24,
  0,0,12,12,24,12,108,12,24,12,24,24,24,24,0,0,108,0,0,
  72,72,0,72,0,72,0,0,0,0,0,0,48,0,0,0,24,24,48,24,
  216,24,24,24,24,48,24,48,0,0,0,0,0,144,144,0,0,0,144,
  0,0,0,0,0,24,0,0,108,12,12,24,12,108,12,24,12,24,24,
  24,24,36,0,36,0,0,24,24,24,72,72,72,0,0,0,0,24,24,0,
  0,24,24,24,24,216,24,48,24,48,24,48,24,0,0,0,0,0,144,
  0,0,0,144,144,0,0,0,0,24,0,0,0,24,24,24,0,0,24,24,
  0,48,0,48,0,0,216,0,0,0,144,144,144,0,0,0,0,0,0,0,
  0,24,24,24,24,216,24,48,24,48,24,48,24,432,0,0,0,0,
  144,0,144,864,0,144,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,108,36,432,108,
  36,36,216,108,0,216,0,216,0,0,0,0,0,0,0,0,72,0,72,0,
  0,0,0,12,0,12,0,12,24,12,0,24,0,24,0,0,108,0,0,72,
  72,0,72,72,0,0,0,0,0,24,12,0,12,0,12,24,0,24,48,108,
  0,0,0,0,0,0,72,24,24,24,0,0,0,0,24,0,0,24,24,0,48,
  0,48,0,0,216,0,0,864,144,0,144,144,0,0,0,0,0,0,12,
  24,12,24,24,24,24,108,0,108,0,0,0,72,72,72,72,0,0,0,
  0,0,36,0,0,0,432,0,0,0,0,0,0,0,0,0,0,72,72,0,0,0,
  0,0,24,12,0,24,0,24,108,0,108,0,0,72,72,72,24,24,24,
  0,0,0,0,0,48,0,48,0,216,0,0,0,0,0,0,144,144,144,0,
  0,0,0,0,24,48,24,0,108,0,0,0,0,72,0,72,72,72,0,0,0,
  0,0,24,24,24,0,0,216,0,0,144,144,0,144,0,0,0,0,0,0,
  24,24,216,0,0,0,0,0,864,144,0,144,144,0,0,0,0,24,0,
  0,216,0,0,0,144,0,144,0,144,0,0,0,0,216,0,0,0,0,144,
  0,144,0,144,0,0,0,0,0,0,36,0,0,72,72,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,0,72,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,24,24,0,0,0,0,0,0,0,24,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,24,24,0,0,0,0,24,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,20,80,20,0,0,20,0,20,0,0,12,24,12,0,0,0,0,12,
  24,0,12,0,0,0,0,0,0,12,0,0,0,12,48,24,24,48,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,80,80,80,
  0,0,0,1920,960,0,0,0,960,480,480,0,0,0,480,0,0,0,0,
  0,0,3840,0,480,0,0,960,30720,0,0,0,0,0,0,720,0,0,0,
  0,0,0,2160,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,80,20,80,0,20,40,20,0,0,
  60,240,60,0,0,0,480,60,0,0,60,0,240,480,0,0,0,60,0,
  0,0,60,3840,0,240,960,0,0,0,0,0,0,540,0,0,0,0,0,0,
  540,0,0,0,0,80,0,0,0,0,480,0,480,480,0,0,480,3840,
  480,240,480,0,0,0,480,480,0,0,0,0,0,480,0,0,0,0,480,
  0,0,0,0,0,0,2160,360,0,0,0,0,0,0,0,0,0,0,0,0,20,0,
  20,240,0,60,0,60,0,240,960,240,60,240,240,60,0,0,240,
  0,0,0,60,0,0,0,60,960,0,240,0,0,240,0,0,0,0,180,0,
  0,0,0,0,0,0,0,0,0,0,80,0,0,480,0,1920,480,480,0,960,
  960,960,0,480,0,480,960,1920,0,0,0,0,480,0,1920,0,0,0,
  1920,1920,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2160,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  20,240,0,60,240,60,0,480,0,0,60,0,0,60,0,240,0,0,0,
  0,60,0,240,0,60,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,80,480,0,240,480,480,0,480,960,480,0,1920,
  480,480,0,0,960,0,0,0,480,0,1920,0,0,480,480,0,0,0,
  1920,0,0,0,360,2160,0,0,0,0,0,0,0,0,0,0,0,0,60,0,
  60,0,0,0,0,60,0,0,60,0,240,0,0,0,0,60,0,480,0,60,0,
  0,240,0,0,240,0,0,0,60,60,0,0,0,0,0,0,60,0,0,0,0,
  0,0,0,0,24,0,48,24,24,24,24,0,24,0,0,0,0,24,0,48,0,
  24,48,24,48,0,0,0,0,0,0,0,72,0,0,0,0,0,0,72,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,0,0,
  0,24,12,24,24,12,48,24,24,0,0,0,12,0,24,0,12,48,24,
  24,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,0,0,0,0,0,0,
  24,24,48,48,24,48,48,24,0,0,0,24,0,24,0,24,96,48,24,
  0,0,24,0,0,0,72,0,0,0,0,0,0,0,72,0,0,0,0,0,0,24,
  12,24,24,12,0,24,24,0,0,0,12,0,24,0,12,48,24,24,0,0,
  24,0,0,0,12,12,0,0,0,0,0,0,12,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,48,24,24,24,24,0,48,0,0,0,0,
  24,0,48,0,24,48,24,48,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  72,0,0,0,96,48,48,48,48,0,48,0,0,0,0,48,0,96,0,48,
  48,48,96,0,0,0,0,0,0,0,144,0,0,0,0,0,0,144,0,0,0,
  24,24,0,0,0,24,24,0,0,0,0,0,48,0,24,0,0,48,48,0,0,
  0,0,0,72,0,0,0,0,0,0,0,72,0,0,0,0,24,12,48,0,24,0,
  0,0,12,0,0,0,12,48,24,0,48,0,24,0,0,0,0,36,0,0,0,
  0,0,0,0,0,0,0,48,24,48,24,24,0,0,0,24,0,0,0,0,96,
  48,0,96,0,48,0,0,0,0,72,0,0,0,0,0,0,72,0,0,0,0,0,
  48,0,0,0,0,24,0,48,0,24,0,0,24,0,0,48,0,0,0,72,0,
  0,0,0,0,0,0,0,0,0,0,0,24,0,0,0,0,12,0,24,0,12,0,
  0,24,0,0,24,0,0,0,36,36,0,0,0,0,0,0,36,0,0,0,48,0,
  0,0,0,0,0,96,0,48,0,0,96,0,0,0,0,0,0,144,0,0,0,0,
  0,0,0,144,0,0,0,24,0,0,0,24,0,0,0,24,96,48,0,96,0,
  48,0,0,0,72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,48,
  0,24,0,0,48,48,0,0,0,0,0,0,0,0,0,0,0,0,0,72,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,24,0,12,0,0,24,0,0,24,0,0,
  0,12,12,0,0,0,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,48,24,24,48,0,24,
  0,0,0,72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,48,24,24,48,0,24,0,0,
  0,12,12,0,0,0,0,0,0,12,0,0,0,0,48,0,0,0,0,0,0,144,
  144,0,0,0,0,0,0,0,0,0,0,24,0,0,0,0,0,0,72,72,0,0,
  0,0,0,0,0,0,0,0,48,0,24,0,0,0,0,72,0,0,0,0,0,0,
  72,0,0,0,0,0,0,0,0,144,0,0,0,0,0,0,0,144,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,0,0,0,
  0,0,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  12,0,0,0,0,0,0,12,0,0,0,0,0,0,0,0,0,12,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,80,80,0,0,0,0,80,0,0,0,480,1920,960,0,0,0,480,
  960,1920,1920,0,0,480,0,0,960,0,7680,0,0,0,0,0,480,
  7680,0,0,0,0,0,0,0,0,0,0,720,0,0,0,0,0,0,0,20,0,0,
  20,0,20,0,0,60,240,60,960,0,0,0,60,0,0,60,0,480,240,
  0,0,480,60,3840,0,0,60,3840,0,240,960,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,80,0,0,0,0,80,0,480,480,0,1920,
  960,1920,0,0,0,960,0,1920,0,1920,960,0,0,1920,960,0,0,
  0,480,0,960,0,0,7680,480,0,0,0,0,0,0,0,0,0,720,720,
  0,0,0,0,0,0,20,0,20,0,480,60,480,60,0,240,0,0,60,0,
  0,60,0,0,240,0,0,0,60,0,0,0,60,1920,480,480,0,0,480,
  0,0,0,0,0,0,0,0,540,540,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,20,0,240,
  60,0,60,0,240,0,0,60,960,0,60,0,240,240,0,0,240,60,
  3840,0,0,60,0,0,0,0,960,240,0,0,0,0,0,0,0,0,0,0,
  540,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,60,0,60,0,0,0,0,60,0,0,60,0,240,240,0,
  0,240,60,0,0,0,60,7680,480,0,0,0,0,0,0,0,0,0,0,0,0,
  60,60,60,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,24,24,48,0,0,0,24,24,48,24,0,0,24,0,0,48,
  24,0,0,0,0,0,0,24,48,0,24,0,0,0,0,0,0,0,0,72,0,72,
  0,0,0,0,24,12,24,0,0,0,12,24,24,12,0,24,24,0,0,24,
  12,48,0,0,12,0,0,24,48,0,0,0,0,0,0,0,0,0,0,36,0,0,
  0,0,0,0,24,24,0,0,0,0,48,24,24,0,0,48,0,0,0,0,0,0,
  0,24,0,48,24,0,48,24,0,0,0,0,0,0,0,0,0,72,72,0,0,
  0,0,0,24,0,0,12,0,0,12,0,24,24,0,0,24,12,48,0,0,12,
  48,24,24,48,48,24,0,0,0,0,0,0,0,0,12,12,12,0,0,0,0,
  24,0,0,24,0,24,0,0,48,0,0,0,48,24,96,0,0,24,48,24,
  24,48,96,48,0,0,0,0,0,0,0,0,0,72,72,0,0,0,0,0,0,
  24,48,24,24,0,0,48,0,0,0,24,0,0,0,24,0,24,0,0,0,0,
  0,0,0,0,0,0,0,0,72,0,72,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,24,24,12,0,0,
  24,0,0,0,12,0,0,0,12,48,24,0,0,48,24,0,0,0,0,0,0,
  0,0,0,36,0,0,0,0,0,0,0,0,24,24,0,0,24,24,48,0,0,
  24,96,48,48,96,48,24,0,0,0,0,0,0,0,0,72,0,72,0,0,0,
  0,0,0,48,0,0,0,48,24,96,0,0,24,48,24,0,48,96,48,0,
  0,0,0,0,0,0,0,0,0,72,0,0,0,0,0,24,0,0,0,24,12,48,
  0,0,12,48,24,24,48,48,24,0,0,0,0,0,0,0,0,36,36,36,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,24,0,0,24,0,0,0,0,24,48,24,
  0,0,0,0,0,0,0,0,0,0,0,0,72,0,0,0,0,0,0,0,0,24,24,
  48,0,0,24,96,48,48,96,48,48,0,0,0,0,0,0,0,0,0,72,
  72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,24,48,24,0,0,
  96,48,0,0,0,0,0,0,0,0,72,72,0,0,0,0,0,0,0,0,12,0,
  24,0,0,48,24,0,0,0,0,0,0,0,0,12,12,12,0,0,0,0,0,0,
  0,0,48,0,0,0,0,0,0,0,0,0,0,0,0,144,144,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  24,48,0,0,0,0,0,0,0,0,0,0,12,12,12,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,144,144,0,0,0,0,0,48,96,0,0,0,
  0,0,0,0,0,0,0,72,72,0,0,0,0,0,0,48,24,0,0,0,0,0,
  0,0,0,0,72,0,0,0,0,0,48,48,0,0,0,0,0,0,0,0,0,144,
  144,0,0,0,0,0,0,0,0,0,0,0,0,0,144,0,144,0,0,0,0,0,
  0,0,0,0,0,0,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,12,12,0,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,5,5,5,5,5,5,5,5,30,30,30,30,30,30,0,0,30,30,30,
  120,30,240,0,0,270,0,30,30,0,30,0,30,0,30,0,120,0,0,
  0,0,0,0,0,0,0,2160,2160,0,0,2160,0,2160,0,5,5,5,5,5,
  5,5,30,30,30,30,30,30,0,0,30,30,30,120,30,240,0,0,
  270,0,30,30,0,30,0,30,0,30,0,120,0,0,0,0,0,0,0,0,0,
  2160,2160,0,0,2160,0,2160,0,5,5,5,5,5,5,30,30,30,30,
  30,30,60,0,30,30,30,0,30,0,60,0,0,0,30,30,120,30,0,
  30,120,30,120,0,60,0,0,0,0,0,0,0,0,0,0,0,77760,0,
  77760,0,0,5,5,5,5,5,30,30,30,30,30,30,60,0,30,30,30,
  60,30,0,0,0,0,0,30,30,60,30,0,30,60,30,60,60,60,0,0,
  270,0,0,0,0,0,2160,2160,0,2160,2160,2160,2160,0,5,5,5,
  5,30,30,30,30,30,30,60,0,30,30,30,60,30,0,0,0,0,0,
  30,30,60,30,0,30,60,30,60,60,60,0,0,270,0,0,0,0,0,
  2160,2160,0,2160,2160,2160,2160,0,5,5,5,30,30,30,30,30,
  30,120,0,30,30,30,120,30,120,120,0,0,0,30,30,0,30,0,
  30,0,30,0,120,120,0,0,0,0,0,77760,77760,0,77760,77760,
  77760,0,77760,0,77760,77760,5,5,30,30,30,30,30,30,120,0,
  30,30,30,0,30,120,60,0,270,0,30,30,120,30,270,30,0,30,
  0,0,120,0,0,90,0,0,360,360,0,2160,2160,360,2160,2160,
  2160,2160,360,5,30,30,30,30,30,30,0,0,30,30,30,0,30,
  60,60,0,0,0,30,30,60,30,270,30,60,30,60,0,0,0,0,0,0,
  0,2160,2160,0,0,0,2160,2160,0,2160,0,2160,30,30,30,30,
  30,30,0,0,30,30,30,0,30,60,60,0,0,0,30,30,60,30,270,
  30,60,30,60,0,0,0,0,0,0,0,2160,2160,0,0,0,2160,2160,
  0,2160,0,2160,30,30,30,30,30,240,0,30,30,30,120,30,120,
  120,0,810,0,30,30,120,30,810,30,120,30,120,120,0,0,0,
  270,0,0,1080,1080,0,6480,6480,0,6480,6480,6480,6480,0,30,
  30,30,30,0,0,30,30,30,120,30,120,0,0,810,0,30,30,120,
  30,810,30,120,30,120,120,0,0,0,0,0,0,1080,0,0,6480,
  6480,0,6480,6480,6480,0,1080,30,30,30,0,0,30,30,30,120,
  30,60,0,0,810,0,30,30,60,30,810,30,120,30,120,120,0,0,
  0,0,0,0,6480,0,0,2160,6480,0,6480,6480,2160,2160,2160,
  30,30,0,0,30,30,30,120,30,60,60,0,0,0,30,30,60,30,
  810,30,120,30,120,0,120,0,0,0,0,0,0,0,0,6480,0,1080,
  6480,6480,6480,0,1080,30,0,0,30,30,30,120,30,0,240,0,0,
  0,30,30,120,30,270,30,120,30,120,120,120,0,0,0,0,0,0,
  0,0,6480,2160,2160,2160,2160,6480,0,0,120,0,30,30,30,
  120,30,0,60,0,0,0,30,30,60,30,810,30,120,30,120,0,60,
  0,0,270,0,0,0,1080,0,6480,0,1080,6480,6480,6480,6480,0,
  0,120,120,60,120,120,240,120,0,0,0,60,120,60,60,3240,
  60,60,120,240,0,0,0,0,0,0,0,0,0,0,25920,0,0,12960,
  25920,12960,0,2160,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,30,30,
  0,30,0,240,0,0,0,30,30,120,30,810,30,120,30,0,0,60,0,
  0,270,0,0,0,1080,0,0,0,1080,6480,6480,0,6480,1080,30,
  120,30,240,0,0,0,0,30,30,120,30,810,30,120,30,0,120,
  120,0,0,0,0,0,0,2160,0,2160,0,0,6480,6480,0,0,360,0,
  30,120,0,0,810,0,30,30,120,30,810,30,120,30,60,120,60,
  0,0,270,0,0,1080,1080,0,0,6480,0,6480,6480,0,6480,1080,
  0,0,960,0,0,0,0,120,0,120,6480,120,0,240,60,480,60,0,
  0,1080,0,0,4320,2160,0,0,0,2160,25920,12960,0,12960,2160,
  120,120,0,810,0,30,30,120,30,0,30,120,30,0,120,60,0,0,
  90,0,0,6480,360,0,0,6480,1080,0,0,2160,2160,2160,60,0,
  0,0,120,60,60,60,3240,120,480,60,0,60,480,0,0,2160,0,
  0,0,4320,0,0,0,4320,25920,12960,0,0,0,0,0,0,120,60,60,
  60,3240,120,120,60,0,240,0,0,0,0,0,0,0,0,0,12960,
  25920,0,12960,12960,12960,0,4320,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,810,0,
  3240,810,0,270,0,810,6480,0,3240,0,0,0,0,0,0,0,0,0,0,
  1080,2160,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,30,0,30,0,30,0,30,0,120,120,
  0,0,270,0,0,1080,1080,0,0,6480,1080,0,0,0,6480,1080,60,
  30,0,30,0,30,120,0,120,0,0,270,0,0,2160,0,0,0,0,360,
  0,2160,0,6480,6480,60,0,0,60,0,0,480,0,0,0,1080,0,0,
  2160,6480,0,0,25920,2160,0,0,0,25920,4320,0,30,60,30,
  120,120,0,0,0,0,0,0,1080,0,0,6480,6480,1080,0,0,6480,
  0,1080,0,0,0,0,0,3240,0,0,0,0,0,1080,0,0,0,2160,1080,
  0,0,0,0,0,0,30,120,120,0,0,0,0,0,0,360,1080,0,6480,
  2160,2160,2160,0,0,0,1080,0,480,0,0,0,0,0,0,0,2160,
  2160,0,25920,12960,2160,0,0,12960,12960,4320,0,120,120,0,
  0,270,0,0,1080,1080,0,0,6480,1080,0,0,6480,6480,0,0,60,
  0,0,1080,0,0,0,2160,0,0,25920,2160,0,25920,0,12960,2160,
  120,0,0,0,0,0,0,4320,0,12960,0,2160,12960,12960,25920,0,
  2160,0,0,0,0,0,2160,0,0,25920,25920,0,0,0,12960,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,2160,0,0,0,0,2160,0,360,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2160,
  360,2160,2160,0,0,0,0,2160,0,0,0,0,2160,2160,360,0,0,
  0,0,0,0,0,0,0,0,0,0,0,2160,2160,2160,0,2160,0,0,0,
  2160,2160,0,0,0,0,0,0,0,0,0,0,0,2160,2160,5,5,5,5,5,
  5,5,30,30,30,30,30,30,0,0,30,30,30,120,30,240,0,0,
  270,0,30,30,0,30,0,30,0,30,0,120,0,0,0,0,0,0,0,0,0,
  2160,2160,0,0,2160,0,2160,0,5,5,5,5,5,5,30,30,30,30,
  30,30,0,240,30,30,30,0,30,0,0,120,0,270,30,30,120,30,
  0,30,240,30,240,0,0,240,270,0,0,2160,0,0,2160,0,0,0,
  2160,0,2160,0,0,5,5,5,5,5,30,30,30,30,30,30,0,240,30,
  30,30,0,30,0,0,120,0,270,30,30,120,30,0,30,240,30,240,
  0,0,240,270,0,0,2160,0,0,2160,0,0,0,2160,0,2160,0,0,
  5,5,5,5,30,30,30,30,30,30,0,60,30,30,30,60,30,60,0,
  60,0,0,30,30,60,30,0,30,60,30,60,60,0,60,0,0,0,77760,
  0,0,77760,77760,77760,0,77760,77760,77760,77760,0,5,5,5,
  30,30,30,30,30,30,0,120,30,30,30,60,30,60,0,120,0,0,
  30,30,0,30,0,30,0,30,0,60,0,120,0,0,0,77760,0,0,
  77760,0,0,0,0,0,0,0,0,5,5,30,30,30,30,30,30,0,60,30,
  30,30,0,30,60,0,60,0,0,30,30,60,30,0,30,0,30,0,0,0,
  60,0,0,270,2160,0,0,2160,0,0,0,2160,0,2160,0,0,5,30,
  30,30,30,30,30,0,60,30,30,30,0,30,60,0,60,0,0,30,30,
  60,30,0,30,0,30,0,0,0,60,0,0,270,2160,0,0,2160,0,0,
  0,2160,0,2160,0,0,30,30,30,30,30,30,0,60,30,30,30,0,
  30,120,0,120,270,270,30,30,60,30,0,30,60,30,60,0,0,60,
  270,0,90,360,0,0,360,360,360,0,2160,360,2160,360,0,30,
  30,30,30,30,0,0,30,30,30,120,30,120,0,0,810,0,30,30,
  120,30,0,30,120,30,60,60,0,0,810,0,0,0,0,0,0,1080,
  1080,0,6480,1080,6480,1080,0,30,30,30,30,0,0,30,30,30,
  60,30,240,0,120,810,0,30,30,120,30,0,30,60,30,120,120,
  0,240,810,0,270,0,0,0,1080,1080,1080,0,6480,1080,6480,0,
  0,30,30,30,0,0,30,30,30,120,30,60,0,0,810,0,30,30,
  120,30,0,30,120,30,120,120,0,0,270,0,0,0,0,0,0,360,
  6480,0,6480,0,2160,0,0,30,30,0,240,30,30,30,120,30,0,
  0,60,0,810,30,30,120,30,0,30,120,30,120,0,0,120,810,0,
  270,1080,0,0,1080,1080,0,0,6480,1080,6480,0,0,30,0,0,
  30,30,30,120,30,0,0,120,0,0,30,30,120,30,0,30,120,30,
  60,0,0,60,810,0,270,0,0,0,6480,6480,0,0,2160,360,6480,
  0,0,0,120,30,30,30,120,30,60,0,60,0,810,30,30,120,30,
  0,30,120,30,60,60,0,60,810,0,0,1080,0,0,0,1080,0,0,
  6480,1080,6480,1080,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  120,60,60,60,120,0,0,0,3240,0,60,120,240,60,0,60,60,
  120,60,480,0,0,3240,0,0,0,0,0,0,4320,12960,0,12960,
  4320,12960,8640,0,30,30,0,30,0,0,120,0,810,30,30,120,
  30,0,30,120,30,60,60,0,60,0,0,270,1080,0,0,1080,0,0,
  0,6480,1080,0,1080,0,30,0,30,0,0,120,0,810,30,30,120,
  30,0,30,60,30,0,240,0,120,0,0,90,6480,0,0,360,0,0,0,
  6480,6480,2160,2160,0,60,30,120,0,120,810,0,30,30,120,
  30,0,30,60,30,0,60,0,60,0,0,270,0,0,0,1080,0,1080,0,
  6480,1080,0,1080,0,0,0,0,0,0,3240,60,240,240,60,0,60,
  60,0,120,480,0,480,0,0,2160,4320,0,0,4320,0,0,0,25920,
  0,0,4320,0,120,0,0,0,0,30,30,120,30,0,30,120,30,60,
  60,0,60,270,0,270,6480,0,0,360,360,6480,0,0,1080,0,360,
  0,0,60,0,3240,120,120,120,60,0,120,480,60,240,120,0,60,
  0,0,3240,12960,0,0,4320,0,0,0,25920,2160,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,3240,0,240,60,0,60,0,0,240,0,0,0,0,120,0,0,0,
  0,0,0,0,4320,4320,0,0,0,12960,4320,0,0,810,0,0,810,0,
  270,0,810,0,0,0,0,0,0,0,1080,0,0,0,0,0,0,2160,1080,
  0,0,0,810,270,0,810,0,0,3240,0,3240,3240,0,0,0,0,0,0,
  0,0,0,0,1080,0,0,1080,0,0,0,30,0,30,0,30,60,30,0,
  240,0,120,0,0,270,1080,0,0,1080,0,1080,0,0,0,0,1080,0,
  120,30,0,30,0,30,240,0,0,120,0,0,270,360,0,0,0,0,
  2160,0,2160,0,0,0,0,0,0,0,60,0,0,960,0,0,0,0,12960,
  2160,0,0,0,4320,2160,0,0,0,0,2160,0,0,30,0,30,120,0,
  0,0,810,0,270,1080,0,0,1080,1080,1080,0,0,0,6480,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,60,
  30,120,120,0,120,810,0,270,360,0,0,1080,6480,360,0,0,
  360,6480,1080,0,0,240,0,0,0,0,0,0,2160,0,0,0,4320,
  4320,0,0,0,0,0,0,120,120,0,0,810,0,0,1080,0,0,0,1080,
  1080,0,0,0,0,1080,0,60,0,60,0,0,1080,6480,0,0,2160,0,
  4320,0,0,0,0,4320,0,0,60,0,0,1080,4320,0,0,4320,0,0,
  0,12960,0,25920,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,3240,0,0,0,0,0,0,4320,4320,0,0,25920,12960,12960,0,0,
  270,0,0,0,1080,0,0,0,0,0,0,1080,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,360,0,0,0,0,2160,360,0,0,0,0,0,
  360,0,2160,360,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,360,0,0,0,0,2160,360,0,0,0,0,0,2160,360,0,
  0,2160,360,0,0,0,0,0,0,0,0,2160,0,0,0,0,0,0,2160,0,
  0,5,5,5,5,5,5,30,30,30,30,30,30,0,0,30,30,30,0,30,
  0,60,60,0,0,30,30,60,30,270,30,60,30,60,0,0,0,0,0,0,
  2160,0,0,2160,0,0,0,2160,0,2160,0,0,5,5,5,5,5,30,30,
  30,30,30,30,0,0,30,30,30,120,30,0,0,120,30,30,30,30,
  60,30,30,30,120,30,120,120,0,0,30,30,30,60,60,60,60,
  60,60,60,60,60,60,60,60,5,5,5,5,30,30,30,30,30,30,0,
  0,30,30,30,0,30,0,60,60,0,0,30,30,60,30,270,30,60,30,
  60,0,0,0,0,0,0,2160,0,0,2160,0,0,0,2160,0,2160,0,0,
  5,5,5,30,30,30,30,30,30,0,0,30,30,30,120,30,60,120,
  120,90,0,30,30,0,30,0,30,0,30,0,120,0,0,0,270,270,
  2160,2160,2160,2160,360,360,2160,0,360,0,360,2160,5,5,30,
  30,30,30,30,30,0,0,30,30,30,0,30,0,0,120,30,30,30,30,
  120,30,30,30,0,30,0,0,0,0,30,30,30,60,60,60,60,60,60,
  60,60,60,60,60,60,5,30,30,30,30,30,30,0,0,30,30,30,
  60,30,60,60,60,270,0,30,30,60,30,90,30,60,30,60,60,0,
  0,0,0,270,360,2160,2160,360,0,0,2160,2160,0,2160,0,2160,
  30,30,30,30,30,30,0,0,30,30,30,0,30,0,0,120,30,30,30,
  30,120,30,30,30,0,30,0,0,0,0,30,30,30,60,60,60,60,60,
  60,60,60,60,60,60,60,30,30,30,30,30,0,0,30,30,30,120,
  30,120,0,0,270,0,30,30,120,30,270,30,0,30,60,60,0,0,
  0,0,0,0,1080,0,0,1080,1080,0,1080,1080,1080,1080,0,30,
  30,30,30,0,0,30,30,30,60,30,120,0,0,270,0,30,30,120,
  30,270,30,60,30,120,120,0,0,270,0,0,0,0,0,0,1080,1080,
  0,1080,1080,1080,0,1080,30,30,30,0,0,30,30,30,240,30,
  60,0,0,90,90,30,30,60,30,90,30,120,30,240,120,0,0,30,
  30,30,180,180,60,60,60,180,180,180,180,60,60,60,30,30,
  0,0,30,30,30,120,30,120,120,120,0,270,30,30,60,30,0,
  30,120,30,120,0,0,0,270,0,270,0,0,0,1080,1080,0,1080,
  1080,1080,1080,0,0,30,0,0,30,30,30,120,30,0,120,120,30,
  30,30,30,240,30,30,30,240,30,60,0,0,0,90,90,90,60,60,
  180,180,180,60,60,60,60,180,180,180,0,0,30,30,30,120,
  30,60,120,60,0,270,30,30,120,30,270,30,120,30,120,0,0,
  0,270,270,0,1080,0,1080,0,0,0,1080,0,1080,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,30,30,0,30,0,120,120,0,0,30,30,0,30,270,30,120,30,
  60,60,0,0,0,270,270,1080,0,1080,0,0,0,0,1080,1080,0,
  1080,1080,30,0,30,120,0,120,90,90,30,30,120,30,90,30,
  60,30,0,120,0,0,30,30,30,180,180,60,60,60,180,180,180,
  180,60,60,60,0,30,0,120,0,0,0,30,30,120,30,0,30,120,
  30,0,60,0,0,0,270,0,0,1080,1080,1080,0,0,0,0,0,0,
  1080,1080,0,0,240,0,0,0,60,120,480,60,0,60,60,120,120,
  0,0,0,0,1080,0,720,6480,8640,4320,0,0,0,2160,720,0,
  4320,4320,120,120,0,90,90,30,30,120,30,90,30,120,30,0,
  60,0,0,30,30,30,180,180,60,60,60,180,180,180,180,60,60,
  60,120,60,0,0,120,60,60,60,1080,240,0,120,240,60,0,0,
  0,0,1080,4320,0,2160,2160,0,0,0,0,8640,6480,2160,720,0,
  2160,0,120,120,60,60,1080,120,120,120,0,0,0,0,0,0,0,0,
  2160,0,0,4320,2160,0,2160,720,0,0,4320,0,0,120,120,120,
  120,0,120,960,120,240,60,0,0,1080,0,0,0,2160,0,0,0,
  720,0,2160,2160,0,4320,0,30,270,30,2160,0,30,30,1080,
  270,0,0,0,0,0,0,0,120,0,0,0,0,0,0,120,120,0,0,0,
  270,30,0,0,30,30,0,0,0,2160,0,0,0,0,0,0,120,0,0,0,
  0,0,0,0,0,0,0,30,0,30,0,30,60,30,0,120,0,0,0,270,
  270,0,1080,0,1080,0,1080,1080,0,0,0,1080,0,60,30,30,30,
  0,30,120,0,0,0,90,90,90,60,60,180,180,180,60,60,60,60,
  180,180,180,60,0,0,0,120,0,480,0,0,0,0,2160,4320,8640,
  2160,720,2160,0,0,0,0,0,2160,2160,0,30,0,30,120,120,0,
  0,270,0,270,1080,1080,0,1080,0,1080,1080,0,0,1080,0,
  1080,30,0,0,0,1080,0,0,0,0,0,120,120,0,0,0,120,120,0,
  0,0,0,0,0,30,120,0,0,0,90,90,90,60,60,180,180,180,60,
  60,60,60,180,180,180,0,480,0,0,0,1080,1080,0,8640,4320,
  2160,0,2160,4320,0,0,0,720,2160,6480,240,120,0,0,270,
  270,0,1080,1080,1080,0,0,0,1080,0,0,0,1080,0,60,0,0,0,
  1080,0,2160,720,4320,8640,0,2160,2160,0,6480,0,4320,0,0,
  0,1080,0,1080,4320,0,6480,720,720,0,8640,4320,0,2160,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,30,30,0,0,120,120,0,0,0,0,0,0,0,
  120,30,0,0,0,0,120,0,0,0,0,0,0,120,0,0,0,0,0,0,0,
  0,0,0,120,0,60,0,0,0,60,60,60,60,0,0,0,0,0,0,60,60,
  60,60,0,0,0,60,60,0,0,0,0,60,60,60,60,0,0,0,0,60,
  60,60,0,0,0,0,60,60,60,60,60,60,0,0,0,60,60,0,0,0,
  60,0,0,0,0,0,0,60,60,60,5,5,5,5,5,30,30,30,30,30,
  30,0,240,30,30,30,0,30,0,0,120,0,270,30,30,120,30,0,
  30,240,30,240,0,0,240,270,0,0,2160,0,0,2160,0,0,0,
  2160,0,2160,0,0,5,5,5,5,30,30,30,30,30,30,0,0,30,30,
  30,0,30,0,60,60,0,0,30,30,60,30,270,30,60,30,60,0,0,
  0,0,0,0,2160,0,0,2160,0,0,0,2160,0,2160,0,0,5,5,5,
  30,30,30,30,30,30,120,0,30,30,30,0,30,0,240,0,0,0,30,
  30,0,30,0,30,0,30,0,0,120,0,0,270,0,0,2160,2160,0,0,
  0,2160,0,0,0,0,2160,5,5,30,30,30,30,30,30,120,0,30,
  30,30,0,30,0,240,0,0,0,30,30,0,30,0,30,0,30,0,0,120,
  0,0,270,0,0,2160,2160,0,0,0,2160,0,0,0,0,2160,5,30,
  30,30,30,30,30,60,60,30,30,30,0,30,0,60,60,0,0,30,30,
  60,30,0,30,60,30,60,0,60,60,0,0,0,77760,77760,77760,
  77760,0,0,77760,77760,0,77760,0,77760,30,30,30,30,30,30,
  0,60,30,30,30,0,30,0,120,60,0,270,30,30,120,30,90,30,
  60,30,60,0,0,60,270,270,0,2160,360,360,2160,0,0,360,
  360,0,360,0,360,30,30,30,30,30,240,0,30,30,30,0,30,0,
  120,0,0,0,30,30,240,30,270,30,120,30,60,0,60,60,810,
  810,0,0,1080,1080,0,0,0,0,1080,0,1080,0,0,30,30,30,30,
  60,60,30,30,30,0,30,0,0,120,0,0,30,30,120,30,270,30,
  60,30,120,0,0,120,810,0,0,0,1080,0,6480,0,0,0,1080,0,
  1080,0,1080,30,30,30,0,240,30,30,30,0,30,0,120,120,0,
  0,30,30,60,30,270,30,120,30,120,0,240,0,270,0,0,0,0,
  2160,2160,0,0,0,0,0,360,0,0,30,30,0,120,30,30,30,0,
  30,0,0,120,0,810,30,30,60,30,270,30,120,30,240,0,120,
  120,810,0,0,6480,0,0,6480,0,0,1080,1080,0,1080,0,1080,
  30,240,0,30,30,30,0,30,0,0,120,0,0,30,30,120,30,90,
  30,120,30,60,0,0,60,810,0,0,2160,2160,0,6480,0,0,0,
  360,0,6480,0,6480,0,120,30,30,30,0,30,0,60,0,0,810,30,
  30,60,30,270,30,0,30,60,0,60,0,810,810,0,6480,0,1080,
  0,0,0,1080,1080,0,1080,0,0,60,120,0,60,0,120,0,120,
  960,0,0,60,120,0,60,1080,60,60,120,480,0,480,0,3240,0,
  0,25920,0,0,12960,0,0,0,4320,0,4320,0,0,120,60,0,0,
  120,0,240,0,0,0,60,120,0,60,1080,60,60,120,60,0,120,
  240,3240,0,0,0,4320,4320,0,0,0,0,2160,0,6480,0,0,30,
  30,0,30,0,240,120,0,810,30,30,120,30,270,30,240,30,0,
  0,60,60,0,810,0,6480,0,1080,6480,0,0,1080,1080,0,0,0,
  1080,30,0,30,0,0,120,0,810,30,30,120,30,270,30,60,30,
  0,0,120,120,0,0,0,6480,0,0,2160,0,0,6480,6480,0,0,0,
  360,0,30,0,120,120,0,0,30,30,120,30,270,30,60,30,60,0,
  60,60,0,810,0,0,1080,1080,6480,0,0,0,1080,0,0,0,1080,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,120,0,0,810,30,30,0,30,270,30,
  120,30,60,0,60,60,0,270,0,6480,1080,360,0,0,0,6480,
  1080,0,360,0,360,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,120,0,0,120,60,60,
  60,3240,120,60,60,0,0,0,480,3240,0,0,0,0,0,25920,0,0,
  0,4320,0,12960,0,2160,0,0,120,120,0,120,12960,120,0,0,
  240,0,240,60,0,0,0,0,2160,2160,0,0,0,4320,0,0,2160,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,810,270,0,810,270,270,3240,810,3240,0,0,0,0,
  0,0,0,1080,0,0,0,0,0,1080,0,0,0,0,30,0,30,0,30,0,
  30,0,0,120,120,0,810,0,6480,1080,1080,6480,0,0,1080,0,
  0,0,0,1080,0,30,0,30,0,30,0,0,120,120,0,810,0,2160,0,
  6480,6480,0,0,360,0,0,0,0,0,60,0,0,120,0,0,0,0,240,
  0,3240,0,12960,4320,4320,0,0,0,4320,0,0,0,0,0,0,30,60,
  30,120,0,120,120,810,0,0,6480,1080,0,6480,0,0,1080,0,0,
  1080,0,1080,270,0,0,0,0,2160,0,0,0,0,2160,360,0,0,0,
  0,360,0,0,0,0,0,60,30,120,0,0,120,0,0,0,0,360,6480,
  0,0,0,360,360,0,6480,0,1080,0,0,0,480,0,0,0,0,12960,
  12960,4320,0,0,0,4320,0,0,0,0,25920,120,0,0,0,0,810,0,
  0,1080,1080,0,0,0,1080,0,0,1080,0,0,0,60,60,0,3240,0,
  12960,8640,12960,12960,0,0,4320,0,0,0,0,4320,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,60,3240,0,0,0,4320,0,25920,
  0,0,0,4320,0,4320,0,0,3240,0,0,0,0,4320,0,0,0,4320,0,
  0,2160,0,0,0,0,0,0,1080,0,0,0,0,0,0,0,0,1080,0,0,0,
  0,2160,0,0,0,0,0,1080,0,1080,0,0,0,0,0,0,0,0,0,0,0,
  0,2160,0,0,0,0,2160,2160,0,0,0,0,0,0,0,0,360,360,0,
  0,0,0,2160,0,0,0,0,0,360,0,360,0,0,0,0,0,2160,0,
  2160,0,0,0,0,0,0,0,0,0,0,0,0,0,360,0,0,0,0,0,0,0,
  0,0,0,0,0,360,0,5,5,5,5,30,30,30,30,30,30,60,0,30,
  30,30,60,30,0,0,0,0,0,30,30,60,30,0,30,60,30,60,60,
  60,0,0,270,0,0,0,0,0,2160,2160,0,2160,2160,2160,2160,0,
  5,5,5,30,30,30,30,30,30,60,60,30,30,30,60,30,0,0,60,
  270,0,30,30,0,30,0,30,0,30,0,60,60,60,0,0,0,2160,
  2160,2160,2160,0,0,2160,0,0,0,0,2160,5,5,30,30,30,30,
  30,30,120,120,30,30,30,0,30,0,0,60,30,30,30,30,120,30,
  30,30,0,30,0,0,120,120,30,30,30,60,60,60,60,60,60,60,
  60,60,60,60,60,5,30,30,30,30,30,30,60,60,30,30,30,60,
  30,0,0,60,270,0,30,30,0,30,0,30,0,30,0,60,60,60,0,0,
  0,2160,2160,2160,2160,0,0,2160,0,0,0,0,2160,30,30,30,
  30,30,30,0,120,30,30,30,0,30,0,0,120,30,30,30,30,120,
  30,30,30,120,30,120,0,0,120,30,30,30,60,60,60,60,60,
  60,60,60,60,60,60,60,30,30,30,30,30,120,0,30,30,30,
  120,30,0,0,0,0,0,30,30,120,30,0,30,120,30,60,120,120,
  120,270,270,0,0,0,1080,0,0,1080,0,1080,1080,1080,0,0,
  30,30,30,30,120,120,30,30,30,120,30,0,0,120,270,0,30,
  30,120,30,270,30,60,30,120,0,0,120,270,0,270,0,1080,0,
  1080,1080,0,0,1080,0,1080,0,0,30,30,30,0,120,30,30,30,
  120,30,0,0,0,90,90,30,30,60,30,90,30,120,30,120,120,
  120,0,30,30,30,180,180,60,60,60,180,180,180,180,60,60,
  60,30,30,0,120,30,30,30,0,30,0,0,60,0,270,30,30,60,
  30,270,30,120,30,120,0,0,120,270,0,270,1080,0,0,1080,0,
  0,0,1080,1080,1080,0,1080,30,120,0,30,30,30,120,30,0,0,
  120,30,30,30,30,240,30,30,30,120,30,60,0,0,60,90,90,
  90,60,60,180,180,180,60,60,60,60,180,180,180,120,0,30,
  30,30,0,30,0,0,0,0,0,30,30,60,30,0,30,120,30,0,60,
  120,60,270,270,0,0,0,1080,0,1080,0,1080,1080,1080,0,
  1080,0,120,0,0,120,0,120,0,0,240,0,2160,120,120,60,60,
  1080,120,60,120,0,0,480,0,1080,0,0,2160,0,0,4320,0,0,
  2160,720,2160,4320,0,4320,120,60,60,60,120,0,0,0,0,0,
  60,120,240,0,1080,60,60,0,60,0,60,240,1080,0,0,0,0,
  2160,0,2160,4320,0,0,4320,720,2160,0,30,30,0,30,0,0,
  120,0,270,30,30,120,30,270,30,120,30,120,120,120,60,0,
  270,270,1080,0,0,1080,0,0,1080,1080,0,0,1080,0,30,0,30,
  0,0,240,90,90,30,30,120,30,90,30,60,30,0,120,120,120,
  30,30,30,180,180,60,60,60,180,180,180,180,60,60,60,60,
  30,0,0,120,270,0,30,30,120,30,270,30,0,30,60,120,60,0,
  0,270,270,0,1080,1080,0,0,1080,0,0,1080,0,1080,1080,0,
  0,0,0,0,0,120,120,0,120,0,120,120,0,60,480,60,0,0,
  1080,1080,720,0,0,0,0,0,2160,2160,2160,0,4320,4320,0,0,
  0,90,90,30,30,120,30,90,30,120,30,60,120,120,60,30,30,
  30,180,180,60,60,60,180,180,180,180,60,60,60,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,120,60,60,60,1080,120,480,60,
  240,60,0,60,0,1080,0,0,0,8640,0,4320,720,6480,720,0,
  4320,4320,0,30,270,30,1080,270,30,30,1080,0,0,0,1080,
  1080,0,0,0,0,0,0,0,0,0,120,120,120,0,0,0,270,30,0,
  270,30,30,2160,270,1080,1080,0,1080,0,0,0,0,0,0,0,0,
  120,0,120,0,0,0,0,30,0,30,0,30,120,30,0,120,120,120,
  0,0,0,1080,0,1080,1080,0,1080,0,0,0,0,0,1080,0,30,30,
  30,0,30,120,0,120,120,90,90,90,60,60,180,180,180,60,60,
  60,60,180,180,180,0,0,0,60,60,0,240,0,480,0,0,1080,
  4320,4320,720,720,6480,8640,4320,0,0,0,0,0,0,30,60,30,
  0,120,0,120,0,0,0,0,1080,0,1080,1080,1080,1080,0,0,0,
  0,1080,30,0,0,0,0,1080,0,0,0,0,120,120,0,0,0,120,120,
  0,0,0,0,0,60,30,120,120,0,120,90,90,90,60,60,180,180,
  180,60,60,60,60,180,180,180,0,240,0,0,0,1080,1080,0,0,
  4320,2160,0,2160,0,4320,0,0,2160,720,2160,0,120,0,0,270,
  0,0,0,1080,1080,0,1080,1080,1080,0,0,0,1080,0,120,60,
  60,0,0,1080,720,2160,4320,0,0,2160,2160,0,0,0,0,4320,0,
  60,2160,0,1080,4320,0,0,720,2160,0,0,4320,4320,2160,0,
  2160,120,0,0,0,0,4320,0,2160,2160,0,0,0,4320,720,0,
  2160,2160,1080,0,2160,720,0,0,4320,2160,2160,0,2160,0,
  4320,0,30,30,0,0,120,120,0,0,0,0,0,0,0,0,30,0,0,0,
  120,120,0,0,0,0,0,0,120,0,0,120,0,120,0,0,0,0,120,
  120,0,60,0,0,0,60,60,60,60,0,0,0,0,0,0,60,60,60,60,
  0,0,0,60,60,0,0,0,0,60,60,60,60,0,0,0,0,60,60,60,0,
  0,0,0,60,60,60,60,60,60,0,0,0,60,60,0,0,0,60,0,0,0,
  0,0,0,60,60,60,5,5,5,30,30,30,30,30,30,0,60,30,30,
  30,0,30,60,60,60,0,0,30,30,0,30,0,30,0,30,0,0,0,60,
  0,0,270,2160,0,0,2160,2160,2160,0,0,2160,0,2160,0,5,5,
  30,30,30,30,30,30,60,60,30,30,30,0,30,60,60,60,0,0,
  30,30,60,30,270,30,0,30,0,0,60,60,0,270,90,2160,0,0,
  2160,2160,2160,0,360,2160,360,2160,0,5,30,30,30,30,30,
  30,30,30,30,30,30,30,30,30,30,30,0,0,30,30,30,30,0,
  30,30,30,30,30,30,30,0,0,0,77760,0,0,77760,0,0,0,
  77760,0,77760,0,0,30,30,30,30,30,30,0,60,30,30,30,0,
  30,60,60,60,0,0,30,30,0,30,0,30,0,30,0,0,0,60,0,0,
  270,2160,0,0,2160,2160,2160,0,0,2160,0,2160,0,30,30,30,
  30,30,60,0,30,30,30,240,30,0,60,0,0,0,30,30,60,30,
  810,30,60,30,30,30,30,30,0,810,0,0,0,0,0,6480,6480,0,
  1080,6480,1080,6480,0,30,30,30,30,30,30,30,30,30,30,30,
  60,0,60,0,0,30,30,120,30,810,30,30,30,240,60,0,60,0,
  0,270,0,0,0,6480,6480,6480,0,1080,6480,1080,0,0,30,30,
  30,0,60,30,30,30,60,30,30,30,30,0,0,30,30,30,30,0,30,
  240,30,60,0,60,0,0,270,270,6480,0,0,0,2160,0,0,1080,
  6480,360,0,0,30,30,0,60,30,30,30,60,30,30,30,30,0,0,
  30,30,30,30,810,30,240,30,60,0,60,0,0,0,270,6480,0,0,
  6480,6480,0,0,1080,6480,1080,0,0,30,60,0,30,30,30,0,30,
  0,60,0,0,0,30,30,60,30,270,30,60,30,30,30,30,30,0,
  810,0,0,0,0,0,0,0,0,360,2160,6480,6480,0,60,60,30,30,
  30,60,30,30,30,30,0,0,30,30,30,30,810,30,60,30,30,30,
  30,30,0,810,0,6480,0,0,0,6480,0,0,1080,6480,1080,6480,
  0,30,60,30,30,30,60,60,30,60,0,0,30,0,30,30,0,30,30,
  0,60,0,60,0,0,0,1080,12960,0,0,12960,12960,0,0,0,0,
  2160,0,0,0,30,30,30,0,0,60,0,0,0,30,60,60,30,1620,30,
  30,60,30,60,30,60,0,1620,0,0,0,0,0,0,0,0,2160,12960,
  51840,12960,0,30,30,0,30,0,60,240,0,0,30,30,60,30,810,
  30,60,30,30,30,30,30,0,810,270,6480,0,0,6480,0,0,0,
  1080,6480,0,6480,0,30,30,30,60,0,60,0,0,30,30,240,30,
  0,30,30,30,0,60,0,60,0,0,90,0,0,0,2160,0,6480,0,6480,
  0,2160,2160,0,30,30,60,60,60,0,0,30,30,60,30,810,30,
  30,30,30,30,30,30,0,810,270,0,0,0,6480,0,6480,0,1080,
  6480,0,6480,0,0,0,60,0,0,0,30,60,60,30,1620,30,30,60,
  30,60,30,60,0,1620,0,0,0,0,12960,0,0,0,2160,12960,0,
  12960,0,0,60,0,0,0,30,30,60,30,810,30,60,30,30,30,30,
  30,0,270,0,0,0,0,0,0,0,0,1080,6480,0,2160,0,30,30,0,
  0,60,30,30,30,1620,60,60,30,60,30,60,30,0,1620,0,0,0,
  0,0,0,0,0,2160,12960,0,12960,0,30,0,0,0,30,30,30,0,
  240,30,30,0,60,0,60,0,0,540,0,0,0,12960,12960,12960,0,
  51840,51840,2160,0,0,0,0,60,30,30,30,1620,60,60,30,60,
  30,60,30,0,1620,0,0,0,0,0,12960,0,0,2160,12960,2160,
  12960,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,30,0,30,0,30,30,30,0,60,0,
  60,0,810,270,6480,0,0,6480,0,6480,0,0,0,0,6480,0,30,
  30,0,30,0,30,60,0,60,240,0,0,0,2160,0,0,6480,6480,
  2160,0,2160,0,1080,6480,0,30,0,0,30,30,0,60,0,60,0,
  6480,540,0,0,0,12960,12960,12960,0,0,0,0,0,0,0,30,30,
  30,60,60,60,60,0,0,270,6480,0,0,6480,6480,6480,0,0,0,
  1080,0,0,0,0,0,0,6480,0,0,0,0,0,0,0,0,0,0,2160,0,0,
  0,0,0,0,30,30,0,60,0,60,0,810,270,0,0,0,6480,0,2160,
  0,0,0,0,6480,0,0,60,0,60,0,0,0,0,12960,0,0,0,12960,
  12960,0,0,0,2160,51840,0,60,0,60,0,0,0,0,6480,0,0,0,
  6480,0,0,0,0,1080,0,0,30,30,30,0,0,540,0,0,0,12960,0,
  12960,0,0,12960,0,12960,0,30,30,0,0,6480,12960,0,0,0,
  12960,0,0,0,12960,2160,0,0,30,0,0,540,0,0,0,12960,0,
  12960,0,0,12960,0,0,0,0,3240,0,12960,0,0,0,12960,0,0,
  0,0,2160,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,270,0,0,
  0,2160,0,0,0,0,0,1080,0,0,0,0,0,0,2160,0,0,0,0,360,
  2160,0,0,0,0,0,0,0,2160,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2160,0,0,0,0,0,0,
  2160,2160,0,0,2160,2160,0,0,0,0,0,0,0,0,2160,0,0,0,0,
  0,0,2160,0,0,5,5,30,30,30,30,30,30,120,0,30,30,30,0,
  30,0,240,0,0,0,30,30,0,30,0,30,0,30,0,0,120,0,0,270,
  0,0,2160,2160,0,0,0,2160,0,0,0,0,2160,5,30,30,30,30,
  30,30,60,60,30,30,30,60,30,0,0,60,270,0,30,30,0,30,0,
  30,0,30,0,60,60,60,0,0,0,2160,2160,2160,2160,0,0,2160,
  0,0,0,0,2160,30,30,30,30,30,30,0,60,30,30,30,0,30,60,
  60,60,0,0,30,30,0,30,0,30,0,30,0,0,0,60,0,0,270,
  2160,0,0,2160,2160,2160,0,0,2160,0,2160,0,30,30,30,30,
  30,120,0,30,30,30,120,30,120,120,0,270,0,30,30,0,30,0,
  30,0,30,0,60,0,0,0,810,0,0,6480,6480,0,1080,1080,0,0,
  1080,0,1080,0,30,30,30,30,0,0,30,30,30,60,30,240,0,
  120,270,0,30,30,0,30,0,30,0,30,0,120,0,120,0,0,810,0,
  6480,0,6480,1080,1080,0,0,1080,0,0,6480,30,30,30,0,120,
  30,30,30,120,30,60,60,60,270,0,30,30,0,30,0,30,0,30,
  0,120,120,0,0,0,0,0,6480,0,0,360,0,0,0,6480,0,2160,
  2160,30,30,0,120,30,30,30,120,30,60,60,60,0,0,30,30,0,
  30,0,30,0,30,0,0,120,120,0,0,810,6480,0,0,6480,1080,0,
  6480,0,1080,0,0,6480,30,120,0,30,30,30,120,30,0,240,
  120,0,0,30,30,0,30,0,30,0,30,0,120,120,120,0,0,810,0,
  0,0,6480,0,2160,2160,0,360,0,0,6480,120,120,30,30,30,0,
  30,60,60,60,0,0,30,30,0,30,0,30,0,30,0,60,120,60,0,
  810,0,6480,0,6480,0,1080,0,6480,0,1080,0,1080,0,0,120,
  120,120,120,120,240,60,480,0,0,120,120,0,0,0,120,0,120,
  0,0,480,0,0,0,0,25920,0,0,12960,2160,4320,12960,0,2160,
  0,0,12960,120,120,60,60,120,0,0,0,1080,0,120,120,0,120,
  0,0,0,0,0,240,60,480,0,6480,0,0,12960,25920,0,2160,
  2160,0,0,2160,0,0,25920,30,30,0,30,0,120,120,0,0,30,
  30,0,30,0,30,0,30,0,0,120,120,0,810,810,6480,0,6480,
  6480,0,0,6480,0,1080,0,1080,6480,30,120,30,240,0,120,0,
  0,30,30,0,30,0,30,0,30,0,0,120,120,0,0,270,6480,0,
  2160,2160,2160,0,6480,0,0,0,0,2160,60,30,0,120,120,270,
  0,30,30,0,30,0,30,0,30,0,60,0,120,0,810,810,0,6480,
  6480,6480,0,1080,0,0,1080,0,1080,6480,0,0,480,0,0,0,0,
  0,0,60,0,60,0,120,0,0,60,0,0,3240,3240,12960,0,25920,
  0,0,0,25920,0,0,0,2160,0,120,120,0,0,0,30,30,0,30,0,
  30,0,30,0,60,120,0,0,270,0,0,0,2160,2160,2160,1080,
  6480,0,1080,0,360,0,60,60,0,0,120,0,0,60,0,120,0,60,
  0,120,960,120,0,0,3240,12960,0,25920,12960,0,0,12960,0,
  4320,0,0,12960,60,2160,0,120,60,0,0,0,120,0,60,0,240,
  0,480,0,0,3240,0,0,0,25920,4320,4320,0,0,0,0,0,12960,
  1080,0,120,60,0,60,0,120,0,0,0,60,0,60,0,3240,0,0,
  25920,25920,0,2160,6480,0,0,4320,0,2160,0,0,270,0,0,270,
  0,90,0,270,0,0,1080,0,0,0,0,2160,0,0,0,0,0,2160,0,
  360,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,30,0,30,0,30,0,30,0,240,120,120,0,
  810,810,6480,6480,6480,6480,0,1080,6480,0,0,0,1080,6480,
  0,30,0,30,0,30,0,0,120,120,0,810,810,2160,2160,6480,
  6480,0,0,2160,0,2160,0,6480,6480,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,30,0,30,0,120,120,
  120,0,0,810,6480,6480,0,6480,1080,1080,6480,0,0,0,0,
  6480,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,30,0,120,0,120,0,810,0,2160,2160,6480,0,1080,360,0,0,
  2160,0,6480,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,120,240,0,0,810,0,6480,6480,6480,0,1080,1080,0,0,
  0,0,1080,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  120,60,0,0,3240,12960,0,0,12960,0,0,25920,0,2160,0,0,
  25920,0,0,0,6480,0,12960,0,25920,2160,2160,0,0,2160,0,
  4320,12960,0,0,0,12960,12960,12960,0,2160,2160,25920,0,
  4320,0,2160,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  2160,1080,0,0,0,0,0,0,0,0,0,2160,0,1080,0,0,0,0,0,
  1080,0,0,0,0,0,2160,0,0,2160,0,0,0,0,0,0,2160,2160,0,
  2160,0,0,0,0,2160,0,0,0,0,0,2160,2160,2160,0,0,0,0,0,
  2160,0,0,0,0,0,0,360,2160,2160,0,360,0,0,0,0,2160,0,
  0,0,0,0,0,0,0,0,0,0,0,2160,5,30,30,30,30,30,30,0,
  60,30,30,30,0,30,60,0,60,0,0,30,30,60,30,0,30,0,30,
  0,0,0,60,0,0,270,2160,0,0,2160,0,0,0,2160,0,2160,0,0,
  30,30,30,30,30,30,0,0,30,30,30,0,30,0,0,120,30,30,30,
  30,120,30,30,30,0,30,0,0,0,0,30,30,30,60,60,60,60,60,
  60,60,60,60,60,60,60,30,30,30,30,30,120,0,30,30,30,0,
  30,120,120,0,270,0,30,30,120,30,270,30,0,30,0,0,60,60,
  270,270,0,0,1080,1080,0,1080,0,0,1080,0,0,1080,0,30,30,
  30,30,60,60,30,30,30,0,30,120,0,0,270,0,30,30,120,30,
  270,30,0,30,0,0,0,120,0,0,270,0,1080,0,1080,0,1080,0,
  0,1080,1080,0,1080,30,30,30,0,120,30,30,30,0,30,120,60,
  60,90,90,30,30,120,30,90,30,0,30,0,0,120,0,30,30,30,
  180,180,60,60,60,180,180,180,180,60,60,60,30,30,0,120,
  30,30,30,0,30,120,120,60,0,270,30,30,120,30,270,30,0,
  30,0,0,120,120,270,0,0,1080,0,0,1080,1080,0,1080,1080,
  0,0,0,1080,30,120,0,30,30,30,0,30,0,120,120,30,30,30,
  30,120,30,30,30,0,30,0,0,0,60,90,90,90,60,60,180,180,
  180,60,60,60,60,180,180,180,120,120,30,30,30,0,30,60,
  60,60,0,270,30,30,120,30,270,30,0,30,0,0,60,0,0,0,0,
  1080,0,1080,0,1080,0,0,1080,1080,1080,1080,0,60,120,0,
  60,0,0,0,60,480,0,1080,60,120,60,0,1080,60,0,120,0,0,
  0,0,2160,0,1080,2160,0,0,4320,8640,6480,720,720,0,4320,
  0,0,120,60,0,0,120,0,240,0,1080,0,60,240,240,120,0,0,
  0,240,0,0,120,480,0,0,0,0,4320,2160,0,2160,4320,0,8640,
  0,2160,720,6480,30,30,0,30,0,120,120,0,270,30,30,0,30,
  0,30,0,30,0,0,60,60,0,270,270,1080,0,1080,1080,0,0,
  1080,0,1080,0,0,1080,30,0,30,120,0,240,90,90,30,30,120,
  30,90,30,0,30,0,0,120,240,30,30,30,180,180,60,60,60,
  180,180,180,180,60,60,60,0,30,120,0,120,270,0,30,30,0,
  30,0,30,0,30,0,0,0,120,0,0,0,0,1080,0,0,0,1080,0,
  1080,1080,0,1080,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,120,240,0,90,90,
  30,30,120,30,90,30,0,30,0,0,60,0,30,30,30,180,180,60,
  60,60,180,180,180,180,60,60,60,120,60,0,0,0,0,0,120,0,
  120,0,120,0,0,240,120,0,2160,1080,0,0,2160,2160,0,0,
  4320,0,4320,0,2160,720,60,0,0,120,60,60,60,1080,120,0,
  120,0,0,0,0,0,0,1080,6480,2160,0,0,0,2160,0,2160,720,
  4320,0,8640,0,0,120,60,120,120,2160,120,0,120,0,0,480,
  0,0,2160,0,0,2160,0,0,0,2160,2160,720,2160,4320,8640,0,
  30,0,30,0,270,30,30,0,270,0,0,1080,1080,0,0,0,0,0,0,
  0,0,0,120,0,120,0,0,0,0,30,1080,270,30,30,0,270,0,0,
  0,1080,0,0,0,0,0,0,0,0,120,0,120,120,0,0,0,30,0,30,
  0,30,0,30,0,0,120,0,0,270,270,1080,1080,1080,1080,0,0,
  1080,0,0,0,1080,1080,0,30,30,30,0,30,0,0,240,120,90,
  90,90,60,60,180,180,180,60,60,60,60,180,180,180,60,0,0,
  0,120,0,0,0,960,0,0,0,0,4320,720,2160,0,0,0,0,0,0,
  2160,2160,0,30,0,30,0,0,120,120,270,0,270,0,0,0,0,
  1080,1080,0,0,0,1080,0,1080,30,0,0,0,0,0,0,0,0,0,0,
  120,0,0,0,0,0,0,0,0,0,0,0,30,0,0,0,120,90,90,90,60,
  60,180,180,180,60,60,60,60,180,180,180,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,120,0,0,270,0,0,
  1080,0,0,1080,1080,0,0,0,1080,1080,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,60,0,0,0,0,4320,0,2160,0,8640,0,4320,4320,720,
  6480,720,0,1080,0,720,2160,4320,0,0,2160,2160,0,6480,
  8640,4320,0,30,30,0,0,0,0,0,0,0,0,0,0,120,0,30,0,0,
  0,120,0,0,0,0,0,120,0,120,0,0,120,0,120,0,0,0,0,120,
  120,0,60,0,0,0,60,60,60,60,0,0,0,0,0,0,60,60,60,60,
  0,0,0,60,60,0,0,0,0,60,60,60,60,0,0,0,0,60,60,60,0,
  0,0,0,60,60,60,60,60,60,0,0,0,60,60,0,0,0,60,0,0,0,
  0,0,0,60,60,60,30,30,30,30,30,30,0,0,30,30,30,0,30,
  60,60,0,0,0,30,30,60,30,270,30,60,30,60,0,0,0,0,0,0,
  0,2160,2160,0,0,0,2160,2160,0,2160,0,2160,30,30,30,30,
  30,60,0,30,30,30,0,30,0,60,0,810,0,30,30,60,30,270,
  30,60,30,30,30,30,30,0,0,0,0,6480,6480,0,0,0,0,6480,
  0,6480,0,0,30,30,30,30,30,30,30,30,30,30,30,60,0,60,
  810,0,30,30,240,30,270,30,30,30,0,60,0,60,0,0,810,0,
  6480,0,1080,0,0,0,6480,0,6480,0,6480,30,30,30,0,60,30,
  30,30,60,30,30,30,30,0,0,30,30,30,30,0,30,240,30,60,
  0,60,0,0,0,0,1080,6480,2160,2160,0,0,6480,6480,0,2160,
  0,0,30,30,0,60,30,30,30,60,30,30,30,30,0,0,30,30,30,
  30,270,30,0,30,60,0,60,240,0,0,810,1080,0,0,1080,0,0,
  6480,6480,0,6480,0,6480,30,60,0,30,30,30,0,30,0,60,240,
  0,0,30,30,60,30,90,30,60,30,30,30,30,30,0,0,0,2160,
  2160,6480,6480,0,0,0,2160,0,0,0,0,60,60,30,30,30,60,
  30,30,30,30,0,0,30,30,30,30,270,30,60,30,30,30,30,30,
  0,0,0,1080,0,6480,0,0,0,6480,6480,0,6480,0,0,30,60,30,
  30,30,60,60,30,60,0,0,30,0,30,30,6480,30,30,0,60,0,
  60,0,0,0,6480,2160,0,0,0,0,0,12960,0,0,12960,0,12960,
  240,30,30,30,0,0,60,0,0,0,30,60,60,30,540,30,30,60,
  30,60,30,60,0,0,0,0,12960,12960,0,0,0,0,12960,0,0,0,
  12960,30,30,0,30,0,60,120,0,0,30,30,60,30,270,30,60,
  30,30,30,30,30,0,0,810,1080,0,6480,1080,0,0,6480,6480,
  0,0,0,6480,30,30,30,60,0,60,810,0,30,30,0,30,0,30,30,
  30,0,60,0,60,0,0,270,6480,6480,0,360,0,0,0,0,0,0,0,
  2160,30,30,60,60,60,810,0,30,30,60,30,270,30,30,30,30,
  30,30,30,0,0,810,0,6480,6480,1080,0,0,0,6480,0,0,0,
  6480,0,0,60,0,0,0,30,60,60,30,540,30,30,60,30,60,30,
  60,0,0,0,0,0,12960,0,0,0,0,12960,0,0,0,12960,240,60,
  0,810,0,30,30,60,30,270,30,60,30,30,30,30,30,0,0,0,0,
  6480,2160,0,0,0,0,6480,0,0,0,0,30,30,0,0,60,30,30,30,
  540,60,60,30,60,30,60,30,0,0,0,2160,0,12960,51840,0,0,
  12960,12960,0,0,0,51840,30,1620,0,0,30,30,30,0,0,30,30,
  0,60,0,60,0,0,1620,0,12960,0,2160,0,0,0,0,0,0,0,
  12960,6480,0,60,30,30,30,540,60,60,30,60,30,60,30,0,0,
  0,0,0,12960,0,0,0,12960,12960,0,0,0,0,0,810,270,1620,
  810,270,270,3240,0,1620,0,1620,0,0,0,0,1080,0,0,0,0,0,
  0,2160,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,30,0,30,0,30,30,30,0,60,240,60,
  0,0,810,1080,6480,6480,1080,0,0,6480,0,0,0,0,6480,30,
  30,270,30,0,30,60,0,60,240,0,0,0,360,0,0,1080,0,0,
  2160,0,0,6480,0,6480,30,0,0,30,30,0,60,0,60,0,0,1620,
  2160,12960,0,2160,0,0,12960,0,0,0,0,12960,0,30,30,30,
  60,60,60,60,0,0,810,1080,6480,0,1080,0,0,6480,0,0,6480,
  0,6480,0,0,0,0,1080,0,0,0,0,0,360,2160,0,0,0,0,2160,
  0,0,0,0,0,30,30,0,60,0,60,0,0,810,0,2160,0,1080,0,0,
  0,0,0,0,0,6480,0,60,0,60,0,0,0,0,2160,0,0,0,0,0,
  12960,0,0,12960,0,0,60,0,60,0,0,0,0,1080,0,0,0,0,0,
  6480,0,0,6480,0,0,30,30,30,0,0,1620,51840,12960,0,2160,
  0,0,0,0,0,0,0,12960,30,30,0,0,0,2160,0,0,0,0,0,
  12960,12960,0,12960,0,0,30,0,0,1620,0,12960,0,2160,0,0,
  0,12960,0,0,0,12960,0,0,0,2160,51840,12960,0,0,0,12960,
  0,0,12960,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,2160,0,0,0,0,0,0,0,0,0,
  2160,0,0,0,0,2160,2160,0,0,0,0,0,0,0,0,2160,0,0,0,0,
  0,2160,0,0,0,0,0,0,0,2160,0,0,0,0,0,2160,0,2160,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,30,30,30,30,30,0,0,30,30,30,0,30,120,0,0,270,0,
  30,30,120,30,270,30,120,30,120,0,0,0,0,270,0,0,0,1080,
  0,0,1080,0,1080,1080,1080,0,0,30,30,30,30,0,0,30,30,
  30,0,30,120,0,120,0,0,30,30,120,30,270,30,120,30,120,
  0,0,120,0,0,0,0,1080,0,1080,1080,0,0,1080,0,1080,0,0,
  30,30,30,0,120,30,30,30,0,30,60,60,60,90,90,30,30,60,
  30,90,30,120,30,0,0,0,0,30,30,30,180,180,60,60,60,180,
  180,180,180,60,60,60,30,30,0,120,30,30,30,0,30,60,60,
  60,0,270,30,30,60,30,270,30,120,30,120,0,0,120,270,0,
  270,1080,0,0,1080,0,0,0,1080,1080,1080,0,1080,30,0,0,
  30,30,30,0,30,0,120,120,30,30,30,30,120,30,30,30,0,30,
  120,0,0,120,90,90,90,60,60,180,180,180,60,60,60,60,180,
  180,180,0,120,30,30,30,0,30,60,60,60,0,270,30,30,60,
  30,270,30,120,30,120,0,0,0,270,0,0,0,0,1080,0,1080,0,
  1080,1080,1080,0,1080,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  120,120,0,0,120,0,960,0,0,0,120,0,0,120,1080,0,0,240,
  60,0,0,480,0,2160,0,0,4320,2160,0,0,0,0,4320,4320,720,
  2160,2160,30,30,0,30,0,120,120,0,0,30,30,120,30,0,30,
  120,30,0,0,0,120,0,0,270,1080,0,0,1080,0,0,1080,1080,
  0,0,1080,0,30,0,30,120,0,120,90,90,30,30,120,30,90,30,
  120,30,0,0,0,0,30,30,30,180,180,60,60,60,180,180,180,
  180,60,60,60,0,30,0,0,0,270,0,30,30,0,30,0,30,120,30,
  0,0,0,120,0,270,0,0,1080,1080,0,0,1080,0,0,1080,0,
  1080,1080,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,120,240,0,90,90,30,30,
  120,30,90,30,120,30,0,0,0,0,30,30,30,180,180,60,60,60,
  180,180,180,180,60,60,60,60,60,0,1080,0,60,60,60,0,240,
  240,0,960,0,0,120,0,2160,1080,0,0,2160,2160,0,0,4320,0,
  0,0,720,720,60,2160,0,120,60,60,60,1080,120,120,0,0,0,
  0,240,1080,0,0,0,720,0,0,4320,2160,0,2160,720,0,0,0,
  1080,0,120,60,60,60,0,120,240,60,0,0,0,60,0,1080,0,0,
  2160,0,0,0,720,2160,720,2160,4320,4320,0,30,270,30,1080,
  0,30,30,1080,270,2160,0,0,1080,0,0,0,120,0,0,0,0,0,0,
  0,120,0,0,0,0,30,0,270,30,30,1080,270,0,0,0,2160,0,0,
  0,0,120,0,0,0,120,0,0,120,0,0,0,30,0,30,0,30,0,30,
  0,0,0,120,0,270,270,1080,0,1080,1080,0,1080,0,0,0,0,0,
  1080,60,30,30,30,0,30,120,0,0,120,90,90,90,60,60,180,
  180,180,60,60,60,60,180,180,180,60,0,0,60,60,0,0,0,
  240,0,1080,0,4320,4320,720,720,2160,0,0,0,0,0,2160,2160,
  0,30,0,30,120,0,0,120,270,0,270,0,1080,0,1080,1080,
  1080,1080,0,0,0,0,1080,30,0,0,0,0,0,0,0,0,0,120,0,0,
  0,0,120,0,0,0,0,0,0,0,30,120,0,0,120,90,90,90,60,60,
  180,180,180,60,60,60,60,180,180,180,0,480,0,0,0,2160,
  1080,0,4320,0,0,0,2160,4320,4320,0,0,0,720,0,240,0,0,
  0,270,270,0,1080,0,1080,0,1080,1080,1080,0,0,1080,0,0,
  0,0,0,0,0,1080,720,2160,0,4320,0,2160,0,0,2160,0,4320,
  4320,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,1080,1080,0,0,720,4320,0,4320,
  0,2160,0,0,4320,0,0,30,30,0,0,120,0,0,0,0,0,0,0,120,
  120,30,0,0,0,0,0,0,0,0,0,120,0,120,0,0,120,0,0,0,0,
  0,0,120,0,0,60,0,0,0,60,60,60,60,0,0,0,0,0,0,60,60,
  60,60,0,0,0,60,60,0,0,0,0,60,60,60,60,0,0,0,0,60,
  60,60,0,0,0,0,60,60,60,60,60,60,0,0,0,60,60,0,0,0,
  60,0,0,0,0,0,0,60,60,60,6,6,6,6,0,0,6,6,6,12,6,12,
  0,0,18,0,6,6,12,6,18,6,12,6,12,12,0,0,18,0,0,0,36,
  0,0,36,36,0,36,36,36,0,0,6,6,6,0,0,6,6,6,12,6,12,
  0,0,18,0,6,6,12,6,18,6,12,6,12,12,0,0,18,0,0,0,36,
  0,0,36,36,0,36,36,36,0,0,6,6,0,0,6,6,6,12,6,12,12,
  0,0,0,6,6,12,6,18,6,12,6,12,0,12,12,18,0,0,0,0,0,
  0,36,0,0,36,36,36,0,0,6,12,0,6,6,6,12,6,0,12,0,0,
  0,6,6,12,6,18,6,12,6,6,6,6,6,18,18,0,0,0,36,0,36,
  0,0,36,36,36,36,0,12,0,6,6,6,12,6,0,12,0,0,0,6,6,
  12,6,18,6,12,6,6,6,6,6,18,18,0,0,0,36,0,36,0,0,36,
  36,36,36,0,0,12,12,12,12,12,24,12,0,0,0,12,12,12,12,
  36,12,12,12,24,0,0,0,36,0,0,0,0,0,0,72,72,0,72,72,
  72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,0,6,0,12,
  0,0,0,6,6,12,6,18,6,12,6,6,6,6,6,0,18,0,0,0,36,0,
  0,0,0,36,36,0,36,0,6,12,6,12,0,0,18,0,6,6,12,6,18,
  6,12,6,0,12,12,12,18,18,0,0,36,36,0,36,36,0,36,36,
  36,36,0,0,6,12,12,0,18,0,6,6,12,6,18,6,12,6,6,6,6,
  6,0,18,0,0,36,36,0,0,36,0,36,36,0,36,0,0,0,24,0,0,
  0,12,12,24,12,36,12,12,12,12,12,12,12,0,36,0,0,72,72,
  0,0,0,0,72,72,0,72,0,12,12,0,18,0,6,6,12,6,18,6,12,
  6,6,6,6,6,0,18,0,0,36,36,0,0,36,0,36,36,0,36,0,12,
  0,0,0,12,12,12,12,36,12,24,12,12,12,12,12,0,36,0,0,
  0,72,0,0,0,0,72,72,72,72,0,0,36,0,12,12,12,12,36,12,
  12,12,0,24,0,0,36,0,0,0,72,0,0,72,72,0,72,72,72,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,18,18,36,18,18,18,36,18,36,0,36,36,0,
  0,0,0,0,0,0,0,0,0,36,36,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,0,6,0,6,
  12,6,0,12,12,12,0,18,0,0,36,36,0,0,36,0,0,0,0,36,0,
  12,6,18,6,0,6,12,0,12,12,18,18,0,0,36,36,0,36,36,0,
  36,36,36,36,0,12,0,0,12,0,0,24,0,0,0,36,0,0,72,72,
  0,72,72,0,0,0,0,72,0,0,6,12,6,12,12,12,12,18,0,0,0,
  36,0,0,36,36,0,0,0,36,0,0,0,0,0,0,36,0,0,0,0,0,0,
  36,0,0,0,36,0,0,0,0,0,0,0,6,12,12,0,0,18,18,0,0,
  36,36,0,36,36,0,0,0,36,36,0,0,24,0,0,0,36,36,0,0,
  72,72,0,72,72,0,0,0,72,72,0,12,12,0,0,18,18,0,0,36,
  36,0,36,36,0,0,0,36,36,0,6,6,6,0,36,0,0,72,72,0,0,
  72,0,0,72,0,72,0,6,6,36,0,0,0,0,72,0,72,0,0,72,72,
  72,0,0,6,36,0,0,0,72,0,0,72,72,0,0,72,72,72,0,36,0,
  0,0,72,0,0,72,72,0,0,72,72,72,0,18,0,0,0,36,0,0,0,
  0,0,0,0,36,0,0,0,0,0,0,36,0,0,0,0,36,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,
  0,36,36,0,0,0,0,36,0,0,0,0,36,36,0,0,0,0,0,0,0,0,
  0,0,0,0,0,36,36,0,0,36,36,0,0,0,0,0,0,0,0,36,0,0,
  0,0,0,0,36,0,0,6,6,6,0,0,6,6,6,12,6,12,0,0,18,0,
  6,6,12,6,18,6,12,6,12,12,0,0,18,0,0,0,36,0,0,36,36,
  0,36,36,36,0,0,6,6,0,0,6,6,6,12,6,0,0,12,0,0,6,6,
  12,6,18,6,12,6,12,0,0,12,18,0,18,0,0,0,36,36,0,0,
  36,36,36,0,36,6,0,0,6,6,6,12,6,0,0,12,0,0,6,6,12,
  6,18,6,12,6,12,0,0,12,18,0,18,0,0,0,36,36,0,0,36,
  36,36,0,36,12,12,6,6,6,12,6,12,0,12,0,0,6,6,12,6,
  18,6,12,6,12,12,0,12,18,0,0,0,0,0,0,36,0,0,36,36,
  36,0,0,6,12,6,6,6,12,0,0,0,0,0,6,12,12,6,36,6,6,
  12,12,0,0,0,36,0,0,0,0,0,0,72,72,0,72,72,72,0,72,
  12,6,6,6,12,0,0,0,0,0,6,12,12,6,36,6,6,12,12,0,0,
  0,36,0,0,0,0,0,0,72,72,0,72,72,72,0,72,6,6,0,6,0,
  0,12,0,0,6,6,12,6,18,6,12,6,12,12,0,12,0,0,18,0,0,
  0,36,0,0,0,36,36,0,0,36,6,6,6,12,0,12,18,0,6,6,12,
  6,18,6,6,6,0,12,0,12,0,0,18,0,36,0,36,0,36,0,36,36,
  0,0,36,6,6,12,0,12,18,0,6,6,12,6,18,6,6,6,0,12,0,
  12,0,0,18,0,36,0,36,0,36,0,36,36,0,0,36,0,0,0,0,0,
  0,6,12,12,6,36,6,6,12,12,24,0,24,0,0,36,0,72,0,72,
  0,0,0,72,72,0,0,72,12,0,0,18,0,6,6,12,6,18,6,12,6,
  12,12,0,12,18,0,18,0,36,0,36,36,36,0,36,36,36,0,36,
  0,12,0,0,12,12,12,12,36,12,24,12,24,12,0,12,0,0,36,
  0,0,0,72,0,0,0,72,72,72,0,72,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,12,
  12,12,12,36,12,24,12,24,12,0,12,36,0,0,0,72,0,0,72,
  72,0,72,72,72,0,0,0,18,18,36,18,18,18,36,18,36,0,0,
  36,0,0,0,0,0,0,0,0,0,0,36,36,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,0,6,0,
  6,6,6,0,12,0,12,0,0,18,0,36,0,36,0,36,0,0,0,0,0,
  36,12,6,18,6,0,6,12,0,0,12,18,0,18,0,36,0,36,36,36,
  0,36,36,36,0,36,0,0,0,12,12,0,24,0,24,0,0,36,0,72,
  0,72,72,72,0,0,0,0,0,72,0,6,6,6,12,12,0,12,18,0,18,
  0,36,0,36,36,36,0,0,0,36,0,36,0,0,0,0,36,0,0,0,0,
  0,0,36,0,0,0,36,0,0,0,0,0,0,6,6,12,12,0,12,18,0,
  18,0,36,0,36,36,36,0,0,0,36,0,36,0,12,0,0,0,36,0,0,
  0,72,0,0,72,72,0,0,0,72,0,72,12,12,0,0,18,0,0,0,36,
  0,0,36,36,0,0,0,36,0,0,12,0,12,0,0,36,0,72,0,72,0,
  72,0,0,72,0,0,72,0,12,36,0,36,0,0,0,72,72,0,0,72,
  72,72,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,
  0,0,72,0,0,72,72,0,0,72,72,0,0,0,18,0,0,0,36,0,0,
  0,0,0,0,0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  36,0,0,0,0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,
  0,36,36,0,0,0,0,0,0,0,0,0,0,0,0,36,0,0,0,0,36,0,
  36,0,0,0,0,36,0,36,0,36,36,0,0,0,0,0,0,0,0,36,0,0,
  0,0,0,0,0,36,0,6,6,0,12,6,6,6,12,6,6,6,6,0,18,6,
  6,6,6,18,6,12,6,12,0,12,0,18,0,0,36,0,0,0,36,0,36,
  36,36,36,0,0,6,0,0,6,6,6,12,6,0,12,12,18,18,6,6,12,
  6,18,6,12,6,12,12,12,0,18,18,18,36,36,36,36,36,36,36,
  36,36,36,36,36,0,12,6,6,6,12,6,6,6,6,0,18,6,6,6,6,
  18,6,12,6,12,0,12,0,18,0,0,36,0,0,0,36,0,36,36,36,
  36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,12,12,12,
  12,0,0,0,36,0,12,12,24,12,36,12,12,12,12,24,12,0,36,
  0,0,0,72,0,0,72,72,0,72,72,72,72,72,6,6,0,6,0,12,
  12,0,18,6,6,12,6,18,6,12,6,12,12,12,0,0,18,18,36,0,
  36,36,0,0,36,36,36,0,36,36,6,0,6,12,0,0,18,18,6,6,
  12,6,18,6,12,6,0,12,12,0,6,6,6,36,36,12,12,12,36,36,
  36,36,12,12,12,12,6,12,12,12,18,0,6,6,12,6,18,6,12,
  6,12,12,12,0,0,18,18,0,36,36,36,0,36,0,36,36,0,36,
  36,0,0,0,0,0,36,12,12,24,12,36,12,12,12,12,24,12,0,
  0,36,36,72,72,72,72,0,0,72,72,72,0,72,72,12,0,0,18,
  18,6,6,12,6,18,6,12,6,0,12,12,0,6,6,6,36,36,12,12,
  12,36,36,36,36,12,12,12,6,6,0,36,12,6,6,6,36,12,12,
  6,24,12,24,0,0,36,36,72,0,72,72,0,0,72,72,72,72,72,
  72,6,36,0,12,6,6,6,36,12,12,6,0,12,0,0,36,0,0,0,72,
  0,0,72,72,0,72,72,72,0,0,36,0,12,6,6,6,36,12,12,6,
  0,12,0,0,36,0,0,0,72,0,0,72,72,0,72,72,72,0,0,18,
  18,18,36,18,18,18,36,18,36,0,36,0,0,0,0,36,0,0,0,0,
  0,36,36,36,0,0,0,18,18,36,18,18,18,36,18,36,36,0,0,
  0,0,0,0,36,0,0,0,36,0,36,36,0,0,0,6,0,6,0,6,12,6,
  0,12,12,0,0,18,18,36,36,36,36,0,36,36,0,0,0,36,36,6,
  6,18,6,0,6,12,0,12,0,18,18,18,36,36,36,36,36,36,36,
  36,36,36,36,36,6,0,0,12,6,0,12,0,0,0,36,36,72,72,72,
  72,72,72,72,0,0,0,72,72,0,6,0,6,12,12,12,0,18,0,0,
  36,36,0,0,36,36,36,0,0,36,0,0,18,0,0,0,36,36,0,0,0,
  0,36,36,0,0,0,36,36,0,0,0,0,0,12,6,12,12,0,0,18,18,
  18,36,36,36,36,36,36,36,36,36,36,36,36,0,24,0,24,0,
  36,0,0,72,72,0,0,72,72,72,0,0,72,72,72,12,12,12,0,
  18,0,0,36,36,0,0,36,36,36,0,0,36,0,0,12,12,0,0,36,
  36,72,72,72,72,0,72,72,0,72,0,72,72,12,0,36,0,0,72,
  0,72,72,72,0,72,72,72,72,0,0,0,36,0,0,0,72,0,0,72,
  72,0,72,72,72,72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  6,6,0,0,12,12,0,0,0,0,0,0,12,12,6,0,0,0,0,12,0,0,
  0,0,12,0,0,0,0,0,0,12,0,0,0,0,12,0,0,36,0,0,0,36,
  36,36,36,0,0,0,0,0,0,36,36,36,36,0,0,0,12,12,0,0,0,
  0,12,12,12,12,0,0,0,0,12,12,12,0,0,0,0,12,12,12,36,
  36,36,0,0,0,36,36,0,0,0,36,0,0,0,0,0,0,12,12,12,6,
  0,0,6,6,6,12,6,0,0,12,0,0,6,6,12,6,18,6,12,6,12,0,
  0,12,18,0,18,0,0,0,36,36,0,0,36,36,36,0,36,0,12,6,
  6,6,12,6,6,6,6,0,18,6,6,6,6,18,6,12,6,12,0,12,0,
  18,0,0,36,0,0,0,36,0,36,36,36,36,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,12,12,12,12,12,0,0,0,0,0,12,12,24,
  12,36,12,12,12,12,0,12,24,36,0,0,0,0,0,0,72,0,0,72,
  72,72,0,72,6,6,0,6,0,0,12,0,18,6,6,12,6,18,6,12,6,
  0,0,12,12,0,0,18,36,0,0,36,0,0,36,36,36,0,0,36,6,0,
  6,0,0,12,0,18,6,6,12,6,18,6,12,6,0,0,12,12,0,0,18,
  36,0,0,36,0,0,36,36,36,0,0,36,12,6,12,12,12,0,0,6,
  6,12,6,18,6,12,6,12,0,12,12,0,0,18,0,0,0,36,0,0,0,
  36,36,0,0,36,0,0,0,0,0,36,12,12,24,12,36,12,12,12,
  12,0,12,24,0,0,36,72,0,0,72,0,0,72,72,72,0,0,72,12,
  12,0,0,18,6,6,12,6,18,6,12,6,12,0,12,12,18,0,18,36,
  0,0,36,36,0,36,36,36,36,0,36,6,6,0,0,12,6,6,6,36,
  12,12,6,0,0,0,12,0,0,36,72,0,0,72,0,0,0,72,72,72,0,
  72,6,0,0,12,6,6,6,36,12,12,6,0,0,0,12,0,0,36,72,0,
  0,72,0,0,0,72,72,72,0,72,0,0,12,6,6,6,36,12,12,6,
  24,0,24,12,36,0,0,0,0,0,0,72,0,72,72,72,72,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,18,18,36,18,18,18,36,18,36,0,0,36,0,0,0,0,0,0,
  0,0,0,0,36,36,0,0,0,6,0,6,0,6,12,6,0,0,12,12,0,0,
  18,36,0,0,36,0,0,36,0,0,0,0,36,6,6,0,6,0,6,12,0,
  12,12,18,0,18,36,0,0,36,36,0,36,0,0,36,0,36,6,0,0,
  12,6,0,0,0,12,0,0,36,72,0,0,72,72,0,72,0,0,0,0,72,
  0,6,0,6,12,0,12,12,18,0,18,36,0,0,36,36,0,36,0,0,
  36,0,36,18,0,0,0,0,36,0,0,0,0,36,0,0,0,0,0,36,0,0,
  0,0,0,12,6,12,0,0,12,18,0,18,36,0,0,36,36,0,36,36,
  36,36,0,36,0,24,0,24,0,36,0,0,72,0,0,0,72,0,72,0,0,
  72,0,72,12,0,12,0,18,0,0,36,0,0,0,36,0,36,0,0,36,0,
  0,0,12,12,0,0,36,72,0,0,72,0,0,72,0,72,0,0,72,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,36,0,36,0,0,0,
  72,72,0,0,72,72,72,0,72,36,0,0,72,0,0,0,72,0,72,0,
  72,72,0,0,0,18,0,0,0,36,0,0,0,0,0,0,0,36,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,0,0,0,36,0,0,0,0,
  0,0,0,36,36,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,36,0,0,0,0,36,0,36,0,0,0,0,36,0,36,0,0,
  0,0,0,0,36,36,0,0,0,36,0,0,0,0,0,0,0,36,0,12,0,6,
  6,6,12,6,0,12,0,0,0,6,6,12,6,18,6,12,6,6,6,6,6,18,
  18,0,0,0,36,0,36,0,0,36,36,36,36,0,0,12,12,12,12,12,
  0,12,24,0,0,12,12,12,12,36,12,12,12,24,0,0,0,36,0,
  36,0,0,0,72,72,72,72,72,72,72,0,72,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,6,6,0,6,0,12,12,0,0,6,6,12,6,18,6,12,6,
  6,6,6,6,0,18,18,0,0,36,36,0,0,0,36,36,0,36,36,6,12,
  6,0,0,12,18,18,6,6,12,6,18,6,12,6,0,12,12,12,18,18,
  18,36,36,36,36,36,36,36,36,36,36,36,36,0,6,0,12,12,0,
  0,6,6,12,6,18,6,12,6,6,6,6,6,0,18,18,0,0,36,36,0,
  0,0,36,36,0,36,36,0,0,24,0,0,0,12,12,24,12,36,12,12,
  12,12,12,12,12,0,36,36,72,72,72,72,0,0,0,72,72,0,72,
  72,0,12,0,18,18,6,6,12,6,18,6,12,6,6,6,6,6,18,18,
  18,36,36,36,36,36,36,36,36,36,36,36,36,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,12,0,0,12,12,12,12,36,12,12,12,0,0,0,24,36,0,36,
  72,72,0,72,72,0,0,72,72,72,0,72,0,0,12,12,12,12,36,
  12,24,12,12,12,12,12,36,36,0,0,0,72,0,72,72,72,72,72,
  72,72,0,6,18,6,36,18,6,6,36,18,36,0,0,36,0,0,0,0,0,
  0,0,0,0,0,12,12,0,0,0,18,6,36,18,6,6,36,18,36,0,0,
  36,0,0,0,0,0,0,0,0,0,0,12,12,0,0,0,6,0,6,0,6,12,
  6,0,12,12,12,0,18,18,36,36,36,36,0,36,36,0,0,0,36,
  36,0,6,6,6,0,6,12,0,0,12,18,18,18,12,12,36,36,36,12,
  12,12,12,36,36,36,12,0,0,12,12,0,0,0,24,0,36,36,72,
  72,72,72,72,72,72,0,0,0,72,72,0,6,12,6,12,12,12,12,
  18,0,18,36,36,0,36,36,36,36,0,0,36,0,36,6,0,0,0,36,
  36,0,0,0,0,12,12,0,0,0,12,12,0,0,0,0,0,0,6,12,0,0,
  12,18,18,18,12,12,36,36,36,12,12,12,12,36,36,36,0,24,
  0,0,0,36,36,0,72,72,72,0,72,72,72,0,0,72,72,72,12,
  12,12,0,18,18,0,36,36,36,0,36,36,36,0,0,36,36,0,6,6,
  6,0,36,36,72,72,72,72,0,72,72,0,72,0,72,72,6,6,36,0,
  36,0,0,0,72,72,0,0,72,72,72,0,72,6,36,0,36,0,0,0,
  72,72,0,0,72,72,72,0,72,36,36,0,72,72,72,0,72,72,72,
  0,72,72,72,0,18,18,0,0,36,36,0,0,0,0,0,0,36,36,18,
  0,0,0,36,36,0,0,0,0,36,0,36,0,0,36,0,36,0,0,0,0,
  36,36,0,12,0,0,0,12,12,12,12,0,0,0,0,0,0,12,12,12,
  12,0,0,0,36,36,0,0,0,0,36,36,36,36,0,0,0,0,36,36,
  36,0,0,0,0,36,36,36,12,12,12,0,0,0,12,12,0,0,0,12,
  0,0,0,0,0,0,36,36,36,12,12,12,12,12,12,12,12,12,0,
  36,12,12,12,12,36,12,12,12,24,0,24,0,36,0,0,72,0,0,
  0,72,0,72,72,72,72,0,0,12,12,12,12,12,0,24,0,0,0,12,
  12,24,12,36,12,12,12,12,12,12,12,36,36,0,0,0,72,0,72,
  0,0,72,72,72,72,0,6,6,0,6,0,12,0,0,18,6,6,12,6,18,
  6,12,6,6,6,6,6,0,18,0,36,0,36,0,0,0,36,36,36,0,36,
  0,6,12,6,12,0,12,0,18,6,6,12,6,18,6,12,6,0,12,12,
  12,18,18,0,36,0,36,0,36,0,36,36,36,36,36,0,12,6,12,
  12,12,0,0,6,6,12,6,18,6,12,6,6,6,6,6,0,18,0,0,0,
  36,0,0,0,0,36,36,0,36,0,0,0,24,0,0,36,12,12,24,12,
  36,12,12,12,12,12,12,12,0,36,0,72,0,72,0,0,0,72,72,
  72,0,72,0,0,12,0,0,18,6,6,12,6,18,6,12,6,6,6,6,6,
  0,18,0,36,0,36,0,0,0,36,36,36,0,36,0,6,6,0,0,12,6,
  6,6,36,12,12,6,12,6,12,6,0,36,0,0,0,72,0,0,0,72,72,
  72,72,72,0,6,0,0,12,6,6,6,36,12,12,6,0,12,0,12,36,
  0,0,72,0,0,0,72,0,0,72,72,72,0,0,0,0,12,6,6,6,36,
  12,12,6,12,6,12,6,0,36,0,0,0,72,0,0,0,72,72,72,72,
  72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,18,18,36,18,18,18,36,18,36,36,0,36,0,
  0,0,0,0,0,0,0,0,0,36,36,0,0,0,6,0,6,0,6,12,6,0,
  12,12,12,0,18,0,36,0,36,0,0,0,36,0,0,0,36,0,6,6,0,
  6,0,6,12,0,12,0,18,18,0,36,0,36,0,36,0,36,0,0,36,
  36,0,6,0,0,12,6,0,12,0,12,0,36,0,72,0,72,0,72,0,72,
  0,0,0,72,0,0,6,12,6,12,12,12,12,18,0,0,36,0,0,0,36,
  0,36,0,0,36,0,0,18,0,0,0,0,36,0,0,0,0,36,0,0,0,0,
  0,36,0,0,0,0,0,12,6,12,12,0,12,18,18,0,36,0,36,0,
  36,0,36,36,36,36,36,0,0,24,0,24,0,36,36,0,72,0,72,0,
  72,0,72,0,0,72,72,0,12,0,12,0,18,18,0,36,0,36,0,36,
  0,36,0,0,36,36,0,6,6,6,0,36,0,72,0,72,0,0,0,72,0,
  72,0,72,0,6,6,36,0,0,72,0,72,0,72,0,72,0,72,72,0,0,
  6,36,0,0,0,0,0,0,72,0,0,72,72,72,72,0,36,0,0,72,0,
  72,0,72,0,72,0,72,72,0,0,18,0,0,0,36,0,0,0,0,0,0,
  0,36,0,0,0,0,0,0,36,0,0,0,0,36,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,36,36,36,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,36,0,0,0,0,36,36,0,0,0,0,0,0,0,0,0,0,0,0,
  0,36,36,0,0,0,0,0,0,0,36,36,0,0,0,36,0,0,0,0,0,0,
  36,0,0,12,6,6,6,12,0,0,0,0,0,6,12,12,6,36,6,6,12,
  12,0,0,0,36,0,0,0,0,0,0,72,72,0,72,72,72,0,72,0,12,
  0,12,0,12,24,0,36,12,12,12,12,36,12,12,12,0,0,24,0,
  0,0,36,72,0,0,72,0,0,72,72,72,0,0,72,6,6,0,0,0,0,
  0,36,6,12,12,6,36,6,6,12,0,0,12,0,0,0,36,72,0,0,72,
  0,0,72,72,72,0,0,72,6,12,24,12,24,0,0,6,12,12,6,36,
  6,6,12,12,0,12,0,0,0,36,0,0,0,72,0,72,0,72,72,0,0,
  72,0,0,0,0,0,36,6,12,12,6,36,6,6,12,0,0,12,0,0,0,
  36,72,0,0,72,0,0,72,72,72,0,0,72,24,12,0,0,36,12,12,
  12,12,36,12,12,12,0,0,24,0,36,0,36,72,0,0,72,72,72,
  72,72,72,72,0,72,0,12,0,0,24,12,12,12,72,24,12,12,0,
  0,0,0,0,0,72,0,0,0,144,0,0,144,144,144,144,0,144,0,
  0,0,12,12,6,6,36,12,6,12,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,72,72,0,0,0,0,0,24,12,12,12,72,24,12,12,0,0,0,
  0,72,0,0,0,0,0,0,144,144,0,144,144,144,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  36,36,72,36,36,36,36,36,72,0,0,0,0,0,0,0,0,0,0,0,
  216,0,72,72,0,0,0,12,0,6,0,6,6,12,0,0,12,0,0,0,36,
  72,0,0,72,0,72,72,0,0,0,0,72,12,0,0,12,0,12,24,0,
  24,0,36,0,36,72,0,0,72,72,72,72,72,0,72,0,72,6,0,0,
  6,12,0,0,0,0,0,0,72,72,0,0,72,72,72,72,0,0,0,0,72,
  0,6,6,0,12,0,12,0,36,0,0,72,0,0,0,72,72,72,0,0,72,
  0,72,36,0,0,0,0,72,0,0,0,0,72,0,0,0,0,216,72,0,0,
  0,0,0,6,12,12,0,0,0,36,0,36,72,0,0,72,72,72,72,72,
  72,72,0,72,0,12,0,12,0,36,0,0,72,0,0,0,72,72,72,0,
  0,72,0,72,24,0,24,0,36,0,0,72,0,0,0,72,72,72,0,0,
  72,0,0,0,12,0,0,0,72,144,0,0,144,0,144,144,0,144,0,
  0,144,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,0,0,
  0,0,0,0,144,144,0,144,144,144,0,144,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,36,0,0,0,72,0,0,0,0,0,0,0,72,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,0,0,0,0,72,0,0,
  0,0,0,0,216,0,72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,72,0,0,0,0,72,0,72,0,0,0,0,72,0,
  72,72,72,216,0,0,0,72,72,0,0,0,0,0,0,0,0,0,0,0,72,
  0,12,0,0,12,0,24,0,0,0,12,12,24,12,36,12,12,12,12,
  12,12,12,0,36,0,0,0,72,0,0,0,0,72,72,0,72,72,6,6,
  12,0,0,0,36,0,6,12,12,6,36,6,6,12,0,24,12,24,36,36,
  0,0,72,72,0,72,72,0,72,72,72,72,72,6,0,0,12,0,0,0,
  6,12,12,6,36,6,6,12,6,12,6,12,0,36,0,0,72,72,0,0,0,
  0,72,72,0,72,72,0,0,12,0,0,0,6,12,12,6,36,6,6,12,6,
  12,6,12,0,36,0,0,72,72,0,0,0,0,72,72,0,72,72,0,24,
  0,36,0,12,12,24,12,36,12,12,12,12,12,12,12,0,36,0,0,
  72,72,0,0,72,0,72,72,72,72,72,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  12,24,12,12,72,12,12,24,0,0,0,0,72,0,0,0,144,0,0,
  144,0,0,144,144,144,0,144,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,36,72,36,
  36,36,36,36,72,0,36,72,0,0,0,0,0,0,0,0,0,0,72,72,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,12,0,6,0,6,6,12,0,24,12,24,0,36,0,0,72,
  72,0,0,72,0,0,0,0,72,72,0,12,36,0,0,12,12,0,12,24,
  36,36,0,0,72,72,0,72,72,0,72,72,72,72,72,12,0,0,0,0,
  0,0,0,0,0,72,0,0,144,144,0,144,144,0,0,0,0,144,144,
  0,6,6,12,12,24,12,24,36,0,0,0,72,0,0,72,72,0,0,0,
  72,0,72,0,0,0,0,72,36,0,0,0,0,0,72,0,0,0,72,0,0,0,
  0,0,0,6,0,12,0,0,0,36,36,0,0,72,72,0,72,72,0,0,0,
  72,72,72,0,12,0,0,0,36,36,0,0,72,72,0,72,72,0,0,0,
  72,72,72,12,24,12,0,36,36,0,0,72,72,0,72,72,0,0,0,
  72,72,0,12,6,12,0,72,0,0,72,72,0,0,72,0,0,72,0,72,
  72,0,12,72,0,0,0,0,144,0,144,0,0,144,144,144,0,144,0,
  36,0,0,0,72,0,0,72,72,0,72,72,72,72,72,72,0,0,0,144,
  0,0,144,144,0,0,144,144,144,0,36,0,0,0,72,0,0,0,0,0,
  0,0,72,216,0,0,0,0,0,72,0,0,0,0,72,0,216,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,
  0,72,72,0,0,0,0,72,0,0,0,0,72,72,216,0,0,0,0,0,0,
  0,0,0,0,0,0,72,72,216,0,72,72,0,0,0,0,0,0,0,0,72,
  0,0,0,0,0,0,72,72,72,6,0,6,0,0,12,0,18,6,6,12,6,
  18,6,12,6,0,0,12,12,0,0,18,36,0,0,36,0,0,36,36,36,
  0,0,36,0,6,0,12,12,0,0,6,6,12,6,18,6,12,6,6,6,6,6,
  0,18,18,0,0,36,36,0,0,0,36,36,0,36,36,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,12,0,0,18,6,6,12,6,18,6,12,6,6,6,6,6,0,18,
  0,36,0,36,0,0,0,36,36,36,0,36,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,
  0,0,12,12,12,12,36,12,12,12,0,0,0,24,0,0,36,72,0,0,
  72,0,0,0,72,72,0,0,72,0,0,12,12,12,12,36,12,24,12,
  12,12,12,12,0,36,0,0,0,72,0,0,0,72,72,72,0,72,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,18,18,36,18,18,18,36,18,36,36,0,36,0,0,0,0,0,
  0,0,0,0,0,36,36,0,0,0,6,0,6,0,6,0,6,0,0,12,12,0,
  18,18,36,0,36,36,0,0,36,0,0,0,36,36,0,6,0,6,0,6,0,
  0,12,12,0,18,18,36,0,36,36,0,0,36,0,0,0,36,36,12,0,
  0,12,12,0,0,0,24,0,36,36,72,0,72,72,0,0,72,0,0,0,
  72,72,0,6,12,6,12,12,12,12,0,0,18,36,0,0,36,0,0,36,
  0,0,0,0,36,18,0,0,0,0,36,0,0,0,0,36,0,0,0,0,0,36,
  0,0,0,0,0,12,6,12,12,0,12,0,18,18,36,0,36,36,0,0,
  36,36,36,0,36,36,0,0,0,24,0,0,36,0,72,0,72,0,0,0,
  72,0,0,0,72,72,12,12,12,0,0,18,0,36,0,36,0,0,0,36,
  0,0,0,36,0,6,6,6,0,0,36,72,0,72,72,0,0,72,0,72,0,
  0,72,6,6,0,0,36,72,0,72,72,0,0,72,0,72,0,0,72,6,0,
  0,36,0,0,0,72,0,0,0,72,72,0,72,72,0,36,0,72,0,72,0,
  0,0,72,0,72,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  18,0,0,0,36,0,0,0,0,0,0,0,36,0,0,36,0,0,0,0,0,0,
  0,36,0,0,0,0,0,0,36,36,36,0,0,0,0,0,0,0,0,0,0,0,
  0,0,36,0,0,0,0,0,0,36,36,0,0,0,0,0,0,36,36,0,0,0,
  0,0,0,0,0,0,0,0,0,0,36,36,0,0,0,36,0,0,0,0,0,0,0,
  0,36,6,6,12,0,12,18,0,6,6,12,6,18,6,6,6,0,12,0,12,
  0,0,18,0,36,0,36,0,36,0,36,36,0,0,36,0,0,0,0,0,36,
  6,12,12,6,36,6,6,12,0,0,12,0,0,0,36,72,0,0,72,0,0,
  72,72,72,0,0,72,12,0,0,18,18,6,6,12,6,18,6,12,6,0,
  12,12,0,6,6,6,36,36,12,12,12,36,36,36,36,12,12,12,0,
  12,0,36,12,12,12,12,36,12,24,12,0,12,24,12,0,0,36,72,
  0,0,72,0,0,72,72,72,72,72,72,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,12,
  12,12,12,36,12,24,12,0,12,24,12,36,36,0,0,72,72,0,72,
  72,72,72,72,72,72,0,18,18,18,36,18,18,18,36,18,0,0,
  36,36,0,0,0,36,0,0,0,0,0,36,36,36,0,0,0,18,18,36,
  18,18,18,36,18,0,36,0,36,0,0,0,0,36,0,0,0,36,0,36,
  36,0,0,0,6,0,6,0,6,6,6,0,12,12,12,0,0,18,36,36,0,
  36,0,36,36,0,0,0,0,36,12,6,18,6,0,6,0,0,12,12,18,
  18,18,36,36,36,36,36,36,36,36,36,36,36,36,0,0,0,12,
  12,0,24,0,24,0,0,36,72,72,72,72,72,72,72,0,0,0,0,72,
  0,6,6,6,0,12,12,12,0,0,18,36,36,0,36,0,36,36,0,0,0,
  0,36,18,0,0,0,36,36,0,0,0,0,36,36,0,0,0,36,36,0,0,
  0,0,0,6,6,0,12,0,12,18,18,18,36,36,36,36,36,36,36,
  36,36,36,36,36,0,0,0,12,0,36,36,0,72,72,72,0,72,72,
  72,0,0,72,72,72,0,12,12,0,18,18,0,36,36,36,0,36,36,
  36,0,0,36,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,12,12,0,0,36,72,0,72,72,72,0,72,72,72,0,0,72,12,0,
  0,36,0,72,0,72,0,72,0,72,72,72,72,72,36,36,0,72,72,
  72,0,72,72,72,0,72,72,72,0,6,6,0,0,0,12,0,0,0,0,0,
  0,0,12,6,0,0,0,12,0,0,0,0,0,0,0,12,0,0,12,0,12,0,
  0,0,0,12,12,0,36,0,0,0,36,36,36,36,0,0,0,0,0,0,36,
  36,36,36,0,0,0,12,12,0,0,0,0,12,12,12,12,0,0,0,0,
  12,12,12,0,0,0,0,12,12,12,36,36,36,0,0,0,36,36,0,0,
  0,36,0,0,0,0,0,0,12,12,12,0,0,12,0,0,0,6,12,12,6,
  36,6,6,12,6,12,6,12,0,36,0,0,72,72,0,0,0,0,72,72,0,
  72,72,12,12,0,18,0,6,6,12,6,18,6,12,6,6,6,6,6,0,18,
  0,0,36,36,0,0,36,0,36,36,0,36,0,12,12,0,0,12,12,12,
  12,36,12,24,12,12,12,12,12,0,36,36,0,0,72,72,0,0,0,
  72,72,0,72,72,12,36,0,12,12,12,12,36,12,12,12,0,24,0,
  24,0,0,36,0,72,0,72,0,72,0,72,72,0,0,72,36,0,12,12,
  12,12,36,12,24,12,12,12,12,12,0,36,0,0,72,72,0,0,72,
  0,72,72,0,72,0,0,18,18,36,18,18,18,36,18,36,0,36,36,
  0,0,0,0,0,0,0,0,0,0,36,36,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,0,6,0,6,
  6,6,0,12,0,12,0,18,18,0,36,36,36,0,36,0,0,0,0,36,
  36,12,6,18,6,0,6,12,0,12,12,0,18,18,0,36,36,36,0,36,
  0,36,36,0,36,36,12,0,0,12,12,0,24,0,24,0,36,36,0,72,
  72,72,0,72,0,0,0,0,72,72,0,6,6,6,12,12,12,12,0,0,
  18,0,36,0,36,0,36,0,0,0,0,0,36,0,0,0,0,36,0,0,0,0,
  0,0,36,0,0,0,36,0,0,0,0,0,0,6,6,0,12,0,12,0,18,18,
  0,36,36,36,0,36,0,0,0,0,36,36,0,12,0,12,0,0,36,0,0,
  72,72,0,0,72,0,0,0,0,72,72,12,12,12,0,0,18,0,0,36,
  36,0,0,36,0,0,0,0,36,0,6,6,6,0,0,36,0,72,0,72,0,
  72,0,0,72,0,72,72,6,6,0,0,36,0,0,72,72,0,0,0,72,72,
  0,0,72,6,0,0,36,0,72,0,72,0,72,0,0,72,0,72,72,0,36,
  0,0,72,72,0,0,72,0,0,72,0,72,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,18,0,0,0,36,0,0,0,0,0,0,0,36,0,0,36,0,
  0,0,0,0,0,0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,
  0,36,36,0,0,0,36,0,0,0,0,0,0,36,36,0,0,0,0,0,0,36,
  36,0,0,0,0,0,0,0,0,36,36,0,0,0,0,0,0,0,0,36,0,0,
  0,0,0,0,0,0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,12,24,12,12,72,12,12,24,0,0,0,0,0,0,72,144,0,0,
  144,0,0,0,144,144,0,0,144,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,36,
  72,36,36,36,36,36,36,72,0,72,0,0,0,0,216,0,0,0,0,0,
  72,72,0,0,0,0,0,6,0,6,6,0,0,0,12,0,0,36,0,72,72,
  72,0,0,0,72,0,0,0,72,72,0,12,0,12,0,12,12,0,12,24,
  0,36,36,72,72,72,72,0,0,72,0,72,0,72,72,12,0,0,0,0,
  0,0,0,0,0,72,0,144,144,144,144,0,0,144,0,0,0,144,0,
  0,6,6,12,12,24,12,24,0,0,36,72,72,0,72,0,0,72,0,0,
  0,0,72,36,0,0,0,0,72,0,0,0,0,72,216,0,0,0,0,72,0,
  0,0,0,0,6,12,12,24,0,24,0,36,36,72,72,72,72,0,0,72,
  72,72,0,72,72,0,0,0,12,0,0,36,0,72,72,72,0,0,0,72,
  0,0,0,72,72,12,24,12,0,0,36,0,72,72,72,0,0,0,72,0,
  0,0,72,0,0,6,0,0,36,36,72,72,72,72,0,0,72,0,72,0,
  72,72,12,12,0,0,72,144,0,144,144,0,0,144,0,144,0,0,
  144,12,0,0,72,0,72,0,72,0,0,0,72,72,0,72,72,0,72,0,
  144,144,144,0,0,0,144,0,0,0,144,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,36,0,0,0,72,0,0,0,0,0,0,0,72,0,0,72,
  0,0,0,0,0,0,0,72,0,72,0,0,0,0,0,72,72,0,0,0,0,0,
  0,0,216,216,72,0,0,0,72,0,0,0,0,0,0,72,72,0,0,0,0,
  0,0,72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,72,72,0,0,0,
  0,0,0,0,0,0,0,0,0,72,12,0,0,36,12,12,12,12,36,12,
  24,12,12,12,12,12,0,36,0,72,0,72,0,0,0,72,72,72,72,
  72,72,0,36,0,12,12,12,12,36,12,12,12,0,24,0,0,36,0,
  36,72,72,0,72,72,72,0,72,72,72,0,72,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,18,
  18,18,36,18,18,18,36,18,36,0,36,36,0,0,0,36,0,0,0,0,
  0,36,36,36,0,0,0,18,18,36,18,18,18,36,18,36,36,0,36,
  0,0,0,0,36,0,0,0,36,0,36,36,0,0,0,6,0,6,0,6,0,6,
  0,12,12,0,0,18,0,36,36,36,0,0,36,36,0,0,0,36,0,12,
  6,18,6,0,6,12,0,12,12,18,18,18,36,36,36,36,36,36,36,
  36,36,36,36,36,12,0,0,12,0,0,24,0,0,0,36,0,72,72,72,
  72,72,72,72,0,0,0,72,0,0,6,12,6,12,12,12,12,18,0,18,
  36,36,0,36,36,36,36,0,0,36,0,36,18,0,0,0,36,36,0,0,
  0,0,36,36,0,0,0,36,36,0,0,0,0,0,12,6,12,12,0,12,18,
  18,18,36,36,36,36,36,36,36,36,36,36,36,36,0,0,0,24,0,
  0,36,0,72,72,72,0,0,72,72,0,0,72,72,72,0,12,12,0,0,
  18,0,36,36,36,0,0,36,36,0,0,0,36,0,6,6,6,0,36,0,72,
  72,72,0,0,72,72,0,72,0,72,0,6,6,36,0,36,72,0,72,72,
  72,0,72,72,72,72,0,72,6,36,0,36,0,72,0,72,72,72,0,
  72,72,72,72,72,0,36,0,72,72,72,0,0,72,72,0,72,0,72,
  0,6,6,0,0,12,0,0,0,0,0,0,0,12,0,6,0,0,0,12,12,0,
  0,0,0,12,0,12,0,0,12,0,0,0,0,0,0,0,12,0,36,0,0,0,
  36,36,36,36,0,0,0,0,0,0,36,36,36,36,0,0,0,12,12,0,
  0,0,0,12,12,12,12,0,0,0,0,12,12,12,0,0,0,0,12,12,
  12,36,36,36,0,0,0,36,36,0,0,0,36,0,0,0,0,0,0,12,12,
  12,6,0,0,12,6,6,6,36,12,12,6,0,0,0,12,0,0,36,72,0,
  0,72,0,0,0,72,72,72,0,72,0,0,12,6,6,6,36,12,12,6,
  12,6,12,6,0,36,0,0,0,72,0,0,0,72,72,72,72,72,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,36,36,36,36,36,36,72,36,72,0,0,72,0,0,0,0,0,0,
  0,0,0,0,72,72,0,0,0,0,0,12,0,12,0,12,0,12,24,12,0,
  36,36,72,0,72,72,0,0,72,0,0,0,72,72,6,6,0,0,0,6,0,
  0,0,12,0,36,36,72,0,72,72,0,0,72,0,0,0,72,72,6,0,0,
  0,6,0,0,0,12,0,36,36,72,0,72,72,0,0,72,0,0,0,72,72,
  0,12,12,6,24,12,24,12,0,0,36,72,0,0,72,0,0,72,0,0,
  72,0,72,36,0,0,0,0,72,0,0,0,0,72,0,0,0,0,0,72,0,0,
  0,0,0,0,12,24,12,0,12,0,36,36,72,0,72,72,0,0,72,72,
  72,72,72,72,0,0,0,0,0,0,72,0,144,0,144,0,0,0,144,0,
  0,0,144,144,24,12,24,0,0,36,0,72,0,72,0,0,0,72,0,0,
  72,72,0,0,12,12,0,0,72,144,0,0,144,0,0,144,0,144,0,
  144,144,0,6,0,0,36,0,0,0,72,0,0,0,0,0,0,0,72,12,0,
  0,72,0,0,0,144,0,0,0,144,144,144,0,144,0,72,0,72,0,
  72,0,0,0,72,0,72,72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,36,0,0,0,72,0,0,0,0,0,216,0,72,0,0,72,0,0,0,0,
  0,0,216,72,0,0,0,0,0,0,72,72,72,0,0,0,0,0,0,0,0,0,
  0,0,0,0,72,0,0,0,0,0,216,0,72,0,0,0,0,0,216,72,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,72,72,0,0,0,72,0,0,0,0,
  0,0,72,72,72,36,0,12,6,6,6,36,12,12,6,0,12,0,0,36,
  0,0,0,72,0,0,72,72,0,72,72,72,0,0,0,36,36,36,36,36,
  36,72,36,0,0,0,72,0,0,0,216,0,0,0,0,0,0,72,72,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,12,0,0,0,12,12,12,0,24,0,24,0,0,36,72,72,0,
  72,0,72,0,0,0,0,0,72,6,6,36,12,0,6,0,0,0,12,36,0,
  36,72,72,0,72,72,72,0,72,72,72,0,72,6,0,0,6,6,0,12,
  0,12,0,0,36,72,72,0,72,72,72,0,0,0,0,0,72,0,0,6,6,
  0,12,0,12,0,0,36,72,72,0,72,72,72,0,0,0,0,0,72,0,0,
  0,0,72,0,0,0,0,0,216,72,0,0,0,72,0,0,0,0,0,0,12,
  12,0,24,0,24,36,0,36,72,72,0,72,72,72,0,0,72,72,0,
  72,0,0,0,0,0,72,0,0,72,72,0,0,72,72,0,0,0,72,0,72,
  0,12,0,0,36,0,0,72,72,0,0,72,72,0,0,0,72,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,0,0,72,144,
  0,0,144,144,0,0,144,144,0,0,144,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,72,0,0,0,144,0,0,144,144,0,0,144,144,
  0,0,0,36,0,0,0,72,0,0,0,0,0,0,0,72,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,72,0,0,0,0,72,0,0,72,0,0,0,
  216,0,216,72,0,0,0,0,0,0,0,0,72,72,0,0,0,0,0,0,0,
  0,0,0,0,0,72,0,0,0,0,72,0,72,0,0,0,0,72,0,72,0,72,
  72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,0,0,36,36,
  36,36,36,36,72,36,72,0,72,36,0,0,0,0,0,0,0,0,0,216,
  72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,12,0,12,0,12,0,0,0,12,24,12,0,
  36,0,0,72,72,0,0,72,72,0,0,0,72,0,6,6,36,12,0,6,24,
  0,24,12,36,36,0,0,72,72,0,72,72,72,72,72,72,72,0,6,
  0,0,0,6,0,12,0,0,0,36,0,0,72,72,0,72,72,72,0,0,0,
  72,0,0,12,12,6,24,12,24,12,36,0,0,0,72,0,0,72,72,72,
  0,0,72,0,0,0,0,0,0,72,0,0,0,0,0,0,72,0,0,0,72,216,
  0,0,0,0,0,0,0,24,12,0,12,36,36,0,0,72,72,0,72,72,
  72,72,0,72,72,0,0,0,0,0,0,0,72,0,0,144,144,0,0,144,
  144,0,0,144,144,0,0,12,0,0,0,36,0,0,72,72,0,72,72,
  72,0,0,0,72,0,12,12,0,0,72,0,0,144,144,0,0,144,144,
  0,144,0,144,0,12,6,72,0,0,0,0,72,0,72,0,72,72,72,72,
  0,0,0,72,0,0,0,144,0,0,144,144,0,144,0,144,144,0,0,
  0,0,0,72,0,0,0,72,0,0,0,0,0,0,36,0,0,0,72,0,0,0,
  0,0,0,0,72,0,0,0,0,0,0,72,0,0,0,0,72,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  216,72,72,0,0,0,0,72,0,0,0,0,72,72,0,0,0,0,0,0,0,
  0,0,0,0,0,0,72,72,0,72,72,72,0,0,0,72,216,0,0,0,0,
  0,0,0,0,0,0,72,0,0,18,6,36,18,6,6,36,18,36,0,0,36,
  0,0,0,0,0,0,0,0,0,0,12,12,0,0,0,0,0,18,0,18,0,18,
  0,0,36,36,0,0,0,36,0,0,0,0,0,36,0,0,0,0,0,0,0,6,
  6,0,0,0,0,36,0,0,0,0,12,0,0,0,0,0,12,0,0,0,0,0,
  36,0,0,0,36,0,0,0,72,0,0,0,72,0,0,0,0,0,72,0,0,0,
  0,0,0,18,0,18,36,0,36,36,0,0,0,36,0,0,0,0,0,36,0,
  0,0,0,0,6,0,0,0,0,36,0,0,0,0,12,0,0,0,0,0,12,0,0,
  0,0,0,36,18,36,0,0,36,0,0,0,12,0,0,0,0,0,12,12,12,
  0,0,0,0,0,0,72,0,0,0,0,72,0,0,0,0,0,72,0,0,0,0,0,
  36,0,36,0,0,0,0,36,0,0,0,0,0,36,0,0,0,0,0,0,36,36,
  0,0,0,72,0,0,0,0,0,72,0,216,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,36,0,0,0,0,0,0,0,0,0,0,72,72,
  0,0,0,0,0,0,72,0,0,0,0,0,72,0,216,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,12,12,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,12,0,0,0,12,0,0,
  0,0,0,0,0,0,0,18,0,18,0,0,36,18,0,36,0,36,0,0,0,0,
  36,0,0,0,36,0,0,0,0,0,0,36,18,6,6,0,18,36,0,0,36,
  0,0,0,0,12,0,0,0,12,0,12,12,0,0,0,0,0,0,0,0,0,72,
  0,0,0,0,0,0,72,0,0,0,72,0,0,0,0,0,0,0,0,36,18,36,
  36,0,36,0,0,0,0,36,0,0,0,36,0,0,0,0,0,0,6,0,0,0,
  36,0,0,0,0,0,0,12,0,0,0,12,0,0,0,0,0,0,0,0,0,36,
  0,0,0,0,0,0,12,0,0,0,12,0,0,0,0,0,0,0,72,0,0,0,0,
  0,0,0,72,0,0,0,72,0,0,0,0,0,0,36,36,0,0,0,0,0,0,
  36,0,0,0,36,0,0,0,0,0,0,36,0,36,0,0,0,0,72,0,0,0,
  72,0,0,216,0,0,0,0,36,0,0,0,0,0,0,0,0,0,0,72,72,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,
  0,0,0,72,0,0,216,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,0,12,12,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,12,12,0,0,0,0,0,0,0,0,12,0,0,0,0,0,0,0,0,0,0,6,
  0,6,0,6,0,0,12,12,0,18,18,36,0,36,36,0,0,36,0,0,0,
  36,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,6,6,6,0,12,12,12,0,0,18,36,36,0,36,0,36,36,
  0,0,0,0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,6,6,0,12,0,12,0,18,18,0,36,36,36,0,36,0,0,
  0,0,36,36,0,0,0,12,0,0,36,0,72,72,72,0,0,0,72,0,0,
  0,72,72,0,12,12,0,0,18,0,36,36,36,0,0,36,36,0,0,0,
  36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,12,0,
  0,36,72,0,72,72,0,0,72,0,0,0,0,72,12,0,0,36,0,72,0,
  72,0,72,0,0,0,0,72,72,0,36,0,72,72,72,0,0,72,72,0,
  0,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,18,0,0,0,36,
  0,0,0,0,0,0,0,36,0,0,36,0,0,0,0,0,0,0,36,0,36,0,
  0,0,36,36,0,0,0,0,0,0,0,0,36,36,0,0,0,0,0,36,0,0,
  0,0,0,0,36,36,0,0,0,0,0,0,36,36,0,0,0,0,0,0,0,36,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,6,0,0,
  0,6,0,0,0,12,0,36,36,72,0,72,72,0,0,72,0,0,0,72,72,
  0,6,0,6,12,0,12,12,18,0,18,36,0,0,36,36,0,36,0,0,
  36,0,36,6,0,0,0,0,36,0,0,0,0,12,0,0,0,0,0,12,0,0,
  0,0,0,0,6,12,0,0,12,18,18,18,12,12,36,36,36,12,12,
  12,12,36,36,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,12,0,12,0,18,18,0,36,0,36,0,36,0,36,0,0,36,
  36,0,0,12,12,0,36,36,72,72,72,72,0,0,72,0,72,0,72,
  72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,36,0,36,
  0,72,0,72,72,72,0,72,72,72,72,72,36,36,0,72,72,72,0,
  72,0,72,0,72,72,72,0,18,18,0,0,36,36,0,0,0,0,0,0,
  36,36,18,0,0,0,36,36,0,0,0,0,36,0,36,0,0,36,0,36,0,
  0,0,0,36,36,0,12,0,0,0,12,12,12,12,0,0,0,0,0,0,12,
  12,12,12,0,0,0,36,36,0,0,0,0,36,36,36,36,0,0,0,0,
  36,36,36,0,0,0,0,36,36,36,12,12,12,0,0,0,12,12,0,0,
  0,12,0,0,0,0,0,0,36,36,36,0,0,6,6,0,12,0,12,0,0,
  36,72,72,0,72,72,72,0,0,0,0,0,72,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,0,0,72,0,
  0,0,0,0,0,0,72,0,0,12,0,0,0,36,0,0,72,72,0,72,72,
  72,0,0,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,12,0,0,72,144,0,144,144,0,0,144,0,0,0,0,144,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,0,144,144,144,0,
  144,144,0,0,0,0,144,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  36,0,0,0,72,216,0,0,0,0,0,0,72,0,0,72,0,216,0,0,0,
  0,0,72,0,72,0,0,0,72,72,0,0,0,0,0,0,0,0,72,72,0,0,
  0,0,0,72,72,0,0,0,0,0,0,72,72,0,0,0,0,0,72,0,0,0,
  0,0,0,216,216,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,6,6,12,12,0,12,18,0,18,0,36,0,36,36,36,0,0,0,
  36,0,36,0,12,0,12,0,36,0,0,72,0,0,0,72,72,72,0,0,
  72,0,72,12,12,12,0,18,0,0,36,36,0,0,36,36,36,0,0,36,
  0,0,12,12,12,0,0,36,72,72,0,72,0,72,72,0,0,0,0,72,
  12,12,36,0,36,72,0,0,72,72,0,72,0,0,72,0,72,12,36,0,
  36,0,72,0,72,72,72,0,0,0,72,0,72,36,0,0,72,72,0,0,
  72,72,72,0,0,72,0,0,0,18,0,0,0,36,0,0,0,0,0,0,0,
  36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,0,0,0,
  36,0,0,36,0,0,0,36,36,0,0,0,0,0,0,0,0,36,36,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,36,0,0,0,0,36,0,36,0,0,0,0,
  36,0,36,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  36,0,0,0,0,36,0,0,0,0,0,0,12,0,0,0,12,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,36,0,0,0,0,72,0,0,0,0,0,72,0,
  0,0,0,0,0,0,0,0,0,72,0,0,0,72,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,12,0,0,0,12,12,0,0,0,0,0,0,0,0,12,12,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,12,0,0,0,36,36,0,0,72,72,0,72,72,0,0,0,72,72,72,
  12,12,0,0,18,18,0,0,36,36,0,36,36,0,0,0,36,36,0,12,
  0,12,0,36,36,72,72,72,72,0,72,0,0,72,0,72,72,0,12,
  36,0,36,72,0,72,72,72,0,72,72,72,72,0,72,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,36,36,0,72,72,72,0,72,72,0,
  0,72,72,72,0,18,18,0,0,36,36,0,0,0,0,0,0,36,36,18,
  0,0,0,36,36,0,0,0,0,36,0,36,0,0,36,0,36,0,0,0,0,
  36,36,0,12,0,0,0,12,12,12,12,0,0,0,0,0,0,12,12,12,
  12,0,0,0,36,36,0,0,0,0,36,36,36,36,0,0,0,0,36,36,
  36,0,0,0,0,36,36,36,12,12,12,0,0,0,12,12,0,0,0,12,
  0,0,0,0,0,0,36,36,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,12,0,0,72,0,144,144,144,0,0,0,144,0,0,
  0,144,144,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,
  0,0,0,144,0,0,144,144,0,0,0,144,144,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,36,0,0,0,72,0,0,0,0,0,0,0,72,
  216,0,0,0,0,0,72,0,0,0,0,72,0,216,0,0,0,0,0,0,0,0,
  0,0,0,0,72,0,0,0,72,72,0,0,0,0,0,0,0,0,72,72,0,0,
  0,0,0,0,72,0,0,0,0,72,0,216,0,0,0,0,0,0,0,0,0,0,
  0,0,0,72,216,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,72,72,72,12,12,0,0,36,0,72,72,72,0,0,72,72,0,0,0,
  72,0,12,0,36,0,0,72,0,72,0,72,0,72,0,0,72,0,0,0,36,
  0,0,0,72,0,0,72,72,0,0,0,72,72,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,18,0,0,0,36,0,0,0,0,0,0,0,36,0,0,0,
  0,0,0,36,0,0,0,0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  36,0,0,0,36,36,0,0,0,0,0,0,0,0,36,36,0,0,0,0,0,0,
  36,0,0,0,0,36,36,0,0,0,0,0,0,0,0,0,0,0,0,0,36,36,
  0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,0,6,
  6,0,0,36,72,0,72,72,0,0,72,0,72,0,0,72,6,0,0,36,0,
  72,0,72,0,72,0,0,72,0,72,72,0,36,0,72,72,72,0,0,72,
  72,0,72,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,0,
  0,0,72,0,0,0,0,0,0,0,72,0,0,72,0,0,0,0,0,0,0,72,
  0,72,0,0,0,72,72,0,72,0,0,0,0,0,0,72,72,0,72,0,0,
  0,72,0,0,0,0,0,0,72,72,0,0,0,0,0,0,72,72,0,0,0,0,
  0,0,0,72,0,216,0,0,0,0,216,0,0,0,0,0,0,0,0,0,0,0,
  0,72,6,36,0,36,0,0,0,72,72,0,0,72,72,72,0,72,36,0,
  0,72,0,72,0,72,0,72,0,72,72,0,0,0,36,0,0,216,72,0,
  0,0,0,0,0,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,216,
  0,72,0,0,0,0,72,0,0,0,0,0,0,0,72,72,72,0,0,0,0,0,
  0,0,0,0,0,0,0,0,72,72,0,0,0,0,216,0,216,72,0,0,0,
  0,72,0,0,0,0,0,0,0,0,72,0,0,0,0,0,0,72,72,0,0,0,
  72,0,0,0,0,0,0,0,72,0,36,0,0,0,72,0,0,72,72,0,0,
  72,72,72,0,0,36,0,0,0,72,0,0,0,0,0,0,216,72,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,0,0,0,0,72,216,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,0,72,72,0,0,0,0,0,
  0,0,0,0,0,0,0,72,0,0,0,0,72,216,0,0,0,0,0,0,216,
  72,0,72,72,0,0,0,0,0,0,0,0,72,0,0,0,0,0,0,72,72,
  72,36,0,0,0,72,0,0,0,0,0,0,0,72,0,0,0,0,0,0,72,0,
  0,0,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,72,0,0,0,72,
  0,0,72,0,0,0,0,0,0,0,72,0,72,0,0,0,0,72,0,0,0,0,
  72,72,0,0,0,0,0,0,0,0,0,0,0,0,0,72,72,0,72,0,216,
  0,0,0,0,216,0,0,0,0,0,0,0,0,0,0,72,0,0,6,0,0,0,
  12,0,0,0,0,0,0,0,12,0,0,12,0,0,0,0,0,0,0,12,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,0,0,
  0,0,0,0,12,12,0,0,0,0,0,0,12,12,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,0,0,0,
  0,12,0,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,0,0,0,0,12,0,
  12,0,0,0,0,12,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,12,0,0,0,0,12,12,0,0,0,0,0,0,0,0,0,0,0,
  0,0,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  12,0,0,0,0,0,12,12,12,12,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,12,12,0,0,0,12,
  12,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,12,12,0,0,0,12,
  12,0,0,0,12,0,0,0,0,0,0,0,0,0,12,0,0,0,0,12,12,12,
  0,0,0,0,12,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,12,12,12,0,0,0,0,12,12,12,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,12,12,12,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,12,12,12,12,12,0,0,0,12,0,0,0,0,0,0,0,
  0,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12 ],

sizesset := 
[ infinity,4,6,8,12,16,24,32,36,48,72,80,96,120,128,144,192,
  200,240,288,336,384,432,480,512,576,648,672,720,768,864,960,
  1008,1152,1296,1440,1536,1728,2160,2304,2400,2688,2880,3072,
  3456,3840,4032,4320,4608,4800,5184,5760,6480,6912,7200,7680,
  8064,8192,8640,9216,10080,10368,11520,12288,12960,13824,14400,
  15360,15552,17280,18432,21504,21600,23040,24192,25920,27648,
  28224,28800,30240,34560,36864,38880,40320,41472,43200,46080,
  50400,51840,55296,56448,57600,60480,62208,69120,73728,77760,
  80640,86400,90720,92160,103680,110592,120960,124416,129600,138240,
  155520,161280,172800,184320,193536,207360,233280,241920,259200,
  276480,294912,302400,311040,322560,345600,362880,373248,387072,
  414720,483840,518400,552960,604800,622080,645120,691200,725760,
  777600,829440,921600,967680,1036800,1088640,1105920,1119744,
  1161216,1209600,1244160,1451520,1555200,1658880,1814400,1935360,
  2073600,2177280,2211840,2332800,2419200,2488320,2903040,3110400,
  3265920,3628800,3732480,3870720,4064256,4147200,4354560,4423680,
  4665600,4838400,4976640,5443200,5806080,6220800,6350400,6531840,
  6773760,7257600,7464960,7962624,8709120,8847360,9676800,9953280,
  10077696,10368000,11612160,12441600,12700800,13996800,14515200,
  17418240,18662400,19353600,19906560,21772800,23224320,24883200,
  26127360,29030400,32659200,34836480,36288000,37324800,38102400,
  40642560,43545600,49766400,52254720,58060800,60963840,62208000,
  65318400,69672960,74649600,76204800,87091200,93312000,101606400,
  104509440,116121600,121927680,124416000,145152000,149299200,
  152409600,154828800,162570240,174182400,185794560,195955200,
  203212800,207360000,209018880,223948800,248832000,261273600,
  265420800,279936000,290304000,313528320,318504960,348364800,
  373248000,398131200,435456000,447897600,522547200,559872000,
  597196800,609638400,668860416,696729600,746496000,783820800,
  1045094400,1119744000,1161216000,1219276800,1306368000,1321205760,
  1343692800,1393459200,1524096000,1567641600,1592524800,1625702400,
  1741824000,1828915200,1866240000,2090188800,2239488000,2286144000,
  2395008000,2438553600,2612736000,2786918400,2799360000,2874009600,
  3135283200,3483648000,3657830400,3919104000,3981312000,4478976000,
  4702924800,4877107200,5016453120,5225472000,5486745600,5643509760,
  6227020800,6270566400,6718464000,6967296000,7315660800,7838208000,
  8360755200,9405849600,10450944000,11147673600,11197440000,12541132800,
  13168189440,14370048000,14631321600,14929920000,15676416000,
  16721510400,18811699200,19160064000,19595520000,19906560000,
  20901888000,23514624000,24385536000,26127360000,26873856000,
  29262643200,31352832000,36578304000,37623398400,39191040000,
  41803776000,43893964800,47029248000,48771072000,50164531200,
  54867456000,56435097600,57480192000,62705664000,65840947200,
  67184640000,75246796800,78033715200,94058496000,100329062400,
  104509440000,109734912000,131681894400,134369280000,141087744000,
  146313216000,156764160000,188116992000,200658124800,201553920000,
  219469824000,235146240000,250822656000,258660864000,263363788800,
  268738560000,275904921600,282175488000,313528320000,322486272000,
  329204736000,376233984000,413857382400,431101440000,438939648000,
  468202291200,470292480000,526727577600,548674560000,627056640000,
  658409472000,877879296000,940584960000,1053455155200,1128701952000,
  1207084032000,1316818944000,1343692800000,1404606873600,1410877440000,
  1551965184000,1609445376000,1755758592000,1843546521600,1881169920000,
  2149908480000,2257403904000,2351462400000,2508226560000,2615348736000,
  2633637888000,2809213747200,3009871872000,3160365465600,3950456832000,
  4389396480000,4514807808000,5267275776000,5643509760000,5794003353600,
  6449725440000,6584094720000,6772211712000,8427641241600,8465264640000,
  8778792960000,9217732608000,10534551552000,11466178560000,
  13436928000000,13544423424000,14485008384000,15049359360000,
  15801827328000,17557585920000,18059231232000,18106260480000,
  21069103104000,22574039040000,23044331520000,23410114560000,
  23702740992000,26336378880000,28443289190400,29496744345600,
  31603654656000,39504568320000,45148078080000,47087773286400,
  47405481984000,52438656614400,52672757760000,56435097600000,
  57940033536000,60197437440000,63207309312000,64497254400000,
  67722117120000,70230343680000,79009136640000,90296156160000,
  96745881600000,101131694899200,105345515520000,110361968640000,
  110612791296000,142216445952000,180592312320000,193491763200000,
  210691031040000,217275125760000,225740390400000,252829237248000,
  253957939200000,260730150912000,269684519731200,270888468480000,
  276531978240000,284432891904000,288947699712000,316036546560000,
  372471644160000,379243855872000,386983526400000,421382062080000,
  474054819840000,526727577600000,564915326976000,651825377280000,
  737418608640000,1083553873920000,1106127912960000,1160950579200000,
  1218998108160000,1264146186240000,1516975423488000,1580182732800000,
  1757514350592000,1769804660736000,1883051089920000,1896219279360000,
  2031663513600000,2370274099200000,2708884684800000,2844328919040000,
  2902376448000000,3539609321472000,3687093043200000,4424511651840000,
  4977575608320000,5214603018240000,5688657838080000,6083703521280000,
  6320730931200000,7964120973312000,8090535591936000,8126654054400000,
  8532986757120000,9481096396800000,10113169489920000,11377315676160000,
  12189981081600000,13273534955520000,14158437285888000,14221644595200000,
  16591918694400000,17382010060800000,17698046607360000,18962192793600000,
  19910302433280000,20858412072960000,22754631352320000,28443289190400000,
  29865453649920000,30474952704000000,30971581562880000,32506616217600000,
  34764020121600000,34828517376000000,36502221127680000,41716824145920000,
  45509262704640000,50565847449600000,56886578380800000,59730907299840000,
  63207309312000000,63712967786496000,72814820327424000,79641209733120000,
  85329867571200000,97339256340480000,113773156761600000,
  142216445952000000,156438090547200000,158176291553280000,
  159282419466240000,176980466073600000,187725708656640000,
  204791682170880000,208584120729600000,227546313523200000,
  292017769021440000,344163799203840000,417168241459200000,
  424753118576640000,426649337856000000,477847258398720000,
  485432135516160000,573416710078464000,573606332006400000,
  682638940569600000,796412097331200000,849506237153280000,
  910185254092800000,1365277881139200000,1462797729792000000,
  2184444609822720000,2336142152171520000,2389236291993600000,
  2730555762278400000,2925595459584000000,3716589787545600000,
  4095833643417600000,6257523621888000000,6371296778649600000,
  7008426456514560000,7509028346265600000,7592461994557440000,
  7747410198528000000,7964120973312000000,10922223049113600000,
  11468334201569280000,13652778811392000000,15768959527157760000,
  16686729658368000000,16724654043955200000,17521066141286400000,
  17839630980218880000,18772570865664000000,21503126627942400000,
  27305557622784000000,34131947028480000000,38227780671897600000,
  40958336434176000000,41299655904460800000,43688892196454400000,
  47784725839872000000,56317712596992000000,57341671007846400000,
  65825897840640000000,75260943197798400000,92156256976896000000,
  100347924263731200000,122875009302528000000,123898967713382400000,
  131066676589363200000,131407996059648000000,153593761628160000000,
  163833345736704000000,170830394877542400000,184312513953792000000,
  210252793695436800000,210642873090048000000,229366684031385600000,
  245750018605056000000,491500037210112000000,567682542977679360000,
  585704211008716800000,1070377858813132800000,1228750093025280000000,
  1238989677133824000000,1337972323516416000000,1351625102327808000000,
  1638333457367040000000,1720250130235392000000,2064300156282470400000,
  2293666840313856000000,2890020218795458560000,4013916970549248000000,
  4281511435252531200000,4816700364659097600000,5529375418613760000000,
  8257200625129881600000,8410111747817472000000,10321500781412352000000,
  11009600833506508800000,14745001116303360000000,16588126255841280000000,
  18062626367471616000000,19266801458636390400000,26491852005625036800000,
  26541002009346048000000,30964502344237056000000,37157402813084467200000,
  41286003125649408000000,44153086676041728000000,81097506139668480000000,
  144501010939772928000000,176940013395640320000000,
  208150265758482432000000,275240020837662720000000,
  330288025005195264000000,385336029172727808000000,
  619290046884741120000000,778536058940817408000000,
  851523814466519040000000,1040751328792412160000000,
  1061640080373841920000000,1083757582048296960000000,
  1105875083722752000000000,1618411322525456793600000,
  1816584137528573952000000,2312016175036366848000000,
  2890020218795458560000000,2998805560364630016000000,
  4768533361012506624000000,4932301173410915942400000,
  6192900468847411200000000,6502545492289781760000000,
  6936048525109100544000000,7077600535825612800000000,
  8859492052209185587200000,9289350703271116800000000,
  9537066722025013248000000,10260073516345196544000000,
  11147220843925340160000000,13872097050218201088000000,
  14862961125233786880000000,15259306755240021196800000,
  15606109181495476224000000,18165841375285739520000000,
  20808145575327301632000000,24276169837881851904000000,
  27248762062928609280000000,30964502344237056000000000,
  31212218362990952448000000,33441662531776020480000000,
  36331682750571479040000000,37157402813084467200000000,
  44588883375701360640000000,45517818446028472320000000,
  53506660050841632768000000,55736104219626700800000000,
  71528000415187599360000000,73470736228933435392000000,
  96430341300475133952000000,110206104343400153088000000,
  124848873451963809792000000,166465164602618413056000000,
  175170613261683916800000000,177116953409035960320000000,
  207817225333268860108800000,208081455753273016320000000,
  238426668050625331200000000,247963734772650344448000000,
  277441941004364021760000000,408731430943929139200000000,
  462403235007273369600000000,490477717132714967040000000,
  495432037507792896000000000,501624937976640307200000000,
  572224003321500794880000000,624244367259819048960000000,
  712100981911200989184000000,873942114163746668544000000,
  1062701720454215761920000000,1486296112523378688000000000,
  1487782408635902066688000000,1618411322525456793600000000,
  1634925723775716556800000000,1747884228327493337088000000,
  1820712737841138892800000000,1948286487499395563520000000,
  1969934115138277736448000000,2179900965034288742400000000,
  2247279722135348576256000000,2663442633641894608896000000,
  2675333002542081638400000000,3329303292052368261120000000,
  3433344019929004769280000000,3932739513736860008448000000,
  3939868230276555472896000000,4046441166344898478080000000,
  4993954938078552391680000000,5785820478028508037120000000,
  8025999007626244915200000000,10195991331910377799680000000,
  11652561522183288913920000000,12588928073073017487360000000,
  13638005412495768944640000000,13733376079716019077120000000,
  16646516460261841305600000000,23970983702777051480064000000,
  35069156774989120143360000000,35956475554165577220096000000,
  37454662035589142937600000000,46818327544486428672000000000,
  48066816279006066769920000000,48940758393169813438464000000,
  58262807610916444569600000000,62424436725981904896000000000,
  67418391664060457287680000000,70138313549978240286720000000,
  79348395127248110223360000000,80259990076262449152000000000,
  84272989580075571609600000000,112363986106767428812800000000,
  136380054124957689446400000000,160519980152524898304000000000,
  168545979160151143219200000000,262182634249124000563200000000,
  294286630279628980224000000000,409899235032340365312000000000,
  441429945419443470336000000000,453201410630628629544960000000,
  481559940457574694912000000000,491592439217107501056000000000,
  602197641590722265088000000000,629238322197897601351680000000,
  840818943656082800640000000000,873942114163746668544000000000,
  1041447686045131446681600000000,1123639861067674288128000000000,
  1310913171245620002816000000000,1412575825342219105075200000000,
  1573095805494744003379200000000,1648005129565922289254400000000,
  1785338890363082480025600000000,1887714966593692804055040000000,
  2060006411957402861568000000000,2400288952599255334256640000000,
  2499474446508315472035840000000,4017012503316935580057600000000,
  4443510127125894172508160000000,4500541786123603751731200000000,
  6696655764585335016652800000000,7855491117597562912112640000000,
  11783236676396344368168960000000,14099599441841779585843200000000,
  14832046166093300603289600000000,16833195251994777668812800000000,
  16919519330210135503011840000000,17674855014594516552253440000000,
  19262397618302987796480000000000,19776061554791067471052800000000,
  26587816090330212933304320000000,33709195832030228643840000000000,
  36256112850450290363596800000000,39655123430180005085184000000000,
  47192874164842320101376000000000,51387796312828304110387200000000,
  63825865330480198660915200000000,74160230830466503016448000000000,
  80718618590983948861440000000000,80902069996872548745216000000000,
  97082483996247058494259200000000,197760615547910674710528000000000,
  207648646325306208446054400000000,224952700185748392483225600000000,
  294580916909908609204224000000000,302034394654990848648806400000000,
  332237834120489933513687040000000,339788693986864704729907200000000,
  424910655906413630246092800000000,466150022362932304674816000000000,
  475704171581610586621870080000000,666526519068884125876224000000000,
  733179170975772538463846400000000,830594585301224833784217600000000,
  1211283770230952882601984000000000,2067958036707558436613652480000000,
  2553034613219207946436608000000000,2610440125232420906178969600000000,
  2855168886972960366133248000000000,3028209425577382206504960000000000,
  3640593149859264693534720000000000,3779205363120572993718190080000000,
  5663144899781078412165120000000000,6398654583061287608411750400000000,
  8198077813559456628001996800000000,11420675547891841464532992000000000,
  16967860814010735890163302400000000,17321357914302626221208371200000000,
  17798455399311960723947520000000000,59084515334501669617532928000000000,
  63955783068194312201384755200000000,65776379280420229261295616000000000,
  68524053287351048787197952000000000,101978557417539271259062272000000000,
  126854445755096156432498688000000000,144976509434395607351427072000000000,
  152710747326096623011469721600000000,186053187107474362767664742400000000,
  203873216392118822837944320000000000,288453633838182510132776140800000000,
  315726620546017100454218956800000000,323931888267477685175844864000000000,
  356325077094225453693429350400000000,523274588739771645284057088000000000,
  747535126771102350405795840000000000,791833504653834341540954112000000000,
  900710611543736563502835302400000000,1269284133619504399056371712000000000,
  1511682145248229197487276032000000000,1903445924648640244088832000000000000
    ,2159545921783184567838965760000000000,
  2616372943698858226420285440000000000,2687434924885740795532935168000000000
    ,2848939427583201179879866368000000000,
  3266313206697066658856435712000000000,9977102158638312703416021811200000000
    ,11337616089361718981154570240000000000,
  14678871580232555244331991040000000000,
  15983660165142115710494834688000000000,
  17125350176937981118387322880000000000,
  19571828773643406992442654720000000000,
  23921124056675275212985466880000000000,
  25332485949667590848517242880000000000,
  39195758480364799906277228544000000000,
  40815417921702188332156452864000000000,
  40828915083713333235705446400000000000,
  54420557228936251109541937152000000000,
  64246491173049740893209231360000000000,
  74421274842989745107065896960000000000,
  79145281546890461349213634560000000000,
  82201680849302309368259149824000000000,
  88886910140595876812251830681600000000,
  89508496924795847978771074252800000000,
  112130269015665352560869376000000000000,
  137002801415503848947098583040000000000,
  181401857429787503698473123840000000000,
  182670401887338465262798110720000000000,
  195718287736434069924426547200000000000,
  274005602831007697894197166080000000000,
  383734698409165873208308531200000000000,
  384831083261763489988903698432000000000,
  720568489234989250802268241920000000000,
  922671927900332043929439436800000000000,
  1130273111677906753813563310080000000000,
  1473541241891196953119904759808000000000,
  1924155416308817449944518492160000000000,
  1932024261761314928713581723648000000000,
  1989280676553115886711871425740800000000,
  2232638245289692353211976908800000000000,
  2938710090362557559915264606208000000000,
  4219686283597518547570636357632000000000,
  6850140070775192447354929152000000000000,
  7263330371488691648086863878553600000000,
  13124640339637304211041314406400000000000,
  15286345807342271963448119132160000000000,
  33154677942551931445197857095680000000000,
  34524705956706969934668842926080000000000,
  41100840424651154684129574912000000000000,
  45713268072306450932015227207680000000000,
  88777815317246494117719881809920000000000,
  102140583349059726301221523292160000000000,
  122290766458738175707584953057280000000000,
  136187444465412968401628697722880000000000,
  162813150610746091918382333952000000000000,
  179121376936384460794835175997440000000000,
  212742516798041560106686249697280000000000,
  231788037349485280725761018717798400000000,
  291830238140170646574918637977600000000000,
  396889123870632079341889347649536000000000,
  538635239538714964821135543238656000000000,
  690494119134139398693376858521600000000000,
  755979283563108722555979709808640000000000,
  7062291762992129647113031039057920000000000,
  7957122706212463546847485702963200000000000,
  9051227078316677284539014987120640000000000,
  10894995557233037472130295817830400000000000,
  12930324397595253263627164267315200000000000,
  17639516616472536859639526562201600000000000,
  22858966345748751305340879686860800000000000,
  23274583915671455874528895681167360000000000,
  40227675903629676820173399942758400000000000,
  47411189457849261966632935646822400000000000,
  63920027028417475764758314903142400000000000,
  75331112138582716235872331083284480000000000,
  79544922524252858729477014101688320000000000,
  89862312463042279964326113130315776000000000,
  615483441325534055348653019124203520000000000,
  775819463855715195817629856038912000000000000,
  782284626054512822449443438172569600000000000,
  853401410241286715399392841642803200000000000,
  1432282087118243438432547426533376000000000000,
  2514486298032362643587496765554688000000000000,
  2982934594659482202355388028813312000000000000,
  3509091113439696424159741195006771200000000000,
  8506887547732597391902402896986112000000000000,
  21085738601646531402062089313996242944000000000,
  25612578127118123520196592568316723200000000000,
  29296679054691343058847560997273600000000000000,
  31090706407529834567326243937474052096000000000,
  41911491480738747578503515111791001600000000000,
  76837734381354370560589777704950169600000000000,
  135596001849448889224570195949912064000000000000,
  182472737898864214056306542140352102400000000000,
  263119171166665808661549165675597004800000000000,
  281622465379624616081799637742125056000000000000,
  282398284843480331277617267598163968000000000000,
  364979238311433260162801444098513305600000000000,
  612495903436747012216973008583000064000000000000,
  1354063570916174921767036642073247744000000000000,
  1564052039133121834768341789774446592000000000000,
  2800449658031179951836371237015126016000000000000,
  3625756999808600616962974149022580736000000000000,
  4640593962263108740521453661968138240000000000000,
  4823851471388873158795068037385945088000000000000,
  5120822231828443340500793119113373286400000000000,
  5415161929859590840072435345455688187904000000000,
  11648929249793563665201712288424263680000000000000,
  15770858061258978500580779713559003136000000000000,
  17972633699681498226311213244997435392000000000000,
  33921681975398857393067386183891455836160000000000,
  35335085391040476451111860608220266496000000000000,
  47249683979406198085309346376402862080000000000000,
  75796368050297442761850409812146257920000000000000,
  77961978566020226840760421521064722432000000000000,
  454200712164258580816726455750499290316800000000000,
  857865356079471907449174281122165555200000000000000,
  1142539659926419166257041303656584760524800000000000,
  1488365045351295239687244410856690155520000000000000,
  1563300091037384756963164702375516569600000000000000,
  3795432948296375652371176076519323729920000000000000,
  6444042696532913099686198061092268943605760000000000,
  7786297922815861385429596384294273548288000000000000,
  84458810113188579077490456647820115968000000000000000,
  97596847241906802489544527681925467340800000000000000,
  168564665816212783660934857498414465155072000000000000,
  216057091502526702866991900682491556528128000000000000,
  310694915725694024449155977945520109780992000000000000,
  447766202075264917588489497627367217037312000000000000,
  1047132734300268252072280599147516644032512000000000000,
  1367722217248081930088477010934503499313971200000000000,
  1786144300585946869061776787238838459170816000000000000,
  8930190272107771438123466465140140933120000000000000000,
  10314369764284476011032603767236862777753600000000000000,
  12274952446751773104038801177538084628070400000000000000,
  34067806143908267855683676462837449799761920000000000000,
  46175889589011694626312979442822244256972800000000000000,
  63110610881590066202654010647406375754058956800000000000,
  228935657287063559990194305830267662654400102400000000000,
  229957691471380808025864816124152786148392960000000000000,
  294747358512920638858548950825799007642583040000000000000,
  510806883564564526260662281806016061374464000000000000000,
  517404805810606818058195836279343768833884160000000000000,
  690156972798674992976391812342982003860176896000000000000,
  1363964738629268518192629546618749368821350400000000000000,
  1485760406521930473017791733137690947747840000000000000000,
  1605393062631488511104938599961764764319744000000000000000,
  11833239540298137412997626996388695453886054400000000000000,
  53045330216320162342453390802932437142732800000000000000000,
  72802597552691397798061590853972354887718010880000000000000,
  207937694447975036382417291947495153599512576000000000000000,
  241455576044949848427158056930360425455812608000000000000000,
  322314905573834980963554411520682561886801100800000000000000,
  1282954865963512744413393069594477862356988723200000000000000,
  1576204097856370538175757898144278132241203200000000000000000,
  1930850019874053909265303425226740711995473920000000000000000,
  3229934632697798967129513681765155778065627873280000000000000,
  7586095189194060692402347679557505730684439756800000000000000,
  14316895332547614542033842434459018168203476992000000000000000,
  95273270910708562124307559219579597649231216640000000000000000,
  150606301550176204922693667167685775535646965760000000000000000,
  203309856010601506784756974959897543417772847923200000000000000,
  367137339493195107604588408425255983952282255360000000000000000,
  4602656851594363528159556996515346983743519719424000000000000000,
  9019644059504997161481320733709181445968190504960000000000000000,
  40892835873780691346340679469040198201721271353344000000000000000,
  414069301492744803877398730844932063579189385625600000000000000000,
  680768353403510937841938168684593013872464593577574400000000000000,
  718903650853486927990900663458197713525244790256435200000000000000,
  5662748757492081648174479072163034297306543578635304960000000000000,
  15256745311454177935034740776449179401805827057647616000000000000000,
  28655640449971978674261065748448779945672889136578560000000000000000,
  40820331972159758582246891549129552601181753599590400000000000000000,
  729963707020463649959991442896016139887171633183457280000000000000000,
  2338296965629187691937956060053684758845187810098413568000000000000000,
  2473198713646256212626684294287551187240102491450245120000000000000000,
  29920085278378455955430818088688800077193521270672588800000000000000000,
  99726314446395706650897740034921186820221957304372690944000000000000000,
  123656465633097235526954253805291186522329149938914557952000000000000000,
  606181929786572414946366339504573856127549040953917440000000000000000000,
  746044378095708569778227724976284260145274325926234030080000000000000000,
  1025253284929720757234334580177384855801351578273919795200000000000000000,
  5999160088288458920039914416517824698032676417763803136000000000000000000,
  505656759763632489467760588203398691653063824995726131200000000000000000000,
  4622251909777153061915274021271721883894813455490140004679680000000000000000
    ,
  3194581314442596224725635569245344117534295586156452616601600000000000000000\
0,
  1131746512366968991284185526785755425980983079853291491819520000000000000000\
00,
  2643469901303426538526387888494414274941118536723442213847040000000000000000\
00,
  1096791358138976952791117509843842061499746685999560822829875200000000000000\
000,
  7156462691890231601481118341613398138747418481416086749184000000000000000000\
000,
  2093089677222789046440236424969549465808470468049707754397368320000000000000\
0000,
  6486757921410452314359132711818673300992511954894920379714974515200000000000\
000000,
  1855568540825824336669747112861188231689509220538599635681280000000000000000\
0000000,
  2797081193180344584086019518304159778919113988503519992813660930048000000000\
0000000,
  2431537015898160210772036616693301058805932882593780962596749312000000000000\
000000000,
  1300082850775315916477872492979237259483130817065255621366032787046400000000\
0000000000,
  3627342905136471469983198081211543622726109874916290402658990161920000000000\
0000000000,
  1174054220765499889262645006348562510234506061097595122625692390517637120000\
000000000000000,
  5531835195610958887067794419326748149270595184868435132170851295111938048000\
0000000000000000,
  5250696046263516092648170987414794689920778272106519961077281857522892800000\
00000000000000000,
  1274840690433857032419929995270250529386172567373975523344266007490750054400\
0000000000000000000,
  1266510491555829687838099172928283249894580032083586792234364527268226662400\
00000000000000000000,
  8957147548733264629940172723773870603298947723338970166011042417045250087321\
60000000000000000000,
  5184259261705608042824431380989661746964670604319535906397749155347672705531\
904000000000000000000000,
  7046403825746979717790151762110084990322572542137773789522100824437770158080\
000000000000000000000000,
  1245907893097207968456281395656302939049053196982972451492656167551971596907\
3152000000000000000000000,
  1806405821099057657625083163216568566706783862903218647666842205749424999628\
8000000000000000000000000,
  1959631333324497649212172216839366354533597776141513874700990752612321808560\
9472000000000000000000000,
  1533746864272984559336263031653558909370031214023943145154280516815744569180\
16000000000000000000000000,
  1158593030939257111194763446896099949300292515915388894364778767380931149824\
000000000000000000000000000,
  4145980273981389153508649245319313335495884320904923764117608120335024032361\
48019200000000000000000000000,
  6218970410972083730262973867978970003243826481357385646176412180502536048542\
22028800000000000000000000000,
  4859136937852978343853298304306116819318132901045030305518791898446326180099\
420009267200000000000000000000,
  3064650049008667159126739690516710727952538530174789215681494517974144286569\
9278028800000000000000000000000,
  1579760477436815558200288076989831408287186817989230475700860826308758898329\
2636535193600000000000000000000000,
  5932397347653089233075558846167571046393009978246902048226886947495715263034\
7923456000000000000000000000000000,
  3614660057009566039293823585327592555235030662561570980158052623494858627623\
2340225130496000000000000000000000000,
  6274176916566953685848959150207403326259616452643266277336548120253731177999\
0400131249902199480569036800000000000000000000000000000000,
  4574037988963764811994905915998328669681764183711672043658607811791123975868\
4309958174081947270879023862906880000000000000000000000000000,
  4794962122703154283780823138328748958614818005487963158725675558725364909900\
7732639759894263022235540258816000000000000000000000000000000,
  5775752629901609642380832200293019921982815338004535532650810056291088317331\
274781843001583802779807034898851700296617295872000000000000000000000000000000\
000,
  2768668094367193327342537963950285098848312852505394350744979631565176762621\
436932021290092862497226231606797746451741933568000000000000000000000000000000\
00000000,
  1651369432361955908463145725927301423425176806747045465707409579192756037770\
415756478460527001118221058623871480092777821767238028727627920616944245536408\
256258873753600000000000000000000000000000000000000000000,
  7065115985045946673536986139001118402705538174395734072025359108433701126993\
558961870245980549651087353428258861053147367587934265517055066894825367817997\
032901067145216000000000000000000000000000000000000000000000000000000,
  1422904478107155693691365040582043426558877618339840148771540855404829317884\
017469243797791886922736473977159826561752756549178369558679621864132661490617\
58424860652994560000000000000000000000000000000000000000000000000000000,
  7165033589793852697531456980706732548435609645091822296777976465116824959213\
549917461783791175492101413818415520493496100407385332345831553946154344805152\
60818409913846161473536000000000000000000000000000000000000000000000000 ],

sizespos :=
[ 14,14,13,37,138,1,13,138,1,37,1,1,95,95,141,95,141,111,
  95,141,111,259,6,29,29,10,29,29,1,1,6,10,29,1,29,1,1,1,
  1,1,10,10,1,6,1,29,1,29,1,1,22,22,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,14,13,37,138,1,13,138,1,37,1,1,95,95,141,
  95,141,111,95,141,111,259,10,6,10,10,10,6,1,1,29,29,29,
  22,29,1,1,1,1,1,6,29,1,29,1,29,22,29,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,34,87,79,1,34,79,1,87,1,1,243,
  243,197,243,197,425,243,197,425,386,29,29,11,10,11,29,1,1,
  29,11,10,1,11,1,1,1,1,1,29,11,1,29,1,11,1,10,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,7,7,1,2,7,1,7,1,1,13,
  13,13,13,13,22,13,13,22,11,4,13,13,4,13,13,1,1,13,13,8,
  22,8,22,22,1,1,1,4,13,1,13,1,8,22,8,1,1,22,22,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,7,1,7,7,1,4,1,1,37,37,90,37,
  90,118,37,90,118,92,17,15,77,17,77,15,1,1,15,77,49,1,51,
  1,1,1,1,1,17,77,1,15,1,51,1,49,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,7,4,1,7,1,1,103,103,62,103,62,334,
  103,62,334,231,17,77,15,17,15,77,1,1,77,15,139,1,20,1,1,
  1,1,1,17,15,1,77,1,20,1,139,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,7,1,7,1,1,13,
  13,13,13,13,22,13,13,22,11,4,13,13,4,13,13,1,1,13,13,8,
  22,8,22,22,1,1,1,4,13,1,13,1,8,22,8,1,1,22,22,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,7,1,1,103,103,62,103,62,334,
  103,62,334,231,17,77,15,17,15,77,1,1,77,15,139,1,20,1,1,
  1,1,1,17,15,1,77,1,20,1,139,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,37,37,90,37,90,118,
  37,90,118,92,17,15,77,17,77,15,1,1,15,77,49,1,51,1,1,1,
  1,1,17,77,1,15,1,51,1,49,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,17,17,16,31,31,16,
  15,42,42,17,38,94,17,94,38,1,1,38,94,77,1,207,250,456,1,
  1,1,17,94,1,38,1,207,1,77,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,17,17,17,8,16,31,31,34,17,38,94,17,94,38,1,
  1,38,94,77,456,207,1,1,1,1,1,17,94,1,38,1,207,250,77,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,31,11,31,15,67,
  18,67,17,66,124,17,124,66,1,1,66,124,96,1,370,1,1,1,1,1,
  17,124,1,66,1,370,1,96,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,17,31,17,17,8,34,17,38,94,17,94,38,1,1,38,94,
  77,1,207,1,1,1,1,1,17,94,1,38,1,207,1,77,1,1,456,250,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,31,31,11,31,19,17,66,124,
  17,124,66,1,1,66,124,96,1,370,1,1,1,1,1,17,124,1,66,1,
  370,1,96,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,42,
  18,18,42,17,49,183,17,183,49,1,1,49,183,25,710,178,1,1,1,
  1,1,17,183,1,49,1,178,718,25,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,17,31,42,17,38,94,17,94,38,1,1,38,94,77,
  1,207,456,250,1,1,1,17,94,1,38,1,207,1,77,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,31,67,17,66,124,17,124,66,1,
  1,66,124,96,1,370,1,1,1,1,1,17,124,1,66,1,370,1,96,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,42,17,49,183,17,
  183,49,1,1,49,183,25,1,178,1,1,1,1,1,17,183,1,49,1,178,
  1,25,1,1,710,718,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,81,
  180,417,81,417,180,1,1,180,417,122,1,167,1,1,1,1,1,81,
  417,1,180,1,167,1,122,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,10,10,4,10,10,1,1,6,6,10,17,10,17,17,1,1,1,4,
  6,1,6,1,10,17,10,1,1,7,7,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,10,10,10,4,1,1,11,11,11,15,11,1,1,1,1,1,6,11,1,11,
  1,11,15,11,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,
  4,10,1,1,11,11,11,34,11,1,1,1,1,1,6,11,1,11,1,11,34,
  11,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,10,1,1,
  10,10,6,17,6,7,7,1,1,1,4,10,1,10,1,6,17,6,1,1,17,17,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,1,1,11,11,11,34,11,
  1,1,1,1,1,6,11,1,11,1,11,34,11,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,11,11,11,15,11,1,1,1,1,1,6,11,
  1,11,1,11,15,11,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,10,11,1,11,1,1,1,1,1,10,10,1,4,1,11,1,11,1,1,15,15,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,11,1,11,1,1,1,1,1,10,
  4,1,10,1,11,1,11,1,1,34,34,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,10,15,15,1,1,1,10,11,1,11,1,10,1,4,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,7,1,1,1,1,
  1,7,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,34,34,
  1,1,1,10,11,1,11,1,4,1,10,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,7,1,1,1,17,1,1,1,1,34,1,15,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,17,1,1,1,1,34,1,15,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,10,1,10,1,10,7,10,1,1,17,17,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,10,1,11,1,11,1,1,34,34,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,11,1,11,1,1,15,15,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,10,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,7,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,3,65,65,152,1,65,152,1,54,1,1,83,83,
  147,83,147,35,83,147,35,277,9,9,9,29,29,10,1,1,9,29,10,
  1,29,1,1,1,1,1,9,9,1,10,1,29,1,29,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,65,65,152,1,65,152,1,54,1,1,83,83,
  147,83,147,35,83,147,35,277,9,9,9,29,29,10,1,1,9,29,10,
  1,29,1,1,1,1,1,9,9,1,10,1,29,1,29,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,39,39,1,13,104,398,104,408,1,89,89,
  210,89,210,311,89,210,311,169,7,48,48,13,67,67,1,1,7,13,
  67,1,67,1,1,1,1,1,13,13,172,13,1,48,172,48,1,1,116,116,
  1,298,1,1,1,1,1,1,1,1,213,1,1,213,1,39,1,39,39,408,30,
  1,1,89,89,210,89,210,311,89,210,311,169,7,13,13,48,67,67,
  1,1,7,48,67,1,67,1,1,1,1,1,48,48,1,13,298,28,1,28,116,
  116,1,1,1,1,1,1,1,1,1,1,1,1,1,213,1,213,1,1,104,138,1,
  104,1,1,248,248,349,248,349,362,248,349,362,427,31,80,80,
  80,46,160,1,1,31,80,160,1,46,1,1,1,1,1,70,70,113,17,52,
  70,113,70,230,230,1,1,1,1,1,1,1,1,1,1,1,1,81,81,1,101,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,39,1,104,1,1,89,89,210,89,210,
  311,89,210,311,169,7,48,48,13,67,67,172,1,7,13,67,1,67,1,
  172,1,1,1,13,13,1,13,1,48,1,48,1,1,116,116,1,298,1,1,1,
  213,1,1,1,1,1,213,1,1,1,1,104,1,1,248,248,349,248,349,
  362,248,349,362,427,31,70,70,70,17,70,113,1,31,70,70,1,17,
  1,113,1,1,1,80,80,1,17,1,160,1,160,1,1,1,1,1,52,1,1,1,
  81,1,1,1,1,1,101,1,81,1,1,1,1,1,1049,1,1,1,1,1,1,1,1,
  76,1,1,1,1,1,1,1,76,1,1,1,1,1,1,1,1,1,1,1,149,1,1,1,
  149,1,1,1,1,1,1,1,1,1,1,516,1,1,1,1,1,1,1,1,1,1,1,
  364,364,452,364,452,145,364,452,145,441,13,134,134,104,42,
  17,205,1,13,104,17,1,42,1,205,1,1,1,104,104,205,7,309,
  134,205,134,327,327,265,265,1,1,1,1,1,323,1,1,1,1,323,1,
  1,1,1,1,1,1,1,1036,1,1,1,1,1,1,76,1,1,1,1,1,149,1,76,
  1,1,1,1,1,149,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,516,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,39,39,31,81,81,
  31,20,57,57,11,108,108,108,95,210,302,1,11,108,210,1,95,1,
  302,1,1,1,108,108,233,13,691,190,233,190,1,1,1,1,1,318,1,
  1,1,453,1,1,1,1,924,924,1,924,1,39,39,39,24,31,81,81,43,
  11,108,108,108,95,210,1,1,11,108,210,1,95,1,1,1,1,1,108,
  108,233,13,1,190,233,190,300,300,1,1,1,1,1,1,1,1,1,1,1,
  1,924,453,1,453,1,81,110,81,20,168,144,168,31,210,210,210,
  137,355,1,1,31,210,355,1,137,1,1,1,1,1,210,210,1,40,656,
  267,1,267,463,463,1,1,1,1,1,1,1,1,1,1,1,1,574,868,1,
  868,1,39,81,39,39,24,43,11,108,108,108,95,210,302,1,11,
  108,210,1,95,1,302,1,1,1,108,108,302,13,318,190,302,190,1,
  1,300,300,1,691,1,1,1,453,1,1,1,1,453,1,1,1,1,81,81,
  110,81,13,31,210,210,210,137,355,1,1,31,210,355,1,137,1,1,
  1,1,1,210,210,320,40,1,267,320,267,593,593,593,593,1,656,
  1,1,1,1,1,1,1,1,868,1,1,1,1,57,144,16,57,13,221,221,
  221,90,62,203,1,13,221,62,1,90,1,203,1,1,1,221,221,203,8,
  555,128,203,128,343,343,238,238,1,555,1,1,1,1,1,1,1,1,
  613,889,1,889,1,39,81,57,11,108,108,108,95,210,233,1,11,
  108,210,1,95,1,233,1,1,1,108,108,1,13,1,190,1,190,1,1,1,
  1,1,691,1,1,1,924,1,1,1,1,1,453,1,453,1,81,168,31,210,
  210,210,137,355,320,1,31,210,355,1,137,1,320,1,1,1,210,
  210,1,40,1,267,1,267,1,1,463,463,1,282,1,1,1,868,1,1,1,
  1,1,1,1,1,1,57,13,221,221,221,90,62,146,1,13,221,62,1,
  90,1,146,1,1,1,221,221,146,8,293,128,146,128,238,238,343,
  343,1,1,1,1,1,889,1,1,1,1,889,1,1,1,1,22,335,335,335,
  87,182,1,1,22,335,182,1,87,1,1,1,1,1,335,335,280,17,1,
  219,280,219,1,1,1,1,1,1,1,1,1,1,1,1,1,1,694,1,1,1,1,
  5,5,5,7,7,9,1,2,5,7,1,7,1,9,1,1,1,5,5,9,4,11,5,9,5,
  5,5,5,5,1,11,1,1,1,11,1,1,1,1,11,11,1,11,1,5,14,14,
  14,1,1,5,14,14,1,14,1,1,1,1,1,9,9,16,10,29,10,16,10,
  29,29,1,1,1,1,1,1,1,1,1,1,1,1,45,45,1,1,1,14,14,14,1,
  1,5,14,14,1,14,1,1,1,1,1,9,9,16,10,29,10,16,10,29,29,
  1,1,1,1,1,1,1,1,1,1,1,1,45,45,1,1,1,14,14,1,1,5,5,
  14,1,14,1,1,1,1,1,10,10,16,10,29,9,16,9,1,1,29,29,1,1,
  1,1,1,1,1,1,1,1,45,45,1,1,1,7,29,1,7,14,7,1,4,1,29,
  1,1,1,14,14,29,6,13,14,29,14,14,14,14,14,1,13,1,1,1,19,
  1,1,1,1,17,17,1,19,1,10,1,7,14,4,1,7,1,10,1,1,1,14,
  14,10,6,36,14,10,14,14,14,14,14,1,36,1,1,1,67,1,1,1,1,
  52,52,1,67,1,1,9,1,10,1,29,1,3,1,1,1,16,16,7,7,55,16,
  7,16,1,1,1,1,1,1,1,1,1,1,1,1,1,1,102,102,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,5,7,1,7,1,9,1,1,1,5,5,9,4,11,5,
  9,5,5,5,5,5,1,11,1,1,1,11,1,1,1,1,11,11,1,11,1,14,1,
  14,1,1,1,1,1,10,10,16,10,29,9,16,9,1,1,29,29,1,1,1,1,
  1,1,1,1,1,1,45,45,1,1,1,1,7,1,10,1,1,1,14,14,10,6,36,
  14,10,14,14,14,14,14,1,36,1,1,1,67,1,1,1,1,52,52,1,67,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,29,1,1,1,14,14,29,6,13,14,29,14,
  14,14,14,14,1,13,1,1,1,19,1,1,1,1,17,17,1,19,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,16,16,7,7,55,16,7,16,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,102,102,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,5,1,10,1,
  14,1,14,1,1,29,29,1,29,1,1,1,45,1,1,1,1,1,1,1,45,1,1,
  10,1,14,1,14,1,1,29,29,1,29,1,1,1,45,1,1,1,1,1,1,1,
  45,1,7,1,1,3,1,1,1,1,1,1,55,1,1,1,102,1,1,1,1,1,1,1,
  102,1,1,10,7,10,13,13,13,13,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,5,29,29,1,1,1,29,1,1,1,48,1,1,1,1,1,1,1,48,1,1,1,
  1,1,1,1,55,1,1,1,102,1,1,1,1,1,1,1,102,1,29,29,1,1,1,
  29,1,1,1,48,1,1,1,1,1,1,1,48,1,5,14,14,1,1,1,1,1,318,
  1,1,1,1,1,318,1,1,1,14,14,1,1,1,1,1,318,1,1,1,1,1,
  318,1,1,1,5,1,1,1,1,1,1,1,1,1,1,1,318,1,318,1,1,1,1,
  1,1,1,1,1,1,1,1,318,1,318,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,81,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,81,1,1,1,1,1,1,1,1,1,65,65,152,1,65,
  152,1,54,1,1,83,83,147,83,147,35,83,147,35,277,9,9,9,29,
  29,10,1,1,9,29,10,1,29,1,1,1,1,1,9,9,1,10,1,29,1,29,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,39,39,1,13,
  104,1,104,1,1,89,89,210,89,210,311,89,210,311,169,13,7,13,
  13,13,13,1,1,48,67,67,116,48,172,1,1,298,1,7,48,1,67,1,
  67,116,48,1,172,1,1,1,1,1,1,1,1,1,1,213,1,1,1,1,1,213,
  39,1,39,39,1,30,1,408,89,89,210,89,210,311,89,210,311,169,
  48,7,48,48,48,13,116,116,13,67,67,1,28,1,1,1,1,1,7,13,
  1,67,1,67,1,28,1,1,1,1,1,1,298,1,213,1,1,1,1,1,1,1,1,
  1,213,1,104,138,1,104,1,1,248,248,349,248,349,362,248,349,
  362,427,80,31,80,70,70,17,1,1,70,17,70,1,160,113,1,1,52,
  1,31,70,1,70,1,17,1,160,1,113,1,1,1,1,1,1,81,1,1,1,81,
  1,1,1,1,1,101,408,1,1,1,1,1,1,1036,1,1,1,1,1,1,1,1,1,
  76,1,1,1,1,1,1,1,1,1,1,1,149,1,1,1,1,76,1,1,1,1,1,1,
  1,1,149,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,516,39,1,104,
  1,398,89,89,210,89,210,311,89,210,311,169,13,7,13,13,13,
  13,1,1,48,67,67,116,48,1,1,172,298,1,7,48,1,67,1,67,116,
  48,1,1,1,172,1,1,1,1,213,1,213,1,1,1,1,1,1,1,1,1,104,
  1,1,248,248,349,248,349,362,248,349,362,427,70,31,70,80,80,
  17,230,230,80,46,160,1,70,1,1,113,1,1,31,80,1,160,1,46,
  1,70,1,1,1,113,1,1,52,1,101,1,81,1,1,1,1,1,1,1,81,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,364,364,452,364,452,145,364,452,145,
  441,104,13,104,104,104,7,327,327,134,42,17,265,134,205,1,
  205,1,1,13,134,1,17,1,42,265,134,1,205,1,205,1,1,309,1,
  1,1,323,1,323,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1049,1,1,
  1,1,1,1,1,76,1,1,1,1,1,1,1,1,1,1,1,1,1,149,1,1,76,1,
  1,1,1,1,1,1,1,1,1,149,1,1,1,1,1,1,1,1,516,1,1,1,1,1,
  1,39,39,31,81,81,31,20,57,57,108,11,108,108,108,13,1,1,
  108,95,210,1,190,233,1,1,691,1,11,108,1,210,1,95,1,190,1,
  233,1,1,1,1,1,1,453,1,1,1,924,1,1,1,1,1,453,39,39,39,
  24,31,81,81,43,108,11,108,108,108,13,1,1,108,95,210,300,
  190,302,1,302,691,1,11,108,1,210,1,95,300,190,1,302,1,302,
  1,1,318,1,1,1,453,1,453,1,1,1,1,1,1,81,110,81,20,168,
  144,168,210,31,210,210,210,40,1,1,210,137,355,463,267,320,
  1,1,282,1,31,210,1,355,1,137,463,267,1,320,1,1,1,1,1,1,
  1,1,1,1,868,1,1,1,1,1,1,39,81,39,39,24,43,108,11,108,
  108,108,13,300,300,108,95,210,1,190,1,1,233,1,1,11,108,1,
  210,1,95,1,190,1,1,1,233,1,1,1,1,453,1,924,1,1,1,1,1,
  1,1,453,81,81,110,81,13,210,31,210,210,210,40,593,593,210,
  137,355,593,267,1,1,320,656,1,31,210,1,355,1,137,593,267,
  1,1,1,320,1,1,1,1,1,1,868,1,1,1,1,1,1,1,1,57,144,16,
  57,221,13,221,221,221,8,238,238,221,90,62,343,128,146,1,
  146,1,1,13,221,1,62,1,90,343,128,1,146,1,146,1,1,293,1,
  1,1,889,1,889,1,1,1,1,1,1,39,81,57,108,11,108,108,108,
  13,1,1,108,95,210,1,190,302,1,233,318,1,11,108,1,210,1,
  95,1,190,1,302,1,233,1,1,691,1,924,1,924,1,453,1,1,1,1,
  1,924,81,168,210,31,210,210,210,40,463,463,210,137,355,1,
  267,1,1,1,1,1,31,210,1,355,1,137,1,267,1,1,1,1,1,1,656,
  1,868,1,574,1,1,1,1,1,1,1,868,57,221,13,221,221,221,8,
  343,343,221,90,62,238,128,203,1,203,555,1,13,221,1,62,1,
  90,238,128,1,203,1,203,1,1,555,1,889,1,613,1,1,1,1,1,1,
  1,889,335,22,335,335,335,17,1,1,335,87,182,1,219,1,1,280,
  1,1,22,335,1,182,1,87,1,219,1,1,1,280,1,1,1,1,1,1,694,
  1,1,1,1,1,1,1,1,5,5,10,10,10,1,1,9,14,14,29,14,16,1,
  1,29,1,5,9,1,14,1,14,29,14,1,16,1,1,1,1,1,1,45,1,1,1,
  45,1,1,1,1,1,1,5,5,5,4,5,5,5,7,7,5,5,9,1,9,11,1,2,
  5,1,7,1,7,5,5,1,9,1,9,1,1,11,1,11,1,11,1,11,1,1,1,1,
  1,11,10,10,10,1,1,9,14,14,29,14,16,1,1,29,1,5,9,1,14,
  1,14,29,14,1,16,1,1,1,1,1,1,45,1,1,1,45,1,1,1,1,1,1,
  5,10,1,1,14,14,14,29,9,1,1,16,1,1,5,14,1,14,1,14,29,9,
  1,1,1,16,1,1,29,1,1,1,45,1,1,1,1,1,1,1,45,10,1,1,14,
  14,14,29,9,1,1,16,1,1,5,14,1,14,1,14,29,9,1,1,1,16,1,
  1,29,1,1,1,45,1,1,1,1,1,1,1,45,13,13,10,6,6,13,10,7,
  1,7,1,1,4,10,1,6,1,6,13,10,1,7,1,7,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,5,29,14,14,14,29,1,1,1,1,1,5,29,1,14,1,
  14,14,29,1,1,1,1,1,1,1,1,1,1,1,1,318,1,1,1,1,1,318,
  29,14,14,14,29,1,1,1,1,1,5,29,1,14,1,14,14,29,1,1,1,1,
  1,1,1,1,1,1,1,1,318,1,1,1,1,1,318,14,14,1,10,1,1,16,
  1,1,5,5,1,14,1,14,1,10,1,1,1,16,1,1,29,1,1,1,45,1,1,
  1,1,1,1,1,45,7,14,14,29,1,29,13,1,7,14,1,7,1,4,14,14,
  1,29,1,29,1,1,13,1,19,1,17,1,19,1,1,1,1,1,17,14,14,10,
  1,10,36,1,7,14,1,4,1,7,14,14,1,10,1,10,1,1,36,1,67,1,
  52,1,67,1,1,1,1,1,52,1,1,1,1,1,1,5,1,1,14,1,14,5,1,
  1,1,1,1,1,1,1,1,318,1,1,1,1,1,1,1,1,1,318,16,1,1,29,
  1,5,10,1,14,1,14,1,5,1,16,1,1,1,1,1,1,48,1,1,1,48,1,
  1,1,1,1,1,1,7,1,1,9,1,1,10,1,29,1,16,1,3,1,7,1,1,55,
  1,1,1,102,1,1,1,1,1,1,1,102,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,55,1,9,16,1,10,
  1,29,1,1,1,7,1,3,1,1,1,1,102,1,1,1,102,1,1,1,1,1,1,
  1,11,1,1,36,1,13,1,29,1,1,1,55,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,5,1,7,1,7,5,5,1,9,1,9,1,1,11,1,11,1,
  11,1,11,1,1,1,1,1,11,1,14,1,14,1,10,1,1,1,16,1,1,29,
  1,1,1,45,1,1,1,1,1,1,1,45,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,7,14,14,1,10,1,10,1,1,36,
  1,67,1,52,1,67,1,1,1,1,1,52,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,14,14,1,29,1,29,1,1,13,1,19,1,17,
  1,19,1,1,1,1,1,17,1,1,1,1,1,1,1,1,1,318,1,1,1,1,1,1,
  1,1,1,318,1,16,1,1,1,1,1,1,48,1,1,1,48,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,7,1,1,55,1,1,
  1,102,1,1,1,1,1,1,1,102,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,102,1,1,1,102,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,81,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,81,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,115,
  115,1,34,47,1,47,1,1,269,269,318,269,318,404,269,318,404,
  544,48,48,31,17,99,70,1,1,48,99,17,209,99,411,1,1,1,1,
  48,31,411,70,1,99,1,17,209,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,115,1,115,115,1,46,1,1,269,269,318,269,318,404,
  269,318,404,544,48,48,31,46,104,160,1,1,48,104,17,1,104,1,
  1,1,1,1,48,31,1,160,1,104,1,46,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,272,47,8,272,47,39,39,299,299,95,299,95,
  336,299,95,336,153,36,36,6,70,10,70,1,39,30,19,13,1,19,
  40,1,1,30,15,30,6,40,70,30,10,1,70,1,1,39,1,44,198,198,
  48,138,172,172,150,138,48,138,138,150,172,172,1,39,39,1,1,
  1,1,1,560,1,566,1,1,501,1,681,1,1,29,1,43,1,1,36,1,84,
  127,1,84,1,1,1,1,1,1,29,1,1,1,43,1,1,1,1,36,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,115,1,47,1,1,269,269,318,269,
  318,404,269,318,404,544,48,48,31,17,99,70,1,209,48,99,17,
  1,99,1,411,411,1,1,48,31,1,70,1,99,1,17,1,1,209,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,39,47,272,272,299,299,95,299,
  95,336,299,95,336,153,30,30,6,70,19,70,1,1,36,10,13,39,
  10,1,40,40,198,44,36,6,1,70,198,19,1,70,39,1,1,1,15,30,
  30,150,172,138,138,48,172,150,172,172,48,138,138,1,1,1,1,
  1053,579,1,604,1,1,501,1,683,1,1,29,1,123,1,1,1,1,84,
  127,1,84,1,1,1,1,1,1,29,1,1,1,123,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,405,405,311,405,311,82,405,
  311,82,442,99,99,10,10,99,6,1,173,99,99,8,173,99,315,315,
  315,1,109,99,10,315,6,1,99,1,10,173,1,173,1,109,1,1,1,1,
  1,1,116,1,116,1,1,1,1,1,39,1,1,501,1,566,1,1,560,1,681,
  1,1,29,1,84,1,1,1,1,43,127,36,43,1,1,1,1,1,1,29,1,1,
  1,84,1,1,36,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,501,1053,604,1,1,579,1,683,1,1,29,1,84,1,1,1,1,123,
  127,1,123,1,1,1,1,1,1,29,1,1,1,84,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,86,86,146,104,104,146,17,165,
  165,154,154,31,137,298,355,1,1,154,298,17,359,298,512,1,1,
  1,1,154,31,512,355,1,298,1,137,359,1,1,1,1,1,1,1,1064,1,
  1,1,1,1,1,1065,1,1065,1064,86,86,86,37,146,104,104,123,
  154,154,31,137,298,355,1,1,154,298,17,396,298,490,1,1,1,1,
  154,31,490,355,1,298,1,137,396,1,1,1,1,1,1,1,1,1,1,1,
  1064,1,1065,1,1,1,1,104,31,104,17,39,80,39,95,95,11,92,
  114,231,1,1,95,114,34,302,114,300,1,1,680,550,95,11,300,
  231,680,114,1,92,302,1,1,1,550,636,636,318,551,496,496,509,
  580,318,508,556,258,556,551,86,104,86,86,37,123,154,154,31,
  137,298,355,1,396,154,298,17,1,298,1,490,490,1,1,154,31,1,
  355,1,298,1,137,1,1,396,1,1,1,1,1,1,1064,1065,1,1,1,1,
  1,1,1,1,104,104,31,104,20,95,95,11,92,114,231,1,233,95,
  114,34,233,114,1,1,1,668,261,95,11,1,231,668,114,1,92,233,
  1,233,1,261,668,668,509,496,551,556,318,551,318,556,496,509,
  496,496,165,80,79,165,285,285,34,72,339,64,1,235,285,339,
  10,235,339,372,368,368,1,1,285,34,372,64,1,339,1,72,235,1,
  235,1,1,1,1,974,1,1,1,974,1,1,1,1,619,1,1,86,104,165,
  154,154,31,137,298,355,1,359,154,298,17,1,298,1,512,512,1,
  1,154,31,1,355,1,298,1,137,1,1,359,1,1,1,1,1,1065,1,1,
  1,1,1,1,1064,1,1064,1065,104,39,95,95,11,92,114,231,1,302,
  95,114,34,1,114,1,300,300,636,550,95,11,1,231,636,114,1,
  92,1,1,302,1,550,680,680,258,556,580,508,318,496,509,496,
  551,318,551,556,165,285,285,34,72,339,64,1,235,285,339,10,
  235,339,368,372,372,1,1,285,34,368,64,1,339,1,72,235,1,
  235,1,1,1,1,619,1,1,1,1,1,974,1,1,974,1,1,301,301,20,
  179,191,418,1,1,301,191,17,1,191,471,471,471,800,240,301,
  20,471,418,800,191,1,179,1,1,1,1,240,800,800,431,650,829,
  829,605,829,605,829,650,431,650,650,5,5,14,14,14,1,1,9,10,
  10,16,10,29,1,1,80,1,9,5,29,14,80,14,1,14,16,1,1,1,17,
  1,1,1,86,1,1,31,86,1,86,86,31,1,1,5,14,14,14,1,1,9,10,
  10,16,10,29,1,1,80,1,9,5,29,14,80,14,1,14,16,1,1,1,17,
  1,1,1,86,1,1,31,86,1,86,86,31,1,1,7,5,7,1,9,5,5,4,9,
  5,5,5,5,10,6,5,2,5,7,10,5,1,7,9,1,9,1,6,10,10,7,16,
  16,16,7,16,7,16,16,7,16,16,14,7,1,29,14,14,6,29,14,14,
  14,14,1,36,14,7,14,7,1,14,1,4,29,1,29,1,36,1,1,67,1,1,
  1,67,1,67,1,1,67,1,1,14,1,1,10,9,10,16,9,1,29,29,22,
  22,10,5,1,14,22,5,1,14,16,1,1,1,16,35,35,43,43,130,130,
  52,43,43,43,43,52,130,130,1,10,14,14,6,10,14,14,14,14,1,
  36,14,7,14,4,1,14,1,7,10,1,10,1,36,1,1,67,1,1,1,67,1,
  67,1,1,67,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,16,16,7,7,
  16,1,1,1,179,1,16,9,1,10,179,1,1,29,7,1,3,1,31,1,1,1,
  113,1,1,34,113,1,113,113,73,331,331,14,10,1,14,1,29,29,1,
  17,5,5,1,14,1,10,1,14,1,1,16,1,1,80,80,31,1,86,86,1,1,
  31,1,1,1,86,86,10,1,5,29,1,1,35,16,14,5,29,14,35,9,1,
  14,1,1,16,1,22,22,22,52,130,43,43,43,130,52,130,130,43,
  43,43,7,10,13,13,13,20,1,10,4,13,6,20,10,1,6,7,1,7,1,
  1,20,20,1,17,17,17,1,17,1,17,17,1,17,17,1,1,1,1,1,31,
  1,9,1,10,1,16,1,29,3,1,7,1,1,179,179,73,331,113,113,1,
  1,34,1,331,1,113,113,29,1,1,35,16,14,5,29,14,35,9,1,14,
  1,1,16,1,22,22,22,52,130,43,43,43,130,52,130,130,43,43,
  43,14,14,1,60,1,5,5,14,1,1,1,14,1,1,1,1,1,159,159,52,
  1,201,201,1,1,52,1,1,1,201,201,5,159,1,29,5,14,14,159,
  29,1,14,1,1,1,1,60,1,1,1,201,1,1,52,201,1,201,201,52,1,
  1,159,1,29,5,14,14,159,29,1,14,1,1,1,1,60,1,1,1,201,1,
  1,52,201,1,201,201,52,1,1,22,1,10,1,1,10,22,1,1,1,1,
  179,1,1,1,1,134,1,1,1,1,1,134,1,1,1,1,1,17,6,60,36,22,
  22,1,36,31,1,1,1,1,1,1,1,81,1,1,1,81,1,81,81,1,1,1,5,
  1,14,1,10,1,14,1,1,16,1,1,80,80,31,1,86,86,1,1,31,1,1,
  1,86,86,5,7,10,5,1,7,9,1,9,1,6,10,10,7,16,16,16,7,16,
  7,16,16,7,16,16,14,1,1,1,14,1,1,1,1,1,159,159,52,1,201,
  201,1,1,52,1,1,1,201,201,1,14,1,7,10,1,10,1,36,1,1,67,
  1,1,1,67,1,67,1,1,67,1,1,22,1,1,1,1,179,1,1,1,1,134,
  1,1,1,1,1,134,1,1,1,1,1,1,14,16,1,1,1,16,35,35,43,43,
  130,130,52,43,43,43,43,52,130,130,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,29,1,29,1,36,1,1,67,1,1,1,67,1,
  67,1,1,67,1,1,1,7,1,1,179,179,73,331,113,113,1,1,34,1,
  331,1,113,113,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,31,
  1,1,1,113,1,1,34,113,1,113,113,73,331,331,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,22,22,1,1,81,81,1,1,1,1,1,1,81,81,
  10,1,1,1,1,134,1,1,1,1,134,1,1,1,1,1,1,134,1,1,1,1,
  134,1,1,43,1,1,1,36,48,36,43,1,1,1,1,1,1,138,36,109,16,
  1,1,1,16,43,1,1,1,1,36,138,109,43,1,1,1,1,36,109,138,1,
  1,1,1,48,36,36,43,16,109,1,1,1,43,36,1,1,1,138,1,1,1,
  1,1,1,43,43,16,3,1,7,7,14,7,1,1,38,38,70,38,70,218,38,
  70,218,54,16,16,48,16,16,70,1,1,16,48,70,1,27,1,1,1,1,
  1,16,16,176,70,1,10,1,33,1,176,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,38,38,38,1,7,7,14,7,1,1,38,38,70,38,70,218,38,70,
  218,54,16,16,48,16,16,70,1,1,16,48,70,1,27,1,1,1,1,1,
  16,16,176,70,1,10,1,33,1,176,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,38,38,38,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,7,1,7,1,1,13,13,
  13,13,13,22,13,13,22,11,4,13,13,4,13,13,1,1,13,13,8,22,
  8,22,22,1,1,1,4,13,1,13,1,8,22,8,1,1,22,22,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,3,14,1,113,113,75,113,75,135,113,
  75,135,35,16,70,16,16,48,16,1,1,70,16,33,134,10,1,134,1,
  1,1,16,48,1,16,1,27,1,70,189,1,1,134,1,1,1,1,1,1,1,1,
  38,38,38,1,1,1,1,1,1,1,1,403,1,419,1,1,1,546,1,329,59,
  1,36,59,1,1,1,1,184,36,122,314,17,273,1,1,1,1,1,379,184,
  198,1,168,184,138,1,476,474,198,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,14,1,113,113,75,113,75,135,113,75,135,35,16,70,
  16,16,48,16,1,1,70,16,33,134,10,1,134,1,1,1,16,48,1,16,
  1,27,1,70,189,1,1,134,1,1,1,1,1,1,1,1,38,38,38,1,1,1,
  1,1,403,1,1,298,546,531,1,435,355,372,59,1,1,59,36,1,1,
  1,198,379,138,1,168,227,235,1,1,1,1,36,1,184,1,17,342,
  122,184,325,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,26,26,31,81,81,31,20,123,123,20,62,156,20,156,117,1,
  1,62,156,151,678,369,675,679,1,838,1,20,156,1,117,1,369,1,
  176,841,841,644,1,1,739,1,1,1,1,1,1,640,407,662,1,640,
  662,397,26,26,26,17,31,81,81,93,20,62,156,20,156,117,1,1,
  62,156,151,679,369,644,1,1,1,1,20,156,841,117,1,369,675,
  176,1,1,1,1,1,1,1,1,1,1,1,1,407,1,640,1,662,397,1,81,
  31,81,20,138,45,138,20,113,233,20,233,161,1,1,113,233,153,
  766,330,828,1,1,1,1,20,233,1,161,1,330,1,111,1,949,1,1,
  1,1,1,1,1,1,1,1,792,1,323,1,323,556,649,26,81,26,26,17,
  93,20,62,156,20,156,117,1,1,62,156,151,1,369,1,1,1,1,1,
  20,156,1,117,1,369,644,176,806,1,638,592,1,1,1,1,1,1,1,
  1,1,397,407,1,397,1,407,81,81,31,81,40,20,113,233,20,233,
  161,1,1,113,233,153,1,330,1,1,1,1,1,20,233,949,161,1,330,
  828,111,994,943,863,898,1,1,1,1,1,1,1,1,1,649,792,1,556,
  649,1,123,45,146,123,54,236,444,54,444,148,1,1,236,444,205,
  930,462,1,1,1,1,1,54,444,988,148,1,462,887,222,934,1,879,
  853,1,985,1,1,1,1,1,1,724,1,557,1,580,745,1,26,81,123,
  20,62,156,20,156,117,1,1,62,156,151,1,369,638,592,1,739,1,
  20,156,1,117,1,369,1,176,1,806,1,678,1,838,1,1,1,1,1,1,
  397,662,1,1,1,407,640,81,138,20,113,233,20,233,161,1,1,
  113,233,153,1,330,863,898,1,1,1,20,233,943,161,1,330,1,
  111,1,1,842,919,1,1,1,1,1,1,1,1,649,556,1,1,649,1,792,
  123,54,236,444,54,444,148,1,1,236,444,205,1,462,879,853,1,
  985,1,54,444,1,148,1,462,1,222,983,1,894,905,1,1,1,1,1,
  1,1,1,1,745,724,1,745,1,724,113,286,554,113,554,228,1,1,
  286,554,139,1,188,1,1,1,767,1,113,554,1,228,1,188,1,133,
  1,1,1,1011,1,767,1,1,1,1,1,1,521,510,416,1,486,510,521,
  10,10,4,10,10,1,1,7,7,10,45,10,23,23,1,59,1,4,7,79,7,
  1,10,45,10,12,12,26,26,1,59,1,1,1,1,1,1,31,1,31,1,31,
  31,1,10,10,10,10,1,1,31,43,43,70,36,151,1,1,198,1,7,31,
  128,43,1,43,70,36,128,151,1,1,1,1,1,1,1,1,1,1,126,1,89,
  1,89,1,126,10,10,10,1,1,31,43,36,99,43,52,1,1,1,1,7,31,
  70,43,1,36,99,43,128,151,1,1,1,1,1,1,1,1,1,1,1,1,70,1,
  11,26,26,10,10,1,1,10,10,6,23,6,26,26,1,1,1,4,10,12,10,
  1,6,23,6,79,1,45,45,1,1,1,1,1,1,1,1,1,31,31,1,31,1,
  31,10,1,1,43,31,43,89,36,1,151,1,1,1,7,43,128,31,1,43,
  89,36,70,226,226,45,1,1,1,1,1,1,1,1,26,26,11,1,70,1,1,
  1,1,43,31,36,116,43,1,52,1,1,1,7,43,70,31,1,36,116,43,
  70,226,45,226,1,198,1,1,1,1,1,1,1,126,89,1,89,126,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,
  43,1,43,1,1,1,1,1,10,10,151,10,1,36,151,36,1,1,70,70,1,
  198,1,1,1,1,1,1,1,89,126,1,1,126,89,43,1,43,1,1,1,1,1,
  10,10,151,10,1,36,52,36,1,181,99,99,1,1,1,1,1,1,1,1,1,
  70,1,1,26,26,11,1,10,119,119,1,122,1,10,36,1,36,1,10,1,
  6,204,84,204,115,1,122,1,1,1,1,1,1,38,1,128,1,1,38,128,
  1,1,1,1,1,1,26,151,139,52,1,1,26,1,209,422,195,1,1,196,
  1,1,1,1,1,1,1,287,1,1,1,102,242,104,104,1,37,1,10,36,1,
  36,1,6,1,10,204,84,115,204,1,17,1,1,1,1,1,1,43,144,144,
  1,29,10,29,26,1,1,1,45,226,226,45,1,100,308,93,1,172,308,
  195,1,620,1,1,1,1,1,1,1,186,205,1,414,449,1,1,1,1,45,
  45,226,226,1,100,195,93,1,139,1,1,1,1,1,1,1,1,1,1,1,1,
  205,1,242,1,102,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,59,1,222,198,1,17,440,122,1,1,
  361,608,1,1,1,1,1,1,1,1,1,149,149,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,1,
  10,1,10,26,10,1,79,23,23,1,59,1,1,1,1,1,1,31,31,1,1,1,
  31,31,226,10,1,43,1,43,181,1,89,89,1,1,1,1,1,1,1,1,26,
  11,26,1,1,1,70,226,1,1,172,1,1,503,1,422,1,514,1,1,1,1,
  1,1,59,295,1,1,1,278,1,1,43,1,43,1,1,116,116,1,1,1,1,
  1,1,1,1,126,89,1,1,126,1,89,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,10,194,130,1,194,1,37,1,1,1,1,
  1,1,10,29,29,1,144,43,144,1,215,1,308,1,1,1,1,1,1,1,1,
  1,186,205,1,1,449,1,345,194,130,194,1,1,122,1,1,1,1,1,1,
  38,128,1,1,128,38,1,1,204,204,1,1,1,1,1,1,1,1,527,323,
  1,1,1,295,59,226,226,1,1,1,1,1,1,1,1,1,59,295,1,527,1,
  278,26,1,1,1,1,1,1,1,1,205,1,186,1,219,414,449,1,1,1,1,
  1,1,1,1,205,1,1,1,102,242,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,149,1,149,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,26,
  26,1,1,1,1,29,1,1,1,1,1,1,1,1,1,1,1,26,29,26,1,7,7,
  14,7,1,1,38,38,70,38,70,218,38,70,218,54,16,16,48,16,16,
  70,1,1,16,48,70,1,27,1,1,1,1,1,16,16,176,70,1,10,1,33,
  1,176,1,1,1,1,1,1,1,1,1,1,1,1,1,1,38,38,38,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,3,1,7,1,14,38,38,70,38,70,218,38,70,218,
  54,16,16,16,16,48,70,176,1,16,16,70,1,10,1,1,176,1,1,16,
  48,1,70,1,27,1,33,1,1,1,1,1,1,1,38,38,1,1,1,1,1,1,38,
  1,1,1,1,7,1,14,38,38,70,38,70,218,38,70,218,54,16,16,16,
  16,48,70,176,1,16,16,70,1,10,1,1,176,1,1,16,48,1,70,1,
  27,1,33,1,1,1,1,1,1,1,38,38,1,1,1,1,1,1,38,1,1,1,1,
  1,1,1,419,546,403,1,1,1,1,1,329,198,1,36,184,379,1,314,
  184,59,36,198,1,17,1,1,198,1,1,1,1,235,122,1,168,1,249,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,17,17,34,
  17,34,17,17,34,17,10,13,6,13,13,13,6,95,95,6,13,13,1,8,
  1,1,95,1,1,13,13,95,6,1,8,1,13,95,95,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,403,1,419,1,1,1,546,
  1,329,1,59,1,184,36,122,1,1,1,379,198,1,168,1,1,235,1,1,
  198,36,198,1,1,17,1,249,184,314,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,26,26,31,81,81,31,20,123,123,62,20,156,62,
  156,151,1,1,20,156,117,1,369,1,1,1,1,1,62,156,592,151,
  838,369,1,166,1,678,1,1,1,1,1,1,397,1,1,1,1,1,1,662,
  662,640,407,26,26,26,17,31,81,81,93,62,20,156,62,156,151,
  1,644,20,156,117,1,369,1,1,679,1,1,62,156,678,151,1,369,
  1,166,675,679,1,1,838,1,739,407,1,1,1,1,1,1,1,397,640,
  407,1,81,31,81,20,138,45,138,113,20,233,113,233,153,1,1,
  20,233,161,1,330,1,1,1,1,1,113,233,919,153,873,330,1,57,
  1,766,1,1,873,1,1,1,649,1,1,1,1,1,1,556,323,792,1,26,
  81,26,26,17,93,62,20,156,62,156,151,679,675,20,156,117,1,
  369,1,1,678,1,838,62,156,679,151,739,369,1,166,644,1,1,1,
  1,1,1,640,407,1,1,1,1,1,1,1,407,1,397,81,81,31,81,40,
  113,20,233,113,233,153,1,828,20,233,161,1,330,1,1,766,1,1,
  113,233,766,153,884,330,1,57,828,1,1,1,1,1,884,792,1,1,1,
  1,1,1,1,649,792,1,649,123,45,146,123,236,54,444,236,444,
  205,1,1,54,444,148,1,462,1,1,930,1,1,236,444,875,205,1,
  462,1,251,887,930,1,1,1,1,907,724,1,1,1,1,1,1,1,745,557,
  724,1,26,81,123,62,20,156,62,156,151,678,1,20,156,117,1,
  369,1,1,592,1,1,62,156,1,151,1,369,1,166,1,1,1,1,1,1,
  838,662,640,1,1,1,1,1,1,407,1,397,662,81,138,113,20,233,
  113,233,153,766,1,20,233,161,1,330,1,1,919,1,873,113,233,
  1,153,1,330,1,57,1,1,1,1,1,1,873,323,792,1,1,1,1,1,1,
  1,1,649,556,123,236,54,444,236,444,205,930,887,54,444,148,
  1,462,1,1,875,1,1,236,444,930,205,907,462,1,251,1,1,1,1,
  1,1,1,557,724,1,1,1,1,1,1,1,724,1,745,286,113,554,286,
  554,139,1,1,113,554,228,1,188,1,1,1038,1,771,286,554,1038,
  139,767,188,1,275,1,1,1,1,771,1,767,416,521,1,1,1,1,1,1,
  510,416,521,510,10,10,31,43,43,1,1,7,31,43,1,43,1,1,1,1,
  1,31,31,1,13,198,21,1,21,70,70,1,1,127,1,1,1,89,1,1,1,
  1,1,1,126,89,126,1,10,10,10,6,26,26,4,10,10,1,6,1,1,23,
  1,1,7,7,45,10,59,10,1,6,23,23,1,1,59,1,59,1,31,1,1,1,
  1,1,1,31,31,1,31,43,31,43,1,1,7,43,31,1,43,1,1,1,1,1,
  31,31,45,21,1,13,1,21,89,89,1,1,17,1,1,1,70,1,1,1,1,1,
  1,1,11,26,26,10,43,1,151,10,10,10,1,36,1,1,70,1,198,31,
  43,70,43,127,36,1,10,151,1,1,1,198,1,127,126,1,1,1,1,1,
  1,1,89,126,1,89,43,151,1,10,10,10,1,36,1,1,89,1,1,31,
  43,99,36,17,43,1,10,52,1,1,1,1,1,84,26,26,1,1,1,1,1,1,
  11,1,1,70,119,119,10,36,36,1,10,1,1,1,1,122,13,21,115,
  16,257,16,1,21,1,1,1,1,257,1,1,38,1,1,1,1,1,1,1,128,
  128,38,1,26,23,1,52,1,104,1,1,1,1,1,70,89,1,1,608,1,1,
  115,278,1,1,1,196,1,1,1,1,1,1,1,1,1,1,287,102,1,242,23,
  52,1,1,104,1,1,1,1,1,70,89,1,1,361,1,1,115,150,278,1,1,
  1,1,1,1,186,1,1,1,1,1,1,205,219,495,345,10,10,1,10,1,1,
  45,1,59,10,10,23,6,59,6,1,6,26,26,1,1,1,1,59,31,1,1,1,
  1,1,1,1,31,1,31,31,10,1,43,1,1,99,1,1,43,31,89,43,84,
  36,1,10,1,151,1,1,1,1,17,1,1,1,1,1,1,1,1,70,26,26,11,
  1,43,1,1,116,1,1,43,31,116,36,127,43,1,10,1,52,1,1,1,1,
  127,1,126,1,1,1,1,1,1,89,1,126,89,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,37,21,13,194,16,162,16,1,21,1,1,1,1,179,1,179,43,
  144,1,1,1,1,1,1,144,29,10,29,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,45,78,115,1,194,1,78,1,1,1,1,1,1,1,1,287,1,1,1,1,
  1,1,1,242,102,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,127,17,1,257,179,179,1,127,1,196,1,
  1,1,1,1,1,149,1,1,1,1,1,1,149,1,1,1,10,1,43,1,43,1,
  21,1,1,1,1,1,1,198,89,126,1,1,1,1,1,1,1,1,89,126,1,43,
  1,43,1,21,1,1,1,1,1,1,1,11,26,1,1,1,1,1,1,26,1,70,1,
  1,1,1,1,78,1,1,1,1,1,1,1,242,102,1,1,1,1,1,1,1,1,287,
  1,1,10,1,21,119,119,1,1,122,1,257,128,38,1,1,1,1,1,1,1,
  38,1,128,179,1,1,1,1,1,1,1,1,1,149,149,1,1,1,1,1,1,1,
  1,1,1,1,21,104,104,1,1,37,1,162,29,10,1,1,1,1,1,1,29,
  43,144,144,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  115,115,1,1,127,1,1,116,116,1,1,1,1,1,1,1,116,116,1,26,
  1,1,1,1,361,219,495,1,1,1,1,1,1,345,1,186,205,1,1,1,1,
  608,102,1,1,1,1,1,1,1,242,1,1,287,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,179,
  1,1,1,1,1,1,1,1,1,1,149,149,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,149,149,1,26,1,1,1,1,1,1,29,1,1,
  1,1,1,1,1,1,1,26,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,26,29,26,14,10,10,14,1,1,1,1,337,1,337,290,658,
  274,1,64,1,1,15,1,52,1,1,110,1,15,1,1,20,110,1,1,1,1,
  232,52,1,1,1,15,215,1,1,367,205,1,1,1,1,1,1,1,1,1,1,1,
  1,1,32,32,32,7,1,3,1,1,113,113,75,113,75,135,113,75,135,
  35,16,70,48,16,16,16,1,189,70,48,33,1,27,134,1,1,1,1,16,
  16,1,16,1,10,134,70,1,1,134,1,1,1,1,1,1,38,38,38,1,1,
  1,1,1,1,1,10,7,10,10,13,13,8,13,8,150,13,8,150,8,13,13,
  6,13,6,13,1,52,13,6,13,1,7,1,1,1,13,13,13,6,1,13,13,7,
  1,13,52,1,1,1,13,13,13,10,10,10,10,10,10,10,10,10,10,10,
  10,1,1,1,1,1,274,1,68,1,1,274,1,58,232,1,15,1,52,1,1,
  1,1,15,1,1,20,293,1,1,1,1,1,52,110,1,1,15,1,1,1,215,1,
  1,1,1,1,1,1,32,32,32,1,1,1,1,1,1,1,1,1,113,113,75,113,
  75,135,113,75,135,35,16,70,48,16,16,16,1,189,70,48,33,1,
  27,134,1,1,1,1,16,16,1,16,1,10,134,70,1,1,134,1,1,1,1,
  1,1,38,38,38,1,1,1,1,1,1,1,3,1,1,65,426,174,1,1,116,1,
  35,1,1,48,116,10,1,1,1,1,48,116,1,27,1,1,1,1,1,1,10,1,
  1,1,10,235,116,86,235,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,65,426,174,1,1,116,1,35,1,1,48,116,10,1,1,1,1,
  48,116,1,27,1,1,1,1,1,1,10,1,1,1,10,235,116,86,235,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,43,43,31,57,57,31,13,
  127,127,131,131,172,131,20,333,1,1,131,172,333,1,388,601,1,
  1,825,1,131,20,686,333,825,34,627,455,1,625,1,1,1,1,1,1,
  614,1,1,1,614,1,1,1,975,975,1,43,43,43,26,31,57,57,52,
  131,131,172,131,20,333,1,1,131,172,333,1,388,1,1,1,1,1,
  131,20,625,333,1,34,601,455,1,704,1,1,1,1,1,1,1,1,1,1,
  1,1,614,614,975,1,1,57,16,57,13,17,43,17,205,205,95,205,
  10,330,1,1,205,95,330,1,243,409,1,1,68,337,205,10,340,330,
  337,13,340,107,804,371,1,1,312,410,410,70,75,205,205,330,
  75,70,57,57,107,330,205,43,57,43,43,26,52,131,131,172,131,
  20,333,1,601,131,172,333,1,388,1,1,1,1,825,131,20,704,333,
  764,34,1,455,957,1,688,1,1,1,1,614,1,1,1,1,1,614,1,1,1,
  1,1,57,57,16,57,15,205,205,95,205,10,330,1,409,205,95,330,
  1,243,1,1,1,337,68,205,10,371,330,274,13,409,107,740,343,
  343,1,95,312,410,75,70,205,330,205,70,75,75,75,330,205,205,
  127,43,45,127,383,383,186,383,179,326,1,640,383,186,326,1,
  388,640,1,1,1,1,383,179,689,326,1,113,665,432,1,1,1,1,1,
  977,952,1,1,1,1,1,1,1,647,647,664,1,1,43,57,127,131,131,
  172,131,20,333,1,627,131,172,333,1,388,688,1,1,764,1,131,
  20,1,333,1,34,1,455,1,528,627,1,1,1,1,1,614,1,1,1,614,
  1,1,1,1,1,975,57,17,205,205,95,205,10,330,1,340,205,95,
  330,1,243,343,1,1,274,337,205,10,343,330,274,13,1,107,1,1,
  371,1,312,95,312,57,75,330,107,205,75,57,70,70,205,205,330,
  127,383,383,186,383,179,326,1,665,383,186,326,1,388,1,1,1,
  1,1,383,179,1,326,1,113,640,432,1,730,774,1,1,1,1,647,1,
  1,731,1,1,647,1,1,1,1,1,480,480,141,480,13,358,1,529,480,
  141,358,1,197,495,1,1,428,190,480,13,461,358,436,10,529,
  215,1,1,1,1,287,540,575,146,214,448,536,448,214,146,146,
  146,536,448,448,29,29,31,10,10,1,1,31,13,84,1,84,70,1,1,
  32,1,31,7,70,31,32,10,13,10,81,31,1,1,227,1,1,1,38,1,1,
  172,38,1,38,38,1,172,1,29,10,10,31,1,1,31,29,29,1,22,70,
  1,1,32,1,31,7,70,10,32,10,31,31,99,13,1,1,1,1,1,1,38,
  1,1,172,38,1,38,38,1,1,172,29,6,29,1,36,13,14,13,1,14,
  13,1,1,15,15,13,6,13,29,15,6,36,29,1,36,36,1,13,29,29,
  48,48,29,29,13,48,48,48,48,13,29,29,10,29,1,70,29,29,31,
  1,22,1,1,1,1,32,31,10,31,10,32,7,70,31,99,1,13,1,227,1,
  227,38,1,1,172,1,1,38,38,38,172,1,1,10,1,1,10,6,10,1,7,
  1,1,1,10,10,7,5,11,7,6,5,11,7,12,16,16,1,13,13,13,7,7,
  20,20,20,7,7,5,5,20,20,20,1,19,84,13,31,1,84,1,1,1,1,
  32,31,10,31,31,32,7,19,10,81,1,31,1,1,227,1,38,1,172,1,
  1,1,38,38,38,172,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,70,
  13,1,1,98,1,1,1,115,1,31,11,89,31,32,16,16,13,81,89,39,
  1,1,1,1,1,110,1,1,394,110,1,38,38,246,246,87,29,10,1,84,
  1,1,1,1,32,10,10,13,31,32,7,70,31,119,119,31,1,1,227,1,
  38,1,1,172,1,1,38,38,38,1,172,1,29,1,14,36,1,1,15,15,
  29,6,36,29,15,6,13,13,1,36,36,1,29,29,13,48,48,29,13,29,
  48,48,48,48,29,29,13,1,84,19,1,1,32,1,10,7,19,31,32,10,
  31,31,119,31,119,1,1,1,227,1,38,172,1,1,38,1,38,38,1,1,
  172,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,98,1,1,20,20,84,7,98,22,20,7,98,
  84,1,30,30,1,84,22,84,27,27,22,84,84,27,27,27,27,84,22,
  84,1,1,1,115,13,16,16,31,32,11,89,31,130,113,89,1,1,389,
  389,110,1,1,394,1,1,110,38,38,87,232,232,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,10,32,10,32,32,10,6,115,32,195,1,32,1,1,1,1,14,1,1,
  1,1,1,14,14,14,1,1,1,32,6,115,32,10,10,32,32,195,32,1,
  1,1,1,1,1,14,1,1,1,14,1,14,14,1,1,1,10,1,29,1,10,1,
  84,1,70,70,1,1,1,227,38,38,172,1,1,38,38,1,1,1,1,172,
  16,10,10,5,1,10,52,1,11,1,13,13,13,5,7,20,20,20,7,5,7,
  7,20,20,20,1,1,1,113,1,1,89,1,1,1,1,1,110,38,232,232,
  87,38,110,1,1,1,1,394,1,10,1,29,1,19,19,1,1,1,1,38,38,
  1,1,172,38,38,1,1,1,1,172,10,1,1,1,115,115,1,1,1,1,14,
  14,1,1,1,14,14,1,1,1,1,1,16,10,52,11,1,1,13,13,13,7,5,
  13,13,13,5,7,7,7,13,13,13,1,1,1,89,1,1,1,1,38,110,394,
  1,1,110,38,1,1,232,232,87,1,70,70,1,227,1,1,38,38,1,1,
  172,38,38,1,1,1,172,1,119,119,1,1,1,1,149,149,1,1,1,149,
  149,1,1,1,1,1,16,1,1,1,389,38,1,87,232,246,1,38,110,110,
  1,1,1,1,389,1,1,1,38,1,1,1,38,1,110,110,246,87,232,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,29,84,1,1,99,1,1,1,1,1,1,
  1,1,99,29,1,1,1,1,99,1,1,1,1,1,1,99,1,1,1,1,99,1,1,
  1,1,1,99,1,7,1,1,1,7,3,7,7,1,1,1,1,1,1,3,7,7,7,1,1,
  1,29,29,1,1,1,1,31,10,31,84,1,1,1,1,31,31,10,1,1,1,1,
  10,31,31,7,7,7,1,1,1,7,7,1,1,1,3,1,1,1,1,1,1,29,84,
  29,1,1,14,1,1,1,298,435,1,546,355,403,1,531,372,1,198,36,
  59,379,184,325,184,1,36,138,1,17,235,227,1,1,1,59,1,1,1,
  1,168,1,122,1,1,1,342,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  3,1,1,1,1,426,116,1,174,1,1,65,1,35,1,1,10,116,48,1,
  235,86,1,10,116,1,10,1,1,1,1,1,1,48,1,1,1,27,1,116,1,
  1,1,235,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,426,
  116,1,174,1,1,65,1,35,1,1,10,116,48,1,235,86,1,10,116,1,
  10,1,1,1,1,1,1,48,1,1,1,27,1,116,1,1,1,235,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,366,573,1,1,298,506,1,403,
  368,1,122,36,184,1,59,293,1,1,36,249,1,17,1,256,1,1,1,
  198,379,1,1,1,168,466,198,338,378,378,338,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,368,368,1,1,676,1,676,429,777,1,600,1,1,777,1,1,
  600,1,1,1,1,712,801,882,1,1,1,1,1,958,848,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,775,522,1,1,139,1,244,104,199,104,1,1,
  402,758,415,788,1,1,1,402,1,1,499,585,1,1,1,1,808,415,1,
  1,1,505,1,1,1,967,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,86,1,88,1,758,402,1,415,1,1,585,1,402,788,1,499,1,
  1,1,1,1,768,415,859,1,1,505,702,1,947,840,816,1008,1,1,1,
  1,1,1,1,1,1,1,1,1,1,533,1,246,185,1,246,1,791,633,791,
  1,904,1,1,1,633,904,1,1,1,1,1,1,1,750,737,992,1,1,706,
  1009,719,978,933,951,917,1,1,1,1,1,1,1,1,1,1,1,1,522,
  778,1,1,1,246,1,877,1,807,832,1022,1055,760,1,945,892,1,
  857,753,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,787,861,368,104,1,822,365,763,363,1,912,
  590,1,365,1,1,489,589,1,1,1,1,1,363,847,1,1,519,1,1,1,
  1,1,1034,1,1,1,1,1,1,1,1,1,1,1,1,1,643,632,1,1,1002,
  803,909,1,854,1,811,1,856,871,1,1,811,1023,1,1,1,903,984,
  1,1,1,844,1,1025,1026,1019,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,835,1,1,167,1,554,867,1,486,1,167,1,1,268,502,1,1,
  1,1,1,554,1,1,1,188,596,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,454,454,454,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,55,55,76,81,81,1,48,48,1,23,52,1,1,1,1,76,70,31,
  1,1,1,1,1,190,194,1,1,1,1,1,1,1,1,1,1,1,1,1,1,43,1,
  144,48,29,48,1,20,1,11,23,1,11,24,1,1,1,1,23,29,1,1,1,
  6,20,1,54,80,24,1,1,1,1,1,1,1,1,1,1,1,1,1,10,19,19,
  55,48,1,52,1,1,76,1,23,81,81,1,1,1,1,1,52,1,1,59,160,
  76,1,1,176,190,1,1,1,1,1,1,1,1,1,1,1,1,144,1,43,1,130,
  1,1,29,1,1,13,1,130,1,1,1,70,43,99,1,1,43,70,70,40,138,
  138,40,1,1,1,1,1,1,1,1,1,1,1,1,1,29,1,121,43,1,23,1,
  1,1,1,74,1,1,1,76,55,149,1,1,59,160,55,36,194,176,194,1,
  1,1,1,1,1,1,1,1,1,1,1,144,43,1,81,1,1,74,1,54,157,1,
  1,1,1,1,1,40,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,155,1,155,1,24,1,1,24,1,157,1,1,1,52,26,126,1,1,26,
  26,31,30,104,36,20,1,1,1,1,1,1,1,1,1,1,1,1,1,52,52,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,48,1,11,20,1,1,1,1,1,13,160,1,1,
  16,61,23,1,1,61,54,1,1,1,1,1,1,1,1,1,1,1,1,19,19,10,
  1,1,43,121,1,1,1,31,70,160,1,1,31,104,76,165,104,165,36,
  1,1,1,1,1,1,1,1,1,1,1,1,1,43,144,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,24,
  54,1,1,1,48,29,80,1,1,16,61,48,19,130,61,19,1,1,1,1,1,
  1,1,1,1,1,1,1,19,10,19,81,1,1,1,31,43,43,1,1,29,76,52,
  20,149,80,30,1,1,1,1,1,1,1,1,1,1,1,1,52,52,1,1,1,1,1,
  95,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,155,
  1,155,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,31,1,48,1,36,104,107,
  1,1,1,1,1,1,1,1,1,1,1,1,1,144,43,43,1,1,43,1,55,1,1,
  70,130,1,1,1,1,1,1,1,1,1,1,1,1,1,1,29,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,130,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,26,1,77,104,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,17,56,56,1,1,1,70,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,192,192,107,130,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  43,144,1,1,165,19,1,1,1,1,1,1,1,1,1,1,1,1,1,174,48,
  138,194,1,1,1,1,1,1,1,1,1,1,1,1,1,1,149,1,1,1,1,1,1,
  1,1,1,1,1,1,1,104,1,1,1,1,1,1,1,1,1,1,1,1,1,1,48,
  174,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,19,19,19,1,7,1,14,38,38,70,38,70,218,38,70,
  218,54,16,16,16,16,48,70,176,1,16,16,70,1,10,1,1,176,1,
  1,16,48,1,70,1,27,1,33,1,1,1,1,1,1,1,38,38,1,1,1,1,1,
  1,38,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,113,113,75,113,75,135,
  113,75,135,35,16,70,48,16,16,16,1,189,70,48,33,1,27,134,
  1,1,1,1,16,16,1,16,1,10,134,70,1,1,134,1,1,1,1,1,1,38,
  38,38,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,419,546,403,1,1,1,
  1,1,329,1,184,379,59,36,198,476,1,1,1,122,474,168,1,273,
  184,1,1,59,36,1,1,1,17,198,138,1,1,314,184,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,26,26,31,81,81,31,20,123,123,20,62,
  156,20,156,117,806,1,62,156,151,1,369,592,638,1,838,1,20,
  156,1,117,1,369,678,176,1,1,1,1,1,739,1,1,407,397,1,662,
  1,1,1,640,1,1,1,26,26,26,17,31,81,81,93,20,62,156,20,
  156,117,1,806,62,156,151,638,369,1,1,1,1,1,20,156,1,117,
  1,369,592,176,1,1,1,644,1,1,1,397,1,1,407,397,1,1,1,407,
  1,1,1,81,31,81,20,138,45,138,20,113,233,20,233,161,1,1,
  113,233,153,842,330,898,863,943,1,1,20,233,1,161,1,330,919,
  111,1,1,1,1,1,1,1,649,1,649,1,556,1,1,1,792,1,1,1,26,
  81,26,26,17,93,20,62,156,20,156,117,1,1,62,156,151,1,369,
  1,644,841,1,1,20,156,1,117,1,369,1,176,1,1,679,675,1,1,
  1,662,397,407,640,1,1,1,1,1,1,1,1,81,81,31,81,40,20,113,
  233,20,233,161,943,994,113,233,153,863,330,1,1,949,1,1,20,
  233,1,161,1,330,898,111,1,1,1,828,1,1,1,556,649,1,792,
  649,1,1,1,1,1,1,1,123,45,146,123,54,236,444,54,444,148,1,
  983,236,444,205,894,462,853,879,1,1,1,54,444,1,148,1,462,
  905,222,1,1,1,1,1,985,1,745,1,1,724,745,1,1,1,724,1,1,
  1,26,81,123,20,62,156,20,156,117,841,841,62,156,151,644,
  369,679,675,1,739,1,20,156,1,117,1,369,1,176,1,1,678,1,1,
  838,1,640,662,640,662,407,1,1,1,397,1,1,1,81,138,20,113,
  233,20,233,161,949,1,113,233,153,1,330,1,828,1,1,1,20,233,
  1,161,1,330,1,111,1,1,766,1,1,1,1,323,556,792,323,1,1,1,
  1,649,1,1,1,123,54,236,444,54,444,148,1,934,236,444,205,
  879,462,1,1,988,985,1,54,444,1,148,1,462,853,222,1,1,930,
  887,1,1,1,580,745,724,557,1,1,1,1,1,1,1,1,113,286,554,
  113,554,228,1,1,286,554,139,1,188,1,1,1,767,1,113,554,1,
  228,1,188,1011,133,1,1,1,1,1,767,1,486,510,521,416,510,1,
  1,1,521,1,1,1,10,10,4,10,10,79,1,7,7,10,23,10,45,45,1,
  59,1,4,7,1,7,1,10,23,10,1,1,26,26,1,59,1,1,31,31,1,31,
  1,1,1,31,1,1,1,10,10,10,10,1,1,31,43,43,70,36,1,1,151,
  198,1,7,31,1,43,1,43,70,36,1,1,1,151,1,1,1,1,126,1,126,
  89,1,1,1,89,1,1,1,10,10,10,1,181,31,43,36,89,43,45,226,
  226,1,1,7,31,1,43,1,36,89,43,1,1,151,1,1,1,1,1,1,26,
  26,11,1,1,1,70,1,1,1,10,10,1,79,10,10,6,45,6,26,26,12,
  1,1,4,10,1,10,1,6,45,6,1,1,23,23,1,1,1,31,1,1,31,31,
  1,1,1,31,1,1,1,10,181,1,43,31,43,99,36,1,1,151,1,1,7,
  43,1,31,1,43,99,36,1,1,1,52,1,1,1,26,26,1,1,70,1,1,1,
  11,1,1,1,1,1,43,31,36,116,43,226,45,226,1,1,7,43,1,31,
  1,36,116,43,1,1,52,1,1,198,1,126,1,126,1,89,1,1,1,89,1,
  1,1,1,151,226,84,226,130,139,172,503,1,1,12,151,1,226,1,
  84,226,130,1,1,422,1,1,1,1,527,1,1,295,59,1,1,1,278,1,
  1,1,128,70,204,204,194,1,1,1,1,1,12,128,1,70,1,204,204,
  194,1,1,209,215,1,1,1,1,295,527,1,323,1,1,1,59,1,1,1,
  10,43,1,43,1,151,128,1,1,10,10,1,10,1,36,1,36,1,1,70,
  70,1,198,1,89,1,126,89,1,1,1,1,126,1,1,1,43,226,43,151,
  1,128,1,1,10,10,1,10,1,36,45,36,1,1,89,89,1,1,1,70,1,
  26,11,26,1,1,1,1,1,1,1,204,10,119,119,1,122,1,10,36,1,
  36,1,10,115,6,1,1,1,1,1,122,1,1,38,38,128,1,1,1,1,128,
  1,1,1,1,1,308,1,1,1,26,1,1,45,1,115,26,194,1,1,195,308,
  1,361,1,219,414,205,186,1,1,1,1,449,1,1,1,100,100,1,37,
  1,10,36,1,36,1,6,194,10,1,1,1,1,1,17,1,144,43,10,29,29,
  1,1,1,144,1,1,1,26,226,1,1,23,1,1,52,1,104,1,93,1,1,1,
  195,1,1,1,242,1,1,205,1,1,1,1,102,1,1,1,226,620,1,23,
  52,1,1,1,104,195,93,1,1,1,308,1,1,1,414,449,1,205,186,1,
  1,1,1,1,1,1,514,1,79,70,1,70,1,1,422,1,1,1,139,172,1,
  222,1,1,278,59,1,295,1,1,1,1,1,1,1,1,59,1,1,198,1,17,
  1,122,1,1,196,1,1,1,1,149,1,1,1,1,1,1,1,149,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,10,1,10,1,10,26,10,1,1,45,45,1,59,1,31,31,31,31,1,1,
  1,1,1,1,1,1,1,10,1,43,1,43,1,1,99,99,1,1,1,11,26,1,
  70,1,1,1,1,26,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,43,226,43,1,1,116,116,1,1,1,89,
  126,1,89,126,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,204,10,1,1,1,1,1,37,1,29,10,43,
  144,144,1,1,1,29,1,1,1,1,1,1,1,1,1,608,1,102,242,205,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,122,1,128,38,38,1,128,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,26,1,1,1,1,102,1,1,
  287,1,1,1,242,1,1,1,1,440,1,449,1,186,1,205,1,1,1,345,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,149,149,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,26,1,1,1,1,1,1,
  29,1,1,1,1,1,1,1,1,1,26,1,1,1,26,26,1,1,1,1,1,1,1,
  29,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,426,116,1,174,1,1,65,1,
  35,1,1,10,116,48,1,235,86,1,10,116,1,10,1,1,1,1,1,1,48,
  1,1,1,27,1,116,1,1,1,235,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,14,1,113,113,75,113,75,135,113,75,135,35,16,70,16,16,
  48,16,1,1,70,16,33,134,10,1,134,1,1,1,16,48,1,16,1,27,
  1,70,189,1,1,134,1,1,1,1,1,1,1,1,38,38,38,1,1,1,1,10,
  658,1,274,1,337,1,1,337,290,64,232,1,52,1,15,1,367,1,1,
  52,1,205,15,1,110,1,1,1,1,15,1,1,1,20,1,1,110,1,1,215,
  1,1,1,32,32,1,1,1,1,1,1,32,1,1,1,1,1,274,1,68,1,1,
  274,1,58,1,1,52,1,15,1,215,1,1,52,1,1,15,1,293,110,1,1,
  232,15,1,1,1,20,1,1,1,1,1,1,1,1,1,1,1,1,1,1,32,32,32,
  1,1,1,1,43,43,31,57,57,31,13,127,127,131,131,20,131,172,
  333,528,1,131,20,333,627,34,1,688,1,1,1,131,172,1,333,1,
  388,1,455,627,1,1,1,1,764,1,1,1,614,1,1,1,1,1,975,1,
  614,1,43,43,43,26,31,57,57,52,131,131,20,131,172,333,1,
  957,131,20,333,688,34,1,1,704,1,1,131,172,1,333,1,388,1,
  455,601,1,1,1,825,1,764,1,1,1,1,614,1,1,1,1,614,1,1,57,
  16,57,13,17,43,17,205,205,10,205,95,330,1,1,205,10,330,
  371,13,1,343,343,95,312,205,95,1,330,312,243,1,107,340,1,
  1,1,337,274,274,205,205,75,70,57,330,205,107,330,57,75,70,
  43,57,43,43,26,52,131,131,20,131,172,333,704,1,131,20,333,
  1,34,1,1,625,1,1,131,172,1,333,1,388,1,455,1,1,1,601,1,
  1,1,975,1,1,614,1,1,1,1,1,1,1,614,57,57,16,57,15,205,
  205,10,205,95,330,343,740,205,10,330,343,13,1,1,371,312,95,
  205,95,1,330,410,243,1,107,409,1,1,409,68,337,274,330,205,
  70,75,75,205,205,330,205,75,70,75,127,43,45,127,383,383,
  179,383,186,326,730,1,383,179,326,774,113,1,1,1,1,1,383,
  186,1,326,1,388,1,432,665,1,1,640,1,1,1,1,1,1,1,647,1,
  1,731,1,647,1,1,43,57,127,131,131,20,131,172,333,625,1,
  131,20,333,1,34,1,601,686,1,1,131,172,1,333,1,388,1,455,
  1,1,1,627,1,825,825,975,975,614,1,1,1,1,1,1,1,614,1,57,
  17,205,205,10,205,95,330,371,804,205,10,330,1,13,1,409,340,
  410,312,205,95,1,330,410,243,1,107,1,1,1,340,337,68,337,
  107,330,75,57,70,205,330,205,205,70,75,57,127,383,383,179,
  383,186,326,1,1,383,179,326,1,113,1,640,689,977,1,383,186,
  1,326,952,388,1,432,640,1,1,665,1,1,1,664,1,1,647,1,1,1,
  1,1,1,1,647,480,480,13,480,141,358,1,1,480,13,358,1,10,1,
  495,461,540,287,480,141,1,358,575,197,1,215,529,1,1,529,
  190,428,436,536,448,214,146,146,448,448,536,448,146,214,146,
  10,10,31,29,29,70,1,31,7,10,70,10,1,13,1,1,1,31,13,1,
  31,227,84,1,84,31,1,1,1,32,32,1,1,1,38,1,38,172,1,1,
  172,38,38,1,10,29,29,31,119,119,31,10,10,31,7,1,1,13,227,
  1,31,13,1,84,1,84,1,31,70,1,1,70,32,1,32,1,172,1,38,38,
  1,1,172,1,38,1,38,10,6,10,1,52,7,5,7,11,5,1,16,16,13,
  13,7,6,1,10,13,7,1,10,11,1,1,1,6,10,10,20,20,7,7,5,20,
  20,20,20,5,7,7,29,10,1,99,10,10,31,13,7,1,1,31,1,227,
  31,29,1,29,227,22,1,31,70,1,1,70,32,1,32,172,1,1,38,38,
  1,1,172,1,38,1,38,29,36,1,29,6,29,36,6,1,36,36,29,29,
  13,14,1,13,13,14,1,13,13,1,1,13,15,15,15,29,29,48,48,48,
  29,29,13,13,48,48,48,19,1,10,7,31,19,10,1,31,1,1,1,31,
  29,1,31,1,22,1,29,31,1,1,1,32,32,1,1,1,38,1,38,1,172,
  1,172,38,38,1,119,13,16,31,16,11,1,113,89,1,1,31,36,1,1,
  389,30,1,70,89,1,1,1,32,1,115,1,1,1,110,38,87,246,232,1,
  38,1,110,99,12,119,119,52,1,130,1,1,1,81,1,1,81,1,1,1,
  1,81,1,1,1,195,195,1,1,1,149,1,149,1,1,1,1,149,149,1,
  10,29,1,10,1,70,70,1,1,29,29,1,31,1,22,1,31,1,1,1,31,
  1,32,32,1,1,38,38,1,1,172,1,172,1,38,38,10,16,5,1,1,11,
  13,13,10,6,1,10,13,7,1,7,1,1,1,11,10,10,6,20,20,7,5,7,
  20,20,20,20,7,7,5,119,10,1,19,19,1,1,84,13,1,31,227,84,
  1,31,1,1,1,31,1,32,32,1,1,38,38,1,172,1,1,172,1,38,38,
  1,1,89,1,1,389,1,36,1,31,1,30,1,70,39,1,1,89,1,32,115,
  246,87,38,110,1,1,1,1,232,1,38,110,1,11,1,13,13,10,6,1,
  7,13,7,1,10,16,1,1,16,10,6,10,13,13,5,7,7,13,13,13,13,
  7,5,7,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,16,389,1,70,13,1,1,389,98,1,31,1,1,
  1,89,115,1,32,87,232,1,38,110,1,1,394,232,110,1,38,1,1,
  70,13,1,31,1,98,1,1,89,1,1,113,115,32,1,1,1,38,1,110,
  232,87,232,394,110,38,1,29,1,29,1,227,29,22,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,99,1,99,1,1,1,227,13,1,1,84,84,1,
  227,1,1,1,1,1,1,1,1,1,1,1,1,99,1,1,99,1,1,1,29,1,10,
  1,84,1,10,1,1,1,13,1,32,32,1,172,38,38,1,1,172,1,1,1,
  38,38,1,29,29,14,1,29,36,1,1,36,15,15,15,13,29,48,48,48,
  29,13,29,29,48,48,48,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,84,1,10,19,1,1,19,32,1,32,172,1,1,
  38,38,172,1,1,1,38,1,38,84,1,1,1,1,1,1,1,1,1,1,99,1,
  1,1,1,99,1,1,1,1,1,1,84,98,1,1,98,20,20,20,84,22,27,
  27,27,22,84,84,84,27,27,27,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,13,1,1,1,32,32,1,1,172,38,1,38,1,172,1,
  1,38,38,1,1,1,16,1,115,32,246,246,110,38,1,1,394,1,87,1,
  110,38,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,32,115,1,232,232,110,1,38,394,1,
  1,87,38,110,1,10,10,1,1,14,14,1,1,1,1,1,1,14,14,10,1,
  1,1,14,14,1,1,1,1,14,1,14,1,1,14,1,14,1,1,1,1,14,14,
  1,29,1,1,1,31,10,31,84,1,1,1,1,1,1,10,31,31,29,1,1,1,
  7,7,1,1,1,1,7,3,7,7,1,1,1,1,7,7,3,1,1,1,1,3,7,7,29,
  29,31,1,1,1,84,31,1,1,1,10,1,1,1,1,1,1,7,7,7,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,391,391,1,1,
  1,1,144,807,1,945,1,1,1,1,1,814,1006,890,1028,846,971,1,
  1,1,1,1,850,760,776,1,1016,922,1018,1,1041,1,1,1,1,1,1,
  1,1,787,709,1,1,1,1,1,1,1,1,128,1,1,168,1,168,763,1,
  365,1,363,1,1,1,762,365,932,976,489,813,1,1017,1,1,1,363,
  590,1,1,519,810,1,1,769,1,1,1,1,1,1,1,1,643,1,1,1,1,1,
  1,1,1,391,1,1,1,1,144,877,1,1,814,850,1,1028,922,1,1006,
  776,1,1,1,1,1039,1,1,1,1,753,890,1,1016,1,1,1,1,1046,1,
  1,1,1,1,1,709,1,1,1,1,1,1,1,1,1,1,1,128,1,17,1,1,176,
  749,206,1,1,885,749,176,1,1,162,1,1,1,1,1,1,206,472,1,1,
  227,885,819,1,698,1,990,1,1,1,1,1,446,1,446,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  168,822,1,365,762,363,1,976,810,1,365,1,1,489,847,1,1014,
  1,1,1,363,589,932,1,519,1,1,1,1,1,1029,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,657,1,85,1,164,1,1,1,1,85,1,1,71,897,1,1,1,1,
  1,164,381,1,1,105,1,651,1,1,1,1012,1,1,1,1,1,355,256,
  355,1,1,1,1,1,1,1,1,1,1,1,1,1,1,76,23,48,190,48,31,1,
  1,1,1,1,70,52,76,1,31,194,31,1,1,294,294,1,1,1,1,1,144,
  1,43,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,13,1,1,1,23,11,23,54,11,1,1,1,1,1,1,29,24,48,1,
  16,80,48,1,20,160,1,1,1,1,1,1,19,19,10,1,1,1,1,1,1,1,
  1,1,1,36,48,48,76,107,23,1,1,1,1,1,1,55,52,55,1,59,130,
  76,1,1,116,1,1,1,1,1,1,1,144,43,1,1,1,1,1,1,1,1,1,1,
  55,29,55,130,29,1,1,1,1,1,1,43,26,70,1,43,99,70,1,175,
  175,95,1,1,1,1,1,1,1,29,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,107,19,36,19,54,1,1,1,1,1,1,130,30,
  194,1,24,194,190,1,1,303,1,1,1,1,1,1,1,174,48,1,1,1,1,
  1,1,1,130,130,194,194,80,1,1,1,1,1,1,99,104,99,1,168,
  168,194,1,1,1,1,1,1,1,1,1,1,1,149,1,1,1,1,1,1,1,48,
  55,1,1,1,1,1,1,1,1,1,31,76,1,59,36,1,1,1,189,1,1,1,1,
  1,1,43,144,1,1,1,1,1,1,1,1,1,19,11,80,1,54,1,1,1,29,
  24,1,1,6,130,23,1,61,1,54,1,1,1,1,1,19,10,19,1,1,1,1,
  1,1,1,194,48,160,1,121,1,1,1,70,43,1,1,1,99,76,1,104,1,
  74,1,1,1,1,1,144,43,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  20,36,1,24,1,1,1,1,111,1,1,1,1,1,1,48,174,1,1,1,1,1,
  1,1,1,160,1,1,1,1,1,13,20,23,1,16,1,1,1,61,84,1,1,1,
  1,1,1,10,19,19,1,1,1,1,1,1,1,1,1,1,1,1,43,43,149,1,
  104,1,104,1,1,342,225,1,1,1,1,1,130,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,95,81,74,1,77,1,81,1,1,378,1,1,1,1,1,
  1,155,1,155,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,43,55,1,43,1,1,
  1,1,116,1,1,1,1,1,1,29,1,1,1,1,1,1,1,1,1,1,1,1,149,
  81,1,76,1,157,1,1,1,1,1,1,52,52,1,1,1,1,1,1,1,1,1,
  194,48,1,160,43,121,1,1,1,1,1,1,43,144,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,168,31,
  1,80,1,77,1,1,1,1,1,56,17,56,1,1,1,1,1,1,1,1,1,1,303,
  1,1,1,1,1,1,149,1,1,1,1,1,1,1,1,1,1,160,160,1,1,1,1,
  1,1,43,1,144,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,175,1,1,1,1,1,1,192,192,1,1,1,1,1,1,1,1,
  294,1,1,1,1,1,1,171,1,1,1,1,1,1,1,1,1,1,1,1,1,155,1,
  155,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,19,19,1,1,1,
  1,1,1,1,19,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,506,1,1,366,1,403,1,
  573,298,368,198,1,379,184,36,1,378,338,122,1,249,378,168,
  256,1,1,1,1,1,36,1,59,1,17,338,198,1,293,1,466,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,185,185,122,116,116,
  122,26,81,81,156,198,424,156,424,54,921,1,198,424,248,908,
  479,1,911,1,1,1,156,424,906,54,1,479,908,264,1,993,1,1,1,
  1,1,1,1,580,1,703,799,513,703,1,1,1,1,185,185,185,20,122,
  116,116,144,156,198,424,156,424,54,1,946,198,424,248,911,
  479,1,1,1,1,941,156,424,993,54,1,479,1,264,937,1,1,1,1,
  1,997,1,1,1,513,580,513,1,799,1,1,1,1,116,45,116,26,91,
  38,91,169,195,356,169,356,81,1,1,195,356,269,858,493,881,1,
  950,1,1,169,356,1015,81,1,493,968,136,940,1,1,1,1,1,1,1,
  1,635,914,862,1,914,1,1,1,1,1,185,116,185,185,20,144,156,
  198,424,156,424,54,1,937,198,424,248,1,479,1,1,993,1,1,
  156,424,1,54,997,479,1,264,946,1,911,1,941,1,1,1,1,513,
  799,1,1,580,513,1,1,1,1,116,116,45,116,32,169,195,356,169,
  356,81,950,1,195,356,269,1,493,1,1,1,1,1,169,356,1,81,1,
  493,881,136,1,950,1,881,1,1,1,1,1,914,1,635,914,635,1,1,
  1,1,1,81,38,31,81,174,148,341,174,341,13,588,624,148,341,
  37,594,254,624,606,672,1,1,174,341,642,13,1,254,597,147,
  606,672,606,624,1,1,878,1,1,1,655,697,655,1,438,1,1,1,1,
  185,116,81,156,198,424,156,424,54,993,1,198,424,248,1,479,
  911,1,906,1,1,156,424,1,54,1,479,1,264,1,921,908,908,1,1,
  1,1,1,799,703,513,580,703,1,1,1,1,1,116,91,169,195,356,
  169,356,81,1,940,195,356,269,1,493,1,881,1015,1,1,169,356,
  950,81,1,493,1,136,1,1,858,968,1,1,1,1,1,1,1,914,635,
  862,914,1,1,1,1,81,174,148,341,174,341,13,672,606,148,341,
  37,606,254,606,624,642,1,1,174,341,672,13,878,254,624,147,
  624,588,594,597,1,1,1,1,1,655,438,1,1,697,655,1,1,1,1,
  252,77,440,252,440,17,1,1,77,440,185,1,142,980,980,981,1,
  700,252,440,981,17,721,142,1004,215,1,1,1,1004,700,1,721,1,
  1,343,194,419,343,419,194,1,1,1,1,43,43,31,10,10,89,1,13,
  31,21,1,21,1,89,1,1,1,31,31,89,7,127,43,89,43,102,102,
  185,185,198,1,1,1,1,89,1,126,126,1,89,1,1,1,1,43,43,43,
  6,189,189,16,21,21,144,16,104,1,104,1,1,13,36,1,10,257,
  10,144,36,1,104,1,104,257,1,122,1,1,1,38,128,38,1,128,1,
  1,1,1,10,31,10,1,89,36,43,10,99,43,102,185,185,1,84,31,
  10,102,10,1,36,181,10,99,1,89,1,17,1,1,1,1,26,26,11,1,
  70,1,1,1,1,1,43,10,1,99,43,43,10,89,36,185,185,102,1,
  127,31,10,102,10,198,36,1,10,99,1,89,1,127,1,198,1,1,1,
  126,89,1,89,126,1,1,1,1,10,99,1,21,31,21,181,13,1,89,1,
  1,1,31,43,89,7,17,43,99,31,102,185,185,102,1,1,1,1,1,70,
  1,1,26,26,11,1,1,1,1,67,67,10,7,6,67,10,39,14,39,1,59,
  7,10,14,4,59,6,67,10,14,39,14,39,59,1,1,1,1,31,1,31,1,
  31,31,1,1,1,1,189,104,185,19,185,100,330,205,284,1,275,
  102,1,150,39,1,144,39,181,330,1,284,1,217,1,564,1,1,1,
  298,99,1,172,562,1,1,1,1,1,102,116,116,1,1,284,1,1,1,
  102,99,284,14,198,189,189,89,128,330,173,284,1,1,1,1,1,
  213,1,1,287,1,65,1,1,1,1,43,21,1,10,1,104,1,1,257,43,
  43,104,6,122,16,104,36,189,189,144,144,1,1,257,1,1,38,128,
  1,1,128,38,1,1,1,1,21,185,43,89,1,89,1,1,10,31,1,10,1,
  13,102,31,1,99,181,99,1,1,17,1,1,26,11,26,70,1,1,1,1,1,
  1,116,21,160,160,160,1,1,21,10,160,6,127,21,19,10,116,19,
  116,19,1,1,127,1,1,116,116,1,116,1,116,1,1,1,1,1,1,330,
  1,1,1,185,89,330,14,392,144,185,99,173,284,128,330,1,1,1,
  1,1,65,287,1,1,1,213,1,1,1,1,1,104,1,1,162,43,36,104,6,
  37,16,100,43,189,144,144,189,179,1,179,1,1,10,29,29,144,
  144,43,1,1,1,1,185,39,1,1,89,185,185,14,198,104,330,102,
  284,205,330,150,1,1,1,1,1,213,1,1,1,287,65,1,1,1,1,185,
  1,1,1,102,39,39,1,1,150,102,1,330,1,330,1,1,198,1,1,1,
  65,287,213,1,1,1,1,1,1,1,1,89,102,185,14,275,104,284,185,
  284,150,330,205,564,1,1,1,1,99,1,298,562,1,194,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,198,17,564,59,179,179,198,127,1,217,1,392,1,1,1,1,
  1,1,1,1,149,1,149,1,1,1,1,43,1,10,1,21,185,43,1,89,1,
  89,1,1,127,1,1,126,89,1,89,126,1,1,1,1,1,185,10,1,43,1,
  10,89,1,99,181,84,1,1,1,1,1,1,70,26,11,26,1,1,1,1,39,
  1,1,205,185,1,284,1,284,1,1,275,1,1,562,194,1,99,298,1,
  1,1,1,1,1,10,39,10,67,67,67,67,59,1,59,1,1,1,31,31,31,
  31,1,1,1,1,1,179,1,1,1,564,1,1,1,1,1,1,1,1,1,1,149,
  149,1,1,1,1,1,189,43,1,100,1,100,162,1,37,1,1,144,43,
  144,10,29,29,1,1,1,1,1,284,1,330,1,392,1,1,1,1,1,1,213,
  287,65,1,1,1,1,1,89,181,99,1,127,1,1,1,1,126,1,89,126,
  89,1,1,1,1,1,189,116,189,1,1,198,1,1,287,65,1,213,1,1,
  1,1,1,1,185,39,275,1,1,1,1,1,562,172,1,99,298,1,1,1,1,
  185,1,1,392,1,1,1,213,1,65,1,287,1,1,1,1,198,1,1,1,1,
  287,1,65,1,213,1,1,1,1,1,1,179,1,1,149,149,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,149,1,149,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  26,26,1,1,1,1,1,1,1,29,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  26,26,1,1,1,1,29,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,65,
  426,174,1,1,116,1,35,1,1,48,116,10,1,1,1,1,48,116,1,27,
  1,1,1,1,1,1,10,1,1,1,10,235,116,86,235,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,244,246,246,1,1,1,
  807,945,1,1,1,877,832,892,1,1,1,753,1,1,1,1,1,1,1022,1,
  857,1,1,760,1055,1,1,1,1,1,1,787,1,1,1,1,1,1,861,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,368,86,368,1,104,185,104,1,1,363,763,365,1,1,1,
  822,363,1,1,519,1,589,847,1,1,1,365,1,1,1,489,1034,1,590,
  912,1,1,1,1,1,1,643,1,1,1,1,1,1,632,1,1,1,1,368,1,1,
  1,1,712,1,801,676,1,1,848,958,676,429,777,1,1,1,600,882,
  1,1,1,1,1,777,1,1,1,1,600,1,1,1,1,1,1,775,522,1,1,1,
  1,1,1,1,1,1,1,1,1,139,1,88,768,1,415,1,402,1,840,947,
  758,415,788,816,505,1,1,859,1,1,1,402,1,1,1,499,1008,1,
  585,1,1,702,1,1,1,1,533,1,1,1,1,1,1,1,1,1,1,1,199,1,
  1,903,1,984,909,856,1,1019,1026,1002,1,871,1,844,1023,811,
  1,1,1,1,803,1,854,1,1,1,1025,811,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,835,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,104,808,1,415,758,402,1,967,1,1,415,1,
  1,505,1,585,1,1,1,1,402,1,788,1,499,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,246,750,1,737,791,633,1,933,
  978,791,1,904,951,706,1,1,992,1,1,1,633,1,904,1,1,917,
  719,1,1,1,1009,1,1,1,522,778,1,1,1,1,1,1,1,1,1,1,1,1,
  554,1,167,1,1,1,1,554,1,1,188,1,502,1,1,1,1,167,1,867,
  1,268,1,1,486,1,1,596,1,1,1,454,454,1,1,1,1,1,1,454,1,
  1,1,1,1,1,1,1,36,1,76,70,31,104,31,1,31,1,1,1,1,23,1,
  76,1,48,107,48,52,1,1,1,1,1,1,1,144,1,1,1,1,1,1,43,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,13,1,1,1,
  70,43,70,70,43,95,43,43,1,1,1,29,1,55,1,29,130,55,26,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,29,1,1,1,1,1,1,1,55,55,76,
  176,59,81,81,52,1,1,1,48,1,48,1,23,190,76,52,1,1,160,1,
  1,1,144,1,1,1,1,1,1,1,43,1,1,1,1,1,1,48,29,48,61,16,
  1,20,160,1,1,1,11,1,23,1,11,54,23,24,1,1,61,1,1,1,19,
  19,1,1,1,1,1,1,10,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,194,138,104,138,104,1,149,1,1,1,1,80,1,194,1,130,194,
  130,104,1,1,1,1,1,1,1,1,1,1,1,1,1,1,149,1,1,1,190,40,
  165,165,77,1,20,1,1,1,1,54,1,36,1,19,19,107,30,1,1,1,1,
  1,1,1,174,1,1,1,1,1,1,48,1,1,1,55,48,1,1,1,52,31,1,1,
  1,1,1,76,1,23,1,1,81,81,1,1,1,1,1,43,1,1,1,1,1,1,1,
  144,1,1,1,1,138,43,130,1,99,1,1,1,29,1,1,1,13,40,70,1,
  130,1,70,1,1,1,1,29,1,1,1,1,1,1,1,1,1,1,165,31,121,43,
  160,1,1,1,23,1,1,1,1,36,76,1,74,1,104,1,1,1,1,43,1,1,
  1,1,1,1,144,1,1,1,1,1,80,1,1,1,1,24,1,176,1,61,1,1,
  36,1,1,70,1,1,1,104,1,1,1,1,1,1,1,1,1,1,1,1,29,1,1,
  1,1,6,1,59,1,16,1,1,26,1,1,26,1,1,1,17,56,1,1,1,1,1,
  1,56,1,1,1,81,1,1,1,1,1,1,74,1,54,1,1,157,1,1,1,1,1,
  1,155,1,1,1,1,1,1,1,155,1,1,1,43,1,1,1,24,1,1,1,24,
  30,52,1,157,1,76,1,1,1,52,52,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,149,1,80,1,1,126,40,1,1,1,1,1,1,1,1,1,1,1,1,1,
  130,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,48,1,11,1,1,20,1,1,20,1,1,1,
  10,19,1,1,1,1,1,1,19,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,194,55,43,121,1,160,1,1,
  1,144,43,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,19,48,24,54,1,61,1,1,1,19,10,1,
  1,1,1,1,1,19,1,1,1,1,20,1,1,1,1,1,1,48,174,1,1,1,1,
  1,1,1,1,1,1,31,1,1,1,1,1,1,43,144,1,1,1,1,1,1,1,1,1,
  1,81,1,26,1,1,1,1,52,1,1,1,1,1,1,52,1,1,1,1,1,1,1,1,
  155,1,1,1,1,1,1,1,155,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,192,1,1,1,1,1,1,192,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,19,1,1,1,1,1,1,19,1,1,1,1,1,1,1,
  1,1,19,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,391,391,1,1,1,1,144,1,1,1,814,1006,890,1,1,
  1,850,776,1046,1016,1,1,753,1,1,877,1,1039,1,1,1,1,1,922,
  1028,1,1,1,1,1,1,1,1,1,1,709,1,1,1,1,1,1,1,128,1,1,
  168,1,168,1,1,363,762,365,932,1,1,1,363,1,1,519,1,847,
  589,1,1,822,365,1014,1,1,489,1029,1,810,976,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,391,1,1,1,1,144,1,814,850,1,
  1006,776,1041,1,1,1,890,1,1016,1,971,760,1,1,807,945,1,1,
  1,846,1,1018,1,1,1028,922,1,1,1,1,1,1,1,1,1,709,787,1,
  1,1,1,1,1,128,1,17,1,749,206,749,176,1,698,1,1,206,1,1,
  227,1,1,472,1,1,1,176,1,1,1,162,990,819,885,1,1,885,1,1,
  1,1,1,1,1,1,446,446,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,168,1,762,363,1,365,1,769,1,1,
  363,932,1,519,1,813,590,1,1,763,365,1017,1,1,489,1,1,1,1,
  976,810,1,1,1,1,1,1,1,1,1,1,643,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,164,1,85,1,1,1,1,
  164,1,1,105,1,897,381,1,1,657,85,1,1,1,71,1012,651,1,1,
  1,1,1,1,1,1,1,1,1,1,355,355,256,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,48,48,76,1,1,1,55,55,189,
  59,1,1,31,1,1,76,23,1,1,1,1,1,1,130,107,1,36,1,1,1,1,
  1,1,1,1,43,1,144,1,1,1,1,55,29,55,1,1,1,43,70,116,43,
  1,43,43,1,1,70,29,95,1,1,13,1,1,99,130,1,1,1,1,1,1,1,
  1,1,1,29,1,1,1,1,1,1,48,55,1,1,1,1,76,116,59,1,1,52,
  1,1,1,1,1,1,1,23,1,76,36,1,107,130,1,1,1,1,1,1,1,1,1,
  43,144,1,1,1,1,1,61,1,1,29,1,1,6,1,80,24,1,1,23,11,54,
  1,1,11,54,23,130,19,19,130,1,1,1,1,1,1,1,1,19,19,10,1,
  1,1,1,160,1,1,70,1,43,1,1,149,1,1,1,76,48,74,1,1,23,
  121,48,99,194,36,194,1,1,1,1,1,1,1,1,1,144,43,1,1,1,1,
  1,1,175,104,175,80,1,1,76,1,1,1,20,1,1,1,61,1,160,1,1,
  1,1,1,1,1,1,1,1,1,1,192,1,192,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,55,175,43,1,1,26,1,
  1,1,13,1,1,1,29,95,70,1,1,130,99,1,1,1,1,1,1,1,1,1,
  29,1,1,1,1,1,1,1,1,160,43,1,1,48,23,121,1,1,48,74,76,
  194,36,194,99,1,1,1,1,1,1,1,1,144,1,43,1,1,1,1,1,1,
  342,1,1,1,294,160,378,1,1,84,294,160,1,303,111,303,1,1,1,
  1,1,1,1,1,1,1,171,1,1,1,1,1,104,1,1,1,31,16,77,1,1,
  16,77,31,168,24,24,168,1,1,1,1,1,1,1,1,56,56,17,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,43,1,1,31,1,1,1,1,160,225,104,1,1,1,1,
  1,1,1,1,1,1,1,1,130,1,1,1,1,1,1,1,1,52,24,81,1,1,20,
  157,81,104,30,20,149,1,1,1,1,1,1,1,1,1,52,52,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,48,294,31,1,1,190,194,1,1,1,1,1,1,
  1,1,144,43,1,1,1,1,1,1,1,1,11,1,48,1,1,54,80,1,1,1,1,
  1,1,1,1,19,10,19,1,1,1,1,1,1,1,1,81,1,1,1,1,1,1,1,1,
  1,1,1,1,155,155,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,80,54,1,1,1,1,1,1,1,1,1,1,10,19,
  19,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,155,155,1,1,1,
  1,1,194,190,1,1,1,1,1,1,1,1,1,1,43,144,1,1,1,1,1,1,
  194,168,1,1,1,1,1,1,1,1,1,149,1,1,1,1,1,19,194,1,1,1,
  1,1,1,1,1,1,48,174,1,1,1,1,1,1,1,1,1,1,1,1,1,48,1,
  174,1,1,1,1,1,1,1,1,1,1,1,1,149,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,19,19,1,1,1,1,19,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,3,7,7,7,5,7,7,7,38,38,54,38,54,136,1,1,38,54,136,
  379,239,538,1,1,532,1,38,54,1,136,1,239,1,54,1,379,1,1,
  1,1,1,1,1,1,1,296,296,1,1,296,1,296,1,7,7,7,5,7,7,7,
  38,38,54,38,54,136,1,1,38,54,136,379,239,538,1,1,532,1,
  38,54,1,136,1,239,1,54,1,379,1,1,1,1,1,1,1,1,1,296,296,
  1,1,296,1,296,1,7,7,5,5,5,5,31,31,31,31,31,31,126,1,31,
  31,31,1,45,1,126,1,1,1,31,31,215,31,1,45,212,11,212,1,
  126,1,1,1,1,1,1,1,1,1,1,1,518,1,518,1,1,3,7,5,7,7,54,
  54,131,54,131,75,227,1,54,131,75,347,202,1,1,1,1,1,54,
  131,319,75,1,202,258,35,258,347,227,1,1,404,1,1,1,1,1,
  500,500,1,296,500,296,500,1,7,5,7,7,54,54,131,54,131,75,
  227,1,54,131,75,347,202,1,1,1,1,1,54,131,319,75,1,202,
  258,35,258,347,227,1,1,404,1,1,1,1,1,500,500,1,296,500,
  296,500,1,5,7,7,31,31,31,31,31,31,212,1,31,31,31,212,45,
  215,215,1,1,1,31,31,1,31,1,45,1,11,1,212,212,1,1,1,1,1,
  518,518,1,518,518,518,1,518,1,518,518,7,7,20,20,20,20,20,
  26,195,1,20,20,26,1,22,194,79,1,211,1,20,20,194,26,211,
  22,1,10,1,1,195,1,1,101,1,1,81,81,1,148,148,81,148,148,
  148,148,81,3,131,131,143,131,143,59,1,1,131,143,59,1,131,
  319,271,1,1,1,131,143,258,59,404,131,251,35,251,1,1,1,1,
  1,1,1,626,626,1,1,1,626,707,1,707,1,626,131,131,143,131,
  143,59,1,1,131,143,59,1,131,319,271,1,1,1,131,143,258,59,
  404,131,251,35,251,1,1,1,1,1,1,1,626,626,1,1,1,626,707,
  1,707,1,626,38,38,31,40,40,351,1,11,31,102,191,102,158,
  284,1,377,1,31,31,283,20,509,123,260,123,128,128,1,1,1,
  313,1,1,193,393,1,586,537,1,654,741,537,654,1,38,38,38,20,
  1,1,31,40,40,128,38,283,1,1,509,1,11,31,158,102,377,102,
  128,93,191,260,1,1,1,1,1,1,393,1,1,537,654,1,537,586,586,
  1,382,40,31,40,1,1,31,107,31,333,107,107,1,1,687,1,31,31,
  107,157,687,45,354,157,333,354,1,1,1,1,1,1,870,1,1,216,
  883,1,883,864,216,707,707,38,40,1,1,38,38,20,260,38,213,
  213,1,1,1,31,40,63,40,321,38,191,11,268,1,179,1,1,1,1,1,
  1,1,1,654,1,382,586,537,741,1,356,40,1,1,40,31,40,354,26,
  1,517,1,1,1,31,107,283,31,102,107,333,93,128,370,370,1,1,
  1,1,1,1,1,1,883,707,707,216,216,864,1,1,481,1,102,31,31,
  477,102,1,172,1,1,1,20,157,26,31,572,20,477,157,120,1,38,
  1,1,381,1,1,1,467,1,784,1,559,784,747,747,872,1,1,179,
  370,38,370,270,485,302,1,1,1,26,269,49,246,920,54,246,269,
  471,1,1,1,1,1,1,1,1,1,1,820,1,1,995,996,851,1,880,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,38,40,1,102,1,380,1,1,1,38,38,223,
  20,465,38,268,93,1,1,26,1,1,313,1,1,1,382,1,1,1,356,741,
  654,1,741,193,40,370,107,530,1,1,1,1,40,31,305,40,565,26,
  128,93,1,447,269,1,1,1,1,1,1,707,1,707,1,1,864,883,1,1,
  38,1,102,437,1,1,502,1,102,31,437,31,484,102,120,40,246,
  120,246,1,1,381,1,1,467,559,1,1,872,1,747,784,1,696,535,
  1,1,690,1,1,1,1,447,1,120,1021,198,1,546,359,725,95,1,1,
  561,1,1,1045,743,1,1,1,780,1033,1040,1,998,964,255,247,1,
  571,1,123,45,351,20,1,146,451,123,1,198,54,1,1,40,1,1,
  874,102,1,1,965,630,1,1,735,450,735,213,1,1,1,223,160,160,
  26,744,279,607,157,1,233,563,1,1,935,1,1,1,1005,1,1,1,
  963,999,929,1,1,1,1,1,1,213,145,224,224,969,376,186,194,1,
  332,1,1,1,1,1,1,1,1,1,1020,1030,1,1000,989,1007,1,722,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,465,1,744,558,1,269,1,413,1021,1,682,1,1,1,1,1,
  1,1,1,1,1,728,954,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,38,1,40,1,102,1,123,1,
  268,191,1,1,253,1,1,382,356,1,1,741,193,1,1,1,586,393,
  160,40,1,107,1,157,447,1,221,1,1,526,1,1,707,1,1,1,1,38,
  1,707,1,864,870,224,1,1,233,1,1,607,1,1,1,641,1,1,637,
  839,1,1,999,869,1,1,1,973,925,1,102,246,157,514,514,1,1,
  1,1,1,1,559,1,1,872,696,535,1,1,696,1,515,1,1,1,1,1,
  927,1,1,1,1,1,821,1,1,1,954,821,1,1,1,1,1,1,123,358,
  324,1,1,1,1,1,1,102,630,1,965,450,735,735,1,1,1,674,1,
  725,1,1,1,1,1,1,1,961,852,1,1033,1035,732,1,1,1040,1001,
  1056,1,295,481,1,1,263,1,1,394,394,1,1,738,684,1,1,986,
  655,1,1,246,1,1,793,1,1,1,961,1,1,1033,852,1,1063,1,1035,
  732,370,1,1,1,1,1,1,1045,1,1040,1,743,1035,1024,1033,1,
  780,1,1,1,1,1,577,1,1,996,820,1,1,1,995,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,886,1,1,1,1,843,1,569,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,614,45,252,707,1,1,1,1,
  707,1,1,1,1,252,614,45,1,1,1,1,1,1,1,1,1,1,1,1,1,406,
  252,707,1,406,1,1,1,735,252,1,1,1,1,1,1,1,1,1,1,1,735,
  707,7,7,7,5,7,7,7,38,38,54,38,54,136,1,1,38,54,136,379,
  239,538,1,1,532,1,38,54,1,136,1,239,1,54,1,379,1,1,1,1,
  1,1,1,1,1,296,296,1,1,296,1,296,1,3,7,7,7,7,5,38,38,
  54,38,54,136,1,538,38,54,136,1,239,1,1,379,1,532,38,54,
  379,136,1,239,538,54,538,1,1,538,532,1,1,296,1,1,296,1,1,
  1,296,1,296,1,1,7,7,7,7,5,38,38,54,38,54,136,1,538,38,
  54,136,1,239,1,1,379,1,532,38,54,379,136,1,239,538,54,538,
  1,1,538,532,1,1,296,1,1,296,1,1,1,296,1,296,1,1,5,5,5,
  5,20,20,26,20,26,13,1,150,20,26,13,50,26,150,1,50,1,1,
  20,26,128,13,1,26,123,10,123,50,1,150,1,1,1,518,1,1,518,
  518,518,1,518,518,518,518,1,7,7,5,31,31,31,31,31,31,1,
  212,31,31,31,126,45,126,1,215,1,1,31,31,1,31,1,45,1,11,
  1,126,1,212,1,1,1,518,1,1,518,1,1,1,1,1,1,1,1,3,7,54,
  54,131,54,131,75,1,317,54,131,75,1,202,227,1,327,1,1,54,
  131,347,75,1,202,1,35,1,1,1,317,1,1,404,421,1,1,421,1,1,
  1,500,1,500,1,1,7,54,54,131,54,131,75,1,317,54,131,75,1,
  202,227,1,327,1,1,54,131,347,75,1,202,1,35,1,1,1,317,1,
  1,404,421,1,1,421,1,1,1,500,1,500,1,1,38,38,70,38,70,31,
  1,113,38,70,31,1,38,241,1,241,244,308,38,70,113,31,1,38,
  110,10,110,1,1,113,308,1,185,141,1,1,141,306,306,1,488,
  306,488,306,1,38,38,31,40,40,1,1,11,31,102,213,102,191,1,
  1,321,1,31,31,268,20,1,123,223,123,63,99,1,1,321,1,1,1,
  1,1,1,356,193,1,741,382,586,393,1,38,38,38,20,1,1,31,40,
  40,99,38,351,1,284,465,1,11,31,191,102,1,102,63,93,158,
  284,1,351,377,1,313,1,1,1,393,193,393,1,586,356,537,1,1,
  40,31,40,1,1,31,107,31,312,107,38,1,1,565,1,31,31,128,
  157,1,45,305,157,283,399,1,1,102,1,1,1,1,1,1,38,870,1,
  864,1,216,1,1,38,40,1,351,38,38,20,284,38,1,1,99,1,465,
  31,40,128,40,1,38,158,11,283,1,1,191,509,1,253,393,1,1,
  193,393,1,1,537,193,654,1,1,40,1,1,40,31,40,399,26,1,1,
  399,1,1,31,107,333,31,1,107,283,93,107,1,1,38,687,1,526,
  1,1,1,870,870,1,1,216,38,883,1,1,1,481,102,31,31,514,102,
  220,1,220,1,558,20,157,120,31,1,20,437,157,26,220,1,220,
  572,1,1,467,1,1,1,515,1,1,747,535,784,467,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,191,38,220,220,330,1,1,1,942,1,26,221,
  471,38,1,344,344,455,49,548,1,1,920,1,1,1,1,1,1,772,955,
  1,851,567,995,953,1,38,40,1,102,1,1,213,1,321,38,38,260,
  20,1,38,283,93,213,213,1,26,1,1,253,193,1,1,356,1,1,1,
  654,393,1,382,1,40,1,107,1,1,312,1,565,40,31,354,40,1,26,
  107,93,1,517,1,221,1,1,17,870,1,1,38,1,1,1,883,870,707,
  707,1,220,102,481,1,514,558,1,102,31,477,31,1,102,26,40,1,
  172,1,38,1,1,385,1,1,1,515,1,467,1,784,515,1,559,1,1,1,
  1,1,1,910,213,517,332,172,1,246,213,1,396,581,1,504,1,1,
  827,944,1,1,1013,1,1,1,1030,1,1,722,1,330,1,1,1,1,123,
  45,451,20,1,146,351,123,344,246,1,344,570,1,570,874,1,1,
  400,400,874,1,1,674,1,102,1,1,220,1,942,179,370,370,38,1,
  270,563,49,485,302,1,22,1,1,782,955,1,1,772,1,1,1,820,
  577,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,910,1,380,145,1,172,1,1,539,1,1,1,
  1,302,1,1,1,1,1,1,1,1013,944,1,1,1,1007,714,1,1,509,1,
  1,502,1,269,1,464,1,1,1,1,1,1,1,708,1,1,1,1,1,1,960,
  796,1,1,1,377,102,1,484,1,1,824,1,713,639,1,1,1,1,1,1,
  1,1,1,1,708,1,1,708,1,1,1,38,1,40,1,102,213,123,1,380,
  1,179,1,1,313,356,1,1,382,1,382,1,1,1,1,356,1,370,40,1,
  107,1,157,530,1,1,269,1,1,526,38,1,1,1,1,707,1,707,1,1,
  1,1,1,1,1,233,1,1,690,1,1,1,1,1032,961,1,1,1,1045,743,
  1,1,1,1,780,1,1,102,1,157,437,1,1,1,502,1,385,515,1,1,
  535,467,559,1,1,1,872,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,344,123,255,247,1,422,571,1,353,400,1,
  1,674,874,102,1,1,400,965,630,1,1,591,1,1,1,1,1,1,637,1,
  1,1,1005,963,1,1,1,1,1,1,487,313,1,1,661,1,1,684,1,1,1,
  684,394,1,1,1,1,394,1,213,1,344,1,1,541,839,1,1,637,1,
  1005,1,1,1,1,963,1,1,220,1,1,845,714,1,1,944,1,1,1,1020,
  1,1030,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,609,1,1,
  1,1,1,1,567,772,1,1,1060,851,955,1,1,570,1,1,1,821,1,1,
  1,1,1,1,728,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,595,
  1,1,1,1,843,595,1,1,1,1,1,45,1,252,400,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,400,1,1,1,1,252,45,1,
  1,1,1,1,614,45,1,1,707,45,1,1,1,1,1,1,1,1,614,1,1,1,
  1,1,1,707,1,1,7,3,7,7,5,7,54,54,131,54,131,75,1,1,54,
  131,75,1,202,1,227,347,1,1,54,131,327,75,404,202,317,35,
  317,1,1,1,1,1,1,500,1,1,500,1,1,1,421,1,421,1,1,7,5,5,
  5,5,31,31,31,31,31,45,1,1,31,31,45,212,146,1,1,212,128,
  128,31,31,126,45,86,146,215,8,215,212,1,1,128,86,139,26,
  13,13,26,26,26,13,13,26,13,26,13,7,7,5,7,54,54,131,54,
  131,75,1,1,54,131,75,1,202,1,227,347,1,1,54,131,327,75,
  404,202,317,35,317,1,1,1,1,1,1,500,1,1,500,1,1,1,421,1,
  421,1,1,7,5,7,20,20,20,20,20,26,1,1,20,20,26,195,22,79,
  194,194,101,1,20,20,1,26,1,22,1,10,1,195,1,1,1,211,211,
  148,148,148,148,81,81,148,1,81,1,81,148,7,3,136,136,53,
  136,53,213,1,1,136,53,213,1,106,1,1,379,215,185,136,53,
  379,213,215,106,1,17,1,1,1,1,185,215,215,157,134,134,157,
  134,134,134,157,134,157,134,134,7,31,31,38,31,38,43,1,1,
  31,38,43,113,45,92,172,106,308,1,31,38,157,43,185,45,146,
  8,146,113,1,1,1,1,244,34,148,148,34,1,1,148,24,1,24,1,
  148,136,136,53,136,53,213,1,1,136,53,213,1,106,1,1,379,
  215,185,136,53,379,213,215,106,1,17,1,1,1,1,185,215,215,
  157,134,134,157,134,134,134,157,134,157,134,134,40,40,31,
  102,102,1,1,31,20,125,316,125,322,1,1,318,1,31,20,349,20,
  485,146,1,146,145,38,1,1,1,1,1,1,383,1,1,622,460,1,551,
  553,551,498,1,40,40,40,26,1,1,31,102,102,38,26,340,1,1,
  287,1,31,20,383,125,483,125,145,17,383,298,1,1,483,1,1,1,
  1,1,1,460,498,1,551,622,551,1,383,102,11,102,1,1,20,83,
  70,426,83,63,1,1,176,402,20,11,99,81,402,16,213,81,455,
  191,1,1,81,134,134,233,233,34,34,70,356,233,151,356,24,102,
  102,40,102,1,1,40,40,26,298,26,457,457,128,1,374,31,102,
  145,102,1,26,383,36,349,1,1,1,485,1,384,1,1,1,383,498,1,
  383,551,460,551,1,1,102,1,1,102,11,102,191,16,1,186,191,
  140,140,20,83,455,70,81,83,455,24,99,1,1,1,402,365,365,34,
  34,233,233,356,102,102,24,70,151,356,233,1,1,125,70,69,509,
  125,254,235,254,1,384,20,81,304,116,473,81,343,81,304,1,1,
  1,473,384,1,478,1,478,1,1,1,433,1,568,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,40,102,1,
  125,1,347,348,1,1,40,40,1,26,374,26,349,17,318,318,1,1,1,
  545,485,383,1,383,1,1,1,1,551,498,1,553,383,102,1,83,148,
  1,346,176,402,102,11,213,102,402,16,99,24,1,177,1,1,102,
  102,81,233,233,102,20,102,356,233,151,356,81,81,20,1,125,1,
  436,1,1,1,125,70,343,69,1,125,304,22,1,81,1,1,1,491,1,1,
  478,433,524,1,1,1,1,1,1,631,439,1,1,508,1,1,1,318,177,
  504,81,1,34,318,387,302,1,1,1,1,705,1,599,1043,471,790,1,
  1,1,616,761,1,612,729,234,458,1,162,499,146,16,324,81,499,
  20,324,146,1,34,1,1,235,95,235,330,330,26,112,112,493,330,
  208,493,146,81,146,457,254,1,1,228,213,213,146,751,534,1,
  294,497,320,1,1,1,1,893,756,1,551,552,1,1,1,1,818,1010,
  833,715,1,923,1,348,128,285,285,901,235,186,144,1,1,1,1,1,
  1,1,1,876,1,1,896,663,1,726,695,1,1,628,1,1,347,128,430,
  246,1,297,690,430,332,95,1,1,928,1,1,1,666,1,1,1,748,1,
  938,831,1,610,1,140,287,140,705,1,140,22,815,422,1,1,1,1,
  1,1,1,268,1,1,1,1,1,1,410,376,1,1,1,483,81,1,1,131,131,
  1,1,1,615,1,1,1,1,1,1,363,1,1,1,1,1,1,1,1,1,1,40,1,
  102,1,125,318,146,1,307,1,1,1,485,545,1,383,1,383,1,553,
  383,1,1,1,622,1,213,102,140,83,1,81,237,1,1,1,402,365,
  365,20,102,233,233,356,81,20,81,102,151,356,233,285,1,1,1,
  430,1,660,1,1,1,1,900,849,765,802,733,1027,1,1,1,1,1,855,
  716,1,125,1,81,338,477,1,1,543,1,477,524,433,1,439,1,631,
  439,1,1,611,1,524,131,1,1,1,823,1,1,1,1,1,420,352,1,1,
  1,459,375,1,1,1,1,1,1,146,335,1,1,1,499,489,489,112,26,
  330,330,493,81,146,146,112,208,493,330,1,617,1,1,1,785,918,
  1,765,849,665,1,734,959,1,1,1,734,755,1048,532,308,1,1,
  468,468,1,477,477,477,1,1,1,477,1,1,1,525,1,318,1,1,1,
  972,1,802,733,849,765,1,734,665,1,1057,1,959,1,1,1,834,1,
  692,471,1,1043,599,761,1,471,902,1,616,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  102,235,1,1,420,352,1,1,1,1,1,1,1,363,134,1,1,1,1,390,
  1,1,1,1,1,1,363,1,1,1,1,1,1,1,1,1,1,363,1,34,1,1,1,
  70,70,52,112,1,1,1,1,1,1,75,31,52,102,1,1,1,34,102,1,1,
  1,1,52,75,31,112,1,1,1,1,52,70,70,1,1,1,1,57,70,45,102,
  81,70,1,1,1,146,45,1,1,1,57,1,1,1,1,1,1,81,146,102,7,
  7,7,7,5,38,38,54,38,54,136,1,538,38,54,136,1,239,1,1,
  379,1,532,38,54,379,136,1,239,538,54,538,1,1,538,532,1,1,
  296,1,1,296,1,1,1,296,1,296,1,1,7,7,5,7,54,54,131,54,
  131,75,1,1,54,131,75,1,202,1,227,347,1,1,54,131,327,75,
  404,202,317,35,317,1,1,1,1,1,1,500,1,1,500,1,1,1,421,1,
  421,1,1,3,7,7,38,38,54,38,54,136,379,1,38,54,136,1,239,
  1,538,1,1,1,38,54,1,136,1,239,1,54,1,1,379,1,1,532,1,1,
  296,296,1,1,1,296,1,1,1,1,296,7,7,38,38,54,38,54,136,
  379,1,38,54,136,1,239,1,538,1,1,1,38,54,1,136,1,239,1,
  54,1,1,379,1,1,532,1,1,296,296,1,1,1,296,1,1,1,1,296,5,
  20,20,26,20,26,13,50,123,20,26,13,1,26,1,150,128,1,1,20,
  26,50,13,1,26,150,10,150,1,50,123,1,1,1,518,518,518,518,
  1,1,518,518,1,518,1,518,38,38,70,38,70,31,1,110,38,70,31,
  1,38,1,241,113,1,308,38,70,241,31,185,38,113,10,113,1,1,
  110,308,244,1,488,306,306,488,1,1,306,141,1,141,1,306,38,
  38,31,40,40,380,1,11,31,102,1,102,1,179,1,1,1,31,31,380,
  20,313,123,179,123,26,1,213,213,377,509,1,1,356,382,1,1,1,
  1,382,1,356,1,1,38,38,38,20,213,213,31,40,40,1,38,1,1,
  260,1,1,11,31,213,102,253,102,26,93,191,1,1,283,321,1,1,
  1,382,1,654,1,1,1,356,1,193,1,393,40,31,40,1,530,31,107,
  31,1,107,1,370,370,1,1,31,31,145,157,526,45,269,157,221,1,
  517,1,102,1,1,1,1,707,707,1,1,1,1,1,38,1,1,38,40,1,283,
  38,38,20,1,38,1,1,128,1,509,31,40,99,40,253,38,191,11,
  351,1,284,158,465,1,1,654,1,1,537,1,1,393,193,1,393,1,
  193,40,517,1,40,31,40,1,26,1,1,354,1,1,31,107,312,31,17,
  107,221,93,38,1,1,107,565,1,1,707,707,1,883,1,1,1,38,1,
  870,1,870,1,437,102,31,31,1,102,1,38,1,1,502,20,157,172,
  31,385,20,1,157,38,1,172,1,484,502,1,872,1,559,1,1,1,467,
  535,1,515,1,1,213,284,1,172,1,247,1,302,690,1,1,99,399,1,
  220,845,246,220,313,548,1,581,1,639,1,1,1030,1,1,1020,1,1,
  1,944,1,714,1,1,158,107,1,1,255,1,485,1,1,1,63,283,1,26,
  541,344,344,487,49,1,396,591,713,1,1,1,963,1005,1,1,1,1,
  637,1,839,1,1,38,40,1,102,1,351,191,1,377,38,38,284,20,
  313,38,351,93,1,1,99,63,1,465,1,537,1,393,586,1,1,193,
  393,1,1,1,356,40,1,107,1,1,333,1,687,40,31,399,40,526,26,
  38,93,1,1,399,283,1,1,1,883,1,1,216,1,1,870,870,1,1,1,
  38,1,102,1,481,477,1,1,102,31,514,31,385,102,38,40,220,1,
  220,26,1,558,1,1,559,467,784,1,1,1,515,1,1,1,515,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,270,1,1,571,123,45,1,20,353,146,422,123,344,
  1,246,344,1,269,1,965,630,102,1,1,1,874,674,1,400,1,400,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,370,1,1,191,38,220,220,782,330,22,49,1,1,1,
  563,942,1,1,1,1,1,820,1,1,1,772,1,955,1,577,1,1,268,128,
  1,120,1032,451,1,1,471,1,332,233,1,1,1,1,780,743,1,1,1,
  1045,1,1,961,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,321,102,1,572,570,570,609,661,920,
  1,1,1,1,1,1,1,728,1,1,1,1,1,821,1,1,1,1,38,1,40,1,
  102,1,123,1,1,213,223,1,321,1,586,393,193,741,1,1,356,1,
  1,1,1,382,1,40,1,107,1,157,1,1,312,305,1,565,1,216,1,
  870,864,1,1,38,1,1,1,1,1,220,1,1,302,1,1,1,1,539,1,910,
  1,1007,714,944,1,1,1,1013,1,1,1,1,1,1,102,220,157,481,1,
  514,437,558,1,1,784,467,1,747,1,1,515,1,1,467,1,535,570,
  1,1,1,1,827,1,1,1,1,843,595,1,1,1,1,595,1,1,1,1,1,344,
  123,330,1,1,351,1,1,1,1,102,874,1,1,1,400,400,1,874,1,
  674,1,1,1,504,1,1,1,1,851,955,772,1,1,1,567,1,1,1,1,
  1060,455,1,1,1,1,464,1,1,394,394,1,1,1,684,1,1,684,1,1,
  1,220,344,1,942,1,995,953,955,851,1,1,772,1,1,1,1,567,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,213,910,1,1,1,722,1,
  1030,1,1,1,1013,1,944,1,1,824,1,1,1,1,963,1,1,1,1005,1,
  1,637,1,1,1,1,1,1,708,1,1,1,1,1,1,1,1,708,1,1,1,1,
  960,1,1,1,1,1,708,1,796,1,1,1,1,1,1,1,1,1,1,1,1,707,
  1,1,1,1,614,252,1,1,1,1,1,1,1,1,45,45,1,1,1,1,707,1,
  1,1,1,1,45,1,45,1,1,1,1,1,252,1,614,1,1,1,1,1,1,1,1,
  1,1,1,1,1,400,1,1,1,1,1,1,1,1,1,1,1,1,400,1,7,5,7,7,
  54,54,131,54,131,75,227,1,54,131,75,347,202,1,1,1,1,1,54,
  131,319,75,1,202,258,35,258,347,227,1,1,404,1,1,1,1,1,
  500,500,1,296,500,296,500,1,7,3,7,54,54,131,54,131,75,347,
  258,54,131,75,227,202,1,1,319,404,1,54,131,1,75,1,202,1,
  35,1,227,347,258,1,1,1,296,500,500,296,1,1,500,1,1,1,1,
  500,7,5,31,31,31,31,31,45,212,215,31,31,45,1,146,1,1,126,
  86,128,31,31,212,45,139,146,1,8,1,1,212,215,128,128,86,13,
  26,26,13,13,13,26,26,13,26,13,26,7,54,54,131,54,131,75,
  347,258,54,131,75,227,202,1,1,319,404,1,54,131,1,75,1,202,
  1,35,1,227,347,258,1,1,1,296,500,500,296,1,1,500,1,1,1,
  1,500,26,26,20,26,20,26,1,194,26,20,26,1,17,1,1,195,86,
  34,26,20,195,26,86,17,194,8,194,1,1,194,34,86,86,81,102,
  102,81,102,102,102,81,102,81,102,102,40,40,31,102,102,347,
  1,31,20,125,348,125,1,1,1,1,1,31,20,307,20,1,146,228,146,
  107,128,457,457,318,485,1,1,1,383,1,1,383,1,553,383,622,1,
  1,40,40,40,26,457,457,31,102,102,128,26,1,1,298,374,1,31,
  20,316,125,384,125,107,17,322,1,1,340,318,1,545,1,383,1,
  498,383,1,1,622,1,460,1,1,102,11,102,1,148,20,83,70,346,
  83,1,1,1,402,176,20,11,26,81,365,16,158,81,346,191,186,1,
  17,102,102,356,233,102,102,20,233,233,356,233,70,34,34,40,
  102,1,340,40,40,26,1,26,1,1,38,1,287,31,102,38,102,485,
  26,322,36,340,1,1,322,287,1,485,498,1,1,460,1,1,1,460,
  383,498,1,383,102,186,1,102,11,102,191,16,1,1,191,140,140,
  20,83,426,70,81,83,346,24,63,1,1,63,176,402,365,102,102,
  233,356,233,34,34,70,20,356,233,233,436,1,125,70,69,1,125,
  1,1,1,1,1,20,81,54,116,1,81,536,81,1,254,235,254,419,543,
  1,1,1,433,1,524,1,478,568,439,1,478,1,457,1,1,235,1,458,
  1,1,508,1,923,128,191,95,254,837,139,254,391,1,1,652,1,
  578,1,1,663,1,1,896,1,1,876,748,666,610,1,836,322,63,254,
  254,234,1,1,1,1,1,107,346,547,1,773,423,423,1,132,1,233,
  520,667,1,1,1,1,551,1,552,756,1,1,492,685,781,1,40,102,1,
  125,1,1,316,1,318,40,40,298,26,545,26,340,17,457,457,128,
  107,1,374,384,460,1,1,622,1,1,383,498,1,1,383,1,102,1,83,
  1,1,426,402,176,102,11,191,102,365,16,63,24,1,186,191,346,
  140,140,81,356,233,34,70,34,233,233,356,233,102,102,20,254,
  125,1,1,509,384,1,125,70,509,69,477,125,1,22,254,235,254,
  1,1,384,477,1,433,478,1,1,478,1,1,524,1,433,524,1,1,1,1,
  1,1,457,186,1,235,1,139,457,1,233,652,150,1,1,670,915,748,
  1,1,1,1,1,666,663,876,1,628,866,1,1,1,499,162,146,16,252,
  81,489,20,200,146,423,139,139,423,205,95,205,493,330,26,112,
  112,330,330,493,330,112,26,112,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,307,26,163,54,692,252,563,163,547,95,1,320,1,786,1,1,1,
  471,1,790,599,1043,720,1,888,471,1,140,485,102,786,543,102,
  95,895,1,1,1,670,834,1,1,1,1,1,1,1,1,1,363,390,375,1,1,
  1,318,17,1,419,205,205,634,422,667,578,1,653,1,1,1,1,1,1,
  1,1,268,1,418,1,1,1,1,40,1,102,1,125,457,146,1,347,348,
  228,1,1,1,622,1,383,553,1,383,1,1,1,1,1,383,1,102,102,
  83,1,81,148,1,346,158,176,402,365,70,34,233,356,233,102,20,
  102,34,356,233,233,1,1,1,320,163,1,508,1,563,1,1,692,888,
  471,599,720,1043,471,790,1,1,1,1,1,1,125,254,81,1,436,1,
  536,1,1,1,1,478,1,568,478,433,524,1,1,1,1,439,205,1,1,1,
  1,915,1,1,1,1,459,420,1,1,1,379,363,1,1,1,1,1,423,146,
  234,458,1,200,162,499,489,112,26,330,493,330,26,112,112,112,
  493,330,330,1,520,1,1,1,653,834,1,1,756,552,1,551,1,492,
  1,1,833,715,979,1,391,1,1,422,1,1,1,477,477,1,477,477,
  477,1,1,1,477,1,457,254,423,1,1,773,685,781,756,1,1,551,
  552,1,1,1,1,492,1,254,923,1,837,610,1,1,748,876,1,1,896,
  836,663,1,666,457,1,1,1,1,628,1,663,666,1,1,1,866,748,1,
  876,634,895,1,833,715,1,1,492,552,551,1,979,1,756,1,140,
  205,1,1,268,418,1,1,1,1,1,1,1,1,102,1,1,1,390,363,1,1,
  1,1,1,1,375,1,1,379,1,363,1,1,1,1,459,420,1,102,1,1,1,
  70,75,38,112,1,1,1,1,1,1,70,31,70,34,1,1,1,102,34,1,1,
  1,1,70,70,31,112,1,1,1,1,38,70,75,1,1,1,1,75,31,31,34,
  102,31,1,1,1,112,31,1,1,1,75,1,1,1,1,1,1,102,112,34,7,
  7,3,131,131,143,131,143,59,1,251,131,143,59,1,131,271,319,
  258,1,1,131,143,1,59,1,131,1,35,1,1,1,251,1,1,404,707,1,
  1,707,626,626,1,1,626,1,626,1,5,7,31,31,38,31,38,43,113,
  146,31,38,43,1,45,172,92,157,1,1,31,38,106,43,244,45,1,8,
  1,1,113,146,1,308,185,24,1,1,24,148,148,1,34,148,34,148,
  1,7,31,31,45,31,45,16,86,128,31,45,16,86,31,128,128,139,
  1,1,31,45,139,16,1,31,128,8,128,86,86,128,1,1,1,518,1,1,
  518,1,1,1,518,1,518,1,1,131,131,143,131,143,59,1,251,131,
  143,59,1,131,271,319,258,1,1,131,143,1,59,1,131,1,35,1,1,
  1,251,1,1,404,707,1,1,707,626,626,1,1,626,1,626,1,107,
  107,31,75,75,292,1,13,89,70,546,70,1,292,1,1,1,31,89,292,
  10,423,59,292,59,102,102,134,134,1,423,1,1,1,1,1,826,830,
  1,470,646,547,623,1,107,107,107,10,134,134,31,75,75,102,31,
  186,1,245,1,1,13,89,435,70,598,70,102,36,546,245,1,186,1,
  1,266,1,1,1,623,830,623,1,547,826,543,1,1,75,69,75,1,255,
  89,172,13,297,172,81,200,200,1,1,89,34,134,57,1,89,573,57,
  297,1,297,1,1,570,570,783,1,1,1,495,1,1,699,991,66,1,1,
  107,75,1,186,107,107,10,245,31,134,134,102,1,1,31,75,102,
  75,579,31,546,13,292,1,292,1,1,1,366,623,1,1,830,623,1,1,
  543,830,412,1,1,75,297,1,75,69,75,1,34,1,297,1,1,1,89,
  172,281,13,269,172,297,76,134,200,200,81,1,523,1,1,1,1,1,
  1,1,1,66,495,891,783,1,274,68,70,13,8,337,70,174,19,174,
  1,1,10,57,19,8,310,10,274,57,19,174,19,174,1,310,1,542,1,
  1,1,542,1,1,395,569,221,542,1,134,292,200,19,200,292,245,
  203,324,1,1,102,1,128,65,1,102,65,1,328,1,328,1,1,1,711,
  797,1,1,1042,736,1,1,1,1,618,1,1,1,81,174,174,1,1,245,1,
  1,1,26,255,313,13,584,134,134,120,95,241,235,126,1,779,1,
  1,1,1,1,1,1,1,445,752,1047,926,1,107,75,1,70,1,292,546,
  1,1,107,107,292,10,443,31,292,36,200,200,102,26,1,443,366,
  830,1,1,826,1,1,1,412,623,1,646,1,75,200,172,255,1,297,1,
  1,75,69,573,75,1,34,134,76,1,297,1,255,1,1,40,1,1,1,495,
  1,783,1,891,1,794,794,1,174,70,68,274,337,1,1,70,13,337,
  8,350,70,19,8,65,19,65,13,1,350,246,1,1,1,542,1,542,1,
  221,542,1,569,1,1,1,324,1,1,1,200,297,328,19,648,102,200,
  288,235,328,150,313,1,603,1,1,1,1,1052,1,1,1,673,916,1,
  865,1,1,292,1,1,1,59,89,245,10,507,31,292,59,134,102,102,
  134,1,102,1,1,1,1,1,1,1,1,401,759,1,216,1,134,174,1,1,
  186,194,194,13,584,186,245,32,297,146,297,22,1,779,1,1,1,
  1,1,1,1,1,445,752,1,926,1,200,1,1,1,128,116,116,1,546,
  128,144,1,328,1,297,1,1,392,1,1,1,736,1042,797,1,1050,
  1062,987,1,1,1,1,245,128,189,19,727,292,324,189,324,150,
  328,146,1,587,1,1,1,1,1,1052,1,1,742,812,939,757,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,107,1,75,1,70,200,59,1,245,1,186,1,579,266,826,1,
  1,646,1,646,1,1,1,1,826,1,194,75,1,172,1,57,297,1,297,
  593,1,1,1,495,1,1,991,783,794,1,794,1,397,991,1,116,1,1,
  203,189,1,328,1,241,1,1059,549,1,1,1,916,1061,757,1,1,1,
  1,1,1,1,70,65,57,274,337,274,68,1,1,246,542,1,1,569,542,
  569,1,1,1,221,1,1,1,1,1,1,962,1,1,1,1,1,1,1,1,1,1,
  960,1,1,1,1,1,1,134,59,1,292,1,186,1,754,192,1,1,1,759,
  1,216,1,1,1,1,982,1,1,328,1,328,1,1,1,1,736,1,1,1,797,
  1037,1,1,1,746,1058,1,288,1,288,1,1,1,1,645,1,1,1,645,1,
  1,1,1,352,1,1,200,65,134,1,1,392,1,1,1,736,1,797,1,1,
  1066,1,1037,1,200,174,1,1,1044,757,1,1,1,916,1,1,1,1054,
  673,1,1,134,1,1,602,1,1,1,797,1,736,1,1,1051,1,1,1,1,
  966,1,926,1,1,1,752,1,1,1,1,445,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,570,1,1,1,954,1,1,1,1,1,821,1,1,1,1,1,1,
  899,1,1,1,1,595,886,1,1,1,1,1,1,1,252,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,252,1,1,
  1,1,1,1,647,406,1,1,794,406,1,1,1,1,1,1,1,1,647,1,1,1,
  1,1,1,794,1,1,7,7,38,38,54,38,54,136,379,1,38,54,136,1,
  239,1,538,1,1,1,38,54,1,136,1,239,1,54,1,1,379,1,1,532,
  1,1,296,296,1,1,1,296,1,1,1,1,296,7,54,54,131,54,131,75,
  347,258,54,131,75,227,202,1,1,319,404,1,54,131,1,75,1,202,
  1,35,1,227,347,258,1,1,1,296,500,500,296,1,1,500,1,1,1,
  1,500,131,131,143,131,143,59,1,251,131,143,59,1,131,271,
  319,258,1,1,131,143,1,59,1,131,1,35,1,1,1,251,1,1,404,
  707,1,1,707,626,626,1,1,626,1,626,1,38,38,31,40,40,268,1,
  11,31,102,191,102,213,223,1,253,1,31,31,1,20,1,123,1,123,
  1,26,1,1,1,465,1,1,586,741,1,193,356,1,1,393,1,382,1,38,
  38,38,20,1,1,31,40,40,26,38,380,1,223,313,1,11,31,1,102,
  1,102,1,93,1,179,1,268,1,1,465,1,741,1,741,356,382,1,1,
  193,1,1,654,40,31,40,1,447,31,107,31,221,107,145,160,160,
  526,1,31,31,1,157,1,45,1,157,1,269,447,1,1,1,1,1,864,1,
  1,38,1,1,1,870,1,707,707,38,40,1,268,38,38,20,179,38,213,
  213,63,1,1,31,40,1,40,1,38,1,11,1,1,260,191,1,1,321,741,
  1,1,586,382,1,654,1,356,1,1,537,40,447,1,40,31,40,269,26,
  1,530,305,1,1,31,107,1,31,1,107,1,93,1,370,370,128,1,1,
  565,1,1,1,864,1,707,707,1,38,1,1,883,514,514,102,31,31,1,
  102,224,26,224,1,1,20,157,1,31,1,20,1,157,1,246,120,246,
  1,558,1,696,1,696,1,535,1,872,1,515,1,559,1,1,260,370,
  120,370,324,332,233,607,1,1,128,354,1,1,1,198,1,295,1,1,
  725,1,1,1,1,1033,1,1,1035,743,1045,1040,1,780,1,1,1024,
  191,128,246,246,358,1,1,1,793,1,128,333,1,120,1,1,1,1,1,
  471,359,725,1,1021,1,1,1035,1033,1,852,961,1,1,732,1,1,
  1063,38,40,1,102,1,283,158,1,1,38,38,1,20,1,38,1,93,1,1,
  128,128,1,509,377,586,1,654,537,1,1,537,1,382,1,393,586,
  40,370,107,517,1,283,1,1,40,31,1,40,1,26,1,93,1,1,354,
  333,1,1,102,864,1,707,216,707,1,883,1,1,1,1,216,246,102,
  1,437,437,381,1,102,31,1,31,1,102,1,40,1,38,1,120,1,502,
  484,1,696,872,747,1,559,1,1,535,1,467,784,1,1,563,1,1,1,
  1,1,1,38,1,54,1,481,1,1,95,1,1,682,927,995,1,820,1,1,1,
  996,1,1,1,577,1,376,279,1,1,1,123,45,1,20,1,146,1,123,1,
  54,198,1,1,269,1,1,1,450,735,735,630,965,1,674,1,102,1,
  213,224,1,1,284,1,1,172,1,247,1,194,1,302,690,186,1,1,
  969,1007,1,1030,1000,1,1,1020,1,722,1,1,989,160,935,1,158,
  107,1,1,1,255,1,157,1,485,1,607,1,1,744,1,1,1,999,963,
  1005,1,1,1,1,1,929,641,1,283,107,1,26,1,351,1,1,1,49,1,
  233,1,744,1,1,973,999,1,869,839,1,1,925,1,637,1,1,313,1,
  1,381,1,40,1,263,1,1,561,1,1,1,1,843,1,1,1,1,1,886,1,
  569,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,38,1,40,1,102,1,123,1,351,191,260,1,377,
  509,537,654,537,654,1,393,586,1,1,1,193,741,1,40,1,107,1,
  157,1,1,333,354,1,687,687,216,707,883,883,1,1,216,1,707,1,
  870,864,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,102,1,157,1,481,477,477,1,1,572,747,872,1,784,559,
  467,784,1,1,1,1,747,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,123,1,270,1,451,1,571,1,735,450,965,1,630,
  102,1,1,735,1,874,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,269,546,1,1,413,1,986,655,738,1,684,394,1,1,1,
  1,394,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,370,246,
  1,1,920,851,1,1,995,1,1,820,1,880,1,1,996,1,1,1,1021,1,
  998,1,1033,780,743,1,1,964,1,1045,1040,1,1,1,1040,1001,
  1035,1,732,852,1033,1,1056,1,961,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,954,728,1,1,1,1,1,1,1,1,1,954,1,821,
  1,1,1,1,1,821,1,1,1,1,1,252,1,1,735,1,1,1,1,1,1,614,
  406,1,707,1,1,1,1,707,1,1,1,1,1,614,406,735,1,1,1,1,1,
  252,1,1,1,1,1,1,45,252,707,1,45,1,1,1,1,252,1,1,1,1,1,
  1,1,1,1,1,1,1,707,7,54,54,131,54,131,75,1,317,54,131,75,
  1,202,227,1,327,1,1,54,131,347,75,1,202,1,35,1,1,1,317,
  1,1,404,421,1,1,421,1,1,1,500,1,500,1,1,136,136,53,136,
  53,213,1,1,136,53,213,1,106,1,1,379,215,185,136,53,379,
  213,215,106,1,17,1,1,1,1,185,215,215,157,134,134,157,134,
  134,134,157,134,157,134,134,40,40,31,102,102,307,1,31,20,
  125,1,125,348,228,1,485,1,31,20,347,20,545,146,1,146,1,1,
  318,318,483,287,1,1,622,553,1,383,1,1,383,1,1,383,1,40,
  40,40,26,318,318,31,102,102,1,26,347,1,1,545,1,31,20,348,
  125,485,125,1,17,1,1,1,349,1,1,374,1,553,1,551,1,383,1,
  1,383,383,1,498,102,11,102,1,237,20,83,70,1,83,128,213,
  213,365,402,20,11,128,81,365,16,1,81,1,1,177,1,81,140,140,
  151,356,81,81,20,233,356,233,233,20,102,102,40,102,1,349,
  40,40,26,1,26,457,457,145,1,485,31,102,128,102,384,26,1,
  36,1,1,298,383,374,1,1,551,1,1,551,383,1,498,383,1,1,1,
  460,102,177,1,102,11,102,1,16,1,148,213,102,102,20,83,346,
  70,81,83,1,24,1,1,1,99,402,176,402,81,81,356,151,233,102,
  102,20,20,233,233,356,477,338,125,70,69,1,125,285,146,285,
  1,543,20,81,246,116,477,81,1,81,1,1,81,1,1,1,1,611,1,
  631,1,439,1,1,439,524,524,433,1,318,298,1,81,1,1,1,320,
  660,1,834,38,191,95,1,692,34,1,308,1,1,1,1,615,1,823,616,
  1,1,902,471,1043,761,599,1,471,1,1,383,99,1,1,335,1,497,
  1,972,1,145,455,332,304,1,1,1,532,1,1,302,617,1,1,1,1,
  959,734,1,665,849,1,765,1,802,733,1057,40,102,1,125,1,340,
  383,1,483,40,40,1,26,1,26,1,17,1,1,38,145,1,287,483,551,
  1,498,551,1,1,460,1,383,1,1,622,102,1,83,186,1,455,365,
  402,102,11,191,102,365,16,1,24,1,1,191,455,140,140,81,151,
  356,102,24,102,233,356,233,233,34,34,70,1,125,436,1,343,
  491,1,125,70,1,69,1,125,1,22,1,1,1,304,1,1,1,1,631,1,1,
  1,433,1,524,439,1,478,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,235,534,1,489,
  499,146,16,297,81,489,20,1,146,1,1,34,1,131,22,131,208,
  493,81,146,146,330,493,330,330,112,26,112,457,285,1,1,1,1,
  1,235,1,458,1,144,1,1,508,186,1,923,901,1,1,663,726,1,1,
  896,1,628,1,876,695,213,1,1,322,63,254,254,893,234,1,294,
  1,1,1,1,1,1,751,1010,833,1,1,1,551,1,552,715,756,1,818,
  1,1,349,99,430,304,900,324,1,430,1,1,504,1,1,705,1,1,855,
  1,1,1,802,1027,733,716,849,765,1,102,1,134,1,384,134,95,1,
  468,1,1,705,918,1,1,1,1,1,1,1,1,1,390,1,363,1,1,1,1,
  81,928,473,235,235,1,468,1,1,1,785,1,1,1,1,1,1,1,1,420,
  1,352,363,1,1,1,40,1,102,1,125,1,146,1,1,316,1,1,318,
  485,551,498,460,551,1,1,622,1,1,1,383,553,1,102,134,83,1,
  81,1,1,426,213,402,176,402,24,102,356,151,233,34,70,34,102,
  233,233,356,254,1,1,1,430,1,1,1,690,1,1,1,1,610,748,938,
  1,1,1,1,1,1,666,831,1,125,1,81,1,1,509,343,384,1,473,1,
  1,1,1,433,478,1,1,1,478,1,568,235,1,1,1,1,1,1,1,1,1,1,
  363,1,1,1,1,1,1,1,1,1,1,1,146,1,1,1,324,499,162,499,
  146,81,493,208,330,26,112,112,146,330,330,493,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,387,1,1,422,1,1,525,
  1,1,477,477,1,1,1,477,477,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,318,
  1,1,1,1,612,1,616,1,471,1,790,729,599,1043,761,1,815,1,
  734,755,959,1,1,665,734,1,1048,765,849,1,140,131,1,1,1,1,
  1,1,1,1,1,1,363,1,140,1,1,1,410,1,1,1,1,1,268,1,376,1,
  1,459,1,375,1,1,1,1,420,352,1,81,1,1,1,52,57,52,146,1,
  1,1,1,1,1,75,70,70,102,1,1,1,81,102,1,1,1,1,70,75,70,
  146,1,1,1,1,52,52,57,1,1,1,1,70,31,45,102,34,31,1,1,1,
  112,45,1,1,1,70,1,1,1,1,1,1,34,112,102,131,131,143,131,
  143,59,1,1,131,143,59,1,131,319,271,1,1,1,131,143,258,59,
  404,131,251,35,251,1,1,1,1,1,1,1,626,626,1,1,1,626,707,
  1,707,1,626,107,107,31,75,75,245,1,13,89,70,1,70,1,186,1,
  579,1,31,89,245,10,266,59,186,59,26,102,200,200,1,1,1,1,
  826,646,1,1,1,1,646,1,826,1,1,107,107,107,10,200,200,31,
  75,75,102,31,292,1,292,443,1,13,89,546,70,366,70,26,36,1,
  292,1,292,1,1,443,1,646,1,412,1,1,1,826,1,830,1,623,75,
  69,75,1,297,89,172,13,297,172,128,194,194,1,1,89,34,128,
  57,1,89,593,57,255,1,297,1,1,1,1,397,991,794,794,1,1,783,
  991,1,495,1,1,107,75,1,292,107,107,10,292,31,134,134,102,
  1,1,31,75,102,75,366,31,1,13,186,1,245,546,1,1,579,412,1,
  1,543,1,1,623,830,1,623,1,830,75,297,1,75,69,75,1,34,1,
  255,573,1,1,89,172,297,13,40,172,255,76,81,200,200,134,1,
  1,1,794,794,783,891,1,1,1,495,1,1,1,1,337,274,70,13,8,
  274,70,116,13,116,1,1,10,57,19,8,246,10,68,57,13,65,19,
  65,1,1,1,221,1,569,1,1,1,542,569,1,542,1,1,200,245,200,
  19,200,292,328,146,328,1,1,102,1,150,174,1044,102,174,1,
  241,1,328,1,1,1,962,673,1,1,1,1,1,916,1,1,757,1,1054,
  546,134,65,65,1,1,297,1,1,1,102,297,324,19,392,134,134,
  288,95,328,235,328,1,1,1,1,1037,797,1,1,1,1,736,1,1,1,
  1066,107,75,1,70,1,186,435,1,1,107,107,245,10,266,31,186,
  36,134,134,102,102,1,1,598,543,1,623,547,1,1,830,623,1,1,
  1,826,75,200,172,297,1,281,523,1,75,69,1,75,1,34,81,76,1,
  297,1,297,1,1,269,891,783,1,66,1,1,1,1,1,1,1,495,65,70,
  274,68,337,350,1,70,13,337,8,246,70,13,8,174,19,174,19,1,
  1,350,1,569,542,221,1,1,1,542,1,1,1,542,1,1,297,1,1,1,
  134,297,328,19,602,102,134,288,235,328,150,328,1,1,1,1,1,
  736,1,1,1,1,797,1,1,1,1051,546,186,1,754,1,59,89,292,10,
  192,31,186,59,134,102,102,134,1,1,1,1,982,216,1,1,1,1,
  759,1,1,1,1,134,116,1,1,292,200,200,19,392,292,297,144,
  245,203,324,128,1,1,1,987,1,797,1050,1,1,1042,736,1,1,1,
  1062,194,779,1,1,81,174,174,1,1,22,32,1,245,1,245,1,1,
  584,1,926,1,445,1,1,1,1,1,1,1,752,1059,1,292,134,189,19,
  549,245,241,189,313,128,328,203,1,1,1,1,1,757,1,1,1,1061,
  916,1,1,1,1,1,423,570,587,310,570,102,966,1,779,1,603,1,
  1,1,1,821,1,1,1,1,1,1,954,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,107,1,75,1,70,
  134,59,1,292,546,292,1,1,423,547,623,830,470,1,1,826,1,1,
  1,1,646,200,75,570,172,1,57,255,1,297,573,1,1,1,66,1,1,
  699,1,1,495,1,1,783,1,991,174,1,1,146,189,1,324,1,324,1,
  1,727,939,757,1,742,1,1,1052,1,1,1,1,812,1,70,174,57,68,
  274,337,274,1,1,310,221,542,1,395,1,1,542,1,1,542,1,569,
  1,1,1,1,711,1,1,1,1,1,595,886,1,1,1,1,899,1,1,1,1,1,
  134,59,1,292,1,292,1,1,507,1,216,1,401,1,1,1,1,1,1,1,
  759,1,126,1,313,1,1,1,1,445,1,1,1,1,1,752,1,1,926,1,1,
  120,1,288,1,1,1,1,352,1,1,1,1,1,645,1,1,645,1,1,134,
  174,134,1,1,584,1047,926,1,445,1,1,1,1,1,1,1,752,200,65,
  1,1,1,618,1,1,1,1,1,736,1042,1,797,1,1,200,1,1,648,1,
  865,1,673,1,1,1,1052,1,1,1,916,1,1,1,746,1058,1037,1,1,
  1,797,1,1,736,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,960,1,1,1,1,1,1,1,1,1,794,1,
  1,1,1,647,252,1,1,1,1,1,1,1,1,406,1,1,1,1,1,794,1,1,
  1,1,1,1,1,406,1,1,1,1,1,252,1,647,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,200,200,116,
  186,186,1,1,22,81,102,1,102,376,1,1,491,1,116,81,308,20,
  473,81,373,81,198,1,1,1,1,477,1,1,1,590,1,1,701,1,576,
  590,469,1,1,200,200,200,20,1,1,116,186,186,1,70,379,1,383,
  1,1,22,81,330,102,384,102,198,104,358,1,1,390,1,1,1,1,
  590,1,551,701,1,1,469,1,498,1,1,186,20,186,1,287,81,147,
  26,1,147,110,364,364,483,148,81,20,34,146,502,16,276,146,1,
  1,1,1,22,235,235,318,494,205,205,95,494,494,318,494,22,131,
  131,200,186,1,390,200,200,20,1,70,360,360,54,1,419,116,186,
  54,186,543,70,358,32,390,1,1,358,419,1,543,551,1,1,498,1,
  1,1,498,701,551,1,701,186,1,1,186,20,186,1,16,1,310,212,
  218,218,81,147,395,26,95,147,1,32,139,1,1,139,148,483,502,
  205,205,494,318,494,131,131,22,95,318,494,494,1,514,102,26,
  31,1,102,220,172,220,1,287,20,146,38,31,483,20,477,146,120,
  1,1,1,318,1,1,1,1,511,1,482,1,669,434,717,1,669,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,358,139,1,1,191,1,690,1,1,1,198,
  1,1,120,670,1,1,475,150,1,1,652,1,936,1,1,948,621,1,1,1,
  1,659,693,671,970,1031,200,186,1,102,1,379,330,1,1,200,200,
  383,20,1,70,390,104,1,1,1,198,1,1,384,498,1,1,469,1,1,
  701,551,1,1,590,1,186,1,147,310,1,395,483,148,186,20,212,
  186,502,16,139,32,1,1,1,1,218,218,95,318,494,131,22,131,
  494,494,318,494,205,205,95,1,102,1,1,1,545,1,102,26,1,31,
  1,102,120,36,1,1,1,120,1,545,1,1,511,669,1,1,669,1,1,
  482,1,511,482,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,213,455,1,365,176,81,16,
  177,20,402,11,186,81,1,1,1,1,140,81,140,253,419,81,140,
  140,419,419,253,419,140,81,140,360,220,1,770,1,364,364,172,
  1,455,332,1,690,1,1,186,1,900,913,1,1,629,677,1,1,1003,1,
  1,1,809,817,364,900,1,376,110,220,220,913,213,186,1,1,1,1,
  332,770,1,1,1,809,1,1,1003,629,1,677,817,1,1,1,798,1,308,
  34,163,38,1,177,471,163,1,1,1,280,1,578,1,1,542,1,1,1,
  805,956,582,611,860,723,1,218,477,235,578,1,235,81,931,468,
  936,1,1,789,1,1,1,422,1,1,1,1,1,1,1,462,1,1,1,1,22,1,
  318,140,140,578,422,1,1,1,923,1,1,1,1,357,1,1,1,432,1,1,
  357,1,1,1,200,1,186,1,102,1,81,1,1,1,373,1,491,473,469,
  1,701,576,1,590,1,1,1,1,1,590,364,186,235,147,1,146,287,
  1,1,276,148,483,502,22,131,494,318,494,205,95,205,131,318,
  494,494,220,1,1,280,163,1,1,1,471,1,798,1,860,723,805,582,
  956,1,1,1,1,1,542,611,1,102,1,146,514,1,1,477,287,1,483,
  1,669,1,434,669,511,482,1,1,1,1,717,140,1,1,1,1,1,1,1,
  1,1,315,1,1,1,1,391,1,1,1,1,1,1,1,81,191,1,1,186,176,
  365,402,140,81,419,253,419,81,140,140,140,253,419,419,1,652,
  1,1,1,923,789,1,659,1,1,1,621,948,693,1,1,1,795,1,475,1,
  1,1,422,468,1,551,1,583,1,583,583,583,1,1,551,1,1,1,1,1,
  1,1,670,671,970,1,659,1,621,1,1,1031,1,948,693,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,578,931,1,1,795,948,1,693,1,621,1,1,659,1,1,218,
  140,1,1,432,1,1,1,1,1,1,1,357,357,235,1,1,1,1,1,1,1,1,
  1,422,1,462,1,1,391,1,1,1,1,1,1,315,1,1,205,1,1,1,89,
  146,45,140,1,1,1,1,1,1,214,86,89,131,1,1,1,205,131,1,1,
  1,1,89,214,86,140,1,1,1,1,45,89,146,1,1,1,1,146,86,41,
  131,205,86,1,1,1,140,41,1,1,1,146,1,1,1,1,1,1,205,140,
  131,3,7,7,7,1,1,5,7,7,10,7,10,1,1,19,1,5,5,10,7,19,7,
  10,7,10,10,1,1,10,1,1,1,26,1,1,16,26,1,26,26,16,1,1,7,
  7,7,1,1,5,7,7,10,7,10,1,1,19,1,5,5,10,7,19,7,10,7,10,
  10,1,1,10,1,1,1,26,1,1,16,26,1,26,26,16,1,1,7,7,1,1,
  5,5,5,9,5,16,16,1,1,1,4,5,7,5,11,5,9,5,7,1,16,16,11,
  1,1,1,1,1,1,11,1,1,11,11,11,1,1,3,14,1,7,5,7,14,7,1,
  10,1,1,1,5,7,10,5,10,7,14,7,5,7,7,7,19,19,1,1,1,26,1,
  26,1,1,16,16,26,26,1,14,1,7,5,7,14,7,1,10,1,1,1,5,7,
  10,5,10,7,14,7,5,7,7,7,19,19,1,1,1,26,1,26,1,1,16,16,
  26,26,1,1,9,16,10,16,10,29,9,1,1,1,7,9,13,16,36,10,16,
  10,20,1,1,1,20,1,1,1,1,1,1,31,149,1,52,31,31,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,5,7,1,7,1,9,1,1,1,5,5,9,4,11,5,
  9,5,5,5,5,5,1,11,1,1,1,11,1,1,1,1,11,11,1,11,1,7,16,
  7,9,1,1,10,1,5,4,9,5,10,5,7,5,1,7,16,16,16,16,1,1,11,
  16,1,16,11,1,11,11,16,16,1,1,3,14,14,1,19,1,7,5,14,5,
  19,7,10,7,7,5,7,7,1,10,1,1,29,16,1,1,29,1,29,29,1,16,
  1,1,1,29,1,1,1,16,9,29,7,36,10,16,10,16,9,10,10,1,20,
  1,1,149,31,1,1,1,1,31,52,1,31,1,14,14,1,19,1,7,5,14,5,
  19,7,10,7,7,5,7,7,1,10,1,1,29,16,1,1,29,1,29,29,1,16,
  1,16,1,1,1,9,16,16,7,20,10,29,10,9,16,10,10,1,36,1,1,
  1,31,1,1,1,1,31,31,149,52,1,1,32,1,9,7,16,16,32,10,13,
  10,1,20,1,1,29,1,1,1,38,1,1,39,70,1,70,38,39,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,11,16,20,11,16,10,29,10,16,1,43,43,1,1,1,1,1,1,
  1,1,1,1,31,31,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,5,1,5,1,7,16,7,1,7,16,16,1,
  11,1,1,11,11,1,1,11,1,1,1,1,11,1,16,5,16,7,1,7,7,1,
  16,16,10,10,1,1,16,11,1,11,16,1,16,16,11,11,1,16,1,1,9,
  1,1,20,1,1,1,36,1,1,31,52,1,149,31,1,1,1,1,31,1,1,7,
  16,7,6,6,10,10,11,1,1,1,11,1,1,11,11,1,1,1,11,1,1,1,
  1,1,1,26,1,1,1,1,1,1,31,1,1,1,31,1,1,1,1,1,1,1,3,10,
  10,1,1,19,19,1,1,16,29,1,29,16,1,1,1,29,29,1,1,20,1,1,
  1,32,32,1,1,39,70,1,70,39,1,1,1,38,38,1,10,10,1,1,19,
  19,1,1,16,29,1,29,16,1,1,1,29,29,1,5,7,7,1,26,1,1,16,
  26,1,1,11,1,1,26,1,26,1,7,7,16,1,1,1,1,26,1,16,1,1,
  26,26,11,1,1,3,43,1,1,1,59,1,1,38,38,1,1,278,59,278,1,
  43,1,1,1,59,1,1,38,38,1,1,278,59,278,1,16,1,1,1,31,1,
  1,1,1,1,1,1,31,1,1,1,1,1,1,31,1,1,1,1,31,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,38,1,
  11,16,1,1,1,1,16,1,1,1,1,11,38,1,1,1,1,1,1,1,1,1,1,
  1,1,1,38,11,1,1,16,11,1,1,1,1,1,1,1,1,38,1,1,1,1,1,
  1,16,1,1,7,7,7,1,1,5,7,7,10,7,10,1,1,19,1,5,5,10,7,
  19,7,10,7,10,10,1,1,10,1,1,1,26,1,1,16,26,1,26,26,16,
  1,1,3,7,1,1,7,7,7,10,5,1,1,10,1,1,5,7,10,7,10,7,10,
  5,10,1,1,10,19,1,19,1,1,1,26,26,1,1,16,16,26,1,26,7,1,
  1,7,7,7,10,5,1,1,10,1,1,5,7,10,7,10,7,10,5,10,1,1,10,
  19,1,19,1,1,1,26,26,1,1,16,16,26,1,26,10,10,5,5,5,6,5,
  16,1,16,1,1,4,5,7,5,11,5,6,5,7,16,1,16,11,1,1,1,1,1,
  1,11,1,1,11,11,11,1,1,3,16,7,7,7,16,1,1,1,1,1,5,16,
  10,7,43,7,7,16,10,1,1,1,43,1,1,1,1,1,1,38,278,1,59,38,
  59,1,278,16,7,7,7,16,1,1,1,1,1,5,16,10,7,43,7,7,16,10,
  1,1,1,43,1,1,1,1,1,1,38,278,1,59,38,59,1,278,7,7,1,5,
  1,1,9,1,1,5,5,9,5,11,5,7,4,16,16,1,7,1,1,11,1,1,1,
  11,1,1,1,11,11,1,1,11,3,7,7,14,1,14,19,1,7,5,14,7,19,
  7,5,5,1,10,1,10,1,1,10,1,26,1,16,1,26,1,26,26,1,1,16,
  7,7,14,1,14,19,1,7,5,14,7,19,7,5,5,1,10,1,10,1,1,10,
  1,26,1,16,1,26,1,26,26,1,1,16,1,1,1,1,1,1,5,7,9,5,16,
  5,5,7,16,20,1,20,1,1,26,1,26,1,26,1,1,1,11,16,1,1,26,
  9,1,1,10,1,5,4,9,5,10,5,7,5,16,7,1,16,16,1,16,1,10,1,
  16,16,10,1,10,10,16,1,16,1,16,1,1,9,16,16,10,20,10,20,
  7,29,9,1,13,1,1,36,1,1,1,31,1,1,1,31,31,149,1,52,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,32,1,9,7,16,10,32,10,20,16,29,13,1,9,29,1,1,1,
  70,1,1,39,38,1,38,70,39,1,1,1,11,16,20,10,16,10,26,11,
  36,1,1,36,1,1,1,1,1,1,1,1,1,1,31,31,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,5,1,7,
  1,7,5,5,1,9,1,9,1,1,11,1,11,1,11,1,11,1,1,1,1,1,11,
  16,7,16,7,1,5,9,1,1,9,10,1,10,1,16,1,11,11,16,1,16,16,
  11,1,11,1,1,1,16,16,1,29,1,29,1,1,36,1,31,1,52,149,31,
  1,1,1,1,1,31,1,3,7,7,14,14,1,14,19,1,19,1,16,1,29,29,
  16,1,1,1,29,1,29,1,1,1,1,29,1,1,1,1,1,1,31,1,1,1,31,
  1,1,1,1,1,1,7,7,14,14,1,14,19,1,19,1,16,1,29,29,16,1,
  1,1,29,1,29,1,9,1,1,1,16,1,1,1,26,1,1,11,26,1,1,1,16,
  1,26,9,9,1,1,11,1,1,1,11,1,1,11,11,1,1,1,11,1,1,16,1,
  16,1,1,20,1,52,1,31,1,31,1,1,149,1,1,31,1,16,32,1,32,
  1,1,1,38,38,1,1,39,39,70,1,70,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,20,1,1,1,52,1,1,31,31,1,1,149,31,1,1,1,16,
  1,1,1,31,1,1,1,1,1,1,1,31,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,31,1,1,1,1,31,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,38,1,11,16,1,1,1,1,1,1,1,1,1,1,1,1,16,1,1,1,1,
  11,1,38,1,1,1,1,38,1,11,1,16,11,1,1,1,1,1,1,1,1,38,1,
  1,1,1,1,1,1,16,1,7,3,1,14,7,7,5,14,7,7,7,7,1,19,5,7,
  5,7,19,5,14,7,10,1,10,1,10,1,1,26,1,1,1,16,1,26,26,26,
  16,1,1,7,1,1,5,5,5,9,5,1,16,16,10,10,4,5,7,5,6,5,9,
  5,7,16,16,1,6,10,10,16,16,16,16,7,16,16,7,7,7,16,16,1,
  14,7,7,5,14,7,7,7,7,1,19,5,7,5,7,19,5,14,7,10,1,10,1,
  10,1,1,26,1,1,1,16,1,26,26,26,16,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,9,10,16,16,10,1,1,1,80,1,7,9,20,10,24,16,16,
  10,13,29,9,1,17,1,1,1,86,1,1,20,86,1,34,34,16,74,74,7,
  5,1,7,1,16,16,1,10,5,5,7,5,10,4,9,5,16,16,7,1,1,16,
  16,11,1,16,16,1,1,11,11,11,1,16,16,7,1,3,10,1,1,11,11,
  7,5,10,7,11,5,10,7,1,10,10,1,7,7,7,43,43,7,7,7,43,43,
  43,43,7,7,7,16,7,6,10,10,10,1,5,4,6,5,10,5,7,5,16,7,
  16,1,1,16,16,1,11,16,16,1,11,1,11,11,1,16,16,1,1,1,1,
  1,23,16,9,20,10,23,7,16,10,9,29,13,1,1,43,43,67,187,43,
  43,1,1,38,38,67,1,43,43,10,1,1,11,11,7,5,10,7,11,5,10,
  7,1,10,10,1,7,7,7,43,43,7,7,7,43,43,43,43,7,7,7,7,7,
  1,17,7,5,5,5,10,7,9,5,20,16,20,1,1,26,26,17,1,16,16,1,
  1,17,11,11,10,26,26,3,48,1,16,5,7,7,48,16,10,7,1,10,1,
  1,26,1,1,1,81,1,1,26,81,1,81,81,26,1,1,48,1,16,5,7,7,
  48,16,10,7,1,10,1,1,26,1,1,1,81,1,1,26,81,1,81,81,26,
  1,1,10,10,10,10,19,10,6,80,19,23,1,24,1,1,1,1,36,1,1,
  1,1,1,36,36,36,1,1,1,10,6,17,19,10,10,24,19,23,24,1,1,
  1,1,1,1,36,1,1,1,36,1,36,36,1,1,1,5,1,7,1,5,16,7,1,
  9,9,1,1,16,16,11,11,16,16,1,11,11,1,1,1,16,16,5,7,10,
  5,1,7,9,1,9,1,6,10,10,7,16,16,16,7,16,7,16,16,7,16,16,
  7,1,1,16,7,1,9,1,1,1,26,26,17,11,26,26,10,11,17,1,1,1,
  16,16,1,7,1,3,14,14,14,1,10,1,1,26,26,1,1,16,26,26,1,
  1,16,1,1,10,1,1,1,80,80,1,1,1,1,36,36,1,1,1,36,36,1,
  1,1,1,1,16,7,9,9,1,1,6,10,10,16,7,10,10,6,7,16,16,16,
  6,10,10,1,29,1,29,1,17,1,1,34,86,1,1,20,86,34,1,1,16,
  74,74,14,14,14,1,10,1,1,26,26,1,1,16,26,26,1,1,16,1,1,
  16,16,1,1,43,43,67,67,43,43,1,38,38,1,187,1,43,43,16,1,
  17,1,1,34,1,74,74,16,1,34,86,86,20,1,1,1,17,1,1,1,34,
  1,1,20,34,1,86,86,16,74,74,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,7,7,1,1,10,10,1,1,1,1,1,1,10,10,3,1,1,1,1,16,1,
  1,1,1,16,1,1,1,1,1,1,16,1,1,1,1,16,1,1,16,1,1,1,13,
  48,13,16,1,1,1,1,1,1,48,13,13,16,1,1,1,3,7,1,1,1,1,5,
  7,7,7,1,1,1,1,5,7,7,1,1,1,1,5,5,5,16,16,13,1,1,1,16,
  13,1,1,1,48,1,1,1,1,1,1,7,7,3,7,1,1,7,7,7,10,5,1,1,
  10,1,1,5,7,10,7,10,7,10,5,10,1,1,10,19,1,19,1,1,1,26,
  26,1,1,16,16,26,1,26,1,14,7,7,5,14,7,7,7,7,1,19,5,7,
  5,7,19,5,14,7,10,1,10,1,10,1,1,26,1,1,1,16,1,26,26,26,
  16,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,10,16,16,9,1,1,1,
  1,1,7,10,20,10,20,16,16,9,13,1,29,29,36,1,1,1,1,1,1,
  31,1,1,31,31,52,1,149,3,7,1,7,1,1,10,1,19,7,7,10,7,19,
  5,10,5,1,1,10,10,1,1,10,26,1,1,16,1,1,26,26,26,1,1,16,
  7,1,7,1,1,10,1,19,7,7,10,7,19,5,10,5,1,1,10,10,1,1,
  10,26,1,1,16,1,1,26,26,26,1,1,16,16,5,10,10,6,1,1,5,5,
  6,5,11,5,7,4,16,1,16,7,1,1,11,1,1,1,11,1,1,1,11,11,1,
  1,11,1,1,1,1,1,32,16,10,20,10,32,7,16,9,29,1,13,29,1,
  1,29,38,1,1,39,1,1,70,70,38,1,1,39,16,16,1,1,10,5,5,7,
  5,10,4,9,5,16,1,7,16,16,1,16,10,1,1,16,16,1,10,10,10,
  16,1,16,3,7,1,1,16,7,7,7,43,16,10,5,1,1,1,10,1,1,43,
  278,1,1,38,1,1,1,38,59,278,1,59,7,1,1,16,7,7,7,43,16,
  10,5,1,1,1,10,1,1,43,278,1,1,38,1,1,1,38,59,278,1,59,
  1,1,7,5,5,5,16,7,9,5,20,1,20,16,26,1,1,1,1,1,1,26,1,
  26,16,11,26,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,11,10,26,10,16,16,20,11,36,1,1,36,
  1,1,1,1,1,1,1,1,1,1,31,31,1,1,1,7,1,7,1,5,16,5,1,1,
  9,9,1,1,11,11,1,1,11,1,1,11,1,1,1,1,11,7,3,1,7,1,7,
  14,1,14,14,19,1,19,16,1,1,26,26,1,16,1,1,26,1,26,7,1,
  1,16,5,1,1,1,9,1,1,16,26,1,1,16,26,1,26,1,1,1,1,11,1,
  7,1,7,14,1,14,14,19,1,19,16,1,1,26,26,1,16,1,1,26,1,
  26,16,1,1,1,1,29,1,1,1,1,31,1,1,1,1,1,31,1,1,1,1,1,
  16,5,9,1,1,9,10,1,10,16,1,1,10,10,1,16,16,16,10,1,10,
  1,29,1,29,1,36,1,1,31,1,1,1,31,1,31,1,1,52,1,149,9,1,
  9,1,11,1,1,11,1,1,1,11,1,11,1,1,11,1,1,1,16,16,1,1,
  20,52,1,1,31,1,1,31,1,149,1,1,31,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,16,32,1,32,1,1,1,70,70,1,1,39,39,38,1,
  38,20,1,1,52,1,1,1,31,1,31,1,149,31,1,1,1,16,1,1,1,31,
  1,1,1,1,1,1,1,31,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  31,1,1,1,1,31,1,1,1,1,1,1,1,38,11,16,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,16,1,1,1,1,11,1,38,1,
  1,1,1,38,1,11,1,1,1,1,1,1,16,11,1,1,1,38,1,1,1,1,1,
  1,1,16,1,14,1,7,5,7,14,7,1,10,1,1,1,5,7,10,5,10,7,14,
  7,5,7,7,7,19,19,1,1,1,26,1,26,1,1,16,16,26,26,1,1,10,
  16,10,16,9,1,29,29,1,1,7,10,13,16,17,10,16,9,20,1,1,1,
  24,1,80,1,1,1,86,34,74,74,16,20,34,1,86,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,7,3,1,7,1,14,14,1,1,7,7,14,5,10,7,10,5,7,
  7,7,5,1,19,19,1,1,26,26,1,1,1,16,16,1,26,26,7,16,5,1,
  1,9,10,10,5,5,9,5,6,5,7,4,1,16,16,7,10,10,6,16,16,16,
  7,16,16,16,7,7,16,16,7,1,7,1,14,14,1,1,7,7,14,5,10,7,
  10,5,7,7,7,5,1,19,19,1,1,26,26,1,1,1,16,16,1,26,26,1,
  1,29,1,1,1,16,10,29,7,17,10,16,9,16,10,10,9,1,24,80,74,
  74,34,86,1,1,1,20,16,1,34,86,1,9,1,10,10,5,5,9,4,6,5,
  9,5,5,5,5,5,10,6,10,10,10,7,16,16,10,10,6,6,16,7,16,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,16,1,1,9,10,16,16,17,10,13,7,1,1,1,20,80,1,
  24,74,74,1,34,86,1,1,20,16,86,1,34,1,1,9,10,16,7,17,10,
  29,16,9,10,10,16,80,24,1,1,1,34,1,86,74,74,16,20,86,34,
  1,3,16,7,43,16,7,7,43,16,26,1,1,26,1,1,1,1,1,1,1,1,1,
  1,16,16,1,1,1,16,7,43,16,7,7,43,16,26,1,1,26,1,1,1,1,
  1,1,1,1,1,1,16,16,1,1,1,7,1,5,1,7,16,5,1,16,16,7,1,
  10,10,16,16,11,11,1,16,16,1,1,1,11,11,1,7,7,3,1,7,10,
  1,1,10,11,11,11,7,7,43,43,43,7,7,7,7,43,43,43,16,1,1,
  29,16,1,1,1,20,1,23,23,43,43,67,67,187,43,43,1,1,1,38,
  38,1,7,16,5,6,10,10,6,10,1,10,16,16,1,11,11,16,16,1,1,
  11,1,11,7,1,1,1,26,26,1,1,1,1,10,10,1,1,1,10,10,1,1,
  1,1,1,1,7,10,1,1,10,11,11,11,7,7,43,43,43,7,7,7,7,43,
  43,43,1,20,1,1,1,23,23,1,43,43,38,1,38,43,43,1,1,67,67,
  187,7,16,16,1,10,10,1,16,16,11,1,11,16,16,1,1,11,11,1,
  7,7,5,1,17,10,26,26,17,11,1,16,16,1,10,1,17,11,3,7,48,
  1,48,1,1,1,81,81,1,1,26,26,81,1,81,7,48,1,48,1,1,1,81,
  81,1,1,26,26,81,1,81,10,17,1,26,26,17,1,11,16,16,1,10,
  11,17,1,10,10,1,1,36,36,1,1,1,1,1,1,36,36,10,1,1,1,36,
  36,1,1,1,1,36,1,36,1,1,36,1,36,1,1,1,1,36,36,1,3,1,1,
  1,7,7,5,7,1,1,1,1,1,1,7,7,5,7,1,1,1,16,16,1,1,1,1,
  13,48,13,16,1,1,1,1,13,13,48,1,1,1,1,48,13,13,3,7,5,1,
  1,1,7,5,1,1,1,5,1,1,1,1,1,1,16,16,16,10,14,10,6,10,
  14,14,10,14,1,61,6,14,6,10,61,6,10,14,13,1,13,1,19,1,1,
  99,1,1,1,32,1,99,99,99,32,1,1,14,6,10,10,14,1,13,1,1,
  1,6,14,13,6,19,10,10,14,6,14,10,14,61,61,1,1,1,99,1,99,
  1,1,32,32,99,99,1,7,7,1,3,1,10,1,1,19,7,7,10,5,19,5,
  14,7,7,7,5,7,1,10,1,29,1,16,1,1,1,29,29,29,1,16,1,5,
  10,7,16,1,16,1,10,5,5,7,5,10,4,6,5,1,16,7,16,16,16,1,
  11,1,16,1,16,1,11,11,11,16,16,1,10,7,10,6,10,1,1,5,5,
  6,4,11,5,6,5,5,5,5,5,1,11,1,1,1,11,1,1,1,1,11,11,1,
  11,1,1,1,13,1,1,61,10,14,13,6,61,6,10,14,10,14,6,14,1,
  19,1,99,1,32,1,1,1,99,99,99,1,32,1,1,10,1,1,19,7,7,10,
  5,19,5,14,7,7,7,5,7,1,10,1,29,1,16,1,1,1,29,29,29,1,
  16,1,7,3,1,1,16,7,7,5,43,16,14,7,16,7,16,7,1,43,1,1,
  1,38,1,1,1,278,59,38,278,59,1,7,1,1,7,5,5,5,16,7,6,5,
  1,16,1,16,26,1,1,26,1,1,1,26,1,1,11,16,26,1,1,1,1,16,
  7,7,5,43,16,14,7,16,7,16,7,1,43,1,1,1,38,1,1,1,278,59,
  38,278,59,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,11,10,26,11,16,16,19,10,16,43,1,43,1,
  1,1,1,1,1,1,1,1,1,31,31,1,1,1,7,1,5,1,5,10,7,1,16,7,
  16,1,11,1,11,1,11,1,1,1,11,1,1,1,11,1,7,7,1,7,1,3,10,
  1,10,1,19,19,1,16,1,26,1,26,1,16,1,1,26,26,1,5,1,1,10,
  7,1,16,1,16,1,16,1,26,1,16,1,26,1,26,1,1,1,11,1,1,5,
  10,7,6,10,6,10,11,1,1,11,1,1,1,11,1,11,1,1,11,1,1,16,
  1,1,1,1,26,1,1,1,1,31,1,1,1,1,1,31,1,1,1,1,1,10,7,7,
  16,1,16,10,10,1,16,1,10,1,10,1,16,16,16,10,10,1,1,13,1,
  13,1,61,61,1,32,1,99,1,99,1,32,1,1,99,99,1,10,1,10,1,
  19,19,1,16,1,26,1,26,1,16,1,1,26,26,1,7,5,7,1,26,1,16,
  1,26,1,1,1,11,1,26,1,26,1,7,3,43,1,1,59,1,278,1,59,1,
  38,1,278,38,1,1,7,16,1,1,1,1,1,1,11,1,1,26,26,16,26,1,
  43,1,1,59,1,278,1,59,1,38,1,278,38,1,1,16,1,1,1,31,1,
  1,1,1,1,1,1,31,1,1,1,1,1,1,31,1,1,1,1,31,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,38,11,16,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,16,1,1,1,1,11,38,1,1,1,1,1,1,1,1,1,1,
  1,1,1,38,11,1,1,1,1,1,1,1,16,11,1,1,1,38,1,1,1,1,1,
  1,16,1,1,16,7,7,7,16,1,1,1,1,1,5,16,10,7,43,7,7,16,
  10,1,1,1,43,1,1,1,1,1,1,38,278,1,59,38,59,1,278,1,10,
  1,14,1,9,29,1,32,10,10,13,16,32,7,16,9,1,1,20,1,1,1,
  29,70,1,1,39,1,1,38,38,70,1,1,39,7,3,1,1,1,1,1,48,7,
  16,10,7,48,5,7,16,1,1,10,1,1,1,26,81,1,1,26,1,1,81,81,
  81,1,1,26,7,10,13,10,13,1,1,5,7,6,5,16,5,5,7,16,1,16,
  1,1,1,26,1,1,1,26,1,26,1,16,11,1,1,26,1,1,1,1,1,48,7,
  16,10,7,48,5,7,16,1,1,10,1,1,1,26,81,1,1,26,1,1,81,81,
  81,1,1,26,29,29,1,1,23,10,9,13,16,23,7,16,10,1,1,20,1,
  43,1,43,39,1,1,43,43,97,38,38,39,43,1,43,1,14,1,1,20,
  10,10,16,63,20,16,9,1,1,1,1,1,1,149,1,1,1,106,1,1,129,
  38,70,291,1,170,1,1,1,16,16,7,7,55,16,7,16,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,102,102,1,1,1,1,1,20,9,10,16,76,20,
  16,10,1,1,1,1,95,1,1,1,1,1,1,95,229,1,102,62,95,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,16,17,40,43,26,26,43,36,76,1,1,1,1,1,1,1,1,1,1,1,
  192,1,70,81,1,1,1,10,1,7,1,5,7,10,1,1,9,1,1,1,26,11,
  1,1,26,1,26,16,1,1,1,1,26,10,1,1,10,1,14,29,1,29,1,24,
  1,80,20,1,1,86,34,74,16,74,1,34,1,86,7,1,1,7,10,1,1,1,
  1,1,1,40,17,1,1,40,26,26,40,1,1,1,1,17,1,7,3,1,14,1,
  14,1,43,1,1,38,1,1,1,38,278,59,1,1,59,1,278,26,1,1,1,
  1,149,1,1,1,1,81,1,1,1,1,192,70,1,1,1,1,1,7,10,9,1,1,
  1,10,1,17,16,1,1,17,10,10,26,26,16,10,1,17,1,14,1,14,1,
  43,1,1,38,1,1,1,38,278,59,1,1,59,1,278,29,1,29,1,20,1,
  1,31,1,1,1,31,149,52,1,1,31,1,1,1,16,1,1,1,95,122,1,1,
  95,1,318,89,1,262,1,1,95,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,63,1,1,1,1,1,1,70,291,1,170,106,38,1,129,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,43,1,1,1,70,1,1,1,1,1,
  1,1,70,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,81,1,1,1,
  1,81,1,1,1,1,1,1,60,1,26,16,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,43,1,1,1,1,16,1,110,1,1,1,1,38,
  1,16,74,74,60,1,1,1,26,26,1,1,1,1,1,1,1,1,1,1,1,43,1,
  10,1,1,14,1,29,1,1,1,10,10,29,7,20,16,16,9,10,10,16,9,
  1,36,1,1,1,31,1,1,1,1,31,31,1,52,149,7,7,10,1,1,1,17,
  1,5,7,9,5,10,5,5,7,1,20,16,20,26,26,1,1,17,16,1,16,17,
  1,11,11,26,26,10,3,1,1,14,1,1,1,7,16,14,5,43,7,7,16,7,
  16,7,16,1,43,1,1,278,38,1,1,1,1,38,59,1,59,278,1,1,14,
  1,1,1,7,16,14,5,43,7,7,16,7,16,7,16,1,43,1,1,278,38,1,
  1,1,1,38,59,1,59,278,1,29,1,80,1,10,9,29,7,24,16,16,10,
  10,9,16,10,1,17,1,1,100,20,1,1,100,1,36,36,74,16,74,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,9,20,16,16,63,10,10,20,1,1,1,1,149,1,
  1,1,129,1,1,106,1,1,70,38,170,1,291,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,26,43,95,
  26,43,26,26,29,40,1,179,149,1,1,1,1,1,1,1,1,1,1,81,81,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,10,1,5,1,7,7,10,1,20,16,20,1,16,1,1,26,11,
  1,1,26,1,1,1,1,16,26,1,10,43,1,1,14,13,1,9,29,23,23,1,
  1,43,38,1,38,43,1,43,43,67,67,187,16,1,1,1,1,1,1,1,1,
  1,76,1,1,95,122,1,262,95,1,1,1,1,89,318,1,7,7,10,6,13,
  10,13,16,1,1,1,26,1,1,11,26,1,1,1,16,1,26,1,1,1,1,95,
  179,1,1,1,1,1,70,1,1,1,70,1,1,1,1,1,1,3,1,10,1,1,1,
  48,48,1,1,26,59,1,59,26,1,1,1,65,65,289,1,10,1,1,1,48,
  48,1,1,26,59,1,59,26,1,1,1,65,65,289,13,29,29,1,32,32,
  1,1,39,70,1,70,39,1,1,1,38,38,1,10,7,10,1,40,1,1,40,
  17,1,1,17,1,1,26,1,40,26,1,14,76,1,1,1,1,318,1,89,1,1,
  95,95,122,1,262,1,31,1,1,1,113,1,1,34,113,1,113,113,73,
  331,331,63,1,1,1,170,1,1,70,106,1,1,291,38,129,1,26,1,1,
  1,81,1,1,1,1,1,1,1,70,192,1,1,1,1,1,81,1,1,1,1,70,1,
  192,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,110,1,16,43,1,1,1,1,16,1,1,1,1,26,149,60,1,1,1,
  1,1,1,1,1,1,1,1,1,149,26,60,1,43,16,1,1,1,1,1,1,1,1,
  38,1,1,1,1,1,1,26,74,74,7,1,7,1,1,10,1,19,7,7,10,7,
  19,5,10,5,1,1,10,10,1,1,10,26,1,1,16,1,1,26,26,26,1,1,
  16,1,7,1,14,14,1,1,7,7,14,5,10,7,10,5,7,7,7,5,1,19,
  19,1,1,26,26,1,1,1,16,16,1,26,26,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,
  1,1,19,7,7,10,5,19,5,14,7,7,7,5,7,1,10,1,29,1,16,1,1,
  1,29,29,29,1,16,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,16,1,1,10,10,16,16,36,9,
  13,7,1,1,1,20,1,1,20,149,1,1,31,1,1,1,31,52,1,1,31,1,
  1,10,10,16,7,36,9,29,16,10,10,9,16,1,20,1,1,1,31,1,1,
  1,149,52,31,1,31,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,10,10,29,11,16,16,20,11,43,43,
  1,16,1,1,1,1,1,1,1,1,1,1,31,31,1,1,1,3,1,7,1,7,1,7,
  1,1,10,10,1,19,19,16,1,26,26,1,1,16,1,1,1,26,26,1,7,1,
  7,1,7,1,1,10,10,1,19,19,16,1,26,26,1,1,16,1,1,1,26,26,
  16,1,1,9,16,1,1,1,20,1,32,32,39,1,38,38,1,1,39,1,1,1,
  70,70,1,5,16,5,10,10,6,6,1,1,11,11,1,1,11,1,1,11,1,1,
  1,1,11,16,1,1,1,1,26,1,1,1,1,31,1,1,1,1,1,31,1,1,1,
  1,1,16,5,16,16,1,7,1,10,10,16,1,10,10,1,1,16,16,16,1,
  10,10,1,1,1,20,1,1,36,1,31,1,31,1,1,1,31,1,1,1,52,149,
  16,16,7,1,1,11,1,11,1,11,1,1,1,11,1,1,1,11,1,3,7,7,1,
  1,43,59,1,278,59,1,1,38,1,278,1,1,38,7,7,1,1,43,59,1,
  278,59,1,1,38,1,278,1,1,38,5,1,1,16,1,1,1,11,1,1,1,26,
  26,1,26,16,1,26,1,16,1,26,1,1,1,11,1,26,1,26,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,16,1,1,1,31,1,1,1,1,1,1,1,31,
  1,1,31,1,1,1,1,1,1,1,31,1,1,1,1,1,1,38,11,16,1,1,1,
  1,1,1,1,1,1,1,1,1,1,16,1,1,1,1,1,1,38,11,1,1,1,1,1,
  1,11,38,1,1,1,1,1,1,1,1,1,1,1,1,1,16,11,1,1,1,38,1,
  1,1,1,1,1,1,1,16,7,7,14,1,14,19,1,7,5,14,7,19,7,5,5,
  1,10,1,10,1,1,10,1,26,1,16,1,26,1,26,26,1,1,16,1,1,1,
  1,1,48,7,16,10,7,48,5,7,16,1,1,10,1,1,1,26,81,1,1,26,
  1,1,81,81,81,1,1,26,10,1,1,11,11,7,5,10,7,11,5,10,7,1,
  10,10,1,7,7,7,43,43,7,7,7,43,43,43,43,7,7,7,1,16,1,80,
  10,16,16,10,24,9,20,7,1,29,29,13,1,1,17,86,1,1,20,1,1,
  86,34,34,74,74,16,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,23,1,10,7,16,10,23,9,20,16,
  1,13,29,29,43,43,1,1,38,43,1,43,67,187,67,38,43,43,1,10,
  19,10,24,19,10,6,17,10,1,1,24,23,1,1,1,36,1,1,1,1,1,
  36,36,36,1,1,1,19,6,80,19,10,10,10,10,1,24,1,23,1,1,1,
  1,36,1,1,1,36,1,36,36,1,1,1,7,1,3,1,7,7,7,1,14,14,14,
  1,1,10,26,26,1,16,1,26,26,1,1,1,1,16,16,7,10,5,1,5,1,
  1,9,9,10,10,6,7,16,16,7,16,16,7,16,16,16,16,7,1,1,1,
  16,16,1,29,1,29,1,1,17,86,34,74,16,74,34,86,1,1,1,1,20,
  1,7,7,7,1,14,14,14,1,1,10,26,26,1,16,1,26,26,1,1,1,1,
  16,10,1,1,1,80,80,1,1,1,1,36,36,1,1,1,36,36,1,1,1,1,
  1,5,5,1,9,1,9,10,10,6,16,7,10,6,10,7,16,16,16,10,10,6,
  1,1,1,9,1,26,26,1,11,17,16,1,16,17,11,1,1,26,26,10,1,
  9,9,1,16,16,1,11,11,16,1,16,11,11,1,1,16,16,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,16,16,1,1,17,34,1,74,16,
  74,1,34,86,86,1,1,20,16,1,1,17,1,34,1,20,1,34,1,86,86,
  74,74,16,43,43,1,67,67,43,1,43,38,38,1,187,43,43,1,3,7,
  1,1,1,16,1,1,1,1,1,1,1,16,7,1,1,1,16,1,1,1,1,1,1,1,
  16,1,1,10,1,10,1,1,1,1,10,10,1,16,1,1,1,13,48,13,16,1,
  1,1,1,1,1,48,13,13,16,1,1,1,7,3,1,1,1,1,7,7,5,7,1,1,
  1,1,5,5,5,1,1,1,1,7,7,5,16,16,13,1,1,1,16,13,1,1,1,
  48,1,1,1,1,1,1,3,7,7,1,1,14,1,1,1,7,16,14,5,43,7,7,
  16,7,16,7,16,1,43,1,1,278,38,1,1,1,1,38,59,1,59,278,14,
  14,1,19,1,7,5,14,5,19,7,10,7,7,5,7,7,1,10,1,1,29,16,
  1,1,29,1,29,29,1,16,1,10,10,1,1,14,10,10,6,19,14,13,6,
  14,10,14,6,1,61,61,1,1,99,99,1,1,1,32,32,1,99,99,10,61,
  1,14,6,10,10,61,14,6,6,1,13,1,13,1,1,19,1,99,1,32,1,
  99,1,99,99,1,1,32,61,1,14,6,10,6,61,14,13,10,14,6,14,
  10,1,19,1,1,99,32,1,1,99,1,99,99,1,32,1,1,10,16,19,11,
  16,10,26,11,43,1,43,16,1,1,1,1,1,1,1,1,1,1,31,31,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,7,1,7,1,3,7,7,1,10,1,10,1,19,19,1,16,29,29,1,
  16,1,1,1,1,29,29,10,5,16,7,1,5,16,1,16,7,1,10,10,1,16,
  11,11,1,16,1,16,16,1,11,11,10,1,1,10,10,1,13,1,13,1,61,
  61,1,32,99,99,1,32,1,1,1,1,99,99,1,7,5,5,10,6,10,6,1,
  1,11,1,11,1,11,1,11,1,1,1,1,1,11,1,1,1,1,26,1,1,1,1,
  1,1,31,1,1,1,31,1,1,1,1,1,1,7,7,1,10,1,10,1,19,19,1,
  16,29,29,1,16,1,1,1,1,29,29,1,16,1,16,1,1,16,1,1,26,
  11,1,1,26,1,1,1,1,16,26,16,7,16,1,1,11,1,1,11,11,1,1,
  11,1,1,1,1,11,1,7,3,7,1,1,43,1,59,1,38,1,38,1,1,278,
  1,278,59,7,5,1,1,16,1,1,26,16,1,1,1,26,26,1,1,11,7,1,
  1,43,1,59,1,38,1,38,1,1,278,1,278,59,1,26,1,1,16,26,1,
  1,11,1,1,26,1,26,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,16,1,
  1,1,31,1,1,1,1,1,1,1,31,1,1,31,1,1,1,1,1,1,1,31,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,38,1,11,16,1,1,1,16,1,1,1,
  1,1,1,38,11,1,1,1,1,1,1,11,38,1,1,1,1,1,1,1,1,16,11,
  1,1,1,1,1,1,1,1,38,1,1,1,1,1,1,1,1,16,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,10,20,16,16,76,9,10,20,1,1,1,1,1,
  1,95,229,1,1,95,1,1,1,62,102,1,1,95,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,43,
  17,149,16,26,26,43,36,31,63,1,76,1,1,1,1,192,1,1,1,1,1,
  81,70,1,1,1,1,1,7,1,7,3,1,1,1,10,1,1,43,1,59,278,38,
  1,1,1,38,1,1,1,59,278,1,10,1,10,1,14,9,1,13,29,1,24,
  80,16,74,34,86,1,1,20,1,74,1,34,86,16,1,1,1,1,1,1,1,1,
  1,63,1,170,291,38,129,1,1,106,1,1,1,70,1,1,5,7,10,10,
  13,6,13,1,1,26,16,26,1,26,1,1,11,1,1,1,1,26,26,1,1,1,
  1,40,1,1,1,1,70,192,1,1,1,1,81,1,1,1,1,1,7,10,16,20,
  1,20,1,10,17,26,10,10,17,1,1,16,16,26,1,10,17,1,1,1,10,
  1,1,43,1,59,278,38,1,1,1,38,1,1,1,59,278,29,29,13,1,1,
  20,1,52,149,31,1,1,1,31,1,1,1,31,1,1,7,1,1,179,179,73,
  331,113,113,1,1,34,1,331,1,113,113,10,14,1,1,149,38,1,
  291,170,1,1,70,1,129,1,1,106,10,1,1,40,1,26,1,17,1,1,1,
  17,40,1,26,40,1,95,1,102,229,95,1,1,1,62,1,1,1,95,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,43,1,1,1,70,1,1,1,1,1,1,
  1,70,1,1,81,1,1,1,1,1,1,1,81,1,74,1,1,1,1,1,26,26,1,
  1,1,1,1,1,1,60,60,74,1,1,1,43,1,1,1,1,1,1,38,16,1,1,
  1,1,1,1,16,110,1,1,1,1,1,1,1,1,1,1,1,1,1,16,26,1,1,
  1,1,1,1,1,1,1,1,1,1,43,16,1,1,80,10,16,16,7,24,9,29,
  10,10,16,9,10,1,17,1,100,1,20,1,1,1,100,36,36,74,16,74,
  1,23,1,10,7,16,16,23,9,13,10,1,20,1,1,43,1,43,97,38,1,
  43,43,39,1,39,38,43,1,43,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,19,10,24,10,10,6,80,
  19,48,1,10,48,1,1,1,17,1,1,1,1,1,17,17,17,1,1,1,19,6,
  80,10,10,10,24,19,48,10,1,48,1,1,1,1,17,1,1,1,17,1,17,
  17,1,1,1,7,1,7,1,7,1,3,1,10,10,1,1,10,1,29,29,16,1,1,
  29,29,1,1,1,16,1,16,5,10,5,1,7,16,1,7,16,10,6,10,6,10,
  7,16,16,10,6,10,10,16,7,16,16,1,1,29,1,1,20,1,1,1,17,
  1,100,36,16,74,74,36,100,1,1,1,20,1,1,5,16,7,10,6,6,10,
  16,1,16,10,10,1,16,16,10,10,1,1,16,1,16,10,1,1,1,17,17,
  1,1,1,1,17,17,1,1,1,17,17,1,1,1,1,1,16,7,16,7,1,16,
  10,6,10,10,6,6,10,10,6,10,10,10,10,6,10,1,1,1,20,1,1,
  17,1,36,100,20,1,1,100,36,1,1,74,16,74,1,10,10,1,1,10,
  1,29,29,16,1,1,29,29,1,1,1,16,1,7,7,3,1,26,1,65,65,26,
  1,1,59,59,1,289,1,26,1,5,7,26,1,26,10,1,10,26,26,1,10,
  17,17,16,1,16,7,26,1,26,1,10,1,16,16,10,1,17,17,26,10,
  26,1,26,1,65,65,26,1,1,59,59,1,289,1,26,1,7,3,1,1,16,
  1,1,1,1,1,1,1,16,1,7,1,1,1,10,10,1,1,1,1,10,1,10,1,
  1,16,1,1,1,1,1,1,1,16,1,10,1,1,1,8,27,8,10,1,1,1,1,
  1,1,27,8,8,10,1,1,1,7,7,1,1,1,1,5,5,5,3,1,1,1,1,7,
  5,7,1,1,1,1,7,5,7,10,10,8,1,1,1,10,8,1,1,1,27,1,1,1,
  1,1,1,7,3,7,7,1,1,16,7,7,7,43,16,10,5,1,1,1,10,1,1,
  43,278,1,1,38,1,1,1,38,59,278,1,59,1,1,16,7,7,5,43,16,
  14,7,16,7,16,7,1,43,1,1,1,38,1,1,1,278,59,38,278,59,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,29,26,26,26,43,43,95,26,149,1,1,40,1,1,1,1,1,1,
  1,1,1,1,81,81,1,1,1,1,1,10,1,14,1,10,1,9,29,13,1,32,
  32,39,1,70,70,1,1,39,1,1,1,38,38,3,7,1,1,1,7,1,1,1,
  10,1,48,48,26,1,81,81,1,1,26,1,1,1,81,81,7,1,1,1,7,1,
  1,1,10,1,48,48,26,1,81,81,1,1,26,1,1,1,81,81,1,10,16,
  5,13,10,13,6,1,1,16,26,1,1,11,1,1,26,1,1,26,1,16,43,1,
  1,1,1,95,1,1,1,1,70,1,1,1,1,1,70,1,1,1,1,1,1,10,29,
  29,1,13,1,23,23,43,1,39,39,1,1,43,43,43,97,38,38,1,1,1,
  1,1,1,76,1,95,1,62,1,1,1,95,1,1,1,102,229,20,16,20,1,
  1,16,1,26,1,11,1,1,1,26,1,1,26,16,1,1,14,10,1,1,63,
  170,1,1,70,1,1,106,1,291,1,129,38,1,7,1,1,55,1,1,1,102,
  1,1,1,1,1,1,1,102,10,1,1,76,1,1,1,62,1,1,1,95,95,229,
  1,102,1,40,1,40,1,17,1,1,1,17,1,26,26,40,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,26,1,1,1,81,1,1,1,1,1,192,1,70,1,
  1,81,1,1,1,1,1,1,192,70,1,1,1,1,1,1,110,16,43,1,1,1,
  1,1,1,1,1,1,1,1,1,1,16,1,1,1,1,1,60,1,26,1,1,1,1,1,
  60,26,1,1,1,1,1,1,1,1,1,1,1,1,1,1,43,16,1,1,1,38,1,
  1,1,1,1,1,74,74,26,48,1,16,5,7,7,48,16,10,7,1,10,1,1,
  26,1,1,1,81,1,1,26,81,1,81,81,26,1,1,1,36,26,43,43,26,
  17,40,16,1,1,1,76,1,1,1,192,1,1,1,1,1,1,81,70,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,10,1,1,1,14,10,10,1,29,1,29,1,1,20,149,52,1,31,1,
  31,1,1,1,1,1,31,7,7,26,10,1,5,1,1,1,9,17,1,10,10,26,
  1,11,17,16,1,16,26,17,1,11,3,1,1,7,7,1,14,1,14,1,1,43,
  278,59,1,59,278,38,1,1,1,1,1,38,1,1,7,7,1,14,1,14,1,1,
  43,278,59,1,59,278,38,1,1,1,1,1,38,1,1,1,1,149,1,1,1,
  1,1,192,70,1,1,1,81,1,1,1,1,1,1,10,10,1,29,1,29,80,1,
  24,74,16,1,36,100,20,1,1,74,100,1,36,1,1,1,1,1,40,1,1,
  26,40,1,1,17,17,1,1,1,40,1,26,1,9,1,1,26,1,1,26,16,1,
  1,26,11,1,1,1,26,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,16,1,1,63,291,1,1,38,129,1,1,106,170,1,1,70,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,95,1,1,1,102,1,1,95,
  62,1,1,229,95,1,1,1,43,1,1,1,81,1,1,1,1,1,1,1,81,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,70,1,1,1,1,70,1,1,74,
  1,1,1,60,1,60,74,1,1,1,1,1,1,1,1,26,26,1,1,1,1,1,1,
  1,1,1,1,1,1,43,1,1,1,1,16,1,38,1,1,1,1,110,1,16,1,16,
  26,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,43,1,1,36,26,43,
  16,26,17,149,43,76,1,63,55,1,1,1,1,1,1,1,1,1,192,70,81,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,10,1,10,1,14,1,1,1,13,29,9,1,20,1,1,31,31,
  1,1,52,149,1,1,1,31,1,7,5,26,10,1,7,20,1,20,16,17,10,
  1,1,16,11,1,17,26,10,26,16,17,11,1,7,1,1,1,3,1,10,1,1,
  1,43,1,1,38,59,1,278,59,278,1,1,1,38,1,1,10,16,7,13,6,
  13,10,26,1,1,1,11,1,1,26,16,26,1,1,26,1,1,1,1,1,1,40,
  1,1,1,1,1,1,81,1,1,1,70,192,1,1,1,1,1,1,1,29,13,1,29,
  80,24,1,1,20,36,1,100,16,74,74,1,100,36,1,1,1,1,1,1,1,
  63,1,1,106,70,1,1,170,291,1,1,129,38,1,1,10,1,1,1,43,1,
  1,38,59,1,278,59,278,1,1,1,38,1,10,14,1,1,95,1,1,89,95,
  1,1,122,262,1,318,1,95,1,10,7,40,1,1,1,1,26,1,40,1,26,
  40,17,17,1,1,1,149,1,1,1,70,1,1,106,38,1,129,1,170,291,
  1,1,1,1,1,102,1,1,1,102,1,1,1,1,1,1,43,1,1,1,81,1,1,
  1,1,1,1,1,81,1,1,1,1,1,1,70,1,1,1,1,70,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,60,
  26,16,1,1,1,1,43,1,1,1,1,16,38,1,1,1,1,1,1,1,1,1,1,
  1,1,1,110,16,1,74,26,26,1,1,1,74,60,1,1,1,1,1,1,1,1,
  1,1,43,1,1,16,7,43,16,7,7,43,16,26,1,1,26,1,1,1,1,1,
  1,1,1,1,1,16,16,1,1,1,1,1,19,1,10,1,19,1,1,20,32,1,1,
  1,31,1,1,1,1,1,31,1,1,1,1,1,1,1,3,7,1,1,1,1,43,1,1,
  1,1,16,1,1,1,1,1,16,1,1,1,1,1,43,1,1,1,43,1,1,1,63,
  1,1,1,70,1,1,1,1,1,70,1,1,1,1,1,1,10,1,19,61,1,19,61,
  1,1,1,31,1,1,1,1,1,31,1,1,1,1,1,7,1,1,1,1,43,1,1,1,
  1,16,1,1,1,1,1,16,1,1,1,1,1,26,10,17,1,1,17,1,1,1,10,
  1,1,1,1,1,10,10,10,1,1,1,1,1,1,95,1,1,1,1,81,1,1,1,
  1,1,81,1,1,1,1,1,32,1,20,1,1,1,1,31,1,1,1,1,1,31,1,
  1,1,1,1,1,43,48,1,1,1,70,1,1,1,1,1,81,1,192,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,43,1,1,1,1,1,1,1,1,
  1,1,70,70,1,1,1,1,1,1,70,1,1,1,1,1,81,1,192,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,14,9,10,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,9,1,1,1,14,1,1,1,
  1,1,1,1,1,1,10,1,19,1,1,43,19,1,20,1,32,1,1,1,1,31,1,
  1,1,31,1,1,1,1,1,1,26,10,7,7,1,10,17,1,1,17,1,1,1,1,
  10,1,1,1,10,1,10,10,1,1,1,1,1,1,1,1,1,95,1,1,1,1,1,
  1,81,1,1,1,81,1,1,1,1,1,1,1,1,43,19,61,19,1,61,1,1,1,
  1,31,1,1,1,31,1,1,1,1,1,1,3,1,1,1,43,1,1,1,1,1,1,16,
  1,1,1,16,1,1,1,1,1,1,1,1,1,43,1,1,1,1,1,1,16,1,1,1,
  16,1,1,1,1,1,1,1,63,1,1,1,1,1,1,1,70,1,1,1,70,1,1,1,
  1,1,1,32,20,1,1,1,1,1,1,31,1,1,1,31,1,1,1,1,1,1,43,
  1,48,1,1,1,1,70,1,1,1,81,1,1,192,1,1,1,1,43,1,1,1,1,
  1,1,1,1,1,1,70,70,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,70,1,1,1,81,1,1,192,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,14,1,9,
  10,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,10,9,1,1,1,1,1,1,1,1,14,1,1,1,1,1,1,1,1,1,
  1,7,1,7,1,7,1,1,10,10,1,19,19,16,1,26,26,1,1,16,1,1,
  1,26,26,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,7,7,7,1,14,14,14,1,1,10,26,26,1,16,1,26,26,1,
  1,1,1,16,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,7,7,1,10,1,10,1,19,19,1,16,29,29,1,16,1,1,1,1,29,29,
  1,1,1,10,1,1,43,1,59,278,38,1,1,1,38,1,1,1,59,278,1,
  10,10,1,1,10,1,29,29,16,1,1,29,29,1,1,1,16,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,16,16,1,1,36,31,1,149,52,
  1,1,31,1,1,1,1,31,16,1,1,36,1,31,1,31,1,31,1,1,1,1,
  149,52,1,29,1,38,38,39,1,1,70,70,1,1,1,39,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,16,1,1,1,31,1,1,1,1,1,1,1,31,1,1,
  31,1,1,1,1,1,1,1,31,1,16,1,1,1,11,38,1,1,1,1,1,1,1,
  1,38,11,1,1,1,1,1,16,1,1,1,1,1,1,38,11,1,1,1,1,1,1,
  11,38,1,1,1,1,1,1,1,16,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,16,7,1,1,1,7,1,1,1,10,1,48,48,26,1,81,81,1,
  1,26,1,1,1,81,81,1,7,1,7,14,1,14,14,19,1,19,16,1,1,26,
  26,1,16,1,1,26,1,26,7,1,1,1,1,43,1,1,1,1,16,1,1,1,1,
  1,16,1,1,1,1,1,1,7,10,1,1,10,11,11,11,7,7,43,43,43,7,
  7,7,7,43,43,43,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,10,1,10,1,19,19,1,16,1,26,1,26,1,16,1,1,26,26,1,1,
  16,16,1,80,24,16,74,86,34,1,1,20,1,74,1,86,34,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,16,23,1,23,1,43,1,38,38,43,
  1,43,43,67,187,67,24,80,1,16,74,86,1,34,1,20,1,74,34,86,
  1,10,10,1,1,36,36,1,1,1,1,1,1,36,36,10,1,1,1,36,36,1,
  1,1,1,36,1,36,1,1,36,1,36,1,1,1,1,36,36,1,7,1,1,1,5,
  5,5,7,1,1,1,1,1,1,7,5,7,3,1,1,1,16,16,1,1,1,1,13,48,
  13,16,1,1,1,1,13,13,48,1,1,1,1,48,13,13,7,3,7,1,1,1,
  7,5,1,1,1,7,1,1,1,1,1,1,16,16,16,1,1,7,7,1,14,1,14,
  1,1,43,278,59,1,59,278,38,1,1,1,1,1,38,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,55,1,1,1,102,1,1,
  1,1,1,1,1,102,1,1,10,1,1,1,43,1,1,38,59,1,278,59,278,
  1,1,1,38,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  16,1,1,76,95,1,229,102,1,1,95,1,1,1,1,62,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,149,1,129,38,170,1,291,70,1,1,1,
  1,106,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,26,1,1,1,70,192,
  1,1,1,1,1,1,81,1,1,70,1,192,1,1,1,1,1,81,1,43,1,1,1,
  16,110,1,1,1,1,1,1,1,1,38,16,1,1,1,1,1,26,74,1,1,1,1,
  1,1,26,74,1,1,1,1,1,26,1,1,1,1,1,1,60,60,43,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,16,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,7,7,14,14,1,14,19,1,19,1,
  16,1,29,29,16,1,1,1,29,1,29,1,14,1,14,1,43,1,1,38,1,1,
  1,38,278,59,1,1,59,1,278,14,14,14,1,10,1,1,26,26,1,1,
  16,26,26,1,1,16,1,1,10,10,10,1,1,19,99,99,1,32,1,99,99,
  1,1,1,1,32,10,10,61,1,61,32,1,1,99,99,1,32,1,1,99,1,
  99,10,61,1,61,1,32,1,99,99,32,1,1,1,99,1,99,19,1,1,99,
  99,1,1,32,99,99,1,1,32,1,1,1,16,1,1,1,31,1,1,1,1,1,1,
  1,31,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,31,1,1,1,1,
  31,1,1,16,1,1,1,11,38,1,1,1,1,1,1,1,1,38,11,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,16,1,1,1,1,11,1,38,1,1,1,1,38,1,
  11,16,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,16,1,1,1,
  1,43,1,1,1,1,1,1,16,1,1,1,16,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,26,1,1,1,1,81,1,1,1,1,1,81,1,1,1,1,1,1,1,1,1,1,
  81,1,1,1,81,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,1,1,1,9,14,1,1,
  1,1,1,1,1,1,14,9,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,10,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,10,1,1,1,48,48,1,1,26,59,1,59,26,1,
  1,1,65,65,289,10,10,1,1,19,19,1,1,16,29,1,29,16,1,1,1,
  29,29,1,16,1,16,1,80,24,74,16,100,36,1,20,1,1,74,1,100,
  36,1,16,23,1,23,43,1,97,38,38,1,43,43,43,39,1,39,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,24,80,1,74,16,100,1,36,20,
  1,1,74,36,100,1,10,10,1,1,17,17,1,1,1,1,1,1,17,17,10,
  1,1,1,17,17,1,1,1,1,17,1,17,1,1,17,1,17,1,1,1,1,17,
  17,1,7,1,1,1,5,7,7,3,1,1,1,1,1,1,5,5,5,7,1,1,1,10,
  10,1,1,1,1,8,27,8,10,1,1,1,1,8,8,27,1,1,1,1,27,8,8,
  7,7,5,1,1,1,3,7,1,1,1,7,1,1,1,1,1,1,10,10,10,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,16,1,1,149,1,38,
  129,106,1,1,1,70,1,1,1,170,291,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,76,1,1,1,95,1,1,62,95,1,1,1,102,229,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,26,1,1,1,81,1,1,1,1,
  1,1,1,70,192,1,1,1,1,1,81,1,1,1,1,70,1,192,1,1,1,1,1,
  1,1,1,1,1,1,1,43,1,1,1,16,38,1,1,1,1,1,1,1,1,110,16,
  1,1,1,1,1,1,16,1,1,1,1,26,1,60,1,1,1,1,1,1,1,1,1,1,
  1,1,1,26,60,43,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,26,
  74,74,16,16,1,1,29,1,38,38,39,1,1,70,70,1,1,1,39,1,16,
  1,36,1,1,31,1,149,1,52,1,31,1,1,31,1,1,1,36,1,1,1,31,
  1,1,31,31,1,1,1,52,149,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,16,1,1,1,31,1,1,1,1,1,1,1,31,1,1,1,1,1,1,31,1,1,
  1,1,31,1,1,1,1,1,1,1,1,1,1,1,1,1,1,16,1,1,1,11,38,1,
  1,1,1,1,1,1,1,38,11,1,1,1,1,1,1,16,1,1,1,1,11,38,1,
  1,1,1,1,1,1,1,1,1,1,1,1,38,11,1,16,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,16,1,1,7,7,1,1,43,59,1,278,59,1,1,
  38,1,278,1,1,38,7,1,1,43,1,59,1,38,1,38,1,1,278,1,278,
  59,1,26,1,65,65,26,1,1,59,59,1,289,1,26,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,43,1,1,1,81,1,1,1,1,1,1,1,81,1,1,70,
  1,1,1,1,1,1,1,70,1,26,1,1,1,26,149,1,74,1,1,1,1,1,1,
  149,26,1,74,1,1,1,43,1,1,1,1,1,1,110,16,1,1,1,1,1,1,
  16,38,1,1,1,1,1,1,1,16,1,60,1,1,1,1,60,1,1,1,1,1,1,
  1,1,1,1,1,1,43,7,48,1,48,1,1,1,81,81,1,1,26,26,81,1,
  81,43,1,1,59,1,278,1,59,1,38,1,278,38,1,1,1,26,1,1,192,
  70,1,1,1,1,1,1,1,81,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  192,1,70,1,1,1,1,81,1,1,1,1,1,1,1,38,16,43,1,1,1,1,1,
  1,1,1,1,1,1,1,1,74,74,1,1,1,1,60,1,60,26,1,1,1,1,26,
  1,1,1,1,1,1,1,1,26,1,1,1,1,1,1,43,16,1,1,1,110,1,1,
  1,1,1,1,1,16,1,43,1,1,1,59,1,1,38,38,1,1,278,59,278,1,
  1,26,1,1,1,81,1,1,1,1,1,1,192,70,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,81,1,1,1,1,70,192,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,38,1,16,43,1,1,1,1,1,1,1,1,1,1,1,1,16,1,
  1,1,1,26,60,1,1,1,1,1,1,60,26,1,43,16,1,1,1,1,1,1,1,
  1,110,1,1,1,1,1,1,74,26,74,43,1,1,1,70,1,1,1,1,1,1,1,
  70,1,1,1,1,1,1,81,1,1,1,1,81,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,26,1,1,1,26,1,1,74,1,1,1,1,1,1,1,26,1,74,1,1,
  1,1,43,1,1,1,1,16,110,1,1,1,1,1,1,1,1,1,1,1,1,1,38,
  16,1,16,1,60,1,1,1,1,60,1,1,1,1,1,1,1,1,1,1,43,1,1,
  7,1,1,1,16,1,1,1,1,1,1,1,16,1,1,16,1,1,1,1,1,1,1,16,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,1,1,
  1,1,1,1,14,9,1,1,1,1,1,1,9,14,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,1,1,1,1,16,1,
  1,1,1,16,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,10,1,1,1,1,9,1,14,1,1,1,1,14,
  1,9,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,10,1,1,1,1,
  9,14,1,1,1,1,1,1,1,1,1,1,1,1,1,14,9,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,10,1,1,1,1,1,7,7,5,7,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,7,
  5,5,1,1,1,7,7,1,1,1,7,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,7,7,7,1,1,1,
  5,5,1,1,1,7,1,1,1,1,1,1,1,1,1,7,1,1,1,1,5,7,7,1,1,
  1,1,7,7,5,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,7,5,
  7,1,1,1,1,7,5,7,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
  1,5,7,7,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,7,7,5,
  7,5,1,1,1,7,1,1,1,1,1,1,1,1,1,7,1,1,1,1,1,1,1,1,1,
  1,1,1,1,1,1,1,1,1,7 ],

sizes := List(~.sizespos,i->~.sizesset[i]),

trsstatusset := 
[ "> 1 orbit (mod m)", "Mod(U DecreasingOn) exceeded <maxmod>", 
  "U DecreasingOn stable for <maxeq> steps", "exceeded memory bound", 
  "finite", "finite orbits", "transitive", 
  "transitivity on small points unclear", 
  "intransitive, but finitely many orbits", 
  "seemingly only finite orbits (very short)", 
  "seemingly only finite orbits (short)", 
  "seemingly only finite orbits (medium)", 
  "seemingly only finite orbits (long)", 
  "seemingly only finite orbits (very long)", 
  "seemingly finite and infinite orbits (linear growth)", 
  "seemingly finite and infinite orbits (exponential growth)",
  "U DecreasingOn stable and exceeded memory bound",
  "seemingly finite and infinite orbits (growth unclear)",
  "seemingly only finite orbits (very long, very unclear)" ],

trsstatuspos :=
[ 5,5,5,5,5,1,5,5,3,5,3,1,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,15,3,5,5,5,15,5,15,15,3,3,3,5,5,15,5,3,5,15,5,
  16,16,5,5,3,3,3,11,15,15,11,15,15,11,15,15,15,15,11,5,
  5,5,5,3,5,5,18,5,18,3,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,16,16,5,5,5,5,5,3,3,3,3,3,5,5,15,5,15,5,5,5,15,
  3,3,3,15,3,3,15,15,15,15,11,15,15,11,11,11,15,15,5,5,5,
  7,5,5,7,5,7,7,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,7,7,
  5,5,5,7,5,7,7,7,7,7,5,5,7,5,7,5,7,5,7,7,7,7,7,7,7,
  16,16,16,16,16,7,7,7,16,7,7,7,5,5,3,5,5,15,5,15,3,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,15,5,5,5,5,5,5,5,
  15,3,1,5,5,15,5,15,5,5,5,15,15,5,5,15,3,1,11,11,11,11,
  11,11,11,11,11,11,11,11,5,7,5,5,7,5,7,7,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,16,3,5,5,5,15,5,15,15,3,3,3,5,5,
  16,5,3,5,15,5,3,3,3,3,3,3,3,3,15,15,3,15,15,3,15,15,
  15,15,3,7,5,5,7,5,7,7,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,16,3,5,5,5,15,5,15,15,3,3,3,5,5,16,5,3,5,15,5,3,
  3,3,3,3,3,3,3,15,15,3,15,15,3,15,15,15,15,3,3,7,7,7,
  7,3,1,3,3,1,3,3,1,1,7,3,1,3,3,1,3,3,7,3,1,1,7,7,7,
  3,3,3,7,3,1,1,7,1,7,7,7,7,1,1,3,3,7,7,3,3,17,17,3,
  3,2,17,2,3,2,2,17,5,15,5,15,3,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,15,15,5,5,5,5,5,5,5,15,3,1,5,5,15,5,15,5,
  5,5,15,15,5,5,15,3,1,11,11,11,11,11,11,11,11,11,11,11,
  11,7,5,7,7,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,16,3,5,
  5,5,15,5,15,15,3,3,3,5,5,16,5,3,5,15,5,3,3,3,3,3,3,
  3,3,15,15,3,15,15,3,15,15,15,15,3,7,15,7,16,16,18,3,3,
  7,16,3,3,3,15,18,16,15,16,18,18,18,3,3,7,3,7,3,3,7,7,
  7,15,3,16,3,16,7,3,7,18,7,7,7,16,7,7,16,16,16,16,17,
  17,3,16,17,16,17,3,7,7,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,16,3,5,5,5,15,5,15,15,3,3,3,5,5,16,5,3,5,15,5,3,
  3,3,3,3,3,3,3,15,15,3,15,15,3,15,15,15,15,3,7,16,16,
  18,3,3,7,16,3,3,3,15,18,16,15,16,18,18,18,3,3,7,3,7,3,
  3,7,7,7,15,3,16,3,16,7,3,7,18,7,7,7,16,7,7,16,16,16,
  16,17,17,3,16,17,16,17,3,1,3,3,1,3,3,1,1,7,3,1,3,3,1,
  3,3,7,3,1,1,7,7,7,3,3,3,7,3,1,1,7,1,7,7,7,7,1,1,3,
  3,7,7,3,3,17,17,3,3,2,17,2,3,2,2,17,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,16,16,5,5,5,15,5,5,5,16,3,1,5,5,16,5,
  16,5,15,5,15,15,15,15,16,3,1,15,14,14,15,13,13,14,11,13,
  11,13,14,5,5,5,5,5,5,5,5,5,5,5,5,5,5,16,16,5,5,5,5,
  5,15,15,16,1,3,5,5,15,5,16,5,5,5,16,16,15,15,16,1,3,
  13,15,15,13,14,14,15,13,14,13,14,15,5,5,5,5,5,5,5,5,5,
  5,5,5,5,16,16,5,5,5,15,5,15,15,15,3,3,5,5,16,5,16,5,
  15,5,16,16,15,15,16,3,3,15,15,15,15,14,14,15,15,14,15,
  14,15,5,5,5,5,5,5,5,5,5,5,5,5,16,16,5,5,5,15,5,15,15,
  15,1,1,5,5,15,5,3,5,15,5,15,16,5,5,3,1,1,11,13,13,11,
  15,15,13,14,15,14,15,13,5,5,5,5,5,5,5,5,5,5,5,15,16,5,
  5,5,15,5,15,15,16,1,3,5,5,16,5,3,5,15,5,16,16,15,15,3,
  1,3,14,15,15,14,15,15,15,14,15,14,15,15,5,5,5,5,5,5,5,
  5,5,5,16,16,5,5,5,5,5,1,1,16,3,3,5,5,16,5,3,5,5,5,
  15,15,15,15,3,3,3,14,15,15,14,14,14,15,14,14,14,14,15,5,
  5,5,5,5,5,5,5,5,15,15,5,5,5,15,5,5,5,15,3,1,5,5,16,
  5,16,5,15,5,16,15,15,15,16,3,1,13,11,11,13,13,13,11,15,
  13,15,13,11,5,5,5,5,5,5,5,5,16,16,5,5,5,15,5,15,15,16,
  3,1,5,5,16,5,3,5,15,5,16,16,15,15,3,3,1,15,14,14,15,
  15,15,14,15,15,15,15,14,5,5,5,5,5,5,5,16,16,5,5,5,15,
  5,15,15,16,3,3,5,5,16,5,3,5,15,5,16,16,5,5,3,3,3,14,
  14,14,14,15,15,14,14,15,14,15,14,5,5,5,5,5,5,16,16,5,5,
  5,16,5,15,15,16,3,3,5,5,16,5,3,5,16,5,16,16,16,16,3,3,
  3,19,16,16,19,16,16,16,19,16,19,16,16,5,5,5,5,5,15,15,
  5,5,5,5,5,5,5,15,3,3,5,5,15,5,15,5,5,5,15,15,5,5,15,
  3,3,11,11,11,11,11,11,11,11,11,11,11,11,5,5,5,5,16,16,
  5,5,5,5,5,3,3,3,3,3,5,5,15,5,15,5,5,5,15,3,3,3,15,3,
  3,15,15,15,15,11,15,15,11,11,11,15,15,5,5,5,16,16,5,5,
  5,5,5,3,3,3,3,3,5,5,15,5,15,5,5,5,15,3,3,3,15,3,3,
  15,15,15,15,11,15,15,11,11,11,15,15,5,5,15,15,5,5,5,5,
  5,5,5,15,3,3,5,5,15,5,15,5,5,5,15,15,5,5,15,3,3,11,
  11,11,11,11,11,11,11,11,11,11,11,5,16,16,5,5,5,5,5,3,3,
  3,3,3,5,5,15,5,15,5,5,5,15,3,3,3,15,3,3,15,15,15,15,
  11,15,15,11,11,11,15,15,16,16,5,5,5,5,5,3,3,3,3,3,5,5,
  15,5,15,5,5,5,15,3,3,3,15,3,3,15,15,15,15,11,15,15,11,
  11,11,15,15,16,15,15,15,15,15,15,15,16,4,3,15,15,15,15,
  16,15,15,15,16,7,3,3,16,4,3,16,16,16,16,15,16,16,15,15,
  15,16,16,3,3,3,3,3,3,3,3,3,4,15,3,16,3,16,3,3,3,15,3,
  7,7,16,3,4,16,16,16,16,15,16,16,15,15,15,16,16,5,5,15,
  5,15,15,3,3,3,5,5,15,5,3,5,15,5,16,16,5,5,3,3,3,11,
  15,15,11,15,15,11,15,15,15,15,11,5,15,5,15,15,3,3,3,5,
  5,15,5,3,5,15,5,16,16,5,5,3,3,3,11,15,15,11,15,15,11,
  15,15,15,15,11,3,5,5,5,16,3,3,5,5,16,5,3,5,3,5,15,3,
  15,15,3,3,3,15,11,11,15,15,11,15,15,15,15,11,15,3,15,15,
  3,3,3,5,15,15,15,7,3,5,3,7,3,3,3,7,3,3,16,13,13,16,
  15,18,12,15,15,15,18,12,5,5,16,3,3,5,5,16,5,3,5,3,5,
  15,3,15,15,3,3,3,15,11,11,15,15,11,15,15,15,15,11,15,5,
  3,7,3,5,15,15,15,3,5,15,5,3,3,3,3,3,7,3,13,15,15,13,
  18,15,16,14,18,14,15,16,3,7,3,5,15,15,15,3,5,15,5,3,3,
  3,3,3,7,3,13,15,15,13,18,15,16,14,18,14,15,16,3,4,15,3,
  16,3,3,16,3,16,3,3,3,3,3,3,4,16,15,15,16,16,3,16,16,
  16,16,3,16,3,3,3,3,3,3,3,3,3,3,4,3,3,3,3,3,3,17,17,
  3,3,17,3,3,3,3,17,3,3,3,3,3,3,3,3,3,3,3,9,9,3,3,3,
  3,3,3,3,3,3,17,3,3,3,3,17,5,15,5,15,5,5,5,15,15,5,5,
  15,3,3,11,11,11,11,11,11,11,11,11,11,11,11,15,5,3,5,15,
  5,16,16,5,5,3,3,3,11,15,15,11,15,15,11,15,15,15,15,11,
  15,16,16,15,16,15,3,3,3,16,3,3,16,15,15,16,16,15,16,16,
  16,16,15,16,3,5,15,5,16,16,5,5,3,3,3,11,15,15,11,15,15,
  11,15,15,15,15,11,3,7,3,16,3,3,3,15,3,3,16,18,18,16,17,
  3,3,16,17,16,3,3,3,5,15,3,15,15,3,3,3,15,11,11,15,15,
  11,15,15,15,15,11,15,3,7,3,3,3,7,3,3,16,13,13,16,15,18,
  12,15,15,15,18,12,15,3,15,15,3,3,3,15,11,11,15,15,11,15,
  15,15,15,11,15,16,15,15,16,3,3,15,16,16,15,16,16,15,16,
  16,16,16,15,3,3,3,4,3,3,16,16,3,16,16,3,16,16,16,16,3,
  5,3,3,9,15,16,16,15,12,14,15,16,12,16,14,15,3,3,9,15,
  16,16,15,12,14,15,16,12,16,14,15,3,3,16,18,18,16,17,3,3,
  16,17,16,3,3,3,3,17,17,3,3,17,3,3,3,3,17,3,3,3,3,3,3,
  3,17,3,3,3,3,17,15,15,11,15,3,3,15,15,15,3,17,11,15,15,
  15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,3,17,
  15,15,15,3,17,15,15,15,11,15,15,15,15,15,15,15,11,15,15,
  15,15,15,11,15,11,15,15,15,15,15,15,15,15,5,5,5,5,16,5,
  5,16,5,3,3,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,3,3,5,5,
  5,15,5,15,3,3,15,3,5,5,15,5,15,5,15,5,15,15,3,3,3,3,
  3,15,11,11,11,15,11,15,11,11,15,11,11,5,5,5,16,5,5,16,
  5,3,3,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,3,3,5,5,5,15,
  5,15,3,3,15,3,5,5,15,5,15,5,15,5,15,15,3,3,3,3,3,15,
  11,11,11,15,11,15,11,11,15,11,11,5,5,1,5,5,5,5,5,1,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,3,5,5,5,15,5,15,15,
  3,3,3,5,5,5,5,15,5,5,5,15,15,5,5,3,5,3,11,15,11,11,
  15,11,11,5,11,11,5,11,5,16,5,5,5,5,3,15,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,15,3,5,5,5,15,5,15,15,3,15,3,5,
  5,15,5,5,5,15,5,5,5,15,15,3,15,3,11,11,11,15,11,11,15,
  11,5,11,5,11,15,5,5,15,5,3,3,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,15,3,5,5,5,16,5,16,15,3,3,3,5,5,5,5,5,5,5,
  5,5,5,15,15,3,3,1,3,15,15,11,11,15,3,5,5,15,5,15,15,
  16,16,16,3,1,15,15,16,15,16,16,15,16,16,16,15,15,15,15,
  16,16,15,16,15,15,16,16,16,16,15,16,16,3,15,15,15,15,15,
  15,15,15,15,15,16,16,3,3,7,16,16,3,3,16,16,16,15,15,16,
  15,16,5,16,5,3,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  3,5,5,5,15,5,15,5,3,15,3,5,5,15,5,15,5,15,5,15,15,5,
  5,3,5,3,11,11,5,11,11,15,11,11,5,15,11,11,16,5,3,3,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,3,5,5,5,16,5,16,5,
  3,15,1,5,5,15,5,15,5,15,5,15,15,15,15,3,5,3,15,15,5,3,
  15,11,11,15,5,15,5,3,7,3,3,15,5,15,15,15,15,15,15,15,
  15,5,15,15,3,3,3,3,3,5,3,3,15,3,15,3,3,16,3,15,15,5,
  3,15,3,5,3,18,18,3,3,4,3,3,16,15,5,12,13,15,14,15,4,
  16,3,3,3,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,16,5,
  5,5,16,5,16,5,16,16,3,5,5,5,5,5,5,5,5,5,5,5,5,3,15,
  3,16,15,5,15,15,15,15,5,11,15,11,16,3,15,15,15,5,15,15,
  15,15,3,16,5,3,3,3,15,15,5,3,5,3,15,3,15,3,5,3,3,3,
  15,15,3,3,3,15,3,15,3,3,3,3,3,15,3,13,12,4,15,14,3,3,
  3,5,16,15,15,15,15,16,15,15,15,16,16,16,16,15,1,1,1,15,
  15,1,16,15,1,15,16,15,16,1,16,7,3,1,1,3,15,3,1,3,1,1,
  1,1,1,4,15,3,3,3,2,16,16,16,16,15,16,16,15,16,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,1,5,5,5,15,5,15,5,1,16,3,5,
  5,5,5,5,5,5,5,15,15,15,15,3,5,3,13,11,5,13,11,12,12,5,
  5,11,5,11,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,1,5,5,5,15,
  5,15,15,1,3,3,5,5,5,5,15,5,5,5,5,5,15,15,3,15,3,11,
  13,11,11,12,11,13,5,5,11,5,13,5,5,5,5,5,5,5,5,5,5,5,
  5,5,15,1,5,5,5,3,5,3,15,1,3,3,5,5,15,5,5,5,15,5,5,5,
  15,15,3,15,1,14,14,11,15,14,13,15,5,5,14,5,14,5,5,5,5,
  5,5,5,5,5,5,5,5,5,1,5,5,5,15,5,15,5,1,16,1,5,5,5,5,
  5,5,5,5,15,15,5,5,1,5,3,11,11,5,11,13,13,11,5,11,12,
  11,11,5,5,5,5,5,5,5,5,5,5,5,15,3,5,5,5,16,5,16,15,3,
  3,3,5,5,5,5,15,5,5,5,5,5,5,5,3,5,3,13,15,11,14,15,14,
  14,5,11,14,11,15,5,5,5,5,5,5,5,5,5,5,5,16,5,5,5,16,5,
  16,5,16,3,3,5,5,5,5,5,5,5,5,5,5,5,5,3,5,3,14,14,11,
  14,14,14,14,5,5,14,5,14,5,5,5,5,5,5,5,5,5,5,1,5,5,5,
  15,5,15,5,1,16,3,5,5,15,5,15,5,15,5,15,15,15,15,3,5,1,
  11,11,5,12,11,11,11,11,5,13,5,11,5,5,5,5,5,5,5,5,5,1,
  5,5,5,16,5,16,5,1,16,3,5,5,15,5,15,5,15,5,15,15,5,5,
  3,5,3,14,14,5,12,14,15,13,11,11,15,11,14,5,5,5,5,5,5,
  5,5,16,5,5,5,18,5,18,5,16,16,3,5,5,5,5,5,5,5,5,5,5,
  5,5,3,15,3,14,14,5,14,14,14,14,5,11,14,11,14,5,5,5,5,
  5,5,15,3,5,5,5,16,5,16,15,3,15,3,5,5,5,5,15,5,5,5,15,
  15,15,15,3,15,3,14,15,11,12,15,13,13,5,11,12,11,15,5,5,
  5,5,5,5,3,5,5,5,15,5,15,5,3,15,3,5,5,5,5,5,5,5,5,5,
  5,5,5,3,5,3,11,11,5,11,11,11,11,5,5,11,5,11,5,5,5,5,
  3,3,5,5,5,15,5,15,3,3,15,3,5,5,5,5,5,5,5,5,5,5,15,
  15,3,3,3,15,11,11,11,11,11,15,5,5,11,11,11,5,5,5,3,3,
  5,5,5,15,5,15,3,3,15,3,5,5,5,5,5,5,5,5,5,5,15,15,3,
  3,3,15,11,11,11,11,11,15,5,5,11,11,11,5,5,15,3,5,5,5,
  15,5,15,15,3,3,3,5,5,5,5,5,5,5,5,15,15,5,5,3,3,3,11,
  15,11,11,11,15,11,5,5,11,11,11,5,5,7,5,5,5,7,5,7,5,7,
  3,3,5,5,5,5,5,5,5,5,5,5,5,5,3,5,3,15,15,5,15,15,15,
  15,5,5,3,5,3,5,7,5,5,5,7,5,7,5,7,3,3,5,5,5,5,5,5,5,
  5,5,5,5,5,3,5,3,15,15,5,15,15,15,15,5,5,3,5,3,3,5,15,
  5,15,5,15,5,3,3,3,5,5,5,5,5,5,5,5,3,3,3,3,3,3,3,12,
  18,12,11,11,13,18,5,5,11,12,11,3,3,7,7,7,7,3,3,3,4,3,
  3,3,15,15,3,3,3,3,3,3,3,3,15,4,4,3,15,4,3,3,4,15,15,
  3,15,7,5,5,15,5,15,5,3,15,3,5,5,5,5,5,5,5,5,5,5,5,5,
  3,5,3,11,11,5,11,11,11,11,5,5,11,5,11,5,15,5,15,15,3,
  3,3,5,5,5,5,5,5,5,5,15,15,5,5,3,3,3,11,15,11,11,11,
  15,11,5,5,11,11,11,7,5,7,5,7,3,3,5,5,5,5,5,5,5,5,5,
  5,5,5,3,5,3,15,15,5,15,15,15,15,5,5,3,5,3,7,15,15,7,
  16,3,15,15,15,15,15,15,15,15,15,15,15,15,4,15,3,16,16,
  15,3,16,16,3,15,15,16,15,16,7,5,7,3,3,5,5,5,5,5,5,5,
  5,5,5,5,5,3,5,3,15,15,5,15,15,15,15,5,5,3,5,3,15,7,
  16,3,15,15,15,15,15,15,15,15,15,15,15,15,4,15,3,16,16,
  15,3,16,16,3,15,15,16,15,16,3,3,3,5,5,5,5,5,5,5,5,3,
  3,3,3,3,3,3,12,18,12,11,11,13,18,5,5,11,12,11,3,4,3,3,
  3,15,15,3,3,3,3,3,3,3,3,15,4,4,3,15,4,3,3,4,15,15,3,
  15,7,3,15,15,15,15,15,15,15,15,18,18,15,15,3,3,3,16,3,
  3,3,16,16,3,15,15,16,15,15,3,3,3,3,3,3,3,3,3,3,3,3,3,
  3,3,3,3,3,4,3,3,3,1,1,3,3,3,5,15,5,15,5,15,5,15,15,
  5,5,3,5,3,11,11,5,11,15,11,11,11,11,15,5,11,15,5,15,5,
  15,5,15,15,5,5,3,5,3,11,11,5,11,15,11,11,11,11,15,5,11,
  5,15,3,5,3,15,15,3,3,3,5,3,11,11,5,16,12,11,11,12,12,
  18,5,12,3,5,5,5,5,5,5,5,3,15,3,11,11,11,11,11,11,11,
  11,11,11,11,11,3,15,3,15,15,3,3,3,3,3,11,11,11,15,15,
  11,11,15,3,15,11,15,3,5,5,5,15,15,3,5,3,11,11,5,15,11,
  11,11,11,11,11,5,15,3,15,15,3,3,3,5,3,11,11,5,16,12,11,
  11,12,12,18,5,12,5,5,15,15,3,5,3,11,11,5,15,11,11,11,
  11,11,11,5,15,5,5,5,3,3,3,12,14,5,12,14,13,11,15,5,14,
  15,11,5,5,3,3,3,12,14,5,12,14,13,11,15,5,14,15,11,5,3,
  3,3,14,11,15,13,11,11,14,15,5,12,5,14,3,3,3,14,11,15,
  13,11,11,14,15,5,12,5,14,3,3,3,4,3,3,3,3,3,3,3,3,3,3,
  3,15,15,3,11,11,15,15,12,11,11,15,11,4,3,1,3,3,3,4,3,
  3,3,1,3,15,14,11,16,15,3,11,11,16,13,16,12,14,15,15,15,
  11,11,15,14,14,11,11,14,13,10,10,11,5,11,15,14,15,14,14,
  3,11,3,11,14,14,13,15,11,15,15,11,11,15,12,14,11,11,14,
  14,11,5,13,11,14,14,10,12,11,3,11,5,5,5,16,5,5,16,5,3,
  3,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,3,3,5,5,5,15,5,
  15,3,3,15,3,5,5,15,5,15,5,15,5,15,15,3,3,3,3,3,15,11,
  11,11,15,11,15,11,11,15,11,11,5,5,3,5,5,15,5,15,3,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,15,5,5,5,5,5,5,3,15,
  5,3,5,5,15,5,15,5,5,5,15,5,3,15,15,3,15,15,11,15,11,
  11,5,11,11,11,11,11,5,5,16,5,5,15,5,1,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,5,15,3,15,15,3,5,5,
  15,5,15,5,15,5,15,15,3,15,3,3,5,11,5,11,11,15,11,11,15,
  11,11,11,5,3,5,5,15,5,3,3,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,15,15,5,5,5,15,5,5,3,3,5,3,5,5,3,5,3,5,15,5,3,
  5,3,3,1,1,3,15,5,11,15,11,5,15,15,15,15,15,5,5,16,16,
  15,3,3,15,5,15,15,15,3,15,15,16,16,15,5,15,3,3,3,3,3,
  15,3,3,16,3,5,3,3,15,3,5,15,15,3,16,3,16,3,15,5,3,3,
  3,3,3,16,4,3,3,16,15,14,15,15,14,12,5,5,16,5,16,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,15,5,5,5,5,5,15,3,5,
  5,3,5,5,15,5,15,5,5,5,15,15,3,5,15,3,15,11,5,11,5,11,
  11,15,11,11,11,15,11,16,5,15,3,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,15,5,15,3,5,15,3,5,5,16,5,16,5,
  15,5,16,15,3,5,3,3,5,15,5,3,5,15,15,11,11,15,15,3,5,1,
  16,3,16,16,16,15,16,16,15,16,15,16,15,15,15,15,15,15,15,
  15,15,15,15,15,15,15,16,15,15,4,15,15,16,15,16,15,15,15,
  16,15,16,15,16,7,3,16,15,16,15,16,15,16,16,16,16,3,3,16,
  7,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  16,5,15,3,5,5,16,5,3,5,5,5,16,5,16,5,3,3,5,15,11,15,
  5,15,5,15,15,16,16,15,11,3,16,15,16,15,16,15,16,16,16,
  15,15,15,15,15,15,15,15,15,15,16,16,1,15,15,16,15,3,3,
  15,15,16,16,16,16,1,15,16,15,16,15,16,3,15,16,15,16,15,
  16,3,3,16,16,16,16,15,15,15,15,5,15,1,15,15,15,15,3,5,
  3,3,3,3,3,3,15,15,15,3,3,3,3,5,3,4,5,15,3,15,3,15,3,
  3,3,3,3,5,3,3,15,4,3,15,4,14,5,14,12,3,16,15,15,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,15,15,5,5,5,15,5,5,1,15,5,
  3,5,5,15,5,16,5,15,5,15,5,1,15,16,3,15,13,5,12,11,12,
  5,11,11,11,11,11,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,15,
  5,5,5,5,5,5,1,5,5,1,5,5,15,5,16,5,5,5,15,5,1,5,16,3,
  5,11,11,13,5,11,5,13,11,12,12,12,11,5,5,5,5,5,5,5,5,5,
  5,5,5,5,15,15,5,5,5,5,5,5,1,15,5,3,5,5,16,5,16,5,5,
  5,16,5,1,15,15,1,15,15,11,15,11,13,5,14,14,14,12,14,11,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,5,15,1,5,15,3,
  5,5,15,5,15,5,15,5,15,15,1,5,3,3,15,11,5,11,5,13,11,
  11,12,13,11,13,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,15,1,5,5,3,5,5,3,5,3,5,5,5,3,15,1,5,3,3,15,14,11,
  14,5,14,11,15,14,15,13,15,11,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,18,5,15,3,5,5,16,5,3,5,5,5,16,5,18,5,3,
  3,5,14,11,14,5,14,5,14,14,14,14,14,11,5,5,5,5,5,5,5,5,
  5,15,15,5,5,5,15,5,5,1,5,5,3,5,5,15,5,16,5,15,5,15,5,
  1,5,16,1,5,12,5,11,5,11,5,11,13,11,13,11,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,15,5,15,3,15,15,3,5,5,16,5,16,5,15,5,
  16,15,3,15,3,3,5,12,5,13,5,15,11,14,15,14,14,14,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,16,5,5,3,5,5,16,5,3,5,5,
  5,16,5,16,5,3,3,5,14,5,14,5,14,11,14,14,14,14,14,5,5,
  5,5,5,5,5,15,15,5,5,5,15,5,15,1,5,15,3,5,5,3,5,3,5,
  15,5,3,15,1,5,3,3,15,12,11,13,5,13,11,15,12,15,14,15,
  12,5,5,5,5,5,15,15,5,5,5,5,5,5,3,3,5,3,5,5,15,5,15,
  5,5,5,15,5,3,3,15,3,3,15,5,11,11,11,5,15,11,11,11,11,
  11,5,5,5,5,5,5,5,5,5,5,5,5,3,5,5,3,5,5,15,5,15,5,5,
  5,15,5,3,5,15,3,5,11,5,11,5,11,5,11,11,11,11,11,5,5,5,
  5,15,15,5,5,5,5,5,5,3,3,5,3,5,5,15,5,15,5,5,5,15,5,
  3,3,15,3,3,15,5,11,11,11,5,15,11,11,11,11,11,5,5,15,15,
  5,5,5,5,5,3,3,5,3,3,5,5,15,5,15,5,5,5,15,3,3,5,15,3,
  5,11,11,15,5,11,11,11,11,11,11,15,5,5,15,15,5,5,5,5,5,
  3,3,5,3,3,5,5,15,5,15,5,5,5,15,3,3,5,15,3,5,11,11,15,
  5,11,11,11,11,11,11,15,5,5,5,5,5,5,5,5,5,15,5,3,3,5,
  5,15,5,15,5,5,5,15,5,15,5,15,3,3,11,11,11,11,11,11,11,
  11,11,11,11,11,5,5,5,5,5,5,3,3,3,3,3,5,5,15,5,15,5,5,
  5,15,3,3,3,15,3,3,13,15,14,15,11,5,14,12,11,12,14,5,5,
  5,5,5,5,3,3,3,3,3,5,5,15,5,15,5,5,5,15,3,3,3,15,3,3,
  13,15,14,15,11,5,14,12,11,12,14,5,5,5,15,5,15,3,5,15,3,
  5,5,15,5,15,5,15,5,15,15,3,5,3,3,5,11,11,11,5,15,11,
  11,11,11,15,11,5,5,5,5,5,7,5,5,3,5,5,7,5,3,5,5,5,7,
  5,7,5,3,3,5,15,5,15,5,15,5,15,15,15,15,15,5,5,5,5,7,
  5,5,3,5,5,7,5,3,5,5,5,7,5,7,5,3,3,5,15,5,15,5,15,5,
  15,15,15,15,15,5,15,15,3,3,15,3,5,15,15,5,18,5,5,15,15,
  15,3,3,3,3,3,12,5,11,15,13,15,11,14,14,14,11,5,5,3,15,
  5,3,5,5,15,5,3,5,15,5,15,5,3,15,15,3,15,11,5,11,11,11,
  5,11,15,15,11,11,11,3,5,15,3,5,15,15,5,15,5,15,5,15,5,
  3,5,3,3,5,11,12,12,5,18,12,11,11,11,13,15,5,3,15,4,3,
  3,7,7,3,7,3,3,7,3,3,3,3,4,15,4,15,4,15,3,15,4,3,3,3,
  4,15,5,3,5,5,15,5,3,5,3,15,15,5,3,5,15,3,15,18,5,11,
  12,11,5,13,16,11,11,11,12,3,5,15,15,5,15,5,15,5,15,15,
  15,5,3,3,3,11,3,15,11,15,15,11,11,11,15,15,11,3,3,3,3,
  3,3,3,3,3,3,4,3,3,3,3,3,3,3,3,3,3,4,3,3,3,4,3,5,15,
  5,15,5,5,5,15,5,3,5,15,3,5,11,5,11,5,11,5,11,11,11,11,
  11,5,15,5,15,5,15,5,15,15,3,5,3,3,5,11,11,11,5,15,11,
  11,11,11,15,11,5,7,16,7,15,15,15,15,7,15,16,3,15,16,15,
  16,15,16,15,3,16,7,16,3,15,3,5,5,5,7,5,7,5,3,3,5,15,
  5,15,5,15,5,15,15,15,15,15,5,3,18,3,16,15,3,3,15,3,3,
  16,15,16,16,16,15,3,16,3,16,3,3,5,5,7,5,7,5,3,3,5,15,
  5,15,5,15,5,15,15,15,15,15,5,15,15,15,3,3,3,3,3,12,5,
  11,15,13,15,11,14,14,14,11,5,15,5,3,15,15,3,15,11,5,11,
  11,11,5,11,15,15,11,11,11,15,7,15,16,3,15,16,15,16,15,
  16,15,3,16,7,16,3,15,3,5,3,3,5,11,12,12,5,18,12,11,11,
  11,13,15,5,3,3,4,15,4,15,4,15,3,15,4,3,3,3,4,15,15,3,
  15,18,5,11,12,11,5,13,16,11,11,11,12,3,15,16,16,15,15,3,
  3,3,16,3,16,3,15,3,3,3,3,1,3,3,4,3,3,3,3,1,15,11,11,
  3,11,11,15,15,15,11,11,15,11,15,12,14,11,15,15,15,14,15,
  14,13,10,14,5,11,11,11,14,14,10,11,15,14,11,14,14,15,3,
  11,11,10,14,13,14,11,11,5,12,14,15,11,15,15,11,11,11,11,
  14,13,11,15,15,14,15,12,15,11,16,14,16,14,13,15,11,11,5,
  5,3,5,5,3,5,15,1,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,3,
  15,5,5,5,5,5,5,15,15,3,15,5,5,5,5,3,5,3,5,5,3,15,3,
  15,3,3,11,15,15,15,11,12,11,12,15,11,11,11,5,16,5,5,15,
  5,1,1,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,3,15,5,5,5,
  15,5,15,15,15,16,15,5,5,15,5,16,5,16,5,15,16,15,3,15,3,
  3,11,11,3,3,11,15,11,15,11,11,12,12,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,3,5,5,5,5,15,5,5,16,16,
  5,5,5,5,5,5,5,5,15,5,15,15,5,3,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,15,5,5,3,3,3,15,15,5,16,5,15,15,5,15,5,15,
  15,5,3,5,3,3,5,15,5,5,15,5,15,3,3,16,15,15,5,15,3,16,
  5,15,3,15,15,5,3,3,3,3,14,3,3,3,15,15,14,15,3,15,3,3,
  5,16,5,3,3,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,3,5,5,5,
  5,15,5,15,5,5,16,15,5,5,15,5,16,5,16,5,15,16,5,3,15,3,
  3,11,12,11,11,11,15,11,15,12,11,3,3,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,3,3,5,5,5,5,5,16,5,5,5,5,5,5,
  16,5,5,5,15,5,5,15,3,3,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,3,3,3,16,5,5,16,5,16,16,5,16,5,15,15,5,3,5,3,3,3,
  15,5,5,15,5,15,3,3,16,15,15,5,15,3,16,5,15,3,15,15,3,
  3,3,3,3,14,3,3,3,15,15,14,15,3,15,3,3,15,3,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,16,5,5,5,5,5,5,5,5,5,3,5,5,5,
  5,5,3,5,16,5,5,16,5,16,5,3,3,11,15,15,15,5,15,5,15,15,
  11,15,15,5,16,16,5,15,5,15,16,5,15,5,3,3,5,15,5,15,3,
  3,15,5,5,5,5,15,15,15,3,3,15,5,15,15,3,5,3,15,5,3,3,
  3,15,7,7,3,15,15,15,14,3,3,3,15,14,15,15,15,15,5,5,5,
  16,15,5,16,5,3,3,5,15,5,15,3,3,15,5,5,15,5,3,15,15,3,
  3,15,5,3,15,3,5,3,15,15,3,3,3,15,7,7,3,3,15,15,14,3,
  3,3,3,14,15,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,1,15,5,
  5,5,5,5,5,15,15,15,15,5,5,5,5,15,5,16,5,5,16,15,1,15,
  15,15,11,5,12,12,11,12,11,12,5,11,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,1,15,5,5,5,5,5,5,15,15,16,15,5,5,5,5,16,
  5,3,5,5,3,15,1,15,15,15,11,12,12,12,11,5,11,5,12,11,14,
  14,5,5,5,5,5,5,5,5,5,5,5,5,5,1,15,5,5,5,5,5,5,15,15,
  5,5,5,5,5,5,5,5,15,5,5,15,15,1,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,3,5,5,5,5,15,5,
  15,5,5,15,15,5,5,15,5,15,5,16,5,15,16,5,3,15,16,16,11,
  12,5,5,11,12,11,12,12,11,13,13,5,5,5,5,5,5,5,5,5,5,5,
  1,5,5,5,5,5,5,15,15,15,5,5,5,5,15,5,5,5,15,5,5,15,5,
  1,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,16,5,5,5,5,5,5,5,5,5,16,15,5,5,5,5,16,5,18,5,5,18,
  5,16,15,16,16,5,16,16,16,5,14,11,14,16,5,16,16,5,5,5,5,
  5,5,5,5,5,1,5,5,5,5,15,5,15,5,5,15,15,5,5,15,5,15,5,
  16,5,15,16,5,1,15,15,15,11,5,12,12,11,12,11,12,5,11,5,
  5,5,5,5,5,5,5,5,5,1,5,5,5,5,15,5,15,5,5,5,5,5,5,15,
  5,5,5,15,5,15,15,5,1,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,16,5,5,5,5,5,5,5,5,5,15,15,5,5,5,5,15,
  5,16,5,5,16,5,16,15,16,16,5,16,14,14,11,16,5,16,16,5,
  16,16,5,5,5,5,5,5,1,15,5,5,5,15,5,5,5,5,5,5,5,5,5,5,
  5,5,16,5,15,16,15,1,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,3,3,5,5,5,5,5,5,15,15,5,16,5,5,5,5,5,5,15,5,
  5,15,3,3,5,3,3,15,5,11,11,5,5,15,5,5,5,11,11,5,5,5,5,
  3,3,5,5,5,5,5,5,15,15,5,16,5,5,5,5,5,5,15,5,5,15,3,
  3,5,3,3,15,5,11,11,5,5,15,5,5,5,11,11,5,5,5,3,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,5,5,15,5,3,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,7,5,5,5,5,5,5,5,5,5,3,5,
  5,5,5,5,3,5,7,5,5,7,5,7,5,3,3,5,15,15,15,5,3,5,3,15,
  5,15,15,5,3,15,5,5,5,5,5,15,5,5,5,5,5,5,15,5,5,5,15,
  5,5,15,15,3,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,7,5,5,5,
  5,5,5,5,5,5,3,5,5,5,5,5,3,5,7,5,5,7,5,7,5,3,3,5,15,
  15,15,5,3,5,3,15,5,15,15,3,3,3,15,3,3,3,3,3,3,15,3,3,
  3,7,3,3,7,7,3,7,3,3,15,7,7,15,3,7,7,15,3,15,3,3,15,
  9,9,5,5,5,5,5,3,3,3,5,15,5,5,3,5,5,15,15,5,5,15,5,3,
  5,3,3,13,5,3,3,5,5,13,5,5,5,5,5,5,5,15,5,15,5,5,15,
  5,5,5,15,5,15,5,15,5,15,15,5,3,16,5,5,5,11,5,5,15,11,
  5,11,11,15,5,5,5,3,5,5,15,15,5,5,5,5,5,5,5,5,15,5,3,
  15,5,3,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  15,5,5,5,5,5,5,15,5,5,15,5,15,15,5,5,11,5,5,5,11,5,
  11,5,5,11,5,5,3,15,3,3,15,5,15,5,15,5,15,5,15,5,5,15,
  5,3,15,5,5,5,5,5,5,13,15,5,15,5,13,5,5,5,15,15,5,5,5,
  5,5,5,5,5,15,5,3,15,5,3,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,12,5,15,5,5,5,12,15,15,5,15,15,3,3,16,5,5,5,
  11,5,5,15,11,5,11,11,15,5,5,5,5,16,5,5,5,5,5,5,15,5,
  3,15,3,3,5,12,12,15,5,11,11,5,5,15,5,5,5,11,11,5,16,5,
  5,5,5,5,5,15,5,3,15,3,3,5,12,12,15,5,11,11,5,5,15,5,
  5,5,11,11,5,15,5,12,3,5,5,16,3,15,16,5,3,3,3,3,5,3,3,
  3,15,16,5,15,3,15,3,3,5,5,5,5,5,5,15,5,5,15,15,15,3,
  3,3,16,5,15,15,11,5,16,5,5,11,15,15,5,15,5,15,5,15,5,
  15,15,5,3,16,5,5,5,11,5,5,15,11,5,11,11,15,5,5,5,5,5,
  5,15,5,5,15,5,3,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,12,
  15,15,5,15,15,3,3,16,5,5,5,11,5,5,15,11,5,11,11,15,5,
  5,3,5,7,5,5,7,5,7,5,3,3,5,15,15,15,5,3,5,3,15,5,15,
  15,5,16,3,15,16,5,3,3,3,3,5,3,3,3,15,15,5,16,3,15,3,
  3,15,5,5,15,15,3,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,7,15,
  15,15,7,15,3,3,15,7,3,3,15,16,15,16,7,15,3,3,5,7,5,7,
  5,3,3,5,15,15,15,5,3,5,3,15,5,15,15,15,5,3,15,5,5,5,
  5,5,5,13,15,5,15,5,13,5,5,15,7,15,3,3,15,7,3,3,15,16,
  15,16,7,15,3,3,3,5,3,3,13,5,3,3,5,5,13,5,5,5,5,5,15,
  7,7,15,3,7,7,15,3,15,3,3,15,9,9,5,5,11,15,5,5,15,15,
  11,15,15,16,5,5,5,3,3,3,3,5,3,3,3,3,5,3,3,3,3,3,3,5,
  3,3,3,3,5,3,3,5,11,11,10,5,5,5,5,10,11,11,15,15,11,5,
  5,5,5,11,15,15,5,5,3,11,3,15,5,5,5,5,3,11,3,15,5,5,5,
  11,10,11,11,5,5,5,5,5,5,11,3,3,5,5,10,11,11,5,11,3,3,
  11,15,15,5,5,5,5,7,5,5,5,5,3,3,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,15,15,5,5,5,15,5,15,15,1,15,3,5,5,5,5,3,
  5,15,5,3,5,15,1,3,3,3,11,11,15,3,15,11,11,11,11,5,5,5,
  7,5,5,5,5,3,3,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,
  15,5,5,5,15,5,15,15,1,15,3,5,5,5,5,3,5,15,5,3,5,15,1,
  3,3,3,11,11,15,3,15,11,11,11,11,5,5,5,3,3,3,3,3,3,1,
  3,3,1,1,3,1,1,3,16,1,3,3,1,1,3,3,15,1,1,3,3,7,15,3,
  3,4,3,1,1,3,1,3,16,16,3,1,1,16,3,3,3,3,3,4,4,3,3,3,
  3,3,3,1,1,1,5,15,5,15,3,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,15,15,5,5,5,5,5,5,5,15,3,1,5,5,15,5,15,5,5,5,
  15,15,5,5,15,3,1,11,11,11,11,11,11,11,11,11,11,11,11,3,
  5,5,7,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,15,5,5,5,
  5,5,15,5,15,3,3,5,5,15,5,3,5,15,5,5,15,15,5,3,15,3,3,
  15,11,11,11,5,5,5,15,11,11,11,3,15,3,15,5,15,5,15,15,
  15,5,15,5,5,15,5,5,15,15,15,15,5,5,5,5,5,5,15,15,3,3,
  15,5,5,5,16,5,5,5,15,5,5,5,16,3,3,14,13,11,11,11,10,
  10,11,14,15,3,3,5,7,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  15,15,5,5,5,5,5,15,5,15,3,3,5,5,15,5,3,5,15,5,5,15,
  15,5,3,15,3,3,15,11,11,11,5,5,5,15,11,11,11,3,5,15,15,
  5,5,5,15,5,5,5,5,15,15,5,5,15,15,15,5,5,5,15,5,5,5,
  15,3,3,15,5,15,5,15,5,5,5,5,5,15,15,16,3,3,11,11,13,
  14,14,3,3,15,11,11,10,10,1,3,1,1,3,3,1,1,3,16,1,3,1,
  1,3,3,16,3,1,1,3,3,16,3,1,15,3,3,1,1,3,1,3,7,3,3,1,
  1,3,3,3,4,3,3,4,4,3,3,1,1,1,3,3,3,3,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,15,15,5,5,5,5,5,5,5,15,5,15,5,5,15,
  5,15,5,15,5,5,5,5,15,16,5,15,13,12,11,13,12,5,5,5,11,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,15,5,5,5,5,5,
  5,15,15,15,15,5,5,5,5,16,5,5,5,15,15,15,15,15,15,15,11,
  13,13,11,11,5,11,5,12,5,5,11,5,5,5,5,5,5,5,5,5,5,5,5,
  5,16,15,5,5,5,5,5,5,15,15,15,16,5,5,15,5,16,5,15,5,15,
  5,15,15,16,15,16,15,14,15,13,14,5,10,5,13,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,15,15,5,5,5,15,5,15,15,15,15,15,5,
  5,15,5,15,5,5,5,5,15,5,5,15,15,16,11,11,11,11,13,11,5,
  5,13,5,11,5,5,5,5,5,5,5,5,5,5,5,5,15,15,5,5,5,15,5,
  15,15,15,15,15,5,5,5,5,15,5,5,5,5,5,5,5,15,15,15,13,
  15,13,11,15,10,5,5,14,5,5,10,5,5,5,5,5,5,5,5,5,5,15,
  15,5,5,5,5,5,15,15,15,15,16,5,5,5,5,16,5,5,5,5,15,5,
  5,16,5,16,13,15,15,12,13,5,11,5,12,5,5,11,5,5,5,5,5,5,
  5,5,5,15,15,5,5,5,15,5,5,5,15,5,16,5,5,15,5,15,5,15,
  5,15,5,15,5,15,5,15,11,11,11,12,11,5,5,11,11,11,5,5,5,
  5,5,5,5,5,5,5,15,15,5,5,5,15,5,5,5,15,15,16,5,5,5,5,
  16,5,15,5,15,15,5,5,16,15,16,11,13,11,13,13,5,5,10,15,
  5,10,5,5,5,5,5,5,5,5,15,15,5,5,5,15,5,5,5,15,5,16,5,
  5,15,5,15,5,15,5,5,15,5,5,15,15,16,14,13,12,14,15,11,5,
  5,15,5,11,5,5,5,5,5,5,5,15,15,5,5,5,16,5,15,15,15,5,
  16,5,5,15,5,16,5,15,5,15,15,15,5,16,5,16,14,16,16,15,
  16,5,5,5,16,5,5,5,5,5,5,5,5,15,15,5,5,5,5,5,5,5,15,
  5,1,5,5,5,5,15,5,5,5,5,5,5,5,15,5,1,11,11,11,11,11,5,
  11,5,11,5,5,11,5,5,5,5,15,15,5,5,5,5,5,5,3,15,5,3,5,
  5,5,5,15,5,5,5,5,5,3,15,15,3,15,15,11,15,11,11,5,11,5,
  11,5,11,5,5,5,5,16,15,5,5,5,5,5,5,15,15,15,15,5,5,5,
  5,15,5,5,5,5,5,15,3,15,3,3,11,11,15,15,11,11,11,5,11,
  5,5,5,5,5,15,15,5,5,5,5,5,5,5,15,3,15,5,5,5,5,15,5,
  5,5,5,15,5,5,15,3,15,11,11,11,11,11,11,5,5,11,5,11,5,
  5,15,16,5,5,5,5,5,3,5,15,3,3,5,5,5,5,15,5,5,5,5,5,5,
  5,15,15,15,15,15,11,11,11,5,5,5,11,5,11,11,15,15,5,5,5,
  5,5,15,5,15,3,15,5,5,5,5,15,5,5,5,5,5,5,5,15,5,3,11,
  15,11,15,11,11,5,5,11,5,5,11,15,15,15,15,15,15,15,15,15,
  3,15,15,15,15,15,15,15,15,15,15,3,16,3,15,16,15,15,16,
  16,15,15,15,15,15,15,15,15,15,15,15,15,15,15,3,15,3,15,
  16,15,15,15,15,16,15,15,15,15,15,15,15,16,15,4,16,15,15,
  16,15,15,16,15,15,15,15,15,5,5,15,5,15,15,15,3,15,5,5,
  5,5,15,5,5,5,15,15,5,5,3,5,15,11,15,11,11,15,11,5,5,
  11,11,5,5,5,15,5,15,15,15,15,15,5,5,5,5,15,5,5,5,16,5,
  5,5,3,3,15,11,11,15,11,15,11,5,11,11,5,5,5,15,5,5,5,
  15,5,15,5,5,15,5,16,5,15,5,5,5,5,5,1,5,16,15,11,11,15,
  15,5,11,5,15,11,5,5,3,15,15,3,3,16,5,5,5,5,16,15,5,15,
  5,5,5,15,3,5,3,12,12,12,16,15,15,5,11,14,10,5,5,5,5,
  16,5,16,5,5,15,5,16,5,15,5,5,5,5,5,16,5,16,15,11,11,
  15,15,5,5,5,15,5,5,5,5,15,14,15,5,5,5,5,16,5,5,5,16,
  5,5,5,3,5,15,12,15,14,12,15,11,5,5,16,5,5,15,15,12,3,
  5,5,5,5,16,5,5,5,3,5,3,15,3,3,15,12,14,15,12,16,11,15,
  5,16,5,10,5,15,3,15,15,15,15,15,15,16,15,15,15,15,15,16,
  15,16,16,15,15,16,15,15,15,15,15,15,15,16,3,5,3,5,5,3,
  5,5,5,15,16,5,5,3,3,3,12,4,15,3,15,3,5,5,11,3,3,3,15,
  15,3,15,16,16,16,16,15,15,16,15,1,3,1,3,16,16,16,16,15,
  16,15,16,3,3,4,5,15,5,15,5,5,5,15,5,5,5,15,5,15,11,11,
  11,11,11,5,5,11,11,11,5,5,5,5,3,5,15,5,5,16,5,5,15,15,
  15,11,15,11,11,11,5,5,5,15,11,11,5,5,16,16,5,15,15,5,3,
  5,16,5,16,16,12,12,13,14,5,5,11,14,16,5,15,3,5,15,5,15,
  15,5,5,15,3,15,11,11,15,11,11,5,5,11,15,5,11,5,16,16,1,
  16,3,15,3,15,3,1,16,16,16,16,16,15,15,16,3,15,3,3,15,5,
  5,5,15,5,16,5,16,15,11,11,15,15,5,5,5,15,5,5,5,3,5,3,
  5,3,16,15,3,12,12,12,16,14,5,5,11,15,5,15,5,5,5,5,15,
  16,5,16,15,11,11,15,15,5,5,11,15,5,5,11,15,5,5,16,3,15,
  12,12,16,12,16,5,5,16,14,11,5,5,5,5,16,3,15,12,15,14,
  12,16,11,5,5,16,5,16,5,5,15,3,16,15,16,16,14,11,5,11,5,
  11,5,5,5,15,3,16,14,16,16,15,11,5,11,15,11,5,5,10,3,16,
  16,16,16,16,3,3,3,15,16,16,15,15,3,3,15,4,12,11,3,3,3,
  15,5,3,5,16,16,16,3,16,3,3,3,16,15,15,16,15,15,11,15,
  11,11,11,15,14,13,14,11,15,15,11,11,11,15,13,14,12,15,15,
  14,12,13,15,11,11,11,15,13,14,14,15,11,11,11,12,12,14,11,
  11,11,11,5,5,11,10,11,10,5,11,10,10,11,11,11,10,10,14,
  12,12,5,5,5,7,5,5,5,5,3,3,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,15,15,5,5,5,15,5,15,15,1,15,3,5,5,5,5,3,5,15,
  5,3,5,15,1,3,3,3,11,11,15,3,15,11,11,11,11,5,5,5,15,
  15,15,15,7,3,16,3,15,16,3,16,15,16,16,16,16,15,16,3,15,
  3,3,15,15,3,16,16,7,15,3,3,16,16,16,15,15,15,16,16,16,
  15,16,16,16,15,3,3,3,15,3,4,3,16,16,16,16,16,15,15,15,
  5,15,5,7,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,3,5,5,
  5,15,5,15,15,5,15,3,5,5,15,5,15,5,15,5,15,15,1,15,15,
  3,3,5,5,11,11,11,15,3,15,5,11,11,11,15,5,7,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,3,5,5,5,15,5,15,15,5,15,3,
  5,5,15,5,15,5,15,5,15,15,1,15,15,3,3,5,5,11,11,11,15,
  3,15,5,11,11,11,3,3,3,15,5,5,5,15,15,15,15,15,5,5,15,
  5,5,5,15,5,5,5,5,5,15,5,15,16,5,15,3,15,15,5,5,15,5,
  15,5,15,15,15,15,3,3,3,11,10,11,11,11,14,13,13,10,15,3,
  3,3,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  15,5,15,15,5,15,3,5,5,5,5,3,5,15,5,5,5,15,15,3,15,3,
  11,11,11,11,11,11,11,11,11,11,11,11,3,3,1,1,3,3,1,1,1,
  3,15,3,1,1,3,3,1,3,3,1,1,3,16,16,3,15,1,3,3,1,1,3,3,
  3,7,1,1,15,3,3,16,4,3,4,1,1,3,4,4,4,3,3,1,3,3,3,15,
  5,15,5,15,15,15,5,15,5,15,5,15,5,5,5,15,15,15,5,5,15,
  5,16,15,5,3,3,5,5,5,15,3,5,15,5,5,5,15,15,15,15,3,3,
  3,13,14,14,11,11,11,15,11,10,10,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,15,14,5,5,5,15,5,15,15,15,15,15,5,5,5,5,5,5,
  15,5,15,5,15,15,15,15,15,11,5,12,13,11,12,11,11,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,5,5,5,5,15,5,15,15,
  5,15,15,5,5,5,5,15,5,15,5,5,5,15,15,5,15,5,5,11,13,11,
  12,11,13,12,5,5,5,11,5,5,5,5,5,5,5,5,5,5,5,5,5,15,15,
  5,5,5,15,5,15,15,15,15,15,5,5,5,5,5,5,15,5,15,5,16,15,
  5,16,15,10,5,14,15,13,14,15,13,5,5,5,10,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,15,5,15,15,5,15,5,5,5,5,5,5,5,
  15,5,5,15,15,15,15,15,15,5,5,11,11,13,13,11,11,11,5,11,
  5,5,5,5,5,5,5,5,5,5,5,5,15,5,5,5,5,15,5,16,15,5,15,
  15,5,5,5,5,5,5,15,5,5,14,15,15,15,15,5,5,10,15,13,14,
  15,13,14,5,5,10,5,5,5,5,5,5,5,5,5,5,5,15,15,5,5,5,15,
  5,15,15,5,16,15,5,5,5,5,15,5,15,5,5,5,15,15,15,16,5,5,
  11,15,13,12,13,15,14,5,5,5,11,5,5,5,5,5,5,5,5,5,5,15,
  5,5,5,15,5,15,15,5,15,15,5,5,15,5,15,5,15,5,15,15,15,
  15,15,15,5,5,5,11,11,11,11,11,13,5,11,5,5,5,5,5,5,5,5,
  5,5,5,15,5,5,5,16,5,15,15,5,16,5,5,5,15,5,15,5,16,5,
  15,15,15,15,15,15,5,5,5,13,11,15,13,11,15,10,10,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,15,5,15,15,5,16,15,5,5,5,5,5,
  5,15,5,15,15,15,15,15,16,15,5,5,13,14,15,15,12,13,11,5,
  11,5,5,5,5,5,5,5,16,15,5,5,5,15,5,15,15,5,16,5,5,5,5,
  5,5,5,15,5,15,15,15,15,5,16,5,5,5,16,14,16,16,16,15,5,
  5,5,5,5,5,5,5,5,15,3,5,5,5,15,5,15,15,3,15,3,5,5,15,
  5,5,5,15,5,5,5,15,15,5,15,3,11,5,11,3,11,11,3,11,5,5,
  5,11,5,5,5,5,5,5,5,5,5,15,5,15,15,5,15,3,5,5,5,5,5,
  5,15,5,5,5,15,15,5,1,5,11,5,11,11,11,11,11,11,5,5,11,
  5,5,5,5,3,15,5,5,5,15,5,15,15,15,15,15,5,5,5,5,15,5,
  15,5,5,5,15,16,5,3,3,11,5,3,3,11,11,11,11,11,5,5,5,5,
  5,15,5,5,5,5,15,5,15,15,5,3,5,5,5,5,5,5,5,15,5,5,3,
  15,15,5,3,5,5,11,15,11,11,15,11,11,5,5,11,5,5,5,15,5,
  5,5,15,5,16,15,5,15,15,5,5,5,5,5,5,15,5,5,15,15,15,15,
  15,5,5,5,11,11,11,15,15,11,5,11,11,5,5,5,5,5,5,15,5,
  15,15,15,1,5,5,5,5,5,5,5,15,5,15,15,15,15,5,15,3,5,11,
  11,15,15,15,11,15,5,5,5,11,5,5,15,5,15,5,15,15,15,3,12,
  5,5,15,15,5,15,15,5,5,3,15,3,5,3,3,11,15,15,16,13,13,
  14,12,5,5,11,5,5,5,15,15,5,3,15,3,15,14,5,5,15,15,5,
  15,15,5,5,5,15,16,15,16,3,11,5,14,16,13,13,15,12,5,5,5,
  5,5,5,15,5,15,15,5,15,5,5,5,5,5,5,5,15,5,5,5,15,15,3,
  15,5,5,11,11,11,11,11,11,11,5,11,5,5,5,15,5,15,16,5,15,
  15,5,5,5,5,5,5,15,5,3,5,15,15,3,3,5,11,11,15,11,15,11,
  11,11,5,5,5,5,15,5,15,15,5,15,16,5,5,5,5,5,5,15,5,15,
  5,15,15,3,15,5,11,5,11,11,15,11,15,11,5,11,5,5,16,15,
  15,3,16,15,15,15,15,15,15,15,15,15,15,15,15,15,16,15,15,
  15,15,15,15,16,16,15,15,15,16,15,15,15,15,15,16,5,5,5,5,
  5,5,5,15,5,15,15,15,15,5,15,5,5,5,11,15,15,15,11,15,5,
  5,5,5,15,15,16,15,15,15,15,15,15,15,15,15,15,15,15,15,3,
  16,15,15,15,15,15,16,16,15,15,15,15,15,15,15,16,3,15,15,
  15,15,15,15,15,15,3,15,3,15,15,16,15,15,15,16,15,15,15,
  16,15,15,15,16,15,16,3,15,5,5,5,12,5,15,5,15,15,15,15,
  3,15,3,15,5,13,15,16,12,13,14,11,5,5,11,15,15,15,15,15,
  15,15,15,15,16,15,16,15,3,1,3,15,3,16,15,16,16,16,16,15,
  15,15,15,5,5,3,5,5,5,15,5,15,5,16,15,3,3,3,3,5,3,3,
  15,11,16,12,5,3,4,3,5,15,5,16,5,15,5,15,15,15,15,3,15,
  5,5,5,11,11,15,11,11,15,11,11,5,5,15,5,15,5,16,5,15,15,
  15,15,15,15,15,5,5,11,11,11,15,11,15,5,11,5,11,15,16,3,
  15,5,15,15,3,15,3,16,12,5,5,12,14,12,12,16,15,10,15,5,
  11,3,5,15,5,5,5,15,15,5,16,5,5,5,15,15,11,11,15,15,11,
  5,11,5,5,16,3,15,15,15,4,3,3,3,5,5,3,4,15,12,11,16,3,
  15,3,3,15,5,5,5,16,15,5,16,5,5,5,15,15,11,11,15,15,5,
  5,5,5,16,16,15,15,7,15,15,4,15,15,15,16,15,15,15,16,16,
  15,15,16,5,5,15,16,5,15,3,5,5,11,15,11,11,11,15,15,5,5,
  15,5,15,15,14,15,5,5,5,12,12,15,14,12,16,5,11,5,5,15,
  15,12,3,5,5,10,12,12,14,15,12,15,5,11,15,5,15,15,3,15,
  16,15,16,15,15,15,16,15,15,15,15,15,15,16,16,15,15,15,16,
  15,15,15,16,16,15,15,15,3,5,3,4,12,11,4,15,3,15,3,3,5,
  5,3,4,3,3,15,16,15,16,16,15,15,16,15,3,3,11,4,12,15,3,
  15,3,5,5,3,5,14,12,14,11,11,11,5,10,10,10,12,12,14,11,
  11,11,5,10,10,10,3,15,15,11,15,12,11,11,11,15,15,3,15,
  14,11,11,11,11,15,15,12,11,11,11,15,15,11,14,12,12,15,11,
  13,14,12,11,12,12,14,10,10,11,5,5,5,5,5,5,5,3,3,15,15,
  5,15,5,5,5,5,15,5,15,15,5,15,5,15,15,5,15,5,15,16,5,5,
  15,3,15,3,5,5,15,15,15,5,5,15,3,5,5,15,3,3,3,14,14,3,
  3,3,14,14,14,14,5,5,5,5,7,5,15,15,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,15,5,5,5,5,15,5,5,15,15,3,3,5,5,15,5,
  15,5,5,5,15,15,5,15,15,15,3,11,11,5,5,5,11,11,11,11,15,
  15,3,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,5,
  5,5,5,15,5,15,15,15,5,5,5,5,15,5,5,5,15,5,5,15,15,15,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,7,3,3,15,16,5,16,5,15,
  15,5,15,5,5,15,5,15,5,15,15,15,15,5,15,15,5,5,3,3,3,
  15,15,5,5,15,15,5,15,15,3,5,15,3,3,3,3,14,14,5,5,5,14,
  14,14,14,3,3,3,15,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,15,5,5,5,5,15,5,5,15,15,3,3,5,5,15,5,15,5,5,5,15,
  15,5,15,15,15,3,11,11,5,5,5,11,11,11,11,15,15,3,5,15,
  15,5,5,5,15,15,5,15,5,15,15,5,5,5,15,3,15,15,5,5,15,5,
  3,15,15,3,3,15,5,15,15,3,5,5,5,5,5,15,3,15,15,3,3,3,
  12,12,12,3,3,15,15,12,12,12,15,15,5,5,5,15,15,5,15,5,
  15,15,5,5,5,15,3,15,15,5,5,15,5,3,15,15,3,3,15,5,15,
  15,3,5,5,5,5,5,15,3,15,15,3,3,3,12,12,12,3,3,15,15,12,
  12,12,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,15,5,5,5,15,
  5,5,15,15,5,15,5,5,5,5,5,5,5,5,15,5,15,15,15,15,15,11,
  5,11,11,12,5,11,11,11,5,5,11,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,16,15,5,5,5,15,5,15,16,15,15,15,5,5,5,5,15,5,5,5,
  15,5,15,15,15,15,15,11,11,11,11,11,11,11,5,5,5,11,11,5,
  5,5,5,5,5,5,5,5,5,5,5,5,15,15,5,5,5,15,5,5,15,15,5,
  5,5,5,5,5,5,5,5,5,5,5,15,16,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,5,5,5,5,16,5,15,
  15,15,15,5,5,5,5,5,5,5,15,5,5,15,5,15,15,15,15,5,11,
  11,11,11,11,5,11,11,11,11,11,5,5,5,5,5,5,5,5,5,5,5,15,
  5,5,5,5,15,5,15,16,15,5,5,5,5,5,5,5,5,5,5,5,5,5,16,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  15,5,5,5,5,15,5,5,15,15,15,15,5,5,5,5,15,5,5,5,15,15,
  15,15,15,5,5,12,12,15,12,12,12,12,5,5,5,11,11,5,5,5,5,
  5,5,5,5,5,15,5,5,5,5,15,5,5,15,15,5,15,5,5,15,5,15,5,
  15,5,16,5,5,15,15,15,15,11,5,11,12,11,5,11,11,11,11,11,
  5,5,5,5,5,5,5,5,5,15,5,5,5,5,15,5,5,15,16,5,5,5,5,5,
  5,5,5,15,5,15,15,5,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,15,5,5,5,5,15,5,15,15,15,15,15,5,5,15,
  5,15,5,5,5,15,5,5,15,15,15,15,5,12,12,5,15,12,5,12,12,
  11,11,11,5,5,5,5,5,5,16,5,5,5,5,16,5,5,16,16,5,5,5,5,
  5,5,5,5,5,5,15,15,15,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,15,15,5,5,5,15,5,5,15,3,5,3,5,5,5,5,5,5,
  5,5,5,5,15,15,5,15,15,3,5,11,11,5,5,3,5,5,11,5,11,5,
  5,5,5,15,15,5,5,5,15,5,5,3,3,5,3,5,5,5,5,5,5,5,5,5,
  5,15,15,15,15,15,15,5,11,11,5,5,15,5,5,11,11,5,5,5,5,
  3,5,5,5,5,15,5,5,16,16,5,5,5,5,5,5,5,5,5,5,15,5,5,3,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,16,5,5,5,5,15,5,
  15,15,15,3,5,5,5,5,5,5,5,5,5,5,15,5,15,5,15,5,5,15,
  11,5,11,15,5,5,5,5,11,11,5,15,15,5,5,5,15,5,15,15,15,
  5,5,5,5,5,5,5,5,5,5,5,5,5,15,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,3,5,5,5,5,15,5,15,15,15,3,5,5,5,5,5,5,5,5,5,
  5,15,5,15,15,5,15,5,3,5,11,11,3,5,5,5,5,11,11,15,15,
  16,15,16,3,15,15,3,3,15,15,15,15,15,15,15,15,15,3,3,15,
  3,3,4,3,15,16,15,15,15,16,15,15,15,15,15,15,5,5,15,15,
  5,3,15,3,5,3,5,5,5,5,5,5,5,5,5,5,5,15,15,15,15,13,5,
  15,15,5,5,13,5,5,5,5,5,5,5,16,5,15,15,15,3,5,5,5,5,5,
  5,5,5,5,5,5,5,15,15,5,15,5,15,11,5,11,15,5,5,5,11,5,
  11,5,16,5,5,15,3,5,5,5,5,5,5,5,5,5,5,3,5,5,3,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,15,5,5,15,3,5,3,5,5,5,5,5,
  5,5,5,5,5,5,15,15,15,5,15,5,5,11,11,5,15,5,5,11,11,5,
  7,15,16,7,3,15,15,15,15,15,15,15,15,15,15,15,15,16,15,
  15,15,15,16,15,15,15,16,15,15,15,15,15,15,5,3,7,5,5,5,
  5,5,5,5,5,5,5,7,5,5,7,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,15,15,15,5,5,5,5,5,5,5,5,5,5,5,5,15,15,5,5,5,13,15,
  5,15,13,5,5,5,5,5,5,16,15,3,15,15,15,15,15,15,15,15,3,
  15,3,3,15,15,15,3,15,15,15,15,15,3,15,15,15,15,15,15,3,
  15,15,15,15,15,15,15,15,3,15,3,3,3,3,4,3,15,15,15,15,
  15,3,15,15,15,15,15,5,5,5,5,5,5,5,5,5,5,15,5,15,3,3,
  3,5,7,15,3,15,7,5,5,5,3,3,3,5,5,5,5,5,5,5,5,5,5,15,
  15,3,3,3,7,5,3,3,3,5,7,5,5,3,3,3,5,15,5,3,5,15,5,3,
  5,5,3,15,15,5,5,5,5,11,11,5,5,15,15,11,11,5,5,5,5,5,
  15,5,5,15,5,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,15,
  15,5,15,3,5,3,3,15,15,15,5,5,5,5,5,5,5,13,13,15,15,5,
  3,5,15,5,15,5,5,3,15,15,15,5,5,11,11,5,5,5,15,15,11,
  11,5,5,3,3,3,5,5,3,3,3,3,5,5,3,3,3,5,5,7,7,15,15,3,
  5,5,5,5,15,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,3,15,
  5,3,15,15,15,5,5,5,15,15,5,5,13,13,5,5,5,15,5,5,16,5,
  15,15,5,5,11,11,5,5,5,3,3,11,5,11,5,5,7,14,12,13,5,5,
  11,12,14,5,5,3,3,14,11,12,5,15,15,15,5,5,13,5,5,5,13,
  5,5,5,15,15,15,15,5,15,15,13,5,15,15,15,5,13,5,5,5,5,
  5,15,3,15,15,15,15,15,15,15,15,3,3,15,15,15,5,5,3,3,5,
  15,15,3,3,15,15,15,15,5,5,3,3,15,15,5,3,3,15,15,15,15,
  5,3,3,15,15,5,3,3,3,3,15,5,15,5,12,11,12,5,5,5,5,14,
  12,12,11,11,12,5,5,5,5,12,12,11,5,5,11,12,11,11,5,5,5,
  5,11,11,14,14,5,5,5,12,12,11,11,5,5,5,5,5,5,12,12,11,
  5,5,14,12,12,5,12,11,11,12,11,11,5,5,5,15,15,5,3,3,15,
  5,5,15,5,5,5,15,5,5,15,5,5,5,5,5,5,5,15,5,5,15,5,5,
  5,15,15,3,5,15,15,15,15,5,15,5,15,15,15,5,3,3,3,11,10,
  3,3,15,14,13,13,11,11,11,11,5,3,3,3,15,5,5,15,5,15,15,
  5,15,5,15,15,5,5,5,15,5,5,15,5,5,15,5,15,3,3,15,3,15,
  5,15,15,15,5,15,5,15,15,3,5,3,3,3,12,12,3,3,15,12,12,
  12,12,15,3,3,3,3,3,15,5,5,15,5,15,15,5,15,5,15,15,5,5,
  5,15,5,5,15,5,5,15,5,15,3,3,15,3,15,5,15,15,15,5,15,5,
  15,15,3,5,3,3,3,12,12,3,3,15,12,12,12,12,15,3,3,3,3,
  15,5,5,15,15,5,5,15,5,5,15,5,5,5,15,5,5,15,15,5,5,15,
  5,15,5,15,15,3,5,5,15,15,15,5,5,5,5,5,5,5,3,3,3,14,
  13,3,3,15,10,10,11,14,11,11,11,3,1,1,1,1,3,3,1,1,4,1,
  3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,4,4,3,3,4,4,4,3,3,
  3,3,3,3,3,4,4,4,3,3,4,4,4,4,4,4,3,3,3,3,1,3,1,1,3,
  1,1,1,1,3,1,1,1,1,1,1,3,1,1,1,1,1,1,1,3,1,4,3,1,1,
  3,1,4,1,3,1,1,1,1,1,4,4,3,3,3,3,3,4,3,3,3,4,3,3,3,
  15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,
  15,15,15,15,15,15,16,16,15,15,15,16,16,15,15,16,15,15,15,
  16,4,3,1,16,16,1,1,16,16,16,16,15,15,15,15,15,15,15,15,
  15,5,5,15,15,5,14,5,5,5,15,5,15,15,5,15,15,5,15,15,14,
  14,5,5,5,15,15,15,15,15,5,5,16,15,4,3,1,16,14,1,3,4,
  15,14,14,13,5,5,11,15,5,15,5,5,5,5,15,15,5,5,5,5,15,
  15,15,5,15,15,5,5,15,15,14,15,5,5,15,16,16,5,15,15,15,
  5,15,15,3,3,1,13,13,1,3,16,14,13,14,14,15,15,15,15,15,
  15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,16,15,15,15,
  16,16,16,15,15,15,16,16,15,15,15,15,16,15,15,3,3,1,16,
  15,1,1,16,16,16,15,16,15,16,15,15,15,5,15,5,16,5,5,14,
  5,15,15,5,15,5,5,16,5,15,15,15,15,14,5,5,5,16,14,5,5,
  15,5,5,5,5,3,3,1,13,13,1,1,3,14,13,12,14,15,5,15,5,5,
  15,5,15,5,5,5,15,5,15,15,15,5,5,16,15,15,15,16,16,16,5,
  5,5,16,14,5,5,5,5,5,5,5,3,4,1,12,14,1,3,16,13,14,16,
  13,5,5,11,15,15,5,15,5,15,5,5,5,5,5,15,5,5,15,5,5,16,
  15,14,16,15,15,15,15,16,15,15,15,15,15,15,15,4,4,1,16,
  14,4,1,15,16,16,14,14,12,5,5,5,5,15,5,5,5,5,15,5,5,15,
  5,15,16,5,5,15,16,16,16,15,5,5,15,16,5,15,15,15,15,15,
  5,3,3,1,14,13,3,1,16,11,13,11,11,15,5,5,15,15,5,5,5,
  15,5,15,5,15,5,5,16,15,5,5,15,16,16,5,5,15,16,14,5,15,
  5,5,5,15,15,3,4,1,13,14,4,4,16,16,14,16,14,12,13,5,15,
  15,5,15,5,5,16,5,15,5,15,16,5,5,16,16,16,15,15,5,15,15,
  16,5,5,15,15,15,15,15,3,3,1,14,14,16,16,16,14,14,14,14,
  5,5,5,15,15,15,15,15,15,15,15,15,15,16,15,15,15,15,15,3,
  15,15,15,15,15,15,15,15,15,15,16,16,3,3,3,16,15,3,3,15,
  15,16,15,15,15,15,16,15,5,5,5,5,5,15,5,5,15,5,5,3,3,
  12,3,5,5,5,15,14,15,15,15,5,5,16,16,3,3,3,13,10,3,3,
  15,11,14,14,14,5,11,5,5,5,5,3,5,15,5,5,15,5,5,3,3,11,
  15,5,5,15,15,15,5,5,15,5,5,5,16,3,3,3,13,10,3,3,15,10,
  13,13,13,5,5,5,5,5,15,5,15,15,5,15,5,5,5,15,3,12,15,
  15,5,15,14,5,5,5,16,16,5,5,3,3,3,11,14,3,3,3,14,10,14,
  14,5,11,5,15,5,15,15,5,15,15,5,15,5,15,15,15,5,5,5,15,
  11,5,5,5,5,5,5,5,3,3,3,12,12,3,3,15,12,12,10,10,15,5,
  15,5,5,15,5,15,16,15,15,5,15,3,14,5,5,5,15,11,5,5,5,5,
  5,5,5,3,3,3,14,13,3,3,3,14,14,11,11,5,5,11,5,15,15,5,
  15,5,5,15,15,3,13,3,3,5,3,15,15,3,3,3,3,15,3,3,3,3,
  11,16,4,3,3,12,14,3,3,5,11,5,15,5,15,15,5,15,5,15,11,
  14,5,5,5,15,11,5,5,5,5,5,5,5,3,3,3,12,11,15,15,15,11,
  12,10,10,15,5,5,15,15,15,15,15,15,15,16,15,15,15,15,15,
  15,15,15,15,15,15,15,15,3,3,3,15,16,3,3,16,16,15,15,15,
  16,15,15,5,15,5,5,15,3,15,11,15,5,5,15,15,5,5,5,16,15,
  5,5,3,3,3,10,13,3,3,3,13,10,13,13,5,5,5,15,15,5,5,16,
  14,16,5,5,5,15,12,5,5,5,5,5,5,5,3,3,3,14,14,3,3,16,
  14,14,10,11,11,5,5,16,15,15,16,16,16,15,15,15,15,15,15,
  15,15,15,15,15,15,4,3,3,15,16,3,3,16,16,15,15,15,16,15,
  15,5,5,16,15,15,5,5,5,15,11,5,5,5,5,5,5,5,3,3,3,14,
  14,3,3,3,14,14,11,11,5,5,5,5,15,12,11,5,5,5,15,11,5,5,
  5,5,5,5,5,3,3,3,11,13,15,15,15,13,11,10,10,5,5,15,3,
  13,3,3,5,3,3,15,3,15,3,3,3,3,3,3,3,3,13,14,4,3,3,11,
  16,4,4,5,11,5,15,4,3,15,3,1,15,3,15,3,3,3,3,3,3,3,4,
  16,15,3,3,3,15,16,4,4,15,15,16,15,14,15,13,15,15,11,12,
  11,16,16,11,12,4,3,3,12,4,3,3,16,16,14,14,12,15,15,15,
  12,11,3,15,15,15,15,14,13,12,16,16,3,3,3,3,14,3,3,16,
  14,16,12,14,3,3,3,15,15,15,16,5,15,5,15,5,5,5,3,3,3,
  11,14,3,3,16,14,10,13,14,11,5,5,5,15,15,5,15,5,15,15,5,
  5,3,3,3,10,12,3,3,15,12,10,12,12,15,15,5,3,16,3,15,3,
  15,15,3,3,3,3,3,13,11,3,3,15,11,14,15,3,15,12,5,4,15,
  3,15,15,15,15,15,3,4,3,15,15,4,3,15,15,15,16,16,15,16,
  15,15,16,3,16,15,15,3,3,4,3,14,14,4,3,16,12,14,16,3,15,
  15,15,5,15,5,5,15,16,3,3,3,13,10,16,16,15,10,13,13,13,
  5,5,5,3,15,15,5,3,3,3,3,13,11,16,16,15,11,13,15,3,15,
  5,5,5,5,15,15,3,3,3,14,11,3,3,15,10,12,14,14,5,5,11,
  15,5,5,3,3,3,11,16,3,3,15,15,11,16,14,12,5,5,5,5,3,3,
  3,11,16,3,3,16,15,10,13,14,15,15,5,16,3,3,3,16,11,16,
  16,15,11,16,14,14,5,15,15,3,3,3,16,11,3,3,15,11,15,16,
  14,5,5,12,3,3,4,4,3,3,7,2,4,4,4,3,3,3,3,4,4,4,3,3,
  4,4,4,2,3,3,3,3,4,3,3,3,4,2,4,4,3,3,3,12,4,3,15,11,
  14,11,13,11,11,13,4,4,3,14,11,11,12,12,11,11,3,3,4,4,3,
  3,3,3,3,3,4,3,4,4,3,3,3,16,16,16,16,15,15,15,12,12,11,
  13,11,11,13,11,11,11,11,14,11,12,11,11,12,11,5,5,5,15,5,
  7,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,3,5,5,5,15,5,
  15,15,5,15,3,5,5,15,5,15,5,15,5,15,15,1,15,15,3,3,5,5,
  11,11,11,15,3,15,5,11,11,11,7,15,3,16,16,16,15,3,3,16,
  16,3,16,15,15,15,15,15,16,15,15,16,3,3,16,7,15,3,3,16,
  3,15,15,15,3,16,16,16,3,16,16,3,3,16,3,3,16,15,15,15,
  15,16,3,16,3,16,4,3,15,15,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,15,5,5,5,5,15,5,5,15,15,3,3,5,5,15,5,15,5,5,5,
  15,15,5,15,15,15,3,11,11,5,5,5,11,11,11,11,15,15,3,15,
  16,15,16,3,15,3,15,3,16,16,15,16,15,15,15,15,15,16,3,15,
  3,16,16,3,15,3,3,3,15,3,16,16,16,7,3,3,15,3,3,16,16,
  16,3,15,15,16,16,3,4,3,16,15,16,16,3,14,5,5,5,15,15,15,
  15,15,5,15,5,5,5,5,5,5,15,15,15,5,5,5,15,5,5,3,3,5,5,
  15,15,3,5,5,5,15,15,5,5,15,15,3,3,3,10,11,11,11,11,11,
  15,13,14,13,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,5,5,
  5,15,5,5,5,15,5,15,5,5,15,5,15,5,5,5,15,15,15,15,16,5,
  15,11,5,5,11,5,11,12,11,5,11,12,11,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,15,5,5,5,5,5,5,15,15,15,15,15,5,5,15,5,16,
  5,5,5,15,15,15,5,15,15,15,5,11,11,5,5,12,13,11,5,12,11,
  13,5,5,5,5,5,5,5,5,5,5,5,5,5,15,15,5,5,5,5,5,5,5,5,
  15,15,5,5,15,5,16,5,5,5,15,15,15,15,15,15,16,5,10,5,10,
  5,13,14,11,5,13,14,15,5,5,5,5,5,5,5,5,5,5,5,5,15,15,
  5,5,5,15,5,15,5,5,15,15,5,5,15,5,15,5,15,5,15,15,5,5,
  15,15,16,5,5,5,5,11,13,11,12,11,11,13,11,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,15,15,5,15,15,5,5,15,5,16,5,
  5,5,15,15,15,5,15,15,15,5,5,10,5,5,14,15,13,10,14,15,
  13,5,5,5,5,5,5,5,5,5,5,15,5,5,5,5,5,5,5,5,15,15,15,
  5,5,15,5,16,5,5,5,15,15,15,15,16,5,16,5,11,11,5,5,12,
  15,14,5,14,13,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,15,5,16,5,5,15,5,15,5,15,5,15,15,5,15,15,5,15,5,5,5,
  5,5,11,11,13,5,13,11,11,5,5,5,5,5,5,5,5,5,15,5,5,5,
  15,5,15,5,15,15,16,5,5,15,5,16,5,15,5,15,16,5,15,15,15,
  16,5,5,5,5,10,15,13,14,5,15,13,11,5,5,5,5,5,5,5,15,5,
  5,5,5,5,5,15,15,5,5,16,5,5,15,5,16,5,5,5,15,15,5,5,
  15,15,16,5,5,5,5,11,15,13,12,11,13,15,12,5,5,5,5,5,5,
  15,15,5,5,5,15,5,15,15,15,5,16,5,5,15,5,16,5,5,5,15,
  15,15,15,16,5,16,5,5,5,5,5,16,16,15,5,15,16,16,5,5,5,
  5,5,5,15,5,5,5,5,5,5,5,15,5,1,5,5,15,5,15,5,5,5,15,
  15,5,5,15,5,1,11,5,5,11,5,11,11,11,5,11,11,11,5,5,5,5,
  15,15,5,5,5,5,5,15,3,5,5,3,5,5,15,5,15,5,5,5,15,15,3,
  5,15,3,15,11,5,11,5,5,11,3,11,5,11,3,11,5,5,5,16,5,5,
  5,5,5,5,5,5,5,15,15,5,5,15,5,15,5,5,5,15,15,5,3,15,3,
  3,11,11,5,5,5,11,11,11,5,11,3,3,5,5,15,5,5,5,5,5,5,5,
  5,5,3,15,5,5,15,5,15,5,5,5,15,15,5,5,15,3,15,5,11,11,
  5,5,11,11,11,5,11,11,11,5,5,16,5,5,5,5,5,3,15,5,3,3,
  5,5,15,5,15,5,5,5,15,15,15,5,15,15,15,5,5,11,11,5,3,3,
  11,5,11,11,11,15,15,5,5,5,5,5,5,5,5,3,15,5,5,15,5,15,
  5,5,5,15,15,5,15,15,5,3,5,11,5,11,5,3,11,11,5,11,11,3,
  15,5,5,5,5,5,5,5,5,3,16,5,5,15,5,15,5,5,5,15,3,5,3,
  15,4,3,5,16,11,5,5,14,16,12,5,12,16,15,5,5,5,5,5,3,16,
  3,3,4,5,5,15,5,15,5,5,5,15,15,5,5,15,15,3,11,5,5,16,
  5,16,16,12,5,12,13,14,5,5,15,5,15,5,5,15,15,5,5,15,5,
  15,5,15,5,15,15,5,5,3,5,15,5,11,5,5,11,15,11,11,5,15,
  11,11,5,5,5,5,15,5,15,15,5,5,15,5,15,5,5,5,16,15,5,5,
  15,15,15,5,11,5,5,5,11,11,11,11,15,15,11,5,5,5,5,15,5,
  1,5,5,15,5,16,5,5,5,15,15,15,15,1,5,16,11,5,5,5,11,11,
  15,15,5,15,11,15,3,15,5,3,15,15,5,15,15,5,15,5,5,5,15,
  15,5,5,16,5,15,5,5,5,5,11,16,11,14,5,15,16,11,5,5,16,
  5,16,5,5,15,5,16,5,5,5,15,15,15,15,16,5,16,5,5,5,5,5,
  11,15,15,5,15,11,15,5,5,16,3,5,15,15,5,15,5,15,5,15,15,
  15,5,3,12,16,5,11,11,5,15,15,16,13,5,13,14,16,5,5,3,5,
  5,15,15,15,5,5,5,3,15,3,5,15,14,16,5,5,11,5,5,14,15,
  13,15,13,15,16,5,3,5,5,15,5,16,15,5,15,15,15,5,5,3,5,
  4,16,5,5,11,5,12,14,12,15,16,12,14,3,5,15,15,5,15,5,15,
  5,15,16,5,15,3,3,3,5,3,3,3,15,16,12,11,5,15,3,3,15,15,
  16,15,16,16,15,16,16,16,16,16,1,3,1,3,15,3,4,3,16,3,16,
  15,16,16,16,5,15,5,15,5,5,5,15,15,5,5,15,5,15,5,5,5,5,
  11,11,11,11,11,11,11,11,15,5,15,5,15,5,15,16,5,5,15,15,
  15,5,5,11,5,11,15,11,15,5,11,11,11,15,16,16,15,15,15,15,
  3,16,16,15,15,15,15,15,15,15,15,15,16,16,16,15,15,3,5,5,
  5,15,15,5,5,15,16,15,5,5,11,5,5,11,11,15,11,11,15,11,
  16,15,1,16,16,3,3,15,3,1,15,15,15,15,15,16,16,16,3,16,
  15,16,5,5,15,15,15,15,16,5,16,5,5,5,5,5,11,15,15,5,15,
  11,15,3,15,15,15,3,16,5,3,5,5,5,15,11,15,11,15,11,14,
  16,11,15,15,15,15,16,5,1,5,5,5,11,5,11,15,15,11,15,11,
  15,15,15,15,16,15,16,15,15,15,15,16,15,15,16,16,16,15,15,
  15,15,16,3,15,15,15,15,15,15,16,15,15,15,15,16,15,5,16,
  3,16,11,5,15,11,5,13,15,16,5,12,13,14,16,5,16,5,15,5,
  11,5,13,14,16,5,12,13,15,15,16,15,15,15,15,3,16,16,16,
  15,16,16,16,3,3,3,3,5,5,3,3,15,15,11,16,12,4,3,15,3,
  15,16,16,16,3,16,16,3,5,10,11,10,11,11,11,5,13,14,14,11,
  10,10,11,11,11,5,14,14,12,5,5,14,12,14,10,11,11,11,5,14,
  14,13,10,11,11,11,12,12,14,11,11,11,11,3,15,11,15,11,3,
  15,11,15,3,11,11,11,15,15,14,12,12,15,15,3,3,3,3,15,5,
  5,15,5,15,15,5,15,5,15,15,5,5,5,15,5,5,15,5,5,15,5,15,
  3,3,15,3,15,5,15,15,15,5,15,5,15,15,3,5,3,3,3,12,12,3,
  3,15,12,12,12,12,15,3,3,5,7,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,15,15,5,5,5,5,5,15,5,15,3,3,5,5,15,5,3,5,15,
  5,5,15,15,5,3,15,3,3,15,11,11,11,5,5,5,15,11,11,11,5,
  5,15,5,15,5,15,15,5,5,5,5,15,5,15,5,15,5,3,15,5,15,5,
  5,15,5,15,3,3,15,5,16,15,3,5,15,15,5,15,3,5,15,3,15,5,
  5,14,14,14,3,3,3,5,14,14,14,15,16,5,16,5,15,15,5,15,5,
  15,15,5,15,5,15,5,3,15,5,15,15,5,3,5,5,3,3,5,5,15,15,
  3,5,15,15,15,15,15,15,15,15,3,3,3,14,14,14,5,5,5,3,14,
  14,14,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,16,5,5,5,5,5,
  15,5,15,15,15,5,5,15,5,15,5,15,5,5,15,15,15,15,5,15,11,
  11,5,11,11,12,11,11,5,11,5,11,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,15,5,5,5,5,5,5,15,15,5,15,15,5,5,15,5,15,5,15,
  5,5,15,16,15,5,15,5,11,11,11,11,5,11,11,12,11,5,11,11,
  5,5,5,5,5,5,5,5,5,5,5,5,5,15,15,5,5,5,5,5,16,5,5,5,
  5,5,5,16,5,5,5,16,5,5,15,16,15,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,5,5,5,15,5,
  16,15,5,15,15,5,5,15,5,15,5,15,5,14,16,15,5,15,15,15,5,
  11,11,5,11,11,11,11,11,11,11,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,16,15,5,5,5,5,5,16,5,5,5,16,5,5,15,15,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,15,5,5,5,5,5,15,15,15,15,15,5,5,15,5,15,5,15,5,5,
  15,15,5,15,15,15,11,11,12,12,5,13,15,5,11,5,12,12,5,5,
  5,5,5,5,5,5,5,5,15,5,5,5,15,5,15,5,5,15,15,5,5,16,5,
  15,5,15,5,15,15,15,5,15,5,5,5,5,5,11,11,11,11,11,11,11,
  5,11,5,5,5,5,5,5,5,5,5,5,5,5,5,15,5,15,5,5,5,5,5,5,
  15,5,5,5,16,5,15,16,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,15,15,5,5,5,15,5,15,5,5,5,15,5,5,
  15,5,5,5,15,5,5,15,15,5,15,15,15,5,11,12,5,12,15,12,12,
  11,12,12,5,5,5,5,5,5,5,15,15,5,5,5,15,5,16,5,5,5,5,5,
  5,16,5,5,5,16,5,5,16,16,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,3,5,5,5,5,5,15,5,15,15,15,5,5,15,5,
  5,5,15,5,5,15,15,15,5,5,3,11,11,5,3,5,5,11,11,5,5,5,
  3,5,5,5,5,5,5,5,5,5,5,5,15,15,5,5,15,5,5,15,5,15,5,
  15,5,5,15,3,5,5,3,5,11,5,3,5,5,11,11,5,11,5,3,5,5,5,
  5,15,5,5,5,5,5,5,15,5,5,5,5,5,5,15,5,5,5,15,5,5,15,
  15,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,5,5,5,5,
  5,5,15,15,5,15,5,5,5,15,5,5,5,15,5,5,3,3,5,5,3,5,5,
  11,15,5,5,11,11,5,11,5,15,5,5,5,15,5,5,5,5,5,3,5,5,5,
  5,5,5,15,5,5,5,15,5,5,16,16,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,3,5,5,5,5,5,15,5,15,15,15,5,5,15,5,15,5,15,
  5,5,15,15,15,5,5,3,11,11,5,15,5,11,5,11,5,5,5,15,5,5,
  5,5,5,5,15,5,5,15,15,5,5,15,15,5,5,15,5,5,3,3,3,5,15,
  5,15,15,13,5,5,5,5,5,15,5,13,5,5,5,5,5,5,3,5,3,12,14,
  5,3,15,5,13,7,7,3,5,3,3,3,5,5,3,14,11,5,3,5,11,14,12,
  12,5,5,3,5,5,15,5,16,5,5,15,15,5,5,15,5,15,5,15,5,15,
  15,15,5,3,5,5,11,11,5,5,15,11,5,11,5,15,5,5,5,5,5,15,
  15,5,5,5,5,5,15,5,5,5,15,5,15,15,15,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,3,5,5,15,15,5,5,15,5,5,5,15,5,
  15,15,15,5,3,5,5,11,11,5,5,3,5,11,11,5,3,5,5,15,16,5,
  3,15,5,15,5,15,5,15,5,15,5,5,15,15,5,15,5,5,5,5,5,5,
  13,15,15,15,5,13,5,5,15,5,15,5,5,5,5,15,5,5,5,15,5,5,
  15,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,15,15,3,15,
  16,16,15,15,3,16,15,15,15,3,15,3,15,15,15,15,15,15,16,
  15,15,15,15,16,15,15,5,5,15,5,5,15,15,5,5,15,5,3,15,3,
  5,5,3,5,5,5,13,5,5,15,15,5,5,5,13,5,15,15,5,5,15,5,
  15,5,15,15,5,15,15,5,5,5,3,15,15,5,13,5,5,5,5,5,5,5,
  13,5,15,5,15,5,5,5,15,15,15,15,15,15,3,3,3,15,15,3,3,
  15,15,5,15,5,15,3,3,5,5,15,15,5,5,15,5,15,15,15,15,3,
  3,3,15,15,3,3,3,5,15,15,5,3,3,3,5,16,5,15,5,15,5,15,
  15,15,5,3,5,5,11,5,5,5,15,11,5,11,11,15,5,5,16,5,5,5,
  15,5,5,15,3,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,15,7,
  15,15,15,16,7,15,3,15,15,15,15,15,15,16,15,15,15,15,16,
  15,15,15,5,15,5,5,15,3,5,5,3,5,5,11,15,5,5,5,11,11,11,
  5,15,5,5,15,15,15,15,3,15,3,3,3,15,5,3,3,15,15,5,15,
  15,15,3,3,7,5,5,3,7,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,3,15,15,3,3,15,15,3,15,15,15,3,15,15,15,15,15,15,15,
  3,5,15,15,15,5,5,3,11,5,5,15,5,11,5,11,11,5,5,15,15,
  15,5,15,5,5,5,5,5,5,13,15,5,15,5,13,5,5,16,15,15,3,15,
  15,15,3,15,15,15,15,15,15,15,3,15,15,15,3,15,15,15,3,15,
  15,15,15,15,15,15,3,15,5,5,15,5,5,5,13,5,5,15,15,5,5,
  5,13,5,5,3,3,5,5,7,15,3,15,3,7,5,5,5,3,15,7,5,5,3,3,
  3,15,5,7,5,3,3,5,7,5,3,3,3,3,5,5,7,5,11,11,14,5,5,5,
  5,14,11,11,11,12,11,5,5,5,5,11,11,12,5,5,12,11,12,12,5,
  5,5,5,12,12,14,12,5,5,5,11,11,12,11,5,5,5,5,5,5,11,12,
  12,5,5,11,11,12,5,12,12,14,11,12,12,5,5,5,3,7,3,1,3,4,
  1,1,3,1,1,16,3,1,3,3,1,1,3,3,3,1,1,3,7,3,1,7,3,4,1,
  1,15,1,4,15,3,3,1,16,1,7,3,3,3,4,3,1,1,1,3,3,3,4,3,
  4,3,3,16,15,15,4,3,1,16,1,1,3,15,15,16,15,15,15,16,15,
  1,1,1,1,1,3,1,3,4,4,15,1,15,1,16,1,1,1,16,3,1,3,16,
  4,4,16,16,16,16,3,3,4,16,3,16,3,4,1,3,3,4,1,1,1,3,1,
  1,3,3,3,3,3,3,3,3,4,3,3,4,3,3,3,3,4,4,3,3,3,3,4,3,
  4,3,3,3,3,4,4,4,4,4,4,3,3,3,3,3,3,4,4,4,4,15,15,15,
  15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,
  15,16,15,16,16,16,15,15,15,16,15,15,15,16,15,15,16,15,4,
  3,14,16,15,16,15,16,15,16,15,16,4,4,15,5,5,15,15,15,15,
  5,5,15,5,15,15,15,15,15,5,5,5,5,5,5,16,15,16,16,15,5,
  5,5,16,5,5,5,15,5,15,15,15,4,4,15,14,12,5,5,14,14,16,
  13,16,4,1,15,5,15,15,5,15,5,5,15,5,15,5,15,15,15,5,5,
  5,5,5,5,16,5,14,14,15,5,5,15,16,5,5,15,15,5,16,16,16,
  3,3,11,11,15,5,15,14,11,14,14,16,1,3,5,15,15,15,15,5,5,
  15,15,5,5,15,5,5,15,5,5,15,15,15,16,5,16,14,15,15,5,5,
  14,5,15,15,15,15,5,15,1,4,4,16,14,5,11,12,14,14,16,13,
  16,1,4,15,15,5,15,5,15,15,5,5,5,15,15,5,5,5,15,15,5,
  15,15,15,14,15,15,5,5,15,14,5,5,5,16,5,15,5,1,3,3,13,
  12,5,15,5,12,12,13,12,3,1,1,15,15,15,15,15,16,15,15,15,
  16,15,15,15,15,15,15,15,15,16,16,16,16,16,15,15,15,16,15,
  15,15,16,15,15,15,1,4,4,15,16,16,15,15,16,16,16,16,4,1,
  1,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,16,
  15,16,16,15,15,15,15,16,15,15,16,16,15,16,15,16,4,4,16,
  16,15,15,15,15,16,16,15,16,4,4,15,5,5,15,5,5,5,15,5,5,
  15,5,15,15,5,5,16,5,16,14,15,5,5,5,16,5,15,15,16,15,15,
  5,1,3,3,14,14,15,15,15,13,14,12,13,16,3,1,15,15,15,15,
  15,15,15,15,15,15,15,15,15,15,15,16,15,16,16,16,15,15,15,
  16,15,15,15,15,15,16,16,1,3,4,16,16,15,15,16,16,16,15,
  16,4,1,1,5,15,5,15,5,15,15,15,15,5,15,15,5,5,16,15,14,
  15,15,5,5,15,14,5,15,5,16,15,15,5,1,3,3,11,11,5,5,5,
  12,11,13,14,16,16,16,15,15,15,15,15,16,16,5,5,5,5,5,5,
  3,3,14,3,15,5,5,5,12,5,5,5,15,15,5,5,15,3,3,14,14,5,
  11,5,14,14,11,10,15,3,3,15,15,15,15,15,15,15,15,15,15,
  15,15,16,15,15,4,15,15,15,15,15,15,15,15,16,15,16,15,15,
  7,3,16,15,16,15,15,15,16,15,15,16,3,3,15,5,15,16,15,5,
  5,5,5,5,15,3,3,15,15,15,5,5,5,11,5,5,5,15,5,5,3,15,3,
  3,13,13,5,5,5,13,13,10,10,15,3,3,15,15,15,5,5,5,5,5,5,
  3,3,3,3,14,15,5,5,5,12,5,5,5,15,3,5,3,15,3,3,14,13,
  11,5,5,14,14,11,10,15,3,3,15,15,15,5,5,5,5,5,3,3,3,3,
  3,15,5,5,5,11,5,5,5,15,5,5,5,15,3,3,12,12,15,15,5,12,
  12,10,10,15,3,3,15,15,15,15,15,15,15,3,1,3,3,15,15,15,
  15,15,15,15,15,15,15,3,15,3,15,3,3,15,16,15,16,15,16,15,
  15,15,15,3,3,15,5,5,5,5,5,3,3,3,3,14,15,5,5,5,13,5,5,
  5,15,3,5,3,15,3,3,15,16,12,5,5,14,14,11,11,15,3,3,5,5,
  5,5,5,3,3,3,3,3,15,5,5,5,12,5,5,5,15,3,3,3,15,3,3,
  15,13,15,15,5,14,15,11,10,15,3,3,5,5,15,15,15,3,3,16,
  11,15,15,5,5,14,5,5,15,15,15,5,3,3,3,3,11,14,5,5,11,
  14,10,14,14,16,3,3,15,5,5,5,16,5,11,15,15,5,5,15,15,5,
  5,5,16,5,15,5,3,3,3,13,10,5,5,5,10,13,13,13,3,3,3,5,
  5,5,16,5,11,3,15,5,5,15,14,15,5,5,16,5,16,5,3,3,3,13,
  10,5,5,11,11,14,14,14,3,3,3,16,15,3,3,15,12,15,15,5,5,
  16,5,15,16,15,15,5,3,3,3,3,11,14,5,5,12,16,11,15,14,16,
  3,3,5,3,15,15,11,15,5,5,5,15,5,15,15,15,5,5,15,3,3,3,
  11,14,5,5,5,14,11,14,14,3,3,3,3,3,16,3,15,5,5,5,13,5,
  15,5,15,15,5,5,3,3,3,14,3,5,11,15,15,14,12,11,15,3,3,
  3,3,4,16,15,15,15,16,16,3,15,16,15,16,15,3,4,3,16,4,16,
  15,15,15,16,16,15,3,3,3,15,3,15,5,5,5,16,5,3,5,15,15,
  5,15,3,3,3,16,3,5,11,5,15,11,14,12,3,3,4,3,15,15,11,
  12,15,11,16,14,15,16,12,15,3,3,4,14,3,3,3,15,16,12,14,
  14,16,3,3,15,11,11,14,15,15,11,12,16,15,16,14,3,4,3,4,
  14,4,3,3,14,3,14,12,3,4,3,15,15,15,16,15,15,15,15,15,
  15,15,16,3,3,15,15,15,15,16,15,15,16,16,16,3,3,5,5,15,
  5,15,15,15,15,5,3,3,3,3,10,12,5,15,15,12,10,12,12,15,3,
  3,15,14,15,5,5,15,5,16,5,15,3,3,11,10,15,5,5,10,11,12,
  11,15,15,15,3,15,5,5,16,5,5,5,3,3,3,14,11,11,5,5,10,
  12,14,14,3,3,3,15,16,3,16,12,3,3,16,3,3,14,12,15,15,15,
  14,14,16,4,16,3,3,5,5,16,5,16,5,3,3,3,13,10,5,5,5,10,
  13,13,13,16,16,15,16,15,15,5,3,3,3,3,11,14,5,15,15,14,
  11,15,15,16,3,3,15,5,5,15,3,3,3,10,14,5,11,5,14,10,14,
  13,3,3,3,15,16,15,16,3,3,15,15,15,15,15,15,16,16,16,16,
  3,3,5,3,3,3,3,13,3,5,5,15,15,13,11,11,15,16,16,5,3,3,
  3,16,11,15,5,15,11,16,14,14,3,3,3,3,3,3,14,3,5,11,5,
  15,11,15,13,3,3,4,3,3,15,15,15,15,3,4,4,16,2,16,3,3,3,
  4,4,3,3,3,4,2,4,4,3,4,3,4,2,3,3,3,4,4,4,4,3,3,7,12,
  13,11,11,11,14,11,13,16,3,4,11,12,11,14,11,11,12,16,4,3,
  5,5,11,11,11,11,15,3,3,5,14,11,11,12,15,3,3,11,11,12,
  12,15,3,3,12,12,11,15,4,3,13,11,4,3,4,14,16,4,4,3,4,4,
  3,3,3,15,5,15,15,5,15,5,15,5,5,5,5,15,5,5,5,15,5,5,5,
  15,5,5,5,5,15,15,3,3,15,5,15,5,3,5,5,5,15,5,15,5,15,
  15,3,11,11,10,11,10,3,3,15,11,13,14,13,3,3,15,16,16,16,
  16,3,16,16,16,3,15,3,16,15,16,16,15,15,15,15,16,16,16,
  15,3,3,16,3,16,15,3,7,7,16,3,3,3,3,16,16,16,3,4,3,16,
  16,15,15,15,16,16,16,16,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,15,5,5,5,5,5,15,5,15,15,15,5,5,5,5,15,5,5,5,15,5,
  15,15,15,16,15,15,13,5,11,5,5,5,5,14,14,14,12,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,15,5,5,5,5,5,5,15,15,15,16,5,5,
  5,5,5,15,5,15,5,5,15,15,15,15,16,5,12,15,11,5,5,5,11,
  5,13,14,12,15,5,5,5,5,5,5,5,5,5,5,5,5,5,15,15,5,5,5,
  5,5,5,15,5,15,15,5,5,5,5,15,5,5,5,5,15,15,15,15,16,15,
  14,13,5,5,5,11,5,12,11,14,15,13,5,5,5,5,5,5,5,5,5,5,
  5,5,15,5,5,5,5,15,5,15,15,5,16,15,5,5,15,5,5,5,14,5,
  5,15,5,15,5,16,15,14,12,5,5,11,11,5,5,15,12,15,13,5,5,
  5,5,5,5,5,5,5,5,5,5,15,5,5,5,15,5,15,15,15,16,15,5,5,
  15,5,15,5,5,5,15,5,15,5,15,16,15,15,14,5,11,5,5,5,11,
  13,15,13,11,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  15,15,5,5,5,5,15,5,5,5,5,5,5,5,15,15,5,11,14,11,5,5,
  5,11,5,11,11,11,12,5,5,5,5,5,5,5,5,5,5,15,5,5,5,15,5,
  5,15,5,16,15,5,5,15,5,15,5,15,5,15,5,5,5,15,16,15,14,
  14,5,5,5,5,5,11,12,15,13,14,5,5,5,5,5,5,5,5,15,5,5,5,
  5,15,5,15,5,5,16,15,5,5,5,5,15,5,15,5,15,15,5,5,15,15,
  15,13,15,11,11,5,5,5,5,14,13,11,13,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,15,15,5,5,5,5,5,5,5,5,5,5,5,5,15,15,
  15,13,11,5,5,11,11,5,5,14,11,12,11,5,5,5,5,5,5,15,15,
  5,5,5,15,5,5,5,5,16,5,5,5,5,5,5,5,5,5,15,15,15,5,5,
  16,5,15,14,5,5,5,5,5,5,14,15,14,14,5,5,5,5,5,5,16,5,
  5,5,15,5,15,5,16,15,3,5,5,5,5,5,5,5,5,5,5,5,5,5,15,
  3,3,11,5,11,5,5,11,5,11,11,11,3,5,5,5,5,5,5,5,5,5,5,
  5,5,16,5,15,3,5,5,15,5,5,5,5,5,15,5,16,5,5,1,5,15,11,
  11,5,5,5,11,5,15,15,15,11,5,5,5,16,5,5,5,5,5,5,5,5,5,
  15,5,5,5,5,5,15,5,5,5,5,15,5,16,5,15,15,11,11,5,5,5,
  11,5,11,11,11,15,15,5,5,16,5,5,5,5,5,5,5,5,5,3,5,5,5,
  5,5,5,5,15,5,5,16,5,15,5,3,5,11,15,11,5,5,11,5,5,11,
  11,15,11,5,5,16,5,5,5,5,5,16,5,15,3,3,5,5,5,5,5,5,5,
  5,5,5,5,5,15,15,15,3,3,5,11,11,5,5,5,11,11,11,11,5,5,
  5,5,5,5,5,5,5,5,1,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,3,
  11,11,5,11,5,11,5,5,11,11,11,11,5,5,5,5,5,5,5,5,5,3,
  5,5,15,5,5,15,5,5,5,5,16,5,16,5,16,5,11,16,11,5,5,15,
  5,5,14,12,16,16,15,5,5,5,15,16,5,16,3,3,5,5,5,5,5,5,
  5,5,5,5,5,5,15,15,3,16,16,5,15,14,5,11,5,12,14,12,13,
  5,5,16,5,16,5,15,15,5,5,5,5,5,5,5,5,5,5,5,5,5,3,15,
  5,15,15,5,5,11,11,5,5,11,15,11,15,5,5,5,5,16,5,15,15,
  5,5,15,5,15,5,5,5,16,5,5,5,15,15,5,11,11,5,5,5,5,11,
  11,11,15,15,11,5,5,5,5,5,15,3,5,5,5,5,5,5,5,5,5,5,5,
  5,3,15,5,15,11,5,5,15,5,15,5,11,15,11,11,16,16,5,16,16,
  15,5,5,5,5,5,5,5,5,5,5,5,5,4,15,3,14,13,5,5,11,15,11,
  5,12,16,12,16,15,5,16,16,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  15,5,15,15,5,5,5,5,5,5,11,15,11,15,5,5,16,3,5,5,5,5,
  5,5,5,5,5,5,5,5,3,15,15,16,16,5,14,15,11,5,5,12,13,12,
  14,5,15,3,15,5,5,5,15,15,5,5,16,5,16,5,15,16,5,13,12,
  11,5,5,5,15,14,14,16,16,12,16,3,5,5,5,5,5,5,5,5,5,5,
  5,5,5,15,14,16,14,5,11,5,5,15,5,12,13,12,16,3,15,3,15,
  15,3,15,3,15,15,15,15,16,3,1,3,16,3,3,4,15,16,15,15,15,
  16,16,16,5,5,5,5,5,5,5,5,15,5,3,5,3,3,3,4,11,3,3,3,
  5,3,5,12,15,15,3,5,16,5,3,5,5,5,16,5,15,5,16,15,5,11,
  11,5,5,11,5,5,11,15,15,11,11,5,5,3,5,16,5,5,16,5,5,5,
  15,15,11,15,11,11,5,5,5,5,15,11,11,11,5,14,16,5,5,16,5,
  16,5,15,16,5,13,12,5,5,15,5,5,11,16,15,14,12,3,5,5,5,
  5,5,5,5,5,15,5,11,11,11,5,5,5,5,11,11,11,11,11,5,3,3,
  3,5,3,3,3,3,3,11,12,4,3,3,5,5,3,4,15,15,3,5,5,15,5,
  16,5,5,16,5,15,11,5,5,5,5,5,5,15,15,15,11,16,5,16,5,
  16,5,16,3,11,16,14,15,5,5,5,11,16,12,14,16,5,5,5,16,5,
  15,16,11,11,5,11,5,5,5,11,15,11,11,15,5,5,5,3,3,5,14,
  12,5,5,11,5,14,15,13,16,16,12,5,5,5,3,15,12,16,15,5,5,
  11,5,5,16,11,15,14,5,15,3,5,16,12,15,5,11,5,11,5,16,14,
  13,12,5,3,4,12,14,5,11,5,14,5,15,16,11,15,16,15,5,4,3,
  5,5,3,3,3,15,15,16,12,11,15,15,15,3,15,15,15,15,15,16,
  15,16,16,3,3,5,3,5,4,3,3,15,12,11,16,3,12,12,14,11,11,
  11,15,15,15,3,12,14,12,11,11,11,15,15,15,11,5,5,10,10,
  10,14,11,11,11,5,10,10,10,14,11,11,11,10,10,10,12,11,11,
  11,5,5,11,12,12,14,5,11,14,12,12,11,12,14,13,15,11,15,
  15,15,15,15,15,5,5,5,15,15,5,15,5,15,15,5,5,5,15,3,15,
  15,5,5,15,5,3,15,15,3,3,15,5,15,15,3,5,5,5,5,5,15,3,
  15,15,3,3,3,12,12,12,3,3,15,15,12,12,12,15,15,15,15,15,
  15,5,5,5,15,15,15,5,5,15,15,15,5,5,5,16,15,16,5,15,4,
  4,15,15,15,5,15,5,15,15,5,5,15,15,16,14,16,12,5,15,14,
  16,4,4,16,5,16,14,14,15,15,15,15,15,15,15,15,15,15,15,
  15,15,16,16,15,15,15,15,15,15,15,15,15,4,3,16,15,16,15,
  15,15,15,15,15,16,15,15,16,16,16,15,16,16,16,16,1,1,16,
  15,16,15,16,5,5,5,15,5,5,5,15,15,5,5,5,15,15,15,5,5,
  15,15,5,15,5,5,3,3,15,5,16,15,15,5,5,15,5,5,16,15,16,
  16,16,15,5,12,13,12,3,1,15,5,14,12,12,15,5,15,15,15,15,
  5,15,5,5,15,15,5,5,5,5,5,16,15,15,5,5,3,4,15,15,15,5,
  1,15,15,15,5,15,15,15,14,14,15,5,5,16,11,14,1,3,4,11,
  15,14,12,15,15,5,15,5,5,15,5,15,5,16,5,5,5,5,5,5,5,15,
  15,5,3,3,16,5,16,15,1,5,5,15,5,15,16,5,14,15,14,15,5,
  13,13,13,1,3,16,15,14,13,13,15,5,15,15,5,15,5,5,5,16,5,
  5,5,15,5,15,5,5,5,15,4,3,15,5,16,5,1,15,15,5,5,15,16,
  15,16,16,14,12,13,16,16,14,3,1,4,5,14,14,14,15,15,15,15,
  16,15,15,15,16,15,15,15,15,15,15,15,15,15,15,3,4,15,15,
  16,15,15,15,16,16,15,15,15,15,16,16,16,15,15,16,15,16,1,
  1,16,15,16,16,15,15,5,5,16,5,5,5,16,5,15,15,5,15,15,5,
  15,5,15,3,3,15,5,16,5,1,5,15,15,15,15,15,15,15,14,16,
  15,15,13,14,13,1,1,3,15,14,13,13,5,5,15,5,5,5,16,5,5,
  5,15,5,5,5,15,15,5,4,3,15,5,16,5,1,15,5,5,15,15,16,5,
  15,16,14,5,5,12,16,14,1,3,15,11,13,14,13,15,15,5,15,5,
  15,15,15,15,5,15,15,5,16,5,15,3,3,15,5,16,5,1,5,15,15,
  5,15,16,5,15,16,16,5,5,14,14,14,16,16,16,5,14,14,14,15,
  15,15,15,15,5,15,5,5,5,5,5,3,5,3,3,3,15,5,15,5,15,5,
  5,5,5,3,3,3,12,14,3,11,5,14,14,10,3,3,15,5,11,14,14,
  15,15,15,15,15,15,15,15,15,15,15,15,15,15,3,3,15,15,16,
  15,16,15,15,15,15,15,15,15,15,3,15,16,15,16,15,15,3,3,
  15,15,15,16,15,15,5,15,15,15,5,5,5,5,5,5,5,5,3,3,15,5,
  15,5,15,5,5,5,5,3,3,3,11,3,3,15,15,12,12,10,3,3,15,5,
  10,12,12,15,15,16,16,5,5,5,5,5,5,5,5,3,3,15,5,15,5,15,
  5,5,5,5,3,3,5,12,3,14,5,11,13,14,10,3,3,15,5,11,14,14,
  15,15,16,5,5,5,5,5,3,5,5,3,3,15,5,15,5,15,5,5,5,5,3,
  3,5,11,15,15,5,5,13,13,10,3,3,15,5,10,13,13,15,15,15,
  15,15,15,15,15,15,15,3,3,15,15,15,15,16,15,15,15,15,15,
  15,15,15,15,4,15,16,15,16,15,3,3,16,15,15,15,16,15,5,5,
  5,5,5,3,5,3,3,3,15,5,15,5,15,5,5,5,5,3,3,3,12,3,3,
  15,15,15,14,11,3,3,15,5,11,16,14,5,5,5,5,5,3,5,3,3,3,
  15,5,15,5,15,5,5,5,5,3,3,3,12,16,3,12,5,15,16,11,3,3,
  15,5,11,16,14,5,5,16,15,15,5,5,3,3,15,15,15,5,3,5,16,
  15,5,5,15,15,3,12,14,5,11,11,14,14,3,3,3,5,14,10,12,15,
  5,5,5,15,5,3,3,15,5,15,15,3,5,5,5,15,5,15,5,15,15,11,
  15,5,12,10,12,3,3,15,15,12,12,10,5,5,5,5,5,3,3,15,5,
  16,15,3,15,5,5,15,5,15,5,3,14,12,11,5,14,10,13,3,3,3,
  5,14,14,10,15,3,5,3,3,3,16,5,3,5,3,5,16,15,5,15,15,5,
  16,11,15,5,15,11,14,16,16,15,16,15,16,11,14,15,5,15,3,3,
  15,5,15,5,3,5,16,15,5,15,15,5,15,11,15,5,5,10,13,13,16,
  15,16,5,13,10,13,5,15,3,3,15,15,15,5,3,5,3,15,5,15,15,
  15,3,13,15,5,11,11,15,15,4,3,3,5,12,14,15,5,3,3,15,5,
  15,15,3,5,5,5,15,5,15,5,11,14,11,5,5,12,10,11,15,15,15,
  15,11,12,10,3,3,15,15,15,5,3,5,3,15,5,5,15,15,3,13,4,
  15,11,12,15,14,3,3,3,5,14,11,15,3,3,3,3,3,3,3,3,3,3,
  3,3,3,3,3,4,3,3,4,4,4,4,3,3,3,4,4,2,3,3,3,3,3,3,3,
  3,3,3,4,3,3,4,3,3,3,2,4,4,3,7,3,3,4,4,4,15,15,15,16,
  15,16,15,15,15,15,15,16,15,15,15,15,15,15,16,3,3,16,16,
  16,15,15,15,5,3,5,16,15,5,15,3,5,15,11,15,5,5,10,13,13,
  3,3,3,5,13,10,13,15,16,16,3,15,15,15,16,15,16,16,15,15,
  15,15,15,16,3,3,16,16,16,15,15,3,15,5,5,5,5,16,5,14,16,
  11,5,5,14,11,14,3,3,16,11,14,14,10,3,3,3,15,3,3,3,16,
  3,3,15,15,15,15,2,3,3,16,3,16,4,4,5,5,5,5,16,5,15,15,
  11,5,5,14,11,14,3,3,3,5,14,14,11,15,5,3,3,3,16,12,3,5,
  5,11,16,15,3,3,3,12,16,11,14,5,15,3,15,14,11,3,5,5,10,
  13,14,3,3,3,11,14,11,14,5,15,5,12,11,11,15,5,11,10,13,
  15,15,15,5,13,11,10,3,15,13,3,15,5,11,13,15,14,4,3,3,5,
  11,15,15,3,15,4,3,16,15,16,15,15,3,3,3,15,15,16,15,15,
  12,16,15,5,11,15,13,16,15,16,5,13,11,15,15,15,15,15,12,
  14,4,3,3,16,15,16,14,14,3,3,3,3,14,14,3,3,16,15,14,16,
  12,3,4,14,4,12,4,3,3,3,14,14,3,5,12,11,12,3,3,15,5,11,
  14,11,11,13,11,3,3,15,5,11,11,11,12,12,4,3,16,11,11,14,
  11,13,3,4,16,11,11,11,14,4,4,3,12,14,11,11,3,3,3,4,4,
  3,3,3,4,3,4,15,16,16,4,12,11,11,12,13,12,15,15,15,15,
  15,15,15,15,15,15,15,15,15,15,15,15,16,16,15,15,15,15,16,
  15,15,4,1,15,15,15,16,3,15,15,16,15,15,15,16,16,16,16,1,
  4,16,16,16,15,15,15,4,16,16,16,15,5,5,15,15,15,15,5,15,
  15,15,5,5,5,15,15,15,5,5,5,5,16,15,5,4,1,5,15,5,15,4,
  15,15,15,5,5,15,15,14,16,14,4,1,14,16,13,5,12,11,4,16,
  14,14,15,5,15,15,5,15,5,15,15,5,5,5,5,15,15,15,5,15,15,
  5,16,5,5,3,1,5,5,5,15,3,5,5,15,5,5,15,15,14,16,16,1,
  3,11,11,14,15,15,15,15,14,14,11,5,15,15,15,15,5,15,5,5,
  14,5,5,5,16,15,15,5,15,5,15,5,5,4,1,5,5,15,15,4,5,15,
  5,15,15,5,5,16,16,16,1,4,14,16,13,12,5,5,16,16,14,14,
  15,15,5,15,5,15,5,5,5,5,15,5,15,15,5,15,15,5,15,15,5,
  3,1,15,5,15,15,3,5,5,5,5,15,15,5,15,14,14,3,1,12,13,
  14,5,5,15,16,13,12,12,15,15,15,15,16,15,15,15,15,15,16,
  16,15,15,15,16,15,15,15,15,3,1,15,15,15,15,4,15,16,15,
  15,15,15,15,16,16,16,4,1,16,16,16,15,16,15,16,16,16,16,
  15,15,15,16,15,15,15,15,15,15,16,15,15,15,15,15,15,15,15,
  4,1,15,15,16,16,3,15,16,15,16,15,15,15,16,16,16,1,4,16,
  16,15,15,15,16,16,16,16,16,15,5,15,5,5,15,5,15,5,16,15,
  5,5,16,5,15,5,5,3,1,5,5,5,15,3,5,16,15,15,15,5,5,14,
  14,16,1,1,14,14,13,15,15,5,3,12,13,14,15,15,15,15,15,15,
  15,16,15,15,15,15,15,15,16,15,15,3,1,15,15,16,15,4,15,
  15,15,15,15,16,15,16,16,16,4,1,16,16,16,16,15,15,16,16,
  16,16,15,15,5,15,5,15,15,16,15,5,15,15,5,15,5,5,3,1,5,
  5,15,15,3,5,5,5,15,15,15,15,15,14,14,16,16,11,11,14,5,
  5,5,16,13,12,11,15,15,15,15,15,15,15,15,15,15,15,15,16,
  15,15,3,3,15,15,15,15,3,15,15,15,15,15,15,15,15,15,3,3,
  3,15,16,15,15,16,15,16,15,15,16,15,5,5,5,3,3,15,5,5,5,
  5,3,3,5,3,3,5,5,3,15,3,15,3,15,5,5,15,5,11,3,14,3,3,
  13,14,11,5,11,5,3,11,14,14,5,5,5,3,3,15,5,5,5,5,15,5,
  5,3,3,5,5,5,15,3,5,3,15,5,5,15,15,11,15,15,3,3,12,12,
  10,5,15,15,15,10,12,12,5,5,15,3,15,15,5,5,5,3,3,5,3,3,
  15,15,3,15,3,5,15,5,5,15,5,5,14,3,12,3,3,14,11,14,11,
  5,5,3,14,14,10,15,5,16,15,5,15,15,5,16,5,5,3,3,5,5,5,
  15,3,5,5,5,5,5,5,5,15,11,15,3,3,10,13,13,5,5,5,15,13,
  10,13,5,16,15,5,15,5,15,15,5,15,3,3,5,5,5,15,3,5,5,5,
  5,5,5,5,14,12,3,3,3,10,13,14,11,5,5,15,14,11,14,3,15,
  5,5,5,5,15,1,5,3,3,15,5,15,15,3,5,15,5,15,3,3,3,15,
  16,12,15,16,15,11,13,5,15,5,16,13,15,11,16,15,16,16,16,
  16,15,15,3,4,15,3,15,15,3,3,3,3,3,3,3,3,16,15,4,3,3,
  2,16,16,15,15,15,16,15,15,15,15,15,16,15,16,15,15,7,3,
  15,15,15,15,3,15,15,15,15,15,15,15,4,15,15,3,3,15,15,16,
  16,15,15,16,16,15,15,5,5,5,3,15,5,3,3,15,5,15,15,3,5,
  5,5,15,15,5,5,15,15,11,3,3,12,10,12,15,5,15,15,12,12,
  10,16,15,16,5,5,3,3,5,5,5,15,3,5,5,5,5,5,5,5,3,11,14,
  3,3,11,14,14,5,11,5,15,14,10,12,16,16,5,16,3,3,5,5,5,
  15,3,5,5,5,15,5,5,5,16,12,3,3,3,10,14,16,16,15,5,15,
  16,11,14,16,5,15,3,3,5,5,5,15,3,5,5,5,5,5,5,5,15,11,
  15,16,16,10,13,13,5,5,5,15,13,10,13,3,15,4,3,3,3,3,15,
  3,3,15,15,3,3,3,3,4,15,16,3,3,2,16,16,16,15,15,16,16,
  15,15,5,3,3,5,15,15,15,3,5,5,5,3,3,3,3,16,16,13,3,3,
  4,11,14,5,15,12,16,13,15,11,3,3,5,5,5,15,3,5,5,5,5,5,
  5,5,11,11,12,15,15,10,13,11,15,5,5,15,11,10,13,3,3,3,3,
  3,3,3,3,3,3,3,3,3,4,3,3,3,4,4,4,2,3,3,3,3,4,4,4,3,
  3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,4,3,4,3,3,3,3,4,4,
  2,15,15,15,3,5,5,5,16,16,5,5,15,14,12,3,3,14,11,14,5,
  5,11,16,13,14,10,3,15,3,5,15,5,15,16,5,5,15,15,11,3,3,
  13,10,13,5,5,5,3,13,13,10,15,3,3,3,5,3,3,3,3,15,15,16,
  3,3,4,14,11,5,5,11,16,16,15,11,3,15,15,15,15,15,15,15,
  15,16,15,3,3,16,15,15,15,15,16,16,15,16,15,3,3,3,3,3,3,
  3,3,4,3,3,3,4,4,4,3,3,3,7,4,2,4,3,15,5,5,16,15,11,
  15,15,3,3,14,14,11,5,5,5,3,11,14,14,3,15,3,3,3,14,15,
  3,3,3,15,16,11,5,5,11,4,12,15,12,5,5,16,16,12,14,16,3,
  3,14,14,11,5,5,11,15,10,14,14,15,5,5,16,3,12,3,3,14,11,
  14,15,5,15,3,15,14,11,5,5,15,3,12,3,3,14,11,14,12,5,5,
  3,15,16,11,15,12,3,16,3,3,16,15,11,5,12,5,3,11,14,14,
  11,3,3,3,3,13,15,11,5,15,15,3,11,14,15,15,15,3,4,14,14,
  3,15,15,15,16,16,12,14,15,3,3,4,12,12,15,15,15,16,14,16,
  14,3,3,14,3,14,3,3,3,16,12,14,16,3,4,3,4,3,3,3,3,4,4,
  3,4,4,3,3,3,3,3,3,4,4,12,12,11,11,12,3,11,14,11,13,11,
  13,11,16,11,11,14,12,11,11,16,14,11,11,5,5,15,12,11,11,
  5,15,11,11,11,15,11,13,11,16,16,15,12,13,12,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,15,15,5,5,5,5,5,5,15,15,5,15,5,5,
  15,5,15,5,15,5,15,5,15,15,15,15,15,11,11,11,11,5,5,11,
  12,5,12,5,11,5,5,5,5,5,5,5,5,5,5,5,5,5,15,15,5,5,5,
  5,5,5,15,15,5,15,5,5,15,5,15,5,15,5,15,5,15,15,15,15,
  15,11,11,11,11,5,5,11,12,5,12,5,11,5,5,5,5,5,5,5,5,5,
  5,5,5,5,15,5,5,5,15,5,15,5,15,15,15,5,5,5,5,15,5,5,5,
  5,15,5,15,15,15,15,10,11,11,10,10,10,11,5,10,5,10,11,5,
  5,5,5,5,5,5,5,5,5,5,5,15,5,5,5,5,5,15,15,15,15,15,5,
  5,5,5,15,5,5,5,5,5,5,15,15,5,15,11,11,11,11,5,5,11,5,
  5,5,5,11,5,5,5,5,5,5,5,5,5,5,5,15,5,5,5,5,5,15,15,
  15,15,15,5,5,5,5,15,5,5,5,5,5,5,15,15,5,15,11,11,11,
  11,5,5,11,5,5,5,5,11,5,5,5,5,5,5,5,5,5,5,15,5,5,5,5,
  5,5,5,15,15,15,5,5,15,5,15,5,15,5,15,5,5,15,15,15,15,
  10,5,5,10,5,5,5,10,5,10,5,5,5,5,5,5,5,5,5,5,5,15,5,
  5,5,15,5,5,5,15,5,15,5,5,5,5,5,5,15,5,14,15,5,15,15,
  5,15,10,5,5,10,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,15,
  5,5,5,15,5,5,5,15,15,15,5,5,5,5,5,5,5,5,5,15,15,15,
  15,15,15,11,5,5,11,12,12,5,5,12,5,12,5,5,5,5,5,5,5,15,
  15,5,5,5,15,5,5,5,15,15,15,5,5,5,5,5,5,5,5,5,15,15,
  15,15,15,15,11,5,5,11,12,12,5,5,12,5,12,5,5,5,5,5,5,5,
  15,5,5,5,5,5,5,5,15,5,1,5,5,5,5,5,5,5,5,5,5,15,15,
  15,5,1,13,5,5,13,5,5,12,5,5,5,5,11,5,5,5,5,15,15,5,5,
  5,5,5,5,15,15,5,1,5,5,5,5,5,5,5,5,5,5,15,15,15,3,15,
  13,5,12,11,5,5,11,5,5,5,11,5,5,5,5,16,15,5,5,5,5,5,5,
  15,15,5,15,5,5,5,5,5,5,5,5,5,5,15,16,15,15,15,11,5,11,
  11,5,5,11,5,5,5,5,5,5,5,15,15,5,5,5,5,5,5,5,15,3,15,
  5,5,5,5,5,5,5,5,5,15,5,15,15,3,15,11,12,11,10,5,11,5,
  5,5,5,12,5,5,15,16,5,5,5,5,5,16,5,15,15,15,5,5,5,5,5,
  5,5,5,5,5,5,15,15,15,15,11,11,11,11,5,5,5,5,5,5,11,11,
  5,15,5,5,5,5,5,15,5,15,3,15,5,5,5,5,5,5,5,5,5,15,5,
  15,15,5,3,11,13,5,13,5,13,5,5,5,5,5,12,15,5,5,5,5,5,
  5,5,15,4,15,5,5,5,5,5,5,5,5,5,1,15,1,15,16,15,11,15,
  16,12,5,13,15,5,5,5,14,5,15,15,15,15,15,1,15,1,15,16,
  15,15,15,15,15,15,15,15,15,15,15,15,16,15,1,16,15,15,16,
  15,16,16,15,15,16,15,15,5,5,15,5,15,5,15,16,15,5,5,5,5,
  5,5,5,5,15,15,5,15,1,5,15,10,11,5,10,11,12,5,5,5,11,5,
  5,5,5,5,5,16,15,15,15,5,5,5,5,5,5,5,5,16,5,5,15,15,
  15,15,11,11,5,10,5,11,11,5,5,12,12,5,15,5,5,15,15,5,3,
  5,5,5,5,5,5,5,5,5,5,5,15,3,5,15,13,5,5,11,13,5,12,5,
  5,13,5,5,16,15,5,1,16,15,15,5,15,5,5,5,15,5,5,5,5,15,
  16,5,15,11,5,5,12,16,16,5,5,5,14,5,5,5,5,16,5,15,5,5,
  5,5,15,5,5,5,15,5,5,15,15,5,15,12,5,5,11,11,5,5,11,12,
  5,5,5,5,15,16,15,5,5,5,5,5,5,5,5,15,5,5,15,3,5,15,12,
  14,5,12,16,16,5,5,5,12,12,15,15,13,4,5,5,5,5,5,5,5,5,
  1,5,1,15,15,15,15,12,11,16,12,5,5,16,5,5,5,14,5,15,1,
  15,15,15,15,15,15,15,15,15,15,15,15,16,15,16,16,15,15,16,
  15,16,15,15,15,16,15,16,15,5,15,5,5,15,5,15,5,5,16,5,
  16,1,4,1,13,4,14,14,16,15,5,5,13,16,16,14,15,15,16,15,
  15,15,15,16,15,15,16,15,1,1,1,1,15,15,16,15,15,4,15,15,
  16,16,15,5,15,5,16,5,15,5,15,5,5,15,16,5,15,10,5,5,11,
  12,5,5,11,11,11,5,5,5,5,15,5,16,5,5,16,5,15,15,5,15,
  10,5,11,11,11,12,5,12,5,11,5,5,5,16,16,5,15,15,5,1,15,
  15,5,15,11,5,5,14,12,5,5,16,12,16,5,5,1,5,5,5,5,5,15,
  15,15,3,15,11,5,12,11,5,5,5,13,13,5,12,5,15,16,4,15,15,
  5,4,16,3,1,13,5,16,14,16,5,5,16,4,16,16,16,15,5,5,5,
  16,15,15,15,15,11,5,5,11,5,5,5,5,11,11,11,5,16,5,16,15,
  1,16,15,3,11,5,5,16,5,5,5,16,14,5,5,5,15,5,5,16,15,5,
  16,15,5,5,15,12,5,5,14,16,5,5,14,15,5,15,16,5,15,11,11,
  5,11,14,5,5,16,5,16,5,5,5,15,15,3,15,12,12,5,11,5,16,
  5,5,5,5,16,5,15,15,1,15,14,5,16,11,5,5,16,15,15,5,13,
  15,15,15,16,15,15,15,16,15,15,15,16,16,15,16,16,15,15,16,
  15,15,16,4,16,15,16,16,16,15,15,15,16,14,4,11,5,14,16,
  14,14,5,4,5,16,15,15,1,15,16,15,15,16,15,15,16,11,13,15,
  11,11,11,11,11,12,12,12,11,12,11,5,5,5,5,12,12,11,11,5,
  11,11,11,12,5,5,5,11,11,13,12,11,11,11,11,12,12,10,12,
  11,5,5,5,12,5,10,12,12,5,5,11,11,11,11,11,11,11,11,14,
  12,12,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,15,5,5,5,5,5,
  5,15,15,5,15,5,5,15,5,15,5,15,5,15,5,15,15,15,15,15,11,
  11,11,11,5,5,11,12,5,12,5,11,5,5,5,5,5,5,5,5,5,5,5,5,
  15,5,5,5,5,15,5,15,15,5,15,5,5,5,5,5,15,5,5,5,5,15,
  15,5,5,15,15,5,11,11,5,11,11,11,5,11,5,11,11,5,5,5,5,
  5,5,5,5,5,5,5,15,5,5,5,5,15,5,15,15,5,15,5,5,5,5,5,
  15,5,5,5,5,15,15,5,5,15,15,5,11,11,5,11,11,11,5,11,5,
  11,11,5,5,5,5,5,5,5,5,5,5,14,5,5,5,5,5,5,5,15,5,15,
  15,5,5,5,5,15,5,5,5,5,5,15,5,15,15,15,5,10,10,5,5,5,
  10,5,5,5,5,10,5,5,5,5,5,5,5,5,5,14,5,5,5,5,5,5,5,15,
  5,15,15,5,5,15,5,15,5,15,5,15,5,15,5,15,15,15,5,10,10,
  5,11,11,10,10,11,10,11,10,5,5,5,5,5,5,5,5,15,5,5,5,5,
  15,5,5,15,5,15,15,5,5,5,5,15,5,15,5,15,15,15,5,15,15,
  5,5,11,11,5,11,11,11,5,11,5,11,11,5,5,5,5,5,5,5,15,5,
  5,5,5,15,5,5,15,5,15,15,5,5,5,5,15,5,15,5,15,15,15,5,
  15,15,5,5,11,11,5,11,11,11,5,11,5,11,11,5,5,5,5,5,5,
  15,5,5,5,5,15,5,5,15,5,5,5,5,5,5,5,15,5,5,5,5,15,15,
  5,5,15,5,5,10,10,5,5,5,10,5,5,5,5,10,5,5,5,5,5,15,15,
  5,5,5,5,5,5,15,15,5,3,5,5,5,5,15,5,5,5,5,5,15,15,5,
  15,3,11,10,11,11,5,5,13,5,5,5,5,13,5,5,5,5,15,15,5,5,
  5,5,5,5,15,5,5,1,5,5,5,5,15,5,5,5,5,5,15,5,5,1,5,11,
  11,13,5,5,5,13,5,5,5,12,11,5,5,5,16,15,5,5,5,5,5,5,
  15,15,5,15,5,5,5,5,15,5,5,5,5,5,15,16,5,15,15,11,11,
  11,11,5,5,11,5,11,5,11,11,5,5,15,5,5,5,5,5,5,15,15,5,
  1,5,5,5,5,5,15,5,5,5,5,15,15,5,5,1,5,5,13,13,5,5,12,
  11,5,5,5,11,10,5,15,15,5,5,5,5,5,16,15,5,15,15,5,5,5,
  5,15,5,5,5,5,15,15,5,5,15,5,11,11,11,5,5,11,11,5,5,5,
  11,11,15,5,5,5,5,5,5,5,15,5,1,5,5,5,5,5,15,5,5,5,5,
  5,15,5,5,15,3,5,14,11,11,5,13,11,5,5,5,5,13,15,15,15,
  15,15,15,15,15,15,3,15,15,15,15,15,16,15,15,15,15,1,15,
  1,15,16,15,15,15,16,15,15,15,15,16,15,15,16,15,5,5,5,5,
  5,1,15,1,5,14,5,5,5,5,15,5,5,5,5,5,15,15,5,15,1,15,
  11,11,16,5,5,14,5,5,5,5,11,5,5,15,5,15,15,5,3,5,5,5,
  5,5,15,5,5,5,5,5,15,5,3,15,5,5,13,11,5,12,11,10,5,5,
  11,5,10,5,15,5,15,16,5,15,5,5,5,5,5,15,5,5,5,15,5,15,
  5,15,15,5,5,11,11,5,11,11,11,5,5,5,5,10,5,5,5,15,5,5,
  16,5,5,5,5,15,5,5,5,15,5,15,5,3,15,5,11,11,11,5,13,5,
  13,5,5,13,5,11,16,15,15,1,4,5,5,5,5,5,15,5,5,15,5,5,
  15,5,16,15,5,5,12,12,5,16,16,13,5,11,14,5,13,5,15,16,
  15,15,5,5,5,5,15,5,5,5,5,5,15,5,5,15,5,5,11,10,5,5,5,
  12,12,5,11,5,11,15,5,14,5,5,5,5,5,15,5,5,5,5,5,15,5,
  4,15,5,5,14,11,5,16,15,12,5,5,13,15,11,15,15,3,15,15,
  15,15,16,15,15,15,1,15,1,15,15,16,14,15,15,16,15,15,15,
  15,15,15,16,16,15,5,4,5,5,15,5,16,15,5,15,15,15,15,5,
  16,15,15,16,12,11,15,5,5,11,15,11,5,5,14,15,5,15,15,5,
  15,5,15,5,13,16,15,15,1,1,3,5,1,14,14,16,4,14,5,5,16,
  16,15,5,5,16,5,15,15,5,15,5,5,16,15,3,1,4,4,13,14,16,
  14,5,15,13,5,16,14,16,5,15,5,16,5,5,5,15,5,15,5,15,15,
  5,5,11,10,5,11,5,10,11,12,12,5,11,5,5,15,5,16,5,5,16,
  15,5,15,15,5,5,11,11,11,11,5,10,5,11,11,11,11,15,16,16,
  5,15,15,5,1,15,15,15,5,5,12,11,10,5,5,12,16,12,16,5,12,
  3,5,15,5,5,15,15,15,5,3,5,5,11,13,5,5,5,11,13,13,5,12,
  11,15,16,1,15,16,15,4,16,1,1,15,16,16,16,16,15,16,16,1,
  16,16,16,5,5,5,5,16,5,5,15,5,5,10,12,5,5,5,11,11,5,5,
  5,11,16,5,1,15,1,16,15,3,5,12,12,16,5,5,11,16,14,12,15,
  12,5,5,15,16,5,15,16,5,11,11,14,5,5,15,16,14,12,5,15,5,
  15,5,16,15,5,5,14,12,5,14,5,12,16,12,16,5,11,15,5,13,4,
  5,5,14,12,5,11,15,12,5,15,5,16,13,15,15,1,15,16,15,16,
  15,15,15,16,15,15,16,15,15,5,15,16,15,11,11,16,5,5,11,
  16,5,5,5,14,15,5,16,14,13,5,4,14,14,16,14,16,5,13,15,
  16,15,1,15,15,16,16,15,15,15,4,15,15,14,11,4,5,14,16,14,
  14,5,5,16,11,13,11,12,5,11,5,5,11,12,13,15,11,13,11,15,
  11,11,12,12,15,11,11,13,15,11,12,11,11,15,5,12,13,11,11,
  5,5,11,11,12,11,11,5,5,10,11,5,5,11,11,12,11,10,11,13,
  15,5,12,12,11,12,11,14,5,11,11,5,5,5,5,5,5,5,5,5,5,5,
  5,15,15,5,5,5,15,5,15,5,5,15,15,5,5,5,5,5,5,5,5,5,15,
  15,15,15,15,15,5,11,11,5,12,12,11,5,12,5,12,11,5,5,5,5,
  5,5,5,5,5,5,5,15,15,5,5,5,5,5,15,14,5,5,5,5,5,5,5,5,
  5,5,5,5,5,15,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,15,15,5,5,5,15,5,15,5,5,15,15,5,5,5,
  5,5,5,5,5,5,15,15,15,15,15,15,5,11,11,5,12,12,11,5,12,
  5,12,11,5,5,5,5,5,5,5,5,5,15,15,5,5,5,5,5,5,5,5,5,
  15,5,5,15,5,15,5,15,5,15,5,15,15,15,5,5,5,5,5,5,5,5,
  5,10,5,10,5,5,5,5,5,5,5,5,5,5,15,15,5,5,5,15,5,15,15,
  5,5,5,5,5,5,5,5,5,15,5,15,15,15,15,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,15,5,5,5,5,5,5,5,5,
  5,15,5,5,5,5,5,5,5,5,5,5,15,15,15,15,5,5,5,5,5,10,10,
  5,5,10,5,10,5,5,5,5,5,5,5,15,15,5,5,5,15,5,15,15,5,5,
  5,5,5,5,5,5,5,15,5,15,15,15,15,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,15,16,5,5,5,5,5,5,15,16,5,16,5,
  5,5,5,5,5,15,5,5,5,15,15,15,15,16,13,5,10,15,5,5,15,5,
  5,5,5,14,5,5,5,5,15,15,5,5,5,5,5,5,16,15,5,16,5,5,5,
  5,5,5,5,5,5,5,16,15,5,16,15,15,10,15,10,5,5,14,5,5,5,
  14,5,5,5,5,15,15,5,5,5,5,5,5,15,15,5,5,5,5,5,5,5,5,
  5,5,5,5,15,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,16,
  15,5,5,5,5,5,5,5,5,16,5,5,5,5,5,15,5,5,5,5,16,15,15,
  5,16,5,10,15,14,5,5,14,5,5,5,5,13,10,5,15,15,5,5,5,5,
  5,15,5,5,5,5,5,5,5,5,5,5,5,5,5,15,15,15,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,15,15,5,5,5,5,5,5,5,5,16,5,5,5,5,
  5,5,5,5,5,5,15,15,15,5,5,16,5,13,5,14,11,15,5,11,5,11,
  10,13,15,15,15,15,15,15,15,15,15,16,15,15,15,15,15,15,15,
  15,15,15,1,15,1,15,16,15,15,16,16,15,15,16,14,16,15,15,
  15,15,15,15,15,15,15,1,15,1,15,16,15,15,15,15,15,15,15,
  15,15,15,15,15,15,15,16,16,15,15,15,16,15,15,15,15,16,15,
  16,5,5,16,5,15,5,5,16,15,5,5,15,5,5,5,5,5,5,5,15,15,
  16,5,5,5,14,5,10,14,13,10,5,5,13,5,5,5,15,5,5,15,5,5,
  5,5,5,5,5,5,5,5,5,15,5,15,15,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,15,5,15,5,15,15,15,5,5,5,5,15,5,5,5,15,5,15,
  15,15,5,15,14,5,5,5,15,10,13,11,11,14,5,5,16,1,5,1,16,
  15,5,5,5,5,15,5,5,5,5,15,15,15,14,5,15,5,5,5,5,14,15,
  15,5,5,12,5,5,5,5,16,5,5,5,5,5,5,5,5,5,5,15,5,15,15,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,14,11,5,5,5,5,5,
  5,15,5,5,5,15,15,16,15,5,5,13,5,5,15,12,11,12,5,5,5,5,
  15,5,16,5,5,5,5,5,5,5,5,1,15,1,15,13,16,11,15,5,12,10,
  5,5,15,5,5,12,12,5,15,16,5,5,5,5,15,5,5,5,5,5,15,15,
  5,11,16,15,5,10,12,11,5,15,5,5,12,5,14,5,5,5,5,15,5,5,
  5,5,15,15,15,15,3,3,3,5,16,14,14,14,16,15,5,5,15,14,14,
  5,5,15,15,5,5,12,15,15,5,16,15,3,1,1,16,5,14,14,16,15,
  16,15,15,16,16,14,5,16,5,16,5,5,5,15,5,15,15,16,5,5,10,
  5,10,5,13,5,5,13,14,11,5,10,5,5,5,5,15,5,5,15,15,15,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,14,16,15,5,16,5,1,15,
  16,15,5,5,5,5,5,5,11,11,12,11,16,5,5,16,5,15,5,5,5,15,
  15,5,16,5,5,5,13,5,10,5,5,14,15,5,11,5,5,14,16,15,5,
  16,16,16,1,1,5,5,14,14,14,5,5,16,16,16,14,14,15,5,5,15,
  16,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,16,5,15,15,1,5,
  5,16,5,5,5,15,5,5,12,12,12,5,5,5,5,5,15,16,5,5,16,5,
  5,5,13,11,11,5,15,13,11,5,11,5,15,15,14,5,12,5,5,5,5,
  12,5,5,16,5,12,5,12,15,15,5,16,5,5,12,5,5,5,14,5,5,11,
  5,15,15,15,16,16,15,16,15,16,15,15,15,16,16,16,16,16,15,
  15,15,16,15,15,15,16,15,16,15,16,15,15,15,16,5,5,14,14,
  5,5,15,16,14,16,16,16,15,5,5,14,14,16,15,5,16,14,16,16,
  15,16,5,14,14,15,16,15,16,15,16,16,15,5,16,5,12,11,11,5,
  5,5,5,12,11,11,11,11,11,5,5,5,5,11,11,11,5,5,11,11,12,
  11,5,5,5,5,11,12,11,11,5,5,5,12,11,11,12,5,5,5,5,5,5,
  11,12,11,5,5,12,11,11,5,15,12,11,12,11,11,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,15,5,5,5,5,15,5,15,15,5,15,5,5,5,5,
  5,15,5,5,5,5,15,15,5,5,15,15,5,11,11,5,11,11,11,5,11,
  5,11,11,5,5,5,5,5,5,5,5,5,5,15,15,5,5,5,15,5,15,5,5,
  15,15,5,5,5,5,5,5,5,5,5,15,15,15,15,15,15,5,11,11,5,
  12,12,11,5,12,5,12,11,5,5,5,5,5,5,5,5,5,5,15,5,5,5,
  15,5,15,5,15,15,15,5,5,15,5,15,5,15,5,15,15,5,15,15,5,
  15,12,5,5,12,11,11,5,11,11,11,11,5,5,5,5,5,5,5,5,5,5,
  15,5,5,5,15,5,15,5,15,15,15,5,5,15,5,15,5,15,5,15,15,
  5,15,15,5,15,12,5,5,12,11,11,5,11,11,11,11,5,5,5,5,5,
  5,5,5,5,5,5,5,5,15,5,15,5,5,15,15,5,5,5,5,15,5,5,5,
  5,15,5,5,15,15,15,5,5,5,5,10,10,5,5,10,5,10,5,5,5,5,
  5,5,5,15,5,5,5,5,15,5,15,5,5,15,5,5,5,5,5,5,5,5,5,5,
  15,15,5,5,5,15,5,5,5,5,10,10,5,5,10,5,10,5,5,5,5,5,5,
  5,15,5,5,5,15,5,15,5,15,15,1,5,5,5,5,5,5,5,5,5,15,5,
  5,5,5,1,11,5,5,11,10,10,11,5,11,5,11,11,5,5,5,5,5,5,
  5,5,5,15,5,15,15,5,15,3,5,5,5,5,5,5,5,5,5,15,15,5,5,
  3,15,11,5,11,5,10,11,11,5,10,5,13,5,5,5,5,15,5,5,5,5,
  15,5,15,5,5,15,15,5,5,5,5,5,5,5,5,5,15,5,16,5,15,15,
  11,11,5,5,10,11,11,11,11,5,11,11,5,5,15,5,5,5,5,15,5,
  15,15,5,1,5,5,5,5,5,5,5,5,5,5,15,5,5,5,1,15,5,11,11,
  5,11,13,5,5,10,5,13,5,5,5,16,5,5,5,15,5,15,15,5,15,15,
  5,5,5,5,5,5,5,5,5,15,15,5,5,15,15,5,5,11,5,11,11,11,
  5,10,5,11,5,15,5,5,5,5,15,5,15,5,15,3,5,5,5,5,5,5,5,
  15,5,5,15,5,15,5,5,3,5,11,5,12,11,14,5,5,11,5,11,11,5,
  5,15,5,15,5,15,5,5,4,14,5,5,15,5,5,5,5,5,5,1,5,1,5,
  16,16,5,16,15,5,11,11,12,5,12,5,14,15,5,5,15,15,5,1,5,
  1,15,16,5,5,15,5,5,5,5,5,5,15,5,5,5,13,1,16,5,5,16,
  11,11,14,5,12,5,14,13,5,5,15,5,15,5,5,16,5,5,5,5,5,5,
  5,5,5,15,15,5,5,1,5,15,5,11,5,5,13,13,5,5,11,12,11,5,
  5,15,5,15,16,5,15,5,5,5,5,5,5,5,5,5,16,15,5,5,15,15,
  15,5,11,11,5,11,11,5,5,11,11,11,5,15,5,15,5,5,15,16,5,
  5,5,5,5,5,5,5,5,15,5,5,1,5,15,12,5,5,5,14,11,11,5,11,
  13,11,5,16,15,15,1,16,15,15,15,15,15,15,15,15,15,15,15,
  15,15,16,15,16,16,15,15,15,16,16,15,15,15,16,15,15,15,5,
  16,15,5,5,5,15,5,5,5,5,5,5,15,5,5,15,5,15,5,5,5,11,
  11,11,5,5,11,5,10,5,15,15,16,15,15,15,15,15,15,15,15,15,
  15,15,15,15,3,15,16,16,16,15,15,16,16,15,15,15,15,15,15,
  5,15,4,5,5,5,5,5,5,5,5,1,15,1,5,5,14,15,13,15,15,5,
  11,11,16,5,11,5,14,5,15,3,5,5,15,5,5,5,15,15,5,15,5,5,
  16,15,16,15,5,5,16,13,11,5,11,12,5,11,13,15,15,15,15,15,
  15,15,15,15,15,15,15,15,1,1,1,15,4,15,15,16,16,15,15,16,
  16,16,15,5,5,16,5,5,5,5,5,5,15,16,16,3,1,1,1,5,15,16,
  14,13,16,5,13,16,14,14,5,15,5,16,5,15,5,15,15,5,5,3,5,
  15,5,5,5,5,13,11,5,12,13,11,10,5,15,5,15,5,15,5,15,16,
  5,5,15,5,15,5,11,5,5,11,11,5,11,11,11,11,11,5,15,16,5,
  15,15,15,1,5,4,5,16,5,5,5,15,12,12,5,16,14,16,13,11,3,
  5,5,5,5,15,5,5,5,16,15,5,5,11,5,11,11,5,13,14,5,13,5,
  5,16,1,15,15,5,4,4,3,1,5,5,14,14,16,11,5,15,1,16,14,
  14,5,5,5,15,16,5,15,15,15,11,5,5,11,11,10,5,5,11,5,12,
  5,16,15,15,5,1,15,15,4,5,5,5,16,11,12,5,16,14,15,11,5,
  5,15,15,15,15,5,16,12,5,5,16,15,11,5,14,15,5,11,14,15,
  5,5,14,5,15,5,5,5,5,14,11,5,16,12,15,12,5,15,15,15,3,
  16,15,15,15,16,15,16,15,15,15,15,15,15,5,5,4,15,13,5,16,
  5,12,11,16,5,13,5,11,12,5,15,16,13,15,5,16,12,11,5,16,
  12,5,12,14,15,15,16,14,5,14,1,14,14,16,16,4,14,5,15,16,
  16,4,5,13,14,16,14,15,5,15,5,15,15,15,1,16,16,16,16,16,
  15,16,16,5,11,11,12,11,5,5,11,11,12,13,11,12,13,11,5,5,
  11,12,13,11,5,11,13,11,12,11,5,11,5,11,12,11,12,11,5,11,
  5,15,11,13,15,11,15,10,11,11,15,13,15,11,5,10,12,13,11,
  11,11,12,11,12,15,12,11,5,11,5,5,5,5,5,5,5,5,5,5,5,15,
  5,5,5,5,5,15,15,15,15,15,5,5,5,5,15,5,5,5,5,5,5,15,
  15,5,15,11,11,11,11,5,5,11,5,5,5,5,11,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,15,15,5,5,15,5,5,15,5,15,5,15,5,15,
  5,5,5,15,15,15,5,5,5,5,11,11,5,12,11,12,11,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,15,5,15,15,5,5,5,5,5,5,5,5,5,15,5,
  15,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,15,15,5,5,15,5,5,15,5,15,5,15,5,15,
  5,5,5,15,15,15,5,5,5,5,11,11,5,12,11,12,11,5,5,5,5,5,
  5,5,14,5,5,5,5,15,5,15,15,5,5,5,5,5,5,5,5,5,5,5,5,
  15,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  15,5,5,5,5,5,15,15,16,14,16,5,5,5,5,15,5,5,5,5,5,5,5,
  5,5,16,11,10,5,13,10,5,13,5,5,5,10,15,5,5,5,5,5,5,5,
  5,5,5,5,15,15,5,5,16,5,5,5,5,5,5,5,5,5,15,16,5,5,16,
  5,13,5,13,5,5,10,15,5,10,5,15,10,5,5,5,15,5,5,5,5,5,
  5,15,15,15,5,5,5,5,5,5,5,5,5,5,5,5,5,15,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,15,5,5,5,5,15,5,15,15,5,16,5,
  5,5,5,5,5,5,5,5,5,15,15,5,5,16,5,5,13,15,5,10,15,10,
  5,5,5,14,5,5,5,15,5,5,5,5,5,15,15,5,5,5,5,5,5,5,5,5,
  5,5,5,15,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,5,
  5,5,15,5,15,15,15,16,15,5,5,5,5,15,5,5,5,15,5,5,5,5,
  5,16,10,11,5,15,5,13,5,5,5,11,5,14,5,15,15,5,15,5,15,
  15,5,16,5,5,5,5,5,5,5,5,5,15,1,5,1,5,15,13,5,15,11,5,
  10,15,5,5,5,5,14,5,5,5,5,5,5,1,15,1,11,13,5,5,5,15,5,
  5,5,15,5,15,5,5,5,15,15,11,11,5,15,5,5,12,11,5,5,5,15,
  5,5,16,5,16,15,5,16,5,5,5,5,5,5,5,5,5,5,5,5,5,15,5,
  5,5,15,10,5,15,14,5,5,10,14,5,10,5,15,5,14,15,5,5,5,5,
  5,5,5,5,5,5,5,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,15,15,5,5,16,5,5,5,5,5,5,15,5,5,5,5,15,16,5,5,
  15,5,5,11,13,5,14,11,5,15,5,5,16,15,15,1,15,15,5,5,15,
  5,15,5,5,15,5,5,5,15,16,5,5,5,15,10,11,12,15,5,5,5,12,
  5,5,15,15,16,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,15,15,16,15,14,15,15,15,15,15,15,
  15,15,15,15,15,16,15,15,15,15,15,16,15,15,15,15,15,16,15,
  15,15,15,16,15,15,15,15,15,15,15,15,1,15,1,15,15,16,15,
  16,15,16,15,15,15,16,16,15,15,15,14,15,16,5,5,5,5,5,5,
  5,5,5,5,15,5,12,5,14,15,15,5,12,5,5,5,5,14,5,5,12,5,
  5,5,5,5,5,5,5,15,15,15,5,5,3,1,1,15,15,14,16,14,16,5,
  5,5,14,14,14,5,5,12,5,5,5,5,5,5,5,16,5,1,3,3,16,15,
  14,14,14,5,16,5,15,14,14,14,5,16,5,16,5,5,5,16,5,5,5,
  15,15,15,5,10,5,5,14,5,10,14,15,13,10,5,15,5,5,5,15,5,
  5,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,14,15,5,5,
  1,5,1,5,16,15,5,5,5,5,5,5,5,5,14,12,15,15,15,16,5,5,
  5,15,5,15,5,15,16,15,11,5,14,5,5,5,5,15,13,10,13,5,5,
  14,15,15,13,5,16,3,1,3,5,5,14,16,14,5,5,16,16,16,14,14,
  5,5,5,5,16,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,16,5,15,
  15,1,5,5,16,11,5,5,15,5,11,5,12,13,5,5,5,15,5,15,16,5,
  15,16,11,5,5,14,5,5,5,13,11,11,5,13,5,5,5,14,11,5,5,5,
  5,11,13,5,5,15,15,12,11,5,15,5,5,16,5,5,14,15,5,5,12,
  10,5,5,5,15,5,5,15,15,15,13,5,15,5,5,10,12,11,5,5,15,
  5,5,5,14,5,5,11,11,5,5,5,15,5,11,5,12,5,5,14,14,5,5,
  16,14,14,14,14,16,15,15,5,14,14,16,5,5,14,14,16,14,15,
  16,5,16,14,5,16,5,14,14,16,14,5,5,16,5,10,12,11,5,5,5,
  5,11,11,11,11,11,12,5,5,5,5,11,11,11,5,5,12,11,11,11,5,
  5,5,5,11,11,12,11,5,5,5,11,11,11,11,5,5,5,5,5,5,11,11,
  11,5,5,11,12,11,5,12,11,11,11,11,11,5,5,5,5,5,5,5,5,5,
  5,5,5,15,5,5,5,5,15,5,5,5,5,15,15,5,5,15,5,15,5,15,5,
  15,15,15,5,15,15,5,5,13,13,5,5,5,13,11,5,11,5,13,5,5,
  5,5,5,5,5,5,5,5,5,5,5,15,5,5,5,5,15,15,5,5,5,5,5,5,
  15,5,15,15,5,5,15,5,5,5,10,10,5,5,5,10,5,5,5,5,10,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,15,5,5,5,5,15,5,
  5,5,5,5,5,5,15,15,15,5,10,10,5,10,10,10,5,10,5,10,10,
  5,5,5,5,5,5,15,5,5,5,5,15,5,5,5,5,15,15,5,5,15,5,15,
  5,15,5,15,15,15,5,15,15,5,5,13,13,5,5,5,13,11,5,11,5,
  13,5,5,5,5,5,5,16,5,5,5,5,5,15,5,15,15,3,5,5,5,5,5,
  5,5,5,5,5,5,5,15,5,3,13,11,10,11,5,5,14,5,5,5,5,13,5,
  5,5,5,5,5,5,5,5,5,5,5,16,5,15,3,5,5,5,5,5,5,5,5,5,
  5,16,5,15,1,5,11,10,14,5,5,5,13,5,5,5,12,10,5,5,5,16,
  5,5,5,5,5,5,5,5,5,15,15,5,5,5,5,15,5,5,5,5,15,5,16,
  15,5,5,5,12,12,12,5,11,10,5,5,5,13,13,5,5,16,5,5,5,5,
  5,5,5,5,5,3,15,5,5,5,5,5,5,5,5,5,16,5,15,15,3,5,5,
  14,13,5,5,12,10,5,5,5,12,11,5,5,16,5,5,5,15,5,16,5,15,
  15,15,5,5,5,5,5,5,5,5,5,5,5,5,15,5,15,12,12,10,12,11,
  13,13,5,5,5,5,11,5,5,5,5,5,5,5,5,5,5,1,15,5,5,5,5,5,
  5,5,5,5,5,5,5,14,5,4,5,15,10,12,5,12,10,5,5,5,5,15,5,
  5,5,5,5,5,5,5,5,3,15,5,15,5,5,14,5,5,15,5,15,5,16,15,
  16,5,5,16,14,5,5,15,11,11,12,5,16,16,15,5,5,5,15,16,5,
  16,15,15,5,5,5,5,5,5,5,5,5,5,5,5,15,5,1,14,12,11,16,
  11,15,13,5,5,5,5,12,5,5,16,5,16,5,5,16,15,5,5,5,5,5,
  5,5,5,5,5,5,5,3,5,5,5,13,10,5,12,12,11,5,5,13,5,11,5,
  5,5,5,16,5,15,15,5,5,5,5,15,5,5,5,16,5,15,5,15,15,5,
  11,10,12,5,12,5,11,5,11,5,5,11,5,5,5,5,5,15,16,5,5,5,
  5,5,5,5,5,5,5,5,5,1,5,5,12,10,10,5,12,5,15,5,5,14,5,
  10,16,16,5,16,15,16,5,5,5,5,5,5,5,5,5,5,5,5,16,5,15,
  11,14,11,5,12,16,12,5,5,13,5,14,15,5,16,15,15,5,5,5,5,
  5,5,5,5,5,5,5,5,15,5,15,11,11,10,11,11,12,11,5,5,11,5,
  11,5,5,16,15,5,5,5,5,5,5,5,5,5,5,5,5,3,5,14,15,14,10,
  11,15,13,12,5,5,15,5,11,5,15,3,15,5,5,5,15,5,5,5,16,5,
  15,5,16,16,5,15,11,12,5,5,5,16,5,5,5,14,11,16,1,5,5,5,
  5,5,5,5,5,5,5,5,5,16,5,15,16,12,11,13,5,12,14,5,5,5,
  5,14,15,15,15,15,15,15,15,15,15,15,15,15,15,1,1,1,15,1,
  16,16,15,16,15,15,15,15,15,16,15,15,16,15,15,15,15,15,15,
  15,16,15,1,1,1,4,15,16,15,16,15,16,15,15,16,15,16,5,16,
  5,15,5,5,5,16,5,15,5,3,5,5,5,10,11,5,12,5,11,13,12,12,
  5,10,5,5,15,5,16,5,5,15,5,5,15,15,15,5,12,11,5,5,5,11,
  5,12,5,5,12,5,16,16,5,5,16,5,16,5,1,5,5,16,11,12,5,5,
  5,14,15,14,16,11,11,1,5,5,5,5,5,5,5,15,16,5,5,10,15,5,
  5,5,10,14,12,5,12,10,15,16,3,16,5,16,3,1,1,3,14,14,16,
  16,16,5,14,16,4,14,15,15,5,5,15,5,16,5,15,5,5,11,10,11,
  5,12,5,11,11,11,11,5,10,16,5,15,5,16,15,14,3,5,14,13,
  15,5,5,11,12,13,5,5,16,5,15,5,15,15,15,16,5,14,14,14,5,
  11,11,13,14,5,12,16,5,5,5,16,15,5,13,11,14,5,13,5,13,
  16,5,12,5,11,5,5,15,3,5,5,12,14,11,5,12,11,15,5,5,16,
  12,5,16,3,5,13,11,16,5,12,5,14,15,5,11,15,11,15,5,16,5,
  11,12,14,5,11,11,16,14,5,15,13,15,15,16,16,16,15,1,15,
  16,15,16,4,15,16,5,16,15,1,5,14,16,14,16,16,5,16,13,14,
  16,11,4,5,15,14,14,14,5,5,16,12,12,12,11,11,12,5,11,12,
  12,12,15,13,13,12,15,10,12,11,12,15,12,12,13,15,12,12,11,
  12,15,11,11,13,11,11,5,11,12,12,12,12,12,5,5,11,13,5,5,
  12,12,13,11,11,12,13,15,5,12,12,12,12,12,12,5,11,13,5,5,
  5,5,5,5,5,5,5,15,5,5,5,15,5,15,5,15,15,15,5,5,15,5,
  15,5,15,5,15,15,5,15,15,5,15,12,5,5,12,11,11,5,11,11,
  11,11,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,15,5,5,15,5,
  5,15,5,15,5,15,5,15,5,5,5,15,15,15,5,5,5,5,11,11,5,12,
  11,12,11,5,5,5,5,5,5,5,15,5,5,5,5,15,5,5,5,5,15,15,5,
  5,15,5,15,5,15,5,15,15,15,5,15,15,5,5,13,13,5,5,5,13,
  11,5,11,5,13,5,5,5,5,5,5,15,5,5,5,5,5,5,5,15,5,1,5,
  5,15,5,15,5,15,5,15,5,15,15,15,5,1,11,5,5,12,5,5,11,
  11,5,10,5,11,5,5,5,5,15,15,5,5,5,5,5,5,15,5,5,1,5,5,
  15,5,15,5,15,5,15,5,15,5,15,1,5,12,5,11,5,5,5,11,10,5,
  10,11,5,5,5,5,16,5,5,5,5,5,5,5,5,5,5,15,5,5,15,5,15,
  5,15,5,15,5,5,16,15,15,15,11,5,12,12,5,11,11,11,5,10,5,
  5,5,5,15,5,5,5,5,5,5,5,5,5,3,15,5,5,15,5,15,5,15,5,
  15,15,5,5,15,3,5,5,11,11,5,5,11,5,10,5,11,11,5,5,5,16,
  5,5,5,5,5,15,5,5,15,15,5,5,15,5,15,5,15,5,15,5,5,5,
  15,15,5,12,12,11,5,11,5,5,10,5,11,11,5,5,5,5,5,5,15,5,
  5,5,5,3,15,5,5,15,5,15,5,15,5,15,5,5,5,15,5,1,5,13,5,
  12,5,11,5,11,5,11,5,12,15,5,5,5,5,5,5,5,5,3,15,5,5,
  15,15,15,5,15,5,15,1,5,1,15,16,15,5,16,16,5,5,5,5,11,
  5,11,13,5,5,5,5,5,5,1,15,1,5,16,5,5,15,5,15,15,15,15,
  15,5,5,5,15,5,3,16,5,5,16,5,5,14,12,5,11,10,5,5,5,15,
  5,15,5,5,16,15,5,5,15,5,15,5,15,5,15,15,5,5,1,5,5,5,
  11,5,5,11,11,5,11,5,13,5,5,5,5,5,5,16,5,15,15,5,5,15,
  5,15,5,15,5,16,15,5,5,15,15,5,5,11,5,5,5,11,5,11,11,
  11,11,5,5,5,15,5,5,5,16,5,5,15,5,15,5,15,5,15,5,15,5,
  3,5,5,12,5,5,5,11,5,12,11,5,14,5,5,15,15,5,1,16,15,15,
  15,15,5,15,5,15,5,15,15,5,15,16,5,5,5,13,5,15,16,16,5,
  11,15,14,5,15,5,5,15,15,15,5,5,15,5,15,5,15,5,15,5,5,
  15,15,5,15,11,11,5,5,5,5,5,11,5,11,5,11,5,5,15,15,5,
  15,15,5,15,5,15,5,15,5,5,5,4,14,5,5,13,5,5,16,16,5,11,
  5,11,12,5,5,5,3,5,5,15,15,15,5,15,5,1,5,1,5,15,16,5,
  12,13,16,5,5,5,16,11,15,11,14,5,5,1,5,5,15,5,15,5,15,
  15,15,5,15,5,16,5,16,16,5,5,16,5,5,12,14,5,12,5,12,15,
  5,15,15,5,15,5,15,5,15,16,5,15,1,4,3,5,4,14,14,15,15,
  5,11,5,16,14,14,15,15,16,15,15,15,15,15,16,15,16,16,1,1,
  1,1,15,16,16,15,15,16,16,15,16,16,16,5,15,5,16,5,15,5,
  15,5,5,5,16,5,5,5,5,5,5,11,5,5,13,11,13,5,5,15,5,15,
  5,16,5,15,15,5,5,15,5,5,5,5,5,5,11,11,5,11,5,11,5,5,
  15,16,15,15,15,15,15,1,15,16,15,15,16,15,16,15,15,15,15,
  16,15,16,15,15,3,5,15,5,15,5,5,5,15,16,5,5,5,12,5,5,5,
  5,14,11,11,11,5,15,16,1,16,15,15,4,15,1,1,15,15,16,15,
  16,15,15,16,1,16,16,16,15,5,15,5,16,5,15,5,15,5,5,5,12,
  5,5,11,11,5,11,5,11,16,15,16,15,1,15,16,4,15,16,15,16,
  15,15,15,16,16,15,15,16,15,5,5,16,16,5,3,5,5,5,16,5,5,
  12,15,14,15,5,16,15,15,15,16,15,15,16,15,16,15,15,15,15,
  16,15,16,15,15,5,5,15,4,5,5,14,13,5,15,15,5,11,5,11,16,
  5,15,15,3,5,14,5,16,5,5,5,16,13,5,11,5,5,15,15,16,5,5,
  5,16,5,5,5,15,5,12,5,14,15,15,16,15,15,15,4,15,15,16,
  15,16,15,15,15,16,16,1,5,5,14,16,14,14,13,15,13,16,16,5,
  1,5,16,16,14,16,13,5,4,12,10,12,11,5,11,11,5,11,11,12,
  11,11,11,5,5,11,5,12,12,13,12,5,11,11,12,12,11,5,5,5,
  11,11,12,11,11,5,11,11,12,13,11,11,5,5,5,11,5,13,11,12,
  11,5,12,11,11,11,15,11,11,13,11,12,11,11,5,5,5,5,5,5,5,
  5,15,5,5,5,5,15,5,5,15,5,15,15,5,5,5,5,15,5,15,5,15,
  15,15,5,15,15,5,5,11,11,5,11,11,11,5,11,5,11,11,5,5,5,
  5,5,5,15,15,5,5,5,15,5,15,15,5,5,5,5,5,5,5,5,5,15,5,
  15,15,15,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,15,5,5,5,15,5,5,5,16,5,16,5,5,5,5,5,5,15,5,15,15,5,
  5,5,5,16,13,5,5,11,5,10,11,5,10,10,5,13,5,5,5,5,5,5,
  5,5,5,15,5,5,15,15,5,16,5,5,5,5,5,5,15,5,15,15,16,5,
  15,16,5,11,5,11,5,10,5,13,10,5,5,13,5,5,5,5,15,5,5,5,
  5,15,5,5,5,5,5,5,5,5,5,5,5,5,15,5,15,15,5,15,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,16,5,5,5,5,15,5,5,5,5,
  16,5,5,5,5,5,5,5,15,5,15,16,5,5,5,16,15,5,11,13,5,5,
  13,5,5,10,10,15,5,5,5,15,5,5,5,15,5,15,5,5,5,5,5,5,5,
  5,5,5,15,5,15,15,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,15,5,5,5,5,16,5,5,5,5,5,5,5,15,5,15,15,5,
  15,15,15,15,5,13,5,13,5,11,10,5,5,5,5,15,5,5,15,5,15,
  15,15,5,5,16,5,5,5,5,15,5,5,15,5,15,1,15,1,5,15,5,5,
  15,12,5,5,5,5,5,15,5,12,11,5,5,15,15,5,1,5,1,5,14,5,
  5,5,5,12,15,15,5,15,15,5,5,15,15,15,12,5,5,16,5,5,13,
  5,11,5,5,5,5,5,16,5,15,5,5,16,5,5,5,15,5,15,5,15,5,
  15,15,5,5,16,5,5,5,13,5,5,13,15,5,10,5,15,10,5,5,15,5,
  5,15,5,5,5,5,5,5,5,5,5,15,5,15,15,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,15,5,5,15,5,5,16,5,5,15,5,15,5,15,5,
  15,15,15,5,16,15,15,13,5,10,11,11,5,15,5,5,13,5,11,16,
  16,15,1,15,16,15,15,15,15,15,15,15,15,15,15,15,15,16,15,
  16,16,16,15,16,16,16,15,15,15,15,15,16,5,5,16,5,5,5,5,
  5,5,5,5,15,5,15,15,5,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,16,13,15,15,15,5,11,5,15,5,15,15,5,5,16,5,5,12,
  13,5,5,15,12,5,10,5,15,5,5,5,15,16,5,5,5,5,5,5,15,5,
  1,15,1,15,11,13,5,5,5,11,12,11,5,15,5,5,5,12,5,15,15,
  5,5,5,5,5,5,15,5,15,15,5,15,15,5,14,16,5,11,12,11,5,5,
  5,5,5,5,12,5,15,5,11,5,5,5,15,5,15,16,5,5,1,3,1,15,
  16,16,16,14,16,5,15,5,14,14,16,15,5,5,5,5,5,15,5,15,15,
  14,5,3,3,1,16,15,16,16,14,5,16,5,5,14,14,16,5,16,5,16,
  5,15,5,16,15,5,15,16,5,5,5,5,5,5,15,10,5,15,13,14,5,5,
  15,5,5,5,15,5,15,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,16,15,15,5,15,15,1,5,15,15,15,12,5,5,5,15,10,11,12,
  14,15,5,5,16,5,15,5,15,15,5,5,5,16,5,11,10,15,11,5,5,
  11,13,11,5,14,5,5,16,16,16,15,15,16,1,3,1,15,5,16,16,
  14,15,15,16,16,14,15,16,15,5,15,15,16,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,16,15,15,15,1,15,15,16,15,15,16,15,15,
  15,15,16,16,15,15,15,15,15,5,16,15,5,16,11,5,11,15,5,5,
  11,11,13,5,5,14,15,15,15,16,15,15,16,15,15,15,16,15,16,
  16,16,15,15,15,15,15,15,16,15,15,16,16,16,15,16,15,15,15,
  15,16,15,5,15,16,15,12,5,15,5,15,5,12,5,5,5,5,5,12,5,
  14,5,5,5,12,11,5,5,16,5,5,5,13,5,5,16,16,15,15,16,14,
  16,14,14,16,5,15,5,15,14,16,5,15,14,14,14,14,5,16,5,16,
  14,5,16,5,14,14,14,14,5,5,16,5,11,15,12,5,5,5,5,12,11,
  11,12,11,10,5,5,5,5,11,11,13,5,5,11,12,11,11,5,5,5,5,
  12,11,11,11,5,5,5,11,11,12,11,5,5,5,5,5,5,12,11,11,5,
  5,11,11,12,5,11,11,11,11,11,11,5,5,5,5,5,5,5,5,5,15,
  15,5,5,5,15,5,5,5,15,15,15,5,5,5,5,5,5,5,5,5,15,15,
  15,15,15,15,11,5,5,11,12,12,5,5,12,5,12,5,5,5,5,5,5,5,
  16,5,5,5,15,5,15,5,16,5,3,5,5,5,5,5,5,5,5,5,5,5,5,
  15,15,3,12,5,5,12,11,11,13,5,10,5,10,11,5,5,5,5,5,5,5,
  5,5,5,5,5,16,5,5,3,5,5,5,5,5,5,5,5,15,5,16,5,15,3,5,
  12,5,13,5,11,10,11,5,11,5,14,5,5,5,5,16,5,5,5,5,5,5,
  5,5,5,15,15,5,5,5,5,15,5,5,5,5,15,5,16,15,15,15,5,5,
  5,5,11,12,5,5,11,5,12,12,5,5,16,5,5,5,5,5,5,5,5,5,3,
  15,5,5,5,5,5,5,15,5,5,16,5,5,15,3,5,5,13,11,5,10,14,
  5,5,11,5,13,5,5,5,16,5,5,5,15,5,16,5,5,15,15,5,5,5,5,
  5,5,5,5,5,5,5,5,15,15,15,5,5,5,5,12,12,12,5,11,11,10,
  12,5,5,5,5,5,5,5,5,5,5,1,15,5,5,5,5,5,5,5,5,5,5,5,
  5,15,15,1,5,12,5,14,10,15,5,5,10,5,10,12,5,5,5,5,5,5,
  5,5,5,3,15,5,15,5,5,5,5,5,15,5,16,5,16,15,16,5,5,16,
  13,15,11,14,5,12,12,5,14,5,5,5,5,5,15,16,5,16,15,16,5,
  5,5,5,5,5,5,5,5,5,5,5,16,15,3,12,5,5,16,12,14,12,5,
  11,12,11,5,5,5,15,5,16,5,5,16,15,5,5,5,5,5,5,5,5,5,5,
  5,5,3,15,5,5,11,5,5,14,13,5,5,10,12,10,5,5,5,5,5,16,
  5,5,15,5,5,15,5,15,5,5,5,15,5,15,5,15,15,5,5,5,12,5,
  12,10,12,11,12,13,13,5,5,5,5,5,5,5,16,5,5,5,5,5,5,5,
  5,5,5,5,5,1,15,5,14,5,5,5,15,10,12,5,10,12,10,5,15,16,
  5,16,15,16,5,5,5,5,5,5,5,5,5,5,5,5,16,15,15,11,15,5,
  15,12,16,14,5,11,14,11,5,5,5,16,5,15,5,5,5,5,5,5,5,5,
  5,5,5,5,15,15,15,11,5,5,11,11,11,11,5,10,11,10,11,5,5,
  16,16,5,5,5,5,5,5,5,5,5,5,5,5,3,15,15,5,13,5,5,16,14,
  5,5,11,15,11,5,5,5,3,15,5,5,5,13,15,5,5,16,5,16,5,15,
  16,5,15,5,14,5,12,11,16,11,11,15,13,5,5,1,5,5,5,5,5,5,
  5,5,5,5,5,5,16,15,16,15,11,5,15,14,13,5,5,11,16,11,13,
  15,5,5,5,5,5,5,5,15,5,16,5,14,1,1,1,5,1,16,16,15,15,
  12,5,13,16,14,16,15,15,16,15,15,15,15,15,15,15,16,15,1,
  1,1,4,15,15,15,16,16,16,15,16,16,16,15,5,16,5,3,5,5,5,
  16,5,5,5,16,15,5,5,5,5,5,13,10,5,12,14,12,11,5,5,5,5,
  5,16,5,5,16,5,5,15,15,15,5,12,11,5,10,12,5,12,12,5,11,
  5,5,14,16,5,5,16,5,16,5,15,16,5,5,5,11,5,14,11,5,12,
  12,16,12,5,3,5,5,5,5,5,5,5,15,15,5,5,5,12,5,10,10,5,
  12,15,5,15,5,15,16,1,1,5,15,3,1,1,3,5,5,15,14,14,11,5,
  4,1,14,16,14,5,5,15,5,16,5,15,15,5,11,5,11,5,11,10,11,
  11,11,12,11,5,16,5,16,5,16,15,15,3,5,15,11,16,10,12,5,
  13,14,5,11,14,5,15,5,16,15,15,3,5,11,11,12,11,14,5,14,
  14,5,16,14,5,5,5,15,15,5,5,5,15,5,14,10,11,16,13,13,12,
  5,5,5,15,3,13,5,16,15,11,11,12,5,5,16,5,16,14,5,15,1,
  5,13,5,16,5,12,11,13,5,14,12,14,5,15,15,16,5,5,5,12,11,
  12,5,16,16,5,14,12,15,15,15,16,15,15,1,15,16,15,16,16,
  16,15,15,15,15,4,15,16,16,15,16,16,15,16,15,14,15,5,4,
  14,16,16,16,15,12,13,4,5,11,12,11,10,5,5,11,12,12,12,12,
  12,12,12,5,11,12,12,12,11,5,12,12,12,11,13,11,12,5,11,
  11,11,11,11,5,10,5,15,13,13,15,12,15,11,12,12,15,13,15,
  13,11,11,11,13,12,12,12,13,11,12,15,12,13,11,12,5,5,5,5,
  5,15,16,5,5,5,15,5,5,15,15,5,16,5,5,5,5,5,5,5,5,5,15,
  15,15,15,5,15,14,10,5,12,10,5,13,5,5,5,11,15,5,5,5,5,
  15,15,5,5,5,15,5,5,16,5,15,16,5,5,5,5,5,5,5,5,5,15,
  16,5,15,16,15,12,5,13,5,5,11,15,5,10,5,15,11,5,5,5,16,
  5,5,5,5,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,15,15,15,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,16,5,5,5,5,15,5,
  5,5,5,16,5,5,5,5,5,5,5,5,5,5,16,15,5,5,16,5,5,13,15,
  5,11,15,11,5,5,5,12,5,5,15,16,5,5,5,15,5,16,5,5,5,5,
  5,5,5,5,5,5,15,5,5,15,15,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,5,15,5,5,5,5,15,5,5,5,5,16,5,5,5,5,5,5,5,5,5,5,
  15,15,15,5,15,16,10,14,5,15,5,12,5,5,5,10,5,12,15,15,
  15,15,15,15,15,15,15,16,15,15,15,15,15,16,15,15,15,15,1,
  15,1,15,15,16,15,16,15,15,15,15,14,15,16,15,16,16,5,5,
  15,15,5,1,5,1,13,15,5,15,15,5,5,15,15,5,5,15,15,5,15,
  5,16,16,5,5,16,15,12,12,5,5,5,5,5,5,5,16,5,15,5,5,16,
  15,5,5,5,5,15,5,5,5,15,15,15,5,16,15,5,5,15,11,5,15,
  12,5,5,11,12,5,10,5,15,5,5,16,5,5,5,5,5,5,5,5,5,5,5,
  16,15,15,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,5,15,15,
  15,5,16,5,5,15,5,15,5,5,5,15,15,15,5,16,5,15,15,5,5,
  10,12,5,12,10,5,15,5,5,15,16,15,1,16,15,15,14,15,15,15,
  15,15,15,15,15,15,15,16,15,15,15,15,15,15,16,16,16,15,15,
  15,15,15,5,5,15,5,5,5,5,5,5,5,5,5,5,15,15,15,15,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,14,5,15,5,5,5,12,5,5,
  15,5,15,15,5,16,5,5,11,14,5,5,16,13,5,12,11,15,5,5,5,
  5,16,5,5,5,5,5,5,5,15,1,15,1,5,5,14,12,15,5,14,11,5,
  5,16,5,5,11,14,11,5,16,5,5,5,5,15,5,5,5,15,15,15,5,12,
  5,14,16,5,10,12,11,5,5,5,5,5,5,11,5,5,5,5,15,5,5,5,5,
  5,15,15,5,3,3,1,5,16,16,14,16,16,15,15,5,14,16,16,15,5,
  12,5,5,5,5,5,15,15,16,5,1,3,3,16,5,14,14,14,5,16,15,5,
  14,14,14,5,15,5,15,5,15,5,16,15,15,5,16,5,5,5,11,5,5,
  12,5,10,12,15,14,10,5,5,5,5,5,16,5,5,16,15,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,14,15,5,5,1,15,1,5,16,5,15,
  5,5,5,5,5,10,11,13,12,16,5,5,16,5,15,5,5,15,15,5,5,16,
  5,10,5,12,5,5,5,5,15,12,10,14,5,5,16,16,16,16,15,16,3,
  1,1,5,15,16,14,14,5,15,16,16,14,14,16,15,5,5,15,15,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,16,5,16,15,1,5,5,16,5,
  12,15,16,5,5,5,15,13,15,5,15,5,15,15,16,5,5,16,5,11,5,
  12,5,5,5,12,14,5,11,14,15,15,15,15,13,5,5,5,12,5,13,5,
  14,15,5,16,5,5,15,15,15,16,16,15,16,15,15,15,15,15,15,
  16,15,16,16,15,15,16,15,15,15,16,15,16,15,16,15,15,15,15,
  15,5,5,16,15,5,5,16,5,15,5,16,15,5,12,12,5,5,14,14,5,
  15,16,14,14,14,14,16,5,5,5,15,16,16,15,15,16,16,14,16,5,
  16,5,14,14,5,16,15,16,14,14,16,5,15,16,5,12,12,12,5,5,
  5,5,11,11,12,13,11,12,5,5,5,5,12,12,13,5,5,12,13,11,11,
  5,5,5,5,12,12,12,11,5,5,5,12,11,11,12,5,5,5,5,5,5,11,
  12,12,5,5,11,12,13,5,12,11,12,11,14,12,5,5,5,5,5,5,5,
  3,3,5,5,5,5,5,5,3,3,5,3,5,5,5,5,5,5,5,5,5,5,3,3,5,
  3,3,15,5,11,11,5,5,3,5,5,5,11,11,5,5,5,3,3,5,5,5,5,
  5,5,3,3,5,3,5,5,5,5,5,5,5,5,5,5,3,3,5,3,3,15,5,11,
  11,5,5,3,5,5,5,11,11,5,5,15,3,5,5,5,5,5,5,5,15,3,3,
  5,5,5,5,5,5,5,5,5,15,5,5,5,3,3,11,11,11,11,5,11,11,5,
  5,5,11,11,5,5,7,5,5,5,5,5,3,5,3,3,3,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,3,11,11,5,15,5,11,11,5,5,5,5,3,5,7,5,
  5,5,5,5,3,5,3,3,3,5,5,5,5,5,5,5,5,5,5,5,5,5,5,3,11,
  11,5,15,5,11,11,5,5,5,5,3,3,5,5,5,5,5,5,5,15,3,3,5,
  5,5,5,5,5,5,5,5,3,3,3,5,3,3,11,15,12,11,5,5,16,5,5,
  5,12,11,3,15,3,3,3,3,15,3,3,3,15,3,15,15,15,3,3,3,15,
  3,3,3,15,15,4,3,3,15,4,15,3,4,15,15,15,15,7,5,5,15,5,
  15,5,3,15,3,5,5,5,5,5,5,5,5,5,5,5,5,3,5,3,11,11,5,
  11,11,11,11,5,5,11,5,11,5,5,5,5,15,3,5,3,5,5,5,5,5,5,
  5,5,15,5,5,5,5,5,15,11,5,5,11,5,5,11,5,5,5,5,11,3,5,
  5,5,7,5,3,5,5,5,5,5,5,5,5,5,5,5,5,3,5,3,15,5,5,11,
  11,5,15,5,5,11,5,11,3,15,5,3,15,3,5,5,5,5,5,5,5,5,5,
  5,5,5,3,5,3,16,5,5,11,12,15,11,5,5,12,5,11,5,5,7,5,3,
  5,5,5,5,5,5,5,5,5,5,5,5,3,5,3,15,5,5,11,11,5,15,5,5,
  11,5,11,5,3,15,3,5,5,5,5,5,5,5,5,5,5,5,5,3,5,3,11,
  12,5,11,15,12,11,5,5,5,5,15,3,5,3,5,5,5,5,5,5,5,5,3,
  5,3,3,5,11,15,12,5,11,11,5,5,16,5,5,5,11,11,15,4,3,15,
  3,15,15,3,15,3,3,15,3,3,3,15,3,4,15,15,3,3,15,4,15,15,
  3,15,4,3,5,5,5,5,5,5,5,5,5,11,5,5,3,3,3,11,3,11,15,
  11,15,11,5,5,11,11,15,3,15,3,3,3,3,15,3,3,15,3,3,3,3,
  3,3,1,3,4,3,1,3,1,1,3,3,3,5,15,5,15,5,5,5,15,5,5,5,
  15,5,3,11,5,5,11,11,5,11,11,11,11,5,11,5,5,5,5,15,5,5,
  15,5,5,5,5,3,11,5,5,11,5,5,11,5,5,5,5,11,5,15,3,5,3,
  15,5,3,3,15,5,3,11,5,5,16,5,5,11,12,12,15,5,11,3,5,5,
  5,5,5,5,5,5,15,3,11,5,11,11,5,5,11,11,11,5,11,11,3,11,
  3,15,5,3,3,15,3,3,11,5,11,15,11,5,11,15,3,12,11,15,3,
  5,5,5,15,15,5,5,3,11,5,5,15,5,5,11,11,11,5,5,15,3,5,
  15,3,3,5,5,3,11,5,5,16,5,5,11,11,11,5,5,11,5,5,15,15,
  5,5,3,11,5,5,15,5,5,11,11,11,5,5,15,5,5,5,11,5,15,12,
  5,5,11,10,5,11,15,5,10,5,11,5,5,5,3,3,11,10,5,11,5,10,
  11,5,5,5,15,11,5,5,3,3,14,5,15,13,5,5,14,15,5,5,5,14,
  5,3,3,14,5,15,13,5,5,14,15,5,5,5,14,5,3,15,11,5,11,3,
  11,15,12,11,15,5,11,3,15,11,3,11,5,11,15,11,11,5,15,11,
  4,3,1,3,1,3,3,3,3,1,1,3,11,14,11,14,11,3,11,11,14,13,
  4,10,14,10,5,11,5,5,10,10,12,11,5,10,12,10,10,5,5,11,
  11,13,4,14,14,11,11,3,10,12,10,11,5,5,11,11,5,5,11,10,
  14,11,11,14,14,11,5,11,11,14,10,10,12,5,11,11,5,5,5,3,
  3,5,5,5,5,5,5,3,3,5,3,5,5,5,5,5,5,5,5,5,5,3,3,5,3,
  3,15,5,11,11,5,5,3,5,5,5,11,11,5,5,15,15,5,5,5,5,5,3,
  3,5,3,3,5,5,5,5,5,5,5,5,5,3,3,5,5,3,5,11,11,15,5,5,
  11,11,5,5,5,3,5,5,15,15,5,5,5,5,5,3,3,5,3,3,5,5,5,5,
  5,5,5,5,5,3,3,5,5,3,5,11,11,15,5,5,11,11,5,5,5,3,5,
  5,5,5,5,5,5,5,5,15,5,3,3,5,5,5,5,5,5,5,5,5,5,15,5,
  5,3,3,11,11,11,11,5,11,11,5,5,5,11,11,5,5,5,5,5,5,3,
  3,3,3,3,5,5,5,5,5,5,5,5,5,3,3,3,5,3,3,13,15,14,15,5,
  5,14,5,5,5,14,5,5,5,5,5,5,3,3,3,3,3,5,5,5,5,5,5,5,
  5,5,3,3,3,5,3,3,13,15,14,15,5,5,14,5,5,5,14,5,5,5,15,
  5,15,3,5,15,3,5,5,5,5,5,5,5,5,5,5,15,5,3,3,5,11,11,
  11,5,11,11,11,5,5,11,11,5,5,5,5,5,7,5,5,3,5,5,5,5,5,
  5,5,5,3,5,3,5,3,3,5,15,5,11,5,11,5,15,5,5,11,11,5,5,
  5,5,7,5,5,3,5,5,5,5,5,5,5,5,3,5,3,5,3,3,5,15,5,11,
  5,11,5,15,5,5,11,11,5,15,15,3,3,15,3,5,5,5,5,5,5,5,5,
  5,5,15,5,11,15,5,11,5,11,5,10,15,11,5,5,10,11,5,5,3,
  15,5,3,5,5,5,5,5,5,5,5,5,5,15,5,5,15,5,11,5,11,5,5,
  5,11,5,5,5,11,5,3,5,15,3,5,5,5,5,5,5,5,5,5,5,15,5,3,
  3,5,11,12,11,5,15,12,11,5,5,5,16,5,3,15,4,3,15,3,3,15,
  3,15,15,3,15,3,15,3,3,15,4,15,3,15,3,15,4,15,15,3,4,
  15,5,3,5,5,5,5,5,5,5,5,5,5,15,5,5,15,11,16,5,11,11,5,
  5,13,5,5,5,11,11,3,5,5,5,5,5,5,5,5,5,11,15,5,3,3,3,
  11,3,15,11,11,15,11,5,5,11,15,11,3,15,3,3,3,3,15,3,3,
  15,4,3,3,3,3,3,3,3,3,3,3,3,3,3,3,4,3,5,15,5,15,5,5,
  5,15,5,3,5,15,3,5,11,5,11,5,11,5,11,11,11,11,11,5,5,5,
  5,5,15,5,5,15,3,5,5,3,5,11,5,11,5,5,5,11,5,5,5,11,5,
  3,15,3,5,5,15,5,3,5,15,3,5,11,5,16,5,5,5,11,12,12,15,
  11,5,3,5,5,5,5,5,7,5,5,3,5,11,5,15,5,5,5,11,11,11,5,
  15,5,3,11,3,15,5,3,3,15,3,3,11,5,15,11,11,5,11,15,3,
  11,15,11,5,5,5,5,7,5,5,3,5,11,5,15,5,5,5,11,11,11,5,
  15,5,15,5,15,3,3,5,3,3,11,5,11,15,5,5,11,10,10,5,11,5,
  5,5,3,15,5,3,15,11,5,11,11,5,5,11,11,11,5,11,11,5,3,5,
  15,3,5,16,5,11,5,12,5,11,15,5,12,11,5,3,5,5,3,5,11,11,
  12,5,5,11,11,5,5,5,15,5,3,15,4,15,4,15,4,15,15,15,3,3,
  3,15,4,15,5,3,15,16,5,11,12,5,5,11,15,5,5,11,12,3,5,
  15,11,11,5,3,11,15,11,11,15,11,5,3,3,3,3,1,1,3,4,3,3,
  1,3,1,15,11,11,3,5,11,15,11,11,5,11,15,11,15,12,13,11,
  15,11,11,14,15,14,12,10,10,5,11,5,5,10,14,10,11,11,14,
  11,14,12,11,3,11,5,10,14,10,10,5,11,5,10,14,10,10,5,11,
  5,11,5,5,10,12,10,11,11,13,4,12,5,10,12,10,11,14,11,11,
  5,11,5,5,7,5,5,5,5,5,5,5,5,5,3,5,5,5,5,5,5,5,5,5,5,
  3,5,3,5,3,3,5,15,11,11,5,3,5,5,5,5,11,11,5,3,15,5,5,
  5,5,5,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,7,5,5,5,5,5,5,5,5,5,3,5,5,5,
  5,5,5,5,5,5,5,3,5,3,5,3,3,5,15,11,11,5,3,5,5,5,5,11,
  11,3,3,3,15,3,3,3,3,3,3,15,15,3,15,3,15,15,3,3,15,3,
  15,3,15,4,4,15,3,3,3,15,3,15,15,15,15,3,3,5,5,5,5,5,
  3,3,3,5,15,5,5,5,5,5,5,5,5,5,5,5,15,5,3,3,12,5,15,
  15,5,5,12,5,5,5,5,5,5,5,15,5,15,5,5,15,5,5,5,5,5,5,
  5,5,5,5,5,5,15,15,5,5,5,11,5,5,11,11,5,5,5,11,5,5,5,
  3,5,5,15,15,5,5,5,5,5,5,5,5,5,5,3,5,5,3,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,5,5,5,5,5,5,5,5,
  5,5,5,15,15,5,5,11,5,5,5,11,5,11,5,5,11,5,5,3,15,3,3,
  15,5,5,5,5,5,5,5,5,5,5,5,5,15,11,5,5,5,5,5,5,10,15,
  5,5,5,10,5,5,5,15,15,5,5,5,5,5,5,5,5,5,5,3,5,5,3,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,11,5,5,5,5,5,5,5,5,
  5,5,5,5,15,15,5,5,5,10,5,5,15,10,5,5,5,5,5,5,5,5,3,
  5,5,5,5,5,5,5,5,3,5,3,3,5,11,11,3,5,11,11,5,5,3,5,5,
  5,11,11,5,3,5,5,5,5,5,5,5,5,3,5,3,3,5,11,11,3,5,11,
  11,5,5,3,5,5,5,11,11,5,5,5,5,5,5,5,5,5,5,15,5,15,3,
  3,3,5,3,15,15,11,15,5,5,5,11,15,15,5,5,5,5,5,5,5,5,5,
  5,15,15,3,3,3,3,5,15,15,11,5,16,5,5,11,15,15,5,15,5,
  15,5,5,5,15,5,5,3,15,5,5,5,5,5,5,11,5,5,11,11,11,5,5,
  5,5,5,5,15,5,5,15,5,3,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,5,11,15,5,5,15,5,3,3,15,5,5,5,5,5,5,5,5,5,10,10,15,
  5,5,3,5,3,5,5,5,5,7,5,15,15,5,5,11,11,5,5,5,3,15,5,
  11,11,5,15,3,15,5,5,3,15,3,3,5,5,15,15,11,5,5,16,3,11,
  15,15,5,5,5,5,15,3,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,3,
  5,15,5,3,5,3,3,5,5,15,15,5,5,5,12,12,5,5,5,5,5,5,7,
  5,15,15,5,5,11,11,5,5,5,3,15,5,11,11,5,5,3,11,5,5,5,
  5,5,5,10,5,5,15,5,10,5,5,5,3,5,3,3,5,12,5,5,5,12,5,
  5,5,5,15,15,3,5,3,3,12,5,15,15,5,5,12,5,5,5,5,5,15,4,
  4,15,15,3,3,15,15,15,3,3,15,3,3,5,5,11,11,5,5,15,11,
  11,11,11,15,5,5,5,3,3,3,3,5,3,3,3,15,5,15,15,3,3,3,3,
  5,3,3,3,15,5,15,15,5,11,11,10,5,5,5,5,10,11,11,11,11,
  10,5,5,5,5,10,11,11,5,5,11,11,14,11,5,5,5,5,11,11,14,
  11,5,5,5,10,10,10,10,5,5,5,5,5,5,10,11,11,5,5,10,11,
  11,5,10,11,11,10,11,11,5,5,5,5,15,15,5,5,5,5,5,3,3,5,
  3,3,5,5,5,5,5,5,5,5,5,3,3,5,5,3,5,11,11,15,5,5,11,
  11,5,5,5,3,5,7,5,5,5,5,5,5,5,5,5,3,5,5,5,5,5,5,5,5,
  5,5,3,5,3,5,3,3,5,15,11,11,5,3,5,5,5,5,11,11,15,15,
  15,15,15,15,15,15,15,4,15,15,15,15,15,15,15,15,15,15,7,
  15,3,15,4,3,15,16,16,3,15,16,15,15,15,15,16,16,5,5,5,5,
  5,3,3,3,3,3,5,5,5,5,5,5,5,5,5,15,5,5,5,3,3,12,11,11,
  15,5,11,12,5,5,5,16,5,5,5,15,5,15,15,5,3,5,5,5,5,5,5,
  5,5,5,3,3,5,5,3,3,5,5,15,11,5,11,15,5,5,5,11,11,5,5,
  15,5,15,15,5,3,5,5,5,5,5,5,5,5,5,3,3,5,5,3,3,5,5,15,
  11,5,11,15,5,5,5,11,11,5,5,5,5,5,5,3,15,5,5,5,5,5,5,
  5,5,5,15,5,5,3,3,5,11,11,11,5,11,11,11,5,5,11,11,5,3,
  15,15,3,3,5,5,5,5,5,5,5,5,5,5,15,5,5,11,15,5,5,12,11,
  5,11,15,5,5,5,11,11,5,5,5,15,3,5,5,5,5,5,5,5,5,5,5,
  15,5,5,5,15,5,5,11,11,5,5,11,5,5,5,5,11,5,5,5,3,3,5,
  5,5,5,5,5,5,5,3,3,3,5,3,3,5,5,14,13,5,15,14,15,5,5,
  5,14,5,5,3,3,5,5,5,5,5,5,5,5,3,3,3,5,3,3,5,5,14,13,
  5,15,14,15,5,5,5,14,5,3,3,5,5,5,5,5,5,5,5,5,15,5,5,
  5,15,11,15,11,11,10,5,12,5,5,5,5,11,10,3,3,3,15,3,3,
  15,3,3,3,4,15,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,5,5,
  5,5,5,5,5,5,5,15,11,5,3,3,3,3,11,15,12,12,11,15,5,5,
  11,15,11,5,15,5,15,5,5,5,15,3,5,5,15,3,5,5,11,11,5,11,
  11,5,11,11,11,11,5,5,5,3,5,15,5,5,7,5,5,5,3,5,5,11,
  15,5,5,11,5,11,11,5,15,5,5,11,15,5,5,15,3,3,5,15,3,5,
  5,11,11,5,5,11,5,10,10,15,11,5,3,5,15,5,5,7,5,5,5,3,
  5,5,11,15,5,5,11,5,11,11,5,15,5,5,3,3,15,3,5,3,15,3,
  3,5,11,15,11,12,11,5,15,3,11,15,11,5,5,5,3,15,5,5,3,5,
  5,11,11,5,5,11,5,5,5,5,11,5,3,5,3,5,3,5,3,3,5,11,11,
  15,5,11,5,12,12,5,15,5,5,3,5,15,5,3,15,5,11,11,11,5,
  11,5,11,11,5,11,11,3,5,5,15,3,5,5,16,11,5,12,11,5,15,
  5,12,11,5,3,3,15,4,15,15,4,4,15,15,3,15,3,3,15,4,15,5,
  5,3,5,11,11,16,5,5,11,11,5,5,5,13,5,5,3,15,5,16,11,12,
  5,11,5,15,5,5,11,12,3,5,11,15,11,5,3,15,11,11,12,15,11,
  5,3,3,3,3,3,3,3,3,3,3,3,3,3,12,15,11,3,5,15,11,11,12,
  5,11,15,11,14,11,10,11,5,5,5,10,14,10,11,14,14,15,11,11,
  11,13,15,12,11,11,15,12,13,14,11,15,11,5,14,10,10,10,5,
  11,5,12,10,10,10,5,11,5,11,11,11,14,11,14,5,5,10,14,11,
  5,10,14,10,10,12,10,11,5,11,5,7,5,5,5,5,5,3,5,3,3,3,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,3,11,11,5,15,5,11,11,5,
  5,5,5,3,15,5,5,5,5,5,15,5,5,3,3,5,5,5,5,5,5,5,5,5,
  3,3,3,5,15,5,15,15,12,5,5,5,5,5,5,5,12,5,15,15,15,15,
  15,7,15,3,16,16,15,15,15,15,15,15,15,15,15,15,15,15,15,
  15,3,16,16,15,3,15,18,18,15,15,15,15,3,5,5,15,5,7,5,5,
  15,15,5,5,5,5,5,5,5,5,5,5,5,5,3,5,5,11,11,5,5,15,11,
  11,5,5,3,5,5,5,5,5,3,15,5,5,5,5,5,5,5,5,5,5,5,15,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,5,7,5,5,15,15,
  5,5,5,5,5,5,5,5,5,5,5,5,3,5,5,11,11,5,5,15,11,11,5,
  5,3,5,5,3,3,5,3,15,15,5,5,5,5,5,5,5,5,5,5,5,5,15,5,
  5,5,5,5,5,12,15,15,5,5,12,5,5,3,5,15,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,3,3,4,
  4,3,3,3,15,15,3,15,15,3,3,3,15,3,15,15,3,3,15,15,3,3,
  3,15,15,3,15,15,5,3,3,5,5,5,5,5,5,5,5,3,3,3,5,5,15,
  5,5,5,12,5,5,15,15,5,5,5,12,5,3,3,5,5,5,5,5,5,5,5,5,
  5,5,5,5,5,15,15,15,5,12,5,5,5,5,5,5,5,12,5,5,5,5,5,
  5,5,5,5,5,11,11,5,3,3,3,3,3,3,3,15,15,15,5,5,3,3,3,
  5,5,5,5,5,5,5,5,5,11,11,5,3,3,3,3,3,3,3,15,15,15,5,
  5,3,3,3,5,15,5,15,5,5,5,15,5,5,5,15,5,5,5,5,5,5,11,
  5,5,11,11,11,5,5,3,5,5,5,15,5,5,15,15,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,11,3,5,5,15,3,3,5,15,5,5,5,5,5,
  5,5,5,5,10,10,15,5,5,15,5,5,5,5,5,5,5,5,15,5,5,5,11,
  5,5,5,5,11,11,5,11,5,5,11,15,15,5,5,15,15,3,3,5,5,11,
  11,11,5,5,15,15,11,11,11,15,5,5,15,15,5,5,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,3,5,15,15,3,5,5,3,5,5,5,3,5,5,5,10,
  10,5,5,5,5,5,5,15,5,5,15,5,5,5,11,5,5,5,11,11,5,5,11,
  5,5,5,11,5,5,5,5,5,5,10,5,5,15,5,10,5,5,5,5,5,3,5,
  11,11,3,5,5,11,11,5,5,5,3,5,5,5,3,5,11,11,3,5,5,11,
  11,5,5,5,3,5,5,5,11,5,5,5,10,5,5,5,15,5,5,5,10,5,5,
  15,15,5,5,3,15,15,11,11,15,5,5,5,15,15,3,5,5,15,15,11,
  11,5,16,5,15,15,5,3,5,15,15,11,11,5,5,16,5,11,11,11,5,
  5,5,5,14,11,11,11,11,11,5,5,5,5,14,11,11,5,5,11,11,10,
  10,5,5,5,5,11,11,10,10,5,5,5,11,11,10,10,5,5,5,5,5,5,
  11,11,11,5,5,11,11,11,5,10,10,10,10,10,10,5,5,5,5,5,5,
  5,5,5,5,5,5,3,5,5,5,5,5,5,5,5,5,5,3,5,3,5,15,3,5,
  15,14,12,5,3,5,5,5,5,14,12,5,5,5,5,5,3,5,3,3,3,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,3,14,12,5,3,5,12,14,5,5,5,5,
  3,5,5,7,5,15,5,15,3,5,5,5,5,5,5,5,5,5,5,5,5,5,3,5,
  3,5,15,5,11,11,15,5,5,5,11,5,11,5,5,5,5,15,5,3,5,5,5,
  5,5,5,5,5,5,15,5,5,5,5,5,15,5,11,5,11,5,11,5,5,5,5,
  5,11,5,5,5,5,5,3,15,5,5,5,5,5,5,5,5,5,5,5,5,3,5,3,
  11,11,5,11,11,11,11,5,5,11,5,11,7,3,5,3,3,5,5,5,5,5,
  5,5,5,5,5,5,5,5,15,5,3,5,3,5,12,14,3,5,5,5,14,5,12,
  15,5,15,3,5,5,5,5,5,5,5,5,5,5,5,5,5,3,5,3,5,15,5,11,
  11,15,5,5,5,11,5,11,5,5,3,3,5,5,5,5,5,5,5,5,5,5,5,5,
  3,5,3,15,13,5,13,15,14,5,5,5,5,5,14,5,3,3,5,5,5,5,5,
  5,5,5,15,5,15,5,5,11,15,5,11,10,11,5,11,15,5,5,5,10,
  11,3,3,5,5,5,5,5,5,5,5,5,5,5,5,3,5,3,15,13,5,13,15,
  14,5,5,5,5,5,14,3,3,3,15,3,3,15,3,3,3,3,15,3,3,3,3,
  1,3,3,3,3,3,1,1,1,3,3,4,5,5,5,5,5,5,5,5,5,5,11,5,3,
  3,3,3,11,11,15,11,11,15,5,5,11,11,15,5,15,5,15,5,5,5,
  15,5,5,5,15,5,3,5,11,5,11,11,11,5,11,11,11,5,11,5,5,3,
  5,7,5,5,15,5,15,5,5,3,5,11,5,15,5,11,5,11,11,5,5,15,
  5,11,15,5,5,15,5,15,5,15,5,3,5,11,5,11,5,11,5,10,10,
  15,5,11,3,5,5,5,5,5,5,5,5,15,3,5,11,11,11,5,11,5,11,
  11,5,11,11,5,3,3,15,3,5,3,15,3,3,5,11,11,15,11,11,5,
  15,3,11,11,15,5,5,5,5,15,5,5,5,3,5,11,5,11,5,11,5,5,
  5,5,5,11,7,5,3,5,3,5,5,3,5,12,5,3,5,12,5,14,14,5,5,
  16,5,15,5,15,5,5,3,5,11,5,15,5,11,5,11,11,5,5,15,5,5,
  5,11,5,15,5,12,5,11,10,11,5,15,5,10,5,11,5,5,5,3,3,5,
  14,5,14,5,14,5,15,5,5,15,14,5,5,3,3,10,11,15,11,5,11,
  10,5,5,5,5,12,5,3,3,5,14,5,14,5,14,5,15,5,5,15,14,5,
  3,11,15,5,11,3,15,11,11,11,15,5,11,3,11,15,3,11,5,15,
  11,11,11,5,15,11,3,3,3,3,3,4,3,3,3,3,3,3,11,10,12,10,
  11,5,5,5,10,10,14,12,15,12,3,11,11,11,14,14,11,11,5,14,
  10,10,10,5,5,11,11,15,14,14,13,11,11,15,14,11,11,10,5,5,
  11,11,11,11,12,12,4,5,5,10,10,12,5,10,10,13,10,10,14,5,
  11,11,5,5,5,5,5,3,3,3,3,3,5,5,5,5,5,5,5,5,5,3,3,3,
  5,3,3,13,15,14,15,5,5,14,5,5,5,14,5,15,5,15,5,15,5,5,
  3,5,5,5,5,5,5,5,5,5,3,3,5,3,15,11,5,5,16,11,5,11,13,
  5,5,5,11,11,5,5,5,3,15,15,3,3,5,5,5,5,5,5,5,5,5,3,3,
  5,3,11,11,5,5,15,11,5,11,15,5,5,5,11,11,5,5,5,5,5,5,
  3,15,5,5,5,5,5,5,5,5,5,15,5,15,15,11,5,11,15,10,5,11,
  5,12,5,5,11,10,5,3,15,15,3,3,5,5,5,5,5,5,5,5,5,3,3,
  5,3,11,11,5,5,15,11,5,11,15,5,5,5,11,11,5,5,5,15,3,5,
  5,5,5,5,5,5,5,5,3,3,5,3,5,11,5,5,15,10,5,5,5,5,5,5,
  5,10,5,15,5,3,3,5,5,5,5,5,5,5,5,3,3,3,3,15,14,5,15,
  15,13,5,15,12,5,5,5,5,14,5,3,3,3,5,5,5,5,5,5,5,5,3,
  3,3,3,3,3,3,12,18,12,11,11,13,18,5,5,11,12,11,3,3,5,5,
  5,5,5,5,5,5,3,3,3,3,5,14,11,15,15,12,11,5,5,15,5,5,5,
  11,11,3,3,3,15,3,3,15,3,3,3,4,15,3,3,4,3,3,7,4,3,3,
  3,3,3,3,3,4,3,5,5,5,5,5,5,5,5,5,15,14,15,3,3,3,3,15,
  14,12,11,5,15,5,5,11,14,16,5,15,5,15,5,5,5,15,3,5,3,
  15,11,5,5,15,10,5,11,5,5,12,11,11,10,5,5,15,3,5,15,5,
  5,3,5,3,5,15,5,5,15,12,5,5,5,5,5,15,5,12,5,5,12,15,5,
  5,15,3,3,3,15,11,5,5,15,11,5,5,5,5,11,10,15,10,5,3,5,
  5,3,5,3,5,3,5,3,3,5,15,13,15,5,5,5,14,13,5,14,5,5,3,
  3,15,3,5,3,15,3,3,5,15,15,14,12,5,5,15,3,12,14,16,5,5,
  5,3,15,3,5,11,5,5,15,11,5,5,5,5,5,5,5,11,5,3,5,3,5,
  3,5,3,3,5,15,13,15,5,5,5,14,13,5,14,5,5,3,5,3,5,15,
  15,5,15,12,11,5,5,5,15,11,5,12,11,3,5,3,14,11,5,5,16,
  11,5,11,5,5,15,5,12,11,5,3,3,15,4,3,15,4,3,3,15,3,15,
  3,3,15,4,3,3,5,3,3,14,15,16,15,5,5,14,5,5,5,11,5,15,
  4,4,15,3,3,3,15,3,15,3,3,15,3,3,11,5,11,16,11,5,15,12,
  11,12,12,15,11,5,4,16,16,4,11,11,4,3,3,16,11,3,15,12,
  15,3,3,5,3,3,3,12,5,3,15,15,12,11,10,5,15,5,5,10,12,
  11,16,15,15,16,15,15,15,16,16,15,3,11,14,12,12,11,10,13,
  10,5,14,14,12,11,5,10,5,11,10,10,10,5,10,5,5,5,5,10,14,
  14,5,5,10,12,14,15,11,11,14,10,12,11,11,5,3,5,15,15,5,
  3,5,3,15,15,5,5,5,5,5,5,5,5,5,5,5,5,3,5,3,12,11,5,
  15,11,11,12,5,5,16,5,5,5,5,5,3,15,3,5,11,5,5,5,5,5,5,
  5,5,15,5,5,5,5,5,15,10,5,5,15,5,5,10,5,5,5,5,5,5,3,
  3,5,3,15,18,5,5,5,5,5,5,5,5,5,5,5,5,3,5,3,14,5,5,15,
  14,15,14,5,5,14,5,5,3,3,5,3,15,18,5,5,5,5,5,5,5,5,5,
  5,5,5,3,5,3,14,5,5,15,14,15,14,5,5,14,5,5,3,5,3,5,15,
  5,5,5,5,5,5,5,5,5,5,5,5,3,5,3,14,5,5,15,15,5,14,5,5,
  5,5,5,3,3,4,4,3,3,3,15,15,3,3,3,3,3,3,3,3,15,3,3,3,
  15,3,3,3,3,15,15,3,15,3,3,3,3,5,5,5,5,5,5,5,5,3,3,3,
  3,5,14,15,12,5,13,15,5,15,16,5,5,5,14,5,3,4,3,3,3,15,
  15,3,3,3,3,3,3,3,3,15,4,4,3,15,4,3,3,4,15,15,3,15,3,
  16,5,5,5,5,5,5,5,5,5,11,5,5,3,3,3,3,3,12,16,13,15,15,
  5,5,3,3,3,11,11,11,11,11,11,11,11,12,12,15,14,3,3,4,3,
  3,16,16,16,11,16,11,11,3,3,4,5,15,5,15,5,5,5,15,5,5,5,
  15,5,3,10,5,5,15,11,5,10,11,11,11,5,5,3,5,5,15,15,5,5,
  15,5,5,5,5,3,10,5,5,3,5,5,10,5,5,5,5,5,5,12,3,3,3,
  15,3,3,3,15,5,3,11,5,5,16,5,5,11,12,13,15,5,5,15,5,5,
  5,5,5,5,5,5,15,3,10,5,11,15,5,5,10,11,11,5,11,5,11,11,
  15,15,5,5,15,15,3,3,11,5,11,15,11,5,11,15,15,11,11,12,
  5,3,5,15,15,3,5,5,3,11,5,5,16,5,5,11,11,11,5,5,5,3,5,
  15,15,3,5,5,3,11,5,5,16,5,5,11,11,11,5,5,5,5,5,5,15,
  5,5,3,11,5,5,15,5,5,11,11,11,5,5,12,5,5,5,12,5,15,11,
  5,5,15,10,5,10,15,5,11,5,5,15,5,5,3,3,12,11,5,15,5,11,
  12,5,5,5,16,5,3,5,3,3,13,5,3,3,5,5,13,5,5,5,5,5,5,3,
  3,14,5,15,15,5,5,12,15,5,5,5,12,5,3,15,13,5,15,3,16,
  14,12,11,15,5,5,3,14,16,3,15,5,13,14,11,12,5,15,5,4,3,
  3,4,3,3,4,3,3,3,3,3,16,12,16,13,11,13,10,11,11,12,14,
  11,16,11,5,11,5,5,13,14,14,15,5,11,12,10,10,5,5,5,15,
  16,15,15,16,15,15,3,11,14,11,10,5,5,5,18,5,5,14,14,14,
  11,10,12,12,14,5,11,10,11,10,11,11,5,5,5,5,15,5,15,15,
  5,3,5,5,5,5,5,5,5,5,5,3,3,5,5,3,3,5,5,15,11,5,11,15,
  5,5,5,11,11,5,15,5,7,5,5,15,15,5,5,5,5,5,5,5,5,5,5,
  5,5,3,5,5,11,11,5,5,15,11,11,5,5,3,5,5,7,15,15,3,16,
  15,15,15,15,15,15,15,15,15,15,15,15,15,4,15,3,15,16,15,
  3,16,16,15,15,15,16,15,16,15,5,15,3,5,5,5,5,5,5,5,5,5,
  5,5,5,5,3,5,3,5,15,5,11,11,15,5,5,5,11,5,11,15,15,16,
  3,15,15,15,15,15,15,15,15,15,15,15,15,4,15,15,16,16,15,
  15,16,16,3,15,15,16,15,15,5,3,3,5,5,5,5,5,5,5,5,3,3,
  3,5,3,3,5,5,16,12,5,11,11,15,5,5,11,12,5,3,3,5,5,5,5,
  5,5,5,5,5,5,5,5,3,5,3,15,11,5,12,11,16,5,5,5,11,5,12,
  15,15,15,15,15,15,15,15,15,18,18,15,15,3,3,3,15,3,3,3,
  16,16,15,15,15,15,16,15,5,5,5,5,5,5,5,5,5,5,11,5,3,3,
  3,3,11,11,11,15,11,15,5,5,15,11,11,5,15,5,15,5,15,5,15,
  15,5,5,3,5,5,5,11,5,5,15,11,5,11,11,15,5,5,15,5,15,5,
  15,5,15,15,5,5,3,5,5,5,11,5,5,15,11,5,11,11,15,5,5,5,
  11,3,5,5,15,15,3,5,3,5,5,5,11,5,5,12,11,5,11,11,16,5,
  5,3,5,5,5,5,5,5,5,3,15,5,5,11,11,5,11,11,5,11,11,11,
  11,5,5,15,3,15,15,5,3,3,3,3,5,11,11,11,15,11,5,15,3,
  15,11,11,5,5,5,5,15,5,3,5,5,5,11,5,5,11,11,5,5,5,11,
  5,5,3,15,15,5,3,3,5,3,5,11,5,15,11,11,5,12,12,16,5,5,
  5,5,5,15,3,5,15,5,11,5,11,11,11,5,11,11,11,5,11,5,5,5,
  3,3,5,5,14,5,5,14,13,5,15,5,14,15,5,5,5,3,3,5,5,14,5,
  5,14,13,5,15,5,14,15,5,5,3,3,5,10,11,15,5,11,11,10,5,
  5,12,5,5,15,5,11,5,11,5,10,11,11,5,15,5,11,5,10,3,3,3,
  4,3,3,3,3,3,3,3,3,3,3,5,11,15,3,5,11,15,12,12,11,11,
  15,5,11,15,5,3,11,15,11,11,11,11,5,15,11,10,10,14,11,5,
  5,5,14,10,11,12,12,15,15,11,11,11,15,14,13,5,11,14,11,
  10,10,11,5,5,11,14,11,10,10,11,5,5,11,14,14,12,15,11,11,
  11,11,11,15,12,12,5,5,14,10,10,5,12,10,10,14,10,10,11,
  11,5,5,5,5,7,5,5,3,5,5,5,5,5,5,5,5,3,5,3,5,3,3,5,
  15,5,11,5,11,5,15,5,5,11,11,5,3,15,15,3,3,5,5,5,5,5,
  5,5,5,5,3,3,5,3,11,11,5,5,15,11,5,11,15,5,5,5,11,11,
  5,5,15,15,5,5,5,5,5,5,5,5,5,5,3,5,5,3,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,15,5,15,5,5,5,5,5,5,5,5,5,15,5,5,
  5,3,3,5,5,12,15,5,15,12,5,5,5,5,5,5,15,15,3,15,15,15,
  15,15,15,15,15,7,15,3,15,16,16,15,3,15,16,15,16,15,3,15,
  15,16,16,15,5,3,5,5,5,5,5,5,5,5,15,5,5,5,5,5,11,3,5,
  5,10,5,5,5,5,5,5,5,10,5,5,5,5,5,5,5,5,5,15,15,5,5,3,
  3,3,5,3,15,11,15,3,5,5,5,15,15,11,5,5,5,5,5,5,5,5,15,
  5,15,5,3,3,3,3,5,15,11,15,5,3,5,5,15,15,11,5,15,5,3,
  5,5,5,7,5,5,5,15,15,5,5,5,11,5,11,5,5,15,15,11,11,5,
  5,5,5,5,15,5,3,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,15,15,3,5,5,3,5,3,5,15,15,5,5,5,5,5,5,5,5,12,12,15,
  15,5,3,5,5,5,7,5,5,5,15,15,5,5,5,11,5,11,5,5,15,15,
  11,11,5,5,11,3,3,5,5,3,3,3,3,5,5,15,11,15,5,5,3,3,15,
  15,11,5,5,3,5,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,15,
  3,15,5,3,5,5,15,5,5,5,15,5,5,5,10,10,5,5,5,3,5,5,15,
  5,5,15,5,5,5,11,5,5,5,11,11,5,5,11,3,3,3,4,4,15,15,
  15,3,15,3,15,15,3,3,3,3,15,5,5,3,3,5,5,12,5,5,5,12,5,
  5,5,15,15,5,5,3,3,5,12,5,15,5,15,5,12,5,5,5,5,5,5,5,
  11,5,5,5,10,5,5,5,3,5,5,5,10,5,5,3,3,3,5,3,3,3,15,
  15,15,15,5,5,3,3,3,5,3,3,3,15,15,15,15,5,11,11,5,15,5,
  11,11,11,11,5,5,15,5,11,10,11,5,5,5,5,14,14,10,11,10,
  11,5,5,5,5,11,11,10,5,5,11,11,11,11,5,5,5,5,10,10,10,
  10,5,5,5,11,11,11,11,5,5,5,5,5,5,11,11,10,5,5,11,11,
  10,5,11,11,10,11,11,10,5,5,5,3,3,5,3,15,18,5,5,5,5,5,
  5,5,5,5,5,5,5,3,5,3,14,5,5,15,14,15,14,5,5,14,5,5,5,
  5,7,5,3,5,5,5,5,5,5,5,5,5,5,5,5,3,5,3,15,5,5,11,11,
  5,15,5,5,11,5,11,5,5,15,15,5,5,5,5,5,5,5,5,5,5,5,5,
  3,5,5,12,14,5,5,16,14,12,5,5,3,5,5,5,5,3,5,5,5,5,5,
  5,5,5,3,5,3,5,3,3,5,16,5,14,5,12,5,3,5,5,12,14,5,5,
  3,5,5,5,5,5,5,5,5,5,5,5,5,3,5,3,3,5,5,14,12,5,3,5,
  5,12,5,14,15,5,5,5,5,5,5,5,5,5,11,5,5,3,3,3,11,3,11,
  11,15,15,11,5,5,15,11,11,15,15,15,15,15,15,15,15,18,15,
  18,15,3,3,3,3,15,16,15,16,15,16,15,15,3,3,3,5,7,5,15,
  5,5,5,15,5,15,5,3,5,5,11,5,5,5,15,5,11,11,11,15,5,5,
  5,5,5,5,15,5,5,15,5,5,3,5,5,11,5,5,5,11,5,11,5,5,11,
  5,5,5,15,7,5,5,3,5,3,5,3,5,5,12,5,5,5,3,5,12,14,14,
  3,5,5,3,5,5,5,5,5,5,5,3,15,5,11,5,11,5,11,5,11,11,11,
  11,11,5,15,11,3,15,5,15,3,3,3,3,11,5,11,12,15,5,11,15,
  3,15,11,12,5,5,15,5,15,5,3,5,5,11,5,5,5,15,5,11,11,11,
  15,5,5,15,5,15,5,15,3,5,3,11,5,5,15,11,5,11,10,10,11,
  5,5,5,5,5,15,3,5,15,11,5,5,11,11,5,11,11,11,11,5,11,5,
  5,5,3,3,5,14,5,15,5,13,5,13,15,5,14,5,5,5,5,3,3,5,11,
  10,5,5,11,10,11,5,5,11,15,5,5,3,3,5,14,5,15,5,13,5,13,
  15,5,14,5,5,15,5,11,11,5,5,10,11,5,11,15,5,11,5,10,3,
  3,3,3,1,1,3,3,3,3,3,3,1,1,5,15,11,3,5,11,11,15,11,11,
  11,15,5,15,11,5,3,11,11,15,12,11,11,5,15,11,14,13,15,11,
  15,11,11,15,11,12,10,10,14,5,11,5,5,12,10,10,5,11,10,12,
  10,10,11,5,5,11,10,12,10,10,11,5,5,12,15,12,14,3,11,11,
  11,5,5,14,10,10,11,11,15,14,13,5,14,11,10,12,10,10,11,
  11,5,3,15,7,3,15,3,3,15,15,15,15,3,3,3,3,15,3,4,15,4,
  15,3,15,3,3,3,15,15,15,3,15,3,15,3,16,3,15,15,15,15,15,
  15,15,15,15,15,15,15,4,15,3,15,16,15,3,16,16,3,15,15,16,
  15,15,3,3,3,5,5,5,5,5,5,5,5,3,3,3,3,11,12,5,5,15,13,
  5,11,15,15,5,5,11,12,5,3,3,3,3,3,15,15,3,3,3,3,3,3,3,
  4,15,4,3,3,15,3,3,3,3,15,15,3,15,3,15,15,15,15,15,15,
  15,15,15,15,15,15,15,4,3,3,15,7,3,3,16,16,15,15,15,16,
  15,15,5,5,5,5,5,5,5,5,5,5,12,5,3,3,3,3,5,11,16,14,15,
  16,5,5,14,12,13,15,15,5,18,5,5,3,15,15,5,3,3,5,3,5,5,
  5,15,13,15,5,14,14,14,5,5,15,5,15,5,15,5,5,15,5,5,15,
  5,5,5,5,5,5,12,15,5,15,5,12,5,5,5,14,3,15,3,15,15,3,
  3,3,5,3,5,5,5,5,12,15,5,13,14,15,5,15,3,5,5,5,5,5,5,
  5,11,15,5,5,5,11,5,10,15,5,11,11,10,11,5,5,18,3,15,15,
  5,3,3,3,3,5,5,11,15,14,15,5,16,3,14,12,13,5,5,5,5,15,
  5,11,5,5,5,5,5,5,11,15,5,5,5,11,5,5,3,15,15,5,3,3,5,
  3,5,5,5,15,13,15,5,14,14,14,5,5,5,5,5,15,15,5,15,5,5,
  5,11,12,15,5,11,16,12,5,11,15,5,3,15,5,5,5,5,5,5,13,
  15,5,15,5,13,5,5,5,5,14,15,5,5,11,5,5,14,16,5,15,5,12,
  15,5,5,12,15,5,10,5,15,5,10,15,10,5,5,11,5,5,12,5,11,
  5,5,5,11,12,15,5,15,15,13,5,11,11,4,3,4,11,11,4,16,16,
  16,3,3,11,15,5,11,12,15,5,11,15,11,11,11,11,15,5,3,3,5,
  3,3,16,12,12,3,3,5,15,5,10,14,11,15,15,5,5,12,10,14,11,
  14,14,16,5,5,5,14,11,14,5,11,15,10,10,10,10,5,5,3,15,
  11,11,13,10,5,5,16,12,12,11,13,10,10,15,15,15,16,16,15,
  5,5,12,10,11,15,12,10,11,14,11,14,11,3,5,5,15,15,5,5,5,
  5,5,5,5,5,5,5,5,5,5,3,5,3,5,14,5,15,15,14,5,5,5,5,5,
  5,15,5,3,5,5,5,5,5,5,5,5,3,5,3,3,5,11,5,5,5,10,5,5,
  5,15,5,5,5,10,5,15,3,15,15,15,15,15,15,15,15,15,15,15,
  15,16,15,16,16,15,15,18,18,15,3,15,15,16,15,16,5,5,5,5,
  5,5,5,5,5,5,11,5,5,3,3,3,5,7,11,15,15,7,5,5,5,15,11,
  15,5,5,5,5,5,5,5,5,5,5,11,5,3,3,3,7,5,11,15,15,5,7,
  5,5,15,11,15,5,15,5,3,5,3,5,15,5,5,15,15,5,15,5,5,5,
  11,11,5,5,15,15,11,5,11,5,5,5,5,15,5,5,15,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,5,5,5,15,3,5,15,15,5,3,15,15,5,
  15,5,5,5,5,5,5,5,14,14,15,5,15,3,5,5,5,5,5,5,5,5,15,
  5,5,5,11,5,5,5,5,11,11,5,11,5,5,15,3,3,5,5,3,3,3,3,
  5,5,11,15,15,5,5,7,7,15,11,15,5,5,5,5,15,5,5,5,5,5,5,
  5,5,5,5,5,5,5,5,5,5,3,3,15,5,3,3,5,3,5,5,5,15,15,5,
  5,14,14,5,5,5,15,5,5,15,15,5,15,5,5,5,11,11,5,5,15,15,
  11,5,11,5,5,5,11,5,11,5,5,5,11,11,5,5,16,5,11,5,11,5,
  5,5,15,5,5,11,5,5,5,11,5,5,5,5,15,5,5,5,15,5,11,5,15,
  5,5,5,11,5,5,5,5,5,11,5,11,5,5,5,11,11,5,5,16,5,11,5,
  11,5,5,15,15,5,3,3,15,15,15,15,15,5,15,5,11,11,15,5,5,
  11,11,11,11,5,15,5,15,15,5,3,3,15,15,15,15,15,5,15,5,
  10,12,12,5,5,5,5,11,10,11,10,11,11,5,5,5,5,12,10,12,5,
  5,10,10,10,10,5,5,5,5,12,11,12,12,5,5,5,12,11,12,12,5,
  5,5,5,5,5,11,10,11,5,5,12,11,12,5,12,10,12,11,10,11,5,
  5,5,5,3,3,5,5,5,5,5,5,5,5,3,3,3,5,3,3,5,5,14,13,5,
  15,14,15,5,5,5,14,5,3,3,5,5,5,5,5,5,5,5,5,5,5,5,3,5,
  3,15,13,5,13,15,14,5,5,5,5,5,14,4,11,11,11,11,11,11,11,
  11,14,15,12,11,3,3,3,11,4,16,16,16,16,15,11,11,16,16,16,
  5,5,5,5,5,5,5,5,5,3,11,5,3,3,3,3,3,13,13,16,3,15,5,
  5,3,3,3,15,15,5,15,5,15,5,15,5,5,5,3,5,5,5,11,5,5,15,
  11,5,11,11,13,5,5,5,5,11,3,15,5,15,15,3,5,3,5,5,5,11,
  5,5,15,11,5,11,11,15,5,5,5,11,3,15,5,15,15,3,5,3,5,5,
  5,11,5,5,15,11,5,11,11,15,5,5,15,5,5,5,5,5,5,5,3,15,
  5,5,10,11,5,15,10,5,11,11,5,12,5,5,14,15,15,15,5,15,3,
  3,3,5,11,11,12,15,11,5,15,15,13,11,11,3,5,5,5,15,5,3,
  5,5,5,10,5,5,15,10,5,5,5,5,5,5,3,15,15,3,3,3,5,3,5,
  11,5,15,15,11,5,12,12,15,5,5,5,5,5,15,3,5,15,5,10,5,
  11,15,10,5,11,11,5,5,12,15,5,5,3,3,5,5,14,15,5,15,13,
  5,15,5,12,5,5,3,5,3,3,5,11,12,12,5,18,12,11,11,11,13,
  15,5,5,3,3,5,11,12,15,5,15,12,11,5,5,5,15,5,15,5,12,5,
  11,5,10,15,10,5,15,5,5,5,11,3,3,3,4,3,3,7,4,3,3,3,3,
  3,3,5,16,14,3,5,15,14,12,11,12,5,15,5,13,15,5,3,15,14,
  16,11,11,5,5,15,3,11,11,16,11,5,5,5,14,14,14,12,13,16,
  13,11,10,11,14,12,11,5,15,12,11,10,10,5,15,5,15,13,11,
  10,11,5,5,15,16,16,16,16,16,15,15,3,11,10,14,12,14,5,5,
  14,14,14,5,11,11,11,11,11,11,5,5,5,5,3,5,5,5,5,5,5,5,
  5,3,5,3,3,5,11,11,3,5,11,11,5,5,3,5,5,5,11,11,3,5,5,
  5,5,5,5,5,5,15,14,15,5,3,3,3,5,3,14,11,12,3,15,5,5,
  16,15,12,3,15,3,3,3,3,15,3,3,15,4,3,3,3,3,3,3,4,3,3,
  3,7,3,3,3,4,3,5,15,15,3,5,5,5,3,5,3,5,15,15,5,5,5,
  12,5,11,5,15,11,16,11,12,5,5,5,5,5,15,5,3,15,3,5,5,11,
  5,5,5,10,5,5,5,15,5,5,5,10,5,5,3,3,5,5,3,5,3,5,15,
  18,5,5,5,14,5,5,5,15,14,14,15,14,5,3,3,5,5,3,5,3,5,
  15,18,5,5,5,14,5,5,5,15,14,14,15,14,5,3,12,3,3,5,3,3,
  3,3,3,5,5,14,12,16,5,15,3,3,14,15,11,5,5,3,5,3,5,5,
  15,5,5,5,14,5,5,5,15,15,5,5,14,5,15,3,15,3,3,5,11,15,
  5,5,10,15,5,5,15,10,11,5,11,5,3,5,3,15,5,11,15,5,5,10,
  11,5,5,15,11,11,5,10,11,3,3,3,4,4,15,3,15,3,15,3,15,3,
  3,3,3,3,15,3,5,3,3,5,5,14,12,5,5,13,15,5,5,15,15,5,3,
  3,4,15,4,15,4,15,3,15,4,3,3,3,4,15,5,11,12,15,5,11,13,
  5,5,15,15,5,5,11,12,16,5,3,3,3,5,3,12,16,13,15,15,15,
  5,11,4,3,3,11,3,16,16,16,16,11,16,11,12,11,11,15,5,11,
  16,11,11,5,11,15,5,14,11,14,5,3,5,5,14,14,11,11,10,13,
  15,15,5,5,14,14,11,11,16,12,16,13,12,11,13,10,5,10,15,
  10,11,5,10,5,11,15,11,14,5,11,5,15,5,5,11,12,10,15,15,
  16,16,15,15,11,14,11,13,11,10,16,5,11,3,5,5,5,5,5,5,5,
  5,5,11,5,5,3,3,3,15,3,12,14,16,3,5,5,5,12,11,15,3,15,
  3,3,3,3,15,3,3,15,3,3,3,3,4,4,3,3,4,3,3,7,3,3,3,3,
  4,5,15,5,3,5,3,15,15,5,5,5,15,5,15,15,5,5,12,11,5,5,
  16,11,11,5,12,5,5,5,5,15,5,5,15,5,5,5,5,11,15,5,5,10,
  5,5,5,5,5,5,5,10,5,3,3,3,5,15,5,3,15,15,5,18,15,5,5,
  14,5,5,5,14,13,15,5,14,3,5,5,5,5,5,5,5,5,15,11,15,5,
  11,10,5,5,5,12,11,5,11,10,3,14,3,3,5,3,3,3,3,3,15,5,
  11,15,13,5,5,3,3,16,12,14,3,3,5,5,15,5,5,5,15,15,5,5,
  14,5,5,5,5,15,5,5,14,3,3,15,3,3,3,5,3,15,5,5,16,15,5,
  5,14,13,5,5,12,15,5,3,15,15,5,18,15,5,5,14,5,5,5,14,
  13,15,5,14,5,5,15,11,5,14,15,5,5,12,11,5,5,16,5,11,5,
  12,5,5,5,15,12,15,10,5,11,5,11,5,5,5,5,15,10,3,5,15,
  14,15,5,15,13,5,5,12,5,15,5,5,14,15,3,15,18,5,11,12,11,
  5,13,16,11,11,11,12,5,16,15,12,5,3,3,3,3,3,12,15,5,16,
  11,16,11,15,11,5,11,13,11,12,5,15,11,16,16,11,3,3,3,3,
  3,16,11,11,16,15,15,16,16,15,3,15,15,15,15,16,10,11,11,
  15,5,5,5,11,10,12,11,5,10,11,10,10,5,5,10,18,12,14,12,
  14,11,10,13,14,14,14,11,5,5,11,5,5,5,14,11,11,5,5,14,
  11,14,15,14,10,12,11,11,14,5,16,11,5,5,5,5,5,5,5,5,5,
  11,11,5,3,3,3,3,3,3,3,15,15,15,5,5,3,3,3,15,15,5,15,
  5,15,5,15,15,5,5,3,3,3,5,3,11,11,15,15,5,11,11,15,11,
  11,11,3,5,5,16,3,15,16,5,3,3,3,3,5,3,3,3,15,15,5,15,
  3,15,3,3,5,11,15,15,5,15,12,15,5,3,3,3,5,15,11,11,12,
  15,5,11,11,15,12,12,3,5,3,5,5,15,5,5,3,3,3,5,3,15,11,
  11,15,5,11,11,12,15,12,5,16,3,15,16,5,3,3,3,3,5,3,3,3,
  15,15,5,15,3,15,3,3,5,5,5,11,15,5,3,3,3,5,15,11,11,11,
  15,5,5,5,11,11,11,3,15,16,5,3,3,3,3,5,3,12,16,12,15,5,
  16,3,15,3,3,5,15,5,15,3,3,3,5,3,11,15,11,15,5,11,11,
  11,11,15,18,5,5,3,3,3,5,3,13,11,15,16,5,15,5,14,16,12,
  11,3,3,4,3,11,3,16,16,16,16,11,11,15,16,16,16,5,3,3,3,
  11,15,16,11,11,15,11,5,5,11,11,12,3,3,3,5,4,12,14,11,
  15,5,15,5,11,16,14,3,3,3,4,3,3,4,4,3,3,3,4,3,3,3,3,
  4,4,3,3,4,3,3,3,3,4,3,3,4,3,4,3,4,3,3,3,3,3,4,3,13,
  13,14,15,5,5,5,13,14,14,4,4,4,7,3,3,3,4,4,3,15,15,4,
  14,14,14,3,3,3,15,4,14,13,14,3,3,3,16,14,13,13,15,15,
  15,15,15,15,16,16,16,5,5,13,12,13,5,13,12,14,14,14,13,
  15,15,15,5,3,5,15,15,5,5,15,5,15,5,3,3,3,3,5,11,11,15,
  5,15,11,11,15,11,11,5,5,5,5,15,5,5,15,11,5,3,3,3,15,5,
  11,11,11,5,15,5,5,11,11,11,3,4,4,3,3,3,5,4,3,3,3,3,3,
  5,3,3,3,5,15,3,3,16,12,12,3,3,5,5,5,5,15,5,3,3,3,3,
  5,15,11,11,5,15,11,11,11,15,11,5,11,3,15,5,16,3,3,3,3,
  3,5,15,15,15,5,15,15,3,15,15,15,11,3,15,5,16,3,3,3,3,
  3,5,15,15,15,5,15,15,3,15,15,15,15,5,15,14,15,3,3,3,15,
  5,12,16,11,5,15,11,11,11,11,12,5,5,15,15,3,3,3,3,5,11,
  15,11,5,15,11,11,12,11,15,5,18,5,3,3,3,3,5,16,12,14,5,
  16,15,5,15,14,11,11,5,3,3,3,15,11,12,11,11,11,15,5,5,
  11,15,11,3,3,4,3,3,11,16,16,16,11,16,11,15,16,4,16,3,3,
  3,4,5,15,14,11,5,16,15,5,12,12,14,3,3,4,3,3,3,4,3,4,
  3,3,4,3,3,3,4,3,4,3,3,3,4,3,3,3,4,3,3,3,3,3,3,3,4,
  3,3,3,3,4,3,4,3,4,3,7,3,3,4,4,4,14,13,14,5,15,5,5,
  14,13,13,15,15,12,16,14,14,3,3,3,15,14,16,14,14,3,3,3,
  13,16,13,14,3,3,3,15,5,5,14,14,14,15,15,4,4,4,5,14,14,
  13,14,13,13,15,15,15,15,5,15,5,15,5,15,15,5,5,3,5,5,5,
  11,5,5,15,11,5,11,11,15,5,5,15,16,7,15,15,15,15,3,15,
  16,15,15,16,15,15,15,16,15,3,16,16,16,15,15,3,5,5,5,7,
  5,5,5,15,15,5,5,5,11,5,11,5,5,15,15,11,11,5,15,18,3,
  16,15,15,3,15,3,3,15,15,15,15,16,15,15,16,3,16,3,3,5,5,
  15,5,15,5,3,5,5,11,5,5,5,15,5,11,11,11,15,5,5,3,15,15,
  5,3,3,5,3,5,5,5,15,13,15,5,14,14,14,5,5,15,5,5,15,15,
  5,15,5,5,5,11,11,5,5,15,15,11,5,11,15,15,15,16,3,15,15,
  15,16,15,16,15,15,16,7,16,3,15,5,5,3,3,5,5,12,5,5,15,
  12,5,11,11,11,15,5,5,3,3,5,12,5,15,5,11,5,12,11,11,16,
  5,5,15,5,11,5,5,5,11,11,5,5,16,11,11,5,11,15,15,15,16,
  15,15,3,3,3,16,16,16,15,15,5,11,11,3,5,11,11,11,15,15,
  11,15,5,11,11,5,3,11,11,11,15,15,11,5,15,5,10,10,14,5,
  5,11,11,13,10,10,11,10,14,5,5,11,11,14,10,10,5,11,10,10,
  14,14,11,5,5,11,10,10,13,14,11,5,5,12,12,15,11,15,11,11,
  5,11,11,14,11,11,11,11,14,11,10,15,11,14,14,15,12,12,11,
  11,5,5,11,3,15,5,15,15,3,5,3,5,5,5,11,5,5,15,11,5,11,
  11,15,5,5,3,5,15,5,5,7,5,5,5,3,5,5,11,15,5,5,11,5,11,
  11,5,15,5,5,16,3,15,16,5,3,3,3,3,5,3,3,3,15,15,5,15,
  3,15,3,3,15,5,5,15,15,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
  5,7,15,15,15,3,15,15,3,15,18,15,3,15,16,15,16,18,15,15,
  3,5,15,5,15,5,5,3,5,11,5,15,5,11,5,11,11,5,5,15,15,5,
  5,15,5,5,5,5,5,5,12,15,5,15,5,12,5,5,15,15,15,3,15,15,
  3,3,15,15,18,15,18,3,15,3,15,5,5,3,5,10,5,3,5,5,5,10,
  5,5,5,5,5,5,5,15,5,5,5,12,5,15,5,15,5,5,5,12,5,5,11,
  15,5,5,3,15,11,15,15,3,5,5,5,11,15,3,5,5,15,11,15,15,
  5,3,5,11,15,5,3,5,15,11,15,15,5,5,3,5,10,10,10,5,5,5,
  5,10,10,10,11,11,11,5,5,5,5,11,11,11,5,5,11,10,11,11,5,
  5,5,5,14,10,14,11,5,5,5,11,10,11,11,5,5,5,5,5,5,11,11,
  11,5,5,10,10,10,5,11,11,11,11,11,11,5,5,5,3,3,5,5,3,5,
  3,5,15,18,5,5,5,14,5,5,5,15,14,14,15,14,5,4,15,3,16,
  11,4,3,16,3,3,11,11,16,16,16,11,15,16,4,16,16,16,3,3,3,
  15,7,15,3,15,15,3,15,15,15,3,15,3,3,3,3,15,15,3,15,15,
  3,3,3,5,3,11,11,5,16,12,11,11,12,12,18,5,12,15,5,3,15,
  15,5,18,15,5,5,14,5,5,5,14,13,15,5,14,15,3,15,16,3,15,
  15,15,15,15,16,15,3,16,3,16,3,15,3,5,3,3,5,5,12,5,5,
  15,13,5,11,11,15,15,5,3,3,3,15,3,15,3,15,3,15,3,3,3,3,
  3,15,15,5,14,5,5,5,14,5,5,15,16,12,15,5,13,15,15,16,15,
  15,15,7,3,3,16,15,16,15,15,5,13,11,3,5,5,11,16,14,15,
  15,16,5,16,11,5,3,5,12,13,14,14,15,5,16,5,14,14,14,5,5,
  11,3,16,11,11,11,10,11,5,5,10,11,15,10,11,5,5,10,14,11,
  11,15,15,5,5,10,14,12,12,15,5,15,11,14,14,14,16,5,5,5,
  11,10,16,10,11,3,11,15,11,11,13,16,12,14,16,11,12,15,15,
  5,3,3,3,3,3,3,3,3,3,3,3,3,4,3,3,3,3,3,3,3,3,3,5,5,
  5,5,7,5,5,3,5,11,5,15,5,5,5,11,11,11,5,15,5,3,5,3,5,
  3,5,3,3,5,15,13,15,5,5,5,14,13,5,14,5,5,5,5,7,5,15,
  15,5,5,11,11,5,5,5,3,15,5,11,11,5,5,5,15,15,5,5,5,12,
  5,14,5,5,16,3,14,12,5,5,5,5,3,5,5,14,15,5,5,14,5,12,
  12,5,3,5,5,5,3,5,14,5,3,5,5,5,14,12,12,5,3,5,5,15,15,
  5,5,12,14,5,5,5,3,3,5,12,14,15,5,11,11,11,5,3,11,11,
  15,15,15,11,5,15,3,3,3,15,15,15,15,15,16,15,16,15,11,11,
  11,3,5,11,11,15,15,5,11,15,5,12,10,10,5,5,11,11,10,14,
  11,12,10,10,5,5,11,11,10,14,11,11,11,14,13,15,15,11,15,
  11,5,10,11,14,14,5,11,5,10,10,14,12,5,11,5,5,11,11,10,
  12,10,11,11,10,12,10,15,12,15,14,14,11,14,11,5,11,11,3,
  15,5,16,3,3,3,3,3,5,15,15,15,5,15,15,3,15,15,15,3,15,
  16,11,4,3,3,3,11,3,16,16,16,11,11,16,3,16,3,4,3,3,3,7,
  3,3,3,1,1,3,3,3,1,1,3,3,3,3,3,15,15,3,16,3,3,15,15,
  16,15,16,15,15,16,3,16,3,3,5,3,3,3,3,5,3,3,3,15,15,5,
  15,3,12,15,12,3,3,3,3,3,5,15,12,13,5,16,15,3,3,3,3,3,
  3,3,3,3,3,4,3,3,3,7,7,3,3,4,3,3,16,16,16,15,4,3,3,
  16,4,16,3,3,3,3,3,4,3,3,3,3,4,4,3,4,3,3,3,3,4,3,3,
  3,4,4,3,3,4,5,14,14,13,5,5,15,3,14,13,13,14,14,14,5,5,
  15,3,14,13,13,15,15,14,14,16,4,15,3,3,15,14,12,16,4,15,
  3,3,13,13,16,4,15,3,3,5,15,3,14,14,14,15,3,14,14,14,7,
  16,4,4,4,4,4,15,15,15,3,5,15,15,3,5,5,3,11,5,5,16,5,
  5,11,11,11,5,5,5,5,5,15,15,5,5,3,11,5,5,15,5,5,11,11,
  11,5,5,15,5,15,5,15,5,5,5,5,5,5,14,5,15,15,5,14,5,5,
  15,5,5,3,5,5,10,5,5,5,10,5,5,5,5,15,5,15,15,3,15,3,
  15,16,15,15,15,18,18,3,15,3,15,5,5,15,5,5,5,14,5,5,15,
  15,5,5,5,14,5,5,15,11,5,5,7,11,15,15,15,7,5,5,5,15,11,
  7,5,5,11,15,15,15,5,7,5,15,11,5,7,5,11,15,15,15,5,5,7,
  5,11,12,12,5,5,5,5,12,12,11,10,10,10,5,5,5,5,10,10,10,
  5,5,10,12,12,11,5,5,5,5,10,11,11,12,5,5,5,10,11,11,12,
  5,5,5,5,5,5,10,10,11,5,5,12,11,12,5,12,11,12,12,12,11,
  5,5,5,15,3,15,7,15,15,7,15,3,15,4,15,3,15,4,3,15,15,4,
  15,5,3,14,5,15,5,5,5,15,14,15,5,16,12,14,5,5,15,3,15,
  3,3,15,3,3,3,15,16,15,16,3,15,3,3,3,5,3,3,14,5,15,15,
  5,5,11,11,11,5,5,15,15,3,4,15,3,3,4,15,3,15,4,3,15,3,
  3,5,3,11,13,5,15,3,16,11,14,14,3,5,5,3,11,15,3,15,5,
  13,11,14,15,5,3,5,3,3,3,7,3,3,3,4,4,3,3,7,5,10,15,10,
  5,5,10,11,10,10,11,11,16,11,5,5,11,18,14,14,14,15,5,11,
  10,13,13,5,15,5,15,16,16,16,16,15,15,3,11,10,12,12,15,5,
  5,5,16,11,14,14,14,11,10,10,11,11,13,12,14,14,12,11,16,
  5,5,5,5,5,15,11,5,15,5,5,5,11,11,5,5,16,13,11,5,11,5,
  15,5,3,3,5,12,5,16,5,12,5,11,11,5,15,11,15,5,3,3,12,5,
  15,11,5,5,12,11,11,5,5,16,15,3,16,15,15,16,16,15,15,15,
  4,4,15,3,16,5,15,12,11,5,11,3,11,12,15,15,15,5,11,15,
  11,11,3,11,5,11,11,15,15,5,15,11,15,16,15,3,15,3,3,3,
  15,15,15,16,5,10,13,10,5,5,11,11,10,10,12,10,14,10,5,5,
  11,11,10,10,12,11,5,10,10,12,14,5,5,11,11,11,12,15,15,
  11,11,15,10,10,12,14,5,5,11,5,11,11,11,10,14,11,11,10,
  10,13,3,14,14,15,12,12,15,5,11,11,5,5,3,3,5,5,14,5,5,
  14,13,5,15,5,14,15,5,5,3,3,5,14,5,15,5,13,5,13,15,5,
  14,5,5,11,5,11,5,5,5,11,11,5,5,16,5,11,5,11,4,11,16,
  16,11,11,4,16,16,16,4,16,15,11,5,3,3,3,5,3,13,13,16,3,
  3,15,5,12,11,5,15,11,11,11,16,12,11,5,15,5,14,11,14,5,
  5,15,5,12,12,11,14,10,11,5,5,15,5,12,14,11,5,3,11,11,
  16,14,11,5,5,11,10,10,15,11,10,5,5,12,13,16,13,13,11,10,
  5,15,5,12,11,11,15,5,13,11,10,3,16,15,15,14,14,11,3,11,
  5,5,5,3,5,11,11,3,5,5,11,11,5,5,5,3,5,5,3,3,5,14,5,
  14,5,14,5,15,5,5,15,14,3,5,12,14,5,5,3,14,11,12,16,3,
  15,5,3,3,4,3,3,3,4,3,3,3,3,7,3,12,14,5,3,5,15,12,13,
  16,5,15,3,11,11,11,11,10,5,5,5,10,15,11,14,11,12,13,10,
  10,3,11,16,11,5,5,14,11,14,14,5,3,5,5,11,10,13,14,5,15,
  15,11,10,13,14,15,15,5,11,18,10,12,16,12,5,5,10,15,10,5,
  11,15,11,10,15,11,15,5,15,5,3,3,14,5,15,13,5,5,14,15,5,
  5,5,14,3,5,14,12,15,5,3,12,15,16,13,3,5,5,3,4,3,4,3,
  3,3,4,3,3,3,7,3,15,11,15,3,5,11,15,15,13,5,5,3,11,16,
  12,12,10,13,10,3,14,14,12,16,10,10,5,10,5,5,10,11,11,15,
  15,15,16,16,16,15,3,15,5,11,12,11,11,5,5,15,10,11,11,10,
  15,5,5,11,5,5,10,11,11,18,10,12,14,12,5,14,14,12,14,14,
  14,5,5,5,5,11,11,12,5,11,15,12,12,16,12,15,5,11,4,3,3,
  3,3,5,12,12,16,3,5,15,3,16,16,11,4,11,16,16,16,4,11,15,
  16,5,13,11,10,5,15,15,5,11,14,14,14,12,11,15,5,15,5,10,
  13,11,3,5,11,11,16,14,5,5,11,11,13,12,16,14,10,11,13,10,
  10,15,10,5,5,10,5,15,5,11,11,14,15,5,10,11,12,3,16,16,
  16,11,14,16,5,11,3,5,3,3,3,5,3,3,3,15,15,15,15,5,15,
  15,5,3,3,15,15,15,15,15,5,15,15,14,14,4,3,3,15,15,16,
  14,12,14,14,4,3,3,15,15,16,14,14,5,3,13,13,14,14,15,5,
  5,3,13,13,14,13,15,5,5,4,4,4,4,7,3,3,15,15,3,4,14,14,
  15,3,4,14,14,15,16,14,14,16,13,13,15,15,5,3,3,3,3,5,3,
  3,3,15,5,15,15,15,4,13,14,3,3,3,15,13,4,14,4,13,14,3,
  3,3,15,14,4,14,3,3,4,3,4,4,3,7,3,5,14,14,13,14,5,15,
  5,14,13,14,13,5,15,5,15,15,15,12,16,12,15,15,14,16,13,
  15,13,16,13,13,16,14,15,5,15,15,13,3,14,3,3,3,15,14,14,
  16,14,4,14,3,3,3,15,14,12,16,3,5,13,13,14,14,5,5,15,3,
  4,4,4,4,3,3,7,13,13,14,14,5,5,15,15,15,3,14,14,4,15,3,
  13,14,4,3,14,14,4,13,13,16,5,15,15,11,11,11,5,5,5,5,14,
  11,11,11,11,12,12,10,11,11,11,10,12,11,11,11,12,11,11,11,
  11,12,11,12,14,11,11,5,5,5,10,11,12,5,5,14,12,12,5,11,
  11,10,12,12,11,14,11,11,11,11,11,11,11,11,11,11,11,11,11,
  10,11,11,10,12,12,12,11,12,11,11,10,11,5,5,5,12,12,11,5,
  5,11,11,11,5,11,12,11,11,11,11,11,12,11,5,11,11,14,11,5,
  5,5,11,11,11,11,5,5,5,11,11,10,12,12,11,12,11,11,11,11,
  11,11,12,11,12,12,11,5,5,5,12,11,12,12,5,5,5,11,14,11,
  11,11,12,11,11,10,12,12,14,11,10,14,11,11,12,5,5,5,11,
  11,11,11,11,11,11,11,11,10,11,11,12,11,11,11,11,11,5,5,
  5,5,5,11,11,11,5,11,11,11,11,11,10,11,11,11,5,12,11,12,
  11,11,11,11,12,11,12,11,11,11,12,11,11,11,11,5 ],

trsstatus := List(~.trsstatuspos,i->~.trsstatusset[i]),

eqclsgt1 :=
[[1,2,68],[3,7,226,239,472,485,2413,2450,2696,8924,8937],
 [4,10],[5,8,2539],[6,12],[9,11],[23,31,44,1296,1309],
 [24,25,27,28,33,35,46,48,97,98,99,101,108,110,112,114,154,
      155,159,162,172,175,2303,2304,2309,2312,2323,2325,2368,
      2369,2374,2377,2388,2390,4513,4514,4519,4522,4533,4535],
 [26,32,41,42,1422,1423,1433,1435,1613,1622,1623,1657,1660,
      1928,1930,1955,3643,3644,3646,3868,3869,3871,4141,4166],
 [29,37,1545],[30,38],[34,36],[39,54],[40,55],
 [43,47,1982],[45,53],[49,50],[51,52],[56,59],
 [57,58],[60,64],[61,66],[62,67],[63,65],
 [69,73,219,223,465,469,4558,4579,4825,8917,8921],[70,76],
 [71,74,4684],[72,78],[75,77],
 [89,91,92,93,1289,1290,1292,1293,1333,1334,1335,1376,1378,
      1418,1419,1459,5647,5648,5649,5732,5733,5734,5775,5815],
 [90,94,107,1349,1511],[95,96],[100,113],[102,116,1796],
 [103,117],[104,118,1857],[105,120],[106,121],[109,115],
 [111,119],[122,125],[123,124],[126,130],[127,132],
 [128,133],[129,131],[134,138,6638],[135,141],[136,139],
 [137,143],[140,142],
 [156,158,163,166,173,177,1382,1385,1392,1396,1463,1466,1473,
      1477,1651,1662,1757,1957],[157,164,179],[160,183],
 [161,182,1596],[165,180,1732],[167,174],[168,169],
 [170,185],[171,186],[176,184],[178,181],[187,190],
 [188,189],[191,195],[192,197],[193,198],[194,196],
 [199,200,203,205,263,265,266,327,330,449,451,510,8717,8718,
      8720,8779,8780,8782,8901,8903,10732,10733,10735,10856,10915,
      12745,12806,16406],[201,207,388,453,8840,8905],
 [204,206,450,452,8902,8904],[208,454,8906],[209,455,8907],
 [210,456,8908],[211,457,8909],[212,458,8910],
 [213,459,8911],[214,460,8912],[215,461,8913],
 [216,462,8914],[217,463,8915],[218,464,8916],
 [220,222,466,468,8918,8920],[221,467,8919],[224,470,8922],
 [225,471,8923],[227,237,473,483,8925,8935],
 [228,243,474,489,8926,8941],[229,475,8927],
 [230,241,476,487,8928,8939],[231,477,8929],[232,478,8930],
 [233,479,8931],[234,249,480,495,8932,8947],
 [235,250,481,496,8933,8948],[236,482,8934],[238,484,8936],
 [240,248,486,494,8938,8946],[242,488,8940],[244,490,8942],
 [245,491,8943],[246,492,8944],[247,493,8945],
 [251,254,497,500,8949,8952],[252,253,498,499,8950,8951],
 [255,259,501,505,8953,8957],[256,261,502,507,8954,8959],
 [257,262,503,508,8955,8960],[258,260,504,506,8956,8958],
 [264,270,391,627],[267,269,568,626],[271,628],[272,629],
 [273,630],[274,631],[275,632],[276,633],[277,634],
 [278,635],[279,636],[280,637],[281,638],
 [282,286,639,643],[283,285,640,642],[284,641],[287,644],
 [288,645],[289,302,646,659],[290,300,647,657],
 [291,306,648,663],[292,649],[293,304,650,661],[294,651],
 [295,652],[296,653],[297,312,654,669],[298,313,655,670],
 [299,656],[301,658],[303,311,660,668],[305,662],
 [307,664],[308,665],[309,666],[310,667],
 [314,317,671,674],[315,316,672,673],[318,322,675,679],
 [319,324,676,681],[320,325,677,682],[321,323,678,680],
 [326,329,331,332,389,390,392,509,511,512,570,683],
 [333,513],[334,514],[335,515],[336,516],[337,517],
 [338,518],[339,519],[340,520],[341,521],[342,522],
 [343,523],[344,348,524,528],[345,347,525,527],[346,526],
 [349,529],[350,530],[351,364,531,544],[352,362,532,542],
 [353,368,533,548],[354,534],[355,366,535,546],[356,536],
 [357,537],[358,538],[359,374,539,554],[360,375,540,555],
 [361,541],[363,543],[365,373,545,553],[367,547],
 [369,549],[370,550],[371,551],[372,552],
 [376,379,382,387,556,559,562,567,2218,2223,2245,2250],
 [377,378,381,386,557,558,561,566,2227,2232,2236,2241],
 [380,383,384,385,560,563,564,565,2253,2255,2269,2273],
 [394,739],[395,740],[396,741],[397,742],[398,743],
 [399,744],[400,745],[401,746],[402,747],[403,748],
 [404,749],[405,409,750,754],[406,408,751,753],[407,752],
 [410,755],[411,756],[412,425,757,770],[413,423,758,768],
 [414,429,759,774],[415,760],[416,427,761,772],[417,762],
 [418,763],[419,764],[420,435,765,780],[421,436,766,781],
 [422,767],[424,769],[426,434,771,779],[428,773],
 [430,775],[431,776],[432,777],[433,778],
 [437,440,782,785],[438,439,783,784],[441,445,786,790],
 [442,447,787,792],[443,448,788,793],[444,446,789,791],
 [571,684],[572,685],[573,686],[574,687],[575,688],
 [576,689],[577,690],[578,691],[579,692],[580,693],
 [581,694],[582,586,695,699],[583,585,696,698],[584,697],
 [587,700],[588,701],[589,602,702,715],[590,600,703,713],
 [591,606,704,719],[592,705],[593,604,706,717],[594,707],
 [595,708],[596,709],[597,612,710,725],[598,613,711,726],
 [599,712],[601,714],[603,611,716,724],[605,718],
 [607,720],[608,721],[609,722],[610,723],
 [614,617,727,730],[615,616,728,729],[618,622,731,735],
 [619,624,732,737],[620,625,733,738],[621,623,734,736],
 [794,795,848],[797,798,1004],[801,802,1198],[804,808],
 [805,807],[811,824],[812,822],[813,828],[815,826],
 [819,834],[820,835],[825,833],[836,839],[837,838],
 [840,844],[841,846],[842,847],[843,845],[849,850,953],
 [853,854,1151],[857,861],[858,860],[864,877],[865,875],
 [866,881],[868,879],[872,887],[873,888],[878,886],
 [889,892],[890,891],[893,897],[894,899],[895,900],
 [896,898],[901,903,954],[905,907,1152],[909,913],
 [910,912],[916,929],[917,927],[918,933],[920,931],
 [924,939],[925,940],[930,938],[941,944],[942,943],
 [945,949],[946,951],[947,952],[948,950],[955,956,1103],
 [960,964],[961,963],[967,980],[968,978],[969,984],
 [971,982],[975,990],[976,991],[981,989],[992,995],
 [993,994],[996,1000],[997,1002],[998,1003],[999,1001],
 [1005,1007,1104],[1010,1014],[1011,1013],[1017,1030],
 [1018,1028],[1019,1034],[1021,1032],[1025,1040],
 [1026,1041],[1031,1039],[1042,1045],[1043,1044],
 [1046,1050],[1047,1052],[1048,1053],[1049,1051],
 [1054,1057,1105],[1059,1063],[1060,1062],[1066,1079],
 [1067,1077],[1068,1083],[1070,1081],[1074,1089],
 [1075,1090],[1080,1088],[1091,1094],[1092,1093],
 [1095,1099],[1096,1101],[1097,1102],[1098,1100],
 [1107,1111],[1108,1110],[1114,1127],[1115,1125],
 [1116,1131],[1118,1129],[1122,1137],[1123,1138],
 [1128,1136],[1139,1142],[1140,1141],[1143,1147],
 [1144,1149],[1145,1150],[1146,1148],[1154,1158],
 [1155,1157],[1161,1174],[1162,1172],[1163,1178],
 [1165,1176],[1169,1184],[1170,1185],[1175,1183],
 [1186,1189],[1187,1188],[1190,1194],[1191,1196],
 [1192,1197],[1193,1195],[1200,1204],[1201,1203],
 [1207,1220],[1208,1218],[1209,1224],[1211,1222],
 [1215,1230],[1216,1231],[1221,1229],[1232,1235],
 [1233,1234],[1236,1240],[1237,1242],[1238,1243],
 [1239,1241],[1245,1249],[1246,1248],[1252,1265],
 [1253,1263],[1254,1269],[1256,1267],[1260,1275],
 [1261,1276],[1266,1274],[1277,1280],[1278,1279],
 [1281,1285],[1282,1287],[1283,1288],[1284,1286],
 [1291,1306,1432,38304],[1297,1307],
 [1298,1300,1311,1313,1685,1691,1696,1756,1763,1932,1934,2049],
 [1304,1319],[1305,1320],[1310,1318],[1316,1317,2143],
 [1321,1324],[1322,1323],[1325,1329],[1326,1331],
 [1327,1332],[1328,1330],[1337,1338,1499,1500,1538],
 [1339,1343,1352,1354,1501,1505,1514,1516,1616,1627,1759,2004],
 [1340,1341,1350,1356,1502,1503,1512,1518,1649,1664,1692,1959],
 [1342,1504],[1344,1346,1506,1508,1784],[1345,1507],
 [1347,1362,1509,1524],[1348,1363,1510,1525],[1351,1513],
 [1353,1361,1515,1523],[1355,1517],[1357,1519],
 [1358,1360,1520,1522,2127],[1359,1521],
 [1364,1367,1526,1529],[1365,1366,1527,1528],
 [1368,1372,1530,1534],[1369,1374,1531,1536],
 [1370,1375,1532,1537],[1371,1373,1533,1535],[1379,1460],
 [1380,1461],
 [1381,1383,1394,1398,1462,1464,1475,1479,1614,1629,1694,2006],
 [1384,1465],[1386,1402,1467,1483],
 [1387,1388,1400,1401,1468,1469,1481,1482,1814,1826,1856,2126],
 [1389,1390,1404,1405,1470,1471,1485,1486,1873,1888,1914,2188],
 [1391,1472],[1393,1399,1474,1480],[1395,1403,1476,1484],
 [1397,1478],
 [1406,1407,1408,1409,1487,1488,1489,1490,2213,2214,2225,2234],
 [1410,1414,1491,1495],
 [1411,1412,1416,1417,1492,1493,1497,1498,2258,2263,2267,2278],
 [1413,1415,1494,1496],[1424,1439],[1426,1437],
 [1427,1428,1783],[1430,1445],[1431,1446],[1436,1444],
 [1447,1450],[1448,1449],[1451,1455],[1452,1457],
 [1453,1458],[1454,1456],[1539,1552,1555,1628,2005],
 [1540,1542,1544,1550,1650,1652,1659,1725,1726,1788,1789,1954],
 [1541,1556],[1543,1554],[1547,1562],[1548,1563],
 [1553,1561],[1564,1567],[1565,1566],[1568,1572],
 [1569,1574],[1570,1575],[1571,1573],[1576,1589],
 [1577,1587,1654,1847],[1578,1579,1593,1684,1731],
 [1580,1591,1592,1762,2048],[1584,1599],[1585,1600],
 [1590,1598],[1601,1604],[1602,1603],[1605,1609],
 [1606,1611],[1607,1612],[1608,1610],[1615,1727],
 [1617,1790],[1618,1624,1819,1820,1979],[1619,1849],
 [1620,1635,1877,2012],[1621,1636,1904,2013],
 [1626,1634,2003,2011],[1630,1631,2007,2008,2108],
 [1632,2009],[1633,2010],[1637,1640,2014,2017],
 [1638,1639,2015,2016],[1641,1645,2018,2022],
 [1642,1647,2019,2024],[1643,1648,2020,2025],
 [1644,1646,2021,2023],[1653,1663,1818,1958],
 [1655,1661,1669,1670,1875,1878,1886,1956,1964,1965,2034,2174],
 [1656,1671,1902,1966],[1665,1960],[1666,1961],[1667,1962],
 [1668,1963],[1672,1675,1967,1970],
 [1673,1674,1676,1680,1968,1969,1971,1975,2226,2230,2235,2239],
 [1677,1679,1681,1682,1972,1974,1976,1977,2259,2261,2271,2276],
 [1678,1683,1973,1978],[1686,1794],[1687,1824],
 [1688,1693,1848,1853,1983],[1689,1704,1881,2094],
 [1690,1705,1908,2095],[1695,1703,2028,2093],[1697,2069],
 [1699,1701,2089,2091,2109],[1700,2090],[1702,2092],
 [1706,1709,2096,2099],[1707,1708,2097,2098],
 [1710,1714,2100,2104],[1711,1716,2101,2106],
 [1712,1717,2102,2107],[1713,1715,2103,2105],[1718,1729],
 [1722,1737],[1723,1738],[1724,1730,1933],[1728,1736],
 [1739,1742],[1740,1741],[1743,1747],[1744,1749],
 [1745,1750],[1746,1748],[1751,1792],[1752,1822],
 [1753,1851],[1754,1769,1879,2055],
 [1755,1760,1768,1770,1905,1906,1913,2026,2035,2054,2056,2175],
 [1758,1981],[1764,1767,2050,2053,2110],[1765,2051],
 [1766,2052],
 [1771,1774,1775,1779,2057,2060,2061,2065,2216,2220,2243,2247],
 [1772,1773,2058,2059],[1776,1781,2062,2067],
 [1777,1778,1780,1782,2063,2064,2066,2068,2264,2266,2272,2277],
 [1785,1800],[1786,1801],[1791,1799],[1802,1805],
 [1803,1804],[1806,1810],[1807,1812],[1808,1813],
 [1809,1811],[1815,1830],[1816,1831],[1821,1829],
 [1832,1835],[1833,1834],[1836,1840],[1837,1842],
 [1838,1843],[1839,1841],[1844,1859],[1845,1860],
 [1850,1858],[1861,1864],[1862,1863],[1865,1869],
 [1866,1871],[1867,1872],[1868,1870],[1874,1940],
 [1876,1989],[1880,2075],[1882,2112],[1883,2129],
 [1884,2145],[1885,2160],[1889,1892,2189,2192],
 [1890,1891,2190,2191],[1893,1897,2193,2197],
 [1894,1899,2194,2199],[1895,1900,2195,2200],
 [1896,1898,2196,2198],[1901,1941],[1903,1990],[1907,2076],
 [1909,2113],[1910,2130],[1911,2146],[1912,2161],
 [1916,1919,2201,2204],[1917,1918,2202,2203],
 [1920,1924,2205,2209],[1921,1926,2206,2211],
 [1922,1927,2207,2212],[1923,1925,2208,2210],[1931,1939],
 [1942,1945],[1943,1944],[1946,1950],[1947,1952],
 [1948,1953],[1949,1951],[1980,1988],[1991,1994],
 [1992,1993],[1995,1999],[1996,2001],[1997,2002],
 [1998,2000],[2027,2074],[2029,2111],[2030,2128],
 [2031,2144],[2032,2159],[2036,2039,2176,2179],
 [2037,2038,2177,2178],[2040,2044,2180,2184],
 [2041,2046,2181,2186],[2042,2047,2182,2187],
 [2043,2045,2183,2185],[2077,2080],[2078,2079],[2081,2085],
 [2082,2087],[2083,2088],[2084,2086],[2114,2117],
 [2115,2116],[2118,2122],[2119,2124],[2120,2125],
 [2121,2123],[2131,2134],[2132,2133],[2135,2139],
 [2136,2141],[2137,2142],[2138,2140],[2147,2150],
 [2148,2149],[2151,2155],[2152,2157],[2153,2158],
 [2154,2156],[2162,2165],[2163,2164],[2166,2170],
 [2167,2172],[2168,2173],[2169,2171],[2217,2222,2244,2249],
 [2219,2221,2246,2248],[2228,2233,2237,2242],
 [2229,2231,2238,2240],[2251,2256,2260,2274],
 [2252,2257,2265,2275],[2280,2345,4490],[2281,2346,4491],
 [2282,2347,4492],[2283,2348,4493],[2284,2349,4494],
 [2285,2350,4495],[2286,2351,4496],[2287,2352,4497],
 [2288,2353,4498],[2289,2354,4499],[2290,2355,4500],
 [2291,2356,4501],[2292,2357,4502],[2293,2358,4503],
 [2294,2359,4504],[2295,2360,4505],[2296,2361,4506],
 [2297,2362,4507],[2298,2363,4508],[2299,2364,4509],
 [2300,2301,2302,2308,2318,2319,2365,2366,2367,2373,2383,2384,
      3560,3561,3602,3603,4510,4511,4512,4518,4528,4529,5652,5663,
      5737,5748,7732,7742,7775,7785],
 [2305,2310,2321,2370,2375,2386,4515,4520,4531],
 [2306,2307,2314,2315,2328,2329,2371,2372,2379,2380,2393,2394,
      3807,3808,4067,4068,4516,4517,4524,4525,4538,4539,5901,5915,
      5938,5952,7981,7982,8264,8293],
 [2311,2313,2320,2324,2326,2327,2376,2378,2385,2389,2391,2392,
      3943,3944,4006,4007,4521,4523,4530,4534,4536,4537,6082,6088,
      6338,6426,8229,8232,8418,8421],[2316,2381,4526],
 [2317,2382,4527],[2322,2387,4532],[2330,2395,4540],
 [2331,2396,4541],[2332,2397,4542],[2333,2398,4543],
 [2334,2399,4544],[2335,2400,4545],[2336,2401,4546],
 [2337,2402,4547],[2338,2403,4548],[2339,2404,4549],
 [2340,2405,4550],[2341,2406,4551],[2342,2407,4552],
 [2343,2408,4553],[2344,2409,4554],[2410,2411,2474],
 [2414,2416,2721,2867,2868],[2429,2437,9748],[2430,2431],
 [2432,2438],[2433,2434,2439,2441],[2435,2443,9997],
 [2436,2444],[2440,2442],[2447,2448],[2449,2453,10434],
 [2452,2454],[2455,2456],[2457,2458],[2462,2465],
 [2476,2477,2660],[2492,2500,11701],[2493,2494],
 [2495,2501],[2496,2497,2502,2504],[2498,2506,11950],
 [2499,2507],[2503,2505],[2510,2511],[2512,2516,12387],
 [2515,2517],[2518,2519],[2520,2521],[2529,2530],
 [2538,2541,2662,2852,2858],[2542,2543,2894],
 [2554,2562,13592],[2555,2556],[2557,2563],[2558,2566],
 [2559,2564],[2560,2568,13841],[2561,2569],[2565,2567],
 [2572,2573],[2574,2578,14278],[2577,2579],[2580,2581],
 [2582,2583],[2587,2593,4429],[2588,2592,4438],
 [2594,2595,4480],[2600,2601,2720],[2615,2623,15422],
 [2616,2617],[2618,2624],[2619,2627],[2620,2625],
 [2621,2629,15671],[2622,2630],[2626,2628],[2633,2634],
 [2635,2639,16108],[2638,2640],[2641,2642],[2643,2644],
 [2675,2683,17192],[2676,2677],[2678,2684],
 [2679,2680,2685,2687],[2681,2689,17441],[2682,2690],
 [2686,2688],[2693,2694],[2695,2699,17878],[2698,2700],
 [2701,2702],[2703,2704],[2714,2719],[2734,2742,18903],
 [2735,2736],[2737,2743],[2738,2746],[2739,2744],
 [2740,2748,19152],[2741,2749],[2745,2747],[2752,2753],
 [2754,2758,19589],[2757,2759],[2760,2761],[2762,2763],
 [2769,2777,4452],[2770,2778,4461],[2771,2776,4466],
 [2792,2800,20556],[2793,2794],[2795,2801],[2796,2804],
 [2797,2802],[2798,2806,20805],[2799,2807],[2803,2805],
 [2810,2811],[2812,2816,21242],[2815,2817],[2818,2819],
 [2820,2821],[2849,2857,22152],[2850,2851],[2853,2861],
 [2854,2859],[2855,2863,22401],[2856,2864],[2860,2862],
 [2869,2873,22838],[2872,2874],[2875,2876],[2877,2878],
 [2905,2913,23692],[2906,2907],[2908,2914],[2909,2917],
 [2910,2915],[2911,2919,23941],[2912,2920],[2916,2918],
 [2923,2924],[2925,2929,24378],[2928,2930],[2931,2932],
 [2933,2934],[2960,2968,25177],[2961,2962],[2963,2969],
 [2964,2972],[2965,2970],[2966,2974,25426],[2967,2975],
 [2971,2973],[2978,2979],[2980,2984,25863],[2983,2985],
 [2986,2987],[2988,2989],[3005,3006,3059],[3008,3009,3215],
 [3012,3013,3409],[3014,3022,26608],[3015,3016],
 [3017,3023],[3018,3026],[3019,3024],[3020,3028,26857],
 [3021,3029],[3025,3027],[3032,3033],[3034,3038,27294],
 [3037,3039],[3040,3041],[3042,3043],[3060,3061,3164],
 [3064,3065,3362],[3067,3075,27986],[3068,3069],
 [3070,3076],[3071,3079],[3072,3077],[3073,3081,28235],
 [3074,3082],[3078,3080],[3085,3086],[3087,3091,28672],
 [3090,3092],[3093,3094],[3095,3096],[3112,3114,3165],
 [3116,3118,3363],[3119,3127,29312],[3120,3121],
 [3122,3128],[3123,3131],[3124,3129],[3125,3133,29561],
 [3126,3134],[3130,3132],[3137,3138],[3139,3143,29998],
 [3142,3144],[3145,3146],[3147,3148],[3166,3167,3314],
 [3170,3178,30587],[3171,3172],[3173,3179],[3174,3182],
 [3175,3180],[3176,3184,30836],[3177,3185],[3181,3183],
 [3188,3189],[3190,3194,31273],[3193,3195],[3196,3197],
 [3198,3199],[3216,3218,3315],[3220,3228,31812],
 [3221,3222],[3223,3229],[3224,3232],[3225,3230],
 [3226,3234,32061],[3227,3235],[3231,3233],[3238,3239],
 [3240,3244,32498],[3243,3245],[3246,3247],[3248,3249],
 [3265,3268,3316],[3269,3277,32988],[3270,3271],
 [3272,3278],[3273,3281],[3274,3279],[3275,3283,33237],
 [3276,3284],[3280,3282],[3287,3288],[3289,3293,33674],
 [3292,3294],[3295,3296],[3297,3298],[3317,3325,34116],
 [3318,3319],[3320,3326],[3321,3329],[3322,3327],
 [3323,3331,34365],[3324,3332],[3328,3330],[3335,3336],
 [3337,3341,34802],[3340,3342],[3343,3344],[3345,3346],
 [3364,3372,35197],[3365,3366],[3367,3373],[3368,3376],
 [3369,3374],[3370,3378,35446],[3371,3379],[3375,3377],
 [3382,3383],[3384,3388,35883],[3387,3389],[3390,3391],
 [3392,3393],[3410,3418,36232],[3411,3412],[3413,3419],
 [3414,3422],[3415,3420],[3416,3424,36481],[3417,3425],
 [3421,3423],[3428,3429],[3430,3434,36918],[3433,3435],
 [3436,3437],[3438,3439],[3455,3463,37222],[3456,3457],
 [3458,3464],[3459,3467],[3460,3465],[3461,3469,37471],
 [3462,3470],[3466,3468],[3473,3474],[3475,3479,37908],
 [3478,3480],[3481,3482],[3483,3484],
 [3500,3501,3544,3550,3592,38211,38253,39156],
 [3502,3508,3633,3634,3824,38294,38295,38485],
 [3503,3504,3509,3511,3670,3673,3675,3712,3716,3825,3827,3896,
      38334,38336,38338,38373,38375,38377,38486,38488,41737,41739,
      41741,41852,41921,42591,42662,44661],
 [3505,3513,3750,3829,38411,38417,38490,43298],
 [3506,3514,3787,3830,38448,38491],
 [3510,3512,3826,3828,38487,38489],[3515,3831,38492],
 [3516,3832,38493],[3517,3518,3833,3834,4139,38494,38495,38800]
    ,[3519,3523,3835,3839,38496,38500,38854,44958],
 [3520,3836,38497],[3521,3837,38498],
 [3522,3524,3838,3840,4260,38499,38501,45025],
 [3525,3526,3841,3842,4319,38502,38503,38980],
 [3527,3528,3843,3844,4354,38504,38505,45119],
 [3529,3845,38506],[3530,3846,38507],[3531,3847,38508],
 [3532,3848,38509],[3533,3849,38510],[3534,3850,38511],
 [3535,3851,38512],[3536,3852,38513],[3537,3853,38514],
 [3538,3854,38515],[3539,3855,38516],[3540,3856,38517],
 [3541,3857,38518],[3542,3858,38519],[3543,3859,38520],
 [3545,3546,3547,3551,3552,3554,3587,3588,3589,3593,3594,3596,
      3629,3630,3635,3637,3674,3713,3860,3862],
 [3548,3549,3556,3557,3590,3591,3598,3599,3749,3757,3793,4025,
      39320,39357,40181,40182,43262,43494],[3553,3555,3595,3597],
 [3558,3600],[3559,3601],[3562,3566,3604,3608,39757,40618],
 [3563,3565,3567,3605,3607,3609,4215,4217],[3564,3606],
 [3568,3569,3610,3611],[3570,3571,3612,3613],[3572,3614],
 [3573,3574,3615,3616,4399],[3575,3617],[3576,3618],
 [3577,3578,3619,3620,4445],[3579,3621],[3580,3622],
 [3581,3623],[3582,3624],[3583,3625],[3584,3626],
 [3585,3586,3627,3628,4489],
 [3631,3636,3638,3639,3751,3753,3755,3861,3863,3864,3931,3994,
      41000,41001,43333,43400,43463,45291],[3632,3640,3788,3865],
 [3641,3866],[3642,3867],[3645,3649,3870,3874,41438,45553],
 [3647,3872],[3648,3650,3873,3875],[3651,3652,3876,3877],
 [3653,3654,3878,3879],[3655,3656,3880,3881,4385],
 [3657,3882],[3658,3883],[3659,3884],
 [3660,3662,3885,3887,4446],[3661,3886],[3663,3888],
 [3664,3889],[3665,3890],[3666,3891],
 [3667,3668,3892,3893,4487],[3669,3894],
 [3671,3679,3754,3963,41781,43432],[3672,3680,3791,3964],
 [3676,3678,3929,3962],[3681,3682,3965,3966,4084],
 [3683,3684,3688,3690,3722,3723,3727,3729,3902,3903,3907,3909,
      3967,3968,3972,3974,4143,4145,4168,4170],
 [3685,3689,3969,3973,42218,47138],[3686,3970],[3687,3971],
 [3691,3692,3693,3694,3730,3731,3732,3733,3910,3911,3912,3913,
      3975,3976,3977,3978,4320,4321,4337,4338],
 [3695,3697,3979,3981,4386],[3696,3980],
 [3698,3699,3982,3983,4424],[3700,3984],
 [3701,3702,3985,3986,4454],[3703,3704,3987,3988,4469],
 [3705,3989],[3706,3990],[3707,3709,3991,3993,4488],
 [3708,3992],[3710,3718,3752,3898,42522,43367],
 [3711,3715,3717,3719,3789,3790,3792,3895,3897,3899,3932,3995],
 [3720,3900],[3721,3901],[3724,3728,3904,3908,42959,46114],
 [3725,3905],[3726,3906],[3734,3914],[3735,3915],
 [3736,3916],[3737,3917],[3738,3918],[3739,3919],
 [3740,3920],[3741,3921],[3742,3922],[3743,3923],
 [3744,3924],[3745,3925],[3746,3926],[3747,3927],
 [3748,3928],[3758,4026,43495],[3759,4027,43496],
 [3760,3761,4028,4029,43497,43498],
 [3762,3763,3766,4030,4031,4034,4190,4216,43499,43500,43503,
      43659,43662,48038,48061,50097],[3764,4032,43501],
 [3765,3767,4033,4035,43502,43504],
 [3768,3769,4036,4037,43505,43506],
 [3770,3771,4038,4039,43507,43508],[3772,4040,43509],
 [3773,4041,43510],[3774,4042,43511],[3775,4043,43512],
 [3776,4044,43513],[3777,4045,43514],[3778,4046,43515],
 [3779,4047,43516],[3780,4048,43517],[3781,4049,43518],
 [3782,4050,43519],[3783,4051,43520],[3784,4052,43521],
 [3785,4053,43522],[3786,4054,43523],[3795,4055],
 [3796,4056],[3797,3798,4057,4058],
 [3799,3802,3803,3804,4059,4062,4063,4064,4192,4194,4259,4280,
      44327,44328,48444,48531,50140,50161],[3800,4060],
 [3801,4061],[3805,3806,4065,4066],[3809,4069],[3810,4070],
 [3811,4071],[3812,4072],[3813,4073],[3814,4074],
 [3815,4075],[3816,4076],[3817,4077],[3818,4078],
 [3819,4079],[3820,4080],[3821,4081],[3822,4082],
 [3823,4083],[3933,3996],[3934,3997],
 [3935,3936,3937,3941,3998,3999,4000,4004,4140,4144,4165,4169,
      46589,46642,47579,47603,49498,49798],[3938,4001],
 [3939,4002],[3940,3942,4003,4005],[3945,3946,4008,4009],
 [3947,4010],[3948,4011],[3949,4012],[3950,4013],
 [3951,4014],[3952,4015],[3953,4016],[3954,4017],
 [3955,4018],[3956,4019],[3957,4020],[3958,4021],
 [3959,4022],[3960,4023],[3961,4024],
 [4085,4086,4089,4142,4167],[4087,4091,48822],[4090,4092],
 [4093,4094],[4095,4096],[4112,4113],[4114,4118,49173],
 [4116,4117,4119,4237,4239],[4120,4121],[4122,4123],
 [4146,4147,4171,4172],[4148,4149,4173,4174],[4150,4175],
 [4151,4176],[4152,4177],[4153,4178],
 [4154,4161,4179,4186,4441],[4155,4180],[4156,4181],
 [4157,4182],[4158,4160,4183,4185,4470],[4159,4184],
 [4162,4187],[4163,4188],[4164,4189],[4191,4238,50119],
 [4195,4196,4281,4282,50162,50163],
 [4197,4198,4283,4284,50164,50165],[4199,4285,50166],
 [4200,4286,50167],[4201,4287,50168],[4202,4288,50169],
 [4203,4289,50170],[4204,4290,50171],[4205,4291,50172],
 [4206,4292,50173],[4207,4293,50174],[4208,4294,50175],
 [4209,4295,50176],[4210,4296,50177],[4211,4297,50178],
 [4212,4298,50179],[4213,4299,50180],[4218,4219],
 [4220,4221],[4240,4241],[4242,4243],[4261,4262,4300,4301],
 [4263,4264,4302,4303],[4265,4304],[4266,4305],[4267,4306],
 [4268,4276,4307,4315,4431],[4269,4308],[4270,4309],
 [4271,4310],[4272,4311],[4273,4312],
 [4274,4275,4313,4314,4475],[4277,4316],[4278,4317],
 [4279,4318],[4322,4339],[4323,4340],[4324,4341],
 [4325,4342],[4326,4343],[4327,4344],[4328,4345],
 [4329,4346],[4330,4347],[4331,4348],[4332,4349],
 [4333,4350],[4334,4351],[4335,4352],[4336,4353],
 [4355,4370],[4356,4371],[4357,4372],[4358,4373],
 [4359,4374],[4360,4375],[4361,4376],[4362,4377],
 [4363,4378],[4364,4379],[4365,4380],[4366,4381],
 [4367,4382],[4368,4383],[4369,4384],[4555,4556,4619],
 [4559,4561,4866,4997,4998],[4574,4576],[4575,4592,9801],
 [4577,4578],[4580,4581],[4582,4593],[4583,4584,4595,4597],
 [4585,4598],[4586,4599],[4587,4601,10248],[4588,4602],
 [4589,4603,10309],[4594,4600],[4611,4615],[4621,4622,4805],
 [4637,4639],[4638,4655,11754],[4640,4641],[4643,4644],
 [4645,4656],[4646,4647,4658,4660],[4648,4661],[4649,4662],
 [4650,4664,12201],[4651,4665],[4652,4666,12262],
 [4657,4663],[4672,4676],[4683,4686,4807,4994,4996],
 [4687,4688,5039],[4699,4701],[4700,4717,13645],
 [4702,4703],[4705,4706],[4707,4718],[4708,4722],
 [4709,4720],[4710,4723],[4711,4724],[4712,4726,14092],
 [4713,4727],[4714,4728,14153],[4719,4725],[4732,4738,6574],
 [4733,4737,6583],[4739,4740,6625],[4745,4746,4865],
 [4760,4762],[4761,4778,15475],[4763,4764],[4766,4767],
 [4768,4779],[4769,4783],[4770,4781],[4771,4784],
 [4772,4785],[4773,4787,15922],[4774,4788],
 [4775,4789,15983],[4780,4786],[4820,4822],
 [4821,4838,17245],[4823,4824],[4826,4827],[4828,4839],
 [4829,4830,4841,4843],[4831,4844],[4832,4845],
 [4833,4847,17692],[4834,4848],[4835,4849,17753],
 [4840,4846],[4860,4862],[4879,4881],[4880,4897,18956],
 [4882,4883],[4885,4886],[4887,4898],[4888,4902],
 [4889,4900],[4890,4903],[4891,4904],[4892,4906,19403],
 [4893,4907],[4894,4908,19464],[4899,4905],[4914,4922,6597],
 [4915,4923,6606],[4916,4921,6611],[4937,4939],
 [4938,4955,20609],[4940,4941],[4943,4944],[4945,4956],
 [4946,4960],[4947,4958],[4948,4961],[4949,4962],
 [4950,4964,21056],[4951,4965],[4952,4966,21117],
 [4957,4963],[4995,5012,22205],[5000,5001],[5002,5013],
 [5003,5017],[5004,5015],[5005,5018],[5006,5019],
 [5007,5021,22652],[5008,5022],[5009,5023,22713],
 [5014,5020],[5050,5052],[5051,5068,23745],[5053,5054],
 [5056,5057],[5058,5069],[5059,5073],[5060,5071],
 [5061,5074],[5062,5075],[5063,5077,24192],[5064,5078],
 [5065,5079,24253],[5070,5076],[5105,5107],
 [5106,5123,25230],[5108,5109],[5111,5112],[5113,5124],
 [5114,5128],[5115,5126],[5116,5129],[5117,5130],
 [5118,5132,25677],[5119,5133],[5120,5134,25738],
 [5125,5131],[5150,5151,5204],[5153,5154,5360],
 [5157,5158,5554],[5159,5161],[5160,5177,26661],
 [5162,5163],[5165,5166],[5167,5178],[5168,5182],
 [5169,5180],[5170,5183],[5171,5184],[5172,5186,27108],
 [5173,5187],[5174,5188,27169],[5179,5185],[5205,5206,5309],
 [5209,5210,5507],[5212,5214],[5213,5230,28039],
 [5215,5216],[5218,5219],[5220,5231],[5221,5235],
 [5222,5233],[5223,5236],[5224,5237],[5225,5239,28486],
 [5226,5240],[5227,5241,28547],[5232,5238],[5257,5259,5310],
 [5261,5263,5508],[5264,5266],[5265,5282,29365],
 [5267,5268],[5270,5271],[5272,5283],[5273,5287],
 [5274,5285],[5275,5288],[5276,5289],[5277,5291,29812],
 [5278,5292],[5279,5293,29873],[5284,5290],[5311,5312,5459],
 [5315,5317],[5316,5333,30640],[5318,5319],[5321,5322],
 [5323,5334],[5324,5338],[5325,5336],[5326,5339],
 [5327,5340],[5328,5342,31087],[5329,5343],
 [5330,5344,31148],[5335,5341],[5361,5363,5460],
 [5365,5367],[5366,5383,31865],[5368,5369],[5371,5372],
 [5373,5384],[5374,5388],[5375,5386],[5376,5389],
 [5377,5390],[5378,5392,32312],[5379,5393],
 [5380,5394,32373],[5385,5391],[5410,5413,5461],
 [5414,5416],[5415,5432,33041],[5417,5418],[5420,5421],
 [5422,5433],[5423,5437],[5424,5435],[5425,5438],
 [5426,5439],[5427,5441,33488],[5428,5442],
 [5429,5443,33549],[5434,5440],[5462,5464],
 [5463,5480,34169],[5465,5466],[5468,5469],[5470,5481],
 [5471,5485],[5472,5483],[5473,5486],[5474,5487],
 [5475,5489,34616],[5476,5490],[5477,5491,34677],
 [5482,5488],[5509,5511],[5510,5527,35250],[5512,5513],
 [5515,5516],[5517,5528],[5518,5532],[5519,5530],
 [5520,5533],[5521,5534],[5522,5536,35697],[5523,5537],
 [5524,5538,35758],[5529,5535],[5555,5557],
 [5556,5573,36285],[5558,5559],[5561,5562],[5563,5574],
 [5564,5578],[5565,5576],[5566,5579],[5567,5580],
 [5568,5582,36732],[5569,5583],[5570,5584,36793],
 [5575,5581],[5600,5602],[5601,5618,37275],[5603,5604],
 [5606,5607],[5608,5619],[5609,5623],[5610,5621],
 [5611,5624],[5612,5625],[5613,5627,37722],[5614,5628],
 [5615,5629,37783],[5620,5626],
 [5645,5646,5662,5689,5747,38221,38263,39166],
 [5650,5651,5735,5736],
 [5653,5654,5656,5665,5667,5669,5738,5739,5741,5750,5752,5754,
      6007,6020,6041,6054,6115,6117,6362,6405],
 [5655,5668,5740,5753],[5657,5671,5742,5756,38668,40432],
 [5658,5659,5672,5673,5743,5744,5757,5758,6170,6184,6212,6499,
      38728,38729,40464,40493,47913,48605],[5660,5745],
 [5661,5746],[5664,5670,5749,5755],[5666,5751],[5674,5759],
 [5675,5676,5760,5761,6544],[5677,5762],[5678,5763],
 [5679,5680,5764,5765,6590],[5681,5766],[5682,5767],
 [5683,5768],[5684,5769],[5685,5770],[5686,5771],
 [5687,5688,5772,5773,6634],
 [5690,5691,5774,5788,5828,39207,39247,40928],
 [5692,5867,39286],[5693,5694,5894,5905,5942,39324,39361,43266]
    ,[5695,5706,5978,5979,6284,39397,39398,39703],
 [5696,5697,5708,5710,6005,6013,6016,6047,6052,6286,6288,6360,
      39432,39435,39437,39466,39469,39471,39705,39707,45262,45265,
      45267,45503,45575,46062,46136,49520],
 [5698,5711,6080,6086,6289,39499,39505,39708],
 [5699,5712,6112,6119,6290,39531,39538,39709],
 [5700,5714,6143,6292,39562,39571,39711,47557],
 [5701,5715,6173,6293,39592,39712],
 [5702,5716,6202,6294,39621,39632,39713,48400],
 [5703,6230,39649],[5704,6257,39676],
 [5707,5713,6285,6291,39704,39710],[5709,6287,39706],
 [5717,6295,39714],[5718,6296,39715],[5719,6297,39716],
 [5720,6298,39717],[5721,6299,39718],[5722,6300,39719],
 [5723,6301,39720],[5724,6302,39721],[5725,6303,39722],
 [5726,6304,39723],[5727,6305,39724],[5728,6306,39725],
 [5729,6307,39726],[5730,6308,39727],[5731,6309,39728],
 [5776,5777,5816,5817],
 [5778,5779,5780,5789,5791,5793,5818,5819,5820,5829,5831,5833,
      5969,5970,5981,5983,6014,6048,6311,6313],
 [5781,5794,5821,5834],[5782,5795,5822,5835],
 [5783,5784,5797,5798,5823,5824,5837,5838,6139,6153,6182,6482,
      41252,41282,42032,42033,47447,47747],
 [5785,5799,5825,5839,41313,42093],[5786,5787,5826,5827,6229],
 [5790,5796,5830,5836],[5792,5832],[5800,5840],[5801,5841],
 [5802,5842],[5803,5804,5843,5844,6569],[5805,5845],
 [5806,5846],[5807,5847],[5808,5809,5848,5849,6614],
 [5810,5850],[5811,5851],[5812,5852],[5813,5853],
 [5814,5854],[5855,5856],
 [5857,5861,5868,5874,5972,5985,6113,6315],[5858,5872],
 [5859,5870],[5860,5873],
 [5862,5864,5876,5878,6140,6154,6211,6483,42773,42775,42832,
      42834,47476,47478,47748,48589],[5863,5877],[5869,5875],
 [5895,5906,5932,5943],
 [5896,5897,5898,5908,5910,5911,5933,5934,5935,5945,5947,5948,
      6006,6019,6040,6053,6083,6085,6361,6404],
 [5899,5912,5936,5949],[5900,5914,5937,5951,43476,44142],
 [5902,5916,5939,5953,43537,44203],[5903,5940],[5904,5941],
 [5907,5913,5944,5950],[5909,5946],[5917,5954],[5918,5955],
 [5919,5956],[5920,5957],[5921,5958],[5922,5959],
 [5923,5960],[5924,5961],[5925,5962],[5926,5963],
 [5927,5964],[5928,5965],[5929,5966],[5930,5967],
 [5931,5968],[5971,5984,6081,6314],
 [5973,5980,5986,5987,6144,6145,6151,6310,6316,6317,6341,6464,
      44771,44772,47582,47606,47729,49801],[5974,5988,6174,6318],
 [5975,5989,6203,6319,44833,48425],[5976,5982,6231,6234,6312],
 [5977,6258],[5990,6320],[5991,6321],[5992,6322],
 [5993,6323],[5994,6001,6324,6331,6586],[5995,6325],
 [5996,6326],[5997,6327],[5998,6000,6328,6330,6615],
 [5999,6329],[6002,6332],[6003,6333],[6004,6334],
 [6008,6022,6148,6407,45367,47672],[6009,6023,6178,6408],
 [6010,6024,6207,6409,45428,48515],[6011,6235],
 [6012,6017,6261,6262,6382],[6015,6021,6337,6406],
 [6025,6026,6410,6411,6530],[6027,6412],
 [6028,6036,6413,6421,6576],[6029,6414],
 [6030,6032,6415,6417,6591],[6031,6416],[6033,6418],
 [6034,6035,6419,6420,6620],[6037,6038,6422,6423,6632],
 [6039,6424],[6042,6056,6146,6364,45928,47629],
 [6043,6049,6055,6057,6175,6176,6181,6335,6342,6363,6365,6465],
 [6044,6058,6205,6366,45989,48472],[6045,6233],[6046,6260],
 [6051,6359],[6059,6367],[6060,6368],[6061,6369],
 [6062,6370],[6063,6371],[6064,6372],[6065,6373],
 [6066,6374],[6067,6375],[6068,6376],[6069,6377],
 [6070,6378],[6071,6379],[6072,6380],[6073,6381],
 [6074,6087,6118,6425],[6075,6089,6149,6427,46456,47692],
 [6076,6090,6179,6428],[6077,6091,6208,6429,46517,48535],
 [6078,6236],[6079,6263],[6084,6383],[6092,6430],
 [6093,6431],[6094,6432],[6095,6433],[6096,6434],
 [6097,6435],[6098,6436],[6099,6437],[6100,6438],
 [6101,6439],[6102,6440],[6103,6441],[6104,6442],
 [6105,6443],[6106,6444],[6107,6121,6150,6446,46952,47711],
 [6108,6122,6180,6447],
 [6109,6114,6120,6123,6204,6209,6210,6339,6343,6445,6448,6466,
      47010,47013,48449,48554,48572,50184],[6110,6237],
 [6111,6264],[6116,6384],[6124,6126,6449,6451,6531],
 [6125,6450],[6127,6452],[6128,6453],[6129,6454],
 [6130,6131,6455,6456,6599],[6132,6457],[6133,6458],
 [6134,6459],[6135,6460],[6136,6138,6461,6463,6633],
 [6137,6462],[6141,6239,47504],[6142,6266,47531],
 [6147,6386,47651],[6155,6484,47749],[6156,6485,47750],
 [6157,6486,47751],[6158,6487,47752],[6159,6488,47753],
 [6160,6489,47754],[6161,6490,47755],[6162,6491,47756],
 [6163,6492,47757],[6164,6493,47758],[6165,6494,47759],
 [6166,6495,47760],[6167,6496,47761],[6168,6497,47762],
 [6169,6498,47763],[6171,6240],[6172,6267],[6177,6387],
 [6185,6500],[6186,6501],[6187,6502],[6188,6503],
 [6189,6504],[6190,6505],[6191,6506],[6192,6507],
 [6193,6508],[6194,6509],[6195,6510],[6196,6511],
 [6197,6512],[6198,6513],[6199,6514],[6200,6241,48347],
 [6201,6268,48374],[6206,6388,48494],[6214,6515,48621],
 [6215,6516,48622],[6216,6517,48623],[6217,6518,48624],
 [6218,6519,48625],[6219,6520,48626],[6220,6521,48627],
 [6221,6522,48628],[6222,6523,48629],[6223,6524,48630],
 [6224,6525,48631],[6225,6526,48632],[6226,6527,48633],
 [6227,6528,48634],[6228,6529,48635],[6232,6238],
 [6259,6265],[6336,6385],[6344,6467],[6345,6468],
 [6346,6469],[6347,6470],[6348,6471],[6349,6472],
 [6350,6473],[6351,6474],[6352,6475],[6353,6476],
 [6354,6477],[6355,6478],[6356,6479],[6357,6480],
 [6358,6481],[6635,6636,6699],[6639,6641,6946],[6654,6655],
 [6656,6673,9844],[6657,6679],[6658,6677],[6659,6675],
 [6660,6683],[6661,6682,10048],[6662,6672],[6663,6666],
 [6665,6680,10184],[6667,6674],[6668,6669],[6678,6681],
 [6701,6702,6885],[6717,6718],[6719,6736,11797],
 [6720,6742],[6721,6740],[6722,6738],[6723,6746],
 [6724,6745,12001],[6725,6735],[6726,6729],
 [6728,6743,12137],[6730,6737],[6731,6732],[6741,6744],
 [6763,6766,6887],[6767,6768,7119],[6779,6780],
 [6781,6798,13688],[6782,6784,6800,6804],[6783,6802],
 [6785,6808],[6786,6807,13892],[6787,6797],[6788,6791],
 [6790,6805,14028],[6792,6799],[6793,6794],[6803,6806],
 [6810,6811],[6812,6818,8654],[6813,6817,6819,6820,8663,8705]
    ,[6814,6815],[6822,6823],[6825,6826,6945],[6840,6841],
 [6842,6859,15518],[6843,6865],[6844,6863,15599,15603,16083],
 [6845,6861],[6846,6869],[6847,6868,15722],[6848,6858],
 [6849,6852],[6851,6866,15858],[6853,6860],[6854,6855],
 [6864,6867],[6900,6901],[6902,6919,17288],[6903,6925],
 [6904,6923],[6905,6921],[6906,6929],[6907,6928,17492],
 [6908,6918],[6909,6912],[6911,6926,17628],[6913,6920],
 [6914,6915],[6924,6927],[6959,6960],[6961,6978,18999],
 [6962,6964,6980,6984],[6963,6982],[6965,6988],
 [6966,6987,19203],[6967,6977],[6968,6971],
 [6970,6985,19339],[6972,6979],[6973,6974],[6975,6981],
 [6983,6986],[6993,7000],[6994,6995,7002,7003,8677,8686],
 [6996,7001,8691],[6997,6999],[7017,7018],[7019,7036,20652],
 [7020,7042],[7021,7040],[7022,7038],[7023,7046],
 [7024,7045,20856],[7025,7035],[7026,7029],
 [7028,7043,20992],[7030,7037],[7031,7032],[7041,7044],
 [7074,7075,7078,7097],[7076,7093,22248],[7077,7099,22295],
 [7079,7095],[7080,7103],[7081,7102,22452],
 [7082,7083,7086,7092],[7085,7100,22588],
 [7087,7088,7089,7094],[7098,7101],[7130,7131],
 [7132,7149,23788],[7133,7155],[7134,7153],[7135,7151],
 [7136,7159],[7137,7158,23992],[7138,7148],
 [7139,7142,23778,23781,24047],[7141,7156,24128],[7143,7150],
 [7144,7145],[7154,7157],[7185,7186],[7187,7204,25273],
 [7188,7210],[7189,7208],[7190,7206],[7191,7214],
 [7192,7213,25477],[7193,7203],[7194,7197],
 [7196,7211,25613],[7198,7205],[7199,7200],[7209,7212],
 [7230,7231,7284],[7233,7234,7440],[7237,7238,7634],
 [7239,7240],[7241,7258,26704],[7242,7264],[7243,7262],
 [7244,7260],[7245,7268],[7246,7267,26908],[7247,7257],
 [7248,7251],[7250,7265,27044],[7252,7259],[7253,7254],
 [7263,7266],[7285,7286,7389],[7289,7290,7587],[7292,7293],
 [7294,7311,28082],[7295,7317],[7296,7315],[7297,7313],
 [7298,7321],[7299,7320,28286],[7300,7310],[7301,7304],
 [7303,7318,28422],[7305,7312],[7306,7307],[7316,7319],
 [7337,7339,7390],[7341,7343,7588],[7344,7345],
 [7346,7363,29408],[7347,7369],[7348,7367],[7349,7365],
 [7350,7373],[7351,7372,29612],[7352,7362],[7353,7356],
 [7355,7370,29748],[7357,7364],[7358,7359],[7360,7366],
 [7368,7371],[7375,7376],[7391,7392,7539],[7395,7396],
 [7397,7414,30683],[7398,7420],[7399,7418],[7400,7416],
 [7401,7424],[7402,7423,30887],[7403,7413],[7404,7407],
 [7406,7421,31023],[7408,7415],[7409,7410],[7419,7422],
 [7441,7443,7540],[7445,7446],[7447,7464,31908],
 [7448,7470],[7449,7468],[7450,7466],[7451,7474],
 [7452,7473,32112],[7453,7463],[7454,7457],
 [7456,7471,32248],[7458,7465],[7459,7460],[7461,7467],
 [7469,7472],[7476,7477],[7490,7493,7541],[7494,7495],
 [7496,7513,33084],[7497,7519],[7498,7517],[7499,7515],
 [7500,7523],[7501,7522,33288],[7502,7512],[7503,7506],
 [7505,7520,33424],[7507,7514],[7508,7509],[7518,7521],
 [7542,7543],[7544,7561,34212],[7545,7567],[7546,7565],
 [7547,7563],[7548,7571],[7549,7570,34416],[7550,7560],
 [7551,7554],[7553,7568,34552],[7555,7562],[7556,7557],
 [7566,7569],[7589,7590],[7591,7608,35293],[7592,7614],
 [7593,7612],[7594,7610],[7595,7618],[7596,7617,35497],
 [7597,7607],[7598,7601],[7600,7615,35633],[7602,7609],
 [7603,7604],[7605,7611],[7613,7616],[7620,7621],
 [7635,7636],[7637,7654,36328],[7638,7660],[7639,7658],
 [7640,7656],[7641,7664],[7642,7663,36532],[7643,7653],
 [7644,7647],[7646,7661,36668],[7648,7655],[7649,7650],
 [7659,7662],[7680,7681],[7682,7699,37318],[7683,7705],
 [7684,7703],[7685,7701],[7686,7709],[7687,7708,37522],
 [7688,7698],[7689,7692],[7691,7706,37658],[7693,7700],
 [7694,7695],[7696,7702],[7704,7707],[7711,7712],
 [7715,7716],[7718,7720],
 [7725,7726,7743,7769,7786,38222,38264,39167],
 [7727,7728,7729,7745,7747,7749,7770,7771,7772,7788,7790,7792,
      7854,7873,7895,7911,7915,7952,8440,8485],
 [7730,7731,7752,7753,7773,7774,7795,7796,7974,7995,8033,8579,
      38468,38469,39334,39371,43276,44489],
 [7733,7734,7736,7776,7777,7779,8085,8121],
 [7735,7750,7778,7793,38604,39507],[7737,7744,7780,7787],
 [7738,7739,7781,7782],[7740,7746,7783,7789],[7741,7784],
 [7748,7751,7791,7794],[7754,7797],[7755,7756,7798,7799],
 [7757,7800],[7758,7765,7801,7808],[7759,7760,7802,7803],
 [7761,7804],[7762,7764,7805,7807],[7763,7806],[7766,7809],
 [7767,7768,7810,7811],
 [7812,7814,7830,7834,7855,7869,7871,7948,7954,8391,8395,8442,
      40069,40071,40075,40148,40150,40154,40591,40595,40968,40970,
      40974,41415,41462,42932,42983,49822],
 [7813,7832,7909,7913,8393,40109,40113,40593],
 [7815,7838,7986,8399,40186,40599],
 [7816,7837,8023,8398,40223,40232,40598,44308],
 [7817,7827,8058,8059,8364,40258,40259,40564],
 [7818,7821,8087,8094,8193,40294,40393,45328],
 [7819,8128,40328],[7820,7835,8161,8396,40361,40368,40596,46620 
    ],[7822,7829,8224,8225,8390,40424,40425,40590],
 [7823,7824,8250,8254,8283,40454,40483,47903],
 [7825,7831,8311,8314,8392,40511,40514,40592],
 [7826,8338,40538],[7833,7836,8394,8397,40594,40597],
 [7839,8400,40600],[7840,7841,8401,8402,8624,40601,40602,50004]
    ,[7842,8403,40603],
 [7843,7850,8404,8411,8666,40604,40611,40866],
 [7844,7845,8405,8406,8670,40605,40606,50050],
 [7846,8407,40607],[7847,7849,8408,8410,8695,40608,40610,40895]
    ,[7848,8409,40609],[7851,8412,40612],
 [7852,7853,8413,8414,8714,40613,40614,50094],
 [7856,7879,7992,8528],[7857,7878,8029,8527,41052,44437],
 [7858,7859,7862,7868,7937,7938,7941,7947,8049,8052,8061,8065,
      8093,8096,8100,8192,8195,8199,8366,8370],[7860,8134],
 [7861,7876,8167,8525,41188,46749],
 [7863,7864,7865,7870,7942,7943,7944,7949,8219,8220,8226,8230,
      8255,8256,8260,8284,8285,8289,8415,8419],[7866,8317],
 [7867,7946,8340,8344],[7872,8464],[7874,7877,8505,8526],
 [7880,7959,8447,8529],[7881,8530],[7882,8531],
 [7883,7962,8450,8532],[7884,8533],[7885,8534],[7886,8535],
 [7887,7966,8454,8536],[7888,8537],[7889,7968,8456,8538],
 [7890,8539],[7891,8540],[7892,7971,8459,8541],[7893,8542],
 [7894,8543],[7896,7919,7990,8489],
 [7897,7914,7917,7918,8027,8028,8031,8484,8487,8488,8508,8562,
      41831,41832,44398,44418,44472,50858],
 [7898,7900,7908,8050,8063,8127,8132,8368],
 [7899,7902,8098,8197],[7901,7916,8165,8486,41968,46710],
 [7903,7910,8228,8417],[7904,7905,8258,8287],
 [7906,7907,7912,8309,8315,8341,8342,8462],[7920,8490],
 [7921,7922,8491,8492],
 [7923,7924,7931,8493,8494,8501,8649,8656],
 [7925,7926,8495,8496],[7927,8497],
 [7928,7929,7930,8498,8499,8500,8694,8700],[7932,8502],
 [7933,7934,8503,8504],
 [7935,7953,7956,7958,7988,7991,7994,8441,8444,8446,8509,8563],
 [7936,7957,8025,8445,42573,44355],[7939,8130],
 [7940,7955,8163,8443,42709,46667],[7945,8313],[7951,8439],
 [7960,8448],[7961,8449],[7963,8451],[7964,8452],
 [7965,8453],[7967,8455],[7969,8457],[7970,8458],
 [7972,8460],[7973,8461],[7975,7985,8069,8374],
 [7976,7978,7979,7993,8086,8101,8104,8154,8171,8200,8203,8546,
      43345,43412,45303,46424,46427,46770],[7977,8138],
 [7980,7987,8234,8423],[7983,8321],[7984,8348],[7989,8468],
 [7997,8595],[7998,8596],[7999,8597],[8000,8598],
 [8001,8599],[8002,8600],[8003,8601],[8004,8602],
 [8005,8603],[8006,8604],[8007,8605],[8008,8606],
 [8009,8607],[8010,8608],[8011,8609],
 [8012,8022,8068,8373,43978,44283],
 [8013,8016,8103,8202,44013,44112],
 [8014,8015,8030,8120,8135,8137,8170,8545,44045,44047,44078,
      44080,44455,45864,45866,46769],
 [8017,8024,8233,8422,44143,44332],
 [8018,8019,8263,8292,44173,44202],
 [8020,8026,8320,8467,44230,44377],[8021,8347,44257],
 [8034,8580,44490],[8035,8581,44491],[8036,8582,44492],
 [8037,8583,44493],[8038,8584,44494],[8039,8585,44495],
 [8040,8586,44496],[8041,8587,44497],[8042,8588,44498],
 [8043,8589,44499],[8044,8590,44500],[8045,8591,44501],
 [8046,8592,44502],[8047,8048,8593,8594,44503,44504],
 [8051,8064,8066,8067,8160,8166,8169,8369,8371,8372,8506,8544,
      44708,44709,46595,46730,46768,49586],[8053,8060,8223,8365],
 [8054,8055,8253,8282],[8056,8062,8310,8367],[8057,8337],
 [8070,8375],[8071,8072,8376,8377],[8073,8378],
 [8074,8081,8379,8386],[8075,8076,8380,8381],[8077,8382],
 [8078,8080,8383,8385],[8079,8384],[8082,8387],
 [8083,8084,8388,8389],[8088,8095,8187,8194],
 [8089,8090,8188,8189],[8091,8097,8190,8196],[8092,8191],
 [8099,8102,8198,8201],
 [8105,8106,8107,8204,8205,8206,8610,8611],[8108,8207],
 [8109,8116,8208,8215],
 [8110,8111,8112,8209,8210,8211,8671,8679],
 [8113,8115,8212,8214],[8114,8213],
 [8117,8118,8119,8216,8217,8218,8712,8713],[8122,8129],
 [8123,8124],[8133,8136],[8155,8162,8231,8420,46455,46644],
 [8156,8157,8261,8290,46485,46514],[8158,8318,46542],
 [8159,8345,46569],[8164,8465,46689],[8172,8547,46771],
 [8173,8174,8548,8549,46772,46773],[8175,8550,46774],
 [8176,8183,8551,8558,46775,46782],[8177,8552,46776],
 [8178,8553,46777],[8179,8554,46778],[8180,8555,46779],
 [8181,8556,46780],[8182,8557,46781],[8184,8559,46783],
 [8185,8560,46784],[8186,8561,46785],[8221,8227,8312,8416],
 [8222,8339],[8235,8424],[8236,8237,8425,8426],[8238,8427],
 [8239,8246,8428,8435],[8240,8241,8429,8430],[8242,8431],
 [8243,8245,8432,8434],[8244,8433],[8247,8436],
 [8248,8249,8437,8438],[8251,8257,8280,8286],[8252,8281],
 [8259,8262,8288,8291],[8265,8294],[8266,8267,8295,8296],
 [8268,8297],[8269,8276,8298,8305],[8270,8271,8299,8300],
 [8272,8301],[8273,8275,8302,8304],[8274,8303],[8277,8306],
 [8278,8279,8307,8308],[8316,8319],[8325,8472],[8331,8478],
 [8343,8346],[8353,8360],[8357,8359],[8463,8466],
 [8510,8564],[8511,8565],[8512,8566],[8513,8567],
 [8514,8568],[8515,8569],[8516,8570],[8517,8571],
 [8518,8572],[8519,8573],[8520,8574],[8521,8575],
 [8522,8576],[8523,8577],[8524,8578],[8614,8615],
 [8622,8623],[8629,8641],[8634,8646],[8653,8655],
 [8664,8701],[8676,8684],[8692,8693],[8716,8778,10731],
 [8719,8781,10734],[8721,8722,8783,8784,9135,10736,10737,11088]
    ,[8723,8785,10738],[8724,8786,10739],[8725,8787,10740],
 [8726,8788,10741],[8727,8789,10742],[8728,8790,10743],
 [8729,8791,10744],[8730,8792,10745],[8731,8793,10746],
 [8732,8794,10747],[8733,8795,10748],[8734,8796,10749],
 [8735,8797,10750],[8736,8798,10751],[8737,8799,10752],
 [8738,8800,10753],[8739,8801,10754],[8740,8802,10755],
 [8741,8803,10756],[8742,8804,10757],[8743,8805,10758],
 [8744,8806,10759],[8745,8807,10760],[8746,8808,10761],
 [8747,8809,10762],[8748,8810,10763],[8749,8811,10764],
 [8750,8812,10765],[8751,8813,10766],[8752,8814,10767],
 [8753,8815,10768],[8754,8816,10769],[8755,8817,10770],
 [8756,8818,10771],[8757,8819,10772],[8758,8820,10773],
 [8759,8821,10774],[8760,8822,10775],[8761,8823,10776],
 [8762,8824,10777],[8763,8825,10778],[8764,8826,10779],
 [8765,8827,10780],[8766,8772,8828,8834,10670,10781,10787,12623 
    ],[8767,8771,8829,8833,10679,10782,10786,12632],
 [8768,8830,10783],[8769,8831,10784],[8770,8832,10785],
 [8773,8774,8835,8836,10721,10788,10789,12674],
 [8775,8837,10790],[8776,8838,10791],[8777,8839,10792],
 [8841,8842,8843,8961,9020,14636,14694,18175],
 [8963,9078,18233],[8964,9079,18234],[8965,9080,18235],
 [8966,9081,18236],[8967,9082,18237],[8968,9083,18238],
 [8969,9084,18239],[8970,9085,18240],[8971,9086,18241],
 [8972,9087,18242],[8973,9088,18243],[8974,9089,18244],
 [8975,9090,18245],[8976,9091,18246],[8977,9092,18247],
 [8978,9093,18248],[8979,9094,18249],[8980,9095,18250],
 [8981,9096,18251],[8982,9097,18252],[8983,9098,18253],
 [8984,9099,18254],[8985,9100,18255],[8986,9101,18256],
 [8987,9102,18257],[8988,9103,18258],[8989,9104,18259],
 [8990,9105,18260],[8991,9106,18261],[8992,9107,18262],
 [8993,9108,18263],[8994,9109,18264],[8995,9110,18265],
 [8996,9111,18266],[8997,9112,18267],[8998,9113,18268],
 [8999,9114,18269],[9000,9115,18270],[9001,9116,18271],
 [9002,9117,18272],[9003,9118,18273],[9004,9119,18274],
 [9005,9120,18275],[9006,9121,18276],[9007,9122,18277],
 [9008,9123,18278],[9009,9124,18279],
 [9010,9018,9125,9133,10693,18280,18288,23097],
 [9011,9019,9126,9134,10702,18281,18289,23106],
 [9012,9017,9127,9132,10707,18282,18287,23111],
 [9013,9128,18283],[9014,9129,18284],[9015,9130,18285],
 [9016,9131,18286],[9246,9247,9300],[9249,9250,9456],
 [9253,9254,9650],[9301,9302,9405],[9305,9306,9603],
 [9353,9355,9406],[9357,9359,9604],[9407,9408,9555],
 [9457,9459,9556],[9506,9509,9557],[9741,9742,9785],
 [9744,9745,9911],[9750,9752,10137],[9763,9765,10501],
 [9768,9769,10595],[9786,9787,9870],[9789,9790,9990],
 [9792,9793,10101],[9804,9806,10456],[9828,9830,9871],
 [9834,9837,10103],[9839,9840,10266],[9846,9850,10458],
 [9856,9857,10640],[9860,9861,10686],[9868,9869,10730],
 [9874,9875,10065],[9879,9880,10235],[9885,9887,10407],
 [9914,9916,10066],[9922,9923,10325],[9925,9929,10409],
 [9933,9934,10578],[9939,9940,10665],[9944,9945,10710],
 [9953,9957,10068],[9958,9960,10236],[9964,9970,10411],
 [9972,9974,10579],[9992,9994,10102],[10004,10007,10457],
 [10030,10031,10136],[10043,10044,10500],[10074,10075,10380],
 [10082,10083,10560],[10109,10112,10382],[10121,10122,10626],
 [10126,10128,10687],[10133,10134,10728],[10143,10148,10384],
 [10151,10153,10561],[10176,10182,10385],[10208,10215,10386],
 [10216,10219,10562],[10220,10222,10627],[10226,10227,10695],
 [10232,10234,10729],[10240,10241,10406],[10271,10272,10431],
 [10300,10305,10435],[10327,10330,10408],[10357,10358,10478],
 [10420,10427,10682],[10424,10426,10711],[10509,10517,10672],
 [10515,10516,10716],
 [10793,10794,10795,10855,10914,14575,14576,16405],
 [10857,10916,16407],[10858,10917,16408],[10859,10918,16409],
 [10860,10919,16410],[10861,10920,16411],[10862,10921,16412],
 [10863,10922,16413],[10864,10923,16414],[10865,10924,16415],
 [10866,10925,16416],[10867,10926,16417],[10868,10927,16418],
 [10869,10928,16419],[10870,10929,16420],[10871,10930,16421],
 [10872,10931,16422],[10873,10932,16423],[10874,10933,16424],
 [10875,10934,16425],[10876,10935,16426],[10877,10936,16427],
 [10878,10937,16428],[10879,10938,16429],[10880,10939,16430],
 [10881,10940,16431],[10882,10941,16432],[10883,10942,16433],
 [10884,10943,16434],[10885,10944,16435],[10886,10945,16436],
 [10887,10946,16437],[10888,10947,16438],[10889,10948,16439],
 [10890,10949,16440],[10891,10950,16441],[10892,10951,16442],
 [10893,10952,16443],[10894,10953,16444],[10895,10954,16445],
 [10896,10955,16446],[10897,10956,16447],[10898,10957,16448],
 [10899,10958,16449],[10900,10959,16450],[10901,10960,16451],
 [10902,10961,16452],[10903,10962,16453],
 [10904,10912,10963,10971,12646,16454,16462,18137],
 [10905,10913,10964,10972,12655,16455,16463,18146],
 [10906,10911,10965,10970,12660,16456,16461,18151],
 [10907,10966,16457],[10908,10967,16458],[10909,10968,16459],
 [10910,10969,16460],[11199,11200,11253],[11202,11203,11409],
 [11206,11207,11603],[11254,11255,11358],[11258,11259,11556],
 [11306,11308,11359],[11310,11312,11557],[11360,11361,11508],
 [11410,11412,11509],[11459,11462,11510],[11694,11695,11738],
 [11697,11698,11864],[11703,11705,12090],[11716,11718,12454],
 [11721,11722,12548],[11739,11740,11823],[11742,11743,11943],
 [11745,11746,12054],[11757,11759,12409],[11781,11783,11824],
 [11787,11790,12056],[11792,11793,12219],[11799,11803,12411],
 [11809,11810,12593],[11813,11814,12639],[11821,11822,12683],
 [11827,11828,12018],[11832,11833,12188],[11838,11840,12360],
 [11867,11869,12019],[11875,11876,12278],[11878,11882,12362],
 [11886,11887,12531],[11892,11893,12618],[11897,11898,12663],
 [11906,11910,12021],[11911,11913,12189],[11917,11923,12364],
 [11925,11927,12532],[11945,11947,12055],[11957,11960,12410],
 [11983,11984,12089],[11996,11997,12453],[12027,12028,12333],
 [12035,12036,12513],[12062,12065,12335],[12074,12075,12579],
 [12079,12081,12640],[12086,12087,12681],[12096,12101,12337],
 [12104,12106,12514],[12129,12135,12338],[12161,12168,12339],
 [12169,12172,12515],[12173,12175,12580],[12179,12180,12648],
 [12185,12187,12682],[12193,12194,12359],[12224,12225,12384],
 [12253,12258,12388],[12280,12283,12361],[12310,12311,12431],
 [12373,12380,12635],[12377,12379,12664],[12462,12470,12625],
 [12468,12469,12669],[12684,12687,14577],[12685,12686,12805],
 [12688,12689,14809],[12733,12739,16344],[12734,12738,16353],
 [12740,12741,16395],[12746,12864,16464],
 [12748,12749,12922,12923,16522,16523,16579,21539],
 [12750,12924,16524],[12751,12925,16525],[12752,12926,16526],
 [12753,12927,16527],[12754,12928,16528],[12755,12929,16529],
 [12756,12930,16530],[12757,12931,16531],[12758,12932,16532],
 [12759,12933,16533],[12760,12934,16534],[12761,12935,16535],
 [12762,12936,16536],[12763,12937,16537],[12764,12938,16538],
 [12765,12939,16539],[12766,12940,16540],[12767,12941,16541],
 [12768,12942,16542],[12769,12943,16543],[12770,12944,16544],
 [12771,12945,16545],[12772,12946,16546],[12773,12947,16547],
 [12774,12948,16548],[12775,12949,16549],[12776,12950,16550],
 [12777,12951,16551],[12778,12952,16552],[12779,12953,16553],
 [12780,12954,16554],[12781,12955,16555],[12782,12956,16556],
 [12783,12957,16557],[12784,12958,16558],[12785,12959,16559],
 [12786,12960,16560],[12787,12961,16561],[12788,12962,16562],
 [12789,12963,16563],[12790,12964,16564],[12791,12965,16565],
 [12792,12966,16566],
 [12793,12799,12967,12973,16567,16573,18114,23074],
 [12794,12798,12968,12972,16568,16572,18123,23083],
 [12795,12969,16569],[12796,12970,16570],[12797,12971,16571],
 [12800,12801,12974,12975,16574,16575,18165,23125],
 [12802,12976,16576],[12803,12977,16577],[12804,12978,16578],
 [12807,12808,18290],[12836,19509],[12841,19083],
 [12849,13699],[12851,13967],[12852,12858,19825],
 [12853,12857,19834],[12854,12862,14537],[12855,12863,14546],
 [12856,12861,14551],[12859,12860,19876],[12865,12866,19943],
 [12910,12916,21478],[12911,12915,21487],[12917,12918,21529],
 [12980,13035,23135],[12981,13036,23136],[12982,13037,23137],
 [12983,13038,23138],[12984,13039,23139],[12985,13040,23140],
 [12986,13041,23141],[12987,13042,23142],[12988,13043,23143],
 [12989,13044,23144],[12990,13045,23145],[12991,13046,23146],
 [12992,13047,23147],[12993,13048,23148],[12994,13049,23149],
 [12995,13050,23150],[12996,13051,23151],[12997,13052,23152],
 [12998,13053,23153],[12999,13054,23154],[13000,13055,23155],
 [13001,13056,23156],[13002,13057,23157],[13003,13058,23158],
 [13004,13059,23159],[13005,13060,23160],[13006,13061,23161],
 [13007,13062,23162],[13008,13063,23163],[13009,13064,23164],
 [13010,13065,23165],[13011,13066,23166],[13012,13067,23167],
 [13013,13068,23168],[13014,13069,23169],[13015,13070,23170],
 [13016,13071,23171],[13017,13072,23172],[13018,13073,23173],
 [13019,13074,23174],[13020,13075,23175],[13021,13076,23176],
 [13022,13077,23177],
 [13023,13029,13078,13084,23178,23184,24614,26099],
 [13024,13028,13079,13083,23179,23183,24623,26108],
 [13025,13080,23180],[13026,13081,23181],[13027,13082,23182],
 [13030,13031,13085,13086,23185,23186,24665,26150],
 [13032,13087,23187],[13033,13088,23188],[13034,13089,23189],
 [13090,13091,13144],[13093,13094,13300],[13097,13098,13494],
 [13132,13138,27530],[13133,13137,27539],[13139,13140,27581],
 [13145,13146,13249],[13149,13150,13447],[13185,13191,28908],
 [13186,13190,28917],[13192,13193,28959],[13197,13199,13250],
 [13201,13203,13448],[13237,13243,30234],[13238,13242,30243],
 [13244,13245,30285],[13251,13252,13399],[13288,13294,31509],
 [13289,13293,31518],[13295,13296,31560],[13301,13303,13400],
 [13338,13344,32734],[13339,13343,32743],[13345,13346,32785],
 [13350,13353,13401],[13387,13393,33910],[13388,13392,33919],
 [13394,13395,33961],[13435,13441,35038],[13436,13440,35047],
 [13442,13443,35089],[13482,13488,36119],[13483,13487,36128],
 [13489,13490,36170],[13528,13534,37154],[13529,13533,37163],
 [13535,13536,37205],[13573,13579,38144],[13574,13578,38153],
 [13580,13581,38195],[13585,13586,13629],[13588,13589,13755],
 [13594,13596,13981],[13607,13609,14345],[13612,13613,14439],
 [13617,13623,39090],[13618,13622,39099],[13624,13625,39141],
 [13630,13631,13714],[13633,13634,13834],[13636,13637,13945],
 [13648,13650,14300],[13660,13666,39993],[13661,13665,40002],
 [13667,13668,40044],[13672,13674,13715],[13678,13681,13947],
 [13683,13684,14110],[13690,13694,14302],[13700,13701,14484],
 [13702,13708,40854],[13703,13707,40863],[13704,13705,14530],
 [13709,13710,40905],[13712,13713,14574],[13718,13719,13909],
 [13723,13724,14079],[13729,13731,14251],[13743,13749,41674],
 [13744,13748,41683],[13750,13751,41725],[13758,13760,13910],
 [13766,13767,14169],[13769,13773,14253],[13777,13778,14422],
 [13783,13784,13788,13789,14509,14513,14524,14554,42453,42454,
      42463,42464,52178,52179,52209,52254],[13790,13791,42505],
 [13797,13801,13912],[13802,13804,14080],[13808,13814,14255],
 [13816,13818,14423],[13822,13828,43195],[13823,13827,43204],
 [13829,13830,43246],[13836,13838,13946],[13848,13851,14301],
 [13860,13866,43898],[13861,13865,43907],[13867,13868,43949],
 [13874,13875,13980],[13881,42562],[13887,13888,14344],
 [13897,13903,44564],[13898,13902,44573],[13904,13905,44615],
 [13918,13919,14224],[13926,13927,14404],[13933,13939,45194],
 [13934,13938,45203],[13940,13941,45245],[13953,13956,14226],
 [13965,13966,14470],[13968,13974,45789],[13969,13973,45798],
 [13970,13972,14531],[13975,13976,45840],[13977,13978,14572],
 [13987,13992,14228],[13995,13997,14405],[14002,14008,46350],
 [14003,14007,46359],[14009,14010,46401],[14020,14026,14229],
 [14035,14041,46878],[14036,14040,46887],[14042,14043,46929],
 [14052,14059,14230],[14060,14063,14406],[14064,14066,14471],
 [14067,14073,47374],[14068,14072,47383],[14070,14071,14539],
 [14074,14075,47425],[14076,14078,14573],[14081,45917],
 [14084,14085,14250],[14098,14104,47839],[14099,14103,47848],
 [14105,14106,47890],[14115,14116,14275],[14128,14134,48274],
 [14129,14133,48283],[14135,14136,48325],[14144,14149,14279],
 [14157,14163,48680],[14158,14162,48689],[14164,14165,48731],
 [14171,14174,14252],[14179,48848],[14185,14191,49058],
 [14186,14190,49067],[14192,14193,49109],[14201,14202,14322],
 [14207,49200],[14212,14218,49409],[14213,14217,49418],
 [14219,14220,49460],[14238,14244,49734],[14239,14243,49743],
 [14245,14246,49785],[14263,14269,50034],
 [14264,14268,14270,14271,14525,14526,14555,14556,50043,50045,
      50076,50085,52255,52256,52265,52365],[14276,46112],
 [14287,14293,50310],[14288,14292,50319],[14294,14295,50361],
 [14310,14316,50563],[14311,14315,50572],[14317,14318,50614],
 [14323,43004],[14332,14338,50794],[14333,14337,50803],
 [14339,14340,50845],
 [14353,14359,14360,14361,14515,14516,14560,14561,51004,51005,
      51051,51055,52215,52216,52220,52375],[14354,14358,51013],
 [14373,14379,51194],[14374,14378,51203],[14380,14381,51245],
 [14392,14398,51365],[14393,14397,51374],[14399,14400,51416],
 [14410,14416,51518],[14411,14415,51527],[14417,14418,51569],
 [14427,14433,51654],[14428,14432,51663],[14434,14435,51705],
 [14443,14449,51774],[14444,14448,51783],[14450,14451,51825],
 [14458,14464,51879],[14459,14463,51888],[14465,14466,51930],
 [14472,14478,51970],[14473,14477,51979],[14479,14480,52021],
 [14485,14491,52048],[14486,14490,52057],[14492,14493,52099],
 [14497,14503,52114],[14498,14502,52123],[14504,14505,52165],
 [14510,14533,52188],[14511,14541,52196],[14512,14548,52203],
 [14517,14562,52217],[14518,14563,52218],[14519,14564,52219],
 [14520,14532,52232],[14521,14540,52240],[14522,14547,52247],
 [14527,14557,52257],[14528,14558,52258],[14529,14559,52259],
 [14534,14535,52301],[14542,14543,52329],[14549,14550,52350],
 [14566,14569,52385],[14567,14570,52386],[14568,14571,52387],
 [14637,14695,18176],[14638,14696,18177],[14639,14697,18178],
 [14640,14698,18179],[14641,14699,18180],[14642,14700,18181],
 [14643,14701,18182],[14644,14702,18183],[14645,14703,18184],
 [14646,14704,18185],[14647,14705,18186],[14648,14706,18187],
 [14649,14707,18188],[14650,14708,18189],[14651,14709,18190],
 [14652,14710,18191],[14653,14711,18192],[14654,14712,18193],
 [14655,14713,18194],[14656,14714,18195],[14657,14715,18196],
 [14658,14716,18197],[14659,14717,18198],[14660,14718,18199],
 [14661,14719,18200],[14662,14720,18201],[14663,14721,18202],
 [14664,14722,18203],[14665,14723,18204],[14666,14724,18205],
 [14667,14725,18206],[14668,14726,18207],[14669,14727,18208],
 [14670,14728,18209],[14671,14729,18210],[14672,14730,18211],
 [14673,14731,18212],[14674,14732,18213],[14675,14733,18214],
 [14676,14734,18215],[14677,14735,18216],[14678,14736,18217],
 [14679,14737,18218],[14680,14738,18219],[14681,14739,18220],
 [14682,14740,18221],[14683,14741,18222],
 [14684,14692,14742,14750,16367,18223,18231,21501],
 [14685,14693,14743,14751,16376,18224,18232,21510],
 [14686,14691,14744,14749,16381,18225,18230,21515],
 [14687,14745,18226],[14688,14746,18227],[14689,14747,18228],
 [14690,14748,18229],[14920,14921,14974],[14923,14924,15130],
 [14927,14928,15324],[14975,14976,15079],[14979,14980,15277],
 [15027,15029,15080],[15031,15033,15278],[15081,15082,15229],
 [15131,15133,15230],[15180,15183,15231],[15415,15416,15459],
 [15418,15419,15585],[15424,15426,15811],[15437,15439,16175],
 [15442,15443,16269],[15460,15461,15544],[15463,15464,15664],
 [15466,15467,15775],[15478,15480,16130],[15502,15504,15545],
 [15508,15511,15777],[15513,15514,15940],[15520,15524,16132],
 [15530,15531,16314],[15534,15535,16360],[15542,15543,16404],
 [15548,15549,15739],[15553,15554,15909],[15559,15561,16081],
 [15563,15642],[15564,15643],[15568,15647],
 [15588,15590,15740],[15596,15597,15999],[15607,15608,16252],
 [15613,15614,16339],[15618,15619,16384],[15627,15631,15742],
 [15632,15634,15910],[15638,15644,16085],[15646,15648,16253],
 [15666,15668,15776],[15678,15681,16131],[15704,15705,15810],
 [15717,15718,16174],[15748,15749,16054],[15756,15757,16234],
 [15783,15786,16056],[15785,15819],[15788,15887],
 [15789,15888],[15792,16134],[15793,15892],
 [15795,15796,16300],[15800,15802,16361],[15807,15808,16402],
 [15817,15822,16058],[15825,15827,16235],[15826,16177],
 [15850,15856,16059],[15882,15889,16060],[15890,15893,16236],
 [15891,16216],[15894,15896,16301],[15900,15901,16369],
 [15906,15908,16403],[15914,15915,16080],[15945,15946,16105],
 [15974,15979,16109],[16001,16004,16082],[16019,16020],
 [16031,16032,16152],[16042,16045],[16094,16101,16356],
 [16098,16100,16385],[16135,16217],[16164,16170],
 [16183,16191,16346],[16185,16186],[16189,16190,16390],
 [16193,16194],[16213,16214],[16690,16691,16744],
 [16693,16694,16900],[16697,16698,17094],[16745,16746,16849],
 [16749,16750,17047],[16797,16799,16850],[16801,16803,17048],
 [16851,16852,16999],[16901,16903,17000],[16950,16953,17001],
 [17185,17186,17229],[17188,17189,17355],[17194,17196,17581],
 [17207,17209,17945],[17212,17213,18039],[17230,17231,17314],
 [17233,17234,17434],[17236,17237,17545],[17248,17250,17900],
 [17272,17274,17315],[17278,17281,17547],[17283,17284,17710],
 [17290,17294,17902],[17300,17301,18084],[17304,17305,18130],
 [17312,17313,18174],[17318,17319,17509],[17323,17324,17679],
 [17329,17331,17851],[17358,17360,17510],[17366,17367,17769],
 [17369,17373,17853],[17377,17378,18022],[17383,17384,18109],
 [17388,17389,18154],[17397,17401,17512],[17402,17404,17680],
 [17408,17414,17855],[17416,17418,18023],[17436,17438,17546],
 [17448,17451,17901],[17474,17475,17580],[17487,17488,17944],
 [17518,17519,17824],[17526,17527,18004],[17553,17556,17826],
 [17565,17566,18070],[17570,17572,18131],[17577,17578,18172],
 [17587,17592,17828],[17595,17597,18005],[17620,17626,17829],
 [17652,17659,17830],[17660,17663,18006],[17664,17666,18071],
 [17670,17671,18139],[17676,17678,18173],[17684,17685,17850],
 [17715,17716,17875],[17744,17749,17879],[17771,17774,17852],
 [17801,17802,17922],[17864,17871,18126],[17868,17870,18155],
 [17953,17961,18116],[17959,17960,18160],[18336,18344,24637],
 [18337,18345,24646],[18338,18343,24651],[18391,18399,26122],
 [18392,18400,26131],[18393,18398,26136],[18401,18402,18455],
 [18404,18405,18611],[18408,18409,18805],[18445,18453,27553],
 [18446,18454,27562],[18447,18452,27567],[18456,18457,18560],
 [18460,18461,18758],[18498,18506,28931],[18499,18507,28940],
 [18500,18505,28945],[18508,18510,18561],[18512,18514,18759],
 [18550,18558,30257],[18551,18559,30266],[18552,18557,30271],
 [18562,18563,18710],[18601,18609,31532],[18602,18610,31541],
 [18603,18608,31546],[18612,18614,18711],[18651,18659,32757],
 [18652,18660,32766],[18653,18658,32771],[18661,18664,18712],
 [18700,18708,33933],[18701,18709,33942],[18702,18707,33947],
 [18748,18756,35061],[18749,18757,35070],[18750,18755,35075],
 [18795,18803,36142],[18796,18804,36151],[18797,18802,36156],
 [18841,18849,37177],[18842,18850,37186],[18843,18848,37191],
 [18886,18894,38167],[18887,18895,38176],[18888,18893,38181],
 [18896,18897,18940],[18899,18900,19066],[18905,18907,19292],
 [18918,18920,19656],[18923,18924,19750],[18930,18938,39113],
 [18931,18939,39122],[18932,18937,39127],[18941,18942,19025],
 [18944,18945,19145],[18947,18948,19256],[18959,18961,19611],
 [18973,18981,40016],[18974,18982,40025],[18975,18980,40030],
 [18983,18985,19026],[18989,18992,19258],[18994,18995,19421],
 [19001,19005,19613],[19011,19012,19795],
 [19015,19016,19023,19024,19841,19849,19856,19885,40877,40878,
      40885,40886,52281,52282,52310,52338],[19017,19022,40891],
 [19029,19030,19220],[19034,19035,19390],[19040,19042,19562],
 [19056,19064,41697],[19057,19065,41706],[19058,19063,41711],
 [19069,19071,19221],[19077,19078,19480],[19080,19084,19564],
 [19088,19089,19733],[19094,19095,19820],[19096,19104,42477],
 [19097,19105,42486],[19098,19103,42491],[19099,19100,19865],
 [19108,19112,19223],[19113,19115,19391],[19119,19125,19566],
 [19127,19129,19734],[19135,19143,43218],[19136,19144,43227],
 [19137,19142,43232],[19147,19149,19257],[19159,19162,19612],
 [19169,42539],[19173,19181,43921],[19174,19182,43930],
 [19175,19180,43935],[19185,19186,19291],[19198,19199,19655],
 [19210,19218,44587],[19211,19219,44596],[19212,19217,44601],
 [19229,19230,19535],[19237,19238,19715],[19246,19254,45217],
 [19247,19255,45226],[19248,19253,45231],[19264,19267,19537],
 [19276,19277,19781],
 [19281,19283,19288,19289,19842,19847,19863,19883,45811,45812,
      45826,45827,52287,52288,52308,52357],[19282,19290,45821],
 [19298,19303,19539],[19306,19308,19716],[19315,19323,46373],
 [19316,19324,46382],[19317,19322,46387],[19331,19337,19540],
 [19343,42713],[19348,19356,46901],[19349,19357,46910],
 [19350,19355,46915],[19363,19370,19541],[19371,19374,19717],
 [19375,19377,19782],[19380,19388,47397],
 [19381,19382,19387,19389,19850,19855,19864,19884,47404,47406,
      47411,47413,52315,52317,52337,52358],[19395,19396,19561],
 [19411,19419,47862],[19412,19420,47871],[19413,19418,47876],
 [19426,19427,19586],[19437,45962],[19441,19449,48297],
 [19442,19450,48306],[19443,19448,48311],[19455,19460,19590],
 [19467,45992],[19470,19478,48703],[19471,19479,48712],
 [19472,19477,48717],[19482,19485,19563],[19498,19506,49081],
 [19499,19507,49090],[19500,19505,49095],[19512,19513,19633],
 [19525,19533,49432],[19526,19534,49441],[19527,19532,49446],
 [19551,19559,49757],[19552,19560,49766],[19553,19558,49771],
 [19575,19582,19837],[19576,19584,50057],[19577,19585,50066],
 [19578,19583,50071],[19579,19581,19866],[19600,19608,50333],
 [19601,19609,50342],[19602,19607,50347],[19623,19631,50586],
 [19624,19632,50595],[19625,19630,50600],[19645,19653,50817],
 [19646,19654,50826],[19647,19652,50831],[19664,19672,19827],
 [19666,19674,51027],[19667,19675,51036],[19668,19673,51041],
 [19670,19671,19871],[19686,19694,51217],[19687,19695,51226],
 [19688,19693,51231],[19705,19713,51388],[19706,19714,51397],
 [19707,19712,51402],[19718,50456],[19723,19731,51541],
 [19724,19732,51550],[19725,19730,51555],[19740,19748,51677],
 [19741,19749,51686],[19742,19747,51691],[19756,19764,51797],
 [19757,19765,51806],[19758,19763,51811],[19768,50506],
 [19771,19779,51902],[19772,19780,51911],[19773,19778,51916],
 [19785,19793,51993],[19786,19794,52002],[19787,19792,52007],
 [19798,19806,52071],[19799,19807,52080],[19800,19805,52085],
 [19810,19818,52137],[19811,19819,52146],[19812,19817,52151],
 [19821,19829,52192],[19822,19830,52201],[19823,19828,52206],
 [19831,19839,52237],[19832,19840,52246],[19833,19838,52251],
 [19843,19869,52294],[19844,19874,52299],[19845,19878,52303],
 [19846,19881,52306],[19851,19870,52323],[19852,19875,52328],
 [19853,19879,52332],[19854,19882,52335],[19858,19868,52342],
 [19859,19873,52347],[19860,19877,52351],[19861,19880,52354],
 [20054,20055,20108],[20057,20058,20264],[20061,20062,20458],
 [20109,20110,20213],[20113,20114,20411],[20161,20163,20214],
 [20165,20167,20412],[20215,20216,20363],[20265,20267,20364],
 [20314,20317,20365],[20549,20550,20593],[20552,20553,20719],
 [20558,20560,20945],[20571,20573,21309],[20576,20577,21403],
 [20594,20595,20678],[20597,20598,20798],[20600,20601,20909],
 [20612,20614,21264],[20636,20638,20679],[20640,20760],
 [20642,20645,20911],[20647,20648,21074],[20654,20658,21266],
 [20656,21021],[20661,21351],[20664,20665,21448],
 [20668,20669,21494],[20676,20677,21538],[20682,20683,20873],
 [20687,20688,21043],[20693,20695,21215],[20697,20887],
 [20698,20923],[20702,21222],[20722,20724,20874],
 [20730,20731,21133],[20733,20737,21217],[20738,20957],
 [20741,20742,21386],[20747,20748,21473],[20752,20753,21518],
 [20761,20765,20876],[20766,20768,21044],[20772,20778,21219],
 [20777,21022],[20780,20782,21387],[20800,20802,20910],
 [20812,20815,21265],[20814,20989],[20819,21332],
 [20836,20837],[20838,20839,20944],[20847,20849],
 [20851,20852,21308],[20882,20883,21188],[20890,20891,21368],
 [20917,20920,21190],[20926,21025],[20927,21197],
 [20929,20930,21434],[20934,20936,21495],[20941,20942,21536],
 [20946,21011],[20951,20956,21192],[20959,20961,21369],
 [20984,20990,21193],[21016,21023,21194],[21024,21027,21370],
 [21028,21030,21435],[21034,21035,21503],[21040,21042,21537],
 [21048,21049,21214],[21052,21054],[21079,21080,21239],
 [21108,21113,21243],[21135,21138,21216],[21150,21160],
 [21165,21166,21286],[21182,21186],[21228,21235,21490],
 [21232,21234,21519],[21268,21350],[21303,21305],
 [21317,21325,21480],[21319,21321],[21323,21324,21524],
 [21326,21327],[21393,21394],[21650,21651,21704],
 [21653,21654,21860],[21657,21658,22054],[21705,21706,21809],
 [21709,21710,22007],[21757,21759,21810],[21761,21763,22008],
 [21811,21812,21959],[21861,21863,21960],[21910,21913,21961],
 [22145,22146,22189],[22148,22149,22315],[22154,22156,22541],
 [22167,22169,22905],[22172,22173,22999],[22190,22191,22274],
 [22193,22194,22394],[22196,22197,22505],[22208,22210,22860],
 [22232,22234,22275],[22238,22241,22507],[22243,22244,22670],
 [22250,22254,22862],[22260,22261,23044],[22264,22265,23090],
 [22272,22273,23134],[22278,22279,22469],[22283,22284,22639],
 [22289,22291,22811],[22318,22320,22470],[22326,22327,22729],
 [22329,22333,22813],[22337,22338,22982],[22343,22344,23069],
 [22348,22349,23114],[22357,22361,22472],[22362,22364,22640],
 [22368,22374,22815],[22376,22378,22983],[22396,22398,22506],
 [22408,22411,22861],[22434,22435,22540],[22447,22448,22904],
 [22478,22479,22784],[22486,22487,22964],[22513,22516,22786],
 [22525,22526,23030],[22530,22532,23091],[22537,22538,23132],
 [22547,22552,22788],[22555,22557,22965],[22580,22586,22789],
 [22612,22619,22790],[22620,22623,22966],[22624,22626,23031],
 [22630,22631,23099],[22636,22638,23133],[22644,22645,22810],
 [22675,22676,22835],[22704,22709,22839],[22731,22734,22812],
 [22761,22762,22882],[22824,22831,23086],[22828,22830,23115],
 [22913,22921,23076],[22919,22920,23120],[23190,23191,23244],
 [23193,23194,23400],[23197,23198,23594],[23245,23246,23349],
 [23249,23250,23547],[23297,23299,23350],[23301,23303,23548],
 [23351,23352,23499],[23401,23403,23500],[23450,23453,23501],
 [23685,23686,23729],[23688,23689,23855],[23694,23696,24081],
 [23707,23709,24445],[23712,23713,24539],[23730,23731,23814],
 [23733,23734,23934],[23736,23737,24045],[23748,23750,24400],
 [23772,23774,23815],[23783,23784,24210],[23790,23794,24402],
 [23800,23801,24584],[23804,23805,24630],[23812,23813,24674],
 [23818,23819,24009],[23821,24123],[23822,24155],
 [23823,23824,24179],[23829,23831,24351],[23839,24406],
 [23856,23895],[23858,23860,24010],[23861,24125],
 [23862,24157],[23865,24084],[23866,23867,24269],
 [23869,23873,24353],[23877,23878,24522],[23879,24449],
 [23883,23884,24609],[23888,23889,24654],[23897,23901,24012],
 [23900,24127],[23902,23904,24180],[23908,23914,24355],
 [23916,23918,24523],[23936,23938,24046],[23937,23939],
 [23948,23951,24401],[23950,23952],[23974,23975,24080],
 [23987,23988,24444],[24018,24019,24324],[24026,24027,24504],
 [24053,24056,24326],[24065,24066,24570],[24070,24072,24631],
 [24077,24078,24672],[24087,24092,24328],[24095,24097,24505],
 [24120,24126,24329],[24152,24159,24330],[24160,24163,24506],
 [24164,24166,24571],[24168,24175],[24170,24171,24639],
 [24172,24174],[24176,24178,24673],[24184,24185,24350],
 [24215,24216,24375],[24244,24249,24379],[24271,24274,24352],
 [24301,24302,24422],[24364,24371,24626],[24368,24370,24655],
 [24453,24461,24616],[24459,24460,24660],[24559,24566],
 [24579,24581],[24587,24591],[24602,24608],
 [24675,24676,24729],[24678,24679,24885],[24682,24683,25079],
 [24730,24731,24834],[24734,24735,25032],[24782,24784,24835],
 [24786,24788,25033],[24836,24837,24984],[24886,24888,24985],
 [24935,24938,24986],[25170,25171,25214],[25173,25174,25340],
 [25179,25181,25566],[25180,25346],[25192,25194,25930],
 [25197,25198,26024],[25215,25216,25299],[25218,25219,25419],
 [25221,25222,25530],[25224,25307],[25233,25235,25885],
 [25238,25361],[25257,25259,25300],[25263,25266,25532],
 [25265,25306],[25268,25269,25695],[25275,25279,25887],
 [25280,25400],[25285,25286,26069],[25289,25290,26115],
 [25297,25298,26159],[25303,25304,25494],[25308,25309,25664],
 [25314,25316,25836],[25324,25364],[25341,25435],
 [25343,25345,25495],[25351,25352,25754],[25354,25358,25838],
 [25362,25363,26007],[25368,25369,26094],[25373,25374,26139],
 [25380,25437],[25382,25386,25497],[25387,25389,25665],
 [25393,25399,25840],[25401,25403,26008],[25421,25423,25531],
 [25433,25436,25886],[25450,25452],[25459,25460,25565],
 [25472,25473,25929],[25503,25504,25809],[25511,25512,25989],
 [25538,25541,25811],[25549,25583],[25550,25551,26055],
 [25555,25557,26116],[25562,25563,26157],[25568,25703],
 [25572,25577,25813],[25580,25582,25990],[25605,25611,25814],
 [25606,25612],[25614,25616],[25633,25705],
 [25637,25644,25815],[25638,25642],[25645,25648,25991],
 [25646,25647],[25649,25651,26056],[25652,25653],
 [25655,25656,26124],[25657,25658],[25661,25663,26158],
 [25669,25670,25835],[25700,25701,25860],[25729,25734,25864],
 [25756,25759,25837],[25786,25787,25907],[25841,25888],
 [25849,25856,26111],[25853,25855,26140],[25891,25934],
 [25938,25946,26101],[25944,25945,26145],[26058,26061],
 [26078,26080],[26082,26085],
 [26161,26162,26163,26213,26214,26215,26265,26266,27591,27592,
      27593,27644,27694,28969,29020,30295],[26164,26216,27594],
 [26165,26166,26167,26217,26218,26219,26463,26464,27595,27596,
      27597,27842,27888,29167,29214,35099],[26168,26220,27598],
 [26169,26221,27599],[26170,26222,27600],[26171,26223,27601],
 [26172,26224,27602],[26173,26225,27603],[26174,26226,27604],
 [26175,26227,27605],[26176,26228,27606],[26177,26229,27607],
 [26178,26230,27608],[26179,26231,27609],[26180,26232,27610],
 [26181,26233,27611],[26182,26234,27612],[26183,26235,27613],
 [26184,26236,27614],[26185,26237,27615],[26186,26238,27616],
 [26187,26239,27617],[26188,26240,27618],[26189,26241,27619],
 [26190,26242,27620],[26191,26243,27621],[26192,26244,27622],
 [26193,26245,27623],[26194,26246,27624],[26195,26247,27625],
 [26196,26248,27626],[26197,26249,27627],[26198,26250,27628],
 [26199,26251,27629],[26200,26252,27630],[26201,26253,27631],
 [26202,26254,27632],[26203,26255,27633],[26204,26256,27634],
 [26205,26257,27635],[26206,26258,27636],[26207,26259,27637],
 [26208,26260,27638],[26209,26261,27639],[26210,26262,27640],
 [26211,26263,27641],[26212,26264,27642],[26267,26268,26415],
 [26269,26270,30489],
 [26317,26319,26320,26366,26368,26369,26416,26417,31570,31572,
      31573,31621,31714,32796,32890,34018],[26318,26367,31571],
 [26321,26370,31574],[26322,26371,31575],[26323,26372,31576],
 [26324,26373,31577],[26325,26374,31578],[26326,26375,31579],
 [26327,26376,31580],[26328,26377,31581],[26329,26378,31582],
 [26330,26379,31583],[26331,26380,31584],[26332,26381,31585],
 [26333,26382,31586],[26334,26383,31587],[26335,26384,31588],
 [26336,26385,31589],[26337,26386,31590],[26338,26387,31591],
 [26339,26388,31592],[26340,26389,31593],[26341,26390,31594],
 [26342,26391,31595],[26343,26392,31596],[26344,26393,31597],
 [26345,26394,31598],[26346,26395,31599],[26347,26396,31600],
 [26348,26397,31601],[26349,26398,31602],[26350,26399,31603],
 [26351,26400,31604],[26352,26401,31605],[26353,26402,31606],
 [26354,26403,31607],[26355,26404,31608],[26356,26405,31609],
 [26357,26406,31610],[26358,26407,31611],[26359,26408,31612],
 [26360,26409,31613],[26361,26410,31614],[26362,26411,31615],
 [26363,26412,31616],[26364,26413,31617],[26365,26414,31618],
 [26511,26556,36180],[26512,26557,36181],[26513,26558,36182],
 [26514,26559,36183],[26515,26560,36184],[26516,26561,36185],
 [26517,26562,36186],[26518,26563,36187],[26519,26564,36188],
 [26520,26565,36189],[26521,26566,36190],[26522,26567,36191],
 [26523,26568,36192],[26524,26569,36193],[26525,26570,36194],
 [26526,26571,36195],[26527,26572,36196],[26528,26573,36197],
 [26529,26574,36198],[26530,26575,36199],[26531,26576,36200],
 [26532,26577,36201],[26533,26578,36202],[26534,26579,36203],
 [26535,26580,36204],[26536,26581,36205],[26537,26582,36206],
 [26538,26583,36207],[26539,26584,36208],[26540,26585,36209],
 [26541,26586,36210],[26542,26587,36211],[26543,26588,36212],
 [26544,26589,36213],[26545,26590,36214],[26546,26591,36215],
 [26547,26592,36216],[26548,26593,36217],[26549,26594,36218],
 [26550,26595,36219],[26551,26596,36220],[26552,26597,36221],
 [26553,26598,36222],[26554,26599,36223],[26555,26600,36224],
 [26601,26602,26645],[26604,26605,26771],[26610,26612,26997],
 [26623,26625,27361],[26628,26629,27455],[26646,26647,26730],
 [26649,26650,26850],[26652,26653,26961],[26664,26666,27316],
 [26688,26690,26731],[26694,26697,26963],[26699,26700,27126],
 [26706,26710,27318],[26716,26717,27500],[26720,26721,27546],
 [26728,26729,27590],[26734,26735,26925],[26739,26740,27095],
 [26745,26747,27267],[26774,26776,26926],[26782,26783,27185],
 [26785,26789,27269],[26793,26794,27438],[26799,26800,27525],
 [26804,26805,27570],[26813,26817,26928],[26818,26820,27096],
 [26824,26830,27271],[26832,26834,27439],[26852,26854,26962],
 [26864,26867,27317],[26890,26891,26996],[26903,26904,27360],
 [26934,26935,27240],[26942,26943,27420],[26969,26972,27242],
 [26981,26982,27486],[26986,26988,27547],[26993,26994,27588],
 [27003,27008,27244],[27011,27013,27421],[27036,27042,27245],
 [27068,27075,27246],[27076,27079,27422],[27080,27082,27487],
 [27086,27087,27555],[27092,27094,27589],[27100,27101,27266],
 [27131,27132,27291],[27160,27165,27295],[27187,27190,27268],
 [27217,27218,27338],[27280,27287,27542],[27284,27286,27571],
 [27369,27377,27532],[27375,27376,27576],
 [27645,27646,27647,27695,27696,27697,27793,27794,30296,30297,
      30298,30395,30442,31619,31667,33971],[27648,27698,30299],
 [27649,27699,30300],[27650,27700,30301],[27651,27701,30302],
 [27652,27702,30303],[27653,27703,30304],[27654,27704,30305],
 [27655,27705,30306],[27656,27706,30307],[27657,27707,30308],
 [27658,27708,30309],[27659,27709,30310],[27660,27710,30311],
 [27661,27711,30312],[27662,27712,30313],[27663,27713,30314],
 [27664,27714,30315],[27665,27715,30316],[27666,27716,30317],
 [27667,27717,30318],[27668,27718,30319],[27669,27719,30320],
 [27670,27720,30321],[27671,27721,30322],[27672,27722,30323],
 [27673,27723,30324],[27674,27724,30325],[27675,27725,30326],
 [27676,27726,30327],[27677,27727,30328],[27678,27728,30329],
 [27679,27729,30330],[27680,27730,30331],[27681,27731,30332],
 [27682,27732,30333],[27683,27733,30334],[27684,27734,30335],
 [27685,27735,30336],[27686,27736,30337],[27687,27737,30338],
 [27688,27738,30339],[27689,27739,30340],[27690,27740,30341],
 [27691,27741,30342],[27692,27742,30343],[27693,27743,30344],
 [27744,27747,27795],[27745,27746,32843],[27843,27889,35100],
 [27844,27890,35101],[27845,27891,35102],[27846,27892,35103],
 [27847,27893,35104],[27848,27894,35105],[27849,27895,35106],
 [27850,27896,35107],[27851,27897,35108],[27852,27898,35109],
 [27853,27899,35110],[27854,27900,35111],[27855,27901,35112],
 [27856,27902,35113],[27857,27903,35114],[27858,27904,35115],
 [27859,27905,35116],[27860,27906,35117],[27861,27907,35118],
 [27862,27908,35119],[27863,27909,35120],[27864,27910,35121],
 [27865,27911,35122],[27866,27912,35123],[27867,27913,35124],
 [27868,27914,35125],[27869,27915,35126],[27870,27916,35127],
 [27871,27917,35128],[27872,27918,35129],[27873,27919,35130],
 [27874,27920,35131],[27875,27921,35132],[27876,27922,35133],
 [27877,27923,35134],[27878,27924,35135],[27879,27925,35136],
 [27880,27926,35137],[27881,27927,35138],[27882,27928,35139],
 [27883,27929,35140],[27884,27930,35141],[27885,27931,35142],
 [27886,27932,35143],[27887,27933,35144],[27979,27980,28023],
 [27982,27983,28149],[27988,27990,28375],[28001,28003,28739],
 [28006,28007,28833],[28024,28025,28108],[28027,28028,28228],
 [28030,28031,28339],[28042,28044,28694],[28066,28068,28109],
 [28072,28075,28341],[28077,28078,28504],[28084,28088,28696],
 [28094,28095,28878],[28098,28099,28924],[28106,28107,28968],
 [28112,28113,28303],[28117,28118,28473],[28123,28125,28645],
 [28152,28154,28304],[28160,28161,28563],[28163,28167,28647],
 [28171,28172,28816],[28177,28178,28903],[28182,28183,28948],
 [28191,28195,28306],[28196,28198,28474],[28202,28208,28649],
 [28210,28212,28817],[28230,28232,28340],[28242,28245,28695],
 [28268,28269,28374],[28281,28282,28738],[28312,28313,28618],
 [28320,28321,28798],[28347,28350,28620],[28359,28360,28864],
 [28364,28366,28925],[28371,28372,28966],[28381,28386,28622],
 [28389,28391,28799],[28414,28420,28623],[28446,28453,28624],
 [28454,28457,28800],[28458,28460,28865],[28464,28465,28933],
 [28470,28472,28967],[28478,28479,28644],[28509,28510,28669],
 [28538,28543,28673],[28565,28568,28646],[28595,28596,28716],
 [28658,28665,28920],[28662,28664,28949],[28747,28755,28910],
 [28753,28754,28954],
 [28971,28972,28974,29070,29071,29073,29119,29121,30345,30346,
      30348,30396,30443,32795,32844,33972],[28973,29072,30347],
 [28975,29074,30349],[28976,29075,30350],[28977,29076,30351],
 [28978,29077,30352],[28979,29078,30353],[28980,29079,30354],
 [28981,29080,30355],[28982,29081,30356],[28983,29082,30357],
 [28984,29083,30358],[28985,29084,30359],[28986,29085,30360],
 [28987,29086,30361],[28988,29087,30362],[28989,29088,30363],
 [28990,29089,30364],[28991,29090,30365],[28992,29091,30366],
 [28993,29092,30367],[28994,29093,30368],[28995,29094,30369],
 [28996,29095,30370],[28997,29096,30371],[28998,29097,30372],
 [28999,29098,30373],[29000,29099,30374],[29001,29100,30375],
 [29002,29101,30376],[29003,29102,30377],[29004,29103,30378],
 [29005,29104,30379],[29006,29105,30380],[29007,29106,30381],
 [29008,29107,30382],[29009,29108,30383],[29010,29109,30384],
 [29011,29110,30385],[29012,29111,30386],[29013,29112,30387],
 [29014,29113,30388],[29015,29114,30389],[29016,29115,30390],
 [29017,29116,30391],[29018,29117,30392],[29019,29118,30393],
 [29021,29023,29120],[29022,29024,31668],[29169,29260,35145],
 [29170,29261,35146],[29171,29262,35147],[29172,29263,35148],
 [29173,29264,35149],[29174,29265,35150],[29175,29266,35151],
 [29176,29267,35152],[29177,29268,35153],[29178,29269,35154],
 [29179,29270,35155],[29180,29271,35156],[29181,29272,35157],
 [29182,29273,35158],[29183,29274,35159],[29184,29275,35160],
 [29185,29276,35161],[29186,29277,35162],[29187,29278,35163],
 [29188,29279,35164],[29189,29280,35165],[29190,29281,35166],
 [29191,29282,35167],[29192,29283,35168],[29193,29284,35169],
 [29194,29285,35170],[29195,29286,35171],[29196,29287,35172],
 [29197,29288,35173],[29198,29289,35174],[29199,29290,35175],
 [29200,29291,35176],[29201,29292,35177],[29202,29293,35178],
 [29203,29294,35179],[29204,29295,35180],[29205,29296,35181],
 [29206,29297,35182],[29207,29298,35183],[29208,29299,35184],
 [29209,29300,35185],[29210,29301,35186],[29211,29302,35187],
 [29212,29303,35188],[29213,29304,35189],[29305,29306,29349],
 [29308,29309,29475],[29314,29316,29701],[29327,29329,30065],
 [29332,29333,30159],[29350,29351,29434],[29353,29354,29554],
 [29356,29357,29665],[29368,29370,30020],[29392,29394,29435],
 [29398,29401,29667],[29403,29404,29830],[29410,29414,30022],
 [29420,29421,30204],[29424,29425,30250],[29432,29433,30294],
 [29438,29439,29629],[29443,29444,29799],[29449,29451,29971],
 [29478,29480,29630],[29486,29487,29889],[29489,29493,29973],
 [29497,29498,30142],[29503,29504,30229],[29508,29509,30274],
 [29517,29521,29632],[29522,29524,29800],[29528,29534,29975],
 [29536,29538,30143],[29556,29558,29666],[29568,29571,30021],
 [29594,29595,29700],[29607,29608,30064],[29638,29639,29944],
 [29646,29647,30124],[29673,29676,29946],[29685,29686,30190],
 [29690,29692,30251],[29697,29698,30292],[29707,29712,29948],
 [29715,29717,30125],[29740,29746,29949],[29772,29779,29950],
 [29780,29783,30126],[29784,29786,30191],[29790,29791,30259],
 [29796,29798,30293],[29804,29805,29970],[29835,29836,29995],
 [29864,29869,29999],[29891,29894,29972],[29921,29922,30042],
 [29984,29991,30246],[29988,29990,30275],[30073,30081,30236],
 [30079,30080,30280],[30397,30444,33973],[30398,30445,33974],
 [30399,30446,33975],[30400,30447,33976],[30401,30448,33977],
 [30402,30449,33978],[30403,30450,33979],[30404,30451,33980],
 [30405,30452,33981],[30406,30453,33982],[30407,30454,33983],
 [30408,30455,33984],[30409,30456,33985],[30410,30457,33986],
 [30411,30458,33987],[30412,30459,33988],[30413,30460,33989],
 [30414,30461,33990],[30415,30462,33991],[30416,30463,33992],
 [30417,30464,33993],[30418,30465,33994],[30419,30466,33995],
 [30420,30467,33996],[30421,30468,33997],[30422,30469,33998],
 [30423,30470,33999],[30424,30471,34000],[30425,30472,34001],
 [30426,30473,34002],[30427,30474,34003],[30428,30475,34004],
 [30429,30476,34005],[30430,30477,34006],[30431,30478,34007],
 [30432,30479,34008],[30433,30480,34009],[30434,30481,34010],
 [30435,30482,34011],[30436,30483,34012],[30437,30484,34013],
 [30438,30485,34014],[30439,30486,34015],[30440,30487,34016],
 [30441,30488,34017],[30580,30581,30624],[30583,30584,30750],
 [30589,30591,30976],[30602,30604,31340],[30607,30608,31434],
 [30625,30626,30709],[30628,30629,30829],[30631,30632,30940],
 [30643,30645,31295],[30667,30669,30710],[30673,30676,30942],
 [30678,30679,31105],[30685,30689,31297],[30695,30696,31479],
 [30699,30700,31525],[30707,30708,31569],[30713,30714,30904],
 [30718,30719,31074],[30724,30726,31246],[30753,30755,30905],
 [30761,30762,31164],[30764,30768,31248],[30772,30773,31417],
 [30778,30779,31504],[30783,30784,31549],[30792,30796,30907],
 [30797,30799,31075],[30803,30809,31250],[30811,30813,31418],
 [30831,30833,30941],[30843,30846,31296],[30869,30870,30975],
 [30882,30883,31339],[30913,30914,31219],[30921,30922,31399],
 [30948,30951,31221],[30960,30961,31465],[30965,30967,31526],
 [30972,30973,31567],[30982,30987,31223],[30990,30992,31400],
 [31015,31021,31224],[31047,31054,31225],[31055,31058,31401],
 [31059,31061,31466],[31065,31066,31534],[31071,31073,31568],
 [31079,31080,31245],[31110,31111,31270],[31139,31144,31274],
 [31166,31169,31247],[31196,31197,31317],[31259,31266,31521],
 [31263,31265,31550],[31348,31356,31511],[31354,31355,31555],
 [31622,31715,34019],[31623,31716,34020],[31624,31717,34021],
 [31625,31718,34022],[31626,31719,34023],[31627,31720,34024],
 [31628,31721,34025],[31629,31722,34026],[31630,31723,34027],
 [31631,31724,34028],[31632,31725,34029],[31633,31726,34030],
 [31634,31727,34031],[31635,31728,34032],[31636,31729,34033],
 [31637,31730,34034],[31638,31731,34035],[31639,31732,34036],
 [31640,31733,34037],[31641,31734,34038],[31642,31735,34039],
 [31643,31736,34040],[31644,31737,34041],[31645,31738,34042],
 [31646,31739,34043],[31647,31740,34044],[31648,31741,34045],
 [31649,31742,34046],[31650,31743,34047],[31651,31744,34048],
 [31652,31745,34049],[31653,31746,34050],[31654,31747,34051],
 [31655,31748,34052],[31656,31749,34053],[31657,31750,34054],
 [31658,31751,34055],[31659,31752,34056],[31660,31753,34057],
 [31661,31754,34058],[31662,31755,34059],[31663,31756,34060],
 [31664,31757,34061],[31665,31758,34062],[31666,31759,34063],
 [31805,31806,31849],[31808,31809,31975],[31814,31816,32201],
 [31827,31829,32565],[31832,31833,32659],[31850,31851,31934],
 [31853,31854,32054],[31856,31857,32165],[31868,31870,32520],
 [31892,31894,31935],[31898,31901,32167],[31903,31904,32330],
 [31910,31914,32522],[31920,31921,32704],[31924,31925,32750],
 [31932,31933,32794],[31938,31939,32129],[31943,31944,32299],
 [31949,31951,32471],[31978,31980,32130],[31986,31987,32389],
 [31989,31993,32473],[31997,31998,32642],[32003,32004,32729],
 [32008,32009,32774],[32017,32021,32132],[32022,32024,32300],
 [32028,32034,32475],[32036,32038,32643],[32056,32058,32166],
 [32068,32071,32521],[32094,32095,32200],[32107,32108,32564],
 [32138,32139,32444],[32146,32147,32624],[32173,32176,32446],
 [32185,32186,32690],[32190,32192,32751],[32197,32198,32792],
 [32207,32212,32448],[32215,32217,32625],[32240,32246,32449],
 [32272,32279,32450],[32280,32283,32626],[32284,32286,32691],
 [32290,32291,32759],[32296,32298,32793],[32304,32305,32470],
 [32335,32336,32495],[32364,32369,32499],[32391,32394,32472],
 [32421,32422,32542],[32484,32491,32746],[32488,32490,32775],
 [32573,32581,32736],[32579,32580,32780],[32798,32936,34064],
 [32799,32937,34065],[32800,32938,34066],[32801,32939,34067],
 [32802,32940,34068],[32803,32941,34069],[32804,32942,34070],
 [32805,32943,34071],[32806,32944,34072],[32807,32945,34073],
 [32808,32946,34074],[32809,32947,34075],[32810,32948,34076],
 [32811,32949,34077],[32812,32950,34078],[32813,32951,34079],
 [32814,32952,34080],[32815,32953,34081],[32816,32954,34082],
 [32817,32955,34083],[32818,32956,34084],[32819,32957,34085],
 [32820,32958,34086],[32821,32959,34087],[32822,32960,34088],
 [32823,32961,34089],[32824,32962,34090],[32825,32963,34091],
 [32826,32964,34092],[32827,32965,34093],[32828,32966,34094],
 [32829,32967,34095],[32830,32968,34096],[32831,32969,34097],
 [32832,32970,34098],[32833,32971,34099],[32834,32972,34100],
 [32835,32973,34101],[32836,32974,34102],[32837,32975,34103],
 [32838,32976,34104],[32839,32977,34105],[32840,32978,34106],
 [32841,32979,34107],[32842,32980,34108],[32981,32982,33025],
 [32984,32985,33151],[32990,32992,33377],[33003,33005,33741],
 [33008,33009,33835],[33026,33027,33110],[33029,33030,33230],
 [33032,33033,33341],[33044,33046,33696],[33068,33070,33111],
 [33074,33077,33343],[33079,33080,33506],[33086,33090,33698],
 [33096,33097,33880],[33100,33101,33926],[33108,33109,33970],
 [33114,33115,33305],[33119,33120,33475],[33125,33127,33647],
 [33154,33156,33306],[33162,33163,33565],[33165,33169,33649],
 [33173,33174,33818],[33179,33180,33905],[33184,33185,33950],
 [33193,33197,33308],[33198,33200,33476],[33204,33210,33651],
 [33212,33214,33819],[33232,33234,33342],[33244,33247,33697],
 [33270,33271,33376],[33283,33284,33740],[33314,33315,33620],
 [33322,33323,33800],[33349,33352,33622],[33361,33362,33866],
 [33366,33368,33927],[33373,33374,33968],[33383,33388,33624],
 [33391,33393,33801],[33416,33422,33625],[33448,33455,33626],
 [33456,33459,33802],[33460,33462,33867],[33466,33467,33935],
 [33472,33474,33969],[33480,33481,33646],[33511,33512,33671],
 [33540,33545,33675],[33567,33570,33648],[33597,33598,33718],
 [33660,33667,33922],[33664,33666,33951],[33749,33757,33912],
 [33755,33756,33956],[34109,34110,34153],[34112,34113,34279],
 [34118,34120,34505],[34131,34133,34869],[34136,34137,34963],
 [34154,34155,34238],[34157,34158,34358],[34160,34161,34469],
 [34172,34174,34824],[34196,34198,34239],[34202,34205,34471],
 [34207,34208,34634],[34214,34218,34826],[34224,34225,35008],
 [34228,34229,35054],[34236,34237,35098],[34242,34243,34433],
 [34247,34248,34603],[34253,34255,34775],[34282,34284,34434],
 [34290,34291,34693],[34293,34297,34777],[34301,34302,34946],
 [34307,34308,35033],[34312,34313,35078],[34321,34325,34436],
 [34326,34328,34604],[34332,34338,34779],[34340,34342,34947],
 [34360,34362,34470],[34372,34375,34825],[34398,34399,34504],
 [34411,34412,34868],[34442,34443,34748],[34450,34451,34928],
 [34477,34480,34750],[34489,34490,34994],[34494,34496,35055],
 [34501,34502,35096],[34511,34516,34752],[34519,34521,34929],
 [34544,34550,34753],[34576,34583,34754],[34584,34587,34930],
 [34588,34590,34995],[34594,34595,35063],[34600,34602,35097],
 [34608,34609,34774],[34639,34640,34799],[34668,34673,34803],
 [34695,34698,34776],[34725,34726,34846],[34788,34795,35050],
 [34792,34794,35079],[34877,34885,35040],[34883,34884,35084],
 [35190,35191,35234],[35193,35194,35360],[35199,35201,35586],
 [35212,35214,35950],[35217,35218,36044],[35235,35236,35319],
 [35238,35239,35439],[35241,35242,35550],[35253,35255,35905],
 [35277,35279,35320],[35283,35286,35552],[35288,35289,35715],
 [35295,35299,35907],[35305,35306,36089],[35309,35310,36135],
 [35317,35318,36179],[35323,35324,35514],[35328,35329,35684],
 [35334,35336,35856],[35363,35365,35515],[35371,35372,35774],
 [35374,35378,35858],[35382,35383,36027],[35388,35389,36114],
 [35393,35394,36159],[35402,35406,35517],[35407,35409,35685],
 [35413,35419,35860],[35421,35423,36028],[35441,35443,35551],
 [35453,35456,35906],[35479,35480,35585],[35492,35493,35949],
 [35523,35524,35829],[35531,35532,36009],[35558,35561,35831],
 [35570,35571,36075],[35575,35577,36136],[35582,35583,36177],
 [35592,35597,35833],[35600,35602,36010],[35625,35631,35834],
 [35657,35664,35835],[35665,35668,36011],[35669,35671,36076],
 [35675,35676,36144],[35681,35683,36178],[35689,35690,35855],
 [35720,35721,35880],[35749,35754,35884],[35776,35779,35857],
 [35806,35807,35927],[35869,35876,36131],[35873,35875,36160],
 [35958,35966,36121],[35964,35965,36165],[36225,36226,36269],
 [36228,36229,36395],[36234,36236,36621],[36247,36249,36985],
 [36252,36253,37079],[36270,36271,36354],[36273,36274,36474],
 [36276,36277,36585],[36288,36290,36940],[36312,36314,36355],
 [36318,36321,36587],[36323,36324,36750],[36330,36334,36942],
 [36340,36341,37124],[36344,36345,37170],[36352,36353,37214],
 [36358,36359,36549],[36363,36364,36719],[36369,36371,36891],
 [36398,36400,36550],[36406,36407,36809],[36409,36413,36893],
 [36417,36418,37062],[36423,36424,37149],[36428,36429,37194],
 [36437,36441,36552],[36442,36444,36720],[36448,36454,36895],
 [36456,36458,37063],[36476,36478,36586],[36488,36491,36941],
 [36514,36515,36620],[36527,36528,36984],[36558,36559,36864],
 [36566,36567,37044],[36593,36596,36866],[36605,36606,37110],
 [36610,36612,37171],[36617,36618,37212],[36627,36632,36868],
 [36635,36637,37045],[36660,36666,36869],[36692,36699,36870],
 [36700,36703,37046],[36704,36706,37111],[36710,36711,37179],
 [36716,36718,37213],[36724,36725,36890],[36755,36756,36915],
 [36784,36789,36919],[36811,36814,36892],[36841,36842,36962],
 [36904,36911,37166],[36908,36910,37195],[36993,37001,37156],
 [36999,37000,37200],[37215,37216,37259],[37218,37219,37385],
 [37224,37226,37611],[37237,37239,37975],[37242,37243,38069],
 [37260,37261,37344],[37263,37264,37464],[37266,37267,37575],
 [37278,37280,37930],[37302,37304,37345],[37308,37311,37577],
 [37313,37314,37740],[37320,37324,37932],[37330,37331,38114],
 [37334,37335,38160],[37342,37343,38204],[37348,37349,37539],
 [37353,37354,37709],[37359,37361,37881],[37388,37390,37540],
 [37396,37397,37799],[37399,37403,37883],[37407,37408,38052],
 [37413,37414,38139],[37418,37419,38184],[37427,37431,37542],
 [37432,37434,37710],[37438,37444,37885],[37446,37448,38053],
 [37466,37468,37576],[37478,37481,37931],[37504,37505,37610],
 [37517,37518,37974],[37548,37549,37854],[37556,37557,38034],
 [37583,37586,37856],[37595,37596,38100],[37600,37602,38161],
 [37607,37608,38202],[37617,37622,37858],[37625,37627,38035],
 [37650,37656,37859],[37682,37689,37860],[37690,37693,38036],
 [37694,37696,38101],[37700,37701,38169],[37706,37708,38203],
 [37714,37715,37880],[37745,37746,37905],[37774,37779,37909],
 [37801,37804,37882],[37831,37832,37952],[37894,37901,38156],
 [37898,37900,38185],[37983,37991,38146],[37989,37990,38190],
 [38206,38207,38208,38248,38249,38250,38290,38291,39151,39152,
      39153,39194,39234,40054,40095,40915],
 [38209,38210,38251,38252,38410,39154,39155,40174],
 [38212,38213,38215,38254,38255,38257,38521,38523,39157,39158,
      39160,39426,39460,40285,40321,45256],[38214,38256,39159],
 [38216,38258,39161],
 [38217,38218,38259,38260,38686,39162,39163,39589],
 [38219,38261,39164],[38220,38262,39165],[38223,38265,39168],
 [38224,38226,38228,38266,38268,38270,38876,38878,39169,39171,
      39173,39781,39824,40640,40685,50394],[38225,38267,39170],
 [38227,38269,39172],[38229,38271,39174],[38230,38272,39175],
 [38231,38232,38273,38274,39176,39177,39918,40779],
 [38233,38275,39178],
 [38234,38235,38276,38277,39060,39179,39180,39963],
 [38236,38278,39181],[38237,38279,39182],
 [38238,38239,38280,38281,39106,39183,39184,40009],
 [38240,38282,39185],[38241,38283,39186],[38242,38284,39187],
 [38243,38285,39188],[38244,38286,39189],[38245,38287,39190],
 [38246,38247,38288,38289,39150,39191,39192,40053],
 [38296,38298,41141],[38299,38300,38655],[38305,38307,38827],
 [38309,38311,41505],[38314,38315,41599],[38332,38371,41735],
 [38333,38372,41736],[38335,38374,41738],[38337,38376,41740],
 [38339,38341,38378,38380,38656,41742,41744,42020],
 [38340,38379,41743],
 [38342,38343,38381,38382,38745,41745,41746,42850],
 [38344,38383,41747],
 [38345,38349,38351,38384,38388,38390,38829,38831,41748,41752,
      41754,42195,42285,42934,43026,49865],[38346,38385,41749],
 [38347,38386,41750],[38348,38387,41751],[38350,38389,41753],
 [38352,38391,41755],
 [38353,38354,38355,38392,38393,38394,38998,38999,41756,41757,
      41758,42363,42379,43103,43120,51579],[38356,38395,41759],
 [38357,38396,41760],[38358,38397,41761],
 [38359,38360,38398,38399,39085,41762,41763,43190],
 [38361,38400,41764],[38362,38401,41765],[38363,38402,41766],
 [38364,38365,38403,38404,39130,41767,41768,43235],
 [38366,38405,41769],[38367,38406,41770],[38368,38407,41771],
 [38369,38408,41772],[38370,38409,41773],[38412,38414,38522],
 [38413,38415,43365],[38424,38427,38877],[38426,38428,43729],
 [38431,38432,43823],
 [38450,38451,38452,38556,38590,44031,44064,45850],
 [38463,38464,38465,38920,38941,44395,44415,50855],
 [38529,38532,38802],[38534,38536,45620],[38539,38540,45714],
 [38541,38542,39046],[38546,38548,39107],[38553,38554,39148],
 [38558,38623,45883],[38559,38624,45884],[38560,38625,45885],
 [38561,38626,45886],[38562,38627,45887],
 [38563,38568,38570,38628,38633,38635,38804,38806,45888,45893,
      45895,46066,46181,47088,47205,49565],[38564,38629,45889],
 [38565,38630,45890],[38566,38631,45891],[38567,38632,45892],
 [38569,38634,45894],
 [38571,38573,38574,38636,38638,38639,38981,38982,45896,45898,
      45899,46242,46275,47265,47299,51443],[38572,38637,45897],
 [38575,38577,38640,38642,39047,45900,45902,46307],
 [38576,38641,45901],[38578,38643,45903],[38579,38644,45904],
 [38580,38645,45905],
 [38581,38582,38646,38647,39115,45906,45907,46375],
 [38583,38648,45908],[38584,38649,45909],[38585,38650,45910],
 [38586,38651,45911],
 [38587,38589,38652,38654,39149,45912,45914,46409],
 [38588,38653,45913],[38596,38602,38805],[38601,38603,46709],
 [38606,38607,46803],[38660,38661,38826],[38664,38666,47670],
 [38669,38670,47764],[38691,38692,38851],[38694,38696,48105],
 [38699,38700,48199],[38702,48248],
 [38720,38723,38725,38853,38855,48443,48511,50141],
 [38747,38750,38828],[38751,38753,48889],[38756,38757,48983],
 [38777,38778,38780,38898,38900,49219,49240,50625],
 [38783,38784,49334],[38809,38810,49659],[38834,38835,49959],
 [38840,38847,39102],[38844,38846,39131],[38858,38859,50235],
 [38881,38882,50488],[38899,50653],[38903,38904,50719],
 [38922,38961,50875],[38923,38962,50876],
 [38924,38925,38963,38964,50877,50878,50929,51290],
 [38926,38965,50879],[38927,38966,50880],[38928,38967,50881],
 [38929,38937,38968,38976,39092,50882,50890,51367],
 [38930,38969,50883],[38931,38970,50884],[38932,38971,50885],
 [38933,38972,50886],[38934,38973,50887],
 [38935,38936,38974,38975,39136,50888,50889,51411],
 [38938,38977,50891],[38939,38978,50892],[38940,38979,50893],
 [38944,38945,51119],[39016,39031,51715],[39017,39032,51716],
 [39018,39033,51717],[39019,39034,51718],[39020,39035,51719],
 [39021,39036,51720],[39022,39037,51721],[39023,39038,51722],
 [39024,39039,51723],[39025,39040,51724],[39026,39041,51725],
 [39027,39042,51726],[39028,39043,51727],[39029,39044,51728],
 [39030,39045,51729],
 [39195,39196,39235,39236,40916,40917,40994,41774],
 [39197,39198,39199,39237,39238,39239,39388,39389,40918,40919,
      40920,41070,41105,41849,41885,44625],[39200,39240,40921],
 [39201,39241,40922],
 [39202,39203,39242,39243,39558,40923,40924,42019],
 [39204,39244,40925],
 [39205,39206,39245,39246,39648,40926,40927,41329],
 [39208,39210,39212,39248,39250,39252,39730,39732,40929,40931,
      40933,41413,41460,42191,42240,49820],[39209,39249,40930],
 [39211,39251,40932],[39213,39253,40934],[39214,39254,40935],
 [39215,39255,40936],
 [39216,39217,39256,39257,39901,40937,40938,41582],
 [39218,39258,40939],[39219,39259,40940],[39220,39260,40941],
 [39221,39261,40942],
 [39222,39223,39262,39263,39988,40943,40944,41669],
 [39224,39264,40945],[39225,39265,40946],[39226,39266,40947],
 [39227,39228,39267,39268,40033,40948,40949,41714],
 [39229,39269,40950],[39230,39270,40951],[39231,39271,40952],
 [39232,39272,40953],[39233,39273,40954],[39274,39275,42515],
 [39276,39280,39391],[39277,39278,42626],[39281,39283,39559],
 [39287,39293,39734],[39289,39291,42981],[39295,39297,39902],
 [39314,39351,43256],
 [39315,39316,39317,39352,39353,39354,39425,39459,43257,43258,
      43259,43329,43364,43995,43996,45255],[39318,39355,43260],
 [39319,39356,43261],[39321,39358,43263],[39322,39359,43264],
 [39323,39360,43265],[39325,39362,43267],[39326,39363,43268],
 [39327,39329,39330,39364,39366,39367,39780,39823,43269,43271,
      43272,43684,43728,44350,44351,50393],[39328,39365,43270],
 [39331,39368,43273],[39332,39369,43274],[39333,39370,43275],
 [39335,39372,43277],[39336,39373,43278],[39337,39374,43279],
 [39338,39375,43280],[39339,39376,43281],[39340,39377,43282],
 [39341,39378,43283],[39342,39379,43284],[39343,39380,43285],
 [39344,39381,43286],[39345,39382,43287],[39346,39383,43288],
 [39347,39384,43289],[39348,39385,43290],[39349,39386,43291],
 [39350,39387,43292],[39400,39402,44980],[39405,39406,39883],
 [39427,39461,45257],[39428,39462,45258],[39429,39463,45259],
 [39430,39464,45260],[39431,39465,45261],[39433,39467,45263],
 [39434,39468,45264],[39436,39470,45266],[39438,39472,45268],
 [39439,39473,45269],
 [39440,39442,39474,39476,39884,45270,45272,45680],
 [39441,39475,45271],[39443,39477,45273],
 [39444,39445,39478,39479,39949,45274,45275,46306],
 [39446,39480,45276],[39447,39481,45277],[39448,39482,45278],
 [39449,39451,39483,39485,40010,45279,45281,46367],
 [39450,39484,45280],[39452,39486,45282],[39453,39487,45283],
 [39454,39488,45284],[39455,39489,45285],
 [39456,39457,39490,39491,40051,45286,45287,46408],
 [39458,39492,45288],[39502,39504,46664],[39534,39536,47160],
 [39539,39542,39885],[39543,39545,39950],[39549,39550,40018],
 [39555,39557,40052],[39563,39564,39729],[39565,39567,47625],
 [39594,39595,39597,39754,39756,48037,48060,50096],
 [39623,39628,39758],[39624,39626,48466],[39635,48665],
 [39650,39653,39731],[39652,39654,48844],[39658,48971],
 [39679,39680,39681,39778,39801,49194,49195,50371],
 [39743,39750,40005],[39747,39749,40034],[39782,39825,50395],
 [39783,39826,50396],[39784,39827,50397],[39785,39828,50398],
 [39786,39829,50399],[39787,39830,50400],[39788,39831,50401],
 [39789,39797,39832,39840,39995,50402,50410,50565],
 [39790,39833,50403],[39791,39834,50404],[39792,39835,50405],
 [39793,39836,50406],[39794,39837,50407],
 [39795,39796,39838,39839,40039,50408,50409,50609],
 [39798,39841,50411],[39799,39842,50412],[39800,39843,50413],
 [40056,40135,40955],[40057,40136,40956],
 [40058,40059,40062,40137,40138,40141,40249,40252,40957,40958,
      40961,41072,41107,42590,42628,44627],[40060,40139,40959],
 [40061,40140,40960],
 [40063,40064,40065,40142,40143,40144,40419,40420,40962,40963,
      40964,41240,41270,42760,42791,47435],[40066,40145,40965],
 [40067,40146,40966],[40068,40147,40967],[40070,40149,40969],
 [40072,40151,40971],[40073,40152,40972],[40074,40153,40973],
 [40076,40155,40975],
 [40077,40079,40156,40158,40763,40976,40978,41583],
 [40078,40157,40977],[40080,40159,40979],
 [40081,40082,40160,40161,40980,40981,41644,43165],
 [40083,40162,40982],[40084,40163,40983],
 [40085,40086,40164,40165,40984,40985,41690,43211],
 [40087,40166,40986],[40088,40167,40987],[40089,40168,40988],
 [40090,40169,40989],[40091,40170,40990],[40092,40171,40991],
 [40093,40094,40172,40173,40992,40993,41734,43255],
 [40098,40100,40250],[40099,40102,41887],[40104,40105,42050],
 [40106,40107,40509],[40111,40115,42242],[40117,40118,40762],
 [40121,40122,42424],[40123,40124,40849],[40125,40126,42470],
 [40128,40129,40894],[40133,40134,42514],
 [40176,40178,40179,40286,40354,43331,43398,45289],
 [40188,40191,40192,40641,40705,43686,43749,50414],
 [40198,40199,43868],[40202,40203,43914],[40210,40211,43958],
 [40213,40216,43997],[40214,40215,40320],[40218,40219,44160],
 [40225,40229,44352],[40227,40228,40684],[40235,40236,44534],
 [40239,40240,44580],[40247,40248,44624],[40254,40255,44790],
 [40261,40265,44982],[40266,40267,40744],[40271,40272,45164],
 [40275,40276,45210],[40283,40284,45254],[40288,40387,45322],
 [40289,40290,40388,40389,45323,45324,45385,46970],
 [40291,40390,45325],[40292,40391,45326],
 [40293,40296,40300,40392,40395,40399,40566,40570,45327,45330,
      45334,45505,45577,47086,47162,49522],[40295,40394,45329],
 [40297,40396,45331],[40298,40397,45332],[40299,40398,45333],
 [40301,40304,40400,40403,40746,45335,45338,45681],
 [40302,40401,45336],[40303,40402,45337],
 [40305,40306,40307,40404,40405,40406,40810,40811,45339,45340,
      45341,45746,45759,47330,47344,51940],[40308,40407,45342],
 [40309,40408,45343],
 [40310,40311,40312,40409,40410,40411,40871,40879,45344,45345,
      45346,45805,45814,47390,47391,52275],[40313,40412,45347],
 [40314,40413,45348],[40315,40414,45349],[40316,40415,45350],
 [40317,40318,40319,40416,40417,40418,40912,40913,45351,45352,
      45353,45848,45849,47432,47434,52394],[40323,40324,45946],
 [40327,40332,40568],[40330,40334,46138],[40335,40337,40745],
 [40340,40341,46320],[40344,40345,46366],[40352,40353,46410],
 [40356,40357,46474],[40360,40366,40569],[40363,40367,46666],
 [40372,46838],[40373,40374,46848],[40377,40378,46894],
 [40385,40386,46938],[40426,40430,47627],[40436,40437,47809],
 [40440,40441,47855],[40448,40449,47899],[40451,40480,47900],
 [40452,40481,47901],[40453,40482,47902],
 [40455,40456,40460,40484,40485,40489,40615,40619,47904,47905,
      47909,48039,48062,48441,48468,50098],[40457,40486,47906],
 [40458,40487,47907],[40459,40488,47908],[40461,40490,47910],
 [40462,40491,47911],[40463,40492,47912],[40465,40494,47914],
 [40466,40467,40495,40496,47915,47916,48244,48650],
 [40468,40497,47917],[40469,40498,47918],
 [40470,40471,40499,40500,47919,47920,48290,48696],
 [40472,40501,47921],[40473,40502,47922],[40474,40503,47923],
 [40475,40504,47924],[40476,40505,47925],[40477,40506,47926],
 [40478,40479,40507,40508,47927,47928,48334,48740],
 [40513,40517,48846],[40523,40524,49028],[40527,40528,49074],
 [40535,40536,49118],[40540,40544,49197],[40541,40542,40662],
 [40550,40551,49379],[40554,40555,49425],[40562,40563,49469],
 [40576,40577,49704],[40580,40581,49750],[40588,40589,49794],
 [40625,40626,50280],[40629,40630,50326],[40637,40638,50370],
 [40639,40664,50373],[40643,40725,50434],[40644,40726,50435],
 [40645,40727,50436],[40646,40728,50437],[40647,40729,50438],
 [40648,40649,40730,40731,50439,50440,50533,51335],
 [40650,40732,50441],[40651,40733,50442],
 [40652,40653,40734,40735,50443,50444,50579,51381],
 [40654,40736,50445],[40655,40737,50446],[40656,40738,50447],
 [40657,40739,50448],[40658,40740,50449],[40659,40741,50450],
 [40660,40661,40742,40743,50451,50452,50623,51425],
 [40670,40671,50764],[40674,40675,50810],[40682,40683,50854],
 [40691,40692,50974],[40693,40701,40856],[40695,40696,51020],
 [40699,40700,40900],[40703,40704,51064],[40711,40712,51164],
 [40715,40716,51210],[40723,40724,51254],[40747,51479],
 [40748,40749,51488],[40752,40753,51534],[40760,40761,51578],
 [40765,40766,51624],[40769,40770,51670],[40777,40778,51714],
 [40781,40782,51744],[40785,40786,51790],[40793,40794,51834],
 [40796,40797,51849],[40800,40801,51895],[40808,40809,51939],
 [40814,40815,51986],[40822,40823,52030],[40825,40837,52031],
 [40826,40838,52032],
 [40827,40828,40839,40840,52033,52034,52064,52130],
 [40829,40841,52035],[40830,40842,52036],[40831,40843,52037],
 [40832,40844,52038],[40833,40845,52039],[40834,40846,52040],
 [40835,40836,40847,40848,52041,52042,52108,52174],
 [40850,40851,52185],[40858,40859,52229],[40860,40861,52230],
 [40868,40869,52274],[40872,40880,52276],[40873,40881,52277],
 [40874,40882,52278],[40875,40883,52279],[40876,40884,52280],
 [40892,40893,52359],[40898,40899,52374],[40903,40904,52384],
 [40907,40908,52390],[40910,40911,52393],
 [40995,40996,40998,41071,41106,43293,43295,44626],
 [41006,41008,41011,41414,41461,43635,43638,49821],
 [41032,41033,43959],[41034,41035,41140],[41037,41038,44129],
 [41043,41045,44301],[41047,41048,41504],
 [41073,41074,41108,41109,44628,44629,44759,45354],
 [41075,41110,44630],[41076,41111,44631],[41077,41112,44632],
 [41078,41079,41081,41113,41114,41116,41384,41386,44633,44634,
      44636,44906,44931,45499,45526,49471],[41080,41115,44635],
 [41082,41117,44637],[41083,41118,44638],[41084,41119,44639],
 [41085,41120,44640],
 [41086,41087,41121,41122,41564,44641,44642,45679],
 [41088,41123,44643],[41089,41124,44644],
 [41090,41091,41125,41126,41630,44645,44646,45150],
 [41092,41127,44647],[41093,41128,44648],[41094,41129,44649],
 [41095,41097,41130,41132,41691,44650,44652,45211],
 [41096,41131,44651],[41098,41133,44653],[41099,41134,44654],
 [41100,41135,44655],[41101,41136,44656],
 [41102,41103,41137,41138,41732,44657,44658,45252],
 [41104,41139,44659],[41142,41143,45915],[41147,41152,41388],
 [41148,41150,46087],[41155,41157,41565],[41175,41176,46443],
 [41180,41186,41389],[41181,41183,46615],[41192,46846],
 [41207,41208,46939],[41212,41219,41390],[41213,41215,47111],
 [41220,41223,41566],[41224,41226,41631],[41230,41231,41699],
 [41236,41238,41733],[41241,41271,47436],[41242,41272,47437],
 [41243,41273,47438],
 [41244,41245,41246,41274,41275,41276,41410,41435,47439,47440,
      47441,47576,47600,48010,48011,49795],[41247,41277,47442],
 [41248,41278,47443],[41249,41279,47444],[41250,41280,47445],
 [41251,41281,47446],[41253,41283,47448],[41254,41284,47449],
 [41255,41285,47450],[41256,41286,47451],[41257,41287,47452],
 [41258,41288,47453],[41259,41289,47454],[41260,41290,47455],
 [41261,41291,47456],[41262,41292,47457],[41263,41293,47458],
 [41264,41294,47459],[41265,41295,47460],[41266,41296,47461],
 [41267,41297,47462],[41268,41298,47463],[41269,41299,47464],
 [41303,41305,48417],[41304,41309,41439],
 [41331,41333,41334,41412,41459,48795,48843,49819],
 [41358,41360,49146],[41361,41362,41482],[41367,49339],
 [41416,41463,49823],[41417,41464,49824],[41418,41465,49825],
 [41419,41466,49826],[41420,41467,49827],[41421,41468,49828],
 [41422,41469,49829],[41423,41470,49830],
 [41424,41431,41471,41478,41686,49831,49838,50575],
 [41425,41472,49832],[41426,41473,49833],[41427,41474,49834],
 [41428,41430,41475,41477,41715,49835,49837,50604],
 [41429,41476,49836],[41432,41479,49839],[41433,41480,49840],
 [41434,41481,49841],[41513,41521,41676],[41519,41520,41720],
 [41775,41777,43294],[41776,41778,41886],[41783,41784,43553],
 [41786,41790,43637],[41788,41791,42241],[41794,41795,43806],
 [41800,41801,43893],[41805,41806,43938],
 [41812,41814,41815,41851,41920,43960,43961,44660],
 [41820,41821,44219],
 [41823,41827,41828,42194,42284,44303,44304,49864],
 [41837,41838,44559],[41842,41843,44604],[41853,41922,44662],
 [41854,41923,44663],[41855,41924,44664],
 [41856,41857,41925,41926,44665,44666,44849,46005],
 [41858,41859,41863,41927,41928,41932,42164,42168,44667,44668,
      44672,44908,44933,46060,46089,49473],[41860,41929,44669],
 [41861,41930,44670],[41862,41931,44671],[41864,41933,44673],
 [41865,41934,44674],
 [41866,41867,41868,41935,41936,41937,42344,42345,44675,44676,
      44677,45085,45102,46240,46258,51426],[41869,41938,44678],
 [41870,41939,44679],[41871,41940,44680],[41872,41941,44681],
 [41873,41874,41942,41943,44682,44683,45189,46345],
 [41875,41944,44684],[41876,41945,44685],[41877,41946,44686],
 [41878,41879,41947,41948,44687,44688,45234,46390],
 [41880,41949,44689],[41881,41950,44690],[41882,41951,44691],
 [41883,41952,44692],[41884,41953,44693],[41891,41892,45444],
 [41893,41896,42166],[41894,41898,45528],[41902,41903,45697],
 [41905,41906,42410],[41908,41909,45784],[41910,41912,42471],
 [41913,41914,45829],[41917,41918,42512],[41958,41959,46533],
 [41960,41966,42169],[41961,41965,46617],[41969,41970,46786],
 [41975,41976,46873],[41980,41981,46918],[41990,41991,47029],
 [41992,41999,42170],[41993,41997,47113],[42000,42003,42346],
 [42001,42002,47282],[42004,42006,42411],[42007,42008,47369],
 [42010,42011,42479],[42012,42013,47414],[42016,42018,42513],
 [42021,42022,47494],
 [42024,42025,42028,42190,42217,47578,47602,49797],
 [42038,42039,47834],[42043,42044,47879],[42051,42052,47929],
 [42054,42058,48013],[42055,42056,42215],[42062,42063,48182],
 [42068,42069,48269],[42073,42074,48314],[42080,42081,48335],
 [42083,42087,48419],[42084,42089,42219],[42091,42092,48588],
 [42097,42098,48675],[42102,42103,48720],[42110,42137,48741],
 [42111,42114,42115,42138,42141,42142,42192,42262,48742,48745,
      48746,48797,48866,49147,49148,49842],[42112,42139,48743],
 [42113,42140,48744],[42116,42143,48747],[42117,42144,48748],
 [42118,42145,48749],
 [42119,42120,42146,42147,48750,48751,48966,49317],
 [42121,42148,48752],[42122,42149,48753],[42123,42150,48754],
 [42124,42151,48755],
 [42125,42126,42152,42153,48756,48757,49053,49404],
 [42127,42154,48758],[42128,42155,48759],[42129,42156,48760],
 [42130,42131,42157,42158,48761,48762,49098,49449],
 [42132,42159,48763],[42133,42160,48764],[42134,42161,48765],
 [42135,42162,48766],[42136,42163,48767],[42172,42173,49642],
 [42178,42179,49729],[42183,42184,49774],[42196,42286,49866],
 [42197,42198,42287,42288,49867,49868,49942,50912],
 [42199,42289,49869],[42200,42290,49870],[42201,42291,49871],
 [42202,42292,49872],
 [42203,42204,42211,42293,42294,42301,42456,42466,49873,49874,
      49881,50029,50036,50999,51016,52181],[42205,42295,49875],
 [42206,42296,49876],[42207,42297,49877],
 [42208,42209,42210,42298,42299,42300,42495,42500,49878,49879,
      49880,50074,50080,51044,51045,52360],[42212,42302,49882],
 [42213,42303,49883],[42214,42304,49884],[42216,50129],
 [42221,42222,50218],[42227,42228,50305],[42232,42233,50350],
 [42244,42245,50471],[42250,42251,50558],[42255,42256,50603],
 [42263,50659],[42266,42267,50702],[42272,42273,50789],
 [42277,42278,50834],[42307,42308,51102],[42313,42314,51189],
 [42318,42319,51234],[42326,42327,51273],[42332,42333,51360],
 [42337,42338,51405],[42350,42351,51513],[42355,42356,51558],
 [42364,42380,51580],[42365,42381,51581],[42366,42382,51582],
 [42367,42368,42383,42384,51583,51584,51649,51769],
 [42369,42385,51585],[42370,42386,51586],[42371,42387,51587],
 [42372,42373,42388,42389,51588,51589,51694,51814],
 [42374,42390,51590],[42375,42391,51591],[42376,42392,51592],
 [42377,42393,51593],[42378,42394,51594],[42398,42399,51874],
 [42403,42404,51919],[42412,42413,51965],[42417,42418,52010],
 [42425,42426,52043],[42430,42431,52088],[42437,42438,52109],
 [42442,42443,52154],[42450,42460,52175],[42451,42461,52176],
 [42452,42462,52177],[42455,42465,52180],[42457,42467,52182],
 [42458,42468,52183],[42459,42469,52184],[42472,42473,52290],
 [42480,42481,52318],[42487,42488,52339],[42496,42501,52361],
 [42497,42502,52362],[42498,42503,52363],[42499,42504,52364],
 [42516,42520,43296],[42517,42519,42627],[42521,42523,43464],
 [42527,42533,43639],[42529,42532,42982],[42535,42537,43807],
 [42553,42557,43962],[42555,42556,42661],[42558,42560,44130],
 [42564,42570,44305],[42568,42569,43025],[42572,42574,44473],
 [42592,42695,44694],
 [42594,42596,42728,42730,44727,44729,44760,46940],
 [42595,42729,44728],[42597,42731,44730],[42598,42732,44731],
 [42599,42600,42606,42733,42734,42740,42905,42911,44732,44733,
      44739,44910,44935,47084,47115,49475],[42601,42735,44734],
 [42602,42736,44735],[42603,42737,44736],[42604,42738,44737],
 [42605,42739,44738],
 [42607,42608,42610,42741,42742,42744,43085,43087,44740,44741,
      44743,45086,45103,47264,47283,51427],[42609,42743,44742],
 [42611,42613,42745,42747,43152,44744,44746,45151],
 [42612,42746,44745],[42614,42748,44747],[42615,42749,44748],
 [42616,42750,44749],
 [42617,42618,42751,42752,43220,44750,44751,45219],
 [42619,42753,44752],[42620,42754,44753],[42621,42755,44754],
 [42622,42756,44755],
 [42623,42625,42757,42759,43254,44756,44758,45253],
 [42624,42758,44757],[42629,42631,45355],[42634,42637,42907],
 [42635,42641,45530],[42643,42645,45698],[42646,42647,43151],
 [42651,42653,43212],[42658,42659,43253],[42663,42665,45916],
 [42668,42673,42909],[42669,42675,46091],[42676,42678,43086],
 [42677,42679,46259],[42696,42698,46444],[42701,42707,42910],
 [42702,42708,46619],[42710,42712,46787],[42762,42821,47465],
 [42763,42822,47466],[42764,42823,47467],
 [42765,42766,42771,42824,42825,42830,42931,42960,47468,47469,
      47474,47580,47604,48416,48421,49799],[42767,42826,47470],
 [42768,42827,47471],[42769,42828,47472],[42770,42829,47473],
 [42772,42831,47475],[42774,42833,47477],[42776,42835,47479],
 [42777,42836,47480],[42778,42837,47481],[42779,42838,47482],
 [42780,42839,47483],[42781,42840,47484],[42782,42841,47485],
 [42783,42842,47486],[42784,42843,47487],[42785,42844,47488],
 [42786,42845,47489],[42787,42846,47490],[42788,42847,47491],
 [42789,42848,47492],[42790,42849,47493],[42795,42801,48015],
 [42796,42797,42956],[42803,42805,48183],
 [42852,42855,42858,42933,43005,48799,48868,49844],
 [42860,42862,48967],[42879,42885,49150],[42882,42883,43003],
 [42887,42889,49318],[42913,42915,49643],[42935,43046,49885],
 [42937,43066,49905],
 [42938,42940,43067,43069,49906,49908,49943,51274],
 [42939,43068,49907],[42941,43070,49909],[42942,43071,49910],
 [42943,43072,49911],[42944,43073,49912],
 [42945,42952,43074,43081,43207,49913,49920,51377],
 [42946,43075,49914],[42947,43076,49915],[42948,43077,49916],
 [42949,42951,43078,43080,43236,49917,49919,51406],
 [42950,43079,49918],[42953,43082,49921],[42954,43083,49922],
 [42955,43084,49923],[42962,42964,50219],[42985,42987,50472],
 [43007,43009,50703],[43028,43030,50913],[43034,43042,43197],
 [43040,43041,43241],[43048,43050,51103],[43105,43136,51595],
 [43106,43137,51596],[43107,43138,51597],[43108,43139,51598],
 [43109,43140,51599],[43110,43141,51600],[43111,43142,51601],
 [43112,43143,51602],[43113,43144,51603],[43114,43145,51604],
 [43115,43146,51605],[43116,43147,51606],[43117,43148,51607],
 [43118,43149,51608],[43119,43150,51609],[43302,43303,43608],
 [43305,43308,44981],[43310,43311,43788],[43315,43878],
 [43332,43399,45290],[43334,43401,45292],[43335,43402,45293],
 [43336,43403,45294],
 [43337,43340,43343,43404,43407,43410,43610,43613,45295,45298,
      45301,45504,45576,46590,46665,49521],[43338,43405,45296],
 [43339,43406,45297],[43341,43408,45299],[43342,43409,45300],
 [43344,43411,45302],[43346,43413,45304],[43347,43414,45305],
 [43348,43415,45306],
 [43349,43350,43416,43417,43854,45307,45308,46834],
 [43351,43418,45309],[43352,43419,45310],[43353,43420,45311],
 [43354,43356,43421,43423,43915,45312,45314,46895],
 [43355,43422,45313],[43357,43424,45315],[43358,43425,45316],
 [43359,43426,45317],[43360,43427,45318],
 [43361,43362,43428,43429,43956,45319,45320,46936],
 [43363,43430,45321],[43371,43376,43612],[43374,43377,46137],
 [43379,43381,43789],[43436,43443,43614],[43439,43442,47161],
 [43444,43447,43790],[43448,43450,43855],[43449,43873],
 [43454,43455,43923],[43460,43462,43957],[43468,43469,43634],
 [43470,43473,47626],[43528,43533,43663],[43529,43532,48467],
 [43555,43558,43636],[43557,43560,48845],[43584,43587,49196],
 [43585,43586,43706],[43648,43655,43910],[43652,43654,43939],
 [43683,43707,50372],[43687,43750,50415],[43688,43751,50416],
 [43689,43752,50417],[43690,43753,50418],[43691,43754,50419],
 [43692,43755,50420],[43693,43756,50421],[43694,43757,50422],
 [43695,43758,50423],[43696,43759,50424],[43697,43760,50425],
 [43698,43761,50426],[43699,43762,50427],[43700,43763,50428],
 [43701,43764,50429],[43702,43765,50430],[43703,43766,50431],
 [43704,43767,50432],[43705,43768,50433],[43737,43745,43900],
 [43743,43744,43944],[43968,43969,44274],[43973,43974,45024],
 [43976,43977,44454],[44003,44006,44276],[44008,44009,45619],
 [44015,44016,44520],[44020,44022,44581],[44027,44028,44622],
 [44032,44065,45851],[44033,44066,45852],[44034,44067,45853],
 [44035,44068,45854],[44036,44069,45855],
 [44037,44042,44043,44070,44075,44076,44278,44279,45856,45861,
      45862,46065,46180,46592,46708,49564],[44038,44071,45857],
 [44039,44072,45858],[44040,44073,45859],[44041,44074,45860],
 [44044,44077,45863],[44046,44079,45865],[44048,44081,45867],
 [44049,44082,45868],[44050,44083,45869],[44051,44084,45870],
 [44052,44085,45871],[44053,44086,45872],[44054,44087,45873],
 [44055,44088,45874],[44056,44089,45875],[44057,44090,45876],
 [44058,44091,45877],[44059,44092,45878],[44060,44093,45879],
 [44061,44094,45880],[44062,44095,45881],[44063,44096,45882],
 [44102,44109,44280],[44107,44108,47204],[44110,44113,44456],
 [44114,44116,44521],[44120,44121,44589],[44126,44128,44623],
 [44134,44135,44300],[44138,44139,47669],[44165,44166,44325],
 [44168,44169,48104],[44194,44199,44329],[44197,44198,48510],
 [44221,44224,44302],[44225,44226,48888],[44248,44269],
 [44251,44252,44253,44372,44373,49218,49239,50624],
 [44254,44270],[44314,44321,44576],[44318,44320,44605],
 [44396,44416,50856],[44397,44417,50857],[44399,44419,50859],
 [44400,44420,50860],[44401,44421,50861],[44402,44422,50862],
 [44403,44411,44423,44431,44566,50863,50871,51196],
 [44404,44424,50864],[44405,44425,50865],[44406,44426,50866],
 [44407,44427,50867],[44408,44428,50868],
 [44409,44410,44429,44430,44610,50869,50870,51240],
 [44412,44432,50872],[44413,44433,50873],[44414,44434,50874],
 [44700,44701,44706,44909,44934,46588,46618,49474],
 [44763,44764,44765,44905,44930,47549,47550,49470],
 [44793,44794,47984],[44795,44796,44955],[44801,44802,48164],
 [44822,44823,48390],[44824,44829,44959],[44830,44831,48570],
 [44850,44851,44854,44907,44932,48768,48771,49472],
 [44858,44859,48948],[44877,44878,49119],[44881,44882,45002],
 [44885,44886,49299],
 [44911,44912,44936,44937,49476,49477,49624,49924],
 [44913,44938,49478],[44914,44939,49479],[44915,44940,49480],
 [44916,44941,49481],[44917,44942,49482],[44918,44943,49483],
 [44919,44926,44944,44951,45206,49484,49491,49746],
 [44920,44945,49485],[44921,44946,49486],[44922,44947,49487],
 [44923,44925,44948,44950,45235,49488,49490,49775],
 [44924,44949,49489],[44927,44952,49492],[44928,44953,49493],
 [44929,44954,49494],[44956,50133],[44960,44961,50200],
 [44983,44984,50453],[45005,45006,50684],[45026,45027,50894],
 [45033,45041,45196],[45039,45040,45240],[45046,45047,51084],
 [45065,45066,51255],[45087,45104,51428],[45088,45105,51429],
 [45089,45106,51430],[45090,45107,51431],[45091,45108,51432],
 [45092,45109,51433],[45093,45110,51434],[45094,45111,51435],
 [45095,45112,51436],[45096,45113,51437],[45097,45114,51438],
 [45098,45115,51439],[45099,45116,51440],[45100,45117,51441],
 [45101,45118,51442],[45358,45361,47551],[45359,45360,45525],
 [45370,45371,47795],[45375,45377,47856],[45382,45383,47897],
 [45388,45390,45391,45500,45550,47985,47986,49495],
 [45400,45401,48230],[45405,45407,48291],[45412,45413,48332],
 [45417,45420,48392],[45419,45424,45554],[45429,45430,48636],
 [45431,48673],[45434,45436,48697],[45441,45442,48738],
 [45445,45448,48770],[45446,45449,45527],[45457,45458,49014],
 [45462,45464,49075],[45469,45470,49116],[45472,45475,49121],
 [45476,45477,45597],[45484,45485,49365],[45489,45491,49426],
 [45496,45497,49467],[45502,45574,49519],[45506,45578,49523],
 [45507,45579,49524],[45508,45580,49525],[45509,45581,49526],
 [45510,45511,45582,45583,49527,49528,49690,50519],
 [45512,45584,49529],[45513,45585,49530],[45514,45586,49531],
 [45515,45517,45587,45589,49532,49534,49751,50580],
 [45516,45588,49533],[45518,45590,49535],[45519,45591,49536],
 [45520,45592,49537],[45521,45593,49538],
 [45522,45523,45594,45595,49539,49540,49792,50621],
 [45524,45596,49541],[45535,45536,49990],[45539,45546,45801],
 [45540,45542,50051],[45543,45545,45830],[45547,45548,50092],
 [45559,45560,50266],[45564,45566,50327],[45571,45572,50368],
 [45604,45605,50750],[45609,45611,50811],[45616,45617,50852],
 [45625,45626,50960],[45628,45636,45791],[45630,45632,51021],
 [45634,45635,45835],[45637,45638,51062],[45645,45646,51150],
 [45650,45652,51211],[45657,45658,51252],[45664,45665,51321],
 [45669,45671,51382],[45676,45677,51423],[45682,45683,51474],
 [45687,45689,51535],[45694,45695,51576],[45699,45700,51610],
 [45704,45706,51671],[45711,45712,51712],[45715,45716,51730],
 [45720,45722,51791],[45727,45728,51832],[45730,45731,51835],
 [45732,51866],[45735,45737,51896],[45742,45743,51937],
 [45747,45760,51941],[45748,45761,51942],
 [45749,45751,45762,45764,51943,51945,51987,52065],
 [45750,45763,51944],[45752,45765,51946],[45753,45766,51947],
 [45754,45767,51948],[45755,45768,51949],
 [45756,45757,45769,45770,51950,51951,52028,52106],
 [45758,45771,51952],[45774,45776,52131],[45781,45782,52172],
 [45785,45787,52186],[45792,45793,52227],[45795,45797,52231],
 [45802,45803,52272],[45807,45822,52283],[45808,45823,52284],
 [45809,45824,52285],[45810,45825,52286],[45813,45828,52289],
 [45819,45820,52336],[45832,45833,52372],[45837,45838,52382],
 [45841,45842,52388],[45844,45845,52391],[45919,45924,47553],
 [45920,45921,46086],[45927,45929,47730],[45949,45954,47988],
 [45951,45952,46111],[45957,45959,48165],[45978,45983,48394],
 [45980,45985,46115],[45986,45988,48571],[46006,46011,48772],
 [46007,46010,46088],[46014,46016,48949],
 [46033,46037,46038,46063,46158,49122,49123,49542],
 [46041,46043,49300],[46061,46113,49497],
 [46067,46069,46182,46184,49566,49568,49625,50895],
 [46068,46183,49567],[46070,46185,49569],[46071,46186,49570],
 [46072,46187,49571],[46073,46188,49572],
 [46074,46082,46189,46197,46352,49573,49581,49736],
 [46075,46190,49574],[46076,46191,49575],[46077,46192,49576],
 [46078,46193,49577],[46079,46194,49578],
 [46080,46081,46195,46196,46396,49579,49580,49780],
 [46083,46198,49582],[46084,46199,49583],[46085,46200,49584],
 [46092,46094,49925],[46100,46107,46362],[46104,46106,46391],
 [46116,46118,50201],[46139,46141,50454],[46161,46163,50685],
 [46202,46204,51085],[46221,46223,51256],[46243,46276,51444],
 [46244,46277,51445],[46245,46278,51446],[46246,46279,51447],
 [46247,46280,51448],[46248,46281,51449],[46249,46282,51450],
 [46250,46283,51451],[46251,46284,51452],[46252,46285,51453],
 [46253,46286,51454],[46254,46287,51455],[46255,46288,51456],
 [46256,46289,51457],[46257,46290,51458],
 [46416,46422,46423,46594,46729,47089,47225,49585],
 [46428,46430,46835],[46434,46435,46903],[46440,46442,46937],
 [46447,46453,47554],[46448,46449,46614],[46477,46483,47989],
 [46479,46480,46639],[46506,46512,48395],[46508,46513,46643],
 [46534,46540,48773],[46535,46538,46616],[46561,46567,49124],
 [46565,46566,46686],[46591,46687,49543],[46596,46731,49587],
 [46597,46732,49588],[46598,46733,49589],[46599,46734,49590],
 [46600,46735,49591],[46601,46736,49592],[46602,46737,49593],
 [46603,46738,49594],[46604,46739,49595],[46605,46740,49596],
 [46606,46741,49597],[46607,46742,49598],[46608,46743,49599],
 [46609,46744,49600],[46610,46745,49601],[46611,46746,49602],
 [46612,46747,49603],[46613,46748,49604],[46628,46635,46890],
 [46632,46634,46919],[46717,46725,46880],[46723,46724,46924],
 [46943,46950,47555],[46944,46945,47110],[46951,46954,47731],
 [46955,46957,47796],[46961,46962,47864],[46967,46969,47898],
 [46973,46980,47990],[46975,46976,47135],[46981,46984,48166],
 [46985,46987,48231],[46986,48256],[46991,46992,48299],
 [46997,46999,48333],
 [47002,47004,47009,47085,47139,48391,48396,49499],
 [47014,47016,48637],[47020,47021,48705],[47026,47028,48739],
 [47030,47037,48774],[47031,47034,47112],[47038,47041,48950],
 [47042,47044,49015],[47048,47049,49083],[47054,47056,49117],
 [47057,47064,49125],[47061,47062,47182],[47065,47068,49301],
 [47069,47071,49366],[47075,47076,49434],[47081,47083,49468],
 [47087,47184,49544],
 [47091,47094,47245,47248,49605,49608,49626,51257],
 [47092,47246,49606],[47093,47247,49607],
 [47095,47097,47249,47251,49609,49611,49691,51322],
 [47096,47250,49610],[47098,47252,49612],[47099,47253,49613],
 [47100,47254,49614],
 [47101,47102,47255,47256,49615,49616,49759,51390],
 [47103,47257,49617],[47104,47258,49618],[47105,47259,49619],
 [47106,47260,49620],
 [47107,47109,47261,47263,49621,49623,49793,51424],
 [47108,47262,49622],[47116,47119,49926],[47120,47122,49991],
 [47124,47131,47386],[47126,47127,50059],[47128,47130,47415],
 [47132,47134,50093],[47140,47143,50202],[47144,47146,50267],
 [47150,47151,50335],[47156,47158,50369],[47163,47166,50455],
 [47167,47169,50520],[47173,47174,50588],[47179,47181,50622],
 [47185,47188,50686],[47189,47191,50751],[47195,47196,50819],
 [47201,47203,50853],[47206,47209,50896],[47210,47212,50961],
 [47213,47221,47376],[47216,47217,51029],[47219,47220,47420],
 [47222,47224,51063],[47226,47229,51086],[47230,47232,51151],
 [47236,47237,51219],[47242,47244,51253],
 [47267,47269,47315,47317,51459,51461,51475,51836],
 [47268,47316,51460],[47270,47318,51462],[47271,47319,51463],
 [47272,47320,51464],
 [47273,47274,47321,47322,51465,51466,51543,51904],
 [47275,47323,51467],[47276,47324,51468],[47277,47325,51469],
 [47278,47326,51470],
 [47279,47281,47327,47329,51471,51473,51577,51938],
 [47280,47328,51472],[47284,47286,51611],[47290,47291,51679],
 [47296,47298,51713],[47300,47302,51731],[47306,47307,51799],
 [47312,47314,51833],[47332,47357,51953],[47333,47358,51954],
 [47334,47359,51955],
 [47335,47336,47360,47361,51956,51957,51995,52139],
 [47337,47362,51958],[47338,47363,51959],[47339,47364,51960],
 [47340,47365,51961],
 [47341,47343,47366,47368,51962,51964,52029,52173],
 [47342,47367,51963],[47348,47349,52073],[47354,47356,52107],
 [47371,47372,52194],[47377,47379,52228],[47381,47382,52239],
 [47387,47389,52273],[47396,47398,52309],[47400,47407,52311],
 [47401,47408,52312],[47402,47409,52313],[47403,47410,52314],
 [47405,47412,52316],[47417,47419,52373],[47422,47424,52383],
 [47426,47428,52389],[47429,47431,52392],[47495,47517],
 [47496,47497,47499,47577,47601,48794,48820,49796],
 [47500,47518],[47523,47524,49145],[47526,47527,47647],
 [47581,47605,49800],[47583,47607,49802],[47584,47608,49803],
 [47585,47609,49804],[47586,47610,49805],[47587,47611,49806],
 [47588,47612,49807],
 [47589,47596,47613,47620,47851,49808,49815,50322],
 [47590,47614,49809],[47591,47615,49810],[47592,47616,49811],
 [47593,47595,47617,47619,47880,49812,49814,50351],
 [47594,47618,49813],[47597,47621,49816],[47598,47622,49817],
 [47599,47623,49818],[47678,47686,47841],[47684,47685,47885],
 [47931,47934,48012],[47932,47933,48819],[47959,47960,49170],
 [47961,47962,48082],[48024,48031,48286],[48028,48030,48315],
 [48036,48059,50095],[48040,48063,50099],[48041,48064,50100],
 [48042,48065,50101],[48043,48066,50102],[48044,48067,50103],
 [48045,48068,50104],[48046,48069,50105],[48047,48070,50106],
 [48048,48071,50107],[48049,48072,50108],[48050,48073,50109],
 [48051,48074,50110],[48052,48075,50111],[48053,48076,50112],
 [48054,48077,50113],[48055,48078,50114],[48056,48079,50115],
 [48057,48080,50116],[48058,48081,50117],[48113,48121,48276],
 [48119,48120,48320],[48337,48340,48418],[48338,48343,48823],
 [48365,48370,49174],[48367,48368,48488],[48430,48437,48692],
 [48434,48436,48721],[48442,48490,50120],[48446,48551,50181],
 [48447,48552,50182],[48448,48553,50183],[48450,48555,50185],
 [48451,48556,50186],[48452,48557,50187],[48453,48558,50188],
 [48454,48559,50189],[48455,48560,50190],[48456,48561,50191],
 [48457,48562,50192],[48458,48563,50193],[48459,48564,50194],
 [48460,48565,50195],[48461,48566,50196],[48462,48567,50197],
 [48463,48568,50198],[48464,48569,50199],[48519,48527,48682],
 [48525,48526,48726],[48798,48867,49843],[48800,48869,49845],
 [48801,48870,49846],[48802,48871,49847],[48803,48872,49848],
 [48804,48873,49849],[48805,48874,49850],[48806,48875,49851],
 [48807,48876,49852],
 [48808,48815,48877,48884,49070,49853,49860,50806],
 [48809,48878,49854],[48810,48879,49855],[48811,48880,49856],
 [48812,48814,48881,48883,49099,49857,49859,50835],
 [48813,48882,49858],[48816,48885,49861],[48817,48886,49862],
 [48818,48887,49863],[48891,48977],[48897,48905,49060],
 [48903,48904,49104],[49153,49343],[49159,49166,49421],
 [49163,49165,49450],[49171,49172,50118],[49220,49241,50626],
 [49221,49242,50627],[49222,49243,50628],[49223,49244,50629],
 [49224,49245,50630],[49225,49246,50631],[49226,49247,50632],
 [49227,49235,49248,49256,49411,50633,50641,50796],
 [49228,49249,50634],[49229,49250,50635],[49230,49251,50636],
 [49231,49252,50637],[49232,49253,50638],
 [49233,49234,49254,49255,49455,50639,50640,50840],
 [49236,49257,50642],[49237,49258,50643],[49238,49259,50644],
 [49677,51871],[49894,49901,51206],[49898,49900,51235],
 [49931,49938,51530],[49935,49937,51559],[49948,49955,51666],
 [49952,49954,51695],[49964,49971,51786],[49968,49970,51815],
 [49979,49986,51891],[49983,49985,51920],[49993,50000,51982],
 [49997,49999,52011],[50006,50013,52060],[50010,50012,52089],
 [50018,50025,52126],[50022,50024,52155],[50033,50035,52210],
 [50040,50055,52235],[50041,50063,52243],[50042,50070,52250],
 [50044,50081,52261],[50047,50089,52269],[50048,50090,52270],
 [50049,50091,52271],[50052,50054,52291],[50060,50062,52319],
 [50067,50069,52340],[50077,50086,52366],[50078,50087,52367],
 [50079,50088,52368],[50149,50157,50312],[50155,50156,50356],
 [50900,50908,51520],[50906,50907,51564],[50917,50925,51656],
 [50923,50924,51700],[50933,50941,51776],[50939,50940,51820],
 [50948,50956,51881],[50954,50955,51925],[50962,50970,51972],
 [50968,50969,52016],[50975,50983,52050],[50981,50982,52094],
 [50987,50995,52116],[50993,50994,52160],[51000,51025,52190],
 [51001,51033,52198],[51002,51040,52205],[51003,51046,52211],
 [51007,51059,52224],[51008,51060,52225],[51009,51061,52226],
 [51014,51015,52260],[51023,51024,52296],[51031,51032,52324],
 [51038,51039,52345],[51052,51056,52376],[51053,51057,52377],
 [51054,51058,52378],[51258,51487]],

equalityclasses :=
  Union(~.eqclsgt1,List(Difference([1..Length(~.grps)],
                                   Union(~.eqclsgt1)),i->[i])),

conjugacyclasses :=
  ReadAsFunction(Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                               "/data/3ctsgrp-conjugacyclasses.g"))(),

partitionlengths :=
  ReadAsFunction(Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                               "/data/3ctsgrp-partitionlengths.g"))(),

degrees :=
  ReadAsFunction(Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                               "/data/3ctsgrp-degrees.g"))(),

orbitlengths :=
  ReadAsFunction(Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                               "/data/3ctsgrp-orbitlengths.g"))(),

permgroupgens :=
  ReadAsFunction(Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                               "/data/3ctsgrp-permgroupgens.g"))(),

respectedpartitions :=
  ReadAsFunction(Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                               "/data/3ctsgrp-respectedpartitions.g"))(),

stabilize_digitsum_base2_mod2 :=
[ 10796,10798,10816,10822,11032,11050,11056,11161,11167,11986,
  14753,14771,14777,14882,14888,15707,21612,21618,22437,25462 ],

stabilize_digitsum_base2_mod3 := 
[ 10816,10822,11986,15707 ],

stabilize_digitsum_base3_mod2 :=
[ 6764,6771,6773,6776,6778,6781,6783,6788,6791,6798,6802,6951,
  6953,6956,6958,6961,6963,6968,6971,6978,6982,7338,7341,7343,
  7346,7348,7353,7356,7363,7367,7442,7444,7447,7449,7454,7457,
  7464,7468,7588,7591,7593,7598,7601,7608,7612,7682,7684,7689,
  7692,7699,7703,7813,7818,7821,7828,7832,7899,7902,7909,7913,
  8087,8094,8098,8193,8197,8393,12811,12813,12816,12818,12821,
  12823,12828,12831,12838,12842,13198,13201,13203,13206,13208,
  13213,13216,13223,13227,13302,13304,13307,13309,13314,13317,
  13324,13328,13448,13451,13453,13458,13461,13468,13472,13542,
  13544,13549,13552,13559,13563,13673,13678,13681,13688,13692,
  13759,13762,13769,13773,13947,13954,13958,14053,14057,14253,
  18509,18512,18514,18517,18519,18524,18527,18534,18538,18613,
  18615,18618,18620,18625,18628,18635,18639,18759,18762,18764,
  18769,18772,18779,18783,18853,18855,18860,18863,18870,18874,
  18984,18989,18992,18999,19003,19070,19073,19080,19084,19258,
  19265,19269,19364,19368,19564,29022,29024,29027,29029,29034,
  29037,29044,29048,29168,29171,29173,29178,29181,29188,29192,
  29262,29264,29269,29272,29279,29283,29393,29398,29401,29408,
  29412,29479,29482,29489,29493,29667,29674,29678,29773,29777,
  29973,31668,31671,31673,31678,31681,31688,31692,31762,31764,
  31769,31772,31779,31783,31893,31898,31901,31908,31912,31979,
  31982,31989,31993,32167,32174,32178,32273,32277,32473,35147,
  35149,35154,35157,35164,35168,35278,35283,35286,35293,35297,
  35364,35367,35374,35378,35552,35559,35563,35658,35662,35858,
  37303,37308,37311,37318,37322,37389,37392,37399,37403,37577,
  37584,37588,37683,37687,37883,40099,40102,40109,40113,40287,
  40294,40298,40393,40397,40593,41887,41894,41898,41993,41997,
  42193,45328,45332,45528,47113 ],

freeproductcandidates :=
[ 14962,15870,20352,26912,26913,26916,27231,27527,28923,33592, 
  33874,33916,43904,44207,44547 ],

freeproductlikes :=
[ 421,437,1565,1585,1916,1922,2604,2631,2647,2648,2651,3796,3812,
  4104,4105,4423,8873,8888,8889,8892,10017,10018,10037,10052,
  10368,10371,14892,14908,14912,14946,14961,14965,15431,15446,
  15447,15450,15848,15869,15986,16019,16041,16164,16306,16327,
  16330,16341,16352,20334,20362,26897,27228,33582,33925,38457,
  38472,38473,38476,38791,39686,44191,44261,44262,44265,44550,
  49073,49371,49392,49457 ],

intransitivemodulo :=
  ReadAsFunction(Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                               "/data/3ctsgrp-intransitivemodulo.g"))(),

finiteorbits := ReadAsFunction(
                  Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                                "/data/3ctsgrp-finite-orbits.g"))()

);

#############################################################################
##
#E  3ctsgrpdata.g . . . . . . . . . . . . . . . . . . . . . . . . . ends here