/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.plugins;

import java.io.Closeable;
import java.io.IOException;
import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import org.apache.lucene.analysis.util.CharFilterFactory;
import org.apache.lucene.analysis.util.TokenFilterFactory;
import org.apache.lucene.analysis.util.TokenizerFactory;
import org.apache.lucene.codecs.Codec;
import org.apache.lucene.codecs.DocValuesFormat;
import org.apache.lucene.codecs.PostingsFormat;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.action.admin.cluster.node.info.PluginsInfo;
import org.elasticsearch.bootstrap.JarHell;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.component.LifecycleComponent;
import org.elasticsearch.common.inject.Module;
import org.elasticsearch.common.io.FileSystemUtils;
import org.elasticsearch.common.logging.ESLogger;
import org.elasticsearch.common.logging.Loggers;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.plugins.Plugin;
import org.elasticsearch.plugins.PluginInfo;
import org.elasticsearch.plugins.SitePlugin;

public class PluginsService
extends AbstractComponent {
    private final List<Tuple<PluginInfo, Plugin>> plugins;
    private final PluginsInfo info;
    private final Map<Plugin, List<OnModuleReference>> onModuleReferences;

    public PluginsService(Settings settings, Path pluginsDirectory, Collection<Class<? extends Plugin>> classpathPlugins) {
        super(settings);
        ArrayList<Tuple<PluginInfo, Plugin>> tupleBuilder = new ArrayList<Tuple<PluginInfo, Plugin>>();
        for (Class<? extends Plugin> clazz : classpathPlugins) {
            Plugin plugin = this.loadPlugin(clazz, settings);
            PluginInfo pluginInfo = new PluginInfo(plugin.name(), plugin.description(), false, "NA", true, clazz.getName(), false);
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("plugin loaded from classpath [{}]", pluginInfo);
            }
            tupleBuilder.add(new Tuple<PluginInfo, Plugin>(pluginInfo, plugin));
        }
        if (pluginsDirectory != null) {
            try {
                List<Bundle> bundles = PluginsService.getPluginBundles(pluginsDirectory);
                tupleBuilder.addAll(this.loadBundles(bundles));
            }
            catch (IOException ex) {
                throw new IllegalStateException("Unable to initialize plugins", ex);
            }
        }
        this.plugins = Collections.unmodifiableList(tupleBuilder);
        this.info = new PluginsInfo();
        for (Tuple tuple : this.plugins) {
            this.info.add((PluginInfo)tuple.v1());
        }
        HashMap<String, Plugin> jvmPlugins = new HashMap<String, Plugin>();
        ArrayList<String> arrayList = new ArrayList<String>();
        for (Tuple<PluginInfo, Plugin> tuple : this.plugins) {
            PluginInfo info = tuple.v1();
            if (info.isJvm()) {
                jvmPlugins.put(info.getName(), tuple.v2());
            }
            if (!info.isSite()) continue;
            arrayList.add(info.getName());
        }
        String[] mandatoryPlugins = settings.getAsArray("plugin.mandatory", null);
        if (mandatoryPlugins != null) {
            HashSet<String> missingPlugins = new HashSet<String>();
            for (String mandatoryPlugin : mandatoryPlugins) {
                if (jvmPlugins.containsKey(mandatoryPlugin) || arrayList.contains(mandatoryPlugin) || missingPlugins.contains(mandatoryPlugin)) continue;
                missingPlugins.add(mandatoryPlugin);
            }
            if (!missingPlugins.isEmpty()) {
                throw new ElasticsearchException("Missing mandatory plugins [" + Strings.collectionToDelimitedString(missingPlugins, ", ") + "]", new Object[0]);
            }
        }
        this.logger.info("loaded {}, sites {}", jvmPlugins.keySet(), arrayList);
        HashMap onModuleReferences = new HashMap();
        for (Plugin plugin : jvmPlugins.values()) {
            ArrayList<OnModuleReference> list = new ArrayList<OnModuleReference>();
            for (Method method : plugin.getClass().getMethods()) {
                if (!method.getName().equals("onModule")) continue;
                if (method.getParameterTypes().length == 0 || method.getParameterTypes().length > 1) {
                    this.logger.warn("Plugin: {} implementing onModule with no parameters or more than one parameter", plugin.name());
                    continue;
                }
                Class<?> moduleClass = method.getParameterTypes()[0];
                if (!Module.class.isAssignableFrom(moduleClass)) {
                    this.logger.warn("Plugin: {} implementing onModule by the type is not of Module type {}", plugin.name(), moduleClass);
                    continue;
                }
                list.add(new OnModuleReference(moduleClass, method));
            }
            if (list.isEmpty()) continue;
            onModuleReferences.put(plugin, list);
        }
        this.onModuleReferences = Collections.unmodifiableMap(onModuleReferences);
    }

    public List<Tuple<PluginInfo, Plugin>> plugins() {
        return this.plugins;
    }

    public void processModules(Iterable<Module> modules) {
        for (Module module : modules) {
            this.processModule(module);
        }
    }

    public void processModule(Module module) {
        for (Tuple<PluginInfo, Plugin> plugin : this.plugins()) {
            List<OnModuleReference> references = this.onModuleReferences.get(plugin.v2());
            if (references == null) continue;
            for (OnModuleReference reference : references) {
                if (!reference.moduleClass.isAssignableFrom(module.getClass())) continue;
                try {
                    reference.onModuleMethod.invoke((Object)plugin.v2(), module);
                }
                catch (Exception e) {
                    this.logger.warn("plugin {}, failed to invoke custom onModule method", e, plugin.v2().name());
                }
            }
        }
    }

    public Settings updatedSettings() {
        HashMap<String, String> foundSettings = new HashMap<String, String>();
        Settings.Builder builder = Settings.settingsBuilder();
        for (Tuple<PluginInfo, Plugin> plugin : this.plugins) {
            Settings settings = plugin.v2().additionalSettings();
            for (String setting : settings.getAsMap().keySet()) {
                String oldPlugin = foundSettings.put(setting, plugin.v1().getName());
                if (oldPlugin == null) continue;
                throw new IllegalArgumentException("Cannot have additional setting [" + setting + "] " + "in plugin [" + plugin.v1().getName() + "], already added in plugin [" + oldPlugin + "]");
            }
            builder.put(settings);
        }
        return builder.put(this.settings).build();
    }

    public Collection<Module> nodeModules() {
        ArrayList<Module> modules = new ArrayList<Module>();
        for (Tuple<PluginInfo, Plugin> plugin : this.plugins) {
            modules.addAll(plugin.v2().nodeModules());
        }
        return modules;
    }

    public Collection<Class<? extends LifecycleComponent>> nodeServices() {
        ArrayList<Class<? extends LifecycleComponent>> services = new ArrayList<Class<? extends LifecycleComponent>>();
        for (Tuple<PluginInfo, Plugin> plugin : this.plugins) {
            services.addAll(plugin.v2().nodeServices());
        }
        return services;
    }

    public Collection<Module> indexModules(Settings indexSettings) {
        ArrayList<Module> modules = new ArrayList<Module>();
        for (Tuple<PluginInfo, Plugin> plugin : this.plugins) {
            modules.addAll(plugin.v2().indexModules(indexSettings));
        }
        return modules;
    }

    public Collection<Class<? extends Closeable>> indexServices() {
        ArrayList<Class<? extends Closeable>> services = new ArrayList<Class<? extends Closeable>>();
        for (Tuple<PluginInfo, Plugin> plugin : this.plugins) {
            services.addAll(plugin.v2().indexServices());
        }
        return services;
    }

    public Collection<Module> shardModules(Settings indexSettings) {
        ArrayList<Module> modules = new ArrayList<Module>();
        for (Tuple<PluginInfo, Plugin> plugin : this.plugins) {
            modules.addAll(plugin.v2().shardModules(indexSettings));
        }
        return modules;
    }

    public Collection<Class<? extends Closeable>> shardServices() {
        ArrayList<Class<? extends Closeable>> services = new ArrayList<Class<? extends Closeable>>();
        for (Tuple<PluginInfo, Plugin> plugin : this.plugins) {
            services.addAll(plugin.v2().shardServices());
        }
        return services;
    }

    public PluginsInfo info() {
        return this.info;
    }

    static List<Bundle> getPluginBundles(Path pluginsDirectory) throws IOException {
        ESLogger logger = Loggers.getLogger(PluginsService.class);
        if (!FileSystemUtils.isAccessibleDirectory(pluginsDirectory, logger)) {
            return Collections.emptyList();
        }
        ArrayList<Bundle> bundles = new ArrayList<Bundle>();
        bundles.add(new Bundle());
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(pluginsDirectory);){
            for (Path plugin : stream) {
                Bundle bundle;
                if (FileSystemUtils.isHidden(plugin)) {
                    logger.trace("--- skip hidden plugin file[{}]", plugin.toAbsolutePath());
                    continue;
                }
                logger.trace("--- adding plugin [{}]", plugin.toAbsolutePath());
                PluginInfo info = PluginInfo.readFromProperties(plugin);
                ArrayList<URL> urls = new ArrayList<URL>();
                if (info.isJvm()) {
                    try (DirectoryStream<Path> jarStream = Files.newDirectoryStream(plugin, "*.jar");){
                        for (Path jar : jarStream) {
                            urls.add(jar.toUri().toURL());
                        }
                    }
                }
                if (info.isJvm() && !info.isIsolated()) {
                    bundle = (Bundle)bundles.get(0);
                } else {
                    bundle = new Bundle();
                    bundles.add(bundle);
                }
                bundle.plugins.add(info);
                bundle.urls.addAll(urls);
            }
        }
        return bundles;
    }

    private List<Tuple<PluginInfo, Plugin>> loadBundles(List<Bundle> bundles) {
        ArrayList<Tuple<PluginInfo, SitePlugin>> plugins = new ArrayList<Tuple<PluginInfo, SitePlugin>>();
        for (Bundle bundle : bundles) {
            try {
                ArrayList<URL> jars = new ArrayList<URL>(Arrays.asList(JarHell.parseClassPath()));
                jars.addAll(bundle.urls);
                JarHell.checkJarHell(jars.toArray(new URL[0]));
            }
            catch (Exception e) {
                throw new IllegalStateException("failed to load bundle " + bundle.urls + " due to jar hell", e);
            }
            URLClassLoader loader = URLClassLoader.newInstance(bundle.urls.toArray(new URL[0]), this.getClass().getClassLoader());
            for (PluginInfo pluginInfo : bundle.plugins) {
                Plugin plugin;
                if (pluginInfo.isJvm()) {
                    PluginsService.reloadLuceneSPI(loader);
                    Class<? extends Plugin> pluginClass = this.loadPluginClass(pluginInfo.getClassname(), loader);
                    plugin = this.loadPlugin(pluginClass, this.settings);
                } else {
                    plugin = new SitePlugin(pluginInfo.getName(), pluginInfo.getDescription());
                }
                plugins.add(new Tuple<PluginInfo, SitePlugin>(pluginInfo, (SitePlugin)plugin));
            }
        }
        return Collections.unmodifiableList(plugins);
    }

    static void reloadLuceneSPI(ClassLoader loader) {
        PostingsFormat.reloadPostingsFormats((ClassLoader)loader);
        DocValuesFormat.reloadDocValuesFormats((ClassLoader)loader);
        Codec.reloadCodecs((ClassLoader)loader);
        CharFilterFactory.reloadCharFilters((ClassLoader)loader);
        TokenFilterFactory.reloadTokenFilters((ClassLoader)loader);
        TokenizerFactory.reloadTokenizers((ClassLoader)loader);
    }

    private Class<? extends Plugin> loadPluginClass(String className, ClassLoader loader) {
        try {
            return loader.loadClass(className).asSubclass(Plugin.class);
        }
        catch (ClassNotFoundException e) {
            throw new ElasticsearchException("Could not find plugin class [" + className + "]", (Throwable)e, new Object[0]);
        }
    }

    private Plugin loadPlugin(Class<? extends Plugin> pluginClass, Settings settings) {
        try {
            try {
                return pluginClass.getConstructor(Settings.class).newInstance(settings);
            }
            catch (NoSuchMethodException e) {
                try {
                    return pluginClass.getConstructor(new Class[0]).newInstance(new Object[0]);
                }
                catch (NoSuchMethodException e1) {
                    throw new ElasticsearchException("No constructor for [" + pluginClass + "]. A plugin class must " + "have either an empty default constructor or a single argument constructor accepting a " + "Settings instance", new Object[0]);
                }
            }
        }
        catch (Throwable e) {
            throw new ElasticsearchException("Failed to load plugin class [" + pluginClass.getName() + "]", e, new Object[0]);
        }
    }

    static class Bundle {
        List<PluginInfo> plugins = new ArrayList<PluginInfo>();
        List<URL> urls = new ArrayList<URL>();

        Bundle() {
        }
    }

    static class OnModuleReference {
        public final Class<? extends Module> moduleClass;
        public final Method onModuleMethod;

        OnModuleReference(Class<? extends Module> moduleClass, Method onModuleMethod) {
            this.moduleClass = moduleClass;
            this.onModuleMethod = onModuleMethod;
        }
    }
}

