/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices.memory;

import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledFuture;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.FutureUtils;
import org.elasticsearch.index.IndexService;
import org.elasticsearch.index.engine.EngineClosedException;
import org.elasticsearch.index.engine.FlushNotAllowedEngineException;
import org.elasticsearch.index.shard.IndexShard;
import org.elasticsearch.index.shard.IndexShardState;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.indices.IndicesService;
import org.elasticsearch.monitor.jvm.JvmInfo;
import org.elasticsearch.threadpool.ThreadPool;

public class IndexingMemoryController
extends AbstractLifecycleComponent<IndexingMemoryController> {
    public static final String INDEX_BUFFER_SIZE_SETTING = "indices.memory.index_buffer_size";
    public static final String MIN_INDEX_BUFFER_SIZE_SETTING = "indices.memory.min_index_buffer_size";
    public static final String MAX_INDEX_BUFFER_SIZE_SETTING = "indices.memory.max_index_buffer_size";
    public static final String MIN_SHARD_INDEX_BUFFER_SIZE_SETTING = "indices.memory.min_shard_index_buffer_size";
    public static final String MAX_SHARD_INDEX_BUFFER_SIZE_SETTING = "indices.memory.max_shard_index_buffer_size";
    public static final String TRANSLOG_BUFFER_SIZE_SETTING = "indices.memory.translog_buffer_size";
    public static final String MIN_TRANSLOG_BUFFER_SIZE_SETTING = "indices.memory.min_translog_buffer_size";
    public static final String MAX_TRANSLOG_BUFFER_SIZE_SETTING = "indices.memory.max_translog_buffer_size";
    public static final String MIN_SHARD_TRANSLOG_BUFFER_SIZE_SETTING = "indices.memory.min_shard_translog_buffer_size";
    public static final String MAX_SHARD_TRANSLOG_BUFFER_SIZE_SETTING = "indices.memory.max_shard_translog_buffer_size";
    public static final String SHARD_INACTIVE_TIME_SETTING = "indices.memory.shard_inactive_time";
    public static final String SHARD_INACTIVE_INTERVAL_TIME_SETTING = "indices.memory.interval";
    public static final ByteSizeValue INACTIVE_SHARD_INDEXING_BUFFER = ByteSizeValue.parseBytesSizeValue("500kb", "INACTIVE_SHARD_INDEXING_BUFFER");
    public static final ByteSizeValue INACTIVE_SHARD_TRANSLOG_BUFFER = ByteSizeValue.parseBytesSizeValue("1kb", "INACTIVE_SHARD_TRANSLOG_BUFFER");
    private final ThreadPool threadPool;
    private final IndicesService indicesService;
    private final ByteSizeValue indexingBuffer;
    private final ByteSizeValue minShardIndexBufferSize;
    private final ByteSizeValue maxShardIndexBufferSize;
    private final ByteSizeValue translogBuffer;
    private final ByteSizeValue minShardTranslogBufferSize;
    private final ByteSizeValue maxShardTranslogBufferSize;
    private final TimeValue inactiveTime;
    private final TimeValue interval;
    private volatile ScheduledFuture scheduler;
    private static final EnumSet<IndexShardState> CAN_UPDATE_INDEX_BUFFER_STATES = EnumSet.of(IndexShardState.RECOVERING, IndexShardState.POST_RECOVERY, IndexShardState.STARTED, IndexShardState.RELOCATED);
    private final ShardsIndicesStatusChecker statusChecker;

    @Inject
    public IndexingMemoryController(Settings settings, ThreadPool threadPool, IndicesService indicesService) {
        this(settings, threadPool, indicesService, JvmInfo.jvmInfo().getMem().getHeapMax().bytes());
    }

    protected IndexingMemoryController(Settings settings, ThreadPool threadPool, IndicesService indicesService, long jvmMemoryInBytes) {
        super(settings);
        ByteSizeValue translogBuffer;
        ByteSizeValue indexingBuffer;
        this.threadPool = threadPool;
        this.indicesService = indicesService;
        String indexingBufferSetting = this.settings.get(INDEX_BUFFER_SIZE_SETTING, "10%");
        if (indexingBufferSetting.endsWith("%")) {
            double percent = Double.parseDouble(indexingBufferSetting.substring(0, indexingBufferSetting.length() - 1));
            indexingBuffer = new ByteSizeValue((long)((double)jvmMemoryInBytes * (percent / 100.0)));
            ByteSizeValue minIndexingBuffer = this.settings.getAsBytesSize(MIN_INDEX_BUFFER_SIZE_SETTING, new ByteSizeValue(48L, ByteSizeUnit.MB));
            ByteSizeValue maxIndexingBuffer = this.settings.getAsBytesSize(MAX_INDEX_BUFFER_SIZE_SETTING, null);
            if (indexingBuffer.bytes() < minIndexingBuffer.bytes()) {
                indexingBuffer = minIndexingBuffer;
            }
            if (maxIndexingBuffer != null && indexingBuffer.bytes() > maxIndexingBuffer.bytes()) {
                indexingBuffer = maxIndexingBuffer;
            }
        } else {
            indexingBuffer = ByteSizeValue.parseBytesSizeValue(indexingBufferSetting, INDEX_BUFFER_SIZE_SETTING);
        }
        this.indexingBuffer = indexingBuffer;
        this.minShardIndexBufferSize = this.settings.getAsBytesSize(MIN_SHARD_INDEX_BUFFER_SIZE_SETTING, new ByteSizeValue(4L, ByteSizeUnit.MB));
        this.maxShardIndexBufferSize = this.settings.getAsBytesSize(MAX_SHARD_INDEX_BUFFER_SIZE_SETTING, new ByteSizeValue(512L, ByteSizeUnit.MB));
        String translogBufferSetting = this.settings.get(TRANSLOG_BUFFER_SIZE_SETTING, "1%");
        if (translogBufferSetting.endsWith("%")) {
            double percent = Double.parseDouble(translogBufferSetting.substring(0, translogBufferSetting.length() - 1));
            translogBuffer = new ByteSizeValue((long)((double)jvmMemoryInBytes * (percent / 100.0)));
            ByteSizeValue minTranslogBuffer = this.settings.getAsBytesSize(MIN_TRANSLOG_BUFFER_SIZE_SETTING, new ByteSizeValue(256L, ByteSizeUnit.KB));
            ByteSizeValue maxTranslogBuffer = this.settings.getAsBytesSize(MAX_TRANSLOG_BUFFER_SIZE_SETTING, null);
            if (translogBuffer.bytes() < minTranslogBuffer.bytes()) {
                translogBuffer = minTranslogBuffer;
            }
            if (maxTranslogBuffer != null && translogBuffer.bytes() > maxTranslogBuffer.bytes()) {
                translogBuffer = maxTranslogBuffer;
            }
        } else {
            translogBuffer = ByteSizeValue.parseBytesSizeValue(translogBufferSetting, TRANSLOG_BUFFER_SIZE_SETTING);
        }
        this.translogBuffer = translogBuffer;
        this.minShardTranslogBufferSize = this.settings.getAsBytesSize(MIN_SHARD_TRANSLOG_BUFFER_SIZE_SETTING, new ByteSizeValue(2L, ByteSizeUnit.KB));
        this.maxShardTranslogBufferSize = this.settings.getAsBytesSize(MAX_SHARD_TRANSLOG_BUFFER_SIZE_SETTING, new ByteSizeValue(64L, ByteSizeUnit.KB));
        this.inactiveTime = this.settings.getAsTime(SHARD_INACTIVE_TIME_SETTING, TimeValue.timeValueMinutes(5L));
        this.interval = this.settings.getAsTime(SHARD_INACTIVE_INTERVAL_TIME_SETTING, TimeValue.timeValueSeconds(30L));
        this.statusChecker = new ShardsIndicesStatusChecker();
        this.logger.debug("using indexing buffer size [{}], with {} [{}], {} [{}], {} [{}], {} [{}]", this.indexingBuffer, MIN_SHARD_INDEX_BUFFER_SIZE_SETTING, this.minShardIndexBufferSize, MAX_SHARD_INDEX_BUFFER_SIZE_SETTING, this.maxShardIndexBufferSize, SHARD_INACTIVE_TIME_SETTING, this.inactiveTime, SHARD_INACTIVE_INTERVAL_TIME_SETTING, this.interval);
    }

    @Override
    protected void doStart() {
        this.scheduler = this.threadPool.scheduleWithFixedDelay(this.statusChecker, this.interval);
    }

    @Override
    protected void doStop() {
        FutureUtils.cancel(this.scheduler);
        this.scheduler = null;
    }

    @Override
    protected void doClose() {
    }

    public ByteSizeValue indexingBufferSize() {
        return this.indexingBuffer;
    }

    public ByteSizeValue translogBufferSize() {
        return this.translogBuffer;
    }

    protected List<ShardId> availableShards() {
        ArrayList<ShardId> list = new ArrayList<ShardId>();
        for (IndexService indexService : this.indicesService) {
            for (IndexShard indexShard : indexService) {
                if (!this.shardAvailable(indexShard)) continue;
                list.add(indexShard.shardId());
            }
        }
        return list;
    }

    protected boolean shardAvailable(ShardId shardId) {
        return this.shardAvailable(this.getShard(shardId));
    }

    protected boolean shardAvailable(@Nullable IndexShard shard) {
        return shard != null && shard.canIndex() && CAN_UPDATE_INDEX_BUFFER_STATES.contains((Object)shard.state());
    }

    protected IndexShard getShard(ShardId shardId) {
        IndexService indexService = this.indicesService.indexService(shardId.index().name());
        if (indexService != null) {
            IndexShard indexShard = indexService.shard(shardId.id());
            return indexShard;
        }
        return null;
    }

    protected void updateShardBuffers(ShardId shardId, ByteSizeValue shardIndexingBufferSize, ByteSizeValue shardTranslogBufferSize) {
        IndexShard shard = this.getShard(shardId);
        if (shard != null) {
            try {
                shard.updateBufferSize(shardIndexingBufferSize, shardTranslogBufferSize);
            }
            catch (EngineClosedException engineClosedException) {
            }
            catch (FlushNotAllowedEngineException flushNotAllowedEngineException) {
            }
            catch (Exception e) {
                this.logger.warn("failed to set shard {} index buffer to [{}]", shardId, shardIndexingBufferSize);
            }
        }
    }

    protected Boolean getShardActive(ShardId shardId) {
        IndexShard indexShard = this.getShard(shardId);
        if (indexShard == null) {
            return null;
        }
        return indexShard.getActive();
    }

    public void forceCheck() {
        this.statusChecker.run();
    }

    protected long currentTimeInNanos() {
        return System.nanoTime();
    }

    protected Boolean checkIdle(ShardId shardId, long inactiveTimeNS) {
        String ignoreReason;
        block5: {
            ignoreReason = null;
            IndexShard shard = this.getShard(shardId);
            if (shard != null) {
                try {
                    return shard.checkIdle(inactiveTimeNS);
                }
                catch (EngineClosedException e) {
                    ignoreReason = "EngineClosedException";
                    break block5;
                }
                catch (FlushNotAllowedEngineException e) {
                    ignoreReason = "FlushNotAllowedEngineException";
                    break block5;
                }
            }
            ignoreReason = "shard not found";
        }
        if (ignoreReason != null) {
            this.logger.trace("ignore [{}] while marking shard {} as inactive", ignoreReason, shardId);
        }
        return null;
    }

    private static enum ShardStatusChangeType {
        ADDED,
        DELETED,
        BECAME_ACTIVE,
        BECAME_INACTIVE;

    }

    class ShardsIndicesStatusChecker
    implements Runnable {
        private final Map<ShardId, Boolean> shardWasActive = new HashMap<ShardId, Boolean>();

        ShardsIndicesStatusChecker() {
        }

        @Override
        public synchronized void run() {
            EnumSet<ShardStatusChangeType> changes = this.purgeDeletedAndClosedShards();
            this.updateShardStatuses(changes);
            if (!changes.isEmpty()) {
                this.calcAndSetShardBuffers("[" + changes + "]");
            }
        }

        private void updateShardStatuses(EnumSet<ShardStatusChangeType> changes) {
            for (ShardId shardId : IndexingMemoryController.this.availableShards()) {
                Boolean isActive = IndexingMemoryController.this.getShardActive(shardId);
                if (isActive == null) continue;
                Boolean wasActive = this.shardWasActive.get(shardId);
                if (wasActive == null) {
                    this.shardWasActive.put(shardId, isActive);
                    changes.add(ShardStatusChangeType.ADDED);
                    continue;
                }
                if (!isActive.booleanValue()) continue;
                if (!wasActive.booleanValue()) {
                    changes.add(ShardStatusChangeType.BECAME_ACTIVE);
                    IndexingMemoryController.this.logger.debug("marking shard {} as active indexing wise", shardId);
                    this.shardWasActive.put(shardId, true);
                    continue;
                }
                if (IndexingMemoryController.this.checkIdle(shardId, IndexingMemoryController.this.inactiveTime.nanos()) != Boolean.TRUE) continue;
                changes.add(ShardStatusChangeType.BECAME_INACTIVE);
                IndexingMemoryController.this.logger.debug("marking shard {} as inactive (inactive_time[{}]) indexing wise", shardId, IndexingMemoryController.this.inactiveTime);
                this.shardWasActive.put(shardId, false);
            }
        }

        private EnumSet<ShardStatusChangeType> purgeDeletedAndClosedShards() {
            EnumSet<ShardStatusChangeType> changes = EnumSet.noneOf(ShardStatusChangeType.class);
            Iterator<ShardId> statusShardIdIterator = this.shardWasActive.keySet().iterator();
            while (statusShardIdIterator.hasNext()) {
                ShardId shardId = statusShardIdIterator.next();
                if (IndexingMemoryController.this.shardAvailable(shardId)) continue;
                changes.add(ShardStatusChangeType.DELETED);
                statusShardIdIterator.remove();
            }
            return changes;
        }

        private void calcAndSetShardBuffers(String reason) {
            ByteSizeValue shardTranslogBufferSize;
            int activeShardCount = 0;
            for (Map.Entry<ShardId, Boolean> ent : this.shardWasActive.entrySet()) {
                if (!ent.getValue().booleanValue()) continue;
                ++activeShardCount;
            }
            if (activeShardCount == 0) {
                IndexingMemoryController.this.logger.debug("no active shards (reason={})", reason);
                return;
            }
            ByteSizeValue shardIndexingBufferSize = new ByteSizeValue(IndexingMemoryController.this.indexingBuffer.bytes() / (long)activeShardCount);
            if (shardIndexingBufferSize.bytes() < IndexingMemoryController.this.minShardIndexBufferSize.bytes()) {
                shardIndexingBufferSize = IndexingMemoryController.this.minShardIndexBufferSize;
            }
            if (shardIndexingBufferSize.bytes() > IndexingMemoryController.this.maxShardIndexBufferSize.bytes()) {
                shardIndexingBufferSize = IndexingMemoryController.this.maxShardIndexBufferSize;
            }
            if ((shardTranslogBufferSize = new ByteSizeValue(IndexingMemoryController.this.translogBuffer.bytes() / (long)activeShardCount)).bytes() < IndexingMemoryController.this.minShardTranslogBufferSize.bytes()) {
                shardTranslogBufferSize = IndexingMemoryController.this.minShardTranslogBufferSize;
            }
            if (shardTranslogBufferSize.bytes() > IndexingMemoryController.this.maxShardTranslogBufferSize.bytes()) {
                shardTranslogBufferSize = IndexingMemoryController.this.maxShardTranslogBufferSize;
            }
            IndexingMemoryController.this.logger.debug("recalculating shard indexing buffer (reason={}), total is [{}] with [{}] active shards, each shard set to indexing=[{}], translog=[{}]", reason, IndexingMemoryController.this.indexingBuffer, activeShardCount, shardIndexingBufferSize, shardTranslogBufferSize);
            for (Map.Entry<ShardId, Boolean> ent : this.shardWasActive.entrySet()) {
                if (!ent.getValue().booleanValue()) continue;
                IndexingMemoryController.this.updateShardBuffers(ent.getKey(), shardIndexingBufferSize, shardTranslogBufferSize);
            }
        }
    }
}

