<?php
/**
 * ownCloud - Addressbook
 *
 * @author Thomas Tanghus
 * @copyright 2012-2014 Thomas Tanghus (thomas@tanghus.net)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU AFFERO GENERAL PUBLIC LICENSE
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU AFFERO GENERAL PUBLIC LICENSE for more details.
 *
 * You should have received a copy of the GNU Affero General Public
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

namespace OCA\Contacts\CardDAV;

use OCA\Contacts;

/**
 * This class overrides \Sabre\CardDAV\Card::getACL()
 * to return read/write permissions based on user and shared state.
*/
class Card extends \Sabre\CardDAV\Card {

	/**
	* Array with information about the containing addressbook
	*
	* @var array
	*/
	protected $addressBookInfo;

	/**
	* Constructor
	*
	* @param \Sabre\CardDAV\Backend\AbstractBackend $carddavBackend
	* @param array $addressBookInfo
	* @param array $cardData
	*/
	public function __construct(\Sabre\CardDAV\Backend\AbstractBackend $carddavBackend, array $addressBookInfo, array $cardData) {

		$this->addressBookInfo = $addressBookInfo;
		parent::__construct($carddavBackend, $addressBookInfo, $cardData);

	}

	/**
	* Returns a list of ACE's for this node.
	*
	* Each ACE has the following properties:
	*   * 'privilege', a string such as {DAV:}read or {DAV:}write. These are
	*     currently the only supported privileges
	*   * 'principal', a url to the principal who owns the node
	*   * 'protected' (optional), indicating that this ACE is not allowed to
	*      be updated.
	*
	* @return array
	*/
	public function getACL() {

		$readprincipal = $this->getOwner();
		$writeprincipal = $this->getOwner();
		$uid = $this->carddavBackend->userIDByPrincipal($this->getOwner());
		$currentUid = \OC::$server->getUserSession()->getUser()->getUId();

		if($uid != $currentUid) {
			list(, $id) = explode('::', $this->addressBookInfo['id']);
			$sharedAddressbook = \OCP\Share::getItemSharedWithBySource('addressbook', $id);
			if ($sharedAddressbook && ($sharedAddressbook['permissions'] & \OCP\PERMISSION_READ)) {
				$readprincipal = 'principals/' . $currentUid;
			}
			if ($sharedAddressbook && ($sharedAddressbook['permissions'] & \OCP\PERMISSION_UPDATE)) {
				$writeprincipal = 'principals/' . $currentUid;
			}
		}

		return array(
			array(
				'privilege' => '{DAV:}read',
				'principal' => $readprincipal,
				'protected' => true,
			),
			array(
				'privilege' => '{DAV:}write',
				'principal' => $writeprincipal,
				'protected' => true,
			),

		);

	}

}
