################################################################################
##<#GAPDoc Label="SCNrRegularTorus">
## <ManSection>
## <Func Name="SCNrRegularTorus" Arg="n"/>
## <Returns>an integer upon success, <K>fail</K> otherwise.</Returns>
## <Description>
## Returns the number of simplicial regular maps on the torus with <M>n</M> vertices, 
## cf. <Cite Key="Brehm08EquivMapsTorus"/> for details.
## <Example>
## gap> SCNrRegularTorus(9);
## gap> SCNrRegularTorus(10);
## </Example>
## </Description>
## </ManSection>
##<#/GAPDoc>
################################################################################
InstallGlobalFunction(SCNrRegularTorus,
function(n)

	if not IsPosInt(n) or n < 7 then
		return fail;
	fi;

	if IsSquareInt(n) or IsSquareInt(3*n) then
		return 1;
	else
		return 0;
	fi;
end);

################################################################################
##<#GAPDoc Label="SCNrChiralTori">
## <ManSection>
## <Func Name="SCNrChiralTori" Arg="n"/>
## <Returns>an integer upon success, <K>fail</K> otherwise.</Returns>
## <Description>
## Returns the number of simplicial chiral maps on the torus with <M>n</M> vertices, 
## cf. <Cite Key="Brehm08EquivMapsTorus"/> for details.
## <Example>
## gap> SCNrChiralTori(7);
## gap> SCNrChiralTori(343);
## </Example>
## </Description>
## </ManSection>
##<#/GAPDoc>
################################################################################
InstallGlobalFunction(SCNrChiralTori,
function(n)

	local factors,p,q,b,numSurfs,pSet,qSet,elm;

	if not IsPosInt(n) or n < 7 then
		return fail;
	fi;

	factors:=FactorsInt(n);
	p:=Filtered(factors,x->x mod 6 = 1);
	q:=Filtered(factors,x->x mod 6 = 5);
	b:=[];
	qSet:=Set(q);	
	Add(b,Size(Filtered(factors,x->x=2)));
	for elm in qSet do
		Add(b,Size(Filtered(q,x->x=elm)));
	od;	

	if p = [] or ForAny(b,x->IsOddInt(x)) then
		return 0;
	fi;
	pSet:=Set(p);
	numSurfs:=1;
	for elm in pSet do
		numSurfs:=numSurfs*(Size(Filtered(p,x->x=elm))+1);
	od;
	return Int(1/2*numSurfs);
end);

SCIntFunc.PermList:=function(list)

	local elm,j,pList,tmpList,p;

	if not ForAll(list,x->IsDuplicateFreeList(x)) then
		return fail;
	fi;
	tmpList:=[];
	for elm in list do
		Append(tmpList,elm);
	od;
	if not IsDuplicateFreeList(tmpList) then
		return fail;
	fi;

	pList:=[];
	for elm in list do
		tmpList:=[1..Maximum(elm)];
		for j in [1..Size(elm)-1] do
			tmpList[elm[j]]:=elm[j+1];
		od;
		tmpList[elm[Size(elm)]]:=elm[1];
		Add(pList,PermList(tmpList));
	od;
	p:=();
	for elm in pList do
		p:=p*elm;
	od;

	return p;

end;

################################################################################
##<#GAPDoc Label="SCSeriesSymmetricTorus">
## <ManSection>
## <Func Name="SCSeriesSymmetricTorus" Arg="p, q"/>
## <Returns>a <K>SCSimplicialComplex</K> object upon success, <K>fail</K> otherwise.</Returns>
## <Description>
## Returns the equivarient triangulation of the torus <M>\{ 3,6 \}_{(p,q)}</M> with 
## fundamental domain <M>(p,q)</M> on the <M>2</M>-dimensional integer lattice. 
## See <Cite Key="Brehm08EquivMapsTorus"/> for details.
## <Example>
## gap> c:=SCSeriesSymmetricTorus(2,1);
## gap> SCFVector(c);
## </Example>
## </Description>
## </ManSection>
##<#/GAPDoc>
################################################################################
InstallGlobalFunction(SCSeriesSymmetricTorus,
function(p,q)

	local phi,psi,sigma,x,y,verts,a,b,n,v,imv,i,j,elm,6cycle,3cycle,2cycle,done,idx,mat,trig,G,c;


	# generators of automorphism group
	psi:=[[-1,-1,1],[1,0,0],[0,0,1]];
	phi:=[[0,-1,0],[1,1,0],[0,0,1]];
	if q=0 then
		sigma:=[[0,-1,p],[-1,0,p],[0,0,1]];
	elif p=q then
		sigma:=[[1,0,0],[-1,-1,3*p],[0,0,1]];
	else
		sigma:=[[1,0,0],[0,1,0],[0,0,1]];
	fi;

	# vertices
	verts:=[];
	n:=q^2+p*q+p^2;
	for x in [-q..p-1] do
		for y in [0..p+2*q-1] do	
			a:=(p+q)*x+q*y;
			b:=-q*x+p*y;
			if a < 0 or b < 0 or a >= n or b >= n then continue; fi;
			Add(verts,[x,y]);
		od;
	od;

	# permuations

	# phi: rotation around origin
	6cycle:=[];
	idx:=0;
	for i in [1..n] do
		done:=false;
		for elm in 6cycle do
			if verts[i] in elm then done:=true; break; fi;
		od;
		if done then continue; fi;
		v:=[[verts[i][1]],[verts[i][2]],[1]];
		idx:=idx+1;
		6cycle[idx]:=[];
		mat:=[[1,0,0],[0,1,0],[0,0,1]];
		for j in [1..6] do
			imv:=mat*v;
			mat:=mat*phi;
			if not [imv[1][1],imv[2][1]] in verts then
				a:=(p+q)*imv[1][1]+q*imv[2][1];
				b:=-q*imv[1][1]+p*imv[2][1];
				while a < 0 do
					imv[1][1]:=imv[1][1]+(p);
					imv[2][1]:=imv[2][1]+(q);
					a:=(p+q)*imv[1][1]+q*imv[2][1];
				od;
				while a >= n do
					imv[1][1]:=imv[1][1]+(-p);
					imv[2][1]:=imv[2][1]+(-q);
					a:=(p+q)*imv[1][1]+q*imv[2][1];
				od;
				while b < 0 do
					imv[1][1]:=imv[1][1]+(-q);
					imv[2][1]:=imv[2][1]+(p+q);
					b:=-q*imv[1][1]+p*imv[2][1];
				od;					
				while b >= n do
					imv[1][1]:=imv[1][1]+(q);
					imv[2][1]:=imv[2][1]+(-p-q);
					b:=-q*imv[1][1]+p*imv[2][1];
				od;
			fi;
			if not 	[imv[1][1],imv[2][1]] in verts then
				Error();
			fi;
			if [imv[1][1],imv[2][1]] in 6cycle[idx] then
				break;
			fi;
			Add(6cycle[idx],[imv[1][1],imv[2][1]]);
		od;
	od;

	# psi: rotation around the center of a triangle
	3cycle:=[];
	idx:=0;
	for i in [1..n] do
		done:=false;
		for elm in 3cycle do
			if verts[i] in elm then done:=true; break; fi;
		od;
		if done then continue; fi;
		v:=[[verts[i][1]],[verts[i][2]],[1]];
		idx:=idx+1;
		3cycle[idx]:=[];
		mat:=[[1,0,0],[0,1,0],[0,0,1]];
		for j in [1..3] do
			imv:=mat*v;
			mat:=mat*psi;
			if not [imv[1][1],imv[2][1]] in verts then
				a:=(p+q)*imv[1][1]+q*imv[2][1];
				b:=-q*imv[1][1]+p*imv[2][1];
				while a < 0 do
					imv[1][1]:=imv[1][1]+(p);
					imv[2][1]:=imv[2][1]+(q);
					a:=(p+q)*imv[1][1]+q*imv[2][1];
				od;
				while a >= n do
					imv[1][1]:=imv[1][1]+(-p);
					imv[2][1]:=imv[2][1]+(-q);
					a:=(p+q)*imv[1][1]+q*imv[2][1];
				od;
				while b < 0 do
					imv[1][1]:=imv[1][1]+(-q);
					imv[2][1]:=imv[2][1]+(p+q);
					b:=-q*imv[1][1]+p*imv[2][1];
				od;					
				while b >= n do
					imv[1][1]:=imv[1][1]+(q);
					imv[2][1]:=imv[2][1]+(-p-q);
					b:=-q*imv[1][1]+p*imv[2][1];
				od;
			fi;
			if not 	[imv[1][1],imv[2][1]] in verts then
				Error();
			fi;
			if [imv[1][1],imv[2][1]] in 3cycle[idx] then
				break;
			fi;
			Add(3cycle[idx],[imv[1][1],imv[2][1]]);
		od;
	od;

	# sigma: reflection (only in regular case)
	2cycle:=[];
	idx:=0;
	for i in [1..n] do
		done:=false;
		for elm in 2cycle do
			if verts[i] in elm then done:=true; break; fi;
		od;
		if done then continue; fi;
		v:=[[verts[i][1]],[verts[i][2]],[1]];
		idx:=idx+1;
		2cycle[idx]:=[];
		mat:=[[1,0,0],[0,1,0],[0,0,1]];
		for j in [1..2] do
			imv:=mat*v;
			mat:=mat*sigma;
			if not [imv[1][1],imv[2][1]] in verts then
				a:=(p+q)*imv[1][1]+q*imv[2][1];
				b:=-q*imv[1][1]+p*imv[2][1];
				while a < 0 do
					imv[1][1]:=imv[1][1]+(p);
					imv[2][1]:=imv[2][1]+(q);
					a:=(p+q)*imv[1][1]+q*imv[2][1];
				od;
				while a >= n do
					imv[1][1]:=imv[1][1]+(-p);
					imv[2][1]:=imv[2][1]+(-q);
					a:=(p+q)*imv[1][1]+q*imv[2][1];
				od;
				while b < 0 do
					imv[1][1]:=imv[1][1]+(-q);
					imv[2][1]:=imv[2][1]+(p+q);
					b:=-q*imv[1][1]+p*imv[2][1];
				od;					
				while b >= n do
					imv[1][1]:=imv[1][1]+(q);
					imv[2][1]:=imv[2][1]+(-p-q);
					b:=-q*imv[1][1]+p*imv[2][1];
				od;
			fi;
			if not 	[imv[1][1],imv[2][1]] in verts then
				Error();
			fi;
			if [imv[1][1],imv[2][1]] in 2cycle[idx] then
				break;
			fi;
			Add(2cycle[idx],[imv[1][1],imv[2][1]]);
		od;
	od;

	# triangle of map
	trig:=[[0,0],[1,0],[0,1]];
	for i in [1,2,3] do
		if not trig[i] in verts then
			a:=(p+q)*trig[i][1]+q*trig[i][2];
			b:=-q*trig[i][1]+p*trig[i][2];
			while a < 0 do
				trig[i][1]:=trig[i][1]+(p);
				trig[i][2]:=trig[i][2]+(q);
				a:=(p+q)*trig[i][1]+q*trig[i][2];
			od;
			while a >= n do
				trig[i][1]:=trig[i][1]+(-p);
				trig[i][2]:=trig[i][2]+(-q);
				a:=(p+q)*trig[i][1]+q*trig[i][2];
			od;
			while b < 0 do
				trig[i][1]:=trig[i][1]+(-q);
				trig[i][2]:=trig[i][2]+(p+q);
				b:=-q*trig[i][1]+p*trig[i][2];
			od;					
			while b >= n do
				trig[i][1]:=trig[i][1]+(q);
				trig[i][2]:=trig[i][2]+(-p-q);
				b:=-q*trig[i][1]+p*trig[i][2];
			od;
		fi;	
	od;
	trig:=[Position(verts,trig[1]),Position(verts,trig[2]),Position(verts,trig[3])];
	trig:=Set(trig);

	# map vertices to integers 1,2,...
	for i in [1..Size(6cycle)] do
		for j in [1..Size(6cycle[i])] do
			6cycle[i][j]:=Position(verts,6cycle[i][j]);
		od;
	od;
	for i in [1..Size(3cycle)] do
		for j in [1..Size(3cycle[i])] do
			3cycle[i][j]:=Position(verts,3cycle[i][j]);
		od;
	od;
	for i in [1..Size(2cycle)] do
		for j in [1..Size(2cycle[i])] do
			2cycle[i][j]:=Position(verts,2cycle[i][j]);
		od;
	od;

	# convert lists to permutations
	2cycle:=SCIntFunc.PermList(2cycle);
	3cycle:=SCIntFunc.PermList(3cycle);
	6cycle:=SCIntFunc.PermList(6cycle);
	G:=Group([2cycle,3cycle,6cycle]);
	c:=SCFromGenerators(G,[trig]);
	SCRename(c,Concatenation("{3,6}_(",String(p),",",String(q),")"));
	SetSCTopologicalType(c,"T^2");
	SetSCAutomorphismGroup(c,G);

	return c;
end);

################################################################################
##<#GAPDoc Label="SCRegularTorus">
## <ManSection>
## <Func Name="SCRegularTorus" Arg="n"/>
## <Returns>a <K>SCSimplicialComplex</K> object upon success, <K>fail</K> otherwise.</Returns>
## <Description>
## Returns a list of regular triangulations of the torus with <M>n</M> vertices (the
## length of the list will be at most <M>1</M>).
## See <Cite Key="Brehm08EquivMapsTorus"/> for details.
## <Example>
## gap> cc:=SCRegularTorus(9);
## gap> g:=SCAutomorphismGroup(cc[1]);
## gap> SCNumFaces(cc[1],0)*12 = Size(g);
## </Example>
## </Description>
## </ManSection>
##<#/GAPDoc>
################################################################################
InstallGlobalFunction(SCRegularTorus,
function(n)

	local p,q;

	if SCNrRegularTorus(n) = fail then
		return fail;
	elif SCNrRegularTorus(n) = 0 then
		return [];
	fi;

	if IsSquareInt(n) then
		p := Sqrt(n);
		q := 0;
	elif IsSquareInt(3*n) then
		p := Sqrt(n/3);
		q := Sqrt(n/3);
	else
		return fail;
	fi;

	return [SCSeriesSymmetricTorus(p,q)];
end);

################################################################################
##<#GAPDoc Label="SCChiralTori">
## <ManSection>
## <Func Name="SCChiralTori" Arg="n"/>
## <Returns>a <K>SCSimplicialComplex</K> object upon success, <K>fail</K> otherwise.</Returns>
## <Description>
## Returns a list of chiral triangulations of the torus with <M>n</M> vertices.
## See <Cite Key="Brehm08EquivMapsTorus"/> for details.
## <Example>
## gap> cc:=SCChiralTori(91);
## gap> SCIsIsomorphic(cc[1],cc[2]);
## </Example>
## </Description>
## </ManSection>
##<#/GAPDoc>
################################################################################
InstallGlobalFunction(SCChiralTori,
function(n)

	local p,q,list,done,l,m;

	l := SCNrChiralTori(n);
	if SCNrChiralTori(n) = fail then
		return fail;
	elif l = 0 then
		return [];
	fi;

	done:=false;
	list:=[];
	for q in [1..n] do
		for p in [q+1..n] do
			m:=p*p + p*q + q*q;
			if m < n then
				continue;
			elif m = n then
				Add(list,SCSeriesSymmetricTorus(p,q));
				if Size(list) >= l then
					done:=true;
					break;
				fi;
			elif m > n then
				break;
			fi;
		od;
		if done = true then
			break;
		fi;
	od;

	return list;					
end);


################################################################################
##<#GAPDoc Label="SCRegularMaps">
## <ManSection>
## <Func Name="SCRegularMaps" Arg=""/>
## <Returns>a list of lists upon success, <K>fail</K> otherwise.</Returns>
## <Description>
## Returns a list of all simplicial (hyperbolic) regular maps of orientable genus up to <M>100</M>
## or non-orientable genus up to <M>200</M>. The list was generated with the help of 
## the classification of regular maps by Marston Conder <Cite Key="Conder09RegMapsOfBdChi"/>.
##
## Every regular map is given by a <M>3</M>-tuple <M>(m,g,or)</M> where <M>m</M> is the 
## vertex valence, <M>g</M> is the genus and <M>or</M> is a boolean stating if the
## map is orientable or not.
##
## Use the <M>3</M>-tuples of the list together with <Ref Func="SCRegularMap"/> 
## to get the corresponding triangulations.
## <M>g</M>
## <Example>
## gap> ll:=SCRegularMaps(){[1..10]};
## gap> c:=SCRegularMap(ll[5][1],ll[5][2],ll[5][3]);
## gap> SCHomology(c);
## gap> SCGenerators(c);
## </Example>
## </Description>
## </ManSection>
##<#/GAPDoc>
################################################################################
InstallGlobalFunction(SCRegularMaps,
function()
	local mg;
	mg:=[[7,3,true],[7,7,true],[7,8,false],[7,14,true],
	[7,15,false],[7,147,false],[8,3,true],[8,5,true],
	[8,8,true],[8,9,false],[8,16,true],[8,17,true],
	[8,17,false],[8,22,true],[8,23,false],[8,33,true],
	[8,46,true],[8,47,false],[8,52,true],[8,65,true],
	[8,82,true],[8,85,true],[8,86,false],[9,10,true],
	[9,15,true],[9,16,false],[9,43,true],[9,69,true],
	[9,70,false],[9,96,true],[9,97,false],[10,6,true],
	[10,13,true],[10,14,false],[10,25,true],[10,26,true],
	[10,26,false],[10,45,true],[10,46,false],[10,49,true],
	[10,65,true],[10,73,true],[10,74,false],[10,89,true],
	[10,116,false],[10,134,false],[10,170,false],
	[11,26,true],[12,10,true],[12,13,true],[12,25,true],
	[12,28,true],[12,29,false],[12,33,true],[12,37,true],
	[12,49,true],[12,56,true],[12,57,false],[12,65,true],
	[12,73,true],[12,76,true],[12,82,true],[12,92,true],
	[12,93,false],[12,97,true],[13,50,true],[13,51,false],
	[14,15,true],[14,49,true],[14,99,true],[14,106,false],
	[14,146,false],[15,37,true],[15,73,true],[15,76,true],
	[16,21,true],[16,41,true],[16,81,true],[17,134,false],
	[18,28,true],[18,82,true],[18,170,false],[20,36,true],
	[20,85,true],[20,86,false],[20,177,false],[22,45,true],
	[24,55,true],[24,73,true],[26,66,true],[28,78,true],
	[30,91,true]];
	return mg;
end);

################################################################################
##<#GAPDoc Label="SCRegularMap">
## <ManSection>
## <Func Name="SCRegularMap" Arg="m, g, orient"/>
## <Returns>a <K>SCSimplicialComplex</K> object upon success, <K>fail</K> otherwise.</Returns>
## <Description>
## Returns the (hyperbolic) regular map of vertex valence <Arg>m</Arg>, genus <Arg>g</Arg> and orientability
## <Arg>orient</Arg> if existent and <K>fail</K> otherwise. 
##
## The triangulations were generated with the help of 
## the classification of regular maps by Marston Conder <Cite Key="Conder09RegMapsOfBdChi"/>.
##
## Use <Ref Func="SCRegularMaps"/> to get a list of all regular maps available.
## <Example>
## gap> SCRegularMaps(){[1..10]};
## gap> c:=SCRegularMap(7,7,true);
## gap> g:=SCAutomorphismGroup(c);
## gap> Size(g);
## </Example>
## </Description>
## </ManSection>
##<#/GAPDoc>
################################################################################
InstallGlobalFunction(SCRegularMap,
function(m,g,orient)

	local autGroups,gens,n,f2,G,mg,i,c;

	if not IsPosInt(m) or m < 7 or not IsInt(g) and g >= 2  then
		Info(InfoSimpcomp,1,"SCRegularMap: argument 1 must be an integer > 6, argument 2 must be an integer > 1.");
		return fail;
	fi;
	if not orient in [true, false] then
		Info(InfoSimpcomp,1,"SCRegularMap: argument 3 must be true or false.");
		return fail;
	fi;

	mg:=[];
	gens:=[];
	autGroups:=[];
	if not orient then
		n:=6*(2-g)/(6-m);
		f2:= 2*m*(2-g)/(6-m);
autGroups[1]:=
[ ( 1,24)( 2,22)( 4,16)( 5,23)( 6,10)( 7,26)( 8,25)( 9,28)(11,19)(12,31)
    (14,35)(15,34)(18,27)(20,30)(21,29)(32,33), 
  ( 1,36,18, 4,15,32,17)( 2, 8,33,27,11,14,20)( 3,26,23,22,10,25,16)
    ( 5,30,12,29, 9, 7,21)( 6,35,31,28,13,34,19) ];

autGroups[2]:=
[ ( 1, 6)( 2, 8)( 3,10)( 4,17)( 7,12)( 9,20)(11,19)(14,15)(16,21), 
  ( 1, 9,15,17,14, 3,16, 7)( 2,11,21, 5,10,20,18, 6)( 8,12,19,13) ];

autGroups[3]:=
[ ( 1, 2,15,16,18, 6, 4,21)( 3,20, 9,14,17, 5, 7,13)( 8,12,19,10), 
  ( 1, 9, 5)( 2, 7,20,21,15, 4)( 3, 8,10,13,16, 6)(11,12,14,18,17,19) ];

autGroups[4]:=[ ( 2, 5)( 4, 6)( 7,18)( 8,15)( 9,16)(10,14)(11,13)(12,17), 
  ( 1,11,16, 4,12,14, 6, 7,13, 3,10,15, 2, 8,17)( 5, 9,18) ];

autGroups[5]:=
[ ( 1,20,41,19,22,64,68,73,70,53,33,12,56)( 2,16,69,45,28,60, 4,39,43,35,77,
     55, 3)( 5,47,24,76,40,21,44, 8,52,11, 7,30,59)( 6,54,31,26,38,65,63,71,
     78,74,46,29,37)( 9,61,58,14,32,67,42,36,57,13,72,51,17)(10,27,62,66,75,
     48,23,34,15,49,25,50,18), ( 1,33,29,11, 8, 6)( 2,21,30,35, 9,17)
    ( 3,27,28,38, 7,22)( 4,12,31)( 5,24,32,26,10,39)(13,37,14,23,15,25)
    (16,20,19,34,36,18)(40,70,41,59,72,58)(42,46,51,55,61,54)(43,75,67,49,73,
     53)(44,71,60,62,77,68)(45,74,56)(47,76,69,66,78,57)(48,64,50,65,52,63) ];

autGroups[6]:=
[ ( 1, 3,10)( 2,12,17)( 4,25,26)( 5, 6,22)( 7,16,15)( 8,13,23)( 9,18,20)
    (11,21,28)(14,24,19), ( 1,24,21,10, 3,15,14)( 2,12,11, 8,27, 6, 7)
    ( 4,20, 5,13,25, 9,17)(16,28,18,23,19,22,26) ];

autGroups[7]:=
[ ( 1, 9,39,24, 3,14,43,19)( 2,15,40,22, 4,12,32,25)( 6,45,37,20,13,26,33,18)
    ( 7,21,11,17)( 8,41,29,23,10,36,44,16)(27,28,31,38,35,30,42,34), 
  ( 1,16,37,11, 4,21,36, 6)( 3,22,34,10, 5,20,31,12)( 7,19,42,39,15,23,26,35)
    ( 8,18,14,24)( 9,17,30,29,13,25,43,38)(27,28,40,44,33,41,45,32) ];

autGroups[8]:=
[ ( 1,12,18, 3)( 2, 7,10,15, 5,14,11,13)( 4,17,21, 9, 8,19,16, 6)
    (22,45,24,60,34,53,35,47)(23,55,39,54,40,50,29,51)(25,48,36,52)
    (26,57,27,58,37,61,33,43)(28,44,30,63)(31,49,42,59,41,62,38,46)(32,56), 
  ( 1,39, 2,42, 4,36)( 3,31,10,28, 7,24)( 5,33,17,27,13,30)( 6,26, 9,38,12,34)
    ( 8,32,16,41,19,35)(11,40,20,29,14,37)(15,23,18,25,21,22)(43,63,44,57,46,
     60)(45,49,52)(47,48,59,51,55,54)(50,62,58,56,61,53) ];

autGroups[9]:=
[ ( 2,10)( 3,15)( 4,22)( 6,14)( 7,17)( 8,24)(12,13)(16,34)(18,30)(19,31)
    (20,35)(21,36)(23,29)(26,33)(27,32), 
  ( 1,24,36,29,16,19,32, 8)( 2, 6,12,26)( 3,33,13, 4, 7,27,21,17)
    ( 5, 9,10,20,23,31,18,14)(11,22,30,25,34,28,35,15) ];

autGroups[10]:=
[ ( 1, 9, 5)( 2, 3, 6, 4, 7, 8)(10,23,13,24,16,22)(11,20,14,21,17,19)
    (12,26,15,27,18,25), ( 1,25,18, 9,24,14, 5,20,10)( 2,26,16, 7,22,15, 6,21,
     11)( 3,27,17, 8,23,13, 4,19,12) ];

autGroups[11]:=
[ ( 1,38,27,55)( 2,39,41,32)( 3,11,36,49)( 4,52,18,66)( 5,53,62,64)
    ( 6,54,43,46)( 7,40,24,35)( 8,19,13,50)( 9,16,37,23)(10,21,30,59)
    (12,51,29,61)(14,63,42,33)(15,58,17,26)(20,44,57,31)(22,28,56,25)(34,60)
    (45,47)(48,65), ( 1,46,39,31,37)( 2,29,50,51,41, 3,48,63,60,24)
    ( 4,20,30,35,36, 8,64,54,45,56)( 5,47,44,25,13, 7,49,52,43,59)
    ( 6,34,15,22,27, 9,62,58,65,57)(10,26,16,32,38,17,66,11,33,40)
    (12,14,28,23,21,18,55,53,42,61) ];

autGroups[12]:=
[ ( 1,24,17,25,64,31,62,32,52,51, 4,59)( 2,57,14,23,65, 6,60,15,28,48,39,66)
    ( 3,58,13,29,34,43,50,35,63, 7,54,20)( 5,19,12,27,42,44,38,33,49,46,55,11)
    ( 8,40,21,22,61,41,56,37, 9,47,18,30)(10,36,16,26,53,45), 
  ( 1,26,36,17,30,35,61,66,39, 9,50,40)( 2,48,59,18,28,38,42,11,46,57,21,25)
    ( 3,32,58,20,23,33,64,45, 4,49,60,13)( 5,14,24,15,27,37,53,63,56,43,10,41)
    ( 6, 7,47,19,12,22,34,65,55,52,62,44)( 8,51,54,16,29,31) ];

autGroups[13]:=
[ (  1, 46, 67, 40,131, 57,118,126, 90, 35)(  2, 75,128, 42,116, 20,135, 15,
      59,117)(  3, 74, 14, 41,110, 38,112,103, 88,102)(  4, 72,127,  6, 50,
      54, 44,125, 91,115)(  5, 58,108, 36,109, 56,107, 79, 23, 80)
    (  7, 49, 65, 43, 84,101, 82, 99, 83,100)(  8, 17, 69, 12, 60,130, 39,124,
     92,121)(  9, 26, 48, 37, 98,134,120,113, 24,132)( 10, 73, 89, 45, 13, 70,
     111, 11, 71,104)( 16, 68, 97, 63,114, 22,133,129, 32, 28)
    ( 18, 52, 96, 62,106, 55, 25, 81,105, 31)( 19, 66, 93, 30, 51, 53, 94, 87,
     122, 33)( 21, 78, 77, 61, 76)( 27, 47, 64, 95, 86,119, 85,123, 34, 29), 
  (  1,103, 96, 31,130,135, 27, 70, 94,  6,134,107, 33, 89,133)
    (  2,125,105, 55,110,132, 16, 73, 93, 36,131,121, 53, 71,123)
    (  3,126,  5, 79,108, 80, 48, 74, 47, 41, 95, 42, 86, 88, 85)
    (  4, 15, 77, 76,117, 69, 46, 49, 43, 40, 63,120, 82, 83, 24)
    (  7, 92, 25,  8,128, 78, 28,104, 68, 26,101, 44, 30, 91,118)
    (  9, 56,109, 35, 54, 75, 29, 38, 98, 32, 57,112, 34, 20,124)
    ( 10, 87, 23,113, 14, 81, 18,102, 67, 58, 51, 45, 64, 59,119)
    ( 11, 97, 61,114,111, 66, 17, 99, 12, 62,106,115, 65, 72, 19)
    ( 13,129, 21,116,127, 52, 50,100, 37, 60,122, 39, 84, 90, 22) ];

autGroups[14]:=
[ ( 1,35,24,32, 5,40, 7)( 2,34,14,13,33, 4,26)( 3, 6,16,23,30,19,39)
    ( 8,38,42, 9,22,10,15)(11,20,41,17,25,21,18)(12,29,37,31,27,36,28), 
  ( 1,41,15,30, 8,14)( 2, 5,22,40,37,16)( 3,25,42,12,39,24)( 4,17,26,32,31,23)
    ( 6,38, 7)( 9,36,34,28,19,21)(10,13,35,11,29,20)(18,27,33) ];

autGroups[15]:=
[ ( 2, 5,17,12, 4, 7,19,13, 3, 6,18,11)( 8,20,29,14, 9,21,31,16,10,22,30,15)
    (23,27,36,42,50,52,44,54,48,39,33,24)(25,26,35,41,51,46,55,53,45,47,38,32)
    (28,37,43,49,40,34), ( 1,28,21,41,32, 4,30, 5,55,35,14,40)( 2,52,17,25,46,
     12,24, 7,31,23,50,29)( 3,53,20,47,45,16,38, 6,54,37,49,39)
    ( 8,44,36,33,10,13,26,22,43,15,51,18)( 9,19,27,34,42,11) ];

autGroups[16]:=
[ ( 1,14,44,31,30, 4)( 2,46,41,19,50,47)( 3,22,43)( 5,12,21,48, 7,40)
    ( 6, 9,36,53, 8,17)(10,29,15,33,37,42)(11,34,54,45,24,39)(13,28,26)
    (16,38,27,25,55,51)(18,23,20,52,49,32), ( 1,42, 5,49)( 2,31)( 3,33, 8,24)
    ( 4,29,25, 6)( 7,48,13,22)( 9,40,50,37)(10,55)(11,14,38,28)(12,51,17,18)
    (15,32,27,54)(16,19,21,30)(23,34,41,43)(26,47,45,52)(35,44,39,53)(36,46) 
 ];

autGroups[17]:=
[ (  1, 16,124, 79, 38,110,101, 20, 71)(  2, 93, 31, 75, 57, 44,122, 83,  7)
    (  3, 26,128, 61, 43,113, 96, 27,  4)(  5, 13,126,134, 62,105,116,131, 82)
    (  6, 12, 76, 32,127, 64, 41,112, 97)(  8, 47, 65,136, 73, 92, 48,104, 98)
    (  9, 23, 95, 10, 19, 99, 25, 17, 58)( 11, 56, 30,133, 63,130,106,114,100)
    ( 14, 69, 87, 55,108,117,107, 81, 37)( 15, 78, 86, 46,115,102, 18, 59, 29)
    ( 21, 70, 33, 53, 77,129,111, 90, 42)( 22, 68, 60, 45,121, 84, 35, 91, 49)
    ( 28, 54, 66, 39,123, 80, 52,103, 94)( 34, 40,125,135, 72,118,132, 85, 36)
    ( 50,109,119, 67,120, 74, 88, 51, 89), (  1, 80, 35, 31, 43, 78, 12, 96)
    (  2, 97,  5, 84, 52,134, 46,124)(  3, 90, 36, 28, 41, 63, 53,122)
    (  4, 83,  7, 79, 38,128, 19, 95)(  6, 86, 62,127, 23, 99, 59, 82)
    (  8,129, 72,133,112, 93, 91, 98)(  9, 64, 54, 20)( 10, 61, 75, 17, 24,
      58, 71, 27)( 11,102, 60,135,111, 26,110, 70)( 13,114, 37, 88, 48,125,
      45,121)( 14,116, 65,120,132, 30,118, 74)( 15,113, 76, 81,100, 33,130, 73
     )( 16,115, 56, 85, 42, 87, 55,123)( 18, 25, 57, 68)( 21,101, 66,119,107,
      32,105, 69)( 22, 44, 77, 47,131, 29, 92, 51)( 34,117, 67,136,106,126,
     103, 94)( 39,108, 40,104, 49, 89, 50,109) ];

autGroups[18]:=
[ (  1,  3, 49, 44, 22, 18, 55, 27)(  2, 35, 74, 31, 26, 19,108, 41)
    (  4, 20, 15, 67, 58, 88, 51,  5)(  6, 70, 85, 21)(  7, 90, 29, 16, 47,
      57, 99, 37)(  8, 11, 32, 76, 89, 87, 65, 52)(  9, 38, 66, 13, 23, 83,
      82, 68)( 10, 14, 92, 60)( 12, 25, 43, 64,104, 80, 24, 30)
    ( 17, 62, 33, 97,100, 78, 63, 56)( 28, 95, 53, 59, 93, 61, 81, 46)
    ( 34, 98, 69, 73, 40,106,107, 36)( 39, 54, 94,103, 71, 96, 48, 42)
    ( 45, 86,101, 91, 77, 79,102, 50)( 72, 75, 84,105), 
  (  1, 24,100, 89, 31, 41, 52, 63,104, 44)(  2, 48,103, 45, 61, 66, 30, 79,
      83, 15)(  3, 14, 47, 23, 59, 95, 90, 38, 70, 49)(  4,  7, 86, 64, 16,
      46,102, 34,106, 74)(  5, 88, 67, 22, 21, 68, 91, 99, 60, 27)
    (  6, 94, 93, 62, 28, 98, 92, 11, 19, 32)(  8, 65, 33, 58, 13, 37,105, 96,
     71, 85)(  9, 84, 40, 97, 53, 56, 42, 75, 57, 25)( 10, 36,107, 72, 82, 29,
     51, 17, 87, 76)( 12, 55, 20, 18, 43, 26, 73, 81, 39,108)( 50, 80)
    ( 54, 78, 69, 77,101) ];

autGroups[19]:=
[ (  1, 90,173, 77,151, 69,128, 45,251, 76,233, 81)(  2, 89,174, 78,152, 70,
     127, 46,252, 75,234, 82)(  3,113,237,108,241, 29, 92,105,199,115,217,124)
    (  4,114,238,107,242, 30, 91,106,200,116,218,123)(  5,175,248,167,243,169,
     6,176,247,168,244,170)(  7, 11,206,185, 73,149,211,165,197,109,227,156)
    (  8, 12,205,186, 74,150,212,166,198,110,228,155)(  9,103, 57,177, 38, 33,
     48, 23,214, 96, 66, 14)( 10,104, 58,178, 37, 34, 47, 24,213, 95, 65, 13)
    ( 15,129,139, 17, 56, 51, 16,130,140, 18, 55, 52)( 19,239,145,189,138,236,
     20,240,146,190,137,235)( 21,231,179,216,194, 36,183, 60,131,144, 93,226)
    ( 22,232,180,215,193, 35,184, 59,132,143, 94,225)( 25, 42,220, 64, 98,172,
     26, 41,219, 63, 97,171)( 27,221,118, 83,100, 62,121, 80,181, 39,153,203)
    ( 28,222,117, 84, 99, 61,122, 79,182, 40,154,204)( 31,133,102, 68,141, 43,
     126,245,164,210,250, 88)( 32,134,101, 67,142, 44,125,246,163,209,249, 87)
    ( 49,191,187,224,230, 54, 50,192,188,223,229, 53)( 71,201,158,159,111,195,
     72,202,157,160,112,196)( 85,147,208,162,135,119, 86,148,207,161,136,120),
  (  1,144, 53,251, 62,190, 99,242, 50,131,  4, 86,150,214,202, 31, 44,159,
     177,197,119)(  2,143, 54,252, 61,189,100,241, 49,132,  3, 85,149,213,201,
     32, 43,160,178,198,120)(  5, 28, 35, 63,118,209,135,234, 38,188, 34, 48,
     224, 24,238,147,163, 80, 97, 21, 40)(  6, 27, 36, 64,117,210,136,233, 37,
     187, 33, 47,223, 23,237,148,164, 79, 98, 22, 39)(  7,157, 11,134,169,115,
     232,239,123, 74,192,114, 82,230,110, 89,235,194, 77,176,142)
    (  8,158, 12,133,170,116,231,240,124, 73,191,113, 81,229,109, 90,236,193,
      78,175,141)(  9, 51,103,200,220,102,221,161,184, 84,196,121,225,208,182,
     67,171,151, 66, 56, 14)( 10, 52,104,199,219,101,222,162,183, 83,195,122,
     226,207,181, 68,172,152, 65, 55, 13)( 15,185, 46,146, 94, 76, 71,250,211,
     167,166,246,112,107, 59,137, 29,228, 18,105, 70)( 16,186, 45,145, 93, 75,
     72,249,212,168,165,245,111,108, 60,138, 30,227, 17,106, 69)
    ( 19,154,217,140,179, 92, 41,155, 58,248,126, 88,244, 96,205, 25,128,215,
     130,173,204)( 20,153,218,139,180, 91, 42,156, 57,247,125, 87,243, 95,206,
     26,127,216,129,174,203) ];

autGroups[20]:=
[ ( 2, 3, 5, 8,12,16,20,26,30,33,35,34,32,28,24,19,15,11, 7, 4)
    ( 6, 9,13,14,10)(17,21,22,18)(23,27,31,29,25), 
  ( 1, 2, 3)( 4, 6, 5)( 7,11, 9)( 8,10,12)(13,15,17)(14,18,16)(19,23,21)
    (20,22,25)(24,28,27)(26,29,30)(31,32,33)(34,36,35), 
  ( 2, 3)( 4, 5)( 7, 8)( 9,10)(11,12)(13,14)(15,16)(17,18)(19,20)(21,22)
    (23,25)(24,26)(27,29)(28,30)(32,33)(34,35) ];

autGroups[21]:=
[ ( 1,52,80, 8,10,49,38)( 2,62,89,13,17,91,56)( 3,32,60,25, 5,75,76)
    ( 4,21,46,35,20, 9,85)( 6,57,84,39,14,77,82)( 7,54,31,59,19,41,18)
    (11,70,23,44,36,40,22)(12,65,79,72,43,87,34)(15,86,53,26,63,33,83)
    (16,74,51,50,45,37,90)(24,69,55,27,73,48,42)(28,71,61,47,58,68,67)
    (29,64,88,30,66,78,81), ( 1,57,33,41,88,79, 6,46,10, 4,24,90)
    ( 2,78,21,61,73,68,23,64,39,52,45,60)( 3, 8,84,81,29,62)( 5,63,38,20,77,
     12,11,87,35,72,91,22)( 7,50,82,74,58, 9,30,40,47,16,17,37)
    (13,65,66,31,26,89,25,51,69,44,76,70)(14,49,75,56,86,71,19,43,34,59,67,15)
    (18,27,83,32,36,42,55,85,80,53,48,54) ];

autGroups[22]:=
[ ( 1, 8,71,74,68,69,45,43,82,51,27,14)( 2,20, 7,79,64,89,39,87,42,52,32,16)
    ( 3,78,72,53,29,24)( 4,31,19,11,65,75,44,67,84,37,33,21)( 5,58,57,61,56,
     62,38,60,59,47,28,17)( 6,80,40,86,66,54,26,22, 9,50,30,23)
    (10,77,49,34,18,12,41,73,83,48,35,15)(13,81,76,90,70,91,46,88,85,55,36,25)
    , ( 1,70)( 2,65)( 3,64)( 4,68)( 5,67)( 7,39)( 8,29)( 9,69)(10,56)(11,48)
    (12,66)(13,18)(14,81)(15,76)(16,90)(17,88)(19,46)(20,36)(21,91)(22,63)
    (23,55)(24,85)(26,71)(27,79)(28,78)(30,41)(32,80)(33,58)(34,50)(35,77)
    (37,74)(38,73)(42,75)(43,57)(44,49)(45,72)(47,86)(51,89)(52,61)(54,83)
    (59,87)(62,82) ];

autGroups[23]:=
[ (  1, 48, 90,127, 22, 76,134,143,125,155,118,178,175, 13,180,100,  6,150, 91
     )(  2,131, 89, 16,156,144, 70,149,117,154, 54,179,174, 29,  5,185, 87,
     120, 99)(  3, 43, 46,140, 85, 80, 37, 33, 56, 62,147,164,163,115,168,170,
     190,108, 95)(  4, 26, 74,186,129,124,130,157,146,160, 19,116,176, 69,148,
     181, 94,  9, 98)(  7, 81, 92, 34, 12, 58,133, 65,152,159,142, 52,177,113,
     24, 75,182, 40,101)(  8,109, 93,167, 38,136,128, 82,106, 36,145, 59, 44,
     119,161, 83,102, 32, 97)( 10,137, 88, 79,103, 30,132, 41,138,158, 86,105,
     31,114, 57,172,112,162, 96)( 11, 39,139, 68, 20,104, 35, 50, 60,173, 27,
      55,184,188, 28, 71,183,110,123)( 14, 25, 73, 63,151,126,166, 53,189,171,
     18, 72,111, 66,121, 42,141, 51, 84)( 15,122, 61, 67,153,165, 17, 49,169,
      64, 23, 78,187, 47, 45,135, 21, 77,107), 
  (  1, 62)(  2, 63)(  3, 41)(  4, 56)(  5, 57)(  6, 58)(  7, 61)(  9, 60)
    ( 10, 59)( 11, 42)( 12, 91)( 13,104)( 14,117)( 15,126)( 16,173)( 17,159)
    ( 18, 78)( 19,165)( 20,181)( 21,111)( 22, 95)( 23,108)( 25,112)( 26,109)
    ( 27, 36)( 28,110)( 29,103)( 30, 65)( 31,164)( 32,153)( 33, 93)( 34,147)
    ( 35,158)( 37,119)( 38, 96)( 39,182)( 40,121)( 44,167)( 45, 47)( 46,132)
    ( 48, 98)( 49,179)( 50, 90)( 51,169)( 52,189)( 53,184)( 54, 84)( 55,177)
    ( 64, 76)( 66,124)( 67,186)( 68,144)( 69, 80)( 70,136)( 72,190)( 73, 88)
    ( 74, 97)( 75, 82)( 77,170)( 79,172)( 81,185)( 85,100)( 86,175)( 87,122)
    ( 89,145)( 92,114)( 94,156)(101,129)(102,116)(105,115)(106,113)(107,118)
    (120,151)(123,143)(125,188)(127,138)(128,139)(130,135)(131,137)(133,140)
    (134,141)(142,160)(146,171)(148,180)(149,162)(152,174)(155,166)(157,187)
    (161,176)(168,178) ];

autGroups[24]:=
[ (  1,136,  2,103, 43, 57,135, 21,145)(  3,154, 20,102,  8, 61, 54, 24,125)
    (  4,146, 52,109,134, 58,175, 33,140)(  5, 90, 40, 86, 88, 56, 99, 23, 89)
    (  6,126, 38,112, 26, 60, 19, 28, 44)(  7,148, 50,107, 74, 62,189, 34,144)
    (  9, 10, 45, 41,101, 59, 70, 37, 27)( 11,152, 51,111,113, 64,184, 32, 75)
    ( 12,147, 47,105,128, 63,169, 30,138)( 13,150, 49, 72,127, 68,187, 36,114)
    ( 14, 76, 46,110,132, 69,124, 31,141)( 15,149, 39,104,133, 67,190, 25,139)
    ( 16,153, 42,106,131, 65, 85, 29,143)( 17,151, 53,100,129, 66,180, 22,137)
    ( 18,115, 48,108,130, 55,162, 35,142)( 71, 91, 97, 96, 98, 87, 93, 95, 94)
    ( 73,156,182,177, 84,116,173,181,179)( 77,160,185,186,123,118,171,176, 80)
    ( 78,155,167,166,168,117,163,165,164)( 79,158,178, 82,161,122,172,183,119)
    ( 81,157, 83,159,188,121,174,120,170), 
  (  1,150)(  2,149)(  3,153)(  4,154)(  5,146)(  6,126)(  7,152)(  8,103)
    (  9,151)( 11, 90)( 12,148)( 13,147)( 14, 45)( 15, 28)( 16,136)( 17,115)
    ( 18, 61)( 19, 76)( 20,176)( 21,183)( 22,184)( 23,158)( 24,131)( 25,182)
    ( 26,108)( 27,181)( 29, 95)( 30,171)( 31,165)( 32, 50)( 34,141)( 35,120)
    ( 36, 66)( 37, 81)( 38,179)( 39,180)( 40,157)( 41,130)( 42,178)( 43,107)
    ( 44,177)( 46, 94)( 47,170)( 48,164)( 51,140)( 52,119)( 53, 65)( 54, 80)
    ( 55,190)( 56,161)( 57,134)( 58,188)( 59,111)( 60,186)( 62, 98)( 63,174)
    ( 64,168)( 67,144)( 68,123)( 70, 84)( 71,162)( 72,135)( 73,189)( 74,112)
    ( 75,187)( 77, 99)( 78,175)( 79,169)( 82,145)( 83,124)( 86,127)( 87,160)
    ( 88,104)( 89,159)( 92,156)( 93,155)( 96,137)( 97,116)(100,133)(102,132)
    (105,129)(106,128)(109,125)(110,113)(114,185)(117,173)(118,167)(121,143)
    (138,172)(139,166) ];

autGroups[25]:=
[ ( 1,21,17, 5,70,69,44)( 2,48,66,71,42, 8,74)( 3,28,39,36,20,16,38)
    ( 4,35,78,29,11,34,27)( 6,58,60,75,54,52,12)( 7,37,64,63,10,23,15)
    ( 9,43,22,13,30,46,33)(14,73,19,18,72,65,24)(25,59,31,47,49,45,67)
    (26,53,32,62,40,56,41)(50,76,55,61,57,51,68), 
  ( 1,66,45,56,73,72,25,26,74,63,52,54)( 2,37,40,16,23,24,69,46,59,17,48,58)
    ( 3,75,14,62,68,31,65,12,13,36,77, 9)( 4,30,19,28,15,53,42,49,70,22,18,38)
    ( 5,35,27, 7,41,29,50,20,33,51,11,67)( 6,10,34,71, 8,78,44,60,47,55,61,32)
    (21,64,76,39,43,57) ];

autGroups[26]:=
[ ( 1,25,66,38,65,37)( 2,56,73,69,60,19)( 3,13,47,63,78,54)( 4,31,59)
    ( 5,68,45)( 6,70,64,10,41,28)( 7,35,77,50,75,43)( 8,44,62,71,49,11)
    ( 9,51,23,33,36,15)(12,26,46,20,17,72)(14,57,32,34,55,39)(16,74,24,29,58,
     27)(18,52,76,53,22,67)(21,42,40,30,48,61), 
  ( 1,77,55,51,71,64, 2,62,54,25,69,76)( 3,57,78,70, 5,53)( 4,39,17,33,52,10,
      7,26,34,31,38, 8)( 6,24,11,15,75,65,16,22,12,13,63,72)( 9,73,43,28,61,
     74,21,67,36,56,50,59)(14,47,32,66,27,45,19,58,60,68,29,44)
    (18,37,42,48,49,41,30,35,20,46,23,40) ];

autGroups[27]:=
[ ( 1, 8,70,50, 4, 2,25,74,18, 3, 5,42,66,11)( 6,13,20,10,29, 7,45)
    ( 9,38,77,17,21,16,72,63,12,32,27,73,52,15)(14,19,26,56,40,24,23,41,28,47,
     30,33,44,37)(22,35,55,57,69,36,48,58,43,53,31,62,67,59)(34,71,46,75,65,
     39,64,49,78,60,54,51,61,76), ( 1,16,56,49,15,78,57)( 2,20,27,58,65,43,14)
    ( 3,30,76,51,37,18,39)( 4,59,53,36,11,71,60)( 5,25,42,74,63,61,32)
    ( 6,41,38,26,45, 8,70)( 7,13,77,19,55,28, 9)(10,44,67,34,54,48,33)
    (17,75,23,73,50,35,46)(21,40,69,68,22,24,52)(29,66,47,31,64,62,72) ];

autGroups[28]:=
[ (  1, 34)(  2, 98)(  3,152)(  4,107)(  5,150)(  6, 30)(  8, 33)(  9, 27)
    ( 10, 42)( 11,139)( 12,102)( 13,140)( 14,151)( 15,138)( 16, 83)( 17,118)
    ( 18, 59)( 19,120)( 20, 40)( 21,146)( 22,143)( 23,167)( 24,128)( 25,166)
    ( 26, 32)( 28, 35)( 36, 50)( 37, 94)( 38, 97)( 39, 44)( 41, 56)( 43,105)
    ( 46, 81)( 47,119)( 48,108)( 49,116)( 51, 63)( 52, 82)( 53,117)( 54,114)
    ( 55,115)( 57,132)( 58,109)( 60, 66)( 61, 93)( 62, 95)( 64, 92)( 65, 89)
    ( 67,145)( 68, 86)( 69,144)( 70, 90)( 71, 72)( 73, 80)( 74, 78)( 76, 79)
    ( 84, 96)( 85, 91)( 87,125)( 88,123)( 99,170)(100,134)(101,165)(103,121)
    (104,160)(110,124)(111,159)(112,137)(113,130)(122,141)(127,129)(131,153)
    (133,142)(135,158)(136,163)(147,171)(148,149)(154,162)(155,157)(164,168), 
  (  1, 89, 13, 60,147)(  2, 72,108, 56,155)(  3, 86,141, 27, 96)
    (  4, 90, 33, 46,169)(  5, 40,136, 59,104)(  6, 92,115, 58, 21)
    (  7, 95,148, 28, 52)(  8, 51,161, 48,149)(  9, 98,131, 61,111)
    ( 10,145, 63, 26, 93)( 11,144,120, 47,168)( 12,143,117, 57,162)
    ( 14, 83, 99, 29, 44)( 15,146,135, 49,152)( 16, 75,106, 70, 31)
    ( 17, 87,142,124,165)( 18, 42,133, 82,101)( 19, 55,140,122,171)
    ( 20, 73, 62, 78,109)( 22, 74,107,123,157)( 23, 36,100,116, 45)
    ( 24, 88,138,121,160)( 25, 41,139,114, 84)( 30, 91, 65, 76, 66)
    ( 32, 94,125, 85, 69)( 34, 97,164, 37,110)( 35, 50,156, 64, 77)
    ( 38,112,119,105,127)( 39,113, 81,102,132)( 43,154,118,103,150)
    ( 53,153,163,170,158)( 54,151,134,159,126)( 67, 71, 79, 68, 80)
    (128,129,167,137,166) ];

autGroups[29]:=
[ (  1, 14, 36, 13, 57, 56, 47, 66, 43, 22,  6, 25)(  2, 62, 31, 16, 42, 24)
    (  3, 53, 33, 20, 51, 55, 40, 29,  4, 61, 32, 17)(  5, 45, 28, 10, 63, 34,
     18, 65, 46, 59, 44, 26)(  7, 50, 37, 19, 27,  9, 58, 35, 12, 64, 41, 23)
    (  8, 11, 38, 15, 21, 54, 52, 60, 48, 49, 39, 30)( 67,182,120,151, 76,195,
     122,171, 81,183,131,148)( 68,157,107,173, 78,168,128,137,103,180, 93,150)
    ( 69,146, 75,177, 79,174, 86,196,101,181,102,144)( 70,170,119,138, 82,197,
     95,191, 99,158,108,145)( 71,194, 90,172, 84,159,110,187, 97,163,129,142)
    ( 72,185,112,164, 80,135,132,198, 94,179,123,149)( 73,193,100,175, 87,153)
    ( 74,190,118,162, 85,186,114,169,104,136,130,152)( 77,143,121,176, 83,189,
     126,154, 98,178,117,147)( 88,192, 89,155,106,156,109,166,116,133,127,139)
    ( 91,134,125,161, 96,184,113,160,111,141,115,167)( 92,165,124,140,105,188)
    , (  1,186,126, 19,148,105, 53,180, 90, 65,133,102,  5,159,128, 17,193,
     115, 58,176, 74, 36,166, 99, 39,149,111, 24,182,112, 60,138, 69)
    (  2,174, 73, 15,187,125, 45,152, 96, 54,163,100, 66,140,129,  8,195,114,
      18,171, 71, 38,188, 89, 42,147, 98, 55,158,119, 29,143,117)
    (  3,168,101, 16,154,127, 25,194, 92, 57,164, 67, 37,156,124, 30,146,109,
      52,175, 75, 35,141,123,  6,153,110, 13,181,121, 62,139, 93)
    (  4,189, 86, 12,191, 82, 50,192, 83, 51,165, 81, 49,155, 80, 26,190, 78,
      47,172, 76, 31,184, 84, 43,150, 85, 46,179, 79, 48,134, 87)
    (  7,135,131, 10,196,132, 14,198, 88, 63,167, 94, 64,160,104, 22,157,116,
      23,177, 70, 33,173, 68, 32,145,106, 41,144,107, 56,136,120)
    (  9,183,122, 20,137,130, 11,197, 95, 21,169,103, 61,161, 91, 27,170, 97,
      59,162, 77, 28,185,113, 40,151, 72, 34,178,118, 44,142,108) ];

autGroups[30]:=
[ ( 1,23,10,47)( 2,24,11,48)( 3,21,12,45)( 4,22, 9,46)( 5,15,44,36)
    ( 6,16,41,33)( 7,13,42,34)( 8,14,43,35)(17,71,37,55)(18,72,38,56)
    (19,69,39,53)(20,70,40,54)(25,61,29,49)(26,62,30,50)(27,63,31,51)
    (28,64,32,52)(57,59)(58,60)(65,67)(66,68), 
  ( 1,28,68,70,17,63,38,16)( 2,25,65,71,18,64,39,13)( 3,26,66,72,19,61,40,14)
    ( 4,27,67,69,20,62,37,15)( 5, 6, 7, 8)( 9,58,36,54,32,24,43,47)
    (10,59,33,55,29,21,44,48)(11,60,34,56,30,22,41,45)(12,57,35,53,31,23,42,46
     )(49,52,51,50) ];

autGroups[31]:=
[ (  1, 47,  9, 63, 30, 55, 11, 40, 25, 56, 33, 64, 23, 44)(  2, 38, 17, 49,
      36, 45, 18, 37,  4, 42, 19, 52, 26, 41)(  3, 54, 35, 69, 22, 53, 29, 43,
     15, 66, 24, 57, 12, 50)(  5, 65, 28, 60,  6, 46, 34, 51, 31, 71, 27, 48,
       7, 58)(  8, 70, 16, 59, 20, 72, 14, 61, 21, 68, 13, 62, 32, 67)
    ( 10, 39)( 73, 84, 95,104, 92, 87, 76)( 74, 80, 98, 96,107, 85, 97)
    ( 75, 93, 90, 86,102,105, 89)( 77,100, 99,106, 78,103, 82)
    ( 81, 88,101, 83, 94,108, 91), (  1, 77, 42, 29,101, 48, 36,102, 55)
    (  2, 86, 66, 18, 83, 62, 10, 94, 72)(  3, 96, 56, 27, 90, 59, 19,108, 69)
    (  4, 80, 58, 11,106, 64, 26, 78, 46)(  5, 91, 49, 16, 99, 43, 33, 98, 60)
    (  6,104, 71, 23, 74, 54, 17,103, 67)(  7, 73, 47, 34, 97, 57, 30, 92, 53)
    (  8, 81, 39, 21, 88, 37, 31,100, 41)(  9,105, 51, 13, 95, 68, 24, 82, 63)
    ( 12, 75, 38, 25, 93, 44, 22,107, 45)( 14, 89, 52, 28, 85, 40, 15, 84, 50)
    ( 20, 79, 70, 32, 76, 65, 35, 87, 61) ];

autGroups[32]:=
[ (  1,127,718,867,346,586,841)(  2,128,717,868,345,585,842)(  3,466,779,360,
     785,588,105)(  4,465,780,359,786,587,106)(  5,277,387,866,781,572,852)
    (  6,278,388,865,782,571,851)(  7,468,609,823,669,567,556)
    (  8,467,610,824,670,568,555)(  9,368,777,809,696,591,211)
    ( 10,367,778,810,695,592,212)( 11,477,769,109,150,574,858)
    ( 12,478,770,110,149,573,857)( 13,488,764,710,792,185,309)
    ( 14,487,763,709,791,186,310)( 15,410,430,445,432,416,441)
    ( 16,409,429,446,431,415,442)( 17,470,776,793,579,590, 50)
    ( 18,469,775,794,580,589, 49)( 19,452,508,523,510,493,519)
    ( 20,451,507,524,509,494,520)( 21,229,740,626,783,570,651)
    ( 22,230,739,625,784,569,652)( 23,482, 38,486,762,593,789)
    ( 24,481, 37,485,761,594,790)( 25,475,668,655,545,236,751)
    ( 26,476,667,656,546,235,752)( 27,453,249,756,611,584,679)
    ( 28,454,250,755,612,583,680)( 29,459,543,266,742,566,819)
    ( 30,460,544,265,741,565,820)( 31,484,200,313,389,596,805)
    ( 32,483,199,314,390,595,806)( 33,489,344,836,787, 79,159)
    ( 34,490,343,835,788, 80,160)( 35,458,766,684,642,530,262)
    ( 36,457,765,683,641,529,261)( 39,455,771, 53,472,577,854)
    ( 40,456,772, 54,471,578,853)( 41,463,774,216,300,374,856)
    ( 42,464,773,215,299,373,855)( 43,462,639,559,251,576,621)
    ( 44,461,640,560,252,575,622)( 45,480, 93,163,719,597,356)
    ( 46,479, 94,164,720,598,355)( 47, 73,148,733,798,330,831)
    ( 48, 74,147,734,797,329,832)( 51,179,297,404,796,581,705)
    ( 52,180,298,403,795,582,706)( 55,473,694,859,201,283,400)
    ( 56,474,693,860,202,284,399)( 57,324,767,846, 96,134,729)
    ( 58,323,768,845, 95,133,730)( 59,144,737,363,352,829,103)
    ( 60,143,738,364,351,830,104)( 61,120,384,869,348,702,843)
    ( 62,119,383,870,347,701,844)( 63,145,605,827,336,648,554)
    ( 64,146,606,828,335,647,553)( 65,124,736,814,337,834,209)
    ( 66,123,735,813,338,833,210)( 67,156,728,113,121,725,849)
    ( 68,155,727,114,122,726,850)( 69,166,721,713,357,208,307)
    ( 70,165,722,714,358,207,308)( 71,126,425,450,321,438,440)
    ( 72,125,426,449,322,437,439)( 75,130,504,527,325,516,518)
    ( 76,129,503,528,326,515,517)( 77,118,716,630,349,675,649)
    ( 78,117,715,629,350,676,650)( 81,154,663,659,328,257,749)
    ( 82,153,664,660,327,258,750)( 83,132,245,759,332,815,678)
    ( 84,131,246,760,331,816,677)( 85,138,540,270,341,617,818)
    ( 86,137,539,269,342,618,817)( 87,161,195,317,320,837,803)
    ( 88,162,196,318,319,838,804)( 89,167,339,840,353,101,157)
    ( 90,168,340,839,354,102,158)( 91,135,723,688,333,551,259)
    ( 92,136,724,687,334,552,260)( 97,142,731,219,272,396,847)
    ( 98,141,732,220,271,395,848)( 99,140,636,563,224,748,619)
    (100,139,635,564,223,747,620)(107,116,293,408,362,802,704)
    (108,115,294,407,361,801,703)(111,152,689,864,174,305,397)
    (112,151,690,863,173,306,398)(169,296,376,825,671,633,557)
    (170,295,375,826,672,634,558)(171,273,405,812,698,707,213)
    (172,274,406,811,697,708,214)(175,315,392,711,808,193,311)
    (176,316,391,712,807,194,312)(177,276,366,448,433,423,443)
    (178,275,365,447,434,424,444)(181,279,370,526,512,502,521)
    (182,280,369,525,511,501,522)(183,221,386,628,800,661,653)
    (184,222,385,627,799,662,654)(187,304,379,657,548,244,754)
    (188,303,380,658,547,243,753)(189,281,225,758,613,700,682)
    (190,282,226,757,614,699,681)(191,287,371,267,743,604,821)
    (192,288,372,268,744,603,822)(197,286,394,685,644,537,263)
    (198,285,393,686,643,538,264)(203,292,401,217,302,381,862)
    (204,291,402,218,301,382,861)(205,290,377,562,253,691,623)
    (206,289,378,561,254,692,624)(227,428,418,435,421,419,414)
    (228,427,417,436,422,420,413)(231,505,496,513,499,498,492)
    (232,506,495,514,500,497,491)(233,247,607,616,674,631,534)
    (234,248,608,615,673,632,533)(237,745,602,645,536,239,542)
    (238,746,601,646,535,240,541)(241,637,532,256,666,599,550)
    (242,638,531,255,665,600,549), 
  (  1,825,126,317,467,331,382,186,232,810,732,639,676,841,546)
    (  2,826,125,318,468,332,381,185,231,809,731,640,675,842,545)
    (  3,870,428,271,367,187,244,582,503,776,682,651,786,103,549)
    (  4,869,427,272,368,188,243,581,504,775,681,652,785,104,550)
    (  5,267,434,293,470,602,706,579, 76,824,161,660,747,353,372)
    (  6,268,433,294,469,601,705,580, 75,823,162,659,748,354,371)
    (  7,811,426,297,461,622,696, 80,513,164,863,638,352,397,183)
    (  8,812,425,298,462,621,695, 79,514,163,864,637,351,398,184)
    (  9,488,417,292,456,495,709,592,497,836,844,533, 46,847,413)
    ( 10,487,418,291,455,496,710,591,498,835,843,534, 45,848,414)
    ( 11,406,177,221,474,606,694,570,519,793,107,645,156,849,541)
    ( 12,405,178,222,473,605,693,569,520,794,108,646,155,850,542)
    ( 13,217,228,301,465,610,662,590,510,110,821,137,839,749,328)
    ( 14,218,227,302,466,609,661,589,509,109,822,138,840,750,327)
    ( 15,315,410,286,464,566,502,593,521,655,833,649,552, 47,562)
    ( 16,316,409,285,463,565,501,594,522,656,834,650,551, 48,561)
    ( 17,627,424,283,452,626,707,568,515,846,557, 28,837,440,281)
    ( 18,628,423,284,451,625,708,567,516,845,558, 27,838,439,282)
    ( 19,866,444,287,478,619,538, 23,526,446,312,459,618,704,529)
    ( 20,865,443,288,477,620,537, 24,525,445,311,460,617,703,530)
    ( 21, 55,447,275,277,457,603,571,494,524,860,653,648,829,554)
    ( 22, 56,448,276,278,458,604,572,493,523,859,654,647,830,553)
    ( 25,712,415,279,486,624,634,585,518,560, 51,658,437,307,453)
    ( 26,711,416,280,485,623,633,586,517,559, 52,657,438,308,454)
    ( 29,855,431, 66,475,135,713,575,325,404,214,240,801,728,544)
    ( 30,856,432, 65,476,136,714,576,326,403,213,239,802,727,543)
    ( 31,595,412,314,484,600,702,587,492, 54,862,419,305,480,531)
    ( 32,596,411,313,483,599,701,588,491, 53,861,420,306,479,532)
    ( 33,778,421,310,471, 82,700,134,528,755,358,377,207,260,548)
    ( 34,777,422,309,472, 81,699,133,527,756,357,378,208,259,547)
    ( 35,361,366,172,230,614,689,577,500,853,791, 84,816,157,564)
    ( 36,362,365,171,229,613,690,578,499,854,792, 83,815,158,563)
    ( 37,685,441,300, 73,616,142,598,505,359,402,189,258,803,540)
    ( 38,686,442,299, 74,615,141,597,506,360,401,190,257,804,539)
    ( 39,111,435,119,489,607,338,374,181,266,807,636,768,678,556)
    ( 40,112,436,120,490,608,337,373,182,265,808,635,767,677,555)
    ( 41,735,429,289,482,611, 88,584,130,868,753,333,396,209,234)
    ( 42,736,430,290,481,612, 87,583,129,867,754,334,395,210,233)
    ( 43,165,450,296,324,376,193,235,511,733,774,632,832,787, 78)
    ( 44,166,449,295,323,375,194,236,512,734,773,631,831,788, 77)
    ( 49,796, 72,303,128,630,692,329,370,216,264,644,726,769,535)
    ( 50,795, 71,304,127,629,691,330,369,215,263,643,725,770,536)
    ( 57,758,321,273,180,237,698,573,508,683,852,642,101,818,131)
    ( 58,757,322,274,179,238,697,574,507,684,851,641,102,817,132)
    ( 59,827,146,364,386,174,226,203,245,764,663,771,669,105,784)
    ( 60,828,145,363,385,173,225,204,246,763,664,772,670,106,783)
    ( 61,255,151,738,740,336,400,202, 63,800,143,779,665,356,390)
    ( 62,256,152,737,739,335,399,201, 64,799,144,780,666,355,389)
    ( 67,393,116,269,743,340,388,191,262,781, 89,766,139,858,741)
    ( 68,394,115,270,744,339,387,192,261,782, 90,765,140,857,742)
    ( 69,206,118,813,715,344,379,212,251, 98,724,147,688,751,345)
    ( 70,205,117,814,716,343,380,211,252, 97,723,148,687,752,346)
    ( 85,820,150,114,746,121,407,197,223,392,195,250,671,729,761)
    ( 86,819,149,113,745,122,408,198,224,391,196,249,672,730,762)
    ( 91,350,123,220,248,347,384,199,241,806,719, 94,673,160,798)
    ( 92,349,124,219,247,348,383,200,242,805,720, 93,674,159,797)
    ( 95, 99,154,168,759,342,319,176,170,254,721,717,668,679,789)
    ( 96,100,153,167,760,341,320,175,169,253,722,718,667,680,790) ];

autGroups[33]:=
[ (  1, 89, 33,155,125,  3, 94, 34,156,121)(  2, 95, 30,149,126,  4, 91, 26,
     148,122)(  5, 88, 32,151,124,  6, 96, 36,154,123)(  7, 85, 25,147,130,
       8, 93, 29,150,129)(  9, 90, 35,153,131, 11, 86, 31,152,127)
    ( 10, 87, 27,145,132, 12, 92, 28,146,128)( 13, 73,102,236,165, 47,221,211,
     64, 49)( 14, 77, 98,231,166, 48,217,214, 62, 51)( 15, 81,105,237,157, 39,
     222,208, 63, 50)( 16, 82,106,239,161, 42,218,216, 61, 52)
    ( 17, 74,101,235,162, 41,226,215, 67, 55)( 18, 78, 97,229,158, 40,225,209,
     68, 56)( 19, 79,108,240,167, 45,220,212, 65, 53)( 20, 83,104,234,163, 44,
     219,206, 66, 54)( 21, 75, 99,230,164, 43,227,205, 72, 57)
    ( 22, 76,100,232,168, 46,223,213, 70, 59)( 23, 80,107,238,159, 37,228,207,
     71, 58)( 24, 84,103,233,160, 38,224,210, 69, 60)(109,136,175,185,197,115,
     137,179,190,193)(110,134,176,184,196,112,133,171,192,195)
    (111,135,169,191,198,116,138,177,187,194)(113,139,172,186,203,118,142,180,
     182,199)(114,140,170,183,204,120,141,174,188,200)(117,144,178,181,202,
     119,143,173,189,201)(241,248,246,244,250)(243,252,245,247,249), 
  (  1,223, 13,139)(  2,226, 14,133)(  3,220, 17,140)(  4,228, 18,135)
    (  5,227, 15,143)(  6,221, 16,141)(  7,224, 21,136)(  8,218, 22,134)
    (  9,217, 19,142)( 10,225, 20,137)( 11,219, 23,144)( 12,222, 24,138)
    ( 25, 36)( 26, 35)( 27, 28, 29, 30)( 31, 34, 33, 32)( 37,208, 73,246)
    ( 38,216, 74,252)( 39,211, 77,249)( 40,214, 78,248)( 41,209, 75,243)
    ( 42,215, 76,251)( 43,206, 81,242)( 44,212, 82,250)( 45,207, 79,245)
    ( 46,210, 80,244)( 47,205, 83,241)( 48,213, 84,247)( 49, 60)( 50, 59)
    ( 51, 54, 53, 52)( 55, 56, 57, 58)( 61,202, 89,171)( 62,201, 90,179)
    ( 63,194, 88,176)( 64,198, 87,177)( 65,197, 85,174)( 66,193, 86,180)
    ( 67,204, 95,169)( 68,200, 96,175)( 69,199, 94,172)( 70,203, 93,173)
    ( 71,196, 91,170)( 72,195, 92,178)( 97,127, 99,128)( 98,124,100,122)
    (101,130,102,132)(103,121,104,123)(105,131,107,129)(106,125,108,126)
    (109,164,147,188)(110,165,148,189)(111,159,145,183)(112,167,146,191)
    (113,168,150,192)(114,162,149,186)(115,163,152,187)(116,157,151,181)
    (117,158,155,182)(118,166,156,190)(119,160,153,184)(120,161,154,185)
    (229,240)(230,234,231,236)(232,237)(233,239,235,238) ];

autGroups[34]:=
[ ( 1,38,16,46,20,43,23,40,26,48, 6,49,17,39,27,42,19,55)( 2,36,14,50, 3,37,
      9,53, 5,47, 8,56,25,32,21,44,24,31)( 4,51, 7,41,12,52,10,54,13,33,28,34,
     18,30,22,35,15,45)(11,29)(57,82,84,58,71,67,63,65,69)(59,70,66,73,75,81,
     62,83,74)(60,68,64,79,80,61,72,77,78), 
  ( 1,53,67,24,48,78)( 2,41,68,18,47,73)( 3,39,61,20,32,57)( 4,46,59,17,33,69)
    ( 5,50,60,25,31,80)( 6,52,63,22,38,81)( 7,56,66,14,34,79)( 8,54,64,15,36,
     83)( 9,44,77)(10,43,62,27,35,82)(11,49,76,16,29,65)(12,30,75)
    (13,55,74,26,45,71)(19,42,58,23,40,84)(21,37,72)(28,51,70) ];

autGroups[35]:=
[ ( 1,20,53,73,74,45,15, 3,18,68,70,69,29, 8, 9,26,63,49,60,21)( 2,14,28,16)
    ( 4, 6,65,47,71,54,37,25,11,35,27,39,38,10,23,41,48,72,44,57)( 7,13,31,34)
    (12,32,24,22,66,42,40,46,43,67)(17,51,58,52)(19,50,62,61)(30,56,59,33)
    (36,55,75,64), ( 1,73,30,41,44,14,10,48,62, 9,49,13,42,43,31)
    ( 2,47,11,28, 6,68,17,53,12, 5,22,26,58,18,27)( 3,34,32,65,19,63, 4,16,39,
     20,59,35,24, 7,70)( 8,61,57,66,36,72,21,52,60,23,64,67,38,33,74)
    (15,71,55,46,45,51,29,40,75,25,69,50,54,37,56) ];

gens[1]:=[ 1, 16, 27 ];
gens[2]:=[ 1, 2, 5 ];
gens[3]:=[ 1, 4, 17 ];
gens[4]:=[ 1, 7, 13 ];
gens[5]:=[ 1, 4, 7 ];
gens[6]:=[ 1, 2, 12 ];
gens[7]:=[ 1, 2, 6 ];
gens[8]:=[ 1, 22, 43 ];
gens[9]:=[ 1, 2, 3 ];
gens[10]:=[ 1, 10, 19 ];
gens[11]:=[ 1, 2, 4 ];
gens[12]:=[ 1, 23, 38 ];
gens[13]:=[ 1, 2, 6 ];
gens[14]:=[ 1, 2, 3 ];
gens[15]:=[ 1, 2, 5 ];
gens[16]:=[ 1, 2, 3 ];
gens[17]:=[ 1, 2, 34 ];
gens[18]:=[ 1, 2, 3 ];
gens[19]:=[ 1, 4, 190 ];
gens[20]:=[ 1, 2, 3 ];
gens[21]:=[ 1, 2, 75 ];
gens[22]:=[ 1, 27, 67 ];
gens[23]:=[ 1, 2, 141 ];
gens[24]:=[ 1, 40, 102 ];
gens[25]:=[ 1, 3, 8 ];
gens[26]:=[ 1, 2, 3 ];
gens[27]:=[ 1, 2, 3 ];
gens[28]:=[ 1, 2, 6 ];
gens[29]:=[ 1, 72, 172 ];
gens[30]:=[ 1, 5, 17 ];
gens[31]:=[ 1, 37, 73 ];
gens[32]:=[ 1, 247, 500 ];
gens[33]:=[ 1, 13, 25 ];
gens[34]:=[ 1, 29, 57 ];
gens[35]:=[ 1, 2, 3 ];

mg[1]:=[ 7, 8 ];
mg[2]:=[ 8, 9 ];
mg[3]:=[ 8, 9 ];
mg[4]:=[ 10, 14 ];
mg[5]:=[ 7, 15 ];
mg[6]:=[ 9, 16 ];
mg[7]:=[ 8, 17 ];
mg[8]:=[ 8, 23 ];
mg[9]:=[ 10, 26 ];
mg[10]:=[ 12, 29 ];
mg[11]:=[ 10, 46 ];
mg[12]:=[ 10, 46 ];
mg[13]:=[ 8, 47 ];
mg[14]:=[ 13, 51 ];
mg[15]:=[ 12, 57 ];
mg[16]:=[ 12, 57 ];
mg[17]:=[ 9, 70 ];
mg[18]:=[ 10, 74 ];
mg[19]:=[ 8, 86 ];
mg[20]:=[ 20, 86 ];
mg[21]:=[ 12, 93 ];
mg[22]:=[ 12, 93 ];
mg[23]:=[ 9, 97 ];
mg[24]:=[ 9, 97 ];
mg[25]:=[ 14, 106 ];
mg[26]:=[ 14, 106 ];
mg[27]:=[ 14, 106 ];
mg[28]:=[ 10, 116 ];
mg[29]:=[ 10, 134 ];
mg[30]:=[ 17, 134 ];
mg[31]:=[ 14, 146 ];
mg[32]:=[ 7, 147 ];
mg[33]:=[ 10, 170 ];
mg[34]:=[ 18, 170 ];
mg[35]:=[ 20, 177 ];
	else
		n:=6*(2-2*g)/(6-m);
		f2:= 2*m*(2-2*g)/(6-m);
autGroups[1]:=
  [ ( 1,13,12, 3, 7,16)( 2,21,19)( 4,23, 8, 6,20, 9)( 5,15,18)(10,17,22,11,14,
         24), ( 1,18,17)( 2,10, 4)( 3,20,22)( 5,12,23)( 6,24,15)( 7,11, 9)
        ( 8,16,21)(13,14,19) ];

autGroups[2]:=
[ ( 2, 3)( 4, 5)( 7, 8)(10,11), ( 1, 7, 6,11,12, 5, 9, 2)( 3,10) ];

autGroups[3]:=
[ ( 2, 3, 5, 8,14,12, 7, 4)( 6, 9,15,10)(11,16,20,13,19,23,22,17)(18,21), 
      ( 1, 2, 3)( 4, 6, 5)( 7,11, 9)( 8,10,13)(12,18,16)(14,20,21)(15,17,19)
        (22,24,23), ( 2, 3)( 4, 5)( 7, 8)( 9,10)(11,13)(12,14)(16,20)(17,19)
        (18,21)(22,23) ];

autGroups[4]:=
[ ( 1, 5, 6, 2,12, 7,15,13, 9,11)( 3,14)( 4,10), 
      ( 1,15, 6, 9,12)( 2,10,11, 8, 7, 4, 5,14,13, 3) ];

autGroups[5]:=
[ ( 1,41,55,34,49, 9,26,72,11,23,68,43,51,62)( 2,64,29,59,32,28, 4,70,18,37,
         48,19,17,66)( 3,56,67,22,60,47,15,71,33,12,27,24,61,39)
        ( 5,46,57, 6,30,31,42,69,53, 7,65,52,40,35)( 8,14,44,20,13,16,50,63,
         36,38,25,45,58,21)(10,54), 
      ( 1,70, 3,65,13,25,39, 7,66)( 2,71, 6,45,20,15,57, 4,72)
        ( 5,41,16,68,49,38,62,12,54)( 8,64, 9,27,30,47,35,23,28)
        (10,69,11,58,34,43,44,26,67)(14,56,31,48,60,53,21,37,19)
        (17,63,18,51,22,52,61,42,55)(24,46,50,29,32,36,33,40,59) ];

autGroups[6]:=
[ ( 2, 3, 5, 8,14,12, 7, 4)( 6, 9,15,22,29,23,16,10)(11,17,24,31,26,20,13,18
         )(19,21,27,34,41,40,32,25)(28,35,38,30,37,33,39,36), 
      ( 1, 2, 3)( 4, 6, 5)( 7,11, 9)( 8,10,13)(12,19,17)(14,20,21)(15,18,16)
        (22,23,28)(24,25,30)(26,33,27)(29,36,35)(31,38,39)(32,34,37)
        (40,42,41), ( 2, 3)( 4, 5)( 7, 8)( 9,10)(11,13)(12,14)(15,16)(17,20)
        (19,21)(22,23)(24,26)(25,27)(30,33)(32,34)(35,36)(38,39)(40,41) ];

autGroups[7]:=
[ ( 2, 3, 5, 8,14,12, 7, 4)( 6, 9,15,24,34,26,16,10)(11,17,27,21,13,20,28,18
         )(19,22,30,39,33,37,38,29)(23,31,40,41,36,25,35,32), 
      ( 1, 2, 3)( 4, 6, 5)( 7,11, 9)( 8,10,13)(12,19,17)(14,21,22)(15,18,23)
        (16,25,20)(24,32,33)(26,37,35)(27,29,30)(28,36,31)(34,39,38)
        (40,41,42), ( 2, 3)( 4, 5)( 7, 8)( 9,10)(11,13)(12,14)(15,16)(17,21)
        (18,20)(19,22)(23,25)(24,26)(29,30)(31,36)(32,35)(33,37)(38,39)
        (40,41) ];

autGroups[8]:=
[ ( 1,22,17)( 2,30, 3, 8,28,14,12,29, 7)( 4,10, 9, 5,15,31,20,32,16)
        ( 6,34,23,25,26,24,27,19,33)(11,35,13)(18,36,21), 
      ( 1,27,21,25,18, 6)( 2,12,31, 9,33,24)( 3, 7,34,19,32,15)
        ( 4,11,20,36, 5,13)( 8,23,16)(10,26,14)(17,30,35,29,22,28) ];

autGroups[9]:=
[ ( 2, 3, 5, 8,11,14,17,16,13,10, 7, 4)( 6, 9)(12,15), 
      ( 1, 2, 3)( 4, 6, 5)( 7, 8, 9)(10,12,11)(13,14,15)(16,18,17), 
      ( 2, 3)( 4, 5)( 7, 8)(10,11)(13,14)(16,17) ];

autGroups[10]:=
[ ( 1, 6,15,19,32,36,29,28, 9,16)( 2,27,18,30,17,21,14, 3,26, 8)
        ( 4,20,25, 7,24,13,35,33,22,34)( 5,11)(10,23)(12,31), 
      ( 1,17,23, 2, 9,33, 6, 5,28,13)( 4,25, 8,11,21)( 7,10,34,19,26,29,31,
         15,14,36)(12,24,27,30,22)(16,18)(20,32) ];

autGroups[11]:=
[ ( 2, 3, 5, 8,14,20,23,22,17,12, 7, 4)( 6, 9,10)(11,15,18,13)(16,21,19), 
      ( 1, 2, 3)( 4, 6, 5)( 7,11, 9)( 8,10,13)(12,16,15)(14,18,19)(17,20,21)
        (22,24,23), ( 2, 3)( 4, 5)( 7, 8)( 9,10)(11,13)(12,14)(15,18)(16,19)
        (17,20)(22,23) ];

autGroups[12]:=
[ (  1, 90,109,118, 44, 82,100, 16, 84,115, 61, 78, 53, 95)(  2, 79, 37,146,
         124,145, 43, 13,119,106,113,151, 68, 51)(  3, 57, 77, 83, 92,108, 66
         )(  4, 48, 54,114, 58, 93,123, 30,111, 89,122,131, 75, 69)
        (  5, 34,121, 56, 64,112, 91,  9, 22,152, 21,107, 81,130)
        (  6,141, 40,127,150,116, 19)(  7,110, 96,134, 67, 45,149, 55,147,
          63, 42,101, 33,128)(  8, 87, 86,104, 14, 60, 59, 31, 41,143, 35,
          47, 36,137)( 10, 29, 62, 85,120,133, 80, 49, 76,140,156,125, 46, 17
         )( 11,132, 12,136,135,154, 24, 20,142, 50, 98,144, 70, 28)
        ( 15, 74, 52,153, 38, 65,138, 72,117, 25, 99,126,105, 88)
        ( 18,102, 23,129,155,103, 71, 39,148, 97,139, 94, 26, 32)( 27, 73), 
      (  1,102, 44,110, 80, 60, 91, 11, 69, 98,101, 76, 31,121)
        (  2,128, 64, 74, 49, 73, 59)(  3,132, 14, 48, 71, 87,100,  6, 95,
          41, 32,111, 47,142)(  4,153, 92, 99, 30,143, 35, 18, 88,127,138,
          39,104, 86)(  5, 79, 24, 90, 43, 34, 66, 22, 51, 84, 28,119,  9,141
         )(  7,114,124, 65,103,112, 70, 33, 94,113,126, 89, 56,109)
        (  8, 29, 67,136,123, 20,130, 36, 17,147, 78,148, 16,152)
        ( 10,117,107,149, 13,137, 27)( 12, 62, 58,129,133, 82,116, 53, 46,
         139,131,140, 50, 77)( 15,134,150, 42, 72,144, 61, 52, 63, 83, 96,
          25,118,135)( 19, 57)( 21, 54, 38,146,155, 45,154, 81, 26,105,151,
         122, 55,115)( 23, 85,120, 93,145,108, 68, 75,125,156, 97,106, 40, 37
         ) ];

autGroups[13]:=
[ (  1, 34, 33)(  2,  6, 11)(  3, 22, 18)(  4, 20, 53)(  5, 59, 52)
        (  7, 62, 81)(  8, 70, 89)(  9, 93, 55)( 10, 13, 36)( 12, 75,111)
        ( 14,106, 61)( 15, 91, 83)( 16, 64, 29)( 17, 39, 90)( 19,119, 95)
        ( 21, 72,114)( 23,100,134)( 24,137, 35)( 25,117,131)( 26, 57, 40)
        ( 27, 44,136)( 28, 84, 47)( 30, 96, 48)( 31, 76, 77)( 32, 49, 68)
        ( 37, 42, 88)( 38,139,145)( 41,126,103)( 43,146, 63)( 45, 80, 86)
        ( 46, 97, 58)( 50,109,108)( 51, 82,104)( 54,129, 79)( 56,122, 85)
        ( 60,130,153)( 65,124,112)( 66,149,116)( 67,115, 73)( 69,152,121)
        ( 71,135,144)( 74,102,147)( 78,132, 98)( 87,128,107)( 92,101,133)
        ( 94,123,141)( 99,142,143)(105,156,138)(110,113,127)(118,125,151)
        (120,150,155)(140,154,148), (  1,151)(  2,146)(  3,133)(  4,156)
        (  5,144)(  6,114)(  7,154)(  8,128)(  9, 79)( 10,152)( 11,137)
        ( 12,142)( 13, 97)( 14, 86)( 15,155)( 16,110)( 17,149)( 18,119)
        ( 19,121)( 20, 57)( 21,113)( 22, 89)( 23,150)( 24,131)( 25,148)
        ( 26,153)( 27,109)( 28, 95)( 29,129)( 30, 41)( 32,117)( 33, 59)
        ( 34,111)( 35, 76)( 36,136)( 37,102)( 38,140)( 40,126)( 42,141)
        ( 43,116)( 44, 83)( 45,143)( 46,132)( 48,135)( 49, 67)( 50, 87)
        ( 51,139)( 52, 93)( 53,106)( 54, 63)( 55, 84)( 56, 99)( 60,105)
        ( 61, 64)( 62, 88)( 65,123)( 66,120)( 68,147)( 69,138)( 70, 81)
        ( 71,118)( 72,145)( 73,122)( 74, 78)( 75,134)( 85,101)( 90,115)
        ( 94,130)( 96,103)( 98,125)(100,112)(108,127) ];

autGroups[14]:=
[ (  1, 77, 78)(  2, 28, 98)(  3,114, 55)(  4, 54, 65)(  5, 84, 29)
        (  6, 85,134)(  7, 51,111)(  8,115,106)(  9,124, 18)( 10, 19, 37)
        ( 11, 47,130)( 12, 32, 43)( 13,132,109)( 14,117, 86)( 15,133, 62)
        ( 16,145, 52)( 17, 69, 23)( 20, 45,116)( 21, 81, 76)( 22,118,143)
        ( 24, 40, 48)( 25, 83, 91)( 26, 90, 94)( 27,148, 87)( 30, 95,139)
        ( 31,147, 33)( 34, 68,153)( 35,129,120)( 36,125, 75)( 38, 73,137)
        ( 39, 57, 71)( 41,144, 63)( 42,123, 96)( 44,101,119)( 46,122, 60)
        ( 49, 61,100)( 50,151, 89)( 53, 80, 72)( 56,135,121)( 58,126, 70)
        ( 59,136,146)( 64,149, 97)( 66,128,104)( 67,131, 92)( 74,154,105)
        ( 79,110,141)( 82,113,127)( 88,150,112)( 93,156,142)( 99,140,152)
        (102,107,138)(103,108,155), 
      (  1,152,139,  9, 97,129, 80, 87, 74, 23,128,135)(  2,137,109, 28, 75,
          81, 82,106,123, 37, 73,151)(  3,104,100, 47,105, 41, 60,120,110,
          12,140,141)(  4,155, 52, 19,143,115,113, 86, 36, 65, 83,145)
        (  5,142,148, 31, 63, 64, 94, 49, 95,  7,156,121)(  6, 91, 71, 85,
          42, 67,112, 76,138, 40,108,102)(  8, 99,154, 16, 35, 66,125, 30,
          27, 38,149, 56)( 10,122, 55, 54,130, 32,127, 29, 90, 98, 51,147)
        ( 11,111,116,114, 26,126, 33, 43,153, 84, 46,119)( 13, 61, 25,118,
          79, 14,103,144, 50, 21, 93, 96)( 15,136, 89, 24,146,117, 88, 92,
          22, 62, 39,132)( 17,150, 18, 69,134, 77, 72, 48, 53, 78,133,124)
        ( 20, 57, 45, 68,107, 44, 70,131,101, 58, 59, 34) ];

autGroups[15]:=
[ ( 1,46,28, 9,15,49,53,18,45)( 2,37,17,56,55, 8,10,41,36)( 3,50,48,22,44,
         27,13,16,23)( 4,12,40, 6,35,34,38,47,32)( 5,43,54,39,14,26,19,52,31)
        ( 7,24,21,11,25,42,29,30,51), 
      ( 1,54,21,33,42,14,15,18,37,55,35,22,20,13,38,36,17,53)( 2, 5,32,45,29,
         6, 3,47,24,49,40,19,56,16,43,51,26,48)( 4, 7,11,28,25,30,12,41,46,
         52,50,44,10,27,23,31, 9, 8)(34,39) ];

autGroups[16]:=
[ ( 2, 3, 5, 8,11,14,17,20,19,16,13,10, 7, 4)( 6, 9)(12,15)(18,21), 
      ( 1, 2, 3)( 4, 6, 5)( 7, 8, 9)(10,12,11)(13,14,15)(16,18,17)(19,20,21),
      ( 2, 3)( 4, 5)( 7, 8)(10,11)(13,14)(16,17)(19,20) ];

autGroups[17]:=
[ ( 2, 3, 5, 8,14,12, 7, 4)( 6, 9,15,24,36,26,16,10)(11,17,27,40,56,41,28,18
         )(13,20,30,43,60,45,31,21)(19,22,32,46,65,59,42,29)(23,33,25,37,51,
         67,47,34)(35,48,68,54,39,53,69,49)(38,50,70,62,44,61,72,52)
        (55,73,81,63,80,79,83,74)(57,75,82,64,78,58,77,76)(66,84,88,89,86,71,
         87,85), ( 1, 2, 3)( 4, 6, 5)( 7,11, 9)( 8,10,13)(12,19,17)(14,21,22)
        (15,18,23)(16,25,20)(24,34,35)(26,38,37)(27,29,39)(28,30,33)
        (31,44,32)(36,49,50)(40,54,55)(41,57,43)(42,58,53)(45,63,61)
        (46,62,64)(47,66,48)(51,52,71)(56,74,75)(59,79,77)(60,76,80)
        (65,82,83)(67,86,84)(68,85,73)(69,78,70)(72,81,87)(88,89,90), 
      ( 2, 3)( 4, 5)( 7, 8)( 9,10)(11,13)(12,14)(15,16)(17,21)(18,20)(19,22)
        (23,25)(24,26)(27,31)(28,30)(29,32)(34,37)(35,38)(39,44)(40,45)
        (41,43)(42,46)(47,51)(48,52)(49,50)(53,62)(54,61)(55,63)(56,60)
        (58,64)(59,65)(66,71)(68,72)(69,70)(73,81)(74,80)(75,76)(77,82)
        (79,83)(84,86)(85,87)(88,89) ];

autGroups[18]:=
[ ( 1,35,54,87,86,41,59,11,68,90,38,62)( 2,57,28,91,81, 8,43,63,95,84,60,37)
        ( 3,13,44,78,89,24,45,39,94,92,33, 9)( 4,83,16,55,72,29,71,31,76,49,
         18,79)( 5,22,23,53,36,70,64,19,85,50,74,46)( 6,27,61,69,75,14,77,80,
         51,65,20,17)( 7,96,56,30,26,67,34,42,47,93,10,48)(12,88,82,58,66,73,
         21,25,32,52,40,15), 
      ( 1,46,73,47,69,45,57,52)( 2,62,50,18,89,39,83,61)( 3, 9,20,60,70,77,
         35,22)( 4,81,80,76,78,19,48,90)( 5,33,41,66,88,43,13,17)
        ( 6,37,32,38,24,85,96,12)( 7,40,64,68,91,14,25,93)( 8,65,67,26,23,59,
         27,74)(10,75,94,55,36,11,79,28)(15,92,29,86,63,71,44,21)
        (16,56,82,95,87,58,42,49)(30,84,31,72,54,34,53,51) ];

autGroups[19]:=
[ ( 2, 3, 5, 8,11,14,17,20,23,22,19,16,13,10, 7, 4)( 6, 9)(12,15)(18,21), 
      ( 1, 2, 3)( 4, 6, 5)( 7, 8, 9)(10,12,11)(13,14,15)(16,18,17)(19,20,21)
        (22,24,23), ( 2, 3)( 4, 5)( 7, 8)(10,11)(13,14)(16,17)(19,20)(22,23) ];

autGroups[20]:=
[ (  2,  3,  5,  8, 14, 12,  7,  4)(  6,  9, 15, 24, 36, 26, 16, 10)
        ( 11, 17, 27, 41, 57, 43, 28, 18)( 13, 20, 30, 46, 66, 48, 31, 21)
        ( 19, 22, 32, 49, 71, 62, 44, 29)( 23, 33, 50, 53, 38, 25, 37, 34)
        ( 35, 39, 52, 74, 92, 91, 73, 51)( 40, 54, 76, 87, 68, 47, 67, 55)
        ( 42, 58, 79, 84, 64, 45, 63, 59)( 56, 69, 86, 98,112,106, 93, 77)
        ( 60, 78, 94,107, 99, 88, 70, 80)( 61, 81, 95,109, 97, 85, 65, 82)
        ( 72, 75)( 83, 89,100,114,125,122,110, 96)( 90,101,115,108,121,113,
         116,102)(103,104,118,105,119,123,111,117)(120,124), 
      (  1,  2,  3)(  4,  6,  5)(  7, 11,  9)(  8, 10, 13)( 12, 19, 17)
        ( 14, 21, 22)( 15, 18, 23)( 16, 25, 20)( 24, 34, 35)( 26, 39, 37)
        ( 27, 29, 40)( 28, 42, 33)( 30, 38, 45)( 31, 47, 32)( 36, 51, 52)
        ( 41, 55, 56)( 43, 60, 58)( 44, 61, 54)( 46, 64, 65)( 48, 69, 67)
        ( 49, 68, 70)( 50, 59, 72)( 53, 75, 63)( 57, 77, 78)( 62, 83, 81)
        ( 66, 85, 86)( 71, 88, 89)( 73, 90, 74)( 76, 82, 84)( 79, 80, 87)
        ( 91,103,101)( 92,102,104)( 93,105, 94)( 95, 96,108)( 97,111, 98)
        ( 99,113,100)(106,120,119)(107,118,116)(109,115,117)(110,114,121)
        (112,123,124)(122,126,125), (  2,  3)(  4,  5)(  7,  8)(  9, 10)
        ( 11, 13)( 12, 14)( 15, 16)( 17, 21)( 18, 20)( 19, 22)( 23, 25)
        ( 24, 26)( 27, 31)( 28, 30)( 29, 32)( 33, 38)( 34, 37)( 35, 39)
        ( 40, 47)( 41, 48)( 42, 45)( 43, 46)( 44, 49)( 50, 53)( 51, 52)
        ( 54, 68)( 55, 67)( 56, 69)( 57, 66)( 58, 64)( 59, 63)( 60, 65)
        ( 61, 70)( 62, 71)( 72, 75)( 73, 74)( 76, 87)( 77, 86)( 78, 85)
        ( 79, 84)( 80, 82)( 81, 88)( 83, 89)( 91, 92)( 93, 98)( 94, 97)
        ( 95, 99)( 96,100)(101,102)(103,104)(105,111)(106,112)(107,109)
        (108,113)(110,114)(115,116)(117,118)(119,123)(120,124)(122,125) ];

autGroups[21]:=
[ (  1, 36)(  2, 54)(  3, 53)(  4,106)(  5,103)(  6,121)(  7, 61)(  8, 70)
        (  9, 87)( 10, 95)( 11, 88)( 12, 14)( 13, 94)( 15, 63)( 16, 67)
        ( 17, 21)( 18, 50)( 19, 22)( 20, 56)( 23, 76)( 24, 66)( 25, 84)
        ( 26, 75)( 27,107)( 28, 97)( 29,109)( 30, 91)( 31,101)( 32,102)
        ( 33, 96)( 34, 90)( 35, 92)( 37, 98)( 38, 86)( 39, 99)( 40, 85)
        ( 41, 58)( 42,117)( 43, 46)( 44, 60)( 45,110)( 47, 77)( 48, 72)
        ( 49, 73)( 51,113)( 52,100)( 55,115)( 57,108)( 59, 65)( 62, 68)
        ( 64, 69)( 71, 74)( 78, 89)( 79,125)( 80,116)( 81,112)( 82, 93)
        ( 83,124)(104,126)(105,111)(114,118)(119,122)(120,123), 
      (  1, 59, 87,122, 32, 50, 24, 99, 16, 38, 19,101,111, 57)
        (  2, 34, 98, 55,  5, 30, 41,124, 52,108, 39,116, 31, 18)
        (  3,104, 74,110, 20, 93, 83, 66, 51, 63, 11,105, 90, 82)
        (  4, 10, 21, 68, 27, 15,  8, 49, 96,120, 64,109, 80, 29)
        (  6, 37,126, 23,  9,118, 45, 58, 78,115, 73, 47, 62, 91)
        (  7, 72, 53,113, 81, 86, 14, 88, 56, 79, 92,106, 77, 69)
        ( 12,117,103, 42,112, 95, 46, 22, 28, 60, 94, 67, 85,123)
        ( 13, 71, 97, 76, 33, 43,119, 70, 25,107, 65, 40, 17,114)
        ( 26,125, 36, 84,102, 48, 75, 54, 89, 35, 44, 61,121,100) ];

autGroups[22]:=
[ ( 2, 3, 5, 8,14,23,20,12, 7, 4)( 6, 9,15,24,35,50,36,25,16,10)
        (11,17,26,38,55,43,30,21,13,18)(19,27,39,45,32,22,31,44,40,28)
        (29,33,46,61,59,69,68,54,57,41)(34,47,62,53,37,52,42,58,63,48)
        (49,64,60,56,67,51,66,71,70,65), 
      ( 1, 2, 3)( 4, 6, 5)( 7,11, 9)( 8,10,13)(12,19,17)(14,21,22)(15,18,16)
        (20,29,27)(23,32,33)(24,25,34)(26,28,37)(30,42,31)(35,48,49)
        (36,51,47)(38,53,54)(39,41,56)(40,44,52)(43,59,58)(45,60,46)
        (50,65,66)(55,68,69)(57,62,67)(61,64,63)(70,72,71), 
      ( 2, 3)( 4, 5)( 7, 8)( 9,10)(11,13)(12,14)(15,16)(17,21)(19,22)(20,23)
        (24,25)(26,30)(27,32)(28,31)(29,33)(35,36)(37,42)(38,43)(39,45)
        (40,44)(41,46)(47,48)(49,51)(53,58)(54,59)(56,60)(57,61)(62,63)
        (64,67)(65,66)(68,69)(70,71) ];

autGroups[23]:=
[ ( 2, 3, 5, 8,14,24,37,32,20,12, 7, 4)( 6, 9,15,26,16,10)(11,17,28,18)
        (13,21,33,22)(19,29,30)(23,34,35)(25,38,31,42,44,36,41,27,40,47,46,39
         )(43,45), ( 1, 2, 3)( 4, 6, 5)( 7,11, 9)( 8,10,13)(12,19,17)
        (14,22,23)(15,18,25)(16,27,21)(20,31,29)(24,35,36)(26,39,40)
        (28,30,38)(32,43,42)(33,41,34)(37,44,45)(46,48,47), 
      ( 2, 3)( 4, 5)( 7, 8)( 9,10)(11,13)(12,14)(15,16)(17,22)(18,21)(19,23)
        (20,24)(25,27)(28,33)(29,35)(30,34)(31,36)(32,37)(38,41)(39,40)
        (42,44)(43,45)(46,47) ];

autGroups[24]:=
[ ( 1,32,15,28,23)( 2,11,61,25,13,24,18,44,68,22)( 3,20,63,34,41,14, 4,45,
         29,58)( 5,73,38,12,39, 8,51,59, 7,71)( 6,60,50,52,36,42,10,19,48,37)
        ( 9,54,74,65,16,26,49,75,31,46)(17,21,67,53,64,30,35,55,72,66)
        (27,69,47,56,70,43,57,62,40,33), 
      ( 1,63,27,75,12,18,48,55,64,52,22, 5,74,56,58)( 2,15,62,57,28,24,39, 6,
         59,17,10,29,70,16,25,14, 9,53,72,46, 4,61,26,33,34,42,21,51,60,71)
        ( 3,38,31,66, 7,50,30,47,65,43,40,49,69,35,37, 8,67,54,73,20,32,11,
         45,19,44,68,36,41,13,23) ];

autGroups[25]:=
[ ( 1,26,30,24, 8,48,34,14,22,53)( 2,60, 7,32,25,38,35,39,40,41)
        ( 3,54,47,46,58)( 4, 6,16,27,51,59,23,49,10,28)( 5,44,19,42,52,31,20,
         50,56,21)( 9,57,55,33,45,17,18,36,29,11)(12,43,13,37,15), 
      ( 1,57, 3,44, 9,29,21,15,36, 8,40)( 2,54,17,41,39,33,37,35,28,14,59)
        ( 4,18,31,55,51,46,24,27, 6,26,43)( 7,13,48,19,16,56,53,47,25,22,34)
        (10,20,52,23,12,42,38,11,60,50,58) ];

autGroups[26]:=
[ ( 2, 3, 5, 8,14,23,31,27,19,12, 7, 4)( 6, 9,15,10)(11,16,21,13,20,28,36,
         46,42,33,24,17)(18,25,29,22)(26,34,43,45,51,53,52,50,41,47,37,30)
        (32,39,49,40)(35,38,48,44), ( 1, 2, 3)( 4, 6, 5)( 7,11, 9)( 8,10,13)
        (12,18,16)(14,21,22)(15,17,20)(19,26,25)(23,29,30)(24,32,28)
        (27,35,34)(31,37,38)(33,41,39)(36,40,45)(42,48,47)(43,44,46)
        (49,50,51)(52,54,53), ( 2, 3)( 4, 5)( 7, 8)( 9,10)(11,13)(12,14)
        (16,21)(17,20)(18,22)(19,23)(24,28)(25,29)(26,30)(27,31)(33,36)
        (34,37)(35,38)(39,40)(41,45)(42,46)(43,47)(44,48)(50,51)(52,53) ];

autGroups[27]:=
[ ( 2, 3, 5, 8,14,24,36,32,20,12, 7, 4)( 6, 9,15,26,16,10)(11,17,28,42,48,
         34,22,13,21,33,29,18)(19,30,23,35,44,31)(25,37,40,27,39,50,47,53,51,
         41,49,38)(43,46)(52,54), ( 1, 2, 3)( 4, 6, 5)( 7,11, 9)( 8,10,13)
        (12,19,17)(14,22,23)(15,18,25)(16,27,21)(20,24,30)(26,38,39)
        (28,31,41)(29,43,37)(32,45,36)(33,40,46)(34,47,35)(42,51,52)
        (44,50,49)(48,54,53), ( 2, 3)( 4, 5)( 7, 8)( 9,10)(11,13)(12,14)
        (15,16)(17,22)(18,21)(19,23)(20,24)(25,27)(28,34)(29,33)(31,35)
        (32,36)(37,40)(38,39)(41,47)(42,48)(43,46)(49,50)(51,53)(52,54) ];

autGroups[28]:=
[ ( 1, 7,18,11,21,25)( 2,24,17,15,19, 6)( 3,20,16)( 4,14,22)( 5,27,23, 9,13,
         12)( 8,26,10), ( 1,20, 9,14,15, 8,21, 3,27, 4,24,10,18,16,12,22, 6,
         26)( 2, 7)( 5,13)(11,19)(17,25) ];

autGroups[29]:=
[ (  2,  3,  5,  8, 14, 12,  7,  4)(  6,  9, 15, 24, 36, 26, 16, 10)
        ( 11, 17, 27, 41, 61, 43, 28, 18)( 13, 20, 30, 46, 70, 48, 31, 21)
        ( 19, 22, 32, 49, 75, 66, 44, 29)( 23, 33, 50, 77,111, 79, 51, 34)
        ( 25, 37, 55, 84,123, 86, 56, 38)( 35, 52, 80,116,153,118, 81, 53)
        ( 39, 54, 82,119,155,128, 87, 57)( 40, 58, 88,130,166,132, 89, 59)
        ( 42, 62, 93, 63)( 45, 67, 98, 68)( 47, 71,102,140,176,141,103, 72)
        ( 60, 90,133,110,147,164,129, 91)( 64, 92, 96, 65, 95,135,134, 94)
        ( 69, 99,137,143,106, 74,105,100)( 73,101,138,173,160,126,142,104)
        ( 76,108,145,151,115,150,146,109)( 78,112,121, 83,120,156,149,113)
        ( 85,124,158,162,127,161,159,125)( 97,107,144,179,191,188,171,136)
        (114,148,131,167,174,139,157,122)(117,154)(152,169,170,187,178,172,
         163,182)(165,180,181,183,184,177,190,185)(168,186,189,175), 
      (  1,  2,  3)(  4,  6,  5)(  7, 11,  9)(  8, 10, 13)( 12, 19, 17)
        ( 14, 21, 22)( 15, 18, 23)( 16, 25, 20)( 24, 34, 35)( 26, 39, 37)
        ( 27, 29, 40)( 28, 42, 33)( 30, 38, 45)( 31, 47, 32)( 36, 53, 54)
        ( 41, 59, 60)( 43, 64, 62)( 44, 65, 58)( 46, 68, 69)( 48, 73, 71)
        ( 49, 72, 74)( 50, 63, 76)( 51, 78, 52)( 55, 57, 83)( 56, 85, 67)
        ( 61, 91, 92)( 66, 97, 95)( 70,100,101)( 75,106,107)( 77,109,110)
        ( 79,114,112)( 80,113,115)( 81,117, 82)( 84,121,122)( 86,126,124)
        ( 87,127,120)( 88, 96,129)( 89,131, 90)( 93, 94,108)( 98,125, 99)
        (102,104,139)(103,138,105)(111,133,148)(116,151,152)(118,119,154)
        (123,157,142)(128,163,161)(130,164,165)(132,168,167)(134,169,145)
        (135,136,170)(137,159,172)(140,174,175)(141,177,173)(143,178,144)
        (146,180,147)(149,181,150)(153,182,155)(156,162,183)(158,160,184)
        (166,185,186)(171,179,187)(176,189,190)(188,192,191), 
      (  2,  3)(  4,  5)(  7,  8)(  9, 10)( 11, 13)( 12, 14)( 15, 16)
        ( 17, 21)( 18, 20)( 19, 22)( 23, 25)( 24, 26)( 27, 31)( 28, 30)
        ( 29, 32)( 33, 38)( 34, 37)( 35, 39)( 40, 47)( 41, 48)( 42, 45)
        ( 43, 46)( 44, 49)( 50, 56)( 51, 55)( 52, 57)( 53, 54)( 58, 72)
        ( 59, 71)( 60, 73)( 61, 70)( 62, 68)( 63, 67)( 64, 69)( 65, 74)
        ( 66, 75)( 76, 85)( 77, 86)( 78, 83)( 79, 84)( 80, 87)( 81, 82)
        ( 88,103)( 89,102)( 90,104)( 91,101)( 92,100)( 93, 98)( 94, 99)
        ( 95,106)( 96,105)( 97,107)(108,125)(109,124)(110,126)(111,123)
        (112,121)(113,120)(114,122)(115,127)(116,128)(118,119)(129,138)
        (130,141)(131,139)(132,140)(133,142)(134,137)(135,143)(136,144)
        (145,159)(146,158)(147,160)(148,157)(149,156)(150,162)(151,161)
        (152,163)(153,155)(164,173)(165,177)(166,176)(167,174)(168,175)
        (169,172)(170,178)(171,179)(180,184)(181,183)(185,190)(186,189)
        (188,191) ];

autGroups[30]:=
[ ( 1, 4,49,29,16,30,45,27)( 2,35,46,44,32,15,33,60)( 3,10, 8,25, 5,17,42, 9
         )( 6,58,59,54,34,40,12,36)( 7,37,48,53,50,41,64,21)(11,56,14,39,43,
         52,24,28)(13,18,22,26,55,63,31,62)(19,47,20,51,38,57,61,23), 
      ( 1,17,62,53,59,12)( 2,30,64,32, 8,23,39,14,11,28,57,22)
        ( 3, 9,54,48, 6, 4,40,63,55,43,61,24)( 5,45,60,29,49,36,58,37, 7,15,
         41,13)(10,52,26,38,46,19)(16,33,21)(18,50,31,34,42,27,25,44,51,20,
         47,35) ];

autGroups[31]:=
[ ( 1,43,45)( 2,23,61,24,54,27,56,41,22,20,37,64)( 3, 7,58,26, 8,28,55,40,
         30,19,32,34)( 4,31,60,36, 6,52,63, 9,18,29,49,21)( 5,42,15,38,33,57,
         44,62,59,11,39,25)(10,51,53)(13,35,14)(16,48,47)(17,46,50), 
      ( 1,63,12,36,35, 4,10,29)( 2,20,31,49,34,30,57,15)( 3,54,52, 5,41,19,
         38,60)( 6,56,58, 9,59,28,24,25)( 7,17,32,14,40,45, 8,50)
        (11,44,55,26,21,18,23,37)(13,64,43,22,48,27,16,61)(33,53,42,46,39,51,
         62,47) ];

autGroups[32]:=
[ ( 1, 3)( 4, 6)( 8, 9)(10,12)(14,15)(16,18)(20,21)(22,24)(26,27)(28,30), 
      ( 1,13,21, 5,24,23, 6,19,15, 2,30,17,12,25, 9, 7,27,11,18,29)( 3,14)
        ( 4,20)(10,26)(16,28) ];

autGroups[33]:=
[ ( 1,58,24, 9,34,32,56,36,28,10,20,54)( 2,21,19, 5,40,45,37, 4,23,18, 3,67)
        ( 6,51,11,35,38,12,70,14,39,31,13,50)( 7,53,16,17,68,30,65,22,15,57,
          8,71)(25,61,27,49,59,33,64,44,60,29,55,52)(26,62,48,46,66,47,72,41,
         69,43,42,63), ( 1,59)( 2,22,36,11)( 3,27,32,41)( 4,57,24,38)
        ( 5,69,20,49)( 6,30)( 7,64,14,63)( 8,33,12,42)( 9,61)(10,19)
        (13,25,17,47)(15,35)(16,71)(18,54,34,40)(21,51,28,65)(23,72)(26,44)
        (29,60,48,62)(31,70)(37,50,58,68)(39,52,53,66)(43,45)(46,55)(56,67) ];

autGroups[34]:=
[ ( 2, 3, 5, 8,14,21,28,37,42,35,27,19,12, 7, 4)( 6, 9,15,23,30,39,44,46,47,
         45,40,31,25,16,10)(11,17,13,20,18)(22,24,29)(26,32,33)
        (34,41,38,43,36), ( 1, 2, 3)( 4, 6, 5)( 7,11, 9)( 8,10,13)(12,14,17)
        (15,18,22)(16,24,20)(19,26,21)(23,29,25)(27,34,32)(28,33,36)
        (30,31,38)(35,39,41)(37,43,40)(42,45,44)(46,47,48), 
      ( 2, 3)( 4, 5)( 7, 8)( 9,10)(11,13)(12,14)(15,16)(18,20)(19,21)(22,24)
        (23,25)(27,28)(30,31)(32,33)(34,36)(35,37)(39,40)(41,43)(44,45)
        (46,47) ];

autGroups[35]:=
[ ( 2, 3, 5, 8,14,23,31,38,44,42,35,27,19,12, 7, 4)( 6, 9,15,10)
        (11,16,24,32,39,45,47,43,36,29,21,13,20,28,25,17)(18,26)(22,30)
        (34,40,37,41), ( 1, 2, 3)( 4, 6, 5)( 7,11, 9)( 8,10,13)(12,18,16)
        (14,21,22)(15,17,20)(19,24,26)(23,30,29)(25,33,28)(27,34,32)
        (31,36,37)(35,38,40)(39,41,43)(42,46,44)(45,47,48), 
      ( 2, 3)( 4, 5)( 7, 8)( 9,10)(11,13)(12,14)(16,21)(17,20)(18,22)(19,23)
        (24,29)(25,28)(26,30)(27,31)(32,36)(34,37)(35,38)(39,43)(42,44)
        (45,47) ];

autGroups[36]:=
[ ( 1, 6,22,10)( 2,12,34,40,14, 3,17,27,47,23,13, 4,19,45,46, 8)
        ( 5,11,28,26,31,20,30,24,36,38,33,21, 7,42,16,37)( 9,32,44,29)
        (15,18,39,41)(25,35,48,43), ( 1,37,16, 6, 8, 7,32, 3,19,15,11,34)
        ( 2,35, 4,17,25,12,28, 9,42, 5,18,21)(10,40,46,22,23,33,39,20,36,29,
         26,47)(13,48,24,30,43,27,31,41,45,14,44,38) ];

autGroups[37]:=
[ (  1, 36,121, 80,  6, 60,144,168, 51, 31,134,142, 87, 95,166,129, 84, 15,
         130, 49, 40)(  2, 89,115,155, 24, 18,128,163,123,127, 99, 79, 52,
          57,165, 91, 42,  8,150,157,141)(  3, 43,118,161, 13, 63,147,167,
          86, 98,131,156,124, 28,154,104, 27,  5, 72,148, 54)
        (  4, 65, 81,151, 44, 94, 90,143, 59,159, 29, 82, 14,122,152,120, 11,
         33,146,132, 88)(  7,  9, 20, 78, 73, 23, 26,114,113,102, 96, 50,103,
         160,158,153,108, 56, 48, 93, 69)( 10,106,116, 76, 58,105, 37,162,
          62,109, 64,149, 25,111, 97, 83, 19, 55,125,112,126)
        ( 12, 21,107, 38, 53, 30, 85,117,164, 67,140,135,101,138,137, 47, 35,
         92, 66, 77, 17)( 16,100, 39, 70, 45,133, 46,145, 32,119, 61, 34, 68,
         41, 71, 74, 75,110,136, 22,139), 
      (  1, 87, 92, 88, 82,109)(  2,158, 93, 96, 79,127)(  3,151, 72, 61,142,
         145)(  4,101, 15, 51,149,159)(  5,123,107,139, 50, 83)
        (  6, 71,154, 29,156,110)(  7,103, 56, 24, 42,165)(  8, 86,105, 69,
          34, 85)(  9, 76, 20, 52,135,141)( 10, 28, 59,164,119, 31)
        ( 11,152, 77,134,129, 58)( 12, 14,124,126, 40,100)( 13, 70,130, 16,
         167, 33)( 17,155, 43, 25,160,136)( 18, 66, 57,113, 55,102)
        ( 19,166, 49,140, 90, 44)( 21, 75,121,162,147,120)( 22, 98)
        ( 23, 78,150,115,163,114)( 26, 45,138, 99,104,112)( 27, 68)
        ( 30, 80, 36,125, 81,143)( 32,137,132, 63, 37, 95)( 35, 46, 73,106,
         131, 91)( 38,122,146,116,168, 60)( 39, 84, 97, 54, 65, 47)
        ( 41, 48, 62,161, 89, 67)( 53,128,153, 64,108,157)( 74,117, 94,148,
         111,144)(118,133) ];

autGroups[38]:=
[ (  1, 94, 18,125,  8,119,120,115, 14,107, 56, 49)(  2, 30, 67, 57, 76, 98,
         97, 58, 71, 99, 28, 90)(  3, 65, 63,112, 48, 17,129, 38, 42, 78, 43,
         116)(  4, 82, 32,106, 85,101,109, 81, 84, 11,  7, 27)
        (  5, 96, 92, 50, 26, 22,114, 89, 24,132, 73,104)(  6, 16, 46, 52,
          47,103, 59,130, 25,110, 45, 77)(  9,105, 23, 41, 40, 72,123, 31,
          79, 95, 68, 21)( 10,126, 34,124, 86,131, 19, 36, 55, 35, 12, 70)
        ( 13, 69,111,127, 75, 66, 74, 33, 61, 39, 80,100)( 15,121, 53,102,
          54, 91, 83, 60, 20,118, 44,122)( 29, 51,128,117, 87, 62, 37, 88,
          64, 93,113,108), (  1,102,103, 82, 90,130, 63, 56, 81, 34)
        (  2,117, 17, 47, 62,124, 40, 83,107,123)(  3, 44, 66,106, 26,121,
          78,114, 11,127)(  4, 50, 98, 76, 74,109,111, 55, 36, 89)
        (  5, 64, 16,125, 70,100, 95,120, 45, 71)(  6, 91, 31, 10, 87,119,
          48, 37, 99, 72)(  7,115, 32, 12,131, 85, 46, 84, 58, 67)
        (  8, 20,110,126, 27, 77, 38, 18, 28,101)(  9, 61, 60, 57, 41,108,
         132,129, 19,128)( 13,113, 79, 97, 54, 96, 21, 88, 35,112)
        ( 14, 51, 39, 30, 49, 52, 23, 73, 86, 59)( 15, 24, 69,122, 94,116,
          33, 92, 43, 25)( 22, 75)( 29, 93, 65,104,118,105, 68, 53, 80, 42) 
     ];

autGroups[39]:=
[ (  1, 36, 61, 64, 43, 63, 62, 53,105, 35, 27,132,112, 44, 42, 73, 83, 52,
          96, 23, 84, 85)(  2,117, 99, 18, 65, 51, 50,122, 57,  6,124,130,
          60, 19,102, 97, 90, 88, 13, 39,126, 24)(  3,106,119, 20, 33, 31,
         118, 67, 45, 25, 81,131, 21, 11,123,108,109,  9, 70,104, 79,113)
        (  4, 56, 86, 32,121,114, 74,100,120, 69, 91,128, 40, 68, 76, 14, 22,
         75, 41, 10, 72, 30)(  5, 46,116,110,101, 17, 71, 48, 34, 82, 37,129,
         89, 12, 80, 15,115, 38, 92, 94, 28,111)(  7, 16, 29, 87,107, 77,127,
         78, 93, 54, 55,125,103, 98, 66, 47, 26,  8, 58, 59, 95, 49), 
      (  1, 49, 98, 52)(  2, 89, 96, 79)(  3, 85, 92, 90)(  4, 94,128, 34)
        (  5, 30,129, 91)(  6,119, 70,122)(  7,101,103,  8)(  9, 24, 95, 68)
        ( 10, 66, 88, 20)( 11, 67, 13,126)( 12, 43, 40, 75)( 14, 72, 44, 38)
        ( 15, 16,127,125)( 17,113,110, 22)( 18, 23, 76, 45)( 19, 21, 41, 84)
        ( 25,130, 46, 53)( 26, 63,109, 37)( 27, 48, 51,131)( 28, 33,112, 58)
        ( 29, 62,132, 55)( 31,111, 77, 83)( 32,104,102,105)( 35, 99,106,100)
        ( 36, 78, 82,108)( 39, 42, 57, 64)( 47, 65, 60, 59)( 50,123,120, 87)
        ( 54, 86,118,124)( 56, 74,116, 73)( 61, 71,121, 69)( 80,114,115, 81)
        ( 93,107, 97,117) ];

autGroups[40]:=
[ ( 1,14,24,28,21, 4, 6,20,30,22,15, 3,12,26,33,16, 9, 8,18,32,27,10)( 2, 5)
        ( 7,11)(13,17)(19,23)(25,29), 
      ( 1,23,15,11,27, 2,30,13,18,25, 6,29, 9,17,21, 5,33, 7,24,19,12,31)
        ( 3,10,22,32,20, 8, 4,16,28,26,14) ];

autGroups[41]:=
[ (  1, 27, 83,124, 72,119, 97, 47)(  2, 28, 84,111, 65, 22, 44, 48)
    (  3, 25, 82,130, 74,113, 98, 49)(  4, 26, 64, 24, 46, 96, 99, 50)
    (  5, 30, 86,147,129, 75,100, 51)(  6, 29, 85, 73,118,166,101, 52)
    (  7, 31, 88,148,139,169,109, 58)(  8, 32, 87,149,168,102, 56, 12)
    (  9, 36, 13, 33, 89,150,105, 53)( 10, 37, 95,155,200,170,106, 54)
    ( 11, 38, 94,152,161,167,103, 55)( 14, 34, 90,151,201,173,110, 60)
    ( 15, 35, 59, 17, 39, 91,104, 57)( 16, 18, 40, 21, 43, 63, 23, 45)
    ( 19, 41, 92,153,202,172,108, 62)( 20, 42, 93,154,203,171,107, 61)
    ( 66, 77,121,163,207,225,179,123)( 67,112,174,218,194,143,132, 79)
    ( 68,114,175,220,247,227,180,125)( 69,115,176,183,128, 71,117,126)
    ( 70,116,178,221,248,228,182,127)( 76,120,177,219,226,181,131, 78)
    ( 80,122,162,206,191,140,133, 81)(134,184,229,208,156,165,193,142)
    (135,185,209,158,157,204,189,137)(136,159,146,188,231,234,195,145)
    (138,186,210,160,205,240,232,190)(141,187,230,255,246,258,233,192)
    (164,198,235,260,252,264,241,211)(196,212)(197,213,239,262,253,263,261,
     237)(199,236,259,257,265,242,214,238)(215,243,266,269,251,222,249,244)
    (216,224)(217,223,250,268,256,254,267,245), 
  (  1,  4,  5)(  2,  3,  6)(  7, 15,  9)(  8, 16, 10)( 11, 13, 17)
    ( 12, 14, 18)( 19, 22, 23)( 20, 21, 24)( 25, 48, 66)( 26, 47, 67)
    ( 27, 51, 68)( 28, 52, 69)( 29, 49, 70)( 30, 50, 71)( 31, 53, 73)
    ( 32, 54, 72)( 33, 55, 75)( 34, 56, 74)( 35, 58, 80)( 36, 57, 39)
    ( 37, 45, 65)( 38, 59, 81)( 40, 60, 46)( 41, 63, 78)( 42, 64, 79)
    ( 43, 61, 76)( 44, 62, 77)( 82,123,134)( 83,125,135)( 84,126,136)
    ( 85,127, 88)( 86,128,138)( 87,124,137)( 89,129,141)( 90,130,142)
    ( 91,104,144)( 92,131,139)( 93,132,140)( 94,133,143)( 95,111,145)
    ( 96,110,146)( 97,156,112)( 98,158,116)( 99,159,117)(100,103,114)
    (101,160,115)(102,157,113)(105,164,118)(106,165,119)(107,161,120)
    (108,162,121)(109,163,122)(147,190,187)(148,182,153)(149,189,196)
    (150,192,198)(151,193,170)(152,194,177)(154,191,197)(155,195,199)
    (166,211,205)(167,171,175)(168,212,204)(169,181,207)(172,213,206)
    (173,214,188)(174,208,215)(176,210,216)(178,209,217)(179,222,184)
    (180,223,185)(183,224,186)(200,238,201)(202,228,239)(203,237,220)
    (218,244,219)(221,245,246)(225,226,249)(227,252,250)(229,251,243)
    (230,232,253)(231,242,254)(233,257,235)(234,256,236)(240,241,263)
    (247,261,264)(248,255,262)(258,267,265)(259,268,260)(266,269,270), 
  (  1,  3)(  2,  4)(  5,  6)(  7, 11)(  8, 12)(  9, 13)( 10, 14)( 15, 17)
    ( 16, 18)( 19, 20)( 21, 23)( 22, 24)( 25, 47)( 26, 48)( 27, 49)( 28, 50)
    ( 29, 51)( 30, 52)( 31, 55)( 32, 56)( 33, 53)( 34, 54)( 35, 59)( 37, 60)
    ( 38, 58)( 39, 57)( 40, 45)( 41, 61)( 42, 62)( 43, 63)( 44, 64)( 46, 65)
    ( 66, 67)( 68, 70)( 69, 71)( 72, 74)( 73, 75)( 76, 78)( 77, 79)( 80, 81)
    ( 82, 97)( 83, 98)( 84, 99)( 85,100)( 86,101)( 87,102)( 88,103)( 89,105)
    ( 90,106)( 91,104)( 92,107)( 93,108)( 94,109)( 95,110)( 96,111)(112,123)
    (113,124)(114,127)(115,128)(116,125)(117,126)(118,129)(119,130)(120,131)
    (121,132)(122,133)(134,156)(135,158)(136,159)(137,157)(138,160)(139,161)
    (140,162)(141,164)(142,165)(143,163)(145,146)(147,166)(148,167)(149,168)
    (151,170)(152,169)(153,171)(154,172)(155,173)(174,179)(175,182)(176,183)
    (177,181)(178,180)(184,208)(185,209)(186,210)(187,211)(188,195)(189,204)
    (190,205)(191,206)(192,198)(194,207)(196,212)(197,213)(199,214)(200,201)
    (202,203)(215,222)(216,224)(217,223)(218,225)(219,226)(220,228)(221,227)
    (230,241)(231,234)(232,240)(233,235)(236,242)(237,239)(243,251)(244,249)
    (245,250)(246,252)(247,248)(253,263)(254,256)(255,264)(258,260)(259,265)
    (261,262)(266,269)(267,268) ];

autGroups[42]:=
[ (  1,108,199,  2,107,200)(  3,  8,190,  4,  7,189)(  5,173,198,  6,174,197)
    (  9, 28,141, 10, 27,142)( 11,128,134, 12,127,133)( 13,168, 23, 14,167,
      24)( 15,148, 20, 16,147, 19)( 17,165, 21, 18,166, 22)( 25,144,157, 26,
     143,158)( 29,139, 92, 30,140, 91)( 31, 42, 97, 32, 41, 98)
    ( 33, 37, 99, 34, 38,100)( 35, 40, 93, 36, 39, 94)( 43,115,170, 44,116,
     169)( 45,123,224, 46,124,223)( 47,125,226, 48,126,225)( 49,136,150, 50,
     135,149)( 51, 96,151, 52, 95,152)( 53,138,153, 54,137,154)
    ( 55, 76,187, 56, 75,188)( 57, 77,184, 58, 78,183)( 59, 73,185, 60, 74,
     186)( 61,234,159, 62,233,160)( 63,217,250, 64,218,249)( 65,236,263, 66,
     235,264)( 67, 70, 71, 68, 69, 72)( 79,240,101, 80,239,102)
    ( 81,254,103, 82,253,104)( 83,248,105, 84,247,106)( 85,192,179, 86,191,
     180)( 87,193,130, 88,194,129)( 89,196,181, 90,195,182)(109,260,131,110,
     259,132)(111,261,119,112,262,120)(113,146,212,114,145,211)
    (117,215,252,118,216,251)(121,208,270,122,207,269)(155,171,238,156,172,
     237)(161,202,246,162,201,245)(163,219,222,164,220,221)(175,266,232,176,
     265,231)(177,268,229,178,267,230)(203,256,213,204,255,214)
    (205,243,258,206,244,257)(209,228,241,210,227,242), 
  (  1,189, 59, 39,194, 73,235, 85,229,253)(  2,190, 60, 40,193, 74,236, 86,
     230,254)(  3,127,148,140, 70, 78,233, 88,182,239)(  4,128,147,139, 69,
      77,234, 87,181,240)(  5, 27,145, 51, 12, 76,218, 89,232,248)
    (  6, 28,146, 52, 11, 75,217, 90,231,247)(  7,109,215,178,257,246,149,
      29, 33, 49)(  8,110,216,177,258,245,150, 30, 34, 50)(  9, 81,126,175,
     263,222, 98, 25,169, 54)( 10, 82,125,176,264,221, 97, 26,170, 53)
    ( 13, 46,228,155,201,113, 63,251,106, 95)( 14, 45,227,156,202,114, 64,
     252,105, 96)( 15, 48,269,102,137, 16, 47,270,101,138)( 17, 44,268, 24,
     204,185,117,261,103,136)( 18, 43,267, 23,203,186,118,262,104,135)
    ( 19,243,184,122,259,131,197, 57, 38,196)( 20,244,183,121,260,132,198,
      58, 37,195)( 21,241,188,120,199, 55, 42,191,211,265)( 22,242,187,119,
     200, 56, 41,192,212,266)( 31,174,108,111,207,210,255,153,151, 36)
    ( 32,173,107,112,208,209,256,154,152, 35)( 61,226,161, 92, 80,123,206,
     160,220, 99)( 62,225,162, 91, 79,124,205,159,219,100)( 65,250,163, 93,
     172,223,214,157, 84,115)( 66,249,164, 94,171,224,213,158, 83,116)
    ( 67,180,238,133,129,167,141, 72,165,143)( 68,179,237,134,130,168,142,
      71,166,144) ];

autGroups[43]:=
[ (  1, 35)(  3, 13)(  4, 53)(  5, 99)(  6, 51)(  7,109)(  8, 63)(  9, 62)
    ( 10, 65)( 11, 54)( 12,144)( 14,134)( 15, 73)( 16, 55)( 17,119)( 18,130)
    ( 19,124)( 20,127)( 21,143)( 22, 44)( 23, 49)( 24, 61)( 25, 72)( 26, 38)
    ( 27, 56)( 28,115)( 29,139)( 30, 64)( 31, 39)( 32,121)( 33,131)( 34,133)
    ( 36, 43)( 37, 90)( 40,103)( 41, 52)( 42,113)( 45, 57)( 46, 74)( 47, 58)
    ( 48, 95)( 50, 79)( 59, 76)( 60, 98)( 66,116)( 67,106)( 68, 93)( 69,110)
    ( 70, 91)( 71,114)( 75, 86)( 77,142)( 78,128)( 80,138)( 81,135)( 82,111)
    ( 83,108)( 84,123)( 85,118)( 87,107)( 88,129)( 89,140)( 92, 94)( 96,136)
    ( 97,120)(100,104)(101,125)(102,137)(112,141)(117,122)(126,132), 
  (  1, 81, 93, 48,144,127, 15, 14, 55, 84,118, 73, 57, 64,  5, 10,135,121,
      71, 44)(  2,140,  9, 96,113,124, 77, 49, 25, 98,105, 26,129, 63, 86,
      19, 53,112,108, 79)(  3, 82, 30, 88, 52,143, 70, 16, 23,131,132, 91,
      95, 62,104, 28,111,114,141, 67)(  4, 66, 45, 94, 80, 38, 17,134, 29,
     103,142, 58, 68, 27, 40, 89,133, 32,109,138)(  6, 61,137, 87)
    (  7, 65, 76, 50,122, 56, 85, 90, 75,101,139, 20, 11, 60,125, 92, 35, 36,
     42,117)(  8, 12, 31, 69, 41, 72, 22, 39, 97,106,136, 99,128, 46,100, 83,
     123, 78,130, 33)( 13, 37,119, 74,102,115, 59,116,120, 24,126, 43, 34,
     110, 51, 21, 54, 47, 18,107) ];

autGroups[44]:=
[ ( 1,35,28,44,14,43,24,54,30,47,15,40)( 2,56,32,51,13,39,23,34,25,41,16,46)
    ( 3,57, 5,63, 4,62,11,61,12,58,17,59)( 6,33,29,48,10,37,21,55,27,45,20,49
     )( 7,53,26,42, 9,52,22,36,31,50,19,38)( 8,60,18,64)(65,77,94,66,78,89)
    (67,68,76)(69,75,81)(70,73,91,71,74,95)(79,96,88,80,92,87)
    (83,93,86,84,90,85), ( 1,41,65)( 2,48,85)( 3,34,95)( 4,53,92)( 5,35,89)
    ( 6,44,71)( 7,50,87)( 8,55,93)( 9,52,82)(10,46,68)(11,56,90)(12,54,96)
    (13,43,69)(14,49,75)(15,37,67)(16,40,76)(17,36,94)(18,33,91)(19,38,72)
    (20,39,81)(21,47,86)(22,42,66)(23,45,70)(24,51,88)(25,59,77)(26,63,78)
    (27,64,73)(28,57,74)(29,60,83)(30,61,84)(31,58,79)(32,62,80), 
  ( 1,41)( 2,47)( 3,57)( 4,58)( 5,59)( 6,45)( 7,50)( 8,60)( 9,52)(10,37)
    (11,61)(12,62)(13,43)(14,39)(15,46)(16,40)(17,63)(18,64)(19,38)(20,49)
    (21,48)(22,42)(23,44)(24,51)(25,35)(26,36)(27,33)(28,34)(29,55)(30,56)
    (31,53)(32,54)(67,68)(70,71)(73,91)(74,95)(75,81)(77,89)(78,94)(79,92)
    (80,96)(83,93)(84,90)(85,86) ];

autGroups[45]:=
[ ( 1,34,58,43,28,10, 7)( 2,35,59,44,29,11, 8)( 3,36,60,45,30,12, 9)
    ( 4,46,57,49,37,19,32,14,23,62,53,41,26,16)( 5,47,55,50,38,20,33,15,24,
     63,54,42,27,17)( 6,48,56,51,39,21,31,13,22,61,52,40,25,18)(64,67)(65,68)
    (66,69), ( 1,13, 4)( 2,14, 5)( 3,15, 6)( 7,40,22)( 8,41,23)( 9,42,24)
    (10,19,25)(11,20,26)(12,21,27)(16,38,34)(17,39,35)(18,37,36)(28,47,32)
    (29,48,33)(30,46,31)(43,64,55)(44,65,56)(45,66,57)(49,69,60)(50,67,58)
    (51,68,59)(52,70,61)(53,71,62)(54,72,63), 
  ( 1,10)( 2,11)( 3,12)( 4,19)( 5,20)( 6,21)(13,25)(14,26)(15,27)(16,32)
    (17,33)(18,31)(22,40)(23,41)(24,42)(28,34)(29,35)(30,36)(37,46)(38,47)
    (39,48)(43,58)(44,59)(45,60)(49,57)(50,55)(51,56)(52,61)(53,62)(54,63)
    (64,67)(65,68)(66,69) ];

autGroups[46]:=
[ ( 1,17,58,46,81,78,66,82,68,19,22, 9, 6,26)( 2,18,59,47,79,76,64,83,69,20,
     23, 7, 4,27)( 3,16,60,48,80,77,65,84,67,21,24, 8, 5,25)(10,56,74,40,53,
     30,33,70,35,14,43,37,63,51)(11,57,75,41,54,28,31,71,36,15,44,38,61,49)
    (12,55,73,42,52,29,32,72,34,13,45,39,62,50), 
  ( 1,53,33,71,29,36,16,82,37,51,11,62,57,67)( 2,54,31,72,30,34,17,83,38,49,
     12,63,55,68)( 3,52,32,70,28,35,18,84,39,50,10,61,56,69)( 4,22, 8,25,44,
     74,58,76,46,80,65,41,14,19)( 5,23, 9,26,45,75,59,77,47,81,66,42,15,20)
    ( 6,24, 7,27,43,73,60,78,48,79,64,40,13,21) ];

autGroups[47]:=
[ (  1,130,223,237,123, 81)(  2,129,224,238,124, 82)(  3, 46,139,221,250,109)
    (  4, 45,140,222,249,110)(  5, 76, 55,137,234,236)(  6, 75, 56,138,233,
     235)(  7,198,240,111, 96, 52)(  8,197,239,112, 95, 51)(  9,212,284,269,
     196,256)( 10,211,283,270,195,255)( 11,207,164,271,306,258)
    ( 12,208,163,272,305,257)( 13,205,265,275,263, 20)( 14,206,266,276,264,
      19)( 15,199,114, 84, 65,135)( 16,200,113, 83, 66,136)( 17,103, 86, 54,
     149,219)( 18,104, 85, 53,150,220)( 21,215,253)( 22,216,254)
    ( 23,204,286,251, 34,201)( 24,203,285,252, 33,202)( 25,153,281,274,299,
     182)( 26,154,282,273,300,181)( 27,175,278,162,303,261)( 28,176,277,161,
     304,262)( 29,213,279,183,293,160)( 30,214,280,184,294,159)
    ( 31,210,185,267,174,259)( 32,209,186,268,173,260)( 35, 40,132,218,108,
      67)( 36, 39,131,217,107, 68)( 37, 70, 48,134,106, 80)( 38, 69, 47,133,
     105, 79)( 41,145,232,243,101, 88)( 42,146,231,244,102, 87)
    ( 43,142,156,245,121, 90)( 44,141,155,246,122, 89)( 49, 97, 77)
    ( 50, 98, 78)( 57,126,230,247,117, 74)( 58,125,229,248,118, 73)
    ( 59,128,225,157,119, 94)( 60,127,226,158,120, 93)( 61,148,227,179,116,
      71)( 62,147,228,180,115, 72)( 63,144,178,242,100, 92)( 64,143,177,241,
      99, 91)(151,289,171,295,193,287)(152,290,172,296,194,288)
    (165,169,302,297,189,188)(166,170,301,298,190,187)(167,191,291)
    (168,192,292), (  1,260,116,238)(  2,259,115,237)(  3,181,292,275)
    (  4,182,291,276)(  5,235, 58,271)(  6,236, 57,272)(  7,135,142,137)
    (  8,136,141,138)(  9,202,287,221)( 10,201,288,222)( 11,261,294, 84)
    ( 12,262,293, 83)( 13,257,225,162)( 14,258,226,161)( 15,160,208,269)
    ( 16,159,207,270)( 17, 52,289,111)( 18, 51,290,112)( 19,256,267, 21)
    ( 20,255,268, 22)( 23,254,278,266)( 24,253,277,265)( 25,252, 26,251)
    ( 27,220,165,203)( 28,219,166,204)( 29,109,241, 54)( 30,110,242, 53)
    ( 31,263, 87,183)( 32,264, 88,184)( 33, 81,187,274)( 34, 82,188,273)
    ( 35,192,122,249)( 36,191,121,250)( 37,246, 38,245)( 39,145, 98,134)
    ( 40,146, 97,133)( 41,212,118,218)( 42,211,117,217)( 43,301,123, 79)
    ( 44,302,124, 80)( 45,296,105,157)( 46,295,106,158)( 47,169,103,243)
    ( 48,170,104,244)( 49, 61,119,107)( 50, 62,120,108)( 55,281,114,239)
    ( 56,282,113,240)( 59,229,100,199)( 60,230, 99,200)( 63,304, 67,179)
    ( 64,303, 68,180)( 65, 91,101,247)( 66, 92,102,248)( 69,127,148,150)
    ( 70,128,147,149)( 71,175,298,234)( 72,176,297,233)( 73,193,306, 95)
    ( 74,194,305, 96)( 75,189,231,173)( 76,190,232,174)( 77,151,213,300)
    ( 78,152,214,299)( 85,186,284,286)( 86,185,283,285)( 89,178,172,215)
    ( 90,177,171,216)( 93,196, 94,195)(125,130,143,131)(126,129,144,132)
    (153,209,227,155)(154,210,228,156)(163,206,280,223)(164,205,279,224)
    (167,198,168,197) ];

autGroups[48]:=
[ (  1,148, 91)(  2,147, 92)(  3, 63,146)(  4, 64,145)(  5,213,259)
    (  6,214,260)(  7, 93, 61)(  8, 94, 62)(  9,232,282)( 10,231,281)
    ( 11,305,246)( 12,306,245)( 13,261,120)( 14,262,119)( 15,283,170)
    ( 16,284,169)( 17,194,303)( 18,193,304)( 19,122,211)( 20,121,212)
    ( 21,298,289)( 22,297,290)( 23,171,230)( 24,172,229)( 25,274,295)
    ( 26,273,296)( 27,291,271)( 28,292,272)( 29, 31,302)( 30, 32,301)
    ( 33,247,191)( 34,248,192)( 35, 39, 70)( 36, 40, 69)( 37,129, 82)
    ( 38,130, 81)( 41,131, 86)( 42,132, 85)( 43,150, 79)( 44,149, 80)
    ( 45,136, 68)( 46,135, 67)( 47,140, 72)( 48,139, 71)( 49,128, 95)
    ( 50,127, 96)( 51, 98, 75)( 52, 97, 76)( 53,143, 87)( 54,144, 88)
    ( 55,125, 77)( 56,126, 78)( 57,138, 90)( 58,137, 89)( 59,141, 84)
    ( 60,142, 83)( 65,134, 73)( 66,133, 74)( 99,197,253)(100,198,254)
    (101,216,235)(102,215,236)(103,202,110)(104,201,109)(105,206,160)
    (106,205,159)(107,175,264)(108,176,263)(111,209,256)(112,210,255)
    (113,154,219)(114,153,220)(115,204,257)(116,203,258)(117,207,251)
    (118,208,252)(123,200,181)(124,199,182)(151,234,239)(152,233,240)
    (155,224,164)(156,223,163)(157,177,286)(158,178,285)(161,227,278)
    (162,228,277)(165,222,279)(166,221,280)(167,225,266)(168,226,265)
    (173,218,185)(174,217,186)(179,196,249)(180,195,250)(183,299,241)
    (184,300,242)(187,276,244)(188,275,243)(189,293,238)(190,294,237)
    (267,269,288)(268,270,287), (  1,179)(  2,180)(  3,236)(  4,235)(  5,243)
    (  6,244)(  7,249)(  8,250)(  9,237)( 10,238)( 11, 49)( 12, 50)( 13,217)
    ( 14,218)( 15,200)( 16,199)( 19, 80)( 20, 79)( 21,157)( 22,158)( 23,248)
    ( 24,247)( 25,246)( 26,245)( 27,107)( 28,108)( 29,242)( 30,241)( 31,239)
    ( 32,240)( 33,134)( 34,133)( 35,181)( 36,182)( 37,189)( 38,190)( 39,195)
    ( 40,196)( 41,184)( 42,183)( 45,177)( 46,178)( 47,176)( 48,175)( 51, 73)
    ( 52, 74)( 53,152)( 54,151)( 55,194)( 56,193)( 57,192)( 58,191)( 59,101)
    ( 60,102)( 61,187)( 62,188)( 63,185)( 64,186)( 65,127)( 66,128)( 67,257)
    ( 68,258)( 69,263)( 70,264)( 71,252)( 72,251)( 75,220)( 76,219)( 77,201)
    ( 78,202)( 83,160)( 84,159)( 85,262)( 86,261)( 87,260)( 88,259)( 89,109)
    ( 90,110)( 91,255)( 92,256)( 93,253)( 94,254)( 95,135)( 96,136)( 97,299)
    ( 98,300)( 99,269)(100,270)(103,228)(104,227)(105,209)(106,210)(111,168)
    (112,167)(113,298)(114,297)(115,296)(116,295)(119,288)(120,287)(121,280)
    (122,279)(123,143)(124,144)(125,275)(126,276)(129,234)(130,233)(131,215)
    (132,216)(137,174)(138,173)(139,305)(140,306)(141,303)(142,304)(145,294)
    (146,293)(147,286)(148,285)(153,221)(154,222)(155,204)(156,203)(163,273)
    (164,274)(165,271)(166,272)(169,268)(170,267)(171,266)(172,265)(205,232)
    (206,231)(207,230)(208,229)(211,225)(212,226)(213,223)(214,224)(277,302)
    (278,301)(281,291)(282,292) ];

autGroups[49]:=
[ ( 1, 7,26, 9, 2,32)( 3,33,23,16,24,31)( 4,36,29,10,30,35)( 5,34, 6,13,20,15
     )( 8,27,17,22,18,25)(11,28,12,19,14,21), 
  ( 1,34,30, 4,15,26,18,16,27,14, 6,28,36, 3, 9,32,24,10,21,20,12,22,33, 8)
    ( 2,25)( 5,31)( 7,19)(11,35)(17,29) ];

autGroups[50]:=
[ (  1, 21, 36, 29)(  2, 22, 35, 30)(  3,  4)(  5, 72, 37, 40)
    (  6, 71, 38, 39)(  7, 48, 33, 20)(  8, 47, 34, 19)(  9, 52, 81, 23)
    ( 10, 51, 82, 24)( 11, 97,110, 85)( 12, 98,109, 86)( 13, 99, 87, 25)
    ( 14,100, 88, 26)( 15, 76, 70, 93)( 16, 75, 69, 94)( 17, 77, 41, 46)
    ( 18, 78, 42, 45)( 27,102, 32, 89)( 28,101, 31, 90)( 43, 95,104, 68)
    ( 44, 96,103, 67)( 49, 55, 73, 65)( 50, 56, 74, 66)( 53, 57, 54, 58)
    ( 59, 79, 83,108)( 60, 80, 84,107)( 61, 64, 62, 63)( 91,106, 92,105), 
  (  1, 56, 57, 46)(  2, 55, 58, 45)(  3, 28, 82, 96)(  4, 27, 81, 95)
    (  5, 73, 92, 14)(  6, 74, 91, 13)(  7, 87,  9,100)(  8, 88, 10, 99)
    ( 11, 89, 12, 90)( 15, 62, 19, 84)( 16, 61, 20, 83)( 17, 72, 18, 71)
    ( 21, 75, 67, 40)( 22, 76, 68, 39)( 23, 86, 49, 52)( 24, 85, 50, 51)
    ( 25, 64, 98, 78)( 26, 63, 97, 77)( 29,107, 48,105)( 30,108, 47,106)
    ( 31, 36, 69, 42)( 32, 35, 70, 41)( 37, 54,109, 93)( 38, 53,110, 94)
    ( 43, 79, 60,102)( 44, 80, 59,101)( 65,104, 66,103) ];

autGroups[51]:=
[ (  1, 41,106, 18, 32, 59, 57, 68, 75, 89, 80, 66)(  2, 47,100,109, 95, 21,
      38, 63, 20, 23, 48, 83)(  3, 11, 25, 39, 93, 98, 81, 56, 77, 54,101, 71
     )(  4, 44,105, 13, 34, 61, 58, 69, 76, 88, 79, 65)(  5, 16, 42,108, 72,
      94,103, 17, 26, 37,  6, 40)(  7, 12, 43,107, 74, 92,104, 15, 27, 35,
      10, 36)(  8, 46, 99,110, 97, 19, 33, 60, 24, 22, 45, 85)
    (  9, 14, 28, 31, 91, 96, 82, 55, 78, 52,102, 73)( 29, 49, 67, 84, 64,
      90, 30, 50, 70, 86, 62, 87), (  1, 87, 68, 80, 27, 45, 31, 85, 74, 89)
    (  2,104, 23, 61, 24, 82, 52,  9, 46,105)(  3, 47,106,  8,103, 22, 59,
      20, 81, 54)(  4, 90, 69, 79, 26, 48, 39, 83, 72, 88)(  5, 98, 30, 19,
      76, 51, 65, 97, 64, 11)(  6, 16,100, 84, 77,102, 49, 60, 36, 43)
    (  7, 96, 29, 21, 75, 53, 66, 95, 62, 14)( 10, 12, 99, 86, 78,101, 50,
      63, 40, 42)( 13, 56, 73, 32, 67)( 15, 94, 44, 25,108,110, 38, 35, 91,
      57)( 17, 92, 41, 28,107,109, 33, 37, 93, 58)( 18, 55, 71, 34, 70) ];

autGroups[52]:=
[ (  1,107, 47,127,  8,110, 41,125)(  2,112, 43,123,  3,109, 42,126)
    (  4,106, 46,122,  5,111, 44,124)(  6,108, 48,128,  7,105, 45,121)
    (  9, 36, 33, 15)( 10, 30, 38, 28, 13, 25, 34, 31)( 11, 53, 35, 56, 16,
      49, 39, 52)( 12, 14, 40, 37)( 17,103, 23,101, 20,102, 18,100)
    ( 19, 97, 24, 98, 22,104, 21, 99)( 26, 29)( 27, 55, 32, 51)
    ( 57,117, 58,118, 63,119, 59,115)( 60,116, 64,120, 62,114, 61,113)
    ( 65, 66, 78, 80, 85, 88, 93, 94)( 67, 95, 73, 72, 86, 79, 91, 84)
    ( 68, 81, 75, 96, 87, 71, 92, 77)( 69, 70, 74, 76, 82, 83, 90, 89)
    (129,264,196,294,246,373,166,337)(130,319,198,292,237,369,154,351)
    (131,276,199,289,252,376,179,330)(132,298,193,295,225,372,138,326)
    (133,260,200,290,241,370,162,342)(134,316,194,296,233,374,157,348)
    (135,280,195,293,255,371,183,334)(136,301,197,291,229,375,141,322)
    (137,321,148,300,205,310,230,366)(139,323,170,304,212,272,228,354)
    (140,328,188,302,222,283,231,382)(142,325,152,303,201,306,226,362)
    (143,324,191,297,217,288,227,377)(144,327,173,299,215,267,232,358)
    (145,259,208,311,247,363,168,340)(146,317,202,305,234,361,153,350)
    (147,279,203,308,249,368,177,333)(149,263,204,307,244,367,164,343)
    (150,314,206,309,238,365,158,345)(151,275,207,312,254,364,181,329)
    (155,346,169,313,209,265,240,357)(156,347,187,315,223,286,235,383)
    (159,352,192,320,220,281,239,380)(160,349,174,318,214,270,236,353)
    (161,338,185,262,219,285,242,384)(163,339,171,257,213,266,243,355)
    (165,341,190,258,224,282,245,379)(167,344,176,261,210,269,248,359)
    (172,277,216,271,253,360,182,331)(175,273,211,268,250,356,178,335)
    (180,332,189,278,221,287,251,378)(184,336,186,274,218,284,256,381), 
  (  1,375, 80,286,117,297, 30,338)(  2,362, 75,296,111,304,  9,342)
    (  3,382, 78,270,128,301, 36,339)(  4,354, 73,305,103,300, 49,343)
    (  5,372, 76,281,114,302, 25,341)(  6,366, 79,292,107,299, 14,337)
    (  7,377, 74,265,123,298, 40,344)(  8,358, 77,309, 99,303, 53,340)
    ( 10,348, 59,361, 67,294,109,313)( 11,330, 22,368, 88,291,108,273)
    ( 12,326, 46,367, 93,289,106,261)( 13,351, 64,365, 71,290,105,318)
    ( 15,322, 41,363, 90,293,110,257)( 16,334, 18,364, 83,295,112,277)
    ( 17,371, 82,288,115,278, 27,336)( 19,378, 86,272,122,276, 39,331)
    ( 20,356, 87,306,104,275, 54,329)( 21,376, 85,283,120,274, 32,332)
    ( 23,381, 81,267,125,280, 35,335)( 24,360, 84,310,100,279, 50,333)
    ( 26,347, 57,374, 68,282,118,315)( 28,324, 48,373, 91,287,119,258)
    ( 29,352, 62,369, 72,285,113,320)( 31,328, 43,370, 96,284,116,262)
    ( 33,323, 44,379, 92,268,121,260)( 34,349, 61,380, 70,269,124,316)
    ( 37,327, 47,384, 95,271,126,264)( 38,346, 58,383, 66,266,127,319)
    ( 42,355, 94,308,102,263, 56,325)( 45,359, 89,312, 98,259, 52,321)
    ( 51,345, 60,353, 65,307,101,317)( 55,350, 63,357, 69,311, 97,314)
    (129,166,247,218,187,158,255,196)(130,154,249,219,185,168,246,195)
    (131,183,238,217,191,142,229,194)(132,141,226,220,189,177,237,193)
    (133,162,244,221,192,153,252,200)(134,157,254,224,190,164,241,199)
    (135,179,234,222,188,137,225,198)(136,138,230,223,186,181,233,197)
    (139,227,206,151,184,240,212,173)(140,232,214,172,178,239,201,148)
    (143,228,209,175,182,235,205,152)(144,231,202,147,180,236,215,170)
    (145,161,248,216,174,160,251,204)(146,159,250,213,176,167,242,203)
    (149,165,243,211,169,155,256,208)(150,156,253,210,171,163,245,207) ];

autGroups[53]:=
[ (  1,119, 21, 29,114,  9)(  2, 94, 24, 38, 97, 55)(  3, 80, 22, 35, 73, 16)
    (  4, 92, 23, 28,103, 50)(  5,113, 52, 27,120, 41)(  6, 96, 14, 36, 99,
      44)(  7, 74, 53, 37, 79, 42)(  8, 90, 11, 30,101, 43)( 10, 34, 93, 46,
      61, 98)( 12, 60,116, 20, 33,117)( 13, 58, 75, 19, 31, 78)
    ( 15, 32, 91, 45, 63,104)( 17, 26, 89, 54, 59,102)( 18, 40, 95, 51, 57,
     100)( 25, 76, 47, 62, 77, 56)( 39,115, 48, 64,118, 49)( 65,144,172,122,
     192,162)( 66,184,171,121,129,156)( 67,191,170,124,138,159)
    ( 68,130,169,123,178,165)( 69,190,145,126,181,155)( 70,131,146,125,141,
     161)( 71,179,147,128,189,166)( 72,139,148,127,132,160)( 81,135,174, 87,
     183,168)( 82,133,157, 88,142,151)( 83,134,149, 85,140,164)
    ( 84,136,153, 86,177,176)(105,182,152,111,188,167)(106,143,158,112,186,
     173)(107,137,175,109,185,163)(108,180,154,110,187,150), 
  (  1,125,172, 78, 15, 37, 68,156,115, 47)(  2,110,159, 80, 21, 30,106,169,
     113, 50)(  3, 70,155, 79, 19, 27,123,146,114, 54)(  4,108,147, 77, 11,
      36,112,160,116, 41)(  5,126,176, 89, 14, 35, 67,163, 93, 42)
    (  6, 83,168, 94, 20, 28, 87,173, 90, 51)(  7, 69,164, 99, 22, 29,124,
     150,103, 55)(  8, 85,151,104, 10, 38, 81,167,100, 48)(  9, 57, 86,148,
      75, 23, 32, 82,166,118)( 12, 61,107,152,102, 18, 26,111,157, 98)
    ( 13, 60, 65,171, 76, 17, 25,128,162,117)( 16, 64, 66,175, 91, 24, 31,
     127,153, 95)( 33, 72,149,101, 44, 62,121,154, 97, 53)( 34, 88,170,119,
      43, 59, 84,165, 74, 52)( 39, 71,145,120, 45, 58,122,161, 73, 56)
    ( 40,105,174, 92, 46, 63,109,158, 96, 49)(129,135,179,181,134,142,141,
     185,186,178)(130,138,190,182,144,140,187,143,192,188)(131,183)
    (132,191,139,184,137,177,180,136,133,189) ];

autGroups[54]:=
[ (  1, 11, 23, 28,  3,  9, 20, 31)(  2, 14, 24, 29,  5, 16, 22, 26)
    (  4, 15, 19, 25,  7, 12, 17, 27)(  6, 10, 21, 32,  8, 13, 18, 30)
    ( 33, 83, 56, 69, 35, 81, 54, 66)( 34, 86, 55, 68, 37, 88, 52, 71)
    ( 36, 87, 53, 72, 39, 84, 50, 70)( 38, 82, 51, 65, 40, 85, 49, 67)
    ( 41, 79, 64, 94, 43, 76, 62, 96)( 42, 74, 63, 91, 45, 77, 60, 89)
    ( 44, 75, 61, 90, 47, 73, 58, 93)( 46, 78, 59, 95, 48, 80, 57, 92)
    ( 97,122,115,106, 99,125,113,109)( 98,127,117,111,101,124,114,108)
    (100,126,119,110,103,128,116,112)(102,123,120,107,104,121,118,105), 
  (  1, 45,107, 30, 64,104,  5, 48,112, 31, 61, 99)(  2, 46,110, 28, 58, 97,
       3, 42,105, 32, 62,102)(  4, 47,106, 26, 59,100,  8, 43,108, 27, 63, 98
     )(  6, 41,111, 25, 60,101,  7, 44,109, 29, 57,103)(  9, 55,127, 17, 34,
     120, 13, 51,125, 21, 38,115)( 10, 49,122, 18, 40,113, 11, 52,124, 19,
      37,118)( 12, 53,126, 20, 36,116, 16, 56,121, 24, 33,114)
    ( 14, 54,123, 22, 35,117, 15, 50,128, 23, 39,119)( 65, 67)
    ( 66, 68, 72, 69, 71, 70)( 73, 96, 87, 78, 92, 83)( 74, 90, 82, 76, 91,
      86)( 75, 94, 84, 80, 95, 81)( 77, 93, 85, 79, 89, 88) ];

autGroups[55]:=
[ ( 1, 9)( 2, 8,23,32,31,22, 7, 3,17,26,37,28,13, 4,11,20,35,34,19,10, 5,14,
     29,38,25,16)( 6,15)(12,21)(18,27)(24,33)(30,39), 
  ( 1,38, 5,15,26, 7)( 2, 9,32)( 3,31,24,16,29,39)( 4,37,30,10,35,36)
    ( 6,34,11,21,20,13)( 8,25,18,22,23,33)(12,28,17,27,14,19) ];

autGroups[56]:=
[ (  1, 46,244,119,132,159,136,194, 48,128,160,208,182, 14,  6, 15)
    (  2, 55,107,233,196,150,143,258, 47,186,126,256,113,253, 86, 23)
    (  3, 54,137,265, 25,163,154,220,145, 93,181,112, 69,106,162, 38)
    (  4,171, 68,260,129,141, 90,224, 65,187, 52, 64,  8,217,166,270)
    (  5, 11,168,226, 41, 19, 18,131, 98, 44,251, 96,192,108,248,234)
    (  7,207,102,114, 33,116,183,142, 60, 89,249, 32,243,155,232,272)
    (  9,214, 75,263, 85,167, 26, 99, 49,100,153,144,146, 42,170,189)
    ( 10,200, 78, 13, 82, 37, 74,206,173,178, 43, 80,205,255,176, 77)
    ( 12, 87, 71,105,254,252,240, 66, 39,184, 57, 16,104,246,242, 45)
    ( 17,227, 63,103, 56,250,120, 28,259,264,266,269, 91,225, 97,241)
    ( 20,115,123,148,257,135, 95, 31,211,193,118,117,151,223,191,271)
    ( 21, 83,165,237,219, 27,179,261,221,239,267,213,161,124,229,209)
    ( 22,195,175,262,268,199,127,158, 81,177,122, 30,147, 50, 34, 40)
    ( 24, 67,197,156, 59, 84,185,134,110,238,210, 35, 70,133, 61,218)
    ( 29, 51,101, 72,212,157,130, 36, 76,202,180, 53,139, 88,231,222)
    ( 58,216,201,172,149,236,198,109,174,228, 79,138, 94,247,121,188)
    ( 62,245,169,230,111,152,204,164,125,140,203, 73,190, 92,215,235), 
  (  1,264,144,119,184, 31,120,100)(  2,269,111,233,200,238, 96,141)
    (  3, 68,166, 25, 82,177,192,261)(  4, 84, 37,201, 92, 79,179,134)
    (  5,109,236,199,173, 57, 95,237)(  6, 47, 64,122,156,267,150,204)
    (  7,171,234, 48, 61,227,127,270)(  8, 35, 75, 67,129, 51,181, 36)
    (  9,164,137,230, 85, 77,185,161)( 10,145, 58,153, 94, 69,213,139)
    ( 11,268, 80, 49, 52, 90,146, 21)( 12,272, 40, 72,162, 73,130,135)
    ( 13, 87,239, 32,254, 99,225,197)( 14, 76,209,118,255,258,223, 38)
    ( 15,189,112,115,180,102,195,228)( 16,101,235,154,157, 18,207, 20)
    ( 17,128,265,103,123,242,136, 54)( 19,107,249, 98, 50,183,143, 22)
    ( 23,125,169,186,175,250,132,226)( 24,104,193,256,131,133, 26,149)
    ( 27,252,194, 60, 63,117, 74,152)( 28,196,244, 71,165,121,178,191)
    ( 29, 91,163,105, 55,229,240,205)( 30,114,216,138,155,251,159, 56)
    ( 33,208,247,142,124,222,262,217)( 34,253, 66,257, 70,174,170,218)
    ( 39, 59, 86, 44, 62,241,220,263)( 41,232,231, 45, 93,167,266,172)
    ( 42,151,116, 78,271,221,210,182)( 43, 88, 65, 53, 83,140,198,245)
    ( 46,248,224,113,187,158,259,212)( 81,215,190,108,219,148,246,206)
    ( 89,188, 97,243,260,168,202,176)(106,160,203,147,126,110,211,214) ];

autGroups[57]:=
[ (  1,109, 65,119,261,255, 16,231,251)(  2,258,262, 38,207,239, 68,113, 56)
    (  3, 70,118,242,190,177,  8,137,160,  6,189,179,214, 67,120,  5,124, 80)
    (  4,111,158,269,110, 51,272,140, 52,  7,186, 74,234, 18, 46,241,126,152)
    (  9, 28,246,195,212,232,172,218,102, 33,171,250, 98,206,263, 27,219,196)
    ( 10, 79,176,249, 42,210,267, 21,220,167,125, 34,240,182,215,203,156,224)
    ( 11,175, 85,209, 87, 64,247, 62,168, 39, 36,198,270, 66,114,271,164, 13)
    ( 12,144, 48,235,187,226,145,208,127,166, 94, 49,256,106,217,134,202, 41)
    ( 14, 31, 86, 47,257, 57, 32, 17,122, 95,169,200, 97,233,129,170,155,103)
    ( 15,266, 29, 82,108,252, 61, 92,184,165,228, 78,136, 99,244,163,143, 44)
    ( 19,146,197,205, 25, 93,128,253,174,154,130, 83,268, 55, 60,100,254, 37)
    ( 20, 63,104, 35, 30,183,193, 89,216, 26,162,194,133, 76, 45,101, 81,260)
    ( 22,147, 40, 23,259, 96,204, 50,225,105,131,173, 77,223,141,211,149,237)
    ( 24, 58,139,245,229,213, 73,199,181,153,161,123,238,264,230,157, 84, 43)
    ( 53,227,265,236,185, 71,117, 72,115,150,243,222,201,112,192,180,191,178)
    ( 54,132, 75,135,248,142,138, 88,188,151, 59,159,148,221, 91,121, 90,107)
    ( 69,116), (  1,126,269,258,256,159, 82,111,225,222,201,187,  4,128, 90,
      22)(  2,121,191, 63,240,143, 34,186,  7,125, 83,167, 97, 72,148,109)
    (  3,123,166,100,136, 23, 84,137,132, 53,147,108, 93, 49, 71,188)
    (  5,127, 68, 62,270,261,259,189, 60, 36,202,229,213,141, 39, 99)
    (  6,124,272,168, 98,208,183, 70,227,185,  8,122, 96,218,198, 18)
    (  9, 15,196, 89, 24,179,113,119,120,153, 32,246,154, 28,175, 11)
    ( 10,165,104,195, 91,181, 69,115,214,220,193, 92,250,207,144, 38)
    ( 12,254,206,161, 51, 65, 64,117,156, 25, 48, 45,171, 76, 79,133)
    ( 13,253,257,235,204,203,177,118,267,145,105, 26, 29, 85,266,174)
    ( 14, 33,103,131,252,210,178,114, 56, 46, 43,172, 78, 77, 31,176)
    ( 16,102,197, 17,182, 67,180,116, 58,221,219,200, 19,249,231,173)
    ( 20, 73,140,134,160, 81, 40,162, 54,226,209,243,216,150, 44,199)
    ( 21,239,190, 61,265,155, 30, 86,230,228,248,262,212,241,184, 66)
    ( 27,251,142, 37,264,194, 95,101,236,205,242,255,215,247,146,110)
    ( 35,106,224,158, 87, 59,107,271,152, 42,237,169, 75,238,157, 88)
    ( 41,129, 55, 74,268,163, 52,244,260,223,217,232,138,135,263,149)
    ( 47, 94,170, 80,211,234,245,233,139,130,192, 57,112,164, 50,151) ];

autGroups[58]:=
[ (  1, 30,160,124,266,240,239,230,194, 17, 91,206, 87,261,211,242,214,178)
    (  2, 32, 41,213,143,113,219,188,100, 34,115, 20,216,181, 39,223,207, 85)
    (  3, 29,199, 49, 68,148, 73, 52,153, 10, 75,179,  5, 28,152, 13, 70,176)
    (  4, 26,249,192, 57,111,218,177,  7, 25, 11,243,195, 21,102,215,166, 66)
    (  6, 27,120,189, 63,183, 38,158,121, 81, 43,136,145,116,203,106,182, 40)
    (  8, 31,231,236,263,259,173, 48,139, 89, 98,225,232,260,264,162,131,175)
    (  9,163,132, 18,248,245,165, 72,117, 19,190, 64, 33,244,250,157,125, 86)
    ( 12,141, 93,271,161,135, 24, 69,205, 83,197, 54,270,149, 80,129, 59,156)
    ( 14,187, 99,268,196, 23,169, 65,184, 97,191, 61,272,204,105,193, 22,154)
    ( 15,104,123,212,137, 94,167, 67,202,107,127, 55,237,147, 77,174, 44,155)
    ( 16,255,227,185,103,128,168, 71, 42,130,246,224,146,119,112,186,101, 35)
    ( 36,140, 90,258,138, 96,144,114,108, 82,180, 37,265,142,118,171, 46, 62)
    ( 45,228,254,221,159,122, 50,233,150, 74,217,256,226,172, 47, 58,210,170)
    ( 51,269,151, 78,134,252,234,238,208, 84,267,201,109, 88,253,220,229,200)
    ( 53, 76)( 56, 92, 95, 79,251,209,257,241,164,133, 60,110,126,247,235,
     262,222,198), (  1,240)(  2,139)(  4, 56)(  5,262)(  6, 52)(  7,261)
    (  9,199)( 11,198)( 12, 98)( 13,242)( 14,231)( 15, 95)( 16,230)( 17,211)
    ( 18,221)( 19,179)( 20, 80)( 21,220)( 22,123)( 23, 62)( 24,264)( 25,133)
    ( 26,164)( 27,178)( 28,237)( 29,265)( 30, 82)( 31, 83)( 32,131)( 33,226)
    ( 34,175)( 35,192)( 36, 91)( 37,181)( 38,102)( 40,252)( 41,135)( 42,195)
    ( 43,194)( 44,117)( 45,132)( 46,193)( 47,196)( 48,115)( 49,257)( 50, 72)
    ( 51,142)( 53,270)( 54,256)( 55, 65)( 57,234)( 58,125)( 59,186)( 60,233)
    ( 61,208)( 63,184)( 64, 74)( 66,266)( 67, 86)( 68,212)( 69,168)( 70, 81)
    ( 71,247)( 73,239)( 75,258)( 76,271)( 77,189)( 79,238)( 84,138)( 85,141)
    ( 87,150)( 88,151)( 90,143)( 92,210)( 93,254)( 94,145)( 96,246)( 97,225)
    ( 99,200)(100,197)(101,251)(104,157)(105,108)(106,111)(107,110)(114,169)
    (116,154)(118,255)(120,243)(121,253)(122,204)(124,170)(126,229)(127,165)
    (129,259)(130,214)(134,201)(136,249)(137,236)(140,177)(144,152)(146,156)
    (147,232)(148,171)(149,228)(153,267)(155,272)(159,160)(161,217)(163,216)
    (166,180)(167,241)(172,206)(174,222)(176,269)(183,245)(185,205)(187,227)
    (188,244)(190,213)(191,224)(202,268)(203,250)(207,248)(209,263)(215,223)
    (218,219)(235,260) ];

autGroups[59]:=
[ (  1, 92,154, 56,126,190, 52, 84,158,  3, 79,150, 32,132,172, 50, 94,168,
      19, 74,148, 13,114,160, 62,110,184, 36, 77,152)(  2, 91,153, 55,125,
     189, 51, 83,157,  4, 80,149, 31,131,171, 49, 93,167, 20, 73,147, 14,113,
     159, 61,109,183, 35, 78,151)(  5,108,162, 18, 81,163, 24,102,174, 41,
      86,156, 25, 97,175)(  6,107,161, 17, 82,164, 23,101,173, 42, 85,155,
      26, 98,176)(  7, 75,145,  8, 76,146)(  9, 90,180, 40,120,192, 70,139,
     211, 59,103,169, 12,124,178, 34,129,216, 72, 87,185, 30, 96,166, 66,138,
     208, 43,115,199)( 10, 89,179, 39,119,191, 69,140,212, 60,104,170, 11,
     123,177, 33,130,215, 71, 88,186, 29, 95,165, 65,137,207, 44,116,200)
    ( 15,144,201, 58,105,195, 45,117,197, 53,127,187, 64,135,209, 21,121,193,
     27, 99,203, 67,142,213, 47,111,181, 37,133,205)( 16,143,202, 57,106,196,
     46,118,198, 54,128,188, 63,136,210, 22,122,194, 28,100,204, 68,141,214,
      48,112,182, 38,134,206), (  1,210, 57,163, 29,200, 18,203, 53,167)
    (  2,209, 58,164, 30,199, 17,204, 54,168)(  3,158, 28,192, 21,151, 20,
     197, 33,196)(  4,157, 27,191, 22,152, 19,198, 34,195)(  5,170, 60,174,
      15,205, 35,148, 48,214)(  6,169, 59,173, 16,206, 36,147, 47,213)
    (  7,181, 66,190, 11,175, 71,153, 70,194)(  8,182, 65,189, 12,176, 72,
     154, 69,193)(  9,185)( 10,186)( 13,183, 39,172, 68,161, 23,201, 55,215)
    ( 14,184, 40,171, 67,162, 24,202, 56,216)( 25,156, 63,160, 43,149, 52,
     166, 61,187)( 26,155, 64,159, 44,150, 51,165, 62,188)( 31,208, 38,145,
      45,178, 50,179, 41,212)( 32,207, 37,146, 46,177, 49,180, 42,211)
    ( 73, 78,133,137,112, 74, 77,134,138,111)( 75,122,119,109, 96, 82,115,
     114,130,117)( 76,121,120,110, 95, 81,116,113,129,118)( 79,106,136, 86,
      89, 88, 97,127, 99, 83)( 80,105,135, 85, 90, 87, 98,128,100, 84)
    ( 91, 94,124,125,144,102,108,141,132,139)( 92, 93,123,126,143,101,107,
     142,131,140) ];

autGroups[60]:=
[ (  1, 15,124,109,  7, 17,122,110)(  2,134, 42,111,  9,133,212, 95)
    (  3,132,101, 19,  5,130,102, 13)(  4,135,211, 96, 10,129, 44,107)
    (  6, 74,163,108, 11,131,205,112)(  8, 67,165,106, 12,136,206,105)
    ( 14,126, 52,103, 21,125,214, 85)( 16,127,213, 86, 22,121, 54, 99)
    ( 18, 94,169,100, 23,123,207,104)( 20, 87,171, 98, 24,128,208, 97)
    ( 25, 27,140,117, 31, 29,138,118)( 26,142, 62,119, 33,141,216, 75)
    ( 28,143,215, 76, 34,137, 64,115)( 30, 84,175,116, 35,139,209,120)
    ( 32, 77,177,114, 36,144,210,113)( 37,164,188, 59,197, 93,145,183)
    ( 38, 50,204,162,152, 91, 80,178)( 39,193, 73,157,187, 57,170,184)
    ( 40,196,156,158, 81, 90,172, 48)( 41,194, 69,161,185, 63,174,182)
    ( 43,168,186, 61,198, 89,149,181)( 45,195,153,160, 82, 88,173, 56)
    ( 46, 55,203,159,154, 92, 78,179)( 47,176,192, 49,201, 83,151,191)
    ( 51,202, 79,155,189, 53,180,190)( 58, 60,200,150,146, 71, 70,166)
    ( 65,199,147,148, 72, 68,167, 66), (  1,121, 89, 61,  2,122, 92, 65)
    (  3, 94, 63, 12,  5, 87, 57, 11)(  4,125, 88, 62,  6,128, 96, 66)
    (  7,126, 93, 59, 10,124, 91, 60)(  8,123, 95, 58,  9,127, 90, 64)
    ( 13,137, 69, 51, 14,138, 72, 55)( 15, 84, 43, 36, 17, 77, 37, 35)
    ( 16,141, 68, 52, 18,144, 76, 56)( 19,142, 73, 49, 22,140, 71, 50)
    ( 20,139, 75, 48, 21,143, 70, 54)( 23, 27, 74, 53, 24, 29, 67, 47)
    ( 25,129, 79, 41, 26,130, 82, 45)( 28,133, 78, 42, 30,136, 86, 46)
    ( 31,134, 83, 39, 34,132, 81, 40)( 32,131, 85, 38, 33,135, 80, 44)
    ( 97,173,205,160,174,101,189,199)( 98,190,150,161,169,211,201,119)
    ( 99,116,184,159,145,165,214,198)(100,192,147,157,171,212,202,115)
    (102,191,200,104,172,206,158,170)(103,114,182,162,149,163,213,197)
    (105,167,207,154,180,117,181,195)(106,186,156,155,175,215,193,111)
    (107,108,188,153,151,177,216,194)(109,185,203,113,179,209,148,168)
    (110,187,204,120,178,210,146,164)(112,166,208,152,176,118,183,196) ];

autGroups[61]:=
[ (  2,  3,  5,  9,  6,  4)(  7, 10, 11)(  8, 12, 13)( 14, 15)( 17, 33)
    ( 18, 35, 21, 41, 22, 36)( 19, 37, 25, 38, 20, 34)( 23, 42, 27, 39, 26,
      43)( 24, 44, 29, 40, 28, 45)( 30, 47)( 31, 46)( 32, 48)
    ( 49,130, 68,120, 93,111, 62,138, 71,115, 82, 97)( 50,129, 65,114, 84,
     104, 61,143, 78,122, 87, 99)( 51,135, 75,117, 83,103, 59,133, 67,119,
      91,101)( 52,136, 77,127, 94,106, 55,131, 66,113, 81, 98)
    ( 53,139, 74,126, 95,109, 60,134, 73,128, 96,105)( 54,140, 72,116, 86,
     108, 56,132, 70,124, 88,100)( 57,144, 80,121, 85,107, 58,142, 79,125,
      92,102)( 63,141, 76,118, 89,112, 64,137, 69,123, 90,110), 
  (  1, 97, 49)(  2, 98, 50)(  3, 99, 51)(  4,100, 52)(  5,101, 53)
    (  6,102, 54)(  7,103, 55)(  8,104, 56)(  9,105, 57)( 10,106, 58)
    ( 11,107, 59)( 12,108, 60)( 13,109, 61)( 14,110, 62)( 15,111, 63)
    ( 16,112, 64)( 17,113, 65)( 18,114, 66)( 19,115, 67)( 20,116, 68)
    ( 21,117, 69)( 22,118, 70)( 23,119, 71)( 24,120, 72)( 25,121, 73)
    ( 26,122, 74)( 27,123, 75)( 28,124, 76)( 29,125, 77)( 30,126, 78)
    ( 31,127, 79)( 32,128, 80)( 33,129, 81)( 34,130, 82)( 35,131, 83)
    ( 36,132, 84)( 37,133, 85)( 38,134, 86)( 39,135, 87)( 40,136, 88)
    ( 41,137, 89)( 42,138, 90)( 43,139, 91)( 44,140, 92)( 45,141, 93)
    ( 46,142, 94)( 47,143, 95)( 48,144, 96), 
  (  1, 16)(  2,  9)(  3,  5)(  4,  6)(  7, 11)(  8, 12)( 17, 32)( 18, 25)
    ( 19, 21)( 20, 22)( 23, 27)( 24, 28)( 33, 48)( 34, 41)( 35, 37)( 36, 38)
    ( 39, 43)( 40, 44)( 49,112)( 50,105)( 51,101)( 52,102)( 53, 99)( 54,100)
    ( 55,107)( 56,108)( 57, 98)( 58,106)( 59,103)( 60,104)( 61,109)( 62,110)
    ( 63,111)( 64, 97)( 65,128)( 66,121)( 67,117)( 68,118)( 69,115)( 70,116)
    ( 71,123)( 72,124)( 73,114)( 74,122)( 75,119)( 76,120)( 77,125)( 78,126)
    ( 79,127)( 80,113)( 81,144)( 82,137)( 83,133)( 84,134)( 85,131)( 86,132)
    ( 87,139)( 88,140)( 89,130)( 90,138)( 91,135)( 92,136)( 93,141)( 94,142)
    ( 95,143)( 96,129) ];

autGroups[62]:=
[ ( 1,83,29)( 2,91,36,10,77,40,12,79,25, 4,90,41, 7,73,30)( 3,84,26)
    ( 5,93,35, 9,74,39,11,80,27, 6,89,42, 8,75,28)(13,87,47,19,82,33,15,95,
     45,22,78,37,17,92,31)(14,88,48,21,81,34,16,96,43,20,76,38,18,94,32)
    (23,86,44)(24,85,46)(49,56,68,61,52)(50,59,67,71,64)(51,55,70,62,54)
    (53,60,69,72,63), ( 1, 4, 5)( 2, 3, 6)( 7,13,17)( 8,14,18)( 9,15,11)
    (10,16,12)(19,20,23)(21,22,24)(25,53,93)(26,54,89)(27,50,91)(28,49,83)
    (29,52,90)(30,51,84)(31,65,92)(32,66,94)(33,59,80)(34,60,79)(35,63,95)
    (36,64,96)(37,55,73)(38,56,75)(39,57,74)(40,58,77)(41,61,87)(42,62,88)
    (43,71,86)(44,67,82)(45,72,85)(46,69,81)(47,68,76)(48,70,78), 
  ( 1, 3)( 2, 4)( 5, 6)( 9,11)(10,12)(13,17)(14,18)(19,22)(20,21)(23,24)
    (25,91)(26,83)(27,93)(28,89)(29,84)(30,90)(31,92)(32,94)(33,95)(34,96)
    (35,80)(36,79)(37,87)(38,88)(39,74)(40,77)(41,73)(42,75)(43,81)(44,85)
    (45,82)(46,86)(47,78)(48,76)(49,54)(50,53)(51,52)(55,61)(56,62)(59,63)
    (60,64)(67,72)(68,70)(69,71) ];

autGroups[63]:=
[ ( 1, 4,15,12,48,35,10,36)( 2,13,20,27)( 3,37,21, 6,43,23,11,26)
    ( 5,33,32,17)( 7,44,18,45)( 8,38,42,40,25,47,24,19)( 9,28,46,41,16,22,29,
     30)(14,39,31,34), ( 1,13,38,31,48,18,47,17)( 2,19, 5,16,14,46,32,10,45,
     23,27, 6,44,40,34, 9,33,29,39,15,20,37, 7,26)( 3,42,36)( 4, 8,41,35,25,
     30)(11,21)(12,43,24) ];

autGroups[64]:=
[ (  2,  3,  5,  8,  7,  4)(  6,  9, 13, 18, 14, 10)( 11, 12, 16, 21, 20, 15)
    ( 17, 22, 19, 24, 25, 23)( 26,126, 76,101)( 27,130, 80,107, 32,132, 82,
     105, 30,127, 77,102)( 28,128, 78,104, 29,133, 83,108, 33,129, 79,103)
    ( 31,134, 84,114, 39,143, 93,113, 38,135, 85,106)( 34,131, 81,110, 35,
     138, 88,118, 43,139, 89,109)( 36,141, 91,120, 45,145, 95,116, 41,136,
      86,111)( 37,137, 87,115, 40,146, 96,121, 46,140, 90,112)
    ( 42,149, 99,125, 50,148, 98,119, 44,147, 97,117)( 47,144, 94,123, 48,
     150,100,124, 49,142, 92,122)( 52, 53, 55, 58, 57, 54)( 56, 59, 63, 68,
      64, 60)( 61, 62, 66, 71, 70, 65)( 67, 72, 69, 74, 75, 73), 
  (  1,108, 33)(  2,107, 26)(  3,101, 30)(  4,118, 32)(  5,105, 38)
    (  6,104, 28)(  7,121, 43)(  8,113, 46)(  9,103, 41)( 10,120, 29)
    ( 11,114, 27)( 12,102, 34)( 13,116, 44)( 14,123, 45)( 15,124, 39)
    ( 16,109, 50)( 17,110, 37)( 18,119, 48)( 19,115, 31)( 20,111, 49)
    ( 21,125, 36)( 22,112, 40)( 23,117, 35)( 24,106, 47)( 25,122, 42)
    ( 51,128, 78)( 52,127, 76)( 53,126, 80)( 54,131, 77)( 55,130, 84)
    ( 56,129, 83)( 57,137, 81)( 58,134, 87)( 59,133, 91)( 60,136, 79)
    ( 61,135, 82)( 62,132, 88)( 63,141, 99)( 64,142, 86)( 65,144, 85)
    ( 66,138, 97)( 67,139, 96)( 68,149, 92)( 69,140, 93)( 70,145, 94)
    ( 71,147, 95)( 72,146, 90)( 73,148, 89)( 74,143,100)( 75,150, 98), 
  (  2,  3)(  4,  5)(  7,  8)(  9, 10)( 11, 12)( 13, 14)( 15, 16)( 17, 19)
    ( 20, 21)( 23, 24)( 26,101)( 27,102)( 28,104)( 29,103)( 30,107)( 31,110)
    ( 32,105)( 33,108)( 34,114)( 35,106)( 36,111)( 37,115)( 38,118)( 39,109)
    ( 40,112)( 41,120)( 42,122)( 43,113)( 44,123)( 45,116)( 46,121)( 47,117)
    ( 48,119)( 49,125)( 50,124)( 52, 53)( 54, 55)( 57, 58)( 59, 60)( 61, 62)
    ( 63, 64)( 65, 66)( 67, 69)( 70, 71)( 73, 74)( 76,126)( 77,130)( 78,128)
    ( 79,133)( 80,127)( 81,134)( 82,132)( 83,129)( 84,131)( 85,138)( 86,141)
    ( 87,137)( 88,135)( 89,143)( 90,146)( 91,136)( 92,149)( 93,139)( 94,147)
    ( 95,145)( 96,140)( 97,144)( 98,150)( 99,142)(100,148) ];

autGroups[65]:=
[ (  1, 32, 86, 94, 96, 72, 42, 47, 59, 54, 80, 92, 90, 87, 16)
    (  2, 31, 19, 29, 73,  5, 27, 40, 11, 21, 81, 13, 35, 15,  3)
    (  6, 28, 55)(  7, 24, 38, 79, 64, 52, 71, 39,  8, 23, 56, 46, 97, 58, 44
     )(  9, 34, 98, 88, 99, 18, 25, 49, 93, 62, 82, 74, 50, 89, 61)
    ( 10, 33, 69, 77, 17, 26, 66, 48, 75, 53, 65, 41, 83, 60, 67)
    ( 12, 22, 43)( 14, 36, 95)( 20, 30, 91)( 37, 51, 63)( 45, 85, 57)
    ( 68, 78, 70)( 76, 84,100), (  2, 57, 26)(  3, 10, 45)(  4, 64, 52)
    (  5, 21, 37)(  6, 13, 46)(  7, 54, 25)(  8, 20, 34)(  9, 42, 23)
    ( 11, 74, 51)( 12, 80, 44)( 14, 41, 38)( 15, 94, 33)( 16, 68, 32)
    ( 17, 90, 31)( 18, 36, 24)( 19, 92, 43)( 22, 58, 29)( 27, 63, 82)
    ( 28, 56, 72)( 30, 39, 53)( 35, 55, 96)( 40, 62, 50)( 47, 61, 84)
    ( 48, 60, 88)( 49, 59, 76)( 65, 71, 79)( 66, 85, 67)( 69, 86, 70)
    ( 73, 97, 81)( 75, 98, 91)( 77, 78, 87)( 83, 95, 99)( 89, 93,100), 
  (  2,  3)(  5, 21)(  6, 22)(  7, 23)(  8, 24)(  9, 25)( 10, 26)( 11, 27)
    ( 12, 28)( 13, 29)( 14, 30)( 15, 31)( 16, 32)( 17, 33)( 18, 34)( 19, 35)
    ( 20, 36)( 38, 39)( 41, 53)( 42, 54)( 43, 55)( 44, 56)( 45, 57)( 46, 58)
    ( 47, 59)( 48, 60)( 49, 61)( 50, 62)( 51, 63)( 52, 64)( 66, 67)( 69, 77)
    ( 70, 78)( 71, 79)( 72, 80)( 73, 81)( 74, 82)( 75, 83)( 76, 84)( 86, 87)
    ( 89, 93)( 90, 94)( 91, 95)( 92, 96)( 98, 99) ];

autGroups[66]:=
[ ( 1, 9)( 2,34,23,10,37,14,13,38,11,22,35, 3,25,26)( 4,31,20, 7,40,17,16,41,
     8,19,32, 5,28,29)( 6,15)(12,21)(18,27)(24,33)(30,39)(36,42), 
  ( 1,29,38,18,11,20)( 2, 8,15,41,34,30)( 3,21,37,28,36, 7)( 4,27,31,22,42,13
     )( 5,14, 9,35,40,24)( 6,23,32,12,17,26)(10,33,25,16,39,19) ];

autGroups[67]:=
[ ( 1,54,32,40, 5,61,27,46, 2,56,31,38, 6,63,28,48)( 3,53,30,39, 7,62,25,45,
      4,55,29,37, 8,64,26,47)( 9,50,22,33,13,57,17,43,10,52,21,35,14,59,18,41
     )(11,49,24,34,15,58,19,44,12,51,23,36,16,60,20,42)(65,80,68,77)
    (66,78,67,79)(69,72)(74,75)(81,89,94,85,84,92,95,88)(82,91,93,87,83,90,
     96,86), ( 1,96,47,19,68,61,14,82,33,32,79,51, 4,93,46,18,65,64,15,83,36,
     29,78,50)( 2,95,48,20,67,62,13,81,34,31,80,52, 3,94,45,17,66,63,16,84,
     35,30,77,49)( 5,89,44,21,72,60, 9,88,37,25,76,53, 8,92,41,24,69,57,12,
     85,40,28,73,56)( 6,90,43,22,71,59,10,87,38,26,75,54, 7,91,42,23,70,58,
     11,86,39,27,74,55) ];

autGroups[68]:=
[ ( 1,74,53,18,81,46,13,70,60,31,93,35, 2,73,54,17,82,45,14,69,59,32,94,36)
    ( 3,76,55,20,83,48,15,72,58,29,95,33, 4,75,56,19,84,47,16,71,57,30,96,34)
    ( 5,78,49,22,85,42,11,68,62,25,91,37, 6,77,50,21,86,41,12,67,61,26,92,38)
    ( 7,80,51,24,87,44, 9,66,64,27,89,39, 8,79,52,23,88,43,10,65,63,28,90,40)
    , ( 1,83, 3,82)( 2,84, 4,81)( 5,91, 7,90)( 6,92, 8,89)( 9,85,11,88)
    (10,86,12,87)(13,93,15,96)(14,94,16,95)(17,67,19,66)(18,68,20,65)
    (21,74,23,75)(22,73,24,76)(25,71,27,70)(26,72,28,69)(29,78,31,79)
    (30,77,32,80)(33,52,34,51)(35,49,36,50)(37,60,38,59)(39,57,40,58)
    (41,54,42,53)(43,55,44,56)(45,62,46,61)(47,63,48,64) ];

autGroups[69]:=
[ (  2, 13, 91, 31,  3, 16, 94, 28)(  4, 82,148, 23,  8, 86,146, 19)
    (  5, 83,145, 20,  7, 85,149, 22)(  6, 84,151, 26,  9, 87,152, 25)
    ( 10, 88,150, 24, 11, 89,147, 21)( 12, 90,153, 27)( 14, 92,155, 33, 18,
      96,156, 29)( 15, 93,157, 30, 17, 95,154, 32)( 34,103,262,173, 53,119,
     269,159)( 35,104,263,174, 52,118,268,158)( 36,105,264,175, 54,120,270,
     160)( 37, 97,254,177, 56,113,271,167)( 38, 98,253,176, 55,112,272,168)
    ( 39, 99,255,178, 57,114,273,169)( 40,106,256,170, 47,116,275,162)
    ( 41,107,257,171, 46,115,274,161)( 42,108,258,172, 48,117,276,163)
    ( 43,100,260,180, 50,110,265,164)( 44,101,259,179, 49,109,266,165)
    ( 45,102,261,181, 51,111,267,166)( 58,127,286,185, 65,131,281,183)
    ( 59,128,287,186, 64,130,280,182)( 60,129,288,187, 66,132,282,184)
    ( 61,121,278,189, 68,125,283,191)( 62,122,277,188, 67,124,284,192)
    ( 63,123,279,190, 69,126,285,193)( 70,133, 73,139, 79,142, 76,136)
    ( 71,134,291,199, 81,144,294,196)( 72,135,292,198, 80,143,293,197)
    ( 74,140,295,201, 78,138,290,194)( 75,141,296,200, 77,137,289,195)
    (202,205,298,367,391,343,216,203)(204,208,301,369,415,399,345,219)
    (206,297,225,308,335,226,307,215)(207,299,368,393,351,236,310,218)
    (209,300,228,318,375,392,344,217)(210,302,370,417,423,400,346,220)
    (211,305,373,441,450,403,349,223)(212,306,374,442,448,401,347,221)
    (213,303,371,439,449,404,350,224)(214,304,372,440,447,402,348,222)
    (227,309,337,229,317,235,311,336)(230,319,376,394,352,237,312,338)
    (231,329,380,398,362,240,313,339)(232,330,379,396,360,238,315,340)
    (233,327,378,397,361,241,314,341)(234,328,377,395,359,239,316,342)
    (242,321,381,416,405,353,243,320)(244,322,382,418,429,424,406,354)
    (245,333,390,422,458,452,411,355)(246,334,389,420,456,451,412,356)
    (247,331,388,421,457,454,413,357)(248,332,387,419,455,453,414,358)
    (249,325,385,445,466,427,408,364)(250,326,386,446,464,425,407,363)
    (251,323,383,443,465,428,410,366)(252,324,384,444,463,426,409,365)
    (431,459,469,438,434,462,468,435)(432,460,467,436,433,461,470,437)
    (471,473,481,485,486,484,476,472)(474,479,477,482,483,478,480,475), 
  (  1, 19,  4)(  2, 20,  6)(  3, 21,  5)(  7, 28, 40)(  8, 30, 42)
    (  9, 29, 41)( 10, 31, 34)( 11, 33, 36)( 12, 32, 35)( 13, 25, 43)
    ( 14, 26, 45)( 15, 27, 44)( 16, 22, 37)( 17, 23, 39)( 18, 24, 38)
    ( 46, 61, 79)( 47, 63, 80)( 48, 62, 81)( 49, 76, 52)( 50, 78, 54)
    ( 51, 77, 53)( 55, 70, 64)( 56, 72, 66)( 57, 71, 65)( 58, 67, 73)
    ( 59, 69, 74)( 60, 68, 75)( 82,146,204)( 83,147,202)( 84,145,203)
    ( 85,162,210)( 86,163,208)( 87,161,209)( 88,159,207)( 89,160,205)
    ( 90,158,206)( 91,164,103)( 92,166,212)( 93,165,211)( 94,167,106)
    ( 95,169,214)( 96,168,213)( 97,149,220)( 98,150,218)( 99,148,219)
    (100,152,217)(101,153,215)(102,151,216)(104,154,222)(105,155,221)
    (107,156,224)(108,157,223)(109,174,244)(110,175,242)(111,173,243)
    (112,186,237)(113,187,235)(114,185,236)(115,139,188)(116,198,246)
    (117,199,245)(118,142,191)(119,200,248)(120,201,247)(121,171,230)
    (122,172,228)(123,170,229)(124,183,227)(125,184,225)(126,182,226)
    (127,133,176)(128,194,232)(129,195,231)(130,136,179)(131,196,234)
    (132,197,233)(134,178,250)(135,177,249)(137,181,252)(138,180,251)
    (140,190,239)(141,189,238)(143,193,241)(144,192,240)(253,310,286)
    (254,346,325)(255,345,326)(256,271,317)(257,350,319)(258,349,318)
    (259,307,280)(260,344,323)(261,343,324)(262,265,320)(263,348,322)
    (264,347,321)(266,354,305)(267,353,306)(268,283,297)(269,358,299)
    (270,357,298)(272,352,303)(273,351,304)(274,277,300)(275,356,302)
    (276,355,301)(278,338,315)(279,337,316)(281,342,309)(282,341,308)
    (284,336,313)(285,335,314)(287,340,312)(288,339,311)(289,365,329)
    (290,366,330)(291,363,333)(292,364,334)(293,361,327)(294,362,328)
    (295,359,331)(296,360,332)(367,413,432)(368,414,431)(369,411,420)
    (370,412,419)(371,394,436)(372,393,435)(373,406,426)(374,405,425)
    (375,403,434)(376,404,433)(377,398,429)(378,397,430)(379,410,423)
    (380,409,424)(381,401,422)(382,402,421)(383,392,438)(384,391,437)
    (385,400,428)(386,399,427)(387,396,417)(388,395,418)(389,408,415)
    (390,407,416)(439,467,477)(440,468,478)(441,463,475)(442,464,476)
    (443,469,473)(444,470,474)(445,465,471)(446,466,472)(447,483,457)
    (448,484,458)(449,479,461)(450,480,462)(451,485,455)(452,486,456)
    (453,481,459)(454,482,460), (  2,  3)(  4, 19)(  5, 20)(  6, 21)(  7, 22)
    (  8, 23)(  9, 24)( 10, 25)( 11, 26)( 12, 27)( 13, 31)( 14, 33)( 15, 32)
    ( 16, 28)( 17, 30)( 18, 29)( 34, 43)( 35, 44)( 36, 45)( 37, 40)( 38, 41)
    ( 39, 42)( 46, 55)( 47, 56)( 48, 57)( 49, 52)( 50, 53)( 51, 54)( 58, 67)
    ( 59, 68)( 60, 69)( 61, 64)( 62, 65)( 63, 66)( 70, 79)( 71, 81)( 72, 80)
    ( 74, 75)( 77, 78)( 82,146)( 83,145)( 84,147)( 85,149)( 86,148)( 87,150)
    ( 88,152)( 89,151)( 90,153)( 92,155)( 93,154)( 95,157)( 96,156)( 97,162)
    ( 98,161)( 99,163)(100,159)(101,158)(102,160)(103,164)(104,165)(105,166)
    (106,167)(107,168)(108,169)(109,174)(110,173)(111,175)(112,171)(113,170)
    (114,172)(115,176)(116,177)(117,178)(118,179)(119,180)(120,181)(121,186)
    (122,185)(123,187)(124,183)(125,182)(126,184)(127,188)(128,189)(129,190)
    (130,191)(131,192)(132,193)(133,139)(134,199)(135,198)(136,142)(137,201)
    (138,200)(140,195)(141,194)(143,197)(144,196)(202,203)(205,216)(206,215)
    (207,217)(208,219)(209,218)(210,220)(211,222)(212,221)(213,224)(214,223)
    (225,226)(228,236)(229,235)(230,237)(231,239)(232,238)(233,241)(234,240)
    (242,243)(245,250)(246,249)(247,252)(248,251)(253,274)(254,275)(255,276)
    (256,271)(257,272)(258,273)(259,268)(260,269)(261,270)(262,265)(263,266)
    (264,267)(277,286)(278,287)(279,288)(280,283)(281,284)(282,285)(289,295)
    (290,296)(297,307)(298,343)(299,344)(300,310)(301,345)(302,346)(303,350)
    (304,349)(305,348)(306,347)(308,335)(309,336)(311,337)(312,338)(313,342)
    (314,341)(315,340)(316,339)(318,351)(319,352)(321,353)(322,354)(323,358)
    (324,357)(325,356)(326,355)(327,361)(328,362)(329,359)(330,360)(331,365)
    (332,366)(333,363)(334,364)(367,391)(368,392)(369,399)(370,400)(371,404)
    (372,403)(373,402)(374,401)(375,393)(376,394)(377,398)(378,397)(379,396)
    (380,395)(381,405)(382,406)(383,414)(384,413)(385,412)(386,411)(387,410)
    (388,409)(389,408)(390,407)(417,423)(418,424)(419,428)(420,427)(421,426)
    (422,425)(431,438)(432,437)(433,436)(434,435)(439,449)(440,450)(441,447)
    (442,448)(443,453)(444,454)(445,451)(446,452)(455,465)(456,466)(457,463)
    (458,464)(459,469)(460,470)(461,467)(462,468)(471,485)(472,486)(473,481)
    (474,482)(475,483)(476,484)(477,479)(478,480) ];

autGroups[70]:=
[ (  1,110, 41,160, 16,125, 48,138, 26,131, 33,148)(  2,114, 45,155, 18,126,
      47,143, 22,127, 31,149)(  3,115, 43,159, 14,120, 46,144, 24,130, 35,145
     )(  4,109, 40,158, 11,123, 54,137, 27,135, 29,152)(  5,111, 37,162, 15,
     121, 53,136, 21,133, 34,150)(  6,112, 44,154, 12,124, 52,141, 23,129,
      28,153)(  7,116, 39,156, 17,122, 51,139, 19,128, 32,151)
    (  8,113, 42,157, 10,119, 50,142, 25,134, 30,147)(  9,117, 38,161, 13,
     118, 49,140, 20,132, 36,146)( 55, 58, 60, 56)( 57, 59, 62, 61)
    ( 64, 67, 69, 65)( 66, 68, 71, 70)( 73, 76, 78, 74)( 75, 77, 80, 79)
    ( 82,107, 92, 90,100, 98, 83,108, 91, 89,101, 99)( 84,102, 93)
    ( 85,105, 97, 86,103, 96, 88,104, 94, 87,106, 95), 
  (  1,109, 55)(  2,110, 56)(  3,111, 57)(  4,112, 58)(  5,113, 59)
    (  6,114, 60)(  7,115, 61)(  8,116, 62)(  9,117, 63)( 10,118, 64)
    ( 11,119, 65)( 12,120, 66)( 13,121, 67)( 14,122, 68)( 15,123, 69)
    ( 16,124, 70)( 17,125, 71)( 18,126, 72)( 19,127, 73)( 20,128, 74)
    ( 21,129, 75)( 22,130, 76)( 23,131, 77)( 24,132, 78)( 25,133, 79)
    ( 26,134, 80)( 27,135, 81)( 28,136, 82)( 29,137, 83)( 30,138, 84)
    ( 31,139, 85)( 32,140, 86)( 33,141, 87)( 34,142, 88)( 35,143, 89)
    ( 36,144, 90)( 37,145, 91)( 38,146, 92)( 39,147, 93)( 40,148, 94)
    ( 41,149, 95)( 42,150, 96)( 43,151, 97)( 44,152, 98)( 45,153, 99)
    ( 46,154,100)( 47,155,101)( 48,156,102)( 49,157,103)( 50,158,104)
    ( 51,159,105)( 52,160,106)( 53,161,107)( 54,162,108), 
  (  1,110)(  2,109)(  3,111)(  4,114)(  5,115)(  6,112)(  7,113)(  8,116)
    (  9,117)( 10,128)( 11,127)( 12,129)( 13,132)( 14,133)( 15,130)( 16,131)
    ( 17,134)( 18,135)( 19,119)( 20,118)( 21,120)( 22,123)( 23,124)( 24,121)
    ( 25,122)( 26,125)( 27,126)( 28,154)( 29,155)( 30,156)( 31,158)( 32,157)
    ( 33,160)( 34,159)( 35,162)( 36,161)( 37,145)( 38,146)( 39,147)( 40,149)
    ( 41,148)( 42,151)( 43,150)( 44,153)( 45,152)( 46,136)( 47,137)( 48,138)
    ( 49,140)( 50,139)( 51,142)( 52,141)( 53,144)( 54,143)( 55, 56)( 58, 60)
    ( 59, 61)( 64, 74)( 65, 73)( 66, 75)( 67, 78)( 68, 79)( 69, 76)( 70, 77)
    ( 71, 80)( 72, 81)( 82,100)( 83,101)( 84,102)( 85,104)( 86,103)( 87,106)
    ( 88,105)( 89,108)( 90,107)( 94, 95)( 96, 97)( 98, 99) ];

autGroups[71]:=
[ (  3,  5)(  4,  6)(  7, 25, 19, 17, 51, 41)(  8, 26, 20, 18, 52, 42)
    (  9, 29, 21, 15, 53, 39)( 10, 30, 22, 16, 54, 40)( 11, 27, 23, 13, 49,
      37)( 12, 28, 24, 14, 50, 38)( 31, 43, 35, 45, 33, 47)( 32, 44, 36, 46,
      34, 48)( 55,130, 84,123, 75,110, 56,129, 83,124, 76,109)
    ( 57,128, 80,121, 77,114, 58,127, 79,122, 78,113)( 59,132, 82,125, 73,
     112, 60,131, 81,126, 74,111)( 61,140,102,157, 67,134, 62,139,101,158,
      68,133)( 63,144, 98,161, 69,138, 64,143, 97,162, 70,137)
    ( 65,142,100,159, 71,136, 66,141, 99,160, 72,135)( 85,116, 96,147,107,
     152, 86,115, 95,148,108,151)( 87,120, 92,145,103,156, 88,119, 91,146,
     104,155)( 89,118, 94,149,105,154, 90,117, 93,150,106,153), 
  (  1,109, 55)(  2,110, 56)(  3,111, 57)(  4,112, 58)(  5,113, 59)
    (  6,114, 60)(  7,115, 61)(  8,116, 62)(  9,117, 63)( 10,118, 64)
    ( 11,119, 65)( 12,120, 66)( 13,121, 67)( 14,122, 68)( 15,123, 69)
    ( 16,124, 70)( 17,125, 71)( 18,126, 72)( 19,127, 73)( 20,128, 74)
    ( 21,129, 75)( 22,130, 76)( 23,131, 77)( 24,132, 78)( 25,133, 79)
    ( 26,134, 80)( 27,135, 81)( 28,136, 82)( 29,137, 83)( 30,138, 84)
    ( 31,139, 85)( 32,140, 86)( 33,141, 87)( 34,142, 88)( 35,143, 89)
    ( 36,144, 90)( 37,145, 91)( 38,146, 92)( 39,147, 93)( 40,148, 94)
    ( 41,149, 95)( 42,150, 96)( 43,151, 97)( 44,152, 98)( 45,153, 99)
    ( 46,154,100)( 47,155,101)( 48,156,102)( 49,157,103)( 50,158,104)
    ( 51,159,105)( 52,160,106)( 53,161,107)( 54,162,108), 
  (  7, 51)(  8, 52)(  9, 53)( 10, 54)( 11, 49)( 12, 50)( 13, 27)( 14, 28)
    ( 15, 29)( 16, 30)( 17, 25)( 18, 26)( 31, 45)( 32, 46)( 33, 47)( 34, 48)
    ( 35, 43)( 36, 44)( 55,109)( 56,110)( 57,111)( 58,112)( 59,113)( 60,114)
    ( 61,159)( 62,160)( 63,161)( 64,162)( 65,157)( 66,158)( 67,135)( 68,136)
    ( 69,137)( 70,138)( 71,133)( 72,134)( 73,127)( 74,128)( 75,129)( 76,130)
    ( 77,131)( 78,132)( 79,125)( 80,126)( 81,121)( 82,122)( 83,123)( 84,124)
    ( 85,153)( 86,154)( 87,155)( 88,156)( 89,151)( 90,152)( 91,145)( 92,146)
    ( 93,147)( 94,148)( 95,149)( 96,150)( 97,143)( 98,144)( 99,139)(100,140)
    (101,141)(102,142)(103,119)(104,120)(105,115)(106,116)(107,117)(108,118) 
 ];

autGroups[72]:=
[ ( 1,38,70,12,30,79, 3,37,72,11,29,81, 2,39,71,10,28,80)( 4,44,64,18,33,76,
      6,43,66,17,32,78, 5,45,65,16,31,77)( 7,41,67,15,36,73, 9,40,69,14,35,
     75, 8,42,68,13,34,74)(19,54,57,26,46,61,21,53,56,25,48,63,20,52,55,27,
     47,62)(22,51,60,23,49,58,24,50,59), 
  ( 1,46,25,38,16,30, 7,52,22,44,13,36, 4,49,19,41,10,33)( 2,48,26,37,17,29,
      8,54,23,43,14,35, 5,51,20,40,11,32)( 3,47,27,39,18,28, 9,53,24,45,15,
     34, 6,50,21,42,12,31)(55,66,75,59,68,76,61,72,81,56,65,73,58,69,78,62,
     71,79)(57,64,74,60,67,77,63,70,80) ];

autGroups[73]:=
[ ( 1,47,10,33,17,34,25,36,27,54,26,42,22,38, 6,37, 2,43)( 3,51,11,49,16,29,
     21,32, 9,52,15,53,18,40,20,28, 7,44)( 4,39,12,31, 5,48,14,35,19,50,24,
     30,23,46, 8,41,13,45)(55,70,76,69,79,61)(56,64,74,67,68,75)
    (57,71,60,65,78,77)(58,72,80,81,63,59)(62,73), 
  ( 1,55,28)( 2,56,29)( 3,57,30)( 4,58,31)( 5,59,32)( 6,60,33)( 7,61,34)
    ( 8,62,35)( 9,63,36)(10,64,37)(11,65,38)(12,66,39)(13,67,40)(14,68,41)
    (15,69,42)(16,70,43)(17,71,44)(18,72,45)(19,73,46)(20,74,47)(21,75,48)
    (22,76,49)(23,77,50)(24,78,51)(25,79,52)(26,80,53)(27,81,54), 
  ( 1,43)( 2,47)( 3,51)( 4,31)( 5,45)( 6,33)( 7,49)( 8,35)( 9,53)(10,37)
    (11,44)(12,39)(13,48)(14,41)(15,52)(16,28)(17,38)(18,32)(19,46)(20,29)
    (21,40)(22,34)(23,50)(24,30)(25,42)(26,36)(27,54)(55,70)(56,74)(57,78)
    (59,72)(61,76)(63,80)(65,71)(67,75)(69,79) ];

autGroups[74]:=
[ ( 1,58,30,39,68,79,78,31,57,48,41,76,81,72,53,19,49, 2)( 3,36,27,40,18,50)
    ( 4,60, 7,61,73,69)( 5, 9,35,63,75,65,11,20,54,24,17,67,28,13,62,74,66,43
     )( 6, 8,56,51,23,45,25,15,55,59,47,70,14,21,33,29,37,46)(10,71)
    (12,16,44,42,34,32)(22,64,26,38,80,52), 
  ( 1,58,10,53,19,60, 9,55,42,63,37,50, 8,62,44,59,66,69)( 2, 4,49, 5, 7,70,
     14,80,67,76,27,68,11,26,45,25,73,43)( 3,65,79,22,41,28,18,46, 6)
    (12,29,75,36,81,72,71,30,39,40,13,56,32,24,23,38,21,54)(15,35,34,51,17,
     52,78,31,77,57,48,64,20,33,16,74,47,61) ];

autGroups[75]:=
[ (  1,  3)(  2,  4)(  5,500, 16,458, 36,318, 40,454)(  6,499, 15,457, 35,
     317, 39,453)(  7,498, 14,460, 34,320, 38,456)(  8,497, 13,459, 33,319,
      37,455)(  9,495, 21,327, 53,391, 20,491)( 10,496, 22,328, 54,392, 19,
     492)( 11,493, 23,325, 55,389, 18,489)( 12,494, 24,326, 56,390, 17,490)
    ( 25, 42, 61, 86,113,119, 92, 30)( 26, 41, 62, 85,114,120, 91, 29)
    ( 27, 44, 63, 88,115,117, 90, 32)( 28, 43, 64, 87,116,118, 89, 31)
    ( 45,428, 93,480,181,388,137,324)( 46,427, 94,479,182,387,138,323)
    ( 47,426, 95,478,183,386,139,322)( 48,425, 96,477,184,385,140,321)
    ( 49,396, 65,464,129,444,221,272)( 50,395, 66,463,130,443,222,271)
    ( 51,394, 67,462,131,442,223,270)( 52,393, 68,461,132,441,224,269)
    ( 57,264,101,312,217,440,133,420)( 58,263,102,311,218,439,134,419)
    ( 59,262,103,310,219,438,135,418)( 60,261,104,309,220,437,136,417)
    ( 69, 98,141,202,265,206,145, 78)( 70, 97,142,201,266,205,146, 77)
    ( 71,100,143,204,267,208,147, 80)( 72, 99,144,203,268,207,148, 79)
    ( 73,256,149,384,185,424, 84,452)( 74,255,150,383,186,423, 83,451)
    ( 75,254,151,382,187,422, 82,450)( 76,253,152,381,188,421, 81,449)
    (105,370,225,290,337,122,293,234)(106,369,226,289,338,121,294,233)
    (107,372,227,292,339,124,295,236)(108,371,228,291,340,123,296,235)
    (109,306,153,342,285,127,373,162)(110,305,154,341,286,128,374,161)
    (111,308,155,344,287,125,375,164)(112,307,156,343,288,126,376,163)
    (157,169,297,167,303,231,238,359)(158,170,298,168,304,232,237,360)
    (159,171,299,165,301,229,240,357)(160,172,300,166,302,230,239,358)
    (173,242,283,189,246,274,278,178)(174,241,284,190,245,273,277,177)
    (175,244,281,191,248,276,280,180)(176,243,282,192,247,275,279,179)
    (193,315,214,200,258,252,377,211)(194,316,213,199,257,251,378,212)
    (195,313,216,198,260,250,379,209)(196,314,215,197,259,249,380,210)
    (329,334,402,345)(330,333,401,346)(331,336,404,347)(332,335,403,348)
    (349,414,469,488,483,368,363,354)(350,413,470,487,484,367,364,353)
    (351,416,471,486,481,366,361,356)(352,415,472,485,482,365,362,355)
    (397,409,466,433,430,446,473,406)(398,410,465,434,429,445,474,405)
    (399,411,468,435,432,448,475,408)(400,412,467,436,431,447,476,407)
    (501,504)(502,503), (  1, 35,319)(  2, 36,320)(  3, 33,317)(  4, 34,318)
    (  5,478,267)(  6,477,268)(  7,480,265)(  8,479,266)(  9, 79,440)
    ( 10, 80,439)( 11, 77,438)( 12, 78,437)( 13,383,118)( 14,384,117)
    ( 15,381,120)( 16,382,119)( 17,364,441)( 18,363,442)( 19,362,443)
    ( 20,361,444)( 21,353,390)( 22,354,389)( 23,355,392)( 24,356,391)
    ( 25,358,239)( 26,357,240)( 27,360,237)( 28,359,238)( 29,286,159)
    ( 30,285,160)( 31,288,157)( 32,287,158)( 37,457,114)( 38,458,113)
    ( 39,459,116)( 40,460,115)( 41,229,227)( 42,230,228)( 43,231,225)
    ( 44,232,226)( 45,324,190)( 46,323,189)( 47,322,192)( 48,321,191)
    ( 49,467,395)( 50,468,396)( 51,465,393)( 52,466,394)( 53,501,326)
    ( 54,502,325)( 55,503,328)( 56,504,327)( 57,407,263)( 58,408,264)
    ( 59,405,261)( 60,406,262)( 61,371,388)( 62,372,387)( 63,369,386)
    ( 64,370,385)( 65,411,252)( 66,412,251)( 67,409,250)( 68,410,249)
    ( 69,490,132)( 70,489,131)( 71,492,130)( 72,491,129)( 73,452,173)
    ( 74,451,174)( 75,450,175)( 76,449,176)( 81,214,243)( 82,213,244)
    ( 83,216,241)( 84,215,242)( 85,182,455)( 86,181,456)( 87,184,453)
    ( 88,183,454)( 89,150,126)( 90,149,125)( 91,152,128)( 92,151,127)
    ( 93,195,206)( 94,196,205)( 95,193,208)( 96,194,207)( 97,462,260)
    ( 98,461,259)( 99,464,258)(100,463,257)(101,475,123)(102,476,124)
    (103,473,121)(104,474,122)(105,248,140)(106,247,139)(107,246,138)
    (108,245,137)(109,346,429)(110,345,430)(111,348,431)(112,347,432)
    (133,148,212)(134,147,211)(135,146,210)(136,145,209)(141,197,424)
    (142,198,423)(143,199,422)(144,200,421)(153,223,368)(154,224,367)
    (155,221,366)(156,222,365)(161,279,329)(162,280,330)(163,277,331)
    (164,278,332)(165,300,488)(166,299,487)(167,298,486)(168,297,485)
    (169,343,482)(170,344,481)(171,341,484)(172,342,483)(177,376,255)
    (178,375,256)(179,374,253)(180,373,254)(185,498,202)(186,497,201)
    (187,500,204)(188,499,203)(217,350,495)(218,349,496)(219,352,493)
    (220,351,494)(233,334,275)(234,333,276)(235,336,273)(236,335,274)
    (269,305,433)(270,306,434)(271,307,435)(272,308,436)(281,316,425)
    (282,315,426)(283,314,427)(284,313,428)(289,304,472)(290,303,471)
    (291,302,470)(292,301,469)(293,445,401)(294,446,402)(295,447,403)
    (296,448,404)(309,337,416)(310,338,415)(311,339,414)(312,340,413)
    (377,399,419)(378,400,420)(379,397,417)(380,398,418), 
  (  1,  3)(  2,  4)(  5,500)(  6,499)(  7,498)(  8,497)(  9,491)( 10,492)
    ( 11,489)( 12,490)( 13,455)( 14,456)( 15,453)( 16,454)( 17,494)( 18,493)
    ( 19,496)( 20,495)( 21,391)( 22,392)( 23,389)( 24,390)( 25, 27)( 26, 28)
    ( 29, 43)( 30, 44)( 31, 41)( 32, 42)( 33,319)( 34,320)( 35,317)( 36,318)
    ( 37,459)( 38,460)( 39,457)( 40,458)( 45,452)( 46,451)( 47,450)( 48,449)
    ( 49,264)( 50,263)( 51,262)( 52,261)( 53,327)( 54,328)( 55,325)( 56,326)
    ( 57,396)( 58,395)( 59,394)( 60,393)( 61, 90)( 62, 89)( 63, 92)( 64, 91)
    ( 65,420)( 66,419)( 67,418)( 68,417)( 69, 78)( 70, 77)( 71, 80)( 72, 79)
    ( 73,324)( 74,323)( 75,322)( 76,321)( 81,425)( 82,426)( 83,427)( 84,428)
    ( 85,118)( 86,117)( 87,120)( 88,119)( 93,424)( 94,423)( 95,422)( 96,421)
    ( 97,146)( 98,145)( 99,148)(100,147)(101,272)(102,271)(103,270)(104,269)
    (105,374)(106,373)(107,376)(108,375)(109,294)(110,293)(111,296)(112,295)
    (113,115)(114,116)(121,306)(122,305)(123,308)(124,307)(125,371)(126,372)
    (127,369)(128,370)(129,440)(130,439)(131,438)(132,437)(133,464)(134,463)
    (135,462)(136,461)(137,256)(138,255)(139,254)(140,253)(141,206)(142,205)
    (143,208)(144,207)(149,388)(150,387)(151,386)(152,385)(153,338)(154,337)
    (155,340)(156,339)(157,229)(158,230)(159,231)(160,232)(161,234)(162,233)
    (163,236)(164,235)(165,297)(166,298)(167,299)(168,300)(169,301)(170,302)
    (171,303)(172,304)(173,190)(174,189)(175,192)(176,191)(177,246)(178,245)
    (179,248)(180,247)(181,384)(182,383)(183,382)(184,381)(185,480)(186,479)
    (187,478)(188,477)(193,199)(194,200)(195,197)(196,198)(201,266)(202,265)
    (203,268)(204,267)(209,259)(210,260)(211,257)(212,258)(213,315)(214,316)
    (215,313)(216,314)(217,444)(218,443)(219,442)(220,441)(221,312)(222,311)
    (223,310)(224,309)(225,286)(226,285)(227,288)(228,287)(237,358)(238,357)
    (239,360)(240,359)(241,283)(242,284)(243,281)(244,282)(249,379)(250,380)
    (251,377)(252,378)(273,278)(274,277)(275,280)(276,279)(289,342)(290,341)
    (291,344)(292,343)(329,333)(330,334)(331,335)(332,336)(345,401)(346,402)
    (347,403)(348,404)(349,362)(350,361)(351,364)(352,363)(353,356)(354,355)
    (365,414)(366,413)(367,416)(368,415)(397,410)(398,409)(399,412)(400,411)
    (405,466)(406,465)(407,468)(408,467)(429,446)(430,445)(431,448)(432,447)
    (433,474)(434,473)(435,476)(436,475)(469,482)(470,481)(471,484)(472,483)
    (485,488)(486,487)(501,504)(502,503) ];


autGroups[76]:=
[ (  2,  3)(  5, 10, 20, 31, 53, 47, 28, 14)(  6, 12, 18, 29, 55, 48, 27, 13)
    (  7,  9, 19, 32, 54, 45, 26, 16)(  8, 11, 17, 30, 56, 46, 25, 15)
    ( 21, 34, 60, 74, 84, 77, 62, 40)( 22, 33, 58, 76, 82, 79, 64, 39)
    ( 23, 36, 59, 73, 81, 78, 61, 38)( 24, 35, 57, 75, 83, 80, 63, 37)
    ( 41, 67, 52, 71, 43, 65, 50, 72)( 42, 68, 51, 70, 44, 66, 49, 69)
    ( 85,169,253,346, 94,173,257,337)( 86,171,255,345, 93,175,259,338)
    ( 87,170,254,348, 96,174,258,339)( 88,172,256,347, 95,176,260,340)
    ( 89,182,266,342, 90,181,265,341)( 91,184,268,344, 92,183,267,343)
    ( 97,195,279,358,106,207,291,349)( 98,196,280,357,105,208,292,350)
    ( 99,193,277,360,108,205,289,351)(100,194,278,359,107,206,290,352)
    (101,179,263,354,102,180,264,353)(103,177,261,356,104,178,262,355)
    (109,214,298,385,133,234,318,361)(110,213,297,386,134,233,317,362)
    (111,216,300,387,135,236,320,363)(112,215,299,388,136,235,319,364)
    (113,186,270,369,117,190,274,365)(114,185,269,370,118,189,273,366)
    (115,188,272,371,119,192,276,367)(116,187,271,372,120,191,275,368)
    (121,231,315,377,125,240,324,373)(122,229,313,378,126,237,321,374)
    (123,232,316,379,127,239,323,375)(124,230,314,380,128,238,322,376)
    (129,222,306,381)(130,224,308,382)(131,221,305,383)(132,223,307,384)
    (137,199,283,401,149,211,295,389)(138,200,284,402,150,212,296,390)
    (139,197,281,403,151,209,293,391)(140,198,282,404,152,210,294,392)
    (141,203,287,406,154,219,303,393)(142,201,285,405,153,217,301,394)
    (143,204,288,407,155,220,304,395)(144,202,286,408,156,218,302,396)
    (145,246,330,398,146,245,329,397)(147,248,332,400,148,247,331,399)
    (157,227,311,410,158,228,312,409)(159,225,309,412,160,226,310,411)
    (161,252,336,419,167,243,327,413)(162,249,333,417,165,241,325,414)
    (163,250,334,418,166,244,328,415)(164,251,335,420,168,242,326,416)
    (421,425,433,459,491,456,439,430)(422,427,435,457,492,454,437,432)
    (423,426,434,460,490,455,440,429)(424,428,436,458,489,453,438,431)
    (441,448,487,449,442,447,488,450)(443,446,485,451,444,445,486,452)
    (461,483,499,496,477,471,468,475)(462,481,497,495,478,469,466,476)
    (463,484,500,494,479,472,467,473)(464,482,498,493,480,470,465,474)
    (503,504), (  1, 94,173)(  2, 93,174)(  3, 96,175)(  4, 95,176)
    (  5,149,221)(  6,151,223)(  7,150,222)(  8,152,224)(  9,129,233)
    ( 10,131,235)( 11,130,234)( 12,132,236)( 13,117,209)( 14,119,211)
    ( 15,118,210)( 16,120,212)( 17,133,205)( 18,135,207)( 19,134,206)
    ( 20,136,208)( 21,164,243)( 22,163,244)( 23,162,241)( 24,161,242)
    ( 25,102,189)( 26,104,191)( 27,101,190)( 28,103,192)( 29,106,182)
    ( 30,108,184)( 31,105,181)( 32,107,183)( 33,166,248)( 34,167,246)
    ( 35,168,247)( 36,165,245)( 37,160,252)( 38,158,249)( 39,159,250)
    ( 40,157,251)( 41,123,199)( 42,124,200)( 43,121,197)( 44,122,198)
    ( 45, 86,178)( 46, 88,180)( 47, 85,177)( 48, 87,179)( 49,111,203)
    ( 50,112,204)( 51,109,201)( 52,110,202)( 53, 90,169)( 54, 92,171)
    ( 55, 89,170)( 56, 91,172)( 57,148,240)( 58,147,239)( 59,146,237)
    ( 60,145,238)( 61,155,228)( 62,156,227)( 63,154,226)( 64,153,225)
    ( 65,139,215)( 66,140,216)( 67,137,213)( 68,138,214)( 69,141,230)
    ( 70,142,229)( 71,144,231)( 72,143,232)( 73,126,194)( 74,128,196)
    ( 75,125,193)( 76,127,195)( 77,114,218)( 78,116,220)( 79,113,217)
    ( 80,115,219)( 81,100,187)( 82, 97,186)( 83, 99,188)( 84, 98,185)
    (253,342,456)(254,341,455)(255,344,454)(256,343,453)(257,346,491)
    (258,345,492)(259,348,490)(260,347,489)(261,337,459)(262,338,460)
    (263,339,457)(264,340,458)(265,357,440)(266,358,439)(267,359,438)
    (268,360,437)(269,350,442)(270,349,441)(271,352,444)(272,351,443)
    (273,354,435)(274,353,436)(275,356,433)(276,355,434)(277,377,432)
    (278,378,431)(279,379,430)(280,380,429)(281,373,477)(282,374,478)
    (283,375,479)(284,376,480)(285,361,481)(286,362,482)(287,363,483)
    (288,364,484)(289,385,446)(290,386,445)(291,387,448)(292,388,447)
    (293,369,450)(294,370,449)(295,371,452)(296,372,451)(297,389,486)
    (298,390,485)(299,391,488)(300,392,487)(301,365,428)(302,366,427)
    (303,367,426)(304,368,425)(305,401,494)(306,402,493)(307,403,496)
    (308,404,495)(309,405,462)(310,406,461)(311,408,464)(312,407,463)
    (313,394,424)(314,393,423)(315,396,422)(316,395,421)(317,381,498)
    (318,382,497)(319,383,500)(320,384,499)(321,398,469)(322,397,470)
    (323,399,472)(324,400,471)(325,414,501)(326,413,503)(327,416,504)
    (328,415,502)(329,417,465)(330,419,466)(331,420,467)(332,418,468)
    (333,410,474)(334,411,475)(335,409,473)(336,412,476), 
  (  5, 10)(  6,  9)(  7, 12)(  8, 11)( 13, 19)( 14, 20)( 15, 17)( 16, 18)
    ( 21, 24)( 22, 23)( 25, 30)( 26, 29)( 27, 32)( 28, 31)( 33, 38)( 34, 37)
    ( 35, 40)( 36, 39)( 41, 50)( 42, 49)( 43, 52)( 44, 51)( 45, 55)( 46, 56)
    ( 47, 53)( 48, 54)( 57, 62)( 58, 61)( 59, 64)( 60, 63)( 65, 67)( 66, 68)
    ( 73, 79)( 74, 80)( 75, 77)( 76, 78)( 81, 82)( 83, 84)( 85,169)( 86,170)
    ( 87,171)( 88,172)( 89,178)( 90,177)( 91,180)( 92,179)( 93,174)( 94,173)
    ( 95,176)( 96,175)( 97,187)( 98,188)( 99,185)(100,186)(101,183)(102,184)
    (103,181)(104,182)(105,192)(106,191)(107,190)(108,189)(109,198)(110,197)
    (111,200)(112,199)(113,194)(114,193)(115,196)(116,195)(117,206)(118,205)
    (119,208)(120,207)(121,202)(122,201)(123,204)(124,203)(125,218)(126,217)
    (127,220)(128,219)(129,223)(130,224)(131,221)(132,222)(133,210)(134,209)
    (135,212)(136,211)(137,215)(138,216)(139,213)(140,214)(141,230)(142,229)
    (143,232)(144,231)(145,226)(146,225)(147,228)(148,227)(149,235)(150,236)
    (151,233)(152,234)(153,237)(154,238)(155,239)(156,240)(157,247)(158,248)
    (159,245)(160,246)(161,243)(162,244)(163,241)(164,242)(165,250)(166,249)
    (167,252)(168,251)(253,337)(254,338)(255,339)(256,340)(257,346)(258,345)
    (259,348)(260,347)(261,342)(262,341)(263,344)(264,343)(265,355)(266,356)
    (267,353)(268,354)(269,351)(270,352)(271,349)(272,350)(273,360)(274,359)
    (275,358)(276,357)(277,366)(278,365)(279,368)(280,367)(281,362)(282,361)
    (283,364)(284,363)(285,374)(286,373)(287,376)(288,375)(289,370)(290,369)
    (291,372)(292,371)(293,386)(294,385)(295,388)(296,387)(297,391)(298,392)
    (299,389)(300,390)(301,378)(302,377)(303,380)(304,379)(305,383)(306,384)
    (307,381)(308,382)(309,398)(310,397)(311,400)(312,399)(313,394)(314,393)
    (315,396)(316,395)(317,403)(318,404)(319,401)(320,402)(321,405)(322,406)
    (323,407)(324,408)(325,415)(326,416)(327,413)(328,414)(329,411)(330,412)
    (331,409)(332,410)(333,418)(334,417)(335,420)(336,419)(425,430)(426,429)
    (427,432)(428,431)(433,439)(434,440)(435,437)(436,438)(441,444)(442,443)
    (445,450)(446,449)(447,452)(448,451)(453,458)(454,457)(455,460)(456,459)
    (461,470)(462,469)(463,472)(464,471)(465,475)(466,476)(467,473)(468,474)
    (477,482)(478,481)(479,484)(480,483)(485,487)(486,488)(493,499)(494,500)
    (495,497)(496,498)(501,502)(503,504) ];

autGroups[77]:=
[ ( 1, 4,16,39,57,67,71,65,41,25, 9, 3)( 2, 7,34,40,17, 5)( 6,12,29, 8,10,28,
     61,51,50,54,53,21)(11,15,22,23,32,64,47,46,55,56,24,30)(13,20,43,14,18,
     38)(19,48,52,27,35,49,60,68,72,66,44,31)(26,42,37,36,45,33)
    (58,63,70,59,62,69), ( 1,59,23,69,57,43,61, 5,29,13)( 2, 9,36, 6,62,50,
     42,71,17,56)( 3,67)( 4,39, 8,47,21,65,25,54,11,28)( 7,53,40,19,26,30,63,
     46,45,72)(10,70,60,38,64,34,15,14,52,58)(12,32,27,49,22,51,24,66,31,55)
    (16,33)(18,41)(20,44)(35,37)(48,68) ];

autGroups[78]:=
[ (  1, 68,177,178,100,  7)(  2,  4, 88, 76, 74, 47)(  3, 62,102,224,202, 30)
    (  5,170,162,257,219, 19)(  6, 53,212,108,247, 99)(  8,190,228,220,180,
       9)( 10, 84, 34,217, 48, 39)( 11,147, 50, 91, 64, 65)( 12,205,193,173,
      36, 54)( 13, 25, 16, 15, 33, 35)( 14,186,213,113,116,246)
    ( 17, 73,132,188,189, 55)( 18, 96,253,225,149,160)( 20,157,181,236,142,
     140)( 21,150,183, 45,161, 87)( 22, 89,130,131,125,158)( 23,155,112, 51,
     176,211)( 24, 46,240,256, 97, 37)( 26,164,261,237,243, 56)
    ( 27,138,134,196,195,208)( 28, 29,263,114,241,101)( 31, 82,167,206, 77,
      42)( 32, 41,259)( 38, 95,115, 93,109,201)( 40,165,207,107,232,230)
    ( 43,234, 75,244,118,121)( 44, 85, 49,152,139,168)( 52,144,218, 81, 79,
     171)( 57,231,204,128,133,105)( 58, 98,197,185,143,239)( 59,255,146,145,
      80,119)( 60,249,221,238,226,111)( 61,110,251,120,209,227)
    ( 63,214,153,122,123,106)( 66,242, 67,264,148,127)( 69,192, 72, 71,141,
     103)( 70, 90,124,117,245,260)( 78, 83,179,250,216,191)( 86,233,254,126,
     235,252)( 92,175,163)( 94,182,154,172,200,199)(104,129,215,184,248,174)
    (135,223,262,194,156,203)(136,159,229,198,222,151)(137,210,187)
    (166,258,169), (  1,127,150,191, 73, 61,152,180)(  2, 96,235, 97, 52,207,
     249,219)(  3,264,247, 55,133,209,248,100)(  4,242,109,171,144,251, 63,
      47)(  5, 39, 68,258, 24,157,132,259)(  6,217,176, 56,104,140,155,101)
    (  7, 62,179,115, 17,105,190,123)(  8,225,240,174, 83,232,162, 99)
    (  9,186, 77, 22, 78, 90, 82, 50)( 10, 43,222,143,181,146,156, 36)
    ( 11,160, 41,204,125, 40,166,202)( 12,113,263,214, 58,117,261,201)
    ( 13,124, 86,178, 69,213,238,189)( 14, 92,149, 44, 70,210,230,161)
    ( 15, 54,182,103, 71,239, 27, 35)( 16, 57,254,241, 72,102,111,243)
    ( 18,205,195, 19,165, 98,199, 37)( 20,118,220, 79, 48, 59,216, 74)
    ( 21,169,119,233, 49, 32,244,226)( 23, 76, 42,203, 51, 81,167,151)
    ( 25, 84,175,130,192,236,187, 64)( 26,121,170,137, 28,255, 46,163)
    ( 29,173, 89,148,164,185, 91,227)( 30, 53, 75, 38,128,129, 80,153)
    ( 31,224,197,139,206,231,193,183)( 33,107,229,257,141,253,262,256)
    ( 34,172,260,122,142,134,246, 95)( 45,131,154,135,168, 65,138,136)
    ( 60,108,250,221,126,184,228,252)( 66,116,114,208,110,245,237, 94)
    ( 67,223,198,106,120,159,194, 93)( 85,158, 88,215, 87,147,218,212)
    (112,234,200,177,211,145,196,188) ];

autGroups[79]:=
[ (  1,  2,252,125,127,161,164,165)(  3,181, 12,117,229,159,168,243)
    (  4,232,193,126,222, 75, 59,257)(  5,253,102, 79,228,136,167,207)
    (  6, 82,264,123,230,163,169,259)(  7,255,239,110, 68,162, 92,246)
    (  8,182,262,120,192, 40,152, 47)(  9,108, 61,122, 87,158,170,260)
    ( 10,254,176,124,194,148,166, 74)( 11,119,250,121,225,160,133,177)
    ( 13,114,251,216,111, 67,132, 86)( 14,115, 72, 51,224,173,128,179)
    ( 15, 17,118,217,218,227,135,129)( 16, 90,248,212,226,241,109, 60)
    ( 18,116,249, 98, 78,242,134,178)( 19, 27,184,219,203,189, 35,154)
    ( 20,112, 49, 24,183,244,131,180)( 21,113,145,215, 70, 56,130,171)
    ( 22,185,137,201,190, 33,151, 46)( 23, 94,245,200,188, 38,106, 41)
    ( 25, 96, 85,199,142, 37,153, 43)( 26, 53,256,175,191, 34,149, 42)
    ( 28,187,263,202,104, 39,156, 45)( 29,174,231,197,146, 32, 77, 48)
    ( 30,155, 44, 31,186, 36,157, 50)( 52, 89,172,210,196,143,103, 66)
    ( 54,223, 69, 58,195,144,221, 76)( 55,101, 81,233,140,150,208, 71)
    ( 57,258,139,234,198,147,220, 73)( 62, 88, 91,211,209,240,107, 65)
    ( 63, 99, 83,247,138,235,206,105)( 64,214, 93,236,205,213,238,100)
    ( 80,204,141,237, 97, 84,261, 95), (  1,199, 17,240,127, 43,227, 88)
    (  2,201,115,263,161, 46,173,156)(  3, 31,114,196,229, 50, 67, 52)
    (  4,157,119,235,222, 44,160, 99)(  5,203, 90,260,228, 19,241,122)
    (  6,175,112,255,230, 42,244,162)(  7,197,118,188, 68, 48,135, 23)
    (  8,200,113,251,192, 41, 56,132)(  9,120,116,226, 87, 47,242, 16)
    ( 10,131, 27,209,194, 49,189, 62)( 11,202,117,213,225, 45,243,214)
    ( 12,143,110, 84,168, 89,246,204)( 13,206,125,147,111, 83,165,258)
    ( 14,107, 79,169,224, 91,207,264)( 15,238,124, 33,218, 93, 74,185)
    ( 18,239,126,105, 78, 92,257,247)( 20,216,123, 71,183, 86,259,233)
    ( 21, 38,121,141, 70, 94,177,261)( 22,170,195,248,190, 61, 54,109)
    ( 24,138, 76,184,180, 63, 58, 35)( 25,210, 34,217,142, 66, 53,129)
    ( 26,187,179,249,191, 39, 51,134)( 28,205,193,145,104, 64, 59,130)
    ( 29,211,198,250,146, 65, 57,133)( 30,212,140, 72,186, 60, 55,128)
    ( 32,219,136, 73,174,154,253,234)( 36,159, 95, 69,155,181,237,221)
    ( 37,215,137, 75, 96,171,151,232)( 40, 98,139, 77,182,178,220,231)
    ( 80,166,256,158, 97,176,149,108)( 81,152,254,236,208,262,148,100)
    ( 82,167,223,245,163,102,144,106)( 85,164,101,172,153,252,150,103) ];

autGroups[80]:=
[ ( 1,35,18,37,36,19,39, 2,21,17)( 3,20,38,34,16)( 4,14,32,40,22)
    ( 5,15,23, 6,41,24,31,42,13,33)( 7,27,11, 9,29,12,43,30,25,45)
    ( 8,26,44,28,10), ( 1,37, 4,27,13,20)( 2,32,12,41,16,39)( 3,21,19,14, 9,
     31)( 5,38,18,35,22,45)( 6,43,10,33, 7,26)( 8,15,25)(11,44,24,29,28,42)
    (17,40,30,23,34,36) ];

autGroups[81]:=
[ (  1,143, 76, 36,150, 42,164,114, 52, 70,122, 97,175, 85)(  2,140,125,163,
     113, 40, 72,166, 58,178,118, 16,174, 73)(  3, 31,127,137,155, 41,109,
      68, 59,170,124, 95, 22, 90)(  4, 66,129,171,159, 43, 25,167, 51, 35,
     119, 94,161, 83)(  5,103, 65,179,156, 49,149, 21, 60, 80, 29, 96,135, 87
     )(  6,141,111,108, 67, 47,172,168, 55, 23,123, 74, 34, 86)
    (  7, 19,130, 81, 32, 44,138,160, 53,107, 64, 98,169, 91)
    (  8,132,128,116,104, 38,180,165, 61,147, 17, 99, 79, 27)
    (  9,112,102, 71,157, 46,182,145, 14,177, 75, 28,152, 84)
    ( 10,144,126, 24,158, 39, 37,151, 54,162,110, 92, 69, 89)
    ( 11,142,131,148, 20, 48, 82, 33, 56,136,120, 93,106, 62)
    ( 12, 77, 30,153,133, 45,117,105, 50,176,121,100,146, 15)
    ( 13,139, 18,181, 78, 26,154,134, 57,115,101, 63,173, 88), 
  (  1,174,105, 38,117, 73)(  2,158,151, 50,149, 80)(  3, 12,167, 68, 72, 82)
    (  4, 48,114, 62, 25, 79)(  5, 90, 21, 69,109, 39)(  6,143,160, 64,180,
      74)(  7,177, 33, 67,154, 51)(  8, 60,145, 70, 37, 78)(  9,123,166, 63,
     138, 77)( 10,130,165, 65,172, 76)( 11, 99,134, 66,164, 75)
    ( 13,181,168, 71,182, 81)( 14,146,107, 26,113, 86)( 15, 22,106, 40, 41,
      83)( 16,135,103, 45,110, 89)( 17,173, 92, 42,112, 87)( 18,169,102, 46,
     111, 88)( 19,161,101, 47, 93, 84)( 20,152, 52, 43,115, 27)
    ( 23, 34,104, 44, 53, 85)( 24,175,108, 49,116, 91)( 28,133,140, 57,118,
     176)( 29,157, 96, 54,139,162)( 30,156,127, 58,126,170)( 31,155,119, 59,
      95,136)( 32,150, 56, 55,147, 35)( 36,159,153, 61,148,178)
    ( 94,132,141,120,122, 98)( 97,125,142,128,121,129)(100,137,144,163,124,
     179)(131,171) ];

autGroups[82]:=
[ (  1,112,128, 38, 51, 70,  2,111,127, 37, 52, 69)(  3,119,172,156, 15, 96,
     149,159, 22,181,144,113)(  4,120,171,155, 16, 95,150,160, 21,182,143,114
     )(  5,179, 26, 60, 19, 84,166,134, 74,136,123,115)(  6,180, 25, 59, 20,
      83,165,133, 73,135,124,116)(  7,100,137,110, 50,177,167, 72,106, 88,
      33, 30)(  8, 99,138,109, 49,178,168, 71,105, 87, 34, 29)
    (  9, 42,151,170, 44, 63, 53, 57, 61, 12,104, 81)( 10, 41,152,169, 43,
      64, 54, 58, 62, 11,103, 82)( 13,176, 36,148, 32, 79, 97,153, 48,121,
     117,162)( 14,175, 35,147, 31, 80, 98,154, 47,122,118,161)
    ( 17,126,132, 86, 75, 28, 78, 66, 90,139,163,130)( 18,125,131, 85, 76,
      27, 77, 65, 89,140,164,129)( 23,102, 68,173, 39, 93,141,157, 91, 55,
     108,145)( 24,101, 67,174, 40, 94,142,158, 92, 56,107,146), 
  (  1,159,145, 53,122, 50, 19, 99,174, 33, 35,151, 73,  3)(  2,160,146, 54,
     121, 49, 20,100,173, 34, 36,152, 74,  4)(  5,157,130, 62,120,134, 51,
      98, 79,128, 23,149,  9, 76)(  6,158,129, 61,119,133, 52, 97, 80,127,
      24,150, 10, 75)(  7,138, 60, 26, 12,164, 15, 66, 89,155, 28, 82, 93, 39
     )(  8,137, 59, 25, 11,163, 16, 65, 90,156, 27, 81, 94, 40)
    ( 13,126, 92,116,131,161,135, 42, 96,140, 78,171, 58,107)
    ( 14,125, 91,115,132,162,136, 41, 95,139, 77,172, 57,108)
    ( 17,141,103,180, 86,123,117, 64,105, 71,168,169,176, 56)
    ( 18,142,104,179, 85,124,118, 63,106, 72,167,170,175, 55)
    ( 21, 70,166,147, 88, 44,177, 47,101,111,114,143, 46,182)
    ( 22, 69,165,148, 87, 43,178, 48,102,112,113,144, 45,181)
    ( 29, 68,153, 37, 31, 84,110)( 30, 67,154, 38, 32, 83,109) ];

autGroups[83]:=
[ (  1, 66,330,183,197,350,345,304,272,280,159,153,231, 84, 81,114,118,267,
     371, 31)(  2, 65,329,184,198,349,346,303,271,279,160,154,232, 83, 82,
     113,117,268,372, 32)(  3, 48,132,256,196,374, 26, 69,299,275,188,170,
     311,243, 89,204,130,147,321,241)(  4, 47,131,255,195,373, 25, 70,300,
     276,187,169,312,244, 90,203,129,148,322,242)(  5, 51,366,331,  9, 73,
     341,334,179,290,340,166,291,228, 98,144,123,226,105,217)
    (  6, 52,365,332, 10, 74,342,333,180,289,339,165,292,227, 97,143,124,225,
     106,218)(  7, 57,240, 94,172,139,257,375,302, 18, 64,161,310,174,107,
     212,135,251,288,353)(  8, 58,239, 93,171,140,258,376,301, 17, 63,162,
     309,173,108,211,136,252,287,354)( 11, 42,262,335,191, 37, 62,363,293,
     178,359,158,320,229, 87,214,109,254,246,101)( 12, 41,261,336,192, 38,
      61,364,294,177,360,157,319,230, 88,213,110,253,245,102)
    ( 13, 40,215,125,176,380,343, 33, 56,282,328,142,324,235,104,205,119,259,
     167,317)( 14, 39,216,126,175,379,344, 34, 55,281,327,141,323,236,103,
     206,120,260,168,318)( 15, 72,361, 23, 46,367,325,193,308,278,355,152,
     273,238, 77,207,116, 86,222,133)( 16, 71,362, 24, 45,368,326,194,307,
     277,356,151,274,237, 78,208,115, 85,221,134)( 19, 53,351,156,182,248,
      96,220,122,250,357,163, 21, 49, 99,210,111,270,347,369)
    ( 20, 54,352,155,181,247, 95,219,121,249,358,164, 22, 50,100,209,112,269,
     348,370)( 27, 44,315,234, 79,223,127,266,305,284, 28, 43,316,233, 80,
     224,128,265,306,283)( 29, 59,190,337,186,378,296,285,297,150,313,145,
      92,200, 76,201,138,263, 35, 67)( 30, 60,189,338,185,377,295,286,298,
     149,314,146, 91,199, 75,202,137,264, 36, 68), 
  (  1,165,359,272)(  2,166,360,271)(  3,110,260,281)(  4,109,259,282)
    (  5,148,352, 88)(  6,147,351, 87)(  7,159,280, 18)(  8,160,279, 17)
    (  9,158,358,276)( 10,157,357,275)( 11,146,353,274)( 12,145,354,273)
    ( 13,164,314,204)( 14,163,313,203)( 15,162, 97,120)( 16,161, 98,119)
    ( 19, 43,355,290)( 20, 44,356,289)( 21,144,237,283)( 22,143,238,284)
    ( 23,141,340,287)( 24,142,339,288)( 25,168,327,177)( 26,167,328,178)
    ( 27,149, 28,150)( 29, 77,129,250)( 30, 78,130,249)( 31,153,214,228)
    ( 32,154,213,227)( 33,169,297, 53)( 34,170,298, 54)( 35,156,187,277)
    ( 36,155,188,278)( 37,151, 64,285)( 38,152, 63,286)( 39,135,269,299)
    ( 40,136,270,300)( 41,265,367, 90)( 42,266,368, 89)( 45,346,380,293)
    ( 46,345,379,294)( 47,244,373,291)( 48,243,374,292)( 49,370,324,205)
    ( 50,369,323,206)( 51,363,108,122)( 52,364,107,121)( 55, 70,378,308)
    ( 56, 69,377,307)( 57,220,247,302)( 58,219,248,301)( 59,194,349,305)
    ( 60,193,350,306)( 61,375,338,179)( 62,376,337,180)( 65,104,140,252)
    ( 66,103,139,251)( 67,319,224,229)( 68,320,223,230)( 71,334,197,296)
    ( 72,333,198,295)( 73,303, 74,304)( 75,117,261,100)( 76,118,262, 99)
    ( 79,128,268,329)( 80,127,267,330)( 81,116,263,315)( 82,115,264,316)
    ( 83,134,254,215)( 84,133,253,216)( 85,131, 86,132)( 91,114,225,362)
    ( 92,113,226,361)( 93,112,257,365)( 94,111,258,366)( 95,137,256,189)
    ( 96,138,255,190)(101,123,202,239)(102,124,201,240)(105,126,175,342)
    (106,125,176,341)(171,235,372,309)(172,236,371,310)(173,343,321,209)
    (174,344,322,210)(181,211,246,332)(182,212,245,331)(183,185,348,335)
    (184,186,347,336)(191,312,221,233)(192,311,222,234)(195,325,196,326)
    (199,241,318,207)(200,242,317,208)(217,232,218,231) ];

autGroups[84]:=
[ (  1,289, 52,367,239)(  2,359,183,350,204)(  3,348,371,184,240)
    (  4,328,104,226,241)(  5,190,356,200,242)(  6,120,376,208,243)
    (  7, 89,330, 99,223)(  8,196,346,368,185)(  9,279,266,117,233)
    ( 10, 92,221,254, 85)( 11,352,194,347,234)( 12,251, 77,327,100)
    ( 13,230,248, 81,235)( 14, 41,378,336, 22)( 15, 34,283, 43,236)
    ( 16,219,282,264,110)( 17,259,109,366,216)( 18,357,212,274,237)
    ( 19,358,340, 29,238)( 20,286, 48,137,334)( 21,290, 53,360,311)
    ( 23,285, 47,158,152)( 24,294, 40,379,178)( 25,284, 46,176,339)
    ( 26,288, 51,306,335)( 27,174, 39,295,304)( 28,155, 45, 57,315)
    ( 30,293, 38,324,131)( 31, 58, 44,374,159)( 32,287, 50,139, 71)
    ( 33,292, 55,380,297)( 35,144, 49,313,337)( 36,291, 54,365,325)
    ( 37,136, 42,307,338)( 56,314,370,163,143)( 59,209,276,165,111)
    ( 60,316, 91,156,148)( 61,258,112,161,151)( 62,317,187,153,142)
    ( 63,318,202,157,147)( 64,256, 88,162,102)( 65,197,343,154,141)
    ( 66, 83,332, 98,146)( 67, 96,227,164, 73)( 68,278,260,107,140)
    ( 69,191,354,160,145)( 70,108,373,166,150)( 72,232,245, 76,149)
    ( 74,329, 93,220,244)( 75,333, 90,130,249)( 78,303, 97,172,255)
    ( 79,125, 94,231,247)( 80,296,105,225,180)( 82,326,101,222,253)
    ( 84,167, 95,229,250)( 86,321,106,224,246)( 87,331,103,228,252)
    (113,362,213,271,263)(114,134,217,277,267)(115,305,207,169,257)
    (116,308,210,273,128)(118,299,214,281,171)(119,323,127,275,262)
    (121,377,173,270,269)(122,363,211,126,261)(123,375,215,280,265)
    (124,175,218,272,268)(129,361,182,351,198)(132,205,309,364,181)
    (133,188,312,201,322)(135,345,310,186,319)(138,355,203,342,320)
    (168,189,300,206,341)(170,349,195,302,369)(177,344,298,192,353)
    (179,372,193,301,199), (  1,297,202,181,283,179,380,230,208,353)
    (  2,339,227,184,356)(  3,325,287,170,378,348,218,225, 29,300)
    (  4,315,370,182,104, 62,255,223,264,153)(  5,311, 91, 63,248,242,257,
     156, 81,318)(  6,159, 88,191,371,309, 95, 64,254,240)(  7,239,260,163,
      77,314,372,231, 99, 65)(  8, 15,337,187,183,346,344,228,226,350)
    (  9,335, 28,193,194,322,288,172,367,352)( 10,338,112,138, 52,145,270,
     224,366,301)( 11,340,343,189,221,347,349, 12,336,354)( 13, 71,245,185,
     266,157, 80,232,368,312)( 14,178,374,190,212,341, 24,229,200,319)
    ( 16,304,332,186,109,129, 39,146,274,355)( 17,334,373,192,330,342,118,
     130, 43,143)( 18,131, 47,141,282,345,375,222,327,351)( 19,152,276,188,
     376,302,326,220,117,132)( 20,101, 56,244,238,171,158, 89,209,369)
    ( 21,115,135, 48,144,169,360,357,214,331)( 22,269,160, 79,285,168,307,
      92, 68,253)( 23,252,204,263,155, 82,313,359,213, 97)( 25,195,198,290,
     174,176,358,219,207, 27)( 26,295,196,211,292)( 30,299,206,321,286,123,
     137, 41,151,281)( 31,272,205,323,284,167,365,120,127, 46)
    ( 32,298,203, 90, 58,180,236,259,166, 84)( 33,296,201,114,136, 55,139,
     279,217,377)( 34,215,199, 37,294,177,324,289,173,379)( 35, 78,197,320,
     293,103, 57,251,216,265)( 36, 51,142,273,291,175,306,328,210,124)
    ( 38,111,126, 42,140,280,329,363,121,125)( 40,149,271,116,134)
    ( 44,147,277, 94, 59,250,235,267,107, 74)( 45,150,275,362,305,303,333,
     119,113, 53)( 49,148,243,261,161, 87,316,364,122,106)( 50, 73,278,361,
     308,105, 67,247,110,128)( 54,102, 61,249,233,268,162, 86,108,133)
    ( 60,256,234,262, 98)( 66,241,237,165, 76, 83,317,310, 93, 72)
    ( 69,246,100, 70,164, 85,258,154, 75, 96) ];

autGroups[85]:=
[ (  1, 45,181,171, 81,  3,217,185,247, 93,308,216,357,243, 36,307,197,351,
     110, 20,136,200,370,100, 58,144,276,374,328, 73,125,284,298,325, 54,124,
     170,291)(  2,198,180,114, 77,137,219,376,233, 74,316,273,355,339, 35,
     117,208,294, 97, 39,143,161,367,  4, 64,182,285, 88,327,225,128,246,302,
     306, 44,353,167, 24)(  5,  7,179,190, 95, 80,213,223,242,245,321,311,
     349,358, 34, 21,193,199,103, 96,134,142,371,366, 61, 68,278,279,336,340,
     130,132,303,287, 40, 48,162,158)(  6,350,186, 38, 79,194,218,109,229,
     150,314,368,347, 72, 30,269,202,332,111,115,151,295,363, 42, 67,163,272,
     12,331,187,133, 94,289,211, 52,239,169,310)(  8,102,172,152, 85,365,214,
     71,240,283,317,330,359,129, 37,288,192, 46,105,153,139,  9,369,176, 76,
      87,270,221,338,244,131,322,292,344, 53, 29,155,196)( 10,160,177, 19,
      83,175,228, 90,232,226,319,235,360,320, 25,345,205, 27, 98,191,141,104,
     362,138, 66,373,271, 69,335,282,126,341,301,116, 56,296,154, 43)
    ( 11,274,183,342, 92,118,227,300,230, 55,315,159,348, 15, 26,174,209, 84,
     99,210,147,238,379,309, 63,354,281, 31,326,206,122,113,286,135, 47,372,
     157, 62)( 13,255,184,266, 89,251,222,261,241,264,318,254,356,262, 32,
     250,203,256,108,248,146,257,375,252, 70,259,280,260,337,263,127,265,299,
     249, 51,258,165,253)( 14,236,188,323, 82,346,224, 33,231,207,313,101,
     343,148, 28,364,195, 65,106,267,145,333,378,119, 75,297,268, 50,334,168,
     120, 18,293,173, 57, 86,156,215)( 16,121,189,304, 78, 41,220,166,234,
      17,312,178,361, 91, 22,212,204,237,112,305,140,352,377, 23, 60,201,275,
     107,324,149,123,380,290, 59, 49,277,164,329), 
  (  1,229, 50, 98,177,333,169,370, 38)(  2,234, 47,112,180,342,154,362, 31)
    (  3,240, 48,108,179,330,170,378, 33)(  4,233, 42,100,176,328,157,366, 23
     )(  5,238, 55,103,190,325,153,374, 21)(  6,246, 54,109,175,335,162,375,
      26)(  7,235, 56,111,185,327,163,372, 32)(  8,247, 40, 96,183,326,158,
     371, 36)(  9,245, 46,114,173,324,164,364, 24)( 10,241, 45,101,189,340,
     166,365, 30)( 11,239, 51,102,178,341,168,376, 22)( 12,231, 43,104,188,
     332,171,363, 20)( 13,236, 44,113,187,338,156,373, 29)( 14,232, 49,105,
     184,331,159,380, 35)( 15,243, 53,110,186,339,167,377, 34)
    ( 16,242, 41,106,181,337,160,368, 37)( 17,237, 57, 97,172,336,155,367, 28
     )( 18,244, 52, 99,182,334,165,369, 25)( 19,230, 39,107,174,329,161,379,
      27)( 58,221,193,253,142,131,351,323,268)( 59,210,202,250,139,123,360,
     313,285)( 60,215,200,264,141,133,344,305,279)( 61,214,195,252,138,119,
     347,309,271)( 62,219,207,256,152,116,343,317,269)( 63,227,206,261,137,
     125,353,318,274)( 64,216,208,263,147,118,354,315,280)( 65,228,192,248,
     145,117,348,314,283)( 66,226,199,266,135,115,355,307,272)
    ( 67,222,198,254,151,130,357,308,278)( 68,220,203,255,140,132,359,319,270
     )( 69,212,196,257,150,122,361,306,267)( 70,217,197,265,149,128,346,316,
     277)( 71,213,201,258,146,121,349,322,282)( 72,224,205,262,148,129,358,
     320,281)( 73,223,194,259,143,127,350,311,284)( 74,218,209,249,134,126,
     345,310,276)( 75,225,204,251,144,124,356,312,273)( 76,211,191,260,136,
     120,352,321,275)( 77, 88, 79, 81, 85, 93, 84, 95, 78)( 80, 87, 86, 89,
      83, 82, 94, 92, 90)(286,298,288,291,295,302,294,304,287)
    (289,297,296,299,293,292,303,301,300) ];

autGroups[86]:=
[ (  1,150,176, 64,115,131)(  2,149,174, 62,113,129)(  3,152,175, 63,116,132)
    (  4,151,173, 61,114,130)(  5,154,171, 71,110,128,  6,153,169, 69,112,126
     )(  7,156,172, 72,109,127,  8,155,170, 70,111,125)(  9,145,180, 65,120,
     124, 11,147,179, 66,119,123)( 10,146,178, 67,118,122, 12,148,177, 68,
     117,121)( 13,108,188, 84, 55,167, 15,107,187, 82, 56,165)
    ( 14,106,186, 83, 53,168, 16,105,185, 81, 54,166)( 17,102,183, 76, 51,158
     )( 18,104,181, 75, 49,157)( 19,101,184, 74, 52,160)( 20,103,182, 73, 50,
     159)( 21, 97,192, 77, 58,161, 22, 99,190, 78, 60,162)( 23, 98,191, 79,
      57,163, 24,100,189, 80, 59,164)( 25, 45, 90, 30, 41, 86, 26, 47, 92,
      32, 42, 85)( 27, 46, 89, 29, 43, 88, 28, 48, 91, 31, 44, 87)
    ( 33, 40, 96, 36, 37, 93)( 34, 38, 94)( 35, 39, 95)(133,140,134,138)
    (135,139,136,137)(141,144), (  1, 42,165)(  2, 41,166)(  3, 44,167)
    (  4, 43,168)(  5, 46,158)(  6, 45,157)(  7, 48,160)(  8, 47,159)
    (  9, 37,162)( 10, 38,161)( 11, 39,164)( 12, 40,163)( 13, 32,181)
    ( 14, 30,183)( 15, 31,182)( 16, 29,184)( 17, 27,190)( 18, 25,192)
    ( 19, 28,189)( 20, 26,191)( 21, 36,188)( 22, 34,186)( 23, 35,187)
    ( 24, 33,185)( 49,109,153)( 50,111,155)( 51,110,154)( 52,112,156)
    ( 53,117,151)( 54,119,149)( 55,118,152)( 56,120,150)( 57,113,148)
    ( 58,115,146)( 59,114,147)( 60,116,145)( 61,134,179)( 62,133,177)
    ( 63,136,180)( 64,135,178)( 65,139,176)( 66,140,174)( 67,137,175)
    ( 68,138,173)( 69,141,172)( 70,142,170)( 71,143,171)( 72,144,169)
    ( 73, 91,125)( 74, 89,126)( 75, 92,127)( 76, 90,128)( 77, 85,131)
    ( 78, 87,132)( 79, 86,129)( 80, 88,130)( 81, 93,123)( 82, 95,124)
    ( 83, 94,121)( 84, 96,122)( 97,108,104)( 98,107,103)( 99,106,102)
    (100,105,101), (  1,131)(  2,129)(  3,132)(  4,130)(  5,125)(  6,127)
    (  7,126)(  8,128)(  9,122)( 10,124)( 11,121)( 12,123)( 13,192)( 14,191)
    ( 15,190)( 16,189)( 17,182)( 18,181)( 19,184)( 20,183)( 21,188)( 22,187)
    ( 23,186)( 24,185)( 25, 32)( 26, 30)( 27, 31)( 28, 29)( 34, 35)( 37, 96)
    ( 38, 95)( 39, 94)( 40, 93)( 41, 86)( 42, 85)( 43, 88)( 44, 87)( 45, 92)
    ( 46, 91)( 47, 90)( 48, 89)( 50, 51)( 53, 59)( 54, 57)( 55, 60)( 56, 58)
    ( 61,173)( 62,174)( 63,175)( 64,176)( 65,178)( 66,177)( 67,180)( 68,179)
    ( 69,172)( 70,171)( 71,170)( 72,169)( 73,158)( 74,160)( 75,157)( 76,159)
    ( 77,165)( 78,167)( 79,166)( 80,168)( 81,163)( 82,161)( 83,164)( 84,162)
    ( 97,108)( 98,106)( 99,107)(100,105)(102,103)(109,153)(110,155)(111,154)
    (112,156)(113,149)(114,151)(115,150)(116,152)(117,147)(118,145)(119,148)
    (120,146)(133,140)(134,138)(135,139)(136,137)(142,143) ];

autGroups[87]:=
[ (  1, 96, 21,126, 52, 93, 18, 99)(  2, 80, 48,122, 39,118, 38, 72)
    (  3,101, 42, 78, 44, 91, 47,104)(  4, 81, 20,120, 24,121,  5, 86)
    (  6,125, 28,109, 59,110, 26,105)(  7, 95, 50, 83, 60,115, 15, 74)
    (  8, 66, 16,112, 58,103, 54,102)(  9, 76, 55,100, 23,107, 34, 89)
    ( 10, 71, 31,114, 19,124, 51, 79)( 11, 94, 53, 97, 49,128, 13,127)
    ( 12,119, 36, 87, 43, 98, 25, 73)( 14,108, 27,111, 40, 67, 37,106)
    ( 17, 84, 56, 88, 57, 69, 22, 68)( 29, 82, 35, 65, 32, 85, 62,116)
    ( 30,117, 33,113, 64, 77, 63, 75)( 41, 70, 61, 92, 45,123, 46, 90)
    (129,151,150,182,169,166,139,153)(130,146,140,185,162,174,136,141)
    (131,138,165,188,179,170,178,142)(132,148,190,157,192,191,154,134)
    (133,156,160,152,161,163,187,158)(135,172,175,155,168,159,147,143)
    (137,177,144,180,176,184,164,173)(145,167,189,171,181,183,149,186), 
  (  1,187, 12,139, 60,155, 52,157, 58,150, 14,165)(  2,181, 27,147, 53,133,
      55,145, 15,138, 13,191)(  3,172, 11,154, 16,149, 51,131, 49,163, 25,189
     )(  4,164, 33,130, 26,171)(  5,167, 28,136, 30,144)(  6,137)
    (  7,178, 41,134, 43,184, 40,143, 45,160,  8,177)(  9,180, 42,142, 61,
     132, 38,173, 10,168, 46,158)( 17,148, 34,141, 37,170, 57,161, 39,185,
      50,175)( 18,192, 23,169, 19,188, 21,152, 48,129, 47,135)
    ( 20,162, 62,140, 24,182)( 22,156, 54,174, 44,179, 56,190, 36,146, 31,159
     )( 29,151, 59,166, 32,176)( 35,186, 64,153, 63,183)( 65,111, 76, 82,124,
     87,116, 83,122, 85, 78,112)( 66,113, 91, 89,117, 67,119, 75, 79, 80, 77,
     115)( 68, 90, 97)( 69, 94, 92)( 71,102,105, 74,107,110,104, 73,109,106,
      72,125)( 81, 95, 98, 86,101,118,121,108,103,120,114,100)( 84, 88,126)
    ( 93, 96,123)( 99,127,128) ];

autGroups[88]:=
[ (  2,  3,  5,  8, 14, 23, 34, 30, 20, 12,  7,  4)(  6,  9, 15, 24, 36, 50,
      60, 52, 37, 25, 16, 10)( 11, 17, 26, 21, 13, 18)( 19, 27, 38, 48, 35,
      47, 43, 32, 22, 31, 39, 28)( 29, 40, 54, 63, 59, 62, 64, 57, 45, 33,
      44, 41)( 42, 46, 55)( 49, 53, 56, 51, 61, 58)( 65,129)( 66,133, 69,142,
     78,162, 98,148, 84,135, 71,130)( 67,136, 72,151, 87,158, 94,140, 76,132,
     68,131)( 70,143, 79,164,100,188,124,165,101,144, 80,134)
    ( 73,152, 88,178,114,180,116,153, 89,138, 74,137)( 75,154, 90,141, 77,139
     )( 81,149, 85,146, 82,145)( 83,166,102,163, 99,171,107,150, 86,167,103,
     147)( 91,176,112,175,111,160, 96,159, 95,156, 92,155)( 93,182,118,187,
     123,192,128,173,109,172,108,157)( 97,169,105,168,104,191,127,190,126,
     185,121,161)(106,183,119,174,110,170)(113,184,120,189,125,177)
    (115,186,122,181,117,179), (  1,131, 67)(  2,133, 65)(  3,129, 66)
    (  4,137, 69)(  5,130, 70)(  6,136, 68)(  7,145, 73)(  8,134, 77)
    (  9,132, 75)( 10,146, 72)( 11,143, 71)( 12,155, 81)( 13,142, 74)
    ( 14,141, 86)( 15,139, 80)( 16,152, 82)( 17,135, 83)( 18,138, 79)
    ( 19,154, 76)( 20,168, 91)( 21,159, 78)( 22,151, 85)( 23,150, 97)
    ( 24,144, 99)( 25,175, 88)( 26,147, 95)( 27,140, 93)( 28,167, 90)
    ( 29,166, 84)( 30,174,104)( 31,149, 92)( 32,172, 87)( 33,162, 96)
    ( 34,161,110)( 35,164, 89)( 36,163,113)( 37,189,111)( 38,157,117)
    ( 39,156,103)( 40,148,106)( 41,184,102)( 42,182, 94)( 43,179,108)
    ( 44,160,120)( 45,183, 98)( 46,158,109)( 47,153,115)( 48,181,100)
    ( 49,178,112)( 50,177,123)( 51,171,101)( 52,192,125)( 53,176,105)
    ( 54,170,121)( 55,173,118)( 56,169,107)( 57,191,119)( 58,190,114)
    ( 59,188,122)( 60,187,128)( 61,165,126)( 62,186,116)( 63,185,124)
    ( 64,180,127), (  2,  3)(  4,  5)(  7,  8)(  9, 10)( 11, 13)( 12, 14)
    ( 15, 16)( 17, 21)( 19, 22)( 20, 23)( 24, 25)( 27, 32)( 28, 31)( 29, 33)
    ( 30, 34)( 36, 37)( 38, 43)( 40, 45)( 41, 44)( 42, 46)( 47, 48)( 49, 51)
    ( 50, 52)( 53, 56)( 54, 57)( 58, 61)( 59, 62)( 63, 64)( 65,129)( 66,133)
    ( 67,131)( 68,136)( 69,130)( 70,137)( 71,142)( 72,132)( 73,134)( 74,143)
    ( 75,146)( 76,151)( 77,145)( 78,135)( 79,138)( 80,152)( 81,141)( 82,139)
    ( 83,159)( 84,162)( 85,154)( 86,155)( 87,140)( 88,144)( 89,164)( 90,149)
    ( 91,150)( 92,167)( 93,172)( 94,158)( 95,147)( 96,166)( 97,168)( 98,148)
    ( 99,175)(100,153)(101,178)(102,160)(103,156)(104,161)(105,169)(106,183)
    (107,176)(108,157)(109,182)(110,174)(111,163)(112,171)(113,189)(114,165)
    (115,181)(116,188)(117,179)(118,173)(119,170)(120,184)(121,191)(122,186)
    (123,192)(124,180)(125,177)(126,190)(127,185)(128,187) ];

autGroups[89]:=
[ (  1, 25, 57,117, 58,138,101, 29, 74,107, 61, 83, 56, 13)(  2, 80, 75, 44,
     132, 23,142, 41,143, 24,123, 43, 42, 49)(  3,106, 34,105,103,110,115,
      77,129, 11, 90, 91, 27,137)(  4, 45, 48, 52,140, 96, 36, 99, 32,126,
     120,124,131,116)(  5,127,133, 31, 20, 51, 46, 47, 35,125,118,119,128, 37
     )(  6,147, 14,139, 59, 63,102, 40, 17, 98,108,122, 76,111)
    (  7, 55, 54, 38, 70, 65, 60,130,114, 78,109, 79, 53,146)
    (  8, 67, 30, 95, 39, 15,100, 92,113,144, 82, 66, 89, 85)
    (  9, 87, 50, 84, 64, 68, 97, 12, 81,145, 88, 86, 28, 69)
    ( 10,112, 26,104,141, 73, 71, 94,136, 72, 93, 16, 21, 62)( 18,134)
    ( 19,135)( 22, 33), (  1,142, 16,121, 72, 44, 74,124, 54, 31,106,128, 53,
     120)(  2, 92, 33, 15, 24, 84, 99, 73, 51,  7,118,112,  4,145)
    (  3, 79, 13,100,107, 38, 34, 26,111,122,130, 40, 63,141)
    (  5,137, 36,144, 42,114, 22, 60, 43, 95, 45,105,127, 14)
    (  6, 48, 62,132, 11,135, 77, 23, 94, 96, 59, 37, 69,133)
    (  8,147, 10,113, 83, 39, 71,139, 89, 88, 90,129,115, 50)
    (  9,117, 25, 78, 98, 65, 29,138, 28, 82, 21, 97,136, 30)
    ( 12,134, 68, 41,110,116,101,125, 85, 46, 57, 32, 91, 75)
    ( 17, 49, 70,126, 87,119, 58, 20, 86,131,109,123,108, 18)
    ( 19, 61, 80,102, 52, 67, 47,104, 35, 66,140, 76,143, 56)
    ( 27, 81, 93, 64,103, 55,146) ];

gens[1]:=[ 1, 2, 3 ];
gens[2]:=[ 1, 2, 3 ];
gens[3]:=[ 1, 2, 3 ];
gens[4]:=[ 1, 2, 3 ];
gens[5]:=[ 1, 2, 3 ];
gens[6]:=[ 1, 2, 3 ];
gens[7]:=[ 1, 2, 3 ];
gens[8]:=[ 1, 2, 3 ];
gens[9]:=[ 1, 2, 3 ];
gens[10]:=[ 1, 2, 3 ];
gens[11]:=[ 1, 2, 3 ];
gens[12]:=[ 1, 2, 3 ];
gens[13]:=[ 1, 2, 3 ];
gens[14]:=[ 1, 2, 3 ];
gens[15]:=[ 1, 5, 7 ];
gens[16]:=[ 1, 2, 3 ];
gens[17]:=[ 1, 2, 3 ];
gens[18]:=[ 1, 2, 3 ];
gens[19]:=[ 1, 2, 3 ];
gens[20]:=[ 1, 2, 3 ];
gens[21]:=[ 1, 2, 3 ];
gens[22]:=[ 1, 2, 3 ];
gens[23]:=[ 1, 2, 3 ];
gens[24]:=[ 1, 2, 3 ];
gens[25]:=[ 1, 2, 3 ];
gens[26]:=[ 1, 2, 3 ];
gens[27]:=[ 1, 2, 3 ];
gens[28]:=[ 1, 2, 3 ];
gens[29]:=[ 1, 2, 3 ];
gens[30]:=[ 1, 2, 27 ];
gens[31]:=[ 1, 2, 26 ];
gens[32]:=[ 1, 2, 3 ];
gens[33]:=[ 1, 2, 3 ];
gens[34]:=[ 1, 2, 3 ];
gens[35]:=[ 1, 2, 3 ];
gens[36]:=[ 1, 2, 3 ];
gens[37]:=[ 1, 2, 3 ];
gens[38]:=[ 1, 2, 3 ];
gens[39]:=[ 1, 2, 3 ];
gens[40]:=[ 1, 2, 3 ];
gens[41]:=[ 1, 4, 5 ];
gens[42]:=[ 1, 15, 138 ];
gens[43]:=[ 1, 2, 3 ];
gens[44]:=[ 1, 34, 68 ];
gens[45]:=[ 1, 16, 27 ];
gens[46]:=[ 1, 4, 7 ];
gens[47]:=[ 1, 71, 245 ];
gens[48]:=[ 1, 67, 290 ];
gens[49]:=[ 1, 2, 3 ];
gens[50]:=[ 1, 9, 11 ];
gens[51]:=[ 1, 5, 15 ];
gens[52]:=[ 1, 130, 322 ];
gens[53]:=[ 1, 10, 176 ];
gens[54]:=[ 1, 35, 92 ];
gens[55]:=[ 1, 2, 3 ];
gens[56]:=[ 1, 19, 63 ];
gens[57]:=[ 1, 4, 235 ];
gens[58]:=[ 1, 10, 68 ];
gens[59]:=[ 1, 73, 145 ];
gens[60]:=[ 1, 37, 198 ];
gens[61]:=[ 1, 49, 97 ];
gens[62]:=[ 1, 25, 68 ];
gens[63]:=[ 1, 2, 3 ];
gens[64]:=[ 1, 27, 102 ];
gens[65]:=[ 1, 2, 32 ];
gens[66]:=[ 1, 2, 3 ];
gens[67]:=[ 1, 33, 65 ];
gens[68]:=[ 1, 33, 65 ];
gens[69]:=[ 1, 4, 19 ];
gens[70]:=[ 1, 58, 135 ];
gens[71]:=[ 1, 55, 109 ];
gens[72]:=[ 1, 28, 55 ];
gens[73]:=[ 1, 28, 55 ];
gens[74]:=[ 1, 2, 3 ];
gens[75]:=[ 1, 5, 9 ];
gens[76]:=[ 1, 85, 169 ];
gens[77]:=[ 1, 2, 3 ];
gens[78]:=[ 1, 2, 3 ];
gens[79]:=[ 1, 13, 171 ];
gens[80]:=[ 1, 2, 3 ];
gens[81]:=[ 1, 30, 109 ];
gens[82]:=[ 1, 16, 142 ];
gens[83]:=[ 1, 85, 119 ];
gens[84]:=[ 1, 54, 79 ];
gens[85]:=[ 1, 44, 190 ];
gens[86]:=[ 1, 85, 97 ];
gens[87]:=[ 1, 66, 130 ];
gens[88]:=[ 1, 66, 130 ];
gens[89]:=[ 1, 2, 3 ];

mg[1]:=[ 7, 3 ];
mg[2]:=[ 8, 3 ];
mg[3]:=[ 8, 5 ];
mg[4]:=[ 10, 6 ];
mg[5]:=[ 7, 7 ];
mg[6]:=[ 8, 8 ];
mg[7]:=[ 8, 8 ];
mg[8]:=[ 9, 10 ];
mg[9]:=[ 12, 10 ];
mg[10]:=[ 10, 13 ];
mg[11]:=[ 12, 13 ];
mg[12]:=[ 7, 14 ];
mg[13]:=[ 7, 14 ];
mg[14]:=[ 7, 14 ];
mg[15]:=[ 9, 15 ];
mg[16]:=[ 14, 15 ];
mg[17]:=[ 8, 16 ];
mg[18]:=[ 8, 17 ];
mg[19]:=[ 16, 21 ];
mg[20]:=[ 8, 22 ];
mg[21]:=[ 8, 22 ];
mg[22]:=[ 10, 25 ];
mg[23]:=[ 12, 25 ];
mg[24]:=[ 10, 26 ];
mg[25]:=[ 11, 26 ];
mg[26]:=[ 12, 28 ];
mg[27]:=[ 12, 28 ];
mg[28]:=[ 18, 28 ];
mg[29]:=[ 8, 33 ];
mg[30]:=[ 12, 33 ];
mg[31]:=[ 12, 33 ];
mg[32]:=[ 20, 36 ];
mg[33]:=[ 12, 37 ];
mg[34]:=[ 15, 37 ];
mg[35]:=[ 16, 41 ];
mg[36]:=[ 16, 41 ];
mg[37]:=[ 9, 43 ];
mg[38]:=[ 10, 45 ];
mg[39]:=[ 10, 45 ];
mg[40]:=[ 22, 45 ];
mg[41]:=[ 8, 46 ];
mg[42]:=[ 8, 46 ];
mg[43]:=[ 10, 49 ];
mg[44]:=[ 12, 49 ];
mg[45]:=[ 14, 49 ];
mg[46]:=[ 13, 50 ];
mg[47]:=[ 8, 52 ];
mg[48]:=[ 8, 52 ];
mg[49]:=[ 24, 55 ];
mg[50]:=[ 12, 56 ];
mg[51]:=[ 12, 56 ];
mg[52]:=[ 8, 65 ];
mg[53]:=[ 10, 65 ];
mg[54]:=[ 12, 65 ];
mg[55]:=[ 26, 66 ];
mg[56]:=[ 9, 69 ];
mg[57]:=[ 9, 69 ];
mg[58]:=[ 9, 69 ];
mg[59]:=[ 10, 73 ];
mg[60]:=[ 10, 73 ];
mg[61]:=[ 12, 73 ];
mg[62]:=[ 15, 73 ];
mg[63]:=[ 24, 73 ];
mg[64]:=[ 12, 76 ];
mg[65]:=[ 15, 76 ];
mg[66]:=[ 28, 78 ];
mg[67]:=[ 16, 81 ];
mg[68]:=[ 16, 81 ];
mg[69]:=[ 8, 82 ];
mg[70]:=[ 12, 82 ];
mg[71]:=[ 12, 82 ];
mg[72]:=[ 18, 82 ];
mg[73]:=[ 18, 82 ];
mg[74]:=[ 18, 82 ];
mg[75]:=[ 8, 85 ];
mg[76]:=[ 8, 85 ];
mg[77]:=[ 20, 85 ];
mg[78]:=[ 10, 89 ];
mg[79]:=[ 10, 89 ];
mg[80]:=[ 30, 91 ];
mg[81]:=[ 12, 92 ];
mg[82]:=[ 12, 92 ];
mg[83]:=[ 9, 96 ];
mg[84]:=[ 9, 96 ];
mg[85]:=[ 9, 96 ];
mg[86]:=[ 12, 97 ];
mg[87]:=[ 12, 97 ];
mg[88]:=[ 12, 97 ];
mg[89]:=[ 14, 99 ];
	fi;

	if m > n-1 then
		Info(InfoSimpcomp,1,"SCRegularMap: first argument must be smaller or second argument must be larger.");
		return fail;
	fi;
	i:=Position(mg,[m,g]);
	if i <> fail then
		G:=Group(autGroups[i]);
		c:=SCFromGenerators(G,[gens[i]]);
		if orient then
			SetSCTopologicalType(c,Concatenation("(T^2)^#",String(g)));
			SCRename(c,Concatenation("Orientable regular map {",String(m),",",String(g),"}"));
		else
			SetSCTopologicalType(c,Concatenation("(RP^2)^#",String(g)));
			SCRename(c,Concatenation("Non-orientable regular map {",String(m),",",String(g),"}"));
		fi;
		return c;
	else
		return fail;
	fi;
end);


################################################################################
##<#GAPDoc Label="SCChiralMaps">
## <ManSection>
## <Func Name="SCChiralMaps" Arg=""/>
## <Returns>a list of lists upon success, <K>fail</K> otherwise.</Returns>
## <Description>
## Returns a list of all simplicial (hyperbolic) chiral maps of orientable genus up to <M>100</M>.
## The list was generated with the help of 
## the classification of regular maps by Marston Conder <Cite Key="Conder09RegMapsOfBdChi"/>.
##
## Every chiral map is given by a <M>2</M>-tuple <M>(m,g)</M> where <M>m</M> is the 
## vertex valence and <M>g</M> is the genus of the map.
##
## Use the <M>2</M>-tuples of the list together with <Ref Func="SCChiralMap"/> 
## to get the corresponding triangulations.
## <Example>
## gap> ll:=SCChiralMaps();
## gap> c:=SCChiralMap(ll[18][1],ll[18][2]);
## gap> SCHomology(c);
## </Example>
## </Description>
## </ManSection>
##<#/GAPDoc>
################################################################################
InstallGlobalFunction(SCChiralMaps,
function()
	local mg;
	mg:=[[7,17],[8,10],[8,28],[8,37],[8,46],[8,82],
	[9,43],[10,73],[12,22],[12,33],[12,40],[12,51],
	[12,58],[12,64],[12,85],[12,94],[12,97],[18,28]];
	return mg;
end);

################################################################################
##<#GAPDoc Label="SCChiralMap">
## <ManSection>
## <Func Name="SCChiralMap" Arg="m,g"/>
## <Returns>a <K>SCSimplicialComplex</K> object upon success, <K>fail</K> otherwise.</Returns>
## <Description>
## Returns the (hyperbolic) chiral map of vertex valence <Arg>m</Arg> and genus <Arg>g</Arg> if existent and
## <K>fail</K> otherwise. 
##
## The list was generated with the help of 
## the classification of regular maps by Marston Conder <Cite Key="Conder09RegMapsOfBdChi"/>.
##
## Use <Ref Func="SCChiralMaps"/> to get a list of all
## chiral maps available.
## <Example>
## gap> SCChiralMaps();
## gap> c:=SCChiralMap(8,10);
## gap> c.Homology;
## </Example>
## </Description>
## </ManSection>
##<#/GAPDoc>
################################################################################
InstallGlobalFunction(SCChiralMap,
function(m,g)

	local autGroups,gens,n,f2,G,mg,i,c;

	if not IsPosInt(m) or m < 7 or not IsInt(g) and g >= 2  then
		Info(InfoSimpcomp,1,"SCChiralMap: argument 1 must be an integer > 6, argument 2 must be an integer > 1.");
		return fail;
	fi;

	mg:=[];
	gens:=[];
	autGroups:=[];
	n:=6*(2-2*g)/(6-m);
	f2:= 2*m*(2-2*g)/(6-m);
	
	if m > n-1 then
		Info(InfoSimpcomp,1,"SCChiralMap: first argument must be smaller or second argument must be larger.");
		return fail;
	fi;

autGroups[1]:=
[ ( 2, 4, 3, 6)( 7,31,49,37,13,43,25,19)( 8,34,51,42,14,46,27,24)
    ( 9,36,50,40,15,48,26,22)(10,33,54,38,16,45,30,20)(11,35,53,41,17,47,29,23
     )(12,32,52,39,18,44,28,21), ( 1,54,38)( 2,49,42)( 3,53,40)( 4,51,41)
    ( 5,52,39)( 6,50,37)( 7,18,32)( 8,13,36)( 9,17,34)(10,15,35)(11,16,33)
    (12,14,31)(19,30,44)(20,25,48)(21,29,46)(22,27,47)(23,28,45)(24,26,43) ];

autGroups[2]:=
[ (  1,  7, 36, 80,112, 87, 47)(  2, 25, 68,103,117, 89, 50)(  3, 28, 20, 40,
      77, 85, 48)(  4, 26, 67,101,116, 88, 52)(  5, 29, 70,102,115, 86, 49)
    (  6, 27, 69, 60, 12, 39, 51)(  8, 37, 81,113,123, 91, 53)
    (  9, 30, 76,110,124, 94, 57)( 10, 31, 73,107,120, 96, 58)
    ( 11, 42, 62, 23, 46, 84, 54)( 13, 32, 71,106,119, 95, 64)
    ( 14, 33, 72,108,121, 97, 56)( 15, 34, 74,109,122, 98, 65)
    ( 16, 35, 75,104,100, 63, 24)( 17, 41, 78,105,118, 90, 55)
    ( 18, 43, 79, 92, 59, 19, 38)( 21, 44, 82,111,125, 99, 66)
    ( 22, 45, 83,114,126, 93, 61)(127,167,186,165,149,159,178)
    (128,142,175,181,132,170,133)(129,169,131,152,134,144,145)
    (130,168,187,192,185,162,151)(135,157,137,174,188,190,182)
    (136,147,171,163,153,173,143)(139,166,140,158,177,183,155)
    (141,176,164,150,154,148,172)(160,179,161,180,189,191,184), 
  (  1, 17, 86)(  2,137, 90)(  3,136, 63)(  4,138, 52)(  5,135,167)
    (  6,106, 60)(  7,115, 72)(  8,123, 83)(  9, 12, 71)( 10, 15,142)
    ( 11, 81,152)( 13,141,169)( 14,139, 89)( 16,132,168)( 18,140,120)
    ( 19,144, 22)( 20, 25,118)( 21, 31,128)( 23,112, 75)( 24,143,170)
    ( 26, 88, 99)( 27, 69,146)( 28,100, 68)( 29,127,171)( 30, 32,129)
    ( 33,117, 36)( 34, 58, 97)( 35,130, 46)( 37, 45,134)( 38, 61,158)
    ( 39, 57,159)( 40,105,109)( 41, 47,154)( 42,131,172)( 43,107,101)
    ( 44,133,173)( 48, 70,147)( 49, 55,157)( 50,155,174)( 51,149,119)
    ( 53,156, 91)( 54,114,113)( 56, 96,166)( 59, 76,145)( 62,148, 87)
    ( 64,161,176)( 65,160,175)( 66,116, 73)( 67,125, 79)( 74,121, 77)
    ( 78,150,122)( 80,103,104)( 82,153,124)( 84,151,126)( 85,108,102)
    ( 92,111,110)( 93,162,177)( 94,163,178)( 95,165,180)( 98,164,179)
    (181,184,187)(182,189,186)(183,185,188)(190,192,191) ];

autGroups[3]:=
[ ( 2, 3, 5, 6, 7, 4)( 8,36,29,22)( 9,40,33,28,14,37,30,26,12,42,35,23)
    (10,41,34,25,11,38,31,27,13,39,32,24)(16,17,19,20,21,18), 
  ( 1,27,13)( 2,28, 8)( 3,22,12)( 4,23,14)( 5,26,11)( 6,25,10)( 7,24, 9)
    (15,41,34)(16,42,29)(17,36,33)(18,37,35)(19,40,32)(20,39,31)(21,38,30) ];

autGroups[4]:=
[ (  3,  5)(  4,  6)(  7, 19, 45, 49, 17, 39, 35, 25)(  8, 20, 46, 50, 18, 40,
     36, 26)(  9, 23, 47, 53, 13, 37, 31, 29)( 10, 24, 48, 54, 14, 38, 32, 30)
    ( 11, 21, 43, 51, 15, 41, 33, 27)( 12, 22, 44, 52, 16, 42, 34, 28)
    ( 55,116, 80,135, 87,128, 62,109)( 56,115, 79,136, 88,127, 61,110)
    ( 57,120, 82,133, 89,132, 64,113)( 58,119, 81,134, 90,131, 63,114)
    ( 59,118, 84,137, 85,130, 66,111)( 60,117, 83,138, 86,129, 65,112)
    ( 67,122,108,155, 77,144, 94,145)( 68,121,107,156, 78,143, 93,146)
    ( 69,126,104,153, 73,142, 96,149)( 70,125,103,154, 74,141, 95,150)
    ( 71,124,106,151, 75,140, 92,147)( 72,123,105,152, 76,139, 91,148)
    ( 97,160, 98,159)( 99,158,100,157)(101,162,102,161), 
  (  1,109, 55)(  2,110, 56)(  3,111, 57)(  4,112, 58)(  5,113, 59)
    (  6,114, 60)(  7,115, 61)(  8,116, 62)(  9,117, 63)( 10,118, 64)
    ( 11,119, 65)( 12,120, 66)( 13,121, 67)( 14,122, 68)( 15,123, 69)
    ( 16,124, 70)( 17,125, 71)( 18,126, 72)( 19,127, 73)( 20,128, 74)
    ( 21,129, 75)( 22,130, 76)( 23,131, 77)( 24,132, 78)( 25,133, 79)
    ( 26,134, 80)( 27,135, 81)( 28,136, 82)( 29,137, 83)( 30,138, 84)
    ( 31,139, 85)( 32,140, 86)( 33,141, 87)( 34,142, 88)( 35,143, 89)
    ( 36,144, 90)( 37,145, 91)( 38,146, 92)( 39,147, 93)( 40,148, 94)
    ( 41,149, 95)( 42,150, 96)( 43,151, 97)( 44,152, 98)( 45,153, 99)
    ( 46,154,100)( 47,155,101)( 48,156,102)( 49,157,103)( 50,158,104)
    ( 51,159,105)( 52,160,106)( 53,161,107)( 54,162,108) ];

autGroups[5]:=
[ ( 1,14, 4,11, 8,18, 2,13, 5,10, 9,17, 3,15, 6,12, 7,16)(19,25,21,26,20,27)
    (22,23), ( 1,19,10)( 2,20,11)( 3,21,12)( 4,22,13)( 5,23,14)( 6,24,15)
    ( 7,25,16)( 8,26,17)( 9,27,18) ];

autGroups[6]:=
[ ( 2, 4, 3)( 5, 9,13, 8,11,14)( 6,12,15, 7,10,16)(17,51,36,24,57,47,19,52,33,
     22,58,46)(18,50,34,23,60,45,20,49,35,21,59,48)(25,63,37,30,55,44,27,64,
     40,32,56,41)(26,62,39,29,54,42,28,61,38,31,53,43), 
  ( 1,32,56)( 2,30,53)( 3,29,55)( 4,31,54)( 5,18,62)( 6,20,63)( 7,19,61)
    ( 8,17,64)( 9,26,60)(10,28,57)(11,27,59)(12,25,58)(13,23,51)(14,21,50)
    (15,22,52)(16,24,49)(34,35,36)(37,45,43)(38,47,42)(39,48,44)(40,46,41) ];

autGroups[7]:=
[ (  3,  6, 10, 13, 18, 16, 12,  7)(  4,  5,  9, 14, 17, 15, 11,  8)
    ( 19,164, 92,182,110,199,145, 73)( 20,163, 91,181,109,200,146, 74)
    ( 21,179,107,189,125,211,147, 75)( 22,180,108,190,126,212,148, 76)
    ( 23,178,106,193,124,215,149, 77)( 24,177,105,194,123,216,150, 78)
    ( 25,176,104,185,122,208,151, 79)( 26,175,103,186,121,207,152, 80)
    ( 27,174,102,198,120,213,153, 81)( 28,173,101,197,119,214,154, 82)
    ( 29,172,100,183,118,203,155, 83)( 30,171, 99,184,117,204,156, 84)
    ( 31,170, 98,195,116,210,157, 85)( 32,169, 97,196,115,209,158, 86)
    ( 33,168, 96,187,114,202,159, 87)( 34,167, 95,188,113,201,160, 88)
    ( 35,165, 93,192,111,206,161, 89)( 36,166, 94,191,112,205,162, 90)
    ( 37,128, 38,127)( 39,137, 47,144, 54,135, 45,129)( 40,138, 48,143, 53,
     136, 46,130)( 41,133, 43,141, 51,139, 49,131)( 42,134, 44,142, 52,140,
      50,132)( 55, 56)( 57, 59, 64, 68, 72, 69, 66, 62)( 58, 60, 63, 67, 71,
      70, 65, 61), (  1, 89, 35)(  2, 90, 36)(  3, 78, 20)(  4, 77, 19)
    (  5, 73, 29)(  6, 74, 30)(  7, 79, 24)(  8, 80, 23)(  9, 83, 33)
    ( 10, 84, 34)( 11, 86, 26)( 12, 85, 25)( 13, 88, 28)( 14, 87, 27)
    ( 15, 76, 32)( 16, 75, 31)( 17, 81, 22)( 18, 82, 21)( 37,209,104)
    ( 38,210,103)( 39,205,100)( 40,206, 99)( 41,215,101)( 42,216,102)
    ( 43,201, 97)( 44,202, 98)( 45,207, 94)( 46,208, 93)( 47,199, 95)
    ( 48,200, 96)( 49,214, 92)( 50,213, 91)( 51,212,107)( 52,211,108)
    ( 53,203,105)( 54,204,106)( 55,189,126)( 56,190,125)( 57,183,112)
    ( 58,184,111)( 59,191,121)( 60,192,122)( 61,193,117)( 62,194,118)
    ( 63,185,115)( 64,186,116)( 65,197,124)( 66,198,123)( 67,196,113)
    ( 68,195,114)( 69,181,120)( 70,182,119)( 71,188,110)( 72,187,109)
    (127,175,158)(128,176,157)(129,166,162)(130,165,161)(131,164,149)
    (132,163,150)(133,173,160)(134,174,159)(135,178,152)(136,177,151)
    (137,172,145)(138,171,146)(139,179,154)(140,180,153)(141,169,148)
    (142,170,147)(143,168,155)(144,167,156) ];

autGroups[8]:=
[ ( 2, 3, 5, 8, 7, 4)( 6, 9,13,11,12,10)(14,66,53,40)(15,70,57,46,20,67,54,44,
     18,72,59,41)(16,73,60,43,17,68,55,47,21,69,56,42)(19,78,65,51,25,71,58,
     52,26,77,64,45)(22,76,63,49,23,74,61,50,24,75,62,48)(28,29,31,34,33,30)
    (32,35,39,37,38,36), ( 1,47,21)( 2,46,14)( 3,40,18)( 4,50,20)( 5,44,26)
    ( 6,43,16)( 7,45,24)( 8,52,19)( 9,42,23)(10,48,17)(11,51,15)(12,41,22)
    (13,49,25)(27,73,60)(28,72,53)(29,66,57)(30,76,59)(31,70,65)(32,69,55)
    (33,71,63)(34,78,58)(35,68,62)(36,74,56)(37,77,54)(38,67,61)(39,75,64) ];

autGroups[9]:=
[ (  1,  7, 34, 43, 47, 10,167, 81, 89)(  2,  9, 36, 44, 46, 11,168, 80, 90)
    (  3,  8, 35, 45, 48, 12,166, 79, 88)(  4, 25, 87,145,136, 33, 24, 66,111)
    (  5, 27, 86,146,137, 31, 23, 65,109)(  6, 26, 85,147,138, 32, 22, 64,110)
    ( 13,101, 57,116,164, 37,148, 63,153)( 14,102, 56,115,165, 39,150, 62,151)
    ( 15,100, 55,117,163, 38,149, 61,152)( 16, 98,105, 94, 51,159, 53,123, 68)
    ( 17, 99,103, 95, 49,158, 52,122, 69)( 18, 97,104, 96, 50,157, 54,121, 67)
    ( 19,156, 41,108,139,112, 78,125,130)( 20,155, 40,107,141,113, 76,124,131)
    ( 21,154, 42,106,140,114, 77,126,132)( 28,129,118,161, 74, 72, 83, 93,143)
    ( 29,127,119,160, 75, 71, 84, 91,142)( 30,128,120,162, 73, 70, 82, 92,144)
    (133,134,135), (  1, 25, 43)(  2, 27, 44)(  3, 26, 45)(  4, 23, 47)
    (  5, 22, 46)(  6, 24, 48)(  7, 34, 58)(  8, 35, 59)(  9, 36, 60)
    ( 10, 31, 52)( 11, 32, 53)( 12, 33, 54)( 13, 38, 49)( 14, 37, 50)
    ( 15, 39, 51)( 16, 29, 57)( 17, 28, 56)( 18, 30, 55)( 19, 40, 61)
    ( 20, 42, 62)( 21, 41, 63)( 64,109,133)( 65,111,134)( 66,110,135)
    ( 67,113,128)( 68,112,127)( 69,114,129)( 70,106,131)( 71,107,130)
    ( 72,108,132)( 73,118,140)( 74,119,139)( 75,120,141)( 76,121,136)
    ( 77,122,137)( 78,123,138)( 79,116,142)( 80,115,143)( 81,117,144)
    ( 82,124,145)( 83,126,146)( 84,125,147)( 85, 88, 91)( 86, 90, 93)
    ( 87, 89, 92)( 94, 97,100)( 95, 98,101)( 96, 99,102)(103,104,105)
    (148,151,154)(149,153,156)(150,152,155)(157,164,166)(158,163,167)
    (159,165,168)(160,161,162) ];

autGroups[10]:=
[ (  1,  2)(  3,  6,  8, 15, 10, 41, 43, 19)(  4,  5,  7, 16,  9, 42, 44, 20)
    ( 11, 21, 95, 23,109, 57, 99, 31)( 12, 22, 96, 24,110, 58,100, 32)
    ( 13, 28, 30,101, 34,119,121,111)( 14, 27, 29,102, 33,120,122,112)
    ( 17, 40,106, 56,108,135,137, 66)( 18, 39,105, 55,107,136,138, 65)
    ( 25, 35, 37, 73, 50,124,163, 77)( 26, 36, 38, 74, 49,123,164, 78)
    ( 45,168,128, 83, 61, 63,165, 67)( 46,167,127, 84, 62, 64,166, 68)
    ( 47, 88, 90,211, 94,178, 98,162)( 48, 87, 89,212, 93,177, 97,161)
    ( 51,198,148,186,174,176,116, 60)( 52,197,147,185,173,175,115, 59)
    ( 53,206,208,219, 54,205,207,220)( 69, 79,214, 81,259,255,252, 91)
    ( 70, 80,213, 82,260,256,251, 92)( 71,130, 76,117, 72,129, 75,118)
    ( 85,157,200,181, 86,158,199,182)(103,134,224,232,226,244,236,239)
    (104,133,223,231,225,243,235,240)(113,141,204,143,139,229,270,153)
    (114,142,203,144,140,230,269,154)(125,190,192,250,196,248,131,172)
    (126,189,191,249,195,247,132,171)(145,149,227,151,261,221,233,155)
    (146,150,228,152,262,222,234,156)(159,209,201,238,217,241,257,266)
    (160,210,202,237,218,242,258,265)(169,253,184,268,188,264,194,180)
    (170,254,183,267,187,263,193,179)(215,246)(216,245), 
  (  1,  3,  5)(  2,  4,  6)(  7, 19, 11)(  8, 20, 12)(  9, 25, 27)
    ( 10, 26, 28)( 13, 39, 41)( 14, 40, 42)( 15, 32, 36)( 16, 31, 35)
    ( 17, 22, 44)( 18, 21, 43)( 23, 59, 61)( 24, 60, 62)( 29, 77, 69)
    ( 30, 78, 70)( 33, 85, 87)( 34, 86, 88)( 37, 99, 57)( 38,100, 58)
    ( 45,105,111)( 46,106,112)( 47,117,119)( 48,118,120)( 49,127,129)
    ( 50,128,130)( 51, 96, 66)( 52, 95, 65)( 53,133,135)( 54,134,136)
    ( 55, 67,113)( 56, 68,114)( 63,115,145)( 64,116,146)( 71,161,124)
    ( 72,162,123)( 73,109, 83)( 74,110, 84)( 75,167,122)( 76,168,121)
    ( 79,163, 97)( 80,164, 98)( 81,171,173)( 82,172,174)( 89,182,179)
    ( 90,181,180)( 91,157,102)( 92,158,101)( 93,187,189)( 94,188,190)
    (103,197,138)(104,198,137)(107,153,205)(108,154,206)(125,213,178)
    (126,214,177)(131,150,176)(132,149,175)(139,223,220)(140,224,219)
    (141,165,155)(142,166,156)(143,225,231)(144,226,232)(147,239,201)
    (148,240,202)(151,245,195)(152,246,196)(159,253,255)(160,254,256)
    (169,200,252)(170,199,251)(183,265,258)(184,266,257)(185,209,259)
    (186,210,260)(191,267,242)(192,268,241)(193,263,212)(194,264,211)
    (203,234,244)(204,233,243)(207,270,229)(208,269,230)(215,228,248)
    (216,227,247)(217,262,250)(218,261,249)(221,237,235)(222,238,236) ];

autGroups[11]:=
[ (  1,  3,  2)(  5, 91, 62,  9, 79, 22, 17, 35, 42, 13, 47, 82)
    (  6, 89, 63, 10, 77, 23, 18, 33, 43, 14, 45, 83)(  7, 90, 61, 11, 78, 21,
     19, 34, 41, 15, 46, 81)(  8, 92, 64, 12, 80, 24, 20, 36, 44, 16, 48, 84)
    ( 25, 87, 94, 49, 71, 66, 97, 39, 30, 73, 55, 58)( 26, 85, 95, 50, 69, 67,
     98, 37, 31, 74, 53, 59)( 27, 86, 93, 51, 70, 65, 99, 38, 29, 75, 54, 57)
    ( 28, 88, 96, 52, 72, 68,100, 40, 32, 76, 56, 60), 
  (  1, 61,  9)(  2, 64, 11)(  3, 62, 12)(  4, 63, 10)(  5, 81, 85)
    (  6, 84, 87)(  7, 82, 88)(  8, 83, 86)( 13, 21, 57)( 14, 24, 59)
    ( 15, 22, 60)( 16, 23, 58)( 17, 41, 33)( 18, 44, 35)( 19, 42, 36)
    ( 20, 43, 34)( 25, 77, 89)( 26, 80, 91)( 27, 78, 92)( 28, 79, 90)
    ( 29, 97, 65)( 30,100, 67)( 31, 98, 68)( 32, 99, 66)( 38, 40, 39)
    ( 45, 53, 93)( 46, 56, 95)( 47, 54, 96)( 48, 55, 94)( 49, 73, 69)
    ( 50, 76, 71)( 51, 74, 72)( 52, 75, 70) ];

autGroups[12]:=
[ (  2,  3,  5,  8,  7,  4)(  6,  9, 13, 17, 14, 10)( 11, 12, 16, 19, 18, 15)
    ( 20, 96, 77, 58)( 21,100, 81, 64, 26, 97, 78, 62, 24,102, 83, 59)
    ( 22,103, 84, 61, 23, 98, 79, 65, 27, 99, 80, 60)( 25,108, 89, 71, 33,101,
     82, 70, 32,109, 90, 63)( 28,112, 93, 67, 29,104, 85, 74, 36,105, 86, 66)
    ( 30,111, 92, 75, 37,106, 87, 73, 35,113, 94, 68)( 31,114, 95, 72, 34,107,
     88, 76, 38,110, 91, 69)( 40, 41, 43, 46, 45, 42)( 44, 47, 51, 55, 52, 48)
    ( 49, 50, 54, 57, 56, 53), (  1, 65, 27)(  2, 64, 20)(  3, 58, 24)
    (  4, 74, 26)(  5, 62, 32)(  6, 61, 22)(  7, 76, 36)(  8, 70, 38)
    (  9, 60, 35)( 10, 75, 23)( 11, 71, 21)( 12, 59, 28)( 13, 73, 29)
    ( 14, 69, 37)( 15, 63, 33)( 16, 66, 30)( 17, 67, 31)( 18, 72, 25)
    ( 19, 68, 34)( 39,103, 84)( 40,102, 77)( 41, 96, 81)( 42,112, 83)
    ( 43,100, 89)( 44, 99, 79)( 45,114, 93)( 46,108, 95)( 47, 98, 92)
    ( 48,113, 80)( 49,109, 78)( 50, 97, 85)( 51,111, 86)( 52,107, 94)
    ( 53,101, 90)( 54,104, 87)( 55,105, 88)( 56,110, 82)( 57,106, 91) ];

autGroups[13]:=
[ (  2,  3,  5,  6,  7,  4)(  8, 15)(  9, 17, 12, 20, 14, 18)( 10, 19, 13, 21,
     11, 16)( 22,120, 99, 71, 29,106, 85, 78, 36,113, 92, 64)( 23,124,103, 77,
     35,107, 86, 82, 40,119, 98, 65)( 24,125,104, 74, 32,108, 87, 83, 41,116,
      95, 66)( 25,122,101, 76, 34,109, 88, 80, 38,118, 97, 67)
    ( 26,126,105, 72, 30,110, 89, 84, 42,114, 93, 68)( 27,123,102, 73, 31,111,
     90, 81, 39,115, 94, 69)( 28,121,100, 75, 33,112, 91, 79, 37,117, 96, 70)
    ( 44, 45, 47, 48, 49, 46)( 50, 57)( 51, 59, 54, 62, 56, 60)
    ( 52, 61, 55, 63, 53, 58), (  1, 69, 27)(  2, 70, 22)(  3, 64, 26)
    (  4, 65, 28)(  5, 68, 25)(  6, 67, 24)(  7, 66, 23)(  8, 83, 34)
    (  9, 84, 29)( 10, 78, 33)( 11, 79, 35)( 12, 82, 32)( 13, 81, 31)
    ( 14, 80, 30)( 15, 76, 41)( 16, 77, 36)( 17, 71, 40)( 18, 72, 42)
    ( 19, 75, 39)( 20, 74, 38)( 21, 73, 37)( 43,111, 90)( 44,112, 85)
    ( 45,106, 89)( 46,107, 91)( 47,110, 88)( 48,109, 87)( 49,108, 86)
    ( 50,125, 97)( 51,126, 92)( 52,120, 96)( 53,121, 98)( 54,124, 95)
    ( 55,123, 94)( 56,122, 93)( 57,118,104)( 58,119, 99)( 59,113,103)
    ( 60,114,105)( 61,117,102)( 62,116,101)( 63,115,100) ];

autGroups[14]:=
[ (  1, 13, 43, 96,141,169,109, 54, 24,  4)(  2, 14, 44, 95,142,170,110, 53,
      23,  3)(  5, 16, 46, 98,139,189,172,113, 55, 25)(  6, 15, 45, 97,140,
     190,171,114, 56, 26)(  7, 18, 48, 73, 37, 69,119,111, 57, 29)
    (  8, 17, 47, 74, 38, 70,120,112, 58, 30)(  9, 20, 50, 94,137,187,159,115,
     59, 27)( 10, 19, 49, 93,138,188,160,116, 60, 28)( 11, 22, 52, 99,143,192,
     173,117, 61, 32)( 12, 21, 51,100,144,191,174,118, 62, 31)
    ( 33, 63, 83,131, 89,135, 92,134, 86, 76)( 34, 64, 84,132, 90,136, 91,133,
     85, 75)( 35, 65,123,161,178,214,183,180,130, 77)( 36, 66,124,162,177,213,
     184,179,129, 78)( 39, 67,101,146,175,168,201,181,127, 81)
    ( 40, 68,102,145,176,167,202,182,128, 82)( 41, 72,122,126, 79)
    ( 42, 71,121,125, 80)( 87, 88)(103,149,194,164,203,198,154,105,147,156)
    (104,150,193,163,204,197,153,106,148,155)(107,151,196,165,206,210,185,207,
     199,157)(108,152,195,166,205,209,186,208,200,158)(211,216)(212,215), 
  (  1,  4,  5)(  2,  3,  6)(  7, 12, 10)(  8, 11,  9)( 13, 25, 33)
    ( 14, 26, 34)( 15, 23, 35)( 16, 24, 36)( 17, 27, 37)( 18, 28, 38)
    ( 19, 31, 41)( 20, 32, 42)( 21, 29, 39)( 22, 30, 40)( 43, 76, 86)
    ( 44, 75, 85)( 45, 77, 89)( 46, 78, 90)( 47, 73, 88)( 48, 74, 87)
    ( 49, 79, 83)( 50, 80, 84)( 51, 81, 92)( 52, 82, 91)( 53,103, 65)
    ( 54,104, 66)( 55, 93, 63)( 56, 94, 64)( 57,101, 67)( 58,102, 68)
    ( 59,107, 69)( 60,108, 70)( 61,105, 71)( 62,106, 72)( 95,133,128)
    ( 96,134,127)( 97,131,126)( 98,132,125)( 99,136,129)(100,135,130)
    (109,165,150)(110,166,149)(111,161,146)(112,162,145)(113,164,138)
    (114,163,137)(115,159,151)(116,160,152)(117,168,147)(118,167,148)
    (119,157,178)(120,158,177)(121,154,139)(122,153,140)(123,156,175)
    (124,155,176)(141,181,185)(142,182,186)(143,179,183)(144,180,184)
    (169,210,206)(170,209,205)(171,211,204)(172,212,203)(173,207,201)
    (174,208,202)(187,193,196)(188,194,195)(189,198,215)(190,197,216)
    (191,213,200)(192,214,199) ];

autGroups[15]:=
[ (  1,  3,  2,  4)(  7, 33, 50, 40, 13, 45, 26, 22)(  8, 34, 49, 39, 14, 46,
      25, 21)(  9, 32, 52, 37, 15, 44, 28, 19)( 10, 31, 51, 38, 16, 43, 27, 20
     )( 11, 35, 53, 41, 17, 47, 29, 23)( 12, 36, 54, 42, 18, 48, 30, 24)
    ( 55,165,410,484,121,369,296,226)( 56,166,409,483,122,370,295,225)
    ( 57,164,412,481,123,368,298,223)( 58,163,411,482,124,367,297,224)
    ( 59,167,413,485,125,371,299,227)( 60,168,414,486,126,372,300,228)
    ( 61,195,386,466,115,357,320,244)( 62,196,385,465,116,358,319,243)
    ( 63,194,388,463,117,356,322,241)( 64,193,387,464,118,355,321,242)
    ( 65,197,389,467,119,359,323,245)( 66,198,390,468,120,360,324,246)
    ( 67,207,416,448,109,327,272,262)( 68,208,415,447,110,328,271,261)
    ( 69,206,418,445,111,326,274,259)( 70,205,417,446,112,325,273,260)
    ( 71,209,419,449,113,329,275,263)( 72,210,420,450,114,330,276,264)
    ( 73,171,422,460,157,363,284,250)( 74,172,421,459,158,364,283,249)
    ( 75,170,424,457,159,362,286,247)( 76,169,423,458,160,361,285,248)
    ( 77,173,425,461,161,365,287,251)( 78,174,426,462,162,366,288,252)
    ( 79,183,398,442,151,351,290,268)( 80,184,397,441,152,352,289,267)
    ( 81,182,400,439,153,350,292,265)( 82,181,399,440,154,349,291,266)
    ( 83,185,401,443,155,353,293,269)( 84,186,402,444,156,354,294,270)
    ( 85,213,392,478,145,339,314,232)( 86,214,391,477,146,340,313,231)
    ( 87,212,394,475,147,338,316,229)( 88,211,393,476,148,337,315,230)
    ( 89,215,395,479,149,341,317,233)( 90,216,396,480,150,342,318,234)
    ( 91,177,380,436,139,375,308,256)( 92,178,379,435,140,376,307,255)
    ( 93,176,382,433,141,374,310,253)( 94,175,381,434,142,373,309,254)
    ( 95,179,383,437,143,377,311,257)( 96,180,384,438,144,378,312,258)
    ( 97,189,428,472,133,345,278,220)( 98,190,427,471,134,346,277,219)
    ( 99,188,430,469,135,344,280,217)(100,187,429,470,136,343,279,218)
    (101,191,431,473,137,347,281,221)(102,192,432,474,138,348,282,222)
    (103,201,404,454,127,333,302,238)(104,202,403,453,128,334,301,237)
    (105,200,406,451,129,332,304,235)(106,199,405,452,130,331,303,236)
    (107,203,407,455,131,335,305,239)(108,204,408,456,132,336,306,240), 
  (  1,413,484)(  2,414,483)(  3,410,486)(  4,409,485)(  5,412,481)
    (  6,411,482)(  7,431,460)(  8,432,459)(  9,428,462)( 10,427,461)
    ( 11,430,457)( 12,429,458)( 13,395,436)( 14,396,435)( 15,392,438)
    ( 16,391,437)( 17,394,433)( 18,393,434)( 19,389,472)( 20,390,471)
    ( 21,386,474)( 22,385,473)( 23,388,469)( 24,387,470)( 25,407,466)
    ( 26,408,465)( 27,404,468)( 28,403,467)( 29,406,463)( 30,405,464)
    ( 31,425,442)( 32,426,441)( 33,422,444)( 34,421,443)( 35,424,439)
    ( 36,423,440)( 37,419,478)( 38,420,477)( 39,416,480)( 40,415,479)
    ( 41,418,475)( 42,417,476)( 43,383,454)( 44,384,453)( 45,380,456)
    ( 46,379,455)( 47,382,451)( 48,381,452)( 49,401,448)( 50,402,447)
    ( 51,398,450)( 52,397,449)( 53,400,445)( 54,399,446)( 55,113,184)
    ( 56,114,183)( 57,110,186)( 58,109,185)( 59,112,181)( 60,111,182)
    ( 61,131,178)( 62,132,177)( 63,128,180)( 64,127,179)( 65,130,175)
    ( 66,129,176)( 67,149,208)( 68,150,207)( 69,146,210)( 70,145,209)
    ( 71,148,205)( 72,147,206)( 73,161,190)( 74,162,189)( 75,158,192)
    ( 76,157,191)( 77,160,187)( 78,159,188)( 79,125,166)( 80,126,165)
    ( 81,122,168)( 82,121,167)( 83,124,163)( 84,123,164)( 85,143,214)
    ( 86,144,213)( 87,140,216)( 88,139,215)( 89,142,211)( 90,141,212)
    ( 91,137,196)( 92,138,195)( 93,134,198)( 94,133,197)( 95,136,193)
    ( 96,135,194)( 97,155,172)( 98,156,171)( 99,152,174)(100,151,173)
    (101,154,169)(102,153,170)(103,119,202)(104,120,201)(105,116,204)
    (106,115,203)(107,118,199)(108,117,200)(217,311,352)(218,312,351)
    (219,308,354)(220,307,353)(221,310,349)(222,309,350)(223,275,328)
    (224,276,327)(225,272,330)(226,271,329)(227,274,325)(228,273,326)
    (229,293,376)(230,294,375)(231,290,378)(232,289,377)(233,292,373)
    (234,291,374)(235,305,358)(236,306,357)(237,302,360)(238,301,359)
    (239,304,355)(240,303,356)(241,323,334)(242,324,333)(243,320,336)
    (244,319,335)(245,322,331)(246,321,332)(247,287,364)(248,288,363)
    (249,284,366)(250,283,365)(251,286,361)(252,285,362)(253,281,346)
    (254,282,345)(255,278,348)(256,277,347)(257,280,343)(258,279,344)
    (259,299,340)(260,300,339)(261,296,342)(262,295,341)(263,298,337)
    (264,297,338)(265,317,370)(266,318,369)(267,314,372)(268,313,371)
    (269,316,367)(270,315,368) ];

autGroups[16]:=
[ (  2,  3,  5,  6,  7,  4)(  8, 15, 22)(  9, 17, 26, 13, 21, 25)
    ( 10, 19, 27, 14, 18, 23)( 11, 16, 24, 12, 20, 28)( 29,148,113,106, 50,
     162,127, 99, 43,141,120, 85)( 30,152,117,112, 56,163,128,103, 47,147,126,
     86)( 31,153,118,109, 53,164,129,104, 48,144,123, 87)( 32,150,115,111, 55,
     165,130,101, 45,146,125, 88)( 33,154,119,107, 51,166,131,105, 49,142,121,
     89)( 34,151,116,108, 52,167,132,102, 46,143,122, 90)( 35,149,114,110, 54,
     168,133,100, 44,145,124, 91)( 36,155,134, 92)( 37,159,138, 98, 42,156,
     135, 96, 40,161,140, 93)( 38,160,139, 95, 39,157,136, 97, 41,158,137, 94)
    ( 57, 78, 71)( 58, 80, 75, 62, 84, 74)( 59, 82, 76, 63, 81, 72)
    ( 60, 79, 73, 61, 83, 77)( 65, 66, 68, 69, 70, 67), 
  (  1, 90, 34)(  2, 91, 29)(  3, 85, 33)(  4, 86, 35)(  5, 89, 32)
    (  6, 88, 31)(  7, 87, 30)(  8,111, 41)(  9,112, 36)( 10,106, 40)
    ( 11,107, 42)( 12,110, 39)( 13,109, 38)( 14,108, 37)( 15, 97, 48)
    ( 16, 98, 43)( 17, 92, 47)( 18, 93, 49)( 19, 96, 46)( 20, 95, 45)
    ( 21, 94, 44)( 22,104, 55)( 23,105, 50)( 24, 99, 54)( 25,100, 56)
    ( 26,103, 53)( 27,102, 52)( 28,101, 51)( 57,146,139)( 58,147,134)
    ( 59,141,138)( 60,142,140)( 61,145,137)( 62,144,136)( 63,143,135)
    ( 64,167,132)( 65,168,127)( 66,162,131)( 67,163,133)( 68,166,130)
    ( 69,165,129)( 70,164,128)( 71,153,125)( 72,154,120)( 73,148,124)
    ( 74,149,126)( 75,152,123)( 76,151,122)( 77,150,121)( 78,160,118)
    ( 79,161,113)( 80,155,117)( 81,156,119)( 82,159,116)( 83,158,115)
    ( 84,157,114) ];

autGroups[17]:=
[ (  2,  3,  5,  8,  7,  4)(  6,  9, 13, 18, 14, 10)( 11, 12, 16, 21, 20, 15)
    ( 17, 22, 26, 27, 28, 23)( 19, 24, 29, 31, 30, 25)( 32,156,125, 94)
    ( 33,160,129,100, 38,157,126, 98, 36,162,131, 95)( 34,163,132, 97, 35,158,
     127,101, 39,159,128, 96)( 37,168,137,107, 45,161,130,106, 44,169,138, 99)
    ( 40,173,142,103, 41,164,133,111, 49,165,134,102)( 42,171,140,113, 51,166,
     135,109, 47,175,144,104)( 43,176,145,108, 46,167,136,114, 52,170,139,105)
    ( 48,181,150,121, 59,172,141,119, 57,183,152,110)( 50,184,153,123, 61,174,
     143,122, 60,185,154,112)( 53,182,151,116, 54,177,146,120, 58,178,147,115)
    ( 55,186,155,118, 56,179,148,124, 62,180,149,117)( 64, 65, 67, 70, 69, 66)
    ( 68, 71, 75, 80, 76, 72)( 73, 74, 78, 83, 82, 77)( 79, 84, 88, 89, 90, 85
     )( 81, 86, 91, 93, 92, 87), (  1,101, 39)(  2,100, 32)(  3, 94, 36)
    (  4,111, 38)(  5, 98, 44)(  6, 97, 34)(  7,114, 49)(  8,106, 52)
    (  9, 96, 47)( 10,113, 35)( 11,107, 33)( 12, 95, 40)( 13,109, 57)
    ( 14,124, 51)( 15,120, 45)( 16,102, 60)( 17,103, 43)( 18,119, 62)
    ( 19,108, 37)( 20,116, 58)( 21,122, 54)( 22,105, 59)( 23,123, 41)
    ( 24, 99, 53)( 25,110, 46)( 26,121, 42)( 27,104, 55)( 28,117, 61)
    ( 29,115, 50)( 30,118, 48)( 31,112, 56)( 63,163,132)( 64,162,125)
    ( 65,156,129)( 66,173,131)( 67,160,137)( 68,159,127)( 69,176,142)
    ( 70,168,145)( 71,158,140)( 72,175,128)( 73,169,126)( 74,157,133)
    ( 75,171,150)( 76,186,144)( 77,182,138)( 78,164,153)( 79,165,136)
    ( 80,181,155)( 81,170,130)( 82,178,151)( 83,184,147)( 84,167,152)
    ( 85,185,134)( 86,161,146)( 87,172,139)( 88,183,135)( 89,166,148)
    ( 90,179,154)( 91,177,143)( 92,180,141)( 93,174,149) ];

autGroups[18]:=
[ (  2,  3,  5,  8, 14, 24, 40, 34, 20, 12,  7,  4)(  6,  9, 15, 26, 25, 41,
      57, 58, 45, 28, 16, 10)( 11, 17, 29, 46, 30, 18)( 13, 21, 35, 51, 36, 22
     )( 19, 31, 43, 27, 42, 52, 53, 38, 23, 37, 44, 32)( 33, 48, 61, 60, 47,
      59, 64, 63, 55, 39, 54, 49)( 50, 56, 62)( 65,129)( 66,133, 69,142, 78,
     168,104,148, 84,135, 71,130)( 67,136, 72,152, 88,162, 98,140, 76,132, 68,
     131)( 70,143, 79,153, 89,185,121,173,109,144, 80,134)( 73,154, 90,169,
     105,186,122,156, 92,138, 74,137)( 75,157, 93,158, 94,139)
    ( 77,163, 99,164,100,141)( 81,174,110,146, 82,145)( 83,171,107,170,106,
     181,117,151, 87,172,108,147)( 85,179,115,150, 86,149)( 91,180,116,166,
     102,165,101,160, 96,159, 95,155)( 97,189,125,175,111,192,128,183,119,182,
     118,161)(103,177,113,176,112,188,124,187,123,191,127,167)
    (114,190,126,184,120,178), (  1,131, 67)(  2,133, 65)(  3,129, 66)
    (  4,137, 69)(  5,130, 70)(  6,136, 68)(  7,145, 73)(  8,134, 77)
    (  9,132, 75)( 10,149, 72)( 11,143, 71)( 12,159, 81)( 13,142, 74)
    ( 14,141, 87)( 15,139, 89)( 16,170, 85)( 17,135, 83)( 18,169, 79)
    ( 19,157, 76)( 20,176, 95)( 21,138, 91)( 22,165, 78)( 23,152, 86)
    ( 24,151,103)( 25,154, 82)( 26,153, 90)( 27,163, 80)( 28,160,106)
    ( 29,147,109)( 30,187,105)( 31,140, 97)( 32,156, 93)( 33,171, 84)
    ( 34,184,112)( 35,155,113)( 36,181,101)( 37,150,116)( 38,182, 88)
    ( 39,168,102)( 40,167,120)( 41,146,111)( 42,144,108)( 43,161, 99)
    ( 44,180, 92)( 45,174, 96)( 46,173,123)( 47,185, 94)( 48,148,114)
    ( 49,179,107)( 50,189, 98)( 51,177,117)( 52,172,100)( 53,164,118)
    ( 54,166,115)( 55,190,104)( 56,162,119)( 57,175,128)( 58,192,110)
    ( 59,158,122)( 60,191,121)( 61,178,127)( 62,183,125)( 63,188,126)
    ( 64,186,124) ];

	gens[1]:=[ 1, 8, 42 ];
	gens[2]:=[ 1, 10, 14 ];
	gens[3]:=[ 1, 9, 23 ];
	gens[4]:=[ 1, 55, 109 ];
	gens[5]:=[ 1, 10, 19 ];
	gens[6]:=[ 1, 17, 47 ];
	gens[7]:=[ 1, 22, 76 ];
	gens[8]:=[ 1, 15, 41 ];
	gens[9]:=[ 1, 4, 7 ];
	gens[10]:=[ 1, 13, 102 ];
	gens[11]:=[ 1, 7, 62 ];
	gens[12]:=[ 1, 21, 59 ];
	gens[13]:=[ 1, 23, 65 ];
	gens[14]:=[ 1, 53, 88 ];
	gens[15]:=[ 1, 76, 408 ];
	gens[16]:=[ 1, 30, 86 ];
	gens[17]:=[ 1, 33, 95 ];
	gens[18]:=[ 1, 66, 130 ];

	mg:=[];
	mg[1]:=[ 8, 10 ];
	mg[2]:=[ 7, 17 ];
	mg[3]:=[ 12, 22 ];
	mg[4]:=[ 8, 28 ];
	mg[5]:=[ 18, 28 ];
	mg[6]:=[ 12, 33 ];
	mg[7]:=[ 8, 37 ];
	mg[8]:=[ 12, 40 ];
	mg[9]:=[ 9, 43 ];
	mg[10]:=[ 8, 46 ];
	mg[11]:=[ 12, 51 ];
	mg[12]:=[ 12, 58 ];
	mg[13]:=[ 12, 64 ];
	mg[14]:=[ 10, 73 ];
	mg[15]:=[ 8, 82 ];
	mg[16]:=[ 12, 85 ];
	mg[17]:=[ 12, 94 ];
	mg[18]:=[ 12, 97 ];
	
	i:=Position(mg,[m,g]);
	if i <> fail then
		G:=Group(autGroups[i]);
		c:=SCFromGenerators(G,[gens[i]]);
		SetSCTopologicalType(c,Concatenation("(T^2)^#",String(g)));
		SCRename(c,Concatenation("Chiral map {",String(m),",",String(g),"}"));
		return c;
	else
		return fail;
	fi;
end);
