#############################################################################
##
#W  ctprodclass.g               GAP4 Package `RCWA'               Stefan Kohl
##
##  This file contains a classification of all products of two different
##  class transpositions which interchange residue classes with moduli <= 6,
##  by
##
##  - intersection relations between the two times two residue classes,
##  - order, and
##  - cycle lengths.
##
##  The list `CTPairsProductClassification' has an entry for any of the 17
##  possible intersection types. Any of these entries are lists, as follows:
##
##  - The first entry is a list of four integers in the range 0 .. 4, which
##    specifies the intersection type. Let cl1 and cl2 be the residue classes
##    which are interchanged by the first class transposition, and let cl3
##    and cl4 be the residue classes which are interchanged by the second
##    class transposition. Then cl1 and cl2 are disjoint by definition, and
##    the same holds for cl3 and cl4. The first entry of the list specifies
##    the intersection relation between cl1 and cl3, the second entry speci-
##    fies the intersection relation between cl2 and cl3, the third entry
##    specifies the intersection relation between cl1 and cl4, and the fourth
##    entry specifies the intersection relation between cl2 and cl4:
##
##    cl1    cl2  --> transposed by first class transposition 
##     | \  / |
##    1|  /   |4
##     |/2 3\ |
##    cl3    cl4  --> transposed by second class transposition 
##
##    The meaning of the entries is as follows:
##
##    0 = equality,     1 = subset relation, 2 = superset relation,
##    3 = disjointness, 4 = general intersection relation.
##
##  - The subsequent entries are lists, one for each possible order of the
##    product, as follows:
##
##    - The first entry is the order <ord> of the product.
##
##    - The subsequent entries are lists, one for each possible cycle type
##      of the product, provided that it has order <ord>:
##
##      The first entry of such a list is the cycle type. The cycle type
##      is a list of length 2. The first entry is the set of cycle lengths.
##      All of them occur infinitely often. The second entry is the number
##      of fixed points which stem from fixed points of non-identity affine
##      partial mappings.
##
##      The subsequent entries are the corresponding pairs of
##      class transpositions.
##
##  The function `CTPairsIntersectionTypes' has been used to determine the 
##  18 possible intersection types. These are the 17 which occur in the list
##  `CTPairsProductClassification' and [0,3,3,0], which stands for two equal
##  class transpositions. The list returned by this function is
##
##  [ [ 0, 3, 3, 0 ], [ 0, 3, 3, 1 ], [ 0, 3, 3, 3 ], [ 0, 3, 3, 4 ], 
##    [ 1, 1, 3, 3 ], [ 1, 3, 3, 1 ], [ 1, 3, 3, 2 ], [ 1, 3, 3, 3 ], 
##    [ 1, 3, 3, 4 ], [ 1, 4, 3, 2 ], [ 1, 4, 3, 3 ], [ 1, 4, 3, 4 ], 
##    [ 3, 3, 3, 3 ], [ 3, 3, 3, 4 ], [ 3, 3, 4, 4 ], [ 3, 4, 4, 3 ], 
##    [ 3, 4, 4, 4 ], [ 4, 4, 4, 4 ] ]
##
##  Given either 2 class transpositions or a 4-tuple of residue classes where
##  the first and the second respectively the third and the fourth are
##  disjoint, the function `CTPairIntersectionType' returns the corresponding
##  intersection type, as described above.
##
##  Given either 2 class transpositions or a 4-tuple of residue classes as
##  described in the last sentence, the function `CTPairProductType' returns
##  a list [ <intersection type>, <order>, <cycle type> ], where the entries
##  have the meanings described above.
##
##  The list `CTPairsProductType' contains the precomputed values of
##  `CTPairProductType' for all pairs of class transpositions which
##  interchange residue classes with moduli <= 6. More precisely, we have
##  CTPairsProductTypes = List(CTPairs,CTPairProductType);.
##
##  The entry at position (i,j) of the matrix `OrdersMatrix' is the order
##  of the product of the i-th and the j-th class transposition in the list
##  List(ClassPairs(6),ClassTransposition);.
##
##  For the description of the components CTProds12 and CTProds32, see the
##  files ctprods12.g and ctprods32.g, respectively.
##
#############################################################################

return rec(

CTPairs := Combinations(List(ClassPairs(6),ClassTransposition),2),

CTPairsIntersectionTypes := function ()

  local  types, modV4, subsetnormal;

  modV4 := type->Set([type,Permuted(type,(1,2)(3,4)),
                           Permuted(type,(1,3)(2,4)),
                           Permuted(type,(1,4)(2,3))])[1];

  subsetnormal := function ( type )

    local  i;

    if 2 in type and not 1 in type then
      for i in [1..4] do if type[i] = 2 then type[i] := 1; fi; od;
    fi;
    return type;
  end;

  types := Tuples([0..4],4);
  types := Set(types,modV4);
  types := Filtered(types,type->Intersection([type{[1,2]},type{[3,4]},
                                              type{[1,3]},type{[2,4]}],
                                             [[0,0],[1,2],[2,1]]) = []);
  types := Set(types,subsetnormal);
  types := Filtered(types,type->Number(type,n->n=1)<3
                           and (Number(type,n->n=1)<2
                             or Intersection(type,[0,4])=[]));
  types := Filtered(types,type->type{[1,2]} in [[0,3],[1,1],[1,3],
                                                [1,4],[3,3],[3,4],[4,4]]);
  types := Filtered(types,type->not type{[1,3]} in [[0,1],[1,1],[1,4]]);
  types := Filtered(types,type->type[1]<>0 or not 4 in type{[2,3]});
  types := Difference(types,[[3,4,3,4]]);

  return types;
end,

CTPairIntersectionType := function ( arg )

  local  cts, cls, type, indices, pair,
         equivalentsV4, equivalentslist, i;

  if Length(arg) = 1 then arg := arg[1]; fi;

  if   Length(arg) = 2 and ForAll(arg,IsClassTransposition) then
    cts := arg;
    cls := Concatenation(List(cts,TransposedClasses));
  elif Length(arg) = 4 and ForAll(arg,IsResidueClassOfZ) then
    cls := arg;
    cts := List([cls{[1,2]},cls{[3,4]}],ClassTransposition);
  else Error("usage: CTPairIntersectionType(<2 class transpositions>|",
                                           "<4 residue classes>)\n");
  fi;

  type := [];
  for indices in [[1,3],[1,4],[2,3],[2,4]] do
    pair := cls{indices};
    if   pair[1] = pair[2]                  then Add(type,0);
    elif IsSubset(pair[1],pair[2])          then Add(type,1);
    elif IsSubset(pair[2],pair[1])          then Add(type,2);
    elif Intersection(pair[1],pair[2]) = [] then Add(type,3);
    else                                         Add(type,4); fi;
  od;

  equivalentsV4 := Set([type,Permuted(type,(1,2)(3,4)),
                             Permuted(type,(1,3)(2,4)),
                             Permuted(type,(1,4)(2,3))]);
  type := equivalentsV4[1];

  equivalentslist := [[[0,3,3,2],[0,3,3,1]],
                      [[2,3,2,3],[1,1,3,3]],
                      [[2,3,3,2],[1,3,3,1]],
                      [[2,3,3,3],[1,3,3,3]],
                      [[2,3,3,4],[1,3,3,4]],
                      [[2,3,4,3],[1,4,3,3]],
                      [[2,3,4,4],[1,4,3,4]],
                      [[3,4,3,4],[3,3,4,4]]];

  for i in [1..8] do
    if   type  = equivalentslist[i][1]
    then type := equivalentslist[i][2]; fi;
  od;

  return type;
end,

CTPairProductType := function ( arg )

  local  result, cls, cts, g, intertype, order, cycletype,
         CTPairIntersectionType;

  if Length(arg) = 1 then arg := arg[1]; fi;

  if   Length(arg) = 2 and ForAll(arg,IsClassTransposition) then
    cts := arg;
    cls := Concatenation(List(cts,TransposedClasses));
  elif Length(arg) = 4 and ForAll(arg,IsResidueClassOfZ) then
    cls := arg;
    cts := List([cls{[1,2]},cls{[3,4]}],ClassTransposition);
  else Error("usage: CTPairProductType(<2 class transpositions>|",
                                      "<4 residue classes>)\n");
  fi;

  if   IsBoundGlobal("CTPairIntersectionType")
  then CTPairIntersectionType := ValueGlobal("CTPairIntersectionType");
  else CTPairIntersectionType :=
       LoadDatabaseOfProductsOf2ClassTranspositions().CTPairIntersectionType;
  fi;

  intertype := CTPairIntersectionType(cls);
  g         := Product(cts);
  order     := Order(g);
  cycletype := Set(ShortCycles(g,[-1000..1000],20),Length);

  if Density(Support(g)) = 1 then cycletype := Difference(cycletype,[1]); fi;

  if   IsSubset(cycletype,[1,2,3,4,5,6,7])
  then cycletype := PositiveIntegers;
  elif IsSubset(cycletype,[1,3,5,7])
  then cycletype := ResidueClass(1,2);
  elif IsSubset(cycletype,[2,4,6,8]) and Density(Support(g)) = 1
  then cycletype := ResidueClass(0,2);
  elif IsSubset(cycletype,[2,4,6,8]) and Density(Support(g)) < 1
  then cycletype := Union(ResidueClass(0,2),[1]);
  elif order = infinity then Add(cycletype,infinity); fi;

  cycletype := [cycletype,Length(ExcludedElements(Support(g)))];

  return [ intertype, order, cycletype ];
end,

CTPairsProductClassification :=
[ [ [ 0, 3, 3, 1 ], 
    [ infinity,
          [ [ [ infinity ], 2 ],
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,2,1,4) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,2,3,4) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,2,1,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,2,5,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(1,2,0,4) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(1,2,2,4) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(1,2,0,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(1,2,4,6) ] ],
          [ [ [ 1, infinity ], 2 ],
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,4,1,4) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(0,4,1,4) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(0,3,1,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(0,3,4,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(1,3,0,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(1,3,3,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(0,3,2,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(0,3,5,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(2,3,0,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(2,3,3,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(1,3,2,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(1,3,5,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(2,3,1,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(2,3,4,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(4,6,5,6) ] ], 
          [ [ [ infinity ], 0 ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,2,3,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(1,2,2,6) ] ], 
          [ [ [ 1, infinity ], 0 ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(3,6,4,6) ] ] ] ], 
  [ [ 0, 3, 3, 3 ],
    [ 3,
          [ [ [ 3 ], 0 ],
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,2,3,4) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(1,2,2,4) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(0,3,2,3) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(1,3,2,3) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(1,3,2,3) ] ], 
          [ [ [ 1, 3 ], 0 ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(0,2,3,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(0,2,5,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(0,2,5,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(1,2,2,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(1,2,4,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(1,2,4,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(0,3,2,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(0,3,5,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(1,3,2,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(1,3,5,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(0,3,1,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(0,3,4,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(2,3,1,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(2,3,4,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(0,3,2,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(0,3,4,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(0,3,5,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(2,3,1,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(0,3,4,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(0,3,5,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(1,3,2,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(0,3,5,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(2,3,4,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(1,3,5,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(1,3,0,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(1,3,3,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(2,3,0,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(2,3,3,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(1,3,2,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(1,3,3,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(1,3,5,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(2,3,0,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(1,3,3,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(1,3,5,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(1,3,5,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(2,3,3,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(2,3,1,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(2,3,3,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(2,3,4,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(2,3,3,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(2,3,4,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(2,3,4,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,5,1,5), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(0,5,1,5), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(0,5,1,5), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(0,5,1,5), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(0,5,1,5), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(0,5,1,5), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(0,5,2,5), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(0,5,2,5), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(0,5,2,5), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(0,5,2,5), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(0,5,2,5), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(0,5,3,5), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(0,5,3,5), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(0,5,3,5), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(0,5,3,5), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(0,5,4,5), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(0,5,4,5), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(0,5,4,5), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(1,5,2,5), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(1,5,2,5), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(1,5,2,5), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(1,5,2,5), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(1,5,3,5), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(1,5,3,5), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(1,5,3,5), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(1,5,4,5), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(1,5,4,5), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(2,5,3,5), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(2,5,3,5), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(2,5,4,5), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(0,6,1,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,6,1,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(0,6,1,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(0,6,1,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(0,6,1,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,6,1,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,6,1,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,6,1,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,6,2,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(0,6,2,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(0,6,2,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(0,6,2,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,6,2,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,6,2,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,6,2,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,6,3,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(0,6,3,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(0,6,3,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,6,3,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,6,3,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(0,6,3,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(0,6,4,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(0,6,4,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,6,4,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,6,4,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(0,6,4,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(0,6,5,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,6,5,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,6,5,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(0,6,5,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(1,6,2,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(1,6,2,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(1,6,2,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(1,6,2,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(1,6,2,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(1,6,2,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(1,6,3,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(1,6,3,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(1,6,3,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(1,6,3,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(1,6,3,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(1,6,4,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(1,6,4,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(1,6,4,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(1,6,4,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(1,6,5,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(1,6,5,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(1,6,5,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(2,6,3,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(2,6,3,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(2,6,3,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(2,6,3,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(2,6,4,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(2,6,4,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(2,6,4,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(2,6,5,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(2,6,5,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(3,6,4,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(3,6,4,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(3,6,5,6), ClassTransposition(4,6,5,6) ] ] ] ], 
  [ [ 0, 3, 3, 4 ],
    [ infinity,
          [ [ ResidueClass(1,2), 2 ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,2,1,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,2,3,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,2,5,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(0,2,1,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(0,2,3,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(0,2,5,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(0,3,1,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(2,3,1,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(1,3,3,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(2,3,3,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(0,3,5,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(1,3,5,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(1,2,0,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(1,2,2,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(1,2,4,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(1,2,0,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(1,2,2,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(1,2,4,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(1,3,0,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(2,3,0,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(0,3,2,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(1,3,2,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(0,3,4,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(2,3,4,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(4,6,5,6) ] ] ] ], 
  [ [ 1, 1, 3, 3 ],
    [ 4,
          [ [ [ 4 ], 0 ],
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(1,4,3,4) ] ], 
          [ [ [ 2, 4 ], 0 ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(3,6,5,6) ] ], 
          [ [ [ 1, 4 ], 0 ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(2,6,5,6) ] ], 
          [ [ [ 1, 2, 4 ], 0 ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(3,6,5,6) ] ] ] ], 
  [ [ 1, 3, 3, 1 ],
    [ 2,
          [ [ [ 1, 2 ], 0 ],
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,4,1,4) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(4,6,5,6) ] ] ], 
    [ 4,
          [ [ [ 4 ], 0 ],
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,4,3,4) ],
              [ ClassTransposition(0,2,1,2), ClassTransposition(1,4,2,4) ] ], 
          [ [ [ 2, 4 ], 0 ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(3,6,4,6) ] ], 
          [ [ [ 1, 4 ], 0 ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(2,6,4,6) ] ] ], 
    [ infinity,
          [ [ ResidueClass(0,2), 2 ],
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(3,4,4,6) ] ] ] ], 
  [ [ 1, 3, 3, 2 ],
    [ infinity,
          [ [ [ infinity ], 2 ],
              [ ClassTransposition(0,2,1,4), ClassTransposition(1,2,0,4) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(1,2,0,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(1,2,2,4) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(1,2,4,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(1,2,0,4) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(1,2,0,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(1,2,2,4) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(1,2,4,6) ] ],
          [ [ [ 1, infinity ], 2 ],
              [ ClassTransposition(0,3,1,6), ClassTransposition(1,3,0,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(2,3,0,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(1,3,3,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(2,3,3,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(2,3,1,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(2,3,4,6) ] ], 
          [ [ [ infinity ], 0 ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(1,2,2,4) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(1,2,2,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(1,2,4,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(1,2,0,4) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(1,2,0,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(1,2,2,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(1,2,2,4) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(1,2,2,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(1,2,4,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(1,2,0,4) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(1,2,2,4) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(1,2,0,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(1,2,2,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(1,2,4,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(1,2,0,4) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(1,2,0,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(1,2,2,6) ] ], 
          [ [ [ 1, infinity ], 0 ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(1,3,3,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(2,3,3,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(1,3,0,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(2,3,0,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(2,3,4,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(2,3,1,6) ] ] ] ], 
  [ [ 1, 3, 3, 3 ],
    [ 6,
          [ [ [ 2, 3 ], 0 ],
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(0,4,1,4) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(0,4,1,4) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(2,3,0,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(2,3,1,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(2,3,3,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(2,3,4,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(1,3,0,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(1,3,2,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(1,3,3,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(1,3,5,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(1,3,2,3) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(1,3,2,3) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(1,3,2,3) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(1,3,2,3) ] ], 
          [ [ [ 1, 2, 3 ], 0 ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(1,3,2,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(1,3,5,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(2,3,0,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(2,3,3,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(1,3,0,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(1,3,3,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(2,3,1,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(2,3,4,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(1,3,2,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(1,3,5,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(2,3,0,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(2,3,3,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(1,3,0,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(1,3,3,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(2,3,1,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(2,3,4,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(2,3,1,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(2,3,4,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(2,3,0,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(2,3,3,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(2,3,1,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(2,3,4,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(2,3,0,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(2,3,3,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(3,6,5,6) ] ] ] ], 
  [ [ 1, 3, 3, 4 ],
    [ 12,
          [ [ [ 1, 2, 3, 4 ], 0 ],
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(3,6,4,6) ] ], 
          [ [ [ 1, 3, 4 ], 0 ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(0,4,1,4) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(0,4,1,4) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(1,4,2,4) ] ] ], 
    [ infinity,
          [ [ PositiveIntegers, 2 ],
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(2,4,5,6) ] ], 
          [ [ [ 1, 2, infinity ], 2 ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(0,4,1,4) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(0,4,1,4) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(2,4,3,4) ] ], 
          [ [ PositiveIntegers, 0 ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(2,4,3,6) ] ], 
          [ [ [ 1, 2, infinity ], 0 ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(0,4,1,4) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(0,4,1,4) ] ] ] ], 
  [ [ 1, 4, 3, 2 ],
    [ infinity,
          [ [ [ 1, infinity ], 2 ],
              [ ClassTransposition(0,2,1,6), ClassTransposition(1,3,0,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(0,3,4,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(2,3,4,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(0,3,1,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(2,3,1,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(1,3,5,6) ] ], 
          [ [ [ 1, infinity ], 0 ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(1,3,2,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(0,3,2,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(2,3,0,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(0,3,5,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(2,3,3,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(1,3,3,6) ] ] ] ], 
  [ [ 1, 4, 3, 3 ],
    [ 12,
          [ [ [ 1, 2, 3, 4 ], 0 ],
              [ ClassTransposition(0,2,1,6), ClassTransposition(0,3,2,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(0,3,4,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(2,3,0,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(2,3,4,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(1,3,0,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(1,3,2,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(2,3,0,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(2,3,4,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(0,3,2,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(0,3,4,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(1,3,0,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(1,3,2,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(1,3,3,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(1,3,5,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(2,3,1,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(2,3,3,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(0,3,1,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(0,3,5,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(1,3,3,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(1,3,5,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(0,3,1,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(0,3,5,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(2,3,1,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(2,3,3,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(3,4,2,6) ] ], 
          [ [ [ 1, 3, 4 ], 0 ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(1,3,3,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(1,3,5,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(0,3,1,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(0,3,5,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(2,3,1,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(2,3,3,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(0,3,2,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(0,3,4,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(2,3,0,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(2,3,4,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(1,3,0,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(1,3,2,6) ] ] ] ], 
  [ [ 1, 4, 3, 4 ],
    [ 4,
          [ [ [ 4 ], 0 ],
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,3,4,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,3,5,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(1,3,2,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(1,3,3,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(2,3,0,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(2,3,1,6) ] ] ], 
    [ 6,
          [ [ [ 1, 2, 3, 6 ], 0 ],
              [ ClassTransposition(0,2,1,4), ClassTransposition(2,3,0,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(2,3,4,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(0,3,2,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(0,3,4,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(2,3,1,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(2,3,3,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(0,3,1,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(0,3,5,6) ] ], 
          [ [ [ 1, 3, 6 ], 0 ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(0,3,2,3) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(0,3,2,3) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(0,3,2,3) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(0,3,2,3) ] ] ], 
    [ 12, [ [ [ 1, 3, 4 ], 0 ],
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,3,2,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(1,3,2,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(1,3,0,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(2,3,0,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(0,3,5,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(1,3,5,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(1,3,3,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(2,3,3,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(3,4,2,6) ] ], 
          [ [ [ 1, 3, 4, 6 ], 0 ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(0,4,5,6) ] ] ], 
    [ infinity,
          [ [ [ 2, infinity ], 2 ],
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,3,1,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,3,2,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(1,3,0,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(1,3,5,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(2,3,3,6) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(2,3,4,6) ] ], 
          [ [ [ 1, 2, infinity ], 2 ],
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,3,4,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(1,3,0,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(2,3,4,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(0,3,1,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(1,3,5,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(2,3,1,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(2,4,5,6) ] ], 
          [ [ [ 1, 2, infinity ], 0 ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(1,3,2,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(1,3,3,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(2,4,5,6) ] ], 
          [ [ [ 1, infinity ], 2 ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(0,3,1,3) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(0,3,1,3) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(1,3,2,3) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(0,3,1,3) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(1,3,2,3) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(1,3,2,3) ] ], 
          [ [ [ 1, infinity ], 0 ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(1,3,2,3) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(0,3,1,3) ] ], 
          [ [ PositiveIntegers, 2 ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(3,4,4,6) ] ] ] ], 
  [ [ 3, 3, 3, 3 ],
    [ 2,
          [ [ [ 2 ], 0 ],
              [ ClassTransposition(0,2,1,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(2,3,4,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(1,3,5,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(2,3,1,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(1,3,2,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(2,3,3,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(2,3,0,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(1,4,2,4) ] ], 
          [ [ [ 1, 2 ], 0 ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,5,1,5), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(0,5,1,5), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(0,5,1,5), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(0,5,2,5), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(0,5,2,5), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(0,5,2,5), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(0,5,3,5), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(0,5,3,5), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(0,5,3,5), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(0,5,4,5), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(0,5,4,5), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(0,5,4,5), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(1,5,2,5), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(1,5,3,5), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(1,5,4,5), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(0,6,1,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,6,1,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,6,1,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,6,1,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(0,6,1,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(0,6,1,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(0,6,2,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,6,2,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,6,2,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,6,2,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(0,6,2,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(0,6,2,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(0,6,3,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,6,3,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,6,3,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,6,3,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,6,3,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,6,3,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(0,6,4,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,6,4,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,6,4,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,6,4,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,6,4,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,6,4,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(0,6,5,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,6,5,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,6,5,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,6,5,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,6,5,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,6,5,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(1,6,2,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(1,6,2,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(1,6,2,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(1,6,3,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(1,6,3,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(1,6,3,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(1,6,4,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(1,6,4,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(1,6,4,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(1,6,5,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(1,6,5,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(1,6,5,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(2,6,3,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(2,6,4,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(2,6,5,6), ClassTransposition(3,6,4,6) ] ] ] ], 
  [ [ 3, 3, 3, 4 ],
    [ 6,
          [ [ [ 2, 3 ], 0 ],
              [ ClassTransposition(0,2,1,6), ClassTransposition(0,3,5,6) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(2,3,3,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(1,3,5,6) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(2,3,1,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(0,3,1,6) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(1,3,3,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(1,3,2,6) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(2,3,4,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(0,3,4,6) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(1,3,0,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(0,3,2,6) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(2,3,0,6) ] ], 
          [ [ [ 1, 2, 3 ], 0 ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(2,6,5,6) ] ] ] ], 
  [ [ 3, 3, 4, 4 ],
    [ 4,
          [ [ [ 1, 2, 4 ], 0 ],
              [ ClassTransposition(0,2,1,4), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(1,6,5,6) ] ], 
          [ [ [ 2, 4 ], 0 ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(0,3,2,3) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(0,3,2,3) ] ] ], 
      [ 6,
          [ [ [ 1, 2, 3 ], 0 ],
              [ ClassTransposition(0,2,1,4), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(3,6,5,6) ] ], 
          [ [ [ 2, 3 ], 0 ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(1,3,2,3) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(0,3,1,3) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(1,3,2,3) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(0,3,1,3) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(0,4,2,4) ] ] ], 
    [ 12,
          [ [ [ 1, 2, 3, 4 ], 0 ],
              [ ClassTransposition(0,3,1,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(0,4,5,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(0,4,3,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(0,4,1,6) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(0,4,3,6) ] ] ] ], 
  [ [ 3, 4, 4, 3 ],
    [ 2,
          [ [ [ 1, 2 ], 0 ],
              [ ClassTransposition(0,4,1,4), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(4,6,5,6) ] ] ], 
    [ 4,
          [ [ [ 1, 2, 4 ], 0 ],
              [ ClassTransposition(0,4,1,4), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(1,6,4,6) ] ] ], 
    [ 6,
          [ [ [ 1, 2, 3 ], 0 ],
              [ ClassTransposition(0,4,1,4), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(3,6,4,6) ] ] ], 
    [ infinity,
          [ [ PositiveIntegers, 2 ],
              [ ClassTransposition(0,4,1,6), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(3,4,4,6) ] ], 
          [ [ PositiveIntegers, 0 ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(1,4,4,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(1,4,0,6) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(1,4,2,6) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(2,4,3,6) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(2,4,1,6) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(2,4,5,6) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(3,4,2,6) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(3,4,4,6) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(3,4,0,6) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(3,4,2,6) ] ] ] ], 
  [ [ 3, 4, 4, 4 ],
    [ 6,
          [ [ [ 1, 2, 3 ], 0 ],
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,3,1,6) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(1,3,3,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(1,3,5,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(1,3,0,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(1,3,2,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(2,3,4,6) ] ] ], 
    [ 10,
          [ [ [ 1, 2, 5 ], 0 ],
              [ ClassTransposition(0,2,1,4), ClassTransposition(2,3,1,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(0,3,4,6) ] ] ], 
    [ 12,
          [ [ [ 1, 2, 3, 4 ], 0 ],
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,3,5,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(2,3,3,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(0,3,2,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(2,3,0,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(0,4,1,4) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(0,4,1,4) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(0,4,1,4) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(0,4,1,4) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(0,4,1,4) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(0,4,1,4) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(1,4,2,4) ] ], 
          [ [ [ 1, 3, 4 ], 0 ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(1,3,5,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(1,3,3,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(1,3,2,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(1,3,0,6) ] ] ], 
    [ 20,
          [ [ [ 1, 2, 4, 5 ], 0 ],
              [ ClassTransposition(0,2,3,4), ClassTransposition(2,3,1,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(0,3,4,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(0,4,1,4) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(0,4,1,4) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(0,4,1,4) ] ] ], 
    [ 30,
          [ [ [ 1, 2, 3, 5 ], 0 ],
              [ ClassTransposition(0,2,1,4), ClassTransposition(2,3,3,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(0,3,1,6) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(0,3,5,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(2,3,0,6) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(2,3,4,6) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(0,3,2,6) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(0,4,3,4) ] ] ], 
    [ infinity,
          [ [ [ 1, 2, 3, infinity ], 2 ],
              [ ClassTransposition(0,3,1,6), ClassTransposition(0,4,1,4) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(0,4,1,4) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(0,4,1,4) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(2,4,3,4) ] ] ] ], 
  [ [ 4, 4, 4, 4 ],
    [ 4,
          [ [ [ 1, 4 ], 0 ],
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,3,1,3) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(1,3,2,3) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(3,6,5,6) ] ], 
          [ [ [ 1, 2, 4 ], 0 ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(0,5,2,5), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(0,5,3,5), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(0,5,3,5), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(0,5,3,5), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,5,3,5), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,5,4,5), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(1,5,3,5), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(1,5,4,5), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(1,5,4,5), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(1,5,4,5), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(1,5,4,5), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(2,5,4,5), ClassTransposition(0,6,5,6) ] ], 
          [ [ [ 2, 4 ], 0 ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(0,2,1,2), ClassTransposition(2,5,4,5) ] ] ], 
    [ 6,
          [ [ [ 6 ], 0 ],
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,3,2,3) ] ], 
          [ [ [ 2, 6 ], 0 ],
              [ ClassTransposition(0,2,1,2), ClassTransposition(0,5,4,5) ] ], 
          [ [ [ 1, 2, 6 ], 0 ],
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,5,4,5) ],
              [ ClassTransposition(1,2,2,4), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(0,4,3,4) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(0,5,1,5), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(1,5,2,5), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(2,5,3,5), ClassTransposition(0,6,5,6) ], 
              [ ClassTransposition(3,5,4,5), ClassTransposition(0,6,5,6) ] ], 
          [ [ [ 1, 2, 3 ], 0 ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(0,4,1,4) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(0,4,1,4) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(0,5,1,5), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(0,5,1,5), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,5,1,5), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(0,5,1,5), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,5,1,5), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,5,1,5), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,5,1,5), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,5,1,5), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,5,1,5), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,5,1,5), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(0,5,1,5), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(0,5,1,5), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(0,5,2,5), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(0,5,2,5), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,5,2,5), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,5,2,5), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,5,2,5), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,5,2,5), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,5,2,5), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(0,5,2,5), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(0,5,2,5), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(0,5,3,5), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(0,5,3,5), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(0,5,3,5), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,5,3,5), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,5,3,5), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,5,3,5), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(0,5,3,5), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(0,5,4,5), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(0,5,4,5), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(0,5,4,5), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,5,4,5), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,5,4,5), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(1,5,2,5), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(1,5,2,5), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(1,5,2,5), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(1,5,2,5), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(1,5,2,5), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(1,5,2,5), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(1,5,2,5), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(1,5,2,5), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(1,5,2,5), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(1,5,2,5), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(1,5,2,5), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(1,5,2,5), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(1,5,3,5), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(1,5,3,5), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(1,5,3,5), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(1,5,3,5), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(1,5,3,5), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(1,5,3,5), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(1,5,3,5), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(1,5,3,5), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(1,5,3,5), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(1,5,4,5), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(1,5,4,5), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(1,5,4,5), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(1,5,4,5), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(1,5,4,5), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(1,5,4,5), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(1,5,4,5), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(2,5,3,5), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(2,5,3,5), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(2,5,3,5), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(2,5,3,5), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(2,5,3,5), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(2,5,3,5), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(2,5,3,5), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(2,5,3,5), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(2,5,3,5), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(2,5,3,5), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(2,5,3,5), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(2,5,3,5), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(2,5,4,5), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(2,5,4,5), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(2,5,4,5), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(2,5,4,5), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(2,5,4,5), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(2,5,4,5), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(2,5,4,5), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(2,5,4,5), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(2,5,4,5), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(3,5,4,5), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(3,5,4,5), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(3,5,4,5), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(3,5,4,5), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(3,5,4,5), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(3,5,4,5), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(3,5,4,5), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(3,5,4,5), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(3,5,4,5), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(3,5,4,5), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(3,5,4,5), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(3,5,4,5), ClassTransposition(4,6,5,6) ] ], 
          [ [ [ 1, 6 ], 0 ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(1,6,5,6) ] ] ], 
    [ 12,
          [ [ [ 1, 2, 3, 4 ], 0 ],
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(0,4,1,4), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(0,4,2,4), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(0,4,3,4), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(1,4,2,4), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(1,4,3,4), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(2,4,3,4), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(0,5,1,5), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(0,5,1,5), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,5,2,5), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(0,5,2,5), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(0,5,2,5), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(0,5,2,5), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(0,5,2,5), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(0,5,3,5), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,5,3,5), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,5,3,5), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,5,3,5), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(0,5,4,5), ClassTransposition(0,6,1,6) ], 
              [ ClassTransposition(0,5,4,5), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(0,5,4,5), ClassTransposition(1,6,2,6) ], 
              [ ClassTransposition(0,5,4,5), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(0,5,4,5), ClassTransposition(2,6,3,6) ], 
              [ ClassTransposition(0,5,4,5), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(0,5,4,5), ClassTransposition(3,6,4,6) ], 
              [ ClassTransposition(0,5,4,5), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(0,5,4,5), ClassTransposition(4,6,5,6) ], 
              [ ClassTransposition(1,5,2,5), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(1,5,2,5), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(1,5,3,5), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(1,5,3,5), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(1,5,3,5), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(1,5,3,5), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(1,5,3,5), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(1,5,4,5), ClassTransposition(0,6,2,6) ], 
              [ ClassTransposition(1,5,4,5), ClassTransposition(1,6,3,6) ], 
              [ ClassTransposition(1,5,4,5), ClassTransposition(2,6,4,6) ], 
              [ ClassTransposition(1,5,4,5), ClassTransposition(3,6,5,6) ], 
              [ ClassTransposition(2,5,3,5), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(2,5,3,5), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(2,5,4,5), ClassTransposition(0,6,3,6) ], 
              [ ClassTransposition(2,5,4,5), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(2,5,4,5), ClassTransposition(1,6,4,6) ], 
              [ ClassTransposition(2,5,4,5), ClassTransposition(1,6,5,6) ], 
              [ ClassTransposition(2,5,4,5), ClassTransposition(2,6,5,6) ], 
              [ ClassTransposition(3,5,4,5), ClassTransposition(0,6,4,6) ], 
              [ ClassTransposition(3,5,4,5), ClassTransposition(1,6,5,6) ] ], 
          [ [ [ 1, 3, 4 ], 0 ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(0,3,1,3), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(0,4,1,4) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(1,4,2,4) ], 
              [ ClassTransposition(0,3,2,3), ClassTransposition(2,4,3,4) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(1,3,2,3), ClassTransposition(1,4,3,4) ] ] ], 
    [ 15,
          [ [ [ 1, 3, 5 ], 0 ],
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,3,2,3) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(0,3,2,3) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(0,3,2,3) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(0,3,2,3) ] ] ], 
    [ 20,
          [ [ [ 1, 4, 5 ], 0 ],
              [ ClassTransposition(0,3,1,6), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(0,4,2,4) ] ] ], 
    [ 30,
          [ [ [ 1, 3, 5, 6 ], 0 ],
              [ ClassTransposition(0,2,1,4), ClassTransposition(1,3,2,3) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(0,3,1,3) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(1,3,2,3) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(0,3,1,3) ] ], 
          [ [ [ 1, 2, 3, 5 ], 0 ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(0,5,4,5) ] ], 
          [ [ [ 1, 2, 3, 5, 6 ], 0 ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(2,5,3,5) ] ] ], 
    [ 60,
          [ [ [ 1, 2, 3, 4, 5 ], 0 ],
              [ ClassTransposition(0,3,1,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(0,5,2,5) ] ], 
          [ [ [ 1, 2, 3, 4, 5, 6 ], 0 ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(1,5,4,5) ] ] ], 
    [ infinity,
          [ [ [ 1, 2, infinity ], 4 ],
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,3,1,3) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(1,3,2,3) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(0,3,1,3) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(1,3,2,3) ] ], 
          [ [ [ 1, 2, 3, 4, infinity ], 2 ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(0,2,1,4), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(0,2,3,4), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(0,2,3,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(1,2,0,4), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(1,2,2,4), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(1,2,2,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(3,5,4,5) ] ], 
          [ [ [ 1, 2, 3, 4, infinity ], 0 ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(0,2,1,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(0,2,5,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(1,2,0,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(1,2,4,6), ClassTransposition(0,5,3,5) ] ], 
          [ [ [ 1, 2, 3, infinity ], 2 ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(0,3,1,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(1,5,3,5) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(0,5,2,5) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(0,5,4,5) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(2,5,4,5) ], 
              [ ClassTransposition(2,3,4,6), ClassTransposition(3,5,4,5) ] ], 
          [ [ [ 1, 3, infinity ], 2 ], 
              [ ClassTransposition(0,3,2,6), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(0,3,4,6), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(0,3,5,6), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(1,3,0,6), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(1,3,2,6), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(1,3,3,6), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(1,3,5,6), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(2,3,0,6), ClassTransposition(0,4,2,4) ], 
              [ ClassTransposition(2,3,1,6), ClassTransposition(1,4,3,4) ], 
              [ ClassTransposition(2,3,3,6), ClassTransposition(1,4,3,4) ] ], 
          [ [ Union( ResidueClass(0,2), [ 1 ] ), 4 ], 
              [ ClassTransposition(0,4,1,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(0,4,3,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(0,4,5,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(1,4,0,6), ClassTransposition(0,5,1,5) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(1,4,2,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(1,4,4,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(2,4,1,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(2,4,3,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(2,4,5,6), ClassTransposition(3,5,4,5) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(0,5,3,5) ], 
              [ ClassTransposition(3,4,0,6), ClassTransposition(1,5,2,5) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(1,5,4,5) ], 
              [ ClassTransposition(3,4,2,6), ClassTransposition(2,5,3,5) ], 
              [ ClassTransposition(3,4,4,6), ClassTransposition(3,5,4,5) ] ] 
         ] ] ],

CTPairsProductType :=
[ [ [ 4, 4, 4, 4 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 4 ], 0 ] ], [ [ 4, 4, 4, 4 ], 6, [ [ 6 ], 0 ] ],
  [ [ 1, 3, 3, 1 ], 2, [ [ 1, 2 ], 0 ] ], [ [ 1, 3, 3, 1 ], 4, [ [ 4 ], 0 ] ],
  [ [ 1, 1, 3, 3 ], 4, [ [ 4 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ infinity ], 2 ] ], 
  [ [ 1, 3, 3, 1 ], 2, [ [ 1, 2 ], 0 ] ], [ [ 1, 1, 3, 3 ], 4, [ [ 4 ], 0 ] ],
  [ [ 0, 3, 3, 1 ], infinity, [ [ infinity ], 2 ] ], 
  [ [ 1, 3, 3, 1 ], 4, [ [ 4 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ infinity ], 2 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 2, 6 ], 0 ] ], 
  [ [ 1, 3, 3, 1 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 1 ], 4, [ [ 2, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 2, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 1 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 2, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 1 ], 4, [ [ 2, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 2, infinity ], 2 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 2, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 1 ], 4, [ [ 2, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 2, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 1 ], 4, [ [ 2, 4 ], 0 ] ], [ [ 1, 4, 3, 4 ], 4, [ [ 4 ], 0 ] ],
  [ [ 1, 1, 3, 3 ], 4, [ [ 2, 4 ], 0 ] ], [ [ 1, 4, 3, 4 ], 4, [ [ 4 ], 0 ] ],
  [ [ 1, 4, 3, 4 ], 4, [ [ 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 2, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ infinity ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ infinity ], 2 ] ], 
  [ [ 1, 3, 3, 1 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 2, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 2, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 1 ], 4, [ [ 2, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ infinity ], 2 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 2, 4 ], 0 ] ], [ [ 1, 4, 3, 4 ], 4, [ [ 4 ], 0 ] ],
  [ [ 1, 3, 3, 1 ], 4, [ [ 2, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 2, infinity ], 2 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 2, infinity ], 2 ] ], 
  [ [ 1, 4, 3, 4 ], 4, [ [ 4 ], 0 ] ], [ [ 1, 4, 3, 4 ], 4, [ [ 4 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ infinity ], 2 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ infinity ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ infinity ], 2 ] ], 
  [ [ 1, 3, 3, 1 ], infinity, [ ResidueClass(0,2), 2 ] ], 
  [ [ 1, 3, 3, 1 ], infinity, [ ResidueClass(0,2), 2 ] ], 
  [ [ 1, 3, 3, 1 ], infinity, [ ResidueClass(0,2), 2 ] ], 
  [ [ 1, 3, 3, 1 ], infinity, [ ResidueClass(0,2), 2 ] ], 
  [ [ 1, 3, 3, 1 ], infinity, [ ResidueClass(0,2), 2 ] ], 
  [ [ 1, 3, 3, 1 ], infinity, [ ResidueClass(0,2), 2 ] ], 
  [ [ 1, 3, 3, 1 ], infinity, [ ResidueClass(0,2), 2 ] ], 
  [ [ 1, 3, 3, 1 ], infinity, [ ResidueClass(0,2), 2 ] ], 
  [ [ 1, 3, 3, 1 ], infinity, [ ResidueClass(0,2), 2 ] ], 
  [ [ 1, 3, 3, 1 ], infinity, [ ResidueClass(0,2), 2 ] ], 
  [ [ 1, 3, 3, 1 ], infinity, [ ResidueClass(0,2), 2 ] ], 
  [ [ 1, 3, 3, 1 ], infinity, [ ResidueClass(0,2), 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 3 ], 0 ] ], [ [ 0, 3, 3, 3 ], 3, [ [ 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, infinity ], 4 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, infinity ], 4 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 1 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 1 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 1 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 1 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], [ [ 3, 3, 3, 3 ], 2, [ [ 2 ], 0 ] ],
  [ [ 3, 3, 4, 4 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, infinity ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, infinity ], 4 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, infinity ], 4 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 1 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 2 ], 0 ] ], [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ],
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 1 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, infinity ], 0 ] ], 
  [ [ 1, 3, 3, 1 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 1 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 15, [ [ 1, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 15, [ [ 1, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 15, [ [ 1, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 15, [ [ 1, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 1 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 1 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 2 ], 0 ] ], [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ],
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 6, [ [ 1, 3, 6 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 2, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 1 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 6, [ [ 1, 3, 6 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 1 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 2, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 6, [ [ 1, 3, 6 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 6, [ [ 1, 3, 6 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 2 ], 0 ] ], [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ],
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 6 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 3, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 3, 4, 4, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 3, 4, 4, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 20, [ [ 1, 2, 4, 5 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ [ 1, 2, infinity ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 20, [ [ 1, 2, 4, 5 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 20, [ [ 1, 2, 4, 5 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ [ 1, 2, infinity ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], [ [ 3, 3, 3, 3 ], 2, [ [ 2 ], 0 ] ],
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 6 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 3, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 3, 4, 4, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 3, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], [ [ 3, 3, 3, 3 ], 2, [ [ 2 ], 0 ] ],
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], [ [ 0, 3, 3, 3 ], 3, [ [ 3 ], 0 ] ],
  [ [ 0, 3, 3, 3 ], 3, [ [ 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 3, infinity ], 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 6 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 3, infinity ], 2 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 20, [ [ 1, 4, 5 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 3, infinity ], 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], [ [ 0, 3, 3, 3 ], 3, [ [ 3 ], 0 ] ],
  [ [ 0, 3, 3, 3 ], 3, [ [ 3 ], 0 ] ], [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ],
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 6 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 6, [ [ 1, 2, 3, 6 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 10, [ [ 1, 2, 5 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 6, [ [ 1, 2, 3, 6 ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 6 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 20, [ [ 1, 2, 4, 5 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 20, [ [ 1, 2, 4, 5 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 20, [ [ 1, 2, 4, 5 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ [ 1, 2, infinity ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 3, 4, 4, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 3, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ [ 1, 2, infinity ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 3 ], 0 ] ], [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ],
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 20, [ [ 1, 4, 5 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 3, infinity ], 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 3, infinity ], 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 3, infinity ], 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 3, infinity ], 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 2 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 1, 4, 3, 4 ], 6, [ [ 1, 2, 3, 6 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 6, [ [ 1, 2, 3, 6 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, 2, infinity ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 20, [ [ 1, 2, 4, 5 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 2 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 6 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 3, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 6, [ [ 1, 2, 3, 6 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 10, [ [ 1, 2, 5 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 2 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 6, [ [ 1, 2, 3, 6 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 20, [ [ 1, 2, 4, 5 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, 2, infinity ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 2 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 6, [ [ 1, 2, 3, 6 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 6, [ [ 1, 2, 3, 6 ], 0 ] ], 
  [ [ 3, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ Union( ResidueClass( 0, 2 ), [ 1 ] ), 4 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ Union( ResidueClass( 0, 2 ), [ 1 ] ), 4 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ Union( ResidueClass( 0, 2 ), [ 1 ] ), 4 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ Union( ResidueClass( 0, 2 ), [ 1 ] ), 4 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ Union( ResidueClass( 0, 2 ), [ 1 ] ), 4 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ Union( ResidueClass( 0, 2 ), [ 1 ] ), 4 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ Union( ResidueClass( 0, 2 ), [ 1 ] ), 4 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ Union( ResidueClass( 0, 2 ), [ 1 ] ), 4 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ Union( ResidueClass( 0, 2 ), [ 1 ] ), 4 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ Union( ResidueClass( 0, 2 ), [ 1 ] ), 4 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ Union( ResidueClass( 0, 2 ), [ 1 ] ), 4 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ Union( ResidueClass( 0, 2 ), [ 1 ] ), 4 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ Union( ResidueClass( 0, 2 ), [ 1 ] ), 4 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ Union( ResidueClass( 0, 2 ), [ 1 ] ), 4 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ Union( ResidueClass( 0, 2 ), [ 1 ] ), 4 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ Union( ResidueClass( 0, 2 ), [ 1 ] ), 4 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, 4, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ Union( ResidueClass( 0, 2 ), [ 1 ] ), 4 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ Union( ResidueClass( 0, 2 ), [ 1 ] ), 4 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ Union( ResidueClass( 0, 2 ), [ 1 ] ), 4 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ Union( ResidueClass( 0, 2 ), [ 1 ] ), 4 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5, 6 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 60, [ [ 1, 2, 3, 4, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], infinity, [ [ 1, 2, 3, infinity ], 2 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 4, 4, 4, 4 ], 30, [ [ 1, 2, 3, 5 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 2 ], 0 ] ], [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ],
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], [ [ 3, 3, 3, 3 ], 2, [ [ 2 ], 0 ] ],
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 2 ], infinity, [ [ 1, infinity ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 2 ], 0 ] ], [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ],
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ 1, infinity ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4, 6 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ 1, infinity ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 2 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4, 6 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], [ [ 3, 3, 3, 3 ], 2, [ [ 2 ], 0 ] ],
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ 1, infinity ], 0 ] ], 
  [ [ 1, 4, 3, 2 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4, 6 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], [ [ 3, 3, 3, 3 ], 2, [ [ 2 ], 0 ] ],
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 2 ], 0 ] ], 
  [ [ 1, 4, 3, 2 ], infinity, [ [ 1, infinity ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4, 6 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 2 ], infinity, [ [ 1, infinity ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 2 ], 0 ] ], [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ],
  [ [ 1, 3, 3, 2 ], infinity, [ [ 1, infinity ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4, 6 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 2 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 2 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4, 6 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], [ [ 3, 3, 3, 3 ], 2, [ [ 2 ], 0 ] ],
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 2 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 2 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ 1, infinity ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 2 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4, 6 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 4 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 4, 3, 2 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 2 ], infinity, [ [ 1, infinity ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 2 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ infinity ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 1, 1, 3, 3 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 4, [ [ 1, 2, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 2 ], infinity, [ [ 1, infinity ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 2 ], infinity, [ [ 1, infinity ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, 2, infinity ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4, 6 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4, 6 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 2 ], infinity, [ [ 1, infinity ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4, 6 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 2 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, 2, infinity ], 2 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4, 6 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 3, 3, 4, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], infinity, [ [ 1, 2, infinity ], 0 ] ], 
  [ [ 1, 4, 3, 4 ], 12, [ [ 1, 3, 4, 6 ], 0 ] ], 
  [ [ 3, 3, 3, 4 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 4, 3, 3 ], 12, [ [ 1, 2, 3, 4 ], 0 ] ], 
  [ [ 0, 3, 3, 4 ], infinity, [ ResidueClass( 1, 2 ), 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 1, 3, 3, 4 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 1, 3, 3, 3 ], 6, [ [ 1, 2, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 1 ], infinity, [ [ 1, infinity ], 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 3, 3, 3 ], 2, [ [ 1, 2 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 2 ] ], 
  [ [ 3, 4, 4, 3 ], infinity, [ PositiveIntegers, 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ], 
  [ [ 0, 3, 3, 3 ], 3, [ [ 1, 3 ], 0 ] ] ],

OrdersMatrix :=
[ [ 1, infinity, infinity, infinity, infinity, infinity, 4, infinity, 6, 
      infinity, 4, 4, 2, infinity, 4, 4, infinity, infinity, 4, 4, 4, 6, 2, 
      4, 4, 4, 4, infinity, infinity, infinity, infinity, infinity, infinity, 
      4, 4, 4, infinity, infinity, 4, infinity, 4, infinity, 4, 4, 4, 4, 4, 
      4, 4, 4, 4, infinity, infinity, infinity, 2, infinity, infinity, 4, 4, 
      2, 4, 4, infinity, infinity, infinity, 4, 4, 4, 2 ], 
  [ infinity, 1, infinity, 3, infinity, infinity, infinity, 6, 15, 12, 
      infinity, 12, infinity, infinity, 4, 6, infinity, infinity, infinity, 
      30, 12, 6, infinity, 4, 12, 4, 12, infinity, infinity, infinity, 
      infinity, infinity, infinity, 30, 12, 6, 12, infinity, infinity, 
      infinity, 3, infinity, 12, 30, 30, 12, 6, 12, 4, 6, 10, 30, 6, 
      infinity, 6, infinity, infinity, infinity, 30, 12, 4, infinity, 6, 6, 
      6, 30, infinity, 6, 12 ], 
  [ infinity, infinity, 1, infinity, 3, 3, infinity, infinity, 4, 12, 12, 6, 
      infinity, infinity, 4, 12, 6, 6, infinity, 30, infinity, 12, infinity, 
      4, 6, 4, 6, infinity, infinity, infinity, infinity, infinity, infinity, 
      infinity, infinity, 12, 12, infinity, 12, infinity, 6, infinity, 30, 
      12, infinity, infinity, 3, infinity, 3, 12, infinity, 6, 12, infinity, 
      infinity, 6, 6, 30, 30, 6, 4, 6, infinity, infinity, infinity, 
      infinity, 6, 2, 6 ], 
  [ infinity, 3, infinity, 1, infinity, infinity, 30, 30, 15, 6, 6, 30, 6, 
      infinity, 4, infinity, infinity, infinity, infinity, 30, infinity, 4, 
      12, 4, infinity, 4, 12, infinity, infinity, infinity, infinity, 
      infinity, 12, infinity, infinity, 12, 6, 6, 6, 6, 3, 6, 30, 12, 
      infinity, infinity, 6, 12, 4, 12, 20, 12, infinity, infinity, infinity, 
      infinity, infinity, 30, 30, 12, 4, 12, infinity, infinity, infinity, 
      infinity, 12, 6, infinity ], 
  [ infinity, infinity, 3, infinity, 1, 3, infinity, 12, 6, infinity, 
      infinity, 12, 12, 6, 4, infinity, infinity, 6, 30, 12, infinity, 30, 6, 
      4, infinity, 4, 6, infinity, infinity, infinity, infinity, infinity, 
      12, 6, 12, infinity, 6, infinity, infinity, infinity, 6, infinity, 30, 
      30, 30, 6, 3, 6, 2, 12, 6, infinity, 12, 6, 12, infinity, 6, infinity, 
      30, infinity, 4, 6, infinity, infinity, infinity, 12, infinity, 3, 6 ], 
  [ infinity, infinity, 3, infinity, 3, 1, 6, 6, 6, 12, 12, infinity, 
      infinity, 6, 4, 12, 6, infinity, 30, 30, 30, 30, 6, 4, 6, 4, infinity, 
      infinity, infinity, infinity, infinity, infinity, 12, infinity, 12, 6, 
      infinity, infinity, 12, infinity, 6, infinity, infinity, 30, 12, 6, 2, 
      6, 3, infinity, 12, 12, infinity, 6, infinity, 6, infinity, 12, 30, 6, 
      4, infinity, infinity, infinity, infinity, infinity, 6, 3, infinity ], 
  [ 4, infinity, infinity, 30, infinity, 6, 1, infinity, 3, 3, infinity, 3, 
      6, infinity, 12, 6, infinity, 6, 6, 12, 4, 4, 2, 6, 4, 4, 6, infinity, 
      infinity, 30, 6, infinity, infinity, 3, 3, infinity, 3, infinity, 6, 6, 
      12, infinity, 6, 12, 4, 6, 4, 4, 6, 6, 6, 6, 6, infinity, 6, infinity, 
      6, 6, 12, 6, 6, 2, infinity, 6, infinity, 6, 2, 6, 6 ], 
  [ infinity, 6, infinity, 30, 12, 6, infinity, 1, 3, 3, 3, 3, infinity, 
      infinity, 6, 30, 12, 6, infinity, infinity, 12, 30, infinity, 6, 4, 6, 
      6, infinity, infinity, 6, 12, 12, infinity, 6, 6, infinity, 6, 
      infinity, 12, 6, 20, 6, 12, 12, 30, 3, infinity, 3, 3, 6, 3, 6, 2, 
      infinity, 20, 12, 6, 12, 60, 6, 2, 2, 12, 12, 12, 30, 6, 6, 2 ], 
  [ 6, 15, 4, 15, 6, 6, 3, 3, 1, infinity, 3, infinity, 12, 6, 4, 4, 12, 12, 
      12, 6, 6, 12, 6, 2, 4, 6, 4, 15, 6, 15, 6, 4, 6, 3, 6, 6, 6, 12, 12, 
      12, 4, 6, 12, 6, 6, 6, 6, 2, 6, infinity, 3, infinity, 3, 6, 12, 12, 
      12, 12, 6, 4, 6, 4, 12, 12, 6, 12, 6, 2, 6 ], 
  [ infinity, 12, 12, 6, infinity, 12, 3, 3, infinity, 1, 3, 3, infinity, 6, 
      infinity, 30, infinity, 6, 12, infinity, 30, infinity, 6, infinity, 4, 
      6, 6, 12, 12, 30, infinity, 6, 6, 6, 3, 6, 2, 12, 12, infinity, 6, 6, 
      infinity, 30, 30, 3, 6, 2, 2, infinity, 6, infinity, 6, 12, 20, 12, 12, 
      60, 12, infinity, 3, 3, 12, infinity, 6, 30, 6, 6, 2 ], 
  [ 4, infinity, 12, 6, infinity, 12, infinity, 3, 3, 3, 1, 3, 12, 12, 
      infinity, 30, 12, 12, 30, 12, 30, infinity, 6, 6, 4, infinity, 6, 20, 
      12, 10, 6, infinity, infinity, 6, 6, infinity, 6, 12, infinity, 6, 6, 
      infinity, 60, 30, 30, 2, 6, 3, 2, 6, 2, 6, 3, 6, 12, infinity, 6, 
      infinity, infinity, 6, 3, 2, 12, 6, infinity, 12, infinity, 6, 3 ], 
  [ 4, 12, 6, 30, 12, infinity, 3, 3, infinity, 3, 3, 1, 12, 6, 6, 30, 12, 
      infinity, 30, 60, 30, 30, 6, 6, 4, 6, infinity, 12, infinity, 6, 12, 
      12, 6, 6, 2, 6, 3, infinity, infinity, 6, infinity, 6, 12, 12, 12, 2, 
      6, 2, 3, infinity, 6, infinity, 6, 6, 12, 12, infinity, 12, infinity, 
      6, 2, 3, 12, 12, 12, infinity, 6, infinity, 3 ], 
  [ 2, infinity, infinity, 6, 12, infinity, 6, infinity, 12, infinity, 12, 
      12, 1, infinity, 3, 3, infinity, infinity, 6, 6, 12, 6, 2, 6, 6, 4, 4, 
      infinity, infinity, 6, 12, infinity, infinity, 6, 12, 12, 20, infinity, 
      3, infinity, 3, infinity, 6, 6, 12, 6, 6, 4, 4, 12, 12, 20, 20, 6, 2, 
      6, 6, 6, 6, 2, 6, 6, 6, 6, 6, 6, 6, 6, 2 ], 
  [ infinity, infinity, infinity, infinity, 6, 6, infinity, infinity, 6, 6, 
      12, 6, infinity, 1, 3, infinity, 3, 3, infinity, 6, 60, 12, infinity, 
      6, 6, 4, 6, infinity, infinity, 6, infinity, infinity, infinity, 
      infinity, 12, 12, 12, infinity, 6, infinity, 6, infinity, 12, 12, 12, 
      infinity, 3, infinity, 3, 12, infinity, 6, 12, 3, 6, 2, 2, 60, 4, 6, 6, 
      6, infinity, infinity, infinity, 60, 6, 2, 6 ], 
  [ 4, 4, 4, 4, 4, 4, 12, 6, 4, infinity, infinity, 6, 3, 3, 1, 3, 3, 3, 6, 
      4, 12, 4, 6, 4, 6, 6, 6, 3, 6, 3, 6, 6, infinity, 12, infinity, 6, 6, 
      6, 3, 6, 2, 6, 6, 4, 12, 6, 2, 6, 2, infinity, 6, 6, 20, 3, 3, 3, 3, 6, 
      4, 6, 4, 6, 6, 6, 6, 6, 6, 2, 6 ], 
  [ 4, 6, 12, infinity, infinity, 12, 6, 30, 4, 30, 30, 30, 3, infinity, 3, 
      1, infinity, infinity, 12, 12, 6, 4, 6, 6, 2, 4, 6, infinity, 12, 6, 
      infinity, 12, 30, 6, 30, 30, 30, 6, 2, 6, 3, 6, 12, 12, 6, 4, 6, 4, 4, 
      30, 30, 30, 30, 6, 3, 6, 6, 12, 12, 6, 6, 2, infinity, infinity, 
      infinity, 12, 4, 6, 6 ], 
  [ infinity, infinity, 6, infinity, infinity, 6, infinity, 12, 12, infinity, 
      12, 12, infinity, 3, 3, infinity, 1, 3, 12, 6, infinity, 30, 6, 6, 
      infinity, 4, 6, infinity, infinity, 6, infinity, infinity, 6, 6, 12, 
      infinity, 6, infinity, 6, infinity, 6, infinity, infinity, 6, 60, 6, 3, 
      6, 2, 12, 6, infinity, 12, 2, 6, 3, 2, 12, 12, infinity, 6, 6, 
      infinity, infinity, infinity, 60, infinity, 3, 6 ], 
  [ infinity, infinity, 6, infinity, 6, infinity, 6, 6, 12, 6, 12, infinity, 
      infinity, 3, 3, infinity, 3, 1, 60, 12, 60, 30, 6, 6, 6, 4, infinity, 
      infinity, infinity, 6, infinity, infinity, 6, infinity, 12, 6, 
      infinity, infinity, 6, infinity, 6, infinity, 12, 6, infinity, 6, 2, 6, 
      3, 12, 12, 12, 12, 2, 6, 2, 3, infinity, 6, 6, 6, infinity, infinity, 
      infinity, infinity, 12, 6, 3, infinity ], 
  [ 4, infinity, infinity, infinity, 30, 30, 6, infinity, 12, 12, 30, 30, 6, 
      infinity, 6, 12, 12, 60, 1, 3, 3, 3, 6, 6, 6, 12, 6, infinity, 
      infinity, 30, 12, infinity, infinity, 6, 60, 12, 12, infinity, 6, 12, 
      6, 60, 3, 3, 3, 6, 6, 6, 12, infinity, 12, 30, 30, 12, 6, 60, 60, 2, 2, 
      6, 6, 6, 12, 60, 60, 2, 6, 6, 6 ], 
  [ 4, 30, 30, 30, 12, 30, 12, infinity, 6, infinity, 12, 60, 6, 6, 4, 12, 6, 
      12, 3, 1, 3, 3, 6, 6, 12, 12, 4, 30, 30, 30, 30, 30, 30, 12, 30, 12, 
      60, 6, 6, 6, 4, 12, 3, 2, 2, 6, 6, 12, 12, infinity, infinity, 12, 60, 
      6, 6, 12, 4, 3, 3, 6, 6, 12, 6, 12, 4, 2, 6, 6, 6 ], 
  [ 4, 12, infinity, infinity, infinity, 30, 4, 12, 6, 30, 30, 30, 12, 60, 
      12, 6, infinity, 60, 3, 3, 1, 3, 6, 12, 4, 6, 4, infinity, 12, 30, 30, 
      infinity, 30, 4, 30, 12, 30, 60, 12, infinity, 12, 60, 2, 3, 2, 6, 12, 
      4, 6, 12, 30, 30, 30, infinity, 12, 60, 12, 3, 2, 6, 12, 4, infinity, 
      60, 12, 3, 6, 12, 6 ], 
  [ 6, 6, 12, 4, 30, 30, 4, 30, 12, infinity, infinity, 30, 6, 12, 4, 4, 30, 
      30, 3, 3, 3, 1, 12, 12, 6, 6, 4, 4, 30, 6, 30, 12, 12, 4, 60, 60, 12, 
      12, 6, 30, 4, 30, 2, 2, 3, 12, 12, 6, 6, 30, infinity, infinity, 30, 
      30, 6, 30, 12, 2, 3, 12, 12, 6, 30, 30, 12, 3, 12, 12, 12 ], 
  [ 2, infinity, infinity, 12, 6, 6, 2, infinity, 6, 6, 6, 6, 2, infinity, 6, 
      6, 6, 6, 6, 6, 6, 12, 1, 3, 3, 3, 3, infinity, infinity, 12, 6, 6, 
      infinity, 6, 6, 6, 6, infinity, 6, 6, 6, 6, 6, 6, 6, 3, 3, 3, 3, 3, 3, 
      2, 2, infinity, 2, 6, 6, 6, 6, 2, 2, 2, infinity, 6, 6, 6, 2, 2, 2 ], 
  [ 4, 4, 4, 4, 4, 4, 6, 6, 2, infinity, 6, 6, 6, 6, 4, 6, 6, 6, 6, 6, 12, 
      12, 3, 1, 3, 3, 3, 6, 3, 6, 3, 2, 3, 6, 3, 2, 2, 3, 6, 3, 2, 2, 6, 6, 
      12, 3, 2, 2, 2, infinity, 6, 6, 6, 6, 6, 6, 6, 6, 6, 3, 3, 3, 3, 3, 2, 
      6, 2, 2, 2 ], 
  [ 4, 12, 6, infinity, infinity, 6, 4, 4, 4, 4, 4, 4, 6, 6, 6, 2, infinity, 
      6, 6, 12, 4, 6, 3, 3, 1, 3, 3, 12, infinity, infinity, 6, 6, 3, 2, 2, 
      3, 2, infinity, 4, 6, 6, 6, 6, 12, 4, 2, 3, 2, 2, 3, 2, 3, 2, 6, 6, 
      infinity, 6, 6, 12, 3, 2, 2, infinity, 6, 6, 6, 3, 3, 2 ], 
  [ 4, 4, 4, 4, 4, 4, 4, 6, 6, 6, infinity, 6, 4, 4, 6, 4, 4, 4, 12, 12, 6, 
      6, 3, 3, 3, 1, 3, 4, 3, 4, 2, 3, infinity, 6, 6, 6, 6, 3, 4, 2, 2, 3, 
      12, 12, 6, 2, 2, 3, 2, 3, 2, 2, 3, 4, 4, 4, 4, 12, 12, 2, 3, 2, 3, 2, 
      3, 12, 3, 2, 3 ], 
  [ 4, 12, 6, 12, 6, infinity, 6, 6, 4, 6, 6, infinity, 4, 6, 6, 6, 6, 
      infinity, 6, 4, 4, 4, 3, 3, 3, 3, 1, 12, infinity, 12, 6, 6, 3, 6, 2, 
      2, 3, infinity, 4, 6, 6, 6, 6, 4, 4, 2, 2, 2, 3, infinity, 6, 6, 6, 6, 
      4, 6, infinity, 6, 4, 2, 2, 3, infinity, 6, 6, 6, 2, 3, 3 ], 
  [ infinity, infinity, infinity, infinity, infinity, infinity, infinity, 
      infinity, 15, 12, 20, 12, infinity, infinity, 3, infinity, infinity, 
      infinity, infinity, 30, infinity, 4, infinity, 6, 12, 4, 12, 1, 
      infinity, 3, infinity, infinity, 6, 30, 12, infinity, 12, infinity, 6, 
      infinity, 4, infinity, 30, 12, infinity, 12, 4, 12, 4, 30, 6, 6, 30, 6, 
      6, 6, 6, 30, 30, 12, 6, infinity, infinity, infinity, infinity, 
      infinity, infinity, 4, 12 ], 
  [ infinity, infinity, infinity, infinity, infinity, infinity, infinity, 
      infinity, 6, 12, 12, infinity, infinity, infinity, 6, 12, infinity, 
      infinity, infinity, 30, 12, 30, infinity, 3, infinity, 3, infinity, 
      infinity, 1, infinity, 3, 3, infinity, 6, 6, 12, 12, infinity, 12, 6, 
      4, 6, 12, 30, 30, 6, 4, 6, 4, infinity, 12, 12, 6, infinity, infinity, 
      infinity, infinity, infinity, 30, 6, 2, 6, infinity, 6, 6, 30, 6, 4, 6 ]
    , [ infinity, infinity, infinity, infinity, infinity, infinity, 30, 6, 
      15, 30, 10, 6, 6, 6, 3, 6, 6, 6, 30, 30, 30, 6, 12, 6, infinity, 4, 12, 
      3, infinity, 1, infinity, infinity, 12, infinity, 6, 12, infinity, 
      infinity, infinity, infinity, 4, infinity, infinity, 30, 12, infinity, 
      4, 12, 4, 12, infinity, 12, 6, infinity, infinity, infinity, infinity, 
      12, 30, 12, 6, 12, infinity, infinity, infinity, infinity, 12, 4, 
      infinity ], 
  [ infinity, infinity, infinity, infinity, infinity, infinity, 6, 12, 6, 
      infinity, 6, 12, 12, infinity, 6, infinity, infinity, infinity, 12, 30, 
      30, 30, 6, 3, 6, 2, 6, infinity, 3, infinity, 1, 3, 6, infinity, 
      infinity, 12, 12, 6, infinity, infinity, 4, 6, infinity, 30, infinity, 
      infinity, 4, 6, 4, 12, infinity, infinity, 12, infinity, 12, infinity, 
      infinity, 30, 12, infinity, 3, infinity, 6, infinity, 6, 30, 6, 4, 6 ], 
  [ infinity, infinity, infinity, infinity, infinity, infinity, infinity, 12, 
      4, 6, infinity, 12, infinity, infinity, 6, 12, infinity, infinity, 
      infinity, 30, infinity, 12, 6, 2, 6, 3, 6, infinity, 3, infinity, 3, 1, 
      12, infinity, 12, infinity, infinity, 6, 12, 6, 4, infinity, 30, 12, 
      infinity, 6, 4, infinity, 4, 6, 12, 12, infinity, infinity, infinity, 
      infinity, infinity, 30, 30, 6, 3, 6, 6, 6, infinity, infinity, 
      infinity, 4, infinity ], 
  [ infinity, infinity, infinity, 12, 12, 12, infinity, infinity, 6, 6, 
      infinity, 6, infinity, infinity, infinity, 30, 6, 6, infinity, 30, 30, 
      12, infinity, 3, 3, infinity, 3, 6, infinity, 12, 6, 12, 1, 3, 3, 3, 3, 
      infinity, 12, 6, 6, 12, 60, 12, 30, 6, 6, 4, 6, 3, 6, 2, 6, 12, 20, 12, 
      12, 30, 60, 2, 6, 2, infinity, 6, 12, 12, 6, 2, 6 ], 
  [ 4, 30, infinity, infinity, 6, infinity, 3, 6, 3, 6, 6, 6, 6, infinity, 
      12, 6, 6, infinity, 6, 12, 4, 4, 6, 6, 2, 6, 6, 30, 6, infinity, 
      infinity, infinity, 3, 1, infinity, 3, infinity, 6, 6, infinity, 12, 
      infinity, 6, 12, 4, 2, 6, 4, 4, 3, infinity, 3, infinity, infinity, 6, 
      6, infinity, 6, 12, 6, 4, 4, 6, infinity, infinity, 6, 6, 6, 2 ], 
  [ 4, 12, infinity, infinity, 12, 12, 3, 6, 6, 3, 6, 2, 12, 12, infinity, 
      30, 12, 12, 60, 30, 30, 60, 6, 3, 2, 6, 2, 12, 6, 6, infinity, 12, 3, 
      infinity, 1, 3, 3, 6, infinity, infinity, 6, 12, infinity, infinity, 
      12, infinity, 6, 4, 6, 6, infinity, 6, infinity, infinity, 12, 6, 6, 
      12, 12, 3, infinity, 3, 6, infinity, 12, 12, 6, 2, 6 ], 
  [ 4, 6, 12, 12, infinity, 6, infinity, infinity, 6, 6, infinity, 6, 12, 12, 
      6, 30, infinity, 6, 12, 12, 12, 60, 6, 2, 3, 6, 2, infinity, 12, 12, 
      12, infinity, 3, 3, 3, 1, 3, 6, infinity, 6, infinity, infinity, 12, 
      infinity, 30, 6, infinity, 4, 6, 2, 6, 3, 6, 12, 12, infinity, 6, 
      infinity, 30, 3, 6, 2, 12, 12, 12, 60, infinity, 3, 6 ], 
  [ infinity, 12, 12, 6, 6, infinity, 3, 6, 6, 2, 6, 3, 20, 12, 6, 30, 6, 
      infinity, 12, 60, 30, 12, 6, 2, 2, 6, 3, 12, 12, infinity, 12, 
      infinity, 3, infinity, 3, 3, 1, 12, 12, 12, infinity, 12, 30, 12, 30, 
      6, 6, 4, infinity, 6, infinity, 6, infinity, 12, infinity, 6, infinity, 
      60, 30, 2, 6, 3, 6, 6, infinity, infinity, 6, 3, infinity ], 
  [ infinity, infinity, infinity, 6, infinity, infinity, infinity, infinity, 
      12, 12, 12, infinity, infinity, infinity, 6, 6, infinity, infinity, 
      infinity, 6, 60, 12, infinity, 3, infinity, 3, infinity, infinity, 
      infinity, infinity, 6, 6, infinity, 6, 6, 6, 12, 1, infinity, 3, 3, 3, 
      12, 12, 12, 6, 6, 6, 4, infinity, 6, 12, 6, infinity, 6, infinity, 
      infinity, 60, 4, 6, 2, 6, 3, 2, 2, 60, 6, 6, 6 ], 
  [ 4, infinity, 12, 6, infinity, 12, 6, 12, 12, 12, infinity, infinity, 3, 
      6, 3, 2, 6, 6, 6, 6, 12, 6, 6, 6, 4, 4, 4, 6, 12, infinity, infinity, 
      12, 12, 6, infinity, infinity, 12, infinity, 1, infinity, 3, infinity, 
      6, 6, 12, 2, 6, 6, 4, infinity, infinity, 12, 12, infinity, 3, 
      infinity, infinity, 6, 6, 6, 6, 4, 6, 6, 6, 6, 2, 6, 6 ], 
  [ infinity, infinity, infinity, 6, infinity, infinity, 6, 6, 12, infinity, 
      6, 6, infinity, infinity, 6, 6, infinity, infinity, 12, 6, infinity, 
      30, 6, 3, 6, 2, 6, infinity, 6, infinity, infinity, 6, 6, infinity, 
      infinity, 6, 12, 3, infinity, 1, 3, 3, infinity, 6, 60, infinity, 6, 6, 
      4, 12, infinity, 12, 12, infinity, 6, infinity, infinity, 12, 12, 
      infinity, 3, infinity, 2, 3, 2, 60, 6, 6, 6 ], 
  [ 4, 3, 6, 3, 6, 6, 12, 20, 4, 6, 6, infinity, 3, 6, 2, 3, 6, 6, 6, 4, 12, 
      4, 6, 2, 6, 2, 6, 4, 4, 4, 4, 4, 6, 12, 6, infinity, infinity, 3, 3, 3, 
      1, 3, 6, 4, 12, 6, 4, 6, 6, 6, infinity, infinity, 6, 6, 3, 6, 6, 6, 4, 
      6, 2, 6, 3, 3, 3, 6, 6, 4, 6 ], 
  [ infinity, infinity, infinity, 6, infinity, infinity, infinity, 6, 6, 6, 
      infinity, 6, infinity, infinity, 6, 6, infinity, infinity, 60, 12, 60, 
      30, 6, 2, 6, 3, 6, infinity, 6, infinity, 6, infinity, 12, infinity, 
      12, infinity, 12, 3, infinity, 3, 3, 1, 12, 6, infinity, 6, 6, 
      infinity, 4, 6, 6, 12, infinity, infinity, 6, infinity, infinity, 
      infinity, 6, 6, 3, 6, 2, 2, 3, 12, infinity, 6, infinity ], 
  [ 4, 12, 30, 30, 30, infinity, 6, 12, 12, infinity, 60, 12, 6, 12, 6, 12, 
      infinity, 12, 3, 3, 2, 2, 6, 6, 6, 12, 6, 30, 12, infinity, infinity, 
      30, 60, 6, infinity, 12, 30, 12, 6, infinity, 6, 12, 1, 3, 3, 6, 6, 6, 
      12, 12, infinity, 60, 12, infinity, 6, 12, 60, 3, 3, 6, 6, 6, infinity, 
      12, 60, 2, 6, 6, 6 ], 
  [ 4, 30, 12, 12, 30, 30, 12, 12, 6, 30, 30, 12, 6, 12, 4, 12, 6, 6, 3, 2, 
      3, 2, 6, 6, 12, 12, 4, 12, 30, 30, 30, 12, 12, 12, infinity, infinity, 
      12, 12, 6, 6, 4, 6, 3, 1, 3, 6, 6, 12, 12, 12, 30, 30, 12, 6, 6, 6, 12, 
      3, 2, 6, 6, 12, 6, 6, 12, 3, 6, 6, 6 ], 
  [ 4, 30, infinity, infinity, 30, 12, 4, 30, 6, 30, 30, 12, 12, 12, 12, 6, 
      60, infinity, 3, 2, 2, 3, 6, 12, 4, 6, 4, infinity, 30, 12, infinity, 
      infinity, 30, 4, 12, 30, 30, 12, 12, 60, 12, infinity, 3, 3, 1, 6, 12, 
      4, 6, 30, 30, 30, 12, 60, 12, infinity, 60, 2, 3, 6, 12, 4, 60, 
      infinity, 60, 3, 6, 12, 6 ], 
  [ 4, 12, infinity, infinity, 6, 6, 6, 3, 6, 3, 2, 2, 6, infinity, 6, 4, 6, 
      6, 6, 6, 6, 12, 3, 3, 2, 2, 2, 12, 6, infinity, infinity, 6, 6, 2, 
      infinity, 6, 6, 6, 2, infinity, 6, 6, 6, 6, 6, 1, 3, 3, 3, 6, infinity, 
      6, 6, infinity, 6, 6, 6, 6, 6, 3, 3, 3, 6, infinity, 6, 6, 2, 2, 2 ], 
  [ 4, 6, 3, 6, 3, 2, 4, infinity, 6, 6, 6, 6, 6, 3, 2, 6, 3, 2, 6, 6, 12, 
      12, 3, 2, 3, 2, 2, 4, 4, 4, 4, 4, 6, 6, 6, infinity, 6, 6, 6, 6, 4, 6, 
      6, 6, 12, 3, 1, 3, 3, 2, 3, 3, 2, 3, 6, 3, 2, 6, 6, 3, 2, 2, 6, 6, 6, 
      6, 3, 3, 2 ], 
  [ 4, 12, infinity, 12, 6, 6, 4, 3, 2, 2, 3, 2, 4, infinity, 6, 4, 6, 6, 6, 
      12, 4, 6, 3, 2, 2, 3, 2, 12, 6, 12, 6, infinity, 4, 4, 4, 4, 4, 6, 6, 
      6, 6, infinity, 6, 12, 4, 3, 3, 1, 3, 2, 3, 2, 3, infinity, 4, 6, 6, 6, 
      12, 2, 3, 2, 6, 6, infinity, 6, 3, 2, 3 ], 
  [ 4, 4, 3, 4, 2, 3, 6, 3, 6, 2, 2, 3, 4, 3, 2, 4, 2, 3, 12, 12, 6, 6, 3, 2, 
      2, 2, 3, 4, 4, 4, 4, 4, 6, 4, 6, 6, infinity, 4, 4, 4, 6, 4, 12, 12, 6, 
      3, 3, 3, 1, 6, infinity, 6, 6, 3, 4, 2, 3, 12, 12, 2, 2, 3, 4, 4, 4, 
      12, 2, 3, 3 ], 
  [ 4, 6, 12, 12, 12, infinity, 6, 6, infinity, infinity, 6, infinity, 12, 
      12, infinity, 30, 12, 12, infinity, infinity, 12, 30, 3, infinity, 3, 
      3, infinity, 30, infinity, 12, 12, 6, 3, 3, 6, 2, 6, infinity, 
      infinity, 12, 6, 6, 12, 12, 30, 6, 2, 2, 6, 1, 3, 3, 3, 6, 12, 6, 
      infinity, 12, 60, 6, 6, 4, infinity, 12, 6, 30, 2, 6, 6 ], 
  [ 4, 10, infinity, 20, 6, 12, 6, 3, 3, 6, 2, 6, 12, infinity, 6, 30, 6, 12, 
      12, infinity, 30, infinity, 3, 6, 2, 2, 6, 6, 12, infinity, infinity, 
      12, 6, infinity, infinity, 6, infinity, 6, infinity, infinity, 
      infinity, 6, infinity, 30, 30, infinity, 3, 3, infinity, 3, 1, 3, 3, 
      infinity, 12, 6, 12, 60, 12, 6, 6, 4, 12, 12, 12, 30, 2, 6, 6 ], 
  [ infinity, 30, 6, 12, infinity, 12, 6, 6, infinity, infinity, 6, infinity, 
      20, 6, 6, 30, infinity, 12, 30, 12, 30, infinity, 2, 6, 3, 2, 6, 6, 12, 
      12, infinity, 12, 2, 3, 6, 3, 6, 12, 12, 12, infinity, 12, 60, 30, 30, 
      6, 3, 2, 6, 3, 3, 1, 3, 6, infinity, infinity, 12, infinity, infinity, 
      infinity, 6, 4, 6, infinity, 6, 12, 3, infinity, 6 ], 
  [ infinity, 6, 12, infinity, 12, infinity, 6, 2, 3, 6, 3, 6, 20, 12, 20, 
      30, 12, 12, 30, 60, 30, 30, 2, 6, 2, 3, 6, 30, 6, 6, 12, infinity, 6, 
      infinity, infinity, 6, infinity, 6, 12, 12, 6, infinity, 12, 12, 12, 6, 
      2, 3, 6, 3, 3, 3, 1, 6, infinity, 6, infinity, 12, infinity, 6, 
      infinity, 4, 6, 12, infinity, infinity, 3, 6, infinity ], 
  [ infinity, infinity, infinity, infinity, 6, 6, infinity, infinity, 6, 12, 
      6, 6, 6, 3, 3, 6, 2, 2, 12, 6, infinity, 30, infinity, 6, 6, 4, 6, 6, 
      infinity, infinity, infinity, infinity, 12, infinity, infinity, 12, 12, 
      infinity, infinity, infinity, 6, infinity, infinity, 6, 60, infinity, 
      3, infinity, 3, 6, infinity, 6, 6, 1, infinity, 3, 3, 12, 12, 6, 6, 6, 
      infinity, infinity, infinity, 60, 6, 2, 6 ], 
  [ 2, 6, infinity, infinity, 12, infinity, 6, 20, 12, 20, 12, 12, 2, 6, 3, 
      3, 6, 6, 6, 6, 12, 6, 2, 6, 6, 4, 4, 6, infinity, infinity, 12, 
      infinity, 20, 6, 12, 12, infinity, 6, 3, 6, 3, 6, 6, 6, 12, 6, 6, 4, 4, 
      12, 12, infinity, infinity, infinity, 1, infinity, infinity, 6, 6, 2, 
      6, 6, infinity, infinity, infinity, 6, 6, 6, 2 ], 
  [ infinity, infinity, 6, infinity, infinity, 6, infinity, 12, 12, 12, 
      infinity, 12, 6, 2, 3, 6, 3, 2, 60, 12, 60, 30, 6, 6, infinity, 4, 6, 
      6, infinity, infinity, infinity, infinity, 12, 6, 6, infinity, 6, 
      infinity, infinity, infinity, 6, infinity, 12, 6, infinity, 6, 3, 6, 2, 
      6, 6, infinity, 6, 3, infinity, 1, 3, infinity, 6, infinity, 6, 6, 
      infinity, infinity, infinity, 12, infinity, 3, 6 ], 
  [ infinity, infinity, 6, infinity, 6, infinity, 6, 6, 12, 12, 6, infinity, 
      6, 2, 3, 6, 2, 3, 60, 4, 12, 12, 6, 6, 6, 4, infinity, 6, infinity, 
      infinity, infinity, infinity, 12, infinity, 6, 6, infinity, infinity, 
      infinity, infinity, 6, infinity, 60, 12, 60, 6, 2, 6, 3, infinity, 12, 
      12, infinity, 3, infinity, 3, 1, 12, 6, 6, 6, infinity, infinity, 
      infinity, infinity, infinity, 6, 3, infinity ], 
  [ 4, infinity, 30, 30, infinity, 12, 6, 12, 12, 60, infinity, 12, 6, 60, 6, 
      12, 12, infinity, 2, 3, 3, 2, 6, 6, 6, 12, 6, 30, infinity, 12, 30, 30, 
      30, 6, 12, infinity, 60, 60, 6, 12, 6, infinity, 3, 3, 2, 6, 6, 6, 12, 
      12, 60, infinity, 12, 12, 6, infinity, 12, 1, 3, 6, 6, 6, 12, infinity, 
      12, 3, 6, 6, 6 ], 
  [ 4, 30, 30, 30, 30, 30, 12, 60, 6, 12, infinity, infinity, 6, 4, 4, 12, 
      12, 6, 2, 3, 2, 3, 6, 6, 12, 12, 4, 30, 30, 30, 12, 30, 60, 12, 12, 30, 
      30, 4, 6, 12, 4, 6, 3, 2, 3, 6, 6, 12, 12, 60, 12, infinity, infinity, 
      12, 6, 6, 6, 3, 1, 6, 6, 12, 12, 6, 6, 3, 6, 6, 6 ], 
  [ 2, 12, 6, 12, infinity, 6, 6, 6, 4, infinity, 6, 6, 2, 6, 6, 6, infinity, 
      6, 6, 6, 6, 12, 2, 3, 3, 2, 2, 12, 6, 12, infinity, 6, 2, 6, 3, 3, 2, 
      6, 6, infinity, 6, 6, 6, 6, 6, 3, 3, 2, 2, 6, 6, infinity, 6, 6, 2, 
      infinity, 6, 6, 6, 1, 3, 3, 6, infinity, 6, 6, 3, 3, 2 ], 
  [ 4, 4, 4, 4, 4, 4, 6, 2, 6, 3, 3, 2, 6, 6, 4, 6, 6, 6, 6, 6, 12, 12, 2, 3, 
      2, 3, 2, 6, 2, 6, 3, 3, 6, 4, infinity, 6, 6, 2, 6, 3, 2, 3, 6, 6, 12, 
      3, 2, 3, 2, 6, 6, 6, infinity, 6, 6, 6, 6, 6, 6, 3, 1, 3, 2, 3, 3, 6, 
      3, 2, 3 ], 
  [ 4, infinity, 6, 12, 6, infinity, 2, 2, 4, 3, 2, 3, 6, 6, 6, 2, 6, 
      infinity, 6, 12, 4, 6, 2, 3, 2, 2, 3, infinity, 6, 12, infinity, 6, 2, 
      4, 3, 2, 3, 6, 4, infinity, 6, 6, 6, 12, 4, 3, 2, 2, 3, 4, 4, 4, 4, 6, 
      6, 6, infinity, 6, 12, 3, 3, 1, 6, infinity, 6, 6, 2, 3, 3 ], 
  [ infinity, 6, infinity, infinity, infinity, infinity, infinity, 12, 12, 
      12, 12, 12, 6, infinity, 6, infinity, infinity, infinity, 12, 6, 
      infinity, 30, infinity, 3, infinity, 3, infinity, infinity, infinity, 
      infinity, 6, 6, infinity, 6, 6, 12, 6, 3, 6, 2, 3, 2, infinity, 6, 60, 
      6, 6, 6, 4, infinity, 12, 6, 6, infinity, infinity, infinity, infinity, 
      12, 12, 6, 2, 6, 1, 3, 3, 60, 6, 6, 6 ], 
  [ infinity, 6, infinity, infinity, infinity, infinity, 6, 12, 12, infinity, 
      6, 12, 6, infinity, 6, infinity, infinity, infinity, 60, 12, 60, 30, 6, 
      3, 6, 2, 6, infinity, 6, infinity, infinity, 6, 6, infinity, infinity, 
      12, 6, 2, 6, 3, 3, 2, 12, 6, infinity, infinity, 6, 6, 4, 12, 12, 
      infinity, 12, infinity, infinity, infinity, infinity, infinity, 6, 
      infinity, 3, infinity, 3, 1, 3, 12, 6, 6, 6 ], 
  [ infinity, 6, infinity, infinity, infinity, infinity, infinity, 12, 6, 6, 
      infinity, 12, 6, infinity, 6, infinity, infinity, infinity, 60, 4, 12, 
      12, 6, 2, 6, 3, 6, infinity, 6, infinity, 6, infinity, 12, infinity, 
      12, 12, infinity, 2, 6, 2, 3, 3, 60, 12, 60, 6, 6, infinity, 4, 6, 12, 
      6, infinity, infinity, infinity, infinity, infinity, 12, 6, 6, 3, 6, 3, 
      3, 1, infinity, infinity, 6, infinity ], 
  [ 4, 30, infinity, infinity, 12, infinity, 6, 30, 12, 30, 12, infinity, 6, 
      60, 6, 12, 60, 12, 2, 2, 3, 3, 6, 6, 6, 12, 6, infinity, 30, infinity, 
      30, infinity, 12, 6, 12, 60, infinity, 60, 6, 60, 6, 12, 2, 3, 3, 6, 6, 
      6, 12, 30, 30, 12, infinity, 60, 6, 12, infinity, 3, 3, 6, 6, 6, 60, 
      12, infinity, 1, 6, 6, 6 ], 
  [ 4, infinity, 6, 12, infinity, 6, 2, 6, 6, 6, infinity, 6, 6, 6, 6, 4, 
      infinity, 6, 6, 6, 6, 12, 2, 2, 3, 3, 2, infinity, 6, 12, 6, infinity, 
      6, 6, 6, infinity, 6, 6, 2, 6, 6, infinity, 6, 6, 6, 2, 3, 3, 2, 2, 2, 
      3, 3, 6, 6, infinity, 6, 6, 6, 3, 3, 2, 6, 6, infinity, 6, 1, 3, 3 ], 
  [ 4, 6, 2, 6, 3, 3, 6, 6, 2, 6, 6, infinity, 6, 2, 2, 6, 3, 3, 6, 6, 12, 
      12, 2, 2, 3, 2, 3, 4, 4, 4, 4, 4, 2, 6, 2, 3, 3, 6, 6, 6, 4, 6, 6, 6, 
      12, 2, 3, 2, 3, 6, 6, infinity, 6, 2, 6, 3, 3, 6, 6, 3, 2, 3, 6, 6, 6, 
      6, 3, 1, 3 ], 
  [ 2, 12, 6, infinity, 6, infinity, 6, 2, 6, 2, 3, 3, 2, 6, 6, 6, 6, 
      infinity, 6, 6, 6, 12, 2, 2, 2, 3, 3, 12, 6, infinity, 6, infinity, 6, 
      2, 6, 6, infinity, 6, 6, 6, 6, infinity, 6, 6, 6, 2, 2, 3, 3, 6, 6, 6, 
      infinity, 6, 2, 6, infinity, 6, 6, 2, 3, 3, 6, 6, infinity, 6, 3, 3, 1 
     ] ],

CTProds12 := ReadAsFunction(
               Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                             "/data/ctprods12.g"))(),

CTProds32 := ReadAsFunction(
               Concatenation(PackageInfo("rcwa")[1].InstallationPath,
                             "/data/ctprods32.g"))()

);

#############################################################################
##
#E  ctprodclass.g . . . . . . . . . . . . . . . . . . . . . . . . . ends here