/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.truffle.api.source;

import com.oracle.truffle.api.source.BytesDecoder;
import com.oracle.truffle.api.source.LineLocation;
import com.oracle.truffle.api.source.SourceListener;
import com.oracle.truffle.api.source.SourceSection;
import com.oracle.truffle.api.source.SourceTag;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.lang.ref.WeakReference;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public abstract class Source {
    private static final List<WeakReference<Source>> allSources = Collections.synchronizedList(new ArrayList());
    private static final Map<String, WeakReference<Source>> filePathToSource = new HashMap<String, WeakReference<Source>>();
    private static boolean fileCacheEnabled = true;
    private static final List<SourceListener> sourceListeners = new ArrayList<SourceListener>();
    private final ArrayList<SourceTag> tags = new ArrayList();
    private TextMap textMap = null;

    public static Source fromFileName(String fileName, boolean reset) throws IOException {
        Source source;
        WeakReference<Source> nameRef = filePathToSource.get(fileName);
        Source source2 = source = nameRef == null ? null : (Source)nameRef.get();
        if (source == null) {
            File file = new File(fileName);
            if (!file.canRead()) {
                throw new IOException("Can't read file " + fileName);
            }
            String path = file.getCanonicalPath();
            WeakReference<Source> pathRef = filePathToSource.get(path);
            Source source3 = source = pathRef == null ? null : (Source)pathRef.get();
            if (source == null) {
                source = new FileSource(file, fileName, path);
                filePathToSource.put(path, new WeakReference<Source>(source));
            }
        }
        if (reset) {
            source.reset();
        }
        Source.notifyNewSource(source).tagAs(Tags.FROM_FILE);
        return source;
    }

    public static Source fromFileName(String fileName) throws IOException {
        return Source.fromFileName(fileName, false);
    }

    public static Source fromFileName(CharSequence chars, String fileName) throws IOException {
        Source source;
        WeakReference<Source> nameRef = filePathToSource.get(fileName);
        Source source2 = source = nameRef == null ? null : (Source)nameRef.get();
        if (source == null) {
            File file = new File(fileName);
            String path = file.getCanonicalPath();
            WeakReference<Source> pathRef = filePathToSource.get(path);
            Source source3 = source = pathRef == null ? null : (Source)pathRef.get();
            if (source == null) {
                source = new FileSource(file, fileName, path, chars);
                filePathToSource.put(path, new WeakReference<Source>(source));
            }
        }
        Source.notifyNewSource(source).tagAs(Tags.FROM_FILE);
        return source;
    }

    public static Source fromText(CharSequence chars, String description) {
        assert (chars != null);
        LiteralSource source = new LiteralSource(description, chars.toString());
        Source.notifyNewSource(source).tagAs(Tags.FROM_LITERAL);
        return source;
    }

    public static Source fromURL(URL url, String description) throws IOException {
        URLSource source = URLSource.get(url, description);
        Source.notifyNewSource(source).tagAs(Tags.FROM_URL);
        return source;
    }

    public static Source fromReader(Reader reader, String description) throws IOException {
        LiteralSource source = new LiteralSource(description, Source.read(reader));
        Source.notifyNewSource(source).tagAs(Tags.FROM_READER);
        return source;
    }

    public static Source fromBytes(byte[] bytes, String description, BytesDecoder decoder) {
        return Source.fromBytes(bytes, 0, bytes.length, description, decoder);
    }

    public static Source fromBytes(byte[] bytes, int byteIndex, int length, String description, BytesDecoder decoder) {
        BytesSource source = new BytesSource(description, bytes, byteIndex, length, decoder);
        Source.notifyNewSource(source).tagAs(Tags.FROM_BYTES);
        return source;
    }

    public static Source asPseudoFile(CharSequence chars, String pseudoFileName) {
        LiteralSource source = new LiteralSource(pseudoFileName, chars.toString());
        filePathToSource.put(pseudoFileName, new WeakReference<LiteralSource>(source));
        Source.notifyNewSource(source).tagAs(Tags.FROM_LITERAL);
        return source;
    }

    public static void setFileCaching(boolean enabled) {
        fileCacheEnabled = enabled;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static Collection<Source> findSourcesTaggedAs(SourceTag tag) {
        ArrayList<Source> taggedSources = new ArrayList<Source>();
        List<WeakReference<Source>> list = allSources;
        synchronized (list) {
            for (WeakReference<Source> ref : allSources) {
                Source source = (Source)ref.get();
                if (source == null || tag != null && !source.isTaggedAs(tag)) continue;
                taggedSources.add((Source)ref.get());
            }
        }
        return taggedSources;
    }

    public static void addSourceListener(SourceListener listener) {
        assert (listener != null);
        sourceListeners.add(listener);
    }

    public static void removeSourceListener(SourceListener listener) {
        sourceListeners.remove(listener);
    }

    private static Source notifyNewSource(Source source) {
        allSources.add(new WeakReference<Source>(source));
        for (SourceListener listener : sourceListeners) {
            listener.sourceCreated(source);
        }
        return source;
    }

    private static String read(Reader reader) throws IOException {
        int n;
        BufferedReader bufferedReader = new BufferedReader(reader);
        StringBuilder builder = new StringBuilder();
        char[] buffer = new char[1024];
        while ((n = bufferedReader.read(buffer)) != -1) {
            builder.append(buffer, 0, n);
        }
        return builder.toString();
    }

    Source() {
    }

    protected abstract void reset();

    public final boolean isTaggedAs(SourceTag tag) {
        assert (tag != null);
        return this.tags.contains(tag);
    }

    public final Collection<SourceTag> getSourceTags() {
        return Collections.unmodifiableCollection(this.tags);
    }

    public final Source tagAs(SourceTag tag) {
        assert (tag != null);
        if (!this.tags.contains(tag)) {
            this.tags.add(tag);
            for (SourceListener listener : sourceListeners) {
                listener.sourceTaggedAs(this, tag);
            }
        }
        return this;
    }

    public abstract String getName();

    public abstract String getShortName();

    public abstract String getPath();

    public abstract URL getURL();

    public abstract Reader getReader();

    public final InputStream getInputStream() {
        return new ByteArrayInputStream(this.getCode().getBytes());
    }

    public final int getLength() {
        return this.checkTextMap().length();
    }

    public abstract String getCode();

    public String getCode(int charIndex, int charLength) {
        return this.getCode().substring(charIndex, charIndex + charLength);
    }

    public final String getCode(int lineNumber) {
        this.checkTextMap();
        int offset = this.textMap.lineStartOffset(lineNumber);
        int length = this.textMap.lineLength(lineNumber);
        return this.getCode().substring(offset, offset + length);
    }

    public final int getLineCount() {
        return this.checkTextMap().lineCount();
    }

    public final int getLineNumber(int offset) throws IllegalArgumentException {
        return this.checkTextMap().offsetToLine(offset);
    }

    public final int getColumnNumber(int offset) throws IllegalArgumentException {
        return this.checkTextMap().offsetToCol(offset);
    }

    public final int getLineStartOffset(int lineNumber) throws IllegalArgumentException {
        return this.checkTextMap().lineStartOffset(lineNumber);
    }

    public final int getLineLength(int lineNumber) throws IllegalArgumentException {
        return this.checkTextMap().lineLength(lineNumber);
    }

    public final SourceSection createSection(String identifier, int startLine, int startColumn, int charIndex, int length) {
        return new DefaultSourceSection(this, identifier, startLine, startColumn, charIndex, length);
    }

    public final SourceSection createSection(String identifier, int startLine, int startColumn, int length) {
        this.checkTextMap();
        int lineStartOffset = this.textMap.lineStartOffset(startLine);
        if (startColumn > this.textMap.lineLength(startLine)) {
            throw new IllegalArgumentException("column out of range");
        }
        int startOffset = lineStartOffset + startColumn - 1;
        return new DefaultSourceSection(this, identifier, startLine, startColumn, startOffset, length);
    }

    public final SourceSection createSection(String identifier, int charIndex, int length) throws IllegalArgumentException {
        this.checkRange(charIndex, length);
        this.checkTextMap();
        int startLine = this.getLineNumber(charIndex);
        int startColumn = charIndex - this.getLineStartOffset(startLine) + 1;
        return new DefaultSourceSection(this, identifier, startLine, startColumn, charIndex, length);
    }

    protected void checkRange(int charIndex, int length) {
        if (charIndex < 0 || length < 0 || charIndex + length > this.getCode().length()) {
            throw new IllegalArgumentException("text positions out of range");
        }
    }

    public final SourceSection createSection(String identifier, int lineNumber) {
        this.checkTextMap();
        int charIndex = this.textMap.lineStartOffset(lineNumber);
        int length = this.textMap.lineLength(lineNumber);
        return this.createSection(identifier, charIndex, length);
    }

    public final LineLocation createLineLocation(int lineNumber) {
        return new LineLocationImpl(this, lineNumber);
    }

    private TextMap checkTextMap() {
        if (this.textMap == null) {
            this.textMap = this.createTextMap();
        }
        return this.textMap;
    }

    protected TextMap createTextMap() {
        String code = this.getCode();
        if (code == null) {
            throw new RuntimeException("can't read file " + this.getName());
        }
        return TextMap.fromString(code);
    }

    private static final class BytesSource
    extends Source {
        private final String name;
        private final byte[] bytes;
        private final int byteIndex;
        private final int length;
        private final BytesDecoder decoder;

        public BytesSource(String name, byte[] bytes, int byteIndex, int length, BytesDecoder decoder) {
            this.name = name;
            this.bytes = bytes;
            this.byteIndex = byteIndex;
            this.length = length;
            this.decoder = decoder;
        }

        @Override
        protected void reset() {
        }

        @Override
        public String getName() {
            return this.name;
        }

        @Override
        public String getShortName() {
            return this.name;
        }

        @Override
        public String getPath() {
            return this.name;
        }

        @Override
        public URL getURL() {
            return null;
        }

        @Override
        public Reader getReader() {
            return null;
        }

        @Override
        public String getCode() {
            return this.decoder.decode(this.bytes, this.byteIndex, this.length);
        }

        @Override
        public String getCode(int byteOffset, int codeLength) {
            return this.decoder.decode(this.bytes, this.byteIndex + byteOffset, codeLength);
        }

        @Override
        protected void checkRange(int charIndex, int rangeLength) {
            if (charIndex < 0 || rangeLength < 0 || charIndex + rangeLength > this.length) {
                throw new IllegalArgumentException("text positions out of range");
            }
        }

        @Override
        protected TextMap createTextMap() {
            return TextMap.fromBytes(this.bytes, this.byteIndex, this.length, this.decoder);
        }
    }

    private static final class DefaultSourceSection
    implements SourceSection {
        private final Source source;
        private final String identifier;
        private final int startLine;
        private final int startColumn;
        private final int charIndex;
        private final int charLength;

        public DefaultSourceSection(Source source, String identifier, int startLine, int startColumn, int charIndex, int charLength) {
            this.source = source;
            this.identifier = identifier;
            this.startLine = startLine;
            this.startColumn = startColumn;
            this.charIndex = charIndex;
            this.charLength = charLength;
        }

        @Override
        public Source getSource() {
            return this.source;
        }

        @Override
        public int getStartLine() {
            return this.startLine;
        }

        @Override
        public LineLocation getLineLocation() {
            return this.source.createLineLocation(this.startLine);
        }

        @Override
        public int getStartColumn() {
            return this.startColumn;
        }

        @Override
        public int getEndLine() {
            return this.source.getLineNumber(this.charIndex + this.charLength - 1);
        }

        @Override
        public int getEndColumn() {
            return this.source.getColumnNumber(this.charIndex + this.charLength - 1);
        }

        @Override
        public int getCharIndex() {
            return this.charIndex;
        }

        @Override
        public int getCharLength() {
            return this.charLength;
        }

        @Override
        public int getCharEndIndex() {
            return this.charIndex + this.charLength;
        }

        @Override
        public String getIdentifier() {
            return this.identifier;
        }

        @Override
        public String getCode() {
            return this.getSource().getCode(this.charIndex, this.charLength);
        }

        @Override
        public String getShortDescription() {
            return String.format("%s:%d", this.source.getShortName(), this.startLine);
        }

        public String toString() {
            return this.getCode();
        }

        public int hashCode() {
            int prime = 31;
            int result = 1;
            result = 31 * result + this.charIndex;
            result = 31 * result + this.charLength;
            result = 31 * result + (this.identifier == null ? 0 : this.identifier.hashCode());
            result = 31 * result + (this.source == null ? 0 : this.source.hashCode());
            result = 31 * result + this.startColumn;
            result = 31 * result + this.startLine;
            return result;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (!(obj instanceof DefaultSourceSection)) {
                return false;
            }
            DefaultSourceSection other = (DefaultSourceSection)obj;
            if (this.charIndex != other.charIndex) {
                return false;
            }
            if (this.charLength != other.charLength) {
                return false;
            }
            if (this.identifier == null ? other.identifier != null : !this.identifier.equals(other.identifier)) {
                return false;
            }
            if (this.source == null ? other.source != null : !this.source.equals(other.source)) {
                return false;
            }
            if (this.startColumn != other.startColumn) {
                return false;
            }
            return this.startLine == other.startLine;
        }
    }

    private static final class FileSource
    extends Source {
        private final File file;
        private final String name;
        private final String path;
        private String code = null;
        private long timeStamp;

        public FileSource(File file, String name, String path) {
            this(file, name, path, null);
        }

        public FileSource(File file, String name, String path, CharSequence chars) {
            this.file = file.getAbsoluteFile();
            this.name = name;
            this.path = path;
            if (chars != null) {
                this.code = chars.toString();
            }
        }

        @Override
        public String getName() {
            return this.name;
        }

        @Override
        public String getShortName() {
            return this.file.getName();
        }

        @Override
        public String getCode() {
            if (fileCacheEnabled) {
                if (this.code == null || this.timeStamp != this.file.lastModified()) {
                    try {
                        this.code = Source.read(this.getReader());
                        this.timeStamp = this.file.lastModified();
                    }
                    catch (IOException iOException) {
                        // empty catch block
                    }
                }
                return this.code;
            }
            try {
                return Source.read(new FileReader(this.file));
            }
            catch (IOException iOException) {
                return null;
            }
        }

        @Override
        public String getPath() {
            return this.path;
        }

        @Override
        public URL getURL() {
            return null;
        }

        @Override
        public Reader getReader() {
            if (this.code != null && this.timeStamp == this.file.lastModified()) {
                return new StringReader(this.code);
            }
            try {
                return new FileReader(this.file);
            }
            catch (FileNotFoundException e) {
                throw new RuntimeException("Can't find file " + this.path, e);
            }
        }

        public int hashCode() {
            return this.path.hashCode();
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj instanceof FileSource) {
                FileSource other = (FileSource)obj;
                return this.path.equals(other.path);
            }
            return false;
        }

        @Override
        protected void reset() {
            this.code = null;
        }
    }

    private static final class LineLocationImpl
    implements LineLocation {
        private final Source source;
        private final int line;

        public LineLocationImpl(Source source, int line) {
            assert (source != null);
            this.source = source;
            this.line = line;
        }

        @Override
        public Source getSource() {
            return this.source;
        }

        @Override
        public int getLineNumber() {
            return this.line;
        }

        @Override
        public String getShortDescription() {
            return String.valueOf(this.source.getShortName()) + ":" + this.line;
        }

        public String toString() {
            return "Line[" + this.getShortDescription() + "]";
        }

        public int hashCode() {
            int prime = 31;
            int result = 1;
            result = 31 * result + this.line;
            result = 31 * result + this.source.hashCode();
            return result;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (!(obj instanceof LineLocationImpl)) {
                return false;
            }
            LineLocationImpl other = (LineLocationImpl)obj;
            if (this.line != other.line) {
                return false;
            }
            return this.source.equals(other.source);
        }
    }

    private static final class LiteralSource
    extends Source {
        private final String name;
        private final String code;

        public LiteralSource(String name, String code) {
            this.name = name;
            this.code = code;
        }

        @Override
        public String getName() {
            return this.name;
        }

        @Override
        public String getShortName() {
            return this.name;
        }

        @Override
        public String getCode() {
            return this.code;
        }

        @Override
        public String getPath() {
            return this.name;
        }

        @Override
        public URL getURL() {
            return null;
        }

        @Override
        public Reader getReader() {
            return new StringReader(this.code);
        }

        @Override
        protected void reset() {
        }

        public int hashCode() {
            int prime = 31;
            int result = 1;
            result = 31 * result + this.name.hashCode();
            result = 31 * result + (this.code == null ? 0 : this.code.hashCode());
            return result;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (!(obj instanceof LiteralSource)) {
                return false;
            }
            LiteralSource other = (LiteralSource)obj;
            return this.name.equals(other.name) && this.code.equals(other.code);
        }
    }

    public static enum Tags implements SourceTag
    {
        FROM_BYTES("bytes", "read from bytes"),
        FROM_FILE("file", "read from a file"),
        FROM_LITERAL("literal", "from literal text"),
        FROM_READER("reader", "read from a Java Reader"),
        FROM_URL("URL", "read from a URL");

        private final String name;
        private final String description;

        private Tags(String name, String description) {
            this.name = name;
            this.description = description;
        }

        public String getName() {
            return this.name;
        }

        @Override
        public String getDescription() {
            return this.description;
        }
    }

    private static final class TextMap {
        private final int[] nlOffsets;
        private final int textLength;
        final boolean finalNL;

        public TextMap(int[] nlOffsets, int textLength, boolean finalNL) {
            this.nlOffsets = nlOffsets;
            this.textLength = textLength;
            this.finalNL = finalNL;
        }

        public static TextMap fromString(String text) {
            int textLength = text.length();
            ArrayList<Integer> lines = new ArrayList<Integer>();
            lines.add(0);
            int offset = 0;
            while (offset < text.length()) {
                int nlIndex = text.indexOf(10, offset);
                if (nlIndex < 0) break;
                offset = nlIndex + 1;
                lines.add(offset);
            }
            lines.add(Integer.MAX_VALUE);
            int[] nlOffsets = new int[lines.size()];
            int line = 0;
            while (line < lines.size()) {
                nlOffsets[line] = (Integer)lines.get(line);
                ++line;
            }
            boolean finalNL = textLength > 0 && textLength == nlOffsets[nlOffsets.length - 2];
            return new TextMap(nlOffsets, textLength, finalNL);
        }

        public static TextMap fromBytes(byte[] bytes, int byteIndex, int length, BytesDecoder bytesDecoder) {
            final ArrayList<Integer> lines = new ArrayList<Integer>();
            lines.add(0);
            bytesDecoder.decodeLines(bytes, byteIndex, length, new BytesDecoder.LineMarker(){

                @Override
                public void markLine(int index) {
                    lines.add(index);
                }
            });
            lines.add(Integer.MAX_VALUE);
            int[] nlOffsets = new int[lines.size()];
            int line = 0;
            while (line < lines.size()) {
                nlOffsets[line] = (Integer)lines.get(line);
                ++line;
            }
            boolean finalNL = length > 0 && length == nlOffsets[nlOffsets.length - 2];
            return new TextMap(nlOffsets, length, finalNL);
        }

        public int offsetToLine(int offset) throws IllegalArgumentException {
            if (offset < 0 || offset >= this.textLength) {
                throw new IllegalArgumentException("offset out of bounds");
            }
            int line = 1;
            while (offset >= this.nlOffsets[line]) {
                ++line;
            }
            return line;
        }

        public int offsetToCol(int offset) throws IllegalArgumentException {
            return 1 + offset - this.nlOffsets[this.offsetToLine(offset) - 1];
        }

        public int length() {
            return this.textLength;
        }

        public int lineCount() {
            if (this.textLength == 0) {
                return 0;
            }
            return this.finalNL ? this.nlOffsets.length - 2 : this.nlOffsets.length - 1;
        }

        public int lineStartOffset(int line) throws IllegalArgumentException {
            if (this.textLength == 0 || this.lineOutOfRange(line)) {
                throw new IllegalArgumentException("line out of bounds");
            }
            return this.nlOffsets[line - 1];
        }

        public int lineLength(int line) throws IllegalArgumentException {
            if (this.textLength == 0 || this.lineOutOfRange(line)) {
                throw new IllegalArgumentException("line out of bounds");
            }
            if (line == this.nlOffsets.length - 1 && !this.finalNL) {
                return this.textLength - this.nlOffsets[line - 1];
            }
            return this.nlOffsets[line] - this.nlOffsets[line - 1] - 1;
        }

        private boolean lineOutOfRange(int line) {
            return line <= 0 || line >= this.nlOffsets.length || line == this.nlOffsets.length - 1 && this.finalNL;
        }
    }

    private static final class URLSource
    extends Source {
        private static final Map<URL, WeakReference<URLSource>> urlToSource = new HashMap<URL, WeakReference<URLSource>>();
        private final URL url;
        private final String name;
        private String code = null;

        public static URLSource get(URL url, String name) throws IOException {
            URLSource source;
            WeakReference<URLSource> sourceRef = urlToSource.get(url);
            URLSource uRLSource = source = sourceRef == null ? null : (URLSource)sourceRef.get();
            if (source == null) {
                source = new URLSource(url, name);
                urlToSource.put(url, new WeakReference<URLSource>(source));
            }
            return source;
        }

        public URLSource(URL url, String name) throws IOException {
            this.url = url;
            this.name = name;
            this.code = Source.read(new InputStreamReader(url.openStream()));
        }

        @Override
        public String getName() {
            return this.name;
        }

        @Override
        public String getShortName() {
            return this.name;
        }

        @Override
        public String getPath() {
            return this.url.getPath();
        }

        @Override
        public URL getURL() {
            return this.url;
        }

        @Override
        public Reader getReader() {
            return new StringReader(this.code);
        }

        @Override
        public String getCode() {
            return this.code;
        }

        @Override
        protected void reset() {
        }
    }
}

