"""
Tests specific to the lines module.
"""
from __future__ import (absolute_import, division, print_function,
                        unicode_literals)

from matplotlib.externals import six

import nose
from nose.tools import assert_true, assert_raises
from timeit import repeat
import numpy as np
import matplotlib as mpl
import matplotlib.pyplot as plt
from matplotlib.testing.decorators import cleanup, image_comparison
import sys


@cleanup
def test_invisible_Line_rendering():
    """
    Github issue #1256 identified a bug in Line.draw method

    Despite visibility attribute set to False, the draw method was not
    returning early enough and some pre-rendering code was executed
    though not necessary.

    Consequence was an excessive draw time for invisible Line instances
    holding a large number of points (Npts> 10**6)
    """
    # Creates big x and y data:
    N = 10**7
    x = np.linspace(0,1,N)
    y = np.random.normal(size=N)

    # Create a plot figure:
    fig = plt.figure()
    ax = plt.subplot(111)

    # Create a "big" Line instance:
    l = mpl.lines.Line2D(x,y)
    l.set_visible(False)
    # but don't add it to the Axis instance `ax`

    # [here Interactive panning and zooming is pretty responsive]
    # Time the canvas drawing:
    t_no_line = min(repeat(fig.canvas.draw, number=1, repeat=3))
    # (gives about 25 ms)

    # Add the big invisible Line:
    ax.add_line(l)

    # [Now interactive panning and zooming is very slow]
    # Time the canvas drawing:
    t_unvisible_line = min(repeat(fig.canvas.draw, number=1, repeat=3))
    # gives about 290 ms for N = 10**7 pts

    slowdown_factor = (t_unvisible_line/t_no_line)
    slowdown_threshold = 2 # trying to avoid false positive failures
    assert_true(slowdown_factor < slowdown_threshold)


@cleanup
def test_set_line_coll_dash():
    fig = plt.figure()
    ax = fig.add_subplot(1, 1, 1)

    np.random.seed(0)
    # Testing setting linestyles for line collections.
    # This should not produce an error.
    cs = ax.contour(np.random.randn(20, 30), linestyles=[(0, (3, 3))])

    assert True


@image_comparison(baseline_images=['line_dashes'], remove_text=True)
def test_line_dashes():
    fig = plt.figure()
    ax = fig.add_subplot(1, 1, 1)

    ax.plot(range(10), linestyle=(0, (3, 3)), lw=5)


@cleanup
def test_line_colors():
    fig = plt.figure()
    ax = fig.add_subplot(1, 1, 1)
    ax.plot(range(10), color='none')
    ax.plot(range(10), color='r')
    ax.plot(range(10), color='.3')
    ax.plot(range(10), color=(1, 0, 0, 1))
    ax.plot(range(10), color=(1, 0, 0))
    fig.canvas.draw()
    assert True


@cleanup
def test_linestyle_variants():
    fig = plt.figure()
    ax = fig.add_subplot(1, 1, 1)
    for ls in ["-", "solid", "--", "dashed",
               "-.", "dashdot", ":", "dotted"]:
        ax.plot(range(10), linestyle=ls)

    fig.canvas.draw()
    assert True


@cleanup
def test_valid_linestyles():
    if sys.version_info[:2] < (2, 7):
        raise nose.SkipTest("assert_raises as context manager "
                            "not supported with Python < 2.7")

    line = mpl.lines.Line2D([], [])
    with assert_raises(ValueError):
        line.set_linestyle('aardvark')


@image_comparison(baseline_images=['line_collection_dashes'], remove_text=True)
def test_set_line_coll_dash_image():
    fig = plt.figure()
    ax = fig.add_subplot(1, 1, 1)

    np.random.seed(0)
    cs = ax.contour(np.random.randn(20, 30), linestyles=[(0, (3, 3))])


def test_nan_is_sorted():
    # Exercises issue from PR #2744 (NaN throwing warning in _is_sorted)
    line = mpl.lines.Line2D([],[])
    assert_true(line._is_sorted(np.array([1, 2, 3])))
    assert_true(line._is_sorted(np.array([1, np.nan, 3])))
    assert_true(not line._is_sorted([3, 5] + [np.nan] * 100 + [0, 2]))


if __name__ == '__main__':
    nose.runmodule(argv=['-s', '--with-doctest'], exit=False)
