var baseFindIndex = require('./internal/baseFindIndex'),
    baseIteratee = require('./internal/baseIteratee');

/**
 * This method is like `_.findIndex` except that it iterates over elements
 * of `collection` from right to left.
 *
 * @static
 * @memberOf _
 * @category Array
 * @param {Array} array The array to search.
 * @param {Function|Object|string} [predicate=_.identity] The function invoked per iteration.
 * @returns {number} Returns the index of the found element, else `-1`.
 * @example
 *
 * var users = [
 *   { 'user': 'barney',  'active': true },
 *   { 'user': 'fred',    'active': false },
 *   { 'user': 'pebbles', 'active': false }
 * ];
 *
 * _.findLastIndex(users, function(o) { return o.user == 'pebbles'; });
 * // => 2
 *
 * // using the `_.matches` iteratee shorthand
 * _.findLastIndex(users, { 'user': 'barney', 'active': true });
 * // => 0
 *
 * // using the `_.matchesProperty` iteratee shorthand
 * _.findLastIndex(users, ['active', false]);
 * // => 2
 *
 * // using the `_.property` iteratee shorthand
 * _.findLastIndex(users, 'active');
 * // => 0
 */
function findLastIndex(array, predicate) {
  return (array && array.length)
    ? baseFindIndex(array, baseIteratee(predicate, 3), true)
    : -1;
}

module.exports = findLastIndex;
