# Copyright (c) 2014-2016 Cedric Bellegarde <cedric.bellegarde@adishatz.org>
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.

from gettext import gettext as _
import itertools

from lollypop.sqlcursor import SqlCursor
from lollypop.define import Lp, Type
from lollypop.utils import format_artist_name


class ArtistsDatabase:
    """
        Artists database helper
    """

    def __init__(self):
        """
            Init artists database object
        """
        pass

    def add(self, name, sortname):
        """
            Add a new artist to database
            @param artist name as string
            @param sortname as string
            @return inserted rowid as int
            @warning: commit needed
        """
        if sortname == "":
            sortname = format_artist_name(name)
        with SqlCursor(Lp().db) as sql:
            result = sql.execute("INSERT INTO artists (name, sortname)\
                                  VALUES (?, ?)",
                                 (name, sortname))
            return result.lastrowid

    def set_sortname(self, artist_id, sortname):
        """
            Set sort name
            @param id as int
            @param sort name a str
            @warning: commit needed
        """
        with SqlCursor(Lp().db) as sql:
            sql.execute("UPDATE artists\
                         SET sortname=?\
                         WHERE rowid=?",
                        (sortname, artist_id))

    def get_sortname(self, artist_id):
        """
            Return sortname
            @param artist id as int
            @return sortname as string
        """
        with SqlCursor(Lp().db) as sql:
            result = sql.execute("SELECT sortname from artists\
                                  WHERE rowid=?", (artist_id,))
            v = result.fetchone()
            if v is not None:
                return v[0]
            return self.get_name(artist_id)

    def get_id(self, name):
        """
            Get artist id
            @param Artist name as string
            @return Artist id as int
        """
        with SqlCursor(Lp().db) as sql:
            result = sql.execute("SELECT rowid from artists\
                                  WHERE name=?", (name,))
            v = result.fetchone()
            if v is not None:
                return v[0]
            return None

    def get_name(self, artist_id):
        """
            Get artist name
            @param Artist id as int
            @return Artist name as string
        """
        with SqlCursor(Lp().db) as sql:
            if artist_id == Type.COMPILATIONS:
                return _("Many artists")

            result = sql.execute("SELECT name from artists WHERE rowid=?",
                                 (artist_id,))
            v = result.fetchone()
            if v is not None:
                return v[0]
            return _("Unknown")

    def get_albums(self, artist_id):
        """
            Get all availables albums for artist
            @return Array of id as int
        """
        with SqlCursor(Lp().db) as sql:
            result = sql.execute("SELECT rowid FROM albums\
                                  WHERE artist_id=?\
                                  ORDER BY year", (artist_id,))
            return list(itertools.chain(*result))

    def get_compilations(self, artist_id):
        """
            Get all availables compilations for artist
            @return Array of id as int
        """
        with SqlCursor(Lp().db) as sql:
            result = sql.execute("SELECT DISTINCT albums.rowid FROM albums,\
                                  tracks, track_artists\
                                  WHERE track_artists.artist_id=?\
                                  AND track_artists.track_id=tracks.rowid\
                                  AND albums.rowid=tracks.album_id\
                                  AND albums.artist_id=?\
                                  ORDER BY albums.year", (artist_id,
                                                          Type.COMPILATIONS))
            return list(itertools.chain(*result))

    def get(self, genre_ids):
        """
            Get all available artists
            @param None
            or
            @param Filter genre ids as [int]/Type.ALL/None
            @return Array of (artist id as int, artist name as string)
        """
        with SqlCursor(Lp().db) as sql:
            result = []
            if genre_ids[0] == Type.ALL or not genre_ids:
                # Only artist that really have an album
                result = sql.execute(
                                 "SELECT DISTINCT artists.rowid,\
                                  artists.name\
                                  FROM artists, albums\
                                  WHERE albums.artist_id = artists.rowid\
                                  ORDER BY artists.sortname COLLATE NOCASE")
            else:
                genres = tuple(genre_ids)
                request = "SELECT DISTINCT artists.rowid,\
                           artists.name\
                           FROM artists, albums, album_genres\
                           WHERE artists.rowid == albums.artist_id\
                           AND album_genres.album_id=albums.rowid AND ("
                for genre_id in genre_ids:
                    request += "album_genres.genre_id=? OR "
                request += "1=0) ORDER BY artists.sortname\
                            COLLATE NOCASE"
                result = sql.execute(request, genres)
            return [(row[0], row[1]) for row in result]

    def exists(self, artist_id):
        """
            Return True if artist exist
            @param artist id as int
        """
        with SqlCursor(Lp().db) as sql:
            result = sql.execute("SELECT COUNT(1) from artists WHERE rowid=?",
                                 (artist_id,))
            v = result.fetchone()
            if v is not None:
                return bool(v[0])
            return False

    def search(self, string):
        """
            Search for artists looking like string
            @param string
            @return Array of id as int
        """
        with SqlCursor(Lp().db) as sql:
            result = sql.execute("SELECT rowid FROM artists\
                                  WHERE name LIKE ?\
                                  LIMIT 25", ('%' + string + '%',))
            return list(itertools.chain(*result))

    def count(self):
        """
            Count artists
            @return int
        """
        with SqlCursor(Lp().db) as sql:
            result = sql.execute("SELECT COUNT(1) from artists")
            v = result.fetchone()
            if v is not None:
                return v[0]
            return 0

    def clean(self, artist_id):
        """
            Clean database for artist id
            @param artist id as int
            @warning commit needed
        """
        with SqlCursor(Lp().db) as sql:
            result = sql.execute("SELECT rowid from albums\
                                  WHERE artist_id=?\
                                  LIMIT 1", (artist_id,))
            v = result.fetchone()
            # Check tracks
            if not v:
                result = sql.execute("SELECT track_id from track_artists\
                                     WHERE artist_id=?\
                                     LIMIT 1", (artist_id,))
                v = result.fetchone()
                # Artist with no relation, remove
                if not v:
                    sql.execute("DELETE FROM artists WHERE rowid=?",
                                (artist_id,))
