
/*
 Copyright 2013, Kovid Goyal <kovid at kovidgoyal.net>
 Released under the GPLv3 License
*/


(function() {
  var FontStats, fl_pat, font_dict, font_usage, log, process_font_face_rule, process_sheet, process_sheet_for_pseudo, process_style_rule;

  if (typeof window !== "undefined" && window !== null ? window.calibre_utils : void 0) {
    log = window.calibre_utils.log;
  }

  font_dict = function(style, computed) {
    var f, family, fams, _i, _len;
    if (computed == null) {
      computed = false;
    }
    if (computed) {
      fams = [];
      family = style.getPropertyCSSValue('font-family');
      if (family.cssValueType === CSSValue.CSS_PRIMITIVE_VALUE) {
        fams.push(family.getStringValue());
      } else {
        for (_i = 0, _len = family.length; _i < _len; _i++) {
          f = family[_i];
          fams.push(f.getStringValue());
        }
      }
    } else {
      fams = style.getPropertyValue('font-family');
    }
    return {
      'font-family': fams,
      'font-weight': style.getPropertyValue('font-weight'),
      'font-style': style.getPropertyValue('font-style'),
      'font-stretch': style.getPropertyValue('font-stretch')
    };
  };

  font_usage = function(node) {
    var ans, child, style, text, _i, _len, _ref;
    style = window.getComputedStyle(node, null);
    ans = font_dict(style, true);
    text = [];
    _ref = node.childNodes;
    for (_i = 0, _len = _ref.length; _i < _len; _i++) {
      child = _ref[_i];
      if (child.nodeType === Node.TEXT_NODE) {
        text.push(child.nodeValue);
      }
    }
    ans['text'] = text;
    return ans;
  };

  process_sheet = function(sheet, font_faces) {
    var rule, _i, _len, _ref, _results;
    _ref = sheet.cssRules;
    _results = [];
    for (_i = 0, _len = _ref.length; _i < _len; _i++) {
      rule = _ref[_i];
      if (rule.type === rule.FONT_FACE_RULE) {
        _results.push(process_font_face_rule(rule, font_faces));
      } else if (rule.type === rule.IMPORT_RULE && rule.styleSheet) {
        _results.push(process_sheet(rule.styleSheet, font_faces));
      } else {
        _results.push(void 0);
      }
    }
    return _results;
  };

  process_font_face_rule = function(rule, font_faces) {
    var fd;
    fd = font_dict(rule.style);
    fd['src'] = rule.style.getPropertyValue('src');
    return font_faces.push(fd);
  };

  fl_pat = /:{1,2}(first-letter|first-line)/i;

  process_sheet_for_pseudo = function(sheet, rules) {
    var ff, m, pseudo, rule, st, _i, _len, _ref, _results;
    _ref = sheet.cssRules;
    _results = [];
    for (_i = 0, _len = _ref.length; _i < _len; _i++) {
      rule = _ref[_i];
      if (rule.type === rule.STYLE_RULE) {
        st = rule.selectorText;
        m = fl_pat.exec(st);
        if (m) {
          pseudo = m[1].toLowerCase();
          ff = rule.style.getPropertyValue('font-family');
          if (ff) {
            _results.push(process_style_rule(st, rule.style, rules, pseudo));
          } else {
            _results.push(void 0);
          }
        } else {
          _results.push(void 0);
        }
      } else if (rule.type === rule.IMPORT_RULE && rule.styleSheet) {
        _results.push(process_sheet_for_pseudo(rule.styleSheet, rules));
      } else {
        _results.push(void 0);
      }
    }
    return _results;
  };

  process_style_rule = function(selector_text, style, rules, pseudo) {
    var element, fd, text, _i, _len, _ref, _results;
    selector_text = selector_text.replace(fl_pat, '');
    fd = font_dict(style);
    _ref = document.querySelectorAll(selector_text);
    _results = [];
    for (_i = 0, _len = _ref.length; _i < _len; _i++) {
      element = _ref[_i];
      text = element.innerText;
      if (text) {
        _results.push(rules.push([fd, text, pseudo]));
      } else {
        _results.push(void 0);
      }
    }
    return _results;
  };

  FontStats = (function() {

    FontStats.name = 'FontStats';

    function FontStats() {
      if (!this instanceof arguments.callee) {
        throw new Error('FontStats constructor called as function');
      }
    }

    FontStats.prototype.get_font_face_rules = function() {
      var font_faces, sheet, _i, _len, _ref;
      font_faces = [];
      _ref = document.styleSheets;
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        sheet = _ref[_i];
        process_sheet(sheet, font_faces);
      }
      return py_bridge.value = font_faces;
    };

    FontStats.prototype.get_font_usage = function() {
      var ans, busage, node, usage, _i, _len, _ref;
      ans = [];
      busage = font_usage(document.body);
      if (busage !== null) {
        ans.push(busage);
      }
      _ref = document.body.getElementsByTagName('*');
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        node = _ref[_i];
        usage = font_usage(node);
        if (usage !== null) {
          ans.push(usage);
        }
      }
      return py_bridge.value = ans;
    };

    FontStats.prototype.get_pseudo_element_font_usage = function() {
      var ans, sheet, _i, _len, _ref;
      ans = [];
      _ref = document.styleSheets;
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        sheet = _ref[_i];
        process_sheet_for_pseudo(sheet, ans);
      }
      return py_bridge.value = ans;
    };

    FontStats.prototype.get_font_families = function() {
      var ans, family, node, rule, rules, style, _i, _j, _len, _len1, _ref;
      ans = {};
      _ref = document.getElementsByTagName('*');
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        node = _ref[_i];
        rules = document.defaultView.getMatchedCSSRules(node, '');
        if (rules) {
          for (_j = 0, _len1 = rules.length; _j < _len1; _j++) {
            rule = rules[_j];
            style = rule.style;
            family = style.getPropertyValue('font-family');
            if (family) {
              ans[family] = true;
            }
          }
        }
        if (node.getAttribute('style')) {
          family = node.style.getPropertyValue('font-family');
          if (family) {
            ans[family] = true;
          }
        }
      }
      return py_bridge.value = ans;
    };

    return FontStats;

  })();

  if (typeof window !== "undefined" && window !== null) {
    window.font_stats = new FontStats();
  }

}).call(this);
