
/*
 Copyright 2012, Kovid Goyal <kovid at kovidgoyal.net>
 Released under the GPLv3 License
 */

(function() {
  var CalibreExtract, get_containing_block, get_epub_type, get_page_break, get_parents_and_self, hide_children, inline_styles, is_epub_footnote, is_footnote_link, log, merge, trim,
    indexOf = [].indexOf || function(item) { for (var i = 0, l = this.length; i < l; i++) { if (i in this && this[i] === item) return i; } return -1; },
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  if (typeof window !== "undefined" && window !== null ? window.calibre_utils : void 0) {
    log = window.calibre_utils.log;
  }

  merge = function(node, cnode) {
    var j, len, name, results, rule, rules, style, val;
    rules = node.ownerDocument.defaultView.getMatchedCSSRules(node, '');
    if (rules) {
      results = [];
      for (j = 0, len = rules.length; j < len; j++) {
        rule = rules[j];
        style = rule.style;
        results.push((function() {
          var k, len1, results1;
          results1 = [];
          for (k = 0, len1 = style.length; k < len1; k++) {
            name = style[k];
            val = style.getPropertyValue(name);
            if (val && !cnode.style.getPropertyValue(name)) {
              results1.push(cnode.style.setProperty(name, val));
            } else {
              results1.push(void 0);
            }
          }
          return results1;
        })());
      }
      return results;
    }
  };

  inline_styles = function(node) {
    var cnl, cnode, i, j, len, nl;
    cnode = node.cloneNode(true);
    merge(node, cnode);
    nl = node.getElementsByTagName('*');
    cnl = cnode.getElementsByTagName('*');
    for (i = j = 0, len = nl.length; j < len; i = ++j) {
      node = nl[i];
      merge(node, cnl[i]);
    }
    return cnode;
  };

  get_epub_type = function(node, possible_values) {
    var epub_type, j, len, ref, ref1, x;
    epub_type = node.getAttributeNS("http://www.idpf.org/2007/ops", 'type') || node.getAttribute('epub:type');
    if (!epub_type) {
      ref = node.attributes;
      for (j = 0, len = ref.length; j < len; j++) {
        x = ref[j];
        if (x.nodeName && (ref1 = x.nodeValue, indexOf.call(possible_values, ref1) >= 0) && x.nodeName.slice(-':type'.length) === ':type') {
          epub_type = x.nodeValue;
          break;
        }
      }
    }
    return epub_type;
  };

  get_containing_block = function(node) {
    var ref, ref1;
    while ((ref = node != null ? (ref1 = node.tagName) != null ? ref1.toLowerCase() : void 0 : void 0) !== 'p' && ref !== 'div' && ref !== 'li' && ref !== 'td' && ref !== 'h1' && ref !== 'h2' && ref !== 'h2' && ref !== 'h3' && ref !== 'h4' && ref !== 'h5' && ref !== 'h6' && ref !== 'body') {
      node = node.parentNode;
      if (!node) {
        break;
      }
    }
    return node;
  };

  trim = function(str) {
    return str.replace(/^\s\s*/, '').replace(/\s\s*$/, '');
  };

  is_footnote_link = function(node, url, linked_to_anchors) {
    var cb, children, ctext, eid, epub_type, ltext, num, ref, ref1, ref2, ref3, ref4, style, x;
    if (!url || url.substr(0, 'file://'.length).toLowerCase() !== 'file://') {
      return false;
    }
    epub_type = get_epub_type(node, ['noteref']);
    if (epub_type && epub_type.toLowerCase() === 'noteref') {
      return true;
    }
    if (epub_type && epub_type === 'link') {
      return false;
    }
    ref = [node, 3], x = ref[0], num = ref[1];
    while (x && num > 0) {
      style = window.getComputedStyle(x);
      if ((ref1 = !style.display) !== 'inline' && ref1 !== 'inline-block') {
        break;
      }
      if ((ref2 = style.verticalAlign) === 'sub' || ref2 === 'super' || ref2 === 'top' || ref2 === 'bottom') {
        return true;
      }
      x = x.parentNode;
      num -= 1;
    }
    children = (function() {
      var j, len, ref3, results;
      ref3 = node.childNodes;
      results = [];
      for (j = 0, len = ref3.length; j < len; j++) {
        x = ref3[j];
        if (x.nodeType === Node.ELEMENT_NODE) {
          results.push(x);
        }
      }
      return results;
    })();
    if (children.length === 1) {
      style = window.getComputedStyle(children[0]);
      if (((ref3 = style.display) === 'inline' || ref3 === 'inline-block') && ((ref4 = style.verticalAlign) === 'sub' || ref4 === 'super' || ref4 === 'top' || ref4 === 'bottom')) {
        return true;
      }
    }
    eid = node.getAttribute('id') || node.getAttribute('name');
    if (eid && linked_to_anchors.hasOwnProperty(eid)) {
      cb = get_containing_block(node);
      if (!cb || cb.tagName.toLowerCase() === 'body') {
        return false;
      }
      ltext = node.textContent;
      if (!ltext) {
        return false;
      }
      ctext = cb.textContent;
      if (!ctext) {
        return false;
      }
      if (trim(ctext) === trim(ltext)) {
        return false;
      }
      return true;
    }
    return false;
  };

  is_epub_footnote = function(node) {
    var epub_type, pv, ref;
    pv = ['note', 'footnote', 'rearnote'];
    epub_type = get_epub_type(node, pv);
    if (epub_type && (ref = epub_type.toLowerCase(), indexOf.call(pv, ref) >= 0)) {
      return true;
    }
    return false;
  };

  get_parents_and_self = function(node) {
    var ans;
    ans = [];
    while (node && node !== document.body) {
      ans.push(node);
      node = node.parentNode;
    }
    return ans;
  };

  get_page_break = function(node) {
    var ans, j, len, ref, ref1, style, x;
    style = getComputedStyle(node);
    ans = {};
    ref = ['before', 'after'];
    for (j = 0, len = ref.length; j < len; j++) {
      x = ref[j];
      ans[x] = (ref1 = style.getPropertyValue('page-break-'.concat(x))) === 'always' || ref1 === 'left' || ref1 === 'right';
    }
    return ans;
  };

  hide_children = function(node) {
    var child, j, len, ref, results;
    ref = node.childNodes;
    results = [];
    for (j = 0, len = ref.length; j < len; j++) {
      child = ref[j];
      if (child.nodeType === Node.ELEMENT_NODE) {
        if (child.do_not_hide) {
          results.push(hide_children(child));
        } else {
          results.push(child.style.display = 'none');
        }
      } else {
        results.push(void 0);
      }
    }
    return results;
  };

  CalibreExtract = (function() {
    function CalibreExtract() {
      this.extract = bind(this.extract, this);
      this.mark = bind(this.mark, this);
      if (!this instanceof arguments.callee) {
        throw new Error('CalibreExtract constructor called as function');
      }
      this.marked_node = null;
    }

    CalibreExtract.prototype.mark = function(node) {
      return this.marked_node = node;
    };

    CalibreExtract.prototype.extract = function(node) {
      var cnode;
      if (node == null) {
        node = null;
      }
      if (node === null) {
        node = this.marked_node;
      }
      cnode = inline_styles(node);
      return cnode.outerHTML;
    };

    CalibreExtract.prototype.is_footnote_link = function(a) {
      return is_footnote_link(a, a.href, py_bridge.value);
    };

    CalibreExtract.prototype.show_footnote = function(target, known_targets) {
      var child, elem, in_note, is_footnote_container, j, k, l, len, len1, len2, len3, m, pb, ref, ref1, ref2, ref3, ref4, start_elem, style;
      if (!target) {
        return;
      }
      start_elem = document.getElementById(target);
      if (!start_elem) {
        return;
      }
      in_note = false;
      is_footnote_container = is_epub_footnote(start_elem);
      ref = get_parents_and_self(start_elem);
      for (j = 0, len = ref.length; j < len; j++) {
        elem = ref[j];
        elem.do_not_hide = true;
        style = window.getComputedStyle(elem);
        if (style.display === 'list-item' && ((ref1 = style.listStyleType) !== 'disc' && ref1 !== 'circle' && ref1 !== 'square')) {
          elem.style.listStyleType = 'none';
        }
      }
      ref2 = document.body.getElementsByTagName('*');
      for (k = 0, len1 = ref2.length; k < len1; k++) {
        elem = ref2[k];
        if (in_note) {
          if (known_targets.hasOwnProperty(elem.getAttribute('id'))) {
            in_note = false;
            continue;
          }
          pb = get_page_break(elem);
          if (pb['before']) {
            in_note = false;
          } else if (pb['after']) {
            in_note = false;
            elem.do_not_hide = true;
            ref3 = elem.getElementsByTagName('*');
            for (l = 0, len2 = ref3.length; l < len2; l++) {
              child = ref3[l];
              child.do_not_hide = true;
            }
          } else {
            elem.do_not_hide = true;
          }
        } else {
          if (elem === start_elem) {
            in_note = !is_footnote_container && !get_page_break(elem)['after'];
            if (!in_note) {
              ref4 = elem.getElementsByTagName('*');
              for (m = 0, len3 = ref4.length; m < len3; m++) {
                child = ref4[m];
                child.do_not_hide = true;
              }
            }
          }
        }
      }
      hide_children(document.body);
      return location.hash = '#' + target;
    };

    return CalibreExtract;

  })();

  if (typeof window !== "undefined" && window !== null) {
    window.calibre_extract = new CalibreExtract();
  }

}).call(this);
