/*global CONFIG, SITE:true */
/*
  app.js
  Copyright 2015 Ubiquiti Networks, Inc. All rights reserved.
 */

define(['jquery',
    'backbone',
    'underscore',
    'msg/Msg',
    'unifi/model/CloudUser',
    'unifi/model/AdminPref',
    'ubnt/GoogleAnalytics',
    'text!includes/global/headerTemplate.html',
    'ubnt/view/dialogs/ErrorDialogView',
    'ubnt/view/dialogs/AlertDialogView',
    'ubnt/view/dialogs/LostConnectionDialogView',
    'libs/jquery.cookie'
], function (
    $,
    Backbone,
    _,
    Msg,
    CloudUser,
    AdminPref,
    GoogleAnalytics,
    headerTemplate,
    ErrorDialogView,
    AlertDialogView,
    LostConnectionDialogView
) {
    'use strict';
    var app = {};

    app.initialize = function(routers, collections, AppView) {
        var self = this;

        this.routers = {};
        this.views = {};
        this.data = {};
        this.adminPref = new AdminPref();
        this.collections = collections;

        $.cookie.defaults.path = '/';
        $(document).ajaxSend(function(event, jqxhr, settings) {
            if (settings.type == "POST") {
                jqxhr.setRequestHeader('X-Csrf-Token', $.cookie('csrf_token'));
            }
        });

        this.cloudUser = new CloudUser();
        this.cloudUser.fetch().done(function() {
            self.cloudUserCallback();

            //initialize the main application view
            self.views.app = new AppView({
                config: CONFIG,
                app: self,
                cloudUser: self.cloudUser
            });

            $('#header').html(_.template(headerTemplate, {msg:Msg}));

            if (CONFIG.site === 'Operator') {
                $('#header').find('fieldset.left').remove();
                $('body').addClass('isHotspot');
            }

            //initialize a backbone router for each main section and set up a trigger to change the nav style
            _.each(routers, function(Router, key) {
                var lcase = key[0].toLowerCase() + key.substr(1)
                ;

                self.routers[lcase] = new Router({ app: self });
                self.routers[lcase].on('route', function(route) {
                    self.views.app.navigate(lcase);
                });
            }, self);

            $(document).ajaxError(self.ajaxError);
            self.suppressAjaxErrors = 0;

            // Initalize Google Analytics
            //GoogleAnalytics.init('UA-XXXXX-X');

            //Initialize jQuery UI tooltips
            $(document).tooltip({
                show: false,
                hide: false,
                track: true
            });

            //load the globally used data and defer the app rendering until it's complete
            self.preload(self.collections);
        }).error(function() {
            // we errored while getting the current site information
            // something is wrong with the site context
            // let's logout and have the user login to get the correct data
            window.location.href = ((CONFIG.site === 'Admin') ? '' : '/hotspot') + '/logout';
        });
    };

    app.cloudUserCallback = function(){
        var self = this;
        var cloudUser = this.cloudUser;
        // set SITE configuration
        window.SITE = {
            id: cloudUser.get('site_id'),
            name: cloudUser.get('site_name')
        };

        // set USER configuration
        window.USER = {
            id: cloudUser.get('admin_id'),
            role: cloudUser.get('site_role'),
            super: cloudUser.get('is_super'),
            is_local: cloudUser.get('is_local')
        };

        // is super?
        if (cloudUser.get('site_role') === 'admin' && cloudUser.get('is_super') === true) {
            $.Body.addClass('isSuper');
            self.isSuper = true;
        } else {
            $.Body.removeClass('isSuper');
            self.isSuper = false;
        }

        // is readonly?
        if (cloudUser.get('site_role') === 'readonly') {
            $.Body.addClass('isReadOnly');
            self.isReadOnly = true;
            $(document)
                .off('keyup keypress', 'form input[type="checkbox"],form input[type="text"]', self.ignore_enter_keypress)
                .on('keyup keypress', 'form input[type="checkbox"],form input[type="text"]', self.ignore_enter_keypress);
        } else {
            $.Body.removeClass('isReadOnly');
            self.isReadOnly = false;

            $(document).off('keyup keypress', 'form input[type="checkbox"],form input[type="text"]', self.ignore_enter_keypress);
        }

        // is local user?
        if (cloudUser.get('is_local')) {
            $.Body.addClass('isLocalUser');
            self.isLocalUser = true;
        } else {
            $.Body.removeClass('isLocalUser');
            self.isLocalUser = false;
        }

        if (cloudUser.get('requires_new_password')) {
            $.Body.addClass('passwordChangeRequired');
            displayChangePassword();
        } else {
            $.Body.removeClass('passwordChangeRequired');
        }
    };

    app.ignore_enter_keypress = function(e){
        if(e.keyCode == 13) {
            e.preventDefault();
            return false;
        }
    };

    app.preload = function(collections) {
        var promises = [],
            deferred = {}
        ;

        //set up a promise for each collection
        _.each(collections, function(collection, collectionName) {
            promises.push(deferred[collectionName] = $.Deferred());
        }, this);

        //defer the rendering until the collections have loaded
        $.when.apply($, promises).then(function() {
            app.render();
        }, function() {
            app.fatal();
        });

        //load each collection and resolve the promises
        _.each(collections, function(collection, collectionName) {
            this.collection(collectionName, {
                'fetch': {
                    'reload': false,
                    'deferred': deferred[collectionName]
                }
            },
            collection);
        }, this);
    };

    app.render = function() {
        this.views.app.render();

        //force initialize the views that are required by more than one section
        this.routers.devices && this.routers.devices.index();
        this.routers.clients && this.routers.clients.index();

        //start the history or if it's already been started then reset the paths
        try {
            this.routers[CONFIG.start].route('', 'forward');
            Backbone.history.start({pushState: true, root: getRootURL()});
        } catch (e) {
            Backbone.history.root = getRootURL();

            var $navigation = $('#navigation');
            app.updatePushStateUrls($navigation.find('a').not('a[data-page=settings]'));
        }

        //load the URL of an empty page first to force backbone to be allowed to navigate to the right page
        //note: see line 1470 of backbone where if Backbone.history.fragment matches the new fragment then it
        //won't navigate regardless of whether Backbone.history.root has changed
        var section = Backbone.history.fragment || CONFIG.start;
        Backbone.history.loadUrl('');
        this.routers[section.split('/')[0]].navigate(section, {replace: false, trigger: true});
    };

    app.refresh = function(manual) {
        if (app.lostConnectionDialog && app.lostConnectionDialog.$el.is(':visible')) {
            app.lostConnectionDialog.hide();
        }

        if (!this._refreshing && (this._refreshing = true)) {
            var promises = [],
                deferred = {},
                self = this,
                view = this.views.app
            ;

            //notify the app that it's refreshing
            view.refreshing();

            //set up a promise for each collection
            _.each(this.data, function(data, name) {
                promises.push(deferred[name] = $.Deferred());
            });

            //when the collections have loaded notify the app of the status
            $.when.apply($, promises).then(function() {
                self._refreshing = false;
                view.refreshing(true);
            }, function() {
                self._refreshing = false;
                view.refreshing(false);
            });

            //re-fetch all the currently fetched data for a refresh - the views should handle data changes automatically
            _.each(this.data, function(data, name) {
                self.collection(name, {
                    'fetch': {
                        'reload': true,
                        'method': 'reload',
                        'deferred': deferred[name]
                    }
                });
            }, this);
        }
    };

    app.changeSite = function(id, route) {
        try {
            var site = id ? this.collection('Sites').get(id) : this.collection('Sites').first();
            if (site && site.id) {
                SITE = {
                    'id': site.id,
                    'name': site.get('name'),
                    'route': route || Backbone.history.fragment
                };
                this.adminPref.sync('update', this.adminPref, { 'last_site_name':SITE.name });
            } else {
                throw Msg.E_ChangeSite;
            }

            //reset the application view
            this.views.app.reset();

            //reset the routers which in turn should reset the views
            _.each(this.routers, function(router, name) {
                router.reset && router.reset();
            }, this);

            //clear out and delete all the collections that have the bySite flag
            _.each(this.data, function(data, name) {
                var collection = data.collection;
                if (collection.bySite) {
                    collection.reset();
                    delete this.data[name];
                }
            }, this);

            var self = this;
            this.cloudUser.url = '/api/s/' + site.get('name') + '/self';
            this.cloudUser.fetch().done(function(){
                self.cloudUserCallback();
                self.preload(self.collections);
                self.views.app.cloudUser = self.cloudUser;
            });
        } catch (e) {
            this.views.app.flash('error', 'Error', e);
            return;
        }
    };

    app.collection = function(collectionName, options, Collection) {
        options = options || {};
        options.init = options.init || {};
        options.fetch = options.fetch || {};

        //if the collection can't be instantiated then reject the deferred object if it exists
        if (_.isUndefined(this.data[collectionName])) {
            try {
                if (!_.isObject(Collection)) {
                    Collection = require('unifi/collection/' + collectionName);
                }

                this.data[collectionName] = {
                    collection: new Collection(options.init.models, options.init.options),
                    state: null,
                    deferred: []
                };
            } catch (e) {
                if (options.fetch && options.fetch.deferred) {
                    options.fetch.deferred.reject();
                }
                throw 'Invalid collection: ' + collectionName;
            }
        }

        var data = this.data[collectionName];

        //fetch the collection (or optionally reload it) and resolve or reject the deferred objects
        if (_.size(options.fetch)) {
            if (options.fetch.deferred) {
                data.deferred.push(options.fetch.deferred);
            }

            if (data.state != 'fetching') {
                if (!data.state || options.fetch.reload) {
                    data.state = 'fetching';

                    data.collection[options.fetch.method || 'fetch'](_.extend(options.fetch.options || {}, {
                        'success': function() {
                            data.state = 'fetched';
                            data.deferred && _.each(data.deferred, function(deferred, key) {
                                if (deferred.state() == 'pending') {
                                    deferred.resolve(arguments);
                                    delete data.deferred[key];
                                }
                            });
                        },
                        'error': function() {
                            data.state = 'fetched';
                            data.deferred && _.each(data.deferred, function(deferred, key) {
                                if (deferred.state() == 'pending') {
                                    deferred.reject(arguments);
                                    delete data.deferred[key];
                                }
                            });
                        }
                    }));
                } else {
                    data.deferred && _.each(data.deferred, function(deferred, key) {
                        if (deferred.state() == 'pending') {
                            deferred.resolve();
                            delete data.deferred[key];
                        }
                    });
                }
            }
        }

        return data.collection;
    };

    app.fatal = function() {
        if (this.views.app) {
            this.views.app.flash('error', 'Fatal Error', Msg.E_Fatal, true);
        } else {
            (new ErrorDialogView()).open(Msg.E_Fatal);
        }
    };

    app.ajaxError = function(event, jqxhr, settings, exception) {
        if (app.suppressAjaxErrors) {
            return;
        }

        if (jqxhr.status === 401 || jqxhr.status === 403) {
            if (app.lostConnectionDialog && app.lostConnectionDialog.$el.is(':visible')) {
                app.lostConnectionDialog.hide();
            }

            if (!app.alertDialog) {
                app.alertDialog = new AlertDialogView();
            }

            if (!app.alertDialog.$el.is(':visible')) {
                app.alertDialog.open(Msg.E_SessionExpired);

                _.delay(function() {
                    document.location = '/';
                }, 2000);
            }

        } else if ((jqxhr.readyState === 0 && jqxhr.status === 0 && jqxhr.statusText === 'error')) {
            if (!app.lostConnectionDialog) {
                app.lostConnectionDialog = new LostConnectionDialogView({
                    buttons: {
                        'Try Again': function() {
                            app.refresh(true);
                        }
                    }
                });
            }

            if (!app.lostConnectionDialog.$el.is(':visible')) {
                app.lostConnectionDialog.show();
            }
        }
    };

    app.ajaxErrorOff = function() {
        app.suppressAjaxErrors++;
    };

    app.ajaxErrorOn = function() {
        app.suppressAjaxErrors > 0 && app.suppressAjaxErrors--;
    };

    app.updatePushStateUrls = function($links) {
        $links.each(function() {
            var $this = $(this),
                page = $this.data('page'),
                href = (Backbone.history.root || getRootURL()) + page.toLowerCase();

            $this.attr('href', href);
        });
    };

    function getRootURL() {
        return '/' + (CONFIG.site === 'Admin' ? 'manage' : 'hotspot') + '/s/' + SITE.name + '/';
    }

    function displayChangePassword() {
        _.delay(function() {
            $('#ChangePassword').click();
        }, 1000);
    }

    window.googleMapsNotLoaded = function() {
        $('.google-map-container').html('<div class="googleMapsError">' + Msg.E_GoogleMapsNotLoaded + '</div>');
    };

    return app;
});
