--[[-
LuaJIT-based Scintilla interface.

This is a raw Scintilla interface without much in the way of documentation.

The auto-generated code is generated by sci_gen.lua, in the src/tools directory.

Copyright 2012-2015 The Howl Developers
License: MIT (see LICENSE.md at the top-level directory of the distribution)
]]

local _G = _G
local setmetatable, tonumber, pcall, error, print, tostring, coroutine =
      setmetatable, tonumber, pcall, error, print, tostring, coroutine
local string_format = string.format
local ffi = require('ffi')
local jit = require 'jit'
local bit = require('bit')
local cdefs = require('howl.cdefs')
local offsets = require('howl.offsets')
local colors = require('howl.ui.colors')
local gobject = require('ljglibs.gobject')
local gsignal = gobject.signal

local C, ffi_cast = ffi.C, ffi.cast
local char_arr, char_p, const_char_p = cdefs.char_arr, cdefs.char_p, cdefs.const_char_p
local glib = cdefs.glib
local gchar_arr = glib.gchar_arr

local sci = {}
local sci_mt = { __index = sci }

local _ENV = sci
if setfenv then setfenv(1, _ENV) end

local sci_map = {}
setmetatable(sci_map, { __mode = 'kv' })

ffi.cdef[[
typedef struct {
    long start_pos;
    long end_pos;
} char_range;

typedef struct {
    char_range range;
    char *text;
} text_range;

typedef struct {
    char_range range;
    char *text;
    char_range found_range;
} find_text;

struct Sci_NotifyHeader {
  void *hwndFrom;
  uintptr_t idFrom;
  unsigned int code;
};

typedef struct {
  struct Sci_NotifyHeader nmhdr;
  int position;
  int ch;
  int modifiers;
  int modificationType;
  const char *text;
  int length;
  int linesAdded;
  int message;
  uintptr_t wParam;
  uintptr_t lParam;
  int line;
  int foldLevelNow;
  int foldLevelPrev;
  int margin;
  int listType;
  int x;
  int y;
  int token;
  int annotationLinesAdded;
  int updated;
} SCNotification;

intptr_t scintilla_send_message(void *sci, int message, intptr_t wParam, intptr_t lParam);

GType scintilla_get_type();
GtkWidget *scintilla_new();
]]

local char_range = ffi.typeof('char_range')
local text_range = ffi.typeof('text_range')
local find_text = ffi.typeof('find_text')

local function string_ptr(s)
  return const_char_p(s)
end

local function string_to_color(spec)
  if not spec then return nil end
  spec = colors[spec] or spec
  local r, g, b = spec:match('^#?(%x%x)(%x%x)(%x%x)$')
  if not r then error("Invalid color specification '" .. spec .. "'", 3) end
  return tonumber(b .. g .. r, 16)
end

local function color_to_string(color)
  local hex = string_format('%.6x', tonumber(color))
  local b, g, r = hex:match('^(%x%x)(%x%x)(%x%x)$')
  if not r then error('Illegal color returned from scintilla: ' .. color) end
  local rgb = '#' .. r .. g .. b
  return colors.reverse[rgb] or rgb
end

local function explain_key_code(code, event)
  local effective_code = code == 10 and glib.GDK_KEY_Return or code

  local key_name = C.gdk_keyval_name(effective_code)
  if key_name ~= nil then
    event.key_name = ffi.string(key_name):lower()
  end
  local unicode_char = C.gdk_keyval_to_unicode(code);

  if (unicode_char ~= 0) then
    utf8 = gchar_arr(6)
    nr_utf8 = C.g_unichar_to_utf8(unicode_char, utf8);
    if nr_utf8 > 0 then
      event.character = ffi.string(utf8, nr_utf8)
    end
  end

  event.key_code = code
end

local function on_sci_key_press(sci_ptr, key)
  local get = function(sci_ptr, key)
    key = ffi.cast('GdkEventKey *', key)
    local event = {
      shift = bit.band(key.state, C.GDK_SHIFT_MASK) ~= 0,
      control = bit.band(key.state, C.GDK_CONTROL_MASK) ~= 0,
      alt = bit.band(key.state, C.GDK_MOD1_MASK) ~= 0,
      super = bit.band(key.state, C.GDK_SUPER_MASK) ~= 0,
      meta = bit.band(key.state, C.GDK_META_MASK) ~= 0,
    }
    explain_key_code(key.keyval, event)
    return event
  end
  local status, event = pcall(get, sci_ptr, key)
  if not status then
    print(event)
    return false
  else
    return dispatch(sci_ptr, 'key-press', event)
  end
end

local function on_sci_notify(sci_ptr, ctrl_id, notification)
  local n = ffi.cast('SCNotification *', notification)
  local code = n.nmhdr.code;

    -- fires a lot, ignore for now
  if code == SCN_PAINTED then return false end

  local event = { code = code }
  local mods = n.modifiers;

  if code == SCN_UPDATEUI then
    event.updated = n.updated
  else
    event.position = n.position

    if code == SCN_CHARADDED or code == SCN_KEY or code == SCN_DOUBLECLICK or
       code == SCN_HOTSPOTCLICK or code == SCN_HOTSPOTDOUBLECLICK or
       code == SCN_HOTSPOTRELEASECLICK or code == SCN_INDICATORCLICK or
       code == SCN_INDICATORRELEASE or code == SCN_MARGINCLICK then

      event.shift = bit.band(mods, SCMOD_SHIFT) ~= 0
      event.control = bit.band(mods, SCMOD_CTRL) ~= 0
      event.alt = bit.band(mods, SCMOD_ALT) ~= 0
      event.super = bit.band(mods, SCMOD_SUPER) ~= 0
      event.meta = bit.band(mods, SCMOD_META) ~= 0
    end

    if code == SCN_MODIFIED or code == SCN_DOUBLECLICK then
      event.line = n.line
    end

    if code == SCN_MODIFIED or code == SCN_USERLISTSELECTION or
        code == SCN_AUTOCSELECTION or code == SCN_URIDROPPED then
      if n.length > 0 and n.text ~= nil then
        event.text = ffi.string(n.text, n.length)
      end
    end

    if code == SCN_CHARADDED or code == SCN_KEY then
      explain_key_code(n.ch, event)
    elseif code == SCN_MODIFIED then
      event.type = n.modificationType
      event.length = n.length
      event.lines_affected = n.linesAdded
      event.fold_level_now = n.foldLevelNow
      event.fold_level_previous = n.foldLevelPrev
      event.token = n.token
    elseif code == SCN_USERLISTSELECTION then
      event.list_type = n.listType
    elseif code == SCN_MARGINCLICK then
      event.margin = n.margin
    elseif code == SCN_DWELLSTART or code == SCN_DWELLEND then
      event.x = n.x
      event.y = n.y
    end
  end

  local status, ret = pcall(dispatch, sci_ptr, 'sci', event)
  if status then
    return ret
  else
    print(ret)
    return false
  end
end

setmetatable(sci, {
  __call = function()

    local sci_ptr = gobject.ref_ptr(C.scintilla_new())
    local obj = setmetatable({
      sci_ptr = sci_ptr,
      multibyte = false,
      offsets = offsets()
    }, sci_mt)

    -- set up defaults
    obj:set_code_page(SC_CP_UTF8)
    obj:set_margin_width_n(1, 0) -- no fold margin
    obj:set_mod_event_mask(bit.bor(SC_MOD_INSERTTEXT, SC_MOD_DELETETEXT))
    obj:use_pop_up(false)
    obj:clear_all_cmd_keys()

    -- store in registry
    sci_map[sci_ptr] = obj
    sci_map[tostring(ffi.cast('gpointer', sci_ptr))] = obj

    gsignal.connect('bool3', sci_ptr, 'key-press-event', on_sci_key_press)
    gsignal.connect('bool4', sci_ptr, 'sci-notify', on_sci_notify)

    return obj
  end
})

function sci.string_to_color(rgb)
  if not rgb then return nil end
  local r, g, b = rgb:match('^#?(%x%x)(%x%x)(%x%x)$')
  if not r then error("Invalid color specification '" .. rgb .. "'", 2) end
  return tonumber(b .. g .. r, 16)
end

local function on_text_inserted(instance, args)
  if instance.multibyte == false then
    instance.multibyte = args.text.multibyte
  else
    instance.offsets:invalidate_from(args.at_pos)
  end
end

local function on_text_deleted(instance, args)
  if instance.multibyte then
    instance.multibyte = nil
    instance.offsets:invalidate_from(args.at_pos)
  end
end

function sci:is_multibyte()
  if self.multibyte == nil then
    self:character_count()
  end

  return self.multibyte
end

function sci:char_offset(byte_offset)
  if not self:is_multibyte() then return byte_offset end
  local ptr = self:get_character_pointer()
  return self.offsets:char_offset(ptr, byte_offset)
end

function sci:byte_offset(char_offset)
  if not self:is_multibyte() then return char_offset end
  local ptr = self:get_character_pointer()
  return self.offsets:byte_offset(ptr, char_offset)
end

function sci:character_count()
  local size = self:get_length()
  local count = tonumber(C.g_utf8_strlen(self:get_character_pointer(), size))
  -- Bug or weird behaviour with Scintillas count_characters
  -- local count = self:count_characters(0, self:get_length() + 1)
  self.multibyte = count ~= size
  return count
end

-- Overrides

-- Compact the document buffer and return a read-only pointer to the
-- characters in the document.
function sci:get_character_pointer()
  return ffi_cast(const_char_p, self:send(2520, 0, 0))
end

-- Return a read-only pointer to a range of characters in the document.
-- May move the gap so that the range is contiguous, but will only move up
-- to rangeLength bytes.
function sci:get_range_pointer(position, range_length)
  return ffi_cast(const_char_p, self:send(2643, position, range_length))
end

-- Change the document object used.
function sci:set_doc_pointer(pointer)
  self.offsets:invalidate_from(0)
  self.multibyte = nil
  self:send(2358, 0, pointer)
end

-- End overrides

function sci.dispatch(sci_ptr, event, args)
  local instance = sci_map[sci_ptr] or sci_map[tostring(sci_ptr)]
  if not instance then
    return true
  end
  local handler
  if event == 'sci' then
    code = args.code
    if code == SCN_STYLENEEDED then
      handler = 'on_style_needed'
      args = args.position
    elseif code == SCN_UPDATEUI then
      if bit.band(args.updated, SC_UPDATE_SELECTION) ~= 0 then
        handler = 'on_selection_changed'
      elseif bit.band(args.updated, SC_UPDATE_CONTENT) ~= 0 then
        handler = 'on_changed'
      end
    elseif code == SCN_CHARADDED then handler = 'on_char_added'
    elseif code == SCN_MODIFIED then
      local inserted = bit.band(args.type, SC_MOD_INSERTTEXT) ~= 0
      local deleted = bit.band(args.type, SC_MOD_DELETETEXT) ~= 0
      args = {
        at_pos = args.position,
        text = args.text,
        lines_affected = args.lines_affected,
        as_undo = bit.band(args.type, SC_PERFORMED_UNDO) ~= 0,
        as_redo = bit.band(args.type, SC_PERFORMED_REDO) ~= 0,
      }
      if inserted then
        handler = 'on_text_inserted'
        on_text_inserted(instance, args)
      elseif deleted then
        handler = 'on_text_deleted'
        on_text_deleted(instance, args)
      end
    elseif code == SCN_FOCUSIN then handler = 'on_focus'
    elseif code == SCN_FOCUSOUT then handler = 'on_focus_lost'
    elseif code == SCN_MODIFYATTEMPTRO then handler = 'on_readonly_mod_attempt'
    end
  elseif event == 'key-press' then handler = 'on_keypress'
  end

  local listener = instance.listener
  if listener and handler then
    if listener[handler] then
      local co = coroutine.create(function(...) return listener[handler](...) end)
      local status, ret = coroutine.resume(co, args)
      if not status then
        if listener.on_error then
          listener.on_error(ret)
          return true
        else
          error(ret)
        end
      end
      if coroutine.status(co) == 'dead' then
        if ret == true or ret == false then
          return ret
        end
      else
        return true
      end
    end
  end

  return false
end

function sci:to_gobject()
  return self.sci_ptr
end


function sci:send(message, arg1, arg2)
  return C.scintilla_send_message(self.sci_ptr, message, ffi.cast('intptr_t', arg1), ffi.cast('intptr_t', arg2))
end

function sci:send_with_stringresult(message, arg1)
  size = self:send(message, arg1, nil)
  if size == 0 then return '' end
  buffer = char_arr(size + 1)
  -- for the cases where the additional argument isn't specified,
  -- we should send the length as computed above
  if not arg1 then arg1 = size end
  self:send(message, arg1, buffer)

  -- don't include the trailing zero in the lua string if there is one
  if buffer[size - 1] == 0 then size = size -1 end

  return ffi.string(buffer, size)
end

function sci:send_with_textrange(message, start_pos, end_pos)
  if end_pos < start_pos then
    _G.error('Invalid range: end_pos < start_pos', 3)
  end
  size = end_pos - start_pos
  -- in case of style bytes we actually need two bytes per pos so double up
  buffer = char_arr((size * 2) + 2)
  tr = text_range({ char_range(start_pos, end_pos), buffer })
  real_size = self:send(message, nil, ffi.cast('text_range *', tr))
  -- but always return a string of the right size
  return ffi.string(buffer, real_size)
end

function sci:send_with_findtext(message, start_pos, end_pos, text)
  text = tostring(text)
  buffer = char_arr(#text, text)
  found_range = char_range()
  ft = find_text({ char_range({start_pos, end_pos}), buffer, found_range })
  found_at = self:send(message, nil, ffi.cast('find_text *', ft))
  if found_at >= 0 then return found_at, found_at + #text end
end

-- !! Begin auto generated content

INVALID_POSITION = -1
SCI_START = 2000
SCI_OPTIONAL_START = 3000
SCI_LEXER_START = 4000
SCI_ADDTEXT = 2001
SCI_ADDSTYLEDTEXT = 2002
SCI_INSERTTEXT = 2003
SCI_CLEARALL = 2004
SCI_DELETERANGE = 2645
SCI_CLEARDOCUMENTSTYLE = 2005
SCI_GETLENGTH = 2006
SCI_GETCHARAT = 2007
SCI_GETCURRENTPOS = 2008
SCI_GETANCHOR = 2009
SCI_GETSTYLEAT = 2010
SCI_REDO = 2011
SCI_SETUNDOCOLLECTION = 2012
SCI_SELECTALL = 2013
SCI_SETSAVEPOINT = 2014
SCI_GETSTYLEDTEXT = 2015
SCI_CANREDO = 2016
SCI_MARKERLINEFROMHANDLE = 2017
SCI_MARKERDELETEHANDLE = 2018
SCI_GETUNDOCOLLECTION = 2019
SCWS_INVISIBLE = 0
SCWS_VISIBLEALWAYS = 1
SCWS_VISIBLEAFTERINDENT = 2
SCI_GETVIEWWS = 2020
SCI_SETVIEWWS = 2021
SCI_POSITIONFROMPOINT = 2022
SCI_POSITIONFROMPOINTCLOSE = 2023
SCI_GOTOLINE = 2024
SCI_GOTOPOS = 2025
SCI_SETANCHOR = 2026
SCI_GETCURLINE = 2027
SCI_GETENDSTYLED = 2028
SC_EOL_CRLF = 0
SC_EOL_CR = 1
SC_EOL_LF = 2
SCI_CONVERTEOLS = 2029
SCI_GETEOLMODE = 2030
SCI_SETEOLMODE = 2031
SCI_STARTSTYLING = 2032
SCI_SETSTYLING = 2033
SCI_GETBUFFEREDDRAW = 2034
SCI_SETBUFFEREDDRAW = 2035
SCI_SETTABWIDTH = 2036
SCI_GETTABWIDTH = 2121
SC_CP_UTF8 = 65001
SCI_SETCODEPAGE = 2037
MARKER_MAX = 31
SC_MARK_CIRCLE = 0
SC_MARK_ROUNDRECT = 1
SC_MARK_ARROW = 2
SC_MARK_SMALLRECT = 3
SC_MARK_SHORTARROW = 4
SC_MARK_EMPTY = 5
SC_MARK_ARROWDOWN = 6
SC_MARK_MINUS = 7
SC_MARK_PLUS = 8
SC_MARK_VLINE = 9
SC_MARK_LCORNER = 10
SC_MARK_TCORNER = 11
SC_MARK_BOXPLUS = 12
SC_MARK_BOXPLUSCONNECTED = 13
SC_MARK_BOXMINUS = 14
SC_MARK_BOXMINUSCONNECTED = 15
SC_MARK_LCORNERCURVE = 16
SC_MARK_TCORNERCURVE = 17
SC_MARK_CIRCLEPLUS = 18
SC_MARK_CIRCLEPLUSCONNECTED = 19
SC_MARK_CIRCLEMINUS = 20
SC_MARK_CIRCLEMINUSCONNECTED = 21
SC_MARK_BACKGROUND = 22
SC_MARK_DOTDOTDOT = 23
SC_MARK_ARROWS = 24
SC_MARK_PIXMAP = 25
SC_MARK_FULLRECT = 26
SC_MARK_LEFTRECT = 27
SC_MARK_AVAILABLE = 28
SC_MARK_UNDERLINE = 29
SC_MARK_RGBAIMAGE = 30
SC_MARK_BOOKMARK = 31
SC_MARK_CHARACTER = 10000
SC_MARKNUM_FOLDEREND = 25
SC_MARKNUM_FOLDEROPENMID = 26
SC_MARKNUM_FOLDERMIDTAIL = 27
SC_MARKNUM_FOLDERTAIL = 28
SC_MARKNUM_FOLDERSUB = 29
SC_MARKNUM_FOLDER = 30
SC_MARKNUM_FOLDEROPEN = 31
SC_MASK_FOLDERS = 0xFE000000
SCI_MARKERDEFINE = 2040
SCI_MARKERSETFORE = 2041
SCI_MARKERSETBACK = 2042
SCI_MARKERSETBACKSELECTED = 2292
SCI_MARKERENABLEHIGHLIGHT = 2293
SCI_MARKERADD = 2043
SCI_MARKERDELETE = 2044
SCI_MARKERDELETEALL = 2045
SCI_MARKERGET = 2046
SCI_MARKERNEXT = 2047
SCI_MARKERPREVIOUS = 2048
SCI_MARKERDEFINEPIXMAP = 2049
SCI_MARKERADDSET = 2466
SCI_MARKERSETALPHA = 2476
SC_MAX_MARGIN = 4
SC_MARGIN_SYMBOL = 0
SC_MARGIN_NUMBER = 1
SC_MARGIN_BACK = 2
SC_MARGIN_FORE = 3
SC_MARGIN_TEXT = 4
SC_MARGIN_RTEXT = 5
SCI_SETMARGINTYPEN = 2240
SCI_GETMARGINTYPEN = 2241
SCI_SETMARGINWIDTHN = 2242
SCI_GETMARGINWIDTHN = 2243
SCI_SETMARGINMASKN = 2244
SCI_GETMARGINMASKN = 2245
SCI_SETMARGINSENSITIVEN = 2246
SCI_GETMARGINSENSITIVEN = 2247
SCI_SETMARGINCURSORN = 2248
SCI_GETMARGINCURSORN = 2249
STYLE_DEFAULT = 32
STYLE_LINENUMBER = 33
STYLE_BRACELIGHT = 34
STYLE_BRACEBAD = 35
STYLE_CONTROLCHAR = 36
STYLE_INDENTGUIDE = 37
STYLE_CALLTIP = 38
STYLE_LASTPREDEFINED = 39
STYLE_MAX = 255
SC_CHARSET_ANSI = 0
SC_CHARSET_DEFAULT = 1
SC_CHARSET_BALTIC = 186
SC_CHARSET_CHINESEBIG5 = 136
SC_CHARSET_EASTEUROPE = 238
SC_CHARSET_GB2312 = 134
SC_CHARSET_GREEK = 161
SC_CHARSET_HANGUL = 129
SC_CHARSET_MAC = 77
SC_CHARSET_OEM = 255
SC_CHARSET_RUSSIAN = 204
SC_CHARSET_CYRILLIC = 1251
SC_CHARSET_SHIFTJIS = 128
SC_CHARSET_SYMBOL = 2
SC_CHARSET_TURKISH = 162
SC_CHARSET_JOHAB = 130
SC_CHARSET_HEBREW = 177
SC_CHARSET_ARABIC = 178
SC_CHARSET_VIETNAMESE = 163
SC_CHARSET_THAI = 222
SC_CHARSET_8859_15 = 1000
SCI_STYLECLEARALL = 2050
SCI_STYLESETFORE = 2051
SCI_STYLESETBACK = 2052
SCI_STYLESETBOLD = 2053
SCI_STYLESETITALIC = 2054
SCI_STYLESETSIZE = 2055
SCI_STYLESETFONT = 2056
SCI_STYLESETEOLFILLED = 2057
SCI_STYLERESETDEFAULT = 2058
SCI_STYLESETUNDERLINE = 2059
SC_CASE_MIXED = 0
SC_CASE_UPPER = 1
SC_CASE_LOWER = 2
SCI_STYLEGETFORE = 2481
SCI_STYLEGETBACK = 2482
SCI_STYLEGETBOLD = 2483
SCI_STYLEGETITALIC = 2484
SCI_STYLEGETSIZE = 2485
SCI_STYLEGETFONT = 2486
SCI_STYLEGETEOLFILLED = 2487
SCI_STYLEGETUNDERLINE = 2488
SCI_STYLEGETCASE = 2489
SCI_STYLEGETCHARACTERSET = 2490
SCI_STYLEGETVISIBLE = 2491
SCI_STYLEGETCHANGEABLE = 2492
SCI_STYLEGETHOTSPOT = 2493
SCI_STYLESETCASE = 2060
SC_FONT_SIZE_MULTIPLIER = 100
SCI_STYLESETSIZEFRACTIONAL = 2061
SCI_STYLEGETSIZEFRACTIONAL = 2062
SC_WEIGHT_NORMAL = 400
SC_WEIGHT_SEMIBOLD = 600
SC_WEIGHT_BOLD = 700
SCI_STYLESETWEIGHT = 2063
SCI_STYLEGETWEIGHT = 2064
SCI_STYLESETCHARACTERSET = 2066
SCI_STYLESETHOTSPOT = 2409
SCI_SETSELFORE = 2067
SCI_SETSELBACK = 2068
SCI_GETSELALPHA = 2477
SCI_SETSELALPHA = 2478
SCI_GETSELEOLFILLED = 2479
SCI_SETSELEOLFILLED = 2480
SCI_SETCARETFORE = 2069
SCI_ASSIGNCMDKEY = 2070
SCI_CLEARCMDKEY = 2071
SCI_CLEARALLCMDKEYS = 2072
SCI_SETSTYLINGEX = 2073
SCI_STYLESETVISIBLE = 2074
SCI_GETCARETPERIOD = 2075
SCI_SETCARETPERIOD = 2076
SCI_SETWORDCHARS = 2077
SCI_GETWORDCHARS = 2646
SCI_BEGINUNDOACTION = 2078
SCI_ENDUNDOACTION = 2079
INDIC_PLAIN = 0
INDIC_SQUIGGLE = 1
INDIC_TT = 2
INDIC_DIAGONAL = 3
INDIC_STRIKE = 4
INDIC_HIDDEN = 5
INDIC_BOX = 6
INDIC_ROUNDBOX = 7
INDIC_STRAIGHTBOX = 8
INDIC_DASH = 9
INDIC_DOTS = 10
INDIC_SQUIGGLELOW = 11
INDIC_DOTBOX = 12
INDIC_SQUIGGLEPIXMAP = 13
INDIC_COMPOSITIONTHICK = 14
INDIC_MAX = 31
INDIC_CONTAINER = 8
INDIC0_MASK = 0x20
INDIC1_MASK = 0x40
INDIC2_MASK = 0x80
INDICS_MASK = 0xE0
SCI_INDICSETSTYLE = 2080
SCI_INDICGETSTYLE = 2081
SCI_INDICSETFORE = 2082
SCI_INDICGETFORE = 2083
SCI_INDICSETUNDER = 2510
SCI_INDICGETUNDER = 2511
SCI_SETWHITESPACEFORE = 2084
SCI_SETWHITESPACEBACK = 2085
SCI_SETWHITESPACESIZE = 2086
SCI_GETWHITESPACESIZE = 2087
SCI_SETSTYLEBITS = 2090
SCI_GETSTYLEBITS = 2091
SCI_SETLINESTATE = 2092
SCI_GETLINESTATE = 2093
SCI_GETMAXLINESTATE = 2094
SCI_GETCARETLINEVISIBLE = 2095
SCI_SETCARETLINEVISIBLE = 2096
SCI_GETCARETLINEBACK = 2097
SCI_SETCARETLINEBACK = 2098
SCI_STYLESETCHANGEABLE = 2099
SCI_AUTOCSHOW = 2100
SCI_AUTOCCANCEL = 2101
SCI_AUTOCACTIVE = 2102
SCI_AUTOCPOSSTART = 2103
SCI_AUTOCCOMPLETE = 2104
SCI_AUTOCSTOPS = 2105
SCI_AUTOCSETSEPARATOR = 2106
SCI_AUTOCGETSEPARATOR = 2107
SCI_AUTOCSELECT = 2108
SCI_AUTOCSETCANCELATSTART = 2110
SCI_AUTOCGETCANCELATSTART = 2111
SCI_AUTOCSETFILLUPS = 2112
SCI_AUTOCSETCHOOSESINGLE = 2113
SCI_AUTOCGETCHOOSESINGLE = 2114
SCI_AUTOCSETIGNORECASE = 2115
SCI_AUTOCGETIGNORECASE = 2116
SCI_USERLISTSHOW = 2117
SCI_AUTOCSETAUTOHIDE = 2118
SCI_AUTOCGETAUTOHIDE = 2119
SCI_AUTOCSETDROPRESTOFWORD = 2270
SCI_AUTOCGETDROPRESTOFWORD = 2271
SCI_REGISTERIMAGE = 2405
SCI_CLEARREGISTEREDIMAGES = 2408
SCI_AUTOCGETTYPESEPARATOR = 2285
SCI_AUTOCSETTYPESEPARATOR = 2286
SCI_AUTOCSETMAXWIDTH = 2208
SCI_AUTOCGETMAXWIDTH = 2209
SCI_AUTOCSETMAXHEIGHT = 2210
SCI_AUTOCGETMAXHEIGHT = 2211
SCI_SETINDENT = 2122
SCI_GETINDENT = 2123
SCI_SETUSETABS = 2124
SCI_GETUSETABS = 2125
SCI_SETLINEINDENTATION = 2126
SCI_GETLINEINDENTATION = 2127
SCI_GETLINEINDENTPOSITION = 2128
SCI_GETCOLUMN = 2129
SCI_COUNTCHARACTERS = 2633
SCI_SETHSCROLLBAR = 2130
SCI_GETHSCROLLBAR = 2131
SC_IV_NONE = 0
SC_IV_REAL = 1
SC_IV_LOOKFORWARD = 2
SC_IV_LOOKBOTH = 3
SCI_SETINDENTATIONGUIDES = 2132
SCI_GETINDENTATIONGUIDES = 2133
SCI_SETHIGHLIGHTGUIDE = 2134
SCI_GETHIGHLIGHTGUIDE = 2135
SCI_GETLINEENDPOSITION = 2136
SCI_GETCODEPAGE = 2137
SCI_GETCARETFORE = 2138
SCI_GETREADONLY = 2140
SCI_SETCURRENTPOS = 2141
SCI_SETSELECTIONSTART = 2142
SCI_GETSELECTIONSTART = 2143
SCI_SETSELECTIONEND = 2144
SCI_GETSELECTIONEND = 2145
SCI_SETEMPTYSELECTION = 2556
SCI_SETPRINTMAGNIFICATION = 2146
SCI_GETPRINTMAGNIFICATION = 2147
SC_PRINT_NORMAL = 0
SC_PRINT_INVERTLIGHT = 1
SC_PRINT_BLACKONWHITE = 2
SC_PRINT_COLOURONWHITE = 3
SC_PRINT_COLOURONWHITEDEFAULTBG = 4
SCI_SETPRINTCOLOURMODE = 2148
SCI_GETPRINTCOLOURMODE = 2149
SCFIND_WHOLEWORD = 0x2
SCFIND_MATCHCASE = 0x4
SCFIND_WORDSTART = 0x00100000
SCFIND_REGEXP = 0x00200000
SCFIND_POSIX = 0x00400000
SCI_FINDTEXT = 2150
SCI_FORMATRANGE = 2151
SCI_GETFIRSTVISIBLELINE = 2152
SCI_GETLINE = 2153
SCI_GETLINECOUNT = 2154
SCI_SETMARGINLEFT = 2155
SCI_GETMARGINLEFT = 2156
SCI_SETMARGINRIGHT = 2157
SCI_GETMARGINRIGHT = 2158
SCI_GETMODIFY = 2159
SCI_SETSEL = 2160
SCI_GETSELTEXT = 2161
SCI_GETTEXTRANGE = 2162
SCI_HIDESELECTION = 2163
SCI_POINTXFROMPOSITION = 2164
SCI_POINTYFROMPOSITION = 2165
SCI_LINEFROMPOSITION = 2166
SCI_POSITIONFROMLINE = 2167
SCI_LINESCROLL = 2168
SCI_SCROLLCARET = 2169
SCI_SCROLLRANGE = 2569
SCI_REPLACESEL = 2170
SCI_SETREADONLY = 2171
SCI_NULL = 2172
SCI_CANPASTE = 2173
SCI_CANUNDO = 2174
SCI_EMPTYUNDOBUFFER = 2175
SCI_UNDO = 2176
SCI_CUT = 2177
SCI_COPY = 2178
SCI_PASTE = 2179
SCI_CLEAR = 2180
SCI_SETTEXT = 2181
SCI_GETTEXT = 2182
SCI_GETTEXTLENGTH = 2183
SCI_GETDIRECTFUNCTION = 2184
SCI_GETDIRECTPOINTER = 2185
SCI_SETOVERTYPE = 2186
SCI_GETOVERTYPE = 2187
SCI_SETCARETWIDTH = 2188
SCI_GETCARETWIDTH = 2189
SCI_SETTARGETSTART = 2190
SCI_GETTARGETSTART = 2191
SCI_SETTARGETEND = 2192
SCI_GETTARGETEND = 2193
SCI_REPLACETARGET = 2194
SCI_REPLACETARGETRE = 2195
SCI_SEARCHINTARGET = 2197
SCI_SETSEARCHFLAGS = 2198
SCI_GETSEARCHFLAGS = 2199
SCI_CALLTIPSHOW = 2200
SCI_CALLTIPCANCEL = 2201
SCI_CALLTIPACTIVE = 2202
SCI_CALLTIPPOSSTART = 2203
SCI_CALLTIPSETPOSSTART = 2214
SCI_CALLTIPSETHLT = 2204
SCI_CALLTIPSETBACK = 2205
SCI_CALLTIPSETFORE = 2206
SCI_CALLTIPSETFOREHLT = 2207
SCI_CALLTIPUSESTYLE = 2212
SCI_CALLTIPSETPOSITION = 2213
SCI_VISIBLEFROMDOCLINE = 2220
SCI_DOCLINEFROMVISIBLE = 2221
SCI_WRAPCOUNT = 2235
SC_FOLDLEVELBASE = 0x400
SC_FOLDLEVELWHITEFLAG = 0x1000
SC_FOLDLEVELHEADERFLAG = 0x2000
SC_FOLDLEVELNUMBERMASK = 0x0FFF
SCI_SETFOLDLEVEL = 2222
SCI_GETFOLDLEVEL = 2223
SCI_GETLASTCHILD = 2224
SCI_GETFOLDPARENT = 2225
SCI_SHOWLINES = 2226
SCI_HIDELINES = 2227
SCI_GETLINEVISIBLE = 2228
SCI_GETALLLINESVISIBLE = 2236
SCI_SETFOLDEXPANDED = 2229
SCI_GETFOLDEXPANDED = 2230
SCI_TOGGLEFOLD = 2231
SC_FOLDACTION_CONTRACT = 0
SC_FOLDACTION_EXPAND = 1
SC_FOLDACTION_TOGGLE = 2
SCI_FOLDLINE = 2237
SCI_FOLDCHILDREN = 2238
SCI_EXPANDCHILDREN = 2239
SCI_FOLDALL = 2662
SCI_ENSUREVISIBLE = 2232
SC_AUTOMATICFOLD_SHOW = 0x0001
SC_AUTOMATICFOLD_CLICK = 0x0002
SC_AUTOMATICFOLD_CHANGE = 0x0004
SCI_SETAUTOMATICFOLD = 2663
SCI_GETAUTOMATICFOLD = 2664
SC_FOLDFLAG_LINEBEFORE_EXPANDED = 0x0002
SC_FOLDFLAG_LINEBEFORE_CONTRACTED = 0x0004
SC_FOLDFLAG_LINEAFTER_EXPANDED = 0x0008
SC_FOLDFLAG_LINEAFTER_CONTRACTED = 0x0010
SC_FOLDFLAG_LEVELNUMBERS = 0x0040
SCI_SETFOLDFLAGS = 2233
SCI_ENSUREVISIBLEENFORCEPOLICY = 2234
SCI_SETTABINDENTS = 2260
SCI_GETTABINDENTS = 2261
SCI_SETBACKSPACEUNINDENTS = 2262
SCI_GETBACKSPACEUNINDENTS = 2263
SC_TIME_FOREVER = 10000000
SCI_SETMOUSEDWELLTIME = 2264
SCI_GETMOUSEDWELLTIME = 2265
SCI_WORDSTARTPOSITION = 2266
SCI_WORDENDPOSITION = 2267
SC_WRAP_NONE = 0
SC_WRAP_WORD = 1
SC_WRAP_CHAR = 2
SC_WRAP_WHITESPACE = 3
SCI_SETWRAPMODE = 2268
SCI_GETWRAPMODE = 2269
SC_WRAPVISUALFLAG_NONE = 0x0000
SC_WRAPVISUALFLAG_END = 0x0001
SC_WRAPVISUALFLAG_START = 0x0002
SC_WRAPVISUALFLAG_MARGIN = 0x0004
SCI_SETWRAPVISUALFLAGS = 2460
SCI_GETWRAPVISUALFLAGS = 2461
SC_WRAPVISUALFLAGLOC_DEFAULT = 0x0000
SC_WRAPVISUALFLAGLOC_END_BY_TEXT = 0x0001
SC_WRAPVISUALFLAGLOC_START_BY_TEXT = 0x0002
SCI_SETWRAPVISUALFLAGSLOCATION = 2462
SCI_GETWRAPVISUALFLAGSLOCATION = 2463
SCI_SETWRAPSTARTINDENT = 2464
SCI_GETWRAPSTARTINDENT = 2465
SC_WRAPINDENT_FIXED = 0
SC_WRAPINDENT_SAME = 1
SC_WRAPINDENT_INDENT = 2
SCI_SETWRAPINDENTMODE = 2472
SCI_GETWRAPINDENTMODE = 2473
SC_CACHE_NONE = 0
SC_CACHE_CARET = 1
SC_CACHE_PAGE = 2
SC_CACHE_DOCUMENT = 3
SCI_SETLAYOUTCACHE = 2272
SCI_GETLAYOUTCACHE = 2273
SCI_SETSCROLLWIDTH = 2274
SCI_GETSCROLLWIDTH = 2275
SCI_SETSCROLLWIDTHTRACKING = 2516
SCI_GETSCROLLWIDTHTRACKING = 2517
SCI_TEXTWIDTH = 2276
SCI_SETENDATLASTLINE = 2277
SCI_GETENDATLASTLINE = 2278
SCI_TEXTHEIGHT = 2279
SCI_SETVSCROLLBAR = 2280
SCI_GETVSCROLLBAR = 2281
SCI_APPENDTEXT = 2282
SCI_GETTWOPHASEDRAW = 2283
SCI_SETTWOPHASEDRAW = 2284
SC_EFF_QUALITY_MASK = 0xF
SC_EFF_QUALITY_DEFAULT = 0
SC_EFF_QUALITY_NON_ANTIALIASED = 1
SC_EFF_QUALITY_ANTIALIASED = 2
SC_EFF_QUALITY_LCD_OPTIMIZED = 3
SCI_SETFONTQUALITY = 2611
SCI_GETFONTQUALITY = 2612
SCI_SETFIRSTVISIBLELINE = 2613
SC_MULTIPASTE_ONCE = 0
SC_MULTIPASTE_EACH = 1
SCI_SETMULTIPASTE = 2614
SCI_GETMULTIPASTE = 2615
SCI_GETTAG = 2616
SCI_TARGETFROMSELECTION = 2287
SCI_LINESJOIN = 2288
SCI_LINESSPLIT = 2289
SCI_SETFOLDMARGINCOLOUR = 2290
SCI_SETFOLDMARGINHICOLOUR = 2291
SCI_LINEDOWN = 2300
SCI_LINEDOWNEXTEND = 2301
SCI_LINEUP = 2302
SCI_LINEUPEXTEND = 2303
SCI_CHARLEFT = 2304
SCI_CHARLEFTEXTEND = 2305
SCI_CHARRIGHT = 2306
SCI_CHARRIGHTEXTEND = 2307
SCI_WORDLEFT = 2308
SCI_WORDLEFTEXTEND = 2309
SCI_WORDRIGHT = 2310
SCI_WORDRIGHTEXTEND = 2311
SCI_HOME = 2312
SCI_HOMEEXTEND = 2313
SCI_LINEEND = 2314
SCI_LINEENDEXTEND = 2315
SCI_DOCUMENTSTART = 2316
SCI_DOCUMENTSTARTEXTEND = 2317
SCI_DOCUMENTEND = 2318
SCI_DOCUMENTENDEXTEND = 2319
SCI_PAGEUP = 2320
SCI_PAGEUPEXTEND = 2321
SCI_PAGEDOWN = 2322
SCI_PAGEDOWNEXTEND = 2323
SCI_EDITTOGGLEOVERTYPE = 2324
SCI_CANCEL = 2325
SCI_DELETEBACK = 2326
SCI_TAB = 2327
SCI_BACKTAB = 2328
SCI_NEWLINE = 2329
SCI_FORMFEED = 2330
SCI_VCHOME = 2331
SCI_VCHOMEEXTEND = 2332
SCI_ZOOMIN = 2333
SCI_ZOOMOUT = 2334
SCI_DELWORDLEFT = 2335
SCI_DELWORDRIGHT = 2336
SCI_DELWORDRIGHTEND = 2518
SCI_LINECUT = 2337
SCI_LINEDELETE = 2338
SCI_LINETRANSPOSE = 2339
SCI_LINEDUPLICATE = 2404
SCI_LOWERCASE = 2340
SCI_UPPERCASE = 2341
SCI_LINESCROLLDOWN = 2342
SCI_LINESCROLLUP = 2343
SCI_DELETEBACKNOTLINE = 2344
SCI_HOMEDISPLAY = 2345
SCI_HOMEDISPLAYEXTEND = 2346
SCI_LINEENDDISPLAY = 2347
SCI_LINEENDDISPLAYEXTEND = 2348
SCI_HOMEWRAP = 2349
SCI_HOMEWRAPEXTEND = 2450
SCI_LINEENDWRAP = 2451
SCI_LINEENDWRAPEXTEND = 2452
SCI_VCHOMEWRAP = 2453
SCI_VCHOMEWRAPEXTEND = 2454
SCI_LINECOPY = 2455
SCI_MOVECARETINSIDEVIEW = 2401
SCI_LINELENGTH = 2350
SCI_BRACEHIGHLIGHT = 2351
SCI_BRACEHIGHLIGHTINDICATOR = 2498
SCI_BRACEBADLIGHT = 2352
SCI_BRACEBADLIGHTINDICATOR = 2499
SCI_BRACEMATCH = 2353
SCI_GETVIEWEOL = 2355
SCI_SETVIEWEOL = 2356
SCI_GETDOCPOINTER = 2357
SCI_SETDOCPOINTER = 2358
SCI_SETMODEVENTMASK = 2359
EDGE_NONE = 0
EDGE_LINE = 1
EDGE_BACKGROUND = 2
SCI_GETEDGECOLUMN = 2360
SCI_SETEDGECOLUMN = 2361
SCI_GETEDGEMODE = 2362
SCI_SETEDGEMODE = 2363
SCI_GETEDGECOLOUR = 2364
SCI_SETEDGECOLOUR = 2365
SCI_SEARCHANCHOR = 2366
SCI_SEARCHNEXT = 2367
SCI_SEARCHPREV = 2368
SCI_LINESONSCREEN = 2370
SCI_USEPOPUP = 2371
SCI_SELECTIONISRECTANGLE = 2372
SCI_SETZOOM = 2373
SCI_GETZOOM = 2374
SCI_CREATEDOCUMENT = 2375
SCI_ADDREFDOCUMENT = 2376
SCI_RELEASEDOCUMENT = 2377
SCI_GETMODEVENTMASK = 2378
SCI_SETFOCUS = 2380
SCI_GETFOCUS = 2381
SC_STATUS_OK = 0
SC_STATUS_FAILURE = 1
SC_STATUS_BADALLOC = 2
SCI_SETSTATUS = 2382
SCI_GETSTATUS = 2383
SCI_SETMOUSEDOWNCAPTURES = 2384
SCI_GETMOUSEDOWNCAPTURES = 2385
SC_CURSORNORMAL = -1
SC_CURSORARROW = 2
SC_CURSORWAIT = 4
SC_CURSORREVERSEARROW = 7
SCI_SETCURSOR = 2386
SCI_GETCURSOR = 2387
SCI_SETCONTROLCHARSYMBOL = 2388
SCI_GETCONTROLCHARSYMBOL = 2389
SCI_WORDPARTLEFT = 2390
SCI_WORDPARTLEFTEXTEND = 2391
SCI_WORDPARTRIGHT = 2392
SCI_WORDPARTRIGHTEXTEND = 2393
VISIBLE_SLOP = 0x01
VISIBLE_STRICT = 0x04
SCI_SETVISIBLEPOLICY = 2394
SCI_DELLINELEFT = 2395
SCI_DELLINERIGHT = 2396
SCI_SETXOFFSET = 2397
SCI_GETXOFFSET = 2398
SCI_CHOOSECARETX = 2399
SCI_GRABFOCUS = 2400
CARET_SLOP = 0x01
CARET_STRICT = 0x04
CARET_JUMPS = 0x10
CARET_EVEN = 0x08
SCI_SETXCARETPOLICY = 2402
SCI_SETYCARETPOLICY = 2403
SCI_SETPRINTWRAPMODE = 2406
SCI_GETPRINTWRAPMODE = 2407
SCI_SETHOTSPOTACTIVEFORE = 2410
SCI_GETHOTSPOTACTIVEFORE = 2494
SCI_SETHOTSPOTACTIVEBACK = 2411
SCI_GETHOTSPOTACTIVEBACK = 2495
SCI_SETHOTSPOTACTIVEUNDERLINE = 2412
SCI_GETHOTSPOTACTIVEUNDERLINE = 2496
SCI_SETHOTSPOTSINGLELINE = 2421
SCI_GETHOTSPOTSINGLELINE = 2497
SCI_PARADOWN = 2413
SCI_PARADOWNEXTEND = 2414
SCI_PARAUP = 2415
SCI_PARAUPEXTEND = 2416
SCI_POSITIONBEFORE = 2417
SCI_POSITIONAFTER = 2418
SCI_POSITIONRELATIVE = 2670
SCI_COPYRANGE = 2419
SCI_COPYTEXT = 2420
SC_SEL_STREAM = 0
SC_SEL_RECTANGLE = 1
SC_SEL_LINES = 2
SC_SEL_THIN = 3
SCI_SETSELECTIONMODE = 2422
SCI_GETSELECTIONMODE = 2423
SCI_GETLINESELSTARTPOSITION = 2424
SCI_GETLINESELENDPOSITION = 2425
SCI_LINEDOWNRECTEXTEND = 2426
SCI_LINEUPRECTEXTEND = 2427
SCI_CHARLEFTRECTEXTEND = 2428
SCI_CHARRIGHTRECTEXTEND = 2429
SCI_HOMERECTEXTEND = 2430
SCI_VCHOMERECTEXTEND = 2431
SCI_LINEENDRECTEXTEND = 2432
SCI_PAGEUPRECTEXTEND = 2433
SCI_PAGEDOWNRECTEXTEND = 2434
SCI_STUTTEREDPAGEUP = 2435
SCI_STUTTEREDPAGEUPEXTEND = 2436
SCI_STUTTEREDPAGEDOWN = 2437
SCI_STUTTEREDPAGEDOWNEXTEND = 2438
SCI_WORDLEFTEND = 2439
SCI_WORDLEFTENDEXTEND = 2440
SCI_WORDRIGHTEND = 2441
SCI_WORDRIGHTENDEXTEND = 2442
SCI_SETWHITESPACECHARS = 2443
SCI_GETWHITESPACECHARS = 2647
SCI_SETPUNCTUATIONCHARS = 2648
SCI_GETPUNCTUATIONCHARS = 2649
SCI_SETCHARSDEFAULT = 2444
SCI_AUTOCGETCURRENT = 2445
SCI_AUTOCGETCURRENTTEXT = 2610
SC_CASEINSENSITIVEBEHAVIOUR_RESPECTCASE = 0
SC_CASEINSENSITIVEBEHAVIOUR_IGNORECASE = 1
SCI_AUTOCSETCASEINSENSITIVEBEHAVIOUR = 2634
SCI_AUTOCGETCASEINSENSITIVEBEHAVIOUR = 2635
SC_ORDER_PRESORTED = 0
SC_ORDER_PERFORMSORT = 1
SC_ORDER_CUSTOM = 2
SCI_AUTOCSETORDER = 2660
SCI_AUTOCGETORDER = 2661
SCI_ALLOCATE = 2446
SCI_TARGETASUTF8 = 2447
SCI_SETLENGTHFORENCODE = 2448
SCI_ENCODEDFROMUTF8 = 2449
SCI_FINDCOLUMN = 2456
SCI_GETCARETSTICKY = 2457
SCI_SETCARETSTICKY = 2458
SC_CARETSTICKY_OFF = 0
SC_CARETSTICKY_ON = 1
SC_CARETSTICKY_WHITESPACE = 2
SCI_TOGGLECARETSTICKY = 2459
SCI_SETPASTECONVERTENDINGS = 2467
SCI_GETPASTECONVERTENDINGS = 2468
SCI_SELECTIONDUPLICATE = 2469
SC_ALPHA_TRANSPARENT = 0
SC_ALPHA_OPAQUE = 255
SC_ALPHA_NOALPHA = 256
SCI_SETCARETLINEBACKALPHA = 2470
SCI_GETCARETLINEBACKALPHA = 2471
CARETSTYLE_INVISIBLE = 0
CARETSTYLE_LINE = 1
CARETSTYLE_BLOCK = 2
SCI_SETCARETSTYLE = 2512
SCI_GETCARETSTYLE = 2513
SCI_SETINDICATORCURRENT = 2500
SCI_GETINDICATORCURRENT = 2501
SCI_SETINDICATORVALUE = 2502
SCI_GETINDICATORVALUE = 2503
SCI_INDICATORFILLRANGE = 2504
SCI_INDICATORCLEARRANGE = 2505
SCI_INDICATORALLONFOR = 2506
SCI_INDICATORVALUEAT = 2507
SCI_INDICATORSTART = 2508
SCI_INDICATOREND = 2509
SCI_SETPOSITIONCACHE = 2514
SCI_GETPOSITIONCACHE = 2515
SCI_COPYALLOWLINE = 2519
SCI_GETCHARACTERPOINTER = 2520
SCI_GETRANGEPOINTER = 2643
SCI_GETGAPPOSITION = 2644
SCI_SETKEYSUNICODE = 2521
SCI_GETKEYSUNICODE = 2522
SCI_INDICSETALPHA = 2523
SCI_INDICGETALPHA = 2524
SCI_INDICSETOUTLINEALPHA = 2558
SCI_INDICGETOUTLINEALPHA = 2559
SCI_SETEXTRAASCENT = 2525
SCI_GETEXTRAASCENT = 2526
SCI_SETEXTRADESCENT = 2527
SCI_GETEXTRADESCENT = 2528
SCI_MARKERSYMBOLDEFINED = 2529
SCI_MARGINSETTEXT = 2530
SCI_MARGINGETTEXT = 2531
SCI_MARGINSETSTYLE = 2532
SCI_MARGINGETSTYLE = 2533
SCI_MARGINSETSTYLES = 2534
SCI_MARGINGETSTYLES = 2535
SCI_MARGINTEXTCLEARALL = 2536
SCI_MARGINSETSTYLEOFFSET = 2537
SCI_MARGINGETSTYLEOFFSET = 2538
SC_MARGINOPTION_NONE = 0
SC_MARGINOPTION_SUBLINESELECT = 1
SCI_SETMARGINOPTIONS = 2539
SCI_GETMARGINOPTIONS = 2557
SCI_ANNOTATIONSETTEXT = 2540
SCI_ANNOTATIONGETTEXT = 2541
SCI_ANNOTATIONSETSTYLE = 2542
SCI_ANNOTATIONGETSTYLE = 2543
SCI_ANNOTATIONSETSTYLES = 2544
SCI_ANNOTATIONGETSTYLES = 2545
SCI_ANNOTATIONGETLINES = 2546
SCI_ANNOTATIONCLEARALL = 2547
ANNOTATION_HIDDEN = 0
ANNOTATION_STANDARD = 1
ANNOTATION_BOXED = 2
SCI_ANNOTATIONSETVISIBLE = 2548
SCI_ANNOTATIONGETVISIBLE = 2549
SCI_ANNOTATIONSETSTYLEOFFSET = 2550
SCI_ANNOTATIONGETSTYLEOFFSET = 2551
SCI_RELEASEALLEXTENDEDSTYLES = 2552
SCI_ALLOCATEEXTENDEDSTYLES = 2553
UNDO_MAY_COALESCE = 1
SCI_ADDUNDOACTION = 2560
SCI_CHARPOSITIONFROMPOINT = 2561
SCI_CHARPOSITIONFROMPOINTCLOSE = 2562
SCI_SETMOUSESELECTIONRECTANGULARSWITCH = 2668
SCI_GETMOUSESELECTIONRECTANGULARSWITCH = 2669
SCI_SETMULTIPLESELECTION = 2563
SCI_GETMULTIPLESELECTION = 2564
SCI_SETADDITIONALSELECTIONTYPING = 2565
SCI_GETADDITIONALSELECTIONTYPING = 2566
SCI_SETADDITIONALCARETSBLINK = 2567
SCI_GETADDITIONALCARETSBLINK = 2568
SCI_SETADDITIONALCARETSVISIBLE = 2608
SCI_GETADDITIONALCARETSVISIBLE = 2609
SCI_GETSELECTIONS = 2570
SCI_GETSELECTIONEMPTY = 2650
SCI_CLEARSELECTIONS = 2571
SCI_SETSELECTION = 2572
SCI_ADDSELECTION = 2573
SCI_DROPSELECTIONN = 2671
SCI_SETMAINSELECTION = 2574
SCI_GETMAINSELECTION = 2575
SCI_SETSELECTIONNCARET = 2576
SCI_GETSELECTIONNCARET = 2577
SCI_SETSELECTIONNANCHOR = 2578
SCI_GETSELECTIONNANCHOR = 2579
SCI_SETSELECTIONNCARETVIRTUALSPACE = 2580
SCI_GETSELECTIONNCARETVIRTUALSPACE = 2581
SCI_SETSELECTIONNANCHORVIRTUALSPACE = 2582
SCI_GETSELECTIONNANCHORVIRTUALSPACE = 2583
SCI_SETSELECTIONNSTART = 2584
SCI_GETSELECTIONNSTART = 2585
SCI_SETSELECTIONNEND = 2586
SCI_GETSELECTIONNEND = 2587
SCI_SETRECTANGULARSELECTIONCARET = 2588
SCI_GETRECTANGULARSELECTIONCARET = 2589
SCI_SETRECTANGULARSELECTIONANCHOR = 2590
SCI_GETRECTANGULARSELECTIONANCHOR = 2591
SCI_SETRECTANGULARSELECTIONCARETVIRTUALSPACE = 2592
SCI_GETRECTANGULARSELECTIONCARETVIRTUALSPACE = 2593
SCI_SETRECTANGULARSELECTIONANCHORVIRTUALSPACE = 2594
SCI_GETRECTANGULARSELECTIONANCHORVIRTUALSPACE = 2595
SCVS_NONE = 0
SCVS_RECTANGULARSELECTION = 1
SCVS_USERACCESSIBLE = 2
SCI_SETVIRTUALSPACEOPTIONS = 2596
SCI_GETVIRTUALSPACEOPTIONS = 2597
SCI_SETRECTANGULARSELECTIONMODIFIER = 2598
SCI_GETRECTANGULARSELECTIONMODIFIER = 2599
SCI_SETADDITIONALSELFORE = 2600
SCI_SETADDITIONALSELBACK = 2601
SCI_SETADDITIONALSELALPHA = 2602
SCI_GETADDITIONALSELALPHA = 2603
SCI_SETADDITIONALCARETFORE = 2604
SCI_GETADDITIONALCARETFORE = 2605
SCI_ROTATESELECTION = 2606
SCI_SWAPMAINANCHORCARET = 2607
SCI_CHANGELEXERSTATE = 2617
SCI_CONTRACTEDFOLDNEXT = 2618
SCI_VERTICALCENTRECARET = 2619
SCI_MOVESELECTEDLINESUP = 2620
SCI_MOVESELECTEDLINESDOWN = 2621
SCI_SETIDENTIFIER = 2622
SCI_GETIDENTIFIER = 2623
SCI_RGBAIMAGESETWIDTH = 2624
SCI_RGBAIMAGESETHEIGHT = 2625
SCI_RGBAIMAGESETSCALE = 2651
SCI_MARKERDEFINERGBAIMAGE = 2626
SCI_REGISTERRGBAIMAGE = 2627
SCI_SCROLLTOSTART = 2628
SCI_SCROLLTOEND = 2629
SC_TECHNOLOGY_DEFAULT = 0
SC_TECHNOLOGY_DIRECTWRITE = 1
SCI_SETTECHNOLOGY = 2630
SCI_GETTECHNOLOGY = 2631
SCI_CREATELOADER = 2632
SCI_FINDINDICATORSHOW = 2640
SCI_FINDINDICATORFLASH = 2641
SCI_FINDINDICATORHIDE = 2642
SCI_VCHOMEDISPLAY = 2652
SCI_VCHOMEDISPLAYEXTEND = 2653
SCI_GETCARETLINEVISIBLEALWAYS = 2654
SCI_SETCARETLINEVISIBLEALWAYS = 2655
SC_LINE_END_TYPE_DEFAULT = 0
SC_LINE_END_TYPE_UNICODE = 1
SCI_SETLINEENDTYPESALLOWED = 2656
SCI_GETLINEENDTYPESALLOWED = 2657
SCI_GETLINEENDTYPESACTIVE = 2658
SCI_SETREPRESENTATION = 2665
SCI_GETREPRESENTATION = 2666
SCI_CLEARREPRESENTATION = 2667
SCI_STARTRECORD = 3001
SCI_STOPRECORD = 3002
SCI_SETLEXER = 4001
SCI_GETLEXER = 4002
SCI_COLOURISE = 4003
SCI_SETPROPERTY = 4004
KEYWORDSET_MAX = 8
SCI_SETKEYWORDS = 4005
SCI_SETLEXERLANGUAGE = 4006
SCI_LOADLEXERLIBRARY = 4007
SCI_GETPROPERTY = 4008
SCI_GETPROPERTYEXPANDED = 4009
SCI_GETPROPERTYINT = 4010
SCI_GETSTYLEBITSNEEDED = 4011
SCI_GETLEXERLANGUAGE = 4012
SCI_PRIVATELEXERCALL = 4013
SCI_PROPERTYNAMES = 4014
SC_TYPE_BOOLEAN = 0
SC_TYPE_INTEGER = 1
SC_TYPE_STRING = 2
SCI_PROPERTYTYPE = 4015
SCI_DESCRIBEPROPERTY = 4016
SCI_DESCRIBEKEYWORDSETS = 4017
SCI_GETLINEENDTYPESSUPPORTED = 4018
SCI_ALLOCATESUBSTYLES = 4020
SCI_GETSUBSTYLESSTART = 4021
SCI_GETSUBSTYLESLENGTH = 4022
SCI_GETSTYLEFROMSUBSTYLE = 4027
SCI_GETPRIMARYSTYLEFROMSTYLE = 4028
SCI_FREESUBSTYLES = 4023
SCI_SETIDENTIFIERS = 4024
SCI_DISTANCETOSECONDARYSTYLES = 4025
SCI_GETSUBSTYLEBASES = 4026
SC_MOD_INSERTTEXT = 0x1
SC_MOD_DELETETEXT = 0x2
SC_MOD_CHANGESTYLE = 0x4
SC_MOD_CHANGEFOLD = 0x8
SC_PERFORMED_USER = 0x10
SC_PERFORMED_UNDO = 0x20
SC_PERFORMED_REDO = 0x40
SC_MULTISTEPUNDOREDO = 0x80
SC_LASTSTEPINUNDOREDO = 0x100
SC_MOD_CHANGEMARKER = 0x200
SC_MOD_BEFOREINSERT = 0x400
SC_MOD_BEFOREDELETE = 0x800
SC_MULTILINEUNDOREDO = 0x1000
SC_STARTACTION = 0x2000
SC_MOD_CHANGEINDICATOR = 0x4000
SC_MOD_CHANGELINESTATE = 0x8000
SC_MOD_CHANGEMARGIN = 0x10000
SC_MOD_CHANGEANNOTATION = 0x20000
SC_MOD_CONTAINER = 0x40000
SC_MOD_LEXERSTATE = 0x80000
SC_MODEVENTMASKALL = 0xFFFFF
SC_UPDATE_CONTENT = 0x1
SC_UPDATE_SELECTION = 0x2
SC_UPDATE_V_SCROLL = 0x4
SC_UPDATE_H_SCROLL = 0x8
SCK_DOWN = 300
SCK_UP = 301
SCK_LEFT = 302
SCK_RIGHT = 303
SCK_HOME = 304
SCK_END = 305
SCK_PRIOR = 306
SCK_NEXT = 307
SCK_DELETE = 308
SCK_INSERT = 309
SCK_ESCAPE = 7
SCK_BACK = 8
SCK_TAB = 9
SCK_RETURN = 13
SCK_ADD = 310
SCK_SUBTRACT = 311
SCK_DIVIDE = 312
SCK_WIN = 313
SCK_RWIN = 314
SCK_MENU = 315
SCMOD_NORM = 0
SCMOD_SHIFT = 1
SCMOD_CTRL = 2
SCMOD_ALT = 4
SCMOD_SUPER = 8
SCMOD_META = 16
SCLEX_CONTAINER = 0
SCLEX_NULL = 1
SCLEX_PYTHON = 2
SCLEX_CPP = 3
SCLEX_HTML = 4
SCLEX_XML = 5
SCLEX_PERL = 6
SCLEX_SQL = 7
SCLEX_VB = 8
SCLEX_PROPERTIES = 9
SCLEX_ERRORLIST = 10
SCLEX_MAKEFILE = 11
SCLEX_BATCH = 12
SCLEX_XCODE = 13
SCLEX_LATEX = 14
SCLEX_LUA = 15
SCLEX_DIFF = 16
SCLEX_CONF = 17
SCLEX_PASCAL = 18
SCLEX_AVE = 19
SCLEX_ADA = 20
SCLEX_LISP = 21
SCLEX_RUBY = 22
SCLEX_EIFFEL = 23
SCLEX_EIFFELKW = 24
SCLEX_TCL = 25
SCLEX_NNCRONTAB = 26
SCLEX_BULLANT = 27
SCLEX_VBSCRIPT = 28
SCLEX_BAAN = 31
SCLEX_MATLAB = 32
SCLEX_SCRIPTOL = 33
SCLEX_ASM = 34
SCLEX_CPPNOCASE = 35
SCLEX_FORTRAN = 36
SCLEX_F77 = 37
SCLEX_CSS = 38
SCLEX_POV = 39
SCLEX_LOUT = 40
SCLEX_ESCRIPT = 41
SCLEX_PS = 42
SCLEX_NSIS = 43
SCLEX_MMIXAL = 44
SCLEX_CLW = 45
SCLEX_CLWNOCASE = 46
SCLEX_LOT = 47
SCLEX_YAML = 48
SCLEX_TEX = 49
SCLEX_METAPOST = 50
SCLEX_POWERBASIC = 51
SCLEX_FORTH = 52
SCLEX_ERLANG = 53
SCLEX_OCTAVE = 54
SCLEX_MSSQL = 55
SCLEX_VERILOG = 56
SCLEX_KIX = 57
SCLEX_GUI4CLI = 58
SCLEX_SPECMAN = 59
SCLEX_AU3 = 60
SCLEX_APDL = 61
SCLEX_BASH = 62
SCLEX_ASN1 = 63
SCLEX_VHDL = 64
SCLEX_CAML = 65
SCLEX_BLITZBASIC = 66
SCLEX_PUREBASIC = 67
SCLEX_HASKELL = 68
SCLEX_PHPSCRIPT = 69
SCLEX_TADS3 = 70
SCLEX_REBOL = 71
SCLEX_SMALLTALK = 72
SCLEX_FLAGSHIP = 73
SCLEX_CSOUND = 74
SCLEX_FREEBASIC = 75
SCLEX_INNOSETUP = 76
SCLEX_OPAL = 77
SCLEX_SPICE = 78
SCLEX_D = 79
SCLEX_CMAKE = 80
SCLEX_GAP = 81
SCLEX_PLM = 82
SCLEX_PROGRESS = 83
SCLEX_ABAQUS = 84
SCLEX_ASYMPTOTE = 85
SCLEX_R = 86
SCLEX_MAGIK = 87
SCLEX_POWERSHELL = 88
SCLEX_MYSQL = 89
SCLEX_PO = 90
SCLEX_TAL = 91
SCLEX_COBOL = 92
SCLEX_TACL = 93
SCLEX_SORCUS = 94
SCLEX_POWERPRO = 95
SCLEX_NIMROD = 96
SCLEX_SML = 97
SCLEX_MARKDOWN = 98
SCLEX_TXT2TAGS = 99
SCLEX_A68K = 100
SCLEX_MODULA = 101
SCLEX_COFFEESCRIPT = 102
SCLEX_TCMD = 103
SCLEX_AVS = 104
SCLEX_ECL = 105
SCLEX_OSCRIPT = 106
SCLEX_VISUALPROLOG = 107
SCLEX_LITERATEHASKELL = 108
SCLEX_STTXT = 109
SCLEX_KVIRC = 110
SCLEX_RUST = 111
SCLEX_DMAP = 112
SCLEX_AS = 113
SCLEX_AUTOMATIC = 1000
SCN_STYLENEEDED = 2000
SCN_CHARADDED = 2001
SCN_SAVEPOINTREACHED = 2002
SCN_SAVEPOINTLEFT = 2003
SCN_MODIFYATTEMPTRO = 2004
SCN_KEY = 2005
SCN_DOUBLECLICK = 2006
SCN_UPDATEUI = 2007
SCN_MODIFIED = 2008
SCN_MACRORECORD = 2009
SCN_MARGINCLICK = 2010
SCN_NEEDSHOWN = 2011
SCN_PAINTED = 2013
SCN_USERLISTSELECTION = 2014
SCN_URIDROPPED = 2015
SCN_DWELLSTART = 2016
SCN_DWELLEND = 2017
SCN_ZOOM = 2018
SCN_HOTSPOTCLICK = 2019
SCN_HOTSPOTDOUBLECLICK = 2020
SCN_CALLTIPCLICK = 2021
SCN_AUTOCSELECTION = 2022
SCN_INDICATORCLICK = 2023
SCN_INDICATORRELEASE = 2024
SCN_AUTOCCANCELLED = 2025
SCN_AUTOCCHARDELETED = 2026
SCN_HOTSPOTRELEASECLICK = 2027
SCN_FOCUSIN = 2028
SCN_FOCUSOUT = 2029
SC_CP_DBCS = 1
SCI_GETUSEPALETTE = 2139
SCI_SETUSEPALETTE = 2039

-- Add text to the document at current position.
function sci:add_text(length, text)
  self:send(2001, length, string_ptr(text))
end

-- Add array of cells to document.
function sci:add_styled_text(length, c)
  self:send(2002, length, c)
end

-- Insert string at a position.
function sci:insert_text(pos, text)
  self:send(2003, pos, string_ptr(text))
end

-- Delete all text in the document.
function sci:clear_all()
  self:send(2004, 0, 0)
end

-- Delete a range of text in the document.
function sci:delete_range(pos, delete_length)
  self:send(2645, pos, delete_length)
end

-- Set all style bytes to 0, remove all folding information.
function sci:clear_document_style()
  self:send(2005, 0, 0)
end

-- Returns the number of bytes in the document.
function sci:get_length()
  return tonumber(self:send(2006, 0, 0))
end

-- Returns the character byte at the position.
function sci:get_char_at(pos)
  return tonumber(self:send(2007, pos, 0))
end

-- Returns the position of the caret.
function sci:get_current_pos()
  return tonumber(self:send(2008, 0, 0))
end

-- Returns the position of the opposite end of the selection to the caret.
function sci:get_anchor()
  return tonumber(self:send(2009, 0, 0))
end

-- Returns the style byte at the position.
function sci:get_style_at(pos)
  return tonumber(self:send(2010, pos, 0))
end

-- Redoes the next action on the undo history.
function sci:redo()
  self:send(2011, 0, 0)
end

-- Choose between collecting actions into the undo
-- history and discarding them.
function sci:set_undo_collection(collect_undo)
  self:send(2012, (collect_undo and 1 or 0), 0)
end

-- Select all the text in the document.
function sci:select_all()
  self:send(2013, 0, 0)
end

-- Remember the current position in the undo history as the position
-- at which the document was saved.
function sci:set_save_point()
  self:send(2014, 0, 0)
end

-- Retrieve a buffer of cells.
-- Returns the number of bytes in the buffer not including terminating NULs.
function sci:get_styled_text(start_pos, end_pos)
  return self:send_with_textrange(2015, start_pos, end_pos)
end

-- Are there any redoable actions in the undo history?
function sci:can_redo()
  return 0 ~= self:send(2016, 0, 0)
end

-- Retrieve the line number at which a particular marker is located.
function sci:marker_line_from_handle(handle)
  return tonumber(self:send(2017, handle, 0))
end

-- Delete a marker.
function sci:marker_delete_handle(handle)
  self:send(2018, handle, 0)
end

-- Is undo history being collected?
function sci:get_undo_collection()
  return 0 ~= self:send(2019, 0, 0)
end

-- Are white space characters currently visible?
-- Returns one of SCWS_* constants.
function sci:get_view_ws()
  return tonumber(self:send(2020, 0, 0))
end

-- Make white space characters invisible, always visible or visible outside indentation.
function sci:set_view_ws(view_ws)
  self:send(2021, view_ws, 0)
end

-- Find the position from a point within the window.
function sci:position_from_point(x, y)
  return tonumber(self:send(2022, x, y))
end

-- Find the position from a point within the window but return
-- INVALID_POSITION if not close to text.
function sci:position_from_point_close(x, y)
  return tonumber(self:send(2023, x, y))
end

-- Set caret to start of a line and ensure it is visible.
function sci:goto_line(line)
  self:send(2024, line, 0)
end

-- Set caret to a position and ensure it is visible.
function sci:goto_pos(pos)
  self:send(2025, pos, 0)
end

-- Set the selection anchor to a position. The anchor is the opposite
-- end of the selection from the caret.
function sci:set_anchor(pos_anchor)
  self:send(2026, pos_anchor, 0)
end

-- Retrieve the text of the line containing the caret.
-- Returns the index of the caret on the line.
function sci:get_cur_line(length)
  return self:send_with_stringresult(2027, length)
end

-- Retrieve the position of the last correctly styled character.
function sci:get_end_styled()
  return tonumber(self:send(2028, 0, 0))
end

-- Convert all line endings in the document to one mode.
function sci:convert_eols(eol_mode)
  self:send(2029, eol_mode, 0)
end

-- Retrieve the current end of line mode - one of CRLF, CR, or LF.
function sci:get_eolmode()
  return tonumber(self:send(2030, 0, 0))
end

-- Set the current end of line mode.
function sci:set_eolmode(eol_mode)
  self:send(2031, eol_mode, 0)
end

-- Set the current styling position to pos and the styling mask to mask.
-- The styling mask can be used to protect some bits in each styling byte from modification.
function sci:start_styling(pos, mask)
  self:send(2032, pos, mask)
end

-- Change style from current styling position for length characters to a style
-- and move the current styling position to after this newly styled segment.
function sci:set_styling(length, style)
  self:send(2033, length, style)
end

-- Is drawing done first into a buffer or direct to the screen?
function sci:get_buffered_draw()
  return 0 ~= self:send(2034, 0, 0)
end

-- If drawing is buffered then each line of text is drawn into a bitmap buffer
-- before drawing it to the screen to avoid flicker.
function sci:set_buffered_draw(buffered)
  self:send(2035, (buffered and 1 or 0), 0)
end

-- Change the visible size of a tab to be a multiple of the width of a space character.
function sci:set_tab_width(tab_width)
  self:send(2036, tab_width, 0)
end

-- Retrieve the visible size of a tab.
function sci:get_tab_width()
  return tonumber(self:send(2121, 0, 0))
end

-- Set the code page used to interpret the bytes of the document as characters.
-- The SC_CP_UTF8 value can be used to enter Unicode mode.
function sci:set_code_page(code_page)
  self:send(2037, code_page, 0)
end

-- Set the symbol used for a particular marker number.
function sci:marker_define(marker_number, marker_symbol)
  self:send(2040, marker_number, marker_symbol)
end

-- Set the foreground colour used for a particular marker number.
function sci:marker_set_fore(marker_number, fore)
  self:send(2041, marker_number, string_to_color(fore))
end

-- Set the background colour used for a particular marker number.
function sci:marker_set_back(marker_number, back)
  self:send(2042, marker_number, string_to_color(back))
end

-- Set the background colour used for a particular marker number when its folding block is selected.
function sci:marker_set_back_selected(marker_number, back)
  self:send(2292, marker_number, string_to_color(back))
end

-- Enable/disable highlight for current folding bloc (smallest one that contains the caret)
function sci:marker_enable_highlight(enabled)
  self:send(2293, (enabled and 1 or 0), 0)
end

-- Add a marker to a line, returning an ID which can be used to find or delete the marker.
function sci:marker_add(line, marker_number)
  return tonumber(self:send(2043, line, marker_number))
end

-- Delete a marker from a line.
function sci:marker_delete(line, marker_number)
  self:send(2044, line, marker_number)
end

-- Delete all markers with a particular number from all lines.
function sci:marker_delete_all(marker_number)
  self:send(2045, marker_number, 0)
end

-- Get a bit mask of all the markers set on a line.
function sci:marker_get(line)
  return tonumber(self:send(2046, line, 0))
end

-- Find the next line at or after lineStart that includes a marker in mask.
-- Return -1 when no more lines.
function sci:marker_next(line_start, marker_mask)
  return tonumber(self:send(2047, line_start, marker_mask))
end

-- Find the previous line before lineStart that includes a marker in mask.
function sci:marker_previous(line_start, marker_mask)
  return tonumber(self:send(2048, line_start, marker_mask))
end

-- Define a marker from a pixmap.
function sci:marker_define_pixmap(marker_number, pixmap)
  self:send(2049, marker_number, string_ptr(pixmap))
end

-- Add a set of markers to a line.
function sci:marker_add_set(line, set)
  self:send(2466, line, set)
end

-- Set the alpha used for a marker that is drawn in the text area, not the margin.
function sci:marker_set_alpha(marker_number, alpha)
  self:send(2476, marker_number, alpha)
end

-- Set a margin to be either numeric or symbolic.
function sci:set_margin_type_n(margin, margin_type)
  self:send(2240, margin, margin_type)
end

-- Retrieve the type of a margin.
function sci:get_margin_type_n(margin)
  return tonumber(self:send(2241, margin, 0))
end

-- Set the width of a margin to a width expressed in pixels.
function sci:set_margin_width_n(margin, pixel_width)
  self:send(2242, margin, pixel_width)
end

-- Retrieve the width of a margin in pixels.
function sci:get_margin_width_n(margin)
  return tonumber(self:send(2243, margin, 0))
end

-- Set a mask that determines which markers are displayed in a margin.
function sci:set_margin_mask_n(margin, mask)
  self:send(2244, margin, mask)
end

-- Retrieve the marker mask of a margin.
function sci:get_margin_mask_n(margin)
  return tonumber(self:send(2245, margin, 0))
end

-- Make a margin sensitive or insensitive to mouse clicks.
function sci:set_margin_sensitive_n(margin, sensitive)
  self:send(2246, margin, (sensitive and 1 or 0))
end

-- Retrieve the mouse click sensitivity of a margin.
function sci:get_margin_sensitive_n(margin)
  return 0 ~= self:send(2247, margin, 0)
end

-- Set the cursor shown when the mouse is inside a margin.
function sci:set_margin_cursor_n(margin, cursor)
  self:send(2248, margin, cursor)
end

-- Retrieve the cursor shown in a margin.
function sci:get_margin_cursor_n(margin)
  return tonumber(self:send(2249, margin, 0))
end

-- Clear all the styles and make equivalent to the global default style.
function sci:style_clear_all()
  self:send(2050, 0, 0)
end

-- Set the foreground colour of a style.
function sci:style_set_fore(style, fore)
  self:send(2051, style, string_to_color(fore))
end

-- Set the background colour of a style.
function sci:style_set_back(style, back)
  self:send(2052, style, string_to_color(back))
end

-- Set a style to be bold or not.
function sci:style_set_bold(style, bold)
  self:send(2053, style, (bold and 1 or 0))
end

-- Set a style to be italic or not.
function sci:style_set_italic(style, italic)
  self:send(2054, style, (italic and 1 or 0))
end

-- Set the size of characters of a style.
function sci:style_set_size(style, size_points)
  self:send(2055, style, size_points)
end

-- Set the font of a style.
function sci:style_set_font(style, font_name)
  self:send(2056, style, string_ptr(font_name))
end

-- Set a style to have its end of line filled or not.
function sci:style_set_eolfilled(style, filled)
  self:send(2057, style, (filled and 1 or 0))
end

-- Reset the default style to its state at startup
function sci:style_reset_default()
  self:send(2058, 0, 0)
end

-- Set a style to be underlined or not.
function sci:style_set_underline(style, underline)
  self:send(2059, style, (underline and 1 or 0))
end

-- Get the foreground colour of a style.
function sci:style_get_fore(style)
  return color_to_string(self:send(2481, style, 0))
end

-- Get the background colour of a style.
function sci:style_get_back(style)
  return color_to_string(self:send(2482, style, 0))
end

-- Get is a style bold or not.
function sci:style_get_bold(style)
  return 0 ~= self:send(2483, style, 0)
end

-- Get is a style italic or not.
function sci:style_get_italic(style)
  return 0 ~= self:send(2484, style, 0)
end

-- Get the size of characters of a style.
function sci:style_get_size(style)
  return tonumber(self:send(2485, style, 0))
end

-- Get the font of a style.
-- Returns the length of the fontName
function sci:style_get_font(style)
  return self:send_with_stringresult(2486, style)
end

-- Get is a style to have its end of line filled or not.
function sci:style_get_eolfilled(style)
  return 0 ~= self:send(2487, style, 0)
end

-- Get is a style underlined or not.
function sci:style_get_underline(style)
  return 0 ~= self:send(2488, style, 0)
end

-- Get is a style mixed case, or to force upper or lower case.
function sci:style_get_case(style)
  return tonumber(self:send(2489, style, 0))
end

-- Get the character get of the font in a style.
function sci:style_get_character_set(style)
  return tonumber(self:send(2490, style, 0))
end

-- Get is a style visible or not.
function sci:style_get_visible(style)
  return 0 ~= self:send(2491, style, 0)
end

-- Get is a style changeable or not (read only).
-- Experimental feature, currently buggy.
function sci:style_get_changeable(style)
  return 0 ~= self:send(2492, style, 0)
end

-- Get is a style a hotspot or not.
function sci:style_get_hot_spot(style)
  return 0 ~= self:send(2493, style, 0)
end

-- Set a style to be mixed case, or to force upper or lower case.
function sci:style_set_case(style, case_force)
  self:send(2060, style, case_force)
end

-- Set the size of characters of a style. Size is in points multiplied by 100.
function sci:style_set_size_fractional(style, case_force)
  self:send(2061, style, case_force)
end

-- Get the size of characters of a style in points multiplied by 100
function sci:style_get_size_fractional(style)
  return tonumber(self:send(2062, style, 0))
end

-- Set the weight of characters of a style.
function sci:style_set_weight(style, weight)
  self:send(2063, style, weight)
end

-- Get the weight of characters of a style.
function sci:style_get_weight(style)
  return tonumber(self:send(2064, style, 0))
end

-- Set the character set of the font in a style.
function sci:style_set_character_set(style, character_set)
  self:send(2066, style, character_set)
end

-- Set a style to be a hotspot or not.
function sci:style_set_hot_spot(style, hotspot)
  self:send(2409, style, (hotspot and 1 or 0))
end

-- Set the foreground colour of the main and additional selections and whether to use this setting.
function sci:set_sel_fore(use_setting, fore)
  self:send(2067, (use_setting and 1 or 0), string_to_color(fore))
end

-- Set the background colour of the main and additional selections and whether to use this setting.
function sci:set_sel_back(use_setting, back)
  self:send(2068, (use_setting and 1 or 0), string_to_color(back))
end

-- Get the alpha of the selection.
function sci:get_sel_alpha()
  return tonumber(self:send(2477, 0, 0))
end

-- Set the alpha of the selection.
function sci:set_sel_alpha(alpha)
  self:send(2478, alpha, 0)
end

-- Is the selection end of line filled?
function sci:get_sel_eolfilled()
  return 0 ~= self:send(2479, 0, 0)
end

-- Set the selection to have its end of line filled or not.
function sci:set_sel_eolfilled(filled)
  self:send(2480, (filled and 1 or 0), 0)
end

-- Set the foreground colour of the caret.
function sci:set_caret_fore(fore)
  self:send(2069, string_to_color(fore), 0)
end

-- When key+modifier combination km is pressed perform msg.
function sci:assign_cmd_key(km, msg)
  self:send(2070, km, msg)
end

-- When key+modifier combination km is pressed do nothing.
function sci:clear_cmd_key(km)
  self:send(2071, km, 0)
end

-- Drop all key mappings.
function sci:clear_all_cmd_keys()
  self:send(2072, 0, 0)
end

-- Set the styles for a segment of the document.
function sci:set_styling_ex(length, styles)
  self:send(2073, length, string_ptr(styles))
end

-- Set a style to be visible or not.
function sci:style_set_visible(style, visible)
  self:send(2074, style, (visible and 1 or 0))
end

-- Get the time in milliseconds that the caret is on and off.
function sci:get_caret_period()
  return tonumber(self:send(2075, 0, 0))
end

-- Get the time in milliseconds that the caret is on and off. 0 = steady on.
function sci:set_caret_period(period_milliseconds)
  self:send(2076, period_milliseconds, 0)
end

-- Set the set of characters making up words for when moving or selecting by word.
-- First sets defaults like SetCharsDefault.
function sci:set_word_chars(characters)
  self:send(2077, 0, string_ptr(characters))
end

-- Get the set of characters making up words for when moving or selecting by word.
-- Retuns the number of characters
function sci:get_word_chars()
  return self:send_with_stringresult(2646)
end

-- Start a sequence of actions that is undone and redone as a unit.
-- May be nested.
function sci:begin_undo_action()
  self:send(2078, 0, 0)
end

-- End a sequence of actions that is undone and redone as a unit.
function sci:end_undo_action()
  self:send(2079, 0, 0)
end

-- Set an indicator to plain, squiggle or TT.
function sci:indic_set_style(indic, style)
  self:send(2080, indic, style)
end

-- Retrieve the style of an indicator.
function sci:indic_get_style(indic)
  return tonumber(self:send(2081, indic, 0))
end

-- Set the foreground colour of an indicator.
function sci:indic_set_fore(indic, fore)
  self:send(2082, indic, string_to_color(fore))
end

-- Retrieve the foreground colour of an indicator.
function sci:indic_get_fore(indic)
  return color_to_string(self:send(2083, indic, 0))
end

-- Set an indicator to draw under text or over(default).
function sci:indic_set_under(indic, under)
  self:send(2510, indic, (under and 1 or 0))
end

-- Retrieve whether indicator drawn under or over text.
function sci:indic_get_under(indic)
  return 0 ~= self:send(2511, indic, 0)
end

-- Set the foreground colour of all whitespace and whether to use this setting.
function sci:set_whitespace_fore(use_setting, fore)
  self:send(2084, (use_setting and 1 or 0), string_to_color(fore))
end

-- Set the background colour of all whitespace and whether to use this setting.
function sci:set_whitespace_back(use_setting, back)
  self:send(2085, (use_setting and 1 or 0), string_to_color(back))
end

-- Set the size of the dots used to mark space characters.
function sci:set_whitespace_size(size)
  self:send(2086, size, 0)
end

-- Get the size of the dots used to mark space characters.
function sci:get_whitespace_size()
  return tonumber(self:send(2087, 0, 0))
end

-- Divide each styling byte into lexical class bits (default: 5) and indicator
-- bits (default: 3). If a lexer requires more than 32 lexical states, then this
-- is used to expand the possible states.
function sci:set_style_bits(bits)
  self:send(2090, bits, 0)
end

-- Retrieve number of bits in style bytes used to hold the lexical state.
function sci:get_style_bits()
  return tonumber(self:send(2091, 0, 0))
end

-- Used to hold extra styling information for each line.
function sci:set_line_state(line, state)
  self:send(2092, line, state)
end

-- Retrieve the extra styling information for a line.
function sci:get_line_state(line)
  return tonumber(self:send(2093, line, 0))
end

-- Retrieve the last line number that has line state.
function sci:get_max_line_state()
  return tonumber(self:send(2094, 0, 0))
end

-- Is the background of the line containing the caret in a different colour?
function sci:get_caret_line_visible()
  return 0 ~= self:send(2095, 0, 0)
end

-- Display the background of the line containing the caret in a different colour.
function sci:set_caret_line_visible(show)
  self:send(2096, (show and 1 or 0), 0)
end

-- Get the colour of the background of the line containing the caret.
function sci:get_caret_line_back()
  return color_to_string(self:send(2097, 0, 0))
end

-- Set the colour of the background of the line containing the caret.
function sci:set_caret_line_back(back)
  self:send(2098, string_to_color(back), 0)
end

-- Set a style to be changeable or not (read only).
-- Experimental feature, currently buggy.
function sci:style_set_changeable(style, changeable)
  self:send(2099, style, (changeable and 1 or 0))
end

-- Display a auto-completion list.
-- The lenEntered parameter indicates how many characters before
-- the caret should be used to provide context.
function sci:auto_cshow(len_entered, item_list)
  self:send(2100, len_entered, string_ptr(item_list))
end

-- Remove the auto-completion list from the screen.
function sci:auto_ccancel()
  self:send(2101, 0, 0)
end

-- Is there an auto-completion list visible?
function sci:auto_cactive()
  return 0 ~= self:send(2102, 0, 0)
end

-- Retrieve the position of the caret when the auto-completion list was displayed.
function sci:auto_cpos_start()
  return tonumber(self:send(2103, 0, 0))
end

-- User has selected an item so remove the list and insert the selection.
function sci:auto_ccomplete()
  self:send(2104, 0, 0)
end

-- Define a set of character that when typed cancel the auto-completion list.
function sci:auto_cstops(character_set)
  self:send(2105, 0, string_ptr(character_set))
end

-- Change the separator character in the string setting up an auto-completion list.
-- Default is space but can be changed if items contain space.
function sci:auto_cset_separator(separator_character)
  self:send(2106, separator_character, 0)
end

-- Retrieve the auto-completion list separator character.
function sci:auto_cget_separator()
  return tonumber(self:send(2107, 0, 0))
end

-- Select the item in the auto-completion list that starts with a string.
function sci:auto_cselect(text)
  self:send(2108, 0, string_ptr(text))
end

-- Should the auto-completion list be cancelled if the user backspaces to a
-- position before where the box was created.
function sci:auto_cset_cancel_at_start(cancel)
  self:send(2110, (cancel and 1 or 0), 0)
end

-- Retrieve whether auto-completion cancelled by backspacing before start.
function sci:auto_cget_cancel_at_start()
  return 0 ~= self:send(2111, 0, 0)
end

-- Define a set of characters that when typed will cause the autocompletion to
-- choose the selected item.
function sci:auto_cset_fill_ups(character_set)
  self:send(2112, 0, string_ptr(character_set))
end

-- Should a single item auto-completion list automatically choose the item.
function sci:auto_cset_choose_single(choose_single)
  self:send(2113, (choose_single and 1 or 0), 0)
end

-- Retrieve whether a single item auto-completion list automatically choose the item.
function sci:auto_cget_choose_single()
  return 0 ~= self:send(2114, 0, 0)
end

-- Set whether case is significant when performing auto-completion searches.
function sci:auto_cset_ignore_case(ignore_case)
  self:send(2115, (ignore_case and 1 or 0), 0)
end

-- Retrieve state of ignore case flag.
function sci:auto_cget_ignore_case()
  return 0 ~= self:send(2116, 0, 0)
end

-- Display a list of strings and send notification when user chooses one.
function sci:user_list_show(list_type, item_list)
  self:send(2117, list_type, string_ptr(item_list))
end

-- Set whether or not autocompletion is hidden automatically when nothing matches.
function sci:auto_cset_auto_hide(auto_hide)
  self:send(2118, (auto_hide and 1 or 0), 0)
end

-- Retrieve whether or not autocompletion is hidden automatically when nothing matches.
function sci:auto_cget_auto_hide()
  return 0 ~= self:send(2119, 0, 0)
end

-- Set whether or not autocompletion deletes any word characters
-- after the inserted text upon completion.
function sci:auto_cset_drop_rest_of_word(drop_rest_of_word)
  self:send(2270, (drop_rest_of_word and 1 or 0), 0)
end

-- Retrieve whether or not autocompletion deletes any word characters
-- after the inserted text upon completion.
function sci:auto_cget_drop_rest_of_word()
  return 0 ~= self:send(2271, 0, 0)
end

-- Register an XPM image for use in autocompletion lists.
function sci:register_image(type, xpm_data)
  self:send(2405, type, string_ptr(xpm_data))
end

-- Clear all the registered XPM images.
function sci:clear_registered_images()
  self:send(2408, 0, 0)
end

-- Retrieve the auto-completion list type-separator character.
function sci:auto_cget_type_separator()
  return tonumber(self:send(2285, 0, 0))
end

-- Change the type-separator character in the string setting up an auto-completion list.
-- Default is '?' but can be changed if items contain '?'.
function sci:auto_cset_type_separator(separator_character)
  self:send(2286, separator_character, 0)
end

-- Set the maximum width, in characters, of auto-completion and user lists.
-- Set to 0 to autosize to fit longest item, which is the default.
function sci:auto_cset_max_width(character_count)
  self:send(2208, character_count, 0)
end

-- Get the maximum width, in characters, of auto-completion and user lists.
function sci:auto_cget_max_width()
  return tonumber(self:send(2209, 0, 0))
end

-- Set the maximum height, in rows, of auto-completion and user lists.
-- The default is 5 rows.
function sci:auto_cset_max_height(row_count)
  self:send(2210, row_count, 0)
end

-- Set the maximum height, in rows, of auto-completion and user lists.
function sci:auto_cget_max_height()
  return tonumber(self:send(2211, 0, 0))
end

-- Set the number of spaces used for one level of indentation.
function sci:set_indent(indent_size)
  self:send(2122, indent_size, 0)
end

-- Retrieve indentation size.
function sci:get_indent()
  return tonumber(self:send(2123, 0, 0))
end

-- Indentation will only use space characters if useTabs is false, otherwise
-- it will use a combination of tabs and spaces.
function sci:set_use_tabs(use_tabs)
  self:send(2124, (use_tabs and 1 or 0), 0)
end

-- Retrieve whether tabs will be used in indentation.
function sci:get_use_tabs()
  return 0 ~= self:send(2125, 0, 0)
end

-- Change the indentation of a line to a number of columns.
function sci:set_line_indentation(line, indent_size)
  self:send(2126, line, indent_size)
end

-- Retrieve the number of columns that a line is indented.
function sci:get_line_indentation(line)
  return tonumber(self:send(2127, line, 0))
end

-- Retrieve the position before the first non indentation character on a line.
function sci:get_line_indent_position(line)
  return tonumber(self:send(2128, line, 0))
end

-- Retrieve the column number of a position, taking tab width into account.
function sci:get_column(pos)
  return tonumber(self:send(2129, pos, 0))
end

-- Count characters between two positions.
function sci:count_characters(start_pos, end_pos)
  return tonumber(self:send(2633, start_pos, end_pos))
end

-- Show or hide the horizontal scroll bar.
-- Is the horizontal scroll bar visible?
function sci:set_hscroll_bar(show)
  self:send(2130, (show and 1 or 0), 0)
end

-- Show or hide the horizontal scroll bar.
-- Is the horizontal scroll bar visible?
function sci:get_hscroll_bar()
  return 0 ~= self:send(2131, 0, 0)
end

-- Show or hide indentation guides.
function sci:set_indentation_guides(indent_view)
  self:send(2132, indent_view, 0)
end

-- Are the indentation guides visible?
function sci:get_indentation_guides()
  return tonumber(self:send(2133, 0, 0))
end

-- Set the highlighted indentation guide column.
-- 0 = no highlighted guide.
function sci:set_highlight_guide(column)
  self:send(2134, column, 0)
end

-- Get the highlighted indentation guide column.
function sci:get_highlight_guide()
  return tonumber(self:send(2135, 0, 0))
end

-- Get the position after the last visible characters on a line.
function sci:get_line_end_position(line)
  return tonumber(self:send(2136, line, 0))
end

-- Get the code page used to interpret the bytes of the document as characters.
function sci:get_code_page()
  return tonumber(self:send(2137, 0, 0))
end

-- Get the foreground colour of the caret.
function sci:get_caret_fore()
  return color_to_string(self:send(2138, 0, 0))
end

-- In read-only mode?
function sci:get_read_only()
  return 0 ~= self:send(2140, 0, 0)
end

-- Sets the position of the caret.
function sci:set_current_pos(pos)
  self:send(2141, pos, 0)
end

-- Sets the position that starts the selection - this becomes the anchor.
function sci:set_selection_start(pos)
  self:send(2142, pos, 0)
end

-- Returns the position at the start of the selection.
function sci:get_selection_start()
  return tonumber(self:send(2143, 0, 0))
end

-- Sets the position that ends the selection - this becomes the currentPosition.
function sci:set_selection_end(pos)
  self:send(2144, pos, 0)
end

-- Returns the position at the end of the selection.
function sci:get_selection_end()
  return tonumber(self:send(2145, 0, 0))
end

-- Set caret to a position, while removing any existing selection.
function sci:set_empty_selection(pos)
  self:send(2556, pos, 0)
end

-- Sets the print magnification added to the point size of each style for printing.
function sci:set_print_magnification(magnification)
  self:send(2146, magnification, 0)
end

-- Returns the print magnification.
function sci:get_print_magnification()
  return tonumber(self:send(2147, 0, 0))
end

-- Modify colours when printing for clearer printed text.
function sci:set_print_colour_mode(mode)
  self:send(2148, mode, 0)
end

-- Returns the print colour mode.
function sci:get_print_colour_mode()
  return tonumber(self:send(2149, 0, 0))
end

-- Find some text in the document.
function sci:find_text(start_pos, end_pos, text)
  return self:send_with_findtext(2150, start_pos, end_pos, text)
end

-- On Windows, will draw the document into a display context such as a printer.
function sci:format_range(draw, fr)
  return tonumber(self:send(2151, (draw and 1 or 0), fr))
end

-- Retrieve the display line at the top of the display.
function sci:get_first_visible_line()
  return tonumber(self:send(2152, 0, 0))
end

-- Retrieve the contents of a line.
-- Returns the length of the line.
function sci:get_line(line)
  return self:send_with_stringresult(2153, line)
end

-- Returns the number of lines in the document. There is always at least one.
function sci:get_line_count()
  return tonumber(self:send(2154, 0, 0))
end

-- Sets the size in pixels of the left margin.
function sci:set_margin_left(pixel_width)
  self:send(2155, 0, pixel_width)
end

-- Returns the size in pixels of the left margin.
function sci:get_margin_left()
  return tonumber(self:send(2156, 0, 0))
end

-- Sets the size in pixels of the right margin.
function sci:set_margin_right(pixel_width)
  self:send(2157, 0, pixel_width)
end

-- Returns the size in pixels of the right margin.
function sci:get_margin_right()
  return tonumber(self:send(2158, 0, 0))
end

-- Is the document different from when it was last saved?
function sci:get_modify()
  return 0 ~= self:send(2159, 0, 0)
end

-- Select a range of text.
function sci:set_sel(start_pos, end_pos)
  self:send(2160, start_pos, end_pos)
end

-- Retrieve the selected text.
-- Return the length of the text.
function sci:get_sel_text()
  return self:send_with_stringresult(2161)
end

-- Retrieve a range of text.
-- Return the length of the text.
function sci:get_text_range(start_pos, end_pos)
  return self:send_with_textrange(2162, start_pos, end_pos)
end

-- Draw the selection in normal style or with selection highlighted.
function sci:hide_selection(normal)
  self:send(2163, (normal and 1 or 0), 0)
end

-- Retrieve the x value of the point in the window where a position is displayed.
function sci:point_xfrom_position(pos)
  return tonumber(self:send(2164, 0, pos))
end

-- Retrieve the y value of the point in the window where a position is displayed.
function sci:point_yfrom_position(pos)
  return tonumber(self:send(2165, 0, pos))
end

-- Retrieve the line containing a position.
function sci:line_from_position(pos)
  return tonumber(self:send(2166, pos, 0))
end

-- Retrieve the position at the start of a line.
function sci:position_from_line(line)
  return tonumber(self:send(2167, line, 0))
end

-- Scroll horizontally and vertically.
function sci:line_scroll(columns, lines)
  self:send(2168, columns, lines)
end

-- Ensure the caret is visible.
function sci:scroll_caret()
  self:send(2169, 0, 0)
end

-- Scroll the argument positions and the range between them into view giving
-- priority to the primary position then the secondary position.
-- This may be used to make a search match visible.
function sci:scroll_range(secondary, primary)
  self:send(2569, secondary, primary)
end

-- Replace the selected text with the argument text.
function sci:replace_sel(text)
  self:send(2170, 0, string_ptr(text))
end

-- Set to read only or read write.
function sci:set_read_only(read_only)
  self:send(2171, (read_only and 1 or 0), 0)
end

-- Null operation.
function sci:null()
  self:send(2172, 0, 0)
end

-- Will a paste succeed?
function sci:can_paste()
  return 0 ~= self:send(2173, 0, 0)
end

-- Are there any undoable actions in the undo history?
function sci:can_undo()
  return 0 ~= self:send(2174, 0, 0)
end

-- Delete the undo history.
function sci:empty_undo_buffer()
  self:send(2175, 0, 0)
end

-- Undo one action in the undo history.
function sci:undo()
  self:send(2176, 0, 0)
end

-- Cut the selection to the clipboard.
function sci:cut()
  self:send(2177, 0, 0)
end

-- Copy the selection to the clipboard.
function sci:copy()
  self:send(2178, 0, 0)
end

-- Paste the contents of the clipboard into the document replacing the selection.
function sci:paste()
  self:send(2179, 0, 0)
end

-- Clear the selection.
function sci:clear()
  self:send(2180, 0, 0)
end

-- Replace the contents of the document with the argument text.
function sci:set_text(text)
  self:send(2181, 0, string_ptr(text))
end

-- Retrieve all the text in the document.
-- Returns number of characters retrieved.
function sci:get_text(length)
  return self:send_with_stringresult(2182, length)
end

-- Retrieve the number of characters in the document.
function sci:get_text_length()
  return tonumber(self:send(2183, 0, 0))
end

-- Retrieve a pointer to a function that processes messages for this Scintilla.
function sci:get_direct_function()
  return tonumber(self:send(2184, 0, 0))
end

-- Retrieve a pointer value to use as the first argument when calling
-- the function returned by GetDirectFunction.
function sci:get_direct_pointer()
  return tonumber(self:send(2185, 0, 0))
end

-- Set to overtype (true) or insert mode.
function sci:set_overtype(overtype)
  self:send(2186, (overtype and 1 or 0), 0)
end

-- Returns true if overtype mode is active otherwise false is returned.
function sci:get_overtype()
  return 0 ~= self:send(2187, 0, 0)
end

-- Set the width of the insert mode caret.
function sci:set_caret_width(pixel_width)
  self:send(2188, pixel_width, 0)
end

-- Returns the width of the insert mode caret.
function sci:get_caret_width()
  return tonumber(self:send(2189, 0, 0))
end

-- Sets the position that starts the target which is used for updating the
-- document without affecting the scroll position.
function sci:set_target_start(pos)
  self:send(2190, pos, 0)
end

-- Get the position that starts the target.
function sci:get_target_start()
  return tonumber(self:send(2191, 0, 0))
end

-- Sets the position that ends the target which is used for updating the
-- document without affecting the scroll position.
function sci:set_target_end(pos)
  self:send(2192, pos, 0)
end

-- Get the position that ends the target.
function sci:get_target_end()
  return tonumber(self:send(2193, 0, 0))
end

-- Replace the target text with the argument text.
-- Text is counted so it can contain NULs.
-- Returns the length of the replacement text.
function sci:replace_target(length, text)
  return tonumber(self:send(2194, length, string_ptr(text)))
end

-- Replace the target text with the argument text after \d processing.
-- Text is counted so it can contain NULs.
-- Looks for \d where d is between 1 and 9 and replaces these with the strings
-- matched in the last search operation which were surrounded by \( and \).
-- Returns the length of the replacement text including any change
-- caused by processing the \d patterns.
function sci:replace_target_re(length, text)
  return tonumber(self:send(2195, length, string_ptr(text)))
end

-- Search for a counted string in the target and set the target to the found
-- range. Text is counted so it can contain NULs.
-- Returns length of range or -1 for failure in which case target is not moved.
function sci:search_in_target(length, text)
  return tonumber(self:send(2197, length, string_ptr(text)))
end

-- Set the search flags used by SearchInTarget.
function sci:set_search_flags(flags)
  self:send(2198, flags, 0)
end

-- Get the search flags used by SearchInTarget.
function sci:get_search_flags()
  return tonumber(self:send(2199, 0, 0))
end

-- Show a call tip containing a definition near position pos.
function sci:call_tip_show(pos, definition)
  self:send(2200, pos, string_ptr(definition))
end

-- Remove the call tip from the screen.
function sci:call_tip_cancel()
  self:send(2201, 0, 0)
end

-- Is there an active call tip?
function sci:call_tip_active()
  return 0 ~= self:send(2202, 0, 0)
end

-- Retrieve the position where the caret was before displaying the call tip.
function sci:call_tip_pos_start()
  return tonumber(self:send(2203, 0, 0))
end

-- Set the start position in order to change when backspacing removes the calltip.
function sci:call_tip_set_pos_start(pos_start)
  self:send(2214, pos_start, 0)
end

-- Highlight a segment of the definition.
function sci:call_tip_set_hlt(start_pos, end_pos)
  self:send(2204, start_pos, end_pos)
end

-- Set the background colour for the call tip.
function sci:call_tip_set_back(back)
  self:send(2205, string_to_color(back), 0)
end

-- Set the foreground colour for the call tip.
function sci:call_tip_set_fore(fore)
  self:send(2206, string_to_color(fore), 0)
end

-- Set the foreground colour for the highlighted part of the call tip.
function sci:call_tip_set_fore_hlt(fore)
  self:send(2207, string_to_color(fore), 0)
end

-- Enable use of STYLE_CALLTIP and set call tip tab size in pixels.
function sci:call_tip_use_style(tab_size)
  self:send(2212, tab_size, 0)
end

-- Set position of calltip, above or below text.
function sci:call_tip_set_position(above)
  self:send(2213, (above and 1 or 0), 0)
end

-- Find the display line of a document line taking hidden lines into account.
function sci:visible_from_doc_line(line)
  return tonumber(self:send(2220, line, 0))
end

-- Find the document line of a display line taking hidden lines into account.
function sci:doc_line_from_visible(line_display)
  return tonumber(self:send(2221, line_display, 0))
end

-- The number of display lines needed to wrap a document line
function sci:wrap_count(line)
  return tonumber(self:send(2235, line, 0))
end

-- Set the fold level of a line.
-- This encodes an integer level along with flags indicating whether the
-- line is a header and whether it is effectively white space.
function sci:set_fold_level(line, level)
  self:send(2222, line, level)
end

-- Retrieve the fold level of a line.
function sci:get_fold_level(line)
  return tonumber(self:send(2223, line, 0))
end

-- Find the last child line of a header line.
function sci:get_last_child(line, level)
  return tonumber(self:send(2224, line, level))
end

-- Find the parent line of a child line.
function sci:get_fold_parent(line)
  return tonumber(self:send(2225, line, 0))
end

-- Make a range of lines visible.
function sci:show_lines(line_start, line_end)
  self:send(2226, line_start, line_end)
end

-- Make a range of lines invisible.
function sci:hide_lines(line_start, line_end)
  self:send(2227, line_start, line_end)
end

-- Is a line visible?
function sci:get_line_visible(line)
  return 0 ~= self:send(2228, line, 0)
end

-- Are all lines visible?
function sci:get_all_lines_visible()
  return 0 ~= self:send(2236, 0, 0)
end

-- Show the children of a header line.
function sci:set_fold_expanded(line, expanded)
  self:send(2229, line, (expanded and 1 or 0))
end

-- Is a header line expanded?
function sci:get_fold_expanded(line)
  return 0 ~= self:send(2230, line, 0)
end

-- Switch a header line between expanded and contracted.
function sci:toggle_fold(line)
  self:send(2231, line, 0)
end

-- Expand or contract a fold header.
function sci:fold_line(line, action)
  self:send(2237, line, action)
end

-- Expand or contract a fold header and its children.
function sci:fold_children(line, action)
  self:send(2238, line, action)
end

-- Expand a fold header and all children. Use the level argument instead of the line's current level.
function sci:expand_children(line, level)
  self:send(2239, line, level)
end

-- Expand or contract all fold headers.
function sci:fold_all(action)
  self:send(2662, action, 0)
end

-- Ensure a particular line is visible by expanding any header line hiding it.
function sci:ensure_visible(line)
  self:send(2232, line, 0)
end

-- Set automatic folding behaviours.
function sci:set_automatic_fold(automatic_fold)
  self:send(2663, automatic_fold, 0)
end

-- Get automatic folding behaviours.
function sci:get_automatic_fold()
  return tonumber(self:send(2664, 0, 0))
end

-- Set some style options for folding.
function sci:set_fold_flags(flags)
  self:send(2233, flags, 0)
end

-- Ensure a particular line is visible by expanding any header line hiding it.
-- Use the currently set visibility policy to determine which range to display.
function sci:ensure_visible_enforce_policy(line)
  self:send(2234, line, 0)
end

-- Sets whether a tab pressed when caret is within indentation indents.
function sci:set_tab_indents(tab_indents)
  self:send(2260, (tab_indents and 1 or 0), 0)
end

-- Does a tab pressed when caret is within indentation indent?
function sci:get_tab_indents()
  return 0 ~= self:send(2261, 0, 0)
end

-- Sets whether a backspace pressed when caret is within indentation unindents.
function sci:set_back_space_un_indents(bs_un_indents)
  self:send(2262, (bs_un_indents and 1 or 0), 0)
end

-- Does a backspace pressed when caret is within indentation unindent?
function sci:get_back_space_un_indents()
  return 0 ~= self:send(2263, 0, 0)
end

-- Sets the time the mouse must sit still to generate a mouse dwell event.
function sci:set_mouse_dwell_time(period_milliseconds)
  self:send(2264, period_milliseconds, 0)
end

-- Retrieve the time the mouse must sit still to generate a mouse dwell event.
function sci:get_mouse_dwell_time()
  return tonumber(self:send(2265, 0, 0))
end

-- Get position of start of word.
function sci:word_start_position(pos, only_word_characters)
  return tonumber(self:send(2266, pos, (only_word_characters and 1 or 0)))
end

-- Get position of end of word.
function sci:word_end_position(pos, only_word_characters)
  return tonumber(self:send(2267, pos, (only_word_characters and 1 or 0)))
end

-- Sets whether text is word wrapped.
function sci:set_wrap_mode(mode)
  self:send(2268, mode, 0)
end

-- Retrieve whether text is word wrapped.
function sci:get_wrap_mode()
  return tonumber(self:send(2269, 0, 0))
end

-- Set the display mode of visual flags for wrapped lines.
function sci:set_wrap_visual_flags(wrap_visual_flags)
  self:send(2460, wrap_visual_flags, 0)
end

-- Retrive the display mode of visual flags for wrapped lines.
function sci:get_wrap_visual_flags()
  return tonumber(self:send(2461, 0, 0))
end

-- Set the location of visual flags for wrapped lines.
function sci:set_wrap_visual_flags_location(wrap_visual_flags_location)
  self:send(2462, wrap_visual_flags_location, 0)
end

-- Retrive the location of visual flags for wrapped lines.
function sci:get_wrap_visual_flags_location()
  return tonumber(self:send(2463, 0, 0))
end

-- Set the start indent for wrapped lines.
function sci:set_wrap_start_indent(indent)
  self:send(2464, indent, 0)
end

-- Retrive the start indent for wrapped lines.
function sci:get_wrap_start_indent()
  return tonumber(self:send(2465, 0, 0))
end

-- Sets how wrapped sublines are placed. Default is fixed.
function sci:set_wrap_indent_mode(mode)
  self:send(2472, mode, 0)
end

-- Retrieve how wrapped sublines are placed. Default is fixed.
function sci:get_wrap_indent_mode()
  return tonumber(self:send(2473, 0, 0))
end

-- Sets the degree of caching of layout information.
function sci:set_layout_cache(mode)
  self:send(2272, mode, 0)
end

-- Retrieve the degree of caching of layout information.
function sci:get_layout_cache()
  return tonumber(self:send(2273, 0, 0))
end

-- Sets the document width assumed for scrolling.
function sci:set_scroll_width(pixel_width)
  self:send(2274, pixel_width, 0)
end

-- Retrieve the document width assumed for scrolling.
function sci:get_scroll_width()
  return tonumber(self:send(2275, 0, 0))
end

-- Sets whether the maximum width line displayed is used to set scroll width.
function sci:set_scroll_width_tracking(tracking)
  self:send(2516, (tracking and 1 or 0), 0)
end

-- Retrieve whether the scroll width tracks wide lines.
function sci:get_scroll_width_tracking()
  return 0 ~= self:send(2517, 0, 0)
end

-- Measure the pixel width of some text in a particular style.
-- NUL terminated text argument.
-- Does not handle tab or control characters.
function sci:text_width(style, text)
  return tonumber(self:send(2276, style, string_ptr(text)))
end

-- Sets the scroll range so that maximum scroll position has
-- the last line at the bottom of the view (default).
-- Setting this to false allows scrolling one page below the last line.
function sci:set_end_at_last_line(end_at_last_line)
  self:send(2277, (end_at_last_line and 1 or 0), 0)
end

-- Retrieve whether the maximum scroll position has the last
-- line at the bottom of the view.
function sci:get_end_at_last_line()
  return 0 ~= self:send(2278, 0, 0)
end

-- Retrieve the height of a particular line of text in pixels.
function sci:text_height(line)
  return tonumber(self:send(2279, line, 0))
end

-- Show or hide the vertical scroll bar.
function sci:set_vscroll_bar(show)
  self:send(2280, (show and 1 or 0), 0)
end

-- Is the vertical scroll bar visible?
function sci:get_vscroll_bar()
  return 0 ~= self:send(2281, 0, 0)
end

-- Append a string to the end of the document without changing the selection.
function sci:append_text(length, text)
  self:send(2282, length, string_ptr(text))
end

-- Is drawing done in two phases with backgrounds drawn before faoregrounds?
function sci:get_two_phase_draw()
  return 0 ~= self:send(2283, 0, 0)
end

-- In twoPhaseDraw mode, drawing is performed in two phases, first the background
-- and then the foreground. This avoids chopping off characters that overlap the next run.
function sci:set_two_phase_draw(two_phase)
  self:send(2284, (two_phase and 1 or 0), 0)
end

-- Choose the quality level for text from the FontQuality enumeration.
function sci:set_font_quality(font_quality)
  self:send(2611, font_quality, 0)
end

-- Retrieve the quality level for text.
function sci:get_font_quality()
  return tonumber(self:send(2612, 0, 0))
end

-- Scroll so that a display line is at the top of the display.
function sci:set_first_visible_line(line_display)
  self:send(2613, line_display, 0)
end

-- Change the effect of pasting when there are multiple selections.
function sci:set_multi_paste(multi_paste)
  self:send(2614, multi_paste, 0)
end

-- Retrieve the effect of pasting when there are multiple selections..
function sci:get_multi_paste()
  return tonumber(self:send(2615, 0, 0))
end

-- Retrieve the value of a tag from a regular expression search.
function sci:get_tag(tag_number)
  return self:send_with_stringresult(2616, tag_number)
end

-- Make the target range start and end be the same as the selection range start and end.
function sci:target_from_selection()
  self:send(2287, 0, 0)
end

-- Join the lines in the target.
function sci:lines_join()
  self:send(2288, 0, 0)
end

-- Split the lines in the target into lines that are less wide than pixelWidth
-- where possible.
function sci:lines_split(pixel_width)
  self:send(2289, pixel_width, 0)
end

-- Set the colours used as a chequerboard pattern in the fold margin
function sci:set_fold_margin_colour(use_setting, back)
  self:send(2290, (use_setting and 1 or 0), string_to_color(back))
end

-- Set the colours used as a chequerboard pattern in the fold margin
function sci:set_fold_margin_hi_colour(use_setting, fore)
  self:send(2291, (use_setting and 1 or 0), string_to_color(fore))
end

-- # Start of key messages
-- Move caret down one line.
function sci:line_down()
  self:send(2300, 0, 0)
end

-- Move caret down one line extending selection to new caret position.
function sci:line_down_extend()
  self:send(2301, 0, 0)
end

-- Move caret up one line.
function sci:line_up()
  self:send(2302, 0, 0)
end

-- Move caret up one line extending selection to new caret position.
function sci:line_up_extend()
  self:send(2303, 0, 0)
end

-- Move caret left one character.
function sci:char_left()
  self:send(2304, 0, 0)
end

-- Move caret left one character extending selection to new caret position.
function sci:char_left_extend()
  self:send(2305, 0, 0)
end

-- Move caret right one character.
function sci:char_right()
  self:send(2306, 0, 0)
end

-- Move caret right one character extending selection to new caret position.
function sci:char_right_extend()
  self:send(2307, 0, 0)
end

-- Move caret left one word.
function sci:word_left()
  self:send(2308, 0, 0)
end

-- Move caret left one word extending selection to new caret position.
function sci:word_left_extend()
  self:send(2309, 0, 0)
end

-- Move caret right one word.
function sci:word_right()
  self:send(2310, 0, 0)
end

-- Move caret right one word extending selection to new caret position.
function sci:word_right_extend()
  self:send(2311, 0, 0)
end

-- Move caret to first position on line.
function sci:home()
  self:send(2312, 0, 0)
end

-- Move caret to first position on line extending selection to new caret position.
function sci:home_extend()
  self:send(2313, 0, 0)
end

-- Move caret to last position on line.
function sci:line_end()
  self:send(2314, 0, 0)
end

-- Move caret to last position on line extending selection to new caret position.
function sci:line_end_extend()
  self:send(2315, 0, 0)
end

-- Move caret to first position in document.
function sci:document_start()
  self:send(2316, 0, 0)
end

-- Move caret to first position in document extending selection to new caret position.
function sci:document_start_extend()
  self:send(2317, 0, 0)
end

-- Move caret to last position in document.
function sci:document_end()
  self:send(2318, 0, 0)
end

-- Move caret to last position in document extending selection to new caret position.
function sci:document_end_extend()
  self:send(2319, 0, 0)
end

-- Move caret one page up.
function sci:page_up()
  self:send(2320, 0, 0)
end

-- Move caret one page up extending selection to new caret position.
function sci:page_up_extend()
  self:send(2321, 0, 0)
end

-- Move caret one page down.
function sci:page_down()
  self:send(2322, 0, 0)
end

-- Move caret one page down extending selection to new caret position.
function sci:page_down_extend()
  self:send(2323, 0, 0)
end

-- Switch from insert to overtype mode or the reverse.
function sci:edit_toggle_overtype()
  self:send(2324, 0, 0)
end

-- Cancel any modes such as call tip or auto-completion list display.
function sci:cancel()
  self:send(2325, 0, 0)
end

-- Delete the selection or if no selection, the character before the caret.
function sci:delete_back()
  self:send(2326, 0, 0)
end

-- If selection is empty or all on one line replace the selection with a tab character.
-- If more than one line selected, indent the lines.
function sci:tab()
  self:send(2327, 0, 0)
end

-- Dedent the selected lines.
function sci:back_tab()
  self:send(2328, 0, 0)
end

-- Insert a new line, may use a CRLF, CR or LF depending on EOL mode.
function sci:new_line()
  self:send(2329, 0, 0)
end

-- Insert a Form Feed character.
function sci:form_feed()
  self:send(2330, 0, 0)
end

-- Move caret to before first visible character on line.
-- If already there move to first character on line.
function sci:vchome()
  self:send(2331, 0, 0)
end

-- Like VCHome but extending selection to new caret position.
function sci:vchome_extend()
  self:send(2332, 0, 0)
end

-- Magnify the displayed text by increasing the sizes by 1 point.
function sci:zoom_in()
  self:send(2333, 0, 0)
end

-- Make the displayed text smaller by decreasing the sizes by 1 point.
function sci:zoom_out()
  self:send(2334, 0, 0)
end

-- Delete the word to the left of the caret.
function sci:del_word_left()
  self:send(2335, 0, 0)
end

-- Delete the word to the right of the caret.
function sci:del_word_right()
  self:send(2336, 0, 0)
end

-- Delete the word to the right of the caret, but not the trailing non-word characters.
function sci:del_word_right_end()
  self:send(2518, 0, 0)
end

-- Cut the line containing the caret.
function sci:line_cut()
  self:send(2337, 0, 0)
end

-- Delete the line containing the caret.
function sci:line_delete()
  self:send(2338, 0, 0)
end

-- Switch the current line with the previous.
function sci:line_transpose()
  self:send(2339, 0, 0)
end

-- Duplicate the current line.
function sci:line_duplicate()
  self:send(2404, 0, 0)
end

-- Transform the selection to lower case.
function sci:lower_case()
  self:send(2340, 0, 0)
end

-- Transform the selection to upper case.
function sci:upper_case()
  self:send(2341, 0, 0)
end

-- Scroll the document down, keeping the caret visible.
function sci:line_scroll_down()
  self:send(2342, 0, 0)
end

-- Scroll the document up, keeping the caret visible.
function sci:line_scroll_up()
  self:send(2343, 0, 0)
end

-- Delete the selection or if no selection, the character before the caret.
-- Will not delete the character before at the start of a line.
function sci:delete_back_not_line()
  self:send(2344, 0, 0)
end

-- Move caret to first position on display line.
function sci:home_display()
  self:send(2345, 0, 0)
end

-- Move caret to first position on display line extending selection to
-- new caret position.
function sci:home_display_extend()
  self:send(2346, 0, 0)
end

-- Move caret to last position on display line.
function sci:line_end_display()
  self:send(2347, 0, 0)
end

-- Move caret to last position on display line extending selection to new
-- caret position.
function sci:line_end_display_extend()
  self:send(2348, 0, 0)
end

function sci:home_wrap()
  self:send(2349, 0, 0)
end

function sci:home_wrap_extend()
  self:send(2450, 0, 0)
end

function sci:line_end_wrap()
  self:send(2451, 0, 0)
end

function sci:line_end_wrap_extend()
  self:send(2452, 0, 0)
end

function sci:vchome_wrap()
  self:send(2453, 0, 0)
end

function sci:vchome_wrap_extend()
  self:send(2454, 0, 0)
end

-- Copy the line containing the caret.
function sci:line_copy()
  self:send(2455, 0, 0)
end

-- Move the caret inside current view if it's not there already.
function sci:move_caret_inside_view()
  self:send(2401, 0, 0)
end

-- How many characters are on a line, including end of line characters?
function sci:line_length(line)
  return tonumber(self:send(2350, line, 0))
end

-- Highlight the characters at two positions.
function sci:brace_highlight(pos1, pos2)
  self:send(2351, pos1, pos2)
end

-- Use specified indicator to highlight matching braces instead of changing their style.
function sci:brace_highlight_indicator(use_brace_highlight_indicator, indicator)
  self:send(2498, (use_brace_highlight_indicator and 1 or 0), indicator)
end

-- Highlight the character at a position indicating there is no matching brace.
function sci:brace_bad_light(pos)
  self:send(2352, pos, 0)
end

-- Use specified indicator to highlight non matching brace instead of changing its style.
function sci:brace_bad_light_indicator(use_brace_bad_light_indicator, indicator)
  self:send(2499, (use_brace_bad_light_indicator and 1 or 0), indicator)
end

-- Find the position of a matching brace or INVALID_POSITION if no match.
function sci:brace_match(pos)
  return tonumber(self:send(2353, pos, 0))
end

-- Are the end of line characters visible?
function sci:get_view_eol()
  return 0 ~= self:send(2355, 0, 0)
end

-- Make the end of line characters visible or invisible.
function sci:set_view_eol(visible)
  self:send(2356, (visible and 1 or 0), 0)
end

-- Retrieve a pointer to the document object.
function sci:get_doc_pointer()
  return tonumber(self:send(2357, 0, 0))
end

-- Set which document modification events are sent to the container.
function sci:set_mod_event_mask(mask)
  self:send(2359, mask, 0)
end

-- Retrieve the column number which text should be kept within.
function sci:get_edge_column()
  return tonumber(self:send(2360, 0, 0))
end

-- Set the column number of the edge.
-- If text goes past the edge then it is highlighted.
function sci:set_edge_column(column)
  self:send(2361, column, 0)
end

-- Retrieve the edge highlight mode.
function sci:get_edge_mode()
  return tonumber(self:send(2362, 0, 0))
end

-- The edge may be displayed by a line (EDGE_LINE) or by highlighting text that
-- goes beyond it (EDGE_BACKGROUND) or not displayed at all (EDGE_NONE).
function sci:set_edge_mode(mode)
  self:send(2363, mode, 0)
end

-- Retrieve the colour used in edge indication.
function sci:get_edge_colour()
  return color_to_string(self:send(2364, 0, 0))
end

-- Change the colour used in edge indication.
function sci:set_edge_colour(edge_colour)
  self:send(2365, string_to_color(edge_colour), 0)
end

-- Sets the current caret position to be the search anchor.
function sci:search_anchor()
  self:send(2366, 0, 0)
end

-- Find some text starting at the search anchor.
-- Does not ensure the selection is visible.
function sci:search_next(flags, text)
  return tonumber(self:send(2367, flags, string_ptr(text)))
end

-- Find some text starting at the search anchor and moving backwards.
-- Does not ensure the selection is visible.
function sci:search_prev(flags, text)
  return tonumber(self:send(2368, flags, string_ptr(text)))
end

-- Retrieves the number of lines completely visible.
function sci:lines_on_screen()
  return tonumber(self:send(2370, 0, 0))
end

-- Set whether a pop up menu is displayed automatically when the user presses
-- the wrong mouse button.
function sci:use_pop_up(allow_pop_up)
  self:send(2371, (allow_pop_up and 1 or 0), 0)
end

-- Is the selection rectangular? The alternative is the more common stream selection.
function sci:selection_is_rectangle()
  return 0 ~= self:send(2372, 0, 0)
end

-- Set the zoom level. This number of points is added to the size of all fonts.
-- It may be positive to magnify or negative to reduce.
-- Retrieve the zoom level.
function sci:set_zoom(zoom)
  self:send(2373, zoom, 0)
end

-- Set the zoom level. This number of points is added to the size of all fonts.
-- It may be positive to magnify or negative to reduce.
-- Retrieve the zoom level.
function sci:get_zoom()
  return tonumber(self:send(2374, 0, 0))
end

-- Create a new document object.
-- Starts with reference count of 1 and not selected into editor.
-- Extend life of document.
-- Release a reference to the document, deleting document if it fades to black.
function sci:create_document()
  return tonumber(self:send(2375, 0, 0))
end

-- Create a new document object.
-- Starts with reference count of 1 and not selected into editor.
-- Extend life of document.
-- Release a reference to the document, deleting document if it fades to black.
function sci:add_ref_document(doc)
  self:send(2376, 0, doc)
end

-- Create a new document object.
-- Starts with reference count of 1 and not selected into editor.
-- Extend life of document.
-- Release a reference to the document, deleting document if it fades to black.
function sci:release_document(doc)
  self:send(2377, 0, doc)
end

-- Get which document modification events are sent to the container.
function sci:get_mod_event_mask()
  return tonumber(self:send(2378, 0, 0))
end

-- Change internal focus flag.
-- Get internal focus flag.
function sci:set_focus(focus)
  self:send(2380, (focus and 1 or 0), 0)
end

-- Change internal focus flag.
-- Get internal focus flag.
function sci:get_focus()
  return 0 ~= self:send(2381, 0, 0)
end

-- Change error status - 0 = OK.
-- Get error status.
function sci:set_status(status_code)
  self:send(2382, status_code, 0)
end

-- Change error status - 0 = OK.
-- Get error status.
function sci:get_status()
  return tonumber(self:send(2383, 0, 0))
end

-- Set whether the mouse is captured when its button is pressed.
-- Get whether mouse gets captured.
function sci:set_mouse_down_captures(captures)
  self:send(2384, (captures and 1 or 0), 0)
end

-- Set whether the mouse is captured when its button is pressed.
-- Get whether mouse gets captured.
function sci:get_mouse_down_captures()
  return 0 ~= self:send(2385, 0, 0)
end

-- Sets the cursor to one of the SC_CURSOR* values.
-- Get cursor type.
function sci:set_cursor(cursor_type)
  self:send(2386, cursor_type, 0)
end

-- Sets the cursor to one of the SC_CURSOR* values.
-- Get cursor type.
function sci:get_cursor()
  return tonumber(self:send(2387, 0, 0))
end

-- Change the way control characters are displayed:
-- If symbol is < 32, keep the drawn way, else, use the given character.
-- Get the way control characters are displayed.
function sci:set_control_char_symbol(symbol)
  self:send(2388, symbol, 0)
end

-- Change the way control characters are displayed:
-- If symbol is < 32, keep the drawn way, else, use the given character.
-- Get the way control characters are displayed.
function sci:get_control_char_symbol()
  return tonumber(self:send(2389, 0, 0))
end

-- Move to the previous change in capitalisation.
-- Move to the previous change in capitalisation extending selection
-- to new caret position.
-- Move to the change next in capitalisation.
-- Move to the next change in capitalisation extending selection
-- to new caret position.
function sci:word_part_left()
  self:send(2390, 0, 0)
end

-- Move to the previous change in capitalisation.
-- Move to the previous change in capitalisation extending selection
-- to new caret position.
-- Move to the change next in capitalisation.
-- Move to the next change in capitalisation extending selection
-- to new caret position.
function sci:word_part_left_extend()
  self:send(2391, 0, 0)
end

-- Move to the previous change in capitalisation.
-- Move to the previous change in capitalisation extending selection
-- to new caret position.
-- Move to the change next in capitalisation.
-- Move to the next change in capitalisation extending selection
-- to new caret position.
function sci:word_part_right()
  self:send(2392, 0, 0)
end

-- Move to the previous change in capitalisation.
-- Move to the previous change in capitalisation extending selection
-- to new caret position.
-- Move to the change next in capitalisation.
-- Move to the next change in capitalisation extending selection
-- to new caret position.
function sci:word_part_right_extend()
  self:send(2393, 0, 0)
end

-- Constants for use with SetVisiblePolicy, similar to SetCaretPolicy.
-- Set the way the display area is determined when a particular line
-- is to be moved to by Find, FindNext, GotoLine, etc.
function sci:set_visible_policy(visible_policy, visible_slop)
  self:send(2394, visible_policy, visible_slop)
end

-- Delete back from the current position to the start of the line.
function sci:del_line_left()
  self:send(2395, 0, 0)
end

-- Delete forwards from the current position to the end of the line.
function sci:del_line_right()
  self:send(2396, 0, 0)
end

-- Get and Set the xOffset (ie, horizontal scroll position).
function sci:set_xoffset(new_offset)
  self:send(2397, new_offset, 0)
end

-- Get and Set the xOffset (ie, horizontal scroll position).
function sci:get_xoffset()
  return tonumber(self:send(2398, 0, 0))
end

-- Set the last x chosen value to be the caret x position.
function sci:choose_caret_x()
  self:send(2399, 0, 0)
end

-- Set the focus to this Scintilla widget.
function sci:grab_focus()
  self:send(2400, 0, 0)
end

-- Set the way the caret is kept visible when going sideways.
-- The exclusion zone is given in pixels.
function sci:set_xcaret_policy(caret_policy, caret_slop)
  self:send(2402, caret_policy, caret_slop)
end

-- Set the way the line the caret is on is kept visible.
-- The exclusion zone is given in lines.
function sci:set_ycaret_policy(caret_policy, caret_slop)
  self:send(2403, caret_policy, caret_slop)
end

-- Set printing to line wrapped (SC_WRAP_WORD) or not line wrapped (SC_WRAP_NONE).
function sci:set_print_wrap_mode(mode)
  self:send(2406, mode, 0)
end

-- Is printing line wrapped?
function sci:get_print_wrap_mode()
  return tonumber(self:send(2407, 0, 0))
end

-- Set a fore colour for active hotspots.
function sci:set_hotspot_active_fore(use_setting, fore)
  self:send(2410, (use_setting and 1 or 0), string_to_color(fore))
end

-- Get the fore colour for active hotspots.
function sci:get_hotspot_active_fore()
  return color_to_string(self:send(2494, 0, 0))
end

-- Set a back colour for active hotspots.
function sci:set_hotspot_active_back(use_setting, back)
  self:send(2411, (use_setting and 1 or 0), string_to_color(back))
end

-- Get the back colour for active hotspots.
function sci:get_hotspot_active_back()
  return color_to_string(self:send(2495, 0, 0))
end

-- Enable / Disable underlining active hotspots.
function sci:set_hotspot_active_underline(underline)
  self:send(2412, (underline and 1 or 0), 0)
end

-- Get whether underlining for active hotspots.
function sci:get_hotspot_active_underline()
  return 0 ~= self:send(2496, 0, 0)
end

-- Limit hotspots to single line so hotspots on two lines don't merge.
function sci:set_hotspot_single_line(single_line)
  self:send(2421, (single_line and 1 or 0), 0)
end

-- Get the HotspotSingleLine property
function sci:get_hotspot_single_line()
  return 0 ~= self:send(2497, 0, 0)
end

-- Move caret between paragraphs (delimited by empty lines).
function sci:para_down()
  self:send(2413, 0, 0)
end

-- Move caret between paragraphs (delimited by empty lines).
function sci:para_down_extend()
  self:send(2414, 0, 0)
end

-- Move caret between paragraphs (delimited by empty lines).
function sci:para_up()
  self:send(2415, 0, 0)
end

-- Move caret between paragraphs (delimited by empty lines).
function sci:para_up_extend()
  self:send(2416, 0, 0)
end

-- Given a valid document position, return the previous position taking code
-- page into account. Returns 0 if passed 0.
function sci:position_before(pos)
  return tonumber(self:send(2417, pos, 0))
end

-- Given a valid document position, return the next position taking code
-- page into account. Maximum value returned is the last position in the document.
function sci:position_after(pos)
  return tonumber(self:send(2418, pos, 0))
end

-- Given a valid document position, return a position that differs in a number
-- of characters. Returned value is always between 0 and last position in document.
function sci:position_relative(pos, relative)
  return tonumber(self:send(2670, pos, relative))
end

-- Copy a range of text to the clipboard. Positions are clipped into the document.
function sci:copy_range(start_pos, end_pos)
  self:send(2419, start_pos, end_pos)
end

-- Copy argument text to the clipboard.
function sci:copy_text(length, text)
  self:send(2420, length, string_ptr(text))
end

-- Set the selection mode to stream (SC_SEL_STREAM) or rectangular (SC_SEL_RECTANGLE/SC_SEL_THIN) or
-- by lines (SC_SEL_LINES).
function sci:set_selection_mode(mode)
  self:send(2422, mode, 0)
end

-- Get the mode of the current selection.
function sci:get_selection_mode()
  return tonumber(self:send(2423, 0, 0))
end

-- Retrieve the position of the start of the selection at the given line (INVALID_POSITION if no selection on this line).
function sci:get_line_sel_start_position(line)
  return tonumber(self:send(2424, line, 0))
end

-- Retrieve the position of the end of the selection at the given line (INVALID_POSITION if no selection on this line).
function sci:get_line_sel_end_position(line)
  return tonumber(self:send(2425, line, 0))
end

-- # RectExtended rectangular selection moves
-- Move caret down one line, extending rectangular selection to new caret position.
function sci:line_down_rect_extend()
  self:send(2426, 0, 0)
end

-- Move caret up one line, extending rectangular selection to new caret position.
function sci:line_up_rect_extend()
  self:send(2427, 0, 0)
end

-- Move caret left one character, extending rectangular selection to new caret position.
function sci:char_left_rect_extend()
  self:send(2428, 0, 0)
end

-- Move caret right one character, extending rectangular selection to new caret position.
function sci:char_right_rect_extend()
  self:send(2429, 0, 0)
end

-- Move caret to first position on line, extending rectangular selection to new caret position.
function sci:home_rect_extend()
  self:send(2430, 0, 0)
end

-- Move caret to before first visible character on line.
-- If already there move to first character on line.
-- In either case, extend rectangular selection to new caret position.
function sci:vchome_rect_extend()
  self:send(2431, 0, 0)
end

-- Move caret to last position on line, extending rectangular selection to new caret position.
function sci:line_end_rect_extend()
  self:send(2432, 0, 0)
end

-- Move caret one page up, extending rectangular selection to new caret position.
function sci:page_up_rect_extend()
  self:send(2433, 0, 0)
end

-- Move caret one page down, extending rectangular selection to new caret position.
function sci:page_down_rect_extend()
  self:send(2434, 0, 0)
end

-- Move caret to top of page, or one page up if already at top of page.
function sci:stuttered_page_up()
  self:send(2435, 0, 0)
end

-- Move caret to top of page, or one page up if already at top of page, extending selection to new caret position.
function sci:stuttered_page_up_extend()
  self:send(2436, 0, 0)
end

-- Move caret to bottom of page, or one page down if already at bottom of page.
function sci:stuttered_page_down()
  self:send(2437, 0, 0)
end

-- Move caret to bottom of page, or one page down if already at bottom of page, extending selection to new caret position.
function sci:stuttered_page_down_extend()
  self:send(2438, 0, 0)
end

-- Move caret left one word, position cursor at end of word.
function sci:word_left_end()
  self:send(2439, 0, 0)
end

-- Move caret left one word, position cursor at end of word, extending selection to new caret position.
function sci:word_left_end_extend()
  self:send(2440, 0, 0)
end

-- Move caret right one word, position cursor at end of word.
function sci:word_right_end()
  self:send(2441, 0, 0)
end

-- Move caret right one word, position cursor at end of word, extending selection to new caret position.
function sci:word_right_end_extend()
  self:send(2442, 0, 0)
end

-- Set the set of characters making up whitespace for when moving or selecting by word.
-- Should be called after SetWordChars.
function sci:set_whitespace_chars(characters)
  self:send(2443, 0, string_ptr(characters))
end

-- Get the set of characters making up whitespace for when moving or selecting by word.
function sci:get_whitespace_chars()
  return self:send_with_stringresult(2647)
end

-- Set the set of characters making up punctuation characters
-- Should be called after SetWordChars.
function sci:set_punctuation_chars(characters)
  self:send(2648, 0, string_ptr(characters))
end

-- Get the set of characters making up punctuation characters
function sci:get_punctuation_chars()
  return self:send_with_stringresult(2649)
end

-- Reset the set of characters for whitespace and word characters to the defaults.
function sci:set_chars_default()
  self:send(2444, 0, 0)
end

-- Get currently selected item position in the auto-completion list
function sci:auto_cget_current()
  return tonumber(self:send(2445, 0, 0))
end

-- Get currently selected item text in the auto-completion list
-- Returns the length of the item text
function sci:auto_cget_current_text()
  return self:send_with_stringresult(2610)
end

-- Set auto-completion case insensitive behaviour to either prefer case-sensitive matches or have no preference.
function sci:auto_cset_case_insensitive_behaviour(behaviour)
  self:send(2634, behaviour, 0)
end

-- Get auto-completion case insensitive behaviour.
function sci:auto_cget_case_insensitive_behaviour()
  return tonumber(self:send(2635, 0, 0))
end

-- Set the way autocompletion lists are ordered.
function sci:auto_cset_order(order)
  self:send(2660, order, 0)
end

-- Get the way autocompletion lists are ordered.
function sci:auto_cget_order()
  return tonumber(self:send(2661, 0, 0))
end

-- Enlarge the document to a particular size of text bytes.
function sci:allocate(bytes)
  self:send(2446, bytes, 0)
end

-- Returns the target converted to UTF8.
-- Return the length in bytes.
function sci:target_as_utf8()
  return self:send_with_stringresult(2447)
end

-- Set the length of the utf8 argument for calling EncodedFromUTF8.
-- Set to -1 and the string will be measured to the first nul.
function sci:set_length_for_encode(bytes)
  self:send(2448, bytes, 0)
end

-- Translates a UTF8 string into the document encoding.
-- Return the length of the result in bytes.
-- On error return 0.
function sci:encoded_from_utf8(utf8)
  return self:send_with_stringresult(2449, utf8)
end

-- Find the position of a column on a line taking into account tabs and
-- multi-byte characters. If beyond end of line, return line end position.
function sci:find_column(line, column)
  return tonumber(self:send(2456, line, column))
end

-- Can the caret preferred x position only be changed by explicit movement commands?
function sci:get_caret_sticky()
  return tonumber(self:send(2457, 0, 0))
end

-- Stop the caret preferred x position changing when the user types.
function sci:set_caret_sticky(use_caret_sticky_behaviour)
  self:send(2458, use_caret_sticky_behaviour, 0)
end

-- Switch between sticky and non-sticky: meant to be bound to a key.
function sci:toggle_caret_sticky()
  self:send(2459, 0, 0)
end

-- Enable/Disable convert-on-paste for line endings
function sci:set_paste_convert_endings(convert)
  self:send(2467, (convert and 1 or 0), 0)
end

-- Get convert-on-paste setting
function sci:get_paste_convert_endings()
  return 0 ~= self:send(2468, 0, 0)
end

-- Duplicate the selection. If selection empty duplicate the line containing the caret.
function sci:selection_duplicate()
  self:send(2469, 0, 0)
end

-- Set background alpha of the caret line.
function sci:set_caret_line_back_alpha(alpha)
  self:send(2470, alpha, 0)
end

-- Get the background alpha of the caret line.
function sci:get_caret_line_back_alpha()
  return tonumber(self:send(2471, 0, 0))
end

-- Set the style of the caret to be drawn.
function sci:set_caret_style(caret_style)
  self:send(2512, caret_style, 0)
end

-- Returns the current style of the caret.
function sci:get_caret_style()
  return tonumber(self:send(2513, 0, 0))
end

-- Set the indicator used for IndicatorFillRange and IndicatorClearRange
function sci:set_indicator_current(indicator)
  self:send(2500, indicator, 0)
end

-- Get the current indicator
function sci:get_indicator_current()
  return tonumber(self:send(2501, 0, 0))
end

-- Set the value used for IndicatorFillRange
function sci:set_indicator_value(value)
  self:send(2502, value, 0)
end

-- Get the current indicator value
function sci:get_indicator_value()
  return tonumber(self:send(2503, 0, 0))
end

-- Turn a indicator on over a range.
function sci:indicator_fill_range(position, fill_length)
  self:send(2504, position, fill_length)
end

-- Turn a indicator off over a range.
function sci:indicator_clear_range(position, clear_length)
  self:send(2505, position, clear_length)
end

-- Are any indicators present at position?
function sci:indicator_all_on_for(position)
  return tonumber(self:send(2506, position, 0))
end

-- What value does a particular indicator have at at a position?
function sci:indicator_value_at(indicator, position)
  return tonumber(self:send(2507, indicator, position))
end

-- Where does a particular indicator start?
function sci:indicator_start(indicator, position)
  return tonumber(self:send(2508, indicator, position))
end

-- Where does a particular indicator end?
function sci:indicator_end(indicator, position)
  return tonumber(self:send(2509, indicator, position))
end

-- Set number of entries in position cache
function sci:set_position_cache(size)
  self:send(2514, size, 0)
end

-- How many entries are allocated to the position cache?
function sci:get_position_cache()
  return tonumber(self:send(2515, 0, 0))
end

-- Copy the selection, if selection empty copy the line with the caret
function sci:copy_allow_line()
  self:send(2519, 0, 0)
end

-- Return a position which, to avoid performance costs, should not be within
-- the range of a call to GetRangePointer.
function sci:get_gap_position()
  return tonumber(self:send(2644, 0, 0))
end

-- Always interpret keyboard input as Unicode
function sci:set_keys_unicode(keys_unicode)
  self:send(2521, (keys_unicode and 1 or 0), 0)
end

-- Are keys always interpreted as Unicode?
function sci:get_keys_unicode()
  return 0 ~= self:send(2522, 0, 0)
end

-- Set the alpha fill colour of the given indicator.
function sci:indic_set_alpha(indicator, alpha)
  self:send(2523, indicator, alpha)
end

-- Get the alpha fill colour of the given indicator.
function sci:indic_get_alpha(indicator)
  return tonumber(self:send(2524, indicator, 0))
end

-- Set the alpha outline colour of the given indicator.
function sci:indic_set_outline_alpha(indicator, alpha)
  self:send(2558, indicator, alpha)
end

-- Get the alpha outline colour of the given indicator.
function sci:indic_get_outline_alpha(indicator)
  return tonumber(self:send(2559, indicator, 0))
end

-- Set extra ascent for each line
function sci:set_extra_ascent(extra_ascent)
  self:send(2525, extra_ascent, 0)
end

-- Get extra ascent for each line
function sci:get_extra_ascent()
  return tonumber(self:send(2526, 0, 0))
end

-- Set extra descent for each line
function sci:set_extra_descent(extra_descent)
  self:send(2527, extra_descent, 0)
end

-- Get extra descent for each line
function sci:get_extra_descent()
  return tonumber(self:send(2528, 0, 0))
end

-- Which symbol was defined for markerNumber with MarkerDefine
function sci:marker_symbol_defined(marker_number)
  return tonumber(self:send(2529, marker_number, 0))
end

-- Set the text in the text margin for a line
function sci:margin_set_text(line, text)
  self:send(2530, line, string_ptr(text))
end

-- Get the text in the text margin for a line
function sci:margin_get_text(line)
  return self:send_with_stringresult(2531, line)
end

-- Set the style number for the text margin for a line
function sci:margin_set_style(line, style)
  self:send(2532, line, style)
end

-- Get the style number for the text margin for a line
function sci:margin_get_style(line)
  return tonumber(self:send(2533, line, 0))
end

-- Set the style in the text margin for a line
function sci:margin_set_styles(line, styles)
  self:send(2534, line, string_ptr(styles))
end

-- Get the styles in the text margin for a line
function sci:margin_get_styles(line)
  return self:send_with_stringresult(2535, line)
end

-- Clear the margin text on all lines
function sci:margin_text_clear_all()
  self:send(2536, 0, 0)
end

-- Get the start of the range of style numbers used for margin text
function sci:margin_set_style_offset(style)
  self:send(2537, style, 0)
end

-- Get the start of the range of style numbers used for margin text
function sci:margin_get_style_offset()
  return tonumber(self:send(2538, 0, 0))
end

-- Set the margin options.
function sci:set_margin_options(margin_options)
  self:send(2539, margin_options, 0)
end

-- Get the margin options.
function sci:get_margin_options()
  return tonumber(self:send(2557, 0, 0))
end

-- Set the annotation text for a line
function sci:annotation_set_text(line, text)
  self:send(2540, line, string_ptr(text))
end

-- Get the annotation text for a line
function sci:annotation_get_text(line)
  return self:send_with_stringresult(2541, line)
end

-- Set the style number for the annotations for a line
function sci:annotation_set_style(line, style)
  self:send(2542, line, style)
end

-- Get the style number for the annotations for a line
function sci:annotation_get_style(line)
  return tonumber(self:send(2543, line, 0))
end

-- Set the annotation styles for a line
function sci:annotation_set_styles(line, styles)
  self:send(2544, line, string_ptr(styles))
end

-- Get the annotation styles for a line
function sci:annotation_get_styles(line)
  return self:send_with_stringresult(2545, line)
end

-- Get the number of annotation lines for a line
function sci:annotation_get_lines(line)
  return tonumber(self:send(2546, line, 0))
end

-- Clear the annotations from all lines
function sci:annotation_clear_all()
  self:send(2547, 0, 0)
end

-- Set the visibility for the annotations for a view
function sci:annotation_set_visible(visible)
  self:send(2548, visible, 0)
end

-- Get the visibility for the annotations for a view
function sci:annotation_get_visible()
  return tonumber(self:send(2549, 0, 0))
end

-- Get the start of the range of style numbers used for annotations
function sci:annotation_set_style_offset(style)
  self:send(2550, style, 0)
end

-- Get the start of the range of style numbers used for annotations
function sci:annotation_get_style_offset()
  return tonumber(self:send(2551, 0, 0))
end

-- Release all extended (>255) style numbers
function sci:release_all_extended_styles()
  self:send(2552, 0, 0)
end

-- Allocate some extended (>255) style numbers and return the start of the range
function sci:allocate_extended_styles(number_styles)
  return tonumber(self:send(2553, number_styles, 0))
end

-- Add a container action to the undo stack
function sci:add_undo_action(token, flags)
  self:send(2560, token, flags)
end

-- Find the position of a character from a point within the window.
function sci:char_position_from_point(x, y)
  return tonumber(self:send(2561, x, y))
end

-- Find the position of a character from a point within the window.
-- Return INVALID_POSITION if not close to text.
function sci:char_position_from_point_close(x, y)
  return tonumber(self:send(2562, x, y))
end

-- Set whether switching to rectangular mode while selecting with the mouse is allowed.
function sci:set_mouse_selection_rectangular_switch(mouse_selection_rectangular_switch)
  self:send(2668, (mouse_selection_rectangular_switch and 1 or 0), 0)
end

-- Whether switching to rectangular mode while selecting with the mouse is allowed.
function sci:get_mouse_selection_rectangular_switch()
  return 0 ~= self:send(2669, 0, 0)
end

-- Set whether multiple selections can be made
function sci:set_multiple_selection(multiple_selection)
  self:send(2563, (multiple_selection and 1 or 0), 0)
end

-- Whether multiple selections can be made
function sci:get_multiple_selection()
  return 0 ~= self:send(2564, 0, 0)
end

-- Set whether typing can be performed into multiple selections
function sci:set_additional_selection_typing(additional_selection_typing)
  self:send(2565, (additional_selection_typing and 1 or 0), 0)
end

-- Whether typing can be performed into multiple selections
function sci:get_additional_selection_typing()
  return 0 ~= self:send(2566, 0, 0)
end

-- Set whether additional carets will blink
function sci:set_additional_carets_blink(additional_carets_blink)
  self:send(2567, (additional_carets_blink and 1 or 0), 0)
end

-- Whether additional carets will blink
function sci:get_additional_carets_blink()
  return 0 ~= self:send(2568, 0, 0)
end

-- Set whether additional carets are visible
function sci:set_additional_carets_visible(additional_carets_blink)
  self:send(2608, (additional_carets_blink and 1 or 0), 0)
end

-- Whether additional carets are visible
function sci:get_additional_carets_visible()
  return 0 ~= self:send(2609, 0, 0)
end

-- How many selections are there?
function sci:get_selections()
  return tonumber(self:send(2570, 0, 0))
end

-- Is every selected range empty?
function sci:get_selection_empty()
  return 0 ~= self:send(2650, 0, 0)
end

-- Clear selections to a single empty stream selection
function sci:clear_selections()
  self:send(2571, 0, 0)
end

-- Set a simple selection
function sci:set_selection(caret, anchor)
  return tonumber(self:send(2572, caret, anchor))
end

-- Add a selection
function sci:add_selection(caret, anchor)
  return tonumber(self:send(2573, caret, anchor))
end

-- Drop one selection
function sci:drop_selection_n(selection)
  self:send(2671, selection, 0)
end

-- Set the main selection
function sci:set_main_selection(selection)
  self:send(2574, selection, 0)
end

-- Which selection is the main selection
function sci:get_main_selection()
  return tonumber(self:send(2575, 0, 0))
end

function sci:set_selection_ncaret(selection, pos)
  self:send(2576, selection, pos)
end

function sci:get_selection_ncaret(selection)
  return tonumber(self:send(2577, selection, 0))
end

function sci:set_selection_nanchor(selection, pos_anchor)
  self:send(2578, selection, pos_anchor)
end

function sci:get_selection_nanchor(selection)
  return tonumber(self:send(2579, selection, 0))
end

function sci:set_selection_ncaret_virtual_space(selection, space)
  self:send(2580, selection, space)
end

function sci:get_selection_ncaret_virtual_space(selection)
  return tonumber(self:send(2581, selection, 0))
end

function sci:set_selection_nanchor_virtual_space(selection, space)
  self:send(2582, selection, space)
end

function sci:get_selection_nanchor_virtual_space(selection)
  return tonumber(self:send(2583, selection, 0))
end

-- Sets the position that starts the selection - this becomes the anchor.
function sci:set_selection_nstart(selection, pos)
  self:send(2584, selection, pos)
end

-- Returns the position at the start of the selection.
function sci:get_selection_nstart(selection)
  return tonumber(self:send(2585, selection, 0))
end

-- Sets the position that ends the selection - this becomes the currentPosition.
function sci:set_selection_nend(selection, pos)
  self:send(2586, selection, pos)
end

-- Returns the position at the end of the selection.
function sci:get_selection_nend(selection)
  return tonumber(self:send(2587, selection, 0))
end

function sci:set_rectangular_selection_caret(pos)
  self:send(2588, pos, 0)
end

function sci:get_rectangular_selection_caret()
  return tonumber(self:send(2589, 0, 0))
end

function sci:set_rectangular_selection_anchor(pos_anchor)
  self:send(2590, pos_anchor, 0)
end

function sci:get_rectangular_selection_anchor()
  return tonumber(self:send(2591, 0, 0))
end

function sci:set_rectangular_selection_caret_virtual_space(space)
  self:send(2592, space, 0)
end

function sci:get_rectangular_selection_caret_virtual_space()
  return tonumber(self:send(2593, 0, 0))
end

function sci:set_rectangular_selection_anchor_virtual_space(space)
  self:send(2594, space, 0)
end

function sci:get_rectangular_selection_anchor_virtual_space()
  return tonumber(self:send(2595, 0, 0))
end

function sci:set_virtual_space_options(virtual_space_options)
  self:send(2596, virtual_space_options, 0)
end

function sci:get_virtual_space_options()
  return tonumber(self:send(2597, 0, 0))
end

function sci:set_rectangular_selection_modifier(modifier)
  self:send(2598, modifier, 0)
end

-- Get the modifier key used for rectangular selection.
function sci:get_rectangular_selection_modifier()
  return tonumber(self:send(2599, 0, 0))
end

-- Set the foreground colour of additional selections.
-- Must have previously called SetSelFore with non-zero first argument for this to have an effect.
function sci:set_additional_sel_fore(fore)
  self:send(2600, string_to_color(fore), 0)
end

-- Set the background colour of additional selections.
-- Must have previously called SetSelBack with non-zero first argument for this to have an effect.
function sci:set_additional_sel_back(back)
  self:send(2601, string_to_color(back), 0)
end

-- Set the alpha of the selection.
function sci:set_additional_sel_alpha(alpha)
  self:send(2602, alpha, 0)
end

-- Get the alpha of the selection.
function sci:get_additional_sel_alpha()
  return tonumber(self:send(2603, 0, 0))
end

-- Set the foreground colour of additional carets.
function sci:set_additional_caret_fore(fore)
  self:send(2604, string_to_color(fore), 0)
end

-- Get the foreground colour of additional carets.
function sci:get_additional_caret_fore()
  return color_to_string(self:send(2605, 0, 0))
end

-- Set the main selection to the next selection.
function sci:rotate_selection()
  self:send(2606, 0, 0)
end

-- Swap that caret and anchor of the main selection.
function sci:swap_main_anchor_caret()
  self:send(2607, 0, 0)
end

-- Indicate that the internal state of a lexer has changed over a range and therefore
-- there may be a need to redraw.
function sci:change_lexer_state(start_pos, end_pos)
  return tonumber(self:send(2617, start_pos, end_pos))
end

-- Find the next line at or after lineStart that is a contracted fold header line.
-- Return -1 when no more lines.
function sci:contracted_fold_next(line_start)
  return tonumber(self:send(2618, line_start, 0))
end

-- Centre current line in window.
function sci:vertical_centre_caret()
  self:send(2619, 0, 0)
end

-- Move the selected lines up one line, shifting the line above after the selection
function sci:move_selected_lines_up()
  self:send(2620, 0, 0)
end

-- Move the selected lines down one line, shifting the line below before the selection
function sci:move_selected_lines_down()
  self:send(2621, 0, 0)
end

-- Set the identifier reported as idFrom in notification messages.
function sci:set_identifier(identifier)
  self:send(2622, identifier, 0)
end

-- Get the identifier.
function sci:get_identifier()
  return tonumber(self:send(2623, 0, 0))
end

-- Set the width for future RGBA image data.
function sci:rgbaimage_set_width(width)
  self:send(2624, width, 0)
end

-- Set the height for future RGBA image data.
function sci:rgbaimage_set_height(height)
  self:send(2625, height, 0)
end

-- Set the scale factor in percent for future RGBA image data.
function sci:rgbaimage_set_scale(scale_percent)
  self:send(2651, scale_percent, 0)
end

-- Define a marker from RGBA data.
-- It has the width and height from RGBAImageSetWidth/Height
function sci:marker_define_rgbaimage(marker_number, pixels)
  self:send(2626, marker_number, string_ptr(pixels))
end

-- Register an RGBA image for use in autocompletion lists.
-- It has the width and height from RGBAImageSetWidth/Height
function sci:register_rgbaimage(type, pixels)
  self:send(2627, type, string_ptr(pixels))
end

-- Scroll to start of document.
function sci:scroll_to_start()
  self:send(2628, 0, 0)
end

-- Scroll to end of document.
function sci:scroll_to_end()
  self:send(2629, 0, 0)
end

-- Set the technology used.
function sci:set_technology(technology)
  self:send(2630, technology, 0)
end

-- Get the tech.
function sci:get_technology()
  return tonumber(self:send(2631, 0, 0))
end

-- Create an ILoader*.
function sci:create_loader(bytes)
  return tonumber(self:send(2632, bytes, 0))
end

-- On OS X, show a find indicator.
function sci:find_indicator_show(start_pos, end_pos)
  self:send(2640, start_pos, end_pos)
end

-- On OS X, flash a find indicator, then fade out.
function sci:find_indicator_flash(start_pos, end_pos)
  self:send(2641, start_pos, end_pos)
end

-- On OS X, hide the find indicator.
function sci:find_indicator_hide()
  self:send(2642, 0, 0)
end

-- Move caret to before first visible character on display line.
-- If already there move to first character on display line.
function sci:vchome_display()
  self:send(2652, 0, 0)
end

-- Like VCHomeDisplay but extending selection to new caret position.
function sci:vchome_display_extend()
  self:send(2653, 0, 0)
end

-- Is the caret line always visible?
function sci:get_caret_line_visible_always()
  return 0 ~= self:send(2654, 0, 0)
end

-- Sets the caret line to always visible.
function sci:set_caret_line_visible_always(always_visible)
  self:send(2655, (always_visible and 1 or 0), 0)
end

-- Set the line end types that the application wants to use. May not be used if incompatible with lexer or encoding.
function sci:set_line_end_types_allowed(line_end_bit_set)
  self:send(2656, line_end_bit_set, 0)
end

-- Get the line end types currently allowed.
function sci:get_line_end_types_allowed()
  return tonumber(self:send(2657, 0, 0))
end

-- Get the line end types currently recognised. May be a subset of the allowed types due to lexer limitation.
function sci:get_line_end_types_active()
  return tonumber(self:send(2658, 0, 0))
end

-- Set the way a character is drawn.
function sci:set_representation(encoded_character, representation)
  self:send(2665, string_ptr(encoded_character), string_ptr(representation))
end

-- Set the way a character is drawn.
function sci:get_representation(encoded_character)
  return self:send_with_stringresult(2666, encoded_character)
end

-- Remove a character representation.
function sci:clear_representation(encoded_character)
  self:send(2667, string_ptr(encoded_character), 0)
end

-- Start notifying the container of all key presses and commands.
function sci:start_record()
  self:send(3001, 0, 0)
end

-- Stop notifying the container of all key presses and commands.
function sci:stop_record()
  self:send(3002, 0, 0)
end

-- Set the lexing language of the document.
function sci:set_lexer(lexer)
  self:send(4001, lexer, 0)
end

-- Retrieve the lexing language of the document.
function sci:get_lexer()
  return tonumber(self:send(4002, 0, 0))
end

-- Colourise a segment of the document using the current lexing language.
function sci:colourise(start_pos, end_pos)
  self:send(4003, start_pos, end_pos)
end

-- Set up a value that may be used by a lexer for some optional feature.
function sci:set_property(key, value)
  self:send(4004, string_ptr(key), string_ptr(value))
end

-- Set up the key words used by the lexer.
function sci:set_key_words(keyword_set, key_words)
  self:send(4005, keyword_set, string_ptr(key_words))
end

-- Set the lexing language of the document based on string name.
function sci:set_lexer_language(language)
  self:send(4006, 0, string_ptr(language))
end

-- Load a lexer library (dll / so).
function sci:load_lexer_library(path)
  self:send(4007, 0, string_ptr(path))
end

-- Retrieve a "property" value previously set with SetProperty.
function sci:get_property(key)
  return self:send_with_stringresult(4008, key)
end

-- Retrieve a "property" value previously set with SetProperty,
-- with "$()" variable replacement on returned buffer.
function sci:get_property_expanded(key)
  return self:send_with_stringresult(4009, key)
end

-- Retrieve a "property" value previously set with SetProperty,
-- interpreted as an int AFTER any "$()" variable replacement.
function sci:get_property_int(key)
  return tonumber(self:send(4010, string_ptr(key), 0))
end

-- Retrieve the number of bits the current lexer needs for styling.
function sci:get_style_bits_needed()
  return tonumber(self:send(4011, 0, 0))
end

-- Retrieve the name of the lexer.
-- Return the length of the text.
function sci:get_lexer_language()
  return self:send_with_stringresult(4012)
end

-- For private communication between an application and a known lexer.
function sci:private_lexer_call(operation, pointer)
  return tonumber(self:send(4013, operation, pointer))
end

-- Retrieve a '\n' separated list of properties understood by the current lexer.
function sci:property_names()
  return self:send_with_stringresult(4014)
end

-- Retrieve the type of a property.
function sci:property_type(name)
  return tonumber(self:send(4015, string_ptr(name), 0))
end

-- Describe a property.
function sci:describe_property(name)
  return self:send_with_stringresult(4016, name)
end

-- Retrieve a '\n' separated list of descriptions of the keyword sets understood by the current lexer.
function sci:describe_key_word_sets()
  return self:send_with_stringresult(4017)
end

-- Bit set of LineEndType enumertion for which line ends beyond the standard
-- LF, CR, and CRLF are supported by the lexer.
function sci:get_line_end_types_supported()
  return tonumber(self:send(4018, 0, 0))
end

-- Allocate a set of sub styles for a particular base style, returning start of range
function sci:allocate_sub_styles(style_base, number_styles)
  return tonumber(self:send(4020, style_base, number_styles))
end

-- The starting style number for the sub styles associated with a base style
function sci:get_sub_styles_start(style_base)
  return tonumber(self:send(4021, style_base, 0))
end

-- The number of sub styles associated with a base style
function sci:get_sub_styles_length(style_base)
  return tonumber(self:send(4022, style_base, 0))
end

-- For a sub style, return the base style, else return the argument.
function sci:get_style_from_sub_style(sub_style)
  return tonumber(self:send(4027, sub_style, 0))
end

-- For a secondary style, return the primary style, else return the argument.
function sci:get_primary_style_from_style(style)
  return tonumber(self:send(4028, style, 0))
end

-- Free allocated sub styles
function sci:free_sub_styles()
  self:send(4023, 0, 0)
end

-- Set the identifiers that are shown in a particular style
function sci:set_identifiers(style, identifiers)
  self:send(4024, style, string_ptr(identifiers))
end

-- Where styles are duplicated by a feature such as active/inactive code
-- return the distance between the two types.
function sci:distance_to_secondary_styles()
  return tonumber(self:send(4025, 0, 0))
end

-- Get the set of base styles that can be extended with sub styles
function sci:get_sub_style_bases()
  return self:send_with_stringresult(4026)
end

-- In palette mode?
function sci:get_use_palette()
  return 0 ~= self:send(2139, 0, 0)
end

-- In palette mode, Scintilla uses the environment's palette calls to display
-- more colours. This may lead to ugly displays.
function sci:set_use_palette(use_palette)
  self:send(2039, (use_palette and 1 or 0), 0)
end

-- !! End auto generated content

jit.off(sci.send)

return sci
