/**
 * @fileoverview Comma spacing - validates spacing before and after comma
 * @author Vignesh Anand aka vegetableman.
 * @copyright 2014 Vignesh Anand. All rights reserved.
 */
"use strict";

var astUtils = require("../ast-utils");

//------------------------------------------------------------------------------
// Rule Definition
//------------------------------------------------------------------------------

module.exports = {
    meta: {
        docs: {
            description: "enforce consistent spacing before and after commas",
            category: "Stylistic Issues",
            recommended: false
        },

        fixable: "whitespace",

        schema: [
            {
                "type": "object",
                "properties": {
                    "before": {
                        "type": "boolean"
                    },
                    "after": {
                        "type": "boolean"
                    }
                },
                "additionalProperties": false
            }
        ]
    },

    create: function(context) {

        var sourceCode = context.getSourceCode();
        var tokensAndComments = sourceCode.tokensAndComments;

        var options = {
            before: context.options[0] ? !!context.options[0].before : false,
            after: context.options[0] ? !!context.options[0].after : true
        };

        //--------------------------------------------------------------------------
        // Helpers
        //--------------------------------------------------------------------------

        // list of comma tokens to ignore for the check of leading whitespace
        var commaTokensToIgnore = [];

        /**
         * Determines if a given token is a comma operator.
         * @param {ASTNode} token The token to check.
         * @returns {boolean} True if the token is a comma, false if not.
         * @private
         */
        function isComma(token) {
            return !!token && (token.type === "Punctuator") && (token.value === ",");
        }

        /**
         * Reports a spacing error with an appropriate message.
         * @param {ASTNode} node The binary expression node to report.
         * @param {string} dir Is the error "before" or "after" the comma?
         * @param {ASTNode} otherNode The node at the left or right of `node`
         * @returns {void}
         * @private
         */
        function report(node, dir, otherNode) {
            context.report({
                node: node,
                fix: function(fixer) {
                    if (options[dir]) {
                        if (dir === "before") {
                            return fixer.insertTextBefore(node, " ");
                        } else {
                            return fixer.insertTextAfter(node, " ");
                        }
                    } else {
                        var start, end;
                        var newText = "";

                        if (dir === "before") {
                            start = otherNode.range[1];
                            end = node.range[0];
                        } else {
                            start = node.range[1];
                            end = otherNode.range[0];
                        }

                        return fixer.replaceTextRange([start, end], newText);
                    }
                },
                message: options[dir] ?
                  "A space is required " + dir + " ','." :
                  "There should be no space " + dir + " ','."
            });
        }

        /**
         * Validates the spacing around a comma token.
         * @param {Object} tokens - The tokens to be validated.
         * @param {Token} tokens.comma The token representing the comma.
         * @param {Token} [tokens.left] The last token before the comma.
         * @param {Token} [tokens.right] The first token after the comma.
         * @param {Token|ASTNode} reportItem The item to use when reporting an error.
         * @returns {void}
         * @private
         */
        function validateCommaItemSpacing(tokens, reportItem) {
            if (tokens.left && astUtils.isTokenOnSameLine(tokens.left, tokens.comma) &&
                    (options.before !== sourceCode.isSpaceBetweenTokens(tokens.left, tokens.comma))
            ) {
                report(reportItem, "before", tokens.left);
            }

            if (tokens.right && !options.after && tokens.right.type === "Line") {
                return;
            }

            if (tokens.right && astUtils.isTokenOnSameLine(tokens.comma, tokens.right) &&
                    (options.after !== sourceCode.isSpaceBetweenTokens(tokens.comma, tokens.right))
            ) {
                report(reportItem, "after", tokens.right);
            }
        }

        /**
         * Adds null elements of the given ArrayExpression or ArrayPattern node to the ignore list.
         * @param {ASTNode} node An ArrayExpression or ArrayPattern node.
         * @returns {void}
         */
        function addNullElementsToIgnoreList(node) {
            var previousToken = context.getFirstToken(node);

            node.elements.forEach(function(element) {
                var token;

                if (element === null) {
                    token = context.getTokenAfter(previousToken);

                    if (isComma(token)) {
                        commaTokensToIgnore.push(token);
                    }
                } else {
                    token = context.getTokenAfter(element);
                }

                previousToken = token;
            });
        }

        //--------------------------------------------------------------------------
        // Public
        //--------------------------------------------------------------------------

        return {
            "Program:exit": function() {

                var previousToken,
                    nextToken;

                tokensAndComments.forEach(function(token, i) {

                    if (!isComma(token)) {
                        return;
                    }

                    if (token && token.type === "JSXText") {
                        return;
                    }

                    previousToken = tokensAndComments[i - 1];
                    nextToken = tokensAndComments[i + 1];

                    validateCommaItemSpacing({
                        comma: token,
                        left: isComma(previousToken) || commaTokensToIgnore.indexOf(token) > -1 ? null : previousToken,
                        right: isComma(nextToken) ? null : nextToken
                    }, token);
                });
            },
            "ArrayExpression": addNullElementsToIgnoreList,
            "ArrayPattern": addNullElementsToIgnoreList

        };

    }
};
