"""Bool class.

Represents a boolean using a widget.
"""

# Copyright (c) Jupyter Development Team.
# Distributed under the terms of the Modified BSD License.

from .widget import DOMWidget, register
from traitlets import Unicode, Bool, CaselessStrEnum


class _Bool(DOMWidget):
    """A base class for creating widgets that represent booleans."""
    value = Bool(False, help="Bool value", sync=True)
    description = Unicode('', help="Description of the boolean (label).", sync=True)
    disabled = Bool(False, help="Enable or disable user changes.", sync=True)

    def __init__(self, value=None, **kwargs):
        if value is not None:
            kwargs['value'] = value
        super(_Bool, self).__init__(**kwargs)

@register('IPython.Checkbox')
class Checkbox(_Bool):
    """Displays a boolean `value` in the form of a checkbox.

       Parameters
       ----------
       value : {True,False}
           value of the checkbox: True-checked, False-unchecked
       description : str
	   description displayed next to the checkbox
"""
    _view_name = Unicode('CheckboxView', sync=True)


@register('IPython.ToggleButton')
class ToggleButton(_Bool):
    """Displays a boolean `value` in the form of a toggle button.

       Parameters
       ----------
       value : {True,False}
           value of the toggle button: True-pressed, False-unpressed
       description : str
	   description displayed next to the button
       tooltip: str
           tooltip caption of the toggle button
       icon: str
           font-awesome icon name
"""
    _view_name = Unicode('ToggleButtonView', sync=True)
    tooltip = Unicode(help="Tooltip caption of the toggle button.", sync=True)
    icon = Unicode('', help= "Font-awesome icon.", sync=True)

    button_style = CaselessStrEnum(
        values=['primary', 'success', 'info', 'warning', 'danger', ''],
        default_value='', allow_none=True, sync=True, help="""Use a
        predefined styling for the button.""")


@register('IPython.Valid')
class Valid(_Bool):

    """Displays a boolean `value` in the form of a green check (True / valid)
    or a red cross (False / invalid).

    Parameters
    ----------
    value: {True,False}
        value of the Valid widget
"""
    readout = Unicode(help="Message displayed when the value is False", sync=True)
    _view_name = Unicode('ValidView', sync=True)
