/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.bootstrap;

import java.io.ByteArrayOutputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.nio.file.Path;
import java.util.Locale;
import java.util.concurrent.CountDownLatch;
import org.apache.lucene.util.Constants;
import org.apache.lucene.util.StringHelper;
import org.elasticsearch.Version;
import org.elasticsearch.bootstrap.BootstrapCLIParser;
import org.elasticsearch.bootstrap.ConsoleCtrlHandler;
import org.elasticsearch.bootstrap.JNAKernel32Library;
import org.elasticsearch.bootstrap.JVMCheck;
import org.elasticsearch.bootstrap.JarHell;
import org.elasticsearch.bootstrap.Natives;
import org.elasticsearch.bootstrap.Security;
import org.elasticsearch.bootstrap.StartupError;
import org.elasticsearch.common.PidFile;
import org.elasticsearch.common.SuppressForbidden;
import org.elasticsearch.common.cli.CliTool;
import org.elasticsearch.common.cli.Terminal;
import org.elasticsearch.common.inject.CreationException;
import org.elasticsearch.common.lease.Releasables;
import org.elasticsearch.common.logging.ESLogger;
import org.elasticsearch.common.logging.Loggers;
import org.elasticsearch.common.logging.log4j.LogConfigurator;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.env.Environment;
import org.elasticsearch.monitor.jvm.JvmInfo;
import org.elasticsearch.monitor.os.OsProbe;
import org.elasticsearch.monitor.process.ProcessProbe;
import org.elasticsearch.node.Node;
import org.elasticsearch.node.NodeBuilder;
import org.elasticsearch.node.internal.InternalSettingsPreparer;

final class Bootstrap {
    private static volatile Bootstrap INSTANCE;
    private volatile Node node;
    private final CountDownLatch keepAliveLatch = new CountDownLatch(1);
    private final Thread keepAliveThread = new Thread(new Runnable(){

        @Override
        public void run() {
            try {
                Bootstrap.this.keepAliveLatch.await();
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
    }, "elasticsearch[keepAlive/" + Version.CURRENT + "]");
    static final String SECURITY_SETTING = "security.manager.enabled";
    static final String SECURITY_FILTER_BAD_DEFAULTS_SETTING = "security.manager.filter_bad_defaults";

    Bootstrap() {
        this.keepAliveThread.setDaemon(false);
        Runtime.getRuntime().addShutdownHook(new Thread(){

            @Override
            public void run() {
                Bootstrap.this.keepAliveLatch.countDown();
            }
        });
    }

    public static void initializeNatives(Path tmpFile, boolean mlockAll, boolean seccomp, boolean ctrlHandler) {
        final ESLogger logger = Loggers.getLogger(Bootstrap.class);
        if (Natives.definitelyRunningAsRoot()) {
            if (Boolean.parseBoolean(System.getProperty("es.insecure.allow.root"))) {
                logger.warn("running as ROOT user. this is a bad idea!", new Object[0]);
            } else {
                throw new RuntimeException("don't run elasticsearch as root.");
            }
        }
        if (seccomp) {
            Natives.trySeccomp(tmpFile);
        }
        if (mlockAll) {
            if (Constants.WINDOWS) {
                Natives.tryVirtualLock();
            } else {
                Natives.tryMlockall();
            }
        }
        if (ctrlHandler) {
            Natives.addConsoleCtrlHandler(new ConsoleCtrlHandler(){

                @Override
                public boolean handle(int code) {
                    if (2 == code) {
                        logger.info("running graceful exit on windows", new Object[0]);
                        Bootstrap.stop();
                        return true;
                    }
                    return false;
                }
            });
        }
        try {
            JNAKernel32Library.getInstance();
        }
        catch (Throwable throwable) {
            // empty catch block
        }
        StringHelper.randomId();
    }

    static void initializeProbes() {
        ProcessProbe.getInstance();
        OsProbe.getInstance();
    }

    private void setup(boolean addShutdownHook, Settings settings, Environment environment) throws Exception {
        Bootstrap.initializeNatives(environment.tmpFile(), settings.getAsBoolean("bootstrap.mlockall", (Boolean)false), settings.getAsBoolean("bootstrap.seccomp", (Boolean)true), settings.getAsBoolean("bootstrap.ctrlhandler", (Boolean)true));
        Bootstrap.initializeProbes();
        if (addShutdownHook) {
            Runtime.getRuntime().addShutdownHook(new Thread(){

                @Override
                public void run() {
                    if (Bootstrap.this.node != null) {
                        Bootstrap.this.node.close();
                    }
                }
            });
        }
        JarHell.checkJarHell();
        this.setupSecurity(settings, environment);
        Settings nodeSettings = Settings.settingsBuilder().put(settings).put("config.ignore_system_properties", true).build();
        NodeBuilder nodeBuilder = NodeBuilder.nodeBuilder().settings(nodeSettings);
        this.node = nodeBuilder.build();
    }

    private void setupSecurity(Settings settings, Environment environment) throws Exception {
        if (settings.getAsBoolean(SECURITY_SETTING, (Boolean)true).booleanValue()) {
            Security.configure(environment, settings.getAsBoolean(SECURITY_FILTER_BAD_DEFAULTS_SETTING, (Boolean)true));
        }
    }

    private static Environment initialSettings(boolean foreground) {
        Terminal terminal = foreground ? Terminal.DEFAULT : null;
        return InternalSettingsPreparer.prepareEnvironment(Settings.Builder.EMPTY_SETTINGS, terminal);
    }

    private void start() {
        this.node.start();
        this.keepAliveThread.start();
    }

    static void stop() {
        try {
            Releasables.close(Bootstrap.INSTANCE.node);
        }
        finally {
            Bootstrap.INSTANCE.keepAliveLatch.countDown();
        }
    }

    static void init(String[] args) throws Throwable {
        ESLogger logger;
        boolean foreground;
        System.setProperty("es.logger.prefix", "");
        BootstrapCLIParser bootstrapCLIParser = new BootstrapCLIParser();
        CliTool.ExitStatus status = bootstrapCLIParser.execute(args);
        if (CliTool.ExitStatus.OK != status) {
            Bootstrap.exit(status.status());
        }
        INSTANCE = new Bootstrap();
        boolean bl = foreground = !"false".equals(System.getProperty("es.foreground", System.getProperty("es-foreground")));
        if (System.getProperty("wrapper.service", "XXX").equalsIgnoreCase("true")) {
            foreground = false;
        }
        Environment environment = Bootstrap.initialSettings(foreground);
        Settings settings = environment.settings();
        LogConfigurator.configure(settings, true);
        Bootstrap.checkForCustomConfFile();
        if (environment.pidFile() != null) {
            PidFile.create(environment.pidFile(), true);
        }
        if (System.getProperty("es.max-open-files", "false").equals("true")) {
            logger = Loggers.getLogger(Bootstrap.class);
            logger.info("max_open_files [{}]", ProcessProbe.getInstance().getMaxFileDescriptorCount());
        }
        if (JvmInfo.jvmInfo().getVmName().toLowerCase(Locale.ROOT).contains("client")) {
            logger = Loggers.getLogger(Bootstrap.class);
            logger.warn("jvm uses the client vm, make sure to run `java` with the server vm for best performance by adding `-server` to the command line", new Object[0]);
        }
        try {
            if (!foreground) {
                Loggers.disableConsoleLogging();
                Bootstrap.closeSystOut();
            }
            JVMCheck.check();
            INSTANCE.setup(true, settings, environment);
            INSTANCE.start();
            if (!foreground) {
                Bootstrap.closeSysError();
            }
        }
        catch (Throwable e) {
            if (foreground) {
                Loggers.disableConsoleLogging();
            }
            ESLogger logger2 = Loggers.getLogger(Bootstrap.class);
            if (Bootstrap.INSTANCE.node != null) {
                logger2 = Loggers.getLogger(Bootstrap.class, Bootstrap.INSTANCE.node.settings().get("name"));
            }
            if (e instanceof CreationException) {
                ByteArrayOutputStream os = new ByteArrayOutputStream();
                PrintStream ps = new PrintStream((OutputStream)os, false, "UTF-8");
                new StartupError(e).printStackTrace(ps);
                ps.flush();
                logger2.error("Guice Exception: {}", os.toString("UTF-8"));
            } else {
                logger2.error("Exception", e, new Object[0]);
            }
            if (foreground) {
                Loggers.enableConsoleLogging();
            }
            throw e;
        }
    }

    @SuppressForbidden(reason="System#out")
    private static void closeSystOut() {
        System.out.close();
    }

    @SuppressForbidden(reason="System#err")
    private static void closeSysError() {
        System.err.close();
    }

    @SuppressForbidden(reason="System#err")
    private static void sysError(String line, boolean flush) {
        System.err.println(line);
        if (flush) {
            System.err.flush();
        }
    }

    private static void checkForCustomConfFile() {
        String confFileSetting = System.getProperty("es.default.config");
        Bootstrap.checkUnsetAndMaybeExit(confFileSetting, "es.default.config");
        confFileSetting = System.getProperty("es.config");
        Bootstrap.checkUnsetAndMaybeExit(confFileSetting, "es.config");
        confFileSetting = System.getProperty("elasticsearch.config");
        Bootstrap.checkUnsetAndMaybeExit(confFileSetting, "elasticsearch.config");
    }

    private static void checkUnsetAndMaybeExit(String confFileSetting, String settingName) {
        if (confFileSetting != null && !confFileSetting.isEmpty()) {
            ESLogger logger = Loggers.getLogger(Bootstrap.class);
            logger.info("{} is no longer supported. elasticsearch.yml must be placed in the config directory and cannot be renamed.", settingName);
            Bootstrap.exit(1);
        }
    }

    @SuppressForbidden(reason="Allowed to exit explicitly in bootstrap phase")
    private static void exit(int status) {
        System.exit(status);
    }
}

