###############################################################################
#
# The MIT License (MIT)
#
# Copyright (c) Tavendo GmbH
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
###############################################################################

from __future__ import absolute_import, division, print_function

import os
import sys
import weakref
import inspect

from functools import partial

from twisted.python.failure import Failure
from twisted.internet.defer import maybeDeferred, Deferred, DeferredList
from twisted.internet.defer import succeed, fail
from twisted.internet.interfaces import IReactorTime

from zope.interface import provider

from txaio.interfaces import IFailedFuture, ILogger, log_levels
from txaio._iotype import guess_stream_needs_encoding
from txaio import _Config
from txaio._common import _BatchedTimer

import six

using_twisted = True
using_asyncio = False

config = _Config()
_stderr, _stdout = sys.stderr, sys.stdout

# some book-keeping variables here. _observer is used as a global by
# the "backwards compatible" (Twisted < 15) loggers. The _loggers object
# is a weak-ref set; we add Logger instances to this *until* such
# time as start_logging is called (with the desired log-level) and
# then we call _set_log_level on each instance. After that,
# Logger's ctor uses _log_level directly.
_observer = None     # for Twisted legacy logging support; see below
_loggers = weakref.WeakSet()  # weak-references of each logger we've created
_log_level = 'info'  # global log level; possibly changed in start_logging()
_started_logging = False

_categories = {}

IFailedFuture.register(Failure)

_NEW_LOGGER = False
try:
    # Twisted 15+
    from twisted.logger import Logger as _Logger, formatEvent, ILogObserver
    from twisted.logger import globalLogBeginner, formatTime, LogLevel
    ILogger.register(_Logger)
    _NEW_LOGGER = True

except ImportError:
    # we still support Twisted 12 and 13, which doesn't have new-logger
    from zope.interface import Interface
    from datetime import datetime
    import time

    # provide our own simple versions of what Twisted new-logger does

    class ILogObserver(Interface):
        pass

    def formatTime(t):  # noqa
        dt = datetime.fromtimestamp(t)
        return six.u(dt.strftime("%Y-%m-%dT%H:%M:%S%z"))

    def formatEvent(event):  # noqa
        msg = event['log_format']
        return msg.format(**event)

    class LogLevel:
        critical = 'critical'
        error = 'error'
        warn = 'warn'
        info = 'info'
        debug = 'debug'
        trace = 'trace'

        @classmethod
        def lookupByName(cls, name):
            return getattr(cls, name)

    class _Logger(ILogger):
        def __init__(self, **kwargs):
            self.namespace = kwargs.get('namespace', None)

        def emit(self, level, format='', **kwargs):
            kwargs['log_time'] = time.time()
            kwargs['log_level'] = level
            kwargs['log_format'] = format
            kwargs['log_namespace'] = self.namespace
            # NOTE: the other loggers are ignoring any log messages
            # before start_logging() as well
            if _observer:
                _observer(kwargs)


def _no_op(*args, **kwargs):
    pass


def add_log_categories(categories):
    _categories.update(categories)


# NOTE: beware that twisted.logger._logger.Logger copies itself via an
# overriden __get__ method when used as recommended as a class
# descriptor.  So, we override __get__ to just return ``self`` which
# means ``log_source`` will be wrong, but we don't document that as a
# key that you can depend on anyway :/
class Logger(object):

    def __init__(self, level=None, logger=None, namespace=None, observer=None):

        assert logger, "Should not be instantiated directly."

        self._logger = logger(observer=observer, namespace=namespace)
        self._log_level_set_explicitly = False

        if level:
            self.set_log_level(level)
        else:
            self._set_log_level(_log_level)

        _loggers.add(self)

    def __get__(self, oself, type=None):
        # this causes the Logger to lie about the "source=", but
        # otherwise we create a new Logger instance every time we do
        # "self.log.info()" if we use it like:
        # class Foo:
        #     log = make_logger
        return self

    def _log(self, level, *args, **kwargs):

        # Look for a log_category, switch it in if we have it
        if "log_category" in kwargs and kwargs["log_category"] in _categories:
            args = tuple()
            kwargs["format"] = _categories.get(kwargs["log_category"])

        self._logger.emit(level, *args, **kwargs)

    def emit(self, level, *args, **kwargs):

        if log_levels.index(self._log_level) < log_levels.index(level):
            return

        if level == "trace":
            return self._trace(*args, **kwargs)

        level = LogLevel.lookupByName(level)
        return self._log(level, *args, **kwargs)

    def set_log_level(self, level, keep=True):
        """
        Set the log level. If keep is True, then it will not change along with
        global log changes.
        """
        self._set_log_level(level)
        self._log_level_set_explicitly = keep

    def _set_log_level(self, level):
        # up to the desired level, we don't do anything, as we're a
        # "real" Twisted new-logger; for methods *after* the desired
        # level, we bind to the no_op method
        desired_index = log_levels.index(level)

        for (idx, name) in enumerate(log_levels):
            if name == 'none':
                continue

            if idx > desired_index:
                current = getattr(self, name, None)
                if not current == _no_op or current is None:
                    setattr(self, name, _no_op)
                if name == 'error':
                    setattr(self, 'failure', _no_op)

            else:
                if getattr(self, name, None) in (_no_op, None):

                    if name == 'trace':
                        setattr(self, "trace", self._trace)
                    else:
                        setattr(self, name,
                                partial(self._log, LogLevel.lookupByName(name)))

                    if name == 'error':
                        setattr(self, "failure", self._failure)

        self._log_level = level

    def _failure(self, format=None, *args, **kw):
        return self._logger.failure(format, *args, **kw)

    def _trace(self, *args, **kw):
        # there is no "trace" level in Twisted -- but this whole
        # method will be no-op'd unless we are at the 'trace' level.
        self.debug(*args, txaio_trace=True, **kw)


def make_logger(level=None, logger=_Logger, observer=None):
    # we want the namespace to be the calling context of "make_logger"
    # -- so we *have* to pass namespace kwarg to Logger (or else it
    # will always say the context is "make_logger")
    cf = inspect.currentframe().f_back
    if "self" in cf.f_locals:
        # We're probably in a class init or method
        cls = cf.f_locals["self"].__class__
        namespace = '{0}.{1}'.format(cls.__module__, cls.__name__)
    else:
        namespace = cf.f_globals["__name__"]
        if cf.f_code.co_name != "<module>":
            # If it's not the module, and not in a class instance, add the code
            # object's name.
            namespace = namespace + "." + cf.f_code.co_name
    logger = Logger(level=level, namespace=namespace, logger=logger,
                    observer=observer)
    return logger


@provider(ILogObserver)
class _LogObserver(object):
    """
    Internal helper.

    An observer which formats events to a given file.
    """
    to_tx = {
        'critical': LogLevel.critical,
        'error': LogLevel.error,
        'warn': LogLevel.warn,
        'info': LogLevel.info,
        'debug': LogLevel.debug,
        'trace': LogLevel.debug,
    }

    def __init__(self, out):
        self._file = out
        self._encode = guess_stream_needs_encoding(out)

        self._levels = None

    def _acceptable_level(self, level):
        if self._levels is None:
            target_level = log_levels.index(_log_level)
            self._levels = [
                self.to_tx[lvl]
                for lvl in log_levels
                if log_levels.index(lvl) <= target_level and lvl != "none"
            ]
        return level in self._levels

    def __call__(self, event):
        # it seems if a twisted.logger.Logger() has .failure() called
        # on it, the log_format will be None for the traceback after
        # "Unhandled error in Deferred" -- perhaps this is a Twisted
        # bug?
        if event['log_format'] is None:
            msg = u'{0} {1}{2}'.format(
                formatTime(event["log_time"]),
                failure_format_traceback(event['log_failure']),
                os.linesep,
            )
            if self._encode:
                msg = msg.encode('utf8')
            self._file.write(msg)
        else:
            # although Logger will already have filtered out unwanted
            # levels, bare Logger instances from Twisted code won't have.
            if 'log_level' in event and self._acceptable_level(event['log_level']):
                msg = u'{0} {1}{2}'.format(
                    formatTime(event["log_time"]),
                    formatEvent(event),
                    os.linesep,
                )
                if self._encode:
                    msg = msg.encode('utf8')

                self._file.write(msg)


def start_logging(out=_stdout, level='info'):
    """
    Start logging to the file-like object in ``out``. By default, this
    is stdout.
    """
    global _loggers, _observer, _log_level, _started_logging

    if level not in log_levels:
        raise RuntimeError(
            "Invalid log level '{0}'; valid are: {1}".format(
                level, ', '.join(log_levels)
            )
        )

    if _started_logging:
        return

    _started_logging = True

    _log_level = level
    set_global_log_level(_log_level)

    if out:
        _observer = _LogObserver(out)

    if _NEW_LOGGER:
        _observers = []
        if _observer:
            _observers.append(_observer)
        globalLogBeginner.beginLoggingTo(_observers)
    else:
        assert out, "out needs to be given a value if using Twisteds before 15.2"
        from twisted.python import log
        log.startLogging(out)


def failure_message(fail):
    """
    :param fail: must be an IFailedFuture
    returns a unicode error-message
    """
    try:
        return u'{0}: {1}'.format(
            fail.value.__class__.__name__,
            fail.getErrorMessage(),
        )
    except Exception:
        return 'Failed to produce failure message for "{0}"'.format(fail)


def failure_traceback(fail):
    """
    :param fail: must be an IFailedFuture
    returns a traceback instance
    """
    return fail.tb


def failure_format_traceback(fail):
    """
    :param fail: must be an IFailedFuture
    returns a string
    """
    try:
        f = six.StringIO()
        fail.printTraceback(file=f)
        return f.getvalue()
    except Exception:
        return u"Failed to format failure traceback for '{0}'".format(fail)


_unspecified = object()


def create_future(result=_unspecified, error=_unspecified):
    if result is not _unspecified and error is not _unspecified:
        raise ValueError("Cannot have both result and error.")

    f = Deferred()
    if result is not _unspecified:
        resolve(f, result)
    elif error is not _unspecified:
        reject(f, error)
    return f


# maybe delete, just use create_future()
def create_future_success(result):
    return succeed(result)


# maybe delete, just use create_future()
def create_future_error(error=None):
    return fail(create_failure(error))


def as_future(fun, *args, **kwargs):
    return maybeDeferred(fun, *args, **kwargs)


def is_future(obj):
    return isinstance(obj, Deferred)


def call_later(delay, fun, *args, **kwargs):
    return IReactorTime(_get_loop()).callLater(delay, fun, *args, **kwargs)


def make_batched_timer(bucket_seconds, chunk_size=100):
    """
    Creates and returns an object implementing
    :class:`txaio.IBatchedTimer`.

    :param bucket_seconds: the number of seconds in each bucket. That
        is, a value of 5 means that any timeout within a 5 second window
        will be in the same bucket, and get notified at the same time.

    :param chunk_size: when "doing" the callbacks in a particular
        bucket, this controls how many we do at once before yielding to
        the reactor.
    """
    clock = IReactorTime(_get_loop())
    get_seconds = clock.seconds

    def create_delayed_call(delay, fun, *args, **kwargs):
        return clock.callLater(delay, fun, *args, **kwargs)

    return _BatchedTimer(
        bucket_seconds, chunk_size,
        seconds_provider=get_seconds,
        delayed_call_creator=create_delayed_call,
    )


def is_called(future):
    return future.called


def resolve(future, result=None):
    future.callback(result)


def reject(future, error=None):
    if error is None:
        error = create_failure()
    elif isinstance(error, Exception):
        error = Failure(error)
    else:
        if not isinstance(error, Failure):
            raise RuntimeError("reject requires a Failure or Exception")
    future.errback(error)


def create_failure(exception=None):
    """
    Create a Failure instance.

    if ``exception`` is None (the default), we MUST be inside an
    "except" block. This encapsulates the exception into an object
    that implements IFailedFuture
    """
    if exception:
        return Failure(exception)
    return Failure()


def add_callbacks(future, callback, errback):
    """
    callback or errback may be None, but at least one must be
    non-None.
    """
    assert future is not None
    if callback is None:
        assert errback is not None
        future.addErrback(errback)
    else:
        # Twisted allows errback to be None here
        future.addCallbacks(callback, errback)
    return future


def gather(futures, consume_exceptions=True):
    def completed(res):
        rtn = []
        for (ok, value) in res:
            rtn.append(value)
            if not ok and not consume_exceptions:
                value.raiseException()
        return rtn

    # XXX if consume_exceptions is False in asyncio.gather(), it will
    # abort on the first raised exception -- should we set
    # fireOnOneErrback=True (if consume_exceptions=False?) -- but then
    # we'll have to wrap the errback() to extract the "real" failure
    # from the FirstError that gets thrown if you set that ...

    dl = DeferredList(list(futures), consumeErrors=consume_exceptions)
    # we unpack the (ok, value) tuples into just a list of values, so
    # that the callback() gets the same value in asyncio and Twisted.
    add_callbacks(dl, completed, None)
    return dl


# methods internal to this implementation


def _get_loop():
    if config.loop is None:
        from twisted.internet import reactor
        config.loop = reactor
    return config.loop


def set_global_log_level(level):
    """
    Set the global log level on all loggers instantiated by txaio.
    """
    for item in _loggers:
        if not item._log_level_set_explicitly:
            item._set_log_level(level)
    global _log_level
    _log_level = level


def get_global_log_level():
    return _log_level
