package vtk;

import java.io.File;

/**
 * Enum used to load native library more easily. If you don't want to set the
 * specific environment variable you can provide the path of the directory that
 * contains the VTK library through a Java Property. Like in the following
 * command line:
 *
 * > java -cp vtk.jar -Dvtk.lib.dir=.../vtk-libs vtk.sample.SimpleVTK
 *
 * The directory .../vtk-libs must contain the so/dll/dylib + the jnilib files
 *
 * @author sebastien jourdain - sebastien.jourdain@kitware.com
 */
public enum vtkNativeLibrary {

    VTKINFOVISCORE("vtkInfovisCoreJava", true),
    VTKINFOVISLAYOUT("vtkInfovisLayoutJava", true),
    VTKINFOVISPARALLEL("vtkInfovisParallelJava", false),
    VTKINFOVISBOOSTGRAPHALGORITHMS("vtkInfovisBoostGraphAlgorithmsJava", true),
    VTKDOMAINSCHEMISTRYOPENGL2("vtkDomainsChemistryOpenGL2Java", false),
    VTKDOMAINSCHEMISTRY("vtkDomainsChemistryJava", true),
    VTKINTERACTIONWIDGETS("vtkInteractionWidgetsJava", true),
    VTKINTERACTIONIMAGE("vtkInteractionImageJava", true),
    VTKINTERACTIONSTYLE("vtkInteractionStyleJava", true),
    VTKCOMMONSYSTEM("vtkCommonSystemJava", true),
    VTKCOMMONCORE("vtkCommonCoreJava", true),
    VTKCOMMONDATAMODEL("vtkCommonDataModelJava", true),
    VTKCOMMONEXECUTIONMODEL("vtkCommonExecutionModelJava", true),
    VTKCOMMONCOMPUTATIONALGEOMETRY("vtkCommonComputationalGeometryJava", true),
    VTKCOMMONMISC("vtkCommonMiscJava", true),
    VTKCOMMONCOLOR("vtkCommonColorJava", true),
    VTKCOMMONMATH("vtkCommonMathJava", true),
    VTKCOMMONTRANSFORMS("vtkCommonTransformsJava", true),
    VTKIMAGINGCORE("vtkImagingCoreJava", true),
    VTKIMAGINGMORPHOLOGICAL("vtkImagingMorphologicalJava", true),
    VTKIMAGINGGENERAL("vtkImagingGeneralJava", true),
    VTKIMAGINGFOURIER("vtkImagingFourierJava", true),
    VTKIMAGINGSOURCES("vtkImagingSourcesJava", true),
    VTKIMAGINGHYBRID("vtkImagingHybridJava", true),
    VTKIMAGINGSTENCIL("vtkImagingStencilJava", true),
    VTKIMAGINGCOLOR("vtkImagingColorJava", true),
    VTKIMAGINGMATH("vtkImagingMathJava", true),
    VTKIMAGINGSTATISTICS("vtkImagingStatisticsJava", true),
    VTKGEOVISCORE("vtkGeovisCoreJava", true),
    VTKRENDERINGCORE("vtkRenderingCoreJava", true),
    VTKRENDERINGCONTEXTOPENGL("vtkRenderingContextOpenGLJava", true),
    VTKRENDERINGVOLUMEAMR("vtkRenderingVolumeAMRJava", true),
    VTKRENDERINGVOLUME("vtkRenderingVolumeJava", true),
    VTKRENDERINGEXTERNAL("vtkRenderingExternalJava", false),
    VTKRENDERINGVOLUMEOPENGL2("vtkRenderingVolumeOpenGL2Java", false),
    VTKRENDERINGOPENGL2("vtkRenderingOpenGL2Java", false),
    VTKRENDERINGLICOPENGL2("vtkRenderingLICOpenGL2Java", false),
    VTKRENDERINGVOLUMEOPENGL("vtkRenderingVolumeOpenGLJava", true),
    VTKRENDERINGANNOTATION("vtkRenderingAnnotationJava", true),
    VTKRENDERINGQT("vtkRenderingQtJava", true),
    VTKRENDERINGCONTEXTOPENGL2("vtkRenderingContextOpenGL2Java", false),
    VTKRENDERINGLOD("vtkRenderingLODJava", true),
    VTKRENDERINGOPENGL("vtkRenderingOpenGLJava", true),
    VTKRENDERINGCONTEXT2D("vtkRenderingContext2DJava", true),
    VTKRENDERINGGL2PS("vtkRenderingGL2PSJava", true),
    VTKRENDERINGLABEL("vtkRenderingLabelJava", true),
    VTKRENDERINGFREETYPE("vtkRenderingFreeTypeJava", true),
    VTKRENDERINGPARALLEL("vtkRenderingParallelJava", true),
    VTKRENDERINGMATPLOTLIB("vtkRenderingMatplotlibJava", true),
    VTKRENDERINGIMAGE("vtkRenderingImageJava", true),
    VTKRENDERINGPARALLELLIC("vtkRenderingParallelLICJava", true),
    VTKRENDERINGLIC("vtkRenderingLICJava", true),
    VTKCHARTSCORE("vtkChartsCoreJava", true),
    VTKFILTERSPYTHON("vtkFiltersPythonJava", true),
    VTKFILTERSMODELING("vtkFiltersModelingJava", true),
    VTKFILTERSCORE("vtkFiltersCoreJava", true),
    VTKFILTERSPARALLELIMAGING("vtkFiltersParallelImagingJava", true),
    VTKFILTERSGENERAL("vtkFiltersGeneralJava", true),
    VTKFILTERSIMAGING("vtkFiltersImagingJava", true),
    VTKFILTERSAMR("vtkFiltersAMRJava", true),
    VTKFILTERSSOURCES("vtkFiltersSourcesJava", true),
    VTKFILTERSHYBRID("vtkFiltersHybridJava", true),
    VTKFILTERSREEBGRAPH("vtkFiltersReebGraphJava", true),
    VTKFILTERSGENERIC("vtkFiltersGenericJava", true),
    VTKFILTERSPARALLELSTATISTICS("vtkFiltersParallelStatisticsJava", true),
    VTKFILTERSVERDICT("vtkFiltersVerdictJava", true),
    VTKFILTERSHYPERTREE("vtkFiltersHyperTreeJava", true),
    VTKFILTERSFLOWPATHS("vtkFiltersFlowPathsJava", true),
    VTKFILTERSSELECTION("vtkFiltersSelectionJava", true),
    VTKFILTERSTEXTURE("vtkFiltersTextureJava", true),
    VTKFILTERSPARALLELGEOMETRY("vtkFiltersParallelGeometryJava", true),
    VTKFILTERSPARALLEL("vtkFiltersParallelJava", true),
    VTKFILTERSPROGRAMMABLE("vtkFiltersProgrammableJava", true),
    VTKFILTERSPARALLELFLOWPATHS("vtkFiltersParallelFlowPathsJava", true),
    VTKFILTERSSTATISTICSGNUR("vtkFiltersStatisticsGnuRJava", false),
    VTKFILTERSEXTRACTION("vtkFiltersExtractionJava", true),
    VTKFILTERSGEOMETRY("vtkFiltersGeometryJava", true),
    VTKFILTERSSTATISTICS("vtkFiltersStatisticsJava", true),
    VTKFILTERSPARALLELMPI("vtkFiltersParallelMPIJava", true),
    VTKFILTERSSMP("vtkFiltersSMPJava", true),
    VTKACCELERATORSDAX("vtkAcceleratorsDaxJava", false),
    VTKACCELERATORSPISTON("vtkAcceleratorsPistonJava", false),
    VTKPYTHONINTERPRETER("vtkPythonInterpreterJava", true),
    VTKVIEWSINFOVIS("vtkViewsInfovisJava", true),
    VTKVIEWSCORE("vtkViewsCoreJava", true),
    VTKVIEWSGEOVIS("vtkViewsGeovisJava", true),
    VTKVIEWSCONTEXT2D("vtkViewsContext2DJava", true),
    VTKWEBCORE("vtkWebCoreJava", false),
    VTKWEBGLEXPORTER("vtkWebGLExporterJava", false),
    VTKPARALLELCORE("vtkParallelCoreJava", true),
    VTKPARALLELMPI("vtkParallelMPIJava", true),
    VTKPARALLELMPI4PY("vtkParallelMPI4PyJava", true),
    VTKIOINFOVIS("vtkIOInfovisJava", true),
    VTKIOPARALLELLSDYNA("vtkIOParallelLSDynaJava", true),
    VTKIOSQL("vtkIOSQLJava", true),
    VTKIOCORE("vtkIOCoreJava", true),
    VTKIOLEGACY("vtkIOLegacyJava", true),
    VTKIOENSIGHT("vtkIOEnSightJava", true),
    VTKIOADIOS("vtkIOADIOSJava", false),
    VTKIOMPIPARALLEL("vtkIOMPIParallelJava", true),
    VTKIOGDAL("vtkIOGDALJava", true),
    VTKIOIMPORT("vtkIOImportJava", true),
    VTKIOGEOJSON("vtkIOGeoJSONJava", true),
    VTKIOAMR("vtkIOAMRJava", true),
    VTKIOPOSTGRESQL("vtkIOPostgreSQLJava", true),
    VTKIOMOVIE("vtkIOMovieJava", true),
    VTKIOPLY("vtkIOPLYJava", true),
    VTKIOPARALLELXML("vtkIOParallelXMLJava", true),
    VTKIOXDMF2("vtkIOXdmf2Java", true),
    VTKIOMYSQL("vtkIOMySQLJava", true),
    VTKIOFFMPEG("vtkIOFFMPEGJava", true),
    VTKIOODBC("vtkIOODBCJava", true),
    VTKIOMPIIMAGE("vtkIOMPIImageJava", true),
    VTKIOPARALLELEXODUS("vtkIOParallelExodusJava", true),
    VTKIOVPIC("vtkIOVPICJava", true),
    VTKIOEXPORT("vtkIOExportJava", true),
    VTKIOMINC("vtkIOMINCJava", true),
    VTKIOVIDEO("vtkIOVideoJava", true),
    VTKIOPARALLEL("vtkIOParallelJava", true),
    VTKIOIMAGE("vtkIOImageJava", true),
    VTKIOEXODUS("vtkIOExodusJava", true),
    VTKIOXML("vtkIOXMLJava", true),
    VTKIOXDMF3("vtkIOXdmf3Java", true),
    VTKIONETCDF("vtkIONetCDFJava", true),
    VTKIOXMLPARSER("vtkIOXMLParserJava", true),
    VTKIOGEOMETRY("vtkIOGeometryJava", true),
    VTKIOLSDYNA("vtkIOLSDynaJava", true),
    VTKIOPARALLELNETCDF("vtkIOParallelNetCDFJava", true),
    VTKTESTINGRENDERING("vtkTestingRenderingJava", true);


  /**
   * Try to load all library
   *
   * @return true if all library have been successfully loaded
   */
  public static boolean LoadAllNativeLibraries() {
    boolean isEveryThingLoaded = true;
    for (vtkNativeLibrary lib : values()) {
      try {
        if(lib.IsBuilt()) {
          lib.LoadLibrary();
        }
      } catch (UnsatisfiedLinkError e) {
        isEveryThingLoaded = false;
        e.printStackTrace();
      }
    }

    return isEveryThingLoaded;
  }

  /**
   * Load the set of given library and trows runtime exception if any given
   * library failed in the loading process
   *
   * @param nativeLibraries
   */
  public static void LoadNativeLibraries(vtkNativeLibrary... nativeLibraries) {
    for (vtkNativeLibrary lib : nativeLibraries) {
      lib.LoadLibrary();
    }
  }

  /**
   * Disable the pop-in vtkErrorWindow by writing the error to a log file.
   * If the provided logFile is null the default "vtkError.txt" file will be
   * used.
   *
   * @param logFile
   */
  public static void DisableOutputWindow(File logFile) {
    if(logFile == null) {
      logFile = new File("vtkError.txt");
    }
    vtkFileOutputWindow outputError = new vtkFileOutputWindow();
    outputError.SetFileName(logFile.getAbsolutePath());
    outputError.SetInstance(outputError);
  }

  private vtkNativeLibrary(String nativeLibraryName, boolean built) {
    this.nativeLibraryName = nativeLibraryName;
    this.loaded = false;
    this.built = built;
  }

  /**
   * Load the library and throws runtime exception if the library failed in
   * the loading process
   */
  public void LoadLibrary() throws UnsatisfiedLinkError {
    if (!loaded) {
      if (System.getProperty("vtk.lib.dir") != null) {
        File dir = new File(System.getProperty("vtk.lib.dir"));
        patchJavaLibraryPath(dir.getAbsolutePath());
        File libPath = new File(dir, System.mapLibraryName(nativeLibraryName));
        if (libPath.exists()) {
          try {
            Runtime.getRuntime().load(libPath.getAbsolutePath());
            loaded = true;
            return;
          } catch (UnsatisfiedLinkError e) {
            e.printStackTrace();
          }
        }
      }
      System.loadLibrary(nativeLibraryName);
    }
    loaded = true;
  }

  /**
   * @return true if the library has already been successfully loaded
   */
  public boolean IsLoaded() {
    return loaded;
  }

  /**
   * @return true if the module was enabled and therefore build
   */
  public boolean IsBuilt() {
    return built;
  }

  /**
   * @return the library name
   */
  public String GetLibraryName() {
    return nativeLibraryName;
  }

  private static void patchJavaLibraryPath(String vtkLibDir) {
    if (vtkLibDir != null) {
      String path_separator = System.getProperty("path.separator");
      String s = System.getProperty("java.library.path");
      if (!s.contains(vtkLibDir)) {
        s = s + path_separator + vtkLibDir;
        System.setProperty("java.library.path", s);
      }
    }
  }

  private String nativeLibraryName;
  private boolean loaded;
  private boolean built;
}
