/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.routing.allocation.decider;

import com.google.common.collect.Sets;
import java.util.Map;
import java.util.Set;
import org.elasticsearch.ElasticsearchParseException;
import org.elasticsearch.client.Client;
import org.elasticsearch.cluster.ClusterInfo;
import org.elasticsearch.cluster.ClusterInfoService;
import org.elasticsearch.cluster.DiskUsage;
import org.elasticsearch.cluster.EmptyClusterInfoService;
import org.elasticsearch.cluster.routing.RoutingNode;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.routing.ShardRoutingState;
import org.elasticsearch.cluster.routing.allocation.RoutingAllocation;
import org.elasticsearch.cluster.routing.allocation.decider.AllocationDecider;
import org.elasticsearch.cluster.routing.allocation.decider.Decision;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.unit.RatioValue;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.node.settings.NodeSettingsService;

public class DiskThresholdDecider
extends AllocationDecider {
    public static final String NAME = "disk_threshold";
    private volatile Double freeDiskThresholdLow;
    private volatile Double freeDiskThresholdHigh;
    private volatile ByteSizeValue freeBytesThresholdLow;
    private volatile ByteSizeValue freeBytesThresholdHigh;
    private volatile boolean includeRelocations;
    private volatile boolean enabled;
    private volatile TimeValue rerouteInterval;
    public static final String CLUSTER_ROUTING_ALLOCATION_DISK_THRESHOLD_ENABLED = "cluster.routing.allocation.disk.threshold_enabled";
    public static final String CLUSTER_ROUTING_ALLOCATION_LOW_DISK_WATERMARK = "cluster.routing.allocation.disk.watermark.low";
    public static final String CLUSTER_ROUTING_ALLOCATION_HIGH_DISK_WATERMARK = "cluster.routing.allocation.disk.watermark.high";
    public static final String CLUSTER_ROUTING_ALLOCATION_INCLUDE_RELOCATIONS = "cluster.routing.allocation.disk.include_relocations";
    public static final String CLUSTER_ROUTING_ALLOCATION_REROUTE_INTERVAL = "cluster.routing.allocation.disk.reroute_interval";

    public DiskThresholdDecider(Settings settings) {
        this(settings, new NodeSettingsService(settings), EmptyClusterInfoService.INSTANCE, null);
    }

    @Inject
    public DiskThresholdDecider(Settings settings, NodeSettingsService nodeSettingsService, ClusterInfoService infoService, Client client) {
        super(settings);
        String lowWatermark = settings.get(CLUSTER_ROUTING_ALLOCATION_LOW_DISK_WATERMARK, "85%");
        String highWatermark = settings.get(CLUSTER_ROUTING_ALLOCATION_HIGH_DISK_WATERMARK, "90%");
        if (!this.validWatermarkSetting(lowWatermark, CLUSTER_ROUTING_ALLOCATION_LOW_DISK_WATERMARK)) {
            throw new ElasticsearchParseException("unable to parse low watermark [{}]", lowWatermark);
        }
        if (!this.validWatermarkSetting(highWatermark, CLUSTER_ROUTING_ALLOCATION_HIGH_DISK_WATERMARK)) {
            throw new ElasticsearchParseException("unable to parse high watermark [{}]", highWatermark);
        }
        this.freeDiskThresholdLow = 100.0 - this.thresholdPercentageFromWatermark(lowWatermark);
        this.freeDiskThresholdHigh = 100.0 - this.thresholdPercentageFromWatermark(highWatermark);
        this.freeBytesThresholdLow = this.thresholdBytesFromWatermark(lowWatermark, CLUSTER_ROUTING_ALLOCATION_LOW_DISK_WATERMARK);
        this.freeBytesThresholdHigh = this.thresholdBytesFromWatermark(highWatermark, CLUSTER_ROUTING_ALLOCATION_HIGH_DISK_WATERMARK);
        this.includeRelocations = settings.getAsBoolean(CLUSTER_ROUTING_ALLOCATION_INCLUDE_RELOCATIONS, (Boolean)true);
        this.rerouteInterval = settings.getAsTime(CLUSTER_ROUTING_ALLOCATION_REROUTE_INTERVAL, TimeValue.timeValueSeconds(60L));
        this.enabled = settings.getAsBoolean(CLUSTER_ROUTING_ALLOCATION_DISK_THRESHOLD_ENABLED, (Boolean)true);
        nodeSettingsService.addListener(new ApplySettings());
        infoService.addListener(new DiskListener(client));
    }

    ApplySettings newApplySettings() {
        return new ApplySettings();
    }

    public Double getFreeDiskThresholdLow() {
        return this.freeDiskThresholdLow;
    }

    public Double getFreeDiskThresholdHigh() {
        return this.freeDiskThresholdHigh;
    }

    public Double getUsedDiskThresholdLow() {
        return 100.0 - this.freeDiskThresholdLow;
    }

    public Double getUsedDiskThresholdHigh() {
        return 100.0 - this.freeDiskThresholdHigh;
    }

    public ByteSizeValue getFreeBytesThresholdLow() {
        return this.freeBytesThresholdLow;
    }

    public ByteSizeValue getFreeBytesThresholdHigh() {
        return this.freeBytesThresholdHigh;
    }

    public boolean isIncludeRelocations() {
        return this.includeRelocations;
    }

    public boolean isEnabled() {
        return this.enabled;
    }

    public TimeValue getRerouteInterval() {
        return this.rerouteInterval;
    }

    public static long sizeOfRelocatingShards(RoutingNode node, ClusterInfo clusterInfo, boolean subtractShardsMovingAway, String dataPath) {
        long totalSize = 0L;
        for (ShardRouting routing : node.shardsWithState(ShardRoutingState.RELOCATING, ShardRoutingState.INITIALIZING)) {
            String actualPath = clusterInfo.getDataPath(routing);
            if (!dataPath.equals(actualPath)) continue;
            if (routing.initializing() && routing.relocatingNodeId() != null) {
                totalSize += DiskThresholdDecider.getShardSize(routing, clusterInfo);
                continue;
            }
            if (!subtractShardsMovingAway || !routing.relocating()) continue;
            totalSize -= DiskThresholdDecider.getShardSize(routing, clusterInfo);
        }
        return totalSize;
    }

    static long getShardSize(ShardRouting routing, ClusterInfo clusterInfo) {
        Long shardSize = clusterInfo.getShardSize(routing);
        return shardSize == null ? 0L : shardSize;
    }

    @Override
    public Decision canAllocate(ShardRouting shardRouting, RoutingNode node, RoutingAllocation allocation) {
        boolean primaryHasBeenAllocated;
        ClusterInfo clusterInfo = allocation.clusterInfo();
        Map<String, DiskUsage> usages = clusterInfo.getNodeMostAvailableDiskUsages();
        Decision decision = this.earlyTerminate(allocation, usages);
        if (decision != null) {
            return decision;
        }
        double usedDiskThresholdLow = 100.0 - this.freeDiskThresholdLow;
        double usedDiskThresholdHigh = 100.0 - this.freeDiskThresholdHigh;
        DiskUsage usage = this.getDiskUsage(node, allocation, usages);
        double freeDiskPercentage = usage.getFreeDiskAsPercentage();
        double usedDiskPercentage = usage.getUsedDiskAsPercentage();
        long freeBytes = usage.getFreeBytes();
        if (this.logger.isTraceEnabled()) {
            this.logger.trace("node [{}] has {}% used disk", node.nodeId(), usedDiskPercentage);
        }
        boolean bl = primaryHasBeenAllocated = shardRouting.primary() && shardRouting.allocatedPostIndexCreate();
        if (freeBytes < this.freeBytesThresholdLow.bytes()) {
            if (!shardRouting.primary() || shardRouting.primary() && primaryHasBeenAllocated) {
                if (this.logger.isDebugEnabled()) {
                    this.logger.debug("less than the required {} free bytes threshold ({} bytes free) on node {}, preventing allocation", this.freeBytesThresholdLow, freeBytes, node.nodeId());
                }
                return allocation.decision(Decision.NO, NAME, "less than required [%s] free on node, free: [%s]", this.freeBytesThresholdLow, new ByteSizeValue(freeBytes));
            }
            if (freeBytes > this.freeBytesThresholdHigh.bytes()) {
                if (this.logger.isDebugEnabled()) {
                    this.logger.debug("less than the required {} free bytes threshold ({} bytes free) on node {}, but allowing allocation because primary has never been allocated", this.freeBytesThresholdLow, freeBytes, node.nodeId());
                }
                return allocation.decision(Decision.YES, NAME, "primary has never been allocated before", new Object[0]);
            }
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("less than the required {} free bytes threshold ({} bytes free) on node {}, preventing allocation even though primary has never been allocated", this.freeBytesThresholdHigh, freeBytes, node.nodeId());
            }
            return allocation.decision(Decision.NO, NAME, "less than required [%s] free on node, free: [%s]", this.freeBytesThresholdHigh, new ByteSizeValue(freeBytes));
        }
        if (freeDiskPercentage < this.freeDiskThresholdLow) {
            if (!shardRouting.primary() || shardRouting.primary() && primaryHasBeenAllocated) {
                if (this.logger.isDebugEnabled()) {
                    this.logger.debug("more than the allowed {} used disk threshold ({} used) on node [{}], preventing allocation", Strings.format1Decimals(usedDiskThresholdLow, "%"), Strings.format1Decimals(usedDiskPercentage, "%"), node.nodeId());
                }
                return allocation.decision(Decision.NO, NAME, "more than allowed [%s%%] used disk on node, free: [%s%%]", usedDiskThresholdLow, freeDiskPercentage);
            }
            if (freeDiskPercentage > this.freeDiskThresholdHigh) {
                if (this.logger.isDebugEnabled()) {
                    this.logger.debug("more than the allowed {} used disk threshold ({} used) on node [{}], but allowing allocation because primary has never been allocated", Strings.format1Decimals(usedDiskThresholdLow, "%"), Strings.format1Decimals(usedDiskPercentage, "%"), node.nodeId());
                }
                return allocation.decision(Decision.YES, NAME, "primary has never been allocated before", new Object[0]);
            }
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("less than the required {} free bytes threshold ({} bytes free) on node {}, preventing allocation even though primary has never been allocated", Strings.format1Decimals(this.freeDiskThresholdHigh, "%"), Strings.format1Decimals(freeDiskPercentage, "%"), node.nodeId());
            }
            return allocation.decision(Decision.NO, NAME, "more than allowed [%s%%] used disk on node, free: [%s%%]", usedDiskThresholdHigh, freeDiskPercentage);
        }
        long shardSize = DiskThresholdDecider.getShardSize(shardRouting, allocation.clusterInfo());
        double freeSpaceAfterShard = this.freeDiskPercentageAfterShardAssigned(usage, shardSize);
        long freeBytesAfterShard = freeBytes - shardSize;
        if (freeBytesAfterShard < this.freeBytesThresholdHigh.bytes()) {
            this.logger.warn("after allocating, node [{}] would have less than the required {} free bytes threshold ({} bytes free), preventing allocation", node.nodeId(), this.freeBytesThresholdHigh, freeBytesAfterShard);
            return allocation.decision(Decision.NO, NAME, "after allocation less than required [%s] free on node, free: [%s]", this.freeBytesThresholdLow, new ByteSizeValue(freeBytesAfterShard));
        }
        if (freeSpaceAfterShard < this.freeDiskThresholdHigh) {
            this.logger.warn("after allocating, node [{}] would have more than the allowed {} free disk threshold ({} free), preventing allocation", node.nodeId(), Strings.format1Decimals(this.freeDiskThresholdHigh, "%"), Strings.format1Decimals(freeSpaceAfterShard, "%"));
            return allocation.decision(Decision.NO, NAME, "after allocation more than allowed [%s%%] used disk on node, free: [%s%%]", usedDiskThresholdLow, freeSpaceAfterShard);
        }
        return allocation.decision(Decision.YES, NAME, "enough disk for shard on node, free: [%s]", new ByteSizeValue(freeBytes));
    }

    @Override
    public Decision canRemain(ShardRouting shardRouting, RoutingNode node, RoutingAllocation allocation) {
        if (!shardRouting.currentNodeId().equals(node.nodeId())) {
            throw new IllegalArgumentException("Shard [" + shardRouting + "] is not allocated on node: [" + node.nodeId() + "]");
        }
        ClusterInfo clusterInfo = allocation.clusterInfo();
        Map<String, DiskUsage> usages = clusterInfo.getNodeLeastAvailableDiskUsages();
        Decision decision = this.earlyTerminate(allocation, usages);
        if (decision != null) {
            return decision;
        }
        DiskUsage usage = this.getDiskUsage(node, allocation, usages);
        String dataPath = clusterInfo.getDataPath(shardRouting);
        double freeDiskPercentage = usage.getFreeDiskAsPercentage();
        long freeBytes = usage.getFreeBytes();
        if (this.logger.isTraceEnabled()) {
            this.logger.trace("node [{}] has {}% free disk ({} bytes)", node.nodeId(), freeDiskPercentage, freeBytes);
        }
        if (dataPath == null || !usage.getPath().equals(dataPath)) {
            return allocation.decision(Decision.YES, NAME, "shard is not allocated on the most utilized disk", new Object[0]);
        }
        if (freeBytes < this.freeBytesThresholdHigh.bytes()) {
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("less than the required {} free bytes threshold ({} bytes free) on node {}, shard cannot remain", this.freeBytesThresholdHigh, freeBytes, node.nodeId());
            }
            return allocation.decision(Decision.NO, NAME, "after allocation less than required [%s] free on node, free: [%s]", this.freeBytesThresholdHigh, new ByteSizeValue(freeBytes));
        }
        if (freeDiskPercentage < this.freeDiskThresholdHigh) {
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("less than the required {}% free disk threshold ({}% free) on node {}, shard cannot remain", this.freeDiskThresholdHigh, freeDiskPercentage, node.nodeId());
            }
            return allocation.decision(Decision.NO, NAME, "after allocation less than required [%s%%] free disk on node, free: [%s%%]", this.freeDiskThresholdHigh, freeDiskPercentage);
        }
        return allocation.decision(Decision.YES, NAME, "enough disk for shard to remain on node, free: [%s]", new ByteSizeValue(freeBytes));
    }

    private DiskUsage getDiskUsage(RoutingNode node, RoutingAllocation allocation, Map<String, DiskUsage> usages) {
        ClusterInfo clusterInfo = allocation.clusterInfo();
        DiskUsage usage = usages.get(node.nodeId());
        if (usage == null) {
            usage = this.averageUsage(node, usages);
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("unable to determine disk usage for {}, defaulting to average across nodes [{} total] [{} free] [{}% free]", node.nodeId(), usage.getTotalBytes(), usage.getFreeBytes(), usage.getFreeDiskAsPercentage());
            }
        }
        if (this.includeRelocations) {
            long relocatingShardsSize = DiskThresholdDecider.sizeOfRelocatingShards(node, clusterInfo, true, usage.getPath());
            DiskUsage usageIncludingRelocations = new DiskUsage(node.nodeId(), node.node().name(), usage.getPath(), usage.getTotalBytes(), usage.getFreeBytes() - relocatingShardsSize);
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("usage without relocations: {}", usage);
                this.logger.trace("usage with relocations: [{} bytes] {}", relocatingShardsSize, usageIncludingRelocations);
            }
            usage = usageIncludingRelocations;
        }
        return usage;
    }

    public DiskUsage averageUsage(RoutingNode node, Map<String, DiskUsage> usages) {
        if (usages.size() == 0) {
            return new DiskUsage(node.nodeId(), node.node().name(), "_na_", 0L, 0L);
        }
        long totalBytes = 0L;
        long freeBytes = 0L;
        for (DiskUsage du : usages.values()) {
            totalBytes += du.getTotalBytes();
            freeBytes += du.getFreeBytes();
        }
        return new DiskUsage(node.nodeId(), node.node().name(), "_na_", totalBytes / (long)usages.size(), freeBytes / (long)usages.size());
    }

    public double freeDiskPercentageAfterShardAssigned(DiskUsage usage, Long shardSize) {
        shardSize = shardSize == null ? 0L : shardSize;
        DiskUsage newUsage = new DiskUsage(usage.getNodeId(), usage.getNodeName(), usage.getPath(), usage.getTotalBytes(), usage.getFreeBytes() - shardSize);
        return newUsage.getFreeDiskAsPercentage();
    }

    public double thresholdPercentageFromWatermark(String watermark) {
        try {
            return RatioValue.parseRatioValue(watermark).getAsPercent();
        }
        catch (ElasticsearchParseException ex) {
            return 100.0;
        }
    }

    public ByteSizeValue thresholdBytesFromWatermark(String watermark, String settingName) {
        try {
            return ByteSizeValue.parseBytesSizeValue(watermark, settingName);
        }
        catch (ElasticsearchParseException ex) {
            return ByteSizeValue.parseBytesSizeValue("0b", settingName);
        }
    }

    public boolean validWatermarkSetting(String watermark, String settingName) {
        try {
            RatioValue.parseRatioValue(watermark);
            return true;
        }
        catch (ElasticsearchParseException e) {
            try {
                ByteSizeValue.parseBytesSizeValue(watermark, settingName);
                return true;
            }
            catch (ElasticsearchParseException ex) {
                return false;
            }
        }
    }

    private Decision earlyTerminate(RoutingAllocation allocation, Map<String, DiskUsage> usages) {
        if (!this.enabled) {
            return allocation.decision(Decision.YES, NAME, "disk threshold decider disabled", new Object[0]);
        }
        if (allocation.nodes().dataNodes().size() <= 1) {
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("only a single data node is present, allowing allocation", new Object[0]);
            }
            return allocation.decision(Decision.YES, NAME, "only a single data node is present", new Object[0]);
        }
        ClusterInfo clusterInfo = allocation.clusterInfo();
        if (clusterInfo == null) {
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("cluster info unavailable for disk threshold decider, allowing allocation.", new Object[0]);
            }
            return allocation.decision(Decision.YES, NAME, "cluster info unavailable", new Object[0]);
        }
        if (usages.isEmpty()) {
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("unable to determine disk usages for disk-aware allocation, allowing allocation", new Object[0]);
            }
            return allocation.decision(Decision.YES, NAME, "disk usages unavailable", new Object[0]);
        }
        return null;
    }

    class DiskListener
    implements ClusterInfoService.Listener {
        private final Client client;
        private final Set<String> nodeHasPassedWatermark = Sets.newConcurrentHashSet();
        private long lastRunNS;

        DiskListener(Client client) {
            this.client = client;
        }

        private void warnAboutDiskIfNeeded(DiskUsage usage) {
            if (usage.getFreeBytes() < DiskThresholdDecider.this.freeBytesThresholdHigh.bytes()) {
                DiskThresholdDecider.this.logger.warn("high disk watermark [{}] exceeded on {}, shards will be relocated away from this node", DiskThresholdDecider.this.freeBytesThresholdHigh, usage);
            } else if (usage.getFreeBytes() < DiskThresholdDecider.this.freeBytesThresholdLow.bytes()) {
                DiskThresholdDecider.this.logger.info("low disk watermark [{}] exceeded on {}, replicas will not be assigned to this node", DiskThresholdDecider.this.freeBytesThresholdLow, usage);
            }
            if (usage.getFreeDiskAsPercentage() < DiskThresholdDecider.this.freeDiskThresholdHigh) {
                DiskThresholdDecider.this.logger.warn("high disk watermark [{}] exceeded on {}, shards will be relocated away from this node", Strings.format1Decimals(100.0 - DiskThresholdDecider.this.freeDiskThresholdHigh, "%"), usage);
            } else if (usage.getFreeDiskAsPercentage() < DiskThresholdDecider.this.freeDiskThresholdLow) {
                DiskThresholdDecider.this.logger.info("low disk watermark [{}] exceeded on {}, replicas will not be assigned to this node", Strings.format1Decimals(100.0 - DiskThresholdDecider.this.freeDiskThresholdLow, "%"), usage);
            }
        }

        @Override
        public void onNewInfo(ClusterInfo info) {
            Map<String, DiskUsage> usages = info.getNodeLeastAvailableDiskUsages();
            if (usages != null) {
                boolean reroute = false;
                String explanation = "";
                Set<String> nodes = usages.keySet();
                for (String string : this.nodeHasPassedWatermark) {
                    if (nodes.contains(string)) continue;
                    this.nodeHasPassedWatermark.remove(string);
                }
                for (Map.Entry entry : usages.entrySet()) {
                    String node = (String)entry.getKey();
                    DiskUsage usage = (DiskUsage)entry.getValue();
                    this.warnAboutDiskIfNeeded(usage);
                    if (usage.getFreeBytes() < DiskThresholdDecider.this.freeBytesThresholdHigh.bytes() || usage.getFreeDiskAsPercentage() < DiskThresholdDecider.this.freeDiskThresholdHigh) {
                        if (System.nanoTime() - this.lastRunNS > DiskThresholdDecider.this.rerouteInterval.nanos()) {
                            this.lastRunNS = System.nanoTime();
                            reroute = true;
                            explanation = "high disk watermark exceeded on one or more nodes";
                        } else {
                            DiskThresholdDecider.this.logger.debug("high disk watermark exceeded on {} but an automatic reroute has occurred in the last [{}], skipping reroute", node, DiskThresholdDecider.this.rerouteInterval);
                        }
                        this.nodeHasPassedWatermark.add(node);
                        continue;
                    }
                    if (usage.getFreeBytes() < DiskThresholdDecider.this.freeBytesThresholdLow.bytes() || usage.getFreeDiskAsPercentage() < DiskThresholdDecider.this.freeDiskThresholdLow) {
                        this.nodeHasPassedWatermark.add(node);
                        continue;
                    }
                    if (!this.nodeHasPassedWatermark.contains(node)) continue;
                    if (System.nanoTime() - this.lastRunNS > DiskThresholdDecider.this.rerouteInterval.nanos()) {
                        this.lastRunNS = System.nanoTime();
                        reroute = true;
                        explanation = "one or more nodes has gone under the high or low watermark";
                        this.nodeHasPassedWatermark.remove(node);
                        continue;
                    }
                    DiskThresholdDecider.this.logger.debug("{} has gone below a disk threshold, but an automatic reroute has occurred in the last [{}], skipping reroute", node, DiskThresholdDecider.this.rerouteInterval);
                }
                if (reroute) {
                    DiskThresholdDecider.this.logger.info("rerouting shards: [{}]", explanation);
                    this.client.admin().cluster().prepareReroute().execute();
                }
            }
        }
    }

    class ApplySettings
    implements NodeSettingsService.Listener {
        ApplySettings() {
        }

        @Override
        public void onRefreshSettings(Settings settings) {
            String newLowWatermark = settings.get(DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_LOW_DISK_WATERMARK, null);
            String newHighWatermark = settings.get(DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_HIGH_DISK_WATERMARK, null);
            Boolean newRelocationsSetting = settings.getAsBoolean(DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_INCLUDE_RELOCATIONS, null);
            Boolean newEnableSetting = settings.getAsBoolean(DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_DISK_THRESHOLD_ENABLED, null);
            TimeValue newRerouteInterval = settings.getAsTime(DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_REROUTE_INTERVAL, null);
            if (newEnableSetting != null) {
                DiskThresholdDecider.this.logger.info("updating [{}] from [{}] to [{}]", DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_DISK_THRESHOLD_ENABLED, DiskThresholdDecider.this.enabled, newEnableSetting);
                DiskThresholdDecider.this.enabled = newEnableSetting;
            }
            if (newRelocationsSetting != null) {
                DiskThresholdDecider.this.logger.info("updating [{}] from [{}] to [{}]", DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_INCLUDE_RELOCATIONS, DiskThresholdDecider.this.includeRelocations, newRelocationsSetting);
                DiskThresholdDecider.this.includeRelocations = newRelocationsSetting;
            }
            if (newLowWatermark != null) {
                if (!DiskThresholdDecider.this.validWatermarkSetting(newLowWatermark, DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_LOW_DISK_WATERMARK)) {
                    throw new ElasticsearchParseException("unable to parse low watermark [{}]", newLowWatermark);
                }
                DiskThresholdDecider.this.logger.info("updating [{}] to [{}]", DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_LOW_DISK_WATERMARK, newLowWatermark);
                DiskThresholdDecider.this.freeDiskThresholdLow = 100.0 - DiskThresholdDecider.this.thresholdPercentageFromWatermark(newLowWatermark);
                DiskThresholdDecider.this.freeBytesThresholdLow = DiskThresholdDecider.this.thresholdBytesFromWatermark(newLowWatermark, DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_LOW_DISK_WATERMARK);
            }
            if (newHighWatermark != null) {
                if (!DiskThresholdDecider.this.validWatermarkSetting(newHighWatermark, DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_HIGH_DISK_WATERMARK)) {
                    throw new ElasticsearchParseException("unable to parse high watermark [{}]", newHighWatermark);
                }
                DiskThresholdDecider.this.logger.info("updating [{}] to [{}]", DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_HIGH_DISK_WATERMARK, newHighWatermark);
                DiskThresholdDecider.this.freeDiskThresholdHigh = 100.0 - DiskThresholdDecider.this.thresholdPercentageFromWatermark(newHighWatermark);
                DiskThresholdDecider.this.freeBytesThresholdHigh = DiskThresholdDecider.this.thresholdBytesFromWatermark(newHighWatermark, DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_HIGH_DISK_WATERMARK);
            }
            if (newRerouteInterval != null) {
                DiskThresholdDecider.this.logger.info("updating [{}] to [{}]", DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_REROUTE_INTERVAL, newRerouteInterval);
                DiskThresholdDecider.this.rerouteInterval = newRerouteInterval;
            }
        }
    }
}

