package gnu.trove;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

// made "main method runnable" instead of JUnit runnable to avoid Junit dependency
public class MapTest {
    public static void main(String[] args) {
        testTIntObjectHashMap();
        testTHashMap();
        testClone();
    }

    private static void testTIntObjectHashMap() {
        TIntObjectHashMap<String> map = new TIntObjectHashMap<String>();
        // init correctly
        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity());
        assertTrue(map._size == 0);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // first put placed at one
        int key = 1;
        assertTrue(map._hashingStrategy.computeHashCode(key) == 1);
        String value = "1";
        String prev = map.put(key, value);
        assertTrue(prev == null);

        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-1);
        assertTrue(map._size == 1);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        int index = map.index(key);
        assertTrue(index == 1);
        assertTrue(map.size() == 1);
        assertTrue(((Object[])map._values)[index] == value);

        // remove leaves cell "removed"
        String remove = map.remove(key);
        assertTrue(remove == value);
        assertTrue(map.size() == 0);
        assertTrue(map.capacity() == 7);
        assertTrue(((Object[])map._values)[index] == TObjectHash.REMOVED);

        assertTrue(map._deadkeys == 1);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-1);
        assertTrue(map._size == 0);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // reinsert after remove must reuse the cell
        prev = map.put(key, value);
        assertTrue(prev == null);
        index = map.index(key);
        assertTrue(index == 1);
        assertTrue(map.size() == 1);
        assertTrue(((Object[])map._values)[index] == value);

        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-1);
        assertTrue(map._size == 1);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));


        // second put double hashes
        int key2 = 8;
        assertTrue(map._hashingStrategy.computeHashCode(key2) == 8); // conflict
        prev = map.put(key2, value);
        assertTrue(prev == null);
        index = map.index(key2);
        assertTrue(index == 4);
        assertTrue(map.size() == 2);
        assertTrue(((Object[])map._values)[index] == value);

        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-2);
        assertTrue(map._size == 2);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));


        // remove of the first key leaves gap
        index = map.index(key);
        remove = map.remove(key);
        assertTrue(remove == value);
        assertTrue(map.size() == 1);
        assertTrue(((Object[])map._values)[index] == TObjectHash.REMOVED);

        assertTrue(map._deadkeys == 1);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-2);
        assertTrue(map._size == 1);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // search for the second key must probe through the gap
        String v = map.get(key2);
        assertTrue(v == value); // traverse through REMOVED

        // second remove leaves two removed cells
        index = map.index(key2);
        assertTrue(index == 4);
        remove = map.remove(key2);
        assertTrue(remove == value);
        assertTrue(map.size() == 0);
        assertTrue(map.capacity() == 7);
        assertTrue(((Object[])map._values)[index] == TObjectHash.REMOVED);

        assertTrue(map._deadkeys == 2);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-2);
        assertTrue(map._size == 0);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // clear resets everything
        map.clear();
        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity());
        assertTrue(map._size == 0);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // prepare for rehash
        map.put(1, value);
        map.put(2, value);
        map.put(3, value);
        map.put(4, value);
        map.put(5, value);

        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-5);
        assertTrue(map._size == 5);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // nothing changes
        map.put(5, value);
        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-5);
        assertTrue(map._size == 5);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // rehash
        map.put(6, value);
        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 17);
        assertTrue(map._free == map.capacity()-6);
        assertTrue(map._size == 6);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));
    }

    private static void testTHashMap() {
        THashMap<Integer,String> map = new THashMap<Integer, String>();
        // init correctly
        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity());
        assertTrue(map._size == 0);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // first put placed at one
        int key = 1;
        assertTrue(map._hashingStrategy.computeHashCode(key) == 1);
        String value = "1";
        String prev = map.put(key, value);
        assertTrue(prev == null);

        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-1);
        assertTrue(map._size == 1);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        int index = map.index(key);
        assertTrue(index == 1);
        assertTrue(map.size() == 1);
        assertTrue(((Object[])map._values)[index] == value);

        // remove leaves cell "removed"
        String remove = map.remove(key);
        assertTrue(remove == value);
        assertTrue(map.size() == 0);
        assertTrue(map.capacity() == 7);
        assertTrue(((Object[])map._set)[index] == TObjectHash.REMOVED);

        assertTrue(map._deadkeys == 1);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-1);
        assertTrue(map._size == 0);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // reinsert after remove must reuse the cell
        prev = map.put(key, value);
        assertTrue(prev == null);
        index = map.index(key);
        assertTrue(index == 1);
        assertTrue(map.size() == 1);
        assertTrue(((Object[])map._values)[index] == value);

        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-1);
        assertTrue(map._size == 1);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));


        // second put double hashes
        int key2 = 8;
        assertTrue(map._hashingStrategy.computeHashCode(key2) == 8); // conflict
        prev = map.put(key2, value);
        assertTrue(prev == null);
        index = map.index(key2);
        assertTrue(index == 4);
        assertTrue(map.size() == 2);
        assertTrue(((Object[])map._values)[index] == value);

        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-2);
        assertTrue(map._size == 2);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));


        // remove of the first key leaves gap
        index = map.index(key);
        remove = map.remove(key);
        assertTrue(remove == value);
        assertTrue(map.size() == 1);
        assertTrue(((Object[])map._set)[index] == TObjectHash.REMOVED);

        assertTrue(map._deadkeys == 1);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-2);
        assertTrue(map._size == 1);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // search for the second key must probe through the gap
        String v = map.get(key2);
        assertTrue(v == value); // traverse through REMOVED

        // second remove leaves two removed cells
        index = map.index(key2);
        assertTrue(index == 4);
        remove = map.remove(key2);
        assertTrue(remove == value);
        assertTrue(map.size() == 0);
        assertTrue(map.capacity() == 7);
        assertTrue(((Object[])map._set)[index] == TObjectHash.REMOVED);

        assertTrue(map._deadkeys == 2);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-2);
        assertTrue(map._size == 0);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int) (map.capacity() * map._loadFactor));

        // clear resets everything
        map.clear();
        // _deadkeys was not reset due to optimisation "do nothing on empty map .clear()"
        assertTrue(map._deadkeys == 2);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-2);
        assertTrue(map._size == 0);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // prepare for rehash
        map.put(1, value);
        map.put(2, value);
        map.put(3, value);
        map.put(4, value);
        map.put(5, value);

        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-5);
        assertTrue(map._size == 5);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // nothing changes
        map.put(5, "xxx");
        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-5);
        assertTrue(map._size == 5);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // rehash
        map.put(6, value);
        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 17);
        assertTrue(map._free == map.capacity()-6);
        assertTrue(map._size == 6);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));
    }

    public static void testClone() {
      TIntObjectHashMap<int[]> map = new TIntObjectHashMap<int[]>();
      map.put(0, new int[2]);
      map.put(1, new int[2]);

      TIntObjectHashMap<int[]> clone = map.clone();
      assertEquals(clone.size(), 2);
      int[] keys = clone.keys();
      assertEquals(keys.length, 2);
      Set set01 = new HashSet(Arrays.asList(0, 1));
      Set keySet = new HashSet(Arrays.asList(keys[0], keys[1]));
      assertEquals(set01, keySet);

      map.clear();

      assertEquals(clone.size(), 2);
      keys = clone.keys();
      assertEquals(keys.length, 2);
      keySet = new HashSet(Arrays.asList(keys[0], keys[1]));
      assertEquals(set01, keySet);
    }

    private static void assertEquals(int a, int b) {
        assertTrue(a == b);
    }
    private static void assertEquals(Object a, Object b) {
        assertTrue(a.equals(b));
    }


    private static void assertTrue(boolean b) {
        if (!b) throw new AssertionError();
    }
}