package vtk;

import java.io.File;

/**
 * Enum used to load native library more easily. If you don't want to set the
 * specific environment variable you can provide the path of the directory that
 * contains the VTK library through a Java Property. Like in the following
 * command line:
 *
 * > java -cp vtk.jar -Dvtk.lib.dir=.../vtk-libs vtk.sample.SimpleVTK
 *
 * The directory .../vtk-libs must contain the so/dll/dylib + the jnilib files
 *
 * @author sebastien jourdain - sebastien.jourdain@kitware.com
 */
public enum vtkNativeLibrary {

    VTKWEBCORE("vtkWebCoreJava", false),
    VTKWEBGLEXPORTER("vtkWebGLExporterJava", false),
    VTKFILTERSSELECTION("vtkFiltersSelectionJava", true),
    VTKFILTERSPROGRAMMABLE("vtkFiltersProgrammableJava", true),
    VTKFILTERSGENERIC("vtkFiltersGenericJava", true),
    VTKFILTERSMODELING("vtkFiltersModelingJava", true),
    VTKFILTERSSTATISTICS("vtkFiltersStatisticsJava", true),
    VTKFILTERSVERDICT("vtkFiltersVerdictJava", true),
    VTKFILTERSHYPERTREE("vtkFiltersHyperTreeJava", true),
    VTKFILTERSSTATISTICSGNUR("vtkFiltersStatisticsGnuRJava", false),
    VTKFILTERSAMR("vtkFiltersAMRJava", true),
    VTKFILTERSSOURCES("vtkFiltersSourcesJava", true),
    VTKFILTERSPARALLELIMAGING("vtkFiltersParallelImagingJava", true),
    VTKFILTERSREEBGRAPH("vtkFiltersReebGraphJava", true),
    VTKFILTERSPARALLELGEOMETRY("vtkFiltersParallelGeometryJava", true),
    VTKFILTERSPARALLELMPI("vtkFiltersParallelMPIJava", true),
    VTKFILTERSHYBRID("vtkFiltersHybridJava", true),
    VTKFILTERSIMAGING("vtkFiltersImagingJava", true),
    VTKFILTERSGEOMETRY("vtkFiltersGeometryJava", true),
    VTKFILTERSSMP("vtkFiltersSMPJava", true),
    VTKFILTERSFLOWPATHS("vtkFiltersFlowPathsJava", true),
    VTKFILTERSCORE("vtkFiltersCoreJava", true),
    VTKFILTERSPYTHON("vtkFiltersPythonJava", true),
    VTKFILTERSPARALLEL("vtkFiltersParallelJava", true),
    VTKFILTERSPARALLELSTATISTICS("vtkFiltersParallelStatisticsJava", true),
    VTKFILTERSEXTRACTION("vtkFiltersExtractionJava", true),
    VTKFILTERSTEXTURE("vtkFiltersTextureJava", true),
    VTKFILTERSGENERAL("vtkFiltersGeneralJava", true),
    VTKFILTERSPARALLELFLOWPATHS("vtkFiltersParallelFlowPathsJava", true),
    VTKCOMMONCOLOR("vtkCommonColorJava", true),
    VTKCOMMONEXECUTIONMODEL("vtkCommonExecutionModelJava", true),
    VTKCOMMONMISC("vtkCommonMiscJava", true),
    VTKCOMMONDATAMODEL("vtkCommonDataModelJava", true),
    VTKCOMMONSYSTEM("vtkCommonSystemJava", true),
    VTKCOMMONCORE("vtkCommonCoreJava", true),
    VTKCOMMONMATH("vtkCommonMathJava", true),
    VTKCOMMONCOMPUTATIONALGEOMETRY("vtkCommonComputationalGeometryJava", true),
    VTKCOMMONTRANSFORMS("vtkCommonTransformsJava", true),
    VTKPYTHONINTERPRETER("vtkPythonInterpreterJava", true),
    VTKIOVPIC("vtkIOVPICJava", true),
    VTKIOIMPORT("vtkIOImportJava", true),
    VTKIOLSDYNA("vtkIOLSDynaJava", true),
    VTKIOXDMF3("vtkIOXdmf3Java", true),
    VTKIOPARALLELEXODUS("vtkIOParallelExodusJava", true),
    VTKIOXML("vtkIOXMLJava", true),
    VTKIOADIOS("vtkIOADIOSJava", false),
    VTKIOPLY("vtkIOPLYJava", true),
    VTKIOMPIIMAGE("vtkIOMPIImageJava", true),
    VTKIOENSIGHT("vtkIOEnSightJava", true),
    VTKIOSQL("vtkIOSQLJava", true),
    VTKIOAMR("vtkIOAMRJava", true),
    VTKIOIMAGE("vtkIOImageJava", true),
    VTKIOXDMF2("vtkIOXdmf2Java", true),
    VTKIOMYSQL("vtkIOMySQLJava", true),
    VTKIOLEGACY("vtkIOLegacyJava", true),
    VTKIOFFMPEG("vtkIOFFMPEGJava", true),
    VTKIOEXODUS("vtkIOExodusJava", true),
    VTKIOGEOMETRY("vtkIOGeometryJava", true),
    VTKIOMPIPARALLEL("vtkIOMPIParallelJava", true),
    VTKIOEXPORT("vtkIOExportJava", true),
    VTKIOPARALLELXML("vtkIOParallelXMLJava", true),
    VTKIOPARALLELLSDYNA("vtkIOParallelLSDynaJava", true),
    VTKIOCORE("vtkIOCoreJava", true),
    VTKIOPARALLEL("vtkIOParallelJava", true),
    VTKIOINFOVIS("vtkIOInfovisJava", true),
    VTKIOPOSTGRESQL("vtkIOPostgreSQLJava", true),
    VTKIOMINC("vtkIOMINCJava", true),
    VTKIOGDAL("vtkIOGDALJava", true),
    VTKIOMOVIE("vtkIOMovieJava", true),
    VTKIONETCDF("vtkIONetCDFJava", true),
    VTKIOXMLPARSER("vtkIOXMLParserJava", true),
    VTKIOPARALLELNETCDF("vtkIOParallelNetCDFJava", true),
    VTKIOODBC("vtkIOODBCJava", true),
    VTKIOGEOJSON("vtkIOGeoJSONJava", true),
    VTKIOVIDEO("vtkIOVideoJava", true),
    VTKINTERACTIONIMAGE("vtkInteractionImageJava", true),
    VTKINTERACTIONSTYLE("vtkInteractionStyleJava", true),
    VTKINTERACTIONWIDGETS("vtkInteractionWidgetsJava", true),
    VTKGEOVISCORE("vtkGeovisCoreJava", true),
    VTKIMAGINGMORPHOLOGICAL("vtkImagingMorphologicalJava", true),
    VTKIMAGINGCOLOR("vtkImagingColorJava", true),
    VTKIMAGINGSTATISTICS("vtkImagingStatisticsJava", true),
    VTKIMAGINGSOURCES("vtkImagingSourcesJava", true),
    VTKIMAGINGHYBRID("vtkImagingHybridJava", true),
    VTKIMAGINGCORE("vtkImagingCoreJava", true),
    VTKIMAGINGMATH("vtkImagingMathJava", true),
    VTKIMAGINGFOURIER("vtkImagingFourierJava", true),
    VTKIMAGINGGENERAL("vtkImagingGeneralJava", true),
    VTKIMAGINGSTENCIL("vtkImagingStencilJava", true),
    VTKPARALLELCORE("vtkParallelCoreJava", true),
    VTKPARALLELMPI4PY("vtkParallelMPI4PyJava", true),
    VTKPARALLELMPI("vtkParallelMPIJava", true),
    VTKINFOVISLAYOUT("vtkInfovisLayoutJava", true),
    VTKINFOVISCORE("vtkInfovisCoreJava", true),
    VTKINFOVISPARALLEL("vtkInfovisParallelJava", false),
    VTKINFOVISBOOSTGRAPHALGORITHMS("vtkInfovisBoostGraphAlgorithmsJava", true),
    VTKTESTINGRENDERING("vtkTestingRenderingJava", true),
    VTKRENDERINGOPENGL("vtkRenderingOpenGLJava", true),
    VTKRENDERINGVOLUME("vtkRenderingVolumeJava", true),
    VTKRENDERINGQT("vtkRenderingQtJava", true),
    VTKRENDERINGVOLUMEOPENGL("vtkRenderingVolumeOpenGLJava", true),
    VTKRENDERINGLICOPENGL2("vtkRenderingLICOpenGL2Java", false),
    VTKRENDERINGLIC("vtkRenderingLICJava", true),
    VTKRENDERINGGL2PS("vtkRenderingGL2PSJava", true),
    VTKRENDERINGCONTEXT2D("vtkRenderingContext2DJava", true),
    VTKRENDERINGLOD("vtkRenderingLODJava", true),
    VTKRENDERINGIMAGE("vtkRenderingImageJava", true),
    VTKRENDERINGEXTERNAL("vtkRenderingExternalJava", false),
    VTKRENDERINGCONTEXTOPENGL2("vtkRenderingContextOpenGL2Java", false),
    VTKRENDERINGANNOTATION("vtkRenderingAnnotationJava", true),
    VTKRENDERINGFREETYPE("vtkRenderingFreeTypeJava", true),
    VTKRENDERINGPARALLELLIC("vtkRenderingParallelLICJava", true),
    VTKRENDERINGCORE("vtkRenderingCoreJava", true),
    VTKRENDERINGPARALLEL("vtkRenderingParallelJava", true),
    VTKRENDERINGOPENGL2("vtkRenderingOpenGL2Java", false),
    VTKRENDERINGVOLUMEAMR("vtkRenderingVolumeAMRJava", true),
    VTKRENDERINGLABEL("vtkRenderingLabelJava", true),
    VTKRENDERINGMATPLOTLIB("vtkRenderingMatplotlibJava", true),
    VTKRENDERINGCONTEXTOPENGL("vtkRenderingContextOpenGLJava", true),
    VTKRENDERINGVOLUMEOPENGL2("vtkRenderingVolumeOpenGL2Java", false),
    VTKACCELERATORSPISTON("vtkAcceleratorsPistonJava", false),
    VTKACCELERATORSDAX("vtkAcceleratorsDaxJava", false),
    VTKCHARTSCORE("vtkChartsCoreJava", true),
    VTKVIEWSCONTEXT2D("vtkViewsContext2DJava", true),
    VTKVIEWSGEOVIS("vtkViewsGeovisJava", true),
    VTKVIEWSCORE("vtkViewsCoreJava", true),
    VTKVIEWSINFOVIS("vtkViewsInfovisJava", true),
    VTKDOMAINSCHEMISTRYOPENGL2("vtkDomainsChemistryOpenGL2Java", false),
    VTKDOMAINSCHEMISTRY("vtkDomainsChemistryJava", true);


  /**
   * Try to load all library
   *
   * @return true if all library have been successfully loaded
   */
  public static boolean LoadAllNativeLibraries() {
    boolean isEveryThingLoaded = true;
    for (vtkNativeLibrary lib : values()) {
      try {
        if(lib.IsBuilt()) {
          lib.LoadLibrary();
        }
      } catch (UnsatisfiedLinkError e) {
        isEveryThingLoaded = false;
        e.printStackTrace();
      }
    }

    return isEveryThingLoaded;
  }

  /**
   * Load the set of given library and trows runtime exception if any given
   * library failed in the loading process
   *
   * @param nativeLibraries
   */
  public static void LoadNativeLibraries(vtkNativeLibrary... nativeLibraries) {
    for (vtkNativeLibrary lib : nativeLibraries) {
      lib.LoadLibrary();
    }
  }

  /**
   * Disable the pop-in vtkErrorWindow by writing the error to a log file.
   * If the provided logFile is null the default "vtkError.txt" file will be
   * used.
   *
   * @param logFile
   */
  public static void DisableOutputWindow(File logFile) {
    if(logFile == null) {
      logFile = new File("vtkError.txt");
    }
    vtkFileOutputWindow outputError = new vtkFileOutputWindow();
    outputError.SetFileName(logFile.getAbsolutePath());
    outputError.SetInstance(outputError);
  }

  private vtkNativeLibrary(String nativeLibraryName, boolean built) {
    this.nativeLibraryName = nativeLibraryName;
    this.loaded = false;
    this.built = built;
  }

  /**
   * Load the library and throws runtime exception if the library failed in
   * the loading process
   */
  public void LoadLibrary() throws UnsatisfiedLinkError {
    if (!loaded) {
      if (System.getProperty("vtk.lib.dir") != null) {
        File dir = new File(System.getProperty("vtk.lib.dir"));
        patchJavaLibraryPath(dir.getAbsolutePath());
        File libPath = new File(dir, System.mapLibraryName(nativeLibraryName));
        if (libPath.exists()) {
          try {
            Runtime.getRuntime().load(libPath.getAbsolutePath());
            loaded = true;
            return;
          } catch (UnsatisfiedLinkError e) {
            e.printStackTrace();
          }
        }
      }
      System.loadLibrary(nativeLibraryName);
    }
    loaded = true;
  }

  /**
   * @return true if the library has already been successfully loaded
   */
  public boolean IsLoaded() {
    return loaded;
  }

  /**
   * @return true if the module was enabled and therefore build
   */
  public boolean IsBuilt() {
    return built;
  }

  /**
   * @return the library name
   */
  public String GetLibraryName() {
    return nativeLibraryName;
  }

  private static void patchJavaLibraryPath(String vtkLibDir) {
    if (vtkLibDir != null) {
      String path_separator = System.getProperty("path.separator");
      String s = System.getProperty("java.library.path");
      if (!s.contains(vtkLibDir)) {
        s = s + path_separator + vtkLibDir;
        System.setProperty("java.library.path", s);
      }
    }
  }

  private String nativeLibraryName;
  private boolean loaded;
  private boolean built;
}
