/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.truffle.api.source;

import com.oracle.truffle.api.CompilerAsserts;
import com.oracle.truffle.api.source.LineLocation;
import com.oracle.truffle.api.source.SourceSection;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.io.UnsupportedEncodingException;
import java.lang.ref.WeakReference;
import java.net.URL;
import java.net.URLConnection;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.logging.Level;
import java.util.logging.Logger;

public abstract class Source {
    private static final Logger LOG = Logger.getLogger(Source.class.getName());
    private static final Map<String, WeakReference<Source>> nameToSource = new HashMap<String, WeakReference<Source>>();
    private static boolean fileCacheEnabled = true;
    private static final String NO_FASTPATH_SUBSOURCE_CREATION_MESSAGE = "do not create sub sources from compiled code";
    private String mimeType;
    private TextMap textMap;

    public static Source find(String name) {
        WeakReference<Source> nameRef = nameToSource.get(name);
        return nameRef == null ? null : (Source)nameRef.get();
    }

    public static Source fromFileName(String fileName, boolean reset) throws IOException {
        Source source;
        WeakReference<Source> nameRef = nameToSource.get(fileName);
        Source source2 = source = nameRef == null ? null : (Source)nameRef.get();
        if (source == null) {
            File file = new File(fileName);
            if (!file.canRead()) {
                throw new IOException("Can't read file " + fileName);
            }
            String path = file.getCanonicalPath();
            WeakReference<Source> pathRef = nameToSource.get(path);
            Source source3 = source = pathRef == null ? null : (Source)pathRef.get();
            if (source == null) {
                source = new FileSource(file, fileName, path);
                nameToSource.put(path, new WeakReference<Source>(source));
            }
        }
        if (reset) {
            source.reset();
        }
        return source;
    }

    public static Source fromFileName(String fileName) throws IOException {
        return Source.fromFileName(fileName, false);
    }

    public static Source fromFileName(CharSequence chars, String fileName) throws IOException {
        Source source;
        CompilerAsserts.neverPartOfCompilation("do not call Source.fromFileName from compiled code");
        assert (chars != null);
        WeakReference<Source> nameRef = nameToSource.get(fileName);
        Source source2 = source = nameRef == null ? null : (Source)nameRef.get();
        if (source == null) {
            File file = new File(fileName);
            String path = file.getCanonicalPath();
            WeakReference<Source> pathRef = nameToSource.get(path);
            Source source3 = source = pathRef == null ? null : (Source)pathRef.get();
            if (source == null) {
                source = new ClientManagedFileSource(file, fileName, path, chars);
                nameToSource.put(path, new WeakReference<Source>(source));
                return source;
            }
        }
        if (source instanceof ClientManagedFileSource) {
            ClientManagedFileSource modifiableSource = (ClientManagedFileSource)source;
            modifiableSource.setCode(chars);
            return modifiableSource;
        }
        throw new IOException("Attempt to modify contents of a file Source");
    }

    public static Source fromText(CharSequence chars, String description) {
        CompilerAsserts.neverPartOfCompilation("do not call Source.fromText from compiled code");
        return new LiteralSource(description, chars.toString());
    }

    public static Source fromAppendableText(String description) {
        CompilerAsserts.neverPartOfCompilation("do not call Source.fromAppendableText from compiled code");
        return new AppendableLiteralSource(description);
    }

    public static Source fromNamedText(CharSequence chars, String name) {
        CompilerAsserts.neverPartOfCompilation("do not call Source.fromNamedText from compiled code");
        LiteralSource source = new LiteralSource(name, chars.toString());
        nameToSource.put(name, new WeakReference<LiteralSource>(source));
        return source;
    }

    public static Source fromNamedAppendableText(String name) {
        CompilerAsserts.neverPartOfCompilation("do not call Source.fromNamedAppendable from compiled code");
        AppendableLiteralSource source = new AppendableLiteralSource(name);
        nameToSource.put(name, new WeakReference<AppendableLiteralSource>(source));
        return source;
    }

    public static Source subSource(Source base, int baseCharIndex, int length) {
        CompilerAsserts.neverPartOfCompilation(NO_FASTPATH_SUBSOURCE_CREATION_MESSAGE);
        SubSource subSource = SubSource.create(base, baseCharIndex, length);
        return subSource;
    }

    public static Source subSource(Source base, int baseCharIndex) {
        CompilerAsserts.neverPartOfCompilation(NO_FASTPATH_SUBSOURCE_CREATION_MESSAGE);
        return Source.subSource(base, baseCharIndex, base.getLength() - baseCharIndex);
    }

    public static Source fromURL(URL url, String description) throws IOException {
        CompilerAsserts.neverPartOfCompilation("do not call Source.fromURL from compiled code");
        return URLSource.get(url, description);
    }

    public static Source fromReader(Reader reader, String description) throws IOException {
        CompilerAsserts.neverPartOfCompilation("do not call Source.fromReader from compiled code");
        return new LiteralSource(description, Source.read(reader));
    }

    public static Source fromBytes(byte[] bytes, String description, Charset charset) {
        return Source.fromBytes(bytes, 0, bytes.length, description, charset);
    }

    public static Source fromBytes(byte[] bytes, int byteIndex, int length, String description, Charset charset) {
        CompilerAsserts.neverPartOfCompilation("do not call Source.fromBytes from compiled code");
        return new BytesSource(description, bytes, byteIndex, length, charset);
    }

    public static void setFileCaching(boolean enabled) {
        fileCacheEnabled = enabled;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static String read(Reader reader) throws IOException {
        BufferedReader bufferedReader = new BufferedReader(reader);
        StringBuilder builder = new StringBuilder();
        char[] buffer = new char[1024];
        try {
            int n;
            while ((n = bufferedReader.read(buffer)) != -1) {
                builder.append(buffer, 0, n);
            }
        }
        finally {
            bufferedReader.close();
        }
        return builder.toString();
    }

    private Source() {
    }

    abstract void reset();

    public abstract String getName();

    public abstract String getShortName();

    public abstract String getPath();

    public abstract URL getURL();

    public abstract Reader getReader();

    public final InputStream getInputStream() {
        return new ByteArrayInputStream(this.getCode().getBytes());
    }

    public final int getLength() {
        return this.getTextMap().length();
    }

    public abstract String getCode();

    public String getCode(int charIndex, int charLength) {
        return this.getCode().substring(charIndex, charIndex + charLength);
    }

    public final String getCode(int lineNumber) {
        int offset = this.getTextMap().lineStartOffset(lineNumber);
        int length = this.getTextMap().lineLength(lineNumber);
        return this.getCode().substring(offset, offset + length);
    }

    public final int getLineCount() {
        return this.getTextMap().lineCount();
    }

    public final int getLineNumber(int offset) throws IllegalArgumentException {
        return this.getTextMap().offsetToLine(offset);
    }

    public final int getColumnNumber(int offset) throws IllegalArgumentException {
        return this.getTextMap().offsetToCol(offset);
    }

    public final int getLineStartOffset(int lineNumber) throws IllegalArgumentException {
        return this.getTextMap().lineStartOffset(lineNumber);
    }

    public final int getLineLength(int lineNumber) throws IllegalArgumentException {
        return this.getTextMap().lineLength(lineNumber);
    }

    public void appendCode(CharSequence chars) {
        throw new UnsupportedOperationException();
    }

    public final SourceSection createSection(String identifier, int startLine, int startColumn, int charIndex, int length) {
        this.checkRange(charIndex, length);
        return this.createSectionImpl(identifier, startLine, startColumn, charIndex, length, SourceSection.EMTPY_TAGS);
    }

    @Deprecated
    public final SourceSection createSection(String identifier, int startLine, int startColumn, int charIndex, int length, String ... tags) {
        this.checkRange(charIndex, length);
        return this.createSectionImpl(identifier, startLine, startColumn, charIndex, length, tags);
    }

    private SourceSection createSectionImpl(String identifier, int startLine, int startColumn, int charIndex, int length, String[] tags) {
        return new SourceSection(null, this, identifier, startLine, startColumn, charIndex, length, tags);
    }

    public final SourceSection createSection(String identifier, int startLine, int startColumn, int length) {
        int lineStartOffset = this.getTextMap().lineStartOffset(startLine);
        if (startColumn > this.getTextMap().lineLength(startLine)) {
            throw new IllegalArgumentException("column out of range");
        }
        int startOffset = lineStartOffset + startColumn - 1;
        return this.createSectionImpl(identifier, startLine, startColumn, startOffset, length, SourceSection.EMTPY_TAGS);
    }

    public final SourceSection createSection(String identifier, int charIndex, int length) throws IllegalArgumentException {
        return this.createSection(identifier, charIndex, length, SourceSection.EMTPY_TAGS);
    }

    @Deprecated
    public final SourceSection createSection(String identifier, int charIndex, int length, String ... tags) throws IllegalArgumentException {
        this.checkRange(charIndex, length);
        int startLine = this.getLineNumber(charIndex);
        int startColumn = charIndex - this.getLineStartOffset(startLine) + 1;
        return this.createSectionImpl(identifier, startLine, startColumn, charIndex, length, tags);
    }

    void checkRange(int charIndex, int length) {
        if (charIndex < 0 || length < 0 || charIndex + length > this.getCode().length()) {
            throw new IllegalArgumentException("text positions out of range");
        }
    }

    public final SourceSection createSection(String identifier, int lineNumber) {
        int charIndex = this.getTextMap().lineStartOffset(lineNumber);
        int length = this.getTextMap().lineLength(lineNumber);
        return this.createSection(identifier, charIndex, length);
    }

    public final LineLocation createLineLocation(int lineNumber) {
        return new LineLocation(this, lineNumber);
    }

    Object getHashKey() {
        return this.getName();
    }

    final TextMap getTextMap() {
        if (this.textMap == null) {
            this.textMap = this.createTextMap();
        }
        return this.textMap;
    }

    final void clearTextMap() {
        this.textMap = null;
    }

    TextMap createTextMap() {
        String code = this.getCode();
        if (code == null) {
            throw new RuntimeException("can't read file " + this.getName());
        }
        return TextMap.fromString(code);
    }

    public final Source withMimeType(String mime) {
        try {
            Source another = (Source)this.clone();
            another.mimeType = mime;
            return another;
        }
        catch (CloneNotSupportedException ex) {
            throw new IllegalStateException(ex);
        }
    }

    public String getMimeType() {
        if (this.mimeType == null) {
            this.mimeType = this.findMimeType();
        }
        return this.mimeType;
    }

    String findMimeType() {
        return null;
    }

    final boolean equalMime(Source other) {
        if (this.mimeType == null) {
            return other.mimeType == null;
        }
        return this.mimeType.equals(other.mimeType);
    }

    private static final class TextMap {
        private final int[] nlOffsets;
        private final int textLength;
        final boolean finalNL;

        TextMap(int[] nlOffsets, int textLength, boolean finalNL) {
            this.nlOffsets = nlOffsets;
            this.textLength = textLength;
            this.finalNL = finalNL;
        }

        public static TextMap fromString(String text) {
            int nlIndex;
            int textLength = text.length();
            ArrayList<Integer> lines = new ArrayList<Integer>();
            lines.add(0);
            int offset = 0;
            while (offset < text.length() && (nlIndex = text.indexOf(10, offset)) >= 0) {
                offset = nlIndex + 1;
                lines.add(offset);
            }
            lines.add(Integer.MAX_VALUE);
            int[] nlOffsets = new int[lines.size()];
            for (int line = 0; line < lines.size(); ++line) {
                nlOffsets[line] = (Integer)lines.get(line);
            }
            boolean finalNL = textLength > 0 && textLength == nlOffsets[nlOffsets.length - 2];
            return new TextMap(nlOffsets, textLength, finalNL);
        }

        public int offsetToLine(int offset) throws IllegalArgumentException {
            if (offset < 0 || offset >= this.textLength) {
                if (offset == 0 && this.textLength == 0) {
                    return 1;
                }
                throw new IllegalArgumentException("offset out of bounds");
            }
            int line = 1;
            while (offset >= this.nlOffsets[line]) {
                ++line;
            }
            return line;
        }

        public int offsetToCol(int offset) throws IllegalArgumentException {
            return 1 + offset - this.nlOffsets[this.offsetToLine(offset) - 1];
        }

        public int length() {
            return this.textLength;
        }

        public int lineCount() {
            if (this.textLength == 0) {
                return 0;
            }
            return this.finalNL ? this.nlOffsets.length - 2 : this.nlOffsets.length - 1;
        }

        public int lineStartOffset(int line) throws IllegalArgumentException {
            if (this.textLength == 0) {
                return 0;
            }
            if (this.lineOutOfRange(line)) {
                throw new IllegalArgumentException("line out of bounds");
            }
            return this.nlOffsets[line - 1];
        }

        public int lineLength(int line) throws IllegalArgumentException {
            if (this.textLength == 0) {
                return 0;
            }
            if (this.lineOutOfRange(line)) {
                throw new IllegalArgumentException("line out of bounds");
            }
            if (line == this.nlOffsets.length - 1 && !this.finalNL) {
                return this.textLength - this.nlOffsets[line - 1];
            }
            return this.nlOffsets[line] - this.nlOffsets[line - 1] - 1;
        }

        private boolean lineOutOfRange(int line) {
            return line <= 0 || line >= this.nlOffsets.length || line == this.nlOffsets.length - 1 && this.finalNL;
        }
    }

    private static final class BytesSource
    extends Source
    implements Cloneable {
        private final String name;
        private final byte[] bytes;
        private final int byteIndex;
        private final int length;
        private final CharsetDecoder decoder;

        BytesSource(String name, byte[] bytes, int byteIndex, int length, Charset decoder) {
            this.name = name;
            this.bytes = bytes;
            this.byteIndex = byteIndex;
            this.length = length;
            this.decoder = decoder.newDecoder();
        }

        @Override
        void reset() {
        }

        @Override
        public String getName() {
            return this.name;
        }

        @Override
        public String getShortName() {
            return this.name;
        }

        @Override
        public String getPath() {
            return this.name;
        }

        @Override
        public URL getURL() {
            return null;
        }

        @Override
        public Reader getReader() {
            return null;
        }

        @Override
        public String getCode() {
            CharBuffer chb;
            ByteBuffer bb = ByteBuffer.wrap(this.bytes, this.byteIndex, this.length);
            try {
                chb = this.decoder.decode(bb);
            }
            catch (CharacterCodingException ex) {
                return "";
            }
            return chb.toString();
        }

        @Override
        public String getCode(int byteOffset, int codeLength) {
            CharBuffer chb;
            ByteBuffer bb = ByteBuffer.wrap(this.bytes, this.byteIndex + byteOffset, codeLength);
            try {
                chb = this.decoder.decode(bb);
            }
            catch (CharacterCodingException ex) {
                return "";
            }
            return chb.toString();
        }

        @Override
        void checkRange(int charIndex, int rangeLength) {
            if (charIndex < 0 || rangeLength < 0 || charIndex + rangeLength > this.length) {
                throw new IllegalArgumentException("text positions out of range");
            }
        }

        @Override
        TextMap createTextMap() {
            return TextMap.fromString(this.getCode());
        }
    }

    private static final class SubSource
    extends Source
    implements Cloneable {
        private final Source base;
        private final int baseIndex;
        private final int subLength;

        private static SubSource create(Source base, int baseIndex, int length) {
            if (baseIndex < 0 || length < 0 || baseIndex + length > base.getLength()) {
                throw new IllegalArgumentException("text positions out of range");
            }
            return new SubSource(base, baseIndex, length);
        }

        private SubSource(Source base, int baseIndex, int length) {
            this.base = base;
            this.baseIndex = baseIndex;
            this.subLength = length;
        }

        @Override
        void reset() {
            assert (false);
        }

        @Override
        public String getName() {
            return this.base.getName();
        }

        @Override
        public String getShortName() {
            return this.base.getShortName();
        }

        @Override
        public String getPath() {
            return this.base.getPath();
        }

        @Override
        public URL getURL() {
            return null;
        }

        @Override
        public Reader getReader() {
            assert (false);
            return null;
        }

        @Override
        public String getCode() {
            return this.base.getCode(this.baseIndex, this.subLength);
        }
    }

    private static final class URLSource
    extends Source
    implements Cloneable {
        private static final Map<URL, WeakReference<URLSource>> urlToSource = new HashMap<URL, WeakReference<URLSource>>();
        private final URL url;
        private final String name;
        private String code;

        public static URLSource get(URL url, String name) throws IOException {
            URLSource source;
            WeakReference<URLSource> sourceRef = urlToSource.get(url);
            URLSource uRLSource = source = sourceRef == null ? null : (URLSource)sourceRef.get();
            if (source == null) {
                source = new URLSource(url, name);
                urlToSource.put(url, new WeakReference<URLSource>(source));
            }
            return source;
        }

        URLSource(URL url, String name) throws IOException {
            this.url = url;
            this.name = name;
            URLConnection c = url.openConnection();
            if (((Source)this).mimeType == null) {
                ((Source)this).mimeType = c.getContentType();
            }
            this.code = Source.read(new InputStreamReader(c.getInputStream()));
        }

        @Override
        public String getName() {
            return this.name;
        }

        @Override
        public String getShortName() {
            return this.name;
        }

        @Override
        public String getPath() {
            return this.url.getPath();
        }

        @Override
        public URL getURL() {
            return this.url;
        }

        @Override
        public Reader getReader() {
            return new StringReader(this.code);
        }

        @Override
        public String getCode() {
            return this.code;
        }

        @Override
        void reset() {
        }
    }

    private static final class ClientManagedFileSource
    extends Source
    implements Cloneable {
        private final File file;
        private final String name;
        private final String path;
        private String code;

        ClientManagedFileSource(File file, String name, String path, CharSequence chars) {
            this.file = file.getAbsoluteFile();
            this.name = name;
            this.path = path;
            this.setCode(chars);
        }

        void setCode(CharSequence chars) {
            this.clearTextMap();
            this.code = chars.toString();
        }

        @Override
        public String getName() {
            return this.name;
        }

        @Override
        public String getShortName() {
            return this.file.getName();
        }

        @Override
        Object getHashKey() {
            return this.path;
        }

        @Override
        public String getCode() {
            return this.code;
        }

        @Override
        public String getPath() {
            return this.path;
        }

        @Override
        public URL getURL() {
            return null;
        }

        @Override
        public Reader getReader() {
            return new StringReader(this.code);
        }

        @Override
        String findMimeType() {
            try {
                return Files.probeContentType(this.file.toPath());
            }
            catch (IOException ex) {
                LOG.log(Level.SEVERE, null, ex);
                return null;
            }
        }

        public int hashCode() {
            return this.path.hashCode();
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj instanceof ClientManagedFileSource) {
                ClientManagedFileSource other = (ClientManagedFileSource)obj;
                return this.path.equals(other.path) && this.equalMime(other);
            }
            return false;
        }

        @Override
        void reset() {
            this.code = null;
        }
    }

    private static final class FileSource
    extends Source
    implements Cloneable {
        private final File file;
        private final String name;
        private final String path;
        private String code = null;

        FileSource(File file, String name, String path) {
            this.file = file.getAbsoluteFile();
            this.name = name;
            this.path = path;
        }

        @Override
        public String getName() {
            return this.name;
        }

        @Override
        public String getShortName() {
            return this.file.getName();
        }

        @Override
        Object getHashKey() {
            return this.path;
        }

        @Override
        public String getCode() {
            if (fileCacheEnabled) {
                if (this.code == null) {
                    try {
                        this.code = Source.read(this.getReader());
                    }
                    catch (IOException iOException) {
                        // empty catch block
                    }
                }
                return this.code;
            }
            try {
                return Source.read(new InputStreamReader((InputStream)new FileInputStream(this.file), "UTF-8"));
            }
            catch (IOException iOException) {
                return null;
            }
        }

        @Override
        public String getPath() {
            return this.path;
        }

        @Override
        public URL getURL() {
            return null;
        }

        @Override
        public Reader getReader() {
            if (this.code != null) {
                return new StringReader(this.code);
            }
            try {
                return new InputStreamReader((InputStream)new FileInputStream(this.file), "UTF-8");
            }
            catch (FileNotFoundException e) {
                throw new RuntimeException("Can't find file " + this.path, e);
            }
            catch (UnsupportedEncodingException e) {
                throw new RuntimeException("Unsupported encoding in file " + this.path, e);
            }
        }

        public int hashCode() {
            return this.path.hashCode();
        }

        @Override
        String findMimeType() {
            try {
                return Files.probeContentType(this.file.toPath());
            }
            catch (IOException ex) {
                LOG.log(Level.SEVERE, null, ex);
                return null;
            }
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj instanceof FileSource) {
                FileSource other = (FileSource)obj;
                return this.path.equals(other.path) && this.equalMime(other);
            }
            return false;
        }

        @Override
        void reset() {
            this.code = null;
        }
    }

    private static final class AppendableLiteralSource
    extends Source
    implements Cloneable {
        private final String description;
        final List<CharSequence> codeList = new ArrayList<CharSequence>();

        AppendableLiteralSource(String description) {
            this.description = description;
        }

        @Override
        public String getName() {
            return this.description;
        }

        @Override
        public String getShortName() {
            return this.description;
        }

        @Override
        public String getCode() {
            return this.getCodeFromIndex(0);
        }

        @Override
        public String getPath() {
            return this.description;
        }

        @Override
        public URL getURL() {
            return null;
        }

        @Override
        public Reader getReader() {
            return new StringReader(this.getCode());
        }

        @Override
        void reset() {
        }

        private String getCodeFromIndex(int index) {
            StringBuilder sb = new StringBuilder();
            for (int i = index; i < this.codeList.size(); ++i) {
                CharSequence s = this.codeList.get(i);
                sb.append(s);
            }
            return sb.toString();
        }

        @Override
        public void appendCode(CharSequence chars) {
            this.codeList.add(chars);
            this.clearTextMap();
        }
    }

    private static final class LiteralSource
    extends Source
    implements Cloneable {
        private final String description;
        private final String code;

        LiteralSource(String description, String code) {
            this.description = description;
            this.code = code;
        }

        @Override
        public String getName() {
            return this.description;
        }

        @Override
        public String getShortName() {
            return this.description;
        }

        @Override
        public String getCode() {
            return this.code;
        }

        @Override
        public String getPath() {
            return null;
        }

        @Override
        public URL getURL() {
            return null;
        }

        @Override
        public Reader getReader() {
            return new StringReader(this.code);
        }

        @Override
        void reset() {
        }

        public int hashCode() {
            return Objects.hash(this.description, this.code);
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (obj instanceof LiteralSource) {
                LiteralSource other = (LiteralSource)obj;
                return Objects.equals(this.description, other.description) && this.code.equals(other.code) && this.equalMime(other);
            }
            return false;
        }
    }
}

