/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.truffle.tools;

import com.oracle.truffle.api.CompilerDirectives;
import com.oracle.truffle.api.frame.FrameDescriptor;
import com.oracle.truffle.api.frame.FrameSlot;
import com.oracle.truffle.api.frame.FrameSlotKind;
import com.oracle.truffle.api.frame.FrameSlotTypeException;
import com.oracle.truffle.api.frame.VirtualFrame;
import com.oracle.truffle.api.instrumentation.EventContext;
import com.oracle.truffle.api.instrumentation.ExecutionEventNode;
import com.oracle.truffle.api.instrumentation.ExecutionEventNodeFactory;
import com.oracle.truffle.api.instrumentation.SourceSectionFilter;
import com.oracle.truffle.api.instrumentation.StandardTags;
import com.oracle.truffle.api.instrumentation.TruffleInstrument;
import com.oracle.truffle.api.nodes.NodeCost;
import com.oracle.truffle.api.profiles.ConditionProfile;
import com.oracle.truffle.api.source.SourceSection;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class TruffleProfiler
extends TruffleInstrument {
    public static final String ID = "truffle_profiler";
    private static final int MAX_CODE_LENGTH = 30;
    private final Map<SourceSection, Counter> counters = new HashMap<SourceSection, Counter>();
    private String[] mimeTypes = TruffleProfiler.parseMimeTypes(System.getProperty("truffle.profiling.includeMimeTypes"));
    private boolean timingDisabled = Boolean.getBoolean("truffle.profiling.timingDisabled");
    private Counter activeCounter;
    private boolean disposed;
    private static TestHook testHook;

    @Override
    protected void onCreate(final TruffleInstrument.Env env) {
        if (!TruffleProfiler.isEnabled()) {
            return;
        }
        if (testHook != null) {
            testHook.onCreate(this);
        }
        SourceSectionFilter.Builder filterBuilder = SourceSectionFilter.newBuilder();
        if (this.mimeTypes != null) {
            filterBuilder.mimeTypeIs(this.mimeTypes);
        }
        env.getInstrumenter().attachFactory(filterBuilder.tagIs(StandardTags.RootTag.class).build(), new ExecutionEventNodeFactory(){

            @Override
            public ExecutionEventNode create(EventContext context) {
                return TruffleProfiler.this.createCountingNode(context);
            }
        });
        Runtime.getRuntime().addShutdownHook(new Thread(new Runnable(){

            @Override
            public void run() {
                TruffleProfiler.this.onDispose(env);
            }
        }));
    }

    private static boolean isEnabled() {
        return Boolean.getBoolean("truffle.profiling.enabled") || testHook != null;
    }

    private ExecutionEventNode createCountingNode(EventContext context) {
        SourceSection sourceSection = context.getInstrumentedSourceSection();
        Counter counter = this.counters.get(sourceSection);
        if (counter == null) {
            counter = new Counter(sourceSection);
            this.counters.put(sourceSection, counter);
        }
        if (this.timingDisabled) {
            return new CounterNode(this, counter);
        }
        return TimedCounterNode.create(this, counter, context);
    }

    Map<SourceSection, Counter> getCounters() {
        return this.counters;
    }

    @Override
    protected void onDispose(TruffleInstrument.Env env) {
        if (!this.disposed) {
            this.disposed = true;
            if (TruffleProfiler.isEnabled()) {
                PrintStream out = new PrintStream(env.out());
                this.printHistograms(out);
                out.flush();
            }
        }
    }

    private void printHistograms(PrintStream out) {
        ArrayList<Counter> sortedCounters = new ArrayList<Counter>(this.counters.values());
        boolean hasCompiled = false;
        for (Counter counter : sortedCounters) {
            if (counter.getInvocations(Counter.TimeKind.COMPILED) <= 0L) continue;
            hasCompiled = true;
        }
        if (hasCompiled) {
            this.printHistogram(out, sortedCounters, Counter.TimeKind.INTERPRETED_AND_COMPILED);
            this.printHistogram(out, sortedCounters, Counter.TimeKind.INTERPRETED);
            this.printHistogram(out, sortedCounters, Counter.TimeKind.COMPILED);
        } else {
            this.printHistogram(out, sortedCounters, Counter.TimeKind.INTERPRETED);
        }
    }

    private void printHistogram(PrintStream out, List<Counter> sortedCounters, final Counter.TimeKind time) {
        Collections.sort(sortedCounters, new Comparator<Counter>(){

            @Override
            public int compare(Counter o1, Counter o2) {
                if (TruffleProfiler.this.timingDisabled) {
                    return Long.compare(o2.getInvocations(time), o1.getInvocations(time));
                }
                return Long.compare(o2.getSelfTime(time), o1.getSelfTime(time));
            }
        });
        out.println("Truffle profiler histogram for mode " + (Object)((Object)time));
        out.println(String.format("%12s | %7s | %11s | %7s | %11s | %-30s | %s ", "Invoc", "Total", "PerInvoc", "SelfTime", "PerInvoc", "Source", "Code"));
        for (Counter counter : sortedCounters) {
            long invocations = counter.getInvocations(time);
            if (invocations <= 0L) continue;
            double totalTimems = (double)counter.getTotalTime(time) / 1000000.0;
            double selfTimems = (double)counter.getSelfTime(time) / 1000000.0;
            out.println(String.format("%12d |%6.0fms |%10.3fms |%7.0fms |%10.3fms | %-30s | %s", invocations, totalTimems, totalTimems / (double)invocations, selfTimems, selfTimems / (double)invocations, TruffleProfiler.getShortDescription(counter.getSourceSection()), TruffleProfiler.getShortSource(counter.getSourceSection())));
        }
        out.println();
    }

    private static Object getShortSource(SourceSection sourceSection) {
        if (sourceSection.getSource() == null) {
            return "<unknown>";
        }
        String code = sourceSection.getCode();
        if (code.length() > 30) {
            code = code.substring(0, 30);
        }
        return code.replaceAll("\\n", "\\\\n");
    }

    private static String getShortDescription(SourceSection sourceSection) {
        if (sourceSection.getSource() == null) {
            return sourceSection.getIdentifier();
        }
        StringBuilder b = new StringBuilder();
        if (sourceSection.getIdentifier() != null) {
            b.append(sourceSection.getSource().getShortName());
        } else {
            b.append("<unknown>");
        }
        b.append(":line=");
        if (sourceSection.getStartLine() == sourceSection.getEndLine()) {
            b.append(sourceSection.getStartLine());
        } else {
            b.append(sourceSection.getStartLine()).append("-").append(sourceSection.getEndLine());
        }
        b.append(":chars=");
        if (sourceSection.getCharIndex() == sourceSection.getEndColumn()) {
            b.append(sourceSection.getCharIndex());
        } else {
            b.append(sourceSection.getCharIndex()).append("-").append(sourceSection.getEndColumn());
        }
        return b.toString();
    }

    private static String[] parseMimeTypes(String property) {
        if (property != null) {
            return property.split(";");
        }
        return null;
    }

    static void setTestHook(TestHook testHook) {
        TruffleProfiler.testHook = testHook;
    }

    static interface TestHook {
        public void onCreate(TruffleProfiler var1);
    }

    static final class Counter {
        private final SourceSection sourceSection;
        private long interpretedInvocations;
        private long interpretedChildTime;
        private long interpretedTotalTime;
        private long compiledInvocations;
        private long compiledTotalTime;
        private long compiledChildTime;
        private boolean compiled;

        Counter(SourceSection sourceSection) {
            this.sourceSection = sourceSection;
        }

        public SourceSection getSourceSection() {
            return this.sourceSection;
        }

        public long getInvocations(TimeKind kind) {
            switch (kind) {
                case INTERPRETED_AND_COMPILED: {
                    return this.interpretedInvocations + this.compiledInvocations;
                }
                case COMPILED: {
                    return this.compiledInvocations;
                }
                case INTERPRETED: {
                    return this.interpretedInvocations;
                }
            }
            throw new AssertionError();
        }

        public long getTotalTime(TimeKind kind) {
            switch (kind) {
                case INTERPRETED_AND_COMPILED: {
                    return this.interpretedTotalTime + this.compiledTotalTime;
                }
                case COMPILED: {
                    return this.compiledTotalTime;
                }
                case INTERPRETED: {
                    return this.interpretedTotalTime;
                }
            }
            throw new AssertionError();
        }

        public long getSelfTime(TimeKind kind) {
            switch (kind) {
                case INTERPRETED_AND_COMPILED: {
                    return this.interpretedTotalTime + this.compiledTotalTime - this.compiledChildTime - this.interpretedChildTime;
                }
                case COMPILED: {
                    return this.compiledTotalTime - this.compiledChildTime;
                }
                case INTERPRETED: {
                    return this.interpretedTotalTime - this.interpretedChildTime;
                }
            }
            throw new AssertionError();
        }

        static enum TimeKind {
            INTERPRETED_AND_COMPILED,
            INTERPRETED,
            COMPILED;

        }
    }

    private static class CounterNode
    extends ExecutionEventNode {
        protected final TruffleProfiler profiler;
        protected final Counter counter;

        CounterNode(TruffleProfiler profiler, Counter counter) {
            this.profiler = profiler;
            this.counter = counter;
        }

        @Override
        protected void onEnter(VirtualFrame frame) {
            if (CompilerDirectives.inInterpreter()) {
                this.counter.interpretedInvocations++;
            } else {
                this.counter.compiledInvocations++;
            }
        }

        @Override
        public NodeCost getCost() {
            return NodeCost.NONE;
        }
    }

    private static class TimedCounterNode
    extends CounterNode {
        private final EventContext context;
        private final FrameSlot parentCounterSlot;
        private final FrameSlot timeStartedSlot;
        private final ConditionProfile parentNotNullProfile = ConditionProfile.createBinaryProfile();
        private static final Object KEY_TIME_STARTED = new Object();
        private static final Object KEY_PARENT_COUNTER = new Object();

        TimedCounterNode(TruffleProfiler profiler, Counter counter, EventContext context) {
            super(profiler, counter);
            this.context = context;
            FrameDescriptor frameDescriptor = context.getInstrumentedNode().getRootNode().getFrameDescriptor();
            this.timeStartedSlot = frameDescriptor.findOrAddFrameSlot(KEY_TIME_STARTED, "profiler:timeStarted", FrameSlotKind.Long);
            this.parentCounterSlot = frameDescriptor.findOrAddFrameSlot(KEY_PARENT_COUNTER, "profiler:parentCounter", FrameSlotKind.Object);
        }

        @Override
        protected void onDispose(VirtualFrame frame) {
            FrameDescriptor frameDescriptor = this.context.getInstrumentedNode().getRootNode().getFrameDescriptor();
            if (frameDescriptor.getIdentifiers().contains(KEY_TIME_STARTED)) {
                frameDescriptor.removeFrameSlot(this.timeStartedSlot);
            }
            if (frameDescriptor.getIdentifiers().contains(KEY_PARENT_COUNTER)) {
                frameDescriptor.removeFrameSlot(this.parentCounterSlot);
            }
        }

        @Override
        protected void onEnter(VirtualFrame frame) {
            frame.setLong(this.timeStartedSlot, System.nanoTime());
            super.onEnter(frame);
            frame.setObject(this.parentCounterSlot, this.profiler.activeCounter);
            this.profiler.activeCounter = this.counter;
            if (CompilerDirectives.inInterpreter()) {
                this.counter.compiled = false;
            } else {
                this.counter.compiled = true;
            }
        }

        @Override
        protected void onReturnExceptional(VirtualFrame frame, Throwable exception) {
            this.onReturnValue(frame, null);
        }

        @Override
        protected void onReturnValue(VirtualFrame frame, Object result) {
            Counter counter;
            Counter parentCounter;
            long startTime;
            try {
                startTime = frame.getLong(this.timeStartedSlot);
                parentCounter = (Counter)frame.getObject(this.parentCounterSlot);
            }
            catch (FrameSlotTypeException e) {
                throw new AssertionError();
            }
            long timeNano = System.nanoTime() - startTime;
            if (CompilerDirectives.inInterpreter()) {
                counter = this.counter;
                counter.interpretedTotalTime = counter.interpretedTotalTime + timeNano;
            } else {
                counter = this.counter;
                counter.compiledTotalTime = counter.compiledTotalTime + timeNano;
            }
            if (this.parentNotNullProfile.profile(parentCounter != null)) {
                if (parentCounter.compiled) {
                    counter = parentCounter;
                    counter.compiledChildTime = counter.compiledChildTime + timeNano;
                } else {
                    counter = parentCounter;
                    counter.interpretedChildTime = counter.interpretedChildTime + timeNano;
                }
            }
            this.profiler.activeCounter = parentCounter;
        }

        static CounterNode create(TruffleProfiler profiler, Counter counter, EventContext context) {
            return new TimedCounterNode(profiler, counter, context);
        }
    }
}

