/**
 * Copyright (c) 2011-2016 by Andrew Mustun. All rights reserved.
 * 
 * This file is part of the QCAD project.
 *
 * QCAD is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * QCAD is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with QCAD.
 */

if (new QFileInfo(autoPath("scripts/Navigation/DefaultNavigation/DefaultNavigation.js")).exists()) {
    include("scripts/Navigation/DefaultNavigation/DefaultNavigation.js");
}

Viewport.templateDir = autoPath("scripts/Widgets/Viewport/Templates");

/**
 * A viewport encapsulates one graphics view and might provide means to navigate
 * (scroll bars) and other widgets that are available for every graphics view.
 *
 * \param vpNumber Number of viewport. There can be up to four viewports
 * available for example in an MDI child (tab). This number is used to identify
 * the viewport settings in configuration files and document settings.
 * \param vpWidget Viewport widget that will contain the viewport
 * after initialization.
 */
function Viewport(vpNumber, vpWidget, documentInterface) {
    this.vpNumber = vpNumber;
    this.vpWidget = vpWidget;
    this.documentInterface = documentInterface;
    this.eventHandler = undefined;
}

/**
 * \return Array of Viewport objects, one for each child of widget that
 * is called "ViewportXX".
 */
Viewport.getViewports = function(widget, documentInterface) {
    var vpWidget;
    var c = 0;
    var vps = [];
    do {
        var vpName = sprintf("Viewport%02d", c);
        vpWidget = widget.findChild(vpName);
        if (isNull(vpWidget)) {
            break;
        }
        var viewport = new Viewport(c, vpWidget, documentInterface);
        vps.push(viewport);
        ++c;
    } while (!isNull(vpWidget));
    return vps;
};

Viewport.initEventHandler = function(viewports) {
    for ( var i = 0; i < viewports.length; ++i) {
        viewports[i].initEventHandler();
    }
};

Viewport.initializeViewports = function(viewports) {
    for ( var i = 0; i < viewports.length; ++i) {
        var vp = viewports[i];
        vp.init();
    }
};

Viewport.updateViewports = function(viewports) {
    // TODO auto zoom for each graphics view works only for the first viewport
    for (var i = 0; i < viewports.length; ++i) {
        var vp = viewports[i];
        vp.getEventHandler().viewportChanged();
        var view = vp.getGraphicsView();

        if (RSettings.getAutoZoomOnLoad() || i>0) {
            view.autoZoom(-1, true);
        }
        else {
            // restore first viewport (zoom / pan):
            var di = vp.getDocumentInterface();
            var doc = di.getDocument();
            var c = doc.getVariable("ViewportCenter", undefined);
            var w = doc.getVariable("ViewportWidth", undefined);
            var h = doc.getVariable("ViewportHeight", undefined);
            if (!isNull(c) && !isNull(w) && !isNull(h)) {
                view.zoomTo(new RBox(c, w, h));
            }
        }
    }
};

Viewport.initMdiChild = function(mdiChild, uiFileName) {
    var w = WidgetFactory.createWidget("", Viewport.templateDir + QDir.separator + uiFileName, mdiChild);
    w.setWindowTitle("");
    mdiChild.setWidget(w);
};

Viewport.prototype.initEventHandler = function() {
    this.eventHandler = new EventHandler(this, this.documentInterface);
    if (isOfType(this.graphicsView, RGraphicsViewQt)) {
        this.graphicsView.drop.connect(this.eventHandler, "drop");
        this.graphicsView.dragEnter.connect(this.eventHandler, "dragEnter");
        this.graphicsView.viewportChanged.connect(this.eventHandler, "viewportChanged");
        this.graphicsView.updateSnapInfo.connect(this.eventHandler, "updateSnapInfo");
        this.graphicsView.updateTextLabel.connect(this.eventHandler, "updateTextLabel");
    }

    if (!isNull(this.hsb)) {
        this.hsb.valueChanged.connect(this.eventHandler, "horizontalScrolled");
        this.vsb.valueChanged.connect(this.eventHandler, "verticalScrolled");
    }
};

/**
 * Initializes this viewport.
 */
Viewport.prototype.init = function() {
    // delete placeholder children if there are any (clear out parent window):
    var chs = this.vpWidget.children();
    for (var i = 0; i < chs.length; ++i) {
        var ch = chs[i];
        ch.destroy();
    }
    
    var vpw = WidgetFactory.createWidget("", "scripts/Widgets/Viewport/Viewport.ui", this.vpWidget);
//    var vpw = WidgetFactory.createWidget("", "scripts/Widgets/Viewport/ViewportGl.ui", this.vpWidget);
    var layout = new QVBoxLayout();
    layout.addWidget(vpw, 0, Qt.AlignTop | Qt.AlignLeft);
    layout.setStretch(0, 1);
    layout.setContentsMargins(0,0,0,0);
    if (!isNull(this.vpWidget.layout())) {
        // destroy existing layout manager
        this.vpWidget.layout().destroy();
    }
    this.vpWidget.setLayout(layout);
    // clear style sheet
    this.vpWidget.styleSheet = "";

    this.graphicsView = this.vpWidget.findChild("GraphicsView");
    if (isNull(this.graphicsView)) {
        qWarning("graphics view not found");
        return;
    }
    this.graphicsView.setViewportNumber(this.vpNumber);
    if (isFunction(this.graphicsView.setFocusFrameWidget)) {
        this.graphicsView.setFocusFrameWidget(this.vpWidget);
    }

    this.graphicsView.setAntialiasing(RSettings.getBoolValue("GraphicsView/Antialiasing", false));

    var scene = new RGraphicsSceneQt(this.documentInterface);
    // var scene = new RGraphicsSceneGl(this.documentInterface);
    if (RSettings.getBoolValue("GraphicsView/AutoSwitchLinetypes", false)===true) {
        scene.setScreenBasedLinetypes(true);
    }

    this.graphicsView.setScene(scene);
    if (typeof(DefaultNavigation)!=="undefined") {
        var navigationAction = new DefaultNavigation(this.vpWidget);
        this.graphicsView.setNavigationAction(navigationAction);
    }

    var grid = new ROrthoGrid(this.graphicsView.getRGraphicsView());
    this.graphicsView.setGrid(grid);

    this.graphicsView.setFocus();

    this.hsb = this.vpWidget.findChild("HorizontalScrollBar");
    if (!isNull(this.hsb)) {
        this.hsb.singleStep = 50;
    }
    this.vsb = this.vpWidget.findChild("VerticalScrollBar");
    if (!isNull(this.vsb)) {
        this.vsb.singleStep = 50;
    }
    if (RSettings.getBoolValue("GraphicsView/ShowScrollBars", true)===false) {
        var infoLabel = this.vpWidget.findChild("InfoLabel");
        infoLabel.hide();
        if (!isNull(this.hsb)) {
            this.hsb.hide();
        }
        if (!isNull(this.vsb)) {
            this.vsb.hide();
        }
    }

    if (RSettings.getBoolValue("GraphicsView/ShowRulers", true)) {
        this.hruler = this.vpWidget.findChild("HorizontalRuler");
        this.hruler.setGraphicsView(this.graphicsView);
        this.vruler = this.vpWidget.findChild("VerticalRuler");
        this.vruler.setGraphicsView(this.graphicsView);

        this.documentInterface.addCoordinateListener(this.hruler);
        this.documentInterface.addCoordinateListener(this.vruler);
    } else {
        var hruler = this.vpWidget.findChild("HorizontalRuler");
        hruler.hide();
        var vruler = this.vpWidget.findChild("VerticalRuler");
        vruler.hide();
        var cf = this.vpWidget.findChild("cornerFrame");
        cf.hide();
    }
};

Viewport.prototype.getVpWidget = function() {
    return this.vpWidget;
};

Viewport.prototype.getGraphicsView = function() {
    return this.graphicsView;
};

Viewport.prototype.getDocumentInterface = function() {
    return this.documentInterface;
};

Viewport.prototype.getEventHandler = function() {
    return this.eventHandler;
};



/**
 * Event handler class. For each new document that is created or
 * opened, an event handler is created to handle various events
 * for the life time of the document.
 */
function EventHandler(viewport, documentInterface) {
    REventHandler.call(this, viewport.getVpWidget(), documentInterface);
}

EventHandler.prototype = new REventHandler();

/**
 * Handles drop into the drawing area
 * \param event QDropEvent
 */
EventHandler.prototype.drop = function(event) {
    var mimeData = event.mimeData();
    var urls = getUrlsFromMimeData(mimeData);

    if (urls.length===0) {
        EAction.handleUserWarning(qsTr("Dropped data not supported"));
        event.acceptProposedAction();
        return;
    }

    var appWin = RMainWindowQt.getMainWindow();
    if (!isNull(appWin)) {
        appWin.raise();
        appWin.setFocus(Qt.OtherFocusReason);
    }

    var action;
    if (urls[0].isLocalFile()) {
        var file = urls[0].toLocalFile();
        EAction.handleUserMessage(qsTr("Importing file: ") + file);
        if (new QFileInfo(file).isFile()) {
            include("scripts/Block/InsertScriptItem/InsertScriptItem.js");
            if (InsertScriptItem.isScriptFile(file)) {
                action = RGuiAction.getByScriptFile("scripts/Block/InsertScriptItem/InsertScriptItem.js");
            } else {
                action = RGuiAction.getByScriptFile("scripts/Block/InsertBlockItem/InsertBlockItem.js");
            }
        }
    }
    else {
        EAction.handleUserMessage(qsTr("Importing URL: ") + urls[0].toString());
        action = RGuiAction.getByScriptFile("scripts/Block/InsertBlockItem/InsertBlockItem.js");
    }

    if (isNull(action)) {
        event.acceptProposedAction();
        return;
    }

    action.setData(urls[0]);
    action.slotTrigger();
    event.accept();
};
