# application.py - Copyright (C) 2012 Red Hat, Inc.
# Written by Darryl L. Pierce <dpierce@redhat.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
# MA  02110-1301, USA.  A copy of the GNU General Public License is
# also available at http://www.gnu.org/copyleft/gpl.html.

from appscreen import ApplicationScreen

class Application:
    '''
    Application is the launching point for a Syrup-based app. All such
    apps should extend this type and override the appropriate methods
    to define the initial screen and any command line argument processing
    necessary.
    '''

    def __init__(self, title, fullscreen = False, compact_buttons = False):
        '''
        Creates a new application instance.

        Keyword arguments:
        title           -- the application's title
        fullscreen      -- sets whether the application will be fullscreen (def. False)
        compact_buttons -- use compact buttons (def. False)
        '''
        self.title=title
        self.screen = None
        self.finished = False
        self.cancelled = False
        self.fullscreen = fullscreen
        self.compact_buttons = compact_buttons

    def get_title(self):
        '''
        Returns the title for the application.
        '''
        return self.title

    def get_fullscreen(self):
        '''
        Returns whether the application will be fullscreen.
        '''
        return self.fullscreen

    def get_compact_buttons(self):
        '''
        Returns whether to use compact buttons on pages.
        '''
        return self.compact_buttons

    def create_app_screen(self):
        '''
        Override this method in order to use an application-specific
        instance of ApplicationScreen.
        '''
        return ApplicationScreen(self)

    def get_current_page(self):
        '''
        Returns the current page.
        '''
        return self.current_page

    def next_page(self):
        '''
        Invoked to advance the application by one page.

        Child classes must override this method to define the
        next page based on the application's context.
        '''
        pass

    def previous_page(self):
        '''
        Invoked to move the application back by one page.

        Child classes must override this method to define the
        previous page based on the application's context.
        '''
        pass

    def start(self):
        '''
        Starts the application. Creates the initial background screen and
        then shows the first page.
        '''
        self.app_screen = self.create_app_screen()
        while self.finished == False:
            self.app_screen.show_current_page()

    def set_finished(self, finished):
        '''
        Sets the finished flag for the application.
        '''
        self.finished = finished

    def set_cancelled(self, cancelled):
        '''
        Sets the cancelled flag for the application.
        '''
        self.cancelled = cancelled

