/*
 * libopensync - A synchronization framework
 * Copyright (C) 2004-2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#ifndef OPENSYNC_SINK_STATE_DB_H_
#define OPENSYNC_SINK_STATE_DB_H_

/**
 * @defgroup OSyncHelper OpenSync Helper Module
 * @ingroup OSyncPublic
 * @defgroup OSyncSinkStateDatabaseAPI OpenSync Sink State Database
 * @ingroup OSyncHelper
 * @brief Functions to store and retrieve sink state information
 *
 * The Sink state database allows your plugin to store custom state
 * information from the member in a persistent database, as a key-value pair
 * (both strings) on a per-sink basis. This state information usually relates
 * to some aspect of the state of the sink's peer, which may need to be
 * compared or otherwise used in each subsequent synchronisation. A typical
 * application is to force a slow-sync when some state value on the peer has
 * changed, however it is not limited to this.
 *
 * Example:
 * A random string is generated by the remote device on each sync, eg. XYZ,
 * and is stored as the "anchor". If the device wasn't reset or synced
 * somewhere else in the meantime, on the beginning of the next sync the
 * anchor and the string are compared and will match: XYZ == XYZ. Regular
 * sync occurs, and a new random string ABC is generated and stored. Now, in
 * the interval between, the device is synced with another system and thus a
 * new random string DEF is generated. On the next local sync the anchor is
 * compared with the device's stored string: ABC != DEF - the "anchor"
 * doesn't match with the device string. Consequently, to avoid any data loss
 * and to get the changed entries of the previous sync with the other system,
 * a slow-sync is then requested by the plugin.
 *
 * Example implementation in plugins: syncml, palm-sync, irmc-sync, ...
 *
 */
/*@{*/

/** @brief Compares the anchor/state value of key with the supplied value
 *
 * @param sinkStateDB Pointer to the sink state database object
 * @param key the key of the value to compare
 * @param value the value to compare with the stored anchor/state value
 * @param issame Pointer to an osync_bool that will be set to the comparison result
 * @param error Pointer to error struct, which will be set on any error
 * @returns TRUE if the anchor/state compare completed successfully, FALSE on error.
 *          (Not the return value of the comparison!)
 *
 */
OSYNC_EXPORT osync_bool osync_sink_state_equal(
				OSyncSinkStateDB *sinkStateDB,
				const char *key,
				const char *value,
				osync_bool *issame,
				OSyncError **error);

/** @brief Sets the anchor/state value of the key
 *
 * @param sinkStateDB Pointer to the sink state database object
 * @param key the key of the value to set
 * @param value the new value to set
 * @param error Pointer to error struct, which will be set on any error
 * @returns TRUE if anchor/state update completed successfully, FALSE on error.
 *
 */
OSYNC_EXPORT osync_bool osync_sink_state_set(
				OSyncSinkStateDB *sinkStateDB,
				const char *key,
				const char *value,
				OSyncError **error);

/** @brief Gets the anchor/state value of the supplied key
 *
 * @param sinkStateDB Pointer to the sink state database object
 * @param key The key as string of the anchor/state value to retrieve
 * @param error Pointer to error struct, which will be set on any error
 * @returns the value of the anchor/state if it was found,
 *          otherwise an empty string is returned.
 *          The caller is responsible for freeing the returned value with osync_free().
 *          NULL if there was an error when retrieving the anchor/state information.
 *
 */
OSYNC_EXPORT char *osync_sink_state_get(
				OSyncSinkStateDB *sinkStateDB,
				const char *key,
				OSyncError **error);

/*@}*/

#endif /* OPENSYNC_SINK_STATE_DB_H_ */
