/********************************************************************\
 * ScrubBusiness.h -- Cleanup functions for the business objects.   *
 *                                                                  *
 * This program is free software; you can redistribute it and/or    *
 * modify it under the terms of the GNU General Public License as   *
 * published by the Free Software Foundation; either version 2 of   *
 * the License, or (at your option) any later version.              *
 *                                                                  *
 * This program is distributed in the hope that it will be useful,  *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of   *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the    *
 * GNU General Public License for more details.                     *
 *                                                                  *
 * You should have received a copy of the GNU General Public License*
 * along with this program; if not, contact:                        *
 *                                                                  *
 * Free Software Foundation           Voice:  +1-617-542-5942       *
 * 51 Franklin Street, Fifth Floor    Fax:    +1-617-542-2652       *
 * Boston, MA  02110-1301,  USA       gnu@gnu.org                   *
\********************************************************************/

/** @addtogroup Engine
    @{ */
/** @addtogroup Scrub
    @{ */

/** @file ScrubBusiness.h
 *  @brief Cleanup functions for business objects
 *  @author Created by Geert Janssens August 2014
 *  @author Copyright (c) 2014 Geert Janssens <geert@kobaltwit.be>
 */
#ifndef GNC_SCRUBBUSINESS_H
#define GNC_SCRUBBUSINESS_H

#include "gnc-engine.h"

/** @name Cleanup functions for business objects
 * Provides the high-level API for checking and repairing ('scrubbing
 * clean') the various data objects used by the business functions.
 @{ */

/** The gncScrubBusinessLot() function makes sure that the indicated
 *    lot has all the correct properties required for a lot used in
 *    the business features.
 *
 *    Currently this function only does one thing: eliminate lot link
 *    transactions between invoice lots and payment lots (which were
 *    generated by GnuCash versions 2.6.0-2.6.3). Lot links
 *    between invoices and credit notes will still remain.
 *
 *    Scrubbing the lot may cause subsplits to be merged together,
 *    i.e. for splits to be deleted.  This routine returns true if
 *    any splits were modified or deleted.
 */
gboolean gncScrubBusinessLot (GNCLot *lot);

/** The gncScrubBusinessSplit() function will fix all issues found with
 *    the given split.
 *
 *    Currently this function only does one thing: check if the split is
 *    part of a transaction that was generated as the result of a doubly
 *    posted invoice/bill/credit note. Refer to
 *    https://bugzilla.gnome.org/show_bug.cgi?id=754209 to learn how this
 *    could have happened in the past.
 *    If such a transaction is found, its read-only status is removed and
 *    a warning is written to the trace file. Considering the user may
 *    already have added a correcting transaction we leave it up to the user
 *    to decide whether to also delete the transaction or not.
 */
void gncScrubBusinessSplit (Split *split);

/** The gncScrubBusinessAccountLots() function will call
 *    gncScrubBusinessLot() on each lot in the given account.
 *
 *    This routine is the primary routine for ensuring that the
 *    lot structure of every lot of a business account is in good
 *    order.
 */
void gncScrubBusinessAccountLots (Account *acc);

/** The gncScrubBusinessAccountSplits() function will call
 *    gncScrubBusinessSplit() on each split in the given account.
 */
void gncScrubBusinessAccountSplits (Account *acc);

/** The gncScrubBusinessAccount() function will call
 *    all scrub functions relevant for a given account
 *    on condition the account is a business related account
 *    (Accounts Receivable or Accounts Payable type).
 *
 *    This routine is the primary routine for fixing all
 *    (known) issues in a business account.
 */
void gncScrubBusinessAccount (Account *acc);

/** The gncScrubBusinessAccountTreeLots() function will call
 *    gncScrubBusinessAccount() on the given account
 *    and its sub accounts.
 */
void gncScrubBusinessAccountTree (Account *acc);

/** @} */
#endif /* GNC_SCRUBBUSINESS_H */
/** @} */
/** @} */
