# Volatility
# Copyright (C) 2011-2013 Volatility Foundation
# Copyright (C) 2011 Jamie Levy (Gleeda) <jamie@memoryanalysis.net>
#
# This file is part of Volatility.
#
# Volatility is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Volatility is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Volatility.  If not, see <http://www.gnu.org/licenses/>.
#

"""
@author:       Jamie Levy (Gleeda)
@license:      GNU General Public License 2.0
@contact:      jamie@memoryanalysis.net
@organization: Volatility Foundation
"""

import volatility.win32.rawreg as rawreg
import volatility.debug as debug
import volatility.plugins.registry.registryapi as registryapi
import volatility.plugins.common as common
from volatility.renderers import TreeGrid
import hashlib
import struct

# This is a dictionary of default services from Vista+ machines

servicesids = {
    'S-1-5-80-3476726845-1218940557-3240126423-1396283824-3706223860': '.NET CLR Data',
    'S-1-5-80-3749761688-76038143-2425834820-4129736068-309120712': '.NET CLR Networking',
    'S-1-5-80-603392709-3706100282-1779817366-3290147925-2109454977': '.NET Data Provider for Oracle',
    'S-1-5-80-1168016597-2140435647-491797002-352772175-817350590': '.NET Data Provider for SqlServer',
    'S-1-5-80-255220978-1106536095-1636044468-311807000-281316439': '.NETFramework',
    'S-1-5-80-799694863-4024754253-4060439485-3284853837-2852070736': '1394ohci',
    'S-1-5-80-550892281-1246201444-2906082186-2301917840-2280485454': 'ACPI',
    'S-1-5-80-2750316143-92726786-3671103447-4285640526-595803658': 'AcpiPmi',
    'S-1-5-80-4277731759-3688284049-1726419820-405794046-874834352': 'adp94xx',
    'S-1-5-80-1668430318-2462354215-3771841206-4231263990-2365432302': 'adpahci',
    'S-1-5-80-1558789706-915067316-2610504951-4085128407-2746609837': 'adpu320',
    'S-1-5-80-2580340827-1408356417-1236233457-3361088231-1362281560': 'adsi',
    'S-1-5-80-1452425288-2709461340-3274533413-2407537074-986069024': 'AeLookupSvc',
    'S-1-5-80-958185937-3813565417-3041720555-255702914-2218388865': 'AFD',
    'S-1-5-80-1478021307-2683864309-2840291008-2654641652-1914939368': 'agp440',
    'S-1-5-80-2964793103-1312530465-1873688160-795174673-2945876561': 'aic78xx',
    'S-1-5-80-2387347252-3645287876-2469496166-3824418187-3586569773': 'ALG',
    'S-1-5-80-1587539839-2488332913-1287008632-3751426284-4220573165': 'aliide',
    'S-1-5-80-2808999507-317517852-2612044860-3916887390-3713671788': 'amdagp',
    'S-1-5-80-4100430975-1934021090-490597466-3817433801-2954987127': 'amdide',
    'S-1-5-80-2291534435-3322220689-2735625597-3465650106-1340236923': 'AmdK8',
    'S-1-5-80-4046459391-4016695280-780100908-1621843708-2839135617': 'AmdPPM',
    'S-1-5-80-1967003600-1747618720-202510732-1118110944-2056302645': 'amdsata',
    'S-1-5-80-3946629880-3877146532-1020811794-3209710663-3707805237': 'amdsbs',
    'S-1-5-80-2663151763-304964558-3327380674-1150567875-3378868591': 'amdxata',
    'S-1-5-80-4206070390-3011771559-4179333097-3486196663-2896243697': 'AppID',
    'S-1-5-80-2078495744-2416903469-4072184685-3943858305-976987417': 'AppIDSvc',
    'S-1-5-80-1345931346-2714066941-3624776837-1617505694-3927660246': 'Appinfo',
    'S-1-5-80-3213379692-3546485254-1309469428-3810262102-2442199571': 'AppMgmt',
    'S-1-5-80-2586396289-3967100905-3140788560-3910242148-3554126937': 'arc',
    'S-1-5-80-4275531960-1601664531-2254151532-3075236607-956726506': 'arcsas',
    'S-1-5-80-3772676405-1029441937-3739550121-1000989080-3364480489': 'AsyncMac',
    'S-1-5-80-3126347352-2401679295-1536073615-3396758597-3783091149': 'atapi',
    'S-1-5-80-1580948945-3239616721-2529237571-3761093093-1214243633': 'AudioEndpointBuilder',
    'S-1-5-80-2676549577-1911656217-2625096541-4178041876-1366760775': 'Audiosrv',
    'S-1-5-80-1058592404-331734164-3167594226-3910907650-1299295147': 'AxInstSV',
    'S-1-5-80-1401731874-3996074688-1963706087-3130220608-1140295258': 'b06bdrv',
    'S-1-5-80-528874604-3378394362-3426265968-3876211711-2956305666': 'b57nd60x',
    'S-1-5-80-2490514847-2461341327-10008697-1811907875-602803682': 'BattC',
    'S-1-5-80-2962817144-200689703-2266453665-3849882635-1986547430': 'BDESVC',
    'S-1-5-80-3186183977-1861961257-3523979229-167170737-1516062821': 'Beep',
    'S-1-5-80-1383147646-27650227-2710666058-1662982300-1023958487': 'BFE',
    'S-1-5-80-864916184-135290571-3087830041-1716922880-4237303741': 'BITS',
    'S-1-5-80-3199704608-2688121514-1535149675-608666402-3313731745': 'blbdrive',
    'S-1-5-80-26818074-245702967-483560604-1005139437-3076944027': 'bowser',
    'S-1-5-80-1926592986-1411939489-3259133927-4064956769-2216240612': 'BrFiltLo',
    'S-1-5-80-3843808474-1199403037-3395254522-1605808544-3221186762': 'BrFiltUp',
    'S-1-5-80-764937145-223273921-1726433829-265908364-3948077829': 'Browser',
    'S-1-5-80-3715020542-2003794336-3716799247-4001019941-1245790858': 'Brserid',
    'S-1-5-80-4014097382-2743177720-3750454595-1699596626-866516122': 'BrSerWdm',
    'S-1-5-80-1195671069-1048138941-897119314-1432864274-834752102': 'BrUsbMdm',
    'S-1-5-80-1736549233-1399426098-2600293700-2473969234-3259996387': 'BrUsbSer',
    'S-1-5-80-505608135-4274227953-3632766965-1888639892-3184055934': 'BTHMODEM',
    'S-1-5-80-1409084391-1870647740-2731517552-2815089321-2189562539': 'BTHPORT',
    'S-1-5-80-2586557155-168560303-1373426920-983201488-1499765686': 'bthserv',
    'S-1-5-80-3223837281-1527595016-2901219760-1358189227-808820507': 'cdfs',
    'S-1-5-80-364680967-1232085744-2960737863-915504889-2752576923': 'cdrom',
    'S-1-5-80-3256172449-2363790065-3617575471-4144056108-756904704': 'CertPropSvc',
    'S-1-5-80-4066704878-4231214995-2335031091-3527122690-1574766183': 'circlass',
    'S-1-5-80-1506673549-1532669541-769420574-1605323189-863873827': 'CLFS',
    'S-1-5-80-776041216-1751974135-1557427478-1892253070-796752000': 'clr_optimization_v2.0.50727_32',
    'S-1-5-80-452204072-1743664639-1560983493-2640850116-597529692': 'CmBatt',
    'S-1-5-80-979911607-31916023-2827320217-2656655436-259985251': 'cmdide',
    'S-1-5-80-3573738861-3694853854-361022443-2442358023-2743921644': 'CNG',
    'S-1-5-80-3960644792-2999129865-644014482-29643289-3842828219': 'Compbatt',
    'S-1-5-80-832194277-1022982267-2217674263-2896671990-3011983110': 'CompositeBus',
    'S-1-5-80-593875016-1044814911-1112741138-2143646632-2690613739': 'COMSysApp',
    'S-1-5-80-3158764370-1001901224-1854525633-1718604346-2756706540': 'crcdisk',
    'S-1-5-80-3747264324-1669729390-1715156009-1010652712-2439569381': 'Crusoe',
    'S-1-5-80-3020380856-1381845346-309829523-1810616773-418643442': 'crypt32',
    'S-1-5-80-242729624-280608522-2219052887-3187409060-2225943459': 'CryptSvc',
    'S-1-5-80-3601020880-2087999432-167179594-730776211-2997520967': 'CSC',
    'S-1-5-80-1987853863-1639573247-1110726908-1137832616-3599624523': 'CscService',
    'S-1-5-80-1564160128-141119064-743480990-78466790-746535033': 'DCLocator',
    'S-1-5-80-1601830629-990752416-3372939810-977361409-3075122917': 'DcomLaunch',
    'S-1-5-80-654447679-1163530548-981569129-3608673666-3128964045': 'defragsvc',
    'S-1-5-80-3837255464-839197112-3211601036-3795322556-2690640524': 'DfsC',
    'S-1-5-80-1267473060-1890374259-1137250836-544356534-2546457154': 'DFSR',
    'S-1-5-80-2940520708-3855866260-481812779-327648279-1710889582': 'Dhcp',
    'S-1-5-80-2142581517-3954605861-2373846864-2138305209-1019737370': 'discache',
    'S-1-5-80-1827140278-1118305254-4004251663-1512899043-4081885502': 'Disk',
    'S-1-5-80-859482183-879914841-863379149-1145462774-2388618682': 'Dnscache',
    'S-1-5-80-3787436395-2174616005-3003730137-1094982900-1570567328': 'dot3svc',
    'S-1-5-80-2970612574-78537857-698502321-558674196-1451644582': 'DPS',
    'S-1-5-80-338020179-181244551-1629881386-919369987-4169324252': 'drmkaud',
    'S-1-5-80-3820654016-1545322283-1804062181-1022271772-3696306321': 'DXGKrnl',
    'S-1-5-80-2212058837-3965059022-779215765-3282659977-917192320': 'E1G60',
    'S-1-5-80-3578261754-285310837-913589462-2834155770-667502746': 'EapHost',
    'S-1-5-80-2437473203-2648204866-3612751994-635271166-3967841232': 'Ecache',
    'S-1-5-80-1191957972-1903257272-3657591267-1787121440-2523964525': 'ebdrv',
    'S-1-5-80-730263862-4055390735-403826019-1175694336-1277635259': 'EFS',
    'S-1-5-80-567955335-3455378119-3305749985-2554534624-1867504835': 'ehRecvr',
    'S-1-5-80-3864065939-1897331054-469427076-3133256761-1570309435': 'ehSched',
    'S-1-5-80-2913099195-3001839937-1914692661-1563395363-459793767': 'ehstart',
    'S-1-5-80-3118383011-3159412168-3368304685-4081854189-1392756948': 'elxstor',
    'S-1-5-80-1436322865-2295268783-31549072-3549518694-69512146': 'EmdCache',
    'S-1-5-80-557382581-4103702789-1349398007-826115979-1301810884': 'EMDMgmt',
    'S-1-5-80-1580004045-3657569029-3054886754-3760858607-1347140441': 'ErrDev',
    'S-1-5-80-1163726475-4032819940-2637749356-1655080563-3495319901': 'ESENT',
    'S-1-5-80-880578595-1860270145-482643319-2788375705-1540778122': 'eventlog',
    'S-1-5-80-1772571935-1555666882-3369284645-1675012128-2386634627': 'EventSystem',
    'S-1-5-80-339744372-1785209941-194342311-2969164887-2874010346': 'exfat',
    'S-1-5-80-3825849991-4144931059-247537738-1429287757-2349637904': 'fastfat',
    'S-1-5-80-2117685068-4011115449-2646761356-2137676340-222423812': 'Fax',
    'S-1-5-80-678085088-615808128-1967178352-3804608619-208504977': 'fdc',
    'S-1-5-80-364023826-931424190-487969545-1024119571-74567675': 'fdPHost',
    'S-1-5-80-3215268152-2863950836-530904203-4246843131-2183915461': 'FDResPub',
    'S-1-5-80-3048209083-3162952562-941345871-1437532549-835501875': 'FileInfo',
    'S-1-5-80-1352441077-2188484239-1994186818-620473926-3758853310': 'Filetrace',
    'S-1-5-80-2678475722-3718149211-1393662077-3558562392-2203603517': 'flpydisk',
    'S-1-5-80-916285479-1714977700-1732101595-331036679-1735462769': 'FltMgr',
    'S-1-5-80-3655275221-2954682349-3644260495-855223267-1438849333': 'FontCache',
    'S-1-5-80-3782458156-2098404076-3767342964-3617937256-1389734963': 'FontCache3.0.0.0',
    'S-1-5-80-4244156434-496195918-1908400060-3754471672-3389379472': 'FsDepends',
    'S-1-5-80-1638897150-273717933-3197303335-567190659-606579740': 'Fs_Rec',
    'S-1-5-80-221025945-1494805562-2841517651-3196795133-192498206': 'fvevol',
    'S-1-5-80-1150850083-1108777032-2236282716-3985597815-2701820264': 'gagp30kx',
    'S-1-5-80-2024188204-2445810227-898691311-2942020084-762398166': 'gpsvc',
    'S-1-5-80-2384017851-2441776339-3346382083-2430645704-3475981877': 'hcw85cir',
    'S-1-5-80-2193151998-1100362924-2192368770-2985476713-896696503': 'HDAudBus',
    'S-1-5-80-1648434057-4219984261-1802816958-334501717-1769477291': 'HidBatt',
    'S-1-5-80-191977210-1053814073-2805336524-1775407748-120039257': 'HidBth',
    'S-1-5-80-498696395-104441048-3395182230-3082814586-1375447691': 'HidIr',
    'S-1-5-80-89818136-74175777-88572358-3912780041-2421659406': 'hidserv',
    'S-1-5-80-1586586559-167648910-1414982260-3863830924-1724542190': 'HidUsb',
    'S-1-5-80-1373701630-3910968185-3388013410-2492353-937432973': 'hkmsvc',
    'S-1-5-80-2291748755-1591405548-1905550586-2340871825-1258388485': 'HpCISSs',
    'S-1-5-80-4028305664-2774326660-44957573-2454826285-2129126537': 'HomeGroupListener',
    'S-1-5-80-2620923248-4247863784-3378508180-2659151310-2535246811': 'HomeGroupProvider',
    'S-1-5-80-3952044490-1864224763-1322162546-396143671-1619397437': 'HpSAMD',
    'S-1-5-80-3734987283-965611577-2130035942-3636592211-2616856863': 'HTTP',
    'S-1-5-80-970016657-3034632851-3048190821-4182690298-3323420226': 'i2omp',
    'S-1-5-80-3096896632-2411553352-2084109408-2930423838-4282791216': 'hwpolicy',
    'S-1-5-80-738727139-3255065492-2264176241-1836141076-1899426695': 'i8042prt',
    'S-1-5-80-1156567179-1019273932-444819734-1772733284-2107707318': 'iaStorV',
    'S-1-5-80-2984992224-2588614340-2167448307-2303456600-125847566': 'idsvc',
    'S-1-5-80-3218395955-317132717-2440444880-267201483-2700625476': 'iirsp',
    'S-1-5-80-698886940-375981264-2691324669-2937073286-3841916615': 'IKEEXT',
    'S-1-5-80-3217419572-1740605331-1127140686-2317006352-2064317000': 'inetaccs',
    'S-1-5-80-3664101217-2276051299-423734030-2746486177-2766044424': 'intelide',
    'S-1-5-80-817570274-767070440-2629795609-3336305482-1678804590': 'intelppm',
    'S-1-5-80-2506443892-94066030-1663014834-2885971264-4189966690': 'IPBusEnum',
    'S-1-5-80-2750735467-3008441591-3989401642-3215998983-1344927289': 'IpFilterDriver',
    'S-1-5-80-62724632-2456781206-3863850748-1496050881-1042387526': 'iphlpsvc',
    'S-1-5-80-1361160473-1867727628-1338406996-3302040194-2851723982': 'IpInIp',
    'S-1-5-80-2771164118-4094026282-2266286801-3306161409-3436440840': 'IPMIDRV',
    'S-1-5-80-2368102602-26431353-856636621-1497418614-482242802': 'IPNAT',
    'S-1-5-80-433158070-3235422099-1317741036-1922328546-1834106188': 'IRENUM',
    'S-1-5-80-1308614567-1511795785-2741360970-8197000-3264788676': 'isapnp',
    'S-1-5-80-1446792217-3918178545-2165441202-3760590537-1875255596': 'iScsiPrt',
    'S-1-5-80-2249099846-2157059493-1994460756-1924820827-2369096692': 'iteatapi',
    'S-1-5-80-750512324-770881543-4197932906-3645560491-3779161573': 'iteraid',
    'S-1-5-80-1974511938-2400693546-1685170019-203554928-1466978163': 'kbdclass',
    'S-1-5-80-3058542000-3285469617-40650340-3734485625-1920508542': 'kbdhid',
    'S-1-5-80-1206118541-1677721718-2423781911-3372378849-3903984073': 'KeyIso',
    'S-1-5-80-3810688523-3855579666-1860693470-2666993558-46302070': 'KSecDD',
    'S-1-5-80-638937566-1168471176-3064579757-2631269312-170126454': 'KSecPkg',
    'S-1-5-80-2818357584-3387065753-4000393942-342927828-138088443': 'KtmRm',
    'S-1-5-80-879696042-2351668846-370232824-2524288904-4023536711': 'LanmanServer',
    'S-1-5-80-719998295-2833700043-1566817583-4093942769-1414026312': 'LanmanWorkstation',
    'S-1-5-80-3356507721-3148410333-1453554623-2317622189-363686743': 'ldap',
    'S-1-5-80-1339741203-2503426401-303705627-250156843-1210515524': 'lltdio',
    'S-1-5-80-940647296-341435850-43817331-158078607-2483727905': 'lltdsvc',
    'S-1-5-80-172094073-716411664-54255058-185476446-2329512179': 'lmhosts',
    'S-1-5-80-1037107160-813189200-1860894220-2610408748-1807657940': 'Lsa',
    'S-1-5-80-973905250-3368826558-2408393701-2645888229-3042295110': 'LSI_FC',
    'S-1-5-80-3066312493-2787136058-3895654580-111488809-2262703568': 'LSI_SAS',
    'S-1-5-80-935126585-3333887566-2369146147-2658756633-3860083864': 'LSI_SAS2',
    'S-1-5-80-702453548-2563122194-4165184037-877730421-2039909086': 'LSI_SCSI',
    'S-1-5-80-381203785-1552481550-3565819581-4159540168-38965703': 'luafv',
    'S-1-5-80-3770938798-2726624435-2075025292-3280341113-3618470894': 'Mcx2Svc',
    'S-1-5-80-1503963800-3543347063-2443146678-2767313893-605308357': 'megasas',
    'S-1-5-80-4024713676-1017792628-381990976-3540878265-1306153904': 'MegaSR',
    'S-1-5-80-2799810402-4136494038-1094338311-2889966999-3154753985': 'MMCSS',
    'S-1-5-80-2005225957-2795451222-469338742-3947262705-2044891099': 'Modem',
    'S-1-5-80-4207690787-1085901060-2295361997-2227230598-1253819078': 'monitor',
    'S-1-5-80-675551267-1826535266-117093185-28668227-296166608': 'mouclass',
    'S-1-5-80-3854853272-3832246511-1244659077-3165440039-2262758429': 'mouhid',
    'S-1-5-80-3601998905-441174471-4117363912-32772110-2632366064': 'mountmgr',
    'S-1-5-80-4261667920-1220466518-1749771309-2316901739-273317064': 'mpio',
    'S-1-5-80-3142377179-3443479297-2149323391-1756545698-484011292': 'mpsdrv',
    'S-1-5-80-3088073201-1464728630-1879813800-1107566885-823218052': 'MpsSvc',
    'S-1-5-80-2250298043-1491746124-3447101336-2334414474-2555807208': 'Mraid35x',
    'S-1-5-80-2688027615-1506195528-3802338144-777155390-618458321': 'MRxDAV',
    'S-1-5-80-2162099894-1456621096-2119874347-3743340265-2368304946': 'mrxsmb',
    'S-1-5-80-2676550360-252586896-1701879715-2742386574-1171030092': 'mrxsmb10',
    'S-1-5-80-3970894941-767821303-4047113619-2738918178-2351404876': 'mrxsmb20',
    'S-1-5-80-276420989-3971400029-4249224515-3588854300-972083571': 'msahci',
    'S-1-5-80-827450036-3359053657-3286484322-221598818-2985401197': 'msdsm',
    'S-1-5-80-3960419045-2460139048-4046793004-1809597027-2250574426': 'MSDTC',
    'S-1-5-80-1515650939-3601430262-2496924429-640160050-3998290523': 'MSDTC Bridge 3.0.0.0',
    'S-1-5-80-3825916667-3375043415-3384654478-3177665693-2200644784': 'Msfs',
    'S-1-5-80-4064639957-1408283007-2091294018-2122350837-1986927883': 'mshidkmdf',
    'S-1-5-80-537088188-2896597613-2307397767-3752262660-2081934664': 'msisadrv',
    'S-1-5-80-917953661-2020045820-2727011118-2260243830-4032185929': 'MSiSCSI',
    'S-1-5-80-685333868-2237257676-1431965530-1907094206-2438021966': 'msiserver',
    'S-1-5-80-1314579368-1827054856-3801607513-4137797117-3785845944': 'MSKSSRV',
    'S-1-5-80-3515336427-2373706795-1189292716-3451446183-2383180522': 'MSPCLOCK',
    'S-1-5-80-2550581486-1497628998-1973453189-3108482975-2816921478': 'MSPQM',
    'S-1-5-80-4273119239-1126992662-2069961181-78804100-786965295': 'MsRPC',
    'S-1-5-80-2731410647-2404537004-1422510964-3385838496-1398925663': 'MSSCNTRS',
    'S-1-5-80-2379877105-2122874852-2028670630-1350450415-3977667049': 'mssmbios',
    'S-1-5-80-294111013-494549581-4136661504-3518049416-761106507': 'MSTEE',
    'S-1-5-80-772196467-3194495650-2141286422-1986870660-3602995159': 'MTConfig',
    'S-1-5-80-2851636321-923882121-3805946377-1773657562-2703951580': 'Mup',
    'S-1-5-80-2006800713-1441093265-249754844-3404434343-1444102779': 'napagent',
    'S-1-5-80-3451137062-797777108-3464068327-231871278-2024511519': 'NativeWifiP',
    'S-1-5-80-2183409222-222800135-1539000935-3109909370-1207982808': 'NDIS',
    'S-1-5-80-1310191460-362243386-72972191-123604350-1188038626': 'NdisCap',
    'S-1-5-80-3307576507-4040802919-832577921-47721884-821370673': 'NdisTapi',
    'S-1-5-80-2426641292-1095310648-1538795067-2456674997-547968854': 'Ndisuio',
    'S-1-5-80-3137956796-3050520361-1309400342-955303752-3583020413': 'NdisWan',
    'S-1-5-80-3999445478-1493703614-491198216-2250085872-3662815299': 'NDProxy',
    'S-1-5-80-298519744-3326885196-200884095-1345730765-1206919721': 'NetBIOS',
    'S-1-5-80-3481163626-3922336224-2171110286-845444925-873416656': 'NetBT',
    'S-1-5-80-1589317753-1926951874-3424712441-2302911845-2572860984': 'Netlogon',
    'S-1-5-80-2898649604-2335086160-1904548223-3761738420-3855444835': 'Netman',
    'S-1-5-80-3635958274-2059881490-2225992882-984577281-633327304': 'netprofm',
    'S-1-5-80-1773860938-1487242074-882566118-4272343956-2175834232': 'NetTcpPortSharing',
    'S-1-5-80-3739586395-593861784-2557645679-4197025642-341497066': 'nfrd960',
    'S-1-5-80-3141615172-2057878085-1754447212-2405740020-3916490453': 'NlaSvc',
    'S-1-5-80-1093399993-2276725296-2148262981-2274078422-4284582767': 'Npfs',
    'S-1-5-80-2310782386-4237065203-3688974353-390202159-3511571085': 'nsi',
    'S-1-5-80-4100249314-4086313984-28913695-873679419-2144728263': 'nsiproxy',
    'S-1-5-80-1664281202-2302623734-631624840-3461998672-2259661997': 'NTDS',
    'S-1-5-80-1256884789-1691082103-446998474-1367286246-1639025938': 'Ntfs',
    'S-1-5-80-2470698091-2858014709-2643764839-982706939-3434751516': 'ntrigdigi',
    'S-1-5-80-2407861648-785230825-3529290450-2326204529-1810679516': 'Null',
    'S-1-5-80-3495072887-919096479-2204902451-1048921326-800355041': 'nvraid',
    'S-1-5-80-3611874924-3178792031-3565391826-286563291-3680247785': 'nvstor',
    'S-1-5-80-2661219475-1923594960-1294537542-2454943126-82436970': 'nv_agp',
    'S-1-5-80-4169196349-563482612-2169411968-43761830-802868667': 'NwlnkFlt',
    'S-1-5-80-1643415749-1981533051-3884744798-2669202348-601031005': 'NwlnkFwd',
    'S-1-5-80-1196941233-2569882653-2923823926-962244991-4277418': 'ohci1394',
    'S-1-5-80-967499406-1694984581-2959056265-2481940682-939264259': 'p2pimsvc',
    'S-1-5-80-1971585524-2528565899-3324366483-1300752743-2325226580': 'p2psvc',
    'S-1-5-80-3473791808-4104434288-1928902041-1743473672-1277326840': 'Parport',
    'S-1-5-80-156989346-1343554423-902067029-1673992682-1866693543': 'partmgr',
    'S-1-5-80-4196153372-502005009-1971508045-3354250645-3015555128': 'Parvdm',
    'S-1-5-80-1948712186-1330865447-943413596-1669284603-1648638051': 'PcaSvc',
    'S-1-5-80-2069178898-4023461412-1711560041-390887617-271771820': 'pci',
    'S-1-5-80-4052642423-944120264-588619640-546327341-1110646568': 'pciide',
    'S-1-5-80-2795309555-3957969320-2916397881-2593713121-382316838': 'pcmcia',
    'S-1-5-80-59707871-3298565586-1716270302-948228651-1074156479': 'pcw',
    'S-1-5-80-1570874813-103103538-3327933986-104584388-2119773521': 'PEAUTH',
    'S-1-5-80-3124040864-3101396827-3094488734-3028845762-1939139329': 'PeerDistSvc',
    'S-1-5-80-4023986828-1464965280-3211893748-414212150-4115790068': 'PerfDisk',
    'S-1-5-80-2413971036-1590988147-3808667159-2204172745-1373631640': 'PerfNet',
    'S-1-5-80-3515570427-2977692895-3762163048-1504969852-99088878': 'PerfOS',
    'S-1-5-80-3544016446-4087985546-3773506770-1472693371-3235341583': 'PerfProc',
    'S-1-5-80-2661322625-712705077-2999183737-3043590567-590698655': 'pla',
    'S-1-5-80-1981970923-922788642-3535304421-2999920573-318732269': 'PlugPlay',
    'S-1-5-80-3141781312-1794533130-3616533224-2008760771-2116720301': 'PNRPAutoReg',
    'S-1-5-80-372467825-374176116-1198570892-3192490889-1232022613': 'PNRPsvc',
    'S-1-5-80-3044542841-3639452079-4096941652-1606687743-1256249853': 'PolicyAgent',
    'S-1-5-80-4126081702-1836807445-3803306975-1029803806-2479180530': 'PortProxy',
    'S-1-5-80-2343416411-2961288913-598565901-392633850-2111459193': 'Power',
    'S-1-5-80-3735226416-1729687437-1959510470-190511368-398645692': 'PptpMiniport',
    'S-1-5-80-3367479018-119754134-174380200-3035551807-2744700953': 'Processor',
    'S-1-5-80-2422153244-111630262-1029994140-3645224535-4078427153': 'PROCEXP',
    'S-1-5-80-3816717743-33564931-1112267079-3548917561-928358339': 'ProfSvc',
    'S-1-5-80-656433041-336319937-100815201-2263438610-4002557366': 'ProtectedStorage',
    'S-1-5-80-133730547-3458667493-930392497-3658715967-3359215708': 'Psched',
    'S-1-5-80-1010784341-3590640432-2144716203-2371202623-2111191834': 'ql2300',
    'S-1-5-80-3680784227-2138494325-1045417256-846249285-1494284974': 'ql40xx',
    'S-1-5-80-1659118645-3148100556-861291880-3953320898-4045657812': 'QWAVE',
    'S-1-5-80-3324762131-3390532780-137711907-1761928331-1932425801': 'QWAVEdrv',
    'S-1-5-80-951069737-1097907447-3199478753-2018050253-2083677786': 'RasAcd',
    'S-1-5-80-4022575210-2284560452-710265691-3594820739-387418549': 'RasAgileVpn',
    'S-1-5-80-1802467488-1541022566-2033325545-854566965-652742428': 'RasAuto',
    'S-1-5-80-1290287420-3502600185-382990664-1700026297-1337626153': 'Rasl2tp',
    'S-1-5-80-4176366874-305252471-2256717057-2714189771-3552532790': 'RasMan',
    'S-1-5-80-4122454071-3550668693-4211410744-1298358403-2272725717': 'RasPppoe',
    'S-1-5-80-1331337031-2474836174-2661672254-391271513-2096420174': 'RasSstp',
    'S-1-5-80-2489667-2470848582-3865645512-452901963-4178804252': 'rdbss',
    'S-1-5-80-3687944073-3313860148-3136628839-3387249243-1709534714': 'rdpbus',
    'S-1-5-80-2431288241-149984296-2543083935-4067350611-1975817884': 'RDPCDD',
    'S-1-5-80-981872547-3861006530-3984275202-4085961120-2027028908': 'RDPDD',
    'S-1-5-80-23661045-4033652049-3526044993-1401805078-1749661838': 'RDPDR',
    'S-1-5-80-3464459778-79086046-1894495498-3954672505-2750168721': 'RDPENCDD',
    'S-1-5-80-191927475-3325244020-2133763035-2511185485-3827563125': 'RDPNP',
    'S-1-5-80-1432111213-2818786930-2152807080-3377190559-901933699': 'RDPREFMP',
    'S-1-5-80-1857653372-1313752195-3783661666-502273730-1171188227': 'RDPWD',
    'S-1-5-80-3474873350-2412947251-3085823233-2315640422-3546857610': 'rdyboost',
    'S-1-5-80-1954729425-4294152082-187165618-318331177-3831297489': 'RemoteAccess',
    'S-1-5-80-2822507136-3601578665-1013168651-121944544-1825232178': 'RemoteRegistry',
    'S-1-5-80-521322694-906040134-3864710659-1525148216-3451224162': 'RpcEptMapper',
    'S-1-5-80-4056015446-1496461683-1723632270-3351149576-1119802320': 'RpcLocator',
    'S-1-5-80-979556362-403687129-3954533659-2335141334-1547273080': 'RpcSs',
    'S-1-5-80-25112808-303066962-2306571906-3820953744-554449017': 'rspndr',
    'S-1-5-80-3189092957-1825937568-2097962828-592273195-15751640': 's3cap',
    'S-1-5-80-3453257571-682267348-3447719424-2810041157-893746920': 'SamSs',
    'S-1-5-80-2172748946-1139208647-3745649895-1734051075-2323558886': 'sbp2port',
    'S-1-5-80-1209419826-1829913269-3824447628-1153237837-3789837839': 'SCardSvr',
    'S-1-5-80-3145502940-3408664484-1477142494-2517801300-3177717725': 'scfilter',
    'S-1-5-80-4125092361-1567024937-842823819-2091237918-836075745': 'Schedule',
    'S-1-5-80-1691538513-4084330536-1620899472-1113280783-3554754292': 'SCPolicySvc',
    'S-1-5-80-2983134835-1185273323-1712700529-1489848661-2325612824': 'SDRSVC',
    'S-1-5-80-1722176216-3611007545-3657005850-3814612847-1080390000': 'secdrv',
    'S-1-5-80-1399994486-219206332-302438500-304602034-1537790326': 'seclogon',
    'S-1-5-80-4259241309-1822918763-1176128033-1339750638-3428293995': 'SENS',
    'S-1-5-80-3168472476-176724102-2968832672-2340942973-2241613192': 'SensrSvc',
    'S-1-5-80-1658387481-2925800327-3198882180-3147662777-2274689045': 'Serenum',
    'S-1-5-80-3562253942-857828347-2712713407-944836455-3636585461': 'Serial',
    'S-1-5-80-3369720968-4228855631-3683183521-2094993598-1022421131': 'sermouse',
    'S-1-5-80-675414407-775065359-1035864904-999747831-2072146957': 'ServiceModelEndpoint 3.0.0.0',
    'S-1-5-80-1904953591-2738210791-1061154185-3936071259-221446881': 'ServiceModelOperation 3.0.0.0',
    'S-1-5-80-297390187-2405189348-2222284465-2989988878-4218767654': 'ServiceModelService 3.0.0.0',
    'S-1-5-80-4022436659-1090538466-1613889075-870485073-3428993833': 'SessionEnv',
    'S-1-5-80-1220365695-3871163487-2301282001-885120026-718998505': 'sffdisk',
    'S-1-5-80-1593449009-2408870187-1077724223-1518188577-3728252823': 'sffp_mmc',
    'S-1-5-80-1659054941-531967795-1983128084-3748020815-2241757750': 'sffp_sd',
    'S-1-5-80-1407380289-3518059920-3931497022-2754447733-2222417609': 'sfloppy',
    'S-1-5-80-2009329905-444645132-2728249442-922493431-93864177': 'SharedAccess',
    'S-1-5-80-1690854464-3758363787-3981977099-3843555589-1401248062': 'ShellHWDetection',
    'S-1-5-80-2037654479-150732571-4235160932-1988269395-3027078133': 'sisagp',
    'S-1-5-80-2290943609-1211775869-3660739483-1432647055-1639441565': 'SiSRaid2',
    'S-1-5-80-1016766434-4163349990-2054491751-1265000292-413406215': 'SiSRaid4',
    'S-1-5-80-2119565420-4155874467-2934723793-509086461-374458824': 'slsvc',
    'S-1-5-80-429025866-4105586292-427562881-1309981334-1060966148': 'SLUINotify',
    'S-1-5-80-97513841-1071082959-3069755588-526311685-2961431215': 'Smb',
    'S-1-5-80-2400470686-1781479961-2091307112-2920730856-2901594176': 'SMSvcHost 3.0.0.0',
    'S-1-5-80-3964583643-2633443559-2834438935-3739664028-1580655619': 'SNMPTRAP',
    'S-1-5-80-2246094146-3761615012-3991572358-959820157-1291755210': 'spldr',
    'S-1-5-80-3951239711-1671533544-1416304335-3763227691-3930497994': 'Spooler',
    'S-1-5-80-123231216-2592883651-3715271367-3753151631-4175906628': 'sppsvc',
    'S-1-5-80-2105443381-1869407242-828286827-1344996006-2512971347': 'sppuinotify',
    'S-1-5-80-3318989984-2647182497-3022510041-1919214433-3551303480': 'srv',
    'S-1-5-80-1034188721-156321652-2901307485-3049929104-2850741453': 'srv2',
    'S-1-5-80-385674269-2427993094-4248660116-187565782-2803330530': 'srvnet',
    'S-1-5-80-486568272-975562994-1883531608-2732234258-332540751': 'SSDPSRV',
    'S-1-5-80-3435701886-799518250-3791383489-3228296122-2938884314': 'SstpSvc',
    'S-1-5-80-2502136977-515215333-1091199184-4078967732-698071891': 'stexstor',
    'S-1-5-80-3182985763-1431228038-2757062859-428472846-3914011746': 'StiSvc',
    'S-1-5-80-3877927215-2009774003-1789373229-1350139498-1490546062': 'storflt',
    'S-1-5-80-3355894222-2288616474-3163838539-1515771758-43395969': 'StorSvc',
    'S-1-5-80-2227193670-1472088527-4216801891-1255609005-3742950393': 'storvsc',
    'S-1-5-80-2499453150-1816575225-2698105218-861119070-2299588587': 'swenum',
    'S-1-5-80-1614360071-3471039648-1078047007-3707138327-1664821506': 'swprv',
    'S-1-5-80-3277458932-3608563558-2424252742-1006353051-3439664691': 'Symc8xx',
    'S-1-5-80-714262929-1152213303-426872964-3738532716-4000887735': 'Sym_hi',
    'S-1-5-80-73616012-2741736120-1450548080-3749295283-3869351969': 'Sym_u3',
    'S-1-5-80-2590341223-3996088049-3993122417-23640849-324535191': 'SysMain',
    'S-1-5-80-949921180-3923668869-394927020-528789358-3592448931': 'TabletInputService',
    'S-1-5-80-4230913304-2206818457-801678004-120036174-1892434133': 'TapiSrv',
    'S-1-5-80-4167276341-681140529-2035857140-584847688-708058301': 'TBS',
    'S-1-5-80-2869215396-3426808149-752611693-425565463-2833823703': 'Tcpip',
    'S-1-5-80-842221325-3630721446-2015653073-424833842-1069621030': 'TCPIP6',
    'S-1-5-80-1243767512-207181711-1639953288-846964026-179032965': 'TCPIP6TUNNEL',
    'S-1-5-80-183440435-3873164873-1814133288-2746138770-1127128543': 'tcpipreg',
    'S-1-5-80-517380867-1805075581-15937331-3649701458-2279870393': 'TCPIPTUNNEL',
    'S-1-5-80-1205525636-1316560639-1871536985-2915653626-3847227622': 'TDPIPE',
    'S-1-5-80-2653571336-860310240-1707811817-3246300807-2032786575': 'TDTCP',
    'S-1-5-80-1811008277-2130293716-2312968959-3698054739-726352487': 'tdx',
    'S-1-5-80-600900383-3940208308-3622757659-1160125390-3717916961': 'TermDD',
    'S-1-5-80-446051430-1559341753-4161941529-1950928533-810483104': 'TermService',
    'S-1-5-80-1189432293-2777010110-2640223427-1344437502-1956879817': 'Themes',
    'S-1-5-80-56840347-690487168-3179794702-1332568925-762031181': 'THREADORDER',
    'S-1-5-80-537470750-3688389562-3749243086-269898693-579266445': 'TPAutoConnSvc',
    'S-1-5-80-1495131930-2676463755-2136540566-1190107536-2533052015': 'TPVCGateway',
    'S-1-5-80-768763963-4214222998-2156221936-2953597973-713500239': 'TrkWks',
    'S-1-5-80-956008885-3418522649-1831038044-1853292631-2271478464': 'TrustedInstaller',
    'S-1-5-80-602153688-1728218534-2156437410-2444491971-1703742505': 'TSDDD',
    'S-1-5-80-3250179172-3414919659-2784612865-1947102831-1832745880': 'tssecsrv',
    'S-1-5-80-3666930311-739912689-1101093007-1147922636-412121971': 'tunmp',
    'S-1-5-80-3579196564-3960183121-2393617881-1570124860-2153905208': 'tunnel',
    'S-1-5-80-3249175164-480052304-527258952-251146422-1017202920': 'uagp35',
    'S-1-5-80-4290168682-2694755981-2883756118-2205499398-4079537721': 'udfs',
    'S-1-5-80-2413584400-2834772909-3391057178-2993126719-4094614649': 'UGatherer',
    'S-1-5-80-900581847-2069635957-4095211819-2149323943-1216697729': 'UGTHRSVC',
    'S-1-5-80-997887591-2350776071-3817597635-4146973621-2526406719': 'UI0Detect',
    'S-1-5-80-4194149548-235381792-2829184477-3934495640-667433095': 'uliagpkx',
    'S-1-5-80-2051301031-3598501189-881763489-2611917303-2352103085': 'uliahci',
    'S-1-5-80-4294381996-3573690956-4084941264-2318251564-135754816': 'UlSata',
    'S-1-5-80-2849548708-3602852847-3953931013-1110249439-3333230880': 'ulsata2',
    'S-1-5-80-3018007626-163191633-622627787-1206491734-2917835273': 'umbus',
    'S-1-5-80-2029728201-2796881031-2302868875-2454600822-1203790938': 'UmPass',
    'S-1-5-80-2014626298-1656748749-3847481816-918933055-2469338456': 'UmRdpService',
    'S-1-5-80-448846144-1414373772-1578130625-718576682-2306699751': 'upnphost',
    'S-1-5-80-3724553804-53543757-2557641770-141295351-1687883918': 'usb',
    'S-1-5-80-4022141922-741376770-3260236731-1675477288-3792235576': 'usbccgp',
    'S-1-5-80-2601879200-4032607390-2815923362-3101623786-2213233685': 'usbcir',
    'S-1-5-80-1032545752-2203350250-1701939687-317337126-3231707909': 'usbehci',
    'S-1-5-80-676136802-2607101929-335774531-4135730467-913299484': 'usbhub',
    'S-1-5-80-3434778094-456680973-2488395463-338906152-1015349184': 'usbohci',
    'S-1-5-80-3620574345-1163766744-4010839292-3531329841-768311061': 'usbprint',
    'S-1-5-80-376233901-499118290-773318279-1925188704-297947815': 'USBSTOR',
    'S-1-5-80-2717376493-4290053016-2054941639-3048903775-1780974753': 'usbuhci',
    'S-1-5-80-2815190569-4075358141-1041947382-2198045348-980246365': 'UxSms',
    'S-1-5-80-2901324718-895851292-2096622302-170690027-1637913602': 'VaultSvc',
    'S-1-5-80-2236596344-777810374-464678914-301799185-133794676': 'vdrvroot',
    'S-1-5-80-2196396108-1448510645-203779624-3888580976-3789157697': 'vds',
    'S-1-5-80-1636345116-1749775499-167646407-1402041886-784684825': 'vga',
    'S-1-5-80-1604054522-1120073184-2766342441-3740248177-2194771659': 'VgaSave',
    'S-1-5-80-2349230263-3936233330-585165183-483748113-2063106807': 'vhdmp',
    'S-1-5-80-269018121-2628019534-3958128902-1689023713-3977233287': 'viaagp',
    'S-1-5-80-702914695-4281403409-954615538-3988029004-192649218': 'ViaC7',
    'S-1-5-80-3488702259-1115883433-1783531185-1350626685-2323838072': 'viaide',
    'S-1-5-80-3414199520-1924951526-579304523-1555932441-262361574': 'vm3dmp',
    'S-1-5-80-3316781363-2712907428-2579548995-1296955556-57435734': 'VMAUDIO',
    'S-1-5-80-394042835-174396444-3357755573-789530950-2357907384': 'vmbus',
    'S-1-5-80-3485585108-3288609388-3381644673-894183282-3425970148': 'VMBusHID',
    'S-1-5-80-2053731399-3564616636-592537298-4187980385-3071434599': 'vmci',
    'S-1-5-80-4081816966-3135276745-2345987325-2511854693-3099376874': 'vmdebug',
    'S-1-5-80-2844247271-1920892496-2185725435-2733799570-1491885128': 'vmhgfs',
    'S-1-5-80-2713566713-2012099321-1704287870-164250842-2950185051': 'VMMEMCTL',
    'S-1-5-80-616456234-2657522756-2692773202-1293725715-2143369223': 'vmmouse',
    'S-1-5-80-470576323-3739623512-411527224-1524486745-930631467': 'vmrawdsk',
    'S-1-5-80-994229404-1081919929-268374983-1858992150-4232923339': 'VMTools',
    'S-1-5-80-3615470141-4057994987-1930054357-1444440834-2714780835': 'VMUpgradeHelper',
    'S-1-5-80-3972256235-858188783-2536722634-3029314587-3393749697': 'vmvss',
    'S-1-5-80-1570634675-3893565091-22195573-2267868061-2898682217': 'volmgr',
    'S-1-5-80-2228288927-839465256-4097931996-4258784654-3424789253': 'volmgrx',
    'S-1-5-80-2161309226-1540144261-2901834345-3792977468-1183436922': 'volsnap',
    'S-1-5-80-1269120828-58111527-683397690-4062780901-3407528550': 'vsmraid',
    'S-1-5-80-3195062495-2862850656-3724129271-1847284719-4038691091': 'VSS',
    'S-1-5-80-4271242282-3170619077-2600330701-1558677754-1139114601': 'vwifibus',
    'S-1-5-80-4267341169-2882910712-659946508-2704364837-2204554466': 'W32Time',
    'S-1-5-80-989796750-4090848350-2040919084-978865222-2182970707': 'W3SVC',
    'S-1-5-80-1272828037-3321607953-1682131387-4084423848-3273467238': 'WacomPen',
    'S-1-5-80-145391760-3682396335-1395736941-2543690743-1822485816': 'WANARP',
    'S-1-5-80-3957613141-1606606214-622769385-3049525404-2510868034': 'Wanarpv6',
    'S-1-5-80-1549550529-11381693-4027442525-4081535042-2424139505': 'wbengine',
    'S-1-5-80-1577343513-2244782562-3500840712-2807016722-4230555396': 'WbioSrvc',
    'S-1-5-80-1555863574-1012459212-3842453055-37978308-1142448422': 'wcncsvc',
    'S-1-5-80-4064017820-1559943312-846267769-2219870576-1957141527': 'WcsPlugInService',
    'S-1-5-80-3405261312-3324525412-773550320-3159108954-1126011555': 'Wd',
    'S-1-5-80-2731089040-2526960094-3333867314-868407530-1311763772': 'Wdf01000',
    'S-1-5-80-3139157870-2983391045-3678747466-658725712-1809340420': 'WdiServiceHost',
    'S-1-5-80-3524758515-3090971750-345616940-2322499744-3530715838': 'WdiSystemHost',
    'S-1-5-80-324959683-3395802011-921526492-919036580-1730255754': 'WebClient',
    'S-1-5-80-4059739203-877974739-1245631912-527174227-2996563517': 'Wecsvc',
    'S-1-5-80-3594706986-2537596223-181334840-1741483385-1351671666': 'wercplsupport',
    'S-1-5-80-3299868208-4286319593-1091140620-3583751967-1732444380': 'WerSvc',
    'S-1-5-80-2019001281-2253379323-945087313-3738653069-3773415333': 'WfpLwf',
    'S-1-5-80-4016954646-3779912912-520790876-2627662839-2216516612': 'WIMMount',
    'S-1-5-80-1367312344-4235937835-3348187091-2947416599-1643272376': 'win32dd',
    'S-1-5-80-1913148863-3492339771-4165695881-2087618961-4109116736': 'WinDefend',
    'S-1-5-80-3760743496-293058752-544796799-945139227-648175845': 'Windows Workflow Foundation 3.0.0.0',
    'S-1-5-80-2455429942-3131183193-3617688776-595395669-3772047725': 'WinHttpAutoProxySvc',
    'S-1-5-80-3750560858-172214265-3889451188-1914796615-4100997547': 'Winmgmt',
    'S-1-5-80-569256582-2953403351-2909559716-1301513147-412116970': 'WinRM',
    'S-1-5-80-3758380775-581010763-2947690711-3499621892-3054972477': 'Winsock',
    'S-1-5-80-197470898-1564017914-2276667423-138762734-2890991316': 'WinSock2',
    'S-1-5-80-1428027539-3309602793-2678353003-1498846795-3763184142': 'Wlansvc',
    'S-1-5-80-404760553-4074834012-3606039051-2170089041-3496108291': 'WmiAcpi',
    'S-1-5-80-1672893355-2301755825-1450106782-2724904875-1401714515': 'WmiApRpl',
    'S-1-5-80-1851371743-411767070-3743290205-1090512353-603110601': 'wmiApSrv',
    'S-1-5-80-2375682873-768044350-3534595160-1005545032-2873800392': 'WMPNetworkSvc',
    'S-1-5-80-2153317275-3787551921-2333987345-3394040919-509713777': 'WPCSvc',
    'S-1-5-80-113310567-2163499630-2787090463-221477905-209227094': 'WPDBusEnum',
    'S-1-5-80-1339864866-2803517768-580965624-1158720225-1206284216': 'ws2ifsl',
    'S-1-5-80-3232712927-1625117661-2590453128-1738570065-3637376297': 'wscsvc',
    'S-1-5-80-117416528-2204451360-1913602512-1355018040-1234992034': 'WSearch',
    'S-1-5-80-1961591210-2878639619-2091680054-2529124376-3572759234': 'WSearchIdxPi',
    'S-1-5-80-1014140700-3308905587-3330345912-272242898-93311788': 'wuauserv',
    'S-1-5-80-69171120-2364612362-2758615892-3595098197-2063739924': 'WudfPf',
    'S-1-5-80-1839061227-813336325-324579571-4216704371-1399658985': 'WUDFRd',
    'S-1-5-80-2652678385-582572993-1835434367-1344795993-749280709': 'wudfsvc',
    'S-1-5-80-3981856537-581775623-1136376035-2066872258-409572886': 'WwanSvc',
    'S-1-5-80-2933569122-2468899862-1495779727-289297006-142656920': 'xmlprov',
}

def createservicesid(svc):
    """ Calculate the Service SID """
    uni = ''.join([c + '\x00' for c in svc])
    sha = hashlib.sha1(uni.upper()).digest() # pylint: disable-msg=E1101
    dec = list()
    for i in range(5):
        ## The use of struct here is OK. It doesn't make much sense
        ## to leverage obj.Object inside this loop. 
        dec.append(struct.unpack('<I', sha[i * 4 : i * 4 + 4])[0])
    return 'S-1-5-80-' + '-'.join([str(n) for n in dec])

class GetServiceSids(common.AbstractWindowsCommand):
    """Get the names of services in the Registry and return Calculated SID"""

    def calculate(self):
        #scan for registries and populate them:
        debug.debug("Scanning for registries....")

        #set our current registry of interest and get its path
        #and get current control set
        debug.debug("Getting Current Control Set....")
        regapi = registryapi.RegistryApi(self._config)
        currentcs = regapi.reg_get_currentcontrolset()
        if currentcs == None:
            currentcs = "ControlSet001"

        #set the services root. 
        regapi.set_current('system')
        debug.debug("Getting Services and calculating SIDs....")
        services = regapi.reg_get_key('system', currentcs + '\\' + 'Services')
        if services:
            for s in rawreg.subkeys(services):
                if s.Name not in servicesids.values():
                    sid = createservicesid(str(s.Name))
                    yield sid, str(s.Name)
        for sid in servicesids:
            yield sid, servicesids[sid]

    def unified_output(self, data):
        return TreeGrid([("SID", str),
                       ("Service", str)],
                        self.generator(data))

    def generator(self, data):
        for sid, service in data:
            if not service:
                continue
            yield (0, [str(sid), str(service)])

    def render_text(self, outfd, data):
        """ output to Service SIDs as a dictionary for future use"""
        outfd.write("servicesids = { \n")
        for sid, service in data:
            if not service:
                continue
            outfd.write("    '" + sid + "': '" + service + "',\n")
        outfd.write("}\n")
