/* See license.txt for terms of usage */

define([
    "firebug/chrome/module",
    "firebug/lib/object",
    "firebug/firebug",
    "firebug/chrome/firefox",
    "firebug/lib/locale",
    "firebug/lib/events",
    "firebug/lib/dom",
    "firebug/lib/options",
    "firebug/firefox/browserOverlayLib",
],
function(Module, Obj, Firebug, Firefox, Locale, Events, Dom, Options, BrowserOverlayLib) {

"use strict";

// ********************************************************************************************* //
// Constants

var Cc = Components.classes;
var Ci = Components.interfaces;

// ********************************************************************************************* //
// Module Implementation

/**
 * @module StartButton module represents the UI entry point to Firebug. This "start button"
 * formerly known as "the status bar icon" is automatically appended into Firefox toolbar
 * (since Firefox 4).
 *
 * Start button is associated with a menu (fbStatusContextMenu) that contains basic actions
 * such as panel activation and also indicates whether Firebug is activated/deactivated for
 * the current page (by changing its color).
 */
Firebug.StartButton = Obj.extend(Module,
/** @lends Firebug.StartButton */
{
    dispatchName: "startButton",

    initializeUI: function()
    {
        Module.initializeUI.apply(this, arguments);

        if (FBTrace.DBG_INITIALIZE)
            FBTrace.sysout("StartButton.initializeUI;");

        // When Firebug is full loaded content of the start button tooltip
        // will be generated by this StartButton object.
        var firebugButton = Firefox.getElementById("firebug-buttonTooltip");
        var listener = Firebug.StartButton.onTooltipShowing.bind(this);
        firebugButton.addEventListener("popupshowing", listener);
    },

    shutdown: function()
    {
    },

    // * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //
    // Tooltip

    onTooltipShowing: function(event)
    {
        var tooltip = event.target;
        var doc = tooltip.ownerDocument;

        Dom.eraseNode(tooltip);

        tooltip.appendChild(BrowserOverlayLib.$label(doc, {
            "class": "version",
            value: Locale.$STR("Firebug") + " " + Firebug.getVersion()
        }));

        var status = BrowserOverlayLib.$el(doc, "hbox");
        tooltip.appendChild(status);

        var suspended = Firebug.getSuspended();
        status.appendChild(BrowserOverlayLib.$label(doc, {
            "class": "status",
            value: suspended ? Locale.$STR("startbutton.tip.deactivated") :
                Locale.$STRP("plural.Total_Firebugs2", [Firebug.TabWatcher.contexts.length])
        }));

        if (suspended)
            return;

        status.appendChild(BrowserOverlayLib.$label(doc, {
            "class": "placement",
            value: "(" + Locale.$STR(Firebug.getPlacement()) + ")"
        }));

        if (Options.get("allPagesActivation") === "on")
        {
            tooltip.appendChild(BrowserOverlayLib.$label(doc, {
                "class": "alwaysOn",
                value: Locale.$STR("enablement.on") + " " +
                    Locale.$STR("enablement.for_all_pages")
            }));
        }

        // Panel enablement status info
        tooltip.appendChild(BrowserOverlayLib.$label(doc, {
            "class": "enablement",
            value: Locale.$STR("enablement.Panel_activation_status")
        }));

        var statuses = this.getEnablementStatus();
        for (var i=0; i<statuses.length; i++)
        {
            var status = statuses[i];
            var parent = BrowserOverlayLib.$el(doc, "hbox");
            tooltip.appendChild(parent);

            parent.appendChild(BrowserOverlayLib.$label(doc, {
                "class": "panelName " + status.status,
                value: status.name + ":"
            }));

            parent.appendChild(BrowserOverlayLib.$label(doc, {
                "class": "panelStatus " + status.status,
                value: status.statusLabel
            }));
        }
    },

    // * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //
    // Error count

    showCount: function(errorCount)
    {
        var errorBadge = Firefox.getElementById("firebug-error-badge");
        if (errorCount && Options.get("showErrorCount"))
        {
            if (errorBadge)
            {
                var errorLabel = Firefox.getElementById("firebug-error-label");
                errorBadge.setAttribute("showErrors", "true");
                errorLabel.setAttribute("value", errorCount);
                errorLabel.setAttribute("tooltiptext",
                    Locale.$STRP("plural.startbutton.tip.errors", [errorCount]));
            }
        }
        else
        {
            if (errorBadge)
            {
                errorBadge.removeAttribute("showErrors");

                // Use '0', so the horizontal space for the number is still allocated.
                // The button will cause re-layout if there are more than 9 errors.
                var errorLabel = Firefox.getElementById("firebug-error-label");
                errorLabel.setAttribute("value", "0");
            }
        }
    },

    // * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //
    // Tooltip

    resetTooltip: function()
    {
        var firebugStatus = Firefox.getElementById("firebugStatus");
        if (!firebugStatus)
            return;

        // The start button is colorful only if there is a context
        var active = Firebug.currentContext ? "true" : "false";
        firebugStatus.setAttribute("firebugActive", active);

        if (FBTrace.DBG_TOOLTIP)
            FBTrace.sysout("StartButton.resetTooltip; called: firebug active: " + active);
    },

    getEnablementStatus: function()
    {
        var firebugStatus = Firefox.getElementById("firebugStatus");
        if (!firebugStatus)
            return;

        var panels = Firebug.getActivablePanelTypes();
        var statuses = [];

        var strOn = Locale.$STR("enablement.on");
        var strOff = Locale.$STR("enablement.off");

        for (var i=0; i<panels.length; ++i)
        {
            var panelName = panels[i].prototype.name;
            var status = firebugStatus.getAttribute(panelName);
            var statusLabel = (status == "on") ? strOn : strOff;

            statuses.push({
                name: Firebug.getPanelTitle(panels[i]),
                status: status,
                statusLabel: statusLabel
            });
        }

        return statuses;
    },

    // * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //
    // Activation

    getSuspended: function()
    {
        var suspendMarker = Firefox.getElementById("firebugStatus");
        if (suspendMarker && suspendMarker.hasAttribute("suspended"))
            return suspendMarker.getAttribute("suspended");

        return null;
    },

    setSuspended: function(value)
    {
        var suspendMarker = Firefox.getElementById("firebugStatus");

        if (FBTrace.DBG_ACTIVATION)
            FBTrace.sysout("StartButton.setSuspended; to " + value + ". Browser: " +
                Firebug.chrome.window.document.title);

        if (value == "suspended")
            suspendMarker.setAttribute("suspended", value);
        else
            suspendMarker.removeAttribute("suspended");

        this.resetTooltip();
    }
});

// ********************************************************************************************* //
// Registration

Firebug.registerModule(Firebug.StartButton);

// ********************************************************************************************* //

return Firebug.StartButton;
});
