/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.rounding;

import java.io.IOException;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.rounding.DateTimeUnit;
import org.elasticsearch.common.rounding.Rounding;
import org.elasticsearch.common.unit.TimeValue;
import org.joda.time.DateTimeField;
import org.joda.time.DateTimeZone;

public abstract class TimeZoneRounding
extends Rounding {
    public static Builder builder(DateTimeUnit unit) {
        return new Builder(unit);
    }

    public static Builder builder(TimeValue interval) {
        return new Builder(interval);
    }

    static class TimeIntervalRounding
    extends TimeZoneRounding {
        static final byte ID = 2;
        private long interval;
        private DateTimeZone timeZone;

        TimeIntervalRounding() {
        }

        TimeIntervalRounding(long interval, DateTimeZone timeZone) {
            if (interval < 1L) {
                throw new IllegalArgumentException("Zero or negative time interval not supported");
            }
            this.interval = interval;
            this.timeZone = timeZone;
        }

        @Override
        public byte id() {
            return 2;
        }

        @Override
        public long roundKey(long utcMillis) {
            long roundedUTC;
            long timeLocal = this.timeZone.convertUTCToLocal(utcMillis);
            long rounded = Rounding.Interval.roundValue(Rounding.Interval.roundKey(timeLocal, this.interval), this.interval);
            if (!this.isInDSTGap(rounded)) {
                roundedUTC = this.timeZone.convertLocalToUTC(rounded, true, utcMillis);
                long transition = this.timeZone.previousTransition(utcMillis);
                if (transition != utcMillis && transition > roundedUTC) {
                    roundedUTC = this.roundKey(transition - 1L);
                }
            } else {
                roundedUTC = this.timeZone.previousTransition(utcMillis) + 1L;
            }
            return roundedUTC;
        }

        private boolean isInDSTGap(long instantLocal) {
            int offset;
            if (this.timeZone.isFixed()) {
                return false;
            }
            int offsetLocal = this.timeZone.getOffset(instantLocal);
            if (offsetLocal != (offset = this.timeZone.getOffset(instantLocal - (long)offsetLocal))) {
                long nextAdjusted;
                long nextLocal = this.timeZone.nextTransition(instantLocal - (long)offsetLocal);
                if (nextLocal == instantLocal - (long)offsetLocal) {
                    nextLocal = Long.MAX_VALUE;
                }
                if ((nextAdjusted = this.timeZone.nextTransition(instantLocal - (long)offset)) == instantLocal - (long)offset) {
                    nextAdjusted = Long.MAX_VALUE;
                }
                if (nextLocal != nextAdjusted) {
                    return true;
                }
            }
            return false;
        }

        @Override
        public long valueForKey(long time) {
            assert (this.roundKey(time) == time);
            return time;
        }

        @Override
        public long nextRoundingValue(long time) {
            long timeLocal = time;
            timeLocal = this.timeZone.convertUTCToLocal(time);
            long next = timeLocal + this.interval;
            return this.timeZone.convertLocalToUTC(next, false);
        }

        @Override
        public void readFrom(StreamInput in) throws IOException {
            this.interval = in.readVLong();
            this.timeZone = DateTimeZone.forID((String)in.readString());
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            out.writeVLong(this.interval);
            out.writeString(this.timeZone.getID());
        }
    }

    static class TimeUnitRounding
    extends TimeZoneRounding {
        static final byte ID = 1;
        private DateTimeUnit unit;
        private DateTimeField field;
        private DateTimeZone timeZone;

        TimeUnitRounding() {
        }

        TimeUnitRounding(DateTimeUnit unit, DateTimeZone timeZone) {
            this.unit = unit;
            this.field = unit.field(timeZone);
            this.timeZone = timeZone;
        }

        @Override
        public byte id() {
            return 1;
        }

        @Override
        public long roundKey(long utcMillis) {
            long rounded = this.field.roundFloor(utcMillis);
            if (!this.timeZone.isFixed() && this.timeZone.getOffset(utcMillis) != this.timeZone.getOffset(rounded)) {
                rounded = this.field.roundFloor(rounded);
            }
            assert (rounded == this.field.roundFloor(rounded));
            return rounded;
        }

        @Override
        public long valueForKey(long time) {
            assert (this.roundKey(time) == time);
            return time;
        }

        @Override
        public long nextRoundingValue(long utcMillis) {
            long floor = this.roundKey(utcMillis);
            long next = this.roundKey(this.field.add(floor, 1));
            if (next == floor) {
                next = this.roundKey(this.field.add(floor, 2));
            }
            return next;
        }

        @Override
        public void readFrom(StreamInput in) throws IOException {
            this.unit = DateTimeUnit.resolve(in.readByte());
            this.timeZone = DateTimeZone.forID((String)in.readString());
            this.field = this.unit.field(this.timeZone);
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            out.writeByte(this.unit.id());
            out.writeString(this.timeZone.getID());
        }

        public String toString() {
            return "[" + this.timeZone + "][" + (Object)((Object)this.unit) + "]";
        }
    }

    public static class Builder {
        private final DateTimeUnit unit;
        private final long interval;
        private DateTimeZone timeZone = DateTimeZone.UTC;
        private float factor = 1.0f;
        private long offset;

        public Builder(DateTimeUnit unit) {
            this.unit = unit;
            this.interval = -1L;
        }

        public Builder(TimeValue interval) {
            this.unit = null;
            if (interval.millis() < 1L) {
                throw new IllegalArgumentException("Zero or negative time interval not supported");
            }
            this.interval = interval.millis();
        }

        public Builder timeZone(DateTimeZone timeZone) {
            if (timeZone == null) {
                throw new IllegalArgumentException("Setting null as timezone is not supported");
            }
            this.timeZone = timeZone;
            return this;
        }

        public Builder offset(long offset) {
            this.offset = offset;
            return this;
        }

        public Builder factor(float factor) {
            this.factor = factor;
            return this;
        }

        public Rounding build() {
            Rounding timeZoneRounding = this.unit != null ? new TimeUnitRounding(this.unit, this.timeZone) : new TimeIntervalRounding(this.interval, this.timeZone);
            if (this.offset != 0L) {
                timeZoneRounding = new Rounding.OffsetRounding(timeZoneRounding, this.offset);
            }
            if (this.factor != 1.0f) {
                timeZoneRounding = new Rounding.FactorRounding(timeZoneRounding, this.factor);
            }
            return timeZoneRounding;
        }
    }
}

