package vtk;

import java.io.File;

/**
 * Enum used to load native library more easily. If you don't want to set the
 * specific environment variable you can provide the path of the directory that
 * contains the VTK library through a Java Property. Like in the following
 * command line:
 *
 * > java -cp vtk.jar -Dvtk.lib.dir=.../vtk-libs vtk.sample.SimpleVTK
 *
 * The directory .../vtk-libs must contain the so/dll/dylib + the jnilib files
 *
 * @author sebastien jourdain - sebastien.jourdain@kitware.com
 */
public enum vtkNativeLibrary {

    VTKDOMAINSCHEMISTRYOPENGL2("vtkDomainsChemistryOpenGL2Java", false),
    VTKDOMAINSCHEMISTRY("vtkDomainsChemistryJava", true),
    VTKIOCORE("vtkIOCoreJava", true),
    VTKIOIMAGE("vtkIOImageJava", true),
    VTKIOMPIPARALLEL("vtkIOMPIParallelJava", true),
    VTKIOVIDEO("vtkIOVideoJava", true),
    VTKIOXMLPARSER("vtkIOXMLParserJava", true),
    VTKIOXDMF3("vtkIOXdmf3Java", true),
    VTKIOEXODUS("vtkIOExodusJava", true),
    VTKIOPARALLELLSDYNA("vtkIOParallelLSDynaJava", true),
    VTKIOINFOVIS("vtkIOInfovisJava", true),
    VTKIOEXPORT("vtkIOExportJava", true),
    VTKIOVPIC("vtkIOVPICJava", true),
    VTKIOXDMF2("vtkIOXdmf2Java", true),
    VTKIOADIOS("vtkIOADIOSJava", false),
    VTKIOPARALLELEXODUS("vtkIOParallelExodusJava", true),
    VTKIOIMPORT("vtkIOImportJava", true),
    VTKIOMOVIE("vtkIOMovieJava", true),
    VTKIOMINC("vtkIOMINCJava", true),
    VTKIOGEOMETRY("vtkIOGeometryJava", true),
    VTKIOODBC("vtkIOODBCJava", true),
    VTKIOFFMPEG("vtkIOFFMPEGJava", true),
    VTKIOAMR("vtkIOAMRJava", true),
    VTKIOXML("vtkIOXMLJava", true),
    VTKIOPARALLEL("vtkIOParallelJava", true),
    VTKIOPARALLELXML("vtkIOParallelXMLJava", true),
    VTKIOGDAL("vtkIOGDALJava", true),
    VTKIOMYSQL("vtkIOMySQLJava", true),
    VTKIONETCDF("vtkIONetCDFJava", true),
    VTKIOENSIGHT("vtkIOEnSightJava", true),
    VTKIOLEGACY("vtkIOLegacyJava", true),
    VTKIOGEOJSON("vtkIOGeoJSONJava", true),
    VTKIOPLY("vtkIOPLYJava", true),
    VTKIOPARALLELNETCDF("vtkIOParallelNetCDFJava", true),
    VTKIOMPIIMAGE("vtkIOMPIImageJava", true),
    VTKIOSQL("vtkIOSQLJava", true),
    VTKIOLSDYNA("vtkIOLSDynaJava", true),
    VTKIOPOSTGRESQL("vtkIOPostgreSQLJava", true),
    VTKCHARTSCORE("vtkChartsCoreJava", true),
    VTKINTERACTIONIMAGE("vtkInteractionImageJava", true),
    VTKINTERACTIONWIDGETS("vtkInteractionWidgetsJava", true),
    VTKINTERACTIONSTYLE("vtkInteractionStyleJava", true),
    VTKINFOVISCORE("vtkInfovisCoreJava", true),
    VTKINFOVISLAYOUT("vtkInfovisLayoutJava", true),
    VTKINFOVISBOOSTGRAPHALGORITHMS("vtkInfovisBoostGraphAlgorithmsJava", true),
    VTKINFOVISPARALLEL("vtkInfovisParallelJava", false),
    VTKVIEWSCORE("vtkViewsCoreJava", true),
    VTKVIEWSINFOVIS("vtkViewsInfovisJava", true),
    VTKVIEWSCONTEXT2D("vtkViewsContext2DJava", true),
    VTKVIEWSGEOVIS("vtkViewsGeovisJava", true),
    VTKTESTINGRENDERING("vtkTestingRenderingJava", true),
    VTKPARALLELCORE("vtkParallelCoreJava", true),
    VTKPARALLELMPI("vtkParallelMPIJava", true),
    VTKPARALLELMPI4PY("vtkParallelMPI4PyJava", true),
    VTKWEBCORE("vtkWebCoreJava", false),
    VTKWEBGLEXPORTER("vtkWebGLExporterJava", false),
    VTKFILTERSPROGRAMMABLE("vtkFiltersProgrammableJava", true),
    VTKFILTERSPARALLELSTATISTICS("vtkFiltersParallelStatisticsJava", true),
    VTKFILTERSCORE("vtkFiltersCoreJava", true),
    VTKFILTERSHYPERTREE("vtkFiltersHyperTreeJava", true),
    VTKFILTERSHYBRID("vtkFiltersHybridJava", true),
    VTKFILTERSPARALLELGEOMETRY("vtkFiltersParallelGeometryJava", true),
    VTKFILTERSPARALLELIMAGING("vtkFiltersParallelImagingJava", true),
    VTKFILTERSPARALLELMPI("vtkFiltersParallelMPIJava", true),
    VTKFILTERSFLOWPATHS("vtkFiltersFlowPathsJava", true),
    VTKFILTERSREEBGRAPH("vtkFiltersReebGraphJava", true),
    VTKFILTERSPARALLELFLOWPATHS("vtkFiltersParallelFlowPathsJava", true),
    VTKFILTERSSOURCES("vtkFiltersSourcesJava", true),
    VTKFILTERSGENERIC("vtkFiltersGenericJava", true),
    VTKFILTERSSMP("vtkFiltersSMPJava", true),
    VTKFILTERSGEOMETRY("vtkFiltersGeometryJava", true),
    VTKFILTERSAMR("vtkFiltersAMRJava", true),
    VTKFILTERSPYTHON("vtkFiltersPythonJava", true),
    VTKFILTERSMODELING("vtkFiltersModelingJava", true),
    VTKFILTERSSELECTION("vtkFiltersSelectionJava", true),
    VTKFILTERSPARALLEL("vtkFiltersParallelJava", true),
    VTKFILTERSVERDICT("vtkFiltersVerdictJava", true),
    VTKFILTERSSTATISTICSGNUR("vtkFiltersStatisticsGnuRJava", false),
    VTKFILTERSGENERAL("vtkFiltersGeneralJava", true),
    VTKFILTERSEXTRACTION("vtkFiltersExtractionJava", true),
    VTKFILTERSSTATISTICS("vtkFiltersStatisticsJava", true),
    VTKFILTERSIMAGING("vtkFiltersImagingJava", true),
    VTKFILTERSTEXTURE("vtkFiltersTextureJava", true),
    VTKACCELERATORSDAX("vtkAcceleratorsDaxJava", false),
    VTKACCELERATORSPISTON("vtkAcceleratorsPistonJava", false),
    VTKPYTHONINTERPRETER("vtkPythonInterpreterJava", true),
    VTKRENDERINGEXTERNAL("vtkRenderingExternalJava", false),
    VTKRENDERINGOPENGL2("vtkRenderingOpenGL2Java", false),
    VTKRENDERINGGL2PS("vtkRenderingGL2PSJava", true),
    VTKRENDERINGCORE("vtkRenderingCoreJava", true),
    VTKRENDERINGIMAGE("vtkRenderingImageJava", true),
    VTKRENDERINGVOLUMEOPENGL2("vtkRenderingVolumeOpenGL2Java", false),
    VTKRENDERINGCONTEXT2D("vtkRenderingContext2DJava", true),
    VTKRENDERINGLOD("vtkRenderingLODJava", true),
    VTKRENDERINGPARALLELLIC("vtkRenderingParallelLICJava", true),
    VTKRENDERINGLIC("vtkRenderingLICJava", true),
    VTKRENDERINGVOLUMEAMR("vtkRenderingVolumeAMRJava", true),
    VTKRENDERINGVOLUME("vtkRenderingVolumeJava", true),
    VTKRENDERINGMATPLOTLIB("vtkRenderingMatplotlibJava", true),
    VTKRENDERINGOPENGL("vtkRenderingOpenGLJava", true),
    VTKRENDERINGPARALLEL("vtkRenderingParallelJava", true),
    VTKRENDERINGANNOTATION("vtkRenderingAnnotationJava", true),
    VTKRENDERINGCONTEXTOPENGL("vtkRenderingContextOpenGLJava", true),
    VTKRENDERINGFREETYPE("vtkRenderingFreeTypeJava", true),
    VTKRENDERINGCONTEXTOPENGL2("vtkRenderingContextOpenGL2Java", false),
    VTKRENDERINGLABEL("vtkRenderingLabelJava", true),
    VTKRENDERINGLICOPENGL2("vtkRenderingLICOpenGL2Java", false),
    VTKRENDERINGQT("vtkRenderingQtJava", true),
    VTKRENDERINGVOLUMEOPENGL("vtkRenderingVolumeOpenGLJava", true),
    VTKIMAGINGMATH("vtkImagingMathJava", true),
    VTKIMAGINGCORE("vtkImagingCoreJava", true),
    VTKIMAGINGHYBRID("vtkImagingHybridJava", true),
    VTKIMAGINGSTENCIL("vtkImagingStencilJava", true),
    VTKIMAGINGSOURCES("vtkImagingSourcesJava", true),
    VTKIMAGINGFOURIER("vtkImagingFourierJava", true),
    VTKIMAGINGCOLOR("vtkImagingColorJava", true),
    VTKIMAGINGGENERAL("vtkImagingGeneralJava", true),
    VTKIMAGINGSTATISTICS("vtkImagingStatisticsJava", true),
    VTKIMAGINGMORPHOLOGICAL("vtkImagingMorphologicalJava", true),
    VTKCOMMONMATH("vtkCommonMathJava", true),
    VTKCOMMONCORE("vtkCommonCoreJava", true),
    VTKCOMMONEXECUTIONMODEL("vtkCommonExecutionModelJava", true),
    VTKCOMMONCOMPUTATIONALGEOMETRY("vtkCommonComputationalGeometryJava", true),
    VTKCOMMONCOLOR("vtkCommonColorJava", true),
    VTKCOMMONSYSTEM("vtkCommonSystemJava", true),
    VTKCOMMONDATAMODEL("vtkCommonDataModelJava", true),
    VTKCOMMONMISC("vtkCommonMiscJava", true),
    VTKCOMMONTRANSFORMS("vtkCommonTransformsJava", true),
    VTKGEOVISCORE("vtkGeovisCoreJava", true);


  /**
   * Try to load all library
   *
   * @return true if all library have been successfully loaded
   */
  public static boolean LoadAllNativeLibraries() {
    boolean isEveryThingLoaded = true;
    for (vtkNativeLibrary lib : values()) {
      try {
        if(lib.IsBuilt()) {
          lib.LoadLibrary();
        }
      } catch (UnsatisfiedLinkError e) {
        isEveryThingLoaded = false;
        e.printStackTrace();
      }
    }

    return isEveryThingLoaded;
  }

  /**
   * Load the set of given library and trows runtime exception if any given
   * library failed in the loading process
   *
   * @param nativeLibraries
   */
  public static void LoadNativeLibraries(vtkNativeLibrary... nativeLibraries) {
    for (vtkNativeLibrary lib : nativeLibraries) {
      lib.LoadLibrary();
    }
  }

  /**
   * Disable the pop-in vtkErrorWindow by writing the error to a log file.
   * If the provided logFile is null the default "vtkError.txt" file will be
   * used.
   *
   * @param logFile
   */
  public static void DisableOutputWindow(File logFile) {
    if(logFile == null) {
      logFile = new File("vtkError.txt");
    }
    vtkFileOutputWindow outputError = new vtkFileOutputWindow();
    outputError.SetFileName(logFile.getAbsolutePath());
    outputError.SetInstance(outputError);
  }

  private vtkNativeLibrary(String nativeLibraryName, boolean built) {
    this.nativeLibraryName = nativeLibraryName;
    this.loaded = false;
    this.built = built;
  }

  /**
   * Load the library and throws runtime exception if the library failed in
   * the loading process
   */
  public void LoadLibrary() throws UnsatisfiedLinkError {
    if (!loaded) {
      if (System.getProperty("vtk.lib.dir") != null) {
        File dir = new File(System.getProperty("vtk.lib.dir"));
        patchJavaLibraryPath(dir.getAbsolutePath());
        File libPath = new File(dir, System.mapLibraryName(nativeLibraryName));
        if (libPath.exists()) {
          try {
            Runtime.getRuntime().load(libPath.getAbsolutePath());
            loaded = true;
            return;
          } catch (UnsatisfiedLinkError e) {
            e.printStackTrace();
          }
        }
      }
      System.loadLibrary(nativeLibraryName);
    }
    loaded = true;
  }

  /**
   * @return true if the library has already been successfully loaded
   */
  public boolean IsLoaded() {
    return loaded;
  }

  /**
   * @return true if the module was enabled and therefore build
   */
  public boolean IsBuilt() {
    return built;
  }

  /**
   * @return the library name
   */
  public String GetLibraryName() {
    return nativeLibraryName;
  }

  private static void patchJavaLibraryPath(String vtkLibDir) {
    if (vtkLibDir != null) {
      String path_separator = System.getProperty("path.separator");
      String s = System.getProperty("java.library.path");
      if (!s.contains(vtkLibDir)) {
        s = s + path_separator + vtkLibDir;
        System.setProperty("java.library.path", s);
      }
    }
  }

  private String nativeLibraryName;
  private boolean loaded;
  private boolean built;
}
