#ifndef _ECORE_AUDIO_EO_H_
#define _ECORE_AUDIO_EO_H_

#ifndef _ECORE_AUDIO_EO_CLASS_TYPE
#define _ECORE_AUDIO_EO_CLASS_TYPE

typedef Eo Ecore_Audio;

#endif

#ifndef _ECORE_AUDIO_EO_TYPES
#define _ECORE_AUDIO_EO_TYPES

typedef enum
{
  ECORE_AUDIO_FORMAT_AUTO = 0, /** Automatically detect the format (for inputs)
                                */
  ECORE_AUDIO_FORMAT_RAW, /** RAW samples (float) */
  ECORE_AUDIO_FORMAT_WAV, /** WAV format */
  ECORE_AUDIO_FORMAT_OGG, /** OGG */
  ECORE_AUDIO_FORMAT_FLAC, /** FLAC, the Free Lossless Audio Codec */
  ECORE_AUDIO_FORMAT_MP3, /** MP3 (not supported) */
  ECORE_AUDIO_FORMAT_LAST /** Sentinel value, do not use */
} Ecore_Audio_Format;


#endif
/** Convenience audio class.
 *
 * @ingroup Ecore_Audio
 */
#define ECORE_AUDIO_CLASS ecore_audio_class_get()

EWAPI const Eo_Class *ecore_audio_class_get(void);

/**
 * @brief Name of the object
 *
 * @param[in] name Name
 *
 * @since 1.8
 *
 * @ingroup Ecore_Audio
 */
EOAPI void ecore_audio_obj_name_set(Eo *obj, const char *name);

/**
 * @brief Name of the object
 *
 * @return Name
 *
 * @since 1.8
 *
 * @ingroup Ecore_Audio
 */
EOAPI const char *ecore_audio_obj_name_get(const Eo *obj);

/**
 * @brief Pause state of the object
 *
 * @param[in] paused true if object is paused, false if not
 *
 * @since 1.8
 *
 * @ingroup Ecore_Audio
 */
EOAPI void ecore_audio_obj_paused_set(Eo *obj, Eina_Bool paused);

/**
 * @brief Pause state of the object
 *
 * @return true if object is paused, false if not
 *
 * @since 1.8
 *
 * @ingroup Ecore_Audio
 */
EOAPI Eina_Bool ecore_audio_obj_paused_get(const Eo *obj);

/**
 * @brief Volume of the object
 *
 * @param[in] volume The volume
 *
 * @since 1.8
 *
 * @ingroup Ecore_Audio
 */
EOAPI void ecore_audio_obj_volume_set(Eo *obj, double volume);

/**
 * @brief Volume of the object
 *
 * @return The volume
 *
 * @since 1.8
 *
 * @ingroup Ecore_Audio
 */
EOAPI double ecore_audio_obj_volume_get(const Eo *obj);

/**
 * @brief Source of the object
 *
 * What sources are supported depends on the actual object. For example, the
 * libsndfile class accepts WAV, OGG, FLAC files as source.
 *
 * @return the source to set to (i.e. file, URL, device)
 *
 * @since 1.8
 *
 * @ingroup Ecore_Audio
 */
EOAPI const char *ecore_audio_obj_source_get(const Eo *obj);

/**
 * @brief Source of the object
 *
 * What sources are supported depends on the actual object. For example, the
 * libsndfile class accepts WAV, OGG, FLAC files as source.
 *
 * @param[in] source the source to set to (i.e. file, URL, device)
 *
 * @return true if the source was set correctly (i.e. the file was opened),
 * EINA_FALSE otherwise
 *
 * @since 1.8
 *
 * @ingroup Ecore_Audio
 */
EOAPI Eina_Bool ecore_audio_obj_source_set(Eo *obj, const char *source);

/**
 * @brief Format of the object.
 *
 * Get the format of the object
 *
 * After setting the source if the format was ECORE_AUDIO_FORMAT_AUTO this
 * function will now return the actual format.
 *
 * @return the format of the object
 *
 * @since 1.8
 *
 * @ingroup Ecore_Audio
 */
EOAPI Ecore_Audio_Format ecore_audio_obj_format_get(const Eo *obj);

/**
 * @brief Format of the object.
 *
 * Set the format of the object
 *
 * What formats are supported depends on the actual object. Default is
 * ECORE_AUDIO_FORMAT_AUTO
 *
 * @param[in] format the format of the object
 *
 * @return true if the format was supported, false otherwise
 *
 * @since 1.8
 *
 * @ingroup Ecore_Audio
 */
EOAPI Eina_Bool ecore_audio_obj_format_set(Eo *obj, Ecore_Audio_Format format);

/**
 * @brief Set the virtual IO functions
 *
 * @param[in] vio the @ref Ecore_Audio_Vio struct with the function callbacks
 * @param[in] data user data to pass to the VIO functions
 * @param[in] free_func this function takes care to clean up @c data when he
 * VIO is destroyed. NULL means do nothing.
 *
 * @since 1.8
 *
 * @ingroup Ecore_Audio
 */
EOAPI void ecore_audio_obj_vio_set(Eo *obj, Ecore_Audio_Vio *vio, void *data, eo_key_data_free_func free_func);


#endif
