require 'coveralls'
Coveralls.wear!
require 'test/unit'
require 'openssl/ccm'

# Testclass with Test Vectors from
# http://tools.ietf.org/html/rfc3610#section-8
class CCMTest < Test::Unit::TestCase
  KEY = %W(C0C1C2C3C4C5C6C7C8C9CACBCCCDCECF) * 12 \
      + %W(D7828D13B2B0BDC325A76236DF93CC6B) * 12
  NONCE = %W(
    00000003020100A0A1A2A3A4A5
    00000004030201A0A1A2A3A4A5
    00000005040302A0A1A2A3A4A5
    00000006050403A0A1A2A3A4A5
    00000007060504A0A1A2A3A4A5
    00000008070605A0A1A2A3A4A5
    00000009080706A0A1A2A3A4A5
    0000000A090807A0A1A2A3A4A5
    0000000B0A0908A0A1A2A3A4A5
    0000000C0B0A09A0A1A2A3A4A5
    0000000D0C0B0AA0A1A2A3A4A5
    0000000E0D0C0BA0A1A2A3A4A5
    00412B4EA9CDBE3C9696766CFA
    0033568EF7B2633C9696766CFA
    00103FE41336713C9696766CFA
    00764C63B8058E3C9696766CFA
    00F8B678094E3B3C9696766CFA
    00D560912D3F703C9696766CFA
    0042FFF8F1951C3C9696766CFA
    00920F40E56CDC3C9696766CFA
    0027CA0C7120BC3C9696766CFA
    005B8CCBCD9AF83C9696766CFA
    003EBE94044B9A3C9696766CFA
    008D493B30AE8B3C9696766CFA
  )
  DATA = %W(
    08090A0B0C0D0E0F101112131415161718191A1B1C1D1E
    08090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F
    08090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F20
    0C0D0E0F101112131415161718191A1B1C1D1E
    0C0D0E0F101112131415161718191A1B1C1D1E1F
    0C0D0E0F101112131415161718191A1B1C1D1E1F20
    08090A0B0C0D0E0F101112131415161718191A1B1C1D1E
    08090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F
    08090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F20
    0C0D0E0F101112131415161718191A1B1C1D1E
    0C0D0E0F101112131415161718191A1B1C1D1E1F
    0C0D0E0F101112131415161718191A1B1C1D1E1F20
    08E8CF97D820EA258460E96AD9CF5289054D895CEAC47C
    9020EA6F91BDD85AFA0039BA4BAFF9BFB79C7028949CD0EC
    B916E0EACC1C00D7DCEC68EC0B3BBB1A02DE8A2D1AA346132E
    12DAAC5630EFA5396F770CE1A66B21F7B2101C
    E88B6A46C78D63E52EB8C546EFB5DE6F75E9CC0D
    6435ACBAFB11A82E2F071D7CA4A5EBD93A803BA87F
    8A19B950BCF71A018E5E6701C91787659809D67DBEDD18
    1761433C37C5A35FC1F39F406302EB907C6163BE38C98437
    A434A8E58500C6E41530538862D686EA9E81301B5AE4226BFA
    B96B49E21D621741632875DB7F6C9243D2D7C2
    E2FCFBB880442C731BF95167C8FFD7895E337076
    ABF21C0B02FEB88F856DF4A37381BCE3CC128517D4
  )
  ADD_DATA = %W(0001020304050607) * 3 \
           + %W(000102030405060708090A0B) * 3 \
           + %W(0001020304050607) * 3 \
           + %W(000102030405060708090A0B) * 3 \
           + %W(
             0BE1A88BACE018B1
             63018F76DC8A1BCB
             AA6CFA36CAE86B40
             D0D0735C531E1BECF049C244
             77B60F011C03E1525899BCAE
             CD9044D2B71FDB8120EA60C0
             D85BC7E69F944FB8
             74A0EBC9069F5B37
             44A3AA3AAE6475CA
             EC46BB63B02520C33C49FD70
             47A65AC78B3D594227E85E71
             6E37A6EF546D955D34AB6059
             )
  CTR0001 = %W(
    50859D916DCB6DDDE077C2D1D4EC9F97
    7AC0103DED38F6C0390DBA871C4991F4
    59B8EFFF46147312B47A1D9D393D3CFF
    AE81666A838B886AEEBF4A5B3284508A
    D0FCF5744D8F31E8895B05054B7C90C3
    63CCBE1EE01744984564B23A8D245C80
    093CDBB9C5524FDAC1C5ECD291C470AF
    737C3391CC8E13DDE0AAC54B6DB7EB98
    8A5A106BC0299A555B936B0B0EA0DE5A
    0B392B9B056697063F12568F2B13A10F
    6B66BC0C90A1F112FCBE6F4E122077BC
    CCF2AED9E04AC974E65855B32B9430BF
    4451B0117A8482BF0319AEC1595EBDDA
    DCEBF413383C66A05A7255EF98D7FFAD
    08C4DAC8ECC1C07B4CE1F24C375A47EE
    0608FF95A694D559F40BB79DEFFA41DF
    BDCE955CCFD3810A91EA77A6A45BC04C
    64A2C55650CEE04C7A93D8EEF543E88E
    363834FA28833DB755660D98650D6846
    4F71A5C11242E37D29F0FEE41BE1025F
    568A459E40094867EB85E09E6A2E6476
    88BC194280C1FA3EBEFCEFFB4DC62D54
    0A7E0A6353C8CF9EBC3B6E63159AD097
    58DB19B3889AA38B3CA40B16FF422C73
  )
  CTR0002 = %W(
    7546717AC6DE9AFF640C9C06DE6D0D8F
    D40CDE22D5F92424F7BE9A569DA79F51
    69F122A078C79B8977894C99975C2378
    D1B19206AC939E2FB6DDCE10A774FD8D
    72A0D4219F0DE1D40483BC2D3D0CFC2A
    396DBAA2A7D2CBD4B5E17C107945BBC0
    11578386E2C472B48ECC8AADAB776FCB
    74B77177C5AAC53B04A4F8708E92EB2B
    EA05FDE2AB225CFEB77312CB88D9A54A
    078965252340943B9E69B256CC5EF731
    979E572BBE658AE5CC2011832A9A9B5B
    A2CAAC1163F407E5E5F6E3B3790F79F8
    83EB76E13A44847F9220090776B825C5
    2F542CBA15D66CDFE1EC468F0E68A124
    A7872E6C6DC44E842602504C3FA573C5
    80553A75783804A9648B68DD7FDCDD7A
    432EF232AE36D89222BF6337E6B26CE8
    18E765ACB7B0E9AF092BD0206CA1C83C
    35E96354871672563F0C08AF784431A9
    342BD3F17CB77BC1790B05056159272C
    A600AA929203549AAEEF2CCC59137A57
    3E597DA5AE21CCA4009E4C0C91F62249
    EA2032DA27826E139E1E725C5B0D3EBF
    C32F243D65DC7E9F4B0216AB7FB96B4D
  )
  CIPHER = %W(
    588C979A61C663D2F066D0C2C0F989806D5F6B61DAC384
    72C91A36E135F8CF291CA894085C87E3CC15C439C9E43A3B
    51B1E5F44A197D1DA46B0F8E2D282AE871E838BB64DA859657
    A28C6865939A9A79FAAA5C4C2A9D4A91CDAC8C
    DCF1FB7B5D9E23FB9D4E131253658AD86EBDCA3E
    6FC1B011F006568B5171A42D953D469B2570A4BD87
    0135D1B2C95F41D5D1D4FEC185D166B8094E999DFED96C
    7B75399AC0831DD2F0BBD75879A2FD8F6CAE6B6CD9B7DB24
    82531A60CC24945A4B8279181AB5C84DF21CE7F9B73F42E197
    07342594157785152B074098330ABB141B947B
    676BB20380B0E301E8AB79590A396DA78B834934
    C0FFA0D6F05BDB67F24D43A4338D2AA4BED7B20E43
    4CB97F86A2A4689A877947AB8091EF5386A6FFBDD080F8
    4CCB1E7CA981BEFAA0726C55D378061298C85C92814ABC33
    B1D23A2220DDC0AC900D9AA03C61FCF4A559A4417767089708
    14D253C3967B70609B7CBB7C49916028324526
    5545FF1A085EE2EFBF52B2E04BEE1E2336C73E3F
    009769ECABDF48625594C59251E6035722675E04C8
    BC218DAA947427B6DB386A99AC1AEF23ADE0B52939CB6A
    5810E6FD25874022E80361A478E3E9CF484AB04F447EFFF6
    F2BEED7BC5098E83FEB5B31608F8E29C38819A89C8E776F154
    31D750A09DA3ED7FDDD49A2032AABF17EC8EBF
    E882F1DBD38CE3EDA7C23F04DD65071EB41342AC
    F32905B88A641B04B9C9FFB58CC390900F3DA12AB1
  )
  MAC = %W(
    17E8D12CFDF926E0
    A091D56E10400916
    4ADAA76FBD9FB0C5
    96C861B9C9E61EF1
    51E83F077D9C2D93
    405A0443AC91CB94
    048C56602C97ACBB7490
    C17B4433F434963F34B4
    EA9C07E56B5EB17E5F4E
    566AA9406B4D999988DD
    F53AA2E9107A8B6C022C
    CD1AA31662E7AD65D6DB
    E78CF7CB0CDDD7B3
    C52EE81D7D77C08A
    A776796EDB723506
    9A6F49975BCADEAF
    762C0C7744FE7E3C
    47099E5AE0704551
    637CF9BEC2408897C6BA
    F0A477CC2FC9BF548944
    4D4151A4ED3A8B87B9CE
    7D22C8088C666BE5C197
    DF7E00DCCEC7AE52987D
    6DCE9E82EFA16DA62059
  )

  def test_aes_init
    assert_raise(OpenSSL::CCMError) { OpenSSL::CCM.new('', 'A' * 16, 8) }
    assert_raise(OpenSSL::CCMError) { OpenSSL::CCM.new('AE', 'A' * 16, 8) }

    assert_raise(OpenSSL::CCMError) { OpenSSL::CCM.new('AES', '', 8) }
    assert_raise(OpenSSL::CCMError) { OpenSSL::CCM.new('AES', 'A', 8) }
    assert_raise(OpenSSL::CCMError) { OpenSSL::CCM.new('AES', 'A' * 15, 8) }

    3.step(17, 2) do |l|
      assert_raise(OpenSSL::CCMError) { OpenSSL::CCM.new('AES', 'A' * 16, l) }
    end
  end

  def test_aes_padding
    ccm = OpenSSL::CCM.new('AES', 'A' * 16, 8)

    assert_equal(''.b           , ccm.send(:padding, '').b)
    assert_equal("\x00".b       , ccm.send(:padding, 'A' * 15).b)
    assert_equal(''.b           , ccm.send(:padding, 'A' * 16).b)
    assert_equal(("\x00" * 15).b, ccm.send(:padding, 'A').b)
  end

  def test_aes_valid
    ccm = OpenSSL::CCM.new('AES', 'A' * 16, 8)

    assert(ccm.send(:valid?, '', 'A' * 13, ''))
    assert(ccm.send(:valid?, 'A' * (256**2 - 1), 'A' * 13, ''))
    assert_raise(OpenSSL::CCMError) do
      ccm.send(:valid?, 'A' * 256**2, 'A' * 13, '')
    end

    assert_raise OpenSSL::CCMError do
      ccm.send(:valid?, 'Hello!', 'A' *  6, '')
    end
    assert(ccm.send(:valid?, '', 'A' * 7, ''))
    assert(ccm.send(:valid?, '', 'A' * 13, ''))
    assert_raise OpenSSL::CCMError do
      ccm.send(:valid?, 'Hello!', 'A' * 14, '')
    end
  end

  def test_aes_vectors
    assert(OpenSSL::CCM.ciphers.include?('AES'), 'Missing AES-Cipher')
    KEY.length.times do |i|
      ccm = OpenSSL::CCM.new('AES', [KEY[i]].pack('H*'),
                             [MAC[i]].pack('H*').b.length)

      c = ccm.send(:get_counter, [NONCE[i]].pack('H*'), 1)
      assert_equal(CTR0001[i], c.unpack('H*')[0].upcase,
                   "Wrong CTR0001 in Vector #{i + 1}")

      c = ccm.send(:get_counter, [NONCE[i]].pack('H*'), 2)
      assert_equal(CTR0002[i], c.unpack('H*')[0].upcase,
                   "Wrong CTR0002 in Vector #{i + 1}")

      c = ccm.send(:crypt, [DATA[i]].pack('H*'), [NONCE[i]].pack('H*'))
      assert_equal(CIPHER[i], c.unpack('H*')[0].upcase,
                   "Wrong CIPHER in Vector #{i + 1}")

      c = ccm.send(:mac,
                   [DATA[i]].pack('H*'),
                   [NONCE[i]].pack('H*'),
                   [ADD_DATA[i]].pack('H*'))
      assert_equal(MAC[i], c.unpack('H*')[0].upcase,
                   "Wrong MAC in Vector #{i + 1}")

      c = ccm.send(:encrypt,
                   [DATA[i]].pack('H*'),
                   [NONCE[i]].pack('H*'),
                   [ADD_DATA[i]].pack('H*'))
      assert_equal((CIPHER[i] + MAC[i]), c.unpack('H*')[0].upcase,
                   "Wrong ENCRYPT in Vector #{i + 1}")

      c = ccm.send(:decrypt,
                   [CIPHER[i] + MAC[i]].pack('H*'),
                   [NONCE[i]].pack('H*'),
                   [ADD_DATA[i]].pack('H*'))
      assert_equal(DATA[i], c.unpack('H*')[0].upcase,
                   "Wrong ENCRYPT in Vector #{i + 1}")
    end
  end

  def test_aes_data
    key = %W(
      00000000000000000000000000000000
      001234567890ABCDEFDCAFFEED3921EE
      001234567890ABCDEFDCAFFEED3921EE
      11223344AABB00000000000000000000
    )
    nonce = %W(
      00000000000000000000000000
      00112233445566778899
      001122334455667788990000
      00112233445566778899
    )
    mac_len = [16, 8, 14, 8]

    assert(OpenSSL::CCM.ciphers.include?('AES'), 'Missing AES-Cipher')
    1.upto(3) do |i|
      open("test/data_#{i}", mode = 'r') do |i_file|
        input = i_file.read
        key.length.times do |j|
          open("test/data_#{i}-#{j + 1}_e", mode = 'r') do |o_file|
            output = o_file.read
            ccm = OpenSSL::CCM.new('AES', [key[j]].pack('H*'), mac_len[j])
            c = ccm.encrypt(input, [nonce[j]].pack('H*'))
            assert_equal(output.unpack('H*'), c.unpack('H*'),
                         "Wrong ENCRYPT in Vector #{i + 1}")
          end
        end
      end
    end
  end

  #Test case from https://github.com/weidai11/cryptopp/blob/master/TestVectors/ccm.txt
  def test_aes_data_256
    key =  %W(
      0000000000000000000000000000000000000000000000000000000000000000
      fb7615b23d80891dd470980bc79584c8b2fb64ce60978f4d17fce45a49e830b7
    )

    nonce = %W(
      000000000000000000000000
      dbd1a3636024b7b402da7d6f
    )

    plaintext = %W(
      00000000000000000000000000000000
      a845348ec8c5b5f126f50e76fefd1b1e
    )

    ciphertext = %W(
      c1944044c8e7aa95d2de9513c7f3dd8c
      cc881261c6a7fa72b96a1739176b277f
    )

    mac = %W(
      4b0a3e5e51f151eb0ffae7c43d010fdb
      3472e1145f2c0cbe146349062cf0e423
    )

    assert(OpenSSL::CCM.ciphers.include?('AES'), 'Missing AES-Cipher')
    key.length.times do |i|
      mac_len = mac[i].length / 2
      ccm = OpenSSL::CCM.new('AES', [key[i]].pack('H*'), mac_len)
      c = ccm.encrypt([plaintext[i]].pack('H*'), [nonce[i]].pack('H*'))
      c_unpack = c.unpack('H*')
      assert_equal([mac[i]], c[-mac_len..-1].unpack('H*'),
                         "Wrong MAC ENCRYPT in Test #{i} ")
      assert_equal([ciphertext[i]], c[0..-mac_len - 1].unpack('H*'),
                         "Wrong ciphertext ENCRYPT in Test #{i}")
    end
  end
end
