#ifndef _ELM_INTERFACE_ATSPI_ACCESSIBLE_EO_H_
#define _ELM_INTERFACE_ATSPI_ACCESSIBLE_EO_H_

#ifndef _ELM_INTERFACE_ATSPI_ACCESSIBLE_EO_CLASS_TYPE
#define _ELM_INTERFACE_ATSPI_ACCESSIBLE_EO_CLASS_TYPE

typedef Eo Elm_Interface_Atspi_Accessible;

#endif

#ifndef _ELM_INTERFACE_ATSPI_ACCESSIBLE_EO_TYPES
#define _ELM_INTERFACE_ATSPI_ACCESSIBLE_EO_TYPES

/** Type of accessibility object
 *
 * @ingroup Elm_Atspi
 */
typedef enum
{
  ELM_ATSPI_TYPE_REGULAR = 0, /** default accessibile object */
  ELM_ATSPI_TYPE_DISABLED, /** skip object and its children in accessibility
                            * hierarchy */
  ELM_ATSPI_TYPE_SKIPPED /** skip object in accessibility hierarchy */
} Elm_Atspi_Type;

/** Describes the role of an object visible to AT-SPI Clients.
 *
 * @ingroup Elm_Atspi
 */
typedef enum
{
  ELM_ATSPI_ROLE_INVALID = 0,
  ELM_ATSPI_ROLE_ACCELERATOR_LABEL,
  ELM_ATSPI_ROLE_ALERT,
  ELM_ATSPI_ROLE_ANIMATION,
  ELM_ATSPI_ROLE_ARROW,
  ELM_ATSPI_ROLE_CALENDAR,
  ELM_ATSPI_ROLE_CANVAS,
  ELM_ATSPI_ROLE_CHECK_BOX,
  ELM_ATSPI_ROLE_CHECK_MENU_ITEM,
  ELM_ATSPI_ROLE_COLOR_CHOOSER,
  ELM_ATSPI_ROLE_COLUMN_HEADER,
  ELM_ATSPI_ROLE_COMBO_BOX,
  ELM_ATSPI_ROLE_DATE_EDITOR,
  ELM_ATSPI_ROLE_DESKTOP_ICON,
  ELM_ATSPI_ROLE_DESKTOP_FRAME,
  ELM_ATSPI_ROLE_DIAL,
  ELM_ATSPI_ROLE_DIALOG,
  ELM_ATSPI_ROLE_DIRECTORY_PANE,
  ELM_ATSPI_ROLE_DRAWING_AREA,
  ELM_ATSPI_ROLE_FILE_CHOOSER,
  ELM_ATSPI_ROLE_FILLER,
  ELM_ATSPI_ROLE_FOCUS_TRAVERSABLE,
  ELM_ATSPI_ROLE_FONT_CHOOSER,
  ELM_ATSPI_ROLE_FRAME,
  ELM_ATSPI_ROLE_GLASS_PANE,
  ELM_ATSPI_ROLE_HTML_CONTAINER,
  ELM_ATSPI_ROLE_ICON,
  ELM_ATSPI_ROLE_IMAGE,
  ELM_ATSPI_ROLE_INTERNAL_FRAME,
  ELM_ATSPI_ROLE_LABEL,
  ELM_ATSPI_ROLE_LAYERED_PANE,
  ELM_ATSPI_ROLE_LIST,
  ELM_ATSPI_ROLE_LIST_ITEM,
  ELM_ATSPI_ROLE_MENU,
  ELM_ATSPI_ROLE_MENU_BAR,
  ELM_ATSPI_ROLE_MENU_ITEM,
  ELM_ATSPI_ROLE_OPTION_PANE,
  ELM_ATSPI_ROLE_PAGE_TAB,
  ELM_ATSPI_ROLE_PAGE_TAB_LIST,
  ELM_ATSPI_ROLE_PANEL,
  ELM_ATSPI_ROLE_PASSWORD_TEXT,
  ELM_ATSPI_ROLE_POPUP_MENU,
  ELM_ATSPI_ROLE_PROGRESS_BAR,
  ELM_ATSPI_ROLE_PUSH_BUTTON,
  ELM_ATSPI_ROLE_RADIO_BUTTON,
  ELM_ATSPI_ROLE_RADIO_MENU_ITEM,
  ELM_ATSPI_ROLE_ROOT_PANE,
  ELM_ATSPI_ROLE_ROW_HEADER,
  ELM_ATSPI_ROLE_SCROLL_BAR,
  ELM_ATSPI_ROLE_SCROLL_PANE,
  ELM_ATSPI_ROLE_SEPARATOR,
  ELM_ATSPI_ROLE_SLIDER,
  ELM_ATSPI_ROLE_SPIN_BUTTON,
  ELM_ATSPI_ROLE_SPLIT_PANE,
  ELM_ATSPI_ROLE_STATUS_BAR,
  ELM_ATSPI_ROLE_TABLE,
  ELM_ATSPI_ROLE_TABLE_CELL,
  ELM_ATSPI_ROLE_TABLE_COLUMN_HEADER,
  ELM_ATSPI_ROLE_TABLE_ROW_HEADER,
  ELM_ATSPI_ROLE_TEAROFF_MENU_ITEM,
  ELM_ATSPI_ROLE_TERMINAL,
  ELM_ATSPI_ROLE_TEXT,
  ELM_ATSPI_ROLE_TOGGLE_BUTTON,
  ELM_ATSPI_ROLE_TOOL_BAR,
  ELM_ATSPI_ROLE_TOOL_TIP,
  ELM_ATSPI_ROLE_TREE,
  ELM_ATSPI_ROLE_TREE_TABLE,
  ELM_ATSPI_ROLE_UNKNOWN,
  ELM_ATSPI_ROLE_VIEWPORT,
  ELM_ATSPI_ROLE_WINDOW,
  ELM_ATSPI_ROLE_EXTENDED,
  ELM_ATSPI_ROLE_HEADER,
  ELM_ATSPI_ROLE_FOOTER,
  ELM_ATSPI_ROLE_PARAGRAPH,
  ELM_ATSPI_ROLE_RULER,
  ELM_ATSPI_ROLE_APPLICATION,
  ELM_ATSPI_ROLE_AUTOCOMPLETE,
  ELM_ATSPI_ROLE_EDITBAR,
  ELM_ATSPI_ROLE_EMBEDDED,
  ELM_ATSPI_ROLE_ENTRY,
  ELM_ATSPI_ROLE_CHART,
  ELM_ATSPI_ROLE_CAPTION,
  ELM_ATSPI_ROLE_DOCUMENT_FRAME,
  ELM_ATSPI_ROLE_HEADING,
  ELM_ATSPI_ROLE_PAGE,
  ELM_ATSPI_ROLE_SECTION,
  ELM_ATSPI_ROLE_REDUNDANT_OBJECT,
  ELM_ATSPI_ROLE_FORM,
  ELM_ATSPI_ROLE_LINK,
  ELM_ATSPI_ROLE_INPUT_METHOD_WINDOW,
  ELM_ATSPI_ROLE_TABLE_ROW,
  ELM_ATSPI_ROLE_TREE_ITEM,
  ELM_ATSPI_ROLE_DOCUMENT_SPREADSHEET,
  ELM_ATSPI_ROLE_DOCUMENT_PRESENTATION,
  ELM_ATSPI_ROLE_DOCUMENT_TEXT,
  ELM_ATSPI_ROLE_DOCUMENT_WEB,
  ELM_ATSPI_ROLE_DOCUMENT_EMAIL,
  ELM_ATSPI_ROLE_COMMENT,
  ELM_ATSPI_ROLE_LIST_BOX,
  ELM_ATSPI_ROLE_GROUPING,
  ELM_ATSPI_ROLE_IMAGE_MAP,
  ELM_ATSPI_ROLE_NOTIFICATION,
  ELM_ATSPI_ROLE_INFO_BAR,
  ELM_ATSPI_ROLE_LAST_DEFINED
} Elm_Atspi_Role;

/** Describes a possible states of an object visible to AT-SPI clients.
 *
 * @ingroup Elm_Atspi
 */
typedef enum
{
  ELM_ATSPI_STATE_INVALID = 0,
  ELM_ATSPI_STATE_ACTIVE,
  ELM_ATSPI_STATE_ARMED,
  ELM_ATSPI_STATE_BUSY,
  ELM_ATSPI_STATE_CHECKED,
  ELM_ATSPI_STATE_COLLAPSED,
  ELM_ATSPI_STATE_DEFUNCT,
  ELM_ATSPI_STATE_EDITABLE,
  ELM_ATSPI_STATE_ENABLED,
  ELM_ATSPI_STATE_EXPANDABLE,
  ELM_ATSPI_STATE_EXPANDED,
  ELM_ATSPI_STATE_FOCUSABLE,
  ELM_ATSPI_STATE_FOCUSED,
  ELM_ATSPI_STATE_HAS_TOOLTIP,
  ELM_ATSPI_STATE_HORIZONTAL,
  ELM_ATSPI_STATE_ICONIFIED,
  ELM_ATSPI_STATE_MODAL,
  ELM_ATSPI_STATE_MULTI_LINE,
  ELM_ATSPI_STATE_MULTISELECTABLE,
  ELM_ATSPI_STATE_OPAQUE,
  ELM_ATSPI_STATE_PRESSED,
  ELM_ATSPI_STATE_RESIZABLE,
  ELM_ATSPI_STATE_SELECTABLE,
  ELM_ATSPI_STATE_SELECTED,
  ELM_ATSPI_STATE_SENSITIVE,
  ELM_ATSPI_STATE_SHOWING,
  ELM_ATSPI_STATE_SINGLE_LINE,
  ELM_ATSPI_STATE_STALE,
  ELM_ATSPI_STATE_TRANSIENT,
  ELM_ATSPI_STATE_VERTICAL,
  ELM_ATSPI_STATE_VISIBLE,
  ELM_ATSPI_STATE_MANAGES_DESCENDANTS,
  ELM_ATSPI_STATE_INDETERMINATE,
  ELM_ATSPI_STATE_REQUIRED,
  ELM_ATSPI_STATE_TRUNCATED,
  ELM_ATSPI_STATE_ANIMATED,
  ELM_ATSPI_STATE_INVALID_ENTRY,
  ELM_ATSPI_STATE_SUPPORTS_AUTOCOMPLETION,
  ELM_ATSPI_STATE_SELECTABLE_TEXT,
  ELM_ATSPI_STATE_IS_DEFAULT,
  ELM_ATSPI_STATE_VISITED,
  ELM_ATSPI_STATE_LAST_DEFINED
} Elm_Atspi_State_Type;

/** Describes relation between two objects.
 *
 * @ingroup Elm_Atspi
 */
typedef enum
{
  ELM_ATSPI_RELATION_NULL = 0,
  ELM_ATSPI_RELATION_LABEL_FOR,
  ELM_ATSPI_RELATION_LABELLED_BY,
  ELM_ATSPI_RELATION_CONTROLLER_FOR,
  ELM_ATSPI_RELATION_CONTROLLED_BY,
  ELM_ATSPI_RELATION_MEMBER_OF,
  ELM_ATSPI_RELATION_TOOLTIP_FOR,
  ELM_ATSPI_RELATION_NODE_CHILD_OF,
  ELM_ATSPI_RELATION_NODE_PARENT_OF,
  ELM_ATSPI_RELATION_EXTENDED,
  ELM_ATSPI_RELATION_FLOWS_TO,
  ELM_ATSPI_RELATION_FLOWS_FROM,
  ELM_ATSPI_RELATION_SUBWINDOW_OF,
  ELM_ATSPI_RELATION_EMBEDS,
  ELM_ATSPI_RELATION_EMBEDDED_BY,
  ELM_ATSPI_RELATION_POPUP_FOR,
  ELM_ATSPI_RELATION_PARENT_WINDOW_OF,
  ELM_ATSPI_RELATION_DESCRIPTION_FOR,
  ELM_ATSPI_RELATION_DESCRIBED_BY,
  ELM_ATSPI_RELATION_LAST_DEFINED
} Elm_Atspi_Relation_Type;

/** ATSPI object state set.
 *
 * @ingroup Elm_Atspi
 */
typedef uint64_t Elm_Atspi_State_Set;

/** ATSPI event listener
 *
 * @ingroup Elm_Atspi_Event
 */
typedef struct _Elm_Atspi_Event_Handler Elm_Atspi_Event_Handler;

typedef struct _Elm_Atspi_Event_State_Changed_Data
{
  Elm_Atspi_State_Type type;
  Eina_Bool new_value;
} Elm_Atspi_Event_State_Changed_Data;

typedef struct _Elm_Atspi_Event_Geometry_Changed_Data
{
  int x;
  int y;
  int width;
  int height;
} Elm_Atspi_Event_Geometry_Changed_Data;

typedef struct _Elm_Atspi_Event_Children_Changed_Data
{
  Eina_Bool is_added;
  Eo_Base *child;
} Elm_Atspi_Event_Children_Changed_Data;

typedef struct _Elm_Atspi_Attribute
{
  const char *key;
  const char *value;
} Elm_Atspi_Attribute;

typedef struct _Elm_Atspi_Relation
{
  Elm_Atspi_Relation_Type type;
  Eina_List *objects;
} Elm_Atspi_Relation;

typedef Eina_List *Elm_Atspi_Relation_Set;


#endif
/** ATSPI accessible mixin
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
#define ELM_INTERFACE_ATSPI_ACCESSIBLE_MIXIN elm_interface_atspi_accessible_mixin_get()

EWAPI const Eo_Class *elm_interface_atspi_accessible_mixin_get(void);

#ifdef ELM_INTERFACE_ATSPI_ACCESSIBLE_PROTECTED
/**
 * @brief Gets an localized string describing ATSPI widget role name.
 *
 * @return Localized widget role name
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI const char *elm_interface_atspi_accessible_localized_role_name_get(const Eo *obj);
#endif

/**
 * @brief String describing ATSPI widget role name. Should be free by a user.
 *
 * @param[in] name Obj name
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI void elm_interface_atspi_accessible_name_set(Eo *obj, char *name);

/**
 * @brief String describing ATSPI widget role name. Should be free by a user.
 *
 * @return Obj name
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI char *elm_interface_atspi_accessible_name_get(const Eo *obj);

#ifdef ELM_INTERFACE_ATSPI_ACCESSIBLE_PROTECTED
/** Gets an string describing ATSPI widget role name. Lists and elements Should
 * be free by a user.
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI Elm_Atspi_Relation_Set elm_interface_atspi_accessible_relation_set_get(const Eo *obj);
#endif

/**
 * @brief The role of the widget in ATSPI Accessibility domain.
 *
 * @param[in] role
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI void elm_interface_atspi_accessible_role_set(Eo *obj, Elm_Atspi_Role role);

/** The role of the widget in ATSPI Accessibility domain.
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI Elm_Atspi_Role elm_interface_atspi_accessible_role_get(const Eo *obj);

#ifdef ELM_INTERFACE_ATSPI_ACCESSIBLE_PROTECTED
/** Gets widget's accessible children.
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI Eina_List *elm_interface_atspi_accessible_children_get(const Eo *obj);
#endif

#ifdef ELM_INTERFACE_ATSPI_ACCESSIBLE_PROTECTED
/** Gets human-readable string indentifying widget accessibility role.
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI const char *elm_interface_atspi_accessible_role_name_get(const Eo *obj);
#endif

#ifdef ELM_INTERFACE_ATSPI_ACCESSIBLE_PROTECTED
/** Gets key-value pairs indentifying widget extra attributes. Must be free by
 * a user.
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI Eina_List *elm_interface_atspi_accessible_attributes_get(const Eo *obj);
#endif

#ifdef ELM_INTERFACE_ATSPI_ACCESSIBLE_PROTECTED
/** Gets index of the child in parent's children list.
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI int elm_interface_atspi_accessible_index_in_parent_get(const Eo *obj);
#endif

/**
 * @brief Widget contextual information.
 *
 * @param[in] description Widget contextual information
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI void elm_interface_atspi_accessible_description_set(Eo *obj, const char *description);

/**
 * @brief Widget contextual information.
 *
 * @return Widget contextual information
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI const char *elm_interface_atspi_accessible_description_get(const Eo *obj);

/**
 * @brief Widget accessible parent.
 *
 * @param[in] parent
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI void elm_interface_atspi_accessible_parent_set(Eo *obj, Elm_Interface_Atspi_Accessible *parent);

/** Widget accessible parent.
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI Elm_Interface_Atspi_Accessible *elm_interface_atspi_accessible_parent_get(const Eo *obj);

#ifdef ELM_INTERFACE_ATSPI_ACCESSIBLE_PROTECTED
/** Gets set describing widget accessible states.
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI Elm_Atspi_State_Set elm_interface_atspi_accessible_state_set_get(const Eo *obj);
#endif

/**
 * @brief Sets the translation domain of "name" and "description" properties.
 *
 * Translation domain should be set if application wants to support i18n for
 * accessibily "name" and "description" properties.
 *
 * When translation domain is set values of "name" and "description" properties
 * will be translated with dgettext function using current translation domain
 * as "domainname" parameter.
 *
 * It is application developer responsibility to ensure that translation files
 * are loaded and binded to translation domain when accessibility is enabled.
 *
 * @param[in] domain Translation domain
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI void elm_interface_atspi_accessible_translation_domain_set(Eo *obj, const char *domain);

/**
 * @brief Sets the translation domain of "name" and "description" properties.
 *
 * Translation domain should be set if application wants to support i18n for
 * accessibily "name" and "description" properties.
 *
 * When translation domain is set values of "name" and "description" properties
 * will be translated with dgettext function using current translation domain
 * as "domainname" parameter.
 *
 * It is application developer responsibility to ensure that translation files
 * are loaded and binded to translation domain when accessibility is enabled.
 *
 * @return Translation domain
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI const char *elm_interface_atspi_accessible_translation_domain_get(const Eo *obj);

/**
 * @brief Type of accessibility object
 *
 * @param[in] type
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI void elm_interface_atspi_accessible_type_set(Eo *obj, Elm_Atspi_Type type);

/** Type of accessibility object
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI Elm_Atspi_Type elm_interface_atspi_accessible_type_get(const Eo *obj);

/** Get root object of accessible object hierarchy
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI Eo_Base *elm_interface_atspi_accessible_root_get(const Eo *obj);

#ifdef ELM_INTERFACE_ATSPI_ACCESSIBLE_PROTECTED
/**
 * @brief Register accessibility event listener
 *
 * @param[in] cb callback
 * @param[in] data data
 *
 * @return Event handler
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI Elm_Atspi_Event_Handler *elm_interface_atspi_accessible_event_handler_add(const Eo *obj, Eo_Event_Cb cb, void *data);
#endif

#ifdef ELM_INTERFACE_ATSPI_ACCESSIBLE_PROTECTED
/**
 * @brief Deregister accessibility event listener
 *
 * @param[in] handler Event handler
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI void elm_interface_atspi_accessible_event_handler_del(const Eo *obj, Elm_Atspi_Event_Handler *handler);
#endif

#ifdef ELM_INTERFACE_ATSPI_ACCESSIBLE_PROTECTED
EOAPI void elm_interface_atspi_accessible_event_emit(const Eo *obj, Elm_Interface_Atspi_Accessible *accessible, const Eo_Event_Description *event, void *event_info);
#endif

/**
 * @brief Defines the relationship between two accessible objects.
 *
 * Adds unique relation between source object and relation_object of a given
 * type.
 *
 * Relationships can be queried by Assistive Technology clients to provide
 * customized feedback, improving overall user experience.
 *
 * Relationship_append API is asymmetric, which means that appending, for
 * example, relation ELM_ATSPI_RELATION_FLOWS_TO from object A to B, do NOT
 * append relation ELM_ATSPI_RELATION_FLOWS_FROM from object B to object A.
 *
 * return: EINA_TRUE is relationship was successfully appended, EINA_FALSE
 * otherwise
 *
 * @param[in] type
 * @param[in] relation_object
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI Eina_Bool elm_interface_atspi_accessible_relationship_append(Eo *obj, Elm_Atspi_Relation_Type type, const Elm_Interface_Atspi_Accessible *relation_object);

/**
 * @brief Removes the relationship between two accessible objects.
 *
 * If relation_object is NULL function removes all relations of given type.
 *
 * @param[in] type
 * @param[in] relation_object
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI void elm_interface_atspi_accessible_relationship_remove(Eo *obj, Elm_Atspi_Relation_Type type, const Elm_Interface_Atspi_Accessible *relation_object);

/** Removes all relationships in accessible object.
 *
 * @ingroup Elm_Interface_Atspi_Accessible
 */
EOAPI void elm_interface_atspi_accessible_relationships_clear(Eo *obj);

EOAPI extern const Eo_Event_Description _ELM_INTERFACE_ATSPI_ACCESSIBLE_EVENT_PROPERTY_CHANGED;
EOAPI extern const Eo_Event_Description _ELM_INTERFACE_ATSPI_ACCESSIBLE_EVENT_CHILDREN_CHANGED;
EOAPI extern const Eo_Event_Description _ELM_INTERFACE_ATSPI_ACCESSIBLE_EVENT_STATE_CHANGED;
EOAPI extern const Eo_Event_Description _ELM_INTERFACE_ATSPI_ACCESSIBLE_EVENT_BOUNDS_CHANGED;
EOAPI extern const Eo_Event_Description _ELM_INTERFACE_ATSPI_ACCESSIBLE_EVENT_VISIBLE_DATA_CHANGED;
EOAPI extern const Eo_Event_Description _ELM_INTERFACE_ATSPI_ACCESSIBLE_EVENT_ACTIVE_DESCENDANT_CHANGED;
EOAPI extern const Eo_Event_Description _ELM_INTERFACE_ATSPI_ACCESSIBLE_EVENT_ADDED;
EOAPI extern const Eo_Event_Description _ELM_INTERFACE_ATSPI_ACCESSIBLE_EVENT_REMOVED;

/**
 * No description
 */
#define ELM_INTERFACE_ATSPI_ACCESSIBLE_EVENT_PROPERTY_CHANGED (&(_ELM_INTERFACE_ATSPI_ACCESSIBLE_EVENT_PROPERTY_CHANGED))

/**
 * No description
 */
#define ELM_INTERFACE_ATSPI_ACCESSIBLE_EVENT_CHILDREN_CHANGED (&(_ELM_INTERFACE_ATSPI_ACCESSIBLE_EVENT_CHILDREN_CHANGED))

/**
 * No description
 */
#define ELM_INTERFACE_ATSPI_ACCESSIBLE_EVENT_STATE_CHANGED (&(_ELM_INTERFACE_ATSPI_ACCESSIBLE_EVENT_STATE_CHANGED))

/**
 * No description
 */
#define ELM_INTERFACE_ATSPI_ACCESSIBLE_EVENT_BOUNDS_CHANGED (&(_ELM_INTERFACE_ATSPI_ACCESSIBLE_EVENT_BOUNDS_CHANGED))

/**
 * No description
 */
#define ELM_INTERFACE_ATSPI_ACCESSIBLE_EVENT_VISIBLE_DATA_CHANGED (&(_ELM_INTERFACE_ATSPI_ACCESSIBLE_EVENT_VISIBLE_DATA_CHANGED))

/**
 * No description
 */
#define ELM_INTERFACE_ATSPI_ACCESSIBLE_EVENT_ACTIVE_DESCENDANT_CHANGED (&(_ELM_INTERFACE_ATSPI_ACCESSIBLE_EVENT_ACTIVE_DESCENDANT_CHANGED))

/**
 * No description
 */
#define ELM_INTERFACE_ATSPI_ACCESSIBLE_EVENT_ADDED (&(_ELM_INTERFACE_ATSPI_ACCESSIBLE_EVENT_ADDED))

/**
 * No description
 */
#define ELM_INTERFACE_ATSPI_ACCESSIBLE_EVENT_REMOVED (&(_ELM_INTERFACE_ATSPI_ACCESSIBLE_EVENT_REMOVED))

#endif
