/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.index.shard;

import java.io.IOException;
import java.math.BigInteger;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import org.apache.lucene.util.IOUtils;
import org.elasticsearch.common.logging.ESLogger;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.env.NodeEnvironment;
import org.elasticsearch.env.ShardLock;
import org.elasticsearch.index.shard.IndexShard;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.shard.ShardStateMetaData;

public final class ShardPath {
    public static final String INDEX_FOLDER_NAME = "index";
    public static final String TRANSLOG_FOLDER_NAME = "translog";
    private final Path path;
    private final String indexUUID;
    private final ShardId shardId;
    private final Path shardStatePath;
    private final boolean isCustomDataPath;

    public ShardPath(boolean isCustomDataPath, Path dataPath, Path shardStatePath, String indexUUID, ShardId shardId) {
        assert (dataPath.getFileName().toString().equals(Integer.toString(shardId.id()))) : "dataPath must end with the shard ID but didn't: " + dataPath.toString();
        assert (shardStatePath.getFileName().toString().equals(Integer.toString(shardId.id()))) : "shardStatePath must end with the shard ID but didn't: " + dataPath.toString();
        assert (dataPath.getParent().getFileName().toString().equals(shardId.getIndex())) : "dataPath must end with index/shardID but didn't: " + dataPath.toString();
        assert (shardStatePath.getParent().getFileName().toString().equals(shardId.getIndex())) : "shardStatePath must end with index/shardID but didn't: " + dataPath.toString();
        if (isCustomDataPath && dataPath.equals(shardStatePath)) {
            throw new IllegalArgumentException("shard state path must be different to the data path when using custom data paths");
        }
        this.isCustomDataPath = isCustomDataPath;
        this.path = dataPath;
        this.indexUUID = indexUUID;
        this.shardId = shardId;
        this.shardStatePath = shardStatePath;
    }

    public Path resolveTranslog() {
        return this.path.resolve(TRANSLOG_FOLDER_NAME);
    }

    public Path resolveIndex() {
        return this.path.resolve(INDEX_FOLDER_NAME);
    }

    public Path getDataPath() {
        return this.path;
    }

    public boolean exists() {
        return Files.exists(this.path, new LinkOption[0]);
    }

    public String getIndexUUID() {
        return this.indexUUID;
    }

    public ShardId getShardId() {
        return this.shardId;
    }

    public Path getShardStatePath() {
        return this.shardStatePath;
    }

    public Path getRootDataPath() {
        Path noIndexShardId = this.getDataPath().getParent().getParent();
        return this.isCustomDataPath ? noIndexShardId : noIndexShardId.getParent();
    }

    public Path getRootStatePath() {
        return this.getShardStatePath().getParent().getParent().getParent();
    }

    public boolean isCustomDataPath() {
        return this.isCustomDataPath;
    }

    public static ShardPath loadShardPath(ESLogger logger, NodeEnvironment env, ShardId shardId, Settings indexSettings) throws IOException {
        String indexUUID = indexSettings.get("index.uuid", "_na_");
        Path[] paths = env.availableShardPaths(shardId);
        Path loadedPath = null;
        for (Path path : paths) {
            ShardStateMetaData load = ShardStateMetaData.FORMAT.loadLatestState(logger, path);
            if (load == null) continue;
            if (!load.indexUUID.equals(indexUUID) && !"_na_".equals(load.indexUUID)) {
                logger.warn("{} found shard on path: [{}] with a different index UUID - this shard seems to be leftover from a different index with the same name. Remove the leftover shard in order to reuse the path with the current index", shardId, path);
                throw new IllegalStateException(shardId + " index UUID in shard state was: " + load.indexUUID + " expected: " + indexUUID + " on shard path: " + path);
            }
            if (loadedPath == null) {
                loadedPath = path;
                continue;
            }
            throw new IllegalStateException(shardId + " more than one shard state found");
        }
        if (loadedPath == null) {
            return null;
        }
        Path statePath = loadedPath;
        Path dataPath = NodeEnvironment.hasCustomDataPath(indexSettings) ? env.resolveCustomLocation(indexSettings, shardId) : statePath;
        logger.debug("{} loaded data path [{}], state path [{}]", shardId, dataPath, statePath);
        return new ShardPath(NodeEnvironment.hasCustomDataPath(indexSettings), dataPath, statePath, indexUUID, shardId);
    }

    public static void deleteLeftoverShardDirectory(ESLogger logger, NodeEnvironment env, ShardLock lock, Settings indexSettings) throws IOException {
        Path[] paths;
        String indexUUID = indexSettings.get("index.uuid", "_na_");
        for (Path path : paths = env.availableShardPaths(lock.getShardId())) {
            ShardStateMetaData load = ShardStateMetaData.FORMAT.loadLatestState(logger, path);
            if (load == null || load.indexUUID.equals(indexUUID) || "_na_".equals(load.indexUUID)) continue;
            logger.warn("{} deleting leftover shard on path: [{}] with a different index UUID", lock.getShardId(), path);
            assert (Files.isDirectory(path, new LinkOption[0])) : path + " is not a directory";
            NodeEnvironment.acquireFSLockForPaths(indexSettings, paths);
            IOUtils.rm((Path[])new Path[]{path});
        }
    }

    private static Map<Path, Long> getEstimatedReservedBytes(NodeEnvironment env, long avgShardSizeInBytes, Iterable<IndexShard> shards) throws IOException {
        long totFreeSpace = 0L;
        for (NodeEnvironment.NodePath nodePath : env.nodePaths()) {
            totFreeSpace += nodePath.fileStore.getUsableSpace();
        }
        long estShardSizeInBytes = Math.max(avgShardSizeInBytes, (long)((double)totFreeSpace / 20.0));
        HashMap<Path, Long> reservedBytes = new HashMap<Path, Long>();
        for (IndexShard shard : shards) {
            Path dataPath = NodeEnvironment.shardStatePathToDataPath(shard.shardPath().getShardStatePath());
            Long curBytes = (Long)reservedBytes.get(dataPath);
            if (curBytes == null) {
                curBytes = 0L;
            }
            reservedBytes.put(dataPath, curBytes + estShardSizeInBytes);
        }
        return reservedBytes;
    }

    public static ShardPath selectNewPathForShard(NodeEnvironment env, ShardId shardId, Settings indexSettings, long avgShardSizeInBytes, Map<Path, Integer> dataPathToShardCount) throws IOException {
        Path statePath;
        Path dataPath;
        if (NodeEnvironment.hasCustomDataPath(indexSettings)) {
            dataPath = env.resolveCustomLocation(indexSettings, shardId);
            statePath = env.nodePaths()[0].resolve(shardId);
        } else {
            BigInteger totFreeSpace = BigInteger.ZERO;
            for (NodeEnvironment.NodePath nodePath : env.nodePaths()) {
                totFreeSpace = totFreeSpace.add(BigInteger.valueOf(nodePath.fileStore.getUsableSpace()));
            }
            BigInteger estShardSizeInBytes = BigInteger.valueOf(avgShardSizeInBytes).max(totFreeSpace.divide(BigInteger.valueOf(20L)));
            NodeEnvironment.NodePath[] paths = env.nodePaths();
            NodeEnvironment.NodePath bestPath = null;
            BigInteger maxUsableBytes = BigInteger.valueOf(Long.MIN_VALUE);
            for (NodeEnvironment.NodePath nodePath : paths) {
                FileStore fileStore = nodePath.fileStore;
                BigInteger usableBytes = BigInteger.valueOf(fileStore.getUsableSpace());
                assert (usableBytes.compareTo(BigInteger.ZERO) >= 0);
                Integer count = dataPathToShardCount.get(nodePath.path);
                if (count != null) {
                    usableBytes = usableBytes.subtract(estShardSizeInBytes.multiply(BigInteger.valueOf(count.intValue())));
                }
                if (bestPath != null && usableBytes.compareTo(maxUsableBytes) <= 0) continue;
                maxUsableBytes = usableBytes;
                bestPath = nodePath;
            }
            dataPath = statePath = bestPath.resolve(shardId);
        }
        String indexUUID = indexSettings.get("index.uuid", "_na_");
        return new ShardPath(NodeEnvironment.hasCustomDataPath(indexSettings), dataPath, statePath, indexUUID, shardId);
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        ShardPath shardPath = (ShardPath)o;
        if (this.shardId != null ? !this.shardId.equals(shardPath.shardId) : shardPath.shardId != null) {
            return false;
        }
        if (this.indexUUID != null ? !this.indexUUID.equals(shardPath.indexUUID) : shardPath.indexUUID != null) {
            return false;
        }
        return !(this.path != null ? !this.path.equals(shardPath.path) : shardPath.path != null);
    }

    public int hashCode() {
        int result = this.path != null ? this.path.hashCode() : 0;
        result = 31 * result + (this.indexUUID != null ? this.indexUUID.hashCode() : 0);
        result = 31 * result + (this.shardId != null ? this.shardId.hashCode() : 0);
        return result;
    }

    public String toString() {
        return "ShardPath{path=" + this.path + ", indexUUID='" + this.indexUUID + '\'' + ", shard=" + this.shardId + '}';
    }
}

