/*
 * Copyright © 2008-2011 Kristian Høgsberg
 * Copyright © 2012 Collabora, Ltd.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#ifndef _WAYLAND_SYSTEM_COMPOSITOR_H_
#define _WAYLAND_SYSTEM_COMPOSITOR_H_

#ifdef  __cplusplus
extern "C" {
#endif

#include <stdbool.h>
#include <stdint.h>
#include <time.h>
#include <pixman.h>
#include <xkbcommon/xkbcommon.h>

#define WL_HIDE_DEPRECATED
#include <wayland-server.h>

#include "version.h"
#include "matrix.h"
#include "config-parser.h"
#include "zalloc.h"
#include "timeline-object.h"

struct weston_geometry {
	int32_t x, y;
	int32_t width, height;
};

struct weston_position {
	int32_t x, y;
};

struct weston_size {
	int32_t width, height;
};

struct weston_transform {
	struct weston_matrix matrix;
	struct wl_list link;
};

struct weston_surface;
struct weston_buffer;
struct shell_surface;
struct weston_seat;
struct weston_output;
struct input_method;
struct weston_pointer;
struct linux_dmabuf_buffer;
struct weston_recorder;
struct weston_pointer_constraint;

enum weston_keyboard_modifier {
	MODIFIER_CTRL = (1 << 0),
	MODIFIER_ALT = (1 << 1),
	MODIFIER_SUPER = (1 << 2),
	MODIFIER_SHIFT = (1 << 3),
};

enum weston_keyboard_locks {
	WESTON_NUM_LOCK = (1 << 0),
	WESTON_CAPS_LOCK = (1 << 1),
};

enum weston_led {
	LED_NUM_LOCK = (1 << 0),
	LED_CAPS_LOCK = (1 << 1),
	LED_SCROLL_LOCK = (1 << 2),
};

struct weston_mode {
	uint32_t flags;
	int32_t width, height;
	uint32_t refresh;
	struct wl_list link;
};

struct weston_animation {
	void (*frame)(struct weston_animation *animation,
		      struct weston_output *output, uint32_t msecs);
	int frame_counter;
	struct wl_list link;
};

enum {
	WESTON_SPRING_OVERSHOOT,
	WESTON_SPRING_CLAMP,
	WESTON_SPRING_BOUNCE
};

struct weston_spring {
	double k;
	double friction;
	double current;
	double target;
	double previous;
	double min, max;
	uint32_t timestamp;
	uint32_t clip;
};

struct weston_output_zoom {
	bool active;
	float increment;
	float level;
	float max_level;
	float trans_x, trans_y;
	struct {
		double x, y;
	} current;
	struct weston_seat *seat;
	struct weston_animation animation_z;
	struct weston_spring spring_z;
	struct wl_listener motion_listener;
};

/* bit compatible with drm definitions. */
enum dpms_enum {
	WESTON_DPMS_ON,
	WESTON_DPMS_STANDBY,
	WESTON_DPMS_SUSPEND,
	WESTON_DPMS_OFF
};

struct weston_output {
	uint32_t id;
	char *name;

	void *renderer_state;

	struct wl_list link;
	struct wl_list resource_list;
	struct wl_global *global;
	struct weston_compositor *compositor;

	/** From global to output buffer coordinates. */
	struct weston_matrix matrix;
	/** From output buffer to global coordinates. */
	struct weston_matrix inverse_matrix;

	struct wl_list animation_list;
	int32_t x, y, width, height;
	int32_t mm_width, mm_height;

	/** Output area in global coordinates, simple rect */
	pixman_region32_t region;

	pixman_region32_t previous_damage;
	int repaint_needed;
	int repaint_scheduled;
	struct wl_event_source *repaint_timer;
	struct weston_output_zoom zoom;
	int dirty;
	struct wl_signal frame_signal;
	struct wl_signal destroy_signal;
	int move_x, move_y;
	uint32_t frame_time; /* presentation timestamp in milliseconds */
	uint64_t msc;        /* media stream counter */
	int disable_planes;
	int destroying;
	struct wl_list feedback_list;

	char *make, *model, *serial_number;
	uint32_t subpixel;
	uint32_t transform;
	int32_t native_scale;
	int32_t current_scale;
	int32_t original_scale;

	struct weston_mode *native_mode;
	struct weston_mode *current_mode;
	struct weston_mode *original_mode;
	struct wl_list mode_list;

	void (*start_repaint_loop)(struct weston_output *output);
	int (*repaint)(struct weston_output *output,
			pixman_region32_t *damage);
	void (*destroy)(struct weston_output *output);
	void (*assign_planes)(struct weston_output *output);
	int (*switch_mode)(struct weston_output *output, struct weston_mode *mode);

	/* backlight values are on 0-255 range, where higher is brighter */
	int32_t backlight_current;
	void (*set_backlight)(struct weston_output *output, uint32_t value);
	void (*set_dpms)(struct weston_output *output, enum dpms_enum level);

	int connection_internal;
	uint16_t gamma_size;
	void (*set_gamma)(struct weston_output *output,
			  uint16_t size,
			  uint16_t *r,
			  uint16_t *g,
			  uint16_t *b);

	struct weston_timeline_object timeline;
};

enum weston_pointer_motion_mask {
	WESTON_POINTER_MOTION_ABS = 1 << 0,
	WESTON_POINTER_MOTION_REL = 1 << 1,
	WESTON_POINTER_MOTION_REL_UNACCEL = 1 << 2,
};

struct weston_pointer_motion_event {
	uint32_t mask;
	uint64_t time_usec;
	double x;
	double y;
	double dx;
	double dy;
	double dx_unaccel;
	double dy_unaccel;
};

struct weston_pointer_axis_event {
	uint32_t axis;
	double value;
	bool has_discrete;
	int32_t discrete;
};

struct weston_pointer_grab;
struct weston_pointer_grab_interface {
	void (*focus)(struct weston_pointer_grab *grab);
	void (*motion)(struct weston_pointer_grab *grab, uint32_t time,
		       struct weston_pointer_motion_event *event);
	void (*button)(struct weston_pointer_grab *grab,
		       uint32_t time, uint32_t button, uint32_t state);
	void (*axis)(struct weston_pointer_grab *grab,
		     uint32_t time,
		     struct weston_pointer_axis_event *event);
	void (*axis_source)(struct weston_pointer_grab *grab, uint32_t source);
	void (*frame)(struct weston_pointer_grab *grab);
	void (*cancel)(struct weston_pointer_grab *grab);
};

struct weston_pointer_grab {
	const struct weston_pointer_grab_interface *interface;
	struct weston_pointer *pointer;
};

struct weston_keyboard_grab;
struct weston_keyboard_grab_interface {
	void (*key)(struct weston_keyboard_grab *grab, uint32_t time,
		    uint32_t key, uint32_t state);
	void (*modifiers)(struct weston_keyboard_grab *grab, uint32_t serial,
			  uint32_t mods_depressed, uint32_t mods_latched,
			  uint32_t mods_locked, uint32_t group);
	void (*cancel)(struct weston_keyboard_grab *grab);
};

struct weston_keyboard_grab {
	const struct weston_keyboard_grab_interface *interface;
	struct weston_keyboard *keyboard;
};

struct weston_touch_grab;
struct weston_touch_grab_interface {
	void (*down)(struct weston_touch_grab *grab,
			uint32_t time,
			int touch_id,
			wl_fixed_t sx,
			wl_fixed_t sy);
	void (*up)(struct weston_touch_grab *grab,
			uint32_t time,
			int touch_id);
	void (*motion)(struct weston_touch_grab *grab,
			uint32_t time,
			int touch_id,
			wl_fixed_t sx,
			wl_fixed_t sy);
	void (*frame)(struct weston_touch_grab *grab);
	void (*cancel)(struct weston_touch_grab *grab);
};

struct weston_touch_grab {
	const struct weston_touch_grab_interface *interface;
	struct weston_touch *touch;
};

struct weston_data_offer {
	struct wl_resource *resource;
	struct weston_data_source *source;
	struct wl_listener source_destroy_listener;
	uint32_t dnd_actions;
	enum wl_data_device_manager_dnd_action preferred_dnd_action;
	bool in_ask;
};

struct weston_data_source {
	struct wl_resource *resource;
	struct wl_signal destroy_signal;
	struct wl_array mime_types;
	struct weston_data_offer *offer;
	struct weston_seat *seat;
	bool accepted;
	bool actions_set;
	uint32_t dnd_actions;
	enum wl_data_device_manager_dnd_action current_dnd_action;
	enum wl_data_device_manager_dnd_action compositor_action;

	void (*accept)(struct weston_data_source *source,
		       uint32_t serial, const char *mime_type);
	void (*send)(struct weston_data_source *source,
		     const char *mime_type, int32_t fd);
	void (*cancel)(struct weston_data_source *source);
};

struct weston_pointer_client {
	struct wl_list link;
	struct wl_client *client;
	struct wl_list pointer_resources;
	struct wl_list relative_pointer_resources;
};

struct weston_pointer {
	struct weston_seat *seat;

	struct wl_list pointer_clients;

	struct weston_view *focus;
	struct weston_pointer_client *focus_client;
	uint32_t focus_serial;
	struct wl_listener focus_view_listener;
	struct wl_listener focus_resource_listener;
	struct wl_signal focus_signal;
	struct wl_signal motion_signal;
	struct wl_signal destroy_signal;

	struct weston_view *sprite;
	struct wl_listener sprite_destroy_listener;
	int32_t hotspot_x, hotspot_y;

	struct weston_pointer_grab *grab;
	struct weston_pointer_grab default_grab;
	wl_fixed_t grab_x, grab_y;
	uint32_t grab_button;
	uint32_t grab_serial;
	uint32_t grab_time;

	wl_fixed_t x, y;
	wl_fixed_t sx, sy;
	uint32_t button_count;

	struct wl_listener output_destroy_listener;
};


struct weston_touch {
	struct weston_seat *seat;

	struct wl_list resource_list;
	struct wl_list focus_resource_list;
	struct weston_view *focus;
	struct wl_listener focus_view_listener;
	struct wl_listener focus_resource_listener;
	uint32_t focus_serial;
	struct wl_signal focus_signal;

	uint32_t num_tp;

	struct weston_touch_grab *grab;
	struct weston_touch_grab default_grab;
	int grab_touch_id;
	wl_fixed_t grab_x, grab_y;
	uint32_t grab_serial;
	uint32_t grab_time;
};

void
weston_pointer_motion_to_abs(struct weston_pointer *pointer,
			     struct weston_pointer_motion_event *event,
			     wl_fixed_t *x, wl_fixed_t *y);

struct weston_pointer *
weston_pointer_create(struct weston_seat *seat);
void
weston_pointer_destroy(struct weston_pointer *pointer);
void
weston_pointer_send_motion(struct weston_pointer *pointer, uint32_t time,
			   struct weston_pointer_motion_event *event);
bool
weston_pointer_has_focus_resource(struct weston_pointer *pointer);
void
weston_pointer_send_button(struct weston_pointer *pointer,
			   uint32_t time, uint32_t button, uint32_t state_w);
void
weston_pointer_send_axis(struct weston_pointer *pointer,
			 uint32_t time,
			 struct weston_pointer_axis_event *event);
void
weston_pointer_send_axis_source(struct weston_pointer *pointer,
				uint32_t source);
void
weston_pointer_send_frame(struct weston_pointer *pointer);

void
weston_pointer_set_focus(struct weston_pointer *pointer,
			 struct weston_view *view,
			 wl_fixed_t sx, wl_fixed_t sy);
void
weston_pointer_clear_focus(struct weston_pointer *pointer);
void
weston_pointer_start_grab(struct weston_pointer *pointer,
			  struct weston_pointer_grab *grab);
void
weston_pointer_end_grab(struct weston_pointer *pointer);
void
weston_pointer_clamp(struct weston_pointer *pointer,
			    wl_fixed_t *fx, wl_fixed_t *fy);
void
weston_pointer_move(struct weston_pointer *pointer,
		    struct weston_pointer_motion_event *event);
void
weston_pointer_set_default_grab(struct weston_pointer *pointer,
		const struct weston_pointer_grab_interface *interface);

void
weston_pointer_constraint_destroy(struct weston_pointer_constraint *constraint);

struct weston_keyboard *
weston_keyboard_create(void);
void
weston_keyboard_destroy(struct weston_keyboard *keyboard);
void
weston_keyboard_set_focus(struct weston_keyboard *keyboard,
			  struct weston_surface *surface);
void
weston_keyboard_start_grab(struct weston_keyboard *device,
			   struct weston_keyboard_grab *grab);
void
weston_keyboard_end_grab(struct weston_keyboard *keyboard);
int
/*
 * 'mask' and 'value' should be a bitwise mask of one or more
 * valued of the weston_keyboard_locks enum.
 */
weston_keyboard_set_locks(struct weston_keyboard *keyboard,
			  uint32_t mask, uint32_t value);

bool
weston_keyboard_has_focus_resource(struct weston_keyboard *keyboard);
void
weston_keyboard_send_key(struct weston_keyboard *keyboard,
			 uint32_t time, uint32_t key,
			 enum wl_keyboard_key_state state);
void
weston_keyboard_send_modifiers(struct weston_keyboard *keyboard,
			       uint32_t serial, uint32_t mods_depressed,
			       uint32_t mods_latched,
			       uint32_t mods_locked, uint32_t group);

struct weston_touch *
weston_touch_create(void);
void
weston_touch_destroy(struct weston_touch *touch);
void
weston_touch_set_focus(struct weston_touch *touch,
		       struct weston_view *view);
void
weston_touch_start_grab(struct weston_touch *device,
			struct weston_touch_grab *grab);
void
weston_touch_end_grab(struct weston_touch *touch);

bool
weston_touch_has_focus_resource(struct weston_touch *touch);
void
weston_touch_send_down(struct weston_touch *touch, uint32_t time,
		       int touch_id, wl_fixed_t x, wl_fixed_t y);
void
weston_touch_send_up(struct weston_touch *touch, uint32_t time, int touch_id);
void
weston_touch_send_motion(struct weston_touch *touch, uint32_t time,
			 int touch_id, wl_fixed_t x, wl_fixed_t y);
void
weston_touch_send_frame(struct weston_touch *touch);

void
wl_data_device_set_keyboard_focus(struct weston_seat *seat);

int
wl_data_device_manager_init(struct wl_display *display);


void
weston_seat_set_selection(struct weston_seat *seat,
			  struct weston_data_source *source, uint32_t serial);
void
weston_seat_send_selection(struct weston_seat *seat, struct wl_client *client);

int
weston_pointer_start_drag(struct weston_pointer *pointer,
		       struct weston_data_source *source,
		       struct weston_surface *icon,
		       struct wl_client *client);
int
weston_touch_start_drag(struct weston_touch *touch,
			struct weston_data_source *source,
			struct weston_surface *icon,
			struct wl_client *client);

struct weston_xkb_info {
	struct xkb_keymap *keymap;
	int keymap_fd;
	size_t keymap_size;
	char *keymap_area;
	int32_t ref_count;
	xkb_mod_index_t shift_mod;
	xkb_mod_index_t caps_mod;
	xkb_mod_index_t ctrl_mod;
	xkb_mod_index_t alt_mod;
	xkb_mod_index_t mod2_mod;
	xkb_mod_index_t mod3_mod;
	xkb_mod_index_t super_mod;
	xkb_mod_index_t mod5_mod;
	xkb_led_index_t num_led;
	xkb_led_index_t caps_led;
	xkb_led_index_t scroll_led;
};

struct weston_keyboard {
	struct weston_seat *seat;

	struct wl_list resource_list;
	struct wl_list focus_resource_list;
	struct weston_surface *focus;
	struct wl_listener focus_resource_listener;
	uint32_t focus_serial;
	struct wl_signal focus_signal;

	struct weston_keyboard_grab *grab;
	struct weston_keyboard_grab default_grab;
	uint32_t grab_key;
	uint32_t grab_serial;
	uint32_t grab_time;

	struct wl_array keys;

	struct {
		uint32_t mods_depressed;
		uint32_t mods_latched;
		uint32_t mods_locked;
		uint32_t group;
	} modifiers;

	struct weston_keyboard_grab input_method_grab;
	struct wl_resource *input_method_resource;

	struct weston_xkb_info *xkb_info;
	struct {
		struct xkb_state *state;
		enum weston_led leds;
	} xkb_state;
	struct xkb_keymap *pending_keymap;
};

struct weston_seat {
	struct wl_list base_resource_list;

	struct wl_global *global;
	struct weston_pointer *pointer_state;
	struct weston_keyboard *keyboard_state;
	struct weston_touch *touch_state;
	int pointer_device_count;
	int keyboard_device_count;
	int touch_device_count;

	struct weston_output *output; /* constraint */

	struct wl_signal destroy_signal;
	struct wl_signal updated_caps_signal;

	struct weston_compositor *compositor;
	struct wl_list link;
	enum weston_keyboard_modifier modifier_state;
	struct weston_surface *saved_kbd_focus;
	struct wl_listener saved_kbd_focus_listener;
	struct wl_list drag_resource_list;

	uint32_t selection_serial;
	struct weston_data_source *selection_data_source;
	struct wl_listener selection_data_source_listener;
	struct wl_signal selection_signal;

	void (*led_update)(struct weston_seat *ws, enum weston_led leds);

	struct input_method *input_method;
	char *seat_name;
};

enum {
	WESTON_COMPOSITOR_ACTIVE,	/* normal rendering and events */
	WESTON_COMPOSITOR_IDLE,		/* shell->unlock called on activity */
	WESTON_COMPOSITOR_OFFSCREEN,	/* no rendering, no frame events */
	WESTON_COMPOSITOR_SLEEPING	/* same as offscreen, but also set dpms
                                         * to off */
};

struct weston_layer_entry {
	struct wl_list link;
	struct weston_layer *layer;
};

struct weston_layer {
	struct weston_layer_entry view_list;
	struct wl_list link;
	pixman_box32_t mask;
};

struct weston_plane {
	struct weston_compositor *compositor;
	pixman_region32_t damage; /**< in global coords */
	pixman_region32_t clip;
	int32_t x, y;
	struct wl_list link;
};

struct weston_renderer {
	int (*read_pixels)(struct weston_output *output,
			       pixman_format_code_t format, void *pixels,
			       uint32_t x, uint32_t y,
			       uint32_t width, uint32_t height);
	void (*repaint_output)(struct weston_output *output,
			       pixman_region32_t *output_damage);
	void (*flush_damage)(struct weston_surface *surface);
	void (*attach)(struct weston_surface *es, struct weston_buffer *buffer);
	void (*surface_set_color)(struct weston_surface *surface,
			       float red, float green,
			       float blue, float alpha);
	void (*destroy)(struct weston_compositor *ec);


	/** See weston_surface_get_content_size() */
	void (*surface_get_content_size)(struct weston_surface *surface,
					 int *width, int *height);

	/** See weston_surface_copy_content() */
	int (*surface_copy_content)(struct weston_surface *surface,
				    void *target, size_t size,
				    int src_x, int src_y,
				    int width, int height);

	/** See weston_compositor_import_dmabuf() */
	bool (*import_dmabuf)(struct weston_compositor *ec,
			      struct linux_dmabuf_buffer *buffer);
};

enum weston_capability {
	/* backend/renderer supports arbitrary rotation */
	WESTON_CAP_ROTATION_ANY			= 0x0001,

	/* screencaptures need to be y-flipped */
	WESTON_CAP_CAPTURE_YFLIP		= 0x0002,

	/* backend/renderer has a separate cursor plane */
	WESTON_CAP_CURSOR_PLANE			= 0x0004,

	/* backend supports setting arbitrary resolutions */
	WESTON_CAP_ARBITRARY_MODES		= 0x0008,

	/* renderer supports weston_view_set_mask() clipping */
	WESTON_CAP_VIEW_CLIP_MASK		= 0x0010,
};

/* Configuration struct for an output.
 *
 * This struct is used to pass the configuration for an output
 * to the compositor backend when creating a new output.
 * The backend can subclass this struct to handle backend
 * specific data.
 */
struct weston_backend_output_config {
	uint32_t transform;
	uint32_t width;
	uint32_t height;
	uint32_t scale;
};

/* Configuration struct for a backend.
 *
 * This struct carries the configuration for a backend, and it's
 * passed to the backend's init entry point. The backend will
 * likely want to subclass this in order to handle backend specific
 * data.
 *
 * NOTE: Alternate designs were proposed (Feb 2016) for using opaque
 * structures[1] and for section+key/value getter/setters[2].  The rationale
 * for selecting the transparent structure design is based on several
 * assumptions[3] which may require re-evaluating the design choice if they
 * fail to hold.
 *
 * 1: https://lists.freedesktop.org/archives/wayland-devel/2016-February/026989.html
 * 2: https://lists.freedesktop.org/archives/wayland-devel/2016-February/026929.html
 * 3: https://lists.freedesktop.org/archives/wayland-devel/2016-February/027228.html
 */
struct weston_backend_config {
   /** Major version for the backend-specific config struct
    *
    * This version must match exactly what the backend expects, otherwise
    * the struct is incompatible.
    */
   uint32_t struct_version;

   /** Minor version of the backend-specific config struct
    *
    * This must be set to sizeof(struct backend-specific config).
    * If the value here is smaller than what the backend expects, the
    * extra config members will assume their default values.
    *
    * A value greater than what the backend expects is incompatible.
    */
   size_t struct_size;
};

struct weston_backend {
	void (*destroy)(struct weston_compositor *compositor);
	void (*restore)(struct weston_compositor *compositor);
};

struct weston_desktop_xwayland;
struct weston_desktop_xwayland_interface;

struct weston_compositor {
	struct wl_signal destroy_signal;

	struct wl_display *wl_display;
	struct weston_desktop_xwayland *xwayland;
	const struct weston_desktop_xwayland_interface *xwayland_interface;

	/* surface signals */
	struct wl_signal create_surface_signal;
	struct wl_signal activate_signal;
	struct wl_signal transform_signal;

	struct wl_signal kill_signal;
	struct wl_signal idle_signal;
	struct wl_signal wake_signal;

	struct wl_signal show_input_panel_signal;
	struct wl_signal hide_input_panel_signal;
	struct wl_signal update_input_panel_signal;

	struct wl_signal seat_created_signal;
	struct wl_signal output_created_signal;
	struct wl_signal output_destroyed_signal;
	struct wl_signal output_moved_signal;
	struct wl_signal output_resized_signal; /* callback argument: resized output */

	struct wl_signal session_signal;
	int session_active;

	struct weston_layer fade_layer;
	struct weston_layer cursor_layer;

	struct wl_list output_list;
	struct wl_list seat_list;
	struct wl_list layer_list;
	struct wl_list view_list;	/* struct weston_view::link */
	struct wl_list plane_list;
	struct wl_list key_binding_list;
	struct wl_list modifier_binding_list;
	struct wl_list button_binding_list;
	struct wl_list touch_binding_list;
	struct wl_list axis_binding_list;
	struct wl_list debug_binding_list;

	uint32_t state;
	struct wl_event_source *idle_source;
	uint32_t idle_inhibit;
	int idle_time;			/* timeout, s */

	const struct weston_pointer_grab_interface *default_pointer_grab;

	/* Repaint state. */
	struct weston_plane primary_plane;
	uint32_t capabilities; /* combination of enum weston_capability */

	struct weston_renderer *renderer;

	pixman_format_code_t read_format;

	struct weston_backend *backend;

	struct weston_launcher *launcher;

	struct wl_list plugin_api_list; /* struct weston_plugin_api::link */

	uint32_t output_id_pool;

	struct xkb_rule_names xkb_names;
	struct xkb_context *xkb_context;
	struct weston_xkb_info *xkb_info;

	/* Raw keyboard processing (no libxkbcommon initialization or handling) */
	int use_xkbcommon;

	int32_t kb_repeat_rate;
	int32_t kb_repeat_delay;

	bool vt_switching;

	clockid_t presentation_clock;
	int32_t repaint_msec;

	unsigned int activate_serial;

	struct wl_global *pointer_constraints;

	int exit_code;

	void *user_data;
	void (*exit)(struct weston_compositor *c);
};

struct weston_buffer {
	struct wl_resource *resource;
	struct wl_signal destroy_signal;
	struct wl_listener destroy_listener;

	union {
		struct wl_shm_buffer *shm_buffer;
		void *legacy_buffer;
	};
	int32_t width, height;
	uint32_t busy_count;
	int y_inverted;
};

struct weston_buffer_reference {
	struct weston_buffer *buffer;
	struct wl_listener destroy_listener;
};

struct weston_buffer_viewport {
	struct {
		/* wl_surface.set_buffer_transform */
		uint32_t transform;

		/* wl_surface.set_scaling_factor */
		int32_t scale;

		/*
		 * If src_width != wl_fixed_from_int(-1),
		 * then and only then src_* are used.
		 */
		wl_fixed_t src_x, src_y;
		wl_fixed_t src_width, src_height;
	} buffer;

	struct {
		/*
		 * If width == -1, the size is inferred from the buffer.
		 */
		int32_t width, height;
	} surface;

	int changed;
};

struct weston_region {
	struct wl_resource *resource;
	pixman_region32_t region;
};

/* Using weston_view transformations
 *
 * To add a transformation to a view, create a struct weston_transform, and
 * add it to the list view->geometry.transformation_list. Whenever you
 * change the list, anything under view->geometry, or anything in the
 * weston_transforms linked into the list, you must call
 * weston_view_geometry_dirty().
 *
 * The order in the list defines the order of transformations. Let the list
 * contain the transformation matrices M1, ..., Mn as head to tail. The
 * transformation is applied to view-local coordinate vector p as
 *    P = Mn * ... * M2 * M1 * p
 * to produce the global coordinate vector P. The total transform
 *    Mn * ... * M2 * M1
 * is cached in view->transform.matrix, and the inverse of it in
 * view->transform.inverse.
 *
 * The list always contains view->transform.position transformation, which
 * is the translation by view->geometry.x and y.
 *
 * If you want to apply a transformation in local coordinates, add your
 * weston_transform to the head of the list. If you want to apply a
 * transformation in global coordinates, add it to the tail of the list.
 *
 * If view->geometry.parent is set, the total transformation of this
 * view will be the parent's total transformation and this transformation
 * combined:
 *    Mparent * Mn * ... * M2 * M1
 */

struct weston_view {
	struct weston_surface *surface;
	struct wl_list surface_link;
	struct wl_signal destroy_signal;

	struct wl_list link;             /* weston_compositor::view_list */
	struct weston_layer_entry layer_link; /* part of geometry */
	struct weston_plane *plane;

	/* For weston_layer inheritance from another view */
	struct weston_view *parent_view;

	unsigned int click_to_activate_serial;

	pixman_region32_t clip;          /* See weston_view_damage_below() */
	float alpha;                     /* part of geometry, see below */

	void *renderer_state;

	/* Surface geometry state, mutable.
	 * If you change anything, call weston_surface_geometry_dirty().
	 * That includes the transformations referenced from the list.
	 */
	struct {
		float x, y; /* surface translation on display */

		/* struct weston_transform */
		struct wl_list transformation_list;

		/* managed by weston_surface_set_transform_parent() */
		struct weston_view *parent;
		struct wl_listener parent_destroy_listener;
		struct wl_list child_list; /* geometry.parent_link */
		struct wl_list parent_link;

		/* managed by weston_view_set_mask() */
		bool scissor_enabled;
		pixman_region32_t scissor; /* always a simple rect */
	} geometry;

	/* State derived from geometry state, read-only.
	 * This is updated by weston_view_update_transform().
	 */
	struct {
		int dirty;

		/* Approximations in global coordinates:
		 * - boundingbox is guaranteed to include the whole view in
		 *   the smallest possible single rectangle.
		 * - opaque is guaranteed to be fully opaque, though not
		 *   necessarily include all opaque areas.
		 */
		pixman_region32_t boundingbox;
		pixman_region32_t opaque;

		/* matrix and inverse are used only if enabled = 1.
		 * If enabled = 0, use x, y, width, height directly.
		 */
		int enabled;
		struct weston_matrix matrix;
		struct weston_matrix inverse;

		struct weston_transform position; /* matrix from x, y */
	} transform;

	/*
	 * The primary output for this view.
	 * Used for picking the output for driving internal animations on the
	 * view, inheriting the primary output for related views in shells, etc.
	 */
	struct weston_output *output;

	/*
	 * A more complete representation of all outputs this surface is
	 * displayed on.
	 */
	uint32_t output_mask;

	/* Per-surface Presentation feedback flags, controlled by backend. */
	uint32_t psf_flags;

	bool is_mapped;
};

struct weston_surface_state {
	/* wl_surface.attach */
	int newly_attached;
	struct weston_buffer *buffer;
	struct wl_listener buffer_destroy_listener;
	int32_t sx;
	int32_t sy;

	/* wl_surface.damage */
	pixman_region32_t damage_surface;
	/* wl_surface.damage_buffer */
	pixman_region32_t damage_buffer;

	/* wl_surface.set_opaque_region */
	pixman_region32_t opaque;

	/* wl_surface.set_input_region */
	pixman_region32_t input;

	/* wl_surface.frame */
	struct wl_list frame_callback_list;

	/* presentation.feedback */
	struct wl_list feedback_list;

	/* wl_surface.set_buffer_transform */
	/* wl_surface.set_scaling_factor */
	/* wp_viewport.set_source */
	/* wp_viewport.set_destination */
	struct weston_buffer_viewport buffer_viewport;
};

struct weston_surface_activation_data {
	struct weston_surface *surface;
	struct weston_seat *seat;
};

struct weston_pointer_constraint {
	struct wl_list link;

	struct weston_surface *surface;
	struct weston_view *view;
	struct wl_resource *resource;
	struct weston_pointer_grab grab;
	struct weston_pointer *pointer;
	uint32_t lifetime;

	pixman_region32_t region;
	pixman_region32_t region_pending;
	bool region_is_pending;

	wl_fixed_t hint_x;
	wl_fixed_t hint_y;
	wl_fixed_t hint_x_pending;
	wl_fixed_t hint_y_pending;
	bool hint_is_pending;

	struct wl_listener pointer_destroy_listener;
	struct wl_listener surface_destroy_listener;
	struct wl_listener surface_commit_listener;
	struct wl_listener surface_activate_listener;
};

struct weston_surface {
	struct wl_resource *resource;
	struct wl_signal destroy_signal; /* callback argument: this surface */
	struct weston_compositor *compositor;
	struct wl_signal commit_signal;

	/** Damage in local coordinates from the client, for tex upload. */
	pixman_region32_t damage;

	pixman_region32_t opaque;        /* part of geometry, see below */
	pixman_region32_t input;
	int32_t width, height;
	int32_t ref_count;

	/* Not for long-term storage.  This exists for book-keeping while
	 * iterating over surfaces and views
	 */
	bool touched;

	void *renderer_state;

	struct wl_list views;

	/*
	 * Which output to vsync this surface to.
	 * Used to determine whether to send or queue frame events, and for
	 * other client-visible syncing/throttling tied to the output
	 * repaint cycle.
	 */
	struct weston_output *output;

	/*
	 * A more complete representation of all outputs this surface is
	 * displayed on.
	 */
	uint32_t output_mask;

	struct wl_list frame_callback_list;
	struct wl_list feedback_list;

	struct weston_buffer_reference buffer_ref;
	struct weston_buffer_viewport buffer_viewport;
	int32_t width_from_buffer; /* before applying viewport */
	int32_t height_from_buffer;
	bool keep_buffer; /* for backends to prevent early release */

	/* wp_viewport resource for this surface */
	struct wl_resource *viewport_resource;

	/* All the pending state, that wl_surface.commit will apply. */
	struct weston_surface_state pending;

	/* Matrices representating of the full transformation between
	 * buffer and surface coordinates.  These matrices are updated
	 * using the weston_surface_build_buffer_matrix function. */
	struct weston_matrix buffer_to_surface_matrix;
	struct weston_matrix surface_to_buffer_matrix;

	/*
	 * If non-NULL, this function will be called on
	 * wl_surface::commit after a new buffer has been set up for
	 * this surface. The integer params are the sx and sy
	 * parameters supplied to wl_surface::attach.
	 */
	void (*committed)(struct weston_surface *es, int32_t sx, int32_t sy);
	void *committed_private;
	int (*get_label)(struct weston_surface *surface, char *buf, size_t len);

	/* Parent's list of its sub-surfaces, weston_subsurface:parent_link.
	 * Contains also the parent itself as a dummy weston_subsurface,
	 * if the list is not empty.
	 */
	struct wl_list subsurface_list; /* weston_subsurface::parent_link */
	struct wl_list subsurface_list_pending; /* ...::parent_link_pending */

	/*
	 * For tracking protocol role assignments. Different roles may
	 * have the same configure hook, e.g. in shell.c. Configure hook
	 * may get reset, this will not.
	 * XXX: map configure functions 1:1 to roles, and never reset it,
	 * and replace role_name with configure.
	 */
	const char *role_name;

	struct weston_timeline_object timeline;

	bool is_mapped;

	/* An list of per seat pointer constraints. */
	struct wl_list pointer_constraints;
};

struct weston_subsurface {
	struct wl_resource *resource;

	/* guaranteed to be valid and non-NULL */
	struct weston_surface *surface;
	struct wl_listener surface_destroy_listener;

	/* can be NULL */
	struct weston_surface *parent;
	struct wl_listener parent_destroy_listener;
	struct wl_list parent_link;
	struct wl_list parent_link_pending;

	struct {
		int32_t x;
		int32_t y;
		int set;
	} position;

	int has_cached_data;
	struct weston_surface_state cached;
	struct weston_buffer_reference cached_buffer_ref;

	int synchronized;

	/* Used for constructing the view tree */
	struct wl_list unused_views;
};

enum weston_key_state_update {
	STATE_UPDATE_AUTOMATIC,
	STATE_UPDATE_NONE,
};

enum weston_activate_flag {
	WESTON_ACTIVATE_FLAG_NONE = 0,
	WESTON_ACTIVATE_FLAG_CONFIGURE = 1 << 0,
	WESTON_ACTIVATE_FLAG_CLICKED = 1 << 1,
};

void
weston_version(int *major, int *minor, int *micro);

void
weston_view_update_transform(struct weston_view *view);

void
weston_view_geometry_dirty(struct weston_view *view);

void
weston_view_to_global_fixed(struct weston_view *view,
			    wl_fixed_t sx, wl_fixed_t sy,
			    wl_fixed_t *x, wl_fixed_t *y);
void
weston_view_to_global_float(struct weston_view *view,
			    float sx, float sy, float *x, float *y);

void
weston_view_from_global_float(struct weston_view *view,
			      float x, float y, float *vx, float *vy);
void
weston_view_from_global(struct weston_view *view,
			int32_t x, int32_t y, int32_t *vx, int32_t *vy);
void
weston_view_from_global_fixed(struct weston_view *view,
			      wl_fixed_t x, wl_fixed_t y,
			      wl_fixed_t *vx, wl_fixed_t *vy);

void
weston_surface_to_buffer_float(struct weston_surface *surface,
			       float x, float y, float *bx, float *by);
pixman_box32_t
weston_surface_to_buffer_rect(struct weston_surface *surface,
			      pixman_box32_t rect);

void
weston_surface_to_buffer_region(struct weston_surface *surface,
				pixman_region32_t *surface_region,
				pixman_region32_t *buffer_region);

void
weston_spring_init(struct weston_spring *spring,
		   double k, double current, double target);
void
weston_spring_update(struct weston_spring *spring, uint32_t msec);
int
weston_spring_done(struct weston_spring *spring);

void
weston_view_activate(struct weston_view *view,
		     struct weston_seat *seat,
		     uint32_t flags);

void
notify_motion(struct weston_seat *seat, uint32_t time,
	      struct weston_pointer_motion_event *event);
void
notify_motion_absolute(struct weston_seat *seat, uint32_t time,
		       double x, double y);
void
notify_button(struct weston_seat *seat, uint32_t time, int32_t button,
	      enum wl_pointer_button_state state);
void
notify_axis(struct weston_seat *seat, uint32_t time,
	    struct weston_pointer_axis_event *event);
void
notify_axis_source(struct weston_seat *seat, uint32_t source);

void
notify_pointer_frame(struct weston_seat *seat);

void
notify_key(struct weston_seat *seat, uint32_t time, uint32_t key,
	   enum wl_keyboard_key_state state,
	   enum weston_key_state_update update_state);
void
notify_modifiers(struct weston_seat *seat, uint32_t serial);

void
notify_pointer_focus(struct weston_seat *seat, struct weston_output *output,
		     double x, double y);

void
notify_keyboard_focus_in(struct weston_seat *seat, struct wl_array *keys,
			 enum weston_key_state_update update_state);
void
notify_keyboard_focus_out(struct weston_seat *seat);

void
notify_touch(struct weston_seat *seat, uint32_t time, int touch_id,
	     double x, double y, int touch_type);
void
notify_touch_frame(struct weston_seat *seat);

void
notify_touch_cancel(struct weston_seat *seat);

void
weston_layer_entry_insert(struct weston_layer_entry *list,
			  struct weston_layer_entry *entry);
void
weston_layer_entry_remove(struct weston_layer_entry *entry);
void
weston_layer_init(struct weston_layer *layer, struct wl_list *below);

void
weston_layer_set_mask(struct weston_layer *layer, int x, int y, int width, int height);

void
weston_layer_set_mask_infinite(struct weston_layer *layer);

void
weston_plane_init(struct weston_plane *plane,
			struct weston_compositor *ec,
			int32_t x, int32_t y);
void
weston_plane_release(struct weston_plane *plane);

void
weston_compositor_stack_plane(struct weston_compositor *ec,
			      struct weston_plane *plane,
			      struct weston_plane *above);

/* An invalid flag in presented_flags to catch logic errors. */
#define WP_PRESENTATION_FEEDBACK_INVALID (1U << 31)

void
weston_output_finish_frame(struct weston_output *output,
			   const struct timespec *stamp,
			   uint32_t presented_flags);
void
weston_output_schedule_repaint(struct weston_output *output);
void
weston_output_damage(struct weston_output *output);
void
weston_compositor_schedule_repaint(struct weston_compositor *compositor);
void
weston_compositor_fade(struct weston_compositor *compositor, float tint);
void
weston_compositor_damage_all(struct weston_compositor *compositor);
void
weston_compositor_unlock(struct weston_compositor *compositor);
void
weston_compositor_wake(struct weston_compositor *compositor);
void
weston_compositor_offscreen(struct weston_compositor *compositor);
void
weston_compositor_sleep(struct weston_compositor *compositor);
struct weston_view *
weston_compositor_pick_view(struct weston_compositor *compositor,
			    wl_fixed_t x, wl_fixed_t y,
			    wl_fixed_t *sx, wl_fixed_t *sy);


struct weston_binding;
typedef void (*weston_key_binding_handler_t)(struct weston_keyboard *keyboard,
					     uint32_t time, uint32_t key,
					     void *data);
struct weston_binding *
weston_compositor_add_key_binding(struct weston_compositor *compositor,
				  uint32_t key,
				  enum weston_keyboard_modifier modifier,
				  weston_key_binding_handler_t binding,
				  void *data);

typedef void (*weston_modifier_binding_handler_t)(struct weston_keyboard *keyboard,
					          enum weston_keyboard_modifier modifier,
					          void *data);
struct weston_binding *
weston_compositor_add_modifier_binding(struct weston_compositor *compositor,
				       enum weston_keyboard_modifier modifier,
				       weston_modifier_binding_handler_t binding,
				       void *data);

typedef void (*weston_button_binding_handler_t)(struct weston_pointer *pointer,
						uint32_t time, uint32_t button,
						void *data);
struct weston_binding *
weston_compositor_add_button_binding(struct weston_compositor *compositor,
				     uint32_t button,
				     enum weston_keyboard_modifier modifier,
				     weston_button_binding_handler_t binding,
				     void *data);

typedef void (*weston_touch_binding_handler_t)(struct weston_touch *touch,
					       uint32_t time,
					       void *data);
struct weston_binding *
weston_compositor_add_touch_binding(struct weston_compositor *compositor,
				    enum weston_keyboard_modifier modifier,
				    weston_touch_binding_handler_t binding,
				    void *data);

typedef void (*weston_axis_binding_handler_t)(struct weston_pointer *pointer,
					      uint32_t time,
					      struct weston_pointer_axis_event *event,
					      void *data);
struct weston_binding *
weston_compositor_add_axis_binding(struct weston_compositor *compositor,
			           uint32_t axis,
			           enum weston_keyboard_modifier modifier,
			           weston_axis_binding_handler_t binding,
				   void *data);
struct weston_binding *
weston_compositor_add_debug_binding(struct weston_compositor *compositor,
				    uint32_t key,
				    weston_key_binding_handler_t binding,
				    void *data);
void
weston_binding_destroy(struct weston_binding *binding);

void
weston_install_debug_key_binding(struct weston_compositor *compositor,
				 uint32_t mod);

void
weston_binding_list_destroy_all(struct wl_list *list);

void
weston_compositor_run_key_binding(struct weston_compositor *compositor,
				  struct weston_keyboard *keyboard,
				  uint32_t time,
				  uint32_t key,
				  enum wl_keyboard_key_state state);

void
weston_compositor_run_modifier_binding(struct weston_compositor *compositor,
				       struct weston_keyboard *keyboard,
				       enum weston_keyboard_modifier modifier,
				       enum wl_keyboard_key_state state);
void
weston_compositor_run_button_binding(struct weston_compositor *compositor,
				     struct weston_pointer *pointer, uint32_t time,
				     uint32_t button,
				     enum wl_pointer_button_state value);
void
weston_compositor_run_touch_binding(struct weston_compositor *compositor,
				    struct weston_touch *touch, uint32_t time,
				    int touch_type);
int
weston_compositor_run_axis_binding(struct weston_compositor *compositor,
				   struct weston_pointer *pointer, uint32_t time,
				   struct weston_pointer_axis_event *event);
int
weston_compositor_run_debug_binding(struct weston_compositor *compositor,
				    struct weston_keyboard *keyboard, uint32_t time,
				    uint32_t key,
				    enum wl_keyboard_key_state state);

void
weston_compositor_set_default_pointer_grab(struct weston_compositor *compositor,
			const struct weston_pointer_grab_interface *interface);

int
weston_environment_get_fd(const char *env);

struct wl_list *
weston_compositor_top(struct weston_compositor *compositor);

struct weston_surface *
weston_surface_create(struct weston_compositor *compositor);

struct weston_view *
weston_view_create(struct weston_surface *surface);

void
weston_view_destroy(struct weston_view *view);

void
weston_view_set_position(struct weston_view *view,
			 float x, float y);

void
weston_view_set_transform_parent(struct weston_view *view,
				 struct weston_view *parent);

void
weston_view_set_mask(struct weston_view *view,
		     int x, int y, int width, int height);

void
weston_view_set_mask_infinite(struct weston_view *view);

bool
weston_view_is_mapped(struct weston_view *view);

void
weston_view_schedule_repaint(struct weston_view *view);

bool
weston_surface_is_mapped(struct weston_surface *surface);

void
weston_surface_set_size(struct weston_surface *surface,
			int32_t width, int32_t height);

void
weston_surface_schedule_repaint(struct weston_surface *surface);

void
weston_surface_damage(struct weston_surface *surface);

void
weston_view_damage_below(struct weston_view *view);

void
weston_view_move_to_plane(struct weston_view *view,
			  struct weston_plane *plane);
void
weston_view_unmap(struct weston_view *view);

void
weston_surface_unmap(struct weston_surface *surface);

struct weston_surface *
weston_surface_get_main_surface(struct weston_surface *surface);

int
weston_surface_set_role(struct weston_surface *surface,
			const char *role_name,
			struct wl_resource *error_resource,
			uint32_t error_code);
const char *
weston_surface_get_role(struct weston_surface *surface);

void
weston_surface_set_label_func(struct weston_surface *surface,
			      int (*desc)(struct weston_surface *,
					  char *, size_t));

void
weston_surface_get_content_size(struct weston_surface *surface,
				int *width, int *height);

struct weston_geometry
weston_surface_get_bounding_box(struct weston_surface *surface);

int
weston_surface_copy_content(struct weston_surface *surface,
			    void *target, size_t size,
			    int src_x, int src_y,
			    int width, int height);

struct weston_buffer *
weston_buffer_from_resource(struct wl_resource *resource);

void
weston_buffer_reference(struct weston_buffer_reference *ref,
			struct weston_buffer *buffer);

uint32_t
weston_compositor_get_time(void);

void
weston_compositor_destroy(struct weston_compositor *ec);
struct weston_compositor *
weston_compositor_create(struct wl_display *display, void *user_data);

enum weston_compositor_backend {
	WESTON_BACKEND_DRM,
	WESTON_BACKEND_FBDEV,
	WESTON_BACKEND_HEADLESS,
	WESTON_BACKEND_RDP,
	WESTON_BACKEND_WAYLAND,
	WESTON_BACKEND_X11,
};

int
weston_compositor_load_backend(struct weston_compositor *compositor,
			       enum weston_compositor_backend backend,
			       struct weston_backend_config *config_base);
void
weston_compositor_exit(struct weston_compositor *ec);
void *
weston_compositor_get_user_data(struct weston_compositor *compositor);
int
weston_compositor_set_presentation_clock(struct weston_compositor *compositor,
					 clockid_t clk_id);
int
weston_compositor_set_presentation_clock_software(
					struct weston_compositor *compositor);
void
weston_compositor_read_presentation_clock(
			const struct weston_compositor *compositor,
			struct timespec *ts);

bool
weston_compositor_import_dmabuf(struct weston_compositor *compositor,
				struct linux_dmabuf_buffer *buffer);

void
weston_compositor_shutdown(struct weston_compositor *ec);
void
weston_compositor_exit_with_code(struct weston_compositor *compositor,
				 int exit_code);
void
weston_output_init_zoom(struct weston_output *output);
void
weston_output_update_zoom(struct weston_output *output);
void
weston_output_activate_zoom(struct weston_output *output,
			    struct weston_seat *seat);
void
weston_output_update_matrix(struct weston_output *output);
void
weston_output_move(struct weston_output *output, int x, int y);
void
weston_output_init(struct weston_output *output, struct weston_compositor *c,
		   int x, int y, int width, int height, uint32_t transform, int32_t scale);
void
weston_compositor_add_output(struct weston_compositor *compositor,
                             struct weston_output *output);
void
weston_output_destroy(struct weston_output *output);
void
weston_output_transform_coordinate(struct weston_output *output,
				   double device_x, double device_y,
				   double *x, double *y);

void
weston_seat_init(struct weston_seat *seat, struct weston_compositor *ec,
		 const char *seat_name);
void
weston_seat_init_pointer(struct weston_seat *seat);
void
weston_seat_release_pointer(struct weston_seat *seat);
int
weston_seat_init_keyboard(struct weston_seat *seat, struct xkb_keymap *keymap);
void
weston_seat_release_keyboard(struct weston_seat *seat);
void
weston_seat_init_touch(struct weston_seat *seat);
void
weston_seat_release_touch(struct weston_seat *seat);
void
weston_seat_repick(struct weston_seat *seat);
void
weston_seat_update_keymap(struct weston_seat *seat, struct xkb_keymap *keymap);

void
weston_seat_release(struct weston_seat *seat);
int
weston_compositor_set_xkb_rule_names(struct weston_compositor *ec,
				     struct xkb_rule_names *names);
void
weston_compositor_xkb_destroy(struct weston_compositor *ec);

/* String literal of spaces, the same width as the timestamp. */
#define STAMP_SPACE "               "

typedef int (*log_func_t)(const char *fmt, va_list ap);
void
weston_log_set_handler(log_func_t log, log_func_t cont);
int
weston_vlog(const char *fmt, va_list ap);
int
weston_vlog_continue(const char *fmt, va_list ap);
int
weston_log(const char *fmt, ...)
	__attribute__ ((format (printf, 1, 2)));
int
weston_log_continue(const char *fmt, ...)
	__attribute__ ((format (printf, 1, 2)));

enum {
	TTY_ENTER_VT,
	TTY_LEAVE_VT
};

struct tty *
tty_create(struct weston_compositor *compositor, int tty_nr);

void
tty_destroy(struct tty *tty);

void
tty_reset(struct tty *tty);

int
tty_activate_vt(struct tty *tty, int vt);

enum weston_screenshooter_outcome {
	WESTON_SCREENSHOOTER_SUCCESS,
	WESTON_SCREENSHOOTER_NO_MEMORY,
	WESTON_SCREENSHOOTER_BAD_BUFFER
};

typedef void (*weston_screenshooter_done_func_t)(void *data,
				enum weston_screenshooter_outcome outcome);
int
weston_screenshooter_shoot(struct weston_output *output, struct weston_buffer *buffer,
			   weston_screenshooter_done_func_t done, void *data);
struct weston_recorder *
weston_recorder_start(struct weston_output *output, const char *filename);
void
weston_recorder_stop(struct weston_recorder *recorder);

struct clipboard *
clipboard_create(struct weston_seat *seat);

struct text_backend;

struct text_backend *
text_backend_init(struct weston_compositor *ec);

void
text_backend_destroy(struct text_backend *text_backend);

struct weston_view_animation;
typedef	void (*weston_view_animation_done_func_t)(struct weston_view_animation *animation, void *data);

void
weston_view_animation_destroy(struct weston_view_animation *animation);

struct weston_view_animation *
weston_zoom_run(struct weston_view *view, float start, float stop,
		weston_view_animation_done_func_t done, void *data);

struct weston_view_animation *
weston_fade_run(struct weston_view *view,
		float start, float end, float k,
		weston_view_animation_done_func_t done, void *data);

struct weston_view_animation *
weston_move_scale_run(struct weston_view *view, int dx, int dy,
		      float start, float end, int reverse,
		      weston_view_animation_done_func_t done, void *data);

void
weston_fade_update(struct weston_view_animation *fade, float target);

struct weston_view_animation *
weston_stable_fade_run(struct weston_view *front_view, float start,
		       struct weston_view *back_view, float end,
		       weston_view_animation_done_func_t done, void *data);

struct weston_view_animation *
weston_slide_run(struct weston_view *view, float start, float stop,
		 weston_view_animation_done_func_t done, void *data);

void
weston_surface_set_color(struct weston_surface *surface,
			 float red, float green, float blue, float alpha);

void
weston_surface_destroy(struct weston_surface *surface);

int
weston_output_mode_set_native(struct weston_output *output,
			      struct weston_mode *mode,
			      int32_t scale);
int
weston_output_mode_switch_to_temporary(struct weston_output *output,
				       struct weston_mode *mode,
				       int32_t scale);
int
weston_output_mode_switch_to_native(struct weston_output *output);

int
noop_renderer_init(struct weston_compositor *ec);

int
weston_input_init(struct weston_compositor *compositor);

int
backend_init(struct weston_compositor *c,
	     struct weston_backend_config *config_base);
int
module_init(struct weston_compositor *compositor,
	    int *argc, char *argv[]);

void
weston_transformed_coord(int width, int height,
			 enum wl_output_transform transform,
			 int32_t scale,
			 float sx, float sy, float *bx, float *by);
pixman_box32_t
weston_transformed_rect(int width, int height,
			enum wl_output_transform transform,
			int32_t scale,
			pixman_box32_t rect);
void
weston_matrix_transform_region(pixman_region32_t *dest,
                               struct weston_matrix *matrix,
                               pixman_region32_t *src);
void
weston_transformed_region(int width, int height,
			  enum wl_output_transform transform,
			  int32_t scale,
			  pixman_region32_t *src, pixman_region32_t *dest);

void *
weston_load_module(const char *name, const char *entrypoint);

int
weston_parse_transform(const char *transform, uint32_t *out);

const char *
weston_transform_to_string(uint32_t output_transform);

struct weston_keyboard *
weston_seat_get_keyboard(struct weston_seat *seat);

struct weston_pointer *
weston_seat_get_pointer(struct weston_seat *seat);

struct weston_touch *
weston_seat_get_touch(struct weston_seat *seat);

void
weston_seat_set_keyboard_focus(struct weston_seat *seat,
			       struct weston_surface *surface);

int
weston_compositor_load_xwayland(struct weston_compositor *compositor);

#ifdef  __cplusplus
}
#endif

#endif
