# 18 "parsing/lexer.mll"
 
open Lexing
open Misc
open Parser

type error =
  | Illegal_character of char
  | Illegal_escape of string
  | Unterminated_comment of Location.t
  | Unterminated_string
  | Unterminated_string_in_comment of Location.t * Location.t
  | Keyword_as_label of string
  | Invalid_literal of string
;;

exception Error of error * Location.t;;

(* The table of keywords *)

let keyword_table =
  create_hashtable 149 [
    "and", AND;
    "as", AS;
    "assert", ASSERT;
    "begin", BEGIN;
    "class", CLASS;
    "constraint", CONSTRAINT;
    "do", DO;
    "done", DONE;
    "downto", DOWNTO;
    "else", ELSE;
    "end", END;
    "exception", EXCEPTION;
    "external", EXTERNAL;
    "false", FALSE;
    "for", FOR;
    "fun", FUN;
    "function", FUNCTION;
    "functor", FUNCTOR;
    "if", IF;
    "in", IN;
    "include", INCLUDE;
    "inherit", INHERIT;
    "initializer", INITIALIZER;
    "lazy", LAZY;
    "let", LET;
    "match", MATCH;
    "method", METHOD;
    "module", MODULE;
    "mutable", MUTABLE;
    "new", NEW;
    "nonrec", NONREC;
    "object", OBJECT;
    "of", OF;
    "open", OPEN;
    "or", OR;
(*  "parser", PARSER; *)
    "private", PRIVATE;
    "rec", REC;
    "sig", SIG;
    "struct", STRUCT;
    "then", THEN;
    "to", TO;
    "true", TRUE;
    "try", TRY;
    "type", TYPE;
    "val", VAL;
    "virtual", VIRTUAL;
    "when", WHEN;
    "while", WHILE;
    "with", WITH;

    "lor", INFIXOP3("lor"); (* Should be INFIXOP2 *)
    "lxor", INFIXOP3("lxor"); (* Should be INFIXOP2 *)
    "mod", INFIXOP3("mod");
    "land", INFIXOP3("land");
    "lsl", INFIXOP4("lsl");
    "lsr", INFIXOP4("lsr");
    "asr", INFIXOP4("asr")
]

(* To buffer string literals *)

let initial_string_buffer = Bytes.create 256
let string_buff = ref initial_string_buffer
let string_index = ref 0

let reset_string_buffer () =
  string_buff := initial_string_buffer;
  string_index := 0

let store_string_char c =
  if !string_index >= Bytes.length !string_buff then begin
    let new_buff = Bytes.create (Bytes.length (!string_buff) * 2) in
    Bytes.blit !string_buff 0 new_buff 0 (Bytes.length !string_buff);
    string_buff := new_buff
  end;
  Bytes.unsafe_set !string_buff !string_index c;
  incr string_index

let store_string s =
  for i = 0 to String.length s - 1 do
    store_string_char s.[i];
  done

let store_lexeme lexbuf =
  store_string (Lexing.lexeme lexbuf)

let get_stored_string () =
  let s = Bytes.sub_string !string_buff 0 !string_index in
  string_buff := initial_string_buffer;
  s

(* To store the position of the beginning of a string and comment *)
let string_start_loc = ref Location.none;;
let comment_start_loc = ref [];;
let in_comment () = !comment_start_loc <> [];;
let is_in_string = ref false
let in_string () = !is_in_string
let print_warnings = ref true

(* Escaped chars are interpreted in strings unless they are in comments. *)
let store_escaped_char lexbuf c =
  if in_comment () then store_lexeme lexbuf else store_string_char c

let with_comment_buffer comment lexbuf =
  let start_loc = Location.curr lexbuf  in
  comment_start_loc := [start_loc];
  reset_string_buffer ();
  let end_loc = comment lexbuf in
  let s = get_stored_string () in
  reset_string_buffer ();
  let loc = { start_loc with Location.loc_end = end_loc.Location.loc_end } in
  s, loc

(* To translate escape sequences *)

let char_for_backslash = function
  | 'n' -> '\010'
  | 'r' -> '\013'
  | 'b' -> '\008'
  | 't' -> '\009'
  | c   -> c

let char_for_decimal_code lexbuf i =
  let c = 100 * (Char.code(Lexing.lexeme_char lexbuf i) - 48) +
           10 * (Char.code(Lexing.lexeme_char lexbuf (i+1)) - 48) +
                (Char.code(Lexing.lexeme_char lexbuf (i+2)) - 48) in
  if (c < 0 || c > 255) then
    if in_comment ()
    then 'x'
    else raise (Error(Illegal_escape (Lexing.lexeme lexbuf),
                      Location.curr lexbuf))
  else Char.chr c

let char_for_octal_code lexbuf i =
  let c = 64 * (Char.code(Lexing.lexeme_char lexbuf i) - 48) +
           8 * (Char.code(Lexing.lexeme_char lexbuf (i+1)) - 48) +
               (Char.code(Lexing.lexeme_char lexbuf (i+2)) - 48) in
  Char.chr c

let char_for_hexadecimal_code lexbuf i =
  let d1 = Char.code (Lexing.lexeme_char lexbuf i) in
  let val1 = if d1 >= 97 then d1 - 87
             else if d1 >= 65 then d1 - 55
             else d1 - 48
  in
  let d2 = Char.code (Lexing.lexeme_char lexbuf (i+1)) in
  let val2 = if d2 >= 97 then d2 - 87
             else if d2 >= 65 then d2 - 55
             else d2 - 48
  in
  Char.chr (val1 * 16 + val2)

(* recover the name from a LABEL or OPTLABEL token *)

let get_label_name lexbuf =
  let s = Lexing.lexeme lexbuf in
  let name = String.sub s 1 (String.length s - 2) in
  if Hashtbl.mem keyword_table name then
    raise (Error(Keyword_as_label name, Location.curr lexbuf));
  name
;;

(* Update the current location with file name and line number. *)

let update_loc lexbuf file line absolute chars =
  let pos = lexbuf.lex_curr_p in
  let new_file = match file with
                 | None -> pos.pos_fname
                 | Some s -> s
  in
  lexbuf.lex_curr_p <- { pos with
    pos_fname = new_file;
    pos_lnum = if absolute then line else pos.pos_lnum + line;
    pos_bol = pos.pos_cnum - chars;
  }
;;

let preprocessor = ref None

let escaped_newlines = ref false

(* Warn about Latin-1 characters used in idents *)

let warn_latin1 lexbuf =
  Location.prerr_warning (Location.curr lexbuf)
    (Warnings.Deprecated "ISO-Latin1 characters in identifiers")
;;

let handle_docstrings = ref true
let comment_list = ref []

let add_comment com =
  comment_list := com :: !comment_list

let add_docstring_comment ds =
  let com =
    ("*" ^ Docstrings.docstring_body ds, Docstrings.docstring_loc ds)
  in
    add_comment com

let comments () = List.rev !comment_list

(* Error report *)

open Format

let report_error ppf = function
  | Illegal_character c ->
      fprintf ppf "Illegal character (%s)" (Char.escaped c)
  | Illegal_escape s ->
      fprintf ppf "Illegal backslash escape in string or character (%s)" s
  | Unterminated_comment _ ->
      fprintf ppf "Comment not terminated"
  | Unterminated_string ->
      fprintf ppf "String literal not terminated"
  | Unterminated_string_in_comment (_, loc) ->
      fprintf ppf "This comment contains an unterminated string literal@.\
                   %aString literal begins here"
              Location.print_error loc
  | Keyword_as_label kwd ->
      fprintf ppf "`%s' is a keyword, it cannot be used as label name" kwd
  | Invalid_literal s ->
      fprintf ppf "Invalid literal %s" s

let () =
  Location.register_error_of_exn
    (function
      | Error (err, loc) ->
          Some (Location.error_of_printer loc report_error err)
      | _ ->
          None
    )


# 259 "parsing/lexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base = 
   "\000\000\163\255\164\255\224\000\003\001\038\001\073\001\108\001\
    \143\001\187\255\178\001\215\001\195\255\091\000\252\001\031\002\
    \068\000\071\000\084\000\066\002\212\255\214\255\217\255\101\002\
    \196\002\231\002\089\000\255\000\005\003\236\255\089\003\173\003\
    \001\004\209\004\161\005\113\006\080\007\172\007\124\008\125\000\
    \254\255\001\000\005\000\255\255\006\000\007\000\091\009\121\009\
    \073\010\250\255\249\255\025\011\233\011\247\255\246\255\185\012\
    \013\013\097\013\181\013\009\014\093\014\177\014\005\015\089\015\
    \173\015\001\016\090\000\085\016\169\016\253\016\081\017\165\017\
    \110\000\193\255\235\255\229\002\000\018\096\000\097\000\011\000\
    \234\255\233\255\228\255\118\002\120\000\124\000\109\000\232\255\
    \134\000\110\000\231\255\224\000\003\001\111\000\230\255\076\004\
    \113\000\229\255\217\000\224\255\222\000\223\255\239\000\000\018\
    \222\255\039\018\067\005\231\002\221\255\012\000\024\001\080\001\
    \115\001\150\001\221\255\013\000\085\018\124\018\159\018\197\018\
    \209\255\205\255\206\255\207\255\203\255\232\018\117\000\092\000\
    \196\255\197\255\198\255\183\000\183\255\181\255\190\255\011\019\
    \186\255\188\255\046\019\081\019\116\019\151\019\096\005\243\255\
    \244\255\017\000\245\255\062\002\189\005\253\255\242\000\247\000\
    \255\255\254\255\252\255\227\005\169\019\252\000\003\001\018\000\
    \251\255\250\255\249\255\188\006\248\002\004\001\248\255\002\003\
    \005\001\247\255\032\008\008\001\246\255\059\001\199\001\246\255\
    \247\255\248\255\060\001\242\019\255\255\249\255\199\008\038\001\
    \099\004\253\255\050\001\094\001\095\001\109\004\252\255\158\002\
    \249\003\251\255\196\009\250\255\113\001\181\001\252\255\051\007\
    \254\255\255\255\129\001\130\001\253\255\143\007\025\001\035\001\
    \107\001\142\001\044\001\149\001\043\001\019\000\255\255";
  Lexing.lex_backtrk = 
   "\255\255\255\255\255\255\089\000\088\000\085\000\084\000\077\000\
    \075\000\255\255\066\000\063\000\255\255\056\000\055\000\053\000\
    \051\000\047\000\045\000\080\000\255\255\255\255\255\255\036\000\
    \035\000\042\000\040\000\039\000\061\000\255\255\014\000\014\000\
    \013\000\012\000\011\000\010\000\007\000\004\000\003\000\002\000\
    \255\255\092\000\092\000\255\255\255\255\255\255\083\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\018\000\
    \018\000\016\000\015\000\018\000\015\000\015\000\014\000\016\000\
    \015\000\016\000\255\255\017\000\017\000\014\000\014\000\016\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\027\000\027\000\027\000\027\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\028\000\255\255\029\000\255\255\030\000\087\000\
    \255\255\090\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\037\000\086\000\081\000\044\000\
    \255\255\255\255\255\255\255\255\255\255\054\000\073\000\070\000\
    \255\255\255\255\255\255\071\000\255\255\255\255\255\255\064\000\
    \255\255\255\255\082\000\076\000\079\000\078\000\255\255\255\255\
    \255\255\012\000\255\255\012\000\012\000\255\255\012\000\012\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\009\000\009\000\255\255\255\255\006\000\006\000\
    \006\000\255\255\001\000\006\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\003\000\
    \255\255\255\255\003\000\255\255\255\255\255\255\002\000\255\255\
    \255\255\001\000\255\255\255\255\255\255\255\255\255\255";
  Lexing.lex_default = 
   "\001\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\000\000\000\000\255\255\
    \255\255\255\255\255\255\077\000\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\000\000\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\255\255\255\255\000\000\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\255\255\082\000\255\255\255\255\255\255\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\000\000\255\255\255\255\255\255\000\000\255\255\
    \255\255\000\000\255\255\000\000\255\255\000\000\255\255\255\255\
    \000\000\255\255\110\000\255\255\000\000\255\255\110\000\111\000\
    \110\000\113\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\000\000\000\000\255\255\255\255\255\255\
    \000\000\000\000\000\000\255\255\000\000\000\000\000\000\255\255\
    \000\000\000\000\255\255\255\255\255\255\255\255\143\000\000\000\
    \000\000\255\255\000\000\157\000\255\255\000\000\255\255\255\255\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\255\255\255\255\255\255\000\000\255\255\
    \255\255\000\000\255\255\255\255\000\000\255\255\175\000\000\000\
    \000\000\000\000\255\255\181\000\000\000\000\000\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\255\255\000\000\255\255\
    \255\255\000\000\255\255\000\000\255\255\198\000\000\000\255\255\
    \000\000\000\000\255\255\255\255\000\000\255\255\255\255\255\255\
    \208\000\211\000\255\255\211\000\255\255\255\255\000\000";
  Lexing.lex_trans = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\039\000\040\000\040\000\039\000\041\000\045\000\043\000\
    \043\000\040\000\044\000\044\000\045\000\078\000\108\000\114\000\
    \079\000\109\000\115\000\144\000\158\000\214\000\173\000\159\000\
    \039\000\008\000\029\000\024\000\006\000\004\000\023\000\027\000\
    \026\000\021\000\025\000\007\000\020\000\019\000\018\000\003\000\
    \031\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\017\000\016\000\015\000\014\000\010\000\036\000\
    \005\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\013\000\042\000\012\000\005\000\038\000\
    \022\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\028\000\011\000\009\000\037\000\124\000\
    \126\000\123\000\120\000\098\000\122\000\121\000\039\000\081\000\
    \080\000\039\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\087\000\090\000\094\000\129\000\
    \097\000\128\000\133\000\127\000\131\000\039\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \091\000\091\000\091\000\091\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\130\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\132\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \002\000\003\000\099\000\100\000\003\000\003\000\003\000\101\000\
    \102\000\078\000\003\000\003\000\079\000\003\000\003\000\003\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \101\000\102\000\003\000\153\000\003\000\003\000\003\000\003\000\
    \003\000\152\000\108\000\162\000\003\000\109\000\255\255\003\000\
    \003\000\003\000\161\000\166\000\169\000\003\000\003\000\172\000\
    \003\000\003\000\003\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\186\000\207\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\208\000\144\000\177\000\005\000\
    \173\000\196\000\005\000\005\000\005\000\212\000\213\000\000\000\
    \005\000\005\000\186\000\005\000\005\000\005\000\191\000\191\000\
    \191\000\191\000\108\000\076\000\003\000\109\000\003\000\000\000\
    \005\000\003\000\005\000\005\000\005\000\005\000\005\000\000\000\
    \186\000\186\000\006\000\188\000\188\000\006\000\006\000\006\000\
    \000\000\000\000\113\000\006\000\006\000\209\000\006\000\006\000\
    \006\000\000\000\000\000\177\000\112\000\108\000\196\000\003\000\
    \109\000\003\000\000\000\006\000\005\000\006\000\006\000\006\000\
    \006\000\006\000\000\000\201\000\201\000\117\000\203\000\203\000\
    \117\000\117\000\117\000\112\000\000\000\111\000\117\000\117\000\
    \210\000\117\000\141\000\117\000\000\000\000\000\000\000\210\000\
    \114\000\000\000\005\000\115\000\005\000\000\000\117\000\006\000\
    \117\000\140\000\117\000\117\000\117\000\000\000\000\000\000\000\
    \138\000\000\000\000\000\138\000\138\000\138\000\000\000\000\000\
    \000\000\138\000\138\000\000\000\138\000\138\000\138\000\201\000\
    \000\000\000\000\202\000\000\000\000\000\006\000\000\000\006\000\
    \000\000\138\000\117\000\138\000\139\000\138\000\138\000\138\000\
    \000\000\177\000\000\000\006\000\178\000\000\000\006\000\006\000\
    \006\000\000\000\000\000\000\000\006\000\006\000\000\000\006\000\
    \006\000\006\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \117\000\180\000\117\000\000\000\006\000\138\000\006\000\006\000\
    \006\000\006\000\006\000\000\000\000\000\000\000\000\000\000\000\
    \006\000\000\000\000\000\006\000\006\000\006\000\000\000\255\255\
    \000\000\006\000\006\000\000\000\006\000\006\000\006\000\000\000\
    \000\000\000\000\000\000\138\000\000\000\138\000\000\000\137\000\
    \006\000\006\000\000\000\006\000\006\000\006\000\006\000\006\000\
    \255\255\000\000\000\000\000\000\000\000\006\000\000\000\000\000\
    \006\000\006\000\006\000\179\000\000\000\000\000\006\000\006\000\
    \000\000\006\000\006\000\006\000\000\000\000\000\006\000\136\000\
    \006\000\199\000\000\000\000\000\134\000\006\000\006\000\000\000\
    \006\000\006\000\006\000\006\000\006\000\000\000\000\000\000\000\
    \006\000\000\000\000\000\006\000\006\000\006\000\000\000\000\000\
    \158\000\006\000\006\000\159\000\125\000\006\000\006\000\000\000\
    \255\255\000\000\000\000\135\000\000\000\006\000\000\000\000\000\
    \000\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \000\000\000\000\000\000\117\000\000\000\160\000\117\000\117\000\
    \117\000\000\000\000\000\255\255\117\000\117\000\000\000\117\000\
    \118\000\117\000\000\000\255\255\000\000\000\000\000\000\000\000\
    \006\000\000\000\006\000\000\000\117\000\006\000\117\000\117\000\
    \119\000\117\000\117\000\000\000\000\000\000\000\006\000\000\000\
    \000\000\006\000\006\000\116\000\000\000\000\000\255\255\006\000\
    \006\000\000\000\006\000\006\000\006\000\255\255\255\255\000\000\
    \000\000\000\000\156\000\006\000\000\000\006\000\000\000\006\000\
    \117\000\006\000\006\000\006\000\006\000\006\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \000\000\000\000\000\000\000\000\000\000\200\000\000\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\000\000\117\000\000\000\
    \117\000\000\000\000\000\006\000\000\000\000\000\000\000\176\000\
    \000\000\000\000\000\000\000\000\000\000\107\000\192\000\192\000\
    \192\000\192\000\192\000\192\000\192\000\192\000\000\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\000\000\000\000\000\000\
    \000\000\006\000\000\000\006\000\107\000\105\000\000\000\105\000\
    \105\000\105\000\105\000\000\000\000\000\000\000\105\000\105\000\
    \107\000\105\000\105\000\105\000\106\000\106\000\106\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\106\000\105\000\000\000\
    \105\000\105\000\105\000\105\000\105\000\000\000\000\000\107\000\
    \003\000\000\000\000\000\003\000\003\000\003\000\000\000\000\000\
    \104\000\103\000\003\000\000\000\003\000\003\000\003\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\106\000\106\000\106\000\
    \106\000\003\000\105\000\003\000\003\000\003\000\003\000\003\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\168\000\168\000\168\000\168\000\168\000\168\000\
    \168\000\168\000\168\000\168\000\000\000\000\000\255\255\000\000\
    \105\000\073\000\105\000\000\000\075\000\003\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \000\000\074\000\000\000\003\000\075\000\003\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \055\000\074\000\000\000\000\000\000\000\000\000\000\000\057\000\
    \000\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\055\000\055\000\055\000\055\000\056\000\055\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\000\000\000\000\000\000\000\000\
    \030\000\000\000\055\000\055\000\055\000\055\000\056\000\055\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\055\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\057\000\000\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\055\000\059\000\
    \055\000\055\000\056\000\055\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\060\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\061\000\058\000\058\000\
    \000\000\000\000\000\000\000\000\030\000\000\000\055\000\059\000\
    \055\000\055\000\056\000\055\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\060\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\061\000\058\000\058\000\
    \032\000\193\000\193\000\193\000\193\000\193\000\193\000\193\000\
    \193\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\000\000\000\000\000\000\000\000\
    \032\000\000\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\096\000\096\000\096\000\096\000\
    \096\000\096\000\096\000\096\000\096\000\096\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\096\000\096\000\096\000\
    \096\000\096\000\096\000\189\000\189\000\189\000\189\000\189\000\
    \189\000\189\000\189\000\189\000\189\000\190\000\190\000\190\000\
    \190\000\190\000\190\000\190\000\190\000\190\000\190\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\096\000\096\000\096\000\
    \096\000\096\000\096\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \000\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \033\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\000\000\000\000\000\000\000\000\
    \033\000\000\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\112\000\108\000\000\000\000\000\
    \109\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\112\000\000\000\111\000\000\000\000\000\
    \000\000\000\000\144\000\000\000\000\000\145\000\000\000\000\000\
    \000\000\000\000\000\000\106\000\106\000\106\000\106\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\000\000\000\000\000\000\
    \000\000\000\000\149\000\000\000\000\000\000\000\000\000\147\000\
    \151\000\000\000\150\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \000\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \034\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\148\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\000\000\000\000\000\000\000\000\
    \034\000\000\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\155\000\000\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \000\000\154\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\155\000\255\255\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\000\000\154\000\
    \146\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \000\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \035\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\000\000\000\000\000\000\000\000\
    \035\000\000\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\170\000\170\000\170\000\170\000\
    \170\000\170\000\170\000\170\000\170\000\170\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\170\000\170\000\170\000\
    \170\000\170\000\170\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\170\000\170\000\170\000\
    \170\000\170\000\170\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \000\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \000\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\046\000\000\000\000\000\046\000\046\000\046\000\000\000\
    \000\000\000\000\046\000\046\000\000\000\046\000\046\000\046\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\046\000\000\000\046\000\046\000\046\000\046\000\
    \046\000\000\000\205\000\000\000\205\000\205\000\205\000\205\000\
    \205\000\205\000\205\000\205\000\205\000\205\000\205\000\205\000\
    \205\000\205\000\205\000\205\000\205\000\205\000\205\000\205\000\
    \205\000\205\000\205\000\205\000\205\000\205\000\046\000\052\000\
    \204\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\000\000\046\000\046\000\046\000\000\000\
    \046\000\046\000\046\000\000\000\000\000\000\000\046\000\046\000\
    \000\000\046\000\046\000\046\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\046\000\000\000\
    \046\000\046\000\046\000\046\000\046\000\000\000\205\000\000\000\
    \205\000\205\000\205\000\205\000\205\000\205\000\205\000\205\000\
    \205\000\205\000\205\000\205\000\205\000\205\000\205\000\205\000\
    \205\000\205\000\205\000\205\000\205\000\205\000\205\000\205\000\
    \205\000\205\000\046\000\048\000\204\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\000\000\
    \046\000\000\000\046\000\000\000\000\000\000\000\000\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\000\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \171\000\171\000\171\000\171\000\171\000\171\000\171\000\171\000\
    \171\000\171\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\171\000\171\000\171\000\171\000\171\000\171\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\171\000\171\000\171\000\171\000\171\000\171\000\000\000\
    \000\000\000\000\000\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\035\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\000\000\
    \000\000\000\000\000\000\035\000\000\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\194\000\
    \194\000\194\000\194\000\194\000\194\000\194\000\194\000\194\000\
    \194\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \194\000\194\000\194\000\194\000\194\000\194\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \194\000\194\000\194\000\194\000\194\000\194\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\000\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\000\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\046\000\000\000\000\000\046\000\
    \046\000\046\000\000\000\000\000\000\000\046\000\046\000\000\000\
    \046\000\046\000\046\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\046\000\000\000\046\000\
    \046\000\046\000\046\000\046\000\000\000\000\000\000\000\000\000\
    \047\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\050\000\000\000\000\000\000\000\000\000\
    \000\000\046\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\000\000\000\000\000\000\046\000\
    \047\000\046\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\195\000\195\000\195\000\195\000\
    \195\000\195\000\195\000\195\000\195\000\195\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\195\000\195\000\195\000\
    \195\000\195\000\195\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\195\000\195\000\195\000\
    \195\000\195\000\195\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \000\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \048\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\049\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\000\000\000\000\000\000\000\000\
    \048\000\000\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \000\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \051\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\054\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\000\000\000\000\000\000\000\000\
    \051\000\000\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \000\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \052\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\053\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\000\000\000\000\000\000\000\000\
    \052\000\000\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \000\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \055\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\000\000\000\000\000\000\000\000\
    \055\000\000\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\000\000\000\000\000\000\
    \072\000\000\000\072\000\000\000\000\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \000\000\000\000\000\000\000\000\055\000\000\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\055\000\055\000\055\000\055\000\056\000\055\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\000\000\000\000\000\000\000\000\
    \057\000\000\000\055\000\055\000\055\000\055\000\056\000\055\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\055\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \000\000\000\000\000\000\000\000\055\000\000\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\070\000\070\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\000\000\000\000\000\000\000\000\
    \055\000\000\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\055\000\055\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \000\000\000\000\000\000\000\000\055\000\000\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\000\000\000\000\000\000\000\000\
    \055\000\000\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\063\000\000\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\064\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \000\000\000\000\000\000\000\000\062\000\000\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\064\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \055\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\068\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\000\000\000\000\000\000\000\000\
    \063\000\000\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\068\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\055\000\000\000\000\000\000\000\
    \066\000\000\000\066\000\000\000\000\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \000\000\000\000\000\000\000\000\055\000\000\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\000\000\000\000\000\000\000\000\
    \065\000\000\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\055\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \000\000\000\000\000\000\000\000\055\000\000\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\000\000\000\000\000\000\066\000\000\000\066\000\000\000\
    \000\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\000\000\000\000\000\000\000\000\
    \055\000\000\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\055\000\055\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \000\000\000\000\000\000\000\000\069\000\000\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \055\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\070\000\070\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\000\000\000\000\000\000\000\000\
    \070\000\000\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\055\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \000\000\000\000\000\000\000\000\071\000\000\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \086\000\103\000\086\000\000\000\103\000\103\000\103\000\086\000\
    \000\000\000\000\103\000\103\000\000\000\103\000\103\000\103\000\
    \085\000\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
    \085\000\085\000\103\000\000\000\103\000\103\000\103\000\103\000\
    \103\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \105\000\000\000\105\000\105\000\105\000\105\000\000\000\000\000\
    \000\000\105\000\105\000\000\000\105\000\105\000\105\000\000\000\
    \000\000\000\000\000\000\000\000\086\000\000\000\103\000\000\000\
    \000\000\105\000\086\000\105\000\105\000\105\000\105\000\105\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\086\000\084\000\
    \000\000\000\000\086\000\000\000\086\000\000\000\006\000\000\000\
    \083\000\006\000\006\000\006\000\103\000\000\000\103\000\006\000\
    \006\000\000\000\006\000\006\000\006\000\105\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\006\000\
    \000\000\006\000\006\000\006\000\006\000\006\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\117\000\000\000\000\000\
    \117\000\117\000\117\000\105\000\000\000\105\000\117\000\117\000\
    \000\000\117\000\117\000\117\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\006\000\000\000\000\000\117\000\000\000\
    \117\000\117\000\117\000\117\000\117\000\000\000\000\000\000\000\
    \117\000\000\000\000\000\117\000\117\000\117\000\000\000\000\000\
    \000\000\117\000\117\000\000\000\117\000\117\000\117\000\000\000\
    \000\000\006\000\000\000\006\000\000\000\000\000\000\000\000\000\
    \000\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\117\000\000\000\
    \000\000\117\000\117\000\117\000\000\000\000\000\000\000\117\000\
    \117\000\000\000\117\000\117\000\117\000\000\000\000\000\000\000\
    \117\000\000\000\117\000\000\000\000\000\117\000\000\000\117\000\
    \255\255\117\000\117\000\117\000\117\000\117\000\000\000\000\000\
    \000\000\006\000\000\000\000\000\006\000\006\000\006\000\000\000\
    \000\000\000\000\006\000\006\000\000\000\006\000\006\000\006\000\
    \000\000\000\000\000\000\117\000\000\000\117\000\000\000\000\000\
    \000\000\000\000\006\000\117\000\006\000\006\000\006\000\006\000\
    \006\000\000\000\000\000\000\000\006\000\000\000\000\000\006\000\
    \006\000\006\000\000\000\000\000\000\000\006\000\006\000\000\000\
    \006\000\006\000\006\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\117\000\000\000\117\000\000\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\000\000\000\000\000\000\138\000\
    \000\000\000\000\138\000\138\000\138\000\000\000\000\000\000\000\
    \138\000\138\000\000\000\138\000\138\000\138\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\006\000\000\000\006\000\000\000\
    \138\000\006\000\138\000\138\000\138\000\138\000\138\000\000\000\
    \000\000\000\000\138\000\000\000\000\000\138\000\138\000\138\000\
    \000\000\000\000\000\000\138\000\138\000\000\000\138\000\138\000\
    \138\000\000\000\000\000\000\000\000\000\000\000\000\000\006\000\
    \000\000\006\000\000\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\000\000\000\000\000\000\117\000\000\000\000\000\
    \117\000\117\000\117\000\000\000\000\000\000\000\117\000\117\000\
    \000\000\117\000\117\000\117\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\138\000\000\000\138\000\000\000\117\000\138\000\
    \117\000\117\000\117\000\117\000\117\000\000\000\000\000\000\000\
    \117\000\000\000\000\000\117\000\117\000\117\000\000\000\000\000\
    \000\000\117\000\117\000\000\000\117\000\117\000\117\000\000\000\
    \000\000\165\000\000\000\165\000\000\000\138\000\000\000\138\000\
    \165\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \000\000\164\000\164\000\164\000\164\000\164\000\164\000\164\000\
    \164\000\164\000\164\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \117\000\000\000\117\000\000\000\000\000\117\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\186\000\000\000\000\000\187\000\
    \000\000\000\000\000\000\000\000\000\000\165\000\000\000\000\000\
    \000\000\000\000\000\000\165\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\185\000\117\000\185\000\117\000\000\000\165\000\
    \000\000\185\000\000\000\165\000\000\000\165\000\000\000\000\000\
    \000\000\163\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\185\000\000\000\
    \000\000\000\000\000\000\000\000\185\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \185\000\183\000\000\000\000\000\185\000\000\000\185\000\000\000\
    \000\000\000\000\182\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255";
  Lexing.lex_check = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\041\000\000\000\000\000\041\000\042\000\
    \044\000\045\000\042\000\044\000\045\000\079\000\109\000\115\000\
    \079\000\109\000\115\000\145\000\159\000\213\000\145\000\159\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\016\000\
    \013\000\017\000\018\000\026\000\017\000\017\000\039\000\077\000\
    \078\000\039\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\086\000\089\000\093\000\013\000\
    \096\000\013\000\126\000\013\000\127\000\039\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \084\000\084\000\084\000\084\000\085\000\085\000\085\000\085\000\
    \085\000\085\000\085\000\085\000\085\000\085\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\013\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\131\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\098\000\098\000\003\000\003\000\003\000\100\000\
    \100\000\027\000\003\000\003\000\027\000\003\000\003\000\003\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \102\000\102\000\003\000\150\000\003\000\003\000\003\000\003\000\
    \003\000\151\000\110\000\157\000\004\000\110\000\027\000\004\000\
    \004\000\004\000\158\000\165\000\168\000\004\000\004\000\171\000\
    \004\000\004\000\004\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\092\000\186\000\206\000\004\000\003\000\004\000\
    \004\000\004\000\004\000\004\000\207\000\173\000\178\000\005\000\
    \173\000\178\000\005\000\005\000\005\000\210\000\212\000\255\255\
    \005\000\005\000\186\000\005\000\005\000\005\000\183\000\183\000\
    \183\000\183\000\111\000\027\000\003\000\111\000\003\000\255\255\
    \005\000\004\000\005\000\005\000\005\000\005\000\005\000\255\255\
    \187\000\188\000\006\000\187\000\188\000\006\000\006\000\006\000\
    \255\255\255\255\111\000\006\000\006\000\208\000\006\000\006\000\
    \006\000\255\255\255\255\196\000\112\000\112\000\196\000\004\000\
    \112\000\004\000\255\255\006\000\005\000\006\000\006\000\006\000\
    \006\000\006\000\255\255\202\000\203\000\007\000\202\000\203\000\
    \007\000\007\000\007\000\112\000\255\255\112\000\007\000\007\000\
    \209\000\007\000\007\000\007\000\255\255\255\255\255\255\211\000\
    \113\000\255\255\005\000\113\000\005\000\255\255\007\000\006\000\
    \007\000\007\000\007\000\007\000\007\000\255\255\255\255\255\255\
    \008\000\255\255\255\255\008\000\008\000\008\000\255\255\255\255\
    \255\255\008\000\008\000\255\255\008\000\008\000\008\000\197\000\
    \255\255\255\255\197\000\255\255\255\255\006\000\255\255\006\000\
    \255\255\008\000\007\000\008\000\008\000\008\000\008\000\008\000\
    \255\255\174\000\255\255\010\000\174\000\255\255\010\000\010\000\
    \010\000\255\255\255\255\255\255\010\000\010\000\255\255\010\000\
    \010\000\010\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \007\000\174\000\007\000\255\255\010\000\008\000\010\000\010\000\
    \010\000\010\000\010\000\255\255\255\255\255\255\255\255\255\255\
    \011\000\255\255\255\255\011\000\011\000\011\000\255\255\027\000\
    \255\255\011\000\011\000\255\255\011\000\011\000\011\000\255\255\
    \255\255\255\255\255\255\008\000\255\255\008\000\255\255\010\000\
    \010\000\011\000\255\255\011\000\011\000\011\000\011\000\011\000\
    \110\000\255\255\255\255\255\255\255\255\014\000\255\255\255\255\
    \014\000\014\000\014\000\174\000\255\255\255\255\014\000\014\000\
    \255\255\014\000\014\000\014\000\255\255\255\255\010\000\010\000\
    \010\000\197\000\255\255\255\255\011\000\011\000\014\000\255\255\
    \014\000\014\000\014\000\014\000\014\000\255\255\255\255\255\255\
    \015\000\255\255\255\255\015\000\015\000\015\000\255\255\255\255\
    \147\000\015\000\015\000\147\000\015\000\015\000\015\000\255\255\
    \111\000\255\255\255\255\011\000\255\255\011\000\255\255\255\255\
    \255\255\015\000\014\000\015\000\015\000\015\000\015\000\015\000\
    \255\255\255\255\255\255\019\000\255\255\147\000\019\000\019\000\
    \019\000\255\255\255\255\208\000\019\000\019\000\255\255\019\000\
    \019\000\019\000\255\255\112\000\255\255\255\255\255\255\255\255\
    \014\000\255\255\014\000\255\255\019\000\015\000\019\000\019\000\
    \019\000\019\000\019\000\255\255\255\255\255\255\023\000\255\255\
    \255\255\023\000\023\000\023\000\255\255\255\255\209\000\023\000\
    \023\000\255\255\023\000\023\000\023\000\211\000\113\000\255\255\
    \255\255\255\255\147\000\015\000\255\255\015\000\255\255\023\000\
    \019\000\023\000\023\000\023\000\023\000\023\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \255\255\255\255\255\255\255\255\255\255\197\000\255\255\083\000\
    \083\000\083\000\083\000\083\000\083\000\255\255\019\000\255\255\
    \019\000\255\255\255\255\023\000\255\255\255\255\255\255\174\000\
    \255\255\255\255\255\255\255\255\255\255\024\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\191\000\191\000\255\255\083\000\
    \083\000\083\000\083\000\083\000\083\000\255\255\255\255\255\255\
    \255\255\023\000\255\255\023\000\024\000\024\000\255\255\024\000\
    \024\000\024\000\024\000\255\255\255\255\255\255\024\000\024\000\
    \107\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\255\255\
    \024\000\024\000\024\000\024\000\024\000\255\255\255\255\107\000\
    \025\000\255\255\255\255\025\000\025\000\025\000\255\255\255\255\
    \025\000\025\000\025\000\255\255\025\000\025\000\025\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\025\000\024\000\025\000\025\000\025\000\025\000\025\000\
    \164\000\164\000\164\000\164\000\164\000\164\000\164\000\164\000\
    \164\000\164\000\167\000\167\000\167\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\255\255\255\255\147\000\255\255\
    \024\000\028\000\024\000\255\255\075\000\025\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \255\255\075\000\255\255\025\000\028\000\025\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \030\000\028\000\255\255\255\255\255\255\255\255\255\255\030\000\
    \255\255\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\255\255\255\255\255\255\255\255\
    \030\000\255\255\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\031\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\031\000\255\255\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \255\255\255\255\255\255\255\255\031\000\255\255\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \032\000\192\000\192\000\192\000\192\000\192\000\192\000\192\000\
    \192\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\255\255\255\255\255\255\255\255\
    \032\000\255\255\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\095\000\095\000\095\000\
    \095\000\095\000\095\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\189\000\189\000\189\000\
    \189\000\189\000\189\000\189\000\189\000\189\000\189\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\095\000\095\000\095\000\
    \095\000\095\000\095\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \255\255\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \033\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\255\255\255\255\255\255\255\255\
    \033\000\255\255\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\106\000\106\000\255\255\255\255\
    \106\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\106\000\255\255\106\000\255\255\255\255\
    \255\255\255\255\142\000\255\255\255\255\142\000\255\255\255\255\
    \255\255\255\255\255\255\106\000\106\000\106\000\106\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\255\255\255\255\255\255\
    \255\255\255\255\142\000\255\255\255\255\255\255\255\255\142\000\
    \142\000\255\255\142\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \255\255\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \034\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\142\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\255\255\255\255\255\255\255\255\
    \034\000\255\255\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\148\000\255\255\148\000\148\000\
    \148\000\148\000\148\000\148\000\148\000\148\000\148\000\148\000\
    \148\000\148\000\148\000\148\000\148\000\148\000\148\000\148\000\
    \148\000\148\000\148\000\148\000\148\000\148\000\148\000\148\000\
    \255\255\148\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\155\000\106\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\255\255\155\000\
    \142\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \255\255\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \035\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\255\255\255\255\255\255\255\255\
    \035\000\255\255\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\163\000\163\000\163\000\163\000\
    \163\000\163\000\163\000\163\000\163\000\163\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\163\000\163\000\163\000\
    \163\000\163\000\163\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\163\000\163\000\163\000\
    \163\000\163\000\163\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \255\255\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \255\255\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\036\000\255\255\255\255\036\000\036\000\036\000\255\255\
    \255\255\255\255\036\000\036\000\255\255\036\000\036\000\036\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\036\000\255\255\036\000\036\000\036\000\036\000\
    \036\000\255\255\199\000\255\255\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\036\000\036\000\
    \199\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\255\255\036\000\037\000\036\000\255\255\
    \037\000\037\000\037\000\255\255\255\255\255\255\037\000\037\000\
    \255\255\037\000\037\000\037\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\037\000\255\255\
    \037\000\037\000\037\000\037\000\037\000\255\255\205\000\255\255\
    \205\000\205\000\205\000\205\000\205\000\205\000\205\000\205\000\
    \205\000\205\000\205\000\205\000\205\000\205\000\205\000\205\000\
    \205\000\205\000\205\000\205\000\205\000\205\000\205\000\205\000\
    \205\000\205\000\037\000\037\000\205\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\255\255\
    \037\000\255\255\037\000\255\255\255\255\255\255\255\255\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\255\255\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \170\000\170\000\170\000\170\000\170\000\170\000\170\000\170\000\
    \170\000\170\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\170\000\170\000\170\000\170\000\170\000\170\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\170\000\170\000\170\000\170\000\170\000\170\000\255\255\
    \255\255\255\255\255\255\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\038\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\255\255\
    \255\255\255\255\255\255\038\000\255\255\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\182\000\
    \182\000\182\000\182\000\182\000\182\000\182\000\182\000\182\000\
    \182\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \182\000\182\000\182\000\182\000\182\000\182\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \182\000\182\000\182\000\182\000\182\000\182\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\255\255\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\255\255\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\046\000\255\255\255\255\046\000\
    \046\000\046\000\255\255\255\255\255\255\046\000\046\000\255\255\
    \046\000\046\000\046\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\046\000\255\255\046\000\
    \046\000\046\000\046\000\046\000\255\255\255\255\255\255\255\255\
    \047\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\255\255\255\255\255\255\255\255\
    \255\255\046\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\255\255\255\255\255\255\046\000\
    \047\000\046\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\194\000\194\000\194\000\194\000\
    \194\000\194\000\194\000\194\000\194\000\194\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\194\000\194\000\194\000\
    \194\000\194\000\194\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\194\000\194\000\194\000\
    \194\000\194\000\194\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \255\255\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \048\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\255\255\255\255\255\255\255\255\
    \048\000\255\255\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \255\255\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \051\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\255\255\255\255\255\255\255\255\
    \051\000\255\255\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \255\255\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \052\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\255\255\255\255\255\255\255\255\
    \052\000\255\255\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \255\255\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \055\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\255\255\255\255\255\255\255\255\
    \055\000\255\255\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\056\000\255\255\255\255\255\255\
    \056\000\255\255\056\000\255\255\255\255\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \255\255\255\255\255\255\255\255\056\000\255\255\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \057\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\255\255\255\255\255\255\255\255\
    \057\000\255\255\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\058\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \255\255\255\255\255\255\255\255\058\000\255\255\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \059\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\255\255\255\255\255\255\255\255\
    \059\000\255\255\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\060\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \255\255\255\255\255\255\255\255\060\000\255\255\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \061\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\255\255\255\255\255\255\255\255\
    \061\000\255\255\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\062\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\062\000\255\255\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \255\255\255\255\255\255\255\255\062\000\255\255\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \063\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\255\255\255\255\255\255\255\255\
    \063\000\255\255\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\064\000\255\255\255\255\255\255\
    \064\000\255\255\064\000\255\255\255\255\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \255\255\255\255\255\255\255\255\064\000\255\255\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \065\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\255\255\255\255\255\255\255\255\
    \065\000\255\255\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\067\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \255\255\255\255\255\255\255\255\067\000\255\255\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \068\000\255\255\255\255\255\255\068\000\255\255\068\000\255\255\
    \255\255\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\255\255\255\255\255\255\255\255\
    \068\000\255\255\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\069\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \255\255\255\255\255\255\255\255\069\000\255\255\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \070\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\255\255\255\255\255\255\255\255\
    \070\000\255\255\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\071\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \255\255\255\255\255\255\255\255\071\000\255\255\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \076\000\103\000\076\000\255\255\103\000\103\000\103\000\076\000\
    \255\255\255\255\103\000\103\000\255\255\103\000\103\000\103\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\103\000\255\255\103\000\103\000\103\000\103\000\
    \103\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \105\000\255\255\105\000\105\000\105\000\105\000\255\255\255\255\
    \255\255\105\000\105\000\255\255\105\000\105\000\105\000\255\255\
    \255\255\255\255\255\255\255\255\076\000\255\255\103\000\255\255\
    \255\255\105\000\076\000\105\000\105\000\105\000\105\000\105\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\076\000\076\000\
    \255\255\255\255\076\000\255\255\076\000\255\255\116\000\255\255\
    \076\000\116\000\116\000\116\000\103\000\255\255\103\000\116\000\
    \116\000\255\255\116\000\116\000\116\000\105\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\116\000\
    \255\255\116\000\116\000\116\000\116\000\116\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\117\000\255\255\255\255\
    \117\000\117\000\117\000\105\000\255\255\105\000\117\000\117\000\
    \255\255\117\000\117\000\117\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\116\000\255\255\255\255\117\000\255\255\
    \117\000\117\000\117\000\117\000\117\000\255\255\255\255\255\255\
    \118\000\255\255\255\255\118\000\118\000\118\000\255\255\255\255\
    \255\255\118\000\118\000\255\255\118\000\118\000\118\000\255\255\
    \255\255\116\000\255\255\116\000\255\255\255\255\255\255\255\255\
    \255\255\118\000\117\000\118\000\118\000\118\000\118\000\118\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\119\000\255\255\
    \255\255\119\000\119\000\119\000\255\255\255\255\255\255\119\000\
    \119\000\255\255\119\000\119\000\119\000\255\255\255\255\255\255\
    \117\000\255\255\117\000\255\255\255\255\118\000\255\255\119\000\
    \076\000\119\000\119\000\119\000\119\000\119\000\255\255\255\255\
    \255\255\125\000\255\255\255\255\125\000\125\000\125\000\255\255\
    \255\255\255\255\125\000\125\000\255\255\125\000\125\000\125\000\
    \255\255\255\255\255\255\118\000\255\255\118\000\255\255\255\255\
    \255\255\255\255\125\000\119\000\125\000\125\000\125\000\125\000\
    \125\000\255\255\255\255\255\255\135\000\255\255\255\255\135\000\
    \135\000\135\000\255\255\255\255\255\255\135\000\135\000\255\255\
    \135\000\135\000\135\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\119\000\255\255\119\000\255\255\135\000\125\000\135\000\
    \135\000\135\000\135\000\135\000\255\255\255\255\255\255\138\000\
    \255\255\255\255\138\000\138\000\138\000\255\255\255\255\255\255\
    \138\000\138\000\255\255\138\000\138\000\138\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\125\000\255\255\125\000\255\255\
    \138\000\135\000\138\000\138\000\138\000\138\000\138\000\255\255\
    \255\255\255\255\139\000\255\255\255\255\139\000\139\000\139\000\
    \255\255\255\255\255\255\139\000\139\000\255\255\139\000\139\000\
    \139\000\255\255\255\255\255\255\255\255\255\255\255\255\135\000\
    \255\255\135\000\255\255\139\000\138\000\139\000\139\000\139\000\
    \139\000\139\000\255\255\255\255\255\255\140\000\255\255\255\255\
    \140\000\140\000\140\000\255\255\255\255\255\255\140\000\140\000\
    \255\255\140\000\140\000\140\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\138\000\255\255\138\000\255\255\140\000\139\000\
    \140\000\140\000\140\000\140\000\140\000\255\255\255\255\255\255\
    \141\000\255\255\255\255\141\000\141\000\141\000\255\255\255\255\
    \255\255\141\000\141\000\255\255\141\000\141\000\141\000\255\255\
    \255\255\156\000\255\255\156\000\255\255\139\000\255\255\139\000\
    \156\000\141\000\140\000\141\000\141\000\141\000\141\000\141\000\
    \255\255\156\000\156\000\156\000\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \140\000\255\255\140\000\255\255\255\255\141\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\179\000\255\255\255\255\179\000\
    \255\255\255\255\255\255\255\255\255\255\156\000\255\255\255\255\
    \255\255\255\255\255\255\156\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\179\000\141\000\179\000\141\000\255\255\156\000\
    \255\255\179\000\255\255\156\000\255\255\156\000\255\255\255\255\
    \255\255\156\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\179\000\255\255\
    \255\255\255\255\255\255\255\255\179\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \179\000\179\000\255\255\255\255\179\000\255\255\179\000\255\255\
    \255\255\255\255\179\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\179\000";
  Lexing.lex_base_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\010\000\036\000\012\000\000\000\000\000\000\000\
    \002\000\000\000\027\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\001\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\002\000\004\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_backtrk_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\039\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_default_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\019\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_trans_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\001\000\000\000\036\000\036\000\000\000\036\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \001\000\000\000\000\000\001\000\022\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\007\000\001\000\000\000\000\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\001\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_check_code = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\024\000\111\000\179\000\187\000\111\000\188\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \024\000\255\255\111\000\000\000\112\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\106\000\107\000\255\255\255\255\
    \024\000\024\000\024\000\024\000\024\000\024\000\024\000\024\000\
    \024\000\024\000\106\000\106\000\106\000\106\000\106\000\106\000\
    \106\000\106\000\106\000\106\000\107\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \111\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255";
  Lexing.lex_code = 
   "\255\004\255\255\005\255\255\007\255\006\255\255\003\255\000\004\
    \001\005\255\007\255\255\006\255\007\255\255\000\004\001\005\003\
    \006\002\007\255\001\255\255\000\001\255";
}

let rec token lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 8 (-1) ;   __ocaml_lex_token_rec lexbuf 0
and __ocaml_lex_token_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 308 "parsing/lexer.mll"
                 (
      if not !escaped_newlines then
        raise (Error(Illegal_character (Lexing.lexeme_char lexbuf 0),
                     Location.curr lexbuf));
      update_loc lexbuf None 1 false 0;
      token lexbuf )
# 1867 "parsing/lexer.ml"

  | 1 ->
# 315 "parsing/lexer.mll"
      ( update_loc lexbuf None 1 false 0;
        EOL )
# 1873 "parsing/lexer.ml"

  | 2 ->
# 318 "parsing/lexer.mll"
      ( token lexbuf )
# 1878 "parsing/lexer.ml"

  | 3 ->
# 320 "parsing/lexer.mll"
      ( UNDERSCORE )
# 1883 "parsing/lexer.ml"

  | 4 ->
# 322 "parsing/lexer.mll"
      ( TILDE )
# 1888 "parsing/lexer.ml"

  | 5 ->
# 324 "parsing/lexer.mll"
      ( LABEL (get_label_name lexbuf) )
# 1893 "parsing/lexer.ml"

  | 6 ->
# 326 "parsing/lexer.mll"
      ( warn_latin1 lexbuf; LABEL (get_label_name lexbuf) )
# 1898 "parsing/lexer.ml"

  | 7 ->
# 328 "parsing/lexer.mll"
      ( QUESTION )
# 1903 "parsing/lexer.ml"

  | 8 ->
# 330 "parsing/lexer.mll"
      ( OPTLABEL (get_label_name lexbuf) )
# 1908 "parsing/lexer.ml"

  | 9 ->
# 332 "parsing/lexer.mll"
      ( warn_latin1 lexbuf; OPTLABEL (get_label_name lexbuf) )
# 1913 "parsing/lexer.ml"

  | 10 ->
# 334 "parsing/lexer.mll"
      ( let s = Lexing.lexeme lexbuf in
        try Hashtbl.find keyword_table s
        with Not_found -> LIDENT s )
# 1920 "parsing/lexer.ml"

  | 11 ->
# 338 "parsing/lexer.mll"
      ( warn_latin1 lexbuf; LIDENT (Lexing.lexeme lexbuf) )
# 1925 "parsing/lexer.ml"

  | 12 ->
# 340 "parsing/lexer.mll"
      ( UIDENT(Lexing.lexeme lexbuf) )
# 1930 "parsing/lexer.ml"

  | 13 ->
# 342 "parsing/lexer.mll"
      ( warn_latin1 lexbuf; UIDENT(Lexing.lexeme lexbuf) )
# 1935 "parsing/lexer.ml"

  | 14 ->
# 343 "parsing/lexer.mll"
                ( INT (Lexing.lexeme lexbuf, None) )
# 1940 "parsing/lexer.ml"

  | 15 ->
let
# 344 "parsing/lexer.mll"
                    lit
# 1946 "parsing/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_curr_pos + -1)
and
# 344 "parsing/lexer.mll"
                                              modif
# 1951 "parsing/lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_curr_pos + -1) in
# 345 "parsing/lexer.mll"
      ( INT (lit, Some modif) )
# 1955 "parsing/lexer.ml"

  | 16 ->
# 347 "parsing/lexer.mll"
      ( FLOAT (Lexing.lexeme lexbuf, None) )
# 1960 "parsing/lexer.ml"

  | 17 ->
let
# 348 "parsing/lexer.mll"
                                            lit
# 1966 "parsing/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_curr_pos + -1)
and
# 348 "parsing/lexer.mll"
                                                                      modif
# 1971 "parsing/lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_curr_pos + -1) in
# 349 "parsing/lexer.mll"
      ( FLOAT (lit, Some modif) )
# 1975 "parsing/lexer.ml"

  | 18 ->
# 351 "parsing/lexer.mll"
      ( raise (Error(Invalid_literal (Lexing.lexeme lexbuf),
                     Location.curr lexbuf)) )
# 1981 "parsing/lexer.ml"

  | 19 ->
# 354 "parsing/lexer.mll"
      ( reset_string_buffer();
        is_in_string := true;
        let string_start = lexbuf.lex_start_p in
        string_start_loc := Location.curr lexbuf;
        string lexbuf;
        is_in_string := false;
        lexbuf.lex_start_p <- string_start;
        STRING (get_stored_string(), None) )
# 1993 "parsing/lexer.ml"

  | 20 ->
# 363 "parsing/lexer.mll"
      ( reset_string_buffer();
        let delim = Lexing.lexeme lexbuf in
        let delim = String.sub delim 1 (String.length delim - 2) in
        is_in_string := true;
        let string_start = lexbuf.lex_start_p in
        string_start_loc := Location.curr lexbuf;
        quoted_string delim lexbuf;
        is_in_string := false;
        lexbuf.lex_start_p <- string_start;
        STRING (get_stored_string(), Some delim) )
# 2007 "parsing/lexer.ml"

  | 21 ->
# 374 "parsing/lexer.mll"
      ( update_loc lexbuf None 1 false 1;
        CHAR (Lexing.lexeme_char lexbuf 1) )
# 2013 "parsing/lexer.ml"

  | 22 ->
# 377 "parsing/lexer.mll"
      ( CHAR(Lexing.lexeme_char lexbuf 1) )
# 2018 "parsing/lexer.ml"

  | 23 ->
# 379 "parsing/lexer.mll"
      ( CHAR(char_for_backslash (Lexing.lexeme_char lexbuf 2)) )
# 2023 "parsing/lexer.ml"

  | 24 ->
# 381 "parsing/lexer.mll"
      ( CHAR(char_for_decimal_code lexbuf 2) )
# 2028 "parsing/lexer.ml"

  | 25 ->
# 383 "parsing/lexer.mll"
      ( CHAR(char_for_octal_code lexbuf 3) )
# 2033 "parsing/lexer.ml"

  | 26 ->
# 385 "parsing/lexer.mll"
      ( CHAR(char_for_hexadecimal_code lexbuf 3) )
# 2038 "parsing/lexer.ml"

  | 27 ->
# 387 "parsing/lexer.mll"
      ( let l = Lexing.lexeme lexbuf in
        let esc = String.sub l 1 (String.length l - 1) in
        raise (Error(Illegal_escape esc, Location.curr lexbuf))
      )
# 2046 "parsing/lexer.ml"

  | 28 ->
# 392 "parsing/lexer.mll"
      ( let s, loc = with_comment_buffer comment lexbuf in
        COMMENT (s, loc) )
# 2052 "parsing/lexer.ml"

  | 29 ->
# 395 "parsing/lexer.mll"
      ( let s, loc = with_comment_buffer comment lexbuf in
        if !handle_docstrings then
          DOCSTRING (Docstrings.docstring s loc)
        else
          COMMENT ("*" ^ s, loc)
      )
# 2062 "parsing/lexer.ml"

  | 30 ->
let
# 401 "parsing/lexer.mll"
                    stars
# 2068 "parsing/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 402 "parsing/lexer.mll"
      ( let s, loc =
          with_comment_buffer
            (fun lexbuf ->
               store_string ("*" ^ stars);
               comment lexbuf)
            lexbuf
        in
        COMMENT (s, loc) )
# 2079 "parsing/lexer.ml"

  | 31 ->
# 411 "parsing/lexer.mll"
      ( if !print_warnings then
          Location.prerr_warning (Location.curr lexbuf) Warnings.Comment_start;
        let s, loc = with_comment_buffer comment lexbuf in
        COMMENT (s, loc) )
# 2087 "parsing/lexer.ml"

  | 32 ->
let
# 415 "parsing/lexer.mll"
                   stars
# 2093 "parsing/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_curr_pos + -2) in
# 416 "parsing/lexer.mll"
      ( COMMENT (stars, Location.curr lexbuf) )
# 2097 "parsing/lexer.ml"

  | 33 ->
# 418 "parsing/lexer.mll"
      ( let loc = Location.curr lexbuf in
        Location.prerr_warning loc Warnings.Comment_not_end;
        lexbuf.Lexing.lex_curr_pos <- lexbuf.Lexing.lex_curr_pos - 1;
        let curpos = lexbuf.lex_curr_p in
        lexbuf.lex_curr_p <- { curpos with pos_cnum = curpos.pos_cnum - 1 };
        STAR
      )
# 2108 "parsing/lexer.ml"

  | 34 ->
let
# 425 "parsing/lexer.mll"
                                   num
# 2114 "parsing/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_mem.(1)
and
# 426 "parsing/lexer.mll"
                                            name
# 2119 "parsing/lexer.ml"
= Lexing.sub_lexeme_opt lexbuf lexbuf.Lexing.lex_mem.(3) lexbuf.Lexing.lex_mem.(2) in
# 428 "parsing/lexer.mll"
      ( update_loc lexbuf name (int_of_string num) true 0;
        token lexbuf
      )
# 2125 "parsing/lexer.ml"

  | 35 ->
# 431 "parsing/lexer.mll"
         ( SHARP )
# 2130 "parsing/lexer.ml"

  | 36 ->
# 432 "parsing/lexer.mll"
         ( AMPERSAND )
# 2135 "parsing/lexer.ml"

  | 37 ->
# 433 "parsing/lexer.mll"
         ( AMPERAMPER )
# 2140 "parsing/lexer.ml"

  | 38 ->
# 434 "parsing/lexer.mll"
         ( BACKQUOTE )
# 2145 "parsing/lexer.ml"

  | 39 ->
# 435 "parsing/lexer.mll"
         ( QUOTE )
# 2150 "parsing/lexer.ml"

  | 40 ->
# 436 "parsing/lexer.mll"
         ( LPAREN )
# 2155 "parsing/lexer.ml"

  | 41 ->
# 437 "parsing/lexer.mll"
         ( RPAREN )
# 2160 "parsing/lexer.ml"

  | 42 ->
# 438 "parsing/lexer.mll"
         ( STAR )
# 2165 "parsing/lexer.ml"

  | 43 ->
# 439 "parsing/lexer.mll"
         ( COMMA )
# 2170 "parsing/lexer.ml"

  | 44 ->
# 440 "parsing/lexer.mll"
         ( MINUSGREATER )
# 2175 "parsing/lexer.ml"

  | 45 ->
# 441 "parsing/lexer.mll"
         ( DOT )
# 2180 "parsing/lexer.ml"

  | 46 ->
# 442 "parsing/lexer.mll"
         ( DOTDOT )
# 2185 "parsing/lexer.ml"

  | 47 ->
# 443 "parsing/lexer.mll"
         ( COLON )
# 2190 "parsing/lexer.ml"

  | 48 ->
# 444 "parsing/lexer.mll"
         ( COLONCOLON )
# 2195 "parsing/lexer.ml"

  | 49 ->
# 445 "parsing/lexer.mll"
         ( COLONEQUAL )
# 2200 "parsing/lexer.ml"

  | 50 ->
# 446 "parsing/lexer.mll"
         ( COLONGREATER )
# 2205 "parsing/lexer.ml"

  | 51 ->
# 447 "parsing/lexer.mll"
         ( SEMI )
# 2210 "parsing/lexer.ml"

  | 52 ->
# 448 "parsing/lexer.mll"
         ( SEMISEMI )
# 2215 "parsing/lexer.ml"

  | 53 ->
# 449 "parsing/lexer.mll"
         ( LESS )
# 2220 "parsing/lexer.ml"

  | 54 ->
# 450 "parsing/lexer.mll"
         ( LESSMINUS )
# 2225 "parsing/lexer.ml"

  | 55 ->
# 451 "parsing/lexer.mll"
         ( EQUAL )
# 2230 "parsing/lexer.ml"

  | 56 ->
# 452 "parsing/lexer.mll"
         ( LBRACKET )
# 2235 "parsing/lexer.ml"

  | 57 ->
# 453 "parsing/lexer.mll"
         ( LBRACKETBAR )
# 2240 "parsing/lexer.ml"

  | 58 ->
# 454 "parsing/lexer.mll"
         ( LBRACKETLESS )
# 2245 "parsing/lexer.ml"

  | 59 ->
# 455 "parsing/lexer.mll"
         ( LBRACKETGREATER )
# 2250 "parsing/lexer.ml"

  | 60 ->
# 456 "parsing/lexer.mll"
         ( RBRACKET )
# 2255 "parsing/lexer.ml"

  | 61 ->
# 457 "parsing/lexer.mll"
         ( LBRACE )
# 2260 "parsing/lexer.ml"

  | 62 ->
# 458 "parsing/lexer.mll"
         ( LBRACELESS )
# 2265 "parsing/lexer.ml"

  | 63 ->
# 459 "parsing/lexer.mll"
         ( BAR )
# 2270 "parsing/lexer.ml"

  | 64 ->
# 460 "parsing/lexer.mll"
         ( BARBAR )
# 2275 "parsing/lexer.ml"

  | 65 ->
# 461 "parsing/lexer.mll"
         ( BARRBRACKET )
# 2280 "parsing/lexer.ml"

  | 66 ->
# 462 "parsing/lexer.mll"
         ( GREATER )
# 2285 "parsing/lexer.ml"

  | 67 ->
# 463 "parsing/lexer.mll"
         ( GREATERRBRACKET )
# 2290 "parsing/lexer.ml"

  | 68 ->
# 464 "parsing/lexer.mll"
         ( RBRACE )
# 2295 "parsing/lexer.ml"

  | 69 ->
# 465 "parsing/lexer.mll"
         ( GREATERRBRACE )
# 2300 "parsing/lexer.ml"

  | 70 ->
# 466 "parsing/lexer.mll"
         ( LBRACKETAT )
# 2305 "parsing/lexer.ml"

  | 71 ->
# 467 "parsing/lexer.mll"
           ( LBRACKETATAT )
# 2310 "parsing/lexer.ml"

  | 72 ->
# 468 "parsing/lexer.mll"
           ( LBRACKETATATAT )
# 2315 "parsing/lexer.ml"

  | 73 ->
# 469 "parsing/lexer.mll"
           ( LBRACKETPERCENT )
# 2320 "parsing/lexer.ml"

  | 74 ->
# 470 "parsing/lexer.mll"
           ( LBRACKETPERCENTPERCENT )
# 2325 "parsing/lexer.ml"

  | 75 ->
# 471 "parsing/lexer.mll"
         ( BANG )
# 2330 "parsing/lexer.ml"

  | 76 ->
# 472 "parsing/lexer.mll"
         ( INFIXOP0 "!=" )
# 2335 "parsing/lexer.ml"

  | 77 ->
# 473 "parsing/lexer.mll"
         ( PLUS )
# 2340 "parsing/lexer.ml"

  | 78 ->
# 474 "parsing/lexer.mll"
         ( PLUSDOT )
# 2345 "parsing/lexer.ml"

  | 79 ->
# 475 "parsing/lexer.mll"
         ( PLUSEQ )
# 2350 "parsing/lexer.ml"

  | 80 ->
# 476 "parsing/lexer.mll"
         ( MINUS )
# 2355 "parsing/lexer.ml"

  | 81 ->
# 477 "parsing/lexer.mll"
         ( MINUSDOT )
# 2360 "parsing/lexer.ml"

  | 82 ->
# 480 "parsing/lexer.mll"
            ( PREFIXOP(Lexing.lexeme lexbuf) )
# 2365 "parsing/lexer.ml"

  | 83 ->
# 482 "parsing/lexer.mll"
            ( PREFIXOP(Lexing.lexeme lexbuf) )
# 2370 "parsing/lexer.ml"

  | 84 ->
# 484 "parsing/lexer.mll"
            ( INFIXOP0(Lexing.lexeme lexbuf) )
# 2375 "parsing/lexer.ml"

  | 85 ->
# 486 "parsing/lexer.mll"
            ( INFIXOP1(Lexing.lexeme lexbuf) )
# 2380 "parsing/lexer.ml"

  | 86 ->
# 488 "parsing/lexer.mll"
            ( INFIXOP2(Lexing.lexeme lexbuf) )
# 2385 "parsing/lexer.ml"

  | 87 ->
# 490 "parsing/lexer.mll"
            ( INFIXOP4(Lexing.lexeme lexbuf) )
# 2390 "parsing/lexer.ml"

  | 88 ->
# 491 "parsing/lexer.mll"
            ( PERCENT )
# 2395 "parsing/lexer.ml"

  | 89 ->
# 493 "parsing/lexer.mll"
            ( INFIXOP3(Lexing.lexeme lexbuf) )
# 2400 "parsing/lexer.ml"

  | 90 ->
# 495 "parsing/lexer.mll"
            ( SHARPOP(Lexing.lexeme lexbuf) )
# 2405 "parsing/lexer.ml"

  | 91 ->
# 496 "parsing/lexer.mll"
        ( EOF )
# 2410 "parsing/lexer.ml"

  | 92 ->
# 498 "parsing/lexer.mll"
      ( raise (Error(Illegal_character (Lexing.lexeme_char lexbuf 0),
                     Location.curr lexbuf))
      )
# 2417 "parsing/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; 
      __ocaml_lex_token_rec lexbuf __ocaml_lex_state

and comment lexbuf =
    __ocaml_lex_comment_rec lexbuf 142
and __ocaml_lex_comment_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 504 "parsing/lexer.mll"
      ( comment_start_loc := (Location.curr lexbuf) :: !comment_start_loc;
        store_lexeme lexbuf;
        comment lexbuf;
      )
# 2432 "parsing/lexer.ml"

  | 1 ->
# 509 "parsing/lexer.mll"
      ( match !comment_start_loc with
        | [] -> assert false
        | [_] -> comment_start_loc := []; Location.curr lexbuf
        | _ :: l -> comment_start_loc := l;
                  store_lexeme lexbuf;
                  comment lexbuf;
       )
# 2443 "parsing/lexer.ml"

  | 2 ->
# 517 "parsing/lexer.mll"
      (
        string_start_loc := Location.curr lexbuf;
        store_string_char '\"';
        is_in_string := true;
        begin try string lexbuf
        with Error (Unterminated_string, str_start) ->
          match !comment_start_loc with
          | [] -> assert false
          | loc :: _ ->
            let start = List.hd (List.rev !comment_start_loc) in
            comment_start_loc := [];
            raise (Error (Unterminated_string_in_comment (start, str_start),
                          loc))
        end;
        is_in_string := false;
        store_string_char '\"';
        comment lexbuf )
# 2464 "parsing/lexer.ml"

  | 3 ->
# 535 "parsing/lexer.mll"
      (
        let delim = Lexing.lexeme lexbuf in
        let delim = String.sub delim 1 (String.length delim - 2) in
        string_start_loc := Location.curr lexbuf;
        store_lexeme lexbuf;
        is_in_string := true;
        begin try quoted_string delim lexbuf
        with Error (Unterminated_string, str_start) ->
          match !comment_start_loc with
          | [] -> assert false
          | loc :: _ ->
            let start = List.hd (List.rev !comment_start_loc) in
            comment_start_loc := [];
            raise (Error (Unterminated_string_in_comment (start, str_start),
                          loc))
        end;
        is_in_string := false;
        store_string_char '|';
        store_string delim;
        store_string_char '}';
        comment lexbuf )
# 2489 "parsing/lexer.ml"

  | 4 ->
# 558 "parsing/lexer.mll"
      ( store_lexeme lexbuf; comment lexbuf )
# 2494 "parsing/lexer.ml"

  | 5 ->
# 560 "parsing/lexer.mll"
      ( update_loc lexbuf None 1 false 1;
        store_lexeme lexbuf;
        comment lexbuf
      )
# 2502 "parsing/lexer.ml"

  | 6 ->
# 565 "parsing/lexer.mll"
      ( store_lexeme lexbuf; comment lexbuf )
# 2507 "parsing/lexer.ml"

  | 7 ->
# 567 "parsing/lexer.mll"
      ( store_lexeme lexbuf; comment lexbuf )
# 2512 "parsing/lexer.ml"

  | 8 ->
# 569 "parsing/lexer.mll"
      ( store_lexeme lexbuf; comment lexbuf )
# 2517 "parsing/lexer.ml"

  | 9 ->
# 571 "parsing/lexer.mll"
      ( store_lexeme lexbuf; comment lexbuf )
# 2522 "parsing/lexer.ml"

  | 10 ->
# 573 "parsing/lexer.mll"
      ( match !comment_start_loc with
        | [] -> assert false
        | loc :: _ ->
          let start = List.hd (List.rev !comment_start_loc) in
          comment_start_loc := [];
          raise (Error (Unterminated_comment start, loc))
      )
# 2533 "parsing/lexer.ml"

  | 11 ->
# 581 "parsing/lexer.mll"
      ( update_loc lexbuf None 1 false 0;
        store_lexeme lexbuf;
        comment lexbuf
      )
# 2541 "parsing/lexer.ml"

  | 12 ->
# 586 "parsing/lexer.mll"
      ( store_lexeme lexbuf; comment lexbuf )
# 2546 "parsing/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; 
      __ocaml_lex_comment_rec lexbuf __ocaml_lex_state

and string lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 2 (-1) ;   __ocaml_lex_string_rec lexbuf 174
and __ocaml_lex_string_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 590 "parsing/lexer.mll"
      ( () )
# 2558 "parsing/lexer.ml"

  | 1 ->
let
# 591 "parsing/lexer.mll"
                                  space
# 2564 "parsing/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_curr_pos in
# 592 "parsing/lexer.mll"
      ( update_loc lexbuf None 1 false (String.length space);
        if in_comment () then store_lexeme lexbuf;
        string lexbuf
      )
# 2571 "parsing/lexer.ml"

  | 2 ->
# 597 "parsing/lexer.mll"
      ( store_escaped_char lexbuf
                           (char_for_backslash(Lexing.lexeme_char lexbuf 1));
        string lexbuf )
# 2578 "parsing/lexer.ml"

  | 3 ->
# 601 "parsing/lexer.mll"
      ( store_escaped_char lexbuf (char_for_decimal_code lexbuf 1);
         string lexbuf )
# 2584 "parsing/lexer.ml"

  | 4 ->
# 604 "parsing/lexer.mll"
      ( store_escaped_char lexbuf (char_for_octal_code lexbuf 2);
         string lexbuf )
# 2590 "parsing/lexer.ml"

  | 5 ->
# 607 "parsing/lexer.mll"
      ( store_escaped_char lexbuf (char_for_hexadecimal_code lexbuf 2);
         string lexbuf )
# 2596 "parsing/lexer.ml"

  | 6 ->
# 610 "parsing/lexer.mll"
      ( if not (in_comment ()) then begin
(*  Should be an error, but we are very lax.
          raise (Error (Illegal_escape (Lexing.lexeme lexbuf),
                        Location.curr lexbuf))
*)
          let loc = Location.curr lexbuf in
          Location.prerr_warning loc Warnings.Illegal_backslash;
        end;
        store_lexeme lexbuf;
        string lexbuf
      )
# 2611 "parsing/lexer.ml"

  | 7 ->
# 622 "parsing/lexer.mll"
      ( if not (in_comment ()) then
          Location.prerr_warning (Location.curr lexbuf) Warnings.Eol_in_string;
        update_loc lexbuf None 1 false 0;
        store_lexeme lexbuf;
        string lexbuf
      )
# 2621 "parsing/lexer.ml"

  | 8 ->
# 629 "parsing/lexer.mll"
      ( is_in_string := false;
        raise (Error (Unterminated_string, !string_start_loc)) )
# 2627 "parsing/lexer.ml"

  | 9 ->
# 632 "parsing/lexer.mll"
      ( store_string_char(Lexing.lexeme_char lexbuf 0);
        string lexbuf )
# 2633 "parsing/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; 
      __ocaml_lex_string_rec lexbuf __ocaml_lex_state

and quoted_string delim lexbuf =
    __ocaml_lex_quoted_string_rec delim lexbuf 197
and __ocaml_lex_quoted_string_rec delim lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 637 "parsing/lexer.mll"
      ( update_loc lexbuf None 1 false 0;
        store_lexeme lexbuf;
        quoted_string delim lexbuf
      )
# 2648 "parsing/lexer.ml"

  | 1 ->
# 642 "parsing/lexer.mll"
      ( is_in_string := false;
        raise (Error (Unterminated_string, !string_start_loc)) )
# 2654 "parsing/lexer.ml"

  | 2 ->
# 645 "parsing/lexer.mll"
      (
        let edelim = Lexing.lexeme lexbuf in
        let edelim = String.sub edelim 1 (String.length edelim - 2) in
        if delim = edelim then ()
        else (store_lexeme lexbuf; quoted_string delim lexbuf)
      )
# 2664 "parsing/lexer.ml"

  | 3 ->
# 652 "parsing/lexer.mll"
      ( store_string_char(Lexing.lexeme_char lexbuf 0);
        quoted_string delim lexbuf )
# 2670 "parsing/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; 
      __ocaml_lex_quoted_string_rec delim lexbuf __ocaml_lex_state

and skip_sharp_bang lexbuf =
    __ocaml_lex_skip_sharp_bang_rec lexbuf 206
and __ocaml_lex_skip_sharp_bang_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 657 "parsing/lexer.mll"
       ( update_loc lexbuf None 3 false 0 )
# 2682 "parsing/lexer.ml"

  | 1 ->
# 659 "parsing/lexer.mll"
       ( update_loc lexbuf None 1 false 0 )
# 2687 "parsing/lexer.ml"

  | 2 ->
# 660 "parsing/lexer.mll"
       ( () )
# 2692 "parsing/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; 
      __ocaml_lex_skip_sharp_bang_rec lexbuf __ocaml_lex_state

;;

# 662 "parsing/lexer.mll"
 

  let token_with_comments lexbuf =
    match !preprocessor with
    | None -> token lexbuf
    | Some (_init, preprocess) -> preprocess token lexbuf

  type newline_state =
    | NoLine (* There have been no blank lines yet. *)
    | NewLine
        (* There have been no blank lines, and the previous
           token was a newline. *)
    | BlankLine (* There have been blank lines. *)

  type doc_state =
    | Initial  (* There have been no docstrings yet *)
    | After of docstring list
        (* There have been docstrings, none of which were
           preceeded by a blank line *)
    | Before of docstring list * docstring list * docstring list
        (* There have been docstrings, some of which were
           preceeded by a blank line *)

  and docstring = Docstrings.docstring

  let token lexbuf =
    let post_pos = lexeme_end_p lexbuf in
    let attach lines docs pre_pos =
      let open Docstrings in
        match docs, lines with
        | Initial, _ -> ()
        | After a, (NoLine | NewLine) ->
            set_post_docstrings post_pos (List.rev a);
            set_pre_docstrings pre_pos a;
        | After a, BlankLine ->
            set_post_docstrings post_pos (List.rev a);
            set_pre_extra_docstrings pre_pos (List.rev a)
        | Before(a, f, b), (NoLine | NewLine) ->
            set_post_docstrings post_pos (List.rev a);
            set_post_extra_docstrings post_pos
              (List.rev_append f (List.rev b));
            set_floating_docstrings pre_pos (List.rev f);
            set_pre_extra_docstrings pre_pos (List.rev a);
            set_pre_docstrings pre_pos b
        | Before(a, f, b), BlankLine ->
            set_post_docstrings post_pos (List.rev a);
            set_post_extra_docstrings post_pos
              (List.rev_append f (List.rev b));
            set_floating_docstrings pre_pos
              (List.rev_append f (List.rev b));
            set_pre_extra_docstrings pre_pos (List.rev a)
    in
    let rec loop lines docs lexbuf =
      match token_with_comments lexbuf with
      | COMMENT (s, loc) ->
          add_comment (s, loc);
          let lines' =
            match lines with
            | NoLine -> NoLine
            | NewLine -> NoLine
            | BlankLine -> BlankLine
          in
          loop lines' docs lexbuf
      | EOL ->
          let lines' =
            match lines with
            | NoLine -> NewLine
            | NewLine -> BlankLine
            | BlankLine -> BlankLine
          in
          loop lines' docs lexbuf
      | DOCSTRING doc ->
          add_docstring_comment doc;
          let docs' =
            match docs, lines with
            | Initial, (NoLine | NewLine) -> After [doc]
            | Initial, BlankLine -> Before([], [], [doc])
            | After a, (NoLine | NewLine) -> After (doc :: a)
            | After a, BlankLine -> Before (a, [], [doc])
            | Before(a, f, b), (NoLine | NewLine) -> Before(a, f, doc :: b)
            | Before(a, f, b), BlankLine -> Before(a, b @ f, [doc])
          in
          loop NoLine docs' lexbuf
      | tok ->
          attach lines docs (lexeme_start_p lexbuf);
          tok
    in
      loop NoLine Initial lexbuf

  let init () =
    is_in_string := false;
    comment_start_loc := [];
    comment_list := [];
    match !preprocessor with
    | None -> ()
    | Some (init, _preprocess) -> init ()

  let set_preprocessor init preprocess =
    escaped_newlines := true;
    preprocessor := Some (init, preprocess)


# 2802 "parsing/lexer.ml"
