/*
 * Copyright 2010-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package kotlin.reflect.jvm.internal.impl.load.kotlin

import kotlin.reflect.jvm.internal.impl.descriptors.ClassDescriptor
import kotlin.reflect.jvm.internal.impl.descriptors.PackageFragmentDescriptor
import kotlin.reflect.jvm.internal.impl.load.kotlin.header.KotlinClassHeader
import kotlin.reflect.jvm.internal.impl.protobuf.InvalidProtocolBufferException
import kotlin.reflect.jvm.internal.impl.resolve.scopes.MemberScope
import kotlin.reflect.jvm.internal.impl.serialization.ClassDataWithSource
import kotlin.reflect.jvm.internal.impl.serialization.deserialization.DeserializationComponents
import kotlin.reflect.jvm.internal.impl.serialization.deserialization.ErrorReporter
import kotlin.reflect.jvm.internal.impl.serialization.deserialization.descriptors.DeserializedPackageMemberScope
import kotlin.reflect.jvm.internal.impl.serialization.jvm.JvmProtoBufUtil
import kotlin.reflect.jvm.internal.impl.utils.sure
import javax.inject.Inject

class DeserializedDescriptorResolver(private val errorReporter: ErrorReporter) {
    lateinit var components: DeserializationComponents

    // component dependency cycle
    @Inject
    fun setComponents(components: DeserializationComponentsForJava) {
        this.components = components.components
    }

    fun resolveClass(kotlinClass: KotlinJvmBinaryClass): ClassDescriptor? {
        val data = readData(kotlinClass, KOTLIN_CLASS) ?: return null
        val strings = kotlinClass.classHeader.strings.sure { "String table not found in $kotlinClass" }
        val classData = parseProto(kotlinClass) {
            JvmProtoBufUtil.readClassDataFrom(data, strings)
        }
        val sourceElement = KotlinJvmBinarySourceElement(kotlinClass)
        return components.classDeserializer.deserializeClass(
                kotlinClass.classId,
                ClassDataWithSource(classData, sourceElement)
        )
    }

    fun createKotlinPackagePartScope(descriptor: PackageFragmentDescriptor, kotlinClass: KotlinJvmBinaryClass): MemberScope? {
        val data = readData(kotlinClass, KOTLIN_FILE_FACADE_OR_MULTIFILE_CLASS_PART) ?: return null
        val strings = kotlinClass.classHeader.strings.sure { "String table not found in $kotlinClass" }
        val (nameResolver, packageProto) = parseProto(kotlinClass) {
            JvmProtoBufUtil.readPackageDataFrom(data, strings)
        }
        val source = JvmPackagePartSource(kotlinClass)
        return DeserializedPackageMemberScope(descriptor, packageProto, nameResolver, source, components) {
            // All classes are included into Java scope
            emptyList()
        }
    }

    internal fun readData(kotlinClass: KotlinJvmBinaryClass, expectedKinds: Set<KotlinClassHeader.Kind>): Array<String>? {
        val header = kotlinClass.classHeader
        if (!header.metadataVersion.isCompatible()) {
            errorReporter.reportIncompatibleMetadataVersion(kotlinClass.classId, kotlinClass.location, header.metadataVersion)
        }
        else if (expectedKinds.contains(header.kind)) {
            return header.data
        }

        return null
    }

    private inline fun <T> parseProto(klass: KotlinJvmBinaryClass, block: () -> T): T {
        try {
            return block()
        }
        catch (e: InvalidProtocolBufferException) {
            throw IllegalStateException("Could not read data from ${klass.location}", e)
        }
    }

    companion object {
        internal val KOTLIN_CLASS = setOf(KotlinClassHeader.Kind.CLASS)

        private val KOTLIN_FILE_FACADE_OR_MULTIFILE_CLASS_PART =
                setOf(KotlinClassHeader.Kind.FILE_FACADE, KotlinClassHeader.Kind.MULTIFILE_CLASS_PART)
    }
}
