package Lire::Report::ChartConfig;

use strict;

use base qw/ Lire::Config::Dictionary /;

use Lire::Utils qw/check_object_param/;

use Carp;

=pod

=head1 NAME

Lire::Report::ChartConfig - Chart configuration object.

=head1 SYNOPSIS

    use Lire::Report::ChartConfig;

    my $chart = new Lire::Report::ChartConfig();
    $chart->title( 'Downloads by Visits Scatterplot' )
    $chart->type( Lire::PluginManager->get_plugin( 'chart_type', 'scatterplot' );
    ...

=head1 DESCRIPTION

This is a Lire::Config::Dictionary subclass which is used to define
a chart that will be generated from a Lire::Report::Subreport.

=head1 CONSTRUCTOR

=head2 new()

Creates a new Lire::Report::ChartConfig object.

=head1 ATTRIBUTES

These are methods that wraps around the Lire::Config::Dictionary get()
and set() method.

=head2 basename( [ $new_basename ] )

Returns (and optionnally changes) the Chart's basename.

=head2 title( [ $new_title ] )

Returns (and optionnally changes) the Chart's title.

=head2 type( [ $new_type ] )

Returns (and optionally changes) the ChartType plugin used by this 
Config.

=head2 type_properties()

Returns the Lire::Config::Value object used to store the ChartType's
specific properties.

=head2 case_var()

Returns the Lire::Report::ColumnInfo object that is use to select the
cases that will be part of the chart's data.

=head2 xlabel( [ $new_xlabel ] )

Returns (and optionnally change) the Chart's X axis label..

=head2 ylabel( [ $new_ylabel ] )

Returns (and optionnally change) the Chart's Y axis label..

=cut

sub new {
    my ( $class, %params ) = @_;

    $params{'spec'} ||= new Lire::Config::ChartSpec( 'name' => 'chart' );
    check_object_param( $params{'spec'}, 'spec', 'Lire::Config::ChartSpec' );

    return $class->SUPER::new( %params );
}

sub basename {
    my ( $self, $new_name ) = @_;

    if ( @_ == 2 ) {
        $self->get( 'name' )->set( $new_name );
    }
    return $self->get( 'name' )->as_value();
}

sub title {
    my ( $self, $new_title ) = @_;

    if ( @_ == 2 ) {
        $self->get( 'title' )->set( $new_title );
    }
    return $self->get( 'title' )->as_value();

}

sub xlabel {
    my ( $self, $new_xlabel ) = @_;

    if ( @_ == 2 ) {
        $self->get( 'xlabel' )->set( $new_xlabel );
    }
    return $self->get( 'xlabel' )->as_value();

}

sub ylabel {
    my ( $self, $new_ylabel ) = @_;

    if ( @_ == 2 ) {
        $self->get( 'ylabel' )->set( $new_ylabel );
    }
    return $self->get( 'ylabel' )->as_value();

}

sub type {
    my ( $self, $new_type ) = @_;

    if ( @_ == 2 ) {
        check_object_param( $new_type, 'type', 'Lire::ChartType' );
        $self->get( 'type' )->set_plugin( $new_type->name() );
    }
    return Lire::PluginManager->get_plugin( 'chart_type',
                                            $self->get( 'type' )->get_plugin() );
}

sub type_properties {
    my $self = $_[0];

    return $self->get( 'type' )->get_properties();
}

sub case_var {
    my $self = $_[0];

    return $self->get( 'case_var' )->as_value();
}

sub as_value {
    return $_[0];
}

1;

__END__

=pod

=head1 SEE ALSO

  Lire::Config::ChartSpec(3pm) Lire::ChartType(3pm)

=head1 VERSION

$Id: ChartConfig.pm,v 1.7 2006/07/23 13:16:31 vanbaal Exp $

=head1 COPYRIGHT

Copyright (C) 2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html. 

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut


