/****************************************************************************
**
*W  code.h                      GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file declares the functions of the coder package.
**
**  The  coder package  is   the part of   the interpreter  that creates  the
**  expressions.  Its functions are called from the reader.
*/

#ifndef libGAP_GAP_CODE_H
#define libGAP_GAP_CODE_H


/****************************************************************************
**
*T  Stat  . . . . . . . . . . . . . . . . . . . . . . . .  type of statements
**
**  'Stat' is the type of statements.
**
**  If 'Stat' is different  from 'Expr', then  a lot of things will  probably
**  break.
*/
#define libGAP_Stat            libGAP_UInt8


/****************************************************************************
**
*V  PtrBody . . . . . . . . . . . . . . . . . . . . . pointer to current body
**
**  'PtrBody' is a pointer to the current body.
*/
extern  libGAP_Stat *          libGAP_PtrBody;


/****************************************************************************
**
*V  FIRST_STAT_CURR_FUNC  . . . . . . . .  index of first statement in a body
**
**  'FIRST_STAT_CURR_FUNC' is the index of the first statement in a body.
*/


#define libGAP_FILENAME_BODY(body) (libGAP_ADDR_OBJ(body)[0])
#define libGAP_STARTLINE_BODY(body) (libGAP_ADDR_OBJ(body)[1])
#define libGAP_ENDLINE_BODY(body) (libGAP_ADDR_OBJ(body)[2])
#define libGAP_NUMBER_HEADER_ITEMS_BODY 3

#define libGAP_FIRST_STAT_CURR_FUNC    (sizeof(libGAP_Stat)+libGAP_NUMBER_HEADER_ITEMS_BODY*sizeof(libGAP_Bag))

/****************************************************************************
**
*S  T_<name>  . . . . . . . . . . . . . .  symbolic names for statement types
*S  FIRST_STAT_TNUM . . . . . . . . . . . . . . . . . .  first statement type
*S  LAST_STAT_TNUM  . . . . . . . . . . . . . . . . . . . last statement type
**
**  For every type  of statements there is  a symbolic name  defined for this
**  type.
**
**  As long as statements   are represented by  bags,  these types  must  not
**  overlap with the object types, lest Gasman becomes confused.
*/
#define libGAP_FIRST_STAT_TNUM         (0UL)

#define libGAP_T_PROCCALL_0ARGS        (libGAP_FIRST_STAT_TNUM+ 0)
#define libGAP_T_PROCCALL_1ARGS        (libGAP_FIRST_STAT_TNUM+ 1)
#define libGAP_T_PROCCALL_2ARGS        (libGAP_FIRST_STAT_TNUM+ 2)
#define libGAP_T_PROCCALL_3ARGS        (libGAP_FIRST_STAT_TNUM+ 3)
#define libGAP_T_PROCCALL_4ARGS        (libGAP_FIRST_STAT_TNUM+ 4)
#define libGAP_T_PROCCALL_5ARGS        (libGAP_FIRST_STAT_TNUM+ 5)
#define libGAP_T_PROCCALL_6ARGS        (libGAP_FIRST_STAT_TNUM+ 6)
#define libGAP_T_PROCCALL_XARGS        (libGAP_FIRST_STAT_TNUM+ 7)

#define libGAP_T_SEQ_STAT              (libGAP_FIRST_STAT_TNUM+ 8)
#define libGAP_T_SEQ_STAT2             (libGAP_FIRST_STAT_TNUM+ 9)
#define libGAP_T_SEQ_STAT3             (libGAP_FIRST_STAT_TNUM+10)
#define libGAP_T_SEQ_STAT4             (libGAP_FIRST_STAT_TNUM+11)
#define libGAP_T_SEQ_STAT5             (libGAP_FIRST_STAT_TNUM+12)
#define libGAP_T_SEQ_STAT6             (libGAP_FIRST_STAT_TNUM+13)
#define libGAP_T_SEQ_STAT7             (libGAP_FIRST_STAT_TNUM+14)
#define libGAP_T_IF                    (libGAP_FIRST_STAT_TNUM+15)
#define libGAP_T_IF_ELSE               (libGAP_FIRST_STAT_TNUM+16)
#define libGAP_T_IF_ELIF               (libGAP_FIRST_STAT_TNUM+17)
#define libGAP_T_IF_ELIF_ELSE          (libGAP_FIRST_STAT_TNUM+18)
#define libGAP_T_FOR                   (libGAP_FIRST_STAT_TNUM+19)
#define libGAP_T_FOR2                  (libGAP_FIRST_STAT_TNUM+20)
#define libGAP_T_FOR3                  (libGAP_FIRST_STAT_TNUM+21)
#define libGAP_T_FOR_RANGE             (libGAP_FIRST_STAT_TNUM+22)
#define libGAP_T_FOR_RANGE2            (libGAP_FIRST_STAT_TNUM+23)
#define libGAP_T_FOR_RANGE3            (libGAP_FIRST_STAT_TNUM+24)
#define libGAP_T_WHILE                 (libGAP_FIRST_STAT_TNUM+25)
#define libGAP_T_WHILE2                (libGAP_FIRST_STAT_TNUM+26)
#define libGAP_T_WHILE3                (libGAP_FIRST_STAT_TNUM+27)
#define libGAP_T_REPEAT                (libGAP_FIRST_STAT_TNUM+28)
#define libGAP_T_REPEAT2               (libGAP_FIRST_STAT_TNUM+29)
#define libGAP_T_REPEAT3               (libGAP_FIRST_STAT_TNUM+30)
#define libGAP_T_BREAK                 (libGAP_FIRST_STAT_TNUM+31)
#define libGAP_T_CONTINUE              (libGAP_FIRST_STAT_TNUM+32)
#define libGAP_T_RETURN_OBJ            (libGAP_FIRST_STAT_TNUM+33)
#define libGAP_T_RETURN_VOID           (libGAP_FIRST_STAT_TNUM+34)

#define libGAP_T_ASS_LVAR              (libGAP_FIRST_STAT_TNUM+35)
#define libGAP_T_ASS_LVAR_01           (libGAP_FIRST_STAT_TNUM+36)
#define libGAP_T_ASS_LVAR_02           (libGAP_FIRST_STAT_TNUM+37)
#define libGAP_T_ASS_LVAR_03           (libGAP_FIRST_STAT_TNUM+38)
#define libGAP_T_ASS_LVAR_04           (libGAP_FIRST_STAT_TNUM+39)
#define libGAP_T_ASS_LVAR_05           (libGAP_FIRST_STAT_TNUM+40)
#define libGAP_T_ASS_LVAR_06           (libGAP_FIRST_STAT_TNUM+41)
#define libGAP_T_ASS_LVAR_07           (libGAP_FIRST_STAT_TNUM+42)
#define libGAP_T_ASS_LVAR_08           (libGAP_FIRST_STAT_TNUM+43)
#define libGAP_T_ASS_LVAR_09           (libGAP_FIRST_STAT_TNUM+44)
#define libGAP_T_ASS_LVAR_10           (libGAP_FIRST_STAT_TNUM+45)
#define libGAP_T_ASS_LVAR_11           (libGAP_FIRST_STAT_TNUM+46)
#define libGAP_T_ASS_LVAR_12           (libGAP_FIRST_STAT_TNUM+47)
#define libGAP_T_ASS_LVAR_13           (libGAP_FIRST_STAT_TNUM+48)
#define libGAP_T_ASS_LVAR_14           (libGAP_FIRST_STAT_TNUM+49)
#define libGAP_T_ASS_LVAR_15           (libGAP_FIRST_STAT_TNUM+50)
#define libGAP_T_ASS_LVAR_16           (libGAP_FIRST_STAT_TNUM+51)
#define libGAP_T_UNB_LVAR              (libGAP_FIRST_STAT_TNUM+52)
#define libGAP_T_ASS_HVAR              (libGAP_FIRST_STAT_TNUM+53)
#define libGAP_T_UNB_HVAR              (libGAP_FIRST_STAT_TNUM+54)
#define libGAP_T_ASS_GVAR              (libGAP_FIRST_STAT_TNUM+55)
#define libGAP_T_UNB_GVAR              (libGAP_FIRST_STAT_TNUM+56)
#define libGAP_T_ASS_LIST              (libGAP_FIRST_STAT_TNUM+57)
#define libGAP_T_ASSS_LIST             (libGAP_FIRST_STAT_TNUM+58)
#define libGAP_T_ASS_LIST_LEV          (libGAP_FIRST_STAT_TNUM+59)
#define libGAP_T_ASSS_LIST_LEV         (libGAP_FIRST_STAT_TNUM+60)
#define libGAP_T_UNB_LIST              (libGAP_FIRST_STAT_TNUM+61)
#define libGAP_T_ASS_REC_NAME          (libGAP_FIRST_STAT_TNUM+62)
#define libGAP_T_ASS_REC_EXPR          (libGAP_FIRST_STAT_TNUM+63)
#define libGAP_T_UNB_REC_NAME          (libGAP_FIRST_STAT_TNUM+64)
#define libGAP_T_UNB_REC_EXPR          (libGAP_FIRST_STAT_TNUM+65)
#define libGAP_T_ASS_POSOBJ            (libGAP_FIRST_STAT_TNUM+66)
#define libGAP_T_ASSS_POSOBJ           (libGAP_FIRST_STAT_TNUM+67)
#define libGAP_T_ASS_POSOBJ_LEV        (libGAP_FIRST_STAT_TNUM+68)
#define libGAP_T_ASSS_POSOBJ_LEV       (libGAP_FIRST_STAT_TNUM+69)
#define libGAP_T_UNB_POSOBJ            (libGAP_FIRST_STAT_TNUM+70)
#define libGAP_T_ASS_COMOBJ_NAME       (libGAP_FIRST_STAT_TNUM+71)
#define libGAP_T_ASS_COMOBJ_EXPR       (libGAP_FIRST_STAT_TNUM+72)
#define libGAP_T_UNB_COMOBJ_NAME       (libGAP_FIRST_STAT_TNUM+73)
#define libGAP_T_UNB_COMOBJ_EXPR       (libGAP_FIRST_STAT_TNUM+74)

#define libGAP_T_INFO                  (libGAP_FIRST_STAT_TNUM+75)
#define libGAP_T_ASSERT_2ARGS          (libGAP_FIRST_STAT_TNUM+76)
#define libGAP_T_ASSERT_3ARGS          (libGAP_FIRST_STAT_TNUM+77)

#define libGAP_T_EMPTY                 (libGAP_FIRST_STAT_TNUM+78)

#define libGAP_T_PROCCALL_OPTS         (libGAP_FIRST_STAT_TNUM+ 79)

#define libGAP_T_ATOMIC               (libGAP_FIRST_STAT_TNUM+80)

#define libGAP_LAST_STAT_TNUM          libGAP_T_ATOMIC

#define libGAP_T_NO_STAT		(libGAP_Stat)(-1)


/****************************************************************************
**
*F  TNUM_STAT(<stat>) . . . . . . . . . . . . . . . . . . type of a statement
**
**  'TNUM_STAT' returns the type of the statement <stat>.
*/
#define libGAP_TNUM_STAT(stat) ((libGAP_Int)(libGAP_ADDR_STAT(stat)[-1] & 0xFF))


/****************************************************************************
**
*F  SIZE_STAT(<stat>) . . . . . . . . . . . . . . . . . . size of a statement
**
**  'SIZE_STAT' returns the size of the statement <stat>.
*/
#define libGAP_SIZE_STAT(stat) ((libGAP_Int)(libGAP_ADDR_STAT(stat)[-1] >> 8 & 0xFFFFFF))

/****************************************************************************
**
*F  LINE_STAT(<stat>) . . . . . . . . . . . . . . line number of a statement
**
**  'LINE_STAT' returns the line number of the statement <stat>.
*/
#define libGAP_LINE_STAT(stat) ((libGAP_Int)(libGAP_ADDR_STAT(stat)[-1] >> 32 & 0xFFFF))

/****************************************************************************
**
*F  FILENAMEID_STAT(<stat>) . . . . . . . . . . . . file name of a statement
**
**  'FILENAMEID_STAT' returns the file the statment <stat> was read from.
**  This should be looked up in the FilenameCache variable
*/
#define libGAP_FILENAMEID_STAT(stat) ((libGAP_Int)(libGAP_ADDR_STAT(stat)[-1] >> 48 & 0x7FFF))

/****************************************************************************
**
*F  FILENAME_STAT(<stat>) . . . . . . . . . . . . file name of a statement
**
**  'FILENAME_STAT' returns a gap string containing the file where the statment
**  <stat> was read from.
*/
libGAP_Obj libGAP_FILENAME_STAT(libGAP_Stat stat);

/****************************************************************************
**
*F  VISITED_STAT(<stat>) . . . . . . . . . . . . if statement has even been run
**
**  'VISITED_STAT' returns true if the statement has ever been executed
**  while profiling is turned on.
*/
#define libGAP_VISITED_STAT(stat) (libGAP_ADDR_STAT(stat)[-1] >> 63 && 0x1)



/****************************************************************************
**
*F  ADDR_STAT(<stat>) . . . . . . . . . . . . absolute address of a statement
**
**  'ADDR_STAT' returns   the  absolute address of the    memory block of the
**  statement <stat>.
*/
#define libGAP_ADDR_STAT(stat) ((libGAP_Stat*)(((char*)libGAP_TLS(libGAP_PtrBody))+(stat)))


/****************************************************************************
**
*T  Expr  . . . . . . . . . . . . . . . . . . . . . . . . type of expressions
**
**  'Expr' is the type of expressions.
**
**  If 'Expr' is different  from 'Stat', then  a lot of things will  probably
**  break.
*/
#define libGAP_Expr            libGAP_Stat


/****************************************************************************
**
*F  IS_REFLVAR(<expr>). . . . test if an expression is a reference to a local
*F  REFLVAR_LVAR(<lvar>)  . . . . . convert a local to a reference to a local
*F  LVAR_REFLVAR(<expr>)  . . . . . convert a reference to a local to a local
**
**  'IS_REFLVAR'  returns  1  if  the  expression <expr>  is  an  (immediate)
**  reference to a local variable, and 0 otherwise.
**
**  'REFLVAR_LVAR'  returns  a (immediate) reference  to   the local variable
**  <lvar> (given by its index).
**
**  'LVAR_REFLVAR' returns the local variable (by  its index) to which <expr>
**  is a (immediate) reference.
*/
#define libGAP_IS_REFLVAR(expr)        \
                        (((libGAP_Int)(expr) & 0x03) == 0x03)

#define libGAP_REFLVAR_LVAR(lvar)      \
                        ((libGAP_Expr)(((lvar) << 2) + 0x03))

#define libGAP_LVAR_REFLVAR(expr)      \
                        ((libGAP_Int)(expr) >> 2)


/****************************************************************************
**
*F  IS_INTEXPR(<expr>). . . .  test if an expression is an integer expression
*F  INTEXPR_INT(<i>)  . . . . .  convert a C integer to an integer expression
*F  INT_INTEXPR(<expr>) . . . .  convert an integer expression to a C integer
**
**  'IS_INTEXPR' returns 1 if the expression <expr> is an (immediate) integer
**  expression, and 0 otherwise.
**
**  'INTEXPR_INT' converts    the C integer <i>    to  an (immediate) integer
**  expression.
**
**  'INT_INTEXPR' converts the (immediate) integer  expression <expr> to a  C
**  integer.
*/
#define libGAP_IS_INTEXPR(expr)        \
                        (((libGAP_Int)(expr) & 0x03) == 0x01)

#define libGAP_INTEXPR_INT(indx)       \
                        ((libGAP_Expr)(((libGAP_Int)(indx) << 2) + 0x01))

#define libGAP_INT_INTEXPR(expr)       \
                        (((libGAP_Int)(expr)-0x01) >> 2)


/****************************************************************************
**
*S  T_<name>  . . . . . . . . . . . . . . symbolic names for expression types
*S  FIRST_EXPR_TNUM . . . . . . . . . . . . . . . . . . first expression type
*S  LAST_EXPR_TNUM  . . . . . . . . . . . . . . . . . .  last expression type
**
**  For every type of expressions there  is a symbolic  name defined for this
**  type.
**
**  As long as  expressions  are represented by  bags,  these types must  not
**  overlap with the object types, lest Gasman becomes confused.
*/
#define libGAP_FIRST_EXPR_TNUM         ((libGAP_UInt)128)

#define libGAP_T_FUNCCALL_0ARGS        (libGAP_FIRST_EXPR_TNUM+ 0)
#define libGAP_T_FUNCCALL_1ARGS        (libGAP_FIRST_EXPR_TNUM+ 1)
#define libGAP_T_FUNCCALL_2ARGS        (libGAP_FIRST_EXPR_TNUM+ 2)
#define libGAP_T_FUNCCALL_3ARGS        (libGAP_FIRST_EXPR_TNUM+ 3)
#define libGAP_T_FUNCCALL_4ARGS        (libGAP_FIRST_EXPR_TNUM+ 4)
#define libGAP_T_FUNCCALL_5ARGS        (libGAP_FIRST_EXPR_TNUM+ 5)
#define libGAP_T_FUNCCALL_6ARGS        (libGAP_FIRST_EXPR_TNUM+ 6)
#define libGAP_T_FUNCCALL_XARGS        (libGAP_FIRST_EXPR_TNUM+ 7)
#define libGAP_T_FUNC_EXPR             (libGAP_FIRST_EXPR_TNUM+ 8)

#define libGAP_T_OR                    (libGAP_FIRST_EXPR_TNUM+ 9)
#define libGAP_T_AND                   (libGAP_FIRST_EXPR_TNUM+10)
#define libGAP_T_NOT                   (libGAP_FIRST_EXPR_TNUM+11)
#define libGAP_T_EQ                    (libGAP_FIRST_EXPR_TNUM+12)
#define libGAP_T_NE                    (libGAP_FIRST_EXPR_TNUM+13)
#define libGAP_T_LT                    (libGAP_FIRST_EXPR_TNUM+14)
#define libGAP_T_GE                    (libGAP_FIRST_EXPR_TNUM+15)
#define libGAP_T_GT                    (libGAP_FIRST_EXPR_TNUM+16)
#define libGAP_T_LE                    (libGAP_FIRST_EXPR_TNUM+17)
#define libGAP_T_IN                    (libGAP_FIRST_EXPR_TNUM+18)
#define libGAP_T_SUM                   (libGAP_FIRST_EXPR_TNUM+19)
#define libGAP_T_AINV                  (libGAP_FIRST_EXPR_TNUM+20)
#define libGAP_T_DIFF                  (libGAP_FIRST_EXPR_TNUM+21)
#define libGAP_T_PROD                  (libGAP_FIRST_EXPR_TNUM+22)
#define libGAP_T_INV                   (libGAP_FIRST_EXPR_TNUM+23)
#define libGAP_T_QUO                   (libGAP_FIRST_EXPR_TNUM+24)
#define libGAP_T_MOD                   (libGAP_FIRST_EXPR_TNUM+25)
#define libGAP_T_POW                   (libGAP_FIRST_EXPR_TNUM+26)

#define libGAP_T_INTEXPR               (libGAP_FIRST_EXPR_TNUM+27)
#define libGAP_T_INT_EXPR              (libGAP_FIRST_EXPR_TNUM+28)
#define libGAP_T_TRUE_EXPR             (libGAP_FIRST_EXPR_TNUM+29)
#define libGAP_T_FALSE_EXPR            (libGAP_FIRST_EXPR_TNUM+30)
#define libGAP_T_CHAR_EXPR             (libGAP_FIRST_EXPR_TNUM+31)
#define libGAP_T_PERM_EXPR             (libGAP_FIRST_EXPR_TNUM+32)
#define libGAP_T_PERM_CYCLE            (libGAP_FIRST_EXPR_TNUM+33)
#define libGAP_T_LIST_EXPR             (libGAP_FIRST_EXPR_TNUM+34)
#define libGAP_T_LIST_TILD_EXPR        (libGAP_FIRST_EXPR_TNUM+35)
#define libGAP_T_RANGE_EXPR            (libGAP_FIRST_EXPR_TNUM+36)
#define libGAP_T_STRING_EXPR           (libGAP_FIRST_EXPR_TNUM+37)
#define libGAP_T_REC_EXPR              (libGAP_FIRST_EXPR_TNUM+38)
#define libGAP_T_REC_TILD_EXPR         (libGAP_FIRST_EXPR_TNUM+39)

#define libGAP_T_REFLVAR               (libGAP_FIRST_EXPR_TNUM+40)
#define libGAP_T_REF_LVAR              (libGAP_FIRST_EXPR_TNUM+41)
#define libGAP_T_REF_LVAR_01           (libGAP_FIRST_EXPR_TNUM+42)
#define libGAP_T_REF_LVAR_02           (libGAP_FIRST_EXPR_TNUM+43)
#define libGAP_T_REF_LVAR_03           (libGAP_FIRST_EXPR_TNUM+44)
#define libGAP_T_REF_LVAR_04           (libGAP_FIRST_EXPR_TNUM+45)
#define libGAP_T_REF_LVAR_05           (libGAP_FIRST_EXPR_TNUM+46)
#define libGAP_T_REF_LVAR_06           (libGAP_FIRST_EXPR_TNUM+47)
#define libGAP_T_REF_LVAR_07           (libGAP_FIRST_EXPR_TNUM+48)
#define libGAP_T_REF_LVAR_08           (libGAP_FIRST_EXPR_TNUM+49)
#define libGAP_T_REF_LVAR_09           (libGAP_FIRST_EXPR_TNUM+50)
#define libGAP_T_REF_LVAR_10           (libGAP_FIRST_EXPR_TNUM+51)
#define libGAP_T_REF_LVAR_11           (libGAP_FIRST_EXPR_TNUM+52)
#define libGAP_T_REF_LVAR_12           (libGAP_FIRST_EXPR_TNUM+53)
#define libGAP_T_REF_LVAR_13           (libGAP_FIRST_EXPR_TNUM+54)
#define libGAP_T_REF_LVAR_14           (libGAP_FIRST_EXPR_TNUM+55)
#define libGAP_T_REF_LVAR_15           (libGAP_FIRST_EXPR_TNUM+56)
#define libGAP_T_REF_LVAR_16           (libGAP_FIRST_EXPR_TNUM+57)
#define libGAP_T_ISB_LVAR              (libGAP_FIRST_EXPR_TNUM+58)
#define libGAP_T_REF_HVAR              (libGAP_FIRST_EXPR_TNUM+59)
#define libGAP_T_ISB_HVAR              (libGAP_FIRST_EXPR_TNUM+60)
#define libGAP_T_REF_GVAR              (libGAP_FIRST_EXPR_TNUM+61)
#define libGAP_T_ISB_GVAR              (libGAP_FIRST_EXPR_TNUM+62)
#define libGAP_T_ELM_LIST              (libGAP_FIRST_EXPR_TNUM+63)
#define libGAP_T_ELMS_LIST             (libGAP_FIRST_EXPR_TNUM+64)
#define libGAP_T_ELM_LIST_LEV          (libGAP_FIRST_EXPR_TNUM+65)
#define libGAP_T_ELMS_LIST_LEV         (libGAP_FIRST_EXPR_TNUM+66)
#define libGAP_T_ISB_LIST              (libGAP_FIRST_EXPR_TNUM+67)
#define libGAP_T_ELM_REC_NAME          (libGAP_FIRST_EXPR_TNUM+68)
#define libGAP_T_ELM_REC_EXPR          (libGAP_FIRST_EXPR_TNUM+69)
#define libGAP_T_ISB_REC_NAME          (libGAP_FIRST_EXPR_TNUM+70)
#define libGAP_T_ISB_REC_EXPR          (libGAP_FIRST_EXPR_TNUM+71)
#define libGAP_T_ELM_POSOBJ            (libGAP_FIRST_EXPR_TNUM+72)
#define libGAP_T_ELMS_POSOBJ           (libGAP_FIRST_EXPR_TNUM+73)
#define libGAP_T_ELM_POSOBJ_LEV        (libGAP_FIRST_EXPR_TNUM+74)
#define libGAP_T_ELMS_POSOBJ_LEV       (libGAP_FIRST_EXPR_TNUM+75)
#define libGAP_T_ISB_POSOBJ            (libGAP_FIRST_EXPR_TNUM+76)
#define libGAP_T_ELM_COMOBJ_NAME       (libGAP_FIRST_EXPR_TNUM+77)
#define libGAP_T_ELM_COMOBJ_EXPR       (libGAP_FIRST_EXPR_TNUM+78)
#define libGAP_T_ISB_COMOBJ_NAME       (libGAP_FIRST_EXPR_TNUM+79)
#define libGAP_T_ISB_COMOBJ_EXPR       (libGAP_FIRST_EXPR_TNUM+80)

#define libGAP_T_FUNCCALL_OPTS         (libGAP_FIRST_EXPR_TNUM+81)
#define libGAP_T_FLOAT_EXPR_EAGER      (libGAP_FIRST_EXPR_TNUM+82)
#define libGAP_T_FLOAT_EXPR_LAZY       (libGAP_FIRST_EXPR_TNUM+83)

#define libGAP_T_ELM2_LIST             (libGAP_FIRST_EXPR_TNUM+84)
#define libGAP_T_ELMX_LIST             (libGAP_FIRST_EXPR_TNUM+85)
#define libGAP_T_ASS2_LIST             (libGAP_FIRST_EXPR_TNUM+86)
#define libGAP_T_ASSX_LIST             (libGAP_FIRST_EXPR_TNUM+87)

#define libGAP_LAST_EXPR_TNUM          libGAP_T_ASSX_LIST


/****************************************************************************
**
*F  TNUM_EXPR(<expr>) . . . . . . . . . . . . . . . . . type of an expression
**
**  'TNUM_EXPR' returns the type of the expression <expr>.
*/
#define libGAP_TNUM_EXPR(expr)         \
                        (libGAP_IS_REFLVAR( (expr) ) ? libGAP_T_REFLVAR : \
                         (libGAP_IS_INTEXPR( (expr) ) ? libGAP_T_INTEXPR : \
                          libGAP_TNUM_STAT(expr) ))


/****************************************************************************
**
*F  SIZE_EXPR(<expr>) . . . . . . . . . . . . . . . . . size of an expression
**
**  'SIZE_EXPR' returns the size of the expression <expr>.
**
**  Note  that  it is *fatal*  to apply  'SIZE_EXPR'   to expressions of type
**  'T_REFLVAR' or 'T_INTEXPR'.
*/
#define libGAP_SIZE_EXPR(expr) libGAP_SIZE_STAT(expr)


/****************************************************************************
**
*F  ADDR_EXPR(<expr>) . . . . . . . . . . . absolute address of an expression
**
**  'ADDR_EXPR' returns  the absolute  address  of  the memory  block of  the
**  expression <expr>.
**
**  Note  that  it is *fatal*  to apply  'ADDR_EXPR'   to expressions of type
**  'T_REFLVAR' or 'T_INTEXPR'.
*/
#define libGAP_ADDR_EXPR(expr) libGAP_ADDR_STAT(expr)


/****************************************************************************
**
*F  FUNC_CALL(<call>) . . . . . . . . . . . . .  function for a function call
*F  ARGI_CALL(<call>,<i>) . . . .  <i>-th formal argument for a function call
*F  NARG_SIZE_CALL(<size>)  . . . . . number of arguments for a function call
*F  SIZE_NARG_CALL(<narg>)  . . . . . . . size of the bag for a function call
**
**  'FUNC_CALL'  returns the expression that should  evaluate to the function
**  for the procedure or  function call <call>.   This is a legal left value,
**  so it can be used to set the expression too.
**
**  'ARGI_CALL'  returns  the expression that evaluate   to the <i>-th actual
**  argument for the procedure or function call <call>.  This is a legal left
**  value, so it can be used to set the expression too.
**
**  'NARG_SIZE_CALL' returns the number of  arguments in a function call from
**  the size <size> of the function call bag (as returned by 'SIZE_EXPR').
**
**  'SIZE_NARG_CALL' returns the size a  function call bag  should have for a
**  function call bag with <narg> arguments.
*/
#define libGAP_FUNC_CALL(call)         (* (libGAP_ADDR_EXPR((call)) +0     ) )
#define libGAP_ARGI_CALL(call,i)       (* (libGAP_ADDR_EXPR((call)) +0 +(i)) )
#define libGAP_NARG_SIZE_CALL(size)    (((size) / sizeof(libGAP_Expr)) - 1)
#define libGAP_SIZE_NARG_CALL(narg)    (((narg) + 1) * sizeof(libGAP_Expr))


/****************************************************************************
**
*F  ARGI_INFO(<info>,<i>) . . .  <i>-th formal argument for an Info statement
*F  NARG_SIZE_INFO(<size>)  . . . . number of arguments for an Info statement
*F  SIZE_NARG_INFO(<narg>)  . . . . . . size of the bag for an Info statement
**
**  'ARGI_INFO' returns the expression   that evaluates to the <i>-th  actual
**  argument for the Info  statement <info>.  This is a  legal left value, so
**  it can be used to set the expression too.
**
**  'NARG_SIZE_INFO' returns the number of  arguments in a function call from
**  the size <size> of the function call bag (as returned by 'SIZE_STAT').
**
**  'SIZE_NARG_INFO' returns the size a  function call bag  should have for a
**  function call bag with <narg> arguments.
*/
#define libGAP_ARGI_INFO(info,i)       (* (libGAP_ADDR_STAT((info))+(i) -1) )
#define libGAP_NARG_SIZE_INFO(size)    ((size) / sizeof(libGAP_Expr))
#define libGAP_SIZE_NARG_INFO(narg)    ((narg) * sizeof(libGAP_Expr))


/****************************************************************************
**
*V  CodeResult  . . . . . . . . . . . . . . . . . . . . . .  result of coding
**
**  'CodeResult'  is the result  of the coding, i.e.,   the function that was
**  coded.
*/
extern  libGAP_Obj             libGAP_CodeResult;


/****************************************************************************
**
*F  PushStat(<stat>)  . . . . . . . . . . . . . push statement onto the stack
*F  PopStat() . . . . . . . . . . . . . . . . .  pop statement from the stack
**
**  'StackStat' is the stack of statements that have been coded.
**
**  'CountStat'   is the number   of statements  currently on  the statements
**  stack.
**
**  'PushStat'  pushes the statement  <stat> onto the  statements stack.  The
**  stack is automatically resized if necessary.
**
**  'PopStat' returns the  top statement from the  statements  stack and pops
**  it.  It is an error if the stack is empty.
*/
extern void libGAP_PushStat (
            libGAP_Stat                stat );

extern libGAP_Stat libGAP_PopStat ( void );


/****************************************************************************
**

*F * * * * * * * * * * * * *  coder functions * * * * * * * * * * * * * * * *
*/

/****************************************************************************
**

*F  CodeBegin() . . . . . . . . . . . . . . . . . . . . . . . start the coder
*F  CodeEnd(<error>)  . . . . . . . . . . . . . . . . . . . .  stop the coder
**
**  'CodeBegin'  starts  the  coder.    It is   called  from  the   immediate
**  interpreter   when he encounters  a construct  that it cannot immediately
**  interpret.
**
**  'CodeEnd' stops the coder.  It  is called from the immediate  interpreter
**  when he is done with the construct  that it cannot immediately interpret.
**  If <error> is  non-zero, a syntax error  was detected by the  reader, and
**  the coder should only clean up.
**
**  ...only function expressions inbetween...
*/
extern  void            libGAP_CodeBegin ( void );

extern  libGAP_UInt            libGAP_CodeEnd (
            libGAP_UInt                error );


/****************************************************************************
**
*F  CodeFuncCallBegin() . . . . . . . . . . . . . . code function call, begin
*F  CodeFuncCallEnd(<funccall>,<options>, <nr>)  . code function call, end
**
**  'CodeFuncCallBegin'  is an action to code  a function call.  It is called
**  by the reader  when it encounters the parenthesis  '(', i.e., *after* the
**  function expression is read.
**
**  'CodeFuncCallEnd' is an action to code a  function call.  It is called by
**  the reader when  it  encounters the parenthesis  ')',  i.e.,  *after* the
**  argument expressions are read.   <funccall> is 1  if  this is a  function
**  call,  and 0  if  this  is  a procedure  call.    <nr> is the   number of
**  arguments. <options> is 1 if options were present after the ':' in which
**  case the options have been read already.
*/
extern  void            libGAP_CodeFuncCallBegin ( void );

extern  void            libGAP_CodeFuncCallEnd (
            libGAP_UInt                funccall,
            libGAP_UInt                libGAP_options,
            libGAP_UInt                nr );


/****************************************************************************
**
*F  CodeFuncExprBegin(<narg>,<nloc>,<nams>,<startline>) . code function expression, begin
*F  CodeFuncExprEnd(<nr>) . . . . . . . . . . . code function expression, end
**
**  'CodeFuncExprBegin'  is an action to code  a  function expression.  It is
**  called when the reader encounters the beginning of a function expression.
**  <narg> is the number of  arguments (-1 if the  function takes a  variable
**  number of arguments), <nloc> is the number of locals, <nams> is a list of
**  local variable names.
**
**  'CodeFuncExprEnd'  is an action to  code  a function  expression.  It  is
**  called when the reader encounters the end of a function expression.  <nr>
**  is the number of statements in the body of the function.
**
*/
extern void libGAP_CodeFuncExprBegin (
            libGAP_Int                 narg,
            libGAP_Int                 nloc,
            libGAP_Obj                 nams,
            libGAP_Int startLine);

extern void libGAP_CodeFuncExprEnd (
            libGAP_UInt                nr,
            libGAP_UInt                mapsto );

/****************************************************************************
**
*F  CodeFuncCallOptionsBegin() . . . . . . . . . . . . .  code options, begin
*F  CodeFuncCallOptionsBeginElmName(<rnam>). . .  code options, begin element
*F  CodeFuncCallOptionsBeginElmExpr() . .. . . . .code options, begin element
*F  CodeFuncCallOptionsEndElm() . . .. .  . . . . . code options, end element
*F  CodeFuncCallOptionsEndElmEmpty() .. .  . . . . .code options, end element
*F  CodeFuncCallOptionsEnd(<nr>)  . . . . . . . . . . . . . code options, end
**
**  The net effect of all of these is to leave a record expression on the stack
**  containing the options record. It will be picked up by
**  CodeFuncCallEnd()
**
*/
extern void            libGAP_CodeFuncCallOptionsBegin ( void );


extern void            libGAP_CodeFuncCallOptionsBeginElmName (
    libGAP_UInt                rnam );

extern void            libGAP_CodeFuncCallOptionsBeginElmExpr ( void );

extern void            libGAP_CodeFuncCallOptionsEndElm ( void );


extern void            libGAP_CodeFuncCallOptionsEndElmEmpty ( void );

extern void            libGAP_CodeFuncCallOptionsEnd ( libGAP_UInt nr );


/****************************************************************************
**
*F  CodeIfBegin() . . . . . . . . . . . code if-statement, begin of statement
*F  CodeIfElif()  . . . . . . . . . . code if-statement, begin of elif-branch
*F  CodeIfElse()  . . . . . . . . . . code if-statement, begin of else-branch
*F  CodeIfBeginBody() . . . . . . . . . . .  code if-statement, begin of body
*F  CodeIfEndBody(<nr>) . . . . . . . . . . .  code if-statement, end of body
*F  CodeIfEnd(<nr>) . . . . . . . . . . . code if-statement, end of statement
**
**  'CodeIfBegin' is an  action to code an  if-statement.  It is called  when
**  the reader encounters the 'if', i.e., *before* the condition is read.
**
**  'CodeIfElif' is an action to code an if-statement.  It is called when the
**  reader encounters an 'elif', i.e., *before* the condition is read.
**
**  'CodeIfElse' is an action to code an if-statement.  It is called when the
**  reader encounters an 'else'.
**
**  'CodeIfBeginBody' is  an action to   code an if-statement.  It  is called
**  when  the  reader encounters the beginning   of the statement  body of an
**  'if', 'elif', or 'else' branch, i.e., *after* the condition is read.
**
**  'CodeIfEndBody' is an action to code an if-statement.   It is called when
**  the reader encounters the end of the  statements body of an 'if', 'elif',
**  or 'else' branch.  <nr> is the number of statements in the body.
**
**  'CodeIfEnd' is an action to code an if-statement.  It  is called when the
**  reader encounters the end of the statement.   <nr> is the number of 'if',
**  'elif', or 'else' branches.
*/
extern  void            libGAP_CodeIfBegin ( void );

extern  void            libGAP_CodeIfElif ( void );

extern  void            libGAP_CodeIfElse ( void );

extern  void            libGAP_CodeIfBeginBody ( void );

extern  void            libGAP_CodeIfEndBody (
            libGAP_UInt                nr );

extern  void            libGAP_CodeIfEnd (
            libGAP_UInt                nr );


/****************************************************************************
**
*F  CodeForBegin()  . . . . . . . . .  code for-statement, begin of statement
*F  CodeForIn() . . . . . . . . . . . . . . . . code for-statement, 'in' read
*F  CodeForBeginBody()  . . . . . . . . . . code for-statement, begin of body
*F  CodeForEndBody(<nr>)  . . . . . . . . . . code for-statement, end of body
*F  CodeForEnd()  . . . . . . . . . . .  code for-statement, end of statement
**
**  'CodeForBegin' is  an action to code  a for-statement.  It is called when
**  the reader encounters the 'for', i.e., *before* the variable is read.
**
**  'CodeForIn' is an action to code a for-statement.  It  is called when the
**  reader encounters  the 'in',  i.e., *after*  the  variable  is  read, but
**  *before* the list expression is read.
**
**  'CodeForBeginBody'  is an action to  code a for-statement.   It is called
**  when   the reader encounters the beginning   of the statement body, i.e.,
**  *after* the list expression is read.
**
**  'CodeForEndBody' is an action to code a for-statement.  It is called when
**  the reader encounters the end of the statement  body.  <nr> is the number
**  of statements in the body.
**
**  'CodeForEnd' is an action to code a for-statement.  It is called when the
**  reader encounters  the end of   the  statement, i.e., immediately   after
**  'CodeForEndBody'.
*/
extern  void            libGAP_CodeForBegin ( void );

extern  void            libGAP_CodeForIn ( void );

extern  void            libGAP_CodeForBeginBody ( void );

extern  void            libGAP_CodeForEndBody (
            libGAP_UInt                nr );

extern  void            libGAP_CodeForEnd ( void );

/****************************************************************************
**
*F  CodeAtomicBegin()  . . . . . . .  code atomic-statement, begin of statement
*F  CodeAtomicBeginBody()  . . . . . . . . code atomic-statement, begin of body
*F  CodeAtomicEndBody( <nr> )  . . . . . . . code atomic-statement, end of body
*F  CodeAtomicEnd()  . . . . . . . . .  code atomic-statement, end of statement
**
**  'CodeAtomicBegin'  is an action to  code a atomic-statement.   It is called
**  when the  reader encounters the 'atomic',  i.e., *before* the condition is
**  read.
**
**  'CodeAtomicBeginBody'  is  an action   to code a  atomic-statement.   It is
**  called when  the reader encounters  the beginning  of the statement body,
**  i.e., *after* the condition is read.
**
**  'CodeAtomicEndBody' is an action to  code a atomic-statement.  It is called
**  when the reader encounters  the end of  the statement body.  <nr> is  the
**  number of statements in the body.
**
**  'CodeAtomicEnd' is an action to code a atomic-statement.  It is called when
**  the reader encounters  the end  of the  statement, i.e., immediate  after
**  'CodeAtomicEndBody'.
**
**  These functions are just placeholders for the future HPC-GAP code.
*/

void libGAP_CodeAtomicBegin ( void );

void libGAP_CodeAtomicBeginBody ( libGAP_UInt nrexprs );

void libGAP_CodeAtomicEndBody (
    libGAP_UInt                nrstats );
void libGAP_CodeAtomicEnd ( void );

/****************************************************************************
**
*F  CodeQualifiedExprBegin()  . . . code readonly/readwrite expression start
*F  CodeQualifiedExprEnd()  . . . . . code readonly/readwrite expression end
**
*/

void libGAP_CodeQualifiedExprBegin(libGAP_UInt qual);

void libGAP_CodeQualifiedExprEnd( void );


/****************************************************************************
**
*F  CodeWhileBegin()  . . . . . . .  code while-statement, begin of statement
*F  CodeWhileBeginBody()  . . . . . . . . code while-statement, begin of body
*F  CodeWhileEndBody(<nr>)  . . . . . . . . code while-statement, end of body
*F  CodeWhileEnd()  . . . . . . . . .  code while-statement, end of statement
**
**  'CodeWhileBegin'  is an action to  code a while-statement.   It is called
**  when the  reader encounters the 'while',  i.e., *before* the condition is
**  read.
**
**  'CodeWhileBeginBody'  is  an action   to code a  while-statement.   It is
**  called when  the reader encounters  the beginning  of the statement body,
**  i.e., *after* the condition is read.
**
**  'CodeWhileEndBody' is an action to  code a while-statement.  It is called
**  when the reader encounters  the end of  the statement body.  <nr> is  the
**  number of statements in the body.
**
**  'CodeWhileEnd' is an action to code a while-statement.  It is called when
**  the reader encounters  the end  of the  statement, i.e., immediate  after
**  'CodeWhileEndBody'.
*/
extern  void            libGAP_CodeWhileBegin ( void );

extern  void            libGAP_CodeWhileBeginBody ( void );

extern  void            libGAP_CodeWhileEndBody (
            libGAP_UInt                nr );

extern  void            libGAP_CodeWhileEnd ( void );


/****************************************************************************
**
*F  CodeRepeatBegin() . . . . . . . code repeat-statement, begin of statement
*F  CodeRepeatBeginBody() . . . . . . .  code repeat-statement, begin of body
*F  CodeRepeatEndBody(<nr>) . . . . . . .  code repeat-statement, end of body
*F  CodeRepeatEnd() . . . . . . . . . code repeat-statement, end of statement
**
**  'CodeRepeatBegin' is an action to code a  repeat-statement.  It is called
**  when the reader encounters the 'repeat'.
**
**  'CodeRepeatBeginBody' is an  action  to code  a  repeat-statement.  It is
**  called when the reader encounters  the  beginning of the statement  body,
**  i.e., immediately after 'CodeRepeatBegin'.
**
**  'CodeRepeatEndBody'   is an action  to code   a repeat-statement.  It  is
**  called when  the reader encounters the end  of  the statement body, i.e.,
**  *before* the condition is read.  <nr> is the  number of statements in the
**  body.
**
**  'CodeRepeatEnd' is an action to   code a repeat-statement.  It is  called
**  when  the reader encounters the end  of the statement,  i.e., *after* the
**  condition is read.
*/
extern  void            libGAP_CodeRepeatBegin ( void );

extern  void            libGAP_CodeRepeatBeginBody ( void );

extern  void            libGAP_CodeRepeatEndBody (
            libGAP_UInt                nr );

extern  void            libGAP_CodeRepeatEnd ( void );


/****************************************************************************
**
*F  CodeBreak() . . . . . . . . . . . . . . . . . . . .  code break-statement
**
**  'CodeBreak' is the  action to code a  break-statement.  It is called when
**  the reader encounters a 'break;'.
*/
extern  void            libGAP_CodeBreak ( void );


/****************************************************************************
**
*F  CodeReturnObj() . . . . . . . . . . . . . . . code return-value-statement
**
**  'CodeReturnObj' is the  action to code  a return-value-statement.  It  is
**  called when the reader encounters a 'return <expr>;', but *after* reading
**  the expression <expr>.
*/
extern  void            libGAP_CodeReturnObj ( void );


/****************************************************************************
**
*F  CodeReturnVoid()  . . . . . . . . . . . . . .  code return-void-statement
**
**  'CodeReturnVoid' is the action  to  code a return-void-statement.   It is
**  called when the reader encounters a 'return;'.
*/
extern  void            libGAP_CodeReturnVoid ( void );


/****************************************************************************
**
*F  CodeOr()  . . . . . . . . . . . . . . . . . . . . . .  code or-expression
*F  CodeAnd() . . . . . . . . . . . . . . . . . . . . . . code and-expression
*F  CodeNot() . . . . . . . . . . . . . . . . . . . . . . code not-expression
*F  CodeEq()  . . . . . . . . . . . . . . . . . . . . . . . code =-expression
*F  CodeNe()  . . . . . . . . . . . . . . . . . . . . . .  code <>-expression
*F  CodeLt()  . . . . . . . . . . . . . . . . . . . . . . . code <-expression
*F  CodeGe()  . . . . . . . . . . . . . . . . . . . . . .  code >=-expression
*F  CodeGt()  . . . . . . . . . . . . . . . . . . . . . . . code >-expression
*F  CodeLe()  . . . . . . . . . . . . . . . . . . . . . .  code <=-expression
*F  CodeIn()  . . . . . . . . . . . . . . . . . . . . . .  code in-expression
*F  CodeSum() . . . . . . . . . . . . . . . . . . . . . . . code +-expression
*F  CodeAInv()  . . . . . . . . . . . . . . . . . . . code unary --expression
*F  CodeDiff()  . . . . . . . . . . . . . . . . . . . . . . code --expression
*F  CodeProd()  . . . . . . . . . . . . . . . . . . . . . . code *-expression
*F  CodeInv() . . . . . . . . . . . . . . . . . . . . . . code ^-1-expression
*F  CodeQuo() . . . . . . . . . . . . . . . . . . . . . . . code /-expression
*F  CodeMod() . . . . . . . . . . . . . . . . . . . . . . code mod-expression
*F  CodePow() . . . . . . . . . . . . . . . . . . . . . . . code ^-expression
**
**  'CodeOr', 'CodeAnd', 'CodeNot',  'CodeEq', 'CodeNe',  'CodeGt', 'CodeGe',
**  'CodeIn',  'CodeSum',  'CodeDiff', 'CodeProd', 'CodeQuo',  'CodeMod', and
**  'CodePow' are the actions to   code the respective operator  expressions.
**  They are called by the reader *after* *both* operands are read.
*/
extern  void            libGAP_CodeOrL ( void );

extern  void            libGAP_CodeOr ( void );

extern  void            libGAP_CodeAndL ( void );

extern  void            libGAP_CodeAnd ( void );

extern  void            libGAP_CodeNot ( void );

extern  void            libGAP_CodeEq ( void );

extern  void            libGAP_CodeNe ( void );

extern  void            libGAP_CodeLt ( void );

extern  void            libGAP_CodeGe ( void );

extern  void            libGAP_CodeGt ( void );

extern  void            libGAP_CodeLe ( void );

extern  void            libGAP_CodeIn ( void );

extern  void            libGAP_CodeSum ( void );

extern  void            libGAP_CodeAInv ( void );

extern  void            libGAP_CodeDiff ( void );

extern  void            libGAP_CodeProd ( void );

extern  void            libGAP_CodeInv ( void );

extern  void            libGAP_CodeQuo ( void );

extern  void            libGAP_CodeMod ( void );

extern  void            libGAP_CodePow ( void );


/****************************************************************************
**
*F  CodeIntExpr(<str>)  . . . . . . . . . . . code literal integer expression
**
**  'CodeIntExpr' is the action to code a literal integer expression.  <str>
**  is the integer as a (null terminated) C character string.
*/
extern  void            libGAP_CodeIntExpr (
            libGAP_Char *              str );
extern  void            libGAP_CodeLongIntExpr (
            libGAP_Obj                 string ); 

/****************************************************************************
**
*F  CodeTrueExpr()  . . . . . . . . . . . . . .  code literal true expression
**
**  'CodeTrueExpr' is the action to code a literal true expression.
*/
extern  void            libGAP_CodeTrueExpr ( void );


/****************************************************************************
**
*F  CodeFalseExpr() . . . . . . . . . . . . . . code literal false expression
**
**  'CodeFalseExpr' is the action to code a literal false expression.
*/
extern  void            libGAP_CodeFalseExpr ( void );


/****************************************************************************
**
*F  CodeCharExpr(<chr>) . . . . . . . . . . code literal character expression
**
**  'CodeCharExpr'  is the action  to  code a  literal  character expression.
**  <chr> is the C character.
*/
extern  void            libGAP_CodeCharExpr (
            libGAP_Char                chr );


/****************************************************************************
**
*F  CodePermCycle(<nrx>,<nrc>)  . . . . . code literal permutation expression
*F  CodePerm(<nrc>) . . . . . . . . . . . code literal permutation expression
**
**  'CodePermCycle'  is an action to code  a  literal permutation expression.
**  It is called when one cycles is read completely.  <nrc>  is the number of
**  elements in that cycle.  <nrx> is the number of that  cycles (i.e., 1 for
**  the first cycle, 2 for the second, and so on).
**
**  'CodePerm' is an action to code a  literal permutation expression.  It is
**  called when  the permutation is read completely.   <nrc> is the number of
**  cycles.
*/
extern  void            libGAP_CodePermCycle (
            libGAP_UInt                nrx,
            libGAP_UInt                nrc );

extern  void            libGAP_CodePerm (
            libGAP_UInt                nrc );


/****************************************************************************
**
*F  CodeListExprBegin(<top>)  . . . . . . . . . . code list expression, begin
*F  CodeListExprBeginElm(<pos>) . . . . . code list expression, begin element
*F  CodeListExprEndElm()  . . . . . . . . . code list expression, end element
*F  CodeListExprEnd(<nr>,<range>,<top>,<tilde>) . . code list expression, end
*/
extern  void            libGAP_CodeListExprBegin (
            libGAP_UInt                top );

extern  void            libGAP_CodeListExprBeginElm (
            libGAP_UInt                pos );

extern  void            libGAP_CodeListExprEndElm ( void );

extern  void            libGAP_CodeListExprEnd (
            libGAP_UInt                nr,
            libGAP_UInt                range,
            libGAP_UInt                top,
            libGAP_UInt                tilde );


/****************************************************************************
**
*F  CodeStringExpr(<str>) . . . . . . . . . .  code literal string expression
*/
extern  void            libGAP_CodeStringExpr (
            libGAP_Obj              str );

/****************************************************************************
**
*F  CodeFloatExpr(<str>) . . . . . . . . . .  code literal float expression
*/
extern  void            libGAP_CodeFloatExpr (
            libGAP_Char *              str );

extern  void            libGAP_CodeLongFloatExpr (
            libGAP_Obj              str );


/****************************************************************************
**
*F  CodeRecExprBegin(<top>) . . . . . . . . . . code record expression, begin
*F  CodeRecExprBeginElmName(<rnam>) . . code record expression, begin element
*F  CodeRecExprBeginElmExpr() . . . . . code record expression, begin element
*F  CodeRecExprEndElmExpr() . . . . . . . code record expression, end element
*F  CodeRecExprEnd(<nr>,<top>,<tilde>)  . . . . . code record expression, end
*/
extern  void            libGAP_CodeRecExprBegin (
            libGAP_UInt                top );

extern  void            libGAP_CodeRecExprBeginElmName (
            libGAP_UInt                rnam );

extern  void            libGAP_CodeRecExprBeginElmExpr ( void );

extern  void            libGAP_CodeRecExprEndElm ( void );

extern  void            libGAP_CodeRecExprEnd (
            libGAP_UInt                nr,
            libGAP_UInt                top,
            libGAP_UInt                tilde );


/****************************************************************************
**
*F  CodeAssLVar(<lvar>) . . . . . . . . . . . . . .  code assignment to local
**
**  'CodeAssLVar' is the action  to code an  assignment to the local variable
**  <lvar> (given  by its  index).  It is   called by the  reader *after* the
**  right hand side expression is read.
**
**  An assignment  to a  local variable  is   represented by a  bag with  two
**  subexpressions.  The  *first* is the local variable,  the *second* is the
**  right hand side expression.
*/
extern  void            libGAP_CodeAssLVar (
            libGAP_UInt                lvar );

extern  void            libGAP_CodeUnbLVar (
            libGAP_UInt                lvar );


/****************************************************************************
**
*F  CodeRefLVar(<lvar>) . . . . . . . . . . . . . . . code reference to local
**
**  'CodeRefLVar' is  the action  to code a  reference  to the local variable
**  <lvar> (given  by its   index).  It is   called by  the  reader  when  it
**  encounters a local variable.
**
**  A   reference to   a local  variable    is represented immediately   (see
**  'REFLVAR_LVAR').
*/
extern  void            libGAP_CodeRefLVar (
            libGAP_UInt                lvar );

extern  void            libGAP_CodeIsbLVar (
            libGAP_UInt                lvar );


/****************************************************************************
**
*F  CodeAssHVar(<hvar>) . . . . . . . . . . . . . . code assignment to higher
**
**  'CodeAssHVar' is the action to code an  assignment to the higher variable
**  <hvar> (given by its  level  and  index).  It  is  called by  the  reader
**  *after* the right hand side expression is read.
**
**  An assignment to a higher variable is represented by a statement bag with
**  two subexpressions.  The *first* is the higher  variable, the *second* is
**  the right hand side expression.
*/
extern  void            libGAP_CodeAssHVar (
            libGAP_UInt                hvar );

extern  void            libGAP_CodeUnbHVar (
            libGAP_UInt                hvar );


/****************************************************************************
**
*F  CodeRefHVar(<hvar>) . . . . . . . . . . . . . .  code reference to higher
**
**  'CodeRefHVar' is the  action to code  a reference to the higher  variable
**  <hvar> (given by its level  and index).  It is  called by the reader when
**  it encounters a higher variable.
**
**  A reference to a higher variable is represented by an expression bag with
**  one subexpression.  This is the higher variable.
*/
extern  void            libGAP_CodeRefHVar (
            libGAP_UInt                hvar );

extern  void            libGAP_CodeIsbHVar (
            libGAP_UInt                hvar );


/****************************************************************************
**
*F  CodeAssGVar(<gvar>) . . . . . . . . . . . . . . code assignment to global
**
**  'CodeAssGVar' is the action to code  an assignment to the global variable
**  <gvar>.  It is  called   by  the reader    *after* the right   hand  side
**  expression is read.
**
**  An assignment to a global variable is represented by a statement bag with
**  two subexpressions.  The *first* is the  global variable, the *second* is
**  the right hand side expression.
*/
extern  void            libGAP_CodeAssGVar (
            libGAP_UInt                gvar );

extern  void            libGAP_CodeUnbGVar (
            libGAP_UInt                gvar );


/****************************************************************************
**
*F  CodeRefGVar(<gvar>) . . . . . . . . . . . . . .  code reference to global
**
**  'CodeRefGVar' is the  action to code a  reference to  the global variable
**  <gvar>.  It is called by the reader when it encounters a global variable.
**
**  A reference to a global variable is represented by an expression bag with
**  one subexpression.  This is the global variable.
*/
extern  void            libGAP_CodeRefGVar (
            libGAP_UInt                gvar );

extern  void            libGAP_CodeIsbGVar (
            libGAP_UInt                gvar );


/****************************************************************************
**
*F  CodeAssList() . . . . . . . . . . . . . . . . . code assignment to a list
*F  CodeAsssList()  . . . . . . . . . . .  code multiple assignment to a list
*F  CodeAssListLevel(<level>) . . . . . . .  code assignment to several lists
*F  CodeAsssListLevel(<level>)  . . code multiple assignment to several lists
*/
extern  void            libGAP_CodeAssList ( libGAP_Int narg );

extern  void            libGAP_CodeAsssList ( void );

extern  void            libGAP_CodeAssListLevel ( libGAP_Int narg,
            libGAP_UInt                level );

extern  void            libGAP_CodeAsssListLevel (
            libGAP_UInt                level );

extern  void            libGAP_CodeUnbList ( libGAP_Int narg );


/****************************************************************************
**
*F  CodeElmList() . . . . . . . . . . . . . . . . .  code selection of a list
*F  CodeElmsList()  . . . . . . . . . . . . code multiple selection of a list
*F  CodeElmListLevel(<level>) . . . . . . . . code selection of several lists
*F  CodeElmsListLevel(<level>)  . .  code multiple selection of several lists
*/
extern  void            libGAP_CodeElmList ( libGAP_Int narg );

extern  void            libGAP_CodeElmsList ( void );

extern  void            libGAP_CodeElmListLevel (
					  libGAP_Int narg,
					  libGAP_UInt level);

extern  void            libGAP_CodeElmsListLevel (
            libGAP_UInt                level );

extern  void            libGAP_CodeIsbList ( libGAP_Int narg );


/****************************************************************************
**
*F  CodeAssRecName(<rnam>)  . . . . . . . . . . . code assignment to a record
*F  CodeAssRecExpr()  . . . . . . . . . . . . . . code assignment to a record
*/
extern  void            libGAP_CodeAssRecName (
            libGAP_UInt                rnam );

extern  void            libGAP_CodeAssRecExpr ( void );

extern  void            libGAP_CodeUnbRecName (
            libGAP_UInt                rnam );

extern  void            libGAP_CodeUnbRecExpr ( void );


/****************************************************************************
**
*F  CodeElmRecName(<rnam>)  . . . . . . . . . . .  code selection of a record
*F  CodeElmRecExpr()  . . . . . . . . . . . . . .  code selection of a record
*/
extern  void            libGAP_CodeElmRecName (
            libGAP_UInt                rnam );

extern  void            libGAP_CodeElmRecExpr ( void );

extern  void            libGAP_CodeIsbRecName (
            libGAP_UInt                rnam );

extern  void            libGAP_CodeIsbRecExpr ( void );


/****************************************************************************
**
*F  CodeAssPosObj() . . . . . . . . . . . . . . . . code assignment to a list
*F  CodeAsssPosObj()  . . . . . . . . . .  code multiple assignment to a list
*F  CodeAssPosObjLevel(<level>) . . . . . .  code assignment to several lists
*F  CodeAsssPosObjLevel(<level>)  . code multiple assignment to several lists
*/
extern  void            libGAP_CodeAssPosObj ( void );

extern  void            libGAP_CodeAsssPosObj ( void );

extern  void            libGAP_CodeAssPosObjLevel (
            libGAP_UInt                level );

extern  void            libGAP_CodeAsssPosObjLevel (
            libGAP_UInt                level );

extern  void            libGAP_CodeUnbPosObj ( void );


/****************************************************************************
**
*F  CodeElmPosObj() . . . . . . . . . . . . . . . .  code selection of a list
*F  CodeElmsPosObj()  . . . . . . . . . . . code multiple selection of a list
*F  CodeElmPosObjLevel(<level>) . . . . . . . code selection of several lists
*F  CodeElmsPosObjLevel(<level>)  .  code multiple selection of several lists
*/
extern  void            libGAP_CodeElmPosObj ( void );

extern  void            libGAP_CodeElmsPosObj ( void );

extern  void            libGAP_CodeElmPosObjLevel (
            libGAP_UInt                level );

extern  void            libGAP_CodeElmsPosObjLevel (
            libGAP_UInt                level );

extern  void            libGAP_CodeIsbPosObj ( void );


/****************************************************************************
**
*F  CodeAssComObjName(<rnam>) . . . . . . . . . . code assignment to a record
*F  CodeAssComObjExpr() . . . . . . . . . . . . . code assignment to a record
*/
extern  void            libGAP_CodeAssComObjName (
            libGAP_UInt                rnam );

extern  void            libGAP_CodeAssComObjExpr ( void );

extern  void            libGAP_CodeUnbComObjName (
            libGAP_UInt                rnam );

extern  void            libGAP_CodeUnbComObjExpr ( void );


/****************************************************************************
**
*F  CodeElmComObjName(<rnam>) . . . . . . . . . .  code selection of a record
*F  CodeElmComObjExpr() . . . . . . . . . . . . .  code selection of a record
*/
extern  void            libGAP_CodeElmComObjName (
            libGAP_UInt                rnam );

extern  void            libGAP_CodeElmComObjExpr ( void );

extern  void            libGAP_CodeIsbComObjName (
            libGAP_UInt                rnam );

extern  void            libGAP_CodeIsbComObjExpr ( void );

/****************************************************************************
**
*F  CodeEmpty()  . . . . code an empty statement
**
*/

extern void libGAP_CodeEmpty( void );

/****************************************************************************
**
*F  CodeInfoBegin() . . . . . . . . . . . . .  start coding of Info statement
*F  CodeInfoMiddle()  . . . . . . . . .   shift to coding printable arguments
*F  CodeInfoEnd( <narg> ) . . Info statement complete, <narg> things to print
**
**  These  actions deal  with the  Info  statement, which is coded specially,
**  because not all of its arguments are always evaluated.
*/
extern  void            libGAP_CodeInfoBegin ( void );

extern  void            libGAP_CodeInfoMiddle ( void );

extern  void            libGAP_CodeInfoEnd   (
            libGAP_UInt                narg );


/****************************************************************************
**
*F  CodeAssertBegin() . . . . . . .  start interpretation of Assert statement
*F  CodeAsseerAfterLevel()  . . called after the first argument has been read
*F  CodeAssertAfterCondition() called after the second argument has been read
*F  CodeAssertEnd2Args() . . . . called after reading the closing parenthesis
*F  CodeAssertEnd3Args() . . . . called after reading the closing parenthesis
*/
extern  void            libGAP_CodeAssertBegin ( void );

extern  void            libGAP_CodeAssertAfterLevel ( void );

extern  void            libGAP_CodeAssertAfterCondition ( void );

extern  void            libGAP_CodeAssertEnd2Args ( void );

extern  void            libGAP_CodeAssertEnd3Args ( void );

/*  CodeContinue() .  . . . . . . . . . . . .  code continue-statement */
extern  void            libGAP_CodeContinue ( void );



/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**

*F  InitInfoCode() . . . . . . . . . . . . . . . . .  table of init functions
*/
libGAP_StructInitInfo * libGAP_InitInfoCode ( void );



#endif // GAP_CODE_H

/****************************************************************************
**

*E  code.h  . . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
