#ifndef __TUNER_MT2266_H
#define __TUNER_MT2266_H

/**

@file

@brief   MT2266 tuner module declaration

One can manipulate MT2266 tuner through MT2266 module.
MT2266 module is derived from tunerd module.



@par Example:
@code

// The example is the same as the tuner example in tuner_base.h except the listed lines.



#include "tuner_mt2266.h"


...



int main(void)
{
	TUNER_MODULE        *pTuner;
	MT2266_EXTRA_MODULE *pTunerExtra;

	TUNER_MODULE          TunerModuleMemory;
	BASE_INTERFACE_MODULE BaseInterfaceModuleMemory;
	I2C_BRIDGE_MODULE     I2cBridgeModuleMemory;

	unsigned long BandwidthHz;


	...



	// Build MT2266 tuner module.
	BuildMt2266Module(
		&pTuner,
		&TunerModuleMemory,
		&BaseInterfaceModuleMemory,
		&I2cBridgeModuleMemory,
		0xc0								// I2C device address is 0xc0 in 8-bit format.
		);





	// Get MT2266 tuner extra module.
	pTunerExtra = &(pTuner->Extra.Mt2266);

	// Open MT2266 handle.
	pTunerExtra->OpenHandle(pTuner);





	// ==== Initialize tuner and set its parameters =====

	...

	// Set MT2266 bandwidth.
	pTunerExtra->SetBandwidthHz(pTuner, MT2266_BANDWIDTH_6MHZ);





	// ==== Get tuner information =====

	...

	// Get MT2266 bandwidth.
	pTunerExtra->GetBandwidthHz(pTuner, &BandwidthHz);





	// Close MT2266 handle.
	pTunerExtra->CloseHandle(pTuner);



	// See the example for other tuner functions in tuner_base.h


	return 0;
}


@endcode

*/





// The following context is source code provided by Microtune.





// Microtune source code - mt_errordef.h



/*****************************************************************************
**
**  Name: mt_errordef.h
**
**  Description:    Definition of bits in status/error word used by various
**                  MicroTuner control programs.
**
**  References:     None
**
**  Exports:        None
**
**  CVS ID:         $Id: mt_errordef.h,v 1.1 2006/06/22 20:18:12 software Exp $
**  CVS Source:     $Source: /export/home/cvsroot/software/tuners/MT2266/mt_errordef.h,v $
**	               
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   09-09-2004    JWS    Original
**   088   01-26-2005    DAD    Added MT_TUNER_INIT_ERR.
**   N/A   12-09-2005    DAD    Added MT_TUNER_TIMEOUT (info).
**
*****************************************************************************/

/*
** Note to users:  DO NOT EDIT THIS FILE  
**
** If you wish to rename any of the "user defined" bits,
** it should be done in the user file that includes this
** source file (e.g. mt_userdef.h)
**
*/



/*
**  3 3 2 2 2 2 2 2 2 2 2 2 1 1 1 1 1 1 1 1 1 1 
**  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
**  M U <- Info Codes --> <# Spurs> < User> <----- Err Codes ----->
**
**  31 = MT_ERROR - Master Error Flag.  If set, check Err Codes for reason.
**  30 = MT_USER_ERROR - User-declared error flag.
**  29 = Unused
**  28 = Unused
**  27 = MT_DNC_RANGE
**  26 = MT_UPC_RANGE
**  25 = MT_FOUT_RANGE
**  24 = MT_FIN_OUT_OF_RANGE
**  23 = MT_SPUR_PRESENT - Unavoidable spur in output
**  22 = MT_TUNER_TIMEOUT
**  21 = Unused
**  20 = Unused
**  19 = MT_SPUR_CNT_MASK (MSB) - Count of avoided spurs
**  18 = MT_SPUR_CNT_MASK
**  17 = MT_SPUR_CNT_MASK
**  16 = MT_SPUR_CNT_MASK
**  15 = MT_SPUR_CNT_MASK (LSB)
**  14 = MT_USER_DEFINED4 - User-definable bit (see MT_Userdef.h)
**  13 = MT_USER_DEFINED3 - User-definable bit (see MT_Userdef.h)
**  12 = MT_USER_DEFINED2 - User-definable bit (see MT_Userdef.h)
**  11 = MT_USER_DEFINED1 - User-definable bit (see MT_Userdef.h)
**  10 = Unused
**   9 = MT_TUNER_INIT_ERR - Tuner initialization error
**   8 = MT_TUNER_ID_ERR - Tuner Part Code / Rev Code mismatches expected value
**   7 = MT_TUNER_CNT_ERR - Attempt to open more than MT_TUNER_CNT tuners
**   6 = MT_ARG_NULL - Null pointer passed as argument
**   5 = MT_ARG_RANGE - Argument value out of range
**   4 = MT_INV_HANDLE - Tuner handle is invalid
**   3 = MT_COMM_ERR - Serial bus communications error
**   2 = MT_DNC_UNLOCK - Downconverter PLL is unlocked
**   1 = MT_UPC_UNLOCK - Upconverter PLL is unlocked
**   0 = MT_UNKNOWN - Unknown error
*/
#define MT_ERROR (1 << 31)
#define MT_USER_ERROR (1 << 30)

/*  Macro to be used to check for errors  */
#define MT_IS_ERROR(s) (((s) >> 30) != 0)
#define MT_NO_ERROR(s) (((s) >> 30) == 0)


#define MT_OK                           (0x00000000)

/*  Unknown error  */
#define MT_UNKNOWN                      (0x80000001)

/*  Error:  Upconverter PLL is not locked  */
#define MT_UPC_UNLOCK                   (0x80000002)

/*  Error:  Downconverter PLL is not locked  */
#define MT_DNC_UNLOCK                   (0x80000004)

/*  Error:  Two-wire serial bus communications error  */
#define MT_COMM_ERR                     (0x80000008)

/*  Error:  Tuner handle passed to function was invalid  */
#define MT_INV_HANDLE                   (0x80000010)

/*  Error:  Function argument is invalid (out of range)  */
#define MT_ARG_RANGE                    (0x80000020)

/*  Error:  Function argument (ptr to return value) was NULL  */
#define MT_ARG_NULL                     (0x80000040)

/*  Error: Attempt to open more than MT_TUNER_CNT tuners  */
#define MT_TUNER_CNT_ERR                (0x80000080)

/*  Error: Tuner Part Code / Rev Code mismatches expected value  */
#define MT_TUNER_ID_ERR                 (0x80000100)

/*  Error: Tuner Initialization failure  */
#define MT_TUNER_INIT_ERR               (0x80000200)

/*  User-definable fields (see mt_userdef.h)  */
#define MT_USER_DEFINED1                (0x00001000)
#define MT_USER_DEFINED2                (0x00002000)
#define MT_USER_DEFINED3                (0x00004000)
#define MT_USER_DEFINED4                (0x00008000)
#define MT_USER_MASK                    (0x4000f000)
#define MT_USER_SHIFT                   (12)

/*  Info: Mask of bits used for # of LO-related spurs that were avoided during tuning  */
#define MT_SPUR_CNT_MASK                (0x001f0000)
#define MT_SPUR_SHIFT                   (16)

/*  Info: Tuner timeout waiting for condition  */
#define MT_TUNER_TIMEOUT                (0x00400000)

/*  Info: Unavoidable LO-related spur may be present in the output  */
#define MT_SPUR_PRESENT                 (0x00800000)

/*  Info: Tuner input frequency is out of range */
#define MT_FIN_RANGE                    (0x01000000)

/*  Info: Tuner output frequency is out of range */
#define MT_FOUT_RANGE                   (0x02000000)

/*  Info: Upconverter frequency is out of range (may be reason for MT_UPC_UNLOCK) */
#define MT_UPC_RANGE                    (0x04000000)

/*  Info: Downconverter frequency is out of range (may be reason for MT_DPC_UNLOCK) */
#define MT_DNC_RANGE                    (0x08000000)























// Microtune source code - mt_userdef.h



/*****************************************************************************
**
**  Name: mt_userdef.h
**
**  Description:    User-defined data types needed by MicroTuner source code.
**
**                  Customers must provide the code for these functions
**                  in the file "mt_userdef.c".
**
**                  Customers must verify that the typedef's in the 
**                  "Data Types" section are correct for their platform.
**
**  Functions
**  Requiring
**  Implementation: MT_WriteSub
**                  MT_ReadSub
**                  MT_Sleep
**
**  References:     None
**
**  Exports:        None
**
**  CVS ID:         $Id: mt_userdef.h,v 1.1 2006/06/22 20:18:12 software Exp $
**  CVS Source:     $Source: /export/home/cvsroot/software/tuners/MT2266/mt_userdef.h,v $
**	               
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   03-25-2004    DAD    Original
**   082   12-06-2004    JWS    Multi-tuner support - requires MTxxxx_CNT 
**                              declarations
**
*****************************************************************************/
#if !defined( __MT_USERDEF_H )
#define __MT_USERDEF_H

//#include "mt_errordef.h"

#if defined( __cplusplus )     
extern "C"                     /* Use "C" external linkage                  */
{
#endif

/*
**  Data Types
*/
typedef unsigned char   U8Data;         /*  type corresponds to 8 bits      */
typedef unsigned int    UData_t;        /*  type must be at least 32 bits   */
typedef int             SData_t;        /*  type must be at least 32 bits   */
typedef void *          Handle_t;       /*  memory pointer type             */
typedef double          FData_t;        /*  floating point data type        */

#define MAX_UDATA         (4294967295U)  /*  max value storable in UData_t   */
	

/*
** Define an MTxxxx_CNT macro for each type of tuner that will be built
** into your application (e.g., MT2121, MT2060). MT_TUNER_CNT
** must be set to the SUM of all of the MTxxxx_CNT macros.
**
** #define MT2050_CNT  (1)
** #define MT2060_CNT  (1)
** #define MT2111_CNT  (1)
** #define MT2121_CNT  (3)
*/


#if !defined( MT_TUNER_CNT )
#define MT_TUNER_CNT               (1)  /*  total num of MicroTuner tuners  */
#endif

/*
**  Optional user-defined Error/Info Codes  (examples below)
**
**  This is the area where you can define user-specific error/info return
**  codes to be returned by any of the functions you are responsible for
**  writing such as MT_WriteSub() and MT_ReadSub.  There are four bits
**  available in the status field for your use.  When set, these
**  bits will be returned in the status word returned by any tuner driver
**  call.  If you OR in the MT_ERROR bit as well, the tuner driver code
**  will treat the code as an error.
**
**  The following are a few examples of errors you can provide.
**
**  Example 1:
**  You might check to see the hUserData handle is correct and issue 
**  MY_USERDATA_INVALID which would be defined like this:
**
**  #define MY_USERDATA_INVALID  (MT_USER_ERROR | MT_USER_DEFINED1)
**
**
**  Example 2:
**  You might be able to provide more descriptive two-wire bus errors:
**
**  #define NO_ACK   (MT_USER_ERROR | MT_USER_DEFINED1)
**  #define NO_NACK  (MT_USER_ERROR | MT_USER_DEFINED2)
**  #define BUS_BUSY (MT_USER_ERROR | MT_USER_DEFINED3)
**
**
**  Example 3:
**  You can also provide information (non-error) feedback:
**
**  #define MY_INFO_1   (MT_USER_DEFINED1)
**
**
**  Example 4:
**  You can combine the fields together to make a multi-bit field.
**  This one can provide the tuner number based off of the addr
**  passed to MT_WriteSub or MT_ReadSub.  It assumes that
**  MT_USER_DEFINED4 through MT_USER_DEFINED1 are contiguously. If
**  TUNER_NUM were OR'ed into the status word on an error, you could
**  use this to identify which tuner had the problem (and whether it
**  was during a read or write operation).
**
**  #define TUNER_NUM  ((addr & 0x07) >> 1) << MT_USER_SHIFT
**
*/

/*****************************************************************************
**
**  Name: MT_WriteSub
**
**  Description:    Write values to device using a two-wire serial bus.
**
**  Parameters:     hUserData  - User-specific I/O parameter that was
**                               passed to tuner's Open function.
**                  addr       - device serial bus address  (value passed
**                               as parameter to MTxxxx_Open)
**                  subAddress - serial bus sub-address (Register Address)
**                  pData      - pointer to the Data to be written to the 
**                               device 
**                  cnt        - number of bytes/registers to be written
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_COMM_ERR      - Serial bus communications error
**                      user-defined
**
**  Notes:          This is a callback function that is called from the
**                  the tuning algorithm.  You MUST provide code for this
**                  function to write data using the tuner's 2-wire serial 
**                  bus.
**
**                  The hUserData parameter is a user-specific argument.
**                  If additional arguments are needed for the user's
**                  serial bus read/write functions, this argument can be
**                  used to supply the necessary information.
**                  The hUserData parameter is initialized in the tuner's Open
**                  function.
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   03-25-2004    DAD    Original
**
*****************************************************************************/
UData_t MT2266_WriteSub(Handle_t hUserData, 
                    UData_t addr, 
                    U8Data subAddress, 
                    U8Data *pData, 
                    UData_t cnt);


/*****************************************************************************
**
**  Name: MT_ReadSub
**
**  Description:    Read values from device using a two-wire serial bus.
**
**  Parameters:     hUserData  - User-specific I/O parameter that was
**                               passed to tuner's Open function.
**                  addr       - device serial bus address  (value passed
**                               as parameter to MTxxxx_Open)
**                  subAddress - serial bus sub-address (Register Address)
**                  pData      - pointer to the Data to be written to the 
**                               device 
**                  cnt        - number of bytes/registers to be written
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_COMM_ERR      - Serial bus communications error
**                      user-defined
**
**  Notes:          This is a callback function that is called from the
**                  the tuning algorithm.  You MUST provide code for this
**                  function to read data using the tuner's 2-wire serial 
**                  bus.
**
**                  The hUserData parameter is a user-specific argument.
**                  If additional arguments are needed for the user's
**                  serial bus read/write functions, this argument can be
**                  used to supply the necessary information.
**                  The hUserData parameter is initialized in the tuner's Open
**                  function.
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   03-25-2004    DAD    Original
**
*****************************************************************************/
UData_t MT2266_ReadSub(Handle_t hUserData, 
                   UData_t addr, 
                   U8Data subAddress, 
                   U8Data *pData, 
                   UData_t cnt);


/*****************************************************************************
**
**  Name: MT_Sleep
**
**  Description:    Delay execution for "nMinDelayTime" milliseconds
**
**  Parameters:     hUserData     - User-specific I/O parameter that was
**                                  passed to tuner's Open function.
**                  nMinDelayTime - Delay time in milliseconds
**
**  Returns:        None.
**
**  Notes:          This is a callback function that is called from the
**                  the tuning algorithm.  You MUST provide code that
**                  blocks execution for the specified period of time. 
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   03-25-2004    DAD    Original
**
*****************************************************************************/
void MT2266_Sleep(Handle_t hUserData,
              UData_t nMinDelayTime);


#if defined(MT2060_CNT)
#if MT2060_CNT > 0
/*****************************************************************************
**
**  Name: MT_TunerGain  (for MT2060 only)
**
**  Description:    Measure the relative tuner gain using the demodulator
**
**  Parameters:     hUserData  - User-specific I/O parameter that was
**                               passed to tuner's Open function.
**                  pMeas      - Tuner gain (1/100 of dB scale).
**                               ie. 1234 = 12.34 (dB)
**
**  Returns:        status:
**                      MT_OK  - No errors
**                      user-defined errors could be set
**
**  Notes:          This is a callback function that is called from the
**                  the 1st IF location routine.  You MUST provide
**                  code that measures the relative tuner gain in a dB
**                  (not linear) scale.  The return value is an integer
**                  value scaled to 1/100 of a dB.
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   06-16-2004    DAD    Original
**   N/A   11-30-2004    DAD    Renamed from MT_DemodInputPower.  This name
**                              better describes what this function does.
**
*****************************************************************************/
UData_t MT_TunerGain(Handle_t hUserData,
                     SData_t* pMeas);
#endif
#endif

#if defined( __cplusplus )     
}
#endif

#endif























// Microtune source code - mt2266.h



/*****************************************************************************
**
**  Name: mt2266.h
**
**  Copyright 2007 Microtune, Inc. All Rights Reserved
**
**  This source code file contains confidential information and/or trade
**  secrets of Microtune, Inc. or its affiliates and is subject to the
**  terms of your confidentiality agreement with Microtune, Inc. or one of
**  its affiliates, as applicable.
**
*****************************************************************************/

/*****************************************************************************
**
**  Name: mt2266.h
**
**  Description:    Microtune MT2266 Tuner software interface.
**                  Supports tuners with Part/Rev code: 0x85.
**
**  Functions
**  Implemented:    UData_t  MT2266_Open
**                  UData_t  MT2266_Close
**                  UData_t  MT2266_ChangeFreq
**                  UData_t  MT2266_GetLocked
**                  UData_t  MT2266_GetParam
**                  UData_t  MT2266_GetReg
**                  UData_t  MT2266_GetUHFXFreqs
**                  UData_t  MT2266_GetUserData
**                  UData_t  MT2266_ReInit
**                  UData_t  MT2266_SetParam
**                  UData_t  MT2266_SetPowerModes
**                  UData_t  MT2266_SetReg
**                  UData_t  MT2266_SetUHFXFreqs
**                  UData_t  MT2266_Tune
**
**  References:     AN-00010: MicroTuner Serial Interface Application Note
**                  MicroTune, Inc.
**
**  Exports:        None
**
**  Dependencies:   MT_ReadSub(hUserData, IC_Addr, subAddress, *pData, cnt);
**                  - Read byte(s) of data from the two-wire bus.
**
**                  MT_WriteSub(hUserData, IC_Addr, subAddress, *pData, cnt);
**                  - Write byte(s) of data to the two-wire bus.
**
**                  MT_Sleep(hUserData, nMinDelayTime);
**                  - Delay execution for x milliseconds
**
**  CVS ID:         $Id: mt2266.h,v 1.3 2007/10/02 18:43:17 software Exp $
**  CVS Source:     $Source: /export/home/cvsroot/software/tuners/MT2266/mt2266.h,v $
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   05-30-2006    DAD    Ver 1.0: Modified version of mt2260.c (Ver 1.01).
**   N/A   11-01-2006    RSK    Ver 1.02: Adding Get/Set UHFXFreq access functions.
**   118   05-09-2007    RSK    Ver 1.05: Adding Standard MTxxxx_Tune() API.
**
*****************************************************************************/
#if !defined( __MT2266_H )
#define __MT2266_H

//#include "mt_userdef.h"

#if defined( __cplusplus )
extern "C"                     /* Use "C" external linkage                  */
{
#endif

/*
**  Parameter for function MT2266_GetParam & MT2266_SetParam that
**  specifies the tuning algorithm parameter to be read/written.
*/
typedef enum
{
    /*  tuner address                                  set by MT2266_Open() */
    MT2266_IC_ADDR,

    /*  max number of MT2266 tuners       set by MT2266_CNT in mt_userdef.h */
    MT2266_MAX_OPEN,

    /*  current number of open MT2266 tuners           set by MT2266_Open() */
    MT2266_NUM_OPEN,

    /*  Number of tuner registers                                           */
    MT2266_NUM_REGS,

    /*  crystal frequency                            (default: 18000000 Hz) */
    MT2266_SRO_FREQ,

    /*  min tuning step size                            (default: 50000 Hz) */
    MT2266_STEPSIZE,

    /*  input center frequency                   set by MT2266_ChangeFreq() */
    MT2266_INPUT_FREQ,

    /*  LO Frequency                             set by MT2266_ChangeFreq() */
    MT2266_LO_FREQ,

    /*  output channel bandwidth                      (default: 8000000 Hz) */
    MT2266_OUTPUT_BW,

    /*  Base band filter calibration RC code                 (default: N/A) */
    MT2266_RC2_VALUE,

    /*  Base band filter nominal cal RC code                 (default: N/A) */
    MT2266_RC2_NOMINAL,

    /*  RF attenuator A/D readback                              (read-only) */
    MT2266_RF_ADC,

    /*  BB attenuator A/D readback                              (read-only) */
    MT2266_BB_ADC,

    /*  RF attenuator setting                             (default: varies) */
    MT2266_RF_ATTN,

    /*  BB attenuator setting                             (default: varies) */
    MT2266_BB_ATTN,

    /*  RF external / internal atten control              (default: varies) */
    MT2266_RF_EXT,

    /*  BB external / internal atten control                   (default: 1) */
    MT2266_BB_EXT,

    /*  LNA gain setting (0-15)                           (default: varies) */
    MT2266_LNA_GAIN,

    /*  Decrement LNA Gain (where arg=min LNA Gain value)                   */
    MT2266_LNA_GAIN_DECR,

    /*  Increment LNA Gain (where arg=max LNA Gain value)                   */
    MT2266_LNA_GAIN_INCR,

    /*  Set for UHF max sensitivity mode                                    */
    MT2266_UHF_MAXSENS,

    /*  Set for UHF normal mode                                             */
    MT2266_UHF_NORMAL,

    MT2266_EOP                    /*  last entry in enumerated list         */
} MT2266_Param;


/*
**  Parameter for function MT2266_GetUHFXFreqs & MT2266_SetUHFXFreqs that
**  specifies the particular frequency crossover table to be read/written.
*/
typedef enum
{
    /*  Reference the UHF 0 filter, without any attenuation                 */
    MT2266_UHF0,

    /*  Reference the UHF 1 filter, without any attenuation                 */
    MT2266_UHF1,

    /*  Reference the UHF 0 filter, with attenuation                        */
    MT2266_UHF0_ATTEN,

    /*  Reference the UHF 1 filter, with attenuation                        */
    MT2266_UHF1_ATTEN,

    MT2266_NUMBER_OF_XFREQ_SETS    /*  last entry in enumerated list        */

} MT2266_UHFXFreq_Type;


#define  MT2266_NUM_XFREQS  (16)

typedef  UData_t MT2266_XFreq_Set[ MT2266_NUM_XFREQS ];

/*
**  Constants for Specifying Operating Band of the Tuner
*/
#define MT2266_VHF_BAND (0)
#define MT2266_UHF_BAND (1)
#define MT2266_L_BAND   (2)

/*
**  Constants for specifying power modes these values
**  are bit-mapped and can be added/OR'ed to indicate
**  multiple settings.  Examples:
**     MT2266_SetPowerModes(h, MT2266_NO_ENABLES + MT22266_SROsd);
**     MT2266_SetPowerModes(h, MT2266_ALL_ENABLES | MT22266_SRO_NOT_sd);
**     MT2266_SetPowerModes(h, MT2266_NO_ENABLES + MT22266_SROsd);
**     MT2266_SetPowerModes(h, MT2266_SROen + MT22266_LOen + MT2266_ADCen);
*/
#define MT2266_SROen       (0x01)
#define MT2266_LOen        (0x02)
#define MT2266_ADCen       (0x04)
#define MT2266_PDen        (0x08)
#define MT2266_DCOCen      (0x10)
#define MT2266_BBen        (0x20)
#define MT2266_MIXen       (0x40)
#define MT2266_LNAen       (0x80)
#define MT2266_ALL_ENABLES (0xFF)
#define MT2266_NO_ENABLES  (0x00)
#define MT2266_SROsd       (0x100)
#define MT2266_SRO_NOT_sd  (0x000)

/* ====== Functions which are declared in mt2266.c File ======= */

/******************************************************************************
**
**  Name: MT2266_Open
**
**  Description:    Initialize the tuner's register values.
**
**  Usage:          status = MT2266_Open(0xC0, &hMT2266, NULL);
**                  if (MT_IS_ERROR(status))
**                      //  Check error codes for reason
**
**  Parameters:     MT2266_Addr  - Serial bus address of the tuner.
**                  hMT2266      - Tuner handle passed back.
**                  hUserData    - User-defined data, if needed for the
**                                 MT_ReadSub() & MT_WriteSub functions.
**
**  Returns:        status:
**                      MT_OK             - No errors
**                      MT_TUNER_ID_ERR   - Tuner Part/Rev code mismatch
**                      MT_TUNER_INIT_ERR - Tuner initialization failed
**                      MT_COMM_ERR       - Serial bus communications error
**                      MT_ARG_NULL       - Null pointer argument passed
**                      MT_TUNER_CNT_ERR  - Too many tuners open
**
**  Dependencies:   MT_ReadSub  - Read byte(s) of data from the two-wire bus
**                  MT_WriteSub - Write byte(s) of data to the two-wire bus
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   02-03-2006   DAD/JWS Original.
**
******************************************************************************/
UData_t MT2266_Open(UData_t MT2266_Addr,
                    Handle_t* hMT2266,
                    Handle_t hUserData);


/******************************************************************************
**
**  Name: MT2266_Close
**
**  Description:    Release the handle to the tuner.
**
**  Parameters:     hMT2266      - Handle to the MT2266 tuner
**
**  Usage:          status = MT2266_Close(hMT2266);
**
**  Returns:        status:
**                      MT_OK         - No errors
**                      MT_INV_HANDLE - Invalid tuner handle
**
**  Dependencies:   mt_errordef.h - definition of error codes
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   02-03-2006   DAD/JWS Original.
**
******************************************************************************/
UData_t MT2266_Close(Handle_t hMT2266);


/****************************************************************************
**
**  Name: MT2266_ChangeFreq
**
**  Description:    Change the tuner's tuned frequency to f_in.
**
**  Parameters:     h           - Open handle to the tuner (from MT2266_Open).
**                  f_in        - RF input center frequency (in Hz).
**
**  Usage:          status = MT2266_ChangeFreq(hMT2266, f_in);
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_INV_HANDLE    - Invalid tuner handle
**                      MT_DNC_UNLOCK    - Downconverter PLL unlocked
**                      MT_COMM_ERR      - Serial bus communications error
**                      MT_FIN_RANGE     - Input freq out of range
**                      MT_DNC_RANGE     - Downconverter freq out of range
**
**  Dependencies:   MUST CALL MT2266_Open BEFORE MT2266_ChangeFreq!
**
**                  MT_ReadSub       - Read byte(s) of data from the two-wire-bus
**                  MT_WriteSub      - Write byte(s) of data to the two-wire-bus
**                  MT_Sleep         - Delay execution for x milliseconds
**                  MT2266_GetLocked - Checks to see if the PLL is locked
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   02-03-2006   DAD/JWS Original.
**
******************************************************************************/
UData_t MT2266_ChangeFreq(Handle_t h,
                          UData_t f_in);


/****************************************************************************
**
**  Name: MT2266_GetLocked
**
**  Description:    Checks to see if the PLL is locked.
**
**  Parameters:     h            - Open handle to the tuner (from MT2266_Open).
**
**  Usage:          status = MT2266_GetLocked(hMT2266);
**                  if (status & MT_DNC_UNLOCK)
**                      //  error!, PLL is unlocked
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_DNC_UNLOCK    - Downconverter PLL unlocked
**                      MT_COMM_ERR      - Serial bus communications error
**                      MT_INV_HANDLE    - Invalid tuner handle
**
**  Dependencies:   MT_ReadSub    - Read byte(s) of data from the serial bus
**                  MT_Sleep      - Delay execution for x milliseconds
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   02-03-2006   DAD/JWS Original.
**
****************************************************************************/
UData_t MT2266_GetLocked(Handle_t h);


/****************************************************************************
**
**  Name: MT2266_GetParam
**
**  Description:    Gets a tuning algorithm parameter.
**
**                  This function provides access to the internals of the
**                  tuning algorithm - mostly for testing purposes.
**
**  Parameters:     h           - Tuner handle (returned by MT2266_Open)
**                  param       - Tuning algorithm parameter
**                                (see enum MT2266_Param)
**                  pValue      - ptr to returned value
**
**                  param                   Description
**                  ----------------------  --------------------------------
**                  MT2266_IC_ADDR          Serial Bus address of this tuner
**                  MT2266_MAX_OPEN         Max number of MT2266's that can be open
**                  MT2266_NUM_OPEN         Number of MT2266's currently open
**                  MT2266_NUM_REGS         Number of tuner registers
**                  MT2266_SRO_FREQ         crystal frequency
**                  MT2266_STEPSIZE         minimum tuning step size
**                  MT2266_INPUT_FREQ       input center frequency
**                  MT2266_LO_FREQ          LO Frequency
**                  MT2266_OUTPUT_BW        Output channel bandwidth
**                  MT2266_RC2_VALUE        Base band filter cal RC code (method 2)
**                  MT2266_RF_ADC           RF attenuator A/D readback
**                  MT2266_RF_ATTN          RF attenuation (0-255)
**                  MT2266_RF_EXT           External control of RF atten
**                  MT2266_LNA_GAIN         LNA gain setting (0-15)
**                  MT2266_BB_ADC           BB attenuator A/D readback
**                  MT2266_BB_ATTN          Baseband attenuation (0-255)
**                  MT2266_BB_EXT           External control of BB atten
**
**  Usage:          status |= MT2266_GetParam(hMT2266,
**                                            MT2266_OUTPUT_BW,
**                                            &f_bw);
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_INV_HANDLE    - Invalid tuner handle
**                      MT_ARG_NULL      - Null pointer argument passed
**                      MT_ARG_RANGE     - Invalid parameter requested
**
**  Dependencies:   USERS MUST CALL MT2266_Open() FIRST!
**
**  See Also:       MT2266_SetParam, MT2266_Open
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   02-03-2006   DAD/JWS Original.
**
****************************************************************************/
UData_t MT2266_GetParam(Handle_t     h,
                        MT2266_Param param,
                        UData_t*     pValue);


/****************************************************************************
**
**  Name: MT2266_GetReg
**
**  Description:    Gets an MT2266 register.
**
**  Parameters:     h           - Tuner handle (returned by MT2266_Open)
**                  reg         - MT2266 register/subaddress location
**                  *val        - MT2266 register/subaddress value
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_COMM_ERR      - Serial bus communications error
**                      MT_INV_HANDLE    - Invalid tuner handle
**                      MT_ARG_NULL      - Null pointer argument passed
**                      MT_ARG_RANGE     - Argument out of range
**
**  Dependencies:   USERS MUST CALL MT2266_Open() FIRST!
**
**                  Use this function if you need to read a register from
**                  the MT2266.
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   02-03-2006   DAD/JWS Original.
**
****************************************************************************/
UData_t MT2266_GetReg(Handle_t h,
                      U8Data   reg,
                      U8Data*  val);


/****************************************************************************
**
**  Name: MT2266_GetUHFXFreqs
**
**  Description:    Retrieves the specified set of UHF Crossover Frequencies
**
**  Parameters:     h            - Open handle to the tuner (from MT2266_Open).
**
**  Usage:          MT2266_Freq_Set  tmpFreqs;
**                  status = MT2266_GetUHFXFreqs(hMT2266,
**                                               MT2266_UHF1_WITH_ATTENUATION,
**                                               tmpFreqs );
**                  if (status & MT_ARG_RANGE)
**                      // error, Invalid UHF Crossover Frequency Set requested.
**                  else
**                      for( int i = 0;  i < MT2266_NUM_XFREQS; i++ )
**                         . . .
**
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_ARG_RANGE     - freq_type is out of range.
**                      MT_INV_HANDLE    - Invalid tuner handle
**
**  Dependencies:   freqs_buffer *must* be defined of type MT2266_Freq_Set
**                     to assure sufficient space allocation!
**
**                  USERS MUST CALL MT2266_Open() FIRST!
**
**  See Also:       MT2266_SetUHFXFreqs
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   11-01-2006   RSK     Original.
**
****************************************************************************/
UData_t MT2266_GetUHFXFreqs(Handle_t             h,
                            MT2266_UHFXFreq_Type freq_type,
                            MT2266_XFreq_Set     freqs_buffer);


/****************************************************************************
**
**  Name: MT2266_GetUserData
**
**  Description:    Gets the user-defined data item.
**
**  Parameters:     h           - Tuner handle (returned by MT2266_Open)
**
**  Usage:          status = MT2266_GetUserData(hMT2266, &hUserData);
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_INV_HANDLE    - Invalid tuner handle
**                      MT_ARG_NULL      - Null pointer argument passed
**
**  Dependencies:   USERS MUST CALL MT2266_Open() FIRST!
**
**                  The hUserData parameter is a user-specific argument
**                  that is stored internally with the other tuner-
**                  specific information.
**
**                  For example, if additional arguments are needed
**                  for the user to identify the device communicating
**                  with the tuner, this argument can be used to supply
**                  the necessary information.
**
**                  The hUserData parameter is initialized in the tuner's
**                  Open function to NULL.
**
**  See Also:       MT2266_SetUserData, MT2266_Open
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   02-03-2006   DAD/JWS Original.
**
****************************************************************************/
UData_t MT2266_GetUserData(Handle_t  h,
                           Handle_t* hUserData);


/******************************************************************************
**
**  Name: MT2266_ReInit
**
**  Description:    Initialize the tuner's register values.
**
**  Parameters:     h           - Tuner handle (returned by MT2266_Open)
**
**  Returns:        status:
**                      MT_OK             - No errors
**                      MT_TUNER_ID_ERR   - Tuner Part/Rev code mismatch
**                      MT_TUNER_INIT_ERR - Tuner initialization failed
**                      MT_INV_HANDLE     - Invalid tuner handle
**                      MT_COMM_ERR       - Serial bus communications error
**
**  Dependencies:   MT_ReadSub  - Read byte(s) of data from the two-wire bus
**                  MT_WriteSub - Write byte(s) of data to the two-wire bus
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   02-03-2006   DAD/JWS Original.
**
******************************************************************************/
UData_t MT2266_ReInit(Handle_t h);


/****************************************************************************
**
**  Name: MT2266_SetParam
**
**  Description:    Sets a tuning algorithm parameter.
**
**                  This function provides access to the internals of the
**                  tuning algorithm.  You can override many of the tuning
**                  algorithm defaults using this function.
**
**  Parameters:     h           - Tuner handle (returned by MT2266_Open)
**                  param       - Tuning algorithm parameter
**                                (see enum MT2266_Param)
**                  nValue      - value to be set
**
**                  param                   Description
**                  ----------------------  --------------------------------
**                  MT2266_SRO_FREQ         crystal frequency
**                  MT2266_STEPSIZE         minimum tuning step size
**                  MT2266_INPUT_FREQ       Center of input channel
**                  MT2266_OUTPUT_BW        Output channel bandwidth
**                  MT2266_RF_ATTN          RF attenuation (0-255)
**                  MT2266_RF_EXT           External control of RF atten
**                  MT2266_LNA_GAIN         LNA gain setting (0-15)
**                  MT2266_LNA_GAIN_DECR    Decrement LNA Gain (arg=min)
**                  MT2266_LNA_GAIN_INCR    Increment LNA Gain (arg=max)
**                  MT2266_BB_ATTN          Baseband attenuation (0-255)
**                  MT2266_BB_EXT           External control of BB atten
**                  MT2266_UHF_MAXSENS      Set for UHF max sensitivity mode
**                  MT2266_UHF_NORMAL       Set for UHF normal mode
**
**  Usage:          status |= MT2266_SetParam(hMT2266,
**                                            MT2266_STEPSIZE,
**                                            50000);
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_INV_HANDLE    - Invalid tuner handle
**                      MT_ARG_NULL      - Null pointer argument passed
**                      MT_ARG_RANGE     - Invalid parameter requested
**                                         or set value out of range
**                                         or non-writable parameter
**
**  Dependencies:   USERS MUST CALL MT2266_Open() FIRST!
**
**  See Also:       MT2266_GetParam, MT2266_Open
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   02-03-2006   DAD/JWS Original.
**
****************************************************************************/
UData_t MT2266_SetParam(Handle_t     h,
                        MT2266_Param param,
                        UData_t      nValue);


/****************************************************************************
**
**  Name: MT2266_SetPowerModes
**
**  Description:    Sets the bits in the MT2266_ENABLES register and the
**                  SROsd bit in the MT2266_SROADC_CTRL register.
**
**  Parameters:     h           - Tuner handle (returned by MT2266_Open)
**                  flags       - Bit mask of flags to indicate enabled
**                                bits.
**
**  Usage:          status = MT2266_SetPowerModes(hMT2266, flags);
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_INV_HANDLE    - Invalid tuner handle
**
**  Dependencies:   USERS MUST CALL MT2266_Open() FIRST!
**
**                  The bits in the MT2266_ENABLES register and the
**                  SROsd bit are set according to the supplied flags.
**
**                  The pre-defined flags are as follows:
**                      MT2266_SROen
**                      MT2266_LOen
**                      MT2266_ADCen
**                      MT2266_PDen
**                      MT2266_DCOCen
**                      MT2266_BBen
**                      MT2266_MIXen
**                      MT2266_LNAen
**                      MT2266_ALL_ENABLES
**                      MT2266_NO_ENABLES
**                      MT2266_SROsd
**                      MT2266_SRO_NOT_sd
**
**                  ONLY the enable bits (or SROsd bit) specified in the
**                  flags parameter will be set.  Any flag which is not
**                  included, will cause that bit to be disabled.
**
**                  The ALL_ENABLES, NO_ENABLES, and SRO_NOT_sd constants
**                  are for convenience.  The NO_ENABLES and SRO_NOT_sd
**                  do not actually have to be included, but are provided
**                  for clarity.
**
**  See Also:       MT2266_Open
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   02-03-2006   DAD/JWS Original.
**
****************************************************************************/
UData_t MT2266_SetPowerModes(Handle_t h,
                             UData_t  flags);


/****************************************************************************
**
**  Name: MT2266_SetReg
**
**  Description:    Sets an MT2266 register.
**
**  Parameters:     h           - Tuner handle (returned by MT2266_Open)
**                  reg         - MT2266 register/subaddress location
**                  val         - MT2266 register/subaddress value
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_COMM_ERR      - Serial bus communications error
**                      MT_INV_HANDLE    - Invalid tuner handle
**                      MT_ARG_RANGE     - Argument out of range
**
**  Dependencies:   USERS MUST CALL MT2266_Open() FIRST!
**
**                  Use this function if you need to override a default
**                  register value
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   02-03-2006   DAD/JWS Original.
**
****************************************************************************/
UData_t MT2266_SetReg(Handle_t h,
                      U8Data   reg,
                      U8Data   val);


/****************************************************************************
**
**  Name: MT2266_SetUHFXFreqs
**
**  Description:    Retrieves the specified set of UHF Crossover Frequencies
**
**  Parameters:     h            - Open handle to the tuner (from MT2266_Open).
**
**  Usage:          MT2266_Freq_Set  tmpFreqs;
**                  status = MT2266_SetUHFXFreqs(hMT2266,
**                                               MT2266_UHF1_WITH_ATTENUATION,
**                                               tmpFreqs );
**                  if (status & MT_ARG_RANGE)
**                      // error, Invalid UHF Crossover Frequency Set requested.
**                  else
**                      for( int i = 0;  i < MT2266_NUM_XFREQS; i++ )
**                         . . .
**
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_ARG_RANGE     - freq_type is out of range.
**                      MT_INV_HANDLE    - Invalid tuner handle
**
**  Dependencies:   freqs_buffer *must* be defined of type MT2266_Freq_Set
**                     to assure sufficient space allocation!
**
**                  USERS MUST CALL MT2266_Open() FIRST!
**
**  See Also:       MT2266_SetUHFXFreqs
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   N/A   11_01-2006   RSK     Original.
**
****************************************************************************/
UData_t MT2266_SetUHFXFreqs(Handle_t             h,
                            MT2266_UHFXFreq_Type freq_type,
                            MT2266_XFreq_Set     freqs_buffer);


/****************************************************************************
**
**  Name: MT2266_Tune
**
**  Description:    Change the tuner's tuned frequency to f_in.
**
**  Parameters:     h           - Open handle to the tuner (from MT2266_Open).
**                  f_in        - RF input center frequency (in Hz).
**
**  Usage:          status = MT2266_Tune(hMT2266, f_in);
**
**  Returns:        status:
**                      MT_OK            - No errors
**                      MT_INV_HANDLE    - Invalid tuner handle
**                      MT_DNC_UNLOCK    - Downconverter PLL unlocked
**                      MT_COMM_ERR      - Serial bus communications error
**                      MT_FIN_RANGE     - Input freq out of range
**                      MT_DNC_RANGE     - Downconverter freq out of range
**
**  Dependencies:   MUST CALL MT2266_Open BEFORE MT2266_Tune!
**
**                  MT_ReadSub       - Read byte(s) of data from the two-wire-bus
**                  MT_WriteSub      - Write byte(s) of data to the two-wire-bus
**                  MT_Sleep         - Delay execution for x milliseconds
**                  MT2266_GetLocked - Checks to see if the PLL is locked
**
**  Revision History:
**
**   SCR      Date      Author  Description
**  -------------------------------------------------------------------------
**   118   05-09-2007   RSK     Original API Introduction (was ChangeFreq).
**
******************************************************************************/
UData_t MT2266_Tune(Handle_t h,
                    UData_t f_in);


#if defined( __cplusplus )
}
#endif

#endif























// The following context is MT2266 tuner API source code





/**

@file

@brief   MT2266 tuner module declaration

One can manipulate MT2266 tuner through MT2266 module.
MT2266 module is derived from tuner module.

*/


#include "tuner_base.h"





// Definitions

// MT2266 API option
#define __NO_CACHE__
#define MT2266_CNT		4


// Bandwidth modes
enum MT2266_BANDWIDTH_HZ
{
	MT2266_BANDWIDTH_5MHZ = 5000000,
	MT2266_BANDWIDTH_6MHZ = 6000000,
	MT2266_BANDWIDTH_7MHZ = 7000000,
	MT2266_BANDWIDTH_8MHZ = 8000000,
};





// Builder
void
BuildMt2266Module(
	TUNER_MODULE **ppTuner,
	TUNER_MODULE *pTunerModuleMemory,
	BASE_INTERFACE_MODULE *pBaseInterfaceModuleMemory,
	I2C_BRIDGE_MODULE *pI2cBridgeModuleMemory,
	unsigned char DeviceAddr
	);





// Manipulaing functions
void
mt2266_GetTunerType(
	TUNER_MODULE *pTuner,
	int *pTunerType
	);

void
mt2266_GetDeviceAddr(
	TUNER_MODULE *pTuner,
	unsigned char *pDeviceAddr
	);

int
mt2266_Initialize(
	TUNER_MODULE *pTuner
	);

int
mt2266_SetRfFreqHz(
	TUNER_MODULE *pTuner,
	unsigned long RfFreqHz
	);

int
mt2266_GetRfFreqHz(
	TUNER_MODULE *pTuner,
	unsigned long *pRfFreqHz
	);





// Extra manipulaing functions
int
mt2266_OpenHandle(
	TUNER_MODULE *pTuner
	);

int
mt2266_CloseHandle(
	TUNER_MODULE *pTuner
	);

void
mt2266_GetHandle(
	TUNER_MODULE *pTuner,
	void **pDeviceHandle
	);

int
mt2266_SetBandwidthHz(
	TUNER_MODULE *pTuner,
	unsigned long BandwidthHz
	);

int
mt2266_GetBandwidthHz(
	TUNER_MODULE *pTuner,
	unsigned long *pBandwidthHz
	);















#endif
