/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices.store;

import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterName;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.ClusterStateObserver;
import org.elasticsearch.cluster.ClusterStateUpdateTask;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.routing.IndexRoutingTable;
import org.elasticsearch.cluster.routing.IndexShardRoutingTable;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import org.elasticsearch.index.IndexService;
import org.elasticsearch.index.IndexSettings;
import org.elasticsearch.index.shard.IndexShard;
import org.elasticsearch.index.shard.IndexShardState;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.indices.IndicesService;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.transport.TransportChannel;
import org.elasticsearch.transport.TransportException;
import org.elasticsearch.transport.TransportRequest;
import org.elasticsearch.transport.TransportRequestHandler;
import org.elasticsearch.transport.TransportResponse;
import org.elasticsearch.transport.TransportResponseHandler;
import org.elasticsearch.transport.TransportService;

public class IndicesStore
extends AbstractComponent
implements ClusterStateListener,
Closeable {
    public static final Setting<TimeValue> INDICES_STORE_DELETE_SHARD_TIMEOUT = Setting.positiveTimeSetting("indices.store.delete.shard.timeout", new TimeValue(30L, TimeUnit.SECONDS), Setting.Property.NodeScope);
    public static final String ACTION_SHARD_EXISTS = "internal:index/shard/exists";
    private static final EnumSet<IndexShardState> ACTIVE_STATES = EnumSet.of(IndexShardState.STARTED, IndexShardState.RELOCATED);
    private final IndicesService indicesService;
    private final ClusterService clusterService;
    private final TransportService transportService;
    private final ThreadPool threadPool;
    private TimeValue deleteShardTimeout;

    @Inject
    public IndicesStore(Settings settings, IndicesService indicesService, ClusterService clusterService, TransportService transportService, ThreadPool threadPool) {
        super(settings);
        this.indicesService = indicesService;
        this.clusterService = clusterService;
        this.transportService = transportService;
        this.threadPool = threadPool;
        transportService.registerRequestHandler(ACTION_SHARD_EXISTS, ShardActiveRequest::new, "same", new ShardActiveRequestHandler());
        this.deleteShardTimeout = INDICES_STORE_DELETE_SHARD_TIMEOUT.get(settings);
        if (DiscoveryNode.isDataNode(settings)) {
            clusterService.add(this);
        }
    }

    @Override
    public void close() {
        if (DiscoveryNode.isDataNode(this.settings)) {
            this.clusterService.remove(this);
        }
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        if (!event.routingTableChanged()) {
            return;
        }
        if (event.state().blocks().disableStatePersistence()) {
            return;
        }
        for (IndexRoutingTable indexRoutingTable : event.state().routingTable()) {
            for (IndexShardRoutingTable indexShardRoutingTable : indexRoutingTable) {
                IndexSettings indexSettings;
                if (!this.shardCanBeDeleted(event.state(), indexShardRoutingTable)) continue;
                ShardId shardId = indexShardRoutingTable.shardId();
                IndexService indexService = this.indicesService.indexService(indexRoutingTable.getIndex());
                if (indexService == null) {
                    IndexMetaData indexMetaData = event.state().getMetaData().getIndexSafe(indexRoutingTable.getIndex());
                    indexSettings = new IndexSettings(indexMetaData, this.settings);
                } else {
                    indexSettings = indexService.getIndexSettings();
                }
                if (!this.indicesService.canDeleteShardContent(shardId, indexSettings)) continue;
                this.deleteShardIfExistElseWhere(event.state(), indexShardRoutingTable);
            }
        }
    }

    boolean shardCanBeDeleted(ClusterState state, IndexShardRoutingTable indexShardRoutingTable) {
        if (indexShardRoutingTable.size() == 0) {
            return false;
        }
        for (ShardRouting shardRouting : indexShardRoutingTable) {
            if (!shardRouting.started()) {
                return false;
            }
            DiscoveryNode node = state.nodes().get(shardRouting.currentNodeId());
            if (node == null) {
                return false;
            }
            if (shardRouting.relocatingNodeId() != null && (node = state.nodes().get(shardRouting.relocatingNodeId())) == null) {
                return false;
            }
            String localNodeId = state.getNodes().getLocalNode().getId();
            if (!localNodeId.equals(shardRouting.currentNodeId()) && !localNodeId.equals(shardRouting.relocatingNodeId())) continue;
            return false;
        }
        return true;
    }

    private void deleteShardIfExistElseWhere(ClusterState state, IndexShardRoutingTable indexShardRoutingTable) {
        ArrayList<Tuple<DiscoveryNode, ShardActiveRequest>> requests = new ArrayList<Tuple<DiscoveryNode, ShardActiveRequest>>(indexShardRoutingTable.size());
        String indexUUID = indexShardRoutingTable.shardId().getIndex().getUUID();
        ClusterName clusterName = state.getClusterName();
        for (ShardRouting shardRouting : indexShardRoutingTable) {
            DiscoveryNode discoveryNode = state.nodes().get(shardRouting.currentNodeId());
            assert (discoveryNode != null);
            requests.add(new Tuple<DiscoveryNode, ShardActiveRequest>(discoveryNode, new ShardActiveRequest(clusterName, indexUUID, shardRouting.shardId(), this.deleteShardTimeout)));
            if (shardRouting.relocatingNodeId() == null) continue;
            DiscoveryNode relocatingNode = state.nodes().get(shardRouting.relocatingNodeId());
            assert (relocatingNode != null);
            requests.add(new Tuple<DiscoveryNode, ShardActiveRequest>(relocatingNode, new ShardActiveRequest(clusterName, indexUUID, shardRouting.shardId(), this.deleteShardTimeout)));
        }
        ShardActiveResponseHandler responseHandler = new ShardActiveResponseHandler(indexShardRoutingTable.shardId(), state, requests.size());
        for (Tuple tuple : requests) {
            this.logger.trace("{} sending shard active check to {}", (Object)((ShardActiveRequest)tuple.v2()).shardId, tuple.v1());
            this.transportService.sendRequest((DiscoveryNode)tuple.v1(), ACTION_SHARD_EXISTS, (TransportRequest)tuple.v2(), responseHandler);
        }
    }

    private static class ShardActiveResponse
    extends TransportResponse {
        private boolean shardActive;
        private DiscoveryNode node;

        ShardActiveResponse() {
        }

        ShardActiveResponse(boolean shardActive, DiscoveryNode node) {
            this.shardActive = shardActive;
            this.node = node;
        }

        @Override
        public void readFrom(StreamInput in) throws IOException {
            super.readFrom(in);
            this.shardActive = in.readBoolean();
            this.node = new DiscoveryNode(in);
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            out.writeBoolean(this.shardActive);
            this.node.writeTo(out);
        }
    }

    private static class ShardActiveRequest
    extends TransportRequest {
        protected TimeValue timeout = null;
        private ClusterName clusterName;
        private String indexUUID;
        private ShardId shardId;

        public ShardActiveRequest() {
        }

        ShardActiveRequest(ClusterName clusterName, String indexUUID, ShardId shardId, TimeValue timeout) {
            this.shardId = shardId;
            this.indexUUID = indexUUID;
            this.clusterName = clusterName;
            this.timeout = timeout;
        }

        @Override
        public void readFrom(StreamInput in) throws IOException {
            super.readFrom(in);
            this.clusterName = new ClusterName(in);
            this.indexUUID = in.readString();
            this.shardId = ShardId.readShardId(in);
            this.timeout = new TimeValue(in.readLong(), TimeUnit.MILLISECONDS);
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            this.clusterName.writeTo(out);
            out.writeString(this.indexUUID);
            this.shardId.writeTo(out);
            out.writeLong(this.timeout.millis());
        }
    }

    private class ShardActiveRequestHandler
    implements TransportRequestHandler<ShardActiveRequest> {
        private ShardActiveRequestHandler() {
        }

        @Override
        public void messageReceived(final ShardActiveRequest request, final TransportChannel channel) throws Exception {
            IndexShard indexShard = this.getShard(request);
            if (indexShard == null) {
                channel.sendResponse(new ShardActiveResponse(false, IndicesStore.this.clusterService.localNode()));
            } else {
                ClusterStateObserver observer = new ClusterStateObserver(IndicesStore.this.clusterService, request.timeout, IndicesStore.this.logger, IndicesStore.this.threadPool.getThreadContext());
                boolean shardActive = this.shardActive(indexShard);
                if (shardActive) {
                    channel.sendResponse(new ShardActiveResponse(true, IndicesStore.this.clusterService.localNode()));
                } else {
                    observer.waitForNextChange(new ClusterStateObserver.Listener(){

                        @Override
                        public void onNewClusterState(ClusterState state) {
                            this.sendResult(ShardActiveRequestHandler.this.shardActive(ShardActiveRequestHandler.this.getShard(request)));
                        }

                        @Override
                        public void onClusterServiceClose() {
                            this.sendResult(false);
                        }

                        @Override
                        public void onTimeout(TimeValue timeout) {
                            this.sendResult(ShardActiveRequestHandler.this.shardActive(ShardActiveRequestHandler.this.getShard(request)));
                        }

                        public void sendResult(boolean shardActive) {
                            try {
                                channel.sendResponse(new ShardActiveResponse(shardActive, IndicesStore.this.clusterService.localNode()));
                            }
                            catch (IOException e) {
                                IndicesStore.this.logger.error(() -> new ParameterizedMessage("failed send response for shard active while trying to delete shard {} - shard will probably not be removed", (Object)request.shardId), (Throwable)e);
                            }
                            catch (EsRejectedExecutionException e) {
                                IndicesStore.this.logger.error(() -> new ParameterizedMessage("failed send response for shard active while trying to delete shard {} - shard will probably not be removed", (Object)request.shardId), (Throwable)e);
                            }
                        }
                    }, new ClusterStateObserver.ValidationPredicate(){

                        @Override
                        protected boolean validate(ClusterState newState) {
                            IndexShard indexShard = ShardActiveRequestHandler.this.getShard(request);
                            return indexShard == null || ShardActiveRequestHandler.this.shardActive(indexShard);
                        }
                    });
                }
            }
        }

        private boolean shardActive(IndexShard indexShard) {
            if (indexShard != null) {
                return ACTIVE_STATES.contains((Object)indexShard.state());
            }
            return false;
        }

        private IndexShard getShard(ShardActiveRequest request) {
            ClusterName thisClusterName = IndicesStore.this.clusterService.getClusterName();
            if (!thisClusterName.equals(request.clusterName)) {
                IndicesStore.this.logger.trace("shard exists request meant for cluster[{}], but this is cluster[{}], ignoring request", (Object)request.clusterName, (Object)thisClusterName);
                return null;
            }
            ShardId shardId = request.shardId;
            IndexService indexService = IndicesStore.this.indicesService.indexService(shardId.getIndex());
            if (indexService != null && indexService.indexUUID().equals(request.indexUUID)) {
                return indexService.getShardOrNull(shardId.id());
            }
            return null;
        }
    }

    private class ShardActiveResponseHandler
    implements TransportResponseHandler<ShardActiveResponse> {
        private final ShardId shardId;
        private final int expectedActiveCopies;
        private final ClusterState clusterState;
        private final AtomicInteger awaitingResponses;
        private final AtomicInteger activeCopies;

        public ShardActiveResponseHandler(ShardId shardId, ClusterState clusterState, int expectedActiveCopies) {
            this.shardId = shardId;
            this.expectedActiveCopies = expectedActiveCopies;
            this.clusterState = clusterState;
            this.awaitingResponses = new AtomicInteger(expectedActiveCopies);
            this.activeCopies = new AtomicInteger();
        }

        @Override
        public ShardActiveResponse newInstance() {
            return new ShardActiveResponse();
        }

        @Override
        public void handleResponse(ShardActiveResponse response) {
            IndicesStore.this.logger.trace("{} is {}active on node {}", (Object)this.shardId, (Object)(response.shardActive ? "" : "not "), (Object)response.node);
            if (response.shardActive) {
                this.activeCopies.incrementAndGet();
            }
            if (this.awaitingResponses.decrementAndGet() == 0) {
                this.allNodesResponded();
            }
        }

        @Override
        public void handleException(TransportException exp) {
            IndicesStore.this.logger.debug(() -> new ParameterizedMessage("shards active request failed for {}", (Object)this.shardId), (Throwable)exp);
            if (this.awaitingResponses.decrementAndGet() == 0) {
                this.allNodesResponded();
            }
        }

        @Override
        public String executor() {
            return "same";
        }

        private void allNodesResponded() {
            if (this.activeCopies.get() != this.expectedActiveCopies) {
                IndicesStore.this.logger.trace("not deleting shard {}, expected {} active copies, but only {} found active copies", (Object)this.shardId, (Object)this.expectedActiveCopies, (Object)this.activeCopies.get());
                return;
            }
            ClusterState latestClusterState = IndicesStore.this.clusterService.state();
            if (this.clusterState.getVersion() != latestClusterState.getVersion()) {
                IndicesStore.this.logger.trace("not deleting shard {}, the latest cluster state version[{}] is not equal to cluster state before shard active api call [{}]", (Object)this.shardId, (Object)latestClusterState.getVersion(), (Object)this.clusterState.getVersion());
                return;
            }
            IndicesStore.this.clusterService.submitStateUpdateTask("indices_store ([" + this.shardId + "] active fully on other nodes)", new ClusterStateUpdateTask(){

                @Override
                public boolean runOnlyOnMaster() {
                    return false;
                }

                @Override
                public ClusterState execute(ClusterState currentState) throws Exception {
                    if (ShardActiveResponseHandler.this.clusterState.getVersion() != currentState.getVersion()) {
                        IndicesStore.this.logger.trace("not deleting shard {}, the update task state version[{}] is not equal to cluster state before shard active api call [{}]", (Object)ShardActiveResponseHandler.this.shardId, (Object)currentState.getVersion(), (Object)ShardActiveResponseHandler.this.clusterState.getVersion());
                        return currentState;
                    }
                    try {
                        IndicesStore.this.indicesService.deleteShardStore("no longer used", ShardActiveResponseHandler.this.shardId, currentState);
                    }
                    catch (Exception ex) {
                        IndicesStore.this.logger.debug(() -> new ParameterizedMessage("{} failed to delete unallocated shard, ignoring", (Object)ShardActiveResponseHandler.this.shardId), (Throwable)ex);
                    }
                    return currentState;
                }

                @Override
                public void onFailure(String source, Exception e) {
                    IndicesStore.this.logger.error(() -> new ParameterizedMessage("{} unexpected error during deletion of unallocated shard", (Object)ShardActiveResponseHandler.this.shardId), (Throwable)e);
                }
            });
        }
    }
}

