require 'minitest_helper'

describe Fog::Compute::XenServer::Models::Vif do
  let(:vif_class) do
    class Fog::Compute::XenServer::Models::Vif
      def self.read_identity
        instance_variable_get('@identity')
      end
    end
    Fog::Compute::XenServer::Models::Vif
  end
  let(:vif) { Fog::Compute::XenServer::Models::Vif.new }
  let(:vm) { Fog::Compute::XenServer::Models::Server.new }
  let(:service) { Object.new }

  it 'should associate to a provider class' do
    vif_class.provider_class.must_equal('VIF')
  end

  it 'should have a collection name' do
    vif_class.collection_name.must_equal(:vifs)
  end

  it 'should have an unique id' do
    vif_class.read_identity.must_equal(:reference)
  end

  it 'should have 19 attributes' do
    vif_class.attributes.must_equal([ :reference,
                                      :allowed_operations,
                                      :current_operations,
                                      :currently_attached,
                                      :device,
                                      :ipv4_allowed,
                                      :ipv6_allowed,
                                      :locking_mode,
                                      :mac,
                                      :mac_autogenerated,
                                      :mtu,
                                      :other_config,
                                      :qos_algorithm_params,
                                      :qos_algorithm_type,
                                      :qos_supported_algorithms,
                                      :runtime_properties,
                                      :status_code,
                                      :status_detail,
                                      :uuid ])
  end

  it 'should have 3 associations' do
    vif_class.associations.must_equal(:metrics => :vifs_metrics,
                                      :network => :networks,
                                      :vm => :servers)
  end

  it 'should have 22 masks' do
    vif_class.masks.must_equal(:reference => :reference,
                               :allowed_operations => :allowed_operations,
                               :current_operations => :current_operations,
                               :currently_attached => :currently_attached,
                               :device => :device,
                               :ipv4_allowed => :ipv4_allowed,
                               :ipv6_allowed => :ipv6_allowed,
                               :locking_mode => :locking_mode,
                               :mac => :MAC,
                               :mac_autogenerated => :MAC_autogenerated,
                               :mtu => :MTU,
                               :other_config => :other_config,
                               :qos_algorithm_params => :qos_algorithm_params,
                               :qos_algorithm_type => :qos_algorithm_type,
                               :qos_supported_algorithms => :qos_supported_algorithms,
                               :runtime_properties => :runtime_properties,
                               :status_code => :status_code,
                               :status_detail => :status_detail,
                               :uuid => :uuid,
                               :metrics => :metrics,
                               :network => :network,
                               :vm => :VM)
  end

  it 'should have 4 aliases' do
    vif_class.aliases.must_equal(:MAC => :mac,
                                 :MAC_autogenerated => :mac_autogenerated,
                                 :MTU => :mtu,
                                 :VM => :vm)
  end

  it 'should have 6 default values' do
    vif_class.default_values.must_equal(:mac => '',
                                        :mac_autogenerated => 'True',
                                        :mtu => '0',
                                        :other_config => {},
                                        :qos_algorithm_params => {},
                                        :qos_algorithm_type => 'ratelimit')
  end

  it 'should require 3 attributes before save' do
    vif_class.require_before_save.must_equal([ :vm, :network, :device ])
  end

  describe '#set_device_number' do
    describe 'when vm is nil' do
      it 'should set the device as 0' do
        lambda { vif.set_device_number }.must_raise(ArgumentError, 'vm is required for this operation')
      end
    end

    describe 'when vm is not nil' do
      describe 'and no vif exists' do
        before :each do
          vif.vm = vm
          vif.stub(:vm, vm) do
            vm.stub(:vifs, []) do
              vif.set_device_number
            end
          end
        end

        it 'should set the device as 0' do
          vif.device.must_equal('0')
        end
      end

      describe 'and exists vifs' do
        before :each do
          @vif2 = Fog::Compute::XenServer::Models::Vif.new(:device => 2)
          @vif4 = Fog::Compute::XenServer::Models::Vif.new(:device => 4)
          vif.stub(:vm, vm) do
            vm.stub(:vifs, [@vif2, @vif4]) do
              vif.set_device_number
            end
          end
        end

        it 'should set the device value based on the existent vifs' do
          vif.device.must_equal('5')
        end
      end
    end
  end
end